//	Turns on runtime lookup of globals, strict wave references and runtime checking of wave index bounds. Requires Igor Pro 6.2 or later.
#pragma rtGlobals=3

//	Places all the functions in this procedure in the module EccentricXPS. This prevent naming conflicts with other procedures. To call a function from outside the procedure or from the command line use EccentricXPS#FunctionName()
#pragma ModuleName=EccentricXPS

//	Used by EccentricXPS to create a history of the procedure versions that have been used to analyze an experiment file
#pragma version = 1.100



//	This is an Igor procedure for XPS analysis written by Ole Lytken (ole.lytken@fau.de) at the Lehrstuhl fuer Physikalische Chemie II, Universitaet Erlangen-Nuernberg, Germany.
//	The procedure was written to load and analyse XPS and NEXAFS data, as created by Scienta's SES (version 1.2.2, build 37) for XPS, and KolXPD (version 1.8.0.42) used at
//	the Materials Science beamline at Elettra Sincrotrone Trieste for NEXAFS. However, the procedure is also able to load spectra from generic two-column (X, Y) text files and will work with all waves present in Igor.

//	The focus of the procedure is on a smooth workflow and easy comparison and fitting of data and is able to fit multiple spectra simultaneously.
//	The focus is not on publication-ready figures, although the procedure will create decent starting points for such figures.
//	The heart of the procedure is the View Spectra data browser, which will allows for quick displaying and comparison of data.
//	It is intended for XPS spectra, but could be convenient for displaying and comparing any type of one-dimensional numeric data.

//	The current version of the procedure (Mar 2017) was written in Igor Pro 6.37 and is unlikely to function properly in earlier versions of Igor.
//	Igor Pro 6.34A also seems to work, but I make no promises.
//	To download the latest version of Igor Pro visit https://www.wavemetrics.com. A useful place to find answers to technical question about programming in Igor Pro can be found at http://www.igorexchange.com.

//	At the Igor Exchange you will also find other Igor procedures written for XPS analysis, and if you're interested in a straightforward and easy-to-use procedure
//	with a more narrow focus on just fitting a single spectrum, I can recommend XPS Tools (XPST) by Martin Schmid.

//	To install the procedure, copy this file (EccentricXPS_v1.10.ipf), or a shortcut to the file, into your Igor Procedures folder, typically located in C:\Program Files\WaveMetrics\Igor Pro Folder\Igor Procedures.
//	To have access to the help file you also need to copy the EccentricXPS_Help_v1.10.ihf file, or a shortcut to the file, into your Igor Help Files folder, typically located in C:\Program Files\WaveMetrics\Igor Pro Folder\Igor Help Files.
//	The procedure also comes with an Igor experiment file EccentricXPS_Examples_v1.10.pxp which contains a selection of spectra to test the full functionality of the procedure on.

//	To navigate the procedure file right click the function names in the menus below, and you will be given the option to go to that function.
//	The functions are roughly in the same order as the menu items that use them.



//	All items associated with XPS analysis
Menu "XPS"
	//	Loads the raw 2D text files produced by the Scienta SES software. The first valid path in the semicolon separated string determines the path the load file dialog will start in. The first path is valid on my office computer and the second on my personal laptop (HD: should be the Mac equivalent of C:)
	"Load VG Scienta", /Q, EccentricXPS#XPSLoadVGScienta("D:Work:Erlangen:Scienta:Data;C:Work:Erlangen:Scienta:Data;D:;C:;HD:;:;")
	//	Loads generic two column text files.  The first valid path in the semicolon separated string determines the path the load file dialog will start in. The first path is valid on my office computer and the second on my personal laptop (HD: should be the Mac equivalent of C:)
	"Load XY", /Q, EccentricXPS#XPSLoadXY("D:Work:Erlangen:Scienta:Data;C:Work:Erlangen:Scienta:Data;D:;C:;HD:;:;")
	//	Adds a blank line to the menu
	"-",;
	//	DIsplays the 2D detector images loaded from the Scienta SES text files. Displays the first wave in root:XPS:Images, unless the window already exists. Specify : to start in the default data folder set with SetDataFolder
	"View Images", /Q, EccentricXPS#XPSViewImages("root:XPS:Images:", "", 0)
	//	Browses, displays and compares XPS spectra. The specifier determines which data folder to display. Specify : to start in the default data folder set with SetDataFolder
	"View Spectra", /Q, EccentricXPS#XPSViewSpectra("root:XPS:Spectra:", 0)
	//	Fits XPS spectra. The two parameters determines the data folder and wave to display. Specify : to start in the default data folder set with SetDataFolder
	"Fit Assistant", /Q, EccentricXPS#XPSFitAssistant("root:XPS:Spectra:", "", 0)
	//	Used to compare multiple saved fits
	"Fit Overview", /Q, EccentricXPS#XPSFitOverview("")
	//	Adds a blank line to the menu
	"-",;
	//	Used to calculate coverages based on XPS intensities
	"Coverage AB", /Q, EccentricXPS#XPSCoverageAB()
	//	Used to calculate coverages based on XPS intensities
	"Coverage ABC", /Q, EccentricXPS#XPSCoverageABC()
	//	Adds a blank line to the menu
	"-",;
	//	Opens a notebook, that can be used to make notes in
	"Open Notebook", /Q, EccentricXPS#XPSOpenNoteBook("XPSNoteBook")
	//	Adds a blank line to the menu
	"-",;
	//	Prints the history of procedures used with the current experiment
	"Print Procedure History", /Q, EccentricXPS#XPSPrintProcedureHistory()
	//	Adds a blank line to the menu
	"-",;
	//	Opens the help file for the procedure
	"Help", /Q, DisplayHelpTopic /Z "EccentricXPS"
end



//	All items associated with NEXAFS analysis
Menu "NEXAFS"
	//	Loads the KolXPD NEXAFS files from Elettra. The first valid path in the semicolon separated string determines the path the load file dialog will start in. The first path is valid on my office computer and the second on my personal laptop (HD: should be the Mac equivalent of C:)
	"Load Elettra NEXAFS", /Q, EccentricXPS#LoadNEXAFS("D:Work:Erlangen:Scienta:Data;C:Work:Erlangen:Scienta:Data;D:;C:;HD:;:;")
	//	Adds a blank line to the menu
	"-",;
	//	Converts the raw NEXAFS XY images into waveform images with uniform X and Y steps. The two specifiers determines the data folder and wave to display. Specify : to start in the default data folder set with SetDataFolder
	"Convert to Waveform", /Q, EccentricXPS#RawNEXAFS("root:NEXAFS:RawImages:", "", 0)
	//	Converts the NEXAFS images into spectra. The specifiers determines the data folder, main NEXAFS image, the background image to subtract and the photon flux to divide by. Specify : to start in the default data folder set with SetDataFolder
	"View NEXAFS", /Q, EccentricXPS#ViewNEXAFS("root:NEXAFS:Images:", "", "", "", 0)
	//	Adds a blank line to the menu
	"-",;
	//	Creates the new display for the kinetic and binding energy images and NEXAFS, Auger and XPS spectra, for the 2D NEXAFS images acquired at Elettra
	"New NEXAFS Viewer", /Q, EccentricXPS#NewNEXAFSViewer("root:NEXAFS:Images:", "", 0)
	//	Creates the display for the photoemission cleanup of the 2D NEXAFS images acquired at Elettra
	"Photoemission Cleanup", /Q, EccentricXPS#NEXAFSCleanup("root:NEXAFS:Images:", "", 0)
	//	Adds a blank line to the menu
	"-",;
	//	Opens a notebook, that can be used to make notes in
	"Open Notebook", /Q, EccentricXPS#XPSOpenNoteBook("NEXAFSNoteBook")
	//	Adds a blank line to the menu
	"-",;
	//	Opens the help file for the procedure
	"Help", /Q, DisplayHelpTopic /Z "EccentricXPS"
end





//     -----<<<<<     XPS     >>>>>-----
//	The following sections contains the functions used by the menu items under the XPS header
//	The functions are grouped together by menu item as much as possible (some cross talk cannot be avoided), with a section of shared functions at the end

//     -----<<<<<     Load VG Scienta     >>>>>-----
//	This section contains the functions used by the Load VG Scienta menu item

Static Function XPSLoadVGScienta(DataPathList)
//	Loads the raw XPS detector images and creates the corresponding spectra
String DataPathList
String FileExtension="Raw XPS Data Files (*.txt):.txt;All Files (*.*):.*;"

	//	Updates the history of procedures used with the current experiment
	XPSUpdateProcedureHistory()
	
	//	Creates the necessary folders, if they do not already exist
	XPSCreateFolders()
	
	//	Specifies the default data directory any open file dialog will start in
	CreateDataPath(DataPathList)
	
	//	Display an open file dialog, showing only files with the extension .txt. Allows multiple files to be selected
	Open/D/F=FileExtension /R/MULT=1 /P=EccentricXPSDataPath RefNum
	String AllFileNames=S_FileName
	
	if (StrLen(AllFileNames)!=0)

		//	Prompts the user for the location of the new waves. The spectra and images will be places in subfolders named :Spectra and :Images
		XPSViewSaveWaveDialog("Select destination folder", "root:XPS:", "", "EccentricXPS#XPSLoadVGScientaLoadWaves", "", AllFileNames)
	endif
end



Static Function XPSLoadVGScientaLoadWaves(SaveFolder, SaveName, AllFileNames)
//	Loads the raw XPS detector images and creates the corresponding spectra
DFREF SaveFolder
String SaveName, AllFileNames

	//	Considers the full name as a data folder rather than the a folder and a wave (as the save dialog was intended for)
	if (StrLen(SaveName)>0)
		NewDataFolder/O SaveFolder:$SaveName
		DFREF LoadFolder=SaveFolder:$SaveName
	else
		DFREF LoadFolder=SaveFolder
	endif
	
	//	Checks if the folder exists
	if (DataFolderRefStatus(LoadFolder)==0)
		ErrorMessage("The data folder does not exist/could not be created!")
	else
	
		//	Creates the folders to places the spectra and images in
		NewDataFolder/O LoadFolder:Images
		NewDataFolder/O LoadFolder:Spectra

		//	Calculates the number of files selected to be loaded
		Variable NumberOfFilesLoaded=ItemsInList(AllFileNames, "\r")
		
		//	Prints a blank line
		Print("\r")
		
		//	Loads each file one at a time
		Variable i=0
		for (i=0; i<NumberOfFilesLoaded; i+=1)
			XPSLoadVGScientaSingleFile(StringFromList(i, AllFileNames, "\r"), LoadFolder)
		endfor

		//	Deletes all temporary waves
		DFREF TempFolder=root:Programming:Temp
		KillWavesInFolder(TempFolder)

		//	Indicates the loading is finished
		Print("\rDone Loading")
		
		//	Opens the View Spectra window, showing the data folder the waves were loaded into. The displayed data folder is not changed if the View Spectra window already exists
		DoWindow XPSViewSpectraWindow
		if (V_Flag==0)

			//	Creates the View Spectra window
			String SpectrumFolderName=GetDataFolder(1, LoadFolder:Spectra)
			XPSViewSpectraCreateWindow(SpectrumFolderName, 1)
		else

			//	Finds the active data folder in the View Spectra window
			DFREF ActiveFolder=XPSViewGetDataFolder("XPSViewSpectraWindow", "DataFolderPopUp")

			//	Refreshes the graph
			XPSViewSpectraUpdateGraph("XPSViewSpectraWindow", ActiveFolder, $"")
		endif
	endif
end



Static Function XPSCreateFolders()
//	Creates the necessay XPS folders, if they do not already exist

	//	The main XPS folder to hold all the subfolders.
	NewDataFolder/O root:XPS
	
	//	The folder to hold all the XPS spectra
	NewDataFolder/O root:XPS:Spectra
	
	//	The folder to hold all the raw two-dimensional spectral images
	NewDataFolder/O root:XPS:Images
	
	//	The folder to hold all the permanent waves, used for different bureaucratic purposes, such as listboxes, displaying waves, etc...
	NewDataFolder/O root:Programming
	
	//	The folder to hold all the temporary waves, created during a function call and deleted again at the end of the function call.
	NewDataFolder/O root:Programming:Temp
end



Static Function XPSLoadVGScientaSingleFile(FullFileName, LoadFolder)
//	Loads all regions of the VG Scienta text file FullFileName and places the images and spectra in LoadFolder:Images: and LoadFolder:Spectra:, respectively
String FullFileName
DFREF LoadFolder

	//	Removes the path information, e.g. 'c:data:xps:20141028xps0005.txt' becomes '20141028xps0005.txt' and adds the filename to the information to be printed in the history
	String FileName=ParseFilePath(0, FullFileName, ":", 1, 0)
	String PrintString=FileName
	
	//	Removes the extension from the filename, e.g '20141028xps0005.txt' becomes '20141028xps0005'
	String BaseWaveName=ParseFilePath(3, FullFileName, ":", 0, 0)

	//	Shortens basename to 20141028_5 if it conforms to the above name type
	Variable Num1=0, Num2=0
	String  Str1="", Str2=""
	SScanF BaseWaveName, "%d%3s%d%s", Num1, Str1, Num2, Str2
	if ((V_flag==3) && (NumType(Num1)==0) && (NumType(Num2)==0) && (StrLen(Str2)==0) && ((CmpStr(Str1, "xps")==0) || (CmpStr(Str1, "ups")==0) || (CmpStr(Str1, "aes")==0) || (CmpStr(Str1, "iss")==0)))
		BaseWaveName=Num2iStr(Num1)+"_"+Num2iStr(Num2)
	endif
	
	//	Deletes all temporary waves
	DFREF TempFolder=root:Programming:Temp
	KillWavesInFolder(TempFolder)

	//	Loads the file as a one long text wave
	//	200 ms
	DFREF CurrentFolder=GetDataFolderDFR()
	SetDataFolder TempFolder
	LoadWave /O/N=LoadedTextWave /K=2 /L={0, 0, 0, 0, 1} /Q /J FullFileName
	Wave/T LoadedTextWave=TempFolder:LoadedTextWave0
	Variable NumberOfLines=NumPnts(LoadedTextWave)
	
	//	The VG Scienta text file has the format:
	//	0	[Info]
	//	1	Number of Regions=1
	//	2	Version=1.2.2
	//	3	
	//	4	[Region 1]
	//	5	Region Name=Ag3d                            
	//	6	Dimension 1 name=Binding Energy [eV]
	//	7	Dimension 1 size=221
	//	8	Dimension 1 scale=382.00000 381.90000 381.80000 381.70000 381.60000 381.50000 381.40000 381.30000 ...
	//	9	Dimension 2 name=Y-Scale [mm]
	//	10	Dimension 2 size=826
	//	11	Dimension 2 scale=-2.92632 -2.91920 -2.91208 -2.90496 -2.89784 -2.89072 -2.88360 -2.87648 -2.86936 -2.86224 ...
	//	12	
	//	13	[Info 1]
	//	14	Instrument=SES 200-49U
	//	15	Location=Erlangen
	//	16	User=Scienta
	//	17	Sample=Ag(100)
	//	18	Comments=commentary needed
	//	19							!!!	Sometimes this extra line is not there     !!!
	//	20	Date=1/13/2015
	//	21	Time=11:53:36 AM
	//	22	Region Name=Ag3d
	//	23	Excitation Energy=1486.6
	//	24	Energy Scale=Binding
	//	25	Acquisition Mode=Swept
	//	26	Center Energy=9
	//	27	Low Energy=1104.6
	//	28	High Energy=1126.6
	//	29	Energy Step=0.1
	//	30	Step Time=200
	//	31	Detector First X-Channel=1
	//	32	Detector Last X-Channel=1064
	//	33	Detector First Y-Channel=90
	//	34	Detector Last Y-Channel=915
	//	35	Number of Slices=826
	//	36	Lens Mode=Transmission
	//	37	Pass Energy=500
	//	38	Number of Sweeps=1
	//	39	Time per Spectrum Channel=91
	//	40	
	//	41	[Data 1]
	//	42	   382.00000      0.00000      0.00000      0.00000      0.00000      0.00000      0.00000      0.00000      0.00000      0.00000      0.00000      ...
	//		...
	
	//	If any error occurs during loading the value of ErrorCode will change. The new value can be used to determine where the error occured.
	Variable ErrorCode=0
	
	//	Checks that the first line is [Info]
	if (CmpStr(LoadedTextWave[0], "[Info]")!=0)
		ErrorCode=1
	else	

		//	Searches for lines starting with [ and ending with ]
		//	< 1 ms
		Make/O/T/FREE/N=0 TempGrepWave
		Grep/INDX/E="(^\\[)(.+)(\\]$)" LoadedTextWave as TempGrepWave
		Wave W_Index=TempFolder:W_Index
		Duplicate/O/FREE W_Index, TempIndexWave
		Variable GrepWaveSize=NumPnts(TempGrepWave)
	
		//	The TempGrepWave will now have the format:
		//	0	[Info]
		//	1	[Region 1]
		//	2	[Info 1]
		//	3	[Data 1]
		//	4	[Region 2]
		//	5	[Info 2]
		//	6	[Data 2]
		//	7	[Region 3]
		//	8	[Info 3]
		//	9	[Data 3]
		//	10	[Region 4]
		//	11	[Info 4]
		//	12	[Data 4]
		
		//	Adds a data point to the index wave with the position where the next region would have been at
		InsertPoints GrepWaveSize, 1, TempIndexWave
		TempIndexWave[GrepWaveSize]=NumberOfLines
		
		//	Variables used to limit the searches below. Typically between FilePosition and NextItemPosition
		Variable CurrentItem=0
		Variable FilePosition=TempIndexWave[CurrentItem]
		Variable NextItemPosition=TempIndexWave[CurrentItem+1]

		//	Finds and stores the number of regions under [Info]
		Variable SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Number of Regions=", LoadedTextWave)
		if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
			ErrorCode=2
		else
			FilePosition=SearchResult
			Variable NumberOfRegions=0
			SScanF LoadedTextWave[FilePosition], "Number of Regions=%f", NumberOfRegions
		
			if (V_flag!=1)
				ErrorCode=3
			else

				//	Checks that the number of regions agree with the GrepWave size
				if (GrepWaveSize!=1+NumberOfRegions*3)
					ErrorCode=4
				else
				
					//	Adds the number of regions to the print string
					if (NumberOfRegions==1)
						PrintString+=" ("+Num2iStr(NumberOfRegions)+" region):"
					else
						PrintString+=" ("+Num2iStr(NumberOfRegions)+" regions):"
					endif
				
					//	Variables used during the for loop
					Variable i=0, a=0, Point0=0, Point1=0, UniqueRegionSuffix=1
					String BaseRegionName=""
					Make/O/T/FREE/N=(NumberOfRegions) RegionName, RegionInfo, Dimension1Name, Dimension2Name
					Make/O/FREE/N=(NumberOfRegions) Dimension1Size, Dimension1Start, Dimension1Delta, Dimension2Size, Dimension2Start, Dimension2Delta, CropFrom, CropTo
					
					//	Reads the text data for all regions in the loaded text wave one at a time
					for (i=0; (i<NumberOfRegions) && (ErrorCode==0); i+=1)
					
						//	Finds the expected position of [Region <i+1>]
						CurrentItem=i*3+1
						FilePosition=TempIndexWave[CurrentItem]
						NextItemPosition=TempIndexWave[CurrentItem+1]
						
						//	Checks that the current position is [Region <i+1>]
						if (CmpStr(LoadedTextWave[FilePosition], "[Region "+Num2iStr(i+1)+"]")!=0)
							ErrorCode=5
						else	
						
							//	Finds and stores the region name under [Region <i+1>] and removes blank spaces, semicolons and illegal characters
							SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Region Name=", LoadedTextWave)
							if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
								ErrorCode=6
							else
								FilePosition=SearchResult
								RegionName[i]=(LoadedTextWave[FilePosition])[12,StrLen(LoadedTextWave[FilePosition])-1]
								RegionName[i]=ReplaceString(" ", RegionName[i], "")
								RegionName[i]=ReplaceString(";", RegionName[i], "")
								RegionName[i]=CleanUpName(RegionName[i], 1)
								
								//	Limits the region name to 10 characters. This is done to prevent illegal wave names with over 32 characters
								if (StrLen(RegionName[i])>10)
									RegionName[i]=(RegionName[i])[0, 9]

									//	Cropping the region names may cause two names to become identical, if so a number is added to the region name making it unique
									BaseRegionName=RegionName[i]
									FindValue /S=0 /TEXT=(RegionName[i]) /TXOP=5 /Z RegionName
									for (UniqueRegionSuffix=2; V_value<i; UniqueRegionSuffix+=1)
										RegionName[i]=BaseRegionName+Num2iStr(UniqueRegionSuffix)
										FindValue /S=0 /TEXT=(RegionName[i]) /TXOP=5 /Z RegionName
									endfor
								endif

								//	Finds and stores the dimension 1 name under [Region <i+1>]
								SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Dimension 1 name=", LoadedTextWave)
								if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
									ErrorCode=7
								else
									FilePosition=SearchResult
									Dimension1Name[i]=(LoadedTextWave[FilePosition])[17,StrLen(LoadedTextWave[FilePosition])-1]
						
									//	Finds and stores the dimension 1 size under [Region <i+1>]
									SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Dimension 1 size=", LoadedTextWave)
									if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
										ErrorCode=8
									else
										FilePosition=SearchResult
										SScanF LoadedTextWave[FilePosition], "Dimension 1 size=%f", Num1
										
										//	V_flag will never be larger than the number of variables SScanF attempts to load, even if the line of text contains additional information
										//	V_flag!=1 is therefore here the same as V_flag<1
										//	To test for additional text, add a string variable %s to the end of SScanF and check that it doesn't load
										if (V_flag!=1)
											ErrorCode=9
										else
											
											Dimension1Size[i]=Num1

											//	Finds and stores the dimension 1 scale under [Region <i+1>]
											SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Dimension 1 scale=", LoadedTextWave)
											if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
												ErrorCode=10
											else
												FilePosition=SearchResult
												SScanF LoadedTextWave[FilePosition], "Dimension 1 scale=%f %f", Point0, Point1

												if (V_flag<2)
													ErrorCode=11
												else

													Dimension1Start[i]=Point0
													Dimension1Delta[i]=Point1-Point0
						
													//	Finds and stores the dimension 2 name under [Region <i+1>]
													SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Dimension 2 name=", LoadedTextWave)
													if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
														ErrorCode=12
													else
														FilePosition=SearchResult
														Dimension2Name[i]=(LoadedTextWave[FilePosition])[17,StrLen(LoadedTextWave[FilePosition])-1]
						
														//	Finds and stores the dimension 2 size under [Region <i+1>]
														SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Dimension 2 size=", LoadedTextWave)
														if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
															ErrorCode=13
														else
															FilePosition=SearchResult
															SScanF LoadedTextWave[FilePosition], "Dimension 2 size=%f", Num1
															
															if (V_flag!=1)
																ErrorCode=14
															else
															
																Dimension2Size[i]=Num1

																//	Finds and stores the dimension 2 scale under [Region <i+1>]
																SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Dimension 2 scale=", LoadedTextWave)
																if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
																	ErrorCode=15
																else
																	FilePosition=SearchResult
																	SScanF LoadedTextWave[FilePosition], "Dimension 2 scale=%f %f", Point0, Point1
																	
																	if (V_flag<2)
																		ErrorCode=16
																	else

																		Dimension2Start[i]=Point0
																		Dimension2Delta[i]=Point1-Point0
																		
																		//	Finds the expected position of [Info <i+1>]
																		CurrentItem=i*3+2
																		FilePosition=TempIndexWave[CurrentItem]
																		NextItemPosition=TempIndexWave[CurrentItem+1]
							
																		//	Checks that the current position is [Info <i>]
																		if (CmpStr(LoadedTextWave[FilePosition], "[Info "+Num2iStr(i+1)+"]")!=0)
																			ErrorCode=17
																		else	

																			//	Stores the entire region information
																			RegionInfo[i]=""
																			for (a=FilePosition+1; a<(NextItemPosition-1); a+=1)
																				if (StrLen(LoadedTextWave[a])>0)
																					RegionInfo[i]+=LoadedTextWave[a]+"\r"
																				endif
																			endfor

																			//	Removes the last linebreak from the region info (if one exists)
																			RegionInfo[i]=RemoveEnding(RegionInfo[i], "\r")

																			//	Replaces ":" with "\CLN" and ";" with "\SCLN". This is needed because the region information will be part of a larger string list attached as a wave note
																			RegionInfo[i]=ReplaceString(":", RegionInfo[i], "\CLN", 1)
																			RegionInfo[i]=ReplaceString(";", RegionInfo[i], "\SCLN", 1)

																			//	Finds the expected position of [Data <i>]
																			CurrentItem=i*3+3
																			FilePosition=TempIndexWave[CurrentItem]
																			NextItemPosition=TempIndexWave[CurrentItem+1]
								
																			//	Checks that the current position is [Data <i+1>]
																			if (CmpStr(LoadedTextWave[FilePosition], "[Data "+Num2iStr(i+1)+"]")!=0)
																				ErrorCode=18
																			else	
																				
																				//	Selects the data range of region <i+1> to crop from the loaded numeric wave
																				CropFrom[i]=FilePosition+1
																				CropTo[i]=NextItemPosition-2
																			endif
																		endif
																	endif
																endif
															endif
														endif
													endif
												endif
											endif
										endif
									endif
								endif
							endif
						endif
					endfor
					
					//	Only proceeds if no error has occured
					if (ErrorCode==0)
					
						//	Determines the maxium number of columns for the regions. This is needed to be able to load all regions with one LoadWave command. The number of columns is usually always the same for all regions, but just in case, I'm doing it this way.
						Variable MaxDimension2Size=WaveMax(Dimension2Size)
					
						//	Loads the file as one long numeric wave
						//	300 ms
						LoadWave /O/M/N=LoadedNumericWave /F={MaxDimension2Size+1, 13, 0} /K=1 /L={0, 0, NumberOfLines, 1, MaxDimension2Size} /Q FullFileName
						Wave  LoadedNumericWave=TempFolder:LoadedNumericWave0
		
						//	The data folders to save the images and spectra in
						DFREF ImageFolder=LoadFolder:Images
						DFREF SpectrumFolder=LoadFolder:Spectra
						
						//	Reads the numeric data for all regions in the loaded numeric wave one at a time
						for (i=0; (i<NumberOfRegions) && (ErrorCode==0); i+=1)

							//	Crops the data range of region <i> from the loaded numeric wave
							Duplicate/O/FREE/R=[CropFrom[i], CropTo[i]][0, Dimension2Size[i]-1] LoadedNumericWave, Cropped2DWave

							//	Checks that the data range of region <i+1> contains no NaN values
							WaveStats /Q Cropped2DWave
							if (V_numNans>0)
								ErrorCode=19
							else
							
								//	Saves the data range of region <i> from the loaded numeric wave. Any existing wave with the same name will be overwritten
								Duplicate/O Cropped2DWave, ImageFolder:$(BaseWaveName+"_"+RegionName[i])/WAVE=Loaded2DWave
																				
								//	The detector in Erlangen is broken at line 143 and 308. Those lines are therefore forced to be the average of the neighboring lines.
								//	0.1 ms
								MultiThread Loaded2DWave[][143]=(Loaded2DWave[p][142]+Loaded2DWave[p][144])/2
								MultiThread Loaded2DWave[][308]=(Loaded2DWave[p][307]+Loaded2DWave[p][309])/2
																					
								//	Calculates the 1D spectrum wave based on the 2D camera image wave. Any existing wave with the same name will be overwritten
								//	0.3 ms
								MatrixOP/NTHR=0/O SpectrumFolder:$(BaseWaveName+"_"+RegionName[i])=sumRows(Loaded2DWave)
								Wave Loaded1DWave=SpectrumFolder:$(BaseWaveName+"_"+RegionName[i])
		
								//	Sets the x- and y-scalings of both waves
								SetScale /P x, Dimension1Start[i], Dimension1Delta[i], "", Loaded1DWave
								SetScale /P x, Dimension1Start[i], Dimension1Delta[i], "", Loaded2DWave
								SetScale /P y, Dimension2Start[i], Dimension2Delta[i], "", Loaded2DWave

								//	Saves the region information as a notes attached to both waves (the region information is not current used for the 2D wave)
								Note/K Loaded1DWave
								Note/K Loaded2DWave
								Note/NOCR Loaded1DWave, "REGION INFO:"+RegionInfo[i]+";"
								Note/NOCR Loaded2DWave, "REGION INFO:"+RegionInfo[i]+";"
								
								//	Adds the region name to the information to be printed in the history
								PrintString+="   ["+RegionName[i]+"]"
							endif
						endfor
					endif
				endif
			endif
		endif
	endif
	
	//	Adds the ErrorCode to the print string if an error occured
	if (ErrorCode!=0)
		PrintString+="   !! An error occured !! (ErrorCode "+Num2Str(ErrorCode)+")"
	endif

	//	Deletes all temporary waves
	KillWavesInFolder(TempFolder)
	
	//	Returns the active folder to the original folder
	SetDataFolder CurrentFolder
	
	//	Prints the filename, regions within the file and information about any error which have occured
	Print(PrintString)
end





//     -----<<<<<     Load XY     >>>>>-----
//	This section contains the functions used by the Load XY menu item


Static Function XPSLoadXY(DataPathList)
//	Loads the raw XPS detector images and creates the corresponding spectra
String DataPathList
String FileExtension="Raw XPS Data Files (*.txt):.txt;All Files (*.*):.*;"

	//	Updates the history of procedures used with the current experiment
	XPSUpdateProcedureHistory()
	
	//	Creates the necessary folders, if they do not already exist
	XPSCreateFolders()
	
	//	Specifies the default data directory any open file dialog will start in
	CreateDataPath(DataPathList)
	
	//	Display an open file dialog, showing only files with the extension .txt. Allows multiple files to be selected
	Open/D/F=FileExtension /R/MULT=1 /P=EccentricXPSDataPath RefNum
	String AllFileNames=S_FileName
	
	if (StrLen(AllFileNames)!=0)

		//	Prompts the user for the location of the new waves. Unlike for LoadVGScienta the spectra will not be placed in a subfolder
		XPSViewSaveWaveDialog("Select destination folder", "root:XPS:Spectra:", "", "EccentricXPS#XPSLoadXYLoadWaves", "", AllFileNames)
	endif
end



Static Function XPSLoadXYLoadWaves(SaveFolder, SaveName, AllFileNames)
//	Will load XPS spectra as two-column text files. The first column is assumed to be binding energy and the second intensity.
//	The binding energy steps must be uniform.
DFREF SaveFolder
String SaveName, AllFileNames
Variable i=0, n=0
String FullFileName="", FileName="", WaveNameString=""

	//	Considers the full name as a data folder rather than the a folder and a wave (as the save dialog was intended for)
	if (StrLen(SaveName)>0)
		NewDataFolder/O SaveFolder:$SaveName
		DFREF SpectrumFolder=SaveFolder:$SaveName
	else
		DFREF SpectrumFolder=SaveFolder
	endif
	
	//	Checks if the folder exists
	if (DataFolderRefStatus(SpectrumFolder)==0)
		ErrorMessage("The data folder does not exist/could not be created!")
	else
	
		//	Calculates the number of files selected to be loaded
		Variable NumberOfFilesLoaded=ItemsInList(AllFileNames, "\r")
		
		//	Saves the active data folder
		DFREF CurrentFolder=GetDataFolderDFR()
		
		//	Sets the active data folder to root:Programming:Temp
		DFREF TempFolder=root:Programming:Temp
		SetDataFolder TempFolder
			
		//	Opens and display each file, one at a time
		for (i=0; i<NumberOfFilesLoaded; i+=1)
		
			FullFileName=StringFromList(i, AllFileNames, "\r")
			
			//	Removes the path information, e.g. 'c:data:xps:20141028xps0005.txt' becomes '20141028xps0005.txt'
			FileName=ParseFilePath(0, FullFileName, ":", 1, 0)
			Print("Loading "+FileName)

			//	Removes the extension from the filename and adds the wave location, e.g '20141028xps0005.txt' becomes root:XPS:Spectra:'20141028xps0005'
			WaveNameString=ParseFilePath(3, FullFileName, ":", 0, 0)

			//	Deletes all temporary waves
			KillWavesInFolder(TempFolder)

			//	Will load the file as a two column text file, starting from the first row with valid numbers, and adding rows until they no longer contain valid numbers.
			//	L={nameLine, firstLine, numLines, firstColumn, numColumns }
			LoadWave /N=LoadedXPSWave /K=1 /L={0, 0, 0, 0, 2} /G/O/Q FullFileName

			Wave LoadedXPSWave0=TempFolder:LoadedXPSWave0
			Wave LoadedXPSWave1=TempFolder:LoadedXPSWave1
			
			n=NumPnts(LoadedXPSWave0)
			if (n>1)

				//	Sets the binding energy scaling
				SetScale /P x, LoadedXPSWave0[0], (LoadedXPSWave0[n-1]-LoadedXPSWave0[0])/(n-1), "", LoadedXPSWave1

				//	Copies the loaded wave to its final name
				Duplicate/O LoadedXPSWave1, SpectrumFolder:$WaveNameString
			else
				Print("Failed to load spectrum from "+FileName)
			endif
		endfor

		Print(" ")
		Print("Done")
		Print(" ")
		
		//	Deletes all temporary waves
		KillWavesInFolder(TempFolder)
		
		//	Returns the active data folder to the original folder
		SetDataFolder CurrentFolder
		
		//	Opens the View Spectra window, showing the data folder the waves were loaded into. The displayed data folder is not changed if the View Spectra window already exists
		DoWindow XPSViewSpectraWindow
		if (V_Flag==0)

			//	Creates the View Spectra window
			String SpectrumFolderName=GetDataFolder(1, SpectrumFolder)
			XPSViewSpectraCreateWindow(SpectrumFolderName, 1)
		else

			//	Finds the active data folder in the View Spectra window
			DFREF ActiveFolder=XPSViewGetDataFolder("XPSViewSpectraWindow", "DataFolderPopUp")

			//	Refreshes the graph
			XPSViewSpectraUpdateGraph("XPSViewSpectraWindow", ActiveFolder, $"")
		endif
	endif
end





//     -----<<<<<     View Images     >>>>>-----
//	This section contains the functions used by the View Images menu item

Static Function XPSViewImages(FolderOrGroup, ActiveName, ForceSelection)
//	Brings the View Image window to the front, or creates it if it does not exist
String FolderOrGroup, ActiveName
Variable ForceSelection
String ActiveWindow="XPSViewImageWindow"

	//	Updates the history of procedures used with the current experiment
	XPSUpdateProcedureHistory()

	//	Does not create the View Images window if it already exists
	DoWindow /F $ActiveWindow
	if ((V_Flag==0) || (ForceSelection==1))

		//	Changes the displayed selection
		XPSViewImagesChangeSelection(FolderOrGroup, ActiveName, ForceSelection)
	else
	
		//	Finds the displayed wave
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")

		//	Refreshes the graph
		XPSViewImageUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
	endif
end



Static Function XPSViewImagesChangeSelection(FolderOrGroup, ActiveName, ForceSelection)
//	Changes the selection in the View Images window without killing the window
String FolderOrGroup, ActiveName
Variable ForceSelection
String ActiveWindow="XPSViewImageWindow"

	//	Does not create the View Images window if it already exists
	DoWindow/F $ActiveWindow
	if (V_Flag==0)

		//	Creates the View Images window
		XPSViewImagesCreateWindow(FolderOrGroup, ActiveName, ForceSelection)
	else
	
		//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is forced
		PopupMenu DataFolderPopUp popmatch=FolderOrGroup, win=$ActiveWindow
		ControlInfo /W=$ActiveWindow DataFolderPopUp
		if (CmpStr(FolderOrGroup, S_Value)!=0)
			PopupMenu DataFolderPopUp mode=1, popvalue=FolderOrGroup, win=$ActiveWindow
		endif

		//	Updates the list of wave selections and select ActiveName, or if ActiveName doesn't exist in the list, the first item in the list is selected
		PopupMenu DisplayedWavePopUp mode=1, popmatch=ActiveName, win=$ActiveWindow

		//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
		if (StrLen(ActiveName)>0)
			ControlInfo /W=$ActiveWindow DisplayedWavePopUp
			if (CmpStr(ActiveName, S_Value)!=0)
				PopupMenu DisplayedWavePopUp mode=1, popvalue=ActiveName, win=$ActiveWindow
			endif
		endif

		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")

		//	Refreshes the View Images graph
		XPSViewImageUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
	endif
end



Static Function XPSViewImagesCreateWindow(FolderOrGroup, ActiveName, ForceSelection)
//	Displays the raw detector image and creates the various popup menus and buttons used for a quick and easy analysis
String FolderOrGroup, ActiveName
Variable ForceSelection
Variable n=120/ScreenResolution
String ActiveWindow="XPSViewImageWindow"
	
	//	Creates the folder to hold all the permanent waves, used for different bureaucratic purposes, such as listboxes, displaying waves, etc...
	NewDataFolder/O root:Programming
	DFREF ProgramFolder=root:Programming
	
	//	Creates the folder to hold all the temporary waves, created during a function call and deleted again at the end of the function call.
	NewDataFolder/O root:Programming:Temp

	//	Creates the wave to hold the displayed image
	Make/O ProgramFolder:XPSViewImageDataWave={{0}}
	Wave DataWave=ProgramFolder:XPSViewImageDataWave

	//	Creates the graph to display the image.
	DoWindow /K $ActiveWindow
	Display /W=(5, 40, 5+365*n, 40+300*n) /K=1 /N=$ActiveWindow
	AppendImage/W=$ActiveWindow DataWave
	ModifyImage/W=$ActiveWindow '' ctab= {*,*,ColdWarm,0}, ctabAutoscale=1, lookup= $""
	ModifyGraph /W=$ActiveWindow margin(top)=40*n
	SetAxis /W=$ActiveWindow /A/R bottom
	ShowInfo /W=$ActiveWindow
	Label /W=$ActiveWindow bottom "Binding Energy (\\ueV)"
	Label /W=$ActiveWindow left "Position (\\umm)"
	
	//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
	SetWindow $ActiveWindow hook(KillHook)=XPSViewImagesHookFunc
		
	//	Selects the active data folder, listing root: and all it's subfolders with the exception of root:Programming, all it's subfolders, and all folders named SavedFits and all their subfolders. To list the default folder (set with SetDataFolder) and all it's subfolders instead, with no exceptions, use EccentricXPS#XPSFitAssGroupsAndFoldersList(:, $"", "")
	PopupMenu DataFolderPopUp bodyWidth=250, mode=1, pos={210, 5}, proc=EccentricXPS#XPSViewDataFolderPopupMenu, value=EccentricXPS#XPSViewGroupsAndFoldersList(root:, root:Programming, "SavedFits"), userdata="EccentricXPS#XPSViewImageUpdateGraph", win=$ActiveWindow, help={"Selects the data folder to look for waves in"}
	//	Popup menu to select the detector image to display. The complicated value call #(String) indicates that the value ActiveWindow had when the control was created should be used
	PopupMenu DisplayedWavePopUp bodyWidth=200, mode=1, pos={415,5}, proc=EccentricXPS#XPSViewDisplayedWavePopupMenu, value=#("EccentricXPS#XPSViewPopUpWaveList(2, \""+ActiveWindow+"\", \"DataFolderPopUp\")"), userdata="EccentricXPS#XPSViewImageUpdateGraph", win=$ActiveWindow, help={"Selects the image to display"}
	//	Popup menu to select the colour scheme to be used. Default is black and white
	PopupMenu ColourSchemePopUp bodyWidth=130, mode=3, pos={550,5}, proc=EccentricXPS#XPSViewImageColours, value="Grays;YellowHot;ColdWarm;SpectrumBlack;", win=$ActiveWindow, help={"Selects the colour scheme to use for the image"}

	//	Sum displays the image normalized to the sum of each line
	Button OffsetButton pos={10,32}, proc=EccentricXPS#XPSViewToggleButton, size={70,20}, title="Offset", userdata="EccentricXPS#XPSViewImageUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Subtracts its sum from each horizontal line"}
	//	Min/Max displays the image normalized to the minimum and maximum values in each line
	Button ScaleButton proc=EccentricXPS#XPSViewToggleButton, size={70,20}, title="Scale", userdata="EccentricXPS#XPSViewImageUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Divides each horizontal line by its maximum value"}
	//	Resample will resample the image to a lower resolution
	Button ResampleButton proc=EccentricXPS#XPSViewToggleButton, size={70,20}, title="Resample", userdata="EccentricXPS#XPSViewImageUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Resamples the image by averaging the specified number of points"}
	//	The number of data points in the X and Y direction to cluster together when resampling
	SetVariable SetResampleX limits={1, inf, 1}, proc=EccentricXPS#XPSViewSetVariable, value=_NUM:2, size={89,20}, pos+={0, 2}, title="X", userdata="EccentricXPS#XPSViewImageUpdateGraph", win=$ActiveWindow, help={"The number of points to average in the X-direction"}
	SetVariable SetResampleY limits={1, inf, 1}, proc=EccentricXPS#XPSViewSetVariable, value=_NUM:30, size={89,20}, title="Y", userdata="EccentricXPS#XPSViewImageUpdateGraph", win=$ActiveWindow, help={"The number of points to average in the Y-direction"}
	//	Convert converts the image inside and outside cursors A and B into two seperate spectra C1 and C2
	Button SpectraButton proc=EccentricXPS#XPSViewImageSpectra, pos+={0, -2}, size={70,20}, title="Spectra", win=$ActiveWindow, help={"Converts the image into a 1D spectrum, in the ranges specified by the cursors"}
	//	Opens the help file
	Button HelpButton proc=EccentricXPS#XPSViewHelp, size={70,20}, pos+={0, 0}, title="Help", userdata="View Images", win=$ActiveWindow, help={"Opens the help file"}

	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$ActiveWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$ActiveWindow B
	
	//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or the default data folder is used instead, depending on the value of ForceSelection
	PopupMenu DataFolderPopUp popmatch=FolderOrGroup, win=$ActiveWindow
	ControlInfo /W=$ActiveWindow DataFolderPopUp
	if (CmpStr(FolderOrGroup, S_Value)!=0)
		if (ForceSelection==1)
			PopupMenu DataFolderPopUp mode=1, popvalue=FolderOrGroup, win=$ActiveWindow
		else
			PopupMenu DataFolderPopUp popmatch=GetDataFolder(1), win=$ActiveWindow
		endif
	endif

	//	Updates the list of wave selections and select ActiveName, or if ActiveName doesn't exist in the list, the first item in the list is selected
	PopupMenu DisplayedWavePopUp mode=1, popmatch=ActiveName, win=$ActiveWindow

	//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
	if (StrLen(ActiveName)>0)
		ControlInfo /W=$ActiveWindow DisplayedWavePopUp
		if (CmpStr(ActiveName, S_Value)!=0)
			if (ForceSelection==1)
				PopupMenu DisplayedWavePopUp mode=1, popvalue=ActiveName, win=$ActiveWindow
			endif
		endif
	endif

	DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
	Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
	
	//	Refreshes the graph
	XPSViewImageUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
end



Function XPSViewImagesHookFunc(s)
//	The hook function for the View Images window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1
	
			//	Kills the wave that was used to hold the displayed image, but delays the execution until the window has been killed
			Execute/P/Q "KillWaves/Z root:Programming:XPSViewImageDataWave"
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function XPSViewImageColours(PU_Struct) : PopupMenuControl
//	Changes the colour scheme of the active window
//	This function is also used by the NEXAFS image windows
STRUCT WMPopupAction &PU_Struct
	if  (PU_Struct.eventCode==2)
		//	Updates the graph with the selected colour scheme
		ModifyImage/W=$PU_Struct.Win '' ctab= {*,*,$PU_Struct.PopStr,0}	//	' ' indicates all images in graph
	endif
	Return 0
end



Static Function/S XPSViewPopUpWaveList(Dim, ActiveWindow, DataFolderPopupStr)
//	Returns a list of all waves in the selected group or data folder
Variable Dim
String ActiveWindow, DataFolderPopupStr
String ListOfWaves=""

	//	Finds the selected group or folder in the active window
	ControlInfo /W=$ActiveWindow $DataFolderPopupStr
	String GroupOrFolderName=S_Value
	DFREF Folder=$GroupOrFolderName
	
	//	It GroupOrFolderName is not a valid data folder it is assumed to be a group
	if (DataFolderRefStatus(Folder)!=0)
	
		//	Returns a list of the waves in the selected data folder
		ListOfWaves=XPSViewListWaves(Folder, Dim)
	else
	
		//	Extracts the folder, waves and their colours for the selected group
		String GroupInfo=XPSViewGetGroupInfo(GroupOrFolderName)
		
		if (StrLen(GroupInfo)>0)

			//	Returns the data folder for the selected group
			Folder=XPSViewGroupDataFolder(GroupInfo)

			//	Checks if the View Images window is active
			if (CmpStr(ActiveWindow, "XPSViewImageWindow")==0)

				//	Checks if the group data folder is Spectra
				if (CmpStr("Spectra", GetDataFolder(0, Folder))==0)

					//	If the group folder is Spectra the neighbouring Images folder is returned, if it exists
					DFREF ParentFolder=$(GetDataFolder(1, Folder)+":")
					DFREF ImageFolder=ParentFolder:Images
				
					//	If the Images folder does not exist and invalid folder is returned instead
					if ((DataFolderRefStatus(Folder)!=0) && (DataFolderRefStatus(ParentFolder)!=0) && (DataFolderRefStatus(ImageFolder)!=0))
						Folder=ImageFolder
					else
						Folder=$""
					endif
				else
					Folder=$""
				endif
			endif

			//	Checks if the data folder exists
			if (DataFolderRefStatus(Folder)!=0)
			
				//	Creates a semicolon separated list of all waves in the selected group
				ListOfWaves=XPSViewGroupWaveList(GroupInfo)
		
				//	If no waves exist in the group, it is treated as a data folder shortcut, and all waves in the group data folder are displayed instead
				if (StrLen(ListOfWaves)==0)

					//	Returns a list of the waves in the group data folder
					ListOfWaves=XPSViewListWaves(Folder, Dim)
				else
				
					//	Removes waves in the group that no longer exist
					String TempList=""
					Variable n=ItemsInList(ListOfWaves, ";")
					Variable i=0, a=0, b=0
					
					//	Counts through all waves in the group			
					for (i=0; i<n; i+=1)
						b=StrSearch(ListOfWaves, ";", a)
						Wave/Z ActiveWave=Folder:$(ListOfWaves[a,b-1])
						if (WaveExists(ActiveWave))
							TempList+=ListOfWaves[a,b]
						endif
						a=b+1
					endfor
					
					//	Displays an error if no existing waves are left in the group
					if (StrLen(TempList)>0)
						ListOfWaves=TempList
					else
						ListOfWaves="No existing waves left in group;"
					endif
				endif
			else
			
				//	Displays an error if the group data folder does not exist anymore
				ListOfWaves="Group data folder does not exist;"
			endif
		else
			//	Displays an error in both the displayed wave and data folder popup menus
			ListOfWaves="Group or data folder does not exist;"
		endif
	endif
	
	//	Returns the list of waves
	Return ListOfWaves
end



Static Function/S XPSViewGroupsAndFoldersList(StartFolder, ExcludeFolder, ExcludeFolders)
//	Returns a semicolon separated string list of groups and data folders
DFREF StartFolder, ExcludeFolder
String ExcludeFolders

	//	The string to be returned
	String GroupsAndFolders=""

	//	Creates a semicolon separated list of all saved groups
	String Groups=XPSViewGroupList()
	
	//	Creates a blank space in the popup menu between the default selection and the groups
	if (StrLen(Groups)>0)
		GroupsAndFolders=" ;"+Groups
	endif

	//	Returns a semicolon separated string list of the full paths to StartFolder and all it's subfolders, with the exception of the specific folder ExcludeFolder and all folders with the name ExcludeFolders and all their subfolders
	String Folders=XPSViewDataFoldersList(StartFolder, ExcludeFolder, ExcludeFolders)
	
	//	Creates two blank spaces in the popup menu between the groups and the folders. Varying the number of blank spaces makes the two blank selections different, which means menu scrolling works
	if (StrLen(Folders)>0)
		GroupsAndFolders+="  ;"+Folders
	endif

	//	Returns the combined list of groups and folders
	Return GroupsAndFolders
end



Static Function/S XPSViewGroupWaveList(GroupInfo)
//	Creates a semicolon separated list of all waves in the selected group
String GroupInfo
String ListOfWaves="", SingleWaveInfo="", WaveNameStr=""
Variable a=0, i=0, n=0

	//	Counts the number of waves in the list
	n=ItemsInList(GroupInfo, ";")-1
				
	//	Extracts wave names and colour information from the saved group
	for (i=0; i<n; i+=1)
	
		//	Finds the end of the wave name, before the colour information starts (WaveName:ColourNumber)
		SingleWaveInfo=StringFromList(i+1, GroupInfo, ";")
		a=StrSearch(SingleWaveInfo, ":", 0)
				
		//	Restores all commas, equal signs, semicolons and colons to the wave name
		WaveNameStr=SingleWaveInfo[0,a-1]
		WaveNameStr=ReplaceString("\COMMA", WaveNameStr, ",", 1)
		WaveNameStr=ReplaceString("\EQUAL", WaveNameStr, "=", 1)
		WaveNameStr=ReplaceString("\SCLN", WaveNameStr, ";", 1)
		WaveNameStr=ReplaceString("\CLN", WaveNameStr, ":", 1)
			
		//	Adds the wave name to the list
		ListOfWaves+=WaveNameStr+";"
	endfor

	//	Returns the list of waves
	Return ListOfWaves
end



Static Function/S XPSViewListWaves(Folder, Dim)
//	Returns a alphanumerically sorted, semi-colon seperated list of all numeric waves of the correct dimension in the specified folder
DFREF Folder
Variable Dim
DFREF TempFolder=root:Programming:Temp
String ReturnString=""

	if (DataFolderRefStatus(Folder)==0)

		//	Returns data folder does not exist, if the data folder does not exist		
		ReturnString="Data folder does not exist;"
	else

		//	Saves the active data folder
		DFREF CurrentFolder=GetDataFolderDFR()

		//	Changes the active folder to Folder
		SetDataFolder Folder
	
		//	Returns a semi-colon seperated list of numeric waves of the correct dimension in the active folder
		ReturnString=WaveList("*", ";", "DIMS:"+Num2Str(Dim)+",DF:0,CMPLX:0,TEXT:0,WAVE:0")
	
		//	Returns the active folder to the original folder
		SetDataFolder CurrentFolder

		//	Sorts the list alphabetically, using a case-insensitive alphanumeric sort that sorts wave0 and wave9 before wave10
		ReturnString=SortList(ReturnString, ";", 16)
	
		//	If the list is empty "No waves in folder" is added
		if (StrLen(ReturnString)==0)
			ReturnString="No waves in folder;"
		endif
	endif
	
	//	Returns the list of waves
	Return ReturnString
end



Static Function/WAVE XPSViewListWaveRefs(Folder, Dim, Type, SearchStr)
//	Returns a list of wave references of all numeric waves of the correct dimension in the specified folder
DFREF Folder
Variable Dim, Type
String SearchStr

	if (DataFolderRefStatus(Folder)==0)

		//	Returns an empty wave, if the datafolder is invalid
		Make/FREE/O/WAVE/N=0 ListOfWaves
	else

		//	Counts the number of waves in the folder
		Variable n=CountObjectsDFR(Folder, 1)

		//	Creates a list of all waves in the folder
		Make/FREE/O/WAVE/N=(n) AllWaves=WaveRefIndexedDFR(Folder, p)
		
		//	Treats a search string without wildcard characters as a search string with wildcard characters at both ends of the string
		//	*C1s searches for wave names ending in C1s. 2014* searches for wave names starting with 2014. *Ag3d* or Ag3d searches for wave names containing Ag3d somewhere in the name.
		if (StrSearch(SearchStr, "*", 0)==-1)
			SearchStr="*"+SearchStr+"*"
		endif

		//	Creates lists of the types and dimensions of the waves in the folder matching the string SearchStr
		//	Type=0 for a null wave, 1 if numeric, 2 if string, 3 if the wave holds data folder references or 4 if the wave holds wave references.
		Make/FREE/B/U/O/N=(n) IncludeInList=((WaveType(AllWaves[p], 1)==Type) && (WaveDims(AllWaves[p])==Dim) && (StringMatch(NameOfWave(AllWaves[p]), SearchStr)))
		
		//	Extracts the waves of the correct type and dimension
		Extract/FREE/O AllWaves, ListOfWaves, IncludeInList
	endif
	
	//	Returns a reference to the list of waves
	Return ListOfWaves
end



Static Function XPSViewDisplayedWavePopupMenu(PU_Struct) : PopupMenuControl
//	Runs the update function if the selection is changed
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Ignores further calls to the popup menu procedure until this one has finished
		PU_Struct.blockReentry=1
		
		//	Finds the active folder	
		DFREF ActiveFolder=XPSViewGetDataFolder(PU_Struct.win, "DataFolderPopUp")

		//	Finds the name of the wave to display
		Wave/Z DisplayedWave=ActiveFolder:$PU_Struct.popStr

		//	The update function
		FUNCREF XPSViewProtoUpdate UpdateFunction=$PU_Struct.userdata

		//	Runs the update function
		UpdateFunction(PU_Struct.win, ActiveFolder, DisplayedWave)
		
		//	If the user scrolls through the items in the list too fast, the selection will change before the update has finished, and the selection will not match the displayed data. This will prevent that
		PopupMenu $PU_Struct.ctrlName popmatch=PU_Struct.popStr, win=$PU_Struct.win
	endif
	
	//	Needed, but not used
	Return 0
end



Static Function XPSViewDataFolderPopupMenu(PU_Struct) : PopupMenuControl
//	Runs the update function if the selection is changed
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Ignores further calls to the popup menu procedure until this one has finished
		PU_Struct.blockReentry=1
		
		//	Finds the active folder or group
		String GroupOrDataFolderName=PU_Struct.popStr
		DFREF ActiveFolder=$GroupOrDataFolderName

		//	If the data folder is invalid it is assumed to be the name of a saved group (for all windows, except for View Spectra and the NEXAFS windows)
		String ActiveWindow=PU_Struct.win
		if ((DataFolderRefStatus(ActiveFolder)==0) && (CmpStr(ActiveWindow, "XPSViewSpectraWindow")!=0) && (CmpStr(ActiveWindow[0,14], "RawNEXAFSWindow")!=0))

			//	Returns the data folder for the selected group
			ActiveFolder=XPSViewGroupDataFolder(XPSViewGetGroupInfo(GroupOrDataFolderName))
		
			//	Checks if the View Images window is active
			if (CmpStr(PU_Struct.win, "XPSViewImageWindow")==0)

				//	Checks if the group data folder is Spectra
				if (CmpStr("Spectra", GetDataFolder(0, ActiveFolder))==0)

					//	If the group folder is Spectra the neighbouring Images folder is returned, if it exists
					DFREF ParentFolder=$(GetDataFolder(1, ActiveFolder)+":")
					DFREF ImageFolder=ParentFolder:Images
				
					//	If the Images folder does not exist and invalid folder is returned instead
					if ((DataFolderRefStatus(ActiveFolder)!=0) && (DataFolderRefStatus(ParentFolder)!=0) && (DataFolderRefStatus(ImageFolder)!=0))
						ActiveFolder=ImageFolder
					else
						ActiveFolder=$""
					endif
				else
					ActiveFolder=$""
				endif
			endif
		endif
	
		if (CmpStr(PU_Struct.win, "XPSViewSpectraWindow")!=0)
	
			//	Changes the displayed wave selection to the first item in the list and updates the list
			PopupMenu DisplayedWavePopUp mode=1, win=$PU_Struct.win

			//	Finds the name of the wave to display
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(PU_Struct.win, ActiveFolder, "DisplayedWavePopUp")	
		else
		
			//	The View Spectra window doesn't need a wave to display
			Wave/Z DisplayedWave=$""
		endif
		
		//	The update function
		FUNCREF XPSViewProtoUpdate UpdateFunction=$PU_Struct.userdata

		//	Runs the update function
		UpdateFunction(PU_Struct.win, ActiveFolder, DisplayedWave)
		
		//	If the user scrolls through the items in the list too fast, the selection will change before the update has finished, and the selection will not match the displayed data. This will prevent that
		PopupMenu $PU_Struct.ctrlName popmatch=PU_Struct.popStr, win=$PU_Struct.win
	endif
	
	//	Needed, but not used
	Return 0
end



Function XPSViewProtoUpdate(ActiveWindow, ActiveFolder, ActiveWave)
//	Proto function used to define the update function type used in XPSViewDataFolder. Proto functions are not alllowed to be static
String ActiveWindow
DFREF ActiveFolder
Wave/Z ActiveWave
end



Static Function XPSViewSetAxis(ActiveWindow, DisplayType, AutoScaleX, AxisXMin, AxisXMax)
//	Sets the axis scaling and labels to match the type of spectrum displayed
String ActiveWindow, DisplayType
Variable AutoScaleX, AxisXMin, AxisXMax
	
	StrSwitch(DisplayType)

		//	Selects labels appropriate for XPS spectra
		case "XPS":
			Label /W=$ActiveWindow bottom "Binding Energy (\\ueV)"
			Label /W=$ActiveWindow left "Intensity (\\ucounts)"
			SetAxis/W=$ActiveWindow /A=2 left
			if (AutoScaleX)
				SetAxis/W=$ActiveWindow /A/R bottom
			else
				SetAxis/W=$ActiveWindow bottom, AxisXMax, AxisXMin
			endif
			break

		//	Selects labels appropriate for NEXAFS spectra
		case "NEXAFS":
			Label /W=$ActiveWindow bottom "Photon Energy (\\ueV)"
			Label /W=$ActiveWindow left "Intensity (\\ucounts)"
			SetAxis/W=$ActiveWindow /A=2 left
			if (AutoScaleX)
				SetAxis/W=$ActiveWindow /A bottom
			else
				SetAxis/W=$ActiveWindow bottom, AxisXMin, AxisXMax
			endif
			break
			
		//	Selects labels appropriate for Auger spectra
		case "AES":
			Label /W=$ActiveWindow bottom "Kinetic Energy (\\ueV)"
			Label /W=$ActiveWindow left "Intensity (\\ucounts)"
			SetAxis/W=$ActiveWindow /A=2 left
			if (AutoScaleX)
				SetAxis/W=$ActiveWindow /A bottom
			else
				SetAxis/W=$ActiveWindow bottom, AxisXMin, AxisXMax
			endif
			break
				
		//	Selects labels appropriate for IR spectra
		case "IR":
			Label /W=$ActiveWindow bottom "Wavenumber (\\ucm\S-1\M)"
			Label /W=$ActiveWindow left "Absorbance (\\uxxx)"
			SetAxis/W=$ActiveWindow /A=2 left
			if (AutoScaleX)
				SetAxis/W=$ActiveWindow /A bottom
			else
				SetAxis/W=$ActiveWindow bottom, AxisXMin, AxisXMax
			endif
			break

//		//	Selects labels appropriate for TPD spectra
//		case "TPD":
//			Label /W=$ActiveWindow bottom "Temperature (\\uK)"
//			Label /W=$ActiveWindow left "Intensity (\\uA)"
//			SetAxis/W=$ActiveWindow /A=2 left
//			if (AutoScaleX)
//				SetAxis/W=$ActiveWindow /A bottom
//			else
//				SetAxis/W=$ActiveWindow bottom, AxisXMin, AxisXMax
//			endif
//			break

		//	Selects labels appropriate for mass spectra
		case "QMS":
			Label /W=$ActiveWindow bottom "Mass (\\uamu)"
			Label /W=$ActiveWindow left "Intensity (\\uA)"
			if (AutoScaleX)
				SetAxis/W=$ActiveWindow /A bottom
			else
				SetAxis/W=$ActiveWindow bottom, AxisXMin, AxisXMax
			endif
			SetAxis/W=$ActiveWindow /A=2 left
			break

//		//	Removes all labels
//		case "None":
//			Label /W=$ActiveWindow bottom "(\\uxxx)"
//			Label /W=$ActiveWindow left "(\\uxxx)"
//			SetAxis/W=$ActiveWindow /A=2 left
//			if (AutoScaleX)
//				SetAxis/W=$ActiveWindow /A bottom
//			else
//				SetAxis/W=$ActiveWindow bottom, AxisXMin, AxisXMax
//			endif
//			break
			
		//	Sets the default labels to XPS
		default:
			Label /W=$ActiveWindow bottom "Binding Energy (\\ueV)"
			Label /W=$ActiveWindow left "Intensity (\\ucounts)"
			if (AutoScaleX)
				SetAxis/W=$ActiveWindow /A/R bottom
			else
				SetAxis/W=$ActiveWindow bottom, AxisXMax, AxisXMin
			endif
			SetAxis/W=$ActiveWindow /A=2 left

	endswitch
end



Static Function XPSViewFolderType(ActiveFolder, ActiveWindow)
//	Sets the folder type popup menu based on the active data folder
DFREF ActiveFolder
String ActiveWindow
String FolderType=""

	if (DataFolderRefStatus(ActiveFolder)!=0)

		//	Finds the full path of ActiveFolder
		String ActiveFolderString=GetDataFolder(1, ActiveFolder)
	
		//	Finds the base datafolder, e.g. root:XPS: or root:NEXAFS: 
		Variable a=StrSearch(ActiveFolderString, ":", 5)
		if (a>5)
			FolderType=ActiveFolderString[5, a-1]
		endif
		
		//	Displays XSW spectra as XPS spectra
		if (CmpStr(FolderType, "XSW")==0)
			FolderType="XPS"
		endif

		//	Sets the display type popup menu to folder type. If the selection is invalid the old selection is kept
		PopupMenu DisplayTypePopUp popmatch=FolderType, win=$ActiveWindow
	endif
end



Static Function XPSViewSetVariable(SV_Struct) : SetVariableControl
//	Runs the update function if the value of the control is changed
STRUCT WMSetVariableAction &SV_Struct

	//	If a value was changed
	if ((SV_Struct.eventCode==1) || (SV_Struct.eventCode==2))
	
		//	Ignores further calls to the set variable procedure until this one has finished
		SV_Struct.blockReentry=1
		
		//	Prevent the user from changing the value of the control faster than it can update
		SetVariable $SV_Struct.ctrlName disable=2

		//	Finds the displayed wave
		DFREF ActiveFolder=XPSViewGetDataFolder(SV_Struct.win, "DataFolderPopUp")
		
		if (CmpStr(SV_Struct.win, "XPSViewSpectraWindow")!=0)
	
			//	Finds the name of the wave to display
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(SV_Struct.win, ActiveFolder, "DisplayedWavePopUp")	
		else
		
			//	The View Spectra window doesn't need a wave to display
			Wave/Z DisplayedWave=$""
		endif
		
		//	Runs the specified update function
		FUNCREF XPSViewProtoUpdate UpdateFunction=$SV_Struct.userdata
		UpdateFunction(SV_Struct.win, ActiveFolder, DisplayedWave)
		
		//	Prevent the user from changing the value of the control faster than it can update
		SetVariable $SV_Struct.ctrlName disable=0
	endif

	//	This is required for a variable control, but not used for anything
	return 0
end



Static Function XPSViewToggleButton(B_Struct) : ButtonControl
//	Toggles the appearance of the button and runs the update function if the button is pressed
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Finds the status of the button (1 = clicked, 0 = not clicked)
		String StatusStr=GetUserData(B_Struct.win, B_Struct.ctrlName, "Status")

		//	Changes the status userdata and the colour of the button
		If (CmpStr(StatusStr, "0")==0)
			Button $B_Struct.ctrlName fColor=(30000,30000,30000), userData(Status)="1", win=$B_Struct.win
		else
			Button $B_Struct.ctrlName fColor=(0,0,0), userData(Status)="0", win=$B_Struct.win
		endif

		//	Finds the displayed data folder
		DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
		
		if (CmpStr(B_Struct.win, "XPSViewSpectraWindow")!=0)
	
			//	Finds the name of the wave to display
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")	
		else
		
			//	The View Spectra window doesn't need a wave to display
			Wave/Z DisplayedWave=$""
		endif
		
		//	Ensures that the value for the control has been updated before running the UpdateGraph function
		ControlUpdate /W=$B_Struct.win $B_Struct.ctrlName

		//	Runs the specified update function
		FUNCREF XPSViewProtoUpdate UpdateFunction=$B_Struct.userdata
		UpdateFunction(B_Struct.win, ActiveFolder, DisplayedWave)
	endif
	
	//	Needed, but not used
	Return 0
end



Static Function XPSViewToggleCheckBox(CB_Struct) : CheckBoxControl
//	Switches between listing all waves or just waves with associated fit waves
STRUCT WMCheckboxAction & CB_Struct

	//	If the state of the checkbox has changed. eventCode 2 = mouse up
	if (CB_Struct.eventCode==2)
	
		//	Ignores further calls to the checkbox procedure until this one has finished
		CB_Struct.blockReentry=1

		//	Finds the displayed wave
		DFREF ActiveFolder=XPSViewGetDataFolder(CB_Struct.win, "DataFolderPopUp")
		
		if (CmpStr(CB_Struct.win, "XPSViewSpectraWindow")!=0)
	
			//	Finds the name of the wave to display
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(CB_Struct.win, ActiveFolder, "DisplayedWavePopUp")	
		else
		
			//	The View Spectra window doesn't need a wave to display
			Wave/Z DisplayedWave=$""
		endif
		
		//	Ensures that the value for the control has been updated before running the UpdateGraph function
		ControlUpdate /W=$CB_Struct.win $CB_Struct.ctrlName

		//	Runs the specified update function
		FUNCREF XPSViewProtoUpdate UpdateFunction=$CB_Struct.userdata
		UpdateFunction(CB_Struct.win, ActiveFolder, DisplayedWave)
	endif
	Return 0
end



Static Function/DF XPSViewGetDataFolder(ActiveWindow, DataFolderPopupStr)
//	Finds the data folder selected by the popup menu
String ActiveWindow, DataFolderPopupStr

	//	Finds the active folder or group
	ControlInfo /W=$ActiveWindow $DataFolderPopupStr
	String GroupOrDataFolderName=S_Value
	DFREF ActiveFolder=$GroupOrDataFolderName
	
	//	If the data folder is invalid it is assumed to be the name of a saved group (for all windows, except for View Spectra and the NEXAFS windows)
	if ((DataFolderRefStatus(ActiveFolder)==0) && (CmpStr(ActiveWindow, "XPSViewSpectraWindow")!=0) && (CmpStr(ActiveWindow[0,14], "RawNEXAFSWindow")!=0) && (CmpStr(ActiveWindow[0,15], "ViewNEXAFSWindow")!=0))

		//	Returns the data folder for the selected group
		ActiveFolder=XPSViewGroupDataFolder(XPSViewGetGroupInfo(GroupOrDataFolderName))
		
		//	Checks if the View Images window is active
		if (CmpStr(ActiveWindow, "XPSViewImageWindow")==0)

			//	Checks if the group data folder is Spectra
			if (CmpStr("Spectra", GetDataFolder(0, ActiveFolder))==0)

				//	If the group folder is Spectra the neighbouring Images folder is returned, if it exists
				DFREF ParentFolder=$(GetDataFolder(1, ActiveFolder)+":")
				DFREF ImageFolder=ParentFolder:Images
				
				//	If the Images folder does not exist and invalid folder is returned instead
				if ((DataFolderRefStatus(ActiveFolder)!=0) && (DataFolderRefStatus(ParentFolder)!=0) && (DataFolderRefStatus(ImageFolder)!=0))
					ActiveFolder=ImageFolder
				else
					ActiveFolder=$""
				endif
			else
				ActiveFolder=$""
			endif
		endif
	endif

	//	Returns the active data folder
	Return ActiveFolder
end



Static Function/WAVE XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, DisplayedWavePopUpStr)
//	Finds the wave selected by the popup menu
String ActiveWindow
DFREF ActiveFolder
String DisplayedWavePopUpStr

	Wave/Z DisplayedWave=$""

	//	Checks if the data folder exists, if not the displayed wave selection is updated to display an error
	if (DataFolderRefStatus(ActiveFolder)==0)
		ControlUpdate /W=$ActiveWindow $DisplayedWavePopUpStr
	else

		//	Finds the name of the displayed wave
		ControlInfo /W=$ActiveWindow $DisplayedWavePopUpStr

		//	Igor 6.34A 400 character limit compatibility (Needed at ELETTRA)
		Variable a=(CmpStr(S_Value, "Wave does not exist")!=0) && (CmpStr(S_Value, "Data folder does not exist")!=0) && (CmpStr(S_Value, "No existing waves left in group")!=0) && (CmpStr(S_Value, "Group data folder does not exist")!=0)
		a=a && (CmpStr(S_Value, "Group or data folder does not exist")!=0) && (CmpStr(S_Value, "No waves in folder")!=0) && (CmpStr(S_Value, "- select wave to subtract -")!=0) && (CmpStr(S_Value, "- select photon flux wave -")!=0)

		if (a)

			//	Checks if the selected wave exists
			Wave/Z DisplayedWave=ActiveFolder:$S_Value
			if (WaveExists(DisplayedWave)==0)
				PopupMenu $DisplayedWavePopUpStr mode=1, popvalue="Wave does not exist", win=$ActiveWindow
			endif
		endif
	endif

	//	Returns the wave displayed in the window
	Return DisplayedWave
end



Static Function XPSViewImageUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
//	Updates the View Image graph. ActiveFolder is not needed, but is included to allow the update functions for all graphs to share the same FUNCREF
String ActiveWindow
DFREF ActiveFolder
Wave/Z DisplayedWave
Variable YSize=0, Scaling=0, Offset=0, i=0, CursorA=0, CursorAX=0, CursorAY=0, CursorB=0, CursorBX=0, CursorBY=0
String OffsetStatus="", ScaleStatus="", ResampleStatus=""

	DFREF ProgramFolder=root:Programming
	
	if (WaveExists(DisplayedWave)==0)

		//	Displays an empty image
		Make/O ProgramFolder:XPSViewImageDataWave={{0}}
		
		//	Deactivates the cursors
		Cursor/K/W=$ActiveWindow A
		Cursor/K/W=$ActiveWindow B
	else
	
		//	Finds the cursor positions
		if (StrLen(CsrWave(A, ActiveWindow))>0)
			CursorA=1
			CursorAX=hcsr(A, ActiveWindow)
			CursorAY=vcsr(A, ActiveWindow)
		endif
		if (StrLen(CsrWave(B, ActiveWindow))>0)
			CursorB=1
			CursorBX=hcsr(B, ActiveWindow)
			CursorBY=vcsr(B, ActiveWindow)
		endif
	
		//	Finds the clicked buttons
		OffsetStatus=GetUserData(ActiveWindow, "OffsetButton", "Status")
		ScaleStatus=GetUserData(ActiveWindow, "ScaleButton", "Status")
		ResampleStatus=GetUserData(ActiveWindow, "ResampleButton", "Status")

		//	Duplicates the selected wave into DataWave, to be displayed
		Wave DataWave=ProgramFolder:XPSViewImageDataWave
		if (WaveRefsEqual(DisplayedWave, DataWave)==0)
			Duplicate/O DisplayedWave ProgramFolder:XPSViewImageDataWave/WAVE=DataWave
			
			//	Removes any unit labels that would otherwise result in double axis labels
			SetScale/P x, DimOffset(DataWave, 0), DimDelta(DataWave, 0), "", DataWave
			SetScale/P y, DimOffset(DataWave, 1), DimDelta(DataWave, 1), "", DataWave
		endif

		//	Resamples the image if the button is pressed
		if (CmpStr(ResampleStatus, "1")==0)
			 XPSViewImageResampleImage()
		endif
		
		//	Normalizes the image by making the sum of each horizontal line equal to 0
		if (CmpStr(OffsetStatus, "1")==0)

			//	Normalizes with the sum of each horizontal line
			Duplicate/FREE/O DataWave TempDataWave
			MatrixOP/NTHR=0/O/S DataWave=subtractMean(TempDataWave,1)
		endif
			
		//	Normalizes the image by setting the maxium value for each horizontal line equal to 1
		if (CmpStr(ScaleStatus, "1")==0)
		
			//	Normalizes with the maxium values in each line
			YSize=DimSize(DataWave, 1)
			for (i=0; i<YSize; i=i+1)
				Duplicate/FREE/O/R=[][i] DataWave TempXWave
				Scaling=1/WaveMax(TempXWave)
				if (NumType(Scaling)!=0)
					Scaling=1				
				endif
				FastOP TempXWave=(Scaling)*TempXWave
				ImageTransform /D=TempXWave /G=(i) putCol DataWave
			endfor
		endif
		
		//	Repositions the cursors
		if (CursorA==1)
			Cursor/I /W=$ActiveWindow A XPSViewImageDataWave, CursorAX, CursorAY
		endif
		if (CursorB==1)
			Cursor/I /W=$ActiveWindow B XPSViewImageDataWave, CursorBX, CursorBY
		endif
	endif
end



Static Function XPSViewImageResampleImage()
//	Resamples the image to a lower resolution given by the new x and y spacings X and Y
String CtrlName
Variable OldOffsetX=0, OldOffsetY=0, OldDeltaX=0, OldDeltaY=0, OldNumX=0, OldNumY=0, NewOffsetX=0, NewOffsetY=0, NewDeltaX=0, NewDeltaY=0, NewNumX=0, NewNumY=0, SkipX=0, SkipY=0
Variable iX=0, iY=0, a=0, b=0, ResampleX=0, ResampleY=0
DFREF ProgramFolder=root:Programming
String ActiveWindow="XPSViewImageWindow"

	//	Finds the resample values
	ControlInfo /W=$ActiveWindow SetResampleX
	ResampleX=V_Value
	ControlInfo /W=$ActiveWindow SetResampleY
	ResampleY=V_Value
	
	//	Saves the original scaling and number of points
	Wave DataWave=ProgramFolder:XPSViewImageDataWave

	OldOffsetX=DimOffset(DataWave, 0)
	OldOffsetY=DimOffset(DataWave, 1)

	OldDeltaX=DimDelta(DataWave, 0)
	OldDeltaY=DimDelta(DataWave, 1)

	OldNumX=DimSize(DataWave, 0)
	OldNumY=DimSize(DataWave, 1)

	//	Forces the resample values to be positive integer values, smaller than the dimensions of the image
	ResampleX=Min(OldNumX, Max(1, Round(Abs(ResampleX))))
	ResampleY=Min(OldNumY, Max(1, Round(Abs(ResampleY))))
		
	//	Calculates the new scaling and number of points
	NewDeltaX=OldDeltaX*ResampleX
	NewDeltaY=OldDeltaY*ResampleY
		
	NewNumX=Trunc(OldNumX/ResampleX)
	NewNumY=Trunc(OldNumY/ResampleY)
	
	//	Calculates the number of data points to skip, to align the new image to be centered with the old image
	SkipX=Trunc((OldNumX-NewNumX*ResampleX)/2)
	SkipY=Trunc((OldNumY-NewNumY*ResampleY)/2)
	
	//	Calculates the new offsets, corresponding to the initial number of data points skipped
	NewOffsetX=OldOffsetX+OldDeltaX*0.5*(ResampleX-1+SkipX)
	NewOffsetY=OldOffsetY+OldDeltaY*0.5*(ResampleY-1+SkipY)
	
	//	Temporary wave used to store the resampling
	Make/FREE/O/N=(NewNumX, NewNumY) TempWave=0
		
	//	Resamples the data
	for (iX=0; iX<ResampleX; iX+=1)
		a=iX+SkipX
		for (iY=0; iY<ResampleY; iY+=1)
			b=iY+SkipY
			MultiThread TempWave[][]+=DataWave[p*ResampleX+a][q*ResampleY+b]		
		endfor
	endfor
		
	//	Copies the resampled wave into the displayed datawave
	Duplicate/O TempWave ProgramFolder:XPSViewImageDataWave

	//	Copies the axis labels from the old wave into the new resampled wave
	SetScale /P x, NewOffsetX, NewDeltaX, "", DataWave
	SetScale /P y, NewOffsetY, NewDeltaY, "", DataWave	
end



Static Function XPSViewImageSpectra(B_Struct) : ButtonControl
//	If no cursors are placed on the image, the selected group, or the selected data folder and spectrum is displayed in View Spectra
//	If at least one cursor is active, the spectra above and below Cursor A or the spectra outside and inside Cursors A and B are displayed together with the combined spectrum
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Finds the displayed wave
		DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")
		
		//	Finds the name of the displayed wave
		String ActiveName=NameOfWave(DisplayedWave)

		//	If the image is located in :Images the spectrum will be saved in the neighbouring :Spectra if such a folder exists
		DFREF ParentFolder=$(GetDataFolder(1, ActiveFolder)+":")
		DFREF SpectraFolder=ParentFolder:Spectra

		//	If the active folder is not :Images or a neighbouring :Spectra folder doesn't exist, the active folder is used instead
		if ((DataFolderRefStatus(ParentFolder)==0) || (DataFolderRefStatus(SpectraFolder)==0) || (CmpStr("Images", GetDataFolder(0, ActiveFolder))!=0))
			DFREF SpectraFolder=ActiveFolder
		endif

		//	Checks if the cursors are active
		Variable CursorA=StrLen(CsrInfo(A, B_Struct.win))
		Variable CursorB=StrLen(CsrInfo(B, B_Struct.win))

		//	If at least one cursor is active, the spectra above and below Cursor A or the spectra outside and inside Cursors A and B are displayed together with the combined spectrum
		if (WaveExists(DisplayedWave) && (DataFolderRefStatus(ActiveFolder)!=0) && ((CursorA>0) || (CursorB>0)))

			//	If the resulting spectra are to be saved in the same folder as the image an "_" is added to the names of the spectra
			if (DataFolderRefsEqual(SpectraFolder, ActiveFolder))
				ActiveName+="_"
			endif
		
			//	Creates a save wave dialog where the OK button will execute XPSViewImageCreateSpectrum
			XPSViewSaveWaveDialog("Save spectrum as", GetDataFolder(1, SpectraFolder), ActiveName, "EccentricXPS#XPSViewImageCreateSpectrum", "", B_Struct.win)

		//	If no cursors are active on the image, the selected group, or the selected data folder and spectrum, are displayed in View Spectra
		else

			//	Finds the selected data folder or group
			ControlInfo /W=$B_Struct.win DataFolderPopUp
			String FolderOrGroup=S_Value
		
			//	If a data folder was selected instead of a group, the displayed wave is selected in the View Spectra window
			if (CmpStr(FolderOrGroup, GetDataFolder(1, ActiveFolder))==0)

				//	Displays the folder or group in the View Spectra window
				XPSViewSpectraChangeSelection(GetDataFolder(1, SpectraFolder), 1)
				
				Wave/Z SpectrumWave=SpectraFolder:$ActiveName
		
				if ((DataFolderRefStatus(SpectraFolder)!=0) && WaveExists(SpectrumWave))

					//	Creates the necessary waves for the View Spectra listbox
					DFREF ProgramFolder=root:Programming
					Make/T/O ProgramFolder:XPSViewWaveList={{ActiveName}, {""}, {""}}
					Make/I/U/O/N=(1, 3, 2) ProgramFolder:XPSViewWaveListSel=0
					Make/WAVE/O ProgramFolder:XPSViewWaveListWaves={SpectrumWave}
			
					//	Defines the first column in the listbox as a checked (bit 4) checkbox (bit 5)
					Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
					XPSViewWaveListSel[0][0][0]=48
	
					//	Makes the colours affect the background colour, change to forecolors to affect foreground (text) colours
					SetDimLabel 2,1, backColors, XPSViewWaveListSel
			
					//	Refreshes the View Spectra graph
					XPSViewSpectraUpdateGraph("XPSViewSpectraWindow", SpectraFolder, $"")
				endif
			else
			
				//	Displays the group in the View Spectra window
				XPSViewSpectraChangeSelection(FolderOrGroup, 1)
			endif
		endif
	endif
	Return 0
end



Static Function XPSViewImageCreateSpectrum(SaveFolder, SaveName, ActiveWindow)
//	Converts the two-dimensional image into spectra
//	If at least one cursor is active, the spectra above and below Cursor A or the spectra outside and inside Cursors A and B are displayed together with the combined spectrum
DFREF SaveFolder
String SaveName, ActiveWindow

	//	Finds the displayed wave
	DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
	Wave/Z DataWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
	
	//	Checks if the source wave exists
	if (WaveExists(DataWave) && (DataFolderRefStatus(ActiveFolder)!=0) && (DataFolderRefStatus(SaveFolder)!=0))
		
		//	Creates the spectra as designated by target name
		Duplicate/O /R=[][0] DataWave, SaveFolder:$SaveName/WAVE=Full1DWave
		MatrixOP/NTHR=0/O/S Full1DWave=sumRows(DataWave)

		//	Checks if the cursors are active
		Variable CursorA=StrLen(CsrInfo(A, ActiveWindow))
		Variable CursorB=StrLen(CsrInfo(B, ActiveWindow))
		
		//	Finds the positions of the cursors A and B
		Variable A=vcsr(A, ActiveWindow)
		Variable B=vcsr(B, ActiveWindow)
		
		//	Creates the waves to hold the spectrum inside (S1) and outside (S2) of the cursors or, if only one cursor is active, below (S1) and above (S2) the cursor
		Duplicate/O Full1DWave, SaveFolder:$(SaveName+"_S1")/WAVE=S1Wave, SaveFolder:$(SaveName+"_S2")/WAVE=S2Wave
		
		if ((CursorA>0) && (CursorB>0))

			//	Calculates the spectrum inside the cursors
			Duplicate/FREE/O /R=()(A, B) DataWave, TempDataWave
			MatrixOP/NTHR=0/O/S S1Wave=sumRows(TempDataWave)
			
			//	Calculates the spectrum outside the cursors
			FastOP S2Wave=Full1DWave-S1Wave
			
		elseif (CursorA>0)
		
			//	Calculates the spectrum below cursor A
			Duplicate/FREE/O/R=()(DimOffset(DataWave, 1), A) DataWave,TempDataWave
			MatrixOP/NTHR=0/O/S S1Wave=sumRows(TempDataWave)
				
			//	Calculates the spectrum above cursor A
			FastOP S2Wave=Full1DWave-S1Wave
		
		elseif (CursorB>0)
		
			//	Calculates the spectrum below cursor B
			Duplicate/FREE/O/R=()(DimOffset(DataWave, 1), B) DataWave,TempDataWave
			MatrixOP/NTHR=0/O/S S1Wave=sumRows(TempDataWave)
				
			//	Calculates the spectrum above cursor B
			FastOP S2Wave=Full1DWave-S1Wave
		
		endif

		//	Creates the View Spectra window
		XPSViewSpectraChangeSelection(GetDataFolder(1, SaveFolder), 1)

		//	Creates the necessary waves for the View Spectra Listbox
		DFREF ProgramFolder=root:Programming
		Make/T/O ProgramFolder:XPSViewWaveList/WAVE=XPSViewWaveList={{NameOfWave(Full1DWave), NameOfWave(S1Wave), NameOfWave(S2Wave)}, {"", "", ""}, {"", "", ""}}
		Make/I/U/O ProgramFolder:XPSViewWaveListSel/WAVE=XPSViewWaveListSel={{{48, 48, 48}, {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {1, 2, 3}}}
		Make/WAVE/O ProgramFolder:XPSViewWaveListWaves/WAVE=XPSViewWaveListWaves={Full1DWave, S1Wave, S2Wave}

		//	Makes the colours affect the background colour, change to forecolors to affect foreground (text) colours
		SetDimLabel 2,1,backColors, XPSViewWaveListSel

		//	Updates the View Spectra graph in the Scale mode
		XPSViewSpectraUpdateGraph("XPSViewSpectraWindow", SaveFolder, $"")
	endif
end





//     -----<<<<<     View Spectra     >>>>>-----

Static Function XPSViewSpectra(FolderOrGroup, ForceSelection)
//	Brings the View Spectra window to the front, or creates it if it does not exist
String FolderOrGroup
Variable ForceSelection
String ActiveWindow="XPSViewSpectraWindow"

	//	Updates the history of procedures used with the current experiment
	XPSUpdateProcedureHistory()

	//	Does not create the View Spectra window if it already exists
	DoWindow /F $ActiveWindow
	if ((V_Flag==0) || (ForceSelection==1))

		//	Changes the displayed selection
		XPSViewSpectraChangeSelection(FolderOrGroup, ForceSelection)
	else

		//	Finds the active data folder in the View Spectra window
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")

		//	Refreshes the graph
		XPSViewSpectraUpdateGraph(ActiveWindow, ActiveFolder, $"")
	endif
end



Static Function XPSViewSpectraChangeSelection(FolderOrGroup, ForceSelection)
//	Changes the selection in the View Spectra window without killing the window
String FolderOrGroup
Variable ForceSelection
String ActiveWindow="XPSViewSpectraWindow"

	//	Does not create the View Spectra window if it already exists
	DoWindow/F $ActiveWindow
	if (V_Flag==0)

		//	Creates the View Spectra window
		XPSViewSpectraCreateWindow(FolderOrGroup, ForceSelection)
	else

		//	If FolderOrGroup starts with root: and ends in : it is assumed to be a data folder
		Variable a=StrLen(FolderOrGroup)-1
		if ((CmpStr(FolderOrGroup[0,4], "root:")==0) && (CmpStr(FolderOrGroup[a,a], ":")==0))

			//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is forced
			PopupMenu DataFolderPopUp popmatch=FolderOrGroup, win=$ActiveWindow
			ControlInfo /W=$ActiveWindow DataFolderPopUp
			if (CmpStr(FolderOrGroup, S_Value)!=0)
				PopupMenu DataFolderPopUp mode=1, popvalue=FolderOrGroup, win=$ActiveWindow
			endif
			DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")

			//	Sets the axis scaling and labels to match the type of spectrum displayed and updates the graph
			XPSViewSpectraUpdateGraphDF(ActiveWindow, ActiveFolder, $"")
		else

			//	Plots the selected group from the saved groups
			XPSViewDisplayGroup(FolderOrGroup)
		endif
	endif
end



Static Function XPSViewSpectraCreateWindow(FolderOrGroup, ForceSelection)
//	Creates the View Spectra window for displaying multiple XPS spectra
String FolderOrGroup
Variable ForceSelection
Variable i=0, n=120/ScreenResolution
String ActiveWindow="XPSViewSpectraWindow"

	//	Creates the folder to hold all the permanent waves, used for different bureaucratic purposes, such as listboxes, displaying waves, etc...
	NewDataFolder/O root:Programming
	DFREF ProgramFolder=root:Programming
	
	//	Creates the folder to hold all the temporary waves, created during a function call and deleted again at the end of the function call.
	NewDataFolder/O root:Programming:Temp
	
	//	Checks if the waves used for the listbox exists
	Wave/Z/T XPSViewWaveList=ProgramFolder:XPSViewWaveList
	Wave/Z/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
	Wave/Z/WAVE XPSViewWaveListWaves=ProgramFolder:XPSViewWaveListWaves

	//	If one of the listbox waves does not exist, blank waves are created instead
	if ((WaveExists(XPSViewWaveList)==0) || (WaveExists(XPSViewWaveListSel)==0) || (WaveExists(XPSViewWaveListWaves)==0))
	
		//	Creates empty listbox waves
		XPSViewCreateEmptyListBoxWaves("")
		Wave/T XPSViewWaveList=ProgramFolder:XPSViewWaveList
		Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
		Wave/WAVE XPSViewWaveListWaves=ProgramFolder:XPSViewWaveListWaves
	endif

	//	Creates the wave used to define the colours of the XPS View Spectra traces
	XPSViewSpectraCreateColourWave()
	Wave/W/U ColourWave=ProgramFolder:XPSViewColours
		
	//	Creates a new zero length datawave as a placeholder in the empty graph. This will keep the graph up when all other traces have been removed.
	Make/O/N=0 ProgramFolder:XPSViewEmptyWave /Wave=EmptyWave
	
	//	Creates the graph.
	DoWindow /K $ActiveWindow
	Display /W=(45, 80, 45+745*n, 80+363*n) /K=1 /N=$ActiveWindow EmptyWave
	ModifyGraph /W=$ActiveWindow hideTrace(XPSViewEmptyWave)=1
	ShowInfo /W=$ActiveWindow
	
	//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
	SetWindow $ActiveWindow hook(KillHook)=XPSViewSpectraHookFunc
	
	//	Adjusts the left margin to make room for the listbox. Using a control bar to make a margin will prevent the margin from being printed.
	ControlBar/L/W=$ActiveWindow 275
	ModifyGraph /W=$ActiveWindow cbRGB=(65534,65534,65534)

	//	Adjusts the top margin to make room for the buttons. I don't use a control bar for the top margin because Igor insists on a horizontal line between the top control bar and the graph
	ModifyGraph /W=$ActiveWindow margin(top)=45
	
	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$ActiveWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$ActiveWindow B

	//	Deletes the marked waves
	Button DeleteButton proc=EccentricXPS#XPSViewDeletePrompt, pos={5,3}, size={55,20}, title="Delete", win=$ActiveWindow, help={"Deletes the selected spectra"}
	//	Refreshes the list of waves. Also holds the close window (eventCode = -1) clean up of waves associated with the window
	Button RefreshButton proc=EccentricXPS#XPSViewRefresh, size={55,20}, title="Refresh", win=$ActiveWindow, help={"Refreshes the list of spectra"}
	//	Clears all checkboxes
	Button ClearButton proc=EccentricXPS#XPSViewClear, size={55,20}, title="Clear", win=$ActiveWindow, help={"Deselects all spectra in the list"}
	//	Selects all checkboxes
	Button AllButton proc=EccentricXPS#XPSViewAll, size={55,20}, title="All", win=$ActiveWindow, help={"Selects all spectra in the list"}
	//	Limits the displayed spectra to those matching the string in the search box
	SetVariable SetSearch, proc=EccentricXPS#XPSViewSetVariable, pos={5,29}, size={227,20}, title="Search ", value=_STR:"", userdata="EccentricXPS#XPSViewSpectraUpdateGraph", win=$ActiveWindow, help={"Limits the waves displayed in the listbox to only those matching the search string"}
	//	Clears the search box
	Button ClearSearchButton proc=EccentricXPS#XPSViewClearSearch, pos={238,28}, size={16,16}, title="\f01X", win=$ActiveWindow, help={"Clears the search string"}

	//	Creates a group of out the selected spectra
	Button GroupButton proc=EccentricXPS#XPSViewGroup, pos={360,3}, size={70,20}, title="Group", win=$ActiveWindow, help={"Creates a group of the displayed spectra"}
	//	Disbands the group with the name specified in the box to the right
	Button DisbandButton proc=EccentricXPS#XPSViewDisband, pos={360,26}, size={70,20}, title="Disband", win=$ActiveWindow, help={"Disbands the group with the selected name"}
	//	Popup menu to select the group to be displayed
	PopupMenu GroupPopUp bodyWidth=170, mode=1, pos={560, 3}, proc=EccentricXPS#XPSViewGroupPopUp, value=EccentricXPS#XPSViewGroupPopUpList(), popvalue="Select a group to display", win=$ActiveWindow, help={"Selects which group of spectra to display"}
	//	Box used to type a group name
	SetVariable SetGroupName bodyWidth=170, pos={560, 29}, title=" ", value=_STR:"", win=$ActiveWindow, help={"The name of the group to create or disband"}

	//	Combines all visible traces into one wave
	Button AverageButton proc=EccentricXPS#XPSViewAverage, pos={640,3}, size={70,20}, title="Average", win=$ActiveWindow, help={"Creates an average of the displayed spectra"}
	//	Shifts the binding energies of all visible traces by a given amount
	Button ShiftButton proc=EccentricXPS#XPSViewShiftPrompt, pos={640,26}, size={70,20}, title="Shift", win=$ActiveWindow, help={"Shifts the spectrum marked by Cursor A. If cursor A is not pressent all displayed spectra will be shifted"}

	//	Offsets all traces by their minimum value
	Button OffsetButton proc=EccentricXPS#XPSViewToggleButton, pos={740,3}, size={70,20}, title="Offset", userdata="EccentricXPS#XPSViewSpectraUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Subtracts its minimum value from each spectrum"}
	//	Scales all traces by their maximum value
	Button ScaleButton proc=EccentricXPS#XPSViewToggleButton, size={70,20}, title="Scale", userdata="EccentricXPS#XPSViewSpectraUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Divides each spectrum by its maximum value"}
	//	Creates a waterfall plot
	Button WaterfallButton proc=EccentricXPS#XPSViewToggleButton, size={70,20}, title="Waterfall", userdata="EccentricXPS#XPSViewSpectraUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Creates a waterfall plot"}
	//	Sets the y-spacing used for the waterfall plot
	SetVariable SetWaterfall, limits={0.01,2,0}, format="%.3f", proc=EccentricXPS#XPSViewSetWaterfall, size={50,20}, title=" ", value=_NUM:0.2, win=$ActiveWindow, help={"Determines the spacing between traces in the waterfall plot"}
	//	Adds a tag to a trace
	Button TagButton proc=EccentricXPS#XPSViewAddTag, pos={740,26}, size={70,20}, title="Add Tag", win=$ActiveWindow, help={"Adds a tag to the trace marked by Cursor A"}
	//	Hides all tags
	Button HideTagsButton proc=EccentricXPS#XPSViewToggleButton, size={70,20}, title="Hide Tags", userdata="EccentricXPS#XPSViewSpectraUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Hides all tags in the graph"}
	//	Shows all hidden traces
	Button ShowHiddenButton proc=EccentricXPS#XPSViewShowHidden, size={70,20}, title="Show All", win=$ActiveWindow, help={"Show all hidden traces in the graph"}

	//	Creates a table with the displayed spectrum and fit waves. Useful for copying and pasting into other programs
	Button EditWavesButton pos={1060,3}, proc=EccentricXPS#XPSViewEditWaves, size={70,20}, title="Table", win=$ActiveWindow, help={"Creates a table with the displayed spectrum and fit waves. Useful for copying and pasting into other programs"}
	//	Selects the displayed spectrum type
	PopupMenu DisplayTypePopUp bodyWidth=70, mode=5, pos={1080,26}, proc=EccentricXPS#XPSViewDisplayTypePopup, value="AES;IR;NEXAFS;QMS;XPS;", win=$ActiveWindow, help={"Selects the displayed spectrum type"}
	//	Opens the associated help file
	Button HelpButton proc=EccentricXPS#XPSViewHelp, pos={1140,3}, size={70,20}, title="Help", userdata="View Spectra", win=$ActiveWindow, help={"Opens the help file"}
	//	Creates a copy of the View Spectra graph, without any controls, meant as a starting point for a more presentable figure
	Button NewWinButton proc=EccentricXPS#XPSViewNewWin, pos={1140,26}, size={70,20}, title="New Win", win=$ActiveWindow, help={"Creates a copy of the graph, without any controls, meant as a starting point for a more presentable figure"}

	//	Selects the active data folder, listing root: and all it's subfolders with the exception of root:Programming, all it's subfolders, and all folders named SavedFits and all their subfolders. To list the default folder (set with SetDataFolder) and all it's subfolders instead, with no exceptions, use EccentricXPS#XPSViewDataFoldersList(:, $"", "")
	PopupMenu DataFolderPopUp bodyWidth=250, mode=1, pos={203, 532}, proc=EccentricXPS#XPSViewDataFolderPopupMenu, value=EccentricXPS#XPSViewDataFoldersList(root:, root:Programming, "SavedFits"), userdata="EccentricXPS#XPSViewSpectraUpdateGraphDF", win=$ActiveWindow, help={"Selects the data folder to look for waves in"}
	//	Creates a checkbox to switch between showing just the raw measured data or the raw data together with the fit waves
	CheckBox FittedCheckBox, mode=0, proc=EccentricXPS#XPSViewToggleCheckBox, pos={5*1.25/n, 560}, side=0, title="Display Fits", value=0, userdata="EccentricXPS#XPSViewSpectraUpdateGraph", win=$ActiveWindow, help={"Limits the list of spectra to only those that have been fitted"}
	//	Changes the display style of the fitted spectra
	CheckBox FancyPlotCheckBox, mode=0, proc=EccentricXPS#XPSViewToggleCheckBox, pos={5*1.25/n, 585}, side=0, title="Fancy Plot", value=0, userdata="EccentricXPS#XPSViewSpectraUpdateGraph", win=$ActiveWindow, help={"Changes the display style of the fitted spectra"}
	//	Does not display the selected spectra (useful when creatng very large groups of spectra)
	CheckBox DoNotDisplay, mode=0, proc=EccentricXPS#XPSViewToggleCheckBox, pos={100*1.25/n, 585}, side=0, title="Do Not Display", value=0, userdata="EccentricXPS#XPSViewSpectraUpdateGraph", win=$ActiveWindow, help={"Does not display the selected spectra (useful when creatng very large groups of spectra)"}
	//	Creates a checkbox to switch between waveform waves and waves with associated X-axis waves
	//CheckBox XWavesCheckBox, mode=0, proc=EccentricXPS#XPSViewToggleCheckBox,  pos={65*1.25/n, 562}, side=0, title="XWaves", value=0, userdata="EccentricXPS#XPSViewSpectraUpdateGraph", win=$ActiveWindow, help={"Limits the list of spectra to only those with associated X-axis waves in the :XWaves subfolder"}
	//	Changes the default data folder, set by SetDataFolder(), to the displayed folder
	Button SetButton proc=EccentricXPS#XPSViewSet, pos={100,559}, size={70,20}, title="Set", win=$ActiveWindow, help={"Changes the default data folder to the displayed folder"}
	//	Changes the displayed data folder to the default data folder set by SetDataFolder()
	Button RestoreButton proc=EccentricXPS#XPSViewRestore, size={70,20}, title="Restore", win=$ActiveWindow, help={"Changes the displayed data folder to the default data folder"}

	//	Creates a list box with all available waves
 	XPSViewRebuildListBox(ActiveWindow)
 	
	//	If FolderOrGroup starts with root: and ends in : it is assumed to be a data folder
	Variable a=StrLen(FolderOrGroup)-1
	if ((CmpStr(FolderOrGroup[0,4], "root:")==0) && (CmpStr(FolderOrGroup[a,a], ":")==0))

		//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or the default data folder is used instead, depending on the value of ForceSelection
		PopupMenu DataFolderPopUp popmatch=FolderOrGroup, win=$ActiveWindow
		ControlInfo /W=$ActiveWindow DataFolderPopUp
		if (CmpStr(FolderOrGroup, S_Value)!=0)
			if (ForceSelection==1)
				PopupMenu DataFolderPopUp mode=1, popvalue=FolderOrGroup, win=$ActiveWindow
			else
				PopupMenu DataFolderPopUp popmatch=GetDataFolder(1), win=$ActiveWindow
			endif
		endif
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		
		//	Sets the axis scaling and labels to match the type of spectrum displayed and updates the graph
		XPSViewSpectraUpdateGraphDF(ActiveWindow, ActiveFolder, $"")
	else
	
		//	Plots the selected group from the saved groups
		XPSViewDisplayGroup(FolderOrGroup)
	endif
end



Function XPSViewRebuildListBox(ActiveWindow)
//	Recreates the View Spectra listbox
//	This function became necessary because of a bug in Igor Pro 6.36, where selecting Rename or Duplicate from the right-click menu bugs out the listbox
String ActiveWindow
String ListBoxName="WaveListBox"
Variable n=120/ScreenResolution

	//	The listbox waves
	DFREF ProgramFolder=root:Programming
	Wave/T XPSViewWaveList=ProgramFolder:XPSViewWaveList
	Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
	Wave/W/U ColourWave=ProgramFolder:XPSViewColours

	//	Recreates the listbox
	KillControl/W=$ActiveWindow $ListBoxName
	ListBox $ListBoxName, clickEventModifiers=31, colorWave=ColourWave, editStyle=1, listWave=XPSViewWaveList, mode=0, pos={3, 60-5*n}, proc=EccentricXPS#XPSViewWaveListBox, selWave=XPSViewWaveListSel, size={250,525-60+5*n}, widths={185,20,20}, win=$ActiveWindow, help={"Selects the spectra to display in the graph"}
end



Function XPSViewSpectraHookFunc(s)
//	The hook function for the View Spectra window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)

		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1
			
			//	Kills the Edit Waves window, if it exists
			DoWindow/K XPSViewEditWavesWindow
			
			//	Saves all tags present on the graph as wave notes
			XPSViewSaveTags(s.winName)
			
			//	Kills the waves associated with View Spectra, but leaves the listbox waves intact and delays the execution until the window has been killed
			//	Does not delete the colour wave. This ensures changes to the colours are not overwritten when the window is recreated
			Execute/P/Q "KillWaves root:Programming:XPSViewEmptyWave, root:Programming:XPSViewWaveListSel, root:Programming:XPSViewWaveList, root:Programming:XPSViewWaveListWaves"
			break
	endswitch
	
	//	Changes the cursor appearance. Not used, just pasted here as a reference.
	//	s.doSetCursor=1
	//	s.cursorCode=.....
	//
	//	0 = <cursor shape determined by objects in window>
	//	1 = normal cursor
	//	2 = hourglass
	//	3 = crosshair
	//	4 = left + up arrow
	//	5 = right + left arrow
	//	6 = up + down arrow
	//	7 = right + down arrow
	//	8 = up + down + right + left arrow
	//	9 = inverted top hat
	//	10 = left + down arrow
	//	11 = right + up arrow
	//	12 = small crosshair
	//	13 = hand
	//	14 = box with cross
	//	15 = circle with cross
	//	16 = TV screen
	//	17 = question mark
	//	18 = crosshair with small box
	//	19 = jagged line / lightning bolt
	//	20 = <same as 12>
	//	21 = Large up + down + right + left arrow
	//	....

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function XPSViewSpectraCreateColourWave()
//	Creates the wave used to define the colours of the XPS View Spectra traces. The first row is not used for the listbox, but is instead used to define the colour of the fit waves	
DFREF ProgramFolder=root:Programming

	Variable ValidWaveExists=0

	//	Checks if a valid colour wave already exists, if so it is not overwritten
	//	This has one potential problem: If something is wrong with the colour wave, it will never be fixed. The user has no option to reset it.
	Wave/W/U/Z Colours=ProgramFolder:XPSViewColours
	if (WaveExists(Colours))
		ValidWaveExists=((DimSize(Colours, 0)>2) && (DimSize(Colours, 1)==3))
	endif
	
	if (ValidWaveExists==0)

		//	The number of repeating colours
		Variable n=9

		//	Only supports 2047 traces plotted simultaneously, which should be more than plenty
		Make/W/U/O/N=(2048, 3) ProgramFolder:XPSViewColours /WAVE=Colours
	
		//	The colour used for the fit waves in the View Spectra graph
		Colours[0][0]=40000;		Colours[0][1]=40000;		Colours[0][2]=40000		//	Grey
	
		//	The colours of the traces in the View Spectra graph ( [3,*;n] means starting at point 3 until the end of the wave (*) in steps of n )
		Colours[1,*;n][0]=65535;	Colours[1,*;n][1]=0;		Colours[1,*;n][2]=0		//	Red
		Colours[2,*;n][0]=0;		Colours[2,*;n][1]=0;		Colours[2,*;n][2]=65535	//	Blue
		Colours[3,*;n][0]=0;		Colours[3,*;n][1]=0;		Colours[3,*;n][2]=0		//	Black
		Colours[4,*;n][0]=0;		Colours[4,*;n][1]=52428;	Colours[4,*;n][2]=0		//	Green
		Colours[5,*;n][0]=39321;	Colours[5,*;n][1]=26214;	Colours[5,*;n][2]=13107	//	Brown
		Colours[6,*;n][0]=0;		Colours[6,*;n][1]=39321;	Colours[6,*;n][2]=52428	//	Teal
		Colours[7,*;n][0]=65535;	Colours[7,*;n][1]=26214;	Colours[7,*;n][2]=0		//	Orange
		Colours[8,*;n][0]=54998;	Colours[8,*;n][1]=0;		Colours[8,*;n][2]=37779	//	Purple
		Colours[9,*;n][0]=60535;	Colours[9,*;n][1]=60535;	Colours[9,*;n][2]=0		//	Yellow
	endif
end



Static Function XPSViewCreateColourPalette(PanelXPos, PanelYPos, WaveNumber, ViewSpectraWindow)
//	Creates a popup window to select a trace colour from
//	PROBLEM: This is sort of implemented now, but makes little sense unless the position in the waterfall plot and the colours can be separated (maybe with an extra index in the SelWave??)
Variable PanelXPos, PanelYPos, WaveNumber
String ViewSpectraWindow

	//	The wave used to define the colours of the XPS View Spectra traces
	DFREF ProgramFolder=root:Programming
	Wave/W/U ColourWave=ProgramFolder:XPSViewColours
	
	Variable NumberOfColours=0
	Variable ColourIndex=2
	
	//	Creates a one-dimensional wave of the red values in ColourWave for FindLevel to search through
	Duplicate/FREE/O/R=[][0] ColourWave, ColourWaveRed
	
	//	Calculates the number of available colours, by looking for when the first colour is repeated in ColourWave. The code assumes at least two colours in ColourWave
	do
		FindValue /S=(ColourIndex+1) /I=(ColourWave[1][0]) /Z ColourWaveRed
		ColourIndex=V_value
	while (((ColourWave[ColourIndex][0]!=ColourWave[1][0]) || (ColourWave[ColourIndex][1]!=ColourWave[1][1]) || (ColourWave[ColourIndex][2]!=ColourWave[1][2])) && (ColourIndex!=-1))
	
	//	All colours are unique and the number of colours are therefore equal to the number of points in the wave (minus the first point which is not used)
	if (ColourIndex==-1)
		NumberOfColours=DimSize(ColourWave, 0)-1

	//	Uses the first repeating colour to calculate the number of unique colours in the colour wave
	else
		NumberOfColours=ColourIndex-1
	endif

	//	Calculates the size of the square used to hold the colour boxes (N x N square)
	Variable N=Ceil(Sqrt(NumberOfColours))

	//	Defines the size of the colour boxes, the border around them and the space between them
	Variable BorderSize=10
	Variable BoxWidth=25
	Variable BoxHeight=25
	Variable BoxSpacing=5

	//	Calculates the size of the panel to hold the boxes
	Variable PanelWidth=2*BorderSize+N*BoxWidth+(N-1)*BoxSpacing
	Variable PanelHeight=2*BorderSize+N*BoxHeight+(N-1)*BoxSpacing
	
	//	Prevents the colour palette panel from being created behind the View Spectra window, if the View Spectra window wasn't the active window at the time the listbox was clicked
	DoWindow/F $ViewSpectraWindow

	//	Creates the new panel to hold the colour boxes
	String ActiveWindow="XPSViewColourPaletteWindow"
	DoWindow/K $ActiveWindow
	NewPanel /K=1 /N=$ActiveWindow /W=(PanelXPos, PanelYPos, PanelXPos+PanelWidth, PanelYPos+PanelHeight) as "Palette"
	
	//	Saves the parameters used to create the colour palette window in the userdata of the window
	String UserDataStr=""
	UserDataStr=ReplaceNumberByKey("NumberOfColours", UserDataStr, NumberOfColours, "=", ";", 1)
	UserDataStr=ReplaceNumberByKey("N", UserDataStr, N, "=", ";", 1)
	UserDataStr=ReplaceNumberByKey("BorderSize", UserDataStr, BorderSize, "=", ";", 1)
	UserDataStr=ReplaceNumberByKey("BoxWidth", UserDataStr, BoxWidth, "=", ";", 1)
	UserDataStr=ReplaceNumberByKey("BoxHeight", UserDataStr, BoxHeight, "=", ";", 1)
	UserDataStr=ReplaceNumberByKey("BoxSpacing", UserDataStr, BoxSpacing, "=", ";", 1)
	UserDataStr=ReplaceNumberByKey("WaveNumber", UserDataStr, WaveNumber, "=", ";", 1)
	UserDataStr=ReplaceStringByKey("ViewSpectraWindow", UserDataStr, ViewSpectraWindow, "=", ";", 1)
	
	//	Associates a hook function with the window. This will be used to determine which colour box has been clicked
	SetWindow $ActiveWindow hook(KillHook)=XPSViewColourPaletteHookFunc, userdata=(UserDataStr) 

	//	Creates the colour boxes
	Variable i=0, LIstBoxXPos=0, ListBoxYPos=0, Nx=0, Ny=0
	for (i=0; i<NumberOfColours; i+=1)
	
		//	Calculates the box position in the N x N square
		Nx=mod(i, N)
		Ny=trunc(i/N)
		
		//	Calculates the position of the new listbox
		ListBoxXPos=BorderSize+Nx*(BoxWidth+BoxSpacing)
		ListBoxYPos=BorderSize+Ny*(BoxHeight+BoxSpacing)

		//	Creates a colour box
		ValDisplay $("ColourBox"+Num2iStr(i)), barmisc={BoxWidth, 0}, barBackColor=(ColourWave[i+1][0], ColourWave[i+1][1], ColourWave[i+1][2]), disable=2, frame=3, mode=0, pos={ListBoxXPos, ListBoxYPos}, size={BoxWidth, BoxHeight}, value=_NUM:(i+1), win=$ActiveWindow, help={"Selects the colour of the trace"}
	endfor
end



Function XPSViewColourPaletteHookFunc(s)
//	The hook function for the View Spectra colour palette window.
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was deactivated (by clicking the mouse outside the window)
		case 1:
		
			//	Indicates that an action has taken place
			hookResult=1
			
			//	Kills the colour palette window
			KillWindow $s.winName
		break
	
		//	Mouse down (the mouse was clicked)
		case 3:
		
			//	If a left (bit 1) or right (bit 4) mouse click was used
			if ((s.eventMod==1) || (s.eventMod==16))
			
				//	Indicates that an action has taken place
				hookResult=1
				
				//	Retrieves the parameters used to build the colour palette window
				String UserDataStr=GetUserData(s.winName, "", "")

				//	Defines the size of the colour boxes, the border around them and the space between them
				Variable NumberOfColours=NumberByKey("NumberOfColours", UserDataStr, "=", ";", 1)
				Variable N=NumberByKey("N", UserDataStr, "=", ";", 1)
				Variable BorderSize=NumberByKey("BorderSize", UserDataStr, "=", ";", 1)
				Variable BoxWidth=NumberByKey("BoxWidth", UserDataStr, "=", ";", 1)
				Variable BoxHeight=NumberByKey("BoxHeight", UserDataStr, "=", ";", 1)
				Variable BoxSpacing=NumberByKey("BoxSpacing", UserDataStr, "=", ";", 1)
				Variable WaveNumber=NumberByKey("WaveNumber", UserDataStr, "=", ";", 1)
				String ViewSpectraWindow=StringByKey("ViewSpectraWindow", UserDataStr, "=", ";", 1)
				
				//	Calculates the position of the cursor relative to the colour boxes. A value between 0 and 1 means the cursors is somewhere between the left edge of box 0 and the right edge of box 1
				Variable BoxXDecimalNumber=(s.mouseLoc.h-BorderSize)/(BoxWidth+BoxSpacing)
				Variable BoxYDecimalNumber=(s.mouseLoc.v-BorderSize)/(BoxHeight+BoxSpacing)
				
				//	Finds the cursor position relative to the top left corner of the "nearest" colour box
				Variable RelativeBoxXPos=Mod(BoxXDecimalNumber, 1)*(BoxWidth+BoxSpacing)
				Variable RelativeBoxYPos=Mod(BoxYDecimalNumber, 1)*(BoxHeight+BoxSpacing)
				
				//	Checks if the cursor is placed on a box or in the space between two boxes
				Variable CursorXOnBox=(RelativeBoxXPos<BoxWidth && RelativeBoxXPos>=0)
				Variable CursorYOnBox=(RelativeBoxYPos<BoxHeight && RelativeBoxYPos>=0)
				
				//	Checks that both the x and y position of the cursor matches a box location
				if (CursorXOnBox && CursorYOnBox)
				
					//	Finds the column and row of the clicked colour box
					Variable Nx=Trunc(BoxXDecimalNumber)
					Variable Ny=Trunc(BoxYDecimalNumber)
				
					//	Finds the colour number corresponding to the box location
					Variable ColourNumber=1+Nx+N*Ny
					
					//	Checks that the calculated colour number is valid
					if (ColourNumber>0 && ColourNumber<=NumberOfColours)
					
						//	If a left mouse click was used the listbox colour is updated
						if (s.eventMod==1)
						
							//	Finds the listbox sel wave, which holds the colour information
							DFREF ProgramFolder=root:Programming
							Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
							
							//	Creates a list of colours already taken and removes the colour selection of the selected wave
							Duplicate/FREE/O/R=[][2][1] XPSViewWaveListSel, UnavailableColours
							UnavailableColours[WaveNumber]=0
						
							//	ColourNumber might be taken, if so, FullColourNumber represents the first available instance of that colour
							Variable FullColourNumber=ColourNumber-NumberOfColours
							
							//	Finds the first available instance of ColourNumber
							do
								FullColourNumber+=NumberOfColours
								FindValue /S=0 /I=(FullColourNumber) UnavailableColours
							while (V_Value!=-1)
							
							//	Updates the colour in the listbox
							XPSViewWaveListSel[WaveNumber][2][1]=FullColourNumber
						
							//	Kills the colour palette window
							KillWindow $s.winName
						
							//	Updates the View Spectra graph
							DFREF ActiveFolder=XPSViewGetDataFolder(ViewSpectraWindow, "DataFolderPopUp")
							XPSViewSpectraUpdateGraph(ViewSpectraWindow, ActiveFolder, $"")
							
						//	If a right mouse click was used. Allows the user to change the ColourWave values
						elseif (s.eventMod==16)
						
							//	The wave used to define the colours of the XPS View Spectra traces
							DFREF ProgramFolder=root:Programming
							Wave/W/U ColourWave=ProgramFolder:XPSViewColours

							//	Creates a string to control the menu formed
							DFREF TempFolder=root:Programming:Temp
							SVAR/Z MenuStr=TempFolder:XPSViewColourPaletteMenuStr
							if (SVAR_Exists(MenuStr)==0)
								String /G TempFolder:XPSViewColourPaletteMenuStr=""
								SVAR MenuStr=TempFolder:XPSViewColourPaletteMenuStr
							endif
							MenuStr="*COLORPOP*("+Num2iStr(ColourWave[ColourNumber][0])+","+Num2iStr(ColourWave[ColourNumber][1])+","+Num2iStr(ColourWave[ColourNumber][2])+")"
						
							//	Create a popup menu to change the colour of ColourNumber
							PopupContextualMenu /N "XPSViewColourPaletteMenu"
							
							//	Kills the global string used to create the menu
							KillStrings/Z MenuStr
							
							//	If a selection was made
							if (V_flag!=-1)
							
								//	Creates a one-dimensional wave of the red values in ColourWave for FindLevel to search through
								Duplicate/FREE/O/R=[][0] ColourWave, ColourWaveRed
								
								Variable ColourIndex=-1
								Variable ColourMatchFound=0
								Variable EndOfWave=0

								//	Checks that the selected colour doesn't already exist in ColourWave
								do
									FindValue /S=(ColourIndex+1) /I=(V_red) /Z ColourWaveRed
									ColourIndex=V_value
									
									//	Stops the search if a match has been found, or if the end of the wave or the first repeating region of the wave has been reached
									if (ColourIndex!=-1)
										ColourMatchFound=((ColourWave[ColourIndex][0]==V_red) && (ColourWave[ColourIndex][1]==V_green) && (ColourWave[ColourIndex][2]==V_blue))
									endif
									EndOfWave=((ColourIndex==-1) || (ColourIndex>=NumberOfColours))
								while ((EndOfWave==0) && (ColourMatchFound==0))
								
								//	If the new colour is unique the colour wave is updated
								if (ColourMatchFound==0)
								
									//	Changes the colour in the colour wave for every repeating occurence
									ColourWave[ColourNumber, *;NumberOfColours][0]=V_red
									ColourWave[ColourNumber, *;NumberOfColours][1]=V_green
									ColourWave[ColourNumber, *;NumberOfColours][2]=V_blue
								
									//	Updates the relevant colour box
									ValDisplay $("ColourBox"+Num2iStr(ColourNumber-1)), barBackColor=(ColourWave[ColourNumber][0], ColourWave[ColourNumber][1], ColourWave[ColourNumber][2]), win=$s.winName
							
									//	Updates the View Spectra graph
									DFREF ActiveFolder=XPSViewGetDataFolder(ViewSpectraWindow, "DataFolderPopUp")
									XPSViewSpectraUpdateGraph(ViewSpectraWindow, ActiveFolder, $"")

									//	Brings the colour palette window to the front again
									DoWindow/F $s.winName
								endif
							endif
						endif
					endif
				endif
			endif
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function/S XPSViewGetColourPaletteStr()
//	Returns the text in the global string MenuStr

	DFREF ProgramFolder=root:Programming
	DFREF TempFolder=root:Programming:Temp

	//	The global string will always exist when the menu is called, but it seems like Igoir tries to build the menu already at compile time, at which point the global string does not exist yet.
	SVAR/Z MenuStr=TempFolder:XPSViewColourPaletteMenuStr
	if (SVAR_Exists(MenuStr))
		Return MenuStr
	else
		Return "*COLORPOP*(0,0,0)"
	endif
end



Static Function XPSViewDoNothing()
//	Does absolutely nothing.... (see the menu below for an explanation)
end



Menu "XPSViewColourPaletteMenu", dynamic, contextualmenu
//	Creates a colour menu for the View Spectra colour palette
//	Would it be better to use non-dynamic and BuildMenu??
	
	//	"" should produce the same result as the DoNothing function, but, at least in Igor Pro 6.37, it produces an error instead. Hence, the need for the DoNothing function.
	EccentricXPS#XPSViewGetColourPaletteStr(), EccentricXPS#XPSViewDoNothing()
end



Static Function XPSViewEditWaves(B_Struct) : ButtonControl
//	Creates a table with the displayed spectra and fit waves. Useful for copying and pasting into other programs
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Creates the edit waves window, or bring it to the front if it already exists
		String ActiveWindow="XPSViewEditWavesWindow"
		DoWindow/F $ActiveWindow
		if (V_flag==0)
		
			Variable n=120/ScreenResolution
			DFREF ProgramFolder=root:Programming
			Wave EmptyWave=ProgramFolder:XPSViewEmptyWave

			//	Creates an empty table. Edit with no wave is supposed to create an empty table, but is bugged in Igor 6.37, hence the Edit Remove combination.
			Edit /K=1 /N=$ActiveWindow /W=(100, 40, 100+775*n, 40+150*n) EmptyWave.d
			RemoveFromTable /W=$ActiveWindow EmptyWave.d
			
			//	Adds all displayed spectra and fit waves to the Edit Waves table.
			XPSViewEditAddWaves()
		endif
	endif
	Return 0
end



Static Function XPSViewEditAddWaves()
//	Adds all displayed spectra and fit waves to the Edit Waves table.

	//	Only updates the Edit Waves window if it already exists
	String ActiveWindow="XPSViewEditWavesWindow"
	DoWindow $ActiveWindow
	if (V_flag!=0)
		
		//	The listbox waves, if they exist
		DFREF ProgramFolder=root:Programming
		Wave/Z/I/U SelWave=ProgramFolder:XPSViewWaveListSel
		Wave/Z/WAVE RefWave=ProgramFolder:XPSViewWaveListWaves
	
		//	If they exist the information about clicked waves and their colours are saved
		if (WaveExists(SelWave) && WaveExists(RefWave))
	
			//	Extracts the positions of the clicked waves in the wavelist
			Duplicate/FREE/O/R=[][0][0] SelWave TempSelWave
			Extract/FREE/INDX/O TempSelWave, IndexWave, (TempSelWave & 16)==16
			
			//	Calculates the number of clicked waves
			Variable NumberOfClickedWaves=NumPnts(IndexWave)
			
			//	Extracts the information that needs to be copied to the new wave list
			if (NumberOfClickedWaves>0)
				
				//	Checks the status of the Fitted checkbox
				ControlInfo /W=XPSViewSpectraWindow FittedCheckBox
				Variable DisplayFitted=V_Value
					
				//	Counts through all clicked waves in the list
				Variable i=0, ii=0
				String WaveNameStr=""
				for (i=0; i<NumberOfClickedWaves; i+=1)
					
					//	Appends the next measured spectrum in the list
					Wave ActiveWave=RefWave[IndexWave[i]]
					AppendToTable /W=$ActiveWindow ActiveWave.id
						
					//	Appends any displayed fit waves
					if (DisplayFitted)
						
						//	Finds the folder with the saved fits
						DFREF DataFolder=GetWavesDataFolderDFR(ActiveWave)
						DFREF SavedFitsFolder=DataFolder:SavedFits
						WaveNameStr=NameOfWave(ActiveWave)
						DFREF FitFolder=SavedFitsFolder:$WaveNameStr

						//	Checks if a folder with saved fits exists
						if ((DataFolderRefStatus(SavedFitsFolder)!=0) && (DataFolderRefStatus(FitFolder)!=0))

							//	Appends the basic fit waves to the table
							Wave/Z FitWave=FitFolder:$(WaveNameStr+"_fit")
							Wave/Z BackWave=FitFolder:$(WaveNameStr+"_bck")
							if (WaveExists(FitWave) && WaveExists(BackWave))
								AppendToTable /W=$ActiveWindow FitWave.id, BackWave.d
							endif

							//	Appends the peaks to the table
							Wave/Z PeakWave=FitFolder:$(WaveNameStr+"_pk"+Num2iStr(0))
							for (ii=1; WaveExists(PeakWave); ii+=1)
								AppendToTable /W=$ActiveWindow PeakWave.d
								Wave/Z PeakWave=FitFolder:$(WaveNameStr+"_pk"+Num2iStr(ii))
							endfor
						endif
					endif
				endfor
					
				//	Autoscales the columns in the table
				ModifyTable /W=$ActiveWindow autosize={0, 0, -1, 0.1, 1}
			endif
		endif
	endif
end



Static Function XPSViewEditRemoveWaves()
//	Removes all waves from the Edit Waves table

	//	Only updates the Edit Waves window if it already exists
	String ActiveWindow="XPSViewEditWavesWindow"
	DoWindow $ActiveWindow
	if (V_flag!=0)

		//	Finds the name of the first wave in the table
		String WaveNameStr=StringByKey("WAVE", TableInfo(ActiveWindow, 0), ":", ";", 1)
		Wave/Z ActiveWave=$WaveNameStr

		//	Removes the waves from the table one at a time
		if (WaveExists(ActiveWave))
			do
				RemoveFromTable /W=$ActiveWindow ActiveWave.id
				WaveNameStr=StringByKey("WAVE", TableInfo(ActiveWindow, 0), ":", ";", 1)
				Wave/Z ActiveWave=$WaveNameStr
			while(WaveExists(ActiveWave))
		endif
	endif
end



Static Function XPSViewDeletePrompt(B_Struct) : ButtonControl
//	Creates a message box asking the user to confirm the deletetion of the selected waves
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	The listbox wave containing information about which waves are selected
		DFREF ProgramFolder=root:Programming
		Wave XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel

		//	Checks if any waves have been selected
		Duplicate/FREE/O/R=[][0][0] XPSViewWaveListSel TempSelWave
		Extract/O/FREE/INDX TempSelWave, IndexWave, (TempSelWave & 16)==16
		
		//	Will only create the prompt panel if waves are actually selected
		if (NumPnts(IndexWave)>0)
			DoWindow /K XPSViewDeleteWindow
			NewPanel /K=1 /W=(370, 220, 660, 260) /N=XPSViewDeleteWindow as "Delete Selected Waves?"	
			Button CancelButton proc=EccentricXPS#CloseWindow, pos={35,10}, size={100, 20}, title="Cancel", win=XPSViewDeleteWindow
			Button DeleteButton proc=EccentricXPS#XPSViewDelete, pos={155,10}, size={100, 20}, title="Delete", win=XPSViewDeleteWindow
		endif
	endif
	Return 0
end



Static Function XPSViewDelete(B_Struct) : ButtonControl
//	Deletes the selected waves
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Kills the prompt window
		DoWindow /K $B_Struct.win

		//	Finds the displayed folder in the View Spectra window
		String ActiveWindow="XPSViewSpectraWindow"
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		
		//	Checks the status of the Fitted checkbox
		ControlInfo /W=$ActiveWindow FittedCheckBox
		Variable DisplayFitted=V_Value
		
		//	Finds the listbox waves
		DFREF ProgramFolder=root:Programming
		Wave/Z/I/U SelWave=ProgramFolder:XPSViewWaveListSel
		Wave/Z/WAVE RefWave=ProgramFolder:XPSViewWaveListWaves
	
		//	Extracts the clicked waves in the listbox
		Duplicate/FREE/O/R=[][0][0] SelWave TempSelWave
		Extract/FREE/O/INDX RefWave, IndexWave, (TempSelWave & 16)==16
			
		//	Calculates the number of clicked waves
		Variable n=NumPnts(IndexWave)
		
		//	Deletes the waves one at a time		
		Variable i=0
		for (i=0; i<n; i+=1)

			//	Deletes all fit waves associated with RefWave[IndexWave[i]]
			XPSViewListBoxDelete(RefWave[IndexWave[i]], ActiveFolder, DisplayFitted, 1, 1)
		endfor

		//	Updates both the View Spectra, View Images and Fit Assistant windows, if they exist
		XPSViewUpdateAllGraphs()
	endif
	Return 0
end	



Static Function XPSViewRefresh(B_Struct) : ButtonControl
//	Refreshes the wave list
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Finds the selected data folder
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value

		//	Updates the View Spectra graph
		XPSViewSpectraUpdateGraph(B_Struct.win, DataFolder, $"")
	endif
	Return 0
end



Static Function XPSViewClear(B_Struct) : ButtonControl
//	Unchecks all checkboxes and removes all waves from the graph
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Finds the wave that controls the formatting of the list box
		DFREF ProgramFolder=root:Programming
		Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel

		//	Changes all bits to not clicked
		XPSViewWaveListSel[][0][0]=XPSViewWaveListSel[p][0][0] & ~16
		
		//	Finds the selected data folder
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value

		//	Updates the graph
		XPSViewSpectraUpdateGraph(B_Struct.win, DataFolder, $"")
	endif
	Return 0
end



Static Function XPSViewAll(B_Struct) : ButtonControl
//	Checks all checkboxes and adds all waves to the graph
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Finds the wave that controls the formatting of the list box
		DFREF ProgramFolder=root:Programming
		Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
		
		//	Finds the number of spectra in the listbox
		Variable NumberOfTraces=DimSize(XPSViewWaveListSel, 0)

		//	Checks if the Do Not Display checkbox has been checked
		ControlInfo /W=$B_Struct.win DoNotDisplay
		Variable DoNotDisplay=V_Value

		//	Displays a warning before displaying more than 500 spectra
		if ((NumberOfTraces>500) && (DoNotDisplay==0))
		
			//	Creates the warning window
			DoWindow /K XPSViewWarningWindow
			NewPanel /K=1 /W=(370, 220, 660, 300) /N=XPSViewWarningWindow as "Warning!"
			DrawText /W=XPSViewWarningWindow 35,40, "You are attempting to display "+Num2iStr(NumberOfTraces)+" spectra.\rThis might take several seconds."
			Button CancelButton proc=EccentricXPS#CloseWindow, pos={35,50}, size={100, 20}, title="Cancel", win=XPSViewWarningWindow
			Button DisplayButton proc=EccentricXPS#XPSViewWarningDisplay, pos={155,50}, size={100, 20}, title="Display", userdata=B_Struct.win, win=XPSViewWarningWindow
		else

			//	Changes all bits to clicked
			XPSViewWaveListSel[][0][0]=XPSViewWaveListSel[p][0][0] | 16
		
			//	Finds the selected data folder
			ControlInfo /W=$B_Struct.win DataFolderPopUp
			DFREF DataFolder=$S_Value

			//	Updates the graph
			XPSViewSpectraUpdateGraph(B_Struct.win, DataFolder, $"")
		endif
	endif
	Return 0
end



Static Function XPSViewWarningDisplay(B_Struct) : ButtonControl
//	Displays all traces despite the warning
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Closes the warning window
		DoWindow/K $B_Struct.win

		//	Finds the wave that controls the formatting of the list box
		DFREF ProgramFolder=root:Programming
		Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel

		//	Changes all bits to clicked
		XPSViewWaveListSel[][0][0]=XPSViewWaveListSel[p][0][0] | 16
		
		//	Finds the selected data folder in the View Spectra window
		ControlInfo /W=$B_Struct.userdata DataFolderPopUp
		DFREF DataFolder=$S_Value

		//	Updates the graph
		XPSViewSpectraUpdateGraph(B_Struct.userdata, DataFolder, $"")
	endif
	Return 0
end



Static Function XPSViewClearSearch(B_Struct) : ButtonControl
//	Clears the search string
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Clears the search string
		SetVariable SetSearch, value=_STR:"", win=$B_Struct.win
		
		//	Finds the selected data folder
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value
		
		//	Updates the graph
		XPSViewSpectraUpdateGraph(B_Struct.win, DataFolder, $"")
	endif
	Return 0
end



Static Function XPSViewSetWaterfall(SV_Struct) : SetVariableControl
//	Sets the y-spacing used for the waterfall plot
STRUCT WMSetVariableAction &SV_Struct

	//	Updates the View Spectra graph if the value was changed
	if ((SV_Struct.eventCode==1) || (SV_Struct.eventCode==2))

		//	Ignores further calls to the set variable procedure until this one has finished
		SV_Struct.blockReentry=1
		
		//	Finds the selected data folder
		ControlInfo /W=$SV_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value
		
		//	Updates the graph
		XPSViewSpectraUpdateGraph(SV_Struct.win, DataFolder, $"")

	//	Mouse scroll wheel up
	elseif (SV_Struct.eventCode==4)

		//	Ignores further calls to the set variable procedure until this one has finished
		SV_Struct.blockReentry=1
	
		//	Increases the value by 20%, but no higher than 2.
		SetVariable $SV_Struct.ctrlName, value=_NUM:(Limit(SV_Struct.dval*1.2, 0.01, 2)), win=$SV_Struct.win
		
		//	Finds the selected data folder
		ControlInfo /W=$SV_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value

		//	Changing the value with SetVariable will NOT force another run of the function, hence the graph will have to be updated
		XPSViewSpectraUpdateGraph(SV_Struct.win, DataFolder, $"")
	
	//	Mouse scroll wheel down
	elseif (SV_Struct.eventCode==5)

		//	Ignores further calls to the set variable procedure until this one has finished
		SV_Struct.blockReentry=1
	
		//	Decreases the value by ~20%, but no lower than 0.01
		SetVariable $SV_Struct.ctrlName, value=_NUM:(Limit(SV_Struct.dval/1.2, 0.01, 2)), win=$SV_Struct.win

		//	Finds the selected data folder
		ControlInfo /W=$SV_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value

		//	Changing the value with SetVariable will NOT force another run of the function, hence the graph will have to be updated
		XPSViewSpectraUpdateGraph(SV_Struct.win, DataFolder, $"")
	endif
	Return 0
end



Static Function/S XPSViewDataFoldersList(StartFolder, ExcludeFolder, ExcludeFolders)
//	Returns a semicolon separated string list of the full paths to StartFolder and all it's subfolders, with the exception of the specific folder ExcludeFolder and all folders with the name ExcludeFolders and all their subfolders
DFREF StartFolder, ExcludeFolder
String ExcludeFolders
Variable n=0, i=0, a=0, b=1, c=0, MaxFolders=1000
String ListString=GetDataFolder(1, StartFolder)+";", FolderName=""
DFREF ParentFolder=StartFolder, ChildFolder=root:
	
	//	Creates a wave to hold the names of all existing folders. If more than 1000 folders exist the length of the wave will be increased by another 1000.
	Make/FREE/O/DF/N=(MaxFolders) ListOfFolders
	ListOfFolders[0]=StartFolder
	
	//	Counts through the list of folders
	for (a=0; a<b; a+=1)
	
		//	Finds the number of additional folders in the active folder
		ParentFolder=ListOfFolders[a]
		n=CountObjectsDFR(ParentFolder, 4)
		
		//	If more than 1000 folders exist the length of the wave will be increased by another 1000.
		if (b+n>MaxFolders)
			InsertPoints MaxFolders, 1000, ListOfFolders
			MaxFolders+=1000
		endif
		
		//	Adds the additional folders to the list of folders, excluding ExcludeFolder
		for (i=0; i<n; i+=1)
			FolderName=GetIndexedObjNameDFR(ParentFolder, 4, i)
			ChildFolder=ParentFolder:$FolderName
			if (DataFolderRefsEqual(ChildFolder, ExcludeFolder) || (CmpStr(FolderName, ExcludeFolders)==0))
				c+=1
			else
				ListOfFolders[b+i-c]=ChildFolder
				ListString+=GetDataFolder(1, ChildFolder)+";"
			endif
		endfor
		
		//	Increases the number of folders by the number of additonal folders
		b+=n-c
		
		//	Resets the exclude counter
		c=0
	endfor
	
	//	Returns the case-insensitive alphanumerically sorted list of data folders
	Return SortList(ListString, ";", 16)
end



Static Function XPSViewHelp(B_Struct) : ButtonControl
//	Opens the help file associated with the user data label
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
		DisplayHelpTopic /Z B_Struct.userData
	endif

	//	This is required for a button control, but not used for anything
	return 0
end



Function XPSViewPrintFunction(ActiveWindow)
//	Proto function defining the functions that can be called by the XPSViewPrint button control. Proto functions are not alllowed to be static
String ActiveWindow
end



Static Function XPSViewSpectraUpdateGraphDF(ActiveWindow, ActiveFolder, ActiveWave)
//	Sets the axis scaling and labels to match the type of spectrum displayed and updates the graph
String ActiveWindow
DFREF ActiveFolder
Wave/Z ActiveWave

	//	Sets the folder type popup menu based on the active data folder
	XPSViewFolderType(ActiveFolder, ActiveWindow)
	
	//	Finds the selected display type
	ControlInfo /W=$ActiveWindow DisplayTypePopUp
	String DisplayType=S_Value
	
	//	Sets the axis scaling and labels to match the type of spectrum displayed
	XPSViewSetAxis(ActiveWindow, DisplayType, 1, 0, 0)

	//	Refreshes the graph
	XPSViewSpectraUpdateGraph(ActiveWindow, ActiveFolder, ActiveWave)
end



Static Function XPSViewSpectraUpdateGraph(ActiveWindow, ActiveFolder, ActiveWave)
//	Adds new waves to the listbox and removes those that no longer exist, while keeping the selected waves and their colours unchanged. Assigns colours to clicked waves with no colours 
String ActiveWindow
DFREF ActiveFolder
Wave/Z ActiveWave
DFREF ProgramFolder=root:Programming, TempFolder=root:Programming:Temp

	//	Removes all waves from the Edit Waves table, if it exists
	XPSViewEditRemoveWaves()

	//	Creates a empty temporary waves. These will be used by preceding functions and need to exist
	Make/FREE/WAVE/O/N=0 ClickedWaves, ClickedFitWaves, ClickedBckWaves
	Make/FREE/B/U/O/N=0 ClickedFitExists
	Make/FREE/I/U/O/N=0 ClickedColours

	//	Checks if the data folder exists
	if (DataFolderRefStatus(ActiveFolder)==0)

		//	Creates empty listbox waves
		XPSViewCreateEmptyListBoxWaves("Data folder does not exist")
	else

		//	The listbox waves, if they exist
		Wave/Z/T ListWave=ProgramFolder:XPSViewWaveList
		Wave/Z/I/U SelWave=ProgramFolder:XPSViewWaveListSel
		Wave/Z/WAVE RefWave=ProgramFolder:XPSViewWaveListWaves
	
		//	If they exist the information about clicked waves and their colours are saved
		if (WaveExists(ListWave) && WaveExists(SelWave) && WaveExists(RefWave))
	
			//	Extracts the positions of the clicked waves in the wavelist
			Duplicate/FREE/O/R=[][0][0] SelWave TempSelWave
			Extract/FREE/INDX/O TempSelWave, IndexWave, (TempSelWave & 16)==16
			
			//	Calculates the number of clicked waves
			Variable NumberOfOldClickedWaves=NumPnts(IndexWave)
			
			//	Extracts the information that needs to be copied to the new wave list
			if (NumberOfOldClickedWaves>0)
			
				//	Creates a temporary wave to hold the wave references of the clicked waves
				Make/FREE/WAVE/O/N=(NumberOfOldClickedWaves) OldWaves=RefWave[IndexWave[p]]
				
				//	Creates a temporary wave to hold the colour information for the clicked waves
				Make/FREE/I/U/O/N=(NumberOfOldClickedWaves) OldColours=SelWave[IndexWave[p]][2][1]
					
				//	Sorts the wave references numerically
				Sort OldWaves, OldWaves, OldColours
			endif
		endif

		//	Finds the search string
		ControlInfo /W=$ActiveWindow SetSearch
		String SearchStr=S_Value

		//	Creates a list of wave references of all one-dimensional, numeric waves
		Wave/WAVE NewWaves=XPSViewListWaveRefs(ActiveFolder, 1, 1, SearchStr)
		
		//	Calculates the number of waves in the folder
		Variable NumberOfNewWaves=NumPnts(NewWaves)
		
		if (NumberOfNewWaves==0)
			
			//	Creates empty listbox waves if the data folder does not exist	
			XPSViewCreateEmptyListBoxWaves("No waves in folder")
		else

			//	Creates two temporary waves to hold the colours and states (checked/not checked) of the new waves. 32 is a not checked checkbox
			Make/FREE/I/U/O/N=(NumberOfNewWaves) NewColours, NewStates
			FastOP NewColours=0
			FastOP NewStates=32
		
			//	Only copies the states from the old waves, if any of them were actually clicked
			if (NumberOfOldClickedWaves>0)
			
				//	Sorts the list of wave references numerically
				Sort NewWaves, NewWaves
				
				//	Creates a new temporary wave with both new and old waves included
				Concatenate/NP/O  {NewWaves, OldWaves}, TempFolder:NewAndOldWaves
				Wave/WAVE NewAndOldWaves=TempFolder:NewAndOldWaves
				Sort NewAndOldWaves, NewAndOldWaves
				
				//	Counts though the old and new waves and updates the new wave with information from the old wave
				Variable New=0, Old=0, NewOld=0
				do
					if (WaveRefsEqual(NewWaves[New], OldWaves[Old]))

						//	Copies the colour information
						NewColours[New]=OldColours[Old]
						
						//	(Bit 4 and 5 set = checked checkbox)
						NewStates[New][0][0]=48

						//	Updates the counters
						New+=1
						Old+=1
						NewOld+=2

					elseif (WaveRefsEqual(NewWaves[New], NewAndOldWaves[NewOld]))

						//	Updates the counters
						New+=1
						NewOld+=1
					else
					
						//	Updates the counters
						Old+=1
						NewOld+=1
					endif
				while((New<NumberOfNewWaves) && (Old<NumberOfOldClickedWaves))
			endif
			
			//	Overwrites the old list of waves references with the new list
			Duplicate/O/WAVE NewWaves, ProgramFolder:XPSViewWaveListWaves/WAVE=RefWave
			
			//	Creates the list of wave names
			Make/T/O/N=(NumberOfNewWaves) ProgramFolder:XPSViewWaveList=NameOfWave(RefWave[p])
			Wave/T ListWave=ProgramFolder:XPSViewWaveList
			
			//	Sorts the waves alphanumerically based on the wave names. This is the slowest bit of the whole function
			Sort/A ListWave, ListWave, NewStates, NewColours, RefWave
			
			//	Adds the extra dimension to the wavelist wave needed for the list box
			Redimension /N=(-1 , 3) ListWave
				
			//	Creates a necessary wave holding the formatting of the listbox
			Make/I/U/O/N=(NumberOfNewWaves, 3, 2) ProgramFolder:XPSViewWaveListSel/WAVE=SelWave
			FastOP SelWave=0
					
			//	Makes the colours affect the background colour, change to foreColors to affect foreground (text) colours
			SetDimLabel 2,1, backColors, SelWave
	
			//	Copies the states and colours to the new sel wave
			ImageTransform /D=NewStates /G=0 /P=0 putCol SelWave	//	Equal to, but faster than, SelWave[][0][0]=NewStates[p]
			ImageTransform /D=NewColours /G=2 /P=1 putCol SelWave	//	Equal to, but faster than, SelWave[][2][1]=NewColours[p]

			//	Creates a temporary wave with information about whether a fit exists
			Make/FREE/B/U/O/N=(NumberOfNewWaves) FitExists
			FastOP FitExists=0
			
			//	Creates two empty waves to hold references to the fit waves if they exist
			Make/WAVE/FREE/O/N=(NumberOfNewWaves) FitWaves, BckWaves
			Wave/Z InvalidWave=$""
			FitWaves=InvalidWave
			FastOP BckWaves=FitWaves
			
			//	Only looks for fit waves, if the correct folder exists
			DFREF SavedFitsFolder=ActiveFolder:SavedFits
			if (DataFolderRefStatus(SavedFitsFolder)!=0)
			
				//	The data folders for the fits, if they exists
				Make/DF/FREE/O/N=(NumberOfNewWaves) FitFolders=SavedFitsFolder:$NameOfWave(RefWave[p])
				
				//	The fit waves, if they exist. The function call is necessary to access the data folders in the FitFolders wave. A simple FitFolders[p]:$(NameOfWave(RefWave[p])+"_fit") doesn't work
				MultiThread FitWaves=XPSViewFolderToWaveRef(FitFolders[p], NameOfWave(RefWave[p])+"_fit")
				MultiThread BckWaves=XPSViewFolderToWaveRef(FitFolders[p], NameOfWave(RefWave[p])+"_bck")
				
				//	Searches through all waves in the folder and determines if the corresponding fit waves from a saved fit exist
				MultiThread FitExists[]=((DataFolderRefStatus(FitFolders[p])!=0) && WaveExists(FitWaves[p]) && WaveExists(BckWaves[p]))
				
				//	Adds a + next to the wave name in the listbox when a fit exists
				ListWave[][1]=SelectString(FitExists[p] , "", "+")
			endif
			
			//	Extracts the positions of the clicked waves in the new wavelist
			Duplicate/FREE/O/R=[][0][0] SelWave TempSelWave
			Extract/FREE/INDX/O TempSelWave, IndexWave, (TempSelWave & 16)==16
			
			//	Calculates the number of clicked waves
			Variable NumberOfNewClickedWaves=NumPnts(IndexWave)
			
			if (NumberOfNewClickedWaves>0)

				//	Creates a temporary wave to hold the wave references of the clicked waves. This will be used by preceding functions
				Make/FREE/WAVE/O/N=(NumberOfNewClickedWaves) ClickedWaves=RefWave[IndexWave[p]], ClickedFitWaves=FitWaves[IndexWave[p]], ClickedBckWaves=BckWaves[IndexWave[p]]

				//	Creates a temporary wave to hold the information on whether a fit exists. This will be used by preceding functions
				Make/FREE/B/U/O/N=(NumberOfNewClickedWaves) ClickedFitExists=FitExists[IndexWave[p]]
				
				//	Checks the status of the Fit checkbox
				ControlInfo /W=$ActiveWindow FittedCheckBox
				Variable DisplayFitted=V_Value
		
				//	Forces subsequent functions not to plot or use the saved fits by setting fit exists equal to zero
				if (DisplayFitted==0)
					FastOP ClickedFitExists=0
				endif
				
				//	Creates a temporary wave to hold the colour information for the clicked waves. This will be used later by other functions in XPSViewSpectraUpdateGraph
				Make/FREE/I/U/O/N=(NumberOfNewClickedWaves) ClickedColours=SelWave[IndexWave[p]][2][1]
			
				//	Extracts the positions where the clicked waves have not been assigned a colour yet
				Extract/FREE/INDX/O ClickedColours, AssignColourIndex, ClickedColours==0
				
				//	Calculates the number of waves without a colour assigned to them
				Variable NumberOfColoursToAssign=NumPnts(AssignColourIndex)
				
				//	Creates a sorted list of used colours that can be used to find the lowest unused colour number
				Duplicate/O/FREE ClickedColours, SortedColours
				Sort SortedColours, SortedColours
				
				//	Counts through the waves without colours and assigns a colour to each
				Variable i=0, ii=0, a=0, b=0
				for (i=0; i<NumberOfColoursToAssign; i+=1)
				
					//	Finds the first unused colour number
					do
						ii+=1
					
						//	If ii is larger than the range of numbers in SortedOldColours, V_LevelX and thereby a and b becomes NaN, and there is no need to search the SortedOldColour anymore.
						if (NumType(a)==0)
							FindLevel/P/Q/R=[a, NumberOfNewClickedWaves-1] SortedColours, ii
							a=Trunc(V_LevelX)
							b=V_LevelX
						endif
					while (a==b)
					
					//	Assigns the new colour
					ClickedColours[AssignColourIndex[i]]=ii
					SelWave[IndexWave[AssignColourIndex[i]]][2][1]=ii
				endfor
			endif
		endif
	endif
	
	//	Calculates the offsets and muloffsets to use with the displayed waves
	XPSViewAssignOffsets(ActiveFolder, ClickedWaves, ClickedColours, ClickedFitExists, ClickedFitWaves, ClickedBckWaves)
	
	//	Adds all displayed spectra and fit waves to the Edit Waves table, if it exists
	XPSViewEditAddWaves()
end



ThreadSafe Static Function/WAVE XPSViewFolderToWaveRef(Folder, Name)
//	Returns a wave reference to the named wave in the given folder. Necessary for wave reference assignments using data folder waves
DFREF Folder
String Name
	Return Folder:$Name
end



Static Function XPSViewCreateEmptyListBoxWaves(Error)
//	Creates a set of empty listbox waves displaying an error message, typically "Data folder does not exist" or "No waves in folder"
String Error
DFREF ProgramFolder=root:Programming
	Make/T/O ProgramFolder:XPSViewWaveList={{Error}, {""}, {""}}
	Make/I/U/O/N=(1, 3, 2) ProgramFolder:XPSViewWaveListSel=0
	Make/WAVE/O ProgramFolder:XPSViewWaveListWaves={$""}
	
	//	Makes the colours affect the background colour, change to forecolors to affect foreground (text) colours
	Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
	SetDimLabel 2,1, backColors, XPSViewWaveListSel
end



Static Function XPSViewAssignOffsets(ActiveFolder, ClickedWaves, ClickedColours, ClickedFitExists, ClickedFitWaves, ClickedBckWaves)
//	Calculates the offsets and muloffsets to be used when displaying the waves
DFREF ActiveFolder
Wave/WAVE ClickedWaves, ClickedFitWaves, ClickedBckWaves
Wave/I/U ClickedColours
Wave/B/U ClickedFitExists
String ActiveWindow="XPSViewSpectraWindow", Name=""

	//	-----     Creates a list of all displayed waves in the View Spectra graph and a list of which of the clicked waves that are currently hidden in the graph     -----
	
	//	Creates a list of traces in the graph, excluding the empty place holder wave used to keep the graph up when all other traces have been removed.
	String VisibleTraces=RemoveFromList("XPSViewEmptyWave", TraceNameList(ActiveWindow,";", 5), ";")
	String AllTraces=RemoveFromList("XPSViewEmptyWave", TraceNameList(ActiveWindow,";",1), ";")
	String HiddenTraces= RemoveFromList(VisibleTraces, AllTraces)

	//	Converts the list of trace names into a reference wave sorted by reference number
	Variable NumberOfTraces=ItemsInList(AllTraces, ";")
	Variable i=0, a=0, b=0
	Make/WAVE/FREE/O/N=(NumberOfTraces) AllTraceWaves
	for (i=0; i<NumberOfTraces; i+=1)
		b=StrSearch(AllTraces, ";", a)
		AllTraceWaves[i]=TraceNameToWaveRef(ActiveWindow, AllTraces[a, b-1])
		a=b+1
	endfor
	Sort AllTraceWaves, AllTraceWaves

	//	Converts the list of hidden trace names into a reference wave sorted by reference number
	Variable NumberOfHiddenTraces=ItemsInList(HiddenTraces, ";")
	Make/WAVE/FREE/O/N=(NumberOfHiddenTraces) HiddenTraceWaves
	a=0
	for (i=0; i<NumberOfHiddenTraces; i+=1)
		b=StrSearch(HiddenTraces, ";", a)
		HiddenTraceWaves[i]=TraceNameToWaveRef(ActiveWindow, HiddenTraces[a, b-1])
		a=b+1
	endfor
	Sort HiddenTraceWaves, HiddenTraceWaves
	
	//	Creates a empty temporary waves. These will be used by preceding functions and need to exist
	Make/FREE/B/U/O/N=0 ClickedHidden, PlotTypes
	Make/FREE/O/N=0 Offsets, MulOffsets
	
	//	Checks if the Fancy Plot checkbox is clicked
	ControlInfo/W=$ActiveWindow FancyPlotCheckBox
	Variable FancyPlot=V_Value
	
	//	Checks if the selected data folder exists
	if (DataFolderRefStatus(ActiveFolder)!=0)

		//	Finds the number of clicked waves
		Variable NumberOfClicked=NumPnts(ClickedWaves)

		if (NumberOfClicked>0)

			//	Sorts all waves by reference number
			Sort ClickedWaves, ClickedWaves, ClickedColours, ClickedFitExists, ClickedFitWaves, ClickedBckWaves

			//	Creates a wave to hold information about whether a clicked wave is hidden in the graoh
			Duplicate/O/FREE ClickedFitExists, ClickedHidden
			FastOP ClickedHidden=0
			
			if (NumberOfHiddenTraces>0)

				//	Creates a new temporary wave with both clicked and hidden waves included. This is used to numerically compare wave references (larger or smaller than), which is otherwise not possible
				DFREF TempFolder=root:Programming:Temp
				Concatenate/NP/O  {ClickedWaves, HiddenTraceWaves}, TempFolder:XPSViewBothWaves
				Wave/WAVE BothWaves=TempFolder:XPSViewBothWaves
				Sort BothWaves, BothWaves
				
				//	Counts though the waves, finding the hidden waves among the clicked waves
				Variable Clicked=0, Hidden=0, Both=0
				do
					if (WaveRefsEqual(ClickedWaves[Clicked], HiddenTraceWaves[Hidden]))

						//	The wave is both hidden and clicked. The wave is marked as hidden and the counters are updated.
						ClickedHidden[Clicked]=1
						Clicked+=1
						Hidden+=1
						Both+=2

					elseif (WaveRefsEqual(ClickedWaves[Clicked], BothWaves[Both]))
		
						//	Updates the counters
						Clicked+=1
						Both+=1
					else
		
						//	Updates the counters
						Hidden+=1
						Both+=1
					endif
				while((Clicked<NumberOfClicked) && (Hidden<NumberOfHiddenTraces))
			endif

		
			//	-----     Calculates the offsets and muloffsets based on the status of the Offset and Scale buttons     -----


			//	Finds the status of the Offset, Scale and Waterfall buttons (1 = clicked, 0 = not clicked)
			Variable OffsetStatus=Str2Num(GetUserData(ActiveWindow, "OffsetButton", "Status"))
			Variable ScaleStatus=Str2Num(GetUserData(ActiveWindow, "ScaleButton", "Status"))
			Variable WaterfallStatus=Str2Num(GetUserData(ActiveWindow, "WaterfallButton", "Status"))
			
			//	Checks if the Do Not Display box is checked
			ControlInfo /W=$ActiveWindow DoNotDisplay
			Variable DoNotDisplay=V_Value
			
			//	Prevents the offsets from being calculated, when the Do Not Display box is checked
			if (DoNotDisplay==1)
				OffsetStatus=0
				ScaleStatus=0
				WaterfallStatus=0
			endif

			//	Sorts the clicked waves in order of descending colour numbers, if the waterfall button is active. This will determine the vertical order of the waves in the waterfall plot
			if (WaterfallStatus)
				Sort/R ClickedColours, ClickedColours, ClickedWaves, ClickedFitWaves, ClickedBckWaves, ClickedFitExists, ClickedHidden
			endif

			//	Creates temporary waves to calculate the maximum and minimum values for each wave and hold the offsets to be used in the graph
			Make/FREE/O/N=(NumberOfClicked) MinWave, MaxWave, Offsets, MulOffsets
			
			//	Calculates the minimum based on the background wave and the maximum based on the fit wave, if a fit exists
			MultiThread MinWave=(ClickedFitExists[p]) ? (WaveMin(ClickedBckWaves[p])) : (WaveMin(ClickedWaves[p]))
			MultiThread MaxWave=(ClickedFitExists[p]) ? (WaveMax(ClickedFitWaves[p])) : (WaveMax(ClickedWaves[p]))
	
			//	Calculates the offsets and muloffsets for the waves based on the Offset and Scale buttons. If Waterfall is active the offsets will be overwritten, but the muloffsets are kept
			Variable StatusBit=OffsetStatus*2^0+ScaleStatus*2^1
			switch (StatusBit)
	
				//	Neither Offset nor Scale is active. Scaling is removed from all waves
				case 0:
					FastOP Offsets=0
					FastOP MulOffsets=1
					break

				//	Only Offset is active. All waves are offset by their minimum value
				case 1:
					FastOP Offsets=-1*MinWave
					FastOP MulOffsets=1
					break

				//	Only Scale is active. All waves are divided by their maxium value
				case 2:
					FastOP Offsets=0
					MultiThread MulOffsets=(MaxWave[p]==0) ? (1) : (1/Abs(MaxWave[p]))
					break

				//	Offset and Scale are both active. All waves are scaled between zero and one
				case 3:
					MultiThread MulOffsets=(MaxWave[p]==MinWave[p]) ? (1) : (1/(MaxWave[p]-MinWave[p]))
					MultiThread Offsets=-MinWave*MulOffsets
					break
			endswitch
			
			//	Extracts the visible waves
			Extract/FREE/INDX/O ClickedHidden, ClickedVisibleIndex, (ClickedHidden==0)

			//	Calculates the number of visible waves
			Variable NumberOfClickedVisible=NumPnts(ClickedVisibleIndex)

			//	Calculates the offsets for the waterfall plot
			if ((WaterfallStatus) && (NumberOfClickedVisible>1))
		
				//	Finds the spacing to use for the waterfall plot as a fraction of the largest difference between minimum and maxium values in a wave
				Duplicate/O/FREE MaxWave, MinMaxRangeWave
				FastOP MinMaxRangeWave=MaxWave-MinWave
				MinMaxRangeWave=MulOffsets*MinMaxRangeWave
				Extract/FREE/O MinMaxRangeWave, VisibleMinMaxRangeWave, (ClickedHidden==0)
				ControlInfo /W=$ActiveWindow SetWaterfall
				Variable WaterfallSpacing=V_Value*WaveMax(VisibleMinMaxRangeWave)

				//	Makes the first wave start at zero
				Offsets[ClickedVisibleIndex[0]]=-MinWave[ClickedVisibleIndex[0]]*MulOffsets[ClickedVisibleIndex[0]]
			
				//	Calculates the remaining offsets to make the closest distance between two waves equal to WaterfallSpacing
				MultiThread ClickedVisibleIndex[1,*]=XPSViewWaterfallOffsets(ClickedVisibleIndex[p-1], ClickedVisibleIndex[p], ClickedWaves, Offsets, MulOffsets, ClickedFitExists, ClickedFitWaves, ClickedBckWaves)
				ClickedVisibleIndex[1,*]=XPSViewWaterfallAddSpacing(ClickedVisibleIndex[p-1], ClickedVisibleIndex[p], WaterfallSpacing, Offsets)
			endif
				

			//	-----     Increases the ClickedWaves with any existing fit waves     -----
			
			
			//	Creates a temporary wave which will determine how the wave is plotted on the graph (line, markers, etc..)
			Duplicate/O/FREE ClickedHidden, PlotTypes
			if (FancyPlot==1)
				FastOP PlotTypes=2
			else
				FastOP PlotTypes=0
			endif

			if (WaveMax(ClickedFitExists)>0)
		
				//	Increases the the size of the clicked waves, colours and offsets to make room for the fit waves, if they exists. If on average more than 20 fit waves exist per displayed wave, funny stuff will happen
				InsertPoints NumberOfClicked, NumberOfClicked*20, ClickedWaves, ClickedColours, Offsets, MulOffsets, ClickedHidden, PlotTypes
				
				//	The data folder containing the saved fits
				DFREF SavedFits=ActiveFolder:SavedFits
		
				//	Counts through the clicked waves and adds any existing fit waves to the end of the waves
				a=NumberOfClicked
				Variable ii=0
				for (i=0; i<NumberOfClicked; i+=1)
					if (ClickedFitExists[i])
			
						//	Adds the peaks, one at a time, until no more peaks exist
						Name=NameOfWave(ClickedWaves[i])
						DFREF FitFolder=SavedFits:$Name
						Wave/Z PeakWave=FitFolder:$(Name+"_pk0")
						for (ii=1; WaveExists(PeakWave); ii+=1)
							ClickedWaves[a+ii-1]=PeakWave
							Wave/Z PeakWave=FitFolder:$(Name+"_pk"+Num2Str(ii))
						endfor
						//	If the last peak that exists is pk3, the counter ii will end at 5
						
						//	Adds the two basic fit waves after the peaks
						ClickedWaves[a+ii-1]=ClickedBckWaves[i]
						ClickedWaves[a+ii]=ClickedFitWaves[i]
				
						//	Sets the colour of the fitwaves to 0 (grey) and the offsets and muloffsets to the same as the parent waves'
						ClickedColours[a, a+ii]=0
						Offsets[a, a+ii]=Offsets[i]
						MulOffsets[a, a+ii]=MulOffsets[i]
						ClickedHidden[a, a+ii]=ClickedHidden[i]
						
						if (FancyPlot!=0)
						
							//	Plots the measured data as grey markers
							PlotTypes[i]=1
							
							//	Plots the overall fit as a coloured fat line
							PlotTypes[a+ii]=2
							ClickedColours[a+ii]=ClickedColours[i]
							ClickedColours[i]=0
							
							//	Plots the peaks and background as thinner grey lines
							PlotTypes[a, a+ii-1]=3
						endif
				
						//	Updates the counter
						a+=ii+1
					endif
				endfor
	
				//	Removes the unused data points from the ends of the waves
				DeletePoints a, NumberOfClicked*20, ClickedWaves, ClickedColours, Offsets, MulOffsets, ClickedHidden, PlotTypes
			endif
		endif
	endif
	
	//	Appends and removes waves from the View Spectra graph
	XPSViewAppendRemoveWaves(ActiveFolder, AllTraceWaves, ClickedWaves, ClickedColours, Offsets, MulOffsets, ClickedHidden, PlotTypes, FancyPlot)
end



ThreadSafe Static Function XPSViewWaterfallOffsets(PrevNum, NextNum, ClickedWaves, Offsets, MulOffsets, ClickedFitExists, ClickedFitWaves, ClickedBckWaves)
//	Calculates the closest distance between two adjecent waves
Variable PrevNum, NextNum
Wave/WAVE ClickedWaves, ClickedFitWaves, ClickedBckWaves
Wave Offsets, MulOffsets
Wave/B/U ClickedFitExists

	//	Uses the background wave for NextWave, if a fit exists
	if (ClickedFitExists[NextNum])
		Wave NextWave=ClickedBckWaves[NextNum]
	else
		Wave NextWave=ClickedWaves[NextNum]
	endif
	
	//	Uses the combined fit wave for PrevWave, if a fit exists
	if (ClickedFitExists[PrevNum])
		Wave PrevWave=ClickedFitWaves[PrevNum]
	else
		Wave PrevWave=ClickedWaves[PrevNum]
	endif
	
	//	Finds the x-range of NextWave
	Variable Temp1=LeftX(NextWave)
	Variable Temp2=Pnt2X(NextWave, NumPnts(NextWave)-1)
	Variable NextXMin=Min(Temp1, Temp2)
	Variable NextXMax=Max(Temp1, Temp2)

	//	Finds the x-range of PrevWave
	Temp1=LeftX(PrevWave)
	Temp2=Pnt2X(PrevWave, NumPnts(PrevWave)-1)
	Variable PrevXMin=Min(Temp1, Temp2)
	Variable PrevXMax=Max(Temp1, Temp2)
	
	//	Finds the combined overlap
	Variable XMin=Max(NextXMin, PrevXMin)
	Variable XMax=Min(NextXMax, PrevXMax)

	//	Reduces the overlap to the nearest x values of the measured wave within the overlapping region
	Variable XOffset=LeftX(ClickedWaves[NextNum])
	Variable XStep=DeltaX(ClickedWaves[NextNum])
	if (XStep>0)
		XMin=XOffset+XStep*Ceil((XMin-XOffset)/XStep)
		XMax=XOffset+XStep*Floor((XMax-XOffset)/XStep)
	else
		XMin=XOffset+XStep*Floor((XMin-XOffset)/XStep)
		XMax=XOffset+XStep*Ceil((XMax-XOffset)/XStep)
	endif
	
	//	Tests if there is an overlap
	if (XMin<XMax)
	
		//	Creates three waves with the correct x-range. The measured wave determines the data point spacing
		Duplicate/O/FREE/R=(XMin, XMax) ClickedWaves[NextNum], OverlapNextWave, OverlapPrevWave, DiffWave
		
		//	Calculates the number of points in the overlap
		Variable NumberOfPoints=NumPnts(OverlapNextWave)
		
		//	Checks if there are any data points in the overlap
		if (NumberOfPoints>0)
		
			//	Interpolates the values of PrevWave at the x positions of NextWave
			OverlapNextWave=NextWave(x)
			OverlapPrevWave=PrevWave(x)

			//	Finds the closest distance between the two waves
			FastOP DiffWave=(MulOffsets[NextNum])*OverlapNextWave-(MulOffsets[PrevNum])*OverlapPrevWave

			//	Calculates the offset needed to make the two waves touch
			Offsets[NextNum]=-WaveMin(DiffWave)
		else
		
			//	Uses an offset of zero, if the waves have no overlap
			Offsets[NextNum]=0
		endif
	else	
	
		//	Uses an offset of zero, if the waves have no overlap
		Offsets[NextNum]=0
	endif
	
	//	Returns the index to NextWave, this is necessary since ClickedVisibleIndex is used to multithread the calculations
	Return NextNum
end



Static Function XPSViewWaterfallAddSpacing(PrevNum, NextNum, WaterfallSpacing, Offsets)
//	Adjusts the spacing between waves to make it equal to WaterfallSpacing
Variable PrevNum, NextNum, WaterfallSpacing
Wave Offsets
	Offsets[NextNum]+=Offsets[PrevNum]+WaterfallSpacing
	Return NextNum
end



Static Function XPSViewAppendRemoveWaves(ActiveFolder, AllTraceWaves, ClickedWaves, ClickedColours, Offsets, MulOffsets, ClickedHidden, PlotTypes, FancyPlot)
//	Appends and removes waves from the View Spectra graph
DFREF ActiveFolder
Wave/WAVE AllTraceWaves, ClickedWaves
Wave/I/U ClickedColours
Wave Offsets, MulOffsets
Wave/B/U ClickedHidden, PlotTypes
Variable FancyPlot
String ActiveWindow="XPSViewSpectraWindow", TraceName="", WaveNote="", TagStr="", WaveNoteTagStr="", TextStr="", AStr=""
Variable i=0, a=0, b=0, c=0, d=0, FVar=0, BVar=0, XVar=0, YVar=0, LVar=0, PosVar=0, NumberOfTags=0

	//	Finds the colour wave
	DFREF ProgramFolder=root:Programming
	Wave/W/U Colours=ProgramFolder:XPSViewColours
	
	//	Determines if the cursors are placed on the graph. If they are, their positions and trace names are saved
	Variable CursorAX=0, CursorBX=0
	Wave/Z CursorAWave=CsrWaveRef(A, ActiveWindow)
	if (WaveExists(CursorAWave))
		CursorAX=hcsr(A, ActiveWindow)
	endif
	Wave/Z CursorBWave=CsrWaveRef(B, ActiveWindow)
	if (WaveExists(CursorBWave))
		CursorBX=hcsr(B, ActiveWindow)
	endif
	
	//	Saves all tags present on the graph as wave notes
	XPSViewSaveTags(ActiveWindow)
	
	//	Removes all traces from the graph (except the empty placeholder wave)
	Variable NumberOfTraces=NumPnts(AllTraceWaves)
	for (i=0; i<NumberOfTraces; i+=1)
		TraceName=NameOfWave(AllTraceWaves[i])
		if (StrLen(TraceName)>0)
			RemoveFromGraph /W=$ActiveWindow $TraceName
		endif
	endfor
	
	//	Checks if the Do Not Display box is checked
	ControlInfo /W=$ActiveWindow DoNotDisplay
	if (V_Value==0)
	
	 	//	Finds the status of the hide tags button (1 = clicked, 0 = not clicked)
		Variable HideTags=Str2Num(GetUserData(ActiveWindow, "HideTagsButton", "Status"))

		//	Calculates the number of clicked waves
		Variable NumberOfClickedWaves=NumPnts(ClickedWaves)
	
		//	Appends the waves and tags to the graph and updates the colours and offsets of the waves. This is faster than using ReorderTraces
		if (FancyPlot==0)

			//	Appends the waves in reverse order, with the measured spectra plotted on top and in colour
			for (i=NumberOfClickedWaves-1; i>=0; i+=-1)
				XPSViewAppendWaves(ClickedWaves[i], Colours[ClickedColours[i]][0], Colours[ClickedColours[i]][1], Colours[ClickedColours[i]][2], ClickedHidden[i], Offsets[i], MulOffsets[i], HideTags, PlotTypes[i], ActiveWindow)
			endfor
		else
	
			//	Appends the waves in normal order, with the fitted spectra plotted on top and in colour
			for (i=0; i<NumberOfClickedWaves; i+=1)
				XPSViewAppendWaves(ClickedWaves[i], Colours[ClickedColours[i]][0], Colours[ClickedColours[i]][1], Colours[ClickedColours[i]][2], ClickedHidden[i], Offsets[i], MulOffsets[i], HideTags, PlotTypes[i], ActiveWindow)
			endfor
		endif

		//	Places the cursors back where they were
		if (WaveExists(CursorAWave))
			CheckDisplayed/W=$ActiveWindow CursorAWave
			if (V_flag==1)
				Cursor/W=$ActiveWindow A $NameOfWave(CursorAWave) CursorAX
			endif
		endif
		if (WaveExists(CursorBWave))
			CheckDisplayed/W=$ActiveWindow CursorBWave
			if (V_flag==1)
				Cursor/W=$ActiveWindow B $NameOfWave(CursorBWave) CursorBX
			endif
		endif
	endif

	//	Cleans up the temporary waves
	DFREF TempFolder=root:Programming:Temp
	KillWavesinFolder(TempFolder)
end



Static Function XPSViewRemoveAllTraces()
//	Saves all tags and removes all traces from the View Spectra graph (except the empty placeholder wave)
String ActiveWindow="XPSViewSpectraWindow", TraceName=""
Variable i=0, a=0, b=0

	//	Checks if the View Spectra window exists
	DoWindow $ActiveWindow
	if (V_Flag!=0)

		//	Saves all tags present on the View Spectra graph as wave notes
		XPSViewSaveTags(ActiveWindow)

		//	Creates a list of all traces in the graph, excluding the empty place holder wave used to keep the graph up when all other traces have been removed.
		String AllTraces=RemoveFromList("XPSViewEmptyWave", TraceNameList(ActiveWindow,";",1), ";")

		//	Converts the list of trace names into wave references
		Variable NumberOfTraces=ItemsInList(AllTraces, ";")
		Make/WAVE/FREE/O/N=(NumberOfTraces) AllTraceWaves
		for (i=0; i<NumberOfTraces; i+=1)
			b=StrSearch(AllTraces, ";", a)
			AllTraceWaves[i]=TraceNameToWaveRef(ActiveWindow, AllTraces[a, b-1])
			a=b+1
		endfor

		//	Removes all traces from the graph (except the empty placeholder wave)
		for (i=0; i<NumberOfTraces; i+=1)
			TraceName=NameOfWave(AllTraceWaves[i])
			if (StrLen(TraceName)>0)
				RemoveFromGraph /W=$ActiveWindow $TraceName
			endif
		endfor
	endif
end



Static Function XPSViewAppendWaves(ClickedWave, Red, Green, Blue, Hidden, Offset, MulOffset, HideTags, PlotType, ActiveWindow)
//	Appends the waves and tags to the graph and updates the colours and offsets of the waves
Wave/Z ClickedWave
Variable Red, Green, Blue, Hidden, Offset, MulOffset, HideTags, PlotType
String ActiveWindow

	if (WaveExists(ClickedWave))
		
		//	Appends the wave to the graph and updates the colours and offsets
		AppendToGraph /W=$ActiveWindow ClickedWave
		String TraceName=NameOfWave(ClickedWave)
		ModifyGraph /W=$ActiveWindow rgb($TraceName)=(Red, Green, Blue), hideTrace($TraceName)=(Hidden), offset($TraceName)={0, Offset}, muloffset($TraceName)={1, MulOffset}
		
		//	Updates the plot type
		switch(PlotType)
			case 1:
				ModifyGraph /W=$ActiveWindow mode($TraceName)=3, marker($TraceName)=19, msize($TraceName)=2, useMrkStrokeRGB($TraceName)=1
				break
			case 2:
				ModifyGraph /W=$ActiveWindow lsize($TraceName)=1.5
				break
			case 2:
				ModifyGraph /W=$ActiveWindow lsize($TraceName)=1
				break
		endswitch

		//	Checks if tags should be displayed
		if (HideTags==0)
			
			//	Checks if there are tags associated with the wave
			String WaveNote=Note(ClickedWave)
			String WaveNoteTagStr=StringByKey("TAG INFO", WaveNote, ":", ";", 1)
			
			if (StrLen(WaveNoteTagStr)>0)
					
				//	Finds the number of tags in the list
				Variable NumberOfTags=ItemsInList(WaveNoteTagStr, "/")
					
				//	Appends the tags one at a time
				Variable a=0
				for (a=0; a<NumberOfTags; a+=1)
						
					//	Finds the string containing the single tag
					String TagStr=StringFromList(a, WaveNoteTagStr, "/")
						
					//	Finds the parameters to use with the tag
					Variable FVar=Str2Num(StringByKey("F", TagStr, "=", ",", 1))
					Variable BVar=Str2Num(StringByKey("B", TagStr, "=", ",", 1))
					String AStr=StringByKey("A", TagStr, "=", ",", 1)
					Variable XVar=Str2Num(StringByKey("X", TagStr, "=", ",", 1))
					Variable YVar=Str2Num(StringByKey("Y", TagStr, "=", ",", 1))
					Variable LVar=Str2Num(StringByKey("L", TagStr, "=", ",", 1))
					Variable PosVar=Str2Num(StringByKey("POS", TagStr, "=", ",", 1))

					//	Finds the text and restores any special characters in the text
					String TextStr=StringByKey("TEXT", TagStr, "=", ",", 1)
					TextStr=ReplaceString("\COMMA", TextStr, ",", 1)
					TextStr=ReplaceString("\EQUAL", TextStr, "=", 1)
					TextStr=ReplaceString("\SCLN", TextStr, ";", 1)
					TextStr=ReplaceString("\CLN", TextStr, ":", 1)
					TextStr=ReplaceString("\SLASH", TextStr, "/", 1)
					
					//	Appends the tag
					Tag/W=$ActiveWindow/F=(FVar)/B=1/A=$AStr/X=(XVar)/Y=(YVar)/L=(LVar) $TraceName, PosVar, TextStr
				endfor
			
				//	Removes the tag string from the wave note
				WaveNote=ReplaceStringByKey("TAG INFO", WaveNote, "", ":", ";", 1)
				Note/K ClickedWave
				Note/NOCR ClickedWave, WaveNote
			endif
		endif
	endif
end



Static Function XPSViewSaveTags(ActiveWindow)
//	Saves all tags present on the graph as wave notes
String ActiveWindow
String TagStr="", WaveNoteTagStr="", TraceName="", TextStr="", ExistingWaveNoteTagStr="", WaveNote=""
Variable a=0, b=0, c=0, d=0, i=0

	//	Extracts the tags from the WinRecreation string
	String TagRecreationString=WinRecreation(ActiveWindow, 0)
	TagRecreationString=GrepList(TagRecreationString, "^\tTag/C/N=", 0, "\r")

	//	Counts the number of tags in the list
	Variable NumberOfTags=ItemsInList(TagRecreationString, "\r")
	
	//	Counts through the tags one at a time
	for (i=0; i<NumberOfTags; i+=1)
	
		//	Finds the next tag. Will have the format:	Tag/C/N=text2/F=0/B=1/A=RC/X=-2.00/Y=-2.00/L=1 '01_C1s', 280.462637637637556, "280.46 eV"
		b=StrSearch(TagRecreationString, "\r", a)
		TagStr=TagRecreationString[a, b-1]
		a=b+1
	
		//	Converts the first bit of the tag string into a string list: F=0,B=1,A=RC,X=-2.00,Y=-2.00,L=1,
		c=StrSearch(TagStr, "=", 0)
		c=StrSearch(TagStr, "/", c+1)
		d=StrSearch(TagStr, " ", c+1)
		WaveNoteTagStr=ReplaceString("/", TagStr[c+1, d-1], ",")+","
		
		//	Finds the wave the tag is placed on
		if (CmpStr(TagStr[d+1, d+1], "'")==0)
			c=StrSearch(TagStr, "'", d+2)
			TraceName=TagStr[d+2, c-1]
			c+=1
		else
			c=StrSearch(TagStr, ",", d+1)
			TraceName=TagStr[d+1, c-1]
		endif
		Wave ActiveWave=TraceNameToWaveRef(ActiveWindow, TraceName)
		
		//	Finds the x-position of the tag
		d=StrSearch(TagStr, ",", c+2)
		WaveNoteTagStr+="POS="+TagStr[c+2, d-1]+","
		
		//	Finds the text of the tag
		TextStr=TagStr[d+3, StrLen(TagStr)-2]
		
		//	Removes all characters used to separate the string list items		
		TextStr=ReplaceString(",", TextStr, "\COMMA", 1)
		TextStr=ReplaceString("=", TextStr, "\EQUAL", 1)
		TextStr=ReplaceString(";", TextStr, "\SCLN", 1)
		TextStr=ReplaceString(":", TextStr, "\CLN", 1)
		TextStr=ReplaceString("/", TextStr, "\SLASH", 1)
		
		//	Adds the text to the wave note string
		WaveNoteTagStr+="TEXT="+TextStr+","
		
		//	Finds the note associated with the displayed wave
		WaveNote=Note(ActiveWave)
		ExistingWaveNoteTagStr=StringByKey("TAG INFO", WaveNote, ":", ";", 1)
		
		//	Appends any existing tag notes to the note string
		if (StrLen(ExistingWaveNoteTagStr)>0)
			WaveNoteTagStr+="/"+ExistingWaveNoteTagStr
		endif
		
		//	Updates the wave note
		WaveNote=ReplaceStringByKey("TAG INFO", WaveNote, WaveNoteTagStr, ":", ";", 1)
		Note/K ActiveWave
		Note/NOCR ActiveWave, WaveNote
	endfor
end



Static Function XPSViewWaveListBox(LB_Struct) : ListboxControl
//	Updates the traces on the graph when a checkbox is checked or unchecked
STRUCT WMListboxAction & LB_Struct
Variable i=0, DisplayFitted=0, ValidFitExists=0
DFREF ProgramFolder=root:Programming
String NewWaveName="", OldWaveName=""

	//	If the state of a checkbox was changed (LB_Struct.eventCode==13).
	if (LB_Struct.eventCode==13)

		//	Ignores further calls to the listbox procedure until this one has finished
		LB_Struct.blockReentry=1
		
		//	Finds the selected data folder
		ControlInfo /W=$LB_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value
			
		//	Updates the graph window
		XPSViewSpectraUpdateGraph(LB_Struct.win, DataFolder, $"")
		
	//	If the editing of a checkbox finished (started with Rename or Duplicate from the right-click popup menu)
	elseif (LB_Struct.eventCode==7)
	
		//	Ignores further calls to the listbox procedure until this one has finished
		LB_Struct.blockReentry=1
		
		//	Recreates the listbox. This is necessary because of a bug in Igor Pro 6.36, where selecting Rename or Duplicate from the right-click menu bugs out the listbox
		XPSViewRebuildListBox(LB_Struct.win)
		
		//	Finds the new wave name chosen by the user
		NewWaveName=LB_Struct.listWave[LB_Struct.row][0]
		
		//	Finds the wave holding the wave references corresponding to the listbox rows
		Wave/Z/WAVE RefWave=ProgramFolder:XPSViewWaveListWaves
		
		//	Finds the wave and the name of the wave corresponding to the active row
		Wave/Z ActiveWave=RefWave[LB_Struct.row]
		OldWaveName=NameOfWave(ActiveWave)
		
		//	Checks if the suggested new wave name is different from the old name
		if (CmpStr(OldWaveName, NewWaveName)==0)
		
			//	Clears bit 1 of the selwave making the wave name no longer editable
			LB_Struct.selWave[LB_Struct.row][0][0]=(LB_Struct.selWave[LB_Struct.row][0][0] & ~ 2)
		else
		
			//	Checks if the suggested new name is null
			if (StrLen(NewWaveName)==0)
			
				//	If the new name is invalid, the old name is kept and nothing is changed
				LB_Struct.listWave[LB_Struct.row][0]=OldWaveName
				
				//	Clears bit 1 of the selwave making the wave name no longer editable
				LB_Struct.selWave[LB_Struct.row][0][0]=(LB_Struct.selWave[LB_Struct.row][0][0] & ~ 2)
			else
		
				//	Checks if the new name conflicts with an already existing wave
				DFREF DataFolder=GetWavesDataFolderDFR(ActiveWave)
				Wave/Z Conflict=DataFolder:$NewWaveName
				
				//	Checks the status of the Fitted checkbox
				ControlInfo /W=$LB_Struct.win FittedCheckBox
				DisplayFitted=V_Value
		
				if (WaveExists(Conflict))
			
					//	Prompts the user before overwriting the existing wave
					DoWindow /K XPSViewListBoxOverwriteWindow
					NewPanel /K=1 /W=(370, 220, 660, 260) /N=XPSViewListBoxOverwriteWindow as "Name Already Exists As a Wave!"
			
					//	Creates Overwrite and Cancel buttons
					Button CancelButton proc=EccentricXPS#XPSViewListBoxCancel, pos={35,10}, size={100, 20}, title="Cancel", userData=Num2iStr(LB_Struct.row), win=XPSViewListBoxOverwriteWindow
					Button OverwriteButton proc=EccentricXPS#XPSViewListBoxOverwrite, pos={155,10}, size={100, 20}, title="Overwrite", userData=Num2iStr(LB_Struct.row), win=XPSViewListBoxOverwriteWindow
				else
			
					//	Tries to create a wave with the new name, if it fails the new name is considered invalid.
					Make/O/N=0 DataFolder:$NewWaveName/WAVE=NewWave
					
					if (WaveExists(NewWave))
			
						//	Creates a duplicate of the selected wave and all associated waves
						XPSViewListBoxDuplicate(ActiveWave, DataFolder, NewWaveName)
					
						//	The listbox waves, if they exist
						Wave/Z/T ListWave=ProgramFolder:XPSViewWaveList
						Wave/Z/I/U SelWave=ProgramFolder:XPSViewWaveListSel
						Wave/Z/WAVE RefWave=ProgramFolder:XPSViewWaveListWaves
						
						//	Replaces the old wave with the new wave in the listbox wave	//	PROBLEM: Are the listbox waves required to be alphabetical??? It doesn't seem so
						ListWave[LB_Struct.row][0][0]=NewWaveName
						RefWave[LB_Struct.row]=DataFolder:$NewWaveName
						
						//	Saves the tags from the old wave in the wavenote of the new wave
						XPSViewListBoxSaveTags(ActiveWave, NewWave, LB_Struct.win, DisplayFitted)
						
						//	Deletes all waves associated with ActiveWave
						XPSViewListBoxDelete(ActiveWave, DataFolder, DisplayFitted, LB_Struct.win, 0)
					
						//	Replaces OldWaveName with NewWaveName in all groups with the correct data folder	
						XPSViewListBoxReplaceInGroups(DataFolder, OldWaveName, NewWaveName)
					
						//	Clears bit 1 of the selwave making the wave name no longer editable. Technically this is not needed, since the update function will clear it too
						LB_Struct.selWave[LB_Struct.row][0][0]=(LB_Struct.selWave[LB_Struct.row][0][0] & ~ 2)
						
						//	Updates both the View Spectra, View Images and Fit Assistant windows, if they exist
						XPSViewUpdateAllGraphs()
					else
						
						//	Displays an error message
						DoWindow /K XPSViewListBoxOverwriteWindow
						NewPanel /K=1 /W=(370, 220, 540, 260) /N=XPSViewListBoxOverwriteWindow as "Invalid Name!"
			
						//	Kills the dialog window and returns to editing mode in the View Spectra listbox
						Button OKButton proc=EccentricXPS#XPSViewListBoxCancel, pos={35,10}, size={100, 20}, title="OK", userData=Num2iStr(LB_Struct.row), win=XPSViewListBoxOverwriteWindow
					endif
				endif
			endif
		endif
		
	//	If a mouse button is pressed down (LB_Struct.eventCode==1)
	elseif (LB_Struct.eventCode==1)
	
		switch(LB_Struct.eventMod)

			//	If a shift (bit 1) left (bit 0) mouse click was used (2^1+2^0=3). Will hide the wave in the graph
			case 3:
		
				//	Ignores further calls to the listbox procedure until this one has finished
				LB_Struct.blockReentry=1
				
				//	If the checkbox is clicked (otherwise the spectra will not be on the graph)
				if ((LB_Struct.selWave[LB_Struct.row][0][0] & 16)==16)

					//	Finds the selected data folder
					ControlInfo /W=$LB_Struct.win DataFolderPopUp
					DFREF DataFolder=$S_Value
			
					//	Checks if the selected data folder exists
					if (DataFolderRefStatus(DataFolder)!=0)
		
						//	Finds the selected wave, if it exists
						Wave/Z ActiveWave=DataFolder:$LB_Struct.listWave[LB_Struct.row][0]
				
						//	Checks if the selected wave exists
						if (WaveExists(ActiveWave))
				
							//	Checks if a saved fit exists
							DFREF SavedFitsFolder=DataFolder:SavedFits
							DFREF FitFolder=SavedFitsFolder:$LB_Struct.listWave[LB_Struct.row][0]
							Wave/Z FitWave=FitFolder:$(LB_Struct.listWave[LB_Struct.row][0]+"_fit")
							Wave/Z BackWave=FitFolder:$(LB_Struct.listWave[LB_Struct.row][0]+"_bck")
							ValidFitExists=((DataFolderRefStatus(SavedFitsFolder)!=0) && (DataFolderRefStatus(FitFolder)!=0) && (WaveExists(FitWave)) && (WaveExists(BackWave)))

							//	Checks the status of the Fitted checkbox
							ControlInfo /W=$LB_Struct.win FittedCheckBox
							DisplayFitted=V_Value
							
							//	Determines if the trace is hidden or visible
							if (Str2Num(StringByKey("hideTrace(x)", TraceInfo(LB_Struct.win, LB_Struct.listWave[LB_Struct.row][0], 0), "=", ";"))==0)
							
								//	Hides the selected trace and all corresponding fit waves
								XPSViewListBoxHide(LB_Struct.win, LB_Struct.listWave[LB_Struct.row][0], DataFolder, FitFolder, (DisplayFitted && ValidFitExists), 1)
							else
								//	Unhides the selected trace and all corresponding fit waves
								XPSViewListBoxHide(LB_Struct.win, LB_Struct.listWave[LB_Struct.row][0], DataFolder, FitFolder, (DisplayFitted && ValidFitExists), 0)
							endif
						endif
					endif
				endif
				break

			//	If a shift (bit 1) right (bit 4) mouse click was used (2^1+2^4=18). Will open the wave in Fit Assistant
			case 18:
		
				//	Ignores further calls to the listbox procedure until this one has finished
				LB_Struct.blockReentry=1
			
				//	Finds the selected data folder
				ControlInfo /W=$LB_Struct.win DataFolderPopUp
				DFREF DataFolder=$S_Value
			
				//	Checks if the selected data folder exists
				if (DataFolderRefStatus(DataFolder)!=0)
		
					//	Finds the selected wave, if it exists
					Wave/Z ActiveWave=DataFolder:$LB_Struct.listWave[LB_Struct.row][0]
				
					//	Checks if the selected wave exists
					if (WaveExists(ActiveWave))

						//	Changes the selection in the Fit Assistant window without killing the window
						XPSFitAssChangeSelection(GetDataFolder(1, DataFolder), LB_Struct.listWave[LB_Struct.row][0], 1)

						//	Overwrites the coefficient waves with the saved coefficient waves, if they exist.
						XPSFitAssUseSavedCoefWaves("XPSFitAssistantWindow", DataFolder, ActiveWave)
					endif
				endif
				break
		
			//	If a right (bit 4) mouse click was used (2^4=16). Opens a small popup menu or a small colour palette
			case 16:

				//	Ignores further calls to the listbox procedure until this one has finished
				LB_Struct.blockReentry=1

				//	If the colour box of a selected wave is right clicked. Will open a small colour palette window to allow the colour to be changed
				if ((LB_Struct.col==2) && (LB_Struct.selWave[LB_Struct.row][LB_Struct.col][1]!=0))
				
					//	PROBLEM: This is sort of implemented now, but makes little sense unless the position in the waterfall plot and the colours can be separated (with an extra index in the SelWave??)
				
					//	Finds the absolute position of the cursor in pixels
					Variable n=ScreenResolution/72
					GetWindow $LB_Struct.win wsize	// points wsizeDC should give pixels, but it doesn't work
					Variable XPosition=LB_Struct.mouseLoc.h+V_left*n+50
					Variable YPosition=LB_Struct.mouseLoc.v+V_top*n
					
					//	Creates a popup window at the location of the cursor to select a trace colour from
					XPSViewCreateColourPalette(XPosition, YPosition, LB_Struct.row, LB_Struct.win)

				//	Opens a small popup menu allowing different actions to be performed with the selected wave
				else

					//	Creates the string with the right-click popup menu options
					String MenuStr=""

					String WaveNameStr=""
					String ListOfGroups=""
					Variable NumberOfGroups=0
		
					//	Finds the selected data folder
					ControlInfo /W=$LB_Struct.win DataFolderPopUp
					DFREF DataFolder=$S_Value
			
					//	Checks if the selected data folder exists
					if (DataFolderRefStatus(DataFolder)!=0)
		
						//	Finds the selected wave, if it exists
						WaveNameStr=LB_Struct.listWave[LB_Struct.row][0]
						Wave/Z ActiveWave=DataFolder:$WaveNameStr
				
						//	Checks if the selected wave exists
						if (WaveExists(ActiveWave))

							//	Creates the string with the right-click popup menu options
							MenuStr+="Fit Spectrum '"+WaveNameStr+"';"
							
							//	Finds the groups ActiveWave belongs to, if any
							ListOfGroups=XPSViewBelongsToGroups(ActiveWave)
							NumberOfGroups=ItemsInList(ListOfGroups, ";")
							
							//	Adds the groups to the popup menu string
							for (i=0; i<NumberOfGroups; i+=1)
								MenuStr+="Fit Group '"+StringFromList(i, ListOfGroups, ";")+"';"
							endfor
			
							//	Finds the image folder and wave
							DFREF ParentFolder=$(GetDataFolder(1, DataFolder)+":")
							DFREF ImageFolder=ParentFolder:Images
							Wave/Z ImageWave=ImageFolder:$WaveNameStr
				
							//	Checks if an image exists, and adds the option to view that image if it does
							if ((DataFolderRefStatus(ParentFolder)!=0) && (DataFolderRefStatus(ImageFolder)!=0) && (WaveExists(ImageWave)))

								//	Creates the string with the right-click popup menu options
								MenuStr+="View Image of Spectrum '"+WaveNameStr+"';"
							
								//	Adds the groups to the popup menu string
								for (i=0; i<NumberOfGroups; i+=1)
									MenuStr+="View Images of Group '"+StringFromList(i, ListOfGroups, ";")+"';"
								endfor
							endif

							//	Checks if Martin Schmid's XPS Tools exist. If it does the option to open the spectrum in XPST is added
							if ((Exists("ProcGlobal#LaunchCursorPanel")==6) && (Exists("ProcGlobal#CallSTNewProjectPanel")==6))
								MenuStr+="Open Spectrum in XPS Tools;"
							endif

							//	If the checkbox is clicked (otherwise the spectra will not be on the graph)
							if ((LB_Struct.selWave[LB_Struct.row][0][0] & 16)==16)

								//	Determines if the trace is hidden or visible
								if (Str2Num(StringByKey("hideTrace(x)", TraceInfo(LB_Struct.win, WaveNameStr, 0), "=", ";"))==0)
									MenuStr+="Hide in Graph;"
								else
									MenuStr+="Unhide in Graph;"
								endif
							endif
			
							//	Checks if a saved fit exists
							DFREF SavedFitsFolder=DataFolder:SavedFits
							DFREF FitFolder=SavedFitsFolder:$WaveNameStr
							Wave/Z FitWave=FitFolder:$(WaveNameStr+"_fit")
							Wave/Z BackWave=FitFolder:$(WaveNameStr+"_bck")
							ValidFitExists=((DataFolderRefStatus(SavedFitsFolder)!=0) && (DataFolderRefStatus(FitFolder)!=0) && (WaveExists(FitWave)) && (WaveExists(BackWave)))
						
							//	Adds Clear Saved Fiit as an option, if a fit exists
							if (ValidFitExists)
								MenuStr+="Clear Saved Fit;"
							endif
			
							//	Adds a number of miscellaneous options
							MenuStr+="Rename Wave;Duplicate Wave;Delete Wave;Copy Full Name;Copy Partial Name;Copy Folder Name;"
			
							//	Extracts the region info from the note attached to the wave
							String RegionInfo=StringByKey("REGION INFO", Note(ActiveWave), ":", ";", 1)
				
							//	Adds Display Region Info as an option, if the wave constains region information
							if (StrLen(RegionInfo)>0)
								MenuStr+="Display Region Info;"
							endif
						else
					
							//	Has the popup menu return an error message
							MenuStr="Wave does not exist;"
						endif
					else
			
						//	Has the popup menu return an error message
						MenuStr="Datafolder does not exist;"
					endif
	
					//	Opens a small popup menu at the position of the cursor
					PopupContextualMenu MenuStr
					
					//	Checks the status of the Fitted checkbox
					ControlInfo /W=$LB_Struct.win FittedCheckBox
					DisplayFitted=V_Value
			
					//	Does an action based on the popup menu selection
					//	Because "Open in Fit Assistant" involves the wave and folder names those selection are dealt with under the default case
					StrSwitch(S_selection)
			
						case "Open in View Images":
				
							//	Displays the corresponding image in View Images
							XPSViewImagesChangeSelection(GetDataFolder(1, ImageFolder), WaveNameStr, 1)
							break
					
						case "Open Spectrum in XPS Tools":
		
							//	Opens the spectrum in Martin Schmid's XPS Tools
							XPSViewCallXPSTools(DataFolder, ActiveWave)
							break
					
						case "Hide in Graph":
						
							//	Hides the selected trace and all corresponding fit waves
							XPSViewListBoxHide(LB_Struct.win, WaveNameStr, DataFolder, FitFolder, (DisplayFitted && ValidFitExists), 1)
							break
					
						case "Unhide in Graph":

							//	Unhides the selected trace and all corresponding fit waves
							XPSViewListBoxHide(LB_Struct.win, WaveNameStr, DataFolder, FitFolder, (DisplayFitted && ValidFitExists), 0)
							break
					
						case "Clear Saved Fit":
							
							//	Removes the fit waves from the graph, if they are displayed
							if (DisplayFitted)

								//	Removes the basic fit waves
								RemoveFromGraph/Z /W=$LB_Struct.win $(WaveNameStr+"_fit"), $(WaveNameStr+"_bck")
							
								//	Removes the individual peaks of the fit
								i=0
								Wave/Z PeakWave=FitFolder:$(WaveNameStr+"_pk"+Num2iStr(i))
								for (i=1; WaveExists(PeakWave); i+=1)
									RemoveFromGraph/Z /W=$LB_Struct.win $NameOfWave(PeakWave)
									Wave/Z PeakWave=FitFolder:$(WaveNameStr+"_pk"+Num2iStr(i))
								endfor
							endif
							
							//	Deletes the data folder holding the saved fit
							if (DataFolderRefStatus(FitFolder)!=0)
								KillDataFolder/Z FitFolder
							endif
							
							//	Updates the graph window
							XPSViewSpectraUpdateGraph(LB_Struct.win, DataFolder, $"")
							break
					
						case "Rename Wave":
						
							//	Sets bit 1 of the selwave making the wave name editable
							LB_Struct.selWave[LB_Struct.row][0][0]=(LB_Struct.selWave[LB_Struct.row][0][0]  |  2)
							
							//	Enables editing mode for the selected wave name in the listbox. LB_Struct.eventCode==7 (see above) indicates that the editing has finished
							Listbox $LB_Struct.ctrlName, SetEditCell={LB_Struct.row, 0, StrLen(WaveNameStr), StrLen(WaveNameStr)}, win=$LB_Struct.win
							break
					
						case "Duplicate Wave":
				
							//	Picks a temporary name for the new wave
							i=2
							do
								NewWaveName=WaveNameStr+"_"+Num2iStr(i)
								Wave/Z NewWave=DataFolder:$NewWaveName
								i+=1
							while (WaveExists(NewWave))
					
							//	Creates a duplicate of the selected wave and all associated waves
							XPSViewListBoxDuplicate(ActiveWave, DataFolder, NewWaveName)
							Wave NewWave=DataFolder:$NewWaveName
					
							//	Saves the tags from the old wave in the wavenote of the new wave
							XPSViewListBoxSaveTags(ActiveWave, NewWave, LB_Struct.win, DisplayFitted)
					
							//	The listbox waves, if they exist
							Wave/Z/T ListWave=ProgramFolder:XPSViewWaveList
							Wave/Z/I/U SelWave=ProgramFolder:XPSViewWaveListSel
							Wave/Z/WAVE RefWave=ProgramFolder:XPSViewWaveListWaves
					
							//	Makes room for the duplicate in the listbox waves
							InsertPoints /M=0 LB_Struct.row+1, 1, ListWave, SelWave, RefWave
					
							//	Inserts the new values in the listbox waves	//	PROBLEM: The wave names in the listbox are no longer sorted alphanumerically, is that a problem??
							ListWave[LB_Struct.row+1][0]=NewWaveName
							ListWave[LB_Struct.row+1][1]=ListWave[LB_Struct.row][1]
							SelWave[LB_Struct.row+1][0][0]=34	//	Editable (bit 1) unchecked checkbox (bit 5)
							RefWave[LB_Struct.row+1]=DataFolder:$NewWaveName
					
							//	Enables editing mode for the duplicate wave in the listbox. LB_Struct.eventCode==7 (see above) indicates that the editing has finished.
							Listbox $LB_Struct.ctrlName, SetEditCell={LB_Struct.row+1, 0, StrLen(NewWaveName), StrLen(NewWaveName)}, win=$LB_Struct.win
							break

						case "Delete Wave":
				
							//	Sets bit 1 of the selwave making the wave name editable. This is only done to mark the wave name with a framed box
							LB_Struct.selWave[LB_Struct.row][0][0]=(LB_Struct.selWave[LB_Struct.row][0][0]  |  2)
						
							//	Rebuilds the list box with all available waves. This prevents a bug where the cursor doesn't update properly when moving over the delete prompt window
						 	XPSViewRebuildListBox(LB_Struct.win)
				
							//	Prompts the user before deleting the wave
							DoWindow /K XPSViewListBoxDeleteWindow
							NewPanel /K=1 /W=(370, 220, 660, 260) /N=XPSViewListBoxDeleteWindow as "Delete Selected Wave?"
			
							//	Creates Overwrite and Cancel buttons
							Button CancelButton proc=EccentricXPS#XPSViewListBoxDeleteCancel, pos={35,10}, size={100, 20}, title="Cancel", userData=Num2iStr(LB_Struct.row), win=XPSViewListBoxDeleteWindow
							Button DeleteButton proc=EccentricXPS#XPSViewListBoxDeleteButton, pos={155,10}, size={100, 20}, title="Delete", userData=Num2iStr(LB_Struct.row), win=XPSViewListBoxDeleteWindow
							break
					
						case "Copy Full Name":

							//	Copies the full path to ActiveWave to the clipboard
							PutScrapText GetWavesDataFolder(ActiveWave, 2)
							break
					
						case "Copy Partial Name":

							//	Copies only the name of ActiveWave to the clipboard
							PutScrapText PossiblyQuoteName(NameOfWave(ActiveWave))
							break
					
						case "Copy Folder Name":
				
							//	Copies the full path of the selected data folder to the clipboard
							PutScrapText GetDataFolder(1, DataFolder)
							break
					
						case "Display Region Info":
				
							//	Restores colons and semicolons to the region info
							RegionInfo=ReplaceString("\CLN", RegionInfo, ":")
							RegionInfo=ReplaceString("\SCLN", RegionInfo, ";")

							//	Prints the region info labelled with the name of the wave
							Print("\r\r["+WaveNameStr+"]\r"+RegionInfo+"\r["+WaveNameStr+"]\r\r")
							break
							
						Default:
						
							String ActiveGroup=""
						
							//	Opens the selected spectrum in Fit Assistant
							if (CmpStr(S_selection, "Fit Spectrum '"+WaveNameStr+"'")==0)
							
								//	Changes the selection in the Fit Assistant window without killing the window
								XPSFitAssChangeSelection(GetDataFolder(1, DataFolder), WaveNameStr, 1)

								//	Overwrites the coefficient waves with the saved coefficient waves, if they exist.
								XPSFitAssUseSavedCoefWaves("XPSFitAssistantWindow", DataFolder, ActiveWave)
							else
							
								//	Counts through the groups ActiveWave belongs to
								for (i=0; i<NumberOfGroups; i+=1)
								
									//	Finds the name of the next group
									ActiveGroup=StringFromList(i, ListOfGroups, ";")
								
									//	Opens the selected group in Fit Assistant
									if (CmpStr(S_selection, "Fit Group '"+ActiveGroup+"'")==0)

										//	Changes the selection in the Fit Assistant window without killing the window
										XPSFitAssChangeSelection(ActiveGroup, WaveNameStr, 1)

										//	Overwrites the coefficient waves with the saved coefficient waves, if they exist.
										XPSFitAssUseSavedCoefWaves("XPSFitAssistantWindow", DataFolder, ActiveWave)
									endif
								endfor
							endif

							//	Opens the selected spectrum in View Images
							if (CmpStr(S_selection, "View Image of Spectrum '"+WaveNameStr+"'")==0)

								//	Displays the corresponding image in View Images
								XPSViewImagesChangeSelection(GetDataFolder(1, ImageFolder), WaveNameStr, 1)
							else
							
								//	Counts through the groups ActiveWave belongs to
								for (i=0; i<NumberOfGroups; i+=1)
								
									//	Finds the name of the next group
									ActiveGroup=StringFromList(i, ListOfGroups, ";")
								
									//	Opens the selected group in View Images
									if (CmpStr(S_selection, "View Images of Group '"+ActiveGroup+"'")==0)

										//	Displays the corresponding image in View Images
										XPSViewImagesChangeSelection(ActiveGroup, WaveNameStr, 1)
									endif
								endfor
							endif
					endswitch
				endif
			break
		endswitch
	endif

	//	This is required for a listbox control, but not used for anything
	return 0
end



Static Function XPSViewListBoxSaveTags(OldWave, NewWave, ActiveWindow, DisplayFitted)
//	Saves the tags from the old wave in the wavenote of the new wave
Wave/Z OldWave, NewWave
String ActiveWindow
Variable DisplayFitted
String TagStr="", WaveNoteTagStr="", TraceName="", TextStr="", ExistingWaveNoteTagStr="", WaveNote="", Extension=""
Variable a=0, b=0, c=0, d=0, e=0, i=0

	if (WaveExists(OldWave) && WaveExists(NewWave))

		//	Finds the data folder of the new wave
		DFREF DataFolder=GetWavesDataFolderDFR(NewWave)
		
		if (DataFolderRefStatus(DataFolder)!=0)

			//	The names of the waves
			String OldWaveName=NameOfWave(OldWave)
			String NewWaveName=NameOfWave(NewWave)
	
			//	Extracts all tags from the WinRecreation string
			String TagRecreationString=WinRecreation(ActiveWindow, 0)
			TagRecreationString=GrepList(TagRecreationString, "^\tTag/C/N=", 0, "\r")

			//	Counts the number of tags in the list
			Variable NumberOfTags=ItemsInList(TagRecreationString, "\r")
	
			//	Counts through the tags one at a time
			for (i=0; i<NumberOfTags; i+=1)
	
				//	Finds the next tag. Will have the format:	Tag/C/N=text2/F=0/B=1/A=RC/X=-2.00/Y=-2.00/L=1 '01_C1s', 280.462637637637556, "280.46 eV"
				b=StrSearch(TagRecreationString, "\r", a)
				TagStr=TagRecreationString[a, b-1]
				a=b+1
	
				//	Converts the first bit of the tag string into a string list: F=0,B=1,A=RC,X=-2.00,Y=-2.00,L=1,
				c=StrSearch(TagStr, "=", 0)
				c=StrSearch(TagStr, "/", c+1)
				d=StrSearch(TagStr, " ", c+1)
				WaveNoteTagStr=ReplaceString("/", TagStr[c+1, d-1], ",")+","
		
				//	Finds the trace the tag is placed on
				if (CmpStr(TagStr[d+1, d+1], "'")==0)
					c=StrSearch(TagStr, "'", d+2)
					TraceName=TagStr[d+2, c-1]
					c+=1
				else
					c=StrSearch(TagStr, ",", d+1)
					TraceName=TagStr[d+1, c-1]
				endif
		
				//	Finds the wave corresponding to the trace
				Wave/Z TraceWave=TraceNameToWaveRef(ActiveWindow, TraceName)
		
				//	If the trace is OldWave the tag is transferred to NewWave
				If (WaveRefsEqual(TraceWave, OldWave)==0)
					Wave/Z WaveNoteWave=NewWave

				//	If the trace is associated with OldWave the tag is transferred to corresponding fit wave of NewWave
				elseif (DisplayFitted)

					//	Tests if TraceName is of the type OldWaveName+"_bck", OldWaveName+"_fit" or OldWaveName+"_pk"+Number
					SplitString /E="^"+OldWaveName+"_(fit|bck|pk[0-9]*)$" TraceName, Extension
			
					if (V_Flag==1)
			
						//	 The tag is transferred to corresponding fit wave of NewWave
						Wave/Z WaveNoteWave=DataFolder:$(NewWaveName+"_"+Extension)
					else
			
						//	The tag is not transferred
						Wave/Z WaveNoteWave=$""
					endif
				else
			
					//	The tag is not transferred
					Wave/Z WaveNoteWave=$""
				endif
		
				//	If WaveNoteWave does not exist, the tag is not transferred
				if (WaveExists(WaveNoteWave))
		
					//	Finds the x-position of the tag
					d=StrSearch(TagStr, ",", c+2)
					WaveNoteTagStr+="POS="+TagStr[c+2, d-1]+","
		
					//	Finds the text of the tag
					TextStr=TagStr[d+3, StrLen(TagStr)-2]
		
					//	Removes all characters used to separate the string list items		
					TextStr=ReplaceString(",", TextStr, "\COMMA", 1)
					TextStr=ReplaceString("=", TextStr, "\EQUAL", 1)
					TextStr=ReplaceString(";", TextStr, "\SCLN", 1)
					TextStr=ReplaceString(":", TextStr, "\CLN", 1)
					TextStr=ReplaceString("/", TextStr, "\SLASH", 1)
		
					//	Adds the text to the wave note string
					WaveNoteTagStr+="TEXT="+TextStr+","
		
					//	Finds the note associated with the displayed wave
					WaveNote=Note(WaveNoteWave)
					ExistingWaveNoteTagStr=StringByKey("TAG INFO", WaveNote, ":", ";", 1)
		
					//	Appends any existing tag notes to the note string
					if (StrLen(ExistingWaveNoteTagStr)>0)
						WaveNoteTagStr+="/"+ExistingWaveNoteTagStr
					endif
		
					//	Updates the wave note
					WaveNote=ReplaceStringByKey("TAG INFO", WaveNote, WaveNoteTagStr, ":", ";", 1)
					Note/K WaveNoteWave
					Note/NOCR WaveNoteWave, WaveNote
				endif
			endfor
		endif
	endif
end



Static Function XPSViewListBoxReplaceInGroups(DataFolder, OldWaveName, NewWaveName)
//	Replaces OldWaveName with NewWaveName in all groups with the correct data folder
DFREF DataFolder
String OldWaveName, NewWaveName
Variable a=0, b=-1, i=0, n=0
Variable aa=0, bb=-1, ii=0, nn=0
String GroupFolder="", GroupStr="", GroupWaveName=""

	//	Finds the global string with information about the saved groups. A global string is used to prevent the information from being lost if the View Spectra window is closed
	//	The format of the string is: Group1=Folder1;Wave1:Colour1;Wave2:Colour2;Wave3:Colour3;,Group2=Folder2;Wave4:Colour4;Wave5:Colour5;Wave6:Colour6;, etc....
	DFREF ProgramFolder=root:Programming
	SVAR/Z SavedGroups=ProgramFolder:XPSViewSavedGroups
	if (SVAR_Exists(SavedGroups)==0)
		String /G ProgramFolder:XPSViewSavedGroups=""
		SVAR SavedGroups=ProgramFolder:XPSViewSavedGroups
	endif

	if (StrLen(SavedGroups)>0)
	
		//	Used to check if the group data folder is the same as the data folder in question	
		String DataFolderStr=GetDataFolder(1, DataFolder)
		
		//	Removes all commas, equal signs and semicolons from the folder name
		DataFolderStr=ReplaceString(",", DataFolderStr, "\COMMA", 1)
		DataFolderStr=ReplaceString("=", DataFolderStr, "\EQUAL", 1)
		DataFolderStr=ReplaceString(";", DataFolderStr, "\SCLN", 1)
		
		//	Removes all commas, equal signs, semicolons and colons from the old and new wave names
		OldWaveName=ReplaceString(",", OldWaveName, "\COMMA", 1)
		OldWaveName=ReplaceString("=", OldWaveName, "\EQUAL", 1)
		OldWaveName=ReplaceString(";", OldWaveName, "\SCLN", 1)
		OldWaveName=ReplaceString(":", OldWaveName, "\CLN", 1)

		NewWaveName=ReplaceString(",", NewWaveName, "\COMMA", 1)
		NewWaveName=ReplaceString("=", NewWaveName, "\EQUAL", 1)
		NewWaveName=ReplaceString(";", NewWaveName, "\SCLN", 1)
		NewWaveName=ReplaceString(":", NewWaveName, "\CLN", 1)

		//	Finds the number of saved groups
		n=ItemsInList(SavedGroups, ",")
		
		//	Searches through all groups
		for (i=0; i<n; i+=1)

			//	Finds the data folder of the next group
			a=StrSearch(SavedGroups, "=", b+1)
			b=StrSearch(SavedGroups, ";", a+1)
			GroupFolder=SavedGroups[a+1, b-1]
				
			//	Compares the data folder of the group with the active data folder
			if (CmpStr(GroupFolder, DataFolderStr)==0)
				
				//	Finds the names of the waves in group i
				a=b
				b=StrSearch(SavedGroups, ",", a+1)
				GroupStr=SavedGroups[a+1, b-1]
			
				//	Finds the number of waves in group i
				nn=ItemsInList(GroupStr, ";")
				
				//	Sets the starting position
				b=a-1
				
				//	Searches through all waves in the group
				for (ii=0; ii<nn; ii+=1)

					//	Finds the name of the next wave in the group
					a=StrSearch(SavedGroups, ";", b+1)
					b=StrSearch(SavedGroups, ":", a+1)
					GroupWaveName=SavedGroups[a+1, b-1]
					
					//	Compares the wave name with the wave name to replace
					if (CmpStr(GroupWaveName, OldWaveName)==0)
					
						if (StrLen(NewWaveName)>0)
						
							//	Replaces OldWaveName in group i with NewWaveName
							SavedGroups=SavedGroups[0, a]+NewWaveName+SavedGroups[b, StrLen(SavedGroups)-1]
						else
						
							//	Removes OldWave completely from group i
							b=StrSearch(SavedGroups, ";", b+1)
							SavedGroups=SavedGroups[0, a]+SavedGroups[b+1, StrLen(SavedGroups)-1]
						endif
						
						//	Corrects for the string potentially being shorter now
						b=a+1
					
						//	Terminates the loop
						ii=nn
					endif
				endfor
			endif
			
			//	Finds the data folder of the next group			
			a=StrSearch(SavedGroups, "=", b+1)
			b=StrSearch(SavedGroups, ";", a+1)
		endfor
	endif
end



Static Function XPSViewListBoxDeleteButton(B_Struct) : ButtonControl
//	Deletes the selected wave in the View Spectra listbox
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Kills the dialog window
		DoWindow/K $B_Struct.win
		
		//	Finds the selected data folder
		String ActiveWindow="XPSViewSpectraWindow"
		ControlInfo /W=$ActiveWindow DataFolderPopUp
		DFREF DataFolder=$S_Value
			
		//	Checks if the selected data folder exists
		if (DataFolderRefStatus(DataFolder)!=0)

			//	Finds the row that was being edited in the View Spectra listbox
			Variable ActiveRow=Str2Num(B_Struct.userData)
		
			//	The listbox waves
			DFREF ProgramFolder=root:Programming
			Wave/Z/T ListWave=ProgramFolder:XPSViewWaveList
			Wave/Z/WAVE RefWave=ProgramFolder:XPSViewWaveListWaves
			
			//	Finds the wave to delete
			Wave/Z ActiveWave=RefWave[ActiveRow]
			
			if (WaveExists(ActiveWave))

				//	Checks the status of the Fitted checkbox
				ControlInfo /W=$ActiveWindow FittedCheckBox
				Variable DisplayFitted=V_Value

				//	Removes the selected wave from all groups
				XPSViewListBoxReplaceInGroups(DataFolder, ListWave[ActiveRow], "")
					
				//	Deletes all waves associated with ActiveWave
				XPSViewListBoxDelete(ActiveWave, DataFolder, DisplayFitted, 0, 1)

				//	Updates both the View Spectra, View Images and Fit Assistant windows, if they exist
				XPSViewUpdateAllGraphs()
			endif
		endif
	endif
	
	//	Not used for anything
	Return 0
end



Static Function XPSViewListBoxDeleteCancel(B_Struct) : ButtonControl
//	Cancels the deletion of the selected wave in the View Spectra listbox
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Kills the dialog window
		DoWindow/K $B_Struct.win
		
		//	Finds the row that was being edited in the View Spectra listbox
		Variable ActiveRow=Str2Num(B_Struct.userData)
		
		//	Finds the length text that was being edited
		DFREF ProgramFolder=root:Programming
		Wave/Z/I/U SelWave=ProgramFolder:XPSViewWaveListSel
		
		//	Clears bit 1 of the selwave making the wave name no longer editable
		SelWave[ActiveRow][0][0]=(SelWave[ActiveRow][0][0] & ~ 2)
	endif
	
	//	Not used for anything
	Return 0
end



Static Function XPSViewListBoxCancel(B_Struct) : ButtonControl
//	Kills the dialog window and returns to editing mode in the View Spectra listbox
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Kills the dialog window
		DoWindow/K $B_Struct.win
		
		//	Finds the row that was being edited in the View Spectra listbox
		Variable ActiveRow=Str2Num(B_Struct.userData)
		
		//	Finds the length text that was being edited
		DFREF ProgramFolder=root:Programming
		Wave/Z/T ListWave=ProgramFolder:XPSViewWaveList
		Variable WaveNameLength=StrLen(ListWave[ActiveRow][0])
		
		//	Returns to editing mode
		DoWindow XPSViewSpectraWindow
		if (V_flag)
			Listbox WaveListBox, SetEditCell={ActiveRow, 0, WaveNameLength, WaveNameLength}, win=XPSViewSpectraWindow
		endif
	endif
	
	//	Not used for anything
	Return 0
end



Static Function XPSViewListBoxOverwrite(B_Struct) : ButtonControl
//	Overwrites the existing wave with the renamed wave
STRUCT WMButtonAction & B_Struct
String ActiveWindow="XPSViewSpectraWindow"

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Kills the dialog window
		DoWindow/K $B_Struct.win
		
		//	Finds the selected data folder
		ControlInfo /W=$ActiveWindow DataFolderPopUp
		DFREF DataFolder=$S_Value
			
		//	Checks if the selected data folder exists
		if (DataFolderRefStatus(DataFolder)!=0)
		
			//	The listbox waves
			DFREF ProgramFolder=root:Programming
			Wave/Z/T ListWave=ProgramFolder:XPSViewWaveList
			Wave/Z/I/U SelWave=ProgramFolder:XPSViewWaveListSel
			Wave/Z/WAVE RefWave=ProgramFolder:XPSViewWaveListWaves
			
			//	Finds the row that was being edited in the View Spectra listbox
			Variable ActiveRow=Str2Num(B_Struct.userData)
		
			//	Finds the wave to overwrite
			String NewWaveName=ListWave[ActiveRow][0][0]
			Wave/Z WaveToOverwrite=DataFolder:$NewWaveName
			
			//	Finds the wave to rename
			Wave/Z OldWave=RefWave[ActiveRow]

			if (WaveExists(WaveToOverwrite) && WaveExists(OldWave))

				//	Checks the status of the Fitted checkbox
				ControlInfo /W=$ActiveWindow FittedCheckBox
				Variable DisplayFitted=V_Value

				//	Deletes all waves associated with the wave to overwrite
				XPSViewListBoxDelete(WaveToOverwrite, DataFolder, DisplayFitted, 0, 1)
				
				//	Creates a duplicate of the wave to rename and all associated waves
				XPSViewListBoxDuplicate(OldWave, DataFolder, NewWaveName)
				
				//	Replaces the old wave with the new wave in the listbox waves
				RefWave[ActiveRow]=DataFolder:$NewWaveName
				
				//	Removes WaveToOverwrite from all groups
				XPSViewListBoxReplaceInGroups(DataFolder, NewWaveName, "")

				//	Replaces OldWaveName with NewWaveName in all groups with the correct data folder	
				XPSViewListBoxReplaceInGroups(DataFolder, NameOfWave(OldWave), NewWaveName)

				//	Deletes all waves associated with the old wave
				XPSViewListBoxDelete(OldWave, DataFolder, DisplayFitted, 0, 1)

				//	Clears bit 1 of the selwave making the wave name no longer editable
				SelWave[ActiveRow][0][0]=(SelWave[ActiveRow][0][0] & ~ 2)
						
				//	Updates both the View Spectra, View Images and Fit Assistant windows, if they exist
				XPSViewUpdateAllGraphs()
			endif
		endif
	endif
	
	//	Not used for anything
	Return 0
end



Static Function XPSViewListBoxHide(ActiveWindow, WaveNameStr, DataFolder, FitFolder, FitDisplayed, Hide)
//	Shows or hides the selected trace and all corresponding fit waves
String ActiveWindow, WaveNameStr
DFREF DataFolder, FitFolder
Variable FitDisplayed, Hide

	//	Shows or hides the selected trace
	ModifyGraph /W=$ActiveWindow hideTrace($WaveNameStr)=Hide
							
	//	Hides the displayed fit waves as well		
	if (FitDisplayed)

		//	Shows or hides the basic fit waves
		ModifyGraph /W=$ActiveWindow hideTrace($(WaveNameStr+"_fit"))=Hide, hideTrace($(WaveNameStr+"_bck"))=Hide
							
		//	Shows or hides the individual peaks of the fit
		Variable i=0
		Wave/Z PeakWave=FitFolder:$(WaveNameStr+"_pk"+Num2iStr(i))
		for (i=1; WaveExists(PeakWave); i+=1)
			ModifyGraph /W=$ActiveWindow hideTrace($NameOfWave(PeakWave))=Hide
			Wave/Z PeakWave=FitFolder:$(WaveNameStr+"_pk"+Num2iStr(i))
		endfor
	endif
							
	//	Updates the graph window
	XPSViewSpectraUpdateGraph(ActiveWindow, DataFolder, $"")
end



Static Function XPSViewListBoxDelete(ActiveWave, DataFolder, DisplayFitted, FitOnly, ViewSpectraActive)
//	Deletes all waves associated with ActiveWave. Setting FitOnly=1 means only the fit waves will be affected. The affected graphs will have to be updated afterwards
Wave ActiveWave
DFREF DataFolder
Variable DisplayFitted, FitOnly, ViewSpectraActive

	//	Checks if the both wave and datafolder is valid
	if (WaveExists(ActiveWave) && (DataFolderRefStatus(DataFolder)!=0))
	
		//	Removes all waves from the Edit Waves table, if it exists
		XPSViewEditRemoveWaves()
	
		String ViewSpectraWindow="XPSViewSpectraWindow"

		//	Removes the wave from the View Spectra graph and deletes the wave
		String WaveNameStr=NameOfWave(ActiveWave)
		if (FitOnly==0)
			if (ViewSpectraActive)
				RemoveFromGraph/Z /W=$ViewSpectraWindow $WaveNameStr
			endif
			KillWaves/Z ActiveWave
		endif

		//	Finds the folder with the saved fits
		DFREF SavedFitsFolder=DataFolder:SavedFits
		DFREF FitFolder=SavedFitsFolder:$WaveNameStr

		//	Checks if a folder with saved fits exists
		if ((DataFolderRefStatus(SavedFitsFolder)!=0) && (DataFolderRefStatus(FitFolder)!=0))

			//	Only removes the fit waves from the View Spectra graph, if the Fitted checkbox is checked
			if (DisplayFitted)

				//	Removes the basic fit waves from the graph
				Wave/Z FitWave=FitFolder:$(WaveNameStr+"_fit")
				Wave/Z BackWave=FitFolder:$(WaveNameStr+"_bck")
				RemoveFromGraph/Z /W=$ViewSpectraWindow $NameOfWave(FitWave), $NameOfWave(BackWave)

				//	Removes the individual peaks of the fit
				Variable i=0
				Wave/Z PeakWave=FitFolder:$(WaveNameStr+"_pk"+Num2iStr(i))
				for (i=1; WaveExists(PeakWave); i+=1)
					RemoveFromGraph/Z /W=$ViewSpectraWindow $NameOfWave(PeakWave)
					Wave/Z PeakWave=FitFolder:$(WaveNameStr+"_pk"+Num2iStr(i))
				endfor
			endif

			//	Deletes the folder with the saved fits and all waves in it
			KillDataFolder/Z FitFolder
		endif
	
		if (FitOnly==0)
			//	Finds the image folder and wave
			DFREF ParentFolder=$(GetDataFolder(1, DataFolder)+":")
			DFREF ImageFolder=ParentFolder:Images
				
			//	Checks if the image folder exists
			if ((DataFolderRefStatus(ParentFolder)!=0) && (DataFolderRefStatus(ImageFolder)!=0))

				//	Deletes the image wave if it exists
				Wave/Z ImageWave=ImageFolder:$WaveNameStr
				if (WaveExists(ImageWave))
					KillWaves/Z ImageWave
				endif
			endif
		endif
	endif
end



Static Function XPSViewUpdateAllGraphs()
//	Updates both the View Spectra, View Images and Fit Assistant windows

	//	Checks if the View Spectra window is active
	String ViewSpectraActiveWindow="XPSViewSpectraWindow"
	DoWindow $ViewSpectraActiveWindow
		
	if (V_Flag)

		//	Finds the displayed folder in the View Spectra window
		DFREF ViewSpectraActiveFolder=XPSViewGetDataFolder(ViewSpectraActiveWindow, "DataFolderPopUp")

		//	Refreshes the View Spectra graph
		XPSViewSpectraUpdateGraph(ViewSpectraActiveWindow, ViewSpectraActiveFolder, $"")
	endif

		
	//	Checks if the View Image window is active
	String ViewImageActiveWindow="XPSViewImageWindow"
	DoWindow $ViewImageActiveWindow
		
	if (V_Flag)

		//	Finds the displayed wave in the View Image window
		DFREF ViewImageActiveFolder=XPSViewGetDataFolder(ViewImageActiveWindow, "DataFolderPopUp")
		Wave/Z ViewImageDisplayedWave=XPSViewGetDisplayedWave(ViewImageActiveWindow, ViewImageActiveFolder, "DisplayedWavePopUp")

		//	Refreshes the View Image graph
		XPSViewImageUpdateGraph(ViewImageActiveWindow, ViewImageActiveFolder, ViewImageDisplayedWave)
	endif


	//	Checks if the Fit Assistant window is active
	String FitAssActiveWindow="XPSFitAssistantWindow"
	DoWindow $FitAssActiveWindow
		
	if (V_Flag)

		//	Finds the displayed wave in the Fit Assistant window
		DFREF FitAssActiveFolder=XPSViewGetDataFolder(FitAssActiveWindow, "DataFolderPopUp")
		Wave/Z FitAssDisplayedWave=XPSViewGetDisplayedWave(FitAssActiveWindow, FitAssActiveFolder, "DisplayedWavePopUp")

		//	Refreshes the Fit Assistant graph
		XPSFitAssUpdateGraph(FitAssActiveWindow, FitAssActiveFolder, FitAssDisplayedWave)
	endif
end



Static Function XPSViewListBoxDuplicate(ActiveWave, DataFolder, NewWaveName)
//	Duplicates Active Wave and all associated waves
Wave ActiveWave
DFREF DataFolder
String NewWaveName

	//	Checks if the both wave and datafolder is valid
	if (WaveExists(ActiveWave) && (DataFolderRefStatus(DataFolder)!=0))		//	Test new wave name....??

		//	Duplicates active wave
		String OldWaveName=NameOfWave(ActiveWave)
		Duplicate/O ActiveWave, DataFolder:$NewWaveName

		//	Finds the folder with the saved fits
		DFREF SavedFitsFolder=DataFolder:SavedFits
		DFREF OldFitFolder=SavedFitsFolder:$OldWaveName

		//	Checks if a folder with saved fits exists
		if ((DataFolderRefStatus(SavedFitsFolder)!=0) && (DataFolderRefStatus(OldFitFolder)!=0))
		
			//	Creates the new fit folder
			NewDataFolder/O SavedFitsFolder:$NewWaveName
			DFREF NewFitFolder=SavedFitsFolder:$NewWaveName

			//	The basic fit waves
			Wave/Z FitWave=OldFitFolder:$(OldWaveName+"_fit")
			Wave/Z BackWave=OldFitFolder:$(OldWaveName+"_bck")

			//	The fit coefficient waves
			Wave/Z ListWave=OldFitFolder:$(OldWaveName+"_lst")
			Wave/Z SelWave=OldFitFolder:$(OldWaveName+"_sel")
			
			//	Duplicates the basic fit waves and fit coefficient waves to the new folder
			if (WaveExists(FitWave))
				Duplicate/O FitWave, NewFitFolder:$(NewWaveName+"_fit")
			endif
			if (WaveExists(BackWave))
				Duplicate/O BackWave, NewFitFolder:$(NewWaveName+"_bck")
			endif
			if (WaveExists(ListWave))
				Duplicate/O ListWave, NewFitFolder:$(NewWaveName+"_lst")
			endif
			if (WaveExists(SelWave))
				Duplicate/O SelWave, NewFitFolder:$(NewWaveName+"_sel")
			endif

			//	Duplicates the individual peaks of the fit
			Variable i=0
			Wave/Z PeakWave=OldFitFolder:$(OldWaveName+"_pk"+Num2iStr(i))
			for (i=1; WaveExists(PeakWave); i+=1)
				Duplicate/O PeakWave, NewFitFolder:$(NewWaveName+"_pk"+Num2iStr(i-1))
				Wave/Z PeakWave=OldFitFolder:$(OldWaveName+"_pk"+Num2iStr(i))
			endfor
		endif
	
		//	Finds the image folder and wave
		DFREF ParentFolder=$(GetDataFolder(1, DataFolder)+":")
		DFREF ImageFolder=ParentFolder:Images
				
		//	Checks if the image folder exists
		if ((DataFolderRefStatus(ParentFolder)!=0) && (DataFolderRefStatus(ImageFolder)!=0))

			//	Duplicates the image wave if it exists
			Wave/Z ImageWave=ImageFolder:$OldWaveName
			if (WaveExists(ImageWave))
				Duplicate/O ImageWave, ImageFolder:$NewWaveName
			endif
		endif
	endif
end



Static Function XPSViewGroup(B_Struct) : ButtonControl
//	Stores the current set of selected waves in the View Spectra listbox
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Reads the group name specified in the SetGroupName box
		ControlInfo /W=$B_Struct.win SetGroupName
		String GroupName=S_Value
		
		//	Reads the selected data folder specified in the DataFolderPopupMenu
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		String FolderName=S_Value
		
		//	Stores the current set of selected waves in the View Spectra listbox
		XPSViewSaveGroup(GroupName, FolderName)
		
		//	Sets the saved group as the displayed group in the popup menu. ControlUpdate is needed to update the displayed list to the new values
		ControlUpdate /W=$B_Struct.win GroupPopUp
		PopupMenu GroupPopUp popmatch=GroupName, win=$B_Struct.win
	endif
	Return 0
end



Static Function XPSViewSaveGroup(GroupName, FolderName)
//	Stores the current set of selected waves in the View Spectra listbox
String GroupName, FolderName

	String ActiveWindow="XPSViewSpectraWindow"

	DFREF ProgramFolder=root:Programming
	Wave/T XPSViewWaveList=ProgramFolder:XPSViewWaveList
	Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
		
	//	Removes all commas and equal signs from the group name
	GroupName=ReplaceString(",", GroupName, "\COMMA", 1)
	GroupName=ReplaceString("=", GroupName, "\EQUAL", 1)
		
	//	Removes all commas, equal signs and semicolons from the folder name
	FolderName=ReplaceString(",", FolderName, "\COMMA", 1)
	FolderName=ReplaceString("=", FolderName, "\EQUAL", 1)
	FolderName=ReplaceString(";", FolderName, "\SCLN", 1)
		
	//	Finds the global string with information about the saved groups. A global string is used to prevent the information from being lost if the View Spectra window is closed
	//	The format of the string is: Group1=Folder1;Wave1:Colour1;Wave2:Colour2;Wave3:Colour3;,Group2=Folder2;Wave4:Colour4;Wave5:Colour5;Wave6:Colour6;, etc....
	SVAR/Z SavedGroups=ProgramFolder:XPSViewSavedGroups
	if (SVAR_Exists(SavedGroups)==0)
		String /G ProgramFolder:XPSViewSavedGroups=""
		SVAR SavedGroups=ProgramFolder:XPSViewSavedGroups
	endif

	//	Extracts the positions of the clicked waves in the wavelist
	Duplicate/FREE/O/R=[][0][0] XPSViewWaveListSel TempSelWave
	Extract/FREE/INDX/O TempSelWave, IndexWave, (TempSelWave & 16)==16
		
	//	Calculates the number of clicked waves
	Variable i=0
	Variable n=NumPnts(IndexWave)
		
	//	Adds the folder, wave names and colour numbers to the group info
	String WaveNameStr=""
	String GroupInfo=FolderName+";"
	for (i=0; i<n; i+=1)
		
		//	Removes all commas, equal signs, semicolons and colons from the wave name
		WaveNameStr=XPSViewWaveList[IndexWave[i]][0]
		WaveNameStr=ReplaceString(",", WaveNameStr, "\COMMA", 1)
		WaveNameStr=ReplaceString("=", WaveNameStr, "\EQUAL", 1)
		WaveNameStr=ReplaceString(";", WaveNameStr, "\SCLN", 1)
		WaveNameStr=ReplaceString(":", WaveNameStr, "\CLN", 1)

		//	Adds the wave name and colour to the group info
		GroupInfo+=WaveNameStr+":"+Num2iStr(XPSViewWaveListSel[IndexWave[i]][2][1])+";"
	endfor

	//	Adds the group to the list of saved groups
	SavedGroups=ReplaceStringByKey(GroupName, SavedGroups, GroupInfo, "=", ",", 1)
end



Static Function XPSViewDisband(B_Struct) : ButtonControl
//	Disbands the selected group if it exists
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	If shift was held down all saved groups are removed
		if ((B_Struct.eventMod & 2)==2)
		
			//	Prompts the user before disbanding all groups
			DoWindow /K XPSViewDisbandWindow
			NewPanel /K=1 /W=(370, 220, 660, 260) /N=XPSViewDisbandWindow as "Disband all saved groups?"
			
			//	Creates Disband and Cancel buttons
			Button CancelButton proc=EccentricXPS#CloseWindow, pos={35,10}, size={100, 20}, title="Cancel", win=XPSViewDisbandWindow
			Button DisbandAllButton proc=EccentricXPS#XPSViewDisbandAll, pos={155,10}, size={100, 20}, title="Disband All", win=XPSViewDisbandWindow
		else

			//	Reads the group name specified in the SetGroupName box
			ControlInfo /W=$B_Struct.win SetGroupName
			String GroupName=S_Value
			
			//	Removes all commas and equal signs from the group name
			GroupName=ReplaceString(",", GroupName, "\COMMA", 1)
			GroupName=ReplaceString("=", GroupName, "\EQUAL", 1)
		
			//	Finds the global string with information about the saved groups. A global string is used to prevent the information from being lost if the View Spectra window is closed
			//	The format of the string is: Group1=Folder1;Wave1:Colour1;Wave2:Colour2;Wave3:Colour3;,Group2=Folder2;Wave4:Colour4;Wave5:Colour5;Wave6:Colour6;, etc....
			DFREF ProgramFolder=root:Programming
			SVAR/Z SavedGroups=ProgramFolder:XPSViewSavedGroups
			if (SVAR_Exists(SavedGroups)==0)
				String /G ProgramFolder:XPSViewSavedGroups=""
				SVAR SavedGroups=ProgramFolder:XPSViewSavedGroups
			endif

			//	Removes the group from the list of saved groups
			SavedGroups=RemoveByKey(GroupName, SavedGroups, "=", ",", 1)

			//	Clears the displayed group name
			SetVariable SetGroupName value=_STR:"", win=$B_Struct.win
		
			//	Updates the popup list of groups, keeping the selection if it still exists. ControlUpdate is needed to update the displayed list to the new values
			ControlInfo /W=$B_Struct.win GroupPopUp
			String OldSelection=S_Value
			ControlUpdate /W=$B_Struct.win GroupPopUp
			PopupMenu GroupPopUp popmatch=OldSelection, win=$B_Struct.win
			
			//	Changes the displayed text to Select a group to display" if the selected group was disbanded
			ControlInfo /W=$B_Struct.win GroupPopUp
			if (CmpStr(S_Value, OldSelection)!=0)
				PopupMenu GroupPopUp mode=1, popvalue="Select a group to display", win=XPSViewSpectraWindow
			endif
		endif
	endif
	Return 0
end



Static Function XPSViewDisbandAll(B_Struct) : ButtonControl
//	Disbands all saved groups
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Kills the prompt window
		DoWindow/K $B_Struct.win
		
		//	Finds the global string with information about the saved groups. A global string is used to prevent the information from being lost if the View Spectra window is closed
		//	The format of the string is: Group1=Folder1;Wave1:Colour1;Wave2:Colour2;Wave3:Colour3;,Group2=Folder2;Wave4:Colour4;Wave5:Colour5;Wave6:Colour6;, etc....
		DFREF ProgramFolder=root:Programming
		SVAR/Z SavedGroups=ProgramFolder:XPSViewSavedGroups
		if (SVAR_Exists(SavedGroups)==0)
			String /G ProgramFolder:XPSViewSavedGroups=""
		else
		
			//	Clears the saved groups string
			SavedGroups=""
		endif
		
		//	Updates the group selections
		SetVariable SetGroupName value=_STR:"", win=XPSViewSpectraWindow
		PopupMenu GroupPopUp mode=1, popvalue="Select a group to display", win=XPSViewSpectraWindow
	endif
	Return 0
end



Static Function/S XPSViewGroupPopUpList()
//	Returns a list of saved groups. Returns "No saved groups exist;", if no saved groups exist

	//	Returns a list of saved groups
	String GroupNames=XPSViewGroupList()
	
	//	Returns no saved groups exist, if no saved groups exist
	if (StrLen(GroupNames)==0)
		GroupNames="No saved groups exist;"
	endif

	//	Returns the list	
	Return GroupNames
end



Static Function/S XPSViewGroupList()
//	Returns a list of saved groups

	//	Finds the global string with information about the saved groups. A global string is used to prevent the information from being lost if the View Spectra window is closed
	//	The format of the string is: Group1=Folder1;Wave1:Colour1;Wave2:Colour2;Wave3:Colour3;,Group2=Folder2;Wave4:Colour4;Wave5:Colour5;Wave6:Colour6;, etc....
	DFREF ProgramFolder=root:Programming
	SVAR/Z SavedGroups=ProgramFolder:XPSViewSavedGroups
	if (SVAR_Exists(SavedGroups)==0)
		String /G ProgramFolder:XPSViewSavedGroups=""
		SVAR SavedGroups=ProgramFolder:XPSViewSavedGroups
	endif
	
	String GroupNames=""
	
	if (StrLen(SavedGroups)>0)

		//	Extracts the names from the saved groups
		Variable a=-1
		Variable b=StrSearch(SavedGroups, "=", 0)
		do	
			GroupNames+=SavedGroups[a+1, b-1]+";"
			a=StrSearch(SavedGroups, ",", b+1)
			b=StrSearch(SavedGroups, "=", a+1)
		while ((a>0) && (b>0))
		
		//	Restores all commas and equal signs in group name
		GroupNames=ReplaceString("\COMMA", GroupNames, ",", 1)
		GroupNames=ReplaceString("\EQUAL", GroupNames, "=", 1)
		
		//	Sorts the list alphanumerically
		GroupNames=SortList(GroupNames, ";", 16)
	endif

	//	Returns the semicolon separated list of group names
	Return GroupNames
end



Static Function/S XPSViewGetGroupInfo(GroupName)
//	Extracts the folder, waves and their colours for the selected group
String GroupName

	//	Removes all commas and equal signs from the group name
	GroupName=ReplaceString(",", GroupName, "\COMMA", 1)
	GroupName=ReplaceString("=", GroupName, "\EQUAL", 1)
	
	//	Finds the global string with information about the saved groups. A global string is used to prevent the information from being lost if the View Spectra window is closed
	//	The format of the string is: Group1=Folder1;Wave1:Colour1;Wave2:Colour2;Wave3:Colour3;,Group2=Folder2;Wave4:Colour4;Wave5:Colour5;Wave6:Colour6;, etc....
	DFREF ProgramFolder=root:Programming
	SVAR/Z SavedGroups=ProgramFolder:XPSViewSavedGroups
	if (SVAR_Exists(SavedGroups)==0)
		String /G ProgramFolder:XPSViewSavedGroups=""
		SVAR SavedGroups=ProgramFolder:XPSViewSavedGroups
	endif

	//	Extracts the folder, waves and their colours for the selected group
	Return StringByKey(GroupName, SavedGroups, "=", ",", 1)
end



Static Function/DF XPSViewGroupDataFolder(GroupInfo)
//	Returns the data folder for the selected group
String GroupInfo

	//	Extracts the folder information
	String Folder=StringFromList(0, GroupInfo, ";")
			
	//	Restores all commas, equal signs and semicolons to the folder name
	Folder=ReplaceString("\COMMA", Folder, ",", 1)
	Folder=ReplaceString("\EQUAL", Folder, "=", 1)
	Folder=ReplaceString("\SCLN", Folder, ";", 1)

	//	Returns the data folder for the selected group
	Return $Folder
end



Static Function/S XPSViewGroupDataFolderStr(GroupInfo)
//	Returns the data folder string for the selected group
String GroupInfo

	//	Extracts the folder information
	String Folder=StringFromList(0, GroupInfo, ";")
			
	//	Restores all commas, equal signs and semicolons to the folder name
	Folder=ReplaceString("\COMMA", Folder, ",", 1)
	Folder=ReplaceString("\EQUAL", Folder, "=", 1)
	Folder=ReplaceString("\SCLN", Folder, ";", 1)

	//	Returns the data folder for the selected group
	Return Folder
end



Static Function XPSViewGroupPopUp(PU_Struct) : PopUpMenuControl
//	Plots the selected group from the saved groups
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Ignores further calls to the popup menu procedure until this one has finished
		PU_Struct.blockReentry=1
		
		String GroupName=PU_Struct.popStr
		
		if (CmpStr(GroupName, "No saved groups exist")!=0)

			//	Plots the selected group from the saved groups
			XPSViewDisplayGroup(GroupName)
		endif
		
		//	If the user scrolls through the items in the list too fast, the selection will change before the update has finished, and the selection will not match the displayed data. This will prevent that
		PopupMenu $PU_Struct.ctrlName popmatch=PU_Struct.popStr, win=$PU_Struct.win
	endif
	Return 0
end



Static Function XPSViewDisplayGroup(GroupName)
//	Plots the selected group from the saved groups
String GroupName
String ActiveWindow="XPSViewSpectraWindow"
		
	//	Extracts the folder, waves and their colours for the selected group
	String GroupInfo=XPSViewGetGroupInfo(GroupName)
			
	//	Returns the data folder string for the selected group
	String ActiveFolderStr=XPSViewGroupDataFolderStr(GroupInfo)

	//	Sets the data folder selection to FolderRef and reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or the default data folder is used instead, depending on the value of ForceSelection
	PopupMenu DataFolderPopUp popmatch=ActiveFolderStr, win=$ActiveWindow
	ControlInfo /W=$ActiveWindow DataFolderPopUp
	if (CmpStr(ActiveFolderStr, S_Value)!=0)
		PopupMenu DataFolderPopUp mode=1, popvalue=ActiveFolderStr, win=$ActiveWindow
	endif
	DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
	
	//	Sets the displayed name to the name of the selected group		
	SetVariable SetGroupName value=_STR:GroupName, win=$ActiveWindow
	
	//	Checks if the data folder exists
	if (DataFolderRefStatus(ActiveFolder)==0)
	
		//	Creates empty listbox waves if the data folder does not exist
		XPSViewCreateEmptyListBoxWaves("Data folder does not exist")
	else

		//	Counts the number of waves in the list
		Variable n=ItemsInList(GroupInfo, ";")-1

		//	Checks if any waves were selected in the group
		if (n==0)
				
			//	Creates empty listbox waves if no waves were selected in the group
			XPSViewCreateEmptyListBoxWaves("")
		else
		
			//	Creates the necessary waves for the Listbox
			DFREF ProgramFolder=root:Programming
			Make/T/O/N=(n) ProgramFolder:XPSViewWaveList /WAVE=XPSViewWaveList
			Make/I/U/O/N=(n) ProgramFolder:XPSViewWaveListSel /WAVE=XPSViewWaveListSel
			Make/WAVE/O/N=(n) ProgramFolder:XPSViewWaveListWaves /WAVE=XPSViewWaveListWaves

			//	Extracts wave names and colour information from the saved group
			Variable a=0, i=0
			String SingleWaveInfo="", WaveNameStr=""
			for (i=0; i<n; i=i+1)
				SingleWaveInfo=StringFromList(i+1, GroupInfo, ";")
				a=StrSearch(SingleWaveInfo, ":", 0)
				
				//	Restores all commas, equal signs, semicolons and colons to the folder name
				WaveNameStr=SingleWaveInfo[0,a-1]
				WaveNameStr=ReplaceString("\COMMA", WaveNameStr, ",", 1)
				WaveNameStr=ReplaceString("\EQUAL", WaveNameStr, "=", 1)
				WaveNameStr=ReplaceString("\SCLN", WaveNameStr, ";", 1)
				WaveNameStr=ReplaceString("\CLN", WaveNameStr, ":", 1)
			
				XPSViewWaveList[i]=WaveNameStr
				XPSViewWaveListWaves[i]=ActiveFolder:$WaveNameStr
				XPSViewWaveListSel[i]=Str2Num(SingleWaveInfo[a+1, StrLen(SingleWaveInfo)-1])
			endfor
			
			//	Rescales the waves to the correct dimensions for the listbox
			InsertPoints /M=1 1, 2, XPSViewWaveList
			InsertPoints /M=1 0, 2, XPSViewWaveListSel
			InsertPoints /M=2 0, 1, XPSViewWaveListSel
		
			//	Defines the entire second column with the wave names as checked (bit 4) checkboxes (bit 5)
			XPSViewWaveListSel[][0][0]=48

			//	Makes the colours affect the background colour, change to forecolors to affect foreground (text) colours
			SetDimLabel 2,1,backColors, XPSViewWaveListSel
		endif
	endif
	
	//	Sets the axis scaling and labels to match the type of spectrum displayed and updates the graph
	XPSViewSpectraUpdateGraphDF(ActiveWindow, ActiveFolder, $"")
end



Static Function/S XPSViewBelongsToGroups(ActiveWave)
//	Returns a semicolon-separated list of all groups ActiveWave belongs to
Wave/Z ActiveWave

	//	The semicolon-separated list of groups the ActiveWave belongs to
	String ListOfGroups=""
	
	if (WaveExists(ActiveWave))

		//	Finds the global string with information about the saved groups. A global string is used to prevent the information from being lost if the View Spectra window is closed
		//	The format of the string is: Group1=Folder1;Wave1:Colour1;Wave2:Colour2;Wave3:Colour3;,Group2=Folder2;Wave4:Colour4;Wave5:Colour5;Wave6:Colour6;, etc....
		DFREF ProgramFolder=root:Programming
		SVAR/Z SavedGroups=ProgramFolder:XPSViewSavedGroups
		if (SVAR_Exists(SavedGroups)==0)
			String /G ProgramFolder:XPSViewSavedGroups=""
			SVAR SavedGroups=ProgramFolder:XPSViewSavedGroups
		endif
	
		if (StrLen(SavedGroups)>0)
	
			//	Finds the wave name and removes all commas, equal signs, semicolons and colons
			String WaveNameStr=NameOfWave(ActiveWave)
			WaveNameStr=ReplaceString(",", WaveNameStr, "\COMMA", 1)
			WaveNameStr=ReplaceString("=", WaveNameStr, "\EQUAL", 1)
			WaveNameStr=ReplaceString(";", WaveNameStr, "\SCLN", 1)
			WaveNameStr=ReplaceString(":", WaveNameStr, "\CLN", 1)

			//	Finds the data folder name and removes all commas, equal signs and semicolons
			String WaveFolderName=GetWavesDataFolder(ActiveWave, 1)
			WaveFolderName=ReplaceString(",", WaveFolderName, "\COMMA", 1)
			WaveFolderName=ReplaceString("=", WaveFolderName, "\EQUAL", 1)
			WaveFolderName=ReplaceString(";", WaveFolderName, "\SCLN", 1)
			
			//	Finds the first instance of ActiveWave in the saved groups, if one exists
			Variable ActiveWaveLocation=StrSearch(SavedGroups, ";"+WaveNameStr+":", 0)

			//	Finds the position of the first group in the list
			Variable EndOfPreviousGroup=-1
			Variable EndOfThisGroup=StrSearch(SavedGroups, ",", EndOfPreviousGroup+1)
		
			//	Variables used in the search
			String GroupName=""
			String GroupFolderName=""
			Variable EndOfGroupName=-1
			Variable EndOfGroupFolderName=-1
		
			//	Counts through all instances of ActiveWave in the saved groups
			do
		
				//	Counts through the group until the group ActiveWave belongs to has been reached
				do	

					//	Finds the positions of the next group in the list	
					EndOfPreviousGroup=EndOfThisGroup
					EndOfThisGroup=StrSearch(SavedGroups, ",", EndOfPreviousGroup+1)
			
				//	Continue until the group ActiveWave is a member of has been reached	
				while(EndOfThisGroup<ActiveWaveLocation)

				//	Finds the name and folder of the group ActiveWave belongs to
				EndOfGroupName=StrSearch(SavedGroups, "=", EndOfPreviousGroup+1)
				GroupName=SavedGroups[EndOfPreviousGroup+1, EndOfGroupName-1]
				EndOfGroupFolderName=StrSearch(SavedGroups, ";", EndOfGroupName+1)
				GroupFolderName=SavedGroups[EndOfGroupName+1, EndOfGroupFolderName-1]
			
				//	Checks if the data folder of the saved group mmatches with the data folder of ActiveWave
				if (CmpStr(WaveFolderName, GroupFolderName)==0)
			
					//	Adds the group to the list of groups ActiveWave belongs to
					ListOfGroups+=GroupName+";"		
				endif
		
				//	Finds the next instance of ActiveWave in the saved groups
				ActiveWaveLocation=StrSearch(SavedGroups, ";"+WaveNameStr+":", ActiveWaveLocation+1)

			//	Continue as long as ActiveWave is a member of another group
			while (ActiveWaveLocation>0)
		
			if (StrLen(ListOfGroups)>0)
		
				//	Restores all commas and equal signs in group name
				ListOfGroups=ReplaceString("\COMMA", ListOfGroups, ",", 1)
				ListOfGroups=ReplaceString("\EQUAL", ListOfGroups, "=", 1)
		
				//	Sorts the list alphanumerically
				ListOfGroups=SortList(ListOfGroups, ";", 16)
			endif
		endif
	endif
	
	//	Returns a semicolon-separated list of all groups ActiveWave belongs to
	Return ListOfGroups
end



Static Function XPSViewDisplayTypePopup(PU_Struct) : PopUpMenuControl
//	Sets the axis scaling and labels to match the type of spectrum displayed
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Ignores further calls to the popup menu procedure until this one has finished
		PU_Struct.blockReentry=1
		
		Variable AutoScaleX=1, AxisXMin=0, AxisXMax=0
		
		//	Checks if the Fit Assistant window is the active window
		if (CmpStr(PU_Struct.win, "XPSFitAssistantWindow")==0)

			//	Checks if the x-axis range is manual
			String XAxisFlags=StringByKey("SETAXISFLAGS", AxisInfo(PU_Struct.win,"bottom"))
			if (StrLen(XAxisFlags)==0)
			
				//	Finds the displayed wave
				DFREF ActiveFolder=XPSViewGetDataFolder(PU_Struct.win, "DataFolderPopUp")
				Wave/Z DisplayedWave=XPSViewGetDisplayedWave(PU_Struct.win, ActiveFolder, "DisplayedWavePopUp")
				
				if (WaveExists(DisplayedWave))
	
					//	Finds the x-range of the displayed wave
					Variable DataXMin=Min(LeftX(DisplayedWave), Pnt2X(DisplayedWave, NumPnts(DisplayedWave)-1))
					Variable DataXMax=Max(LeftX(DisplayedWave), Pnt2X(DisplayedWave, NumPnts(DisplayedWave)-1))
	
					//	Finds the current x-axis scaling
					GetAxis /W=$PU_Struct.win /Q bottom
					AxisXMin=Min(V_max, V_min)
					AxisXMax=Max(V_max, V_min)
	
					//	Checks if the displayed wave has any datapoints inside the manual x-axis range
					if (limit(DataXMin, AxisXMin, AxisXMax)!=limit(DataXMax, AxisXMin, AxisXMax))
	
						//	Keeps the manual x-axis scaling
						AutoScaleX=0
					endif
				endif
			endif
		endif

		//	Sets the axis scaling and labels to match the type of spectrum displayed
		XPSViewSetAxis(PU_Struct.win, PU_Struct.popStr, AutoScaleX, AxisXMin, AxisXMax)
	endif
	Return 0
end



Static Function XPSViewSet(B_Struct) : ButtonControl
//	Changes the default data folder, set by SetDataFolder(), to the displayed folder
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Finds the active folder
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value

		//	Changes the default folder to the displayed folder, if the displayed folder is a valid folder
		if (DataFolderRefStatus(DataFolder)!=0)
			SetDataFolder DataFolder
		endif
		
		//	Refreshes the graph
		XPSViewSpectraUpdateGraph(B_Struct.win, DataFolder, $"")
	endif
	Return 0
end



Static Function XPSViewRestore(B_Struct) : ButtonControl
//	Changes the displayed data folder to the default data folder set by SetDataFolder()
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Selects the default data folder set by SetDataFolder() as the active folder
		PopupMenu DataFolderPopUp popmatch=GetDataFolder(1), win=$B_Struct.win
		
		//	Reads back the selected data folder
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value
		
		//	Refreshes the graph
		XPSViewSpectraUpdateGraph(B_Struct.win, DataFolder, $"")
	endif
	Return 0
end



Static Function XPSViewShowHidden(B_Struct) : ButtonControl
//	Shows all hidden waves in the View Spectra graph
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Unhides all traces
		ModifyGraph /W=$B_Struct.win hideTrace=0
		
		//	Hides the empty placeholder wave
		ModifyGraph /W=$B_Struct.win hideTrace(XPSViewEmptyWave)=1
		
		//	Finds the selected data folder
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		DFREF DataFolder=$S_Value
	
		//	Refreshes the graph
		XPSViewSpectraUpdateGraph(B_Struct.win, DataFolder, $"")
	endif
	Return 0
end



Static Function XPSViewShiftPrompt(B_Struct) : ButtonControl
//	Creates a message box asking the user how much the selected waves should be shifted
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
	
		//	Creates a new temp data folder to contain the information about which waves are clicked
		NewDataFolder/O root:Programming:Temp:ShiftWaves
		DFREF ProgramFolder=root:Programming, ShiftTempFolder=root:Programming:Temp:ShiftWaves
		
		//	The listbox wave containing information about which waves are selected
		Wave XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
		Wave/WAVE XPSViewWaveListWaves=ProgramFolder:XPSViewWaveListWaves

		//	Extracts the clicked waves from the wavelist
		Duplicate/FREE/O/R=[][0][0] XPSViewWaveListSel TempSelWave
		Extract/O XPSViewWaveListWaves, ShiftTempFolder:ShiftedWaves, (TempSelWave & 16)==16
		Wave ShiftedWaves=ShiftTempFolder:ShiftedWaves
		
		Variable n=NumPnts(ShiftedWaves)
		
		//	Will only create the message box if waves are actually selected
		if (n>0)
		
			//	Determines the title of the dialog box
			String PanelTitle="Shift All Displayed Waves?"
			Wave/Z CursorWave=CsrWaveRef(A, B_Struct.win)
			if (WaveExists(CursorWave))
				PanelTitle="Shift Wave?"
				Make/O/WAVE ShiftTempFolder:ShiftedWaves={CursorWave}
			endif

			//	Creates the dialog box
			DoWindow /K XPSViewShiftWindow
			NewPanel /K=1 /W=(800, 300, 1100, 395) /N=XPSViewShiftWindow as PanelTitle
			SetVariable XPSViewSpectraShift bodyWidth=100, disable=0, pos={115,15}, title="Shift (eV):", value=_NUM:0, win=XPSViewShiftWindow
			CheckBox XPSViewSpectraInvert noproc, pos={180,15}, title="Invert Energy Axis", value=0, win=XPSViewShiftWindow
			Button XPSViewShiftWindowB proc=EccentricXPS#CloseWindow, pos={15,50}, size={125, 30}, title="Cancel", win=XPSViewShiftWindow
			Button XPSViewShiftB proc=EccentricXPS#XPSViewShift, pos={160,50}, size={125, 30}, title="Shift", win=XPSViewShiftWindow
			
			//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
			SetWindow XPSViewShiftWindow, hook(KillHook)=XPSViewShiftHookFunc
		endif
	endif
	Return 0
end



Function XPSViewShiftHookFunc(s)
//	The hook function for the View Images window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1
	
			//	Kills the temporary data folder
			KillDataFolder /Z root:Programming:Temp:ShiftWaves
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function XPSViewShift(B_Struct) : ButtonControl
//	Shifts the selected waves
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	The list of waves to be shifted
		DFREF ShiftTempFolder=root:Programming:Temp:ShiftWaves
		Wave/WAVE ShiftedWaves=ShiftTempFolder:ShiftedWaves
		
		//	Reads the variables from the dialog window
		ControlInfo /W=$B_Struct.win XPSViewSpectraShift
		Variable Shift=V_Value
		ControlInfo /W=$B_Struct.win XPSViewSpectraInvert
		Variable Invert=V_Value
	
		if (Invert==0)
			Invert=1
		else
			Invert=-1
		endif

		//	Shifts all clicked waves
		ShiftedWaves[]=XPSViewShiftSingleWave(Shift, Invert, ShiftedWaves[p])

		//	Closes the promt window, this has to be at the end or the cleanup function will remove the temporary waves
		DoWindow /K $B_Struct.win
		
		//	Finds the selected data folder
		ControlInfo /W=XPSViewSpectraWindow DataFolderPopUp
		DFREF DataFolder=$S_Value
	
		//	Updates the graph
		XPSViewSpectraUpdateGraph("XPSViewSpectraWindow", DataFolder, $"")
	endif
	Return 0
end



Static Function/WAVE XPSViewShiftSingleWave(Shift, Invert, ShiftWave)
//	Shifts the selected wave and any associated fit waves and updates the saved fit information
//	Meant to correct for binding energy shifts, but, by clicking the Invert Energy Axis checkbox, can also be used to change between kinetic and binding energy.
Variable Shift, Invert
Wave/Z ShiftWave
Variable i=0

	if (WaveExists(ShiftWave))

		//	Shifts the wave by Shift
		Setscale/P x, Invert*LeftX(ShiftWave)+Shift, Invert*DeltaX(ShiftWave), ShiftWave
				
		//	Extracts the location and names of the fit waves
		String ShiftWaveName=NameOfWave(ShiftWave)
		DFREF ActiveFolder=GetWavesDataFolderDFR(ShiftWave)
		DFREF SavedFitsFolder=ActiveFolder:SavedFits
		DFREF FitFolder=SavedFitsFolder:$ShiftWaveName

		if ((DataFolderRefStatus(SavedFitsFolder)!=0) &&	 (DataFolderRefStatus(FitFolder)!=0))
	
			//	Shifts the fit waves
			Wave/Z FitWave=FitFolder:$(ShiftWaveName+"_fit")
			Wave/Z BckWave=FitFolder:$(ShiftWaveName+"_bck")
			
			if (WaveExists(FitWave) && WaveExists(BckWave))
			
				Variable LeftXValue=Invert*LeftX(FitWave)+Shift
				Variable DeltaXValue=Invert*DeltaX(FitWave)
				Setscale/P x, LeftXValue, DeltaXValue, FitWave, BckWave
		
				//	Shifts each individual peak
				Wave/Z PeakWave=FitFolder:$(ShiftWaveName+"_pk0")
				for (i=1; WaveExists(PeakWave); i+=1)
					Setscale/P x, LeftXValue, DeltaXValue, PeakWave
					Wave/Z PeakWave=FitFolder:$(ShiftWaveName+"_pk"+Num2iStr(i))
				endfor
			endif

			//	Shifts the coefficients in the coefficient waves
			Wave/Z/T CoefWave=FitFolder:$(ShiftWaveName+"_lst")
			Wave/Z/I/U CoefSelWave=FitFolder:$(ShiftWaveName+"_sel")
			
			if (WaveExists(CoefWave) && WaveExists(CoefSelWave))
			
				//	Finds the number of peaks in the coefficient wave
				Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)
				Variable Slope=0, Intercept=0, Shirley=0, PeakArea=0
				
				if (NumberOfPeaks>=0)
				
					//	Finds the intercept and slope
					Slope=Invert*Str2Num(CoefWave[1][5])
					Intercept=Str2Num(CoefWave[1][1])-Shift*Slope
					
					for (i=0; i<NumberOfPeaks; i+=1)
				
						//	Finds the change in offset caused by the inverted Shirley background
						Shirley=Invert*Str2Num(CoefWave[4+i][25])
						PeakArea=Str2Num(CoefWave[4+i][5])
						Intercept+=0.5*(1/Invert-1)*PeakArea*Shirley
						
						//	Shifts the peak positions and Shirley background
						CoefWave[4+i][1]=Num2Str(Invert*Str2Num(CoefWave[p][q])+Shift)
						CoefWave[4+i][25]=Num2Str(Shirley)
					endfor
					
					//	Shifts the intercept and slope
					CoefWave[1][1]=Num2Str(Intercept)
					CoefWave[1][5]=Num2Str(Slope)
				endif
			endif
		endif
	endif
	
	//	Not used
	Return ShiftWave
end



Static Function XPSViewNewWin(B_Struct) : ButtonControl
//	Creates a copy of the active graph, without any controls, meant as a starting point for a more presentable figure
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	The temporary folder to place the global string in
		DFREF TempFolder=root:Programming:Temp
		
		//	Finds the recreation macro string for the active window
		KillStrings /Z TempFolder:WinRecreationString
		String/G TempFolder:WinRecreationString=WinRecreation(B_Struct.win, 0)
		SVAR WinRecreationString=TempFolder:WinRecreationString
		
		//	Removes unwanted controls from the recreation string. GrepList(x, y, 1, "\r") means include all items in x where y is NOT true. ^\t(x|y|y) means starting with (^) a tabulator (\t) followed by either x, y or z.
		WinRecreationString=GrepList(WinRecreationString, "^\t(ControlBar|SetWindow|Cursor|ListBox|CheckBox|PopupMenu|ValDisplay|SetVariable|Button|ModifyGraph margin|NewPanel|ModifyPanel|RenameWindow|SetActiveSubwindow|ShowInfo)", 1, "\r")
		
		//	Finds the Display command
		Variable a=StrSearch(WinRecreationString, "\r\tDisplay /W=(", 0)+13
		Variable b=StrSearch(WinRecreationString, ")", a)
		
		//	Finds the position of the old window
		Variable x1=0, y1=0, x2=0, y2=0, Shift=20
		SScanF WinRecreationString[a, b], "(%f,%f,%f,%f)", x1, y1, x2, y2
		
		//	Shifts the new window by 20 points
		WinRecreationString=WinRecreationString[0,a]+Num2Str(x1+Shift)+","+Num2Str(y1+Shift)+","+Num2Str(x2+Shift)+","+Num2Str(y2+Shift)+WinRecreationString[b,StrLen(WinRecreationString)-1]
		
		//	Executes the macro
		Execute/P/Q "Execute root:Programming:Temp:WinRecreationString"
		
		//	Kills the global string
		Execute/P/Q "KillStrings /Z root:Programming:Temp:WinRecreationString"
	endif
	Return 0
end



Static Function XPSViewAverage(B_Struct) : ButtonControl
//	Averages all visible traces into one wave
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Creates a new temp data folder to contain the information about which waves are clicked
		NewDataFolder/O root:Programming:Temp:Combine
		DFREF ProgramFolder=root:Programming, CombineFolder=root:Programming:Temp:Combine
		
		//	The listbox wave containing information about which waves are selected
		Wave XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
		Wave/WAVE XPSViewWaveListWaves=ProgramFolder:XPSViewWaveListWaves

		//	Extracts the positions of the clicked waves in the wavelist
		Duplicate/FREE/O/R=[][0][0] XPSViewWaveListSel TempSelWave
		Extract/FREE/INDX/O TempSelWave, IndexWave, (TempSelWave & 16)==16
	
		//	Finds the number of waves to combine
		Variable n=NumPnts(IndexWave)

		if (n>0)
		
			//	Creates a temporary wave to hold the colour information for the clicked waves
			Make/FREE/I/U/O/N=(n) Colours=XPSViewWaveListSel[IndexWave[p]][2][1]
					
			//	Creates a wave to hold the wave references of the clicked waves
			Make/WAVE/O/N=(n) CombineFolder:CombineWaves=XPSViewWaveListWaves[IndexWave[p]]
			Wave/WAVE CombineWaves=CombineFolder:CombineWaves
			
			//	Sorts the wave references by colour number
			Sort Colours, CombineWaves

			//	Finds the x-axis range where all spectra overlap
			Make/FREE/O/N=(n) LeftXWave=LeftX(CombineWaves[p]), RightXWave=Pnt2X(CombineWaves[p], NumPnts(CombineWaves[p])-1)
			Make/FREE/O/N=(n) StartXWave=Max(LeftXWave, RightXWave), EndXWave=Min(LeftXWave, RightXWave)
			Variable StartX=WaveMin(StartXWave)
			Variable EndX=WaveMax(EndXWave)

			//	If a range exists where all spectra overlap
			if (StartX>EndX)
			
				//	The wave with the lowest colour number will determine the x-scaling of the combined result
				Duplicate/O/R=(StartX, EndX) CombineWaves[0] CombineFolder:CombinedResult/WAVE=CombinedResult
				Variable nn=NumPnts(CombinedResult)
				
				if (nn>0)
				
					//	Finds the active data folder
					ControlInfo /W=$B_Struct.win DataFolderPopUp
					DFREF DataFolder=$S_Value
					
					//	Suggests a new name based on the name of the first wave
					String SaveName=NameOfWave(CombineWaves[0])+"_Avg"
			
					//	Creates a save wave dialog where the OK button will execute XPSViewDoCombine
					XPSViewSaveWaveDialog("Save combined spectrum as", GetDataFolder(1, DataFolder), SaveName, "EccentricXPS#XPSViewDoAverage", "EccentricXPS#XPSViewAverageCleanUp", B_Struct.win)
				else
					ErrorMessage("The selected waves do not overlap!")

					//	Deletes all temporary waves
					XPSViewAverageCleanUp("")
				endif
			else
				ErrorMessage("The selected waves do not overlap!")

				//	Deletes all temporary waves
				XPSViewAverageCleanUp("")
			endif
		endif
	endif
end



Static Function XPSViewAverageCleanUp(ActiveWindow)
//	Deletes all temporary waves associated with the combine functions. ActiveWindow is required for a clean up function, but not used in this function.
String ActiveWindow
	DFREF CombineFolder=root:Programming:Temp:Combine
	KillDataFolder /Z CombineFolder
end



Static Function XPSViewDoAverage(SaveFolder, SaveName, ActiveWindow)
//	Combines the clicked waves into one wave, retaining the original waves
DFREF SaveFolder
String SaveName, ActiveWindow

	//	The folder containing the temporary waves from the last function
	DFREF CombineFolder=root:Programming:Temp:Combine
	
	//	The waves containing the waves references to the clicked waves
	Wave/WAVE CombineWaves=CombineFolder:CombineWaves
	
	//	The wave definining the x-scaling of the resulting combination wave
	Wave CombinedResult=CombineFolder:CombinedResult
	
	//	Calculates the number of waves to combine
	Variable n=NumPnts(CombineWaves)
	
	//	Adds the waves together one by one
	Variable i=1
	for (i=1; i<n; i+=1)
		Wave NextWave=CombineWaves[i]
		CombinedResult+=NextWave(x)
	endfor
	FastOP CombinedResult=(1/n)*CombinedResult
	
	//	Copies the combined wave to its final name
	Duplicate/O CombinedResult, SaveFolder:$SaveName

	//	Deletes all temporary waves
	KillDataFolder /Z CombineFolder
	
	//	Finds the selected data folder
	ControlInfo /W=$ActiveWindow DataFolderPopUp
	DFREF DataFolder=$S_Value
	
	//	Updates the graph
	XPSViewSpectraUpdateGraph(ActiveWindow, DataFolder, $"")
end



Static Function XPSViewAddTag(B_Struct) : ButtonControl
//	Attaches a tag at the position of cursor A
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Extracts the name of the wave cursor A is placed on
		String TraceName=CsrWave(A, B_Struct.win)
	
		//	Checks that cursor A is active and placed on a wave. An inactive cursor returns an empty wavename of ""
		If (StrLen(TraceName)==0)
			ErrorMessage("Use cursor A to select the position of the tag!")
		else
			Tag /A=RC /B=1 /F=0 /L=1 /X=-2 /Y=-2 /W=$B_Struct.win $TraceName, xcsr(A, B_Struct.win), Num2Str(xcsr(A, B_Struct.win))+" eV"
		endif
	endif
	Return 0
end





//     -----<<<<<     Fit Assistant     >>>>>-----
//	This section contains the functions used by the Fit Assistant menu item

Static Function XPSFitAssistant(FolderOrGroup, ActiveName, ForceSelection)
//	Brings the Fit Assistant window to the front, or creates it if it does not exist
String FolderOrGroup, ActiveName
Variable ForceSelection
String ActiveWindow="XPSFitAssistantWindow"

	//	Updates the history of procedures used with the current experiment
	XPSUpdateProcedureHistory()

	//	Does not create the Fit Assistant window if it already exists
	DoWindow /F XPSFitAssistantWindow
	if ((V_Flag==0) || (ForceSelection==1))

		//	Changes the displayed selection
		XPSFitAssChangeSelection(FolderOrGroup, ActiveName, ForceSelection)
	else

		//	Finds the displayed wave
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")

		//	Refreshes the graph
		XPSFitAssUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
	endif
end



Static Function XPSFitAssChangeSelection(FolderOrGroup, ActiveName, ForceSelection)
//	Changes the selection in the Fit Assistant window without killing the window
String FolderOrGroup, ActiveName
Variable ForceSelection
String ActiveWindow="XPSFitAssistantWindow"

	//	Does not create the Fit Assistant window if it already exists
	DoWindow $ActiveWindow
	if (V_Flag==0)

		//	Creates the Fit Assistant window
		XPSFitAssCreateWindow(FolderOrGroup, ActiveName, ForceSelection)
	else
	
		//	Creates a list of graph and panels with names starting with XPSFitAss
		String ListOfWindows=WinList("XPSFitAss*", ";", "WIN:65;VISIBLE:1")
		
		//	Finds the position of the coefficient window in the list
		Variable a=FindListItem("XPSFitAssCoefWindow", ListOfWindows, ";", 0, 1)

		//	Checks if the coefficient window exists
		if (a==-1)
		
			//	Brings the Fit Assistant window to the front.
			DoWindow/F XPSFitAssistantWindow
		else
		
			//	Finds the position of the Fit Assistant window in the list
			Variable b=FindListItem("XPSFitAssistantWindow", ListOfWindows, ";", 0, 1)
			if (a<b)
			
				//	Brings both windows to the front, but keeps the order of the two windows
				DoWindow/F XPSFitAssistantWindow
				DoWindow/F XPSFitAssCoefWindow
			else

				//	Brings both windows to the front, but keeps the order of the two windows
				DoWindow/F XPSFitAssCoefWindow
				DoWindow/F XPSFitAssistantWindow
			endif
		endif

		//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is forced
		PopupMenu DataFolderPopUp popmatch=FolderOrGroup, win=$ActiveWindow
		ControlInfo /W=$ActiveWindow DataFolderPopUp
		if (CmpStr(FolderOrGroup, S_Value)!=0)
			PopupMenu DataFolderPopUp mode=1, popvalue=FolderOrGroup, win=$ActiveWindow
		endif
		
		//	Updates the list of wave selections and select ActiveName, or if ActiveName doesn't exist in the list, the first item in the list is selected
		PopupMenu DisplayedWavePopUp mode=1, popmatch=ActiveName, win=$ActiveWindow

		//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
		if (StrLen(ActiveName)>0)
			ControlInfo /W=$ActiveWindow DisplayedWavePopUp
			if (CmpStr(ActiveName, S_Value)!=0)
				PopupMenu DisplayedWavePopUp mode=1, popvalue=ActiveName, win=$ActiveWindow
			endif
		endif

		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")

		//	Sets the axis scaling and labels to match the type of spectrum displayed and updates the graph
		XPSFitAssUpdateGraphDF(ActiveWindow, ActiveFolder, DisplayedWave)
	endif
end



Static Function XPSFitAssCreateWindow(FolderOrGroup, ActiveName, ForceSelection)
//	Creates a graph used to display and fit XPS data
String FolderOrGroup, ActiveName
Variable ForceSelection
Variable n=120/ScreenResolution
String ActiveWindow="XPSFitAssistantWindow"

	//	Creates the folder to hold all the permanent waves, used for different bureaucratic purposes, such as listboxes, displaying waves, etc...
	NewDataFolder/O root:Programming
	DFREF ProgramFolder=root:Programming
	
	//	Creates s separate folder to hold the waves used for displaying the fit
	NewDataFolder/O ProgramFolder:FitAssFitWaves
	DFREF FitWavesFolder=ProgramFolder:FitAssFitWaves

	//	Kills the Fit Assistant window if it exists together with any old waves that might be around
	DoWindow /K $ActiveWindow
	KillWavesInFolder(FitWavesFolder)		

	//	Creates the folder to hold all the temporary waves, created during a function call and deleted again at the end of the function call.
	NewDataFolder/O root:Programming:Temp

	//	Checks if the coefficient waves exist
	Wave/Z/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	Wave/Z/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel
	if ((WaveExists(CoefWave)==0) || (WaveExists(CoefSelWave)==0))

		//	Creates empty coefficient waves
		XPSFitAssCreateEmptyConstraints(-1)
	else

		//	Creates the small coefficient waves used for the condensed display in the coefficient window
		XPSFitAssCreateSmallCoefWaves()
	endif
	
	//	Checks if the undo waves exists
	DFREF UndoFolder=ProgramFolder:FitAssUndo
	Wave/Z/WAVE UndoWaves=UndoFolder:XPSFitAssUndoWaves
	
	if ((DataFolderRefStatus(UndoFolder)==0) || (WaveExists(UndoWaves)==0))
	
		//	Creates blank undo waves
		XPSFitAssCreateUndoWaves(10)
	else
	
		//	Checks if all waves in the undo wave exists
		Extract/FREE/INDX/O UndoWaves, UndoIndexWave, (WaveExists(UndoWaves[p])==0)
		if (NumPnts(UndoIndexWave)>0)
	
			//	Creates blank undo waves
			XPSFitAssCreateUndoWaves(10)
		endif
	endif
	
	//	Creates an empty wave to display
	Make/O/N=0 ProgramFolder:XPSFitAssDataWave/WAVE=DataWave
	
	//	Creates the graph. To fit on the lab screen it should be no wider than 775*n.
	Display /W=(250, 150, 250+775*n, 150+400*n) /K=1 /N=$ActiveWindow DataWave
	ModifyGraph /W=$ActiveWindow rgb(XPSFitAssDataWave)=(0,0,0), margin(top)=45*n
	ShowInfo /W=$ActiveWindow
	
	//	Associates a hook function with the window. This will be used to clean up waves when the window is killed and to update the window as the cursors are moved
	SetWindow $ActiveWindow hook(HookOne)=XPSFitAssHookFunc

	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$ActiveWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$ActiveWindow B
	
	//	Creates a box with the colour of the wave as it appears in the View Spectra graph
	ValDisplay ColourBox, barmisc={20, 0}, barBackColor=(60000, 60000, 60000), disable=2, frame=4, mode=0, pos={10, 5}, size={20, 20}, value=_NUM:0, win=$ActiveWindow, help={"Displays the colour of the spectrum in the View Spectra window"}
	//	Selects the active data folder, listing root: and all it's subfolders with the exception of root:Programming and it's subfolders. To list the default folder (set with SetDataFolder) and all it's subfolders instead with no exceptions, use XPSViewDataFoldersList(:, $"")
	PopupMenu DataFolderPopUp bodyWidth=250, mode=1, pos={235, 5}, proc=EccentricXPS#XPSViewDataFolderPopupMenu, value=EccentricXPS#XPSViewGroupsAndFoldersList(root:, root:Programming, "SavedFits"), userdata="EccentricXPS#XPSFitAssUseSavedCoefWaves", win=$ActiveWindow, help={"Selects the data folder to look for waves in"}
	//	Popup menu to select the spectrum to display. The complicated value call #(String) indicates that the value ActiveWindow had when the control was created should be used
	PopupMenu DisplayedWavePopUp bodyWidth=200, mode=1, pos={440,5}, proc=EccentricXPS#XPSViewDisplayedWavePopupMenu, value=#("EccentricXPS#XPSViewPopUpWaveList(1, \""+ActiveWindow+"\", \"DataFolderPopUp\")"), userdata="EccentricXPS#XPSFitAssUseSavedCoefWaves", win=$ActiveWindow, help={"Selects the spectrum to display"}
	//	Creates a flat linear background either at the value of cursor A or the average of all points between cursors A and B
	Button OffsetButton pos+={20, 0}, proc=EccentricXPS#XPSFitAssSetOffset, size={70,20}, title="Offset", win=$ActiveWindow, help={"Creates a flat linear background either at the value of cursor A or the average of all points between cursors A and B"}
	//	Calculates a new linear background based on a straight line between cursors A and B
	Button LineButton proc=EccentricXPS#XPSFitAssSetLine, size={70,20}, title="Line", win=$ActiveWindow, help={"Calculates a new linear background based on a straight line between cursors A and B"}
	//	Toggles free/bound cursors (Bound means that the cursors are bound to a wave)
	Button FreeButton proc=EccentricXPS#XPSViewToggleButton, size={70,20}, title="Free", userdata="EccentricXPS#XPSFitAssUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Toggles the cursors between being free or bound to a trace"}
	//	Adds a peak to the fit
	Button AddButton pos+={20, 0}, proc=EccentricXPS#XPSFitAssAdd, size={70,20}, title="Add", win=$ActiveWindow, help={"Adds a peak to the fit at the position of cursor A"}
	//	Removes the last added peak from the fit. If no peaks exist, the background will be removed as well
	Button RemoveButton proc=EccentricXPS#XPSFitAssRemove, size={70,20}, title="Remove", win=$ActiveWindow, help={"Removes the last added peak from the fit"}
	//	Removes the background and all peaks from the fit
	Button ClearButton proc=EccentricXPS#XPSFitAssClear, size={70,20}, title="Clear", win=$ActiveWindow, help={"Removes the background and all peaks from the fit"}
	//	Brings the constraints table to the front
	Button CoefButton proc=EccentricXPS#XPSFitAssCoefButton, size={70,20}, title="Coef", win=$ActiveWindow, help={"Displays the fit coefficients"}
	//	Opens the selected group, or the selected data folder and spectrum in View Spectra
	Button SpectraButton pos+={20, 0}, proc=EccentricXPS#XPSFitAssSpectra, size={70,20}, title="Spectra", win=$ActiveWindow, help={"Opens the selected group, or the selected data folder and spectrum in View Spectra"}
	//	Displays the help file
	Button HelpButton proc=EccentricXPS#XPSViewHelp, size={70,20}, title="Help", userdata="Fit Assistant", win=$ActiveWindow, help={"Opens the help file"}
	//	Creates a table with the displayed spectrum and fit waves. Useful for copying and pasting into other programs
	Button EditWavesButton pos={600,30}, proc=EccentricXPS#XPSFitAssEditWaves, size={70,20}, title="Table", win=$ActiveWindow, help={"Creates a table with the displayed spectrum and fit waves. Useful for copying and pasting into other programs"}
	//	Selects the displayed spectrum type
	PopupMenu DisplayTypePopUp bodyWidth=70, mode=5, pos={700,30}, proc=EccentricXPS#XPSViewDisplayTypePopup, value="AES;IR;NEXAFS;QMS;XPS;", win=$ActiveWindow, help={"Selects the displayed spectrum type"}

	//	Starts the fit
	Button FitButton pos={780, 30}, proc=EccentricXPS#XPSFitAssFitButton, size={70,20}, title="Fit", win=$ActiveWindow, help={"Starts the fit"}
	//	Restores the coefficients from before the fit
	Button UndoButton proc=EccentricXPS#XPSFitAssUndoButton, size={70,20}, title="Undo", win=$ActiveWindow, help={"Restores the previous coefficient values"}
	//	Creates a single printer-friendly page with the fit and coefficients
	Button ReportButton proc=EccentricXPS#XPSFitAssReport, size={70,20}, title="Report", win=$ActiveWindow, help={"Creates a printable page with the fit and the fit coefficients"}
	//	Fits all spectra in the displayed group or folder
	Button FitAllButton proc=EccentricXPS#XPSFitAssFitButton, size={70,20}, title="Fit All", win=$ActiveWindow, help={"Fits all spectra in the displayed group or folder"}
	//	Opens the selected group or data folder in Fit Overview
	Button OverviewButton proc=EccentricXPS#XPSFitAssOverview, pos+={20, 0}, size={70,20}, title="Overview", win=$ActiveWindow, help={"Opens the selected group or data folder in Fit Overview"}
	//	Creates a copy of the graph, without any controls, meant as a starting point for a more presentable figure
	Button NewWinButton proc=EccentricXPS#XPSViewNewWin, size={70,20}, title="New Win", win=$ActiveWindow, help={"Creates a copy of the graph, without any controls, meant as a starting point for a more presentable figure"}


	//	A series of checkboxes and a pop up menu to choose if the next added peak should be linked to any of the previous peaks
	CheckBox PositionCheckBox, mode=0, noproc, pos={40*1.25/n, 30}, side=1, title="Link position", value=0, win=$ActiveWindow, help={"Links the position of the next peak to that of the selected peak"}
	CheckBox AreaCheckBox, mode=0, noproc, pos={115*1.25/n, 30}, side=1, title="area", value=0, win=$ActiveWindow, help={"Links the area of the next peak to that of the selected peak"}
	CheckBox FWHMCheckBox, mode=0, noproc, pos={215*1.25/n, 30}, side=1, title="FWHM", value=0, win=$ActiveWindow, help={"Links the FWHM of the next peak to that of the selected peak"}
	CheckBox GLCheckBox, mode=0, noproc, pos={40*1.25/n, 46}, side=1, title="GL ratio", value=0, win=$ActiveWindow, help={"Links the GL ratio of the next peak to that of the selected peak"}
	CheckBox AsymCheckBox, mode=0, noproc, pos={115*1.25/n, 46}, side=1, title="asymmetry", value=0, win=$ActiveWindow, help={"Links the asymmetry of the next peak to that of the selected peak"}
	CheckBox ShirleyCheckBox, mode=0, noproc, pos={215*1.25/n, 46}, side=1, title="Shirley bck.", value=1, win=$ActiveWindow, help={"Links the Shirley background coefficient of the next peak to that of the selected peak"}
	PopupMenu LinkPopUp bodyWidth=50, mode=1, proc=EccentricXPS#XPSFitAssLinkPopupMenu, pos={315*1.25/n, 35}, title="to peak", userdata="0", value=EccentricXPS#XPSFitAssLinkList(), win=$ActiveWindow, help={"Links the next peak to the selected peak"}

	//	Enables asymmetry for the next added peak
	CheckBox EnableAsymCheckBox, mode=0, noproc, pos={440*1.25/n, 30}, side=1, title="Asymmetric peak", value=0, win=$ActiveWindow, help={"Allows the next peak to be asymmetric"}

	//	Changes the display style of the fitted spectrum
	CheckBox FancyPlotCheckBox, mode=0, proc=EccentricXPS#XPSViewToggleCheckBox, pos={440*1.25/n, 46}, side=1, title="Fancy Plot", value=0, userdata="EccentricXPS#XPSFitAssUpdateGraph", win=$ActiveWindow, help={"Changes the display style of the fitted spectrum"}

	//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or the default data folder is used instead, depending on the value of ForceSelection
	PopupMenu DataFolderPopUp popmatch=FolderOrGroup, win=$ActiveWindow
	ControlInfo /W=$ActiveWindow DataFolderPopUp
	if (CmpStr(FolderOrGroup, S_Value)!=0)
		if (ForceSelection==1)
			PopupMenu DataFolderPopUp mode=1, popvalue=FolderOrGroup, win=$ActiveWindow
		else
			PopupMenu DataFolderPopUp popmatch=GetDataFolder(1), win=$ActiveWindow
		endif
	endif
	
	//	Updates the list of wave selections and select ActiveName, or if ActiveName doesn't exist in the list, the first item in the list is selected
	PopupMenu DisplayedWavePopUp mode=1, popmatch=ActiveName, win=$ActiveWindow

	//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
	if (StrLen(ActiveName)>0)
		ControlInfo /W=$ActiveWindow DisplayedWavePopUp
		if (CmpStr(ActiveName, S_Value)!=0)
			if (ForceSelection==1)
				PopupMenu DisplayedWavePopUp mode=1, popvalue=ActiveName, win=$ActiveWindow
			endif
		endif
	endif

	DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
	Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
	
	//	Updates the Fit Assistant window
	XPSFitAssUseSavedCoefWaves(ActiveWindow, ActiveFolder, DisplayedWave)
end



Static Function XPSFitAssLinkPopupMenu(PU_Struct) : PopupMenuControl
//	Changes the userdata when the popup menu selection is actively changed
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Saves the last active selection in the userdata
		PU_Struct.userdata=PU_Struct.popStr
	endif
	Return 0
end



Function XPSFitAssHookFunc(s)
//	The hook function for the Fit Assistant window. This is used to update the graph when the cursors are moved and for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:
		
			//	Indicates that an action has taken place
			hookResult=1
			
			//	Kills the coefficient and edit waves windows
			DoWindow/K XPSFitAssCoefWindow
			DoWindow/K XPSFitAssEditWavesWindow
			
			//	Kills the waves associated with the Fit Assistant window, except the coefficient waves. Execute/P delays the execution until the window has been killed
			Execute/P/Q "KillDataFolder/Z root:Programming:FitAssFitWaves"
			Execute/P/Q "KillDataFolder/Z root:Programming:FitAssUndo"
			Execute/P/Q "KillWaves/Z root:Programming:XPSFitAssCoefCSmall, root:Programming:XPSFitAssCoefCSelSmall, root:Programming:XPSFitAssDataWave, root:Programming:XPSFitAssColours, root:Programming:XPSFitAssUndoWaves"
			break

		//	MouseUp (s.eventCode = 5): the mouse button was released. This is used to detect the user moving the in-graph cursors. This works better than CursorMoved (s.eventCode = 7), since the update for CursorMoved is continueous for free cursors. The only problem is that it doesn't catch when a cursor is removed from the graph
		case 5:
		
			//	Indicates that an action has taken place
			hookResult=1

			//	Finds the wave to display
			DFREF ActiveFolder=XPSViewGetDataFolder(s.winName, "DataFolderPopUp")
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(s.winName, ActiveFolder, "DisplayedWavePopUp")
			
			//	Updates the Fit Assistant window
			XPSFitAssUpdateGraph(s.winName, ActiveFolder, DisplayedWave)
			break
			
		//	mouseWheel (s.eventCode=22) The mouse wheel was rolled either up (s.wheelDy=1) or down (s.wheelDy=-1)
		case 22:
		
			//	Ignores mouse wheel rolls in the upper part of the graph where the popup menus are
			if (s.mouseLoc.v>(45*120/ScreenResolution+10))
			
				//	Finds the list of waves in the popup menu
				String ListOfWaves=XPSViewPopUpWaveList(1, s.winName, "DataFolderPopUp")
				Variable NumberOfWaves=ItemsInList(ListOfWaves, ";")

				//	Finds the currently selected wave. This does not update the list of existing waves in the popup menu.
				ControlInfo /W=$s.winName DisplayedWavePopUp
				Variable CurrentSelectionNum=WhichListItem(S_Value, ListOfWaves, ";", 0, 1)
				
				//	Sets the new selection to the first item in the list if the current selection no longer exists
				Variable NewSelectionNum=0
				if (CurrentSelectionNum>-1)
				
					//	Changes the selection by -1 or +1	
					NewSelectionNum=CurrentSelectionNum-s.wheelDy
				endif
				
				//	Checks if the new selection is valid
				if ((NewSelectionNum>=0) && (NewSelectionNum<NumberOfWaves))

					//	Finds the name of the new selection
					String NewSelectionStr=StringFromList(NewSelectionNum, ListOfWaves, ";")

					//	Updates the selected wave. This will also update the list of waves
					PopupMenu DisplayedWavePopUp mode=1, popmatch=NewSelectionStr, win=$s.winName

					DFREF ActiveFolder=XPSViewGetDataFolder(s.winName, "DataFolderPopUp")
					Wave/Z DisplayedWave=XPSViewGetDisplayedWave(s.winName, ActiveFolder, "DisplayedWavePopUp")
				
					//	Checks if shift was pressed
					if ((s.eventMod & 2)==2)
				
						//	Updates the Fit Assistant window without changing the fit coefficients
						XPSFitAssUpdateGraph(s.winName, ActiveFolder, DisplayedWave)
					else

						//	Updates the Fit Assistant window
						XPSFitAssUseSavedCoefWaves(s.winName, ActiveFolder, DisplayedWave)
					endif
				endif
			endif
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function XPSFitAssSpectra(B_Struct) : ButtonControl
//	Opens the selected group, or the selected data folder and spectrum in View Spectra
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Finds the selected data folder or group
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		String FolderOrGroup=S_Value
		DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
		
		//	Displays the folder or group in the View Spectra window
		XPSViewSpectraChangeSelection(FolderOrGroup, 1)
		
		//	If a data folder was selected instead of a group. The displayed wave is selected in the View Spectra window
		if (CmpStr(FolderOrGroup, GetDataFolder(1, ActiveFolder))==0)
		
			//	Finds the displayed wave in the Fit Assistant window
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")
			
			if ((DataFolderRefStatus(ActiveFolder)!=0) && WaveExists(DisplayedWave))

				//	Creates the necessary waves for the View Spectra listbox
				DFREF ProgramFolder=root:Programming
				Make/T/O ProgramFolder:XPSViewWaveList={{NameOfWave(DisplayedWave)}, {""}, {""}}
				Make/I/U/O/N=(1, 3, 2) ProgramFolder:XPSViewWaveListSel=0
				Make/WAVE/O ProgramFolder:XPSViewWaveListWaves={DisplayedWave}
			
				//	Defines the first column in the listbox as a checked (bit 4) checkbox (bit 5)
				Wave/I/U XPSViewWaveListSel=ProgramFolder:XPSViewWaveListSel
				XPSViewWaveListSel[0][0][0]=48
	
				//	Makes the colours affect the background colour, change to forecolors to affect foreground (text) colours
				SetDimLabel 2,1, backColors, XPSViewWaveListSel
			
				//	Refreshes the View Spectra graph
				XPSViewSpectraUpdateGraph("XPSViewSpectraWindow", ActiveFolder, $"")
			endif
		endif
	endif
	Return 0
end



Static Function XPSFitAssOverview(B_Struct) : ButtonControl
//	Opens the selected group or data folder in Fit Overview
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Finds the selected data folder or group
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		String FolderOrGroup=S_Value
		DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
		
		//	Displays the folder or group in the Fit Overview window
		XPSFitOverview(FolderOrGroup)
	endif
	Return 0
end



Static Function XPSFitAssEditWaves(B_Struct) : ButtonControl
//	Creates a table with the displayed spectrum and fit waves. Useful for copying and pasting into other programs
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Creates the edit waves window, or bring it to the front if it already exists
		String ActiveWindow="XPSFitAssEditWavesWindow"
		DoWindow/F $ActiveWindow
		if (V_flag==0)
		
			//	Finds the displayed spectrum
			DFREF ProgramFolder=root:Programming
			Wave DataWave=ProgramFolder:XPSFitAssDataWave

			//	Creates a table with the measured spectrum including binding energy values
			Variable n=120/ScreenResolution
			Edit /K=1 /N=$ActiveWindow /W=(150, 50, 150+775*n, 50+150*n) DataWave.id
			ModifyTable /W=$ActiveWindow alignment=2
			
			//	Appends the fit waves to the Edit Waves table
			XPSFitAssEditAddFitWaves()
		endif
	endif
	Return 0
end



Static Function XPSFitAssEditAddFitWaves()
//	Appends the fit waves to the Edit Waves table
String ActiveWindow="XPSFitAssEditWavesWindow"

	//	Checks if the Edit Waves window exists
	DoWindow $ActiveWindow
	if (V_Flag!=0)
			
		//	Finds the basic fit waves
		DFREF ProgramFolder=root:Programming
		DFREF FitWavesFolder=ProgramFolder:FitAssFitWaves
		Wave/Z FitWave=FitWavesFolder:XPSFitAssFitWave
		Wave/Z BckWave=FitWavesFolder:XPSFitAssBckWave
		
		//	Appends the basic fit waves to the table, if they exist
		if (WaveExists(FitWave) && WaveExists(BckWave))
			AppendToTable /W=$ActiveWindow FitWave.id, BckWave.d
		endif
				
		//	Appends the peaks to the table, one at a time
		Variable i=0		
		Wave/Z PeakWave=FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(0))
		for (i=1; WaveExists(PeakWave); i+=1)
			AppendToTable /W=$ActiveWindow PeakWave.d
			Wave/Z PeakWave=FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(i))
		endfor

		//	Autoscales the columns in the table
		ModifyTable /W=$ActiveWindow autosize={0, 0, -1, 0.1, 1}
	endif
end



Static Function XPSFitAssEditRemoveFitWaves()
//	Removes the fit waves from the Edit Waves table
String ActiveWindow="XPSFitAssEditWavesWindow"

	//	Checks if the Edit Waves window exists
	DoWindow $ActiveWindow
	if (V_Flag!=0)

		//	Finds the basic fit waves
		DFREF ProgramFolder=root:Programming
		DFREF FitWavesFolder=ProgramFolder:FitAssFitWaves
		Wave/Z FitWave=FitWavesFolder:XPSFitAssFitWave
		Wave/Z BckWave=FitWavesFolder:XPSFitAssBckWave

		//	Removes the basic fit waves from the table, if they exist
		if (WaveExists(FitWave) && WaveExists(BckWave))
			RemoveFromTable /W=$ActiveWindow FitWave.id, BckWave.d
		endif
				
		//	Removes the peaks from the table, one at a time
		Variable i=0
		Wave/Z PeakWave=FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(0))
		for (i=1; WaveExists(PeakWave); i+=1)
			RemoveFromTable /W=$ActiveWindow PeakWave.d
			Wave/Z PeakWave=FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(i))
		endfor
	endif
end



Static Function XPSFitAssCoefButton(B_Struct) : ButtonControl
//	Creates the constraints table or brings it to the front if it already exists
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Creates the fit coefficient window, or bring it to the front if it already exists
		DoWindow/F XPSFitAssCoefWindow
		if (V_flag==0)
			XPSFitAssCreateCoefWindow("Expand")
		endif
	endif
	Return 0
end



Static Function XPSFitAssCreateCoefWindow(DisplayType)
//	Creates the constraints table. Screen resolution is needed to contro the size of the table, since the size of some objects are in points and others in pixels (which is very confusing)
String DisplayType
Variable n=ScreenResolution/72
String ActiveWindow="XPSFitAssCoefWindow"
	
	//	Creates the wave used to define the colours of the listbox
	DFREF ProgramFolder=root:Programming
	Make /W/U/O/N=(3, 4) ProgramFolder:XPSFitAssColours={{0, 0, 0}, {65000, 58000,45000}, {60000, 60000, 65535}, {51000, 51000, 65535}}	//	Orange, Light Blue, Dark Blue
	Wave/W/U Colours=ProgramFolder:XPSFitAssColours
	MatrixTranspose Colours

	//	The setting used for the expanded display
	Variable ListBoxY=30
	Variable ExportCoefsX=5+90+20+90*3+20, ExportCoefsY=5
	Variable PanelWidth=950*n, PanelHeight=250*n
	String ExpandButtonUserData="Condense"
	
	//	Checks if the condensed or expanded display should be created
	if (CmpStr(DisplayType, "Condense")==0)
	
		//	The setting used for the condensed display
		ListBoxY=30+25
		ExportCoefsX=5+90+20
		ExportCoefsY=30
		PanelWidth=370*n
		PanelHeight=250*n
		ExpandButtonUserData="Expand"
		
		//	Changes the displayed coefficient waves to the small waves without constraints
		XPSFitAssCreateSmallCoefWaves()
		Wave/T CoefWave=ProgramFolder:XPSFitAssCoefCSmall
		Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefCSelSmall
	else
	
		//	The coefficient waves used for the listbox
		Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
		Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel
	endif

	//	Creates a table with the constraints. If the table already exists its position is kept
	GetWindow/Z $ActiveWindow, wSize
	if (V_Flag==0)
		DoWindow /K $ActiveWindow
		NewPanel /K=1 /N=$ActiveWindow /W=(V_left*n, V_top*n, V_left*n+PanelWidth, V_top*n+PanelHeight)
	else
		NewPanel /K=1 /N=$ActiveWindow /W=(10, 40, 10+PanelWidth, 40+PanelHeight)
	endif
	
	//	Creates a list box with the constraints in the condensed mode
	ListBox CoefListBox, colorWave=Colours, editStyle=0, frame=0, listWave=CoefWave, mode=0, proc=EccentricXPS#XPSFitAssCoefListBoxProc, pos={0, ListBoxY}, selWave=CoefSelWave, size={PanelWidth, PanelHeight-ListBoxY}, special={0, 0, 1 }, win=$ActiveWindow, help={"The fit coefficients"}
	//	Switches between showing everything or just the fit values
	Button ExpandButton proc=EccentricXPS#XPSFitAssCoefExpand, pos={5,5}, size={80, 20}, title=ExpandButtonUserData, userdata=ExpandButtonUserData, win=$ActiveWindow, help={"Toggles between displaying the constraints or just the fit parameters"}
	//	Stores the displayed coefficients
	Button CopyButton pos+={20,0}, proc=EccentricXPS#XPSFitAssCoefCopy, size={80,20}, title="Copy", win=$ActiveWindow, help={"Stores the displayed coefficients"}
	//	Overwrites the coefficients with the stored coefficients
	Button PasteButton proc=EccentricXPS#XPSFitAssCoefPaste, size={80,20}, title="Paste", win=$ActiveWindow, help={"Overwrites the coefficients with the stored coefficients"}
	//	Removes the background and all peaks from the fit
	Button ClearButton proc=EccentricXPS#XPSFitAssClear, size={80,20}, title="Clear", win=$ActiveWindow, help={"Removes the background and all peaks from the fit"}
	//	Exports the coefficients as a text file
	Button ExportCoefsButton pos={ExportCoefsX,ExportCoefsY}, proc=EccentricXPS#XPSFitAssExportCoefs, size={80,20}, title="Export", win=$ActiveWindow, help={"Exports the coefficients as a text file"}
	//	Imports the coefficients from a text file
	Button ImportCoefsButton, proc=EccentricXPS#XPSFitAssImportCoefs, size={80,20}, title="Import", win=$ActiveWindow, help={"Imports the coefficients from a text file"}
end



Static Function XPSFitAssCoefCopy(B_Struct) : ButtonControl
//	Stores the displayed coefficients to be pasted later
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	The coefficient waves to copy
		DFREF ProgramFolder=root:Programming
		Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
		Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel
		
		//	The target folder
		DFREF UndoFolder=ProgramFolder:FitAssUndo

		//	Copies the waves into the undo folder
		Duplicate/O CoefWave UndoFolder:XPSFitAssCopyPasteCoef
		Duplicate/O CoefSelWave UndoFolder:XPSFitAssCopyPasteCoefSel
	endif
	Return 0
end



Static Function XPSFitAssCoefPaste(B_Struct) : ButtonControl
//	Overwrites the coefficients with the stored coefficients
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	The coefficient waves to copy
		DFREF ProgramFolder=root:Programming
		DFREF UndoFolder=ProgramFolder:FitAssUndo
		Wave/Z/T CoefWave=UndoFolder:XPSFitAssCopyPasteCoef
		Wave/Z/B/U CoefSelWave=UndoFolder:XPSFitAssCopyPasteCoefSel

		//	Checks if the stored coefficient waves exist
		if ((DataFolderRefStatus(ProgramFolder)!=0) && (DataFolderRefStatus(UndoFolder)!=0) && WaveExists(CoefWave) && WaveExists(CoefSelWave))
		
			//	Copies the waves into the undo folder
			Duplicate/O CoefWave ProgramFolder:XPSFitAssCoefConstraints
			Duplicate/O CoefSelWave ProgramFolder:XPSFitAssCoefConstraintsSel

			//	Creates the small coefficient waves used for the condensed display in the coefficient window
			XPSFitAssCreateSmallCoefWaves()
		
			//	Finds the wave to display
			DFREF ActiveFolder=XPSViewGetDataFolder("XPSFitAssistantWindow", "DataFolderPopUp")
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave("XPSFitAssistantWindow", ActiveFolder, "DisplayedWavePopUp")
				
			//	Updates the Fit Assistant window
			 XPSFitAssUpdateGraph("XPSFitAssistantWindow", ActiveFolder, DisplayedWave)
		endif
	endif
	Return 0
end



Static Function XPSFitAssCoefExpand(B_Struct) : ButtonControl
//	Expands and condenses the CoefConstraints window
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Toggles the condensed and expanded display modes
		XPSFitAssCreateCoefWindow(B_Struct.userdata)
	endif
	Return 0
end



Static Function XPSFitAssCoefListBoxProc(LB_Struct) : ListboxControl
//	Updates the values in the CoefConstraints wave when values are edited in the condensed mode
STRUCT WMListboxAction &LB_Struct

	//	If a value was edited
	if (LB_Struct.eventCode==7)
	
		//	Ignores further calls to the listbox procedure until this one has finished
		LB_Struct.blockReentry=1
		
		DFREF ProgramFolder=root:Programming
		Wave/T FullCoefWave=ProgramFolder:XPSFitAssCoefConstraints
		
		//	Checks if the condensed version of the window is active
		if (WaveRefsEqual(FullCoefWave, LB_Struct.listWave)==0)
		
			//	Copies the values from the small wave to the full coefficient wave
			FullCoefWave[][0]=LB_Struct.listWave[p][0]
			FullCoefWave[][1,25;4]=LB_Struct.listWave[p][(q-1)/4+1]		//	[1, 25;4] means the range from 1-25 in steps of 4, e.g. 1, 5, 9, 13, 17, 21, 25
		endif
		
		//	Finds the wave to display
		DFREF ActiveFolder=XPSViewGetDataFolder("XPSFitAssistantWindow", "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave("XPSFitAssistantWindow", ActiveFolder, "DisplayedWavePopUp")
		
		//	Updates the Fit Assistant window
		XPSFitAssUpdateGraph("XPSFitAssistantWindow", ActiveFolder, DisplayedWave)
	endif
end



Static Function XPSFitAssExportCoefs(B_Struct) : ButtonControl
//	Exports the XPSFitAssCoefConstraints wave with all the coefficients to a text file
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up (mouse down, eventCode 1, in combination with Open seems to bug out the button )
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		DFREF ProgramFolder=root:Programming
		Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
		Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel
	
		//	Suggests a filename based on the wave selected
		ControlInfo /W=XPSFitAssistantWindow DisplayedWavePopUp
		String FileName=S_value+".fit"
	
		//	Opens a dialog where the user can select a filename
		Variable refNum=0
		Open /D/F="Coefficient Files (*.fit):.fit;All Files (*.*):.*;" /P=EccentricXPSDataPath /M="Save Coefficients" refNum as FileName
	
		//	If a filename was selected...
		If (CmpStr(S_fileName, "")!=0)
	
			//	Creates a temporary copy of the fit coefficients
			Duplicate/O/T/FREE CoefWave SaveWave

			//	Copies the hold parameters into the temporary wave
			if (DimSize(SaveWave, 0)>=2)
				SaveWave[1][2,6;4]=Num2iStr((CoefSelWave[p][q][0]&2^4)/(2^4))
				if (DimSize(SaveWave, 0)>=5)
					SaveWave[4,*][2,26;4]=Num2iStr((CoefSelWave[p][q][0]&2^4)/(2^4))
				endif
			endif
	
			//	Saves the temporary wave with the coefficients and hold parameters
			Save /A=0/J/M="\r\n" /O SaveWave as S_fileName
		endif
	endif
	Return 0
end



Static Function XPSFitAssImportCoefs(B_Struct) : ButtonControl
//	Imports the XPSFitAssCoefConstraints wave with all the coefficients from a text file
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up (mouse down, eventCode 1, in combination with Open seems to bug out the button )
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Display an open file dialog, showing only files with the extension .fit
		Variable refNum=0
		Open/D/R/F="Coefficient Files (*.fit):.fit;All Files (*.*):.*;" /P=EccentricXPSDataPath /M="LoadCoefficients" RefNum

		//	If a file was selected
		if (CmpStr(S_FileName, "")!=0)
		
			//	Saves the active data folder
			DFREF CurrentFolder=GetDataFolderDFR()
			
			DFREF ProgramFolder=root:Programming
			DFREF TempFolder=ProgramFolder:Temp

			//	Loads the file specified in the open dialog
			SetDataFolder TempFolder
			LoadWave /N=XPSFitAssLoadedCoefWave /K=2 /L={0,0,0,0,29} /J/M/O/Q/U={0,0,0,0} S_FileName
			SetDataFolder CurrentFolder
			
			//	Checks if a wave was loaded
			Wave/Z/T LoadedWave=TempFolder:XPSFitAssLoadedCoefWave0
			if (WaveExists(LoadedWave))
			
				//	Finds the number of peaks in the saved fit
				Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(LoadedWave)

				//	Creates empty coefficient waves for the listbox
				XPSFitAssCreateEmptyConstraints(NumberOfPeaks)
				Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel
				
				//	Overwrites the coefficient wave with the loaded wave
				Duplicate/O/T LoadedWave, ProgramFolder:XPSFitAssCoefConstraints/WAVE=CoefWave
				
				//	Reads the hold values from the loaded constraints wave
				CoefSelWave[1][2,6;4][0]=2^5+2^4*Str2Num(CoefWave[p][q])
				
				//	Removes the hold values from the coefficient wave
				CoefWave[1][2,6;4]=""

				If (NumberOfPeaks>0)
				
					//	Reads the hold values from the loaded constraints wave
					CoefSelWave[4,3+NumberOfPeaks][2,26;4][0]=2^5+2^4*Str2Num(CoefWave[p][q])
					
					//	Removes the hold values from the coefficient wave
					CoefWave[4,*][2,26;4]=""
				endif
				
				//	Kills all temporary waves
				KillWavesInFolder(TempFolder)
				
				//	Finds the wave to display
				DFREF ActiveFolder=XPSViewGetDataFolder("XPSFitAssistantWindow", "DataFolderPopUp")
				Wave/Z DisplayedWave=XPSViewGetDisplayedWave("XPSFitAssistantWindow", ActiveFolder, "DisplayedWavePopUp")
				
				//	Updates the Fit Assistant window
				 XPSFitAssUpdateGraph("XPSFitAssistantWindow", ActiveFolder, DisplayedWave)
			endif
		endif
	endif
	Return 0
end



Static Function XPSFitAssSaveCoefs(ActiveWindow, ActiveFolder, DisplayedWave)
//	Saves the displayed coefficients as the fit associated with the wave
String ActiveWindow
DFREF ActiveFolder
Wave/Z DisplayedWave

	//	Finds the active coefficient and fit waves
	DFREF ProgramFolder=root:Programming
	Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel

	//	Finds the number of peaks in the coefficient wave
	Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)
		
	if (WaveExists(DisplayedWave))

		//	Finds name of the displayed wave
		String DisplayedWaveName=NameOfWave(DisplayedWave)
			
		//	Removes all waves from the View Spectra Edit Waves table, if it exists. This will allow saved fit waves to be deleted
		XPSViewEditRemoveWaves()
			
		//	Saves all tags and removes all traces from the View Spectra graph, if it exists (except the empty placeholder wave)
		//	This will allow saved fit waves to be deleted during the fitting process.
		XPSViewRemoveAllTraces()

		//	Creates the data folder for the saved fit. Kills any old folder and all waves within which may already exist
		DFREF FitFolder=XPSFitAssSaveFitFolders(DisplayedWave)
		
		//	Leaves the fit folder empty if the coefficient waves are empty
		if (NumberOfPeaks>=0)

			//	Copies the fit coefficient waves into the subfolder
			Duplicate/O/T CoefWave, FitFolder:$(DisplayedWaveName+"_lst")
			Duplicate/O/B/U CoefSelWave, FitFolder:$(DisplayedWaveName+"_sel")

			//	Copies the base fit waves
			DFREF FitWavesFolder=ProgramFolder:FitAssFitWaves
			Duplicate/O FitWavesFolder:XPSFitAssBckWave, FitFolder:$(DisplayedWaveName+"_bck")
			Duplicate/O FitWavesFolder:XPSFitAssFitWave, FitFolder:$(DisplayedWaveName+"_fit")

			//	Copies the peak waves
			Variable i=0
			for (i=0; i<NumberOfPeaks; i+=1)
				Duplicate/O FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(i)), FitFolder:$(DisplayedWaveName+"_pk"+Num2iStr(i))
			endfor
		endif
		
		//	Updates the View Spectra window if it exists
		DoWindow XPSViewSpectraWindow
		if (V_Flag!=0)
		
			//	Finds the selected data folder
			ControlInfo /W=XPSViewSpectraWindow DataFolderPopUp
			DFREF SpectraFolder=$S_Value
		
			//	Updates the View Spectra window
			XPSViewSpectraUpdateGraph("XPSViewSpectraWindow", SpectraFolder, $"")
		endif
	endif
	Return 0
end



Static Function XPSFitAssNumberOfPeaks(CoefWave)
//	Calculates the number of peaks in the coefficient wave. -1 means not even the linear background exists
Wave/Z CoefWave
	if (WaveExists(CoefWave))
		Return Max(-1, DimSize(CoefWave, 0)-7)
	else
		Return NaN
	endif
end



Static Function XPSFitAssClear(B_Struct) : ButtonControl
//	Removes the background and all peaks from the fit
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Removes the background and all peaks from the fit
		XPSFitAssCreateEmptyConstraints(-1)
		
		//	Finds the wave to display
		String ActiveWindow="XPSFitAssistantWindow"
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
		
		//	Updates the Fit Assistant window
		XPSFitAssUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
	endif
	Return 0
end



Static Function XPSFitAssCreateSmallCoefWaves()
//	Creates the small coef waves for the condensed coefficient window display

	//	The coefficient waves used for the coefficient window listbox
	DFREF ProgramFolder=root:Programming
	Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel

	//	The number of rows in the coefficient waves
	Variable NumberOfRows=DimSize(CoefWave, 0)
	
	//	Finds the number of peaks in the saved fit
	Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)
	
	//	Calculates the combined area of all peaks.
	Variable TotalArea=0, i=0
	for (i=0; i<NumberOfPeaks; i+=1)
		TotalArea+=Str2Num(CoefWave[i+4][5])
	endfor
	
	//	Adds the combined area to the full coefficient wave
	CoefWave[NumberOfRows-1][5]=Num2Str(TotalArea)
	
	//	Creates the small coefficient waves for the reduced listbox
	Make/T/O/N=(NumberOfRows, 8) ProgramFolder:XPSFitAssCoefCSmall/WAVE=SmallCoefWave
	Make/B/U/O/N=(NumberOfRows, 8, 2) ProgramFolder:XPSFitAssCoefCSelSmall/WAVE=SmallCoefSelWave
	
	//	Makes the colours affect the background colour
	SetDimLabel 2,1,backColors, SmallCoefSelWave
	
	//	Copies the values from the full to the small coefficient waves, excluding the contraint and hold columns
	if (NumberOfRows>0)
		SmallCoefWave=""
		SmallCoefWave[][0]=CoefWave[p][0]
		SmallCoefWave[][1,*]=CoefWave[p][4*(q-1)+1]
		
		FastOP SmallCoefSelWave=0
		SmallCoefSelWave[][0][]=CoefSelWave[p][0][r]
		SmallCoefSelWave[][1,*][]=CoefSelWave[p][4*(q-1)+1][r]
	endif
end



Static Function XPSFitAssCreateEmptyConstraints(NumberOfPeaks)
//	Creates the skeleton of the coeffficient waves for the listbox. The actual parameters still have to be filled in.
Variable NumberOfPeaks

	//	Calculates the number of rows the coefficient waves should have
	Variable NumberOfRows=0
	if (NumberOfPeaks>=0)
		NumberOfRows=NumberOfPeaks+7
	endif

	//	Creates the wave holding the formatting information for the CoefConstraints listbox
	DFREF ProgramFolder=root:Programming
	Make/B/U/O/N=(NumberOfRows, 29, 2) ProgramFolder:XPSFitAssCoefConstraintsSel/WAVE=CoefSelWave
	
	//	Makes the colours affect the background colour
	SetDimLabel 2, 1, backColors, CoefSelWave

	if (NumberOfRows>0)
	
		//	Creates the huge two-dimensional text wave used to store the constraints and initial guesses. The rows and columns are flipped to make it easier to load values into it
		Make/O/T/N=(29, NumberOfRows) ProgramFolder:XPSFitAssCoefConstraints/WAVE=CoefWave
		CoefWave=""

		FastOP CoefSelWave=0
	
		//	Header for the linear background
		CoefWave[1][0]={"Intercept", " h ", "min", "max", "Slope", " h ", "min", "max"}

		//	Header for the peaks
		CoefWave[1][3]={"Position", " h ", "min", "max", "Area", " h ", "min", "max", "FWHM", " h ", "min", "max", "GL", " h ", "min", "max", "Asym1", " h ", "min", "max", "Asym2", " h ", "min", "max", "Shirley", " h ", "min", "max"}

		//	Sets the background colous of the Intercept and Slope headers to dark blue
		CoefSelWave[0][1][1]=3
		CoefSelWave[0][5][1]=3
		
		//	Sets the background colous of the position, area, FWHM, etc... headers to be the same dark blue
		CoefSelWave[3][1,25;4][1]=3		//	[1,25;4] means from 1 to 25 in steps of 4

		//	Makes the intercept and slope coefficient boxes editable
		CoefSelWave[1][1,8][0]=2
	
		//	Defines the intercept and slope hold boxes as unchecked checkboxes (sets bit 5)
		CoefSelWave[1][2,6;4][0]=32

		//	Sets the background colour of the intercept and slope min, max and hold values to orange
		CoefSelWave[1][1,8][1]=1

		//	Sets the background colous of the Intercept and Slope values to be light blue
		CoefSelWave[1][1,5;4][1]=2
		
		//	Creates a box below the individual area of the peak with the combined area of all the peaks
		CoefWave[5][NumberOfRows-2]="Total"
		CoefSelWave[NumberOfRows-2][5][1]=3
		CoefSelWave[NumberOfRows-1][5][1]=2
		
		//	Makes the combined peak-area box editable, to make it possible to copy the value
		CoefSelWave[NumberOfRows-1][5][0]=2

		if (NumberOfPeaks>0)
	
			//	Enters the peak numbers
			CoefWave[0][4,NumberOfRows-4]=Num2iStr(q-4)

			//	Makes the peak parameter boxes editable
			CoefSelWave[4,NumberOfRows-4][1,28][0]=2
		
			//	Defines the peak hold boxes as unchecked checkboxes (sets bit 5)
			CoefSelWave[4,NumberOfRows-4][2,26;4][0]=32

			//	Sets the background colour of the intercept and slope min, max and hold values to orange
			CoefSelWave[4,NumberOfRows-4][1,28][1]=1

			//	Sets the background colous of the Position, Area, FWHM, etc.. values to be light blue
			CoefSelWave[4,NumberOfRows-4][1,25;4][1]=2
		endif

		//	Flips the rows and columns
		MatrixTranspose CoefWave
	else

		//	Creates the huge two-dimensional text wave used to store the constraints and initial guesses.
		Make/O/T/N=(0, 29) ProgramFolder:XPSFitAssCoefConstraints/WAVE=CoefWave
	endif
	
	//	Creates the small coefficient waves used for the condensed display in the coefficient window
	XPSFitAssCreateSmallCoefWaves()
end



Static Function/S XPSFitAssLinkList()
//	Returns a string with the number of peaks, eg: "0;1;2;3;" if four peaks exist in the constraints wave. The first item is always NA and incated that the new peak should not be linked to an existing peak

	//	Finds the number of peaks in the fit
	DFREF ProgramFolder=root:Programming
	Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)

	//	Adds the peaks to the semicolon-separated list
	String PeakListString="NA;"
	Variable i=0
	for (i=0; i<NumberOfPeaks; i=i+1)
		PeakListString+=Num2Str(i)+";"
	endfor
	
	//	Returns the list
	Return PeakListString
end



Static Function XPSFitAssUndoButton(B_Struct) : ButtonControl
//	Restores the coefficients from before the last fit
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Restores the coefficients from before the last fit
		XPSFitAssUndo()
	endif
end



Static Function XPSFitAssUndo()
//	Restores the coefficients from before the last fit
		
	//	Finds the wave reference wave with the old fit coefficients
	DFREF ProgramFolder=root:Programming
	Wave/WAVE UndoWaves=ProgramFolder:XPSFitAssUndoWaves
		
	//	The newest of the old fit coefficients
	Wave/Z/T OldCoefWave=UndoWaves[0][0]
	Wave/Z/B/U OldCoefSelWave=UndoWaves[0][1]
		
	if (WaveExists(OldCoefWave) && WaveExists(OldCoefSelWave))
		
		//	Overwrite the coefficient waves with the old coefficients
		Duplicate/O/T OldCoefWave, ProgramFolder:XPSFitAssCoefConstraints/WAVE=CoefWave
		Duplicate/O/B/U OldCoefSelWave, ProgramFolder:XPSFitAssCoefConstraintsSel/WAVE=CoefSelWave

		//	Does not rotate the wave references if the next wave is empty
		if (Dimsize(UndoWaves[1][0], 0)>0)

			//	Replaces the old coefficient waves with empty waves
			Make/O/T/N=(0, 29) $GetWavesDataFolder(OldCoefWave, 2)/WAVE=EmptyCoefWave
			Make/O/B/U/N=(0, 29, 2) $GetWavesDataFolder(OldCoefSelWave, 2)/WAVE=EmptyCoefSelWave
			SetDimLabel 2, 1, backColors, EmptyCoefSelWave
		
			//	Rotates the undo wave to make point 1 the new point 0. Simpler ways such as MatrixOP/O UndoWaves=rotateRows(TempUndoWaves, -1) or ImageTransform/G=-1 rotateRows UndoWaves doesn't semm to work for wave reference waves
			Variable n=Dimsize(UndoWaves, 0)
			InsertPoints /M=0 n, 1, UndoWaves
			UndoWaves[n][]=UndoWaves[0][q]
			DeletePoints /M=0 0, 1, UndoWaves
		endif

		//	Creates the small coefficient waves used for the condensed display in the coefficient window
		XPSFitAssCreateSmallCoefWaves()
			
		//	Finds the wave to display
		DFREF ActiveFolder=XPSViewGetDataFolder("XPSFitAssistantWindow", "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave("XPSFitAssistantWindow", ActiveFolder, "DisplayedWavePopUp")
		
		//	Updates the Fit Assistant window
		XPSFitAssUpdateGraph("XPSFitAssistantWindow", ActiveFolder, DisplayedWave)
	endif
end



Static Function XPSFitAssCreateUndoWaves(MaxUndo)
//	Creates the undo waves for the Fit Assistant window
Variable MaxUndo

	//	Creates the wave to hold the references to the old coefficient waves
	DFREF ProgramFolder=root:Programming
	Make/O/WAVE/N=(MaxUndo, 2) ProgramFolder:XPSFitAssUndoWaves/WAVE=UndoWaves

	//	Creates the data folder to hold the old coefficient waves
	NewDataFolder/O ProgramFolder:FitAssUndo
	DFREF UndoFolder=ProgramFolder:FitAssUndo
	
	//	Create an empty coefficient wave pair
	Make/O/T/N=(0, 29) UndoFolder:$("XPSFitAssCoefConstraints"+Num2iStr(0))/WAVE=CoefWave
	Make/O/B/U/N=(0, 29, 2) UndoFolder:$("XPSFitAssCoefConstraintsSel"+Num2iStr(0))/WAVE=CoefSelWave
	SetDimLabel 2, 1, backColors, CoefSelWave
	
	//	Adds the coefficient wave pair to the undo waves
	UndoWaves[0][0]=CoefWave
	UndoWaves[0][1]=CoefSelWave
	
	Variable i=0
	for (i=1; i<MaxUndo; i+=1)
		
		//	Creates the remaining old coefficient waves
		Duplicate/O/T CoefWave, UndoFolder:$("XPSFitAssCoefConstraints"+Num2iStr(i))/WAVE=CoefWave
		Duplicate/O/B/U CoefSelWave, UndoFolder:$("XPSFitAssCoefConstraintsSel"+Num2iStr(i))/WAVE=CoefSelWave
	
		//	Adds the coefficient wave pair to the undo waves
		UndoWaves[i][0]=CoefWave
		UndoWaves[i][1]=CoefSelWave
	endfor
end



Static Function XPSFitAssUndoAdd()
//	Adds the current fit coefficients to the undo waves

	//	Finds the wave reference wave with the old fit coefficients
	DFREF ProgramFolder=root:Programming
	Wave/WAVE UndoWaves=ProgramFolder:XPSFitAssUndoWaves
	Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel
	
	//	Doesn't add empty fits
	Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)
	if (NumberOfPeaks>-1)

		//	Rotates the undo wave to make point 0 the new point 1. Simpler ways such as MatrixOP/O UndoWaves=rotateRows(TempUndoWaves, 1) or ImageTransform/G=1 rotateRows UndoWaves doesn't semm to work for wave reference waves
		Variable n=Dimsize(UndoWaves, 0)
		InsertPoints /M=0 0, 1, UndoWaves
		UndoWaves[0][]=UndoWaves[n][q]
		DeletePoints /M=0 n, 1, UndoWaves
		
		//	The position in the undo wave the coefficients should be added to
		Wave/T UndoCoefWave=UndoWaves[0][0]
		Wave/B/U UndoCoefSelWave=UndoWaves[0][1]
		
		//	Overwrite the old coefficient waves with the new coefficients
		Duplicate/O/T CoefWave, UndoCoefWave
		Duplicate/O/B/U CoefSelWave, UndoCoefSelWave
	endif
end



Static Function XPSFitAssReport(B_Struct) : ButtonControl
//	Creates a layout with the Fit Assistant graph and fitted values meant for printing
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		Variable n=120/ScreenResolution

		//	Ensure the small coefficient waves are updated
		XPSFitAssCreateSmallCoefWaves()
	
		//	Calculates the number of rows and colums needed to display the coefficients
		DFREF ProgramFolder=root:Programming
		Wave/T CoefCSmall=ProgramFolder:XPSFitAssCoefCSmall
		Variable NumberOfRows=DimSize(CoefCSmall, 0)	//186-169=17
		Variable NumberOfColumns=DimSize(CoefCSmall, 1)

		//	Creates the layout
		DoWindow /K XPSFitAssReportLayout
		NewLayout /K=1 /N=XPSFitAssReportLayout /P=Portrait /W=(10, 45, 360, 520)
		
		//	Creates a clean up function for when the window is closed
		SetWindow XPSFitAssReportLayout hook(KillHook)=XPSFitAssReportHookFunc
	
		//	Finds the paper and margin dimensions, needed to display the graph and table properly on the page
		PrintSettings /W=XPSFitAssReportLayout getPageDimensions
		String Temp=StringByKey("PAPER", S_value)
		Variable a=StrSearch(Temp, ",", 0)
		Variable b=StrSearch(Temp, ",", a+1)
		Variable c=StrSearch(Temp, ",", b+1)
		Variable PaperWidth=Str2Num(Temp[b+1,c-1])-Str2Num(Temp[0,a-1])
		Variable PaperHeight=Str2Num(Temp[c+1,StrLen(Temp)-1])-Str2Num(Temp[a+1,b-1])
	
		PrintSettings /W=XPSFitAssReportLayout getPageSettings
		Temp=StringByKey("MARGINS", S_value)
		a=StrSearch(Temp, ",", 0)
		b=StrSearch(Temp, ",", a+1)
		c=StrSearch(Temp, ",", b+1)
		Variable MarginL=Str2Num(Temp[0,a-1])
		Variable MarginR=Str2Num(Temp[b+1,c-1])
		Variable MarginT=Str2Num(Temp[a+1,b-1])
		Variable MarginB=Str2Num(Temp[c+1,StrLen(Temp)-1])

		//	Copies the wave name from the pop up menu and adds it to the layout
		ControlInfo /W=XPSFitAssistantWindow DisplayedWavePopUp
		TextBox /W=XPSFitAssReportLayout /B=0 /A=RT /F=0 /N=Title /X=0 /Y=0 "\Z18"+S_value
	
		//	Calculates the positions of the columns on the page
		Make/O/FREE/N=(NumberOfColumns) ColumnPosition=100-100/(NumberOfColumns-1+0.2)*(p+0.2)

		//	Creates a table using textboxes. This was the only way to create a table with invisble borders
		String BoxName="", BoxString=""
		Variable i=0, ii=0
		for (i=0; i<NumberOfRows; i+=1)
			for (ii=0; ii<NumberOfColumns; ii+=1)
				BoxName="Box"+Num2Str(i)+"_"+Num2Str(ii)
				if (i==0 || i==3 || i==NumberOfRows-2)
					BoxString="\Z10\f01"+CoefCSmall[i][ii]
				else
					BoxString="\Z08"+CoefCSmall[i][ii]
				endif
				if (CmpStr(CoefCSmall[i][ii], "")!=0)
					TextBox /W=XPSFitAssReportLayout /B=0 /A=RT /F=0 /N=$BoxName /X=(ColumnPosition[ii]) /Y=(i*2) BoxString
				endif
			endfor
		endfor
	
		//	Appends a picture of the Fit Assistant graph, showing the fit
		Variable YPosition=2/100*(NumberOfRows)*(PaperHeight-MarginT-MarginB)+MarginT
		SavePICT /B=288 /E=1 /O /P=_PictGallery_ /W=(0,0, PaperWidth-MarginR-MarginL, 2/3*(PaperWidth-MarginR-MarginL)) /WIN=XPSFitAssistantWindow as "XPSFitAssLayoutGraph"
		AppendLayoutObject /F=0 /R=(MarginL, YPosition, PaperWidth-MarginR, YPosition+2/3*(PaperWidth-MarginR-MarginL)) /T=1 /W=XPSFitAssReportLayout picture XPSFitAssLayoutGraph
	endif
	Return 0
end



Function XPSFitAssReportHookFunc(s)
//	The hook function for the Fit Assistant fit report window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1

			//	Deletes the graph image used in the Fit Report layout, but delays the execution until the window has been killed
			Execute/P/Q "KillPICTs/Z XPSFitAssLayoutGraph"
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function XPSFitAssSetOffset(B_Struct) : ButtonControl
//	Calculates a new linear background based on a straight line between cursors A and B
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		Variable Intercept=0

		//	Checks if cursor A or B is placed on a wave
		if (WaveExists(CsrWaveRef(A, B_Struct.win)))
			if (WaveExists(CsrWaveRef(B, B_Struct.win)))
			
				//	Sets the intercept equal to the average value of DataWave between cursors A and B
				DFREF ProgramFolder=root:Programming
				Wave DataWave=ProgramFolder:XPSFitAssDataWave
				Intercept=Mean(DataWave, hcsr(A, B_Struct.win), hcsr(B, B_Struct.win))
			else

				//	Sets the intercept equal to the value of cursor A
				Intercept=vcsr(A, B_Struct.win)
			endif
			
			//	Sets the linear background to Intercept and with no slope
			XPSFitAssSetLinearBackground(Intercept, 0)

		elseif (WaveExists(CsrWaveRef(B, B_Struct.win)))

			//	Sets the intercept equal to the value of cursor B
			Intercept=vcsr(B, B_Struct.win)

			//	Sets the linear background to Intercept and with no slope
			XPSFitAssSetLinearBackground(Intercept, 0)
		else
			ErrorMessage("Please place either cursor A on the point, or cursors A and B on the range, to be used as background!")
		endif
		
	endif
	Return 0
end



Static Function XPSFitAssSetLine(B_Struct) : ButtonControl
//	Calculates a new linear background based on a straight line between cursors A and B
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	If the line button has been clicked both cursors needs to be placed on a waves and they cannot be placed at the same position
		if (WaveExists(CsrWaveRef(A, B_Struct.win)) && WaveExists(CsrWaveRef(B, B_Struct.win)) && hcsr(A, B_Struct.win)!=hcsr(B, B_Struct.win))
				
			//	Calculates the slope between cursors A and B and the corresponding intercept
			Variable Slope=(vcsr(A, B_Struct.win)-vcsr(B, B_Struct.win))/(hcsr(A, B_Struct.win)-hcsr(B, B_Struct.win))
			Variable Intercept=vcsr(A, B_Struct.win)-hcsr(A, B_Struct.win)*Slope

			//	Sets the linear background to Intercept and Slope 
			XPSFitAssSetLinearBackground(Intercept, Slope)
		else
			ErrorMessage("Please place both cursors A and B on the graph and on different points!")
		endif
	endif
	Return 0
end



Static Function XPSFitAssSetLinearBackground(Intercept, Slope)
//	Sets the linear background to Intercept and Slope 
Variable Intercept, Slope
String ActiveWindow="XPSFitAssistantWindow"

	//	If the coefficient wave is empty, it is replaced with an empty wave big enough to hold a linear background
	DFREF ProgramFolder=root:Programming
	Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)

	if (NumberOfPeaks<0)
		XPSFitAssCreateEmptyConstraints(0)
		Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	endif
	
	//	Enter the values for the intercept and slope
	CoefWave[1][1]=Num2Str(Intercept)			//	Intercept
	CoefWave[1][3]="L0 > -inf"					//	   min
	CoefWave[1][4]="L0 < inf"					//	   max
	CoefWave[1][5]=Num2Str(Slope)			//	Slope
	CoefWave[1][7]="L1 > -inf"					//	   min
	CoefWave[1][8]="L1 < inf"					//	   max
		
	//	Sets bit 4, the state of the hold checkboxes, for the intercept and slope. The intercept and slope will now be held constant during the fit
	Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel
	CoefSelWave[1][2][0]=CoefSelWave[1][2][0] | (2^4)	//	Hold for intercept
	CoefSelWave[1][6][0]=CoefSelWave[1][6][0] | (2^4)	//	Hold for slope
	
	//	Finds the wave to display
	DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
	Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
		
	//	Updates the Fit Assistant window
	 XPSFitAssUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
end



Static Function XPSFitAssAdd(B_Struct) : ButtonControl
//	Adds a peak to the fit. The first peak added will also add a linear background
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		if (WaveExists(CsrWaveRef(A, B_Struct.win))==0)
			ErrorMessage("Please place cursor A on the position you wish to add a peak!")
		else
		
			//	Finds the selected display type
			ControlInfo /W=$B_Struct.win DisplayTypePopUp
			String DisplayType=S_Value
		
			//	The coefficient waves
			DFREF ProgramFolder=root:Programming
			Wave DataWave=ProgramFolder:XPSFitAssDataWave
			Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
			Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel

			//	Finds the number of peaks in the coefficient wave
			Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)

			//	If a linear background does not exist one is added
			Variable Intercept=0, Slope=0
			if (NumberOfPeaks<0)
			
				//	Creates empty coefficient waves just big enough to hold a linear background
				XPSFitAssCreateEmptyConstraints(0)
				
				//	Calculates the linear background parameters, based on the selected display type
				StrSwitch(DisplayType)
				
					//	Calculates the linear background for a NEXAFS spectrum
					case "NEXAFS":

						//	The slope is held at zero and the intercept is set to the minimum value in the spectrum
						Slope=0
						Intercept=WaveMin(DataWave)
						
						//	Sets the state of the Slope hold checkbox to a checked (bit 4) checkbox (bit 5)
						CoefSelWave[1][6][0]=48	
						break
						
					//	Calculates the linear background parameters for an XPS spectrum
					default:

						//	Creates the starting guesses for the linear background. The slope is calculated as the slope between the first and the last data point
						Variable n=NumPnts(DataWave)-1
						Slope=(DataWave[n]-DataWave[0])/(n*DeltaX(DataWave))

						//	The intercept is calculated to make the linear background touch the data in only one spot
						Duplicate/O/FREE DataWave TempWave
						TempWave=DataWave-XPSFitAssLine(x, 0, Slope)
						Intercept=WaveMin(TempWave)
				endswitch

				//	Enters the values for the intercept and slope
				CoefWave[1][1]=Num2Str(Intercept)			//	Intercept
				CoefWave[1][3]="L0 > -inf"					//	   min
				CoefWave[1][4]="L0 < inf"					//	   max
				CoefWave[1][5]=Num2Str(Slope)			//	Slope
				CoefWave[1][7]="L1 > -inf"					//	   min
				CoefWave[1][8]="L1 < inf"					//	   max
				
				//	During the rest of the function the coefficient waves will now be treated as waves with a linear background
				NumberOfPeaks=0
			endif
			
			//	Adds an extra row after the last peak in the coefficient waves
			Variable NewPeakPosition=NumberOfPeaks+4
			InsertPoints /M=0 NewPeakPosition, 1, CoefWave, CoefSelWave

			//	Makes the coefficient boxes editable
			CoefSelWave[NewPeakPosition][1,28][0]=2
			
			//	Defines the hold boxes as unchecked checkboxes
			CoefSelWave[NewPeakPosition][2,26;4][0]=32
			
			//	Sets the background colour of the min, mx and hold values to orange
			CoefSelWave[NewPeakPosition][1,28][1]=1
			
			//	Sets the background colous of the position, area, FWHM, etc... to light blue
			CoefSelWave[NewPeakPosition][1,25;4][1]=2
	
			//	Adds the peak number
			String PeakNumber=Num2Str(NumberOfPeaks)
			CoefWave[NewPeakPosition][0]=PeakNumber
	
			//	Finds the link-to peak (a value of "NA" will make the procedure ignore the link checkboxes)
			ControlInfo /W=$B_Struct.win LinkPopUp
			String LinkToPeakStr=S_Value
			Variable LinkToPeakPosition=Str2Num(LinkToPeakStr)+4

			//	Adds the position
			CoefWave[NewPeakPosition][1]=Num2Str(hcsr(A, B_Struct.win))

			//	Checks if the position is linked to that of another peak
			ControlInfo /W=$B_Struct.win PositionCheckBox

			//	If the checkbox is clicked and a link to peak has been selected
			if (V_value && NumType(LinkToPeakPosition)==0)

				//	Calculates the splitting between the new peak and the peak it should be linked to
				Variable PeakSplitting=Str2Num(CoefWave[NewPeakPosition][1])-Str2Num(CoefWave[LinkToPeakPosition][1])
				String PeakSplittingString=""
				if (PeakSplitting<0)
					PeakSplittingString=" - "+Num2Str(Abs(PeakSplitting))
				else
					PeakSplittingString=" + "+Num2Str(Abs(PeakSplitting))
				endif

				//	Copies the hold value from the linked-to peak
				CoefSelWave[NewPeakPosition][2][0]=CoefSelWave[LinkToPeakPosition][2][0]

				//	Adds the min, max values
				CoefWave[NewPeakPosition][3]="P"+PeakNumber+" > P"+LinkToPeakStr+PeakSplittingString
				CoefWave[NewPeakPosition][4]="P"+PeakNumber+" <  P"+LinkToPeakStr+PeakSplittingString
			else

				//	Limits the peak positions to +/- 1 eV or the binding energy range of the datawave whichever is smallest
				CoefWave[NewPeakPosition][3]="P"+PeakNumber+" > "+Num2Str(Max(Str2Num(CoefWave[NewPeakPosition][1])-1, Min(LeftX(DataWave), Pnt2X(DataWave, NumPnts(DataWave)-1))))
				CoefWave[NewPeakPosition][4]="P"+PeakNumber+" < "+Num2Str(Min(Str2Num(CoefWave[NewPeakPosition][1])+1, Max(LeftX(DataWave), Pnt2X(DataWave, NumPnts(DataWave)-1))))
			endif
	
			//	Adds the FWHM
			ControlInfo /W=$B_Struct.win FWHMCheckBox
			if (V_value && NumType(LinkToPeakPosition)==0)

				//	Adds the FWHM
				CoefWave[NewPeakPosition][9]=CoefWave[LinkToPeakPosition][9]

				//	Copies the hold value from the linked-to peak
				CoefSelWave[NewPeakPosition][10][0]=CoefSelWave[LinkToPeakPosition][10][0]

				//	Adds the min, max values
				CoefWave[NewPeakPosition][11]="W"+PeakNumber+" > W"+LinkToPeakStr
				CoefWave[NewPeakPosition][12]="W"+PeakNumber+" < W"+LinkToPeakStr
			else

				//	Adds the FWHM
				CoefWave[NewPeakPosition][9]="0.8"

				//	Adds the min, max values
				CoefWave[NewPeakPosition][11]="W"+PeakNumber+" > 0.4"
				CoefWave[NewPeakPosition][12]="W"+PeakNumber+" < 10"
			endif
	
			//	Adds the GL ratio (0 = Pure Gaussian, 1 = Pure Lorentizian)
			ControlInfo /W=$B_Struct.win GLCheckBox
			if (V_value && NumType(LinkToPeakPosition)==0)

				//	Adds the GL ratio
				CoefWave[NewPeakPosition][13]=CoefWave[LinkToPeakPosition][13]

				//	Copies the hold value from the linked-to peak
				CoefSelWave[NewPeakPosition][14][0]=CoefSelWave[LinkToPeakPosition][14][0]
			
				//	Adds the min, max values
				CoefWave[NewPeakPosition][15]="G"+PeakNumber+" > G"+LinkToPeakStr
				CoefWave[NewPeakPosition][16]="G"+PeakNumber+" < G"+LinkToPeakStr
			else

				//	Adds the GL ratio
				CoefWave[NewPeakPosition][13]="0.2"

				//	Adds the min, max values
				CoefWave[NewPeakPosition][15]="G"+PeakNumber+" > 0"
				CoefWave[NewPeakPosition][16]="G"+PeakNumber+" < 1"
			endif
		
			//	Checks if asymmetry is enabled
			ControlInfo /W=$B_Struct.win EnableAsymCheckBox
			Variable AsymEnabled=V_value, Asym1=0, Asym2=0
			if (AsymEnabled==0)
				Asym1=0
				Asym2=0
			else
				//	The fit works better if the starting values are not zero
				Asym1=0.05
				Asym2=0.05
			endif
	
			//	Adds asym1
			ControlInfo /W=$B_Struct.win AsymCheckBox
			if (V_value && NumType(LinkToPeakPosition)==0)

				//	Adds asym1
				CoefWave[NewPeakPosition][17]=CoefWave[LinkToPeakPosition][17]

				//	Copies the hold value from the linked-to peak
				CoefSelWave[NewPeakPosition][18][0]=CoefSelWave[LinkToPeakPosition][18][0]
			
				//	Adds the min, max values
				CoefWave[NewPeakPosition][19]="Y"+PeakNumber+" > Y"+LinkToPeakStr
				CoefWave[NewPeakPosition][20]="Y"+PeakNumber+" < Y"+LinkToPeakStr
			else

				//	Adds asym1
				CoefWave[NewPeakPosition][17]=Num2Str(Asym1)

				//	Adds the hold value
				if (AsymEnabled==0)

					//	Sets bit 4, the state of the hold checkbox
					CoefSelWave[NewPeakPosition][18][0]=CoefSelWave[NewPeakPosition][18][0] | (2^4)	////////////
				endif

				//	Adds the min, max values
				CoefWave[NewPeakPosition][19]="Y"+PeakNumber+" > 0.01"
				CoefWave[NewPeakPosition][20]="Y"+PeakNumber+" < 1"
			endif
	
			//	Adds asym2
			ControlInfo /W=$B_Struct.win AsymCheckBox
			if (V_value && NumType(LinkToPeakPosition)==0)

				//	Adds asym2
				CoefWave[NewPeakPosition][21]=CoefWave[LinkToPeakPosition][21]

				//	Copies the hold value from the linked-to peak
				CoefSelWave[NewPeakPosition][22][0]=CoefSelWave[LinkToPeakPosition][22][0]
			
				//	Adds the min, max values
				CoefWave[NewPeakPosition][23]="Z"+PeakNumber+" > Z"+LinkToPeakStr
				CoefWave[NewPeakPosition][24]="Z"+PeakNumber+" < Z"+LinkToPeakStr
			else

				//	Adds asym2
				CoefWave[NewPeakPosition][21]=Num2Str(Asym2)

				//	Adds the hold value
				if (AsymEnabled==0)

					//	Sets bit 4, the state of the hold checkbox
					CoefSelWave[NewPeakPosition][22][0]=CoefSelWave[NewPeakPosition][22][0] | (2^4)
				endif
			
				//	Adds the min, max values
				CoefWave[NewPeakPosition][23]="Z"+PeakNumber+" > 0.01"
				CoefWave[NewPeakPosition][24]="Z"+PeakNumber+" < 1"
			endif
	
			//	Adds the Shirley coefficient, based on the selected display type
			StrSwitch(DisplayType)
				
				//	Sets the Shirley coefficient for a NEXAFS spectrum. Linked Shirley coefficients are ignored for NEXAFS spectra
				case "NEXAFS":
					
					//	If a shift (bit 1) click was used to activate the Add button. Will add a NEXAFS edge jump instead of a peak
					if ((B_Struct.eventMod & 2)==2)
						
						//	Adds the Shirley coefficient
						CoefWave[NewPeakPosition][25]="1e6"
					else

						//	Adds the Shirley coefficient
						CoefWave[NewPeakPosition][25]="0"
					endif

					//	Sets the state of the Shirley hold checkbox to a checked (bit 4) checkbox (bit 5)
					CoefSelWave[NewPeakPosition][26][0]=48	
					
					//	Adds the min, max values
					CoefWave[NewPeakPosition][27]="S"+PeakNumber+" > 0"	
					CoefWave[NewPeakPosition][28]="S"+PeakNumber+" < 0.5"
					break
						
				//	Sets the Shirley coefficient for an XPS spectrum
				default:

					ControlInfo /W=$B_Struct.win ShirleyCheckBox
					if (V_value && NumType(LinkToPeakPosition)==0)

						//	Adds the Shirley coefficient
						CoefWave[NewPeakPosition][25]=CoefWave[LinkToPeakPosition][25]

						//	Copies the hold value from the linked-to peak
						CoefSelWave[NewPeakPosition][26][0]=CoefSelWave[LinkToPeakPosition][26][0]
			
						//	Adds the min, max values
						CoefWave[NewPeakPosition][27]="S"+PeakNumber+" > S"+LinkToPeakStr
						CoefWave[NewPeakPosition][28]="S"+PeakNumber+" < S"+LinkToPeakStr
					else

						//	Adds the Shirley coefficient
						CoefWave[NewPeakPosition][25]="0.02"
						
						//	Adds the min, max values
						CoefWave[NewPeakPosition][27]="S"+PeakNumber+" > 0"	
						CoefWave[NewPeakPosition][28]="S"+PeakNumber+" < 0.5"
					endif
			endswitch
	
			//	Adds the peak area
			ControlInfo /W=$B_Struct.win AreaCheckBox
			if (V_value && NumType(LinkToPeakPosition)==0)

				//	Adds the peak area
				CoefWave[NewPeakPosition][5]=CoefWave[LinkToPeakPosition][5]

				//	Copies the hold value from the linked-to peak
				CoefSelWave[NewPeakPosition][6][0]=CoefSelWave[LinkToPeakPosition][6][0]

				//	Adds the min, max values
				CoefWave[NewPeakPosition][7]="A"+PeakNumber+" > A"+LinkToPeakStr
				CoefWave[NewPeakPosition][8]="A"+PeakNumber+" < A"+LinkToPeakStr
			else

				//	The peak coefficients
				Intercept=Str2Num(CoefWave[1][1])
				Slope=Str2Num(CoefWave[1][5])
				Variable Position=Str2Num(CoefWave[NewPeakPosition][1])
				Variable FWHM=Str2Num(CoefWave[NewPeakPosition][9])
				Variable GL=Str2Num(CoefWave[NewPeakPosition][13])
				Asym1=Str2Num(CoefWave[NewPeakPosition][17])
				Asym2=Str2Num(CoefWave[NewPeakPosition][21])
				Variable ShirleyCoef=Str2Num(CoefWave[NewPeakPosition][25])

				Variable PeakArea=0
				
				//	If a shift (bit 1) click was used to activate the Add button for a NEXAFS spectra. Will add a NEXAFS edge jump instead of a peak
				if ((CmpStr(DisplayType, "NEXAFS")==0) && ((B_Struct.eventMod & 2)==2))
				
					//	Finds the last point in the data wave (highest photon energy)
					Variable LastPoint=Max(LeftX(DataWave), Pnt2X(DataWave, NumPnts(DataWave)-1))
					
					//	The edge jump peak area is calculated to make the edge jump touch the data in only one spot
					Duplicate/O/FREE/R=(Position, LastPoint) DataWave TempWave
					MultiThread TempWave=(DataWave(x)-XPSFitAssLine(x, Intercept, Slope))/(ShirleyCoef*XPSFitAssAsymPseudoVoigtShirley(x, Position, FWHM, GL, Asym1, Asym2))
					PeakArea=WaveMin(TempWave)
				else

					//	Calculates a peak area that makes the height of the peak equal to the difference between the measured data and the linear background
					Variable PeakHeight=XPSFitAssAsymPseudoVoigt(Position, Position, FWHM, GL, Asym1, Asym2)+ShirleyCoef*XPSFitAssAsymPseudoVoigtShirley(Position, Position, FWHM, GL, Asym1, Asym2)
					PeakArea=Abs((vcsr(A, B_Struct.win)-XPSFitAssLine(Position, Intercept, Slope))/PeakHeight)
				endif

				//	Adds the peak area
				CoefWave[NewPeakPosition][5]=Num2Str(PeakArea)

				//	Adds the min, max values. 														//	It would be natural to write a constraint expression "K1 > 0", but this can lead to problems. If some iteration tries a negative value of K1, the constraints will set K1 to be exactly zero.
				CoefWave[NewPeakPosition][7]="A"+PeakNumber+" > "+Num2Str(0.001*PeakArea)			//	 But when K1 is zero, the function has no dependence on K2 and a singular matrix error results. The solution is to use a constraint that requires K1 to be greater than some small
				CoefWave[NewPeakPosition][8]="A"+PeakNumber+" < inf"								//	 positive number: "K1 > 0.1", for instance. The value of the limit must be tailored to your application. If you expect the constraint to be inactive when the solution is found, and the fit
			endif

			//	Finds the wave to display
			DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")

			//	Updates the Fit Assistant window
			XPSFitAssUpdateGraph(B_Struct.win, ActiveFolder, DisplayedWave)

			//	Makes peak 0 the default link-to peak, if only one peak exists and no active selection other than 0 has previously been made
			if (NumberOfPeaks==0)
				ControlInfo /W=$B_Struct.win LinkPopUp
				if (CmpStr(S_UserData, "0")==0)
					ControlUpdate /W=$B_Struct.win LinkPopUp
					PopUpMenu LinkPopUp popmatch="0", win=$B_Struct.win
				endif
			endif
			DoWindow/F $B_Struct.win
		endif
	endif
	Return 0
end



Static Function XPSFitAssRemove(B_Struct) : ButtonControl
//	Removes a peak from the fit
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	The coefficient waves
		DFREF ProgramFolder=root:Programming
		Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
		Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel

		//	Finds the number of peaks in the coefficient wave
		Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)
		if (NumberOfPeaks<=0)
		
			//	Replaces the coefficient waves with empty waves
			XPSFitAssCreateEmptyConstraints(-1)
		else
			
			//	Removes the last peak from the coefficient waves
			DeletePoints /M=0 DimSize(CoefWave, 0)-4, 1, CoefWave, CoefSelWave
		endif
		
		//	Finds the wave to display
		DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")
	
		//	Updates the Fit Assistant window
		XPSFitAssUpdateGraph(B_Struct.win, ActiveFolder, DisplayedWave)
	endif
	Return 0
end



Static Function XPSFitAssUseSavedCoefWaves(ActiveWindow, ActiveFolder, ActiveWave)
//	Overwrites the coefficient waves with the saved coefficient waves, if they exist. Updates the Fit Assistant window, and places cursor A at the highest datapoint
String ActiveWindow
DFREF ActiveFolder
Wave/Z ActiveWave

	//	Overwrites the coefficient waves with the saved coefficient waves, if they exist
	XPSFitAssGetSavedCoefs(ActiveFolder, ActiveWave)
	
	//	Sets the axis scaling and labels to match the type of spectrum displayed and updates the graph
	XPSFitAssUpdateGraphDF(ActiveWindow, ActiveFolder, ActiveWave)
end



Static Function XPSFitAssGetSavedCoefs(ActiveFolder, ActiveWave)
//	Overwrites the coefficient waves with the saved coefficient waves, if they exist
DFREF ActiveFolder
Wave/Z ActiveWave

	//	Finds the name of the displayed wave
	String DisplayedName=NameOfWave(ActiveWave)
	
	//	The folder with the saved fits, if it exists
	DFREF SavedFitsFolder=ActiveFolder:SavedFits
	DFREF FitFolder=SavedFitsFolder:$DisplayedName
	Wave/Z/T ListWave=FitFolder:$(DisplayedName+"_lst")
	Wave/Z/B/U SelWave=FitFolder:$(DisplayedName+"_sel")
	
	//	Checks if the saved coeffcient waves exist
	if ((DataFolderRefStatus(ActiveFolder)!=0) && (DataFolderRefStatus(SavedFitsFolder)!=0) && (DataFolderRefStatus(FitFolder)!=0) && WaveExists(ListWave) && WaveExists(SelWave))

		//	Overwrites the coefficient waves with the saved waves
		DFREF ProgramFolder=root:Programming
		Duplicate/O/T ListWave ProgramFolder:XPSFitAssCoefConstraints
		Duplicate/O/B/U SelWave ProgramFolder:XPSFitAssCoefConstraintsSel
	else
	
		//	Replaces the coefficient waves with empty waves
		XPSFitAssCreateEmptyConstraints(-1)
	endif
end



Static Function XPSFitAssUpdateGraphDF(ActiveWindow, ActiveFolder, ActiveWave)
//	Sets the axis scaling and labels to match the type of spectrum displayed and updates the graph
String ActiveWindow
DFREF ActiveFolder
Wave/Z ActiveWave

	//	Checks if the x-axis range is manual
	Variable AutoScaleX=1, AxisXMin=0, AxisXMax=0
	String XAxisFlags=StringByKey("SETAXISFLAGS", AxisInfo(ActiveWindow,"bottom"))
	if (StrLen(XAxisFlags)==0)
	
		if (WaveExists(ActiveWave))
	
			//	Finds the x-range of the displayed wave
			Variable DataXMin=Min(LeftX(ActiveWave), Pnt2X(ActiveWave, NumPnts(ActiveWave)-1))
			Variable DataXMax=Max(LeftX(ActiveWave), Pnt2X(ActiveWave, NumPnts(ActiveWave)-1))
	
			//	Finds the current x-axis scaling
			GetAxis /W=$ActiveWindow /Q bottom
			AxisXMin=Min(V_max, V_min)
			AxisXMax=Max(V_max, V_min)
	
			//	Checks if the displayed wave has any datapoints inside the manual x-axis range
			if (limit(DataXMin, AxisXMin, AxisXMax)!=limit(DataXMax, AxisXMin, AxisXMax))
	
				//	Keeps the manual x-axis scaling
				AutoScaleX=0
			endif
		endif
	endif
	
	//	Sets the folder type popup menu based on the active data folder
	XPSViewFolderType(ActiveFolder, ActiveWindow)
	
	//	Finds the selected display type
	ControlInfo /W=$ActiveWindow DisplayTypePopUp
	String DisplayType=S_Value
	
	//	Sets the axis scaling and labels to match the type of spectrum displayed
	XPSViewSetAxis(ActiveWindow, DisplayType, AutoScaleX, AxisXMin, AxisXMax)

	//	Refreshes the graph
	XPSFitAssUpdateGraph(ActiveWindow, ActiveFolder, ActiveWave)
end



Static Function XPSFitAssUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
//	Updates the Fit Assistant graph
String ActiveWindow
DFREF ActiveFolder
Wave/Z DisplayedWave
Variable CursorAX=0, CursorBX=0, CursorAY=0, CursorBY=0, NumberOfOldDataPoints=0, NumberOfNewDataPoints=0, Overlap=0
String CursorAInfo="", CursorBInfo=""

	//	Removes the fit waves from the Edit Waves table, if it exists
	XPSFitAssEditRemoveFitWaves()
	
	//	Temporarily disables the hook function associated with the window to prevent update running again as the cursors are moved
	SetWindow $ActiveWindow hook(HookOne)=$""
	
	DFREF ProgramFolder=root:Programming

	if (WaveExists(DisplayedWave)==0)
	
		//	Displays an empty wave
		Make/O/N=0 ProgramFolder:XPSFitAssDataWave/WAVE=DataWave
		Wave/Z DisplayedWave=$""
	else
	
		//	The old data wave
		Wave DataWave=ProgramFolder:XPSFitAssDataWave
		
		//	The number of points in the new wave
		NumberOfOldDataPoints=NumPnts(DataWave)
		NumberOfNewDataPoints=NumPnts(DisplayedWave)
		
		if (WaveRefsEqual(DisplayedWave, DataWave)==0)

			//	Finds the x-range of the new displayed wave
			Variable Temp1=LeftX(DisplayedWave)
			Variable Temp2=Pnt2x(DisplayedWave, NumPnts(DisplayedWave)-1)
			Variable NewMin=Min(Temp1, Temp2)
			Variable NewMax=Max(Temp1, Temp2)
			
			Overlap=1

			//	Determines if the cursors are placed on the graph. If they are, their positions are saved
			CursorAInfo=CsrInfo(A, ActiveWindow)
			if (StrLen(CursorAInfo)>0)
				CursorAX=hcsr(A, ActiveWindow)
				CursorAY=vcsr(A, ActiveWindow)
				
				//	Checks if the cursor position overlaps with the x-range of the new wave
				if ((CursorAX<NewMin) || (CursorAX>NewMax))
					Overlap=0
				endif
			endif
			CursorBInfo=CsrInfo(B, ActiveWindow)
			if (StrLen(CursorBInfo)>0)
				CursorBX=hcsr(B, ActiveWindow)
				CursorBY=vcsr(B, ActiveWindow)
				
				//	Checks if the cursor position overlaps with the x-range of the new wave
				if ((CursorBX<NewMin) || (CursorBX>NewMax))
					Overlap=0
				endif
			endif
	
			//	Duplicates the selected wave into DataWave, to be displayed
			Duplicate/O DisplayedWave ProgramFolder:XPSFitAssDataWave/WAVE=DataWave
			
			//	Removes any unit labels that would otherwise result in double axis labels
			SetScale/P x, LeftX(DataWave), DeltaX(DataWave), "", DataWave
		endif
	endif
	
	//	Removes the cursors
	Cursor/K/W=$ActiveWindow A
	Cursor/K/W=$ActiveWindow B
	
	//	Updates the colour of the little coloured box in the top left corner to match the colour in the View Spectra graph (if the wave is selected there)
	XPSFitAssUpdateColourBox(DisplayedWave)

	//	Displays the fit in the coefficient waves and updates the Fit Assistant window
	Variable CursorsActive=((StrLen(CursorAInfo)>0) && (StrLen(CursorBInfo)>0) && (Overlap==1))
	XPSFitAssRedraw(CursorsActive, CursorAX, CursorBX)
	
	//	Creates the small CoefConstraints waves used for the condensed display. This will also update the total area box.
	XPSFitAssCreateSmallCoefWaves()

	//	Changes the link-to peak to NA if it is a peak that no longer exists
	ControlInfo /W=$ActiveWindow LinkPopUp
	ControlUpdate /W=$ActiveWindow LinkPopUp
	PopupMenu LinkPopUp mode=1, popmatch=S_Value, win=$ActiveWindow
	
	if (NumberOfNewDataPoints>0)
	
		//	Removes cursor B and places cursor A on the highest data point, if the old and new datawave do not overlap
		if ((Overlap==0) || (NumberOfOldDataPoints==0))
	
			//	Displays cursor A, but not B
			CursorAInfo="X"
			CursorBInfo=""
		
			//	Sets the position of cursor A to the highest data point
			WaveStats/Q/Z DataWave
			CursorAX=V_maxloc
			CursorAY=DataWave(V_maxloc)
		endif
	
		//	Checks if the free cursors button is toggled
		Variable FreeCursors=(Str2Num(GetUserData(ActiveWindow, "FreeButton", "Status"))==1)

		//	Forces the window to update. This will autoscale the axes before the cursors are placed back on the graph.
		if (FreeCursors)
			DoUpdate /W=$ActiveWindow
		endif
	
		//	Moves the cursors back to where they were before the axis scaling changed. If the windows hook function is not disabled this will trigger UpdateGraph to run again
		if (StrLen(CursorAInfo)>0)
			if (FreeCursors)
				Cursor/F/W=$ActiveWindow A $NameOfWave(DataWave) CursorAX, CursorAY
			else
				Cursor/W=$ActiveWindow A $NameOfWave(DataWave) CursorAX
			endif
		endif
		if (StrLen(CursorBInfo)>0)
			if (FreeCursors)
				Cursor/F/W=$ActiveWindow B $NameOfWave(DataWave) CursorBX, CursorBY
			else
				Cursor/W=$ActiveWindow B $NameOfWave(DataWave) CursorBX
			endif
		endif
	endif
	
	//	Restores the hook function to the window
	SetWindow $ActiveWindow hook(HookOne)=XPSFitAssHookFunc

	//	Saves the displayed coefficients as the fit associated with the wave
	//	This will also update the View Spectra graph if necessary
	XPSFitAssSaveCoefs(ActiveWindow, ActiveFolder, DisplayedWave)

	//	Appends the fit waves to the Edit Waves table, if it exists
	XPSFitAssEditAddFitWaves()
end



Static Function XPSFitAssUpdateColourBox(ActiveWave)
//	Updates the colour of the little coloured box in the top left corner to match the colour in the View Spectra graph (if the wave is selected there)
Wave/Z ActiveWave
Variable Red=60000, Green=60000, Blue=60000

	//	Checks if the View Spectra window exists
	DoWindow XPSViewSpectraWindow
	
	if (WaveExists(ActiveWave) && (V_Flag!=0))

		//	Finds the View Spectra listbox waves
		DFREF ProgramFolder=root:Programming
		Wave/WAVE RefWave=ProgramFolder:XPSViewWaveListWaves
		Wave/B/U SelWave=ProgramFolder:XPSViewWaveListSel
		Wave/W/U ColourWave=ProgramFolder:XPSViewColours

		//	Checks if the active wave has been assigned a colour in the View Spectra window
		Extract/FREE/INDX/O RefWave, IndexWave, WaveRefsEqual(RefWave[p], ActiveWave)
		if (NumPnts(IndexWave)>0)
			if (SelWave[IndexWave[0]][2][1]>0)
				Red=ColourWave[SelWave[IndexWave[0]][2][1]][0]
				Green=ColourWave[SelWave[IndexWave[0]][2][1]][1]
				Blue=ColourWave[SelWave[IndexWave[0]][2][1]][2]
			endif
		endif
	endif
	
	//	Checks if the Fit Assistant window exists
	DoWindow XPSFitAssistantWindow
	
	if (V_Flag!=0)

		//	Updates the colour of the little coloured box in the top left corner to match the colour in the View Spectra graph if the wave is displayed selected there, if not the colour will default to light grey (60000, 60000, 60000)
		ValDisplay ColourBox, barBackColor=(Red, Green, Blue), win=XPSFitAssistantWindow
	endif
end



Static Function XPSFitAssRedraw(CursorsActive, CursorAX, CursorBX)
//	Displays the fit in the coefficient waves and updates the Fit Assistant window
Variable CursorsActive, CursorAX, CursorBX

	String ActiveWindow="XPSFitAssistantWindow"
	
	//	Checks if the Fancy Plot checkbox is clicked
	ControlInfo/W=$ActiveWindow FancyPlotCheckBox
	Variable FancyPlot=V_Value

	DFREF ProgramFolder=root:Programming
	DFREF FitWavesFolder=ProgramFolder:FitAssFitWaves
	Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	Wave DataWave=ProgramFolder:XPSFitAssDataWave

	//	Finds the number of peaks in the coefficient wave
	Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)
	
	//	Doesn't display a fit, if the displayed wave is invalid
	if (NumPnts(DataWave)==0)
		NumberOfPeaks=-1
	endif

	//	Tests if the constraints wave contains any NaNs
	Variable CoefsOK=XPSFitAssTestCoefWave()
	if (CoefsOK==0)
		ErrorMessage("The fit parameters are invalid!")
	endif

	if ((NumberOfPeaks>=0) && CoefsOK)
	
		//	Creates the displayed fit waves. If both cursors exists only the range between the cursors is drawn
		if (CursorsActive)
			XPSFitAssCreateFitWaves(DataWave, CursorAX, CursorBX)
		else
			XPSFitAssCreateFitWaves(DataWave, 0, 0)
		endif

		//	Adds the background to the graph if it does not already exist on the graph
		Wave BckWave=FitWavesFolder:XPSFitAssBckWave
		CheckDisplayed /W=$ActiveWindow BckWave
		if (V_flag==0)
			AppendToGraph /W=$ActiveWindow BckWave
			ModifyGraph /W=$ActiveWindow rgb($NameOfWave(BckWave))=(0, 55000, 0)
		endif

		//	Adds the peaks to the graph, one at a time
		Variable i=0, Position=0
		for (i=0; i<NumberOfPeaks; i=i+1)

			//	Adds the peak to the graph if it does not already exist on the graph
			Wave PeakWave=FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(i))
			CheckDisplayed /W=$ActiveWindow PeakWave
			if (V_flag==0)
				AppendToGraph /W=$ActiveWindow PeakWave
				ModifyGraph /W=$ActiveWindow rgb($NameOfWave(PeakWave))=(0, 0, 65535)
			endif
		
			//	Adds a tag to the peak with the peak number
			Position=Str2Num(CoefWave[4+i][1])
			Tag /W=$ActiveWindow /C/N=$("PeakTag"+Num2Str(i)) /A=MT /B=1 /F=0 /G=(0, 0, 65535) /L=0 /X=0 /Y=-10 $NameOfWave(PeakWave), Position, Num2Str(i)
		endfor
	
		//	Removes any additional peaks from the graph and kills the waves
		Wave/Z PeakWave=FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(NumberOfPeaks))
		for(i=NumberOfPeaks+1; WaveExists(PeakWave); i+=1)
			RemoveFromGraph/Z /W=$ActiveWindow $NameOfWave(PeakWave)
			KillWaves/Z PeakWave
			Wave/Z PeakWave=FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(i))
		endfor
		
		//	Brings the result wave to the front of the graph
		Wave FitWave=FitWavesFolder:XPSFitAssFitWave
		RemoveFromGraph/Z /W=$ActiveWindow $NameOfWave(FitWave)
		AppendToGraph /W=$ActiveWindow FitWave
		
		//	Changes the display style
		String FitWaveName=NameOfWave(FitWave)
		String DataWaveName=NameOfWave(DataWave)
		if (FancyPlot==0)
			ModifyGraph /W=$ActiveWindow mode($DataWaveName)=0, rgb($DataWaveName)=(0,0,0), lsize($FitWaveName)=1
		else
			ModifyGraph /W=$ActiveWindow mode($DataWaveName)=3, marker($DataWaveName)=19, msize($DataWaveName)=2, useMrkStrokeRGB($DataWaveName)=1, rgb($DataWaveName)=(40000,40000,40000), lsize($FitWaveName)=1.5
		endif
	else
	
		//	Removes the background and fit waves from the graph
		RemoveFromGraph/Z /W=$ActiveWindow XPSFitAssBckWave, XPSFitAssFitWave
		
		//	Removes all peaks from the graph
		Wave/Z PeakWave=FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(0))
		for(i=1; WaveExists(PeakWave); i+=1)
			RemoveFromGraph/Z /W=$ActiveWindow $NameOfWave(PeakWave)
			Wave/Z PeakWave=FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(i))
		endfor
		
		//	Cleans up all fit waves
		KillWavesInFolder(FitWavesFolder)
	endif
end



Static Function XPSFitAssCreateFitWaves(DataWave, DrawFrom, DrawTo)
//	Creates the fit waves displayed in the Fit Assistant window. Data wave determines the x-resolution of the created fit waves
Wave DataWave
Variable DrawFrom, DrawTo
Variable i=0

	//	The coefficient wave
	DFREF ProgramFolder=root:Programming
	DFREF FitWavesFolder=ProgramFolder:FitAssFitWaves
	Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints

	//	Finds the number of peaks in the coefficient wave
	Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)
	
	//	Finds the number of data points in the displayed wave
	Variable NumberOfDataPoints=NumPnts(DataWave)

	//	Creates a temporary wave with a ten times finer x-resolution than the measured datawave's
	if (NumberOfDataPoints>0)
		Make/FREE/O/N=((NumberOfDataPoints-1)*10+1) BaseFitWave
	else
		Make/FREE/O/N=0 BaseFitWave
	endif
	SetScale/P x, LeftX(DataWave), 0.1*DeltaX(DataWave), "", BaseFitWave
	
	//	Creates the waves to hold background and fit. If DrawFrom and DrawTo are both different from zero, only the range between them is duplicated
	if ((DrawFrom==0) && (DrawTo==0))
		Duplicate/O BaseFitWave, FitWavesFolder:XPSFitAssBckWave/WAVE=BckWave, FitWavesFolder:XPSFitAssFitWave/WAVE=FitWave
	else
		Duplicate/O/R=(DrawFrom, DrawTo) BaseFitWave, FitWavesFolder:XPSFitAssBckWave/WAVE=BckWave, FitWavesFolder:XPSFitAssFitWave/WAVE=FitWave
	endif
	FastOP FitWave=0
		
	//	Creates a temporary wave used for faster calculations
	Duplicate/O/FREE FitWave, XWave
	XWave=x
	
	//	Creates a temporary wave used for faster calculations
	Make/O/FREE/WAVE/N=(NumberOfPeaks) PeakWaves, BckWaves
	
	//	Creates a temporary wave used for faster calculations
	Make/O/FREE/N=(NumberOfPeaks+1) ConstWave
	ConstWave[0]=Str2Num(CoefWave[1][1])

	//	Calculates the peaks and Shirley backgrounds for each peak
	if (NumberOfPeaks>0)
		ConstWave[1,*]=XPSFitAssCreateSingleFitWave(p-1, PeakWaves, BckWaves, XWave, FitWave, CoefWave, FitWavesFolder)
	endif
	
	//	Add the linear background to the background
	Variable Const=Sum(ConstWave)
	Variable Slope=Str2Num(CoefWave[1][5])
	FastOP BckWave=(Const)+(Slope)*XWave
	
	//	Adds the individual Shirley backgrounds to the combined background and the peaks to the combined fit
	for (i=0; i<NumberOfPeaks; i+=1)
		Wave NextBckWave=BckWaves[i]
		Wave PeakWave=PeakWaves[i]
		FastOP BckWave=BckWave+NextBckWave
		FastOP FitWave=FitWave+PeakWave
	endfor
	
	//	Adds the combined background to the fit
	FastOP FitWave=FitWave+BckWave

	//	Adds the combined background to the peak waves
	for (i=0; i<NumberOfPeaks; i+=1)
		Wave PeakWave=PeakWaves[i]
		FastOP PeakWave=PeakWave+BckWave
	endfor
end



Static Function XPSFitAssCreateSingleFitWave(Num, PeakWaves, BckWaves, XWave, FitWave, CoefWave, FitWavesFolder)
//
Variable Num
Wave/WAVE PeakWaves, BckWaves
Wave XWave, FitWave
Wave/T CoefWave
DFREF FitWavesFolder

	//	Creates the wave to hold the peak
	Duplicate/O FitWave, FitWavesFolder:$("XPSFitAssPkWave"+Num2Str(Num))/WAVE=PeakWave
	PeakWaves[Num]=PeakWave
	
	//	Creates the wave to hold the Shirley background
	Duplicate/O/FREE FitWave, BckWave
	BckWaves[Num]=BckWave
			
	//	Finds the parameters for peak number Num
	Variable Row=Num+4
	Variable Position=Str2Num(CoefWave[Row][1])
	Variable PeakArea=Str2Num(CoefWave[Row][5])
	Variable FWHM=Str2Num(CoefWave[Row][9])
	Variable GL=Str2Num(CoefWave[Row][13])
	Variable Asym1=Str2Num(CoefWave[Row][17])
	Variable Asym2=Str2Num(CoefWave[Row][21])
	Variable ShirleyCoef=Str2Num(CoefWave[Row][25])
			
	//	Calculates the Shirley background for the peak and adds it to the combined background
	Variable Const=XPSFitAssFitFuncAAOOnePeakBck(Position, PeakArea, FWHM, GL, Asym1, Asym2, ShirleyCoef, XWave, BckWave)

	//	Calculates the peak without any background
	XPSFitAssFitFuncAAOOnePeak(Position, PeakArea, FWHM, GL, Asym1, Asym2, 0, XWave, PeakWave)

	//	Returns the constant offset component of the Shirley background
	Return Const
end



Static Function XPSFitAssTestCoefWave()
//	Tests if the coefficient wave contains numbers in all the right places

	Variable CoefsOK=1

	//	Finds the number of peaks in the coefficient wave
	DFREF ProgramFolder=root:Programming
	Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)

	if (NumberOfPeaks>=0)
		
		//	Creates a new wave to hold the relevant values from the coefficient wave
		Make/FREE/O/N=(2+NumberOfPeaks*7) TestWave
		FastOP TestWave=(NaN)

		//	Copies the intercept and slope values
		TestWave[0,1]=Str2Num(CoefWave[1][1+p*4])
	
		if (NumberOfPeaks>0)
			//	Copies the peak values from CoefConstraints into the temporary wave
			TestWave[2,*]=Str2Num(CoefWave[4+Trunc((p-2)/7)][1+4*(p-2-7*Trunc((p-2)/7))])
		endif

		//	Calculates the number of NaNs in the wave
		WaveStats/Q TestWave
	
		//	Returns 1 if the test wave contains only valid numbers		
		CoefsOK=(V_numNans==0) ? (1) : (0)
	endif
	Return CoefsOK
end



Static Function XPSFitAssFitButton(B_Struct) : ButtonControl
//	Starts the fit
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Finds the number of peaks in the coefficient wave
		DFREF ProgramFolder=root:Programming
		Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
		Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel
		Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)
		
		if (NumberOfPeaks<0)
			ErrorMessage("How about adding a peak or at least a linear background?")
		else
		
			//	Removes all waves from the View Spectra Edit Waves table, if it exists. This will allow saved fit waves to be deleted
			XPSViewEditRemoveWaves()
			
			//	Saves all tags and removes all traces from the View Spectra graph, if it exists (except the empty placeholder wave)
			//	This will allow saved fit waves to be deleted during the fitting process.
			XPSViewRemoveAllTraces()

			Variable i=0, Position=0
			String ErrorString=""

			//	Determines if the cursors are placed on the graph. If they are, their positions are saved
			Variable CursorAX=0, CursorAY=0, CursorBX=0, CursorBY=0
			String CursorAInfo=CsrInfo(A, B_Struct.win)
			if (StrLen(CursorAInfo)>0)
				CursorAX=hcsr(A, B_Struct.win)
				CursorAY=vcsr(A, B_Struct.win)
			endif
			String CursorBInfo=CsrInfo(B, B_Struct.win)
			if (StrLen(CursorBInfo)>0)
				CursorBX=hcsr(B, B_Struct.win)
				CursorBY=vcsr(B, B_Struct.win)
			endif
		
			//	If both cursors A and B are on the graph they determine the range to be fitted
			Variable FitFrom=0, FitTo=0
			if ((StrLen(CursorAInfo)>0) && (StrLen(CursorBInfo)>0))
				FitFrom=Min(CursorAX, CursorBX)
				FitTo=Max(CursorAX, CursorBX)
			endif

			//	Finds the selected data folder or the data folder of the selected group
			DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")

			//	Checks the Fit or Fit All has been pressed
			if (CmpStr(B_Struct.ctrlName, "FitButton")==0)
		
				//	Finds the displayed wave
				Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")

				//	Creates a wave reference wave with just one wave in it
				Make/O/FREE/WAVE DataWaves={DisplayedWave}
			else

				//	Returns a list of all waves in the selected group or data folder
				String ListOfWaves=XPSViewPopUpWaveList(1, B_Struct.win, "DataFolderPopUp")
			
				//	Finds the number of waves in the list and creates a wave reference wave to hold them
				Variable NumberOfWaves=ItemsInList(ListOfWaves, ";")
				Make/O/FREE/WAVE/N=(NumberOfWaves) DataWaves
							
				//	Adds the waves in the list to the wave reference wave
				Variable a=0, b=0
				for (i=0; i<NumberOfWaves; i+=1)
					b=StrSearch(ListOfWaves, ";", a)
					DataWaves[i]=ActiveFolder:$(ListOfWaves[a, b-1])
					a=b+1
				endfor
			endif
			
			//	Checks if all waves exists
			Extract/FREE/INDX/O DataWaves, TestWave, (WaveExists(DataWaves[p])==0)
			if ((DataFolderRefStatus(ActiveFolder)!=0) && (NumPnts(TestWave)==0))
			
				//	Creates a backup of the coefficient waves that can be restored using the Undo button
				XPSFitAssUndoAdd()

				//	The defualt behavior 1 is to perform initial fits for all spectra
				//	The initial fits hold positions, GL ratios and asymmetry for constant. Especially the GL ratios tend to run amok, if the initial guesses are not perfect. This can be seen as a way to improve the initial guesses before letting the GL ratios loose.
				//	For troubleshooting purposes this value can be set to -1, which supresses all initial fits, or 0, which suppresses all initial fits, except the first one
				Variable DoInitialFit=1
				
				//	Hides the curve fitting window during the fitting. This will make the fit go faster when fitting a large number of simple fits
				Variable FitOptions=4
				
				//	Can be used to suppress saving the fit. This can significantly speed up the fitting process for very simple fits
				//	The result of the fit can still be accessed in root:Programming:Temp:XPSFitAssFitCoefs. Most normal users do not need to know about this
				Variable SaveFitCoefs=1
				

					////////////			THE FIT IS DONE HERE			////////////////////////
			
				//	Fits the list of waves, using 90,000 as the maxium number of iterations per spectrum
				XPSFitAssFitMany(DataWaves, FitFrom, FitTo, 90000, FitOptions, DoInitialFit, SaveFitCoefs)
				
				//	Finds the wave to display
				Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")
				if (WaveExists(DisplayedWave))
					Duplicate/O DisplayedWave ProgramFolder:XPSFitAssDataWave
				endif

				//	Checks if the free cursors button is toggled
				Variable FreeCursors=(Str2Num(GetUserData(B_Struct.win, "FreeButton", "Status"))==1)
	
				//	Temporarily disables the hook function associated with the window to prevent update running again as the cursors are moved
				SetWindow $B_Struct.win hook(HookOne)=$""

				//	Moves the cursors back to where they were before the axis scaling changed
				if (StrLen(CursorAInfo)>0)
					if (FreeCursors)
						Cursor/F/W=$B_Struct.win A XPSFitAssDataWave CursorAX, CursorAY
					else
						Cursor/W=$B_Struct.win A XPSFitAssDataWave CursorAX
					endif
				endif
				if (StrLen(CursorBInfo)>0)
					if (FreeCursors)
						Cursor/F/W=$B_Struct.win B XPSFitAssDataWave CursorBX, CursorBY
					else
						Cursor/W=$B_Struct.win B XPSFitAssDataWave CursorBX
					endif
				endif
	
				//	Restores the hook function to the window
				SetWindow $B_Struct.win hook(HookOne)=XPSFitAssHookFunc

				//	Overwrites the coefficient waves with the saved coefficient waves for the displayed spectrum, if they exist, and updates the Fit Assistant window
				XPSFitAssUseSavedCoefWaves(B_Struct.win, ActiveFolder, DisplayedWave)
				
				//	Updates the View Spectra window if it exists
				DoWindow XPSViewSpectraWindow
				if (V_Flag!=0)
			
					//	Finds the selected data folder
					ControlInfo /W=XPSViewSpectraWindow DataFolderPopUp
					DFREF SpectraFolder=$S_Value
			
					XPSViewSpectraUpdateGraph("XPSViewSpectraWindow", SpectraFolder, $"")
				endif
			else
				ErrorMessage("One or more waves does not exist!")
			endif
		endif
	endif
	Return 0
end



Static Function XPSFitAssCreateProgressWindow(NumberOfSpectra)
//	Creates the fit progress window
Variable NumberOfSpectra

	//	Creates the fit progress window
	String ActiveWindow="FitProgressWindow"
	DoWindow/K $ActiveWindow
	NewPanel /N=$ActiveWindow /W=(300,100,690,235)
	
	//	Sets the background colour of the panel to something very close to the standard grey. This is done to be able to make the ValDisplay background the exact same shade of grey
	ModifyPanel /W=$ActiveWindow cbRGB=(54000, 53000, 51000)
	
	String NumberOfSpectraStr=Num2iStr(NumberOfSpectra)
	String NumberOfSpectraStrLen=Num2iStr(StrLen(NumberOfSpectraStr))
	String TimeElapsedStr="Time elapsed: 00:00:00"
	
	//	Creates the fitting spectra progress counter and bar
	ValDisplay FitCounter pos={20,5}, size={350,20}, disable=2, frame=0, format="Fitting spectra: %"+NumberOfSpectraStrLen+"u of "+NumberOfSpectraStr, valueBackColor=(54000, 53000, 51000), value=_NUM:0, win=$ActiveWindow, help={"Displays the fitting spectra progress"}
	ValDisplay FitProgressBar, pos={20,25}, size={350,20}, disable=2, limits={0,NumberOfSpectra,0}, barmisc={0,0}, value=_NUM:0, mode=0, highColor=(0,65535,0), win=$ActiveWindow, help={"Displays the fitting spectra progress"}

	//	Creates the saving coefficients progress counter and bar	
	ValDisplay CoefCounter pos={20,55}, size={350,20}, disable=2, frame=0, format="Saving coefficients: %"+NumberOfSpectraStrLen+"u of "+NumberOfSpectraStr, valueBackColor=(54000, 53000, 51000), value=_NUM:0, win=$ActiveWindow, help={"Displays the saving coefficients progress"}
	ValDisplay CoefProgressBar, pos={20,75}, size={350,20}, disable=2, limits={0,NumberOfSpectra,0}, barmisc={0,0}, value=_NUM:0, mode=3, highColor=(0,65535,0), win=$ActiveWindow, help={"Displays the saving coefficients progress"}

	//	Displays the time elapsed since the fitting started
	SetVariable TimeElapsed pos={20,105}, size={350,20}, noedit=1, frame=0, valueBackColor=(54000, 53000, 51000), value=_STR:(TimeElapsedStr), win=$ActiveWindow, help={"Displays the time elapsed since the fitting started"}

	//	Creates an abort button, to abort the fit
	Button AbortButton, pos={300,105}, size={70,20}, title="Abort", win=$ActiveWindow, help={"Aborts the fitting"}
	
	//	Marks the window as a progress window, which will accept mouse events while code is executing (see DisplayHelpTopic "Progress Windows"). This is not needed to display a progress bar. It is only needed for the Abort button to work
	//	This special state of the control panel is automatically cleared when procedure execution finishes and Igor's outer loop again runs.
	DoUpdate /W=$ActiveWindow /E=1
end



Static Function XPSFitAssUpdateProgressWindow(StartTime, Counter, FitOrSave)
//	Updates the counters and progress bars for the fit progress window
Variable StartTime, Counter, FitOrSave

	String ActiveWindow="FitProgressWindow"
	
	//	Checks which counter to update
	if (FitOrSave==0)
	
		//	Updates the fitting spectra progress counter and bar
		ValDisplay FitCounter, value=_NUM:Counter, win=$ActiveWindow
		ValDisplay FitProgressBar, value=_NUM:Counter, win=$ActiveWindow
	else

		//	Updates the saving coefficients progress counter and bar
		ValDisplay CoefCounter, value=_NUM:Counter, win=$ActiveWindow
		ValDisplay CoefProgressBar, value=_NUM:Counter, win=$ActiveWindow
	endif
		
	//	Finds the time elapsed in seconds
	Variable CurrentTime=DateTime-StartTime		//	1 us
		
	//	Converts the time in seconds into hours, minutes and seconds
	Variable CurrentHours=Trunc(CurrentTime/3600)
	Variable CurrentMinutes=Trunc((CurrentTime-3600*CurrentHours)/60)
	Variable CurrentSeconds=Trunc(CurrentTime-3600*CurrentHours-60*CurrentMinutes)
		
	//	Creates the time elapsed string to display with the format hh:mm:ss
	String TimeElapsedStr=""
	SPrintF TimeElapsedStr, "Time elapsed: %02u:%02u:%02u", CurrentHours, CurrentMinutes, CurrentSeconds
		
	//	Updates the time elapsed counter
	SetVariable TimeElapsed, value=_STR:(TimeElapsedStr), win=$ActiveWindow
		
	//	Updates all controls in the window and checks if the Abort button has been pressed
	//	For a window marked as a progress window, DoUpdate sets V_Flag to 2 if a mouse up happened in a button since the last call.
	//	When this occurs, the full path to the subwindow containing the button is stored in S_path and the name of the control is stored in S_name
	DoUpdate /W=$ActiveWindow		//	0.4 ms	->	50 ms between updates should give a smooth update without affecting the overall speed
	Return V_Flag
end



Static Function XPSFitAssFitMany(DataWaves, FitFrom, FitTo, V_FitMaxIters, FitOptions, DoInitialFit, SaveFitCoefs)
//	Fits all waves in the DataWaves list of waves in the range given by FitFrom and FitTo, using a maxium number of iterations per fit of V_FitMaxIters and a V_FitOptions of FitOptions
//	The active coefficient waves are used as initial guesses for the first wave in the list, and the fit for the first wave is used as initial guesses for the second wave in the list, etc...
//	The coefficient waves for each fit are saved in the same data folder as the fitted wave
Wave/WAVE DataWaves
Variable FitFrom, FitTo, V_FitMaxIters, FitOptions, DoInitialFit, SaveFitCoefs
Variable i=0, ii=0, V_FitError=0, NumberOfDataPoints=0
String ConstraintsError=""


	//	TO DO LIST
	//	Asym1 is not allowed to be zero if Asym2 is not held constant. If that is the case Asym1 is forced to be 0.01
//	FitCoefWave[7,*;7]= ((FitCoefWave[p]==0) && (FinalFitPeakHoldWave[p+4]==0)) ? (0.01) : (FitCoefWave[p])



	//	Prints confirmation that the fit has started
	Variable NumberOfWaves=NumPnts(DataWaves)
	if (NumberOfWaves==1)
		Print("\rSingle fit started...")
	else
		Print("\rBulk fit started...")
	endif

	//	Creates a fit progress window
	XPSFitAssCreateProgressWindow(NumberOfWaves)

	//	Times how long the fits take
	Variable StartTime=DateTime
	Variable OverallFitTime=StartMSTimer

	//	Finds the number of peaks in the coefficient wave
	DFREF ProgramFolder=root:Programming
	DFREF TempFolder=ProgramFolder:Temp
	Wave/T CoefWave=ProgramFolder:XPSFitAssCoefConstraints
	Wave/B/U CoefSelWave=ProgramFolder:XPSFitAssCoefConstraintsSel
	Variable NumberOfPeaks=XPSFitAssNumberOfPeaks(CoefWave)

	//	Cleans up all waves in the temp folder	
	KillWavesInFolder(TempFolder)

	//	Duplicates the displayed data wave. This will be used to calculate a number of parameters, used for all the fits
	Wave  DataWave=ProgramFolder:XPSFitAssDataWave
	if ((FitFrom==0) && (FitTo==0))
		Duplicate/O DataWave, TempFolder:XPSFitAssTempDataWave/WAVE=ReducedDataWave
	else
		Duplicate/O/R=(FitFrom, FitTo) DataWave, TempFolder:XPSFitAssTempDataWave/WAVE=ReducedDataWave
	endif

	//	Converts the wave to double precision
	Redimension/D ReducedDataWave
	
	//	Finds the number of points in the wave
	NumberOfDataPoints=NumPnts(ReducedDataWave)



	//	Creates some of the temporary waves used for calculations by the all-at-once fit function. Creating the waves beforehand is faster than having to create them every time the fit function is called
	Make/O/D/N=(NumberOfPeaks+1) TempFolder:XPSFitAssAAOMulThreadConst/WAVE=ConstWave
	Make/O/WAVE/N=(NumberOfPeaks+1) TempFolder:XPSFitAssAAOMulThreadWaves/WAVE=TempWaves
	TempWaves[0]=ConstWave



	//	Creates the double-precision coefficient and epsilon waves for the fit. Before the fit coefficient wave holds the initial guesses, after the fit, the coefficient wave holds the fit result
	Variable NumberOfCoefs=3+NumberOfPeaks*7
	Make/FREE/D/O/N=(NumberOfCoefs) FitCoefWave, FitEpsilonWave
	FastOP FitCoefWave=0

	//	For the all-at-once structure fit function the first parameter has be held constant and set equal to 1
	FitCoefWave[0]=1
			
	//	The next two parameters are the linear background intercept and slope
	FitCoefWave[1]=Str2Num(CoefWave[1][1])	//	Intercept
	FitCoefWave[2]=Str2Num(CoefWave[1][5])	//	Slope
	
	//	Adds the peak parameters to the coef wave
	if (NumberOfPeaks>0)
		FitCoefWave[3,*]=Str2Num(CoefWave[4+Trunc((p-3)/7)][1+4*(p-3)-28*Trunc(4*(p-3)/28)])
	endif
	
	//	Shifting the spectrum close to x=0 will often make the fit faster and more stable, see DisplayHelpTopic "Errors Due to X Values with Large Offsets". Shifting the y values close to zero as well or normalising the areas of the peaks to 1 doesn't make any difference
	Variable XShift=pnt2x(ReducedDataWave, 0.5*(NumberOfDataPoints-1))
	FitCoefWave[1]+=XShift*FitCoefWave[2]		//	Intercept
	if (NumberOfPeaks>0)
		FitCoefWave[3,*;7]+=-XShift				//	Peak Positions
	endif
	
	//	Creates a double-precision wave used to shift the constraints with
	Duplicate/O/FREE FitCoefWave XShiftWave
	FastOP XShiftWave=0
	XShiftWave[1]=XShift*FitCoefWave[2]		//	Intercept
	if (NumberOfPeaks>0)
		XShiftWave[3,*;7]=-XShift					//	Peak Positions
	endif
	
	
	//	Creates a temporary double-precision wave to hold the results of all fits
	Make/O/D/FREE/N=(NumberOfCoefs, NumberOfWaves) AllFitCoefWave, AllShiftWave
	FastOP AllFitCoefWave=(NaN)
	FastOP AllShiftWave=0
	Make/O/D/FREE/N=(NumberOfWaves) ShiftTempWave
	FastOP ShiftTempWave=(XShift)
	for (i=0; i<NumberOfPeaks; i+=1)
		if (NumberOfWaves>1)
			ImageTransform/D=ShiftTempWave /G=(3+i*7) putRow AllShiftWave
		else
			AllShiftWave[3+i*7]=ShiftTempWave[0]
		endif
	endfor

			

	//	Adds values to the episilon wave. The epsilon values doesn't affect the speed of the fit, but if they are too small they may cause the fit to crash
	Variable XRange=Abs(LeftX(ReducedDataWave)-pnt2x(ReducedDataWave, NumPnts(ReducedDataWave)-1))
	if (NumberOfPeaks==0)
		FitEpsilonWave[0]=1e-6								//	Not used
		FitEpsilonWave[1]=1e-6*FitCoefWave[1]					//	Intercept					//	From Igor Exchange:
		FitEpsilonWave[2]=FitEpsilonWave[1]/XRange				//	Slope					//	The epsilon wave sets the size of the perturbation used to calculate the derivative. If you don't use an epsilon wave, Igor sets epsilon to:
	else
		Variable BiggestAreaOverFWHM=0
		//	Checks if the peak is a NEXAFS edge jump
		//	NEW NEW NEW
		if (FitCoefWave[9]==1e6)
			BiggestAreaOverFWHM=FitCoefWave[4]*FitCoefWave[9]	//	The change in intercept is scaled to be similar to a 1e-5 change in NEXAFS edge jump
		else
			BiggestAreaOverFWHM=FitCoefWave[4]/FitCoefWave[5]	//	The change in intercept is scaled to be similar to a 1e-5 change in peak area
		endif
		for (i=1; i<NumberOfPeaks; i+=1)
			//	Checks if the peak is a NEXAFS edge jump
			if (FitCoefWave[9+i*7]==1e6)
				BiggestAreaOverFWHM=Max(BiggestAreaOverFWHM, FitCoefWave[4+i*7]*FitCoefWave[9+i*7])	//	The change in intercept is scaled to be similar to a 1e-5 change in NEXAFS edge jump
			else
				BiggestAreaOverFWHM=Max(BiggestAreaOverFWHM, FitCoefWave[4+i*7]/FitCoefWave[5+i*7])	//	The change in intercept is scaled to be similar to a 1e-5 change in peak area
			endif
		endfor
		FitEpsilonWave[0]=1e-6								//	Not used
		FitEpsilonWave[1]=1e-5*BiggestAreaOverFWHM			//	Intercept					//	if your coefficient wave is single precision,
		FitEpsilonWave[2]=FitEpsilonWave[1]/XRange				//	Slope					//	eps[i] = 1e-4*coef[i], unless coef[i] is zero, in which case eps[i] = 1e-4
		FitEpsilonWave[3,*;7]=1e-4								//	Position
		FitEpsilonWave[4,*;7]=1e-5*FitCoefWave[p]				//	Area					//	if your coefficient wave is double precision,
		FitEpsilonWave[5,*;7]=1e-5*FitCoefWave[p]				//	FWHM					//	eps[i] = 1e-10*coef[i], unless coef[i] is zero, in which case eps[i] = 1e-10
		FitEpsilonWave[6,*;7]=1e-4								//	GL ratio
		FitEpsilonWave[7,*;7]=1e-4								//	Asym1
		FitEpsilonWave[8,*;7]=1e-4								//	Asym2
		FitEpsilonWave[9,*;7]=1e-6								//	Shirley
	endif



	//	Creates temporary waves for the hold values of the linear background parameters
	Make/O/FREE FinalFitLineHoldWave={((CoefSelWave[1][2][0]&16)==16), ((CoefSelWave[1][6][0]&16)==16)}
	Duplicate/O/FREE FinalFitLineHoldWave, InitialFitLineHoldWave
	
	if (NumberOfPeaks>0)		
	
		//	Creates a temporary wave for the hold parameters of the peaks
		Make/FREE/O/N=(7, NumberOfPeaks) FinalFitPeakHoldWave=((CoefSelWave[4+q][2+p*4][0]&16)==16)

		//	Creates a copy of the peak hold wave for the initial fit, with positions, GL ratios, and asymmetry held constant
		Duplicate/O/FREE FinalFitPeakHoldWave InitialFitPeakHoldWave
		InitialFitPeakHoldWave[0][]=1
		InitialFitPeakHoldWave[3,5][]=1
	endif



	//	Creates a temporary wave to hold all the constraints for the linear background
	Make/FREE/O/T LineConstraintsWave={CoefWave[1][3]+";", CoefWave[1][4]+";", CoefWave[1][7]+";", CoefWave[1][8]+";"}
			
	//	Creates temporary waves to hold the constraints for the linear background, where constraints belonging to parameters that are to be held constant are removed
	Duplicate/FREE/O/T LineConstraintsWave, FinalFitLineConstraintsWave, InitialFitLineConstraintsWave
	FinalFitLineConstraintsWave[]=SelectString(FinalFitLineHoldWave[Trunc(0.5*p)] , LineConstraintsWave[p], "")
	InitialFitLineConstraintsWave[]=SelectString(InitialFitLineHoldWave[Trunc(0.5*p)] , LineConstraintsWave[p], "")

	//	Creates the constraint strings for the two fits
	String FinalFitConstraints=FinalFitLineConstraintsWave[0]+FinalFitLineConstraintsWave[1]+FinalFitLineConstraintsWave[2]+FinalFitLineConstraintsWave[3]
	String InitialFitConstraints=InitialFitLineConstraintsWave[0]+InitialFitLineConstraintsWave[1]+InitialFitLineConstraintsWave[2]+InitialFitLineConstraintsWave[3]

	//	Creates a temporary wave to hold all the constraints for the peaks
	if (NumberOfPeaks>0)
		Make/FREE/T/O/N=(14, NumberOfPeaks) PeakConstraintsWave
		PeakConstraintsWave[0,*;2][]=CoefWave[4+q][3+p*2]+";"
		PeakConstraintsWave[1,*;2][]=CoefWave[4+q][4+(p-1)*2]+";"

		//	Creates temporary waves to hold the constraints for the peaks, where constraints belonging to parameters that are to be held constant are removed. Wave = (expression) ? (true) : (false) unfortunately doesn't work for text waves
		Duplicate/FREE/O/T PeakConstraintsWave, FinalFitPeakConstraintsWave, InitialFitPeakConstraintsWave
		FinalFitPeakConstraintsWave[][]=SelectString(FinalFitPeakHoldWave[Trunc(0.5*p)][q] , PeakConstraintsWave[p][q], "")
		InitialFitPeakConstraintsWave[][]=SelectString(InitialFitPeakHoldWave[Trunc(0.5*p)][q] , PeakConstraintsWave[p][q], "")

		//	Redimensions the hold and constraints waves to 1D waves to make the conversion to strings easier
		Variable NumberOfPeakCoefs=7*NumberOfPeaks
		Redimension /N=(NumberOfPeakCoefs) FinalFitPeakHoldWave, InitialFitPeakHoldWave
		Redimension /N=(NumberOfPeakCoefs*2) FinalFitPeakConstraintsWave, InitialFitPeakConstraintsWave
	
		//	Adds the constraints values for the peaks to the strings
		Variable NumberOfConstraints=NumberOfPeakCoefs*2
		for (i=0; i<NumberOfConstraints; i+=1)
			FinalFitConstraints+=FinalFitPeakConstraintsWave[i]
			InitialFitConstraints+=InitialFitPeakConstraintsWave[i]
		endfor
	endif
	


	//	Converts the user-friendly parameters P0, A0, W0, etc.. into the K0, K1, K2 values required by FuncFit
	Wave FinalFitConstraintsWave=XPSFitAssParseConstraintsString(FinalFitConstraints, NumberOfPeaks)
	Wave InitialFitConstraintsWave=XPSFitAssParseConstraintsString(InitialFitConstraints, NumberOfPeaks)
	
	//	Saves the current default data folder and changes the default folder to root:programming:Temp. This will ensure that the wave generated by FuncFit are all created in the temp folder
	DFREF CurrentFolder=GetDataFolderDFR()
	SetDataFolder TempFolder

	//	Creates the double-precision constraint matrix and wave for the initial fit from the text wave. It is much, much easier to apply the XShift to the M_and W_FitConstraint waves, than to the text wave, see DisplayHelpTopic "Constraint Matrix and Vector"
	FuncFit/C/O/N/Q/NTHR=1/W=2 EccentricXPS#XPSFitAssFitFuncDummy FitCoefWave ReducedDataWave /C=InitialFitConstraintsWave
	Wave/Z M_FitConstraint=TempFolder:M_FitConstraint
	Wave/Z W_FitConstraint=TempFolder:W_FitConstraint
	
	//	Checks if the constraints wave and matrix were created, if not an error is returned
	if (WaveExists(M_FitConstraint) && WaveExists(W_FitConstraint))

		//	Renames the constraint matrix
		Duplicate/O/FREE M_FitConstraint InitialM_FitConstraint

		//	Shifts the positions in the constraint wave by XShift
		if (NumPnts(M_FitConstraint)>0)
			MatrixOP/O/FREE InitialW_FitConstraint=W_FitConstraint + (M_FitConstraint x XShiftWave)
		else
			Duplicate/O/FREE W_FitConstraint, InitialW_FitConstraint
		endif
	
		//	Creates the double-precision constraints matrix and wave for the final fit from the text wave
		FuncFit/C/O/N/Q/NTHR=1/W=2 EccentricXPS#XPSFitAssFitFuncDummy FitCoefWave ReducedDataWave /C=FinalFitConstraintsWave

		//	Checks if the constraints wave and matrix were created, if not an error is returned
		if (WaveExists(M_FitConstraint) && WaveExists(W_FitConstraint))
	
			//	Renames the constraint matrix
			Duplicate/O/FREE M_FitConstraint FinalM_FitConstraint

			//	Shifts the positions in the constraint wave by XShift
			if (NumPnts(M_FitConstraint)>0)
				MatrixOP/O/FREE FinalW_FitConstraint=W_FitConstraint + (M_FitConstraint x XShiftWave)
			else
				Duplicate/O/FREE W_FitConstraint, FinalW_FitConstraint
			endif
		else

			//	Returns an error		
			ConstraintsError="The constraint matrix could not be created!"
		endif
	else
	
		//	Returns an error
		ConstraintsError="The constraint matrix could not be created!"
	endif
	
	


	//	Checks that all constraint waves were successfully created
	if (StrLen(ConstraintsError)==0)
	
		//	Creates an index wave equal to 0 if the row is a duplicate of an earlier row and should be ignored, 1 if the row is the first row in a hard-linked constraints pair, and 2 if the row is unique and should be treated as a normal constraint
		//	Hard linked pairs are rows with (<expression> <= <constant>) and ( - <same expression> <= - <same constant>)
		Wave InitialLinkIndex=XPSFitAssCreateLinkIndex(InitialM_FitConstraint, InitialW_FitConstraint)		//	TESTED
		Wave FinalLinkIndex=XPSFitAssCreateLinkIndex(FinalM_FitConstraint, FinalW_FitConstraint)			//	TESTED
	
		//	Extracts the hard-linked constraint rows, based on the final constraints
		Wave LinkM_FitConstraint=XPSFitAssChopConstraints(FinalM_FitConstraint, FinalLinkIndex, 1)		//	TESTED
		Wave LinkW_FitConstraint=XPSFitAssChopConstraints(FinalW_FitConstraint, FinalLinkIndex, 1)		//	TESTED
		
		//	Combines the hard-linked constraints into one matrix, with the constants in the first column (overwriting the K0 values)
		Duplicate/O/FREE LinkM_FitConstraint, LinkFullFitConstraint
		if (NumPnts(LinkM_FitConstraint)>0)
			FastOP LinkW_FitConstraint=-1*LinkW_FitConstraint
			ImageTransform/D=LinkW_FitConstraint /G=0 putCol LinkFullFitConstraint
		endif
		
		//	Does a sort-of Gaussian elimination on the hard-linked constraints to determine which of the parameters should be considered dependent and calculated from the other parameters
		Wave DependentCoefWave=XPSFitAssGaussElimination(LinkFullFitConstraint, NumberOfCoefs)		//	TESTED

		//	Calculates the wave used to hard-link coefficients from the hard-linked constraint rows
		Wave LinkMatrix=XPSFitAssCreateLinkMatrix(LinkFullFitConstraint, DependentCoefWave, NumberOfCoefs)			//	TESTED
		
		//	Removes the hard-linked rows from the initial constraints
		Wave/D InitialM_FitConstRed=XPSFitAssChopConstraints(InitialM_FitConstraint, InitialLinkIndex, 2)		//	TESTED
		Wave/D InitialW_FitConstRed=XPSFitAssChopConstraints(InitialW_FitConstraint, InitialLinkIndex, 2)		//	TESTED

		//	Removes the hard-linked rows from the final constraints
		Wave/D FinalM_FitConstRed=XPSFitAssChopConstraints(FinalM_FitConstraint, FinalLinkIndex, 2)		//	TESTED
		Wave/D FinalW_FitConstRed=XPSFitAssChopConstraints(FinalW_FitConstraint, FinalLinkIndex, 2)		//	TESTED
		
		//	Replaces any dependent, hard-linked coefficients in the constraints with independent coefficients
		XPSFitAssReplaceLinked(InitialM_FitConstRed, InitialW_FitConstRed, LinkMatrix, DependentCoefWave)
		XPSFitAssReplaceLinked(FinalM_FitConstRed, FinalW_FitConstRed, LinkMatrix, DependentCoefWave)
		
		//	FuncFit does not allow empty constraint waves
		if (NumPnts(InitialW_FitConstRed)==0)
			Make/O/FREE/D/N=(1, NumberOfCoefs) InitialM_FitConstRed=0
			Make/O/FREE/D/N=1 InitialW_FitConstRed=0
		endif
		if (NumPnts(FinalW_FitConstRed)==0)
			Make/O/FREE/D/N=(1, NumberOfCoefs) FinalM_FitConstRed=0
			Make/O/FREE/D/N=1 FinalW_FitConstRed=0
		endif

		//	Holds the dependent, hard-linked coefficients in the line hold waves
		FinalFitLineHoldWave[]=((FinalFitLineHoldWave[p]==1) || (DependentCoefWave[p+1]==1))
		InitialFitLineHoldWave[]=((InitialFitLineHoldWave[p]==1) || (DependentCoefWave[p+1]==1))

		//	Creates the hold strings for the two fits. The first parameter is used for the calculations of the hard-linked parameters and must always be equal to one and therefore held constant.
		String FinalFitHoldString="1"+Num2iStr(FinalFitLineHoldWave[0])+Num2iStr(FinalFitLineHoldWave[1])
		String InitialFitHoldString="1"+Num2iStr(InitialFitLineHoldWave[0])+Num2iStr(InitialFitLineHoldWave[1])

		if (NumberOfPeaks>0)
		
			//	Holds the dependent, hard-linked coefficients in the peak hold waves
			FinalFitPeakHoldWave[]=((FinalFitPeakHoldWave[p]==1) || (DependentCoefWave[p+3]==1))
			InitialFitPeakHoldWave[]=((InitialFitPeakHoldWave[p]==1) || (DependentCoefWave[p+3]==1))
			
			//	Adds the hold and constraints values for the peaks to the strings
			for (i=0; i<NumberOfPeakCoefs; i+=1)
				FinalFitHoldString+=Num2iStr(FinalFitPeakHoldWave[i])
				InitialFitHoldString+=Num2iStr(InitialFitPeakHoldWave[i])
			endfor
			
			//	Does not perform the initial fit, if all coefficients in the initial fit are to be held constant
			if ((WaveMin(InitialFitLineHoldWave)==1) && (WaveMin(InitialFitPeakHoldWave)==1))
				DoInitialFit=-1
			endif
		
			//	Returns an error if all coefficients are to be held constant in the final fit
			if ((WaveMin(FinalFitLineHoldWave)==1) && (WaveMin(FinalFitPeakHoldWave)==1))
				ConstraintsError="All parameters cannot be held constant. Check hold and constraint values!"
			endif
		else

			//	Does not perform the initial fit, if all coefficients in the initial fit are to be held constant
			if (WaveMin(InitialFitLineHoldWave)==1)
				DoInitialFit=-1
			endif
		
			//	Returns an error if all coefficients are to be held constant in the final fit
			if (WaveMin(FinalFitLineHoldWave)==1)
				ConstraintsError="All parameters cannot be held constant. Check hold and constraint values!"
			endif
		endif
		
		//	Does not perform the initial fit, if there are no difference between the fits
		if (CmpStr(FinalFitHoldString, InitialFitHoldString)==0)
			DoInitialFit=-1
		endif
	endif
	
	
		
	if (StrLen(ConstraintsError)==0)
	
		//	Creates a temporary wave holding the information from WaveNumber, FitTime, V_FitError, V_FitQuitReason, V_FitNumIters for each of the fits
		Make/FREE/N=(NumberOfWaves*2, 5) InfoWave
		FastOP InfoWave=0

		//	Initializes the structure used by the all-at-once structure fit function
		STRUCT XPSFitAssFitAAOSTRUCT FitFuncStruct
		//Wave FitFuncStruct.coefw						//	The fit coefficients				(will be initialized by FuncFit)
		//Wave FitFuncStruct.yw							//	The y values of the fit				(will be initialized by FuncFit)
		//Wave FitFuncStruct.xw							//	The x values of the fit				(will be initialized by FuncFit)
		FitFuncStruct.NumberOfPeaks=NumberOfPeaks		//	The number of peaks in the fit
		Wave FitFuncStruct.TempWaves=TempWaves		//	Wave references to preexisting waves used for temporary storage during the calculations
		Wave FitFuncStruct.LinkMatrix=LinkMatrix			//	A matrix used to calculate the actual coefficients from the input coefficients




		//	NEW NEW NEW

//		//	Creates a temporary wave used during the fit [I WOULD NEED ONE PER THREAD]
//		Duplicate/O/D/FREE FitCoefWave, UsedFitCoefWave
//		
//		//	Extracts the dependent (hard-linked) coefficients NEW
//		Extract/O/FREE/INDX DependentCoefWave, HardLinkedIndex, (DependentCoefWave==1)
//		InsertPoints 0, 1, HardLinkedIndex
//		HardLinkedIndex[0]=NumPnts(HardLinkedIndex)+1
//		
//		//	Creates an index poiting to the non-zero values in the link matrix NEW
//		Make/O/I/U/FREE/N=(NumberOfCoefs, NumberOfCoefs+1) LinkMatrixIndex
//		FastOP LinkMatrixIndex=0
//		for (i=0; i<NumberOfCoefs; i+=1)
//			Duplicate/O/FREE/R=[i][] LinkMatrix, TempMatrix
//			Extract/O/FREE/INDX TempMatrix, TempIndex, (TempMatrix!=0)
//			LinkMatrixIndex[i][0]=NumPnts(TempIndex)+1
//			if (LinkMatrixIndex[i][0]>0)
//				LinkMatrixIndex[i][1, LinkMatrixIndex[i][0]-1]=TempIndex[p-1]
//			endif
//		endfor
//		
//		//	Initializes the structure used by the threadsafe, all-at-once, structure fit function
//		STRUCT XPSFitAssFitStructNEW FitFuncStruct
//		//Wave FitFuncStruct.coefw						//	The fit coefficients				(will be initialized by FuncFit)
//		//Wave FitFuncStruct.yw							//	The y values of the fit				(will be initialized by FuncFit)
//		//Wave FitFuncStruct.xw							//	The x values of the fit				(will be initialized by FuncFit)
//		FitFuncStruct.NumberOfPeaks=NumberOfPeaks		//	The number of peaks in the fit
//		Wave/WAVE FitFuncStruct.TempWaves=TempWaves		//	Wave references to preexisting waves used for temporary storage during the calculations
//		Wave/D FitFuncStruct.LinkMatrix=LinkMatrix		//	A matrix used to calculate the actual coefficients from the input coefficients
//		Wave/I/U FitFuncStruct.LinkMatrixIndex=LinkMatrixIndex
//		Wave/D FitFuncStruct.UsedCoefs=UsedFitCoefWave
//		Wave/WAVE FitFuncStruct.UsedCoefsIndexWaves=xxxxxxxxxxxxxxxxxxx
//		Wave/I/U FitFuncStruct.HardLinkedIndex=HardLinkedIndex
//		Wave/B/U FitFuncStruct.PeakPertubed=xxxxxxxxxxxxxxxxxxxxx
		
		
		
		//	Initializes the structure for the XPSFitAssFitSingle function
		STRUCT XPSFitAssFitSingleSTRUCT s
		Wave s.DataWave=ReducedDataWave				//	The wave to be fitted
		Wave s.FitCoefWave=FitCoefWave					//	The coefficients for the fit
		Wave s.FitEpsilonWave=FitEpsilonWave			//	The epsilon wave for the fit
		Wave s.InitialM_FitConstraint=InitialM_FitConstRed	//	The constraint matrix for the initial fit
		Wave s.InitialW_FitConstraint=InitialW_FitConstRed	//	The constraint wave for the initial fit
		Wave s.FinalM_FitConstraint=FinalM_FitConstRed	//	The constraint matrix for the final fit
		Wave s.FinalW_FitConstraint=FinalW_FitConstRed	//	The constraint wave for the final fit
		s.InitialFitHoldString=InitialFitHoldString				//	The hold string for the initial fit
		s.FinalFitHoldString=FinalFitHoldString				//	The hold string for the final fit
		s.V_FitMaxIters=V_FitMaxIters						//	The maximum number of iterations per fit
		s.FitOptions=FitOptions							//	The fit options. Set this to 4 to hide the flashing fit window, see DisplayHelpTopic "Special Variables for Curve Fitting"
		Wave s.InfoWave=InfoWave						//	A wave holding the information from FitTime, V_FitError, V_FitQuitReason, V_FitNumIters for each of the fits
		s.i=0											//	The index of the wave currently being fitted
		s.FitFuncStruct=FitFuncStruct						//	The structure used by the all-at-once structure fit function
		s.DoInitialFit=DoInitialFit


		//	Creates variables and waves for the loop
		Variable TempA=0, TempB=0, LowRange=0, HighRange=0, StopLoop=0
		String ErrorStr=""
		Make/O/D/FREE/N=(NumberOfPeaks) RangeTestWave
		FastOP RangeTestWave=0
		
		//	Variables related to the fit progress window
		Variable BatchSize=1
		Variable BatchCounter=1
		Variable BatchTime=0
		Variable BatchTimerRefNum=StartMsTimer
		Variable BatchTargetTime=50e3	//	Will attempt to update the progress window every 50 ms
		Variable AbortPressed0=0
		Variable AbortPressed1=0
		Variable ProcessorCount=ThreadProcessorCount
		Variable MultiThreadStart=0
		Variable MultiThreadEnd=-1
		
		//	Propagates the fit through the list of waves
		for (i=0; (i<NumberOfWaves) && (StopLoop==0); i+=1)
			
			//	The next wave to be fitted.
			Wave/Z SourceWave=DataWaves[i]
	
			//	Returns an error if the wave to be fitted doesn't exist
			if (WaveExists(SourceWave)==0)
		
				//	Stops the loop and returns error code 1
				StopLoop=1
				ErrorStr="Fit failed for spectrum "+Num2iStr(i)+": Wave does not exist!"
			endif
			
			if (StopLoop==0)
				
				//	Reduces the wave to the specified range.			//	4 us
				if ((FitFrom==0) && (FitTo==0))
					Duplicate/O SourceWave, TempFolder:XPSFitAssTempDataWave/WAVE=ReducedDataWave
				else
					Duplicate/O/R=(FitFrom, FitTo) SourceWave, TempFolder:XPSFitAssTempDataWave/WAVE=ReducedDataWave
				endif
				
				//	Converts the wave to double precision. FuncFit seems to do this automatically, but better safe than sorry.
				Redimension/D ReducedDataWave		//	0.8 us
				
				//	Shifts the wave by the predetermined amount to get it closer to x=0
				SetScale/P x, LeftX(ReducedDataWave)-XShift, DeltaX(ReducedDataWave), ReducedDataWave
				
				if (NumberOfPeaks>0)
				
					//	Calculates the number of data points in the reduced wave
					NumberOfDataPoints=NumPnts(ReducedDataWave)
				
					//	Checks if any of the peak positions fall outside the range of the wave		//	2 us
					TempA=LeftX(ReducedDataWave)
					TempB=pnt2x(ReducedDataWave, NumberOfDataPoints-1)
					LowRange=Min(TempA, TempB)
					HighRange=Max(TempA, TempB)
					
					//	A value of one means the peak position is outside the range of the wave, a value of zero means the peak position is inside the range of the wave
					RangeTestWave[]=(FitCoefWave[3+p*7]!=limit(FitCoefWave[3+p*7], LowRange, HighRange))

					//	Stops the loop and returns error code 2 if at least one peak position is outside the range of the wave
					if (WaveMax(RangeTestWave)!=0)
						StopLoop=2
						ErrorStr="Fit failed for spectrum "+Num2iStr(i)+" "+PossiblyQuoteName(NameOfWave(SourceWave))+": One of more peaks are outside the range of the wave to be fitted!"
					endif
				endif
			endif
			
			if (StopLoop==0)	
					
				//	Creates a series of temporary waves used for calculations by the all-at-once fit function. Creating the waves beforehand is faster than having to create them every time the fit function is called
				for (ii=0; ii<NumberOfPeaks; ii+=1)
					Duplicate/O/D ReducedDataWave, TempFolder:$("XPSFitAssFitTempAAOWave"+Num2Str(ii))/WAVE=TempAAOWaveX
					TempWaves[ii+1]=TempAAOWaveX
				endfor				//	6 us
								
				//	Updates the index in the structure
				 s.i=i
	
				////////////////////////////////			Fits the spectrum			/////////////////////////////////////////
				V_FitError=XPSFitAssFitSingle(s)

				if (V_FitError==0)
						
					//	Saves the fit coefficients in the temporary wave
					ImageTransform/D=FitCoefWave /G=(i) putCol AllFitCoefWave		//	2 us
				else
				
					//	Stops the loop and returns error code 3
					StopLoop=3
					ErrorStr="Fit failed for spectrum "+Num2iStr(i)+" "+PossiblyQuoteName(NameOfWave(SourceWave))+": V_FitError = "+Num2iStr(V_FitError)+" \""+XPSFitAssFitError(V_FitError)+"\"!"
				endif
			endif
			
			if (BatchCounter<BatchSize)
				
				//	Increases the batch counter by one
				BatchCounter+=1
			else

				//	Updates the counters and progress bars for the fit progress window. Returns 2 if the abort button has been pressed
				AbortPressed0=XPSFitAssUpdateProgressWindow(StartTime, i+1, 0)			//	0.4 ms
					
				//	Stops the fitting loop and returns error code 4
				if (AbortPressed0==2)
					StopLoop=4
					ErrorStr="Fit cancelled by user at spectrum "+Num2iStr(i)+" "+PossiblyQuoteName(NameOfWave(SourceWave))+"!"
				endif
					
				//	Calculates the time used to finish the last batch of spectra
				BatchTime=StopMSTimer(BatchTimerRefNum)
					
				//	Resets the batch counter
				BatchCounter=1
					
				//	Calculates the batch size needed to give one update of the progress window every BatchTargetTime microseconds
				BatchSize=Ceil(BatchTargetTime*BatchSize/BatchTime)
					
				//	Restarts the batch timer
				BatchTimerRefNum=StartMsTimer
			endif
		endfor
		
		//	Stops the batch timer if it is still running
		BatchTime=StopMSTimer(BatchTimerRefNum)
		
		//	The number of the last wave in the loop
		Variable LastWaveNumber=i-1
		
		//	If an error occurred the last wave in the loop was not fitted successfully. Only the results of the successfully fitted waves (up to and including LastWaveNumber) will be saved
		if ((StopLoop!=0) && (AbortPressed0!=2))
			LastWaveNumber+=-1
		endif
		
		//	The result of the fits will not be saved if Abort has been pressed
		if (AbortPressed0==2)
			SaveFitCoefs=0
		endif

		//	Corrects the fit coefficients for x shift and linked coefficients
		MatrixOP/O/FREE CorrectedAllFitCoefWave=(LinkMatrix x AllFitCoefWave)+AllShiftWave
		MatrixOP/O/FREE CorrectedInterceptWave=(row(CorrectedAllFitCoefWave, 1)-XShift*row(CorrectedAllFitCoefWave, 2))^t
		if (NumberOfWaves>1)
			ImageTransform/D=CorrectedInterceptWave /G=1 putRow CorrectedAllFitCoefWave
		else
			CorrectedAllFitCoefWave[1]=CorrectedInterceptWave[0]
		endif
		
		//	Does not save the fit coefficients if SaveFitCoefs==0
		//	The result of the fit can still be accessed in root:Programming:Temp:XPSFitAssFitCoefs
		if (SaveFitCoefs==1)
		
			//	Deletes all existing fit waves. Fastest way?? Delete folders??
			//	SHOULD OLD FITS BE DELETED EVEN IF THE NEW FITS FAILED??

			//	Creates the new fit waves
			if (LastWaveNumber>=0)
				
				//	Creates a wave to hold the data folders for the saved fits
				Make/O/FREE/DF/N=(NumberOfWaves) FitFolders
		
				//	Uses one spectrum per processor core as the initial batch size
				BatchSize=1
			
				//	Starts from the first spectrum
				MultiThreadStart=0

				//	Saves the fit coefficients in batches, updating the progress window every BatchTargetTime microseconds 
				for (MultiThreadStart=0; (MultiThreadEnd<LastWaveNumber && AbortPressed1!=2); MultiThreadStart=MultiThreadEnd+1)
				
					//	Restarts the batch timer
					BatchTimerRefNum=StartMsTimer
			
					//	Includes Batchsize number of spectra per processor core in the calculation, up until the last spectrum
					MultiThreadEnd=MultiThreadStart+BatchSize*ProcessorCount-1
					MultiThreadEnd=(MultiThreadEnd<LastWaveNumber) ? (MultiThreadEnd) : (LastWaveNumber)

					//	Creates the necessay data folders for the saved fits. This is done outside the multithread, since it has a tendency to crash otherwise, probably because two treads both start creating the same data folder
					//	KillWaves is supposed to be ThreadSafe, but any attempt at using KillWaves in a multithreaded function results in no waves being killed
					//	KillDataFolder is also supposed to be ThreadSafe, but any attempt at using KillDataFolder in a multithreaded function crashes Igor. However, making the function ThreadSafe while NOT multithreading it, speeds it up by a factor of 2...???
					FitFolders[MultiThreadStart, MultiThreadEnd]=XPSFitAssSaveFitFolders(DataWaves[p])
		
					//	Creates coefficient waves for each fitted data wave
					MultiThread DataWaves[MultiThreadStart, MultiThreadEnd]=XPSFitAssSaveFit(p, DataWaves[p], FitFolders[p], CoefWave, CoefSelWave, CorrectedAllFitCoefWave, NumberOfPeaks, FitFrom, FitTo)

					//	Updates the counters and progress bars for the fit progress window. Returns 2 if the abort button has been pressed
					AbortPressed1=XPSFitAssUpdateProgressWindow(StartTime, MultiThreadEnd+1, 1)			//	0.4 ms
					
					//	Calculates the time used to finish the last batch of spectra
					BatchTime=StopMSTimer(BatchTimerRefNum)
					
					//	Calculates the batch size needed to give one update of the progress window every BatchTargetTime microseconds
					BatchSize=Ceil(BatchTargetTime*BatchSize/BatchTime)
				endfor
			endif
		
			//	Deletes any remaining old fits for the waves that were not fitted successfully
			//	SHOULD THESE BE KEPT OR DELETED?
//			if (LastWaveNumber+1<NumberOfWaves)
//				FitFolders[LastWaveNumber+1, *]=XPSFitAssDeleteFitFolders(DataWaves[p])
//			endif
		endif

		//	Stops the timer and creates a string with the time in miliseconds, seconds or minutes
		OverallFitTime=StopMSTimer(OverallFitTime)/1000
		String OverallFitTimeStr=""
		if (OverAllFitTime<1e6)
			OverallFitTimeStr=Num2Str(OverallFitTime/1e3)+" ms"
		elseif (OverAllFitTime<60e6)
			OverallFitTimeStr=Num2Str(OverallFitTime/1e6)+" s"
		else
			OverallFitTimeStr=Num2Str(OverallFitTime/60e6)+" min"
		endif
	
		//	Checks if an error occurred during the loop, and prints a small information string about the error, if any
		String PrintString=""
		if (StrLen(ErrorStr)==0)
			if (NumberOfWaves==1)
				PrintString="\r\t"+PossiblyQuoteName(NameOfWave(DataWaves[0]))+" fitted succesfully in "+OverallFitTimeStr
			else
				PrintString="\r\t"+Num2iStr(NumberOfWaves)+" waves fitted succesfully in "+OverallFitTimeStr
			endif
		else		
			PrintString="\r\t"+ErrorStr
		endif
		
		//	Prints a small information string if the user pressed Abort before the last spectrum during the saving of the fit coefficients
		if ((AbortPressed1==2) && (MultiThreadEnd<LastWaveNumber))
			PrintString+="\r\tSaving of fit coefficients cancelled by user at spectrum "+Num2iStr(MultiThreadEnd)+" "+PossiblyQuoteName(NameOfWave(DataWaves[MultiThreadEnd]))
		endif
		
		//	Extract an index wave pointing to all non-zero V_FitQuitReason values in the info wave
		Duplicate/O/FREE/R=[][3] InfoWave QuitReasonWave
		Extract/FREE/INDX/O QuitReasonWave, IndexWave, (QuitReasonWave!=0)

		Variable NumberOfIndexPoints=NumPnts(IndexWave)
		Variable QuitReason=0
		Variable WaveNumber=0

		//	Prints a small information string about all non-zero V_FitQuitReason values in the info wave
		for (i=0; i<NumberOfIndexPoints; i+=1)
			WaveNumber=InfoWave[IndexWave[i]][0]
			QuitReason=InfoWave[IndexWave[i]][3]
			PrintString+="\r\tProblem at spectrum "+PossiblyQuoteName(NameOfWave(DataWaves[WaveNumber]))+": V_FitQuitReason = "+Num2iStr(QuitReason)+" \""+XPSFitAssQuitReason(QuitReason)+"\"!"
		endfor

		//	Prints the fit summary including error to the history
		Print(PrintString)
	endif
	
	//	Stops the timer, if it is still running
	OverallFitTime=StopMSTimer(OverallFitTime)
	
	//	Cleans up the temporary waves
	KillWavesInFolder(TempFolder)

	//	Copies the temporary wave with the fit information into the temp folder, where it will be accessable for other functions
	if (WaveExists(InfoWave))
		Duplicate/O InfoWave TempFolder:XPSFitAssFitInfoWave
	endif
	
	//	Copies the wave with all fit coefficients into the temp folder, where it will be accessable for other functions. The first coefficient is a dummy and will not be copied
	if (WaveExists(CorrectedAllFitCoefWave))
		Duplicate/O/R=[1,*][] CorrectedAllFitCoefWave, TempFolder:XPSFitAssFitCoefs
	endif
	
	//	Kills the progress window
	DoWindow /K FitProgressWindow

	//	Changes the default data folder back to the original folder
	SetDataFolder CurrentFolder
	
	//	Display an error if the constraint matrix could not be created or if all coefficients are to be held constant in the final fit
	if (StrLen(ConstraintsError)>0)
		ErrorMessage(ConstraintsError)
	endif
end



Static Function/WAVE XPSFitAssCreateLinkMatrix(LinkFullFitConstraint, DependentCoefWave, NumberOfCoefs)
//	Converts the hard-linked constraints into a matrix that will be used by the fit function to calculates the values of the fit coefficients, taking dependency into account.
Wave LinkFullFitConstraint, DependentCoefWave
Variable NumberOfCoefs
	
	//	Creates the hard-link coefficient matrix and converts the integer matrix to double-precision
  	MatrixOP/O/FREE LinkMatrix=Identity(NumberOfCoefs)
  	Redimension/D LinkMatrix
  	
	//	Extracts the dependent (hard-linked) coefficients
	Extract/O/FREE/INDX DependentCoefWave, IndexWave, (DependentCoefWave==1)
	Variable NumberOfDependent=NumPnts(IndexWave)
	
	//	Adjusts the matrix for each dependent coefficient
	Variable i=0
	for (i=0; i<NumberOfDependent; i+=1)
		LinkMatrix[IndexWave[i]][]+=-LinkFullFitConstraint[i][q]
	endfor
	
	//	Returns the hard-link matrix
	Return LinkMatrix
end



Static Function XPSFitAssReplaceLinked(M_FitConstraint, W_FitConstraint, LinkMatrix, DependentCoefWave)
//	Replaces any dependent, hard-linked coefficients in the constraints with independent coefficients
Wave M_FitConstraint, W_FitConstraint, LinkMatrix, DependentCoefWave

	if (NumPnts(W_FitConstraint)>0)
	
		//	Extracts the dependent (hard-linked) coefficients
		Extract/O/FREE/INDX DependentCoefWave, IndexWave, (DependentCoefWave==1)
		Variable NumberOfDependent=NumPnts(IndexWave)

		//	Creates a teporary wave		
		Duplicate/O/FREE M_FitConstraint, TempWave
		
		//	Adjusts the matrix for each dependent coefficient
		Variable i=0
		for (i=0; i<NumberOfDependent; i+=1)
		
			//	Replaces any dependent, hard-linked coefficients in the constraints with independent coefficients
			TempWave[][]=M_FitConstraint[p][IndexWave[i]]*LinkMatrix[IndexWave[i]][q]
			W_FitConstraint[]+=M_FitConstraint[p][IndexWave[i]]*LinkMatrix[IndexWave[i]][0]
			
			FastOP M_FitConstraint=M_FitConstraint+TempWave
			M_FitConstraint[][IndexWave[i]]=0
			M_FitConstraint[][0]=0
		endfor
	endif
end



Static Function/WAVE XPSFitAssCreateLinkIndex(M_FitConstraint, W_FitConstraint)
//	Creates a wave used to hold information about which coefficients are hard linked through (<expression> <= <constant>) and ( - <same expression> <= - <same constant>) constraint pairs
Wave M_FitConstraint, W_FitConstraint

	//	Creates an index wave to hold information about the constraint rows
	Duplicate/O/FREE W_FitConstraint LinkIndex

	if (NumPnts(LinkIndex)>0)

		//	Creates a temporary constraints matrix, which includes the W values
		Duplicate/O/FREE M_FitConstraint, FullMatrix
		InsertPoints/M=1 0, 1, FullMatrix
		ImageTransform/D=W_FitConstraint /G=0 putCol FullMatrix
	
		//	Normalizes the rows to make equivalent rows (eq1 = eq2 * a) identical. The normalization factor is always positive and will not change the signs of the values in the rows
		MatrixOP/O/FREE FullNormMatrix=NormalizeRows(FullMatrix)
	
		//	Analyzes the matrix, row by row. Returns 0 if the row is a duplicate of an earlier row and should be ignored, 1 if the row is the first row in a hard-linked constraints pair, and 2 if the row is unique and should be treated as a normal constraint
		MultiThread LinkIndex=XPSFitAssLinkParseRows(p, FullNormMatrix)
	endif
	
	//	Returns the index wave
	Return LinkIndex
end



ThreadSafe Static Function XPSFitAssLinkParseRows(RowNumber, FullNormMatrix)
//	Returns 0 or 1 depending on whether the constraints in the given row represent a hard link
Variable RowNumber
Wave FullNormMatrix

	//	Caculates the number of constraint rows
	Variable NumberOfRows=DimSize(FullNormMatrix, 0)

	//	Creates a copy of the matrix
	Duplicate/O/FREE FullNormMatrix, TestMatrix1, TestMatrix2
	FastOP TestMatrix1=1
	
	//	Tests if the row is equivalent to an earlier row, or equivalent, but of opposite sign, to an earlier row. The 1e-8 is necessary, because of rounding errors, causing identical numbers to not be exactly identical
	if (RowNumber>0)
		TestMatrix1[0, RowNumber-1][]=(Abs((FullNormMatrix[RowNumber][q]-FullNormMatrix[p][q])/FullNormMatrix[RowNumber][q])>1e-8)
		TestMatrix2[0, RowNumber-1][]=(Abs((FullNormMatrix[RowNumber][q]+FullNormMatrix[p][q])/FullNormMatrix[RowNumber][q])>1e-8)
	endif
	
	//	Tests if the row is equivalent, but of opposite sign, to a later row. The 1e-8 is necessary, because of rounding errors, causing identical numbers to not be exactly identical
	if (RowNumber<NumberOfRows-1)
		TestMatrix2[RowNumber+1, *][]=(Abs((FullNormMatrix[RowNumber][q]+FullNormMatrix[p][q])/FullNormMatrix[RowNumber][q])>1e-8)
	endif
	
	//	Sums up the rows. A sum of zero means the row is equivalent to another row
	MatrixOP/O/FREE TestWave=sumRows(TestMatrix1)*sumRows(TestMatrix2)
	
	//	0 = The row is a duplicate of an erlier row and should be ignored, 2 = The row is unique and should be treated as a normal constraint
	TestWave[]=(TestWave[p]!=0) ? (2) : (0)

	//	1 = The row is the first row in a hard-linked constraints pair
	if (RowNumber<NumberOfRows-1)
		TestWave[RowNumber+1, *]=(TestWave[p]==0) ? (1) : (TestWave[p])
	endif

	//	Does not compare the row with itself
	TestWave[RowNumber]=2
	
	//	Returns 0 if the row is a duplicate of an erlier row and should be ignored, 1 if the row is the first row in a hard-linked constraints pair, and 2 if the row is unique and should be treated as a normal constraint
	Return WaveMin(TestWave)
end



Static Function/WAVE XPSFitAssChopConstraints(FullWave, LinkIndex, LinkValue)
//	Returns the rows in FullWave for which LinkIndex is equal to LinkValue
Wave FullWave, LinkIndex
Variable LinkValue

	//	Extracts the rows in FullWave for which LinkIndex is equal to LinkValue. The extracted wave will always be a 1D wave
	Extract/O/FREE/D FullWave, ExtractedWave, (LinkIndex[p]==LinkValue)
	Variable NumCol=DimSize(FullWave, 1)

	//	Converts the extracted wave back to the original number of columns
	if (NumCol>0)
		Variable NewNumRows=NumPnts(ExtractedWave)/NumCol
		Redimension/N=(NewNumRows, NumCol) ExtractedWave
	endif

	//	Returns the extracted wave
	Return ExtractedWave
end



Static Function/WAVE XPSFitAssGaussElimination(LinkFullFitConstraint, NumberOfCoefs)
//	Does a sort-of Gaussian elimination on the hard-linked constraints matrix
Wave LinkFullFitConstraint
Variable NumberOfCoefs

	//	Calculates the number of hard-linked (dependent) constraints
	Variable NumberOfConstraints=DimSize(LinkFullFitConstraint, 0)
	
	//	Creates a wave to hold information about whether a coefficient should be considered dependent (=1) or independent (=0)
	Make/O/FREE/D/N=(NumberOfCoefs) DependentCoefWave
	FastOP DependentCoefWave=0
	
	if (NumberOfConstraints>0)
	
		if (NumberOfConstraints>=NumberOfCoefs)
	
			//	If there are as many hard-linked constraints as coefficients, only one combination of coefficients is allowed, and it is not possible to vary them.
			FastOP DependentCoefWave=1
		else

			//	The number of dependent parameters should end up equal to the number of hard-linked constraints
			Variable NumberOfDependent=0
	
			//	Counts backward through the coefficients. This will make the later coefficients dependent and the earlier coefficients independent
			Variable i=0, ii=0, n=0, NormFactor=0, NormRow=0, ActiveRow=0
			for (i=NumberOfCoefs-1; (i>0) && (NumberOfDependent<NumberOfConstraints); i+=-1)

				//	Calculates the number of times the coefficient number i appears in the hard-linked constraints
				Duplicate/O/FREE/R=[][i] LinkFullFitConstraint, SingleColumn
				Extract/O/FREE/INDX SingleColumn, IndexWave, (SingleColumn[p]!=0)
				n=NumPnts(IndexWave)
		
				if (n==0)
		
					//	The coefficient is independent
					DependentCoefWave[i]=0
					
				elseif (IndexWave[n-1]<NumberOfDependent)
		
					//	The coefficient is independent
					DependentCoefWave[i]=0
				else

					//	The coefficient is considered dependent
					DependentCoefWave[i]=1
					NumberOfDependent+=1

					//	The row to normalise with
					NormRow=IndexWave[n-1]
					NormFactor=LinkFullFitConstraint[NormRow][i]

					//	Normalises the row with coefficient i
					LinkFullFitConstraint[NormRow][]=LinkFullFitConstraint[NormRow][q]/NormFactor

					//	Counts through the unused hard-linked constraints
					for (ii=0; ii<n-1; ii+=1)

						//	The row to subtract from
						ActiveRow=IndexWave[ii]
						NormFactor=LinkFullFitConstraint[ActiveRow][i]

						//	Makes all other rows independent of coefficient i
						LinkFullFitConstraint[ActiveRow][]+=-LinkFullFitConstraint[NormRow][q]*NormFactor
					endfor
			
					//	Moves the row to the beginning of the matrix
					InsertPoints/M=0 0, 1, LinkFullFitConstraint
					LinkFullFitConstraint[0][]=LinkFullFitConstraint[NormRow+1][q]
					DeletePoints/M=0 NormRow+1, 1, LinkFullFitConstraint
				endif
			endfor
		endif
	endif

	//	Returns the information about which coefficients are dependent
	Return DependentCoefWave
end



Static Function/DF XPSFitAssSaveFitFolders(DataWave)
//	Creates the data folders for the saved fits. Kills any old folder and all waves within which may already exist
Wave/Z DataWave
DFREF FitFolder=$""
	
	if (WaveExists(DataWave))
		
		//	Finds the data folder the wave is located in
		DFREF ActiveFolder=GetWavesDataFolderDFR(DataWave)
	
		//	Creates the subfolders to hold the fit waves and coefficients
		NewDataFolder/O ActiveFolder:SavedFits
		DFREF SavedFitsFolder=ActiveFolder:SavedFits
		String DataWaveName=NameOfWave(DataWave)
		FitFolder=SavedFitsFolder:$DataWaveName
		if (DataFolderRefStatus(FitFolder)!=0)
			KillDataFolder/Z FitFolder
		endif
		NewDataFolder/O SavedFitsFolder:$DataWaveName
		FitFolder=SavedFitsFolder:$DataWaveName
	endif
	
	//	Returns the created folder
	Return FitFolder
end



Static Function/DF XPSFitAssDeleteFitFolders(DataWave)
//	Deletes the data folders for the saved fits
Wave/Z DataWave

	if (WaveExists(DataWave))
		
		//	Finds the data folder the wave is located in
		DFREF ActiveFolder=GetWavesDataFolderDFR(DataWave)
	
		//	Deletes the folder that holds the fit waves and coefficients, if it exists
		DFREF SavedFitsFolder=ActiveFolder:SavedFits
		if (DataFolderRefStatus(SavedFitsFolder)!=0)
			String DataWaveName=NameOfWave(DataWave)
			DFREF FitFolder=SavedFitsFolder:$DataWaveName
			if (DataFolderRefStatus(FitFolder)!=0)
				KillDataFolder/Z FitFolder
			endif
		endif
	endif
	
	//	Returns an invalid folder
	Return $""
end



ThreadSafe Static Function/WAVE XPSFitAssSaveFit(Num, DataWave,  FitFolder, CoefWave, CoefSelWave, AllFitCoefWave, NumberOfPeaks, DrawFrom, DrawTo)
//	Saves the fit and coefficient waves in a subfolder of the data wave folder
Variable Num
Wave/Z DataWave
DFREF FitFolder
Wave/T CoefWave
Wave/B/U CoefSelWave
Wave AllFitCoefWave
Variable NumberOfPeaks, DrawFrom, DrawTo
	
	if (WaveExists(DataWave))
		
		//	Finds the name of the fitted wave
		String DataWaveName=NameOfWave(DataWave)
	
		//	Copies the fit coefficient waves into the subfolder
		Duplicate/O/T CoefWave, FitFolder:$(DataWaveName+"_lst")/WAVE=TargetCoefWave
		Duplicate/O/B/U CoefSelWave, FitFolder:$(DataWaveName+"_sel")
		
		//	Shifts the spectrum back to the correct x position and writes the fitted parameters for the linear background into the coefficient wave
		TargetCoefWave[1][1]=Num2Str(AllFitCoefWave[1][Num])	//	Intercept
		TargetCoefWave[1][5]=Num2Str(AllFitCoefWave[2][Num])	//	Slope
			
		//	Writes the fitted parameters for the peaks into the coefficient wave
		if (NumberOfPeaks>0)
			TargetCoefWave[4,3+NumberOfPeaks][1,*;4]=Num2Str(AllFitCoefWave[3+(q-1)/4+(p-4)*7][Num])
		endif
		
		//	Calculates the total area of all peaks in the fit combined
		Variable TotalArea=0, i=0
		for (i=0; i<NumberOfPeaks; i+=1)
			TotalArea+=AllFitCoefWave[4+i*7][Num]
		endfor
		
		//	Adds the total area to the coefficient wave
		TargetCoefWave[NumberOfPeaks+6][5]=Num2Str(TotalArea)
	
		//	Finds the number of data points in the fitted wave
		Variable NumberOfDataPoints=NumPnts(DataWave)

		//	Creates a temporary wave with a ten times finer x-resolution than the measured datawave's
		if (NumberOfDataPoints>0)
			Make/FREE/O/N=((NumberOfDataPoints-1)*10+1) BaseFitWave
		else
			Make/FREE/O/N=0 BaseFitWave
		endif
		SetScale/P x, LeftX(DataWave), 0.1*DeltaX(DataWave), "", BaseFitWave
	
		//	Creates the waves to hold background and fit. If DrawFrom and DrawTo are both different from zero, only the range between them is duplicated
		if ((DrawFrom==0) && (DrawTo==0))
			Duplicate/O BaseFitWave, FitFolder:$(DataWaveName+"_bck")/WAVE=BckWave, FitFolder:$(DataWaveName+"_fit")/WAVE=FitWave
		else
			Duplicate/O/R=(DrawFrom, DrawTo) BaseFitWave, FitFolder:$(DataWaveName+"_bck")/WAVE=BckWave, FitFolder:$(DataWaveName+"_fit")/WAVE=FitWave
		endif	
		FastOP FitWave=0
		FastOP BckWave=0
		
		//	Creates a temporary wave used for faster calculations
		Duplicate/O/FREE BckWave, XWave, TempWave
		XWave=x
			
		//	Calculates the peaks
		Variable Const=AllFitCoefWave[1][Num]
		for (i=0; i<NumberOfPeaks; i+=1)

			//	Creates the wave to hold the peak
			Duplicate/O BckWave, FitFolder:$(DataWaveName+"_pk"+Num2iStr(i))/WAVE=PeakWave
			
//			Position=AllFitCoefWave[3+i*7][Num]
//			PeakArea=AllFitCoefWave[4+i*7][Num]
//			FWHM=AllFitCoefWave[5+i*7][Num]
//			GL=AllFitCoefWave[6+i*7][Num]
//			Asym1=AllFitCoefWave[7+i*7][Num]
//			Asym2=AllFitCoefWave[8+i*7][Num]
//			ShirleyCoef=AllFitCoefWave[9+i*7][Num]
			
			//	Calculates the Shirley background for the peak and adds it to the combined background
			Const+=XPSFitAssFitFuncAAOOnePeakBck(AllFitCoefWave[3+i*7][Num], AllFitCoefWave[4+i*7][Num], AllFitCoefWave[5+i*7][Num], AllFitCoefWave[6+i*7][Num], AllFitCoefWave[7+i*7][Num], AllFitCoefWave[8+i*7][Num], AllFitCoefWave[9+i*7][Num], XWave, TempWave)
			FastOP BckWave=BckWave+TempWave
				
			//	Calculates the peak without any background
			XPSFitAssFitFuncAAOOnePeak(AllFitCoefWave[3+i*7][Num], AllFitCoefWave[4+i*7][Num], AllFitCoefWave[5+i*7][Num], AllFitCoefWave[6+i*7][Num], AllFitCoefWave[7+i*7][Num], AllFitCoefWave[8+i*7][Num], 0, XWave, PeakWave)
		
			//	Adds the peak to the fit wave
			FastOP FitWave=FitWave+PeakWave
		endfor
			
		//	Add the linear background to the background
		FastOP BckWave=BckWave+(Const)+(AllFitCoefWave[2][Num])*XWave
		
		//	Adds the background to the fit wave
		FastOP FitWave=FitWave+BckWave
	
		//	Adds the combined background to the peak waves
		for (i=0; i<NumberOfPeaks; i+=1)
			Wave PeakWave=FitFolder:$(DataWaveName+"_pk"+Num2iStr(i))
			FastOP PeakWave=PeakWave+BckWave
		endfor
	endif
	
	//	Used for multithreading
	Return DataWave
end



Structure XPSFitAssFitSingleSTRUCT
//	The structure used to pass information to the XPSFitAssFitSingle function
	Wave DataWave											//	The wave to be fitted
	Wave FitCoefWave										//	The coefficients for the fit
	Wave FitEpsilonWave										//	The epsilon wave for the fit
	Wave InitialM_FitConstraint								//	The constraint matrix for the initial fit
	Wave InitialW_FitConstraint								//	The constraint wave for the initial fit
	Wave FinalM_FitConstraint								//	The constraint matrix for the final fit
	Wave FinalW_FitConstraint								//	The constraint wave for the final fit
	String InitialFitHoldString									//	The hold string for the initial fit
	String FinalFitHoldString									//	The hold string for the final fit
	Variable V_FitMaxIters									//	The maximum number of iterations per fit
	Variable FitOptions										//	The fit options, see DisplayHelpTopic "Special Variables for Curve Fitting"
	Wave InfoWave											//	A wave holding the information from FitTime, V_FitError, V_FitQuitReason, V_FitNumIters for each of the fits
	Variable i												//	The index of the wave currently being fitted
	//	OLD
	STRUCT XPSFitAssFitAAOSTRUCT FitFuncStruct			//	The structure used by the all-at-once structure fit function
	//	NEW
	//STRUCT XPSFitAssFitStructNEW FitFuncStruct				//	The structure used by the threadsafe, all-at-once, structure fit function
	Variable DoInitialFit										//
endStructure



Static Function XPSFitAssFitSingle(s)
//	Fits a single spectrum
STRUCT XPSFitAssFitSingleSTRUCT & s
Variable FitTime=0

	//	Special variables. See DisplayHelpTopic "Special Variables for Curve Fitting". V_FitTol should be between 0.1 and 0.00001. 0.001 is default. The initial fit uses a crude tolerance, the final fit a much finer tolerance.
	Variable V_FitOptions=s.FitOptions, V_FitNumIters=0, V_FitQuitReason=0, V_FitTol=0.1, V_FitMaxIters=s.V_FitMaxIters, V_FitError=0
	
	//	Doesn't do the initial fit if the same parameters are held constant in the final fit
	if (s.DoInitialFit>=0)

		//	Holds positions, GL ratios and asymmetry for the initial fit. Especially the GL ratios tend to run amok, if the initial guesses are not perfect. This can be seen as a way to improve the initial guesses before letting the GL ratios loose.
		FitTime=StartMSTimer
		FuncFit/N/Q/NTHR=1/W=2/H=s.InitialFitHoldString EccentricXPS#XPSFitAssFitFuncAAOStruct s.FitCoefWave s.DataWave /C={s.InitialM_FitConstraint, s.InitialW_FitConstraint} /E=s.FitEpsilonWave /STRC=s.FitFuncStruct
		FitTime=StopMSTimer(FitTime)
		
		//	Saves the information about the fit
		s.InfoWave[2*s.i][0]=s.i
		s.InfoWave[2*s.i][1]=FitTime
		s.InfoWave[2*s.i][2]=V_FitError
		s.InfoWave[2*s.i][3]=V_FitQuitReason
		s.InfoWave[2*s.i][4]=V_FitNumIters
		
		//	Only does the initial fit once, if DoInitialFit == 0
		if (s.DoInitialFit==0)
			s.DoInitialFit=-1
		endif
	endif
			
	//	Changes to the finest tolerance. This will create a lot of V_FitQuitReason==3: "Limit of passes without decreasing chi-square was reached" errors
	//	The fine tolerance has proven important for x-ray standing wave fits
	V_FitTol=0.00001

	//	Frees all parameters for the final fit
	FitTime=StartMSTimer
	FuncFit/N/Q/NTHR=1/H=s.FinalFitHoldString EccentricXPS#XPSFitAssFitFuncAAOStruct s.FitCoefWave s.DataWave /C={s.FinalM_FitConstraint, s.FinalW_FitConstraint} /E=s.FitEpsilonWave /STRC=s.FitFuncStruct
	FitTime=StopMSTimer(FitTime)			//	0.02 ms
	
	//	Suppresses the V_FitQuitReason==3: "Limit of passes without decreasing chi-square was reached" errors created by the fine tolerance
	If (V_FitQuitReason==3)
		V_FitQuitReason=0
	endif
	
	//	Saves the information about the fit
	s.InfoWave[2*s.i+1][0]=s.i
	s.InfoWave[2*s.i+1][1]=FitTime
	s.InfoWave[2*s.i+1][2]=V_FitError
	s.InfoWave[2*s.i+1][3]=V_FitQuitReason
	s.InfoWave[2*s.i+1][4]=V_FitNumIters
	
	//	Returns the error to be passed on to any subsequent FitFunc calls
	Return V_FitError
end



Static Function/S XPSFitAssQuitReason(V_FitQuitReason)
//	Creates a string with a description of the reason the fit stopped
Variable V_FitQuitReason
String QuitReasonString=""

	//	Parses the V_FitQuitReason variable
	switch(V_FitQuitReason)
		case 1:
			QuitReasonString="Iteration limit was reached"
			break
		case 2:
			QuitReasonString="User stopped the fit"
			break
		case 3:
			QuitReasonString="Limit of passes without decreasing chi-square was reached"		//	V_FitQuitReason==3 should be suppressed to avoid spam
			break
	endswitch

	//	Returns a string with a description of why the fit stopped
	Return QuitReasonString
end



Static Function/S XPSFitAssFitError(V_FitError)
//	Creates a string with a description of any error that might have occurred during the fit
Variable V_FitError
String FitErrorString=""

		//	Parses the V_FitError variable returned by FuncFit
		if ((V_FitError & 1)==1)
			FitErrorString="Error"
		endif
		if ((V_FitError & 2)==2)
			FitErrorString+=", Singular matrix"
		endif
		if ((V_FitError & 4)==4)
			FitErrorString+=", Out of memory"
		endif
		if ((V_FitError & 8)==8)
			FitErrorString+=", Function returned NaN or INF"
		endif
		if ((V_FitError & 16)==16)
			FitErrorString+=", Fit function requested stop"
		endif
		if ((V_FitError & 32)==32)
			FitErrorString+=", Reentrant curve fitting"
		endif

	//	Returns a string with a description of the error
	Return FitErrorString
end



Static Function/WAVE XPSFitAssParseConstraintsString(ConstraintsString, NumberOfPeaks)
//	Creates a string with the list of constraints for the fit, where the easy to understand symbols P0, A0, W0, etc... are replaced by the required K2, K3, K4, etc....
//	See DisplayHelpTopic "Fitting with Constraints"
String ConstraintsString
Variable NumberOfPeaks
Variable i=0, n=0, a=0, b=0
String PeakStr=""

	//	Removes all blank spaces from the string
	ConstraintsString=ReplaceString(" ", ConstraintsString, "")
	
	//	Counts backward through the peaks to avoid replacing P12 when replacing P1
	for (i=NumberOfPeaks-1; i>=0; i+=-1)
		PeakStr=Num2iStr(i)
		a=i*7
		ConstraintsString=ReplaceString("P"+PeakStr, ConstraintsString, "K"+Num2iStr(a+3), 1)
		ConstraintsString=ReplaceString("A"+PeakStr, ConstraintsString, "K"+Num2iStr(a+4), 1)
		ConstraintsString=ReplaceString("W"+PeakStr, ConstraintsString, "K"+Num2iStr(a+5), 1)
		ConstraintsString=ReplaceString("G"+PeakStr, ConstraintsString, "K"+Num2iStr(a+6), 1)
		ConstraintsString=ReplaceString("Y"+PeakStr, ConstraintsString, "K"+Num2iStr(a+7), 1)
		ConstraintsString=ReplaceString("Z"+PeakStr, ConstraintsString, "K"+Num2iStr(a+8), 1)
		ConstraintsString=ReplaceString("S"+PeakStr, ConstraintsString, "K"+Num2iStr(a+9), 1)
	endfor

	//	Replaces the linear background symbols. K0 is used for the number of parameters in the fit
	ConstraintsString=ReplaceString("L1", ConstraintsString, "K2")
	ConstraintsString=ReplaceString("L0", ConstraintsString, "K1")
	
	//	Removes useless constraints such as ( K<number> > -inf ) and ( K<number> < inf )
	ConstraintsString=GrepList(ConstraintsString, "(<|>-)((?i)inf)$" , 1, ";")
	
	//	Removes empty constraints
	ConstraintsString=GrepList(ConstraintsString, "." , 0, ";")
	
	//	Converts the string list to a wave
	n=ItemsInList(ConstraintsString, ";")
	Make/FREE/T/O/N=(n) ConstraintsWave
	for (i=0; i<n; i+=1)
		b=strsearch(ConstraintsString, ";", a)
		ConstraintsWave[i]=ConstraintsString[a,b-1]
		a=b+1
	endfor
	
	//	Returns the replaced string
	Return ConstraintsWave
end



Static Function XPSFitAssFitFuncDummy(pw, yw, xw) : FitFunc
//	A dummy fit function used when generating the constraint matrix and wave
WAVE pw, yw, xw
end



Structure XPSFitAssFitAAOSTRUCT
//	The structure used for the all-at-once structure fit function
	Wave/D coefw					//	The fit coefficients
	Wave/D yw						//	The y values of the fit
	Wave/D xw						//	The x values of the fit
	STRUCT WMFitInfoStruct FitInfo	//	See DisplayHelpTopic "The WMFitInfoStruct Structure"
	
	Variable NumberOfPeaks			//	The number of peaks in the fit
	Wave/WAVE TempWaves			//	Wave references to preexisting waves used for temporary storage during the calculations
	Wave/D LinkMatrix				//	A matrix used to calculate the actual coefficients from the input coefficients
endStructure



Static Function XPSFitAssFitFuncAAOStruct(s) : FitFunc
//	An all-at-once structure fit function (DisplayHelpTopic "User-Defined Fitting Function: Detailed Description") which calculates all values in the wave in one go, instead of point-by-point. This version is optimized for speed and is faster than the fast point-by-point version
//	FuncFit does not use multiple cores for all-at-once or structure fit functions, but you can still use MultiThread within the fit function, as I do here, or MultiThread several FuncFit calls.
STRUCT XPSFitAssFitAAOSTRUCT & s
	
	//	Calculates the fit coefficients to use, taking the dependency of the coefficients into account
	//MatrixOP/O/FREE UsedCoefs=(s.LinkMatrix x s.coefw)		//	Takes 40 us, independent of the number of peaks
	MatrixMultiply s.LinkMatrix, s.coefw						//	Takes 5 us for 1 peak, 8 us for two, 16 for four, but not threadsafe since the destination wave cannot be controlled
	Wave UsedCoefs=M_product
	
	//Print(s.FitInfo.IterStarted)
	//Print(s.FitInfo.ParamPerturbed)	//	Can be used to speed up fit significantly!!
	//	PROBLEM: s.FitInfo.ParamPerturbed will tell me which FitCoef is being perturbed, but not which UsedFitCoef!!!! Knowing which UsedFitCoefs are being pertubed woud also allow me to reduce the MatrixMultiply to smaller waves
	
	//	Position=UsedCoefs[3+p*7]
	//	PeakArea=UsedCoefs[4+p*7]
	//	FWHM=UsedCoefs[5+p*7]
	//	GL=UsedCoefs[6+p*7]
	//	Asym1=UsedCoefs[7+p*7]
	//	Asym2=UsedCoefs[8+p*7]
	//	ShirleyCoef=UsedCoefs[9+p*7]
	
	//	The result wave
	Wave/D yw=s.yw
	
	//	The x wave
	Wave/D xw=s.xw

	//	A wave with the same number of points as the fit has peaks, which will be used to store the constant offset for each peak
	Wave/D ConstantOffset=s.TempWaves[0]

	//	The offset from the linear background
	ConstantOffset[0]=UsedCoefs[1]

	//	Calculates the spectrum for each peak and saves	 the spectrum and offset. The temp waves are preexisting waves used by all function calls, which means that the function is NOT threadsafe
	if (s.NumberOfPeaks>0)
		MultiThread ConstantOffset[1,*]=XPSFitAssFitFuncAAOOnePeak(UsedCoefs[-4+p*7], UsedCoefs[-3+p*7], UsedCoefs[-2+p*7], UsedCoefs[-1+p*7], UsedCoefs[p*7], UsedCoefs[1+p*7], UsedCoefs[2+p*7], xw, s.TempWaves[p])		//	136 us for one peak (40 us for pure Gaussian)
	endif

	//	Calculates the combined offset
	Variable Offset=Sum(ConstantOffset)		//	1 us for one peak
	
	//	Adds the constant offset and the linear background to the result
	FastOP yw=(UsedCoefs[2])*xw+(Offset)		//	0.7 us

	//	Adds the peaks to the result, one peak at a time
	Variable i=0
	for (i=0; i<s.NumberOfPeaks; i+=1)
		Wave PeakWave=s.TempWaves[i+1]
		FastOP yw=yw+PeakWave
	endfor				//	1.5 us for one peak, 3.6 for four peaks

end



ThreadSafe Static Function XPSFitAssFitFuncAAOOnePeak(Position, PeakArea, FWHM, GL, Asym1, Asym2, ShirleyCoef, xw, R)
//	Calculates a single peak for the all-at-once fit function. Since this function optimized for speed the math is almost impossible to follow. Look to XPSFitAssFitFunc or XPSFitAssFitFuncFast for easier-to-follow equations
Variable Position, PeakArea, FWHM, GL, Asym1, Asym2, ShirleyCoef
Wave/D xw, R
Variable c1=0, c2=0, Const=0
	
	Duplicate/O/FREE/D R, A, A2, B, C		//	8 us
	
	//	The simple calculation for a symmetric pseudo-Voigt peak with the corresponding Shirley background. This will also be the starting point for the asymmetric peak
	c1=2/FWHM
	c2=c1*Position
	FastOP A=(c1)*xw-(c2)	//	0.8 us
	A2=A^2					//	15 us
	Const=0.5*PeakArea*ShirleyCoef

	//	Doesn't calculate the Gaussian component, if the peak is pure Lorentzian
	if (GL!=1)

		//	Calculates the Gaussian component of the peak
		FastOP B=(-0.693148)*A2			//	0.6 us
		B=exp(B)						//	12 us
		c2=(1-GL)*PeakArea
		c1=0.939437*c2/FWHM

		//	Doesn't calculate the Shirley background if the Shirley coefficient is zero
		if (ShirleyCoef!=0)

			//	Calculates the Shirley background for the Gaussian component
			FastOP C=(0.832555)*A		//	0.5 us
			C=erf(C)						//	61 us
			c2=c2*0.5*ShirleyCoef

			//	Adds the peak and background together
			FastOP R=(c1)*B+(c2)*C		//	1.5 us
		else

			//	Only adds the peak and not the background
			FastOP R=(c1)*B
		endif
		
		//	Doesn't calculate the Lorentzian component, if the peak is pure Gaussian
		if (GL!=0)
		
			//	Calculates the Lorentzian component of the peak
			FastOP B=A2+(1)		//	0.7 us
			B=1/B					//	8.5 us
			c2=PeakArea*GL
			c1=c2*0.63662/FWHM

			//	Adds the peak to the result wave
			FastOP R=R+(c1)*B		//	1.4 us

			//	Doesn't calculate the Shirley background if the Shirley coefficient is zero
			if (ShirleyCoef!=0)

				//	Calculates the Shirley background for the Lorentzian component
				C=atan(A)			//	16 us
				c2=c2*0.31831*ShirleyCoef

				//	Adds the background to the result wave
				FastOP R=R+(c2)*C		//	1.3 us
			endif
		endif
	else

		//	Calculates the Lorentzian component of the peak
		FastOP B=A2+(1)
		B=1/B
		c1=PeakArea*0.63662/FWHM

		//	Doesn't calculate the Shirley background if the Shirley coefficient is zero
		if (ShirleyCoef!=0)

			//	Calculates the Shirley background for the Lorentzian component
			C=atan(A)
			c2=PeakArea*0.31831*ShirleyCoef

			//	Adds the peak and background together
			FastOP R=(c1)*B+(c2)*C
		else
		
			//	Only adds the peak and not the background
			FastOP R=(c1)*B
		endif
	endif

//	c1=2/FWHM
//	c2=c1*Position
//	FastOP A=(c1)*xw-(c2)
//	B=atan(A)
//	c1=PeakArea*ShirleyCoef
//	c2=0.31831*c1*GL
//
//	FastOP C=(0.832555)*A
//	C=erf(C)
//	Const=0.5*c1
//	c1=(1-GL)*Const
//	FastOP R=(c2)*B+(c1)*C
//			
//	A=A^2
//	FastOP B=(-0.693148)*A
//	B=exp(B)
//	c1=PeakArea/FWHM
//	c2=c1*GL
//	c1=(c1-c2)*0.939437
//	FastOP R=R+(c1)*B
//
//	c1=c2*0.63662
//	FastOP B=A+(1)
//	B=1/B
//	FastOP R=R+(c1)*B
	
	//	Checks if the peak is asymetric. The asymmetric peaks are calculated as symmetric pseudo-Voigt peaks with four trailing gaussian peaks creating a high binding energy tail
	if (Asym1!=0)

		Variable AsymFWHM=0, AsymPosition=0, AsymArea=0, TotalArea=0

		//	1st Gaussian tail: The simple calculation for a symmetric Gaussian peak with the corresponding Shirley background
		AsymPosition=Position+0.5*FWHM
		AsymFWHM=1.5*FWHM
		AsymArea=PeakArea*Asym1
		TotalArea=PeakArea+AsymArea

		c1=2/AsymFWHM
		c2=c1*AsymPosition
		FastOP A=(c1)*xw-(c2)

		c1=0.5*AsymArea*ShirleyCoef
		Const+=c1
		FastOP B=(0.832555)*A
		B=erf(B)
		FastOP R=R+(c1)*B
			
		A=A^2
		FastOP A=(-0.693148)*A
		A=exp(A)
		c1=0.939437*AsymArea/AsymFWHM
		FastOP R=R+(c1)*A

		if (Asym2!=0)

			//	2nd Gaussian tail: The simple calculation for a symmetric Gaussian peak with the corresponding Shirley background
			AsymPosition=AsymPosition+0.5*AsymFWHM
			AsymFWHM=1.5*AsymFWHM
			AsymArea=AsymArea*Asym2
			TotalArea+=AsymArea

			c1=2/AsymFWHM
			c2=c1*AsymPosition
			FastOP A=(c1)*xw-(c2)

			c1=0.5*AsymArea*ShirleyCoef
			Const+=c1
			FastOP B=(0.832555)*A
			B=erf(B)
			FastOP R=R+(c1)*B
			
			A=A^2
			FastOP A=(-0.693148)*A
			A=exp(A)
			c1=0.939437*AsymArea/AsymFWHM
			FastOP R=R+(c1)*A

			//	Depending on the values of Asym1 and Asym2 the decrease in peak area either accelerates or deccelerates, as described by the parameter alpha
			Variable Asym3=0, Asym4=0, Alpha=0
		
			if (Asym1>=Asym2)
				Alpha=Asym2/Asym1
				Asym3=Asym2*Alpha
				Asym4=Asym3*Alpha
			else
				if (Asym1==1)
					Asym3=0
					Asym4=0
				else
					Alpha=(1-Asym2)/(1-Asym1)
					Asym3=1-(1-Asym2)*Alpha
					Asym4=1-(1-Asym3)*Alpha
				endif
			endif

			//	3rd Gaussian tail: The simple calculation for a symmetric Gaussian peak with the corresponding Shirley background
			AsymPosition=AsymPosition+0.5*AsymFWHM
			AsymFWHM=1.5*AsymFWHM
			AsymArea=AsymArea*Asym3
			TotalArea+=AsymArea

			c1=2/AsymFWHM
			c2=c1*AsymPosition
			FastOP A=(c1)*xw-(c2)

			c1=0.5*AsymArea*ShirleyCoef
			Const+=c1
			FastOP B=(0.832555)*A
			B=erf(B)
			FastOP R=R+(c1)*B
			
			A=A^2
			FastOP A=(-0.693148)*A
			A=exp(A)
			c1=0.939437*AsymArea/AsymFWHM
			FastOP R=R+(c1)*A

			//	4th Gaussian tail: The simple calculation for a symmetric Gaussian peak with the corresponding Shirley background
			AsymPosition=AsymPosition+0.5*AsymFWHM
			AsymFWHM=1.5*AsymFWHM
			AsymArea=AsymArea*Asym4
			TotalArea+=AsymArea

			c1=2/AsymFWHM
			c2=c1*AsymPosition
			FastOP A=(c1)*xw-(c2)

			c1=0.5*AsymArea*ShirleyCoef
			Const+=c1
			FastOP B=(0.832555)*A
			B=erf(B)
			FastOP R=R+(c1)*B
			
			A=A^2
			FastOP A=(-0.693148)*A
			A=exp(A)
			c1=0.939437*AsymArea/AsymFWHM
			FastOP R=R+(c1)*A
		endif
		
		//	Returns the total area to PeakArea
		c1=PeakArea/TotalArea
		Const=Const*c1
		FastOP R=(c1)*R
	endif
	
	//	Returns the constant offset, which should be added to the wave
	Return Const
end



//	FindMe		///////////////////////////////////////////

//	NEW NEW NEW

//
//Structure XPSFitAssFitStructNEW
////	The structure used for the threadsafe, all-at-once, structure fit function
//	Wave/D coefw								//	The fit coefficients
//	Wave/D yw									//	The y values of the fit
//	Wave/D xw									//	The x values of the fit
//	STRUCT WMFitInfoStruct FitInfo				//	See DisplayHelpTopic "The WMFitInfoStruct Structure"
//	
//	Variable NumberOfPeaks						//	The number of peaks in the fit
//	Wave/WAVE TempWaves						//	Preexisting waves used for temporary storage during the calculations
//	Wave/D LinkMatrix							//	A matrix used to calculate the actual used coefficients from the input coefficients
//	Wave/I/U LinkMatrixIndex
//	Wave/D UsedCoefs
//	Wave/D UsedCoefsRef
//	Wave/B/U HoldCoefs
//	Wave/WAVE UsedCoefsIndexWaves
//	Wave/I/U HardLinkedIndex
//	Wave/B/U PeakPertubed
//	Wave/B FixedGL
//	Wave/B FixedShirley
//endStructure

//	NEW NEW NEW

//Static Function XPSFitAssFitFuncNEW(s) : FitFunc
////	An all-at-once structure fit function (DisplayHelpTopic "User-Defined Fitting Function: Detailed Description") which calculates all values in the wave in one go, instead of point-by-point. This version is optimized for speed and is faster than the fast point-by-point version
////	FuncFit does not use multiple cores for all-at-once or structure fit functions, but you can still use MultiThread within the fit function, as I do here, or MultiThread several FuncFit calls.
////	UPDATE DESCRIPTION
//STRUCT XPSFitAssFitStructNEW & s
//Variable i=0, ii=0, PeakPertubed=0, i5=0, i7=0, c1=0, c2=0
//Variable Position=0, PeakArea=0, FWHM=0, GL=0, Asym1=0, Asym2=0, ShirleyCoef=0
////
////
////	Variable t=StartMSTimer
////
////	
////	//	Calculates the fit coefficients to use, taking the dependency of the coefficients into account
////	//MatrixOP/O/FREE UsedCoefs=(s.LinkMatrix x s.coefw)		//	Takes 40 us, independent of the number of peaks
////	//MatrixMultiply s.LinkMatrix, s.coefw						//	Takes 5 us for 1 peak, 8 us for two, 16 for four, but not threadsafe since the destination wave cannot be controlled
////	//Wave UsedCoefs=M_product
////	
////	//Print(s.FitInfo.IterStarted)
////	
////	//	The different waves used for the calculations	
////	Wave/D yw=s.yw
////	Wave/D xw=s.xw
////	Wave/D InputCoefs=s.coefw
////	Wave/D LinkMatrix=s.LinkMatrix
////	Wave/I/U LinkMatrixIndex=s.LinkMatrixIndex
////	
////	//	Calculates the values of the used coefficients based on the input coefficients, taking hard-linked coefficients into account. An s.FitInfo.ParamPerturbed value of -1 means all coefficients not held constant have been changed and all coefficients not held constant are therefore recalculated.
////	if (s.FitInfo.ParamPerturbed==-1)
////	
////		//	Recalculates the reference coefficients
////		Wave/D UsedCoefs=s.UsedCoefsRef
////	
////		//	All coefficients have been changed. First, the values from the Input coefficients are copied
////		FastOP UsedCoefs=InputCoefs
////		
////		//	Then all the dependent, hard-linked coefficents are included in the index of coefficients to be calculated
////		Wave/I/U UsedCoefsIndex=s.HardLinkedIndex
////	else
////
////		//	Calculates the new coefficients based on the changes (s.FitInfo.ParamPerturbed) to the reference coefficients
////		Wave/D UsedCoefs=s.UsedCoefs
////		Wave/D UsedCoefsRef=s.UsedCoefsRef
////		FastOP UsedCoefs=UsedCoefsRef
////	
////		//	The index wave indicating the used coefficients that have changed relative to the reference coeffcients and needs to be recalculated
////		Wave/I/U UsedCoefsIndex=s.UsedCoefsIndexWaves[s.FitInfo.ParamPerturbed]
////	endif
////
////	//	Counts through and recalculates the coefficients
////	for (i=1; i<UsedCoefsIndex[0]; i+=1)
////	
////		//	Dummy coefficient still used
////		//	Test speed..
////		
////		//	Each hard-linked coefficient is calculated as a linear combination of the coefficients stored in s.LinkMatrixIndex
////		UsedCoefs[UsedCoefsIndex[i]]=LinkMatrix[UsedCoefsIndex[i]][LinkMatrixIndex[UsedCoefsIndex[i]][1]]*InputCoefs[LinkMatrixIndex[UsedCoefsIndex[i]][1]]
////		for (ii=2; ii<LinkMatrixIndex[UsedCoefsIndex[i]][0]; ii+=1)
////			UsedCoefs[UsedCoefsIndex[i]]+=LinkMatrix[UsedCoefsIndex[i]][LinkMatrixIndex[UsedCoefsIndex[i]][ii]]*InputCoefs[LinkMatrixIndex[UsedCoefsIndex[i]][ii]]
////		endfor
////	endfor
////
////	t=StopMSTimer(t)
////	Print(Num2Str(s.FitInfo.ParamPerturbed)+", "+Num2Str(t))
////
////
////
////	//	Adds the slope from the linear background to the result
////	FastOP yw=(UsedCoefs[2])*xw
////
////	//	Adds the intercept from the linear background to the total offset
////	Variable TotalOffset=UsedCoefs[1]
////
////	//	Counts through the number of peaks in the fit
////	for (i=0; i<s.NumberOfPeaks; i+=1)
////
////		//	The peak coefficients
////		i7=i*7
////		Position=UsedCoefs[3+i7]
////		PeakArea=UsedCoefs[4+i7]
////		FWHM=UsedCoefs[5+i7]
////		GL=UsedCoefs[6+i7]
////		Asym1=UsedCoefs[7+i7]
////		Asym2=UsedCoefs[8+i7]
////		ShirleyCoef=UsedCoefs[9+i7]
////		
////		//	Adds the constant offset caused by the Shirley background
////		TotalOffset+=0.5*PeakArea*ShirleyCoef	//	Only true if not asymmetric!!!!!!!
////		
////		//	Checks if the perturbed coefficient affects peak i. An s.FitInfo.ParamPerturbed value of -1 means all coefficients not held constant have been changed
////		if (s.FitInfo.ParamPerturbed==-1)
////		
////			//	Checks if the fit function is called for the first time, if so all peaks are calculated from scratch, even if some coefficients are held constant
////			if (s.FitInfo.IterStarted==1)
////			
////				//	Forces a full calculation, this is otherwise only necessary if the position or FWHM has changed
////				PeakPertubed=2
////			else
////
////				//	Checks if a full recalculation of the peak is necessary, this is only necessary if the position or FWHM has changed
////				PeakPertubed=s.PeakPertubed[i][0]
////			endif
////
////			//	The reference waves are used instead of the temporary waves to ensure that any changes are saved
////			i5=i*5
////			Wave/D GaussPeak=TempWaves[7+i5]
////			Wave/D GaussShirley=TempWaves[8+i5]
////			Wave/D LorentzPeak=TempWaves[9+i5]
////			Wave/D LorentzShirley=TempWaves[10+i5]
////			Wave/D VoigtPeak=TempWaves[11+i5]
////		else
////
////			//	Checks if a full recalculation of the peak is necessary
////			PeakPertubed=s.PeakPertubed[i][s.FitInfo.ParamPerturbed]
////			
////			//	PeakPerturbed==2 means that the position or FWHM has changed
////			if (PeakPerturbed==2)
////			
////				//	The position or FWHM has changed forcing a recalculation of the peak waves, but temporary waves are used instead of the reference waves to prevent the changes from being saved
////				Wave/D GaussPeak=TempWaves[2]
////				Wave/D GaussShirley=TempWaves[3]
////				Wave/D LorentzPeak=TempWaves[4]
////				Wave/D LorentzShirley=TempWaves[5]
////				Wave/D VoigtPeak=TempWaves[6]
////			else
////			
////				//	The position or FWHM has not changed and the reference waves can therefore be used, avoiding have to recalculate them
////				i5=i*5
////				Wave/D GaussPeak=TempWaves[7+i5]
////				Wave/D GaussShirley=TempWaves[8+i5]
////				Wave/D LorentzPeak=TempWaves[9+i5]
////				Wave/D LorentzShirley=TempWaves[10+i5]
////				Wave/D VoigtPeak=TempWaves[11+i5]
////			endif
////		endif
////
////		//	The FWHM or position has changed. The peak needs to be completely recalculated
////		if (PeakPertubed==2)
////		
////			//	Temporary waves used for the calculations
////			Wave/D A=TempWaves[0]
////			Wave/D A2=TempWaves[1]
////		
////			//	Waves used for calculating the peaks
////			c1=2/FWHM
////			c2=c1*Position
////			FastOP A=(c1)*xw-(c2)	//	0.8 us
////			A2=A^2					//	15 us
////
////			//	Calculates the Gaussian component of the peak, with or without a Shirley background
////			if (s.FixedGL[i]==0)
////
////				//	Calculates the Gaussian component of the peak
////				FastOP GaussPeak=(-0.693148)*A2			//	0.6 us
////				GaussPeak=exp(GaussPeak)					//	12 us
////				
////				//	Calculates the Shirley background for the Gaussian peak
////				if (s.FixedShirley[i]!=0)			//	PROBLEM WITH HARD-LINKED. SHOULD NOT BE CONSIDERED HELD!!!!!!!
////					FastOP GaussShirley=(0.832555)*A		//	0.5 us
////					GaussShirley=erf(GaussShirley)			//	61 us
////				endif
////
////			//	Calculates the Lorentzian component of the peak, with or without a Shirley background
////			elseif (s.FixedGL[i]==1)						//	PROBLEM WITH HARD-LINKED. SHOULD NOT BE CONSIDERED HELD!!!!!!!
////			
////				//	Calculates the Lorentzian component of the peak
////				FastOP LorentzPeak=A2+(1)					//	0.7 us
////				LorentzPeak=1/LorentzPeak					//	8.5 us
////
////				//	Calculates the Shirley background for the Lorentzian component
////				if (s.FixedShirley[i]!=0)					//	PROBLEM WITH HARD-LINKED. SHOULD NOT BE CONSIDERED HELD!!!!!!!
////					LorentzShirley=atan(A)					//	16 us
////				endif
////
////			//	Calculates both the Gaussian and Lorentzian components of the mixed, pseudo-Voigt peak, with or without a Shirley background
////			else
////
////				//	Calculates the Gaussian component of the peak
////				FastOP GaussPeak=(-0.693148)*A2			//	0.6 us
////				GaussPeak=exp(GaussPeak)					//	12 us
////
////				//	Calculates the Lorentzian component of the peak
////				FastOP LorentzPeak=A2+(1)					//	0.7 us
////				LorentzPeak=1/LorentzPeak					//	8.5 us
////
////				if (s.FixedShirley[i]!=0)				//	PROBLEM WITH HARD-LINKED. SHOULD NOT BE CONSIDERED HELD!!!!!!!
////
////					//	Calculates the Shirley background for the Gaussian peak
////					FastOP GaussShirley=(0.832555)*A		//	0.5 us
////					GaussShirley=erf(GaussShirley)			//	61 us
////
////					//	Calculates the Shirley background for the Lorentzian component
////					LorentzShirley=atan(A)					//	16 us
////				endif
////			endif
////		endif
////
////		//	Only the GL ratio, asymmetry or Shirley coefficient has changed. A simple, fast calculation is sufficient.
////		if (PeakPertubed!=0)
////
////			//	Uses a pure Gaussian peak, with or without a Shirley background
////			if (s.FixedGL[i]==0)
////				c1=0.939437/FWHM
////				if (s.FixedShirley[i]==0)	//	is this worthwhile?
////					FastOP VoigtPeak=(c1)*GaussPeak		//	Could be made faster
////				else
////					c2=0.5*ShirleyCoef
////					FastOP VoigtPeak=(c1)*GaussPeak+(c2)*GaussShirley
////				endif
////
////			//	Uses a pure Lorentzian peak, with or without a Shirley background
////			elseif (s.FixedGL[i]==1)
////				c1=0.63662/FWHM
////				if (s.FixedShirley[i]==0)	//	is this worthwhile?
////					FastOP VoigtPeak=(c1)*LorentzPeak		//	Could be made faster
////				else
////					c2=0.31831*ShirleyCoef
////					FastOP VoigtPeak=(c1)*LorentzPeak+(c2)*LorentzShirley
////				endif
////
////			//	Uses a mixed, pseudo-Voigt peak, with or without a Shirley background	
////			else
////				c1=(1-GL)*0.939437/FWHM
////				if (s.FixedShirley[i]==0)	//	is this worthwhile?
////					c2=GL*0.63662/FWHM
////					FastOP VoigtPeak=(c1)*GaussPeak+(c2)*LorentzPeak
////				else
////					c2=(1-GL)*0.5*ShirleyCoef
////					FastOP VoigtPeak=(c1)*GaussPeak+(c2)*GaussShirley
////					c2=GL*0.63662/FWHM
////					FastOP VoigtPeak=VoigtPeak+(c2)*LorentzPeak
////					c2=GL*0.31831*ShirleyCoef
////					FastOP VoigtPeak=VoigtPeak+(c2)*LorentzShirley
////				endif
////			endif
////		endif
////
////		//	Adds the peak to the fit
////		FastOP yw=yw+(PeakArea)*VoigtPeak
////	endfor
////
////	//	Adds the total offset to the fit
////	FastOP yw=yw+(TotalOffset)
//end









ThreadSafe Static Function XPSFitAssFitFuncAAOOnePeakBck(Position, PeakArea, FWHM, GL, Asym1, Asym2, ShirleyCoef, xw, R)
//	Calculates the Shirley background for a single peak. Since this function optimized for speed the math is almost impossible to follow. Look to XPSFitAssFitFunc or XPSFitAssFitFuncFast for easier-to-follow equations
Variable Position, PeakArea, FWHM, GL, Asym1, Asym2, ShirleyCoef
Wave/D xw, R
Variable c1=0, c2=0, Const=0

	//	Doesn't calculate the Shirley background if the Shirley coefficient is zero
	if (ShirleyCoef!=0)

		Duplicate/O/FREE/D R, A, B, C
	
		//	The simple calculation for a symmetric pseudo-Voigt peak with the corresponding Shirley background. This will also be the starting point for the asymmetric peak
		c1=2/FWHM
		c2=c1*Position
		FastOP A=(c1)*xw-(c2)
		Const=0.5*PeakArea*ShirleyCoef

		//	Doesn't calculate the Gaussian component, if the peak is pure Lorentzian
		if (GL!=1)

			//	Calculates the Shirley background for the Gaussian component
			FastOP B=(0.832555)*A
			B=erf(B)
			c1=(1-GL)*Const

			//	Doesn't calculate the Lorentzian component, if the peak is pure Gaussian
			if (GL!=0)
		
				//	Calculates the Shirley background for the Lorentzian component
				C=atan(A)
				c2=GL*0.63662*Const

				//	Adds both backgrounds
				FastOP R=(c1)*B+(c2)*C
			else
			
				//	Adds only the Gaussian background
				FastOP R=(c1)*B
			endif
		else

			//	Calculates the Shirley background for the Lorentzian component
			C=atan(A)
			c2=GL*0.63662*Const

			//	Adds only the Lorentzian background
			FastOP R=(c2)*C
		endif

	
		//	Checks if the peak is asymetric. The asymmetric peaks are calculated as symmetric pseudo-Voigt peaks with four trailing gaussian peaks creating a high binding energy tail
		if (Asym1!=0)

			Variable AsymFWHM=0, AsymPosition=0, AsymArea=0, TotalArea=0

			//	1st Gaussian tail: The simple calculation for a symmetric Gaussian peak with the corresponding Shirley background
			AsymPosition=Position+0.5*FWHM
			AsymFWHM=1.5*FWHM
			AsymArea=PeakArea*Asym1
			TotalArea=PeakArea+AsymArea

			c1=2/AsymFWHM
			c2=c1*AsymPosition
			FastOP A=(c1)*xw-(c2)

			c1=0.5*AsymArea*ShirleyCoef
			Const+=c1
			FastOP B=(0.832555)*A
			B=erf(B)
			FastOP R=R+(c1)*B
			
//			A=A^2
//			FastOP A=(-0.693148)*A
//			A=exp(A)
//			c1=0.939437*AsymArea/AsymFWHM
//			FastOP R=R+(c1)*A

			if (Asym2!=0)

				//	2nd Gaussian tail: The simple calculation for a symmetric Gaussian peak with the corresponding Shirley background
				AsymPosition=AsymPosition+0.5*AsymFWHM
				AsymFWHM=1.5*AsymFWHM
				AsymArea=AsymArea*Asym2
				TotalArea+=AsymArea

				c1=2/AsymFWHM
				c2=c1*AsymPosition
				FastOP A=(c1)*xw-(c2)

				c1=0.5*AsymArea*ShirleyCoef
				Const+=c1
				FastOP B=(0.832555)*A
				B=erf(B)
				FastOP R=R+(c1)*B
			
//				A=A^2
//				FastOP A=(-0.693148)*A
//				A=exp(A)
//				c1=0.939437*AsymArea/AsymFWHM
//				FastOP R=R+(c1)*A

				//	Depending on the values of Asym1 and Asym2 the decrease in peak area either accelerates or deccelerates, as described by the parameter alpha
				Variable Asym3=0, Asym4=0, Alpha=0
		
				if (Asym1>=Asym2)
					Alpha=Asym2/Asym1
					Asym3=Asym2*Alpha
					Asym4=Asym3*Alpha
				else
					if (Asym1==1)
						Asym3=0
						Asym4=0
					else
						Alpha=(1-Asym2)/(1-Asym1)
						Asym3=1-(1-Asym2)*Alpha
						Asym4=1-(1-Asym3)*Alpha
					endif
				endif

				//	3rd Gaussian tail: The simple calculation for a symmetric Gaussian peak with the corresponding Shirley background
				AsymPosition=AsymPosition+0.5*AsymFWHM
				AsymFWHM=1.5*AsymFWHM
				AsymArea=AsymArea*Asym3
				TotalArea+=AsymArea

				c1=2/AsymFWHM
				c2=c1*AsymPosition
				FastOP A=(c1)*xw-(c2)

				c1=0.5*AsymArea*ShirleyCoef
				Const+=c1
				FastOP B=(0.832555)*A
				B=erf(B)
				FastOP R=R+(c1)*B
			
//				A=A^2
//				FastOP A=(-0.693148)*A
//				A=exp(A)
//				c1=0.939437*AsymArea/AsymFWHM
//				FastOP R=R+(c1)*A

				//	4th Gaussian tail: The simple calculation for a symmetric Gaussian peak with the corresponding Shirley background
				AsymPosition=AsymPosition+0.5*AsymFWHM
				AsymFWHM=1.5*AsymFWHM
				AsymArea=AsymArea*Asym4
				TotalArea+=AsymArea

				c1=2/AsymFWHM
				c2=c1*AsymPosition
				FastOP A=(c1)*xw-(c2)

				c1=0.5*AsymArea*ShirleyCoef
				Const+=c1
				FastOP B=(0.832555)*A
				B=erf(B)
				FastOP R=R+(c1)*B
			
//				A=A^2
//				FastOP A=(-0.693148)*A
//				A=exp(A)
//				c1=0.939437*AsymArea/AsymFWHM
//				FastOP R=R+(c1)*A
			endif
		
			//	Returns the total area to PeakArea
			c1=PeakArea/TotalArea
			Const=Const*c1
			FastOP R=(c1)*R
		endif
	endif
	
	//	Returns the constant offset, which should be added to the wave
	Return Const
end



ThreadSafe Static Function XPSFitAssFitFuncFast(w,x):FitFunc
//	OLD VERSION NOT IN USE
//	Calculates the fitted height at position x. This version is optimized for speed. Because it is a standard fit function, and not an all-at-once or strucuture fit function, FitFunc can run on multiple cores (/NTHR=0) when using this function.
Wave w; Variable x
Variable Result=0, TempResult=0, TotalArea=0, A=0, RelArea=0, n=0
Variable Position=0, PeakArea=0, FWHM=0, GL=0, Asym1=0, Asym2=0, ShirleyCoef=0

Variable temp=0

	//	Adds the linear background. Intercept=w[1], Slope=w[2]
	Result=w[1]+w[2]*x
	
	//	Adds the peaks one at a time. w[0] contains the number of points in w.
	for (n=3; n<w[0]; n+=7)
	
		Position=w[n]
		PeakArea=w[n+1]
		FWHM=w[n+2]
		GL=w[n+3]
		Asym1=w[n+4]
		Asym2=w[n+5]
		ShirleyCoef=w[n+6]

		//	Used for faster calculations
		A=2*(x-Position)/FWHM
		
		//	Checks if the peak is asymetric
		if (Asym1==0)
			//	The simple calculation for a symmetric pseudo-Voigt peak with the corresponding Shirley background
			Result+=PeakArea*((1-GL)*(0.939437/FWHM*exp(-0.693148*A^2)+0.5*ShirleyCoef*(Erf(0.832555*A)+1))+GL*(0.63662/(FWHM*(A^2+1))+ShirleyCoef*(0.31831*atan(A)+0.5)))	
		else
			//	The asymmetric peaks are calculated as symmetric pseudo-Voigt peaks with four gaussian peak creating a high binding energy tail

			//	The main symmetric pseudo-Voigt peak
			TempResult=(1-GL)*(0.939437/FWHM*exp(-0.693148*A^2)+0.5*ShirleyCoef*(Erf(0.832555*A)+1))+GL*(0.63662/(FWHM*(A^2+1))+ShirleyCoef*(0.31831*atan(A)+0.5))
			TotalArea=1
			
			//	Gaussian and Gaussian Shirley 1
			TempResult+=Asym1*(0.626291/FWHM*exp(-0.308066*(A-1)^2)+0.5*ShirleyCoef*(Erf(0.555037*(A-1))+1))
			TotalArea+=Asym1

			//	Gaussian and Gaussian Shirley 2
			RelArea=Asym1*Asym2
			TempResult+=RelArea*(0.417528/FWHM*exp(-0.136918*(A-2.5)^2)+0.5*ShirleyCoef*(Erf(0.370024*(A-2.5))+1))
			TotalArea+=RelArea

			if (Asym1>=Asym2)
				//	Gaussian and Gaussian Shirley 3
				RelArea=Asym2^3
				TempResult+=RelArea*(0.278352/FWHM*exp(-0.0608525*(A-4.75)^2)+0.5*ShirleyCoef*(Erf(0.246683*(A-4.75))+1))
				TotalArea+=RelArea

				//	Gaussian and Gaussian Shirley 4
				RelArea=RelArea^2/Asym1^2
				TempResult+=RelArea*(0.185568/FWHM*exp(-0.0270456*(A-8.125)^2)+0.5*ShirleyCoef*(Erf(0.164455*(A-8.125))+1))
				TotalArea+=RelArea
				
			else
				if (Asym1!=1)
					//	Gaussian and Gaussian Shirley 3
					RelArea=RelArea*(1-(1-Asym2)^2/(1-Asym1))
					TempResult+=RelArea*(0.278352/FWHM*exp(-0.0608525*(A-4.75)^2)+0.5*ShirleyCoef*(Erf(0.246683*(A-4.75))+1))
					TotalArea+=RelArea

					//	Gaussian and Gaussian Shirley 4
					RelArea=RelArea*(1-(1-Asym2)^3/(1-Asym1)^2)
					TempResult+=RelArea*(0.185568/FWHM*exp(-0.0270456*(A-8.125)^2)+0.5*ShirleyCoef*(Erf(0.164455*(A-8.125))+1))
					TotalArea+=RelArea
				endif
			endif
			Result+=PeakArea*TempResult/TotalArea
		endif
	endfor
	return Result
end



ThreadSafe Static Function XPSFitAssFitFunc(w,x):FitFunc
//	OLD VERSION NOT IN USE
//	Calculates the fitted height at position x. This vesion is NOT optimized for speed, but because of that it is easier to read and understand. The function is not used anywhere in the procedure
//	The faster version uses w[0] to send the number of peaks to the fit function instead of a global variable. The two fit functions are therefore not directly interchangeable
Wave w; Variable x
NVAR NumberOfPeaks=XPSFitAssNumberOfPeaks
Variable i=0, Result=0, Intercept=0, Slope=0, Position=0, PeakArea=0, FWHM=0, GL=0, Asym1=0, Asym2=0, ShirleyCoef=0

	//	Adds the linear background
	Intercept=w[0]
	Slope=w[1]
	Result=XPSFitAssLine(x, Intercept, Slope)
	
	//	Adds the peaks one at a time
	for (i=0; i<NumberOfPeaks; i+=1)
	
		Position=w[2+i*7]
		PeakArea=w[3+i*7]
		FWHM=w[4+i*7]
		GL=w[5+i*7]
		Asym1=w[6+i*7]
		Asym2=w[7+i*7]
		ShirleyCoef=w[8+i*7]
	
		Result+=PeakArea*(XPSFitAssAsymPseudoVoigt(x, Position, FWHM, GL, Asym1, Asym2)+ShirleyCoef*XPSFitAssAsymPseudoVoigtShirley(x, Position, FWHM, GL, Asym1, Asym2))
	endfor
	return Result
end



ThreadSafe Static Function XPSFitAssLine(x, Intercept, Slope)
//	Calculates the linear background at position x
Variable x, Intercept, Slope
	Return Intercept+Slope*x
end



ThreadSafe Static Function XPSFitAssGauss(x, Position, FWHM)
//	Returns a Gaussian peak normalized to an area of one
Variable x, Position, FWHM

	//	f(x) = sqrt( 4 * ln(2) / pi ) / FWHM * exp( -4 * ln(2) * (( x - Position ) / FWHM )^2 )
	//	sqrt( 4 ln(2) / pi ) = 0.939437
	//	4*ln(2) = 2.77259

	Return 0.939437/FWHM*exp(-2.77259*((x-Position)/FWHM)^2)
end



ThreadSafe Static Function XPSFitAssGaussShirley(x, Position, FWHM)
//	Calculates the Shirley background (integrated area) of a Gaussian peak with an area of one
//	D.A. Shirley, Phys. Rev. 55 (1972) 4709
Variable x, Position, FWHM

	//	f(x) = 0.5 * ( Erf(sqrt( 4 * ln(2) ) * ( x - Position ) / FWHM ) + 1 )
	//	sqrt( 4 ln(2) ) = 1.66511

	Return 0.5*(Erf(1.66511*(x-Position)/FWHM)+1)
end



ThreadSafe Static Function XPSFitAssLorentz(x, Position, FWHM)
//	Returns a Lorentzian peak normalised to an area of one
Variable x, Position, FWHM

	//	2 / pi * FWHM / (4 * ( x - Position )^2 + FWHM^2 )
	//	2/pi = 0.63662

	Return 0.63662*FWHM/(4*(x-Position)^2+FWHM^2)
end



ThreadSafe Static Function XPSFitAssLorentzShirley(x, Position, FWHM)
//	Calculates the Shirley background (integrated area) of a Lorentzian peak with an area of one
Variable x, Position, FWHM

	//	1 / pi * atan( 2 * ( x - Position ) / FWHM ) + 0.5
	//	1/pi = 0.31831

	Return 0.31831*atan(2*(x-Position)/FWHM)+0.5
end



ThreadSafe Static Function XPSFitAssPseudoVoigt(x, Position, FWHM, GL)
//	Returns a Pseudo-Voigt peak, a linear combination (sum) of a Gaussian and a Lorentzian peak, with an area of one
//	A Gaussian multiplied by a Lorentzian is also commonly used for a Pseudo-Voigt, but this procedure uses a sum
//	Since both the gaussian and lorentzian have the same FWHM the FWHM of the sum will be the same of that of the individual peaks
//	GL=0 pure gaussian
//	GL=1 pure lorentzian
Variable x, Position, FWHM, GL
	Return (1-GL)*XPSFitAssGauss(x, Position, FWHM)+GL*XPSFitAssLorentz(x, Position, FWHM)
end



ThreadSafe Static Function XPSFitAssPseudoVoigtShirley(x, Position, FWHM, GL)
//	Calculates the Shirley background for a Pseudo-Voigt peak, a linear combination of a Gaussian and a Lorentzian peak, with a combined area of one
//	GL=0 pure gaussian
//	GL=1 pure lorentzian
Variable x, Position, FWHM, GL
	Return (1-GL)*XPSFitAssGaussShirley(x, Position, FWHM)+GL*XPSFitAssLorentzShirley(x, Position, FWHM)
end



ThreadSafe Static Function XPSFitAssAsymPseudoVoigt(x, Position, FWHM, GL, Asym1, Asym2)
//	Calculates an asymmetric pseudo-Voigt peak with an area of one using a tail made from the sum of 4 gaussians
//	The combined peak will no longer have a FWHM of FWHM!!!!
Variable x, Position, FWHM, GL, Asym1, Asym2
//Variable i=0, ReturnValue=0, AsymPosition=Position, AsymFWHM=FWHM, AsymArea=1, TotalArea=1

Variable AsymFWHM1=0, AsymFWHM2=0, AsymFWHM3=0, AsymFWHM4=0
Variable AsymPosition1=0, AsymPosition2=0, AsymPosition3=0, AsymPosition4=0
Variable AsymArea1=0, AsymArea2=0, AsymArea3=0, AsymArea4=0
Variable TotalArea=0, Alpha=0, Asym3=0, Asym4=0

//	ReturnValue=XPSFitAssPseudoVoigt(x, Position, FWHM, GL)
//	for (i=1; i<5; i=i+1)
//		AsymPosition=AsymPosition+0.5*AsymFWHM
//		AsymFWHM=1.5*AsymFWHM
//		AsymArea=AsymArea*(1-(1-Asym1)/Asym2^i)
//		TotalArea=TotalArea+AsymArea
//		ReturnValue=ReturnValue+AsymArea*XPSFitAssGauss(x, AsymPosition, AsymFWHM)
//	endfor
//	Return ReturnValue/TotalArea

	if (Asym1==0)
		Return XPSFitAssPseudoVoigt(x, Position, FWHM, GL)
	else
		AsymFWHM1=1.5*FWHM
		AsymFWHM2=1.5*AsymFWHM1
		AsymFWHM3=1.5*AsymFWHM2
		AsymFWHM4=1.5*AsymFWHM3
		
		AsymPosition1=Position+0.5*FWHM
		AsymPosition2=AsymPosition1+0.5*AsymFWHM1
		AsymPosition3=AsymPosition2+0.5*AsymFWHM2
		AsymPosition4=AsymPosition3+0.5*AsymFWHM3
		
		AsymArea1=Asym1
		AsymArea2=AsymArea1*Asym2
		
		if (Asym1>=Asym2)
			Alpha=Asym2/Asym1
			Asym3=Asym2*Alpha
			Asym4=Asym3*Alpha
		else
			if (Asym1==1)
				Asym3=0
				Asym4=0
			else
				Alpha=(1-Asym2)/(1-Asym1)
				Asym3=1-(1-Asym2)*Alpha
				Asym4=1-(1-Asym3)*Alpha
			endif
		endif
		
		AsymArea3=AsymArea2*Asym3
		AsymArea4=AsymArea3*Asym4
		
		TotalArea=1+AsymArea1+AsymArea2+AsymArea3+AsymArea4
		
		Return (XPSFitAssPseudoVoigt(x, Position, FWHM, GL)+AsymArea1*XPSFitAssGauss(x, AsymPosition1, AsymFWHM1)+AsymArea2*XPSFitAssGauss(x, AsymPosition2, AsymFWHM2)+AsymArea3*XPSFitAssGauss(x, AsymPosition3, AsymFWHM3)+AsymArea4*XPSFitAssGauss(x, AsymPosition4, AsymFWHM4))/TotalArea
	endif
end



ThreadSafe Static Function XPSFitAssAsymPseudoVoigtShirley(x, Position, FWHM, GL, Asym1, Asym2)
//	Calculates the Shirley background for an asymmetric pseudo-Voigt peak with an area of one using a tail made from the sum of 4 gaussians
//	The combined peak will no longer have a FWHM of FWHM!!!!
Variable x, Position, FWHM, GL, Asym1, Asym2
Variable AsymFWHM1=0, AsymFWHM2=0, AsymFWHM3=0, AsymFWHM4=0
Variable AsymPosition1=0, AsymPosition2=0, AsymPosition3=0, AsymPosition4=0
Variable AsymArea1=0, AsymArea2=0, AsymArea3=0, AsymArea4=0
Variable TotalArea=0, Alpha=0, Asym3=0, Asym4=0

//Variable i=0, ReturnValue=0, AsymPosition=Position, AsymFWHM=FWHM, AsymArea=1, TotalArea=1
//	ReturnValue=XPSFitAssPseudoVoigtShirley(x, Position, FWHM, GL)
//	for (i=1; i<5; i=i+1)
//		AsymPosition=AsymPosition+0.5*AsymFWHM
//		AsymFWHM=1.5*AsymFWHM
//		AsymArea=AsymArea*(1-(1-Asym1)/Asym2^i)
//		TotalArea=TotalArea+AsymArea
//		ReturnValue=ReturnValue+AsymArea*XPSFitAssGaussShirley(x, AsymPosition, AsymFWHM)
//	endfor
//	Return ReturnValue

	if (Asym1==0)
		Return XPSFitAssPseudoVoigtShirley(x, Position, FWHM, GL)
	else
		AsymFWHM1=1.5*FWHM
		AsymFWHM2=1.5*AsymFWHM1
		AsymFWHM3=1.5*AsymFWHM2
		AsymFWHM4=1.5*AsymFWHM3

		AsymPosition1=Position+0.5*FWHM
		AsymPosition2=AsymPosition1+0.5*AsymFWHM1
		AsymPosition3=AsymPosition2+0.5*AsymFWHM2
		AsymPosition4=AsymPosition3+0.5*AsymFWHM3
	
		AsymArea1=Asym1
		AsymArea2=AsymArea1*Asym2
		
		if (Asym1>=Asym2)
			Alpha=Asym2/Asym1
			Asym3=Asym2*Alpha
			Asym4=Asym3*Alpha
		else
			if (Asym1==1)
				Asym3=0
				Asym4=0
			else
				Alpha=(1-Asym2)/(1-Asym1)
				Asym3=1-(1-Asym2)*Alpha
				Asym4=1-(1-Asym3)*Alpha
			endif
		endif
		
		AsymArea3=AsymArea2*Asym3
		AsymArea4=AsymArea3*Asym4
	
		TotalArea=1+AsymArea1+AsymArea2+AsymArea3+AsymArea4

		Return (XPSFitAssPseudoVoigtShirley(x, Position, FWHM, GL)+AsymArea1*XPSFitAssGaussShirley(x, AsymPosition1, AsymFWHM1)+AsymArea2*XPSFitAssGaussShirley(x, AsymPosition2, AsymFWHM2)+AsymArea3*XPSFitAssGaussShirley(x, AsymPosition3, AsymFWHM3)+AsymArea4*XPSFitAssGaussShirley(x, AsymPosition4, AsymFWHM4))/TotalArea
	endif
end





//     -----<<<<<     Fit Overview     >>>>>-----
//	This section contains the functions used by the Fit Overview menu item

Static Function XPSFitOverview(FolderOrGroup)
//	Creates a window to view to fit coefficients
String FolderOrGroup
Variable n=120/ScreenResolution, i=0
String ActiveWindow="XPSFitOverviewWindow"

	//	Updates the history of procedures used with the current experiment
	XPSUpdateProcedureHistory()
	
	//	Brings the constraints table and graph window to the front if they exists
	DoWindow/F XPSFitOverviewWindow

	//	If the graph window does not exist it is created
	if (V_Flag==0)
		
		//	Creates the folder to hold all the permanent waves, used for different bureaucratic purposes, such as listboxes, displaying waves, etc...
		NewDataFolder/O root:Programming
		DFREF ProgramFolder=root:Programming
	
		//	Creates the folder to hold all the temporary waves, created during a function call and deleted again at the end of the function call.
		NewDataFolder/O root:Programming:Temp
		
		//	Creates the data folders used to hold the displayed waves and the saved fits
		NewDataFolder/O root:Programming:FitOverviewActiveGroupWaves
		NewDataFolder/O root:Programming:FitOverviewActiveTraceWaves
		NewDataFolder/O root:Programming:FitOverviewSavedGroupWaves
		
		//	Creates the wave used to define the colours of the Fit Overview traces
		XPSFitOverviewCreateColourWave()
		
		//	Creates a new zero length datawave as a placeholder in the empty graph. This will keep the graph up when all other traces have been removed.
		Make/O/N=0 ProgramFolder:XPSFitOverviewEmptyWave /Wave=EmptyWave

		//	Creates the graph. To fit on the lab screen it should be no wider than 775*n.
		Display /W=(105, 140, 105+775*n, 140+375*n) /K=1 /N=$ActiveWindow EmptyWave
		ModifyGraph /W=$ActiveWindow hideTrace(XPSFitOverviewEmptyWave)=1
		Label /W=$ActiveWindow bottom "x-Axis (\\u)"
		Label /W=$ActiveWindow left "y-Axis (\\u)"
		
		//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
		SetWindow $ActiveWindow hook(KillHook)=XPSFitOverviewHookFunc
		
		//	Adjusts the left margin to make room for the listbox. Using a control bar to make a margin will prevent the margin from being printed.
		ControlBar/L/W=$ActiveWindow 450
		ModifyGraph /W=$ActiveWindow cbRGB=(65534,65534,65534)
		
		//	Creates a new guide used to determine the right border of the ControlPanel subwindow
		DefineGuide /W=$ActiveWindow ControlPanelRight={FL, 400/n}

		//	Adjusts the top margin to make room for the buttons. I don't use a control bar for the top margin because Igor insists on a horizontal line between the top control bar and the graph
		ModifyGraph /W=$ActiveWindow margin(top)=30
		
		//	Changes the colours of the cursors to red and blue with dashed crosshairs
		Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$ActiveWindow A
		Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$ActiveWindow B

		//	Selects the displayed spectrum type
		PopupMenu DisplayTypePopUp bodyWidth=70, mode=5, pos={480,5}, proc=EccentricXPS#XPSFitOverviewDisplayTypePopup, value="AES;IR;NEXAFS;QMS;XPS;", win=$ActiveWindow, help={"Selects the displayed spectrum type"}
		//	Offsets all traces by their minimum value
		Button OffsetButton proc=EccentricXPS#XPSFitOverviewToggleButton, size={70,20}, title="Offset", userdata="EccentricXPS#XPSFitOverviewUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Subtracts the minimum y-value from each spectrum"}
		//	Scales all traces by their maximum value
		Button ScaleButton proc=EccentricXPS#XPSFitOverviewToggleButton, size={70,20}, title="Scale", userdata="EccentricXPS#XPSFitOverviewUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Divides each spectrum by the maximum y-value"}
		//	Creates a waterfall plot
		//Button WaterfallButton proc=EccentricXPS#XPSViewToggleButton, size={70,20}, title="Waterfall", userdata="EccentricXPS#XPSViewSpectraUpdateGraph", userdata(Status)="0", win=$ActiveWindow, help={"Creates a waterfall plot"}
		//	Sets the y-spacing used for the waterfall plot
		//SetVariable SetWaterfall, limits={0.01,2,0}, format="%.3f", proc=EccentricXPS#XPSViewSetWaterfall, size={50,20}, title=" ", value=_NUM:0.2, win=$ActiveWindow, help={"Determines the spacing between traces in the waterfall plot"}

//	MUST HAVE USED XX FOR A REASON....
//xx
		//	Creates a table with the displayed spectrum and fit waves. Useful for copying and pasting into other programs
		Button EditWavesButton proc=EccentricXPS#XPSFitOverviewEditWaves, size={70,20}, title="Table", win=$ActiveWindow, help={"Creates a table with the displayed traces. Useful for copying and pasting into other programs"}
		//	Creates a copy of the graph, without any controls, meant as a starting point for a more presentable figure
		Button NewWinButton proc=EccentricXPS#XPSViewNewWin, size={70,20}, title="New Win", win=$ActiveWindow, help={"Creates a copy of the graph, without any controls, meant as a starting point for a more presentable figure"}
		//	Opens the associated help file
		Button HelpButton proc=EccentricXPS#XPSViewHelp, size={70,20}, title="Help", userdata="Fit Overview", win=$ActiveWindow, help={"Opens the help file"}

		//	Creates the waves to hold the information about the groups and traces to display		
		XPSFitOverviewChangeGroup(FolderOrGroup, 0)

		//	Creates or recreates the controls for the Fit Overview window
		XPSFitOverviewBuildControls()
		
		//	Updates the Fit Overview graph
		XPSFitOverviewUpdateGraph()
	endif
end



Static Function XPSFitOverviewToggleButton(B_Struct) : ButtonControl
//	WORK IN PROGRESS
//	Toggles the appearance of the button and runs the update function if the button is pressed
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Finds the status of the button (1 = clicked, 0 = not clicked)
		String StatusStr=GetUserData(B_Struct.win, B_Struct.ctrlName, "Status")

		//	Changes the status userdata and the colour of the button
		If (CmpStr(StatusStr, "0")==0)
			Button $B_Struct.ctrlName fColor=(30000,30000,30000), userData(Status)="1", win=$B_Struct.win
		else
			Button $B_Struct.ctrlName fColor=(0,0,0), userData(Status)="0", win=$B_Struct.win
		endif

//		//	Finds the displayed data folder
//		DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
//		
//		if (CmpStr(B_Struct.win, "XPSViewSpectraWindow")!=0)
//	
//			//	Finds the name of the wave to display
//			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")	
//		else
//		
//			//	The View Spectra window doesn't need a wave to display
//			Wave/Z DisplayedWave=$""
//		endif
//		
//		//	Ensures that the value for the control has been updated before running the UpdateGraph function
//		ControlUpdate /W=$B_Struct.win $B_Struct.ctrlName
//
//		//	Runs the specified update function
//		FUNCREF XPSViewProtoUpdate UpdateFunction=$B_Struct.userdata
//		UpdateFunction(B_Struct.win, ActiveFolder, DisplayedWave)
	endif
	
	//	Needed, but not used
	Return 0
end



Static Function XPSFitOverviewDisplayTypePopup(PU_Struct) : PopUpMenuControl
//	Sets the axis scaling and labels to match the type of spectrum displayed
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Ignores further calls to the popup menu procedure until this one has finished
		PU_Struct.blockReentry=1
		
		//	Updates the Fit Overview graph
		XPSFitOverviewUpdateGraph()
		
		//	Prevent multiple calls.....
	endif
	Return 0
end



Static Function XPSFitOverviewChangeGroup(FolderOrGroup, GroupNum)
//	Creates the waves to hold the information about the groups and traces to display
String FolderOrGroup
Variable GroupNum

	//	Finds the maximum number of peaks in the indicated FolderOrGroup and creates the GroupV and GroupS waves
	Variable MaxNumberOfPeaks=XPSFitOverviewExtractGroupWaves(FolderOrGroup, GroupNum)
		
	//	Displays the areas of all peaks, and, if at least two peaks exist, the sum of all peaks, as default
	Variable NumNewTraces=0
	if (MaxNumberOfPeaks>1)
		NumNewTraces=MaxNumberOfPeaks+1
	elseif (MaxNumberOfPeaks==1)
		NumNewTraces=1
	endif

	//	Checks if the waves used to construct the Fit Overview controls exist
	DFREF ProgramFolder=root:Programming
	Wave/T/Z Groups=ProgramFolder:XPSFitOverviewGroups
	Wave/T/Z Traces=ProgramFolder:XPSFitOverviewTraces
	Wave/B/U/Z GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden
	Wave/B/U/Z WhichGroup=ProgramFolder:XPSFitOverviewWhichGroup
	Wave/B/U/Z TraceHidden=ProgramFolder:XPSFitOverviewTraceHidden
	Wave/B/U/Z TraceColour=ProgramFolder:XPSFitOverviewTraceColour

	//	If all the relevant waves do not exist they are created
	if (WaveExists(Groups)==0 || WaveExists(Traces)==0 || WaveExists(GroupHidden)==0 || WaveExists(WhichGroup)==0 || WaveExists(TraceHidden)==0 || WaveExists(TraceColour)==0)
	
		//	Creates the waves to hold the information about the groups and traces to display
		Make/O/T/N=(1) ProgramFolder:XPSFitOverviewGroups/WAVE=Groups
		Make/O/T/N=(NumNewTraces, 4) ProgramFolder:XPSFitOverviewTraces/WAVE=Traces
	
		Groups[0]=FolderOrGroup
	
		//	Creates the waves to hold the information about whether a trace or group is hidden, which group a trace belongs to and which colour the trace should have
		Make/O/B/U/N=(1) ProgramFolder:XPSFitOverviewGroupHidden/WAVE=GroupHidden
		Make/O/B/U/N=(NumNewTraces) ProgramFolder:XPSFitOverviewWhichGroup/WAVE=WhichGroup
		Make/O/B/U/N=(NumNewTraces) ProgramFolder:XPSFitOverviewTraceHidden/WAVE=TraceHidden
		Make/O/B/U/N=(NumNewTraces) ProgramFolder:XPSFitOverviewTraceColour/WAVE=TraceColour
		
		GroupHidden=0
		FastOP WhichGroup=(GroupNum)
		FastOP TraceHidden=0
		TraceColour[]=p
		
		//	Displays the areas of all peaks and the sum of all peaks as default
		if (NumNewTraces>0)
		
			if (MaxNumberOfPeaks>1)
				Traces[0][0]="Area Sum"
				Traces[0][1]="0"

				Traces[1, NumNewTraces-1][0]="Area"
				Traces[1, NumNewTraces-1][1]=Num2iStr(p-1)
				Traces[][2]="Wave No."
				Traces[][3]="0"
			else
				Traces[][0]="Area"
				Traces[][1]=Num2iStr(p)
				Traces[][2]="Wave No."
				Traces[][3]="0"
			endif
		endif
	else
	
		//	Sets the group selection
		Groups[GroupNum]=FolderOrGroup

		//	Finds the old number of traces in group GroupNum
		Extract/O/FREE/INDX WhichGroup, IndexWave, (WhichGroup==GroupNum)
		Variable NumOldTraces=NumPnts(IndexWave)
		
		//	The position to add the new traces in (or remove the old traces from)
		Variable AddRemoveAtPosition=0
		
		//	The position of the first trace in the group is used
		if (NumOldTraces>0)
			AddRemoveAtPosition=IndexWave[0]
		else
			
			//	If no traces exist in the group, the new traces are added after the last trace in the previous groups
			Extract/O/FREE/INDX WhichGroup, IndexWave, (WhichGroup<GroupNum)
			AddRemoveAtPosition=NumPnts(IndexWave)
		endif

		//	Finds the change in the total number of displayed traces
		Variable Increment=NumNewTraces-NumOldTraces

		//	Finds the total number of old traces, including traces from other groups
		Variable NumOldTracesTotal=DimSize(Traces, 0)
				
		//	Checks if the number of traces has changed
		if (Increment!=0)
		
			//	Finds the position from where trace references need to be shifted
			Variable ShiftFromPosition=AddRemoveAtPosition+NumOldTraces
		
			//	Checks if any traces exists with trace numbers higher than those in the current group
			if (ShiftFromPosition<NumOldTracesTotal)
			
				//	Shifts the selection of any traces higher than AddRemoveAtPosition by the selected increment
				Traces[ShiftFromPosition, *][0,2;2]=XPSFitOverviewShiftTraceStr(Traces[p][q], ShiftFromPosition, AddRemoveAtPosition, Increment)
			endif
		endif



		//	Checks if traces needs to be added or removed
		if (Increment<0)

			//	Deletes increment number of traces, starting from FirstOldTrace
			DeletePoints /M=0 AddRemoveAtPosition, Abs(Increment), Traces, WhichGroup, TraceHidden, TraceColour

		elseif (Increment>0)

			//	Checks if any traces exists (including in other groups)
			if (NumOldTracesTotal==0)
			
				//	If the number of traces is zero, the Traces wave will have lost it's dimensionality and it has to be recreated
				Make/O/T/N=(NumNewTraces, 4) ProgramFolder:XPSFitOverviewTraces=""

				//	Adds increment number of new traces, before FirstOldTrace
				InsertPoints /M=0 AddRemoveAtPosition, Increment, WhichGroup, TraceHidden, TraceColour
			else
			
				//	Adds increment number of new traces, before FirstOldTrace
				InsertPoints /M=0 AddRemoveAtPosition, Increment, Traces, WhichGroup, TraceHidden, TraceColour
			endif
		endif

		

		//	Checks if the new group contains any traces
		if (NumNewTraces>0)
			
			Variable LastTraceInGroup=AddRemoveAtPosition+NumNewTraces-1
		
			//	If only one peak exist in the group, only the area of that peak is displayed
			if (NumNewTraces==1)

				Traces[AddRemoveAtPosition][0]="Area"
				Traces[AddRemoveAtPosition][1]="0"
				Traces[AddRemoveAtPosition][2]="Wave No."
				Traces[AddRemoveAtPosition][3]="0"

			//	If more than one peak exist in the group, the sum of all peak areas is displayed together with each individual peak area
			elseif (NumNewTraces>1)

				Traces[AddRemoveAtPosition][0]="Area Sum"
				Traces[AddRemoveAtPosition][1]="0"
			
				Traces[AddRemoveAtPosition+1, LastTraceInGroup][0]="Area"
				Traces[AddRemoveAtPosition+1, LastTraceInGroup][1]=Num2iStr(p-AddRemoveAtPosition-1)
				Traces[AddRemoveAtPosition, LastTraceInGroup][2]="Wave No."
				Traces[AddRemoveAtPosition, LastTraceInGroup][3]="0"
			endif
			
			//	Associates the new traces with group GroupNum and makes them visible
			WhichGroup[AddRemoveAtPosition, LastTraceInGroup]=GroupNum
			TraceHidden[AddRemoveAtPosition, LastTraceInGroup]=0
			
			//	Finds the least used colour and uses that for the new trace (255 is not a colour and will therefore not affect the histogram used to select the least used colour)
			TraceColour[AddRemoveAtPosition, LastTraceInGroup]=255
			TraceColour[AddRemoveAtPosition, LastTraceInGroup]=XPSFitOverviewGiveColour()
		endif
	endif
end



Static Function XPSFitOverviewBuildControls()
//	Creates or recreates the controls for the Fit Overview window

	String BaseWindow="XPSFitOverviewWindow"
	String SubWindow="ControlPanel"
	
	//	Kills the ControlPanel subwindow, if it exists, thereby removing all existing controls. DoWindow does not work for subwindows
	if (WinType(BaseWindow+"#"+SubWindow)>0)
		KillWindow $(BaseWindow+"#"+SubWindow)
	endif
	
	//	Recreates the ControlPanel subwindow and deactivates it, removing the dashed yellow line otherwise around it
	NewPanel /FG=(FL, FT, ControlPanelRight, FB) /HOST=$BaseWindow /K=2 /N=$SubWindow /NA=2
	SetActiveSubwindow $BaseWindow
	
	//	Makes the ControlPanel Invisible (65534 is one point off maximum white, because 65535 has a special meaning on windows machines...)
	ModifyPanel /W=$(BaseWindow+"#"+SubWindow) cbRGB=(65534,65534,65534), frameStyle=0

	//	The number of groups to create
	DFREF ProgramFolder=root:Programming
	Wave/T Groups=ProgramFolder:XPSFitOverviewGroups
	Variable NumberOfGroups=NumPnts(Groups)
	
	//	The number of traces to create
	Wave/T Traces=ProgramFolder:XPSFitOverviewTraces
	Variable NumberOfTraces=DimSize(Traces, 0)

	//	The starting y position of the controls
	Variable YPos=5
	
	//	The number of the next trace to be added
	Variable TraceNum=0

	//	Creates the controls for groups
	Variable i=0
	for (i=0; i<NumberOfGroups; i+=1)
		XPSFitOverviewAddGroupControl(i, TraceNum, NumberOfTraces, YPos)
	endfor
end



Static Function XPSFitOverviewAddGroupControl(GroupNum, TraceNum, NumberOfTraces, YPos)
//	Creates the set of controls for the next group
Variable GroupNum, NumberOfTraces
Variable &TraceNum, &YPos	//	DisplayHelpTopic "Pass-By-Reference"

	//	Show coefs also needed, or goto Fit All
	//	Text box with notes??

	//	The suffix used to make the control names unique
	String Suffix=Num2iStr(GroupNum)
	
	String ActiveWindow="XPSFitOverviewWindow#ControlPanel"
	
	//	Finds the wave with information about whether a group is hidden or not
	DFREF ProgramFolder=root:Programming
	Wave/B/U GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden

	//	Creates a checkbox to hide/unhide the selected dataset
	CheckBox $("HideGroupCheckBox"+Suffix) pos={5, YPos+3}, proc=EccentricXPS#XPSFitOverviewHideCheckBox, size={10,10}, title=" ", value=(GroupHidden[GroupNum]==0), userdata=Suffix, win=$ActiveWindow, help={"Hide/unhides the selected data set"}
	//	Selects the active data folder, listing root: and all it's subfolders with the exception of root:Programming and it's subfolders. To list the default folder (set with SetDataFolder) and all it's subfolders instead with no exceptions, use XPSViewDataFoldersList(:, $"")
	PopupMenu $("DataFolderPopUp"+Suffix) bodyWidth=250, mode=1, pos={225, YPos}, proc=EccentricXPS#XPSFitOverviewSetDF, value="- select group or data folder -;"+EccentricXPS#XPSViewGroupsAndFoldersList(root:, root:Programming, "SavedFits"), userdata=Suffix, win=$ActiveWindow, help={"Selects the group or data folder to display the fits for"}
	//	Removes the group, all traces in the group and associated controls
	Button $("RemoveGroupButton"+Suffix) proc=EccentricXPS#XPSFitOverviewRemoveTrace, size={16,16}, title="\f01X", userdata=Suffix, win=$ActiveWindow, help={"Removes the group, all traces in the group and associated controls"}
	
	//	Finds the group or data folder to display
	Wave/T Groups=ProgramFolder:XPSFitOverviewGroups
	
	//	Changes the popup menu selection to the group or data folder to display
	if (StrLen(Groups[GroupNum])>0)
		PopupMenu $("DataFolderPopUp"+Suffix) popmatch=Groups[GroupNum], win=$ActiveWindow
	endif

	//	Creates the next line of controls
	YPos+=35
		
	//	Is more of a reextract or refresh button
	Button $("ReacquireButton"+Suffix) proc=EccentricXPS#XPSFitOverviewReaquire, pos={25,YPos}, size={76,20}, title="Reacquire", userdata=Suffix, win=$ActiveWindow, help={"xxxxxxxxxxxx"}
	//	Is more of a reextract or refresh button
	Button $("OverwriteButton"+Suffix) noproc, pos+={1,0}, size={76,20}, title="Overwrite", userdata=Suffix, win=$ActiveWindow, help={"xxxxxxxxxxxx"}
	//	Is more of a reextract or refresh button
	Button $("SaveButton"+Suffix) noproc, pos+={1,0}, size={76,20}, title="Save", userdata=Suffix, win=$ActiveWindow, help={"xxxxxxxxxxxx"}

	//	Creates the next line of controls
	YPos+=25

	//	Is more of a reextract or refresh button
	Button $("SpectraButton"+Suffix) noproc, pos={25,YPos}, size={76,20}, title="Spectra", userdata=Suffix, win=$ActiveWindow, help={"xxxxxxxxxxxx"}
	//	Is more of a reextract or refresh button
	Button $("FitAssButton"+Suffix) noproc, pos+={1,0}, size={76,20}, title="Fit Ass", userdata=Suffix, win=$ActiveWindow, help={"xxxxxxxxxxxx"}
	//	Is more of a reextract or refresh button
	Button $("XSWButton"+Suffix) noproc, pos+={1,0}, size={76,20}, title="XSW", userdata=Suffix, win=$ActiveWindow, help={"xxxxxxxxxxxx"}
	
	//	Creates the next line of controls
	YPos+=35
	
	//	Creates the base list of coefficients for the CoefPopUp menus and the list of peaks for the PeakPopUp menus
	String CoefPopUpStr=XPSFitOverviewCoefStr(GroupNum)
	String PeakPopUpStr=XPSFitOverviewPeakStr(GroupNum)

	//	Finds the wave with information about which group each trace belongs to
	Wave/B/U WhichGroup=ProgramFolder:XPSFitOverviewWhichGroup
	
	//	Finds the traces belonging to group GroupNum
	Extract/FREE/INDX/O WhichGroup, IndexWave, (WhichGroup==GroupNum)
	Variable TracesInGroup=NumPnts(IndexWave)
	
	//	Adds the trace controls for the group one at a time
	Variable i=0
	for (i=0; i<TracesInGroup; i+=1)

		//	The trace number of the next control
		TraceNum=IndexWave[i]

		//	Creates the set of controls for the next trace
		XPSFitOverviewAddTraceControl(GroupNum, TraceNum, NumberOfTraces, YPos, CoefPopUpStr, PeakPopUpStr)
	endfor
	
	//	Adds the group number and the number of the next trace to be added to the user data
	String AddUserdata=Suffix+";"+Num2iStr(TraceNum)

	//	Adds an additional trace to the group
	Button $("AddTraceButton"+Suffix) proc=EccentricXPS#XPSFitOverviewAddTrace, pos={70,YPos}, size={100,20}, title="Add Trace", userdata=AddUserdata, win=$ActiveWindow, help={"Adds an additional trace to the group"}
	//	Adds an additional group after this one
	Button $("AddGroupButton"+Suffix) proc=EccentricXPS#XPSFitOverviewAddTrace, pos={175,YPos}, size={100,20}, title="Add Group", userdata=AddUserdata, win=$ActiveWindow, help={"Adds an additional group after this one"}
	
	//	Creates the next line of controls
	YPos+=35
end



Static Function XPSFitOverviewReaquire(B_Struct) : ButtonControl
//	Reaquires the GroupV and GroupS waves
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Finds the group to be reaquired
		Variable GroupNum=Str2Num(B_Struct.userdata)
		
		//	Updates the control waves with the popup menu selections
		XPSFitOverviewUpdateCtrlWaves()
		
		//	Finds the FolderOrGroup selection of group GroupNum
		DFREF ProgramFolder=root:Programming
		Wave/T Groups=ProgramFolder:XPSFitOverviewGroups
		String FolderOrGroup=Groups[GroupNum]
		
		//	Reaquires the GroupV and GroupS waves
		XPSFitOverviewChangeGroup(FolderOrGroup, GroupNum)

		//	Recreates the controls for the Fit Overview window
		XPSFitOverviewBuildControls()
		
		//	Updates the Fit Overview graph
		XPSFitOverviewUpdateGraph()
	endif
	Return 0
end



Static Function XPSFitOverviewAddTraceControl(GroupNum, TraceNum, NumberOfTraces, YPos, CoefPopUpStr, PeakPopUpStr)
//	Creates the set of controls for the next trace
Variable GroupNum, NumberOfTraces
Variable &TraceNum, &YPos	//	DisplayHelpTopic "Pass-By-Reference"
String CoefPopUpStr, PeakPopUpStr

	//	The suffix used to make the control names unique
	String Suffix=Num2iStr(TraceNum)
	
	String ActiveWindow="XPSFitOverviewWindow#ControlPanel"
	
	//	Finds the wave with information about whether a trace is hidden or not
	DFREF ProgramFolder=root:Programming
	Wave/B/U TraceHidden=ProgramFolder:XPSFitOverviewTraceHidden
	Wave/B/U GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden
	Wave/W/U ColourWave=ProgramFolder:XPSFitOverviewColours
	Wave/B/U TraceColour=ProgramFolder:XPSFitOverviewTraceColour
	
	//	The default light grey colour of the ColourBox
	Variable Red=60000
	Variable Green=60000
	Variable Blue=60000
	
	//	If the trace is not hidden the colour determined by TraceColour is used
	if ((TraceHidden[TraceNum]==0) && (GroupHidden[GroupNum]==0))
		Red=ColourWave[TraceColour[TraceNum]][0]
		Green=ColourWave[TraceColour[TraceNum]][1]
		Blue=ColourWave[TraceColour[TraceNum]][2]
	endif
	
	//	Adds the list of traces to the base list of coefficients for the CoefPopUp menus. Only trace with numbers lower than TraceNum can be selected to prevent infinite loops
	CoefPopUpStr+=XPSFitOverviewCoefStrTrace(TraceNum)
	
	//	Creates a checkbox to hide/unhide the selected dataset
	CheckBox $("HideTraceCheckBox"+Suffix) pos={33, YPos+3}, proc=EccentricXPS#XPSFitOverviewHideCheckBox, size={10,10}, side=1, title=Suffix+".  ", value=(TraceHidden[TraceNum]==0), userdata=Suffix, win=$ActiveWindow, help={"Hide/unhides the selected data set"}
	//	Creates a box with the colour of the plotted fit coefficients
	ValDisplay $("ColourBox"+Suffix) pos={45, YPos}, barmisc={20, 0}, barBackColor=(Red, Green, Blue), disable=2, frame=4, mode=0, size={20, 20}, value=_NUM:0, win=$ActiveWindow, help={"Displays the colour of the plotted fit coefficients"}
	//	Selects the fit coefficient to display as y-axis
	PopupMenu $("YCoefPopUp"+Suffix) pos={120, YPos}, bodyWidth=100, disable=0, mode=1, popvalue="- select coef -", proc=EccentricXPS#XPSFitOverviewSelectionPop, value=#CoefPopUpStr, userdata=Suffix, win=$ActiveWindow
	PopupMenu $("YPeakPopUp"+Suffix) pos={225, YPos}, bodyWidth=100, disable=0, mode=1, popvalue="- select peak -", proc=EccentricXPS#XPSFitOverviewSelectionPop, value=#PeakPopUpStr, userdata=Suffix, win=$ActiveWindow
	//	Removes the trace and associated controls
	Button $("RemoveTraceButton"+Suffix) proc=EccentricXPS#XPSFitOverviewRemoveTrace, size={16,16}, title="\f01X", userdata=Suffix, win=$ActiveWindow, help={"Removes the trace and associated controls"}

	//	Creates the next line of controls
	YPos+=25

	//	Selects the fit coefficient to display as x-axis
	PopupMenu $("XCoefPopUp"+Suffix) pos={120, YPos}, title="vs. ", bodyWidth=100, disable=0, mode=1, popvalue="- select coef -", proc=EccentricXPS#XPSFitOverviewSelectionPop, value=#CoefPopUpStr, userdata=Suffix, win=$ActiveWindow
	PopupMenu $("XPeakPopUp"+Suffix) pos={225, YPos}, bodyWidth=100, disable=0, mode=1, popvalue="- select peak -", proc=EccentricXPS#XPSFitOverviewSelectionPop, value=#PeakPopUpStr, userdata=Suffix, win=$ActiveWindow
	
	//	Creates the next line of controls
	YPos+=35

	//	Finds the fit coefficients to display
	Wave/T Traces=ProgramFolder:XPSFitOverviewTraces

	//	Changes the popup menu selections to the coefficients to display
	if (StrLen(Traces[TraceNum][0])>0)
		PopupMenu $("YCoefPopUp"+Suffix) popmatch=Traces[TraceNum][0], win=$ActiveWindow
	endif
	if (StrLen(Traces[TraceNum][1])>0)
		PopupMenu $("YPeakPopUp"+Suffix) popmatch=Traces[TraceNum][1], win=$ActiveWindow
	endif
	if (StrLen(Traces[TraceNum][2])>0)
		PopupMenu $("XCoefPopUp"+Suffix) popmatch=Traces[TraceNum][2], win=$ActiveWindow
	endif
	if (StrLen(Traces[TraceNum][3])>0)
		PopupMenu $("XPeakPopUp"+Suffix) popmatch=Traces[TraceNum][3], win=$ActiveWindow
	endif
	
	//	The number of the next trace to be added
	TraceNum+=1
end



Static Function/S  XPSFitOverviewCoefStr(GroupNum)
//	Creates the base list of coefficients for the CoefPopUp menus
Variable GroupNum

	//	Finds the maximum number of peaks in the fit
	Variable MaxNumberOfPeaks=XPSFitOverviewMaxNumPeaks(GroupNum)
	
	//	The extra quote is necessary for the popup menu
	String CoefPopUpStr="\"- select coef -;"
	
	//	Checks if a linear background exists
	if (MaxNumberOfPeaks>-1)
	
		//	Adds the linear background coefficients to the list of items
		CoefPopUpStr+=" ;Intercept;Slope;"
		
		//	Checks if at least one peak exists
		if (MaxNumberOfPeaks>0)
	
			//	Adds the peak coefficients to the list of items
			CoefPopUpStr+="  ;Position;Area;FWHM;GL Ratio;Asym1;Asym2;Shirley Coef;"
		endif
	endif
	
	//	The wave number is always a choice
	CoefPopUpStr+="   ;Wave No.;"
	
	//	Checks if at least two peaks exist
	if (MaxNumberOfPeaks>1)
	
		//	Adds the sum of all peak areas to the list of items
		CoefPopUpStr+="Area Sum;"
	endif

	//	The XSW energy is always a choice (until I can figure out how to implement it properly)
	CoefPopUpStr+="XSW Energy;"
	
	//	Returns the list of items
	Return CoefPopUpStr
end



Static Function/S  XPSFitOverviewCoefStrTrace(TraceNum)
//	Adds the list of traces to the base list of coefficients for the CoefPopUp menus
Variable TraceNum

	String CoefPopUpStrTrace=""

	if (TraceNum>0)
	
		//	Adds a blank space
		CoefPopUpStrTrace+="    ;"
	
		//	Adds the traces one at a time, up to, but not including trace TraceNum
		Variable i=0
		for (i=0; i<TraceNum; i+=1)
			CoefPopUpStrTrace+="Trace "+Num2iStr(i)+";"
		endfor
	endif
	
	//	The extra quote is necessary for the popup menu
	CoefPopUpStrTrace+="\""

	//	Returns the list of items for the CoefPopUp menus, with the list of traces added
	Return CoefPopUpStrTrace
end



Static Function/S  XPSFitOverviewPeakStr(GroupNum)
//	Creates the list of peaks for the PeakPopUp menus
Variable GroupNum

	//	Finds the maximum number of peaks in the fit
	Variable MaxNumberOfPeaks=XPSFitOverviewMaxNumPeaks(GroupNum)

	//	The extra quote is necessary for the popup menu
	String PeakPopUpStr="\"- select peak -;"
	
	if (MaxNumberOfPeaks>0)
	
		//	Adds a blank space
		PeakPopUpStr+=" ;"
		
		//	Add the peak numbers one at a time
		Variable i=0
		for (i=0; i<MaxNumberOfPeaks; i+=1)
			PeakPopUpStr+=Num2iStr(i)+";"
		endfor
	endif

	//	The extra quote is necessary for the popup menu
	PeakPopUpStr+="\""
	
	//	Returns the list of peaks: "\"- select peak -; ;0;1;2;3;4;\""
	Return PeakPopUpStr
end



Static Function XPSFitOverviewRemoveTrace(B_Struct) : ButtonControl
//	Removes a trace, or a group and all associated traces, from the Fit Overview controls
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	The waves used to construct the Fit Overview controls
		DFREF ProgramFolder=root:Programming
		Wave/T Groups=ProgramFolder:XPSFitOverviewGroups
		Wave/T Traces=ProgramFolder:XPSFitOverviewTraces
		Wave/B/U GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden
		Wave/B/U WhichGroup=ProgramFolder:XPSFitOverviewWhichGroup
		Wave/B/U TraceHidden=ProgramFolder:XPSFitOverviewTraceHidden
		Wave/B/U TraceColour=ProgramFolder:XPSFitOverviewTraceColour

		//	Updates the control waves with the popup menu selections
		XPSFitOverviewUpdateCtrlWaves()
		
		Variable GroupNum=0
		
		//	Finds the total number of traces
		Variable NumberOfTraces=DimSize(Traces, 0)
		
		//	Checks if remove trace or remove group was pressed
		String ButtonName=B_Struct.ctrlname
		if (CmpStr(ButtonName[0,16], "RemoveTraceButton")==0)
		
			//	Removes a trace
		
			//	Finds the trace to be removed
			Variable TraceNum=Str2Num(B_Struct.userdata)
			
			//	Shifts the selection of any traces higher than TraceNum by -1
			if (TraceNum+1<NumberOfTraces)
				Traces[TraceNum+1, *][0,2;2]=XPSFitOverviewShiftTraceStr(Traces[p][q], TraceNum+1, TraceNum, -1)
			endif
			
			//	Finds the group of the trace to be removed
			GroupNum=WhichGroup[TraceNum]
			
			//	Removes the trace from the controls
			DeletePoints /M=0 TraceNum, 1, Traces, WhichGroup, TraceHidden, TraceColour
		else
		

			//	Removes a group


			//	Finds the total number of groups
			Variable NumberOfGroups=NumPnts(Groups)
			
			//	Doesn't allow the last group to be removed
			if (NumberOfGroups>1)

				//	Finds the group to be removed
				GroupNum=Str2Num(B_Struct.userdata)
			
				//	Finds the traces belonging to group GroupNum
				Extract/O/FREE/INDX WhichGroup, IndexWave, (WhichGroup==GroupNum)
				Variable NumberOfTracesInGroup=NumPnts(IndexWave)
				
				if (NumberOfTracesInGroup>0)
				
					//	Finds the first and last traces in the group
					Variable FirstTraceInGroup=IndexWave[0]
					Variable LastTraceInGroup=IndexWave[NumberOfTracesInGroup-1]
				
					//	Shifts the selection of any traces higher than the last trace in the group by the number of traces in the deleted group
					if (LastTraceInGroup+1<NumberOfTraces)
						Traces[LastTraceInGroup+1, *][0,2;2]=XPSFitOverviewShiftTraceStr(Traces[p][q], LastTraceInGroup+1, FirstTraceInGroup, -NumberOfTracesInGroup)
					endif
				
					//	Removes the traces belonging to group GroupNum
					DeletePoints /M=0 FirstTraceInGroup, NumberOfTracesInGroup, Traces, WhichGroup, TraceHidden, TraceColour
				endif
				
				//	Finds all traces in groups with numbers greater than GroupNum and subtracts 1 from their number
				Extract/O/FREE/INDX WhichGroup, IndexWave, (WhichGroup>GroupNum)
				Variable n=NumPnts(IndexWave), i=0
				for (i=0; i<n; i+=1)
					WhichGroup[IndexWave[i]]+=-1
				endfor
				
				//	Reorders the GroupV and GroupS waves to match the new group numbers after removing the group
				XPSFitOverviewShiftGroupWaves(GroupNum, -1, NumberOfGroups)
				
				//	Removes the group from the controls
				DeletePoints /M=0 GroupNum, 1, Groups, GroupHidden
			endif
		endif
		
		//	Creates or recreates the controls for the Fit Overview window
		XPSFitOverviewBuildControls()
			
		//	Updates the traces displayed in the Fit Overview graph
		XPSFitOverviewUpdateGraph()
	endif
	Return 0
end



Static Function XPSFitOverviewShiftGroupWaves(GroupNum, Increment, NumberOfGroups)
//	Reorders the GroupV and GroupS waves to match the new group numbers after adding or removing a group
Variable GroupNum, Increment, NumberOfGroups

	//	The waves used to construct the Fit Overview controls
	DFREF ProgramFolder=root:Programming
	DFREF ActiveGroupWaves=ProgramFolder:FitOverviewActiveGroupWaves
	
	Variable i=0
	
	//	GroupNum is the group to be deleted
	if (Increment==-1)
		for (i=GroupNum+1; i<NumberOfGroups; i+=1)
			Duplicate/O ActiveGroupWaves:$("XPSFitOverviewGroupV"+Num2iStr(i)), ActiveGroupWaves:$("XPSFitOverviewGroupV"+Num2iStr(i-1))
			Duplicate/O/T ActiveGroupWaves:$("XPSFitOverviewGroupS"+Num2iStr(i)), ActiveGroupWaves:$("XPSFitOverviewGroupS"+Num2iStr(i-1))
		endfor
		
		//	Kills the obsolete group waves
		KillWaves ActiveGroupWaves:$("XPSFitOverviewGroupV"+Num2iStr(NumberOfGroups-1)), ActiveGroupWaves:$("XPSFitOverviewGroupS"+Num2iStr(NumberOfGroups-1))
	
	//	GroupNum is the position of the new group to be added
	elseif (Increment==1)
		for (i=NumberOfGroups-1; i>=GroupNum; i+=-1)
			Duplicate/O ActiveGroupWaves:$("XPSFitOverviewGroupV"+Num2iStr(i)), ActiveGroupWaves:$("XPSFitOverviewGroupV"+Num2iStr(i+1))
			Duplicate/O/T ActiveGroupWaves:$("XPSFitOverviewGroupS"+Num2iStr(i)), ActiveGroupWaves:$("XPSFitOverviewGroupS"+Num2iStr(i+1))
		endfor
		
		//	Resets the new group
		Make/O/N=0 ActiveGroupWaves:$("XPSFitOverviewGroupV"+Num2iStr(GroupNum))
		Make/O/T/N=0 ActiveGroupWaves:$("XPSFitOverviewGroupS"+Num2iStr(GroupNum))
	endif
end



Static Function/S XPSFitOverviewShiftTraceStr(CoefStr, ShiftFromTraceNum, RemoveFromTraceNum, Increment)
//	Shifts the selection of any traces higher than TraceNum by the selected increment
String CoefStr
Variable ShiftFromTraceNum, RemoveFromTraceNum, Increment

	//	Checks if the CoefStr selection is a trace
	Variable SelectedTraceNum=-1
	String OverflowStr=""
	SScanF CoefStr, "Trace %u%s", SelectedTraceNum, OverflowStr
			
	//	The selection is a trace
	if (V_flag==1)
	
		if (SelectedTraceNum>=ShiftFromTraceNum)
			
			//	If the CoefStr is a reference to a trace with number ShiftFromTraceNum or higher it is shifted by the number of traces to be added
			CoefStr="Trace "+Num2iStr(SelectedTraceNum+Increment)
				
		elseif (SelectedTraceNum>=RemoveFromTraceNum)
			
			//	If the CoefStr is a reference to a trace with a number between RemoveFromTraceNum and ShiftFromTraceNum the trace reference is removed
			CoefStr=""
			
		endif
	endif
	Return CoefStr
end



Static Function XPSFitOverviewUpdateCtrlWaves()
//	Updates the control waves with the popup menu selections

	//	The waves used to construct the Fit Overview controls
	DFREF ProgramFolder=root:Programming
	Wave/T Groups=ProgramFolder:XPSFitOverviewGroups
	Wave/T Traces=ProgramFolder:XPSFitOverviewTraces
	Wave/B/U GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden
	Wave/B/U TraceHidden=ProgramFolder:XPSFitOverviewTraceHidden
	
	String ActiveWindow="XPSFitOverviewWindow#ControlPanel"
	String Suffix=""

	//	Finds the number of displayed traces and groups		
	Variable NumberOfGroups=NumPnts(GroupHidden)
	Variable NumberOfTraces=NumPnts(TraceHidden)
	
	Variable i=0
		
	//	Counts through the displayed groups
	for (i=0; i<NumberOfGroups; i+=1)
	
		//	The suffix used to make the control names unique
		Suffix=Num2iStr(i)

		//	Copies the hidden selection
		Controlinfo /W=$ActiveWindow $("HideGroupCheckBox"+Suffix)
		GroupHidden[i]=(V_Value==0)

		//	Copies the group or data folder selection
		Controlinfo /W=$ActiveWindow $("DataFolderPopup"+Suffix)
		Groups[i]=S_Value
	endfor

	//	Counts through the displayed traces
	for (i=0; i<NumberOfTraces; i+=1)
	
		//	The suffix used to make the control names unique
		Suffix=Num2iStr(i)
		
		//	Copies the hidden selection
		Controlinfo /W=$ActiveWindow $("HideTraceCheckBox"+Suffix)
		TraceHidden[i]=(V_Value==0)

		//	Copies the y-axis selection
		Controlinfo /W=$ActiveWindow $("YCoefPopUp"+Suffix)
		Traces[i][0]=S_Value
		Controlinfo /W=$ActiveWindow $("YPeakPopUp"+Suffix)
		Traces[i][1]=S_Value

		//	Copies the x-axis selection
		Controlinfo /W=$ActiveWindow $("XCoefPopUp"+Suffix)
		Traces[i][2]=S_Value
		Controlinfo /W=$ActiveWindow $("XPeakPopUp"+Suffix)
		Traces[i][3]=S_Value
	endfor
end



Static Function XPSFitOverviewAddTrace(B_Struct) : ButtonControl
//	Adds a trace, or a group and a trace, to the Fit Overview controls
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	The waves used to construct the Fit Overview controls
		DFREF ProgramFolder=root:Programming
		Wave/T Groups=ProgramFolder:XPSFitOverviewGroups
		Wave/T Traces=ProgramFolder:XPSFitOverviewTraces
		Wave/B/U GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden
		Wave/B/U WhichGroup=ProgramFolder:XPSFitOverviewWhichGroup
		Wave/B/U TraceHidden=ProgramFolder:XPSFitOverviewTraceHidden
		Wave/B/U TraceColour=ProgramFolder:XPSFitOverviewTraceColour

		//	Updates the control waves with the popup menu selections
		XPSFitOverviewUpdateCtrlWaves()
		
		//	Finds the group and position to add the new trace in
		Variable GroupNum=Str2Num(StringFromList(0, B_Struct.userdata, ";"))
		Variable TraceNum=Str2Num(StringFromList(1, B_Struct.userdata, ";"))
		
		//	Checks if add trace or add group was pressed
		String ButtonName=B_Struct.ctrlname
		if (CmpStr(ButtonName[0,13], "AddTraceButton")==0)
		
			//	---    Adds Trace    ---
			
			//	Finds the total number of traces
			Variable NumberOfTraces=DimSize(Traces, 0)
			
			//	Shifts the selection of any traces higher than or equal to TraceNum by +1
			if (TraceNum+1<NumberOfTraces)
				Traces[TraceNum+1, *][0,2;2]=XPSFitOverviewShiftTraceStr(Traces[p][q], TraceNum, TraceNum, 1)
			endif
		
			//	Checks if the group has any previous traces
			Extract/O/FREE/INDX WhichGroup, IndexWave, (WhichGroup==GroupNum)
			Variable TracesInGroup=NumPnts(IndexWave)
			
			//	Finds the least used colour and uses that for the new trace
			Variable NewColour=XPSFitOverviewGiveColour()

			if (NumberOfTraces==0)
			
				//	If the number of traces is zero, the Traces wave will have lost it's dimensionality and has to be recreated
				Make/O/T/N=(1, 4) ProgramFolder:XPSFitOverviewTraces=""

				//	Adds the trace to the controls at position TraceNum
				InsertPoints /M=0 TraceNum, 1, WhichGroup, TraceHidden, TraceColour
			else

				//	Adds the trace to the controls at position TraceNum
				InsertPoints /M=0 TraceNum, 1, Traces, WhichGroup, TraceHidden, TraceColour
			endif
			
			//	Finds the maximum number of peaks in the fit
			Variable MaxNumberOfPeaks=XPSFitOverviewMaxNumPeaks(GroupNum)

			//	Checks if the group has any previous traces
			if (TracesInGroup>0)
			
				//	The position of the previous trace in the group
				Variable PreviousTrace=IndexWave[TracesInGroup-1]
			
				//	Makes the default selection the same as the previous trace's
				Traces[TraceNum][]=Traces[PreviousTrace][q]
			else
			
				if (MaxNumberOfPeaks>0)

					//	Defaults to displaying the area of the first peak				
					Traces[TraceNum][0]="Area"
					Traces[TraceNum][1]="0"
					Traces[TraceNum][2]="Wave No."
					Traces[TraceNum][3]="0"
				else
				
					//	Defaults to no selection
					Traces[TraceNum][]=""
				endif
			endif
			
			//	Adds the trace to group GroupNum
			WhichGroup[TraceNum]=GroupNum
			
			//	Makes the new trace visible
			TraceHidden[TraceNum]=0
			
			//	Gives the new trace the first unused colour
			TraceColour[TraceNum]=NewColour
		else
		
			//	---    Adds Group    ---
		
			//	The new group to be created
			GroupNum+=1
			
			//	Finds all groups with numbers greater than or equal to GroupNum
			Extract/O/FREE/INDX WhichGroup, IndexWave, (WhichGroup>=GroupNum)
			
			//	Counts though them one at a time
			Variable i=0, n=NumPnts(IndexWave)
			for (i=0; i<n; i+=1)

				//	Adds 1 to their number
				WhichGroup[IndexWave[i]]+=1
			endfor
			
			//	Finds the total number of groups
			Variable NumberOfGroups=NumPnts(Groups)
			
			//	Reorders the GroupV and GroupS waves to match the new group numbers after adding the new group
			XPSFitOverviewShiftGroupWaves(GroupNum, +1, NumberOfGroups)

			//	Adds the new group to the controls
			InsertPoints /M=0 GroupNum, 1, Groups, GroupHidden
			
			//	Does not select a default group or data folder to show
			Groups[GroupNum]=""
			
			//	Makes the new group visible
			GroupHidden[GroupNum]=0
		endif
		
		//	Creates or recreates the controls for the Fit Overview window
		XPSFitOverviewBuildControls()
		
		//	Updates the Fit Overview graph
		XPSFitOverviewUpdateGraph()
	endif
	Return 0
end



Static Function XPSFitOverviewUpdateGraph()
//	Updates the traces displayed in the Fit Overview graph


	//	Removes all waves from the Edit Waves table, if it exists
	//	CREATE FIT OVERVIEW EQUIVALENT
	//XPSViewEditRemoveWaves()
	
	//	Adds all displayed spectra and fit waves to the Edit Waves table, if it exists
	//	CREATE FIT OVERVIEW EQUIVALENT
	//XPSViewEditAddWaves()




	String ActiveWindow="XPSFitOverviewWindow"

	//	The waves used to construct the Fit Overview controls
	DFREF ProgramFolder=root:Programming
	DFREF ActiveGroupWaves=ProgramFolder:FitOverviewActiveGroupWaves
	Wave/B/U GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden
	Wave/B/U WhichGroup=ProgramFolder:XPSFitOverviewWhichGroup
	Wave/B/U TraceHidden=ProgramFolder:XPSFitOverviewTraceHidden
	Wave/B/U TraceColour=ProgramFolder:XPSFitOverviewTraceColour
	Wave/T Traces=ProgramFolder:XPSFitOverviewTraces
	Wave/W/U ColourWave=ProgramFolder:XPSFitOverviewColours
	
	//	Creates a list of all traces in the Fit Overview graph, excluding the empty place holder wave used to keep the graph up when all other traces have been removed.
	String AllTraces=RemoveFromList("XPSFitOverviewEmptyWave", TraceNameList(ActiveWindow,";",1), ";")

	//	Finds the number of traces on the graph
	Variable NumberOfOldTraces=ItemsInList(AllTraces, ";")
	
	//	Removes all traces from the graph (except the empty placeholder wave)
	Variable i=0, a=-1, b=0
	for (i=0; i<NumberOfOldTraces; i+=1)
		b=StrSearch(AllTraces, ";", a+1)
		RemoveFromGraph/Z /W=$ActiveWindow $(AllTraces[a+1,b-1])
		a=b
	endfor

	//	Kills all existing traces
	DFREF ActiveTraceWaves=ProgramFolder:FitOverviewActiveTraceWaves
	if (DataFolderRefStatus(ActiveTraceWaves)!=0)
		KillDataFolder/Z ActiveTraceWaves
	endif
	NewDataFolder/O ProgramFolder:FitOverviewActiveTraceWaves
	DFREF ActiveTraceWaves=ProgramFolder:FitOverviewActiveTraceWaves
	
	//	Finds the new number of traces to be displayed
	Variable NumberOfNewTraces=NumPnts(TraceHidden)
	
	//	Used hold the references to the displayed traces
	Make/O/FREE/WAVE/N=(NumberOfNewTraces) YWaves, XWaves
	
	Variable Hidden=0, GroupNum=0
	String TraceName=""
		
	//	Creates the waves to display in the Fit Overview graph
	for (i=0; i<NumberOfNewTraces; i+=1)

		//	Finds the group the trace belongs to
		GroupNum=WhichGroup[i]
	
		//	Finds the GroupV wave to create the trace from
		Wave GroupV=ActiveGroupWaves:$("XPSFitOverviewGroupV"+Num2iStr(GroupNum))

		//	Creates the y-wave used to display the Fit Overview traces
		YWaves[i]=XPSFitOverviewCreateSingleTrace(Traces[i][0], Traces[i][1], GroupV, "XPSFitOverviewYWave"+Num2iStr(i), ActiveTraceWaves)

		//	Creates the x-wave used to display the Fit Overview traces
		XWaves[i]=XPSFitOverviewCreateSingleTrace(Traces[i][2], Traces[i][3], GroupV, "XPSFitOverviewXWave"+Num2iStr(i), ActiveTraceWaves)
	endfor
	//	Can I multithread this??

//	MUST HAVE USED XXX FOR A REASON HERE
//	xxx
	
	//	Creates temporary waves to calculate the maximum and minimum values for each wave and hold the offsets to be used in the graph
	Make/FREE/O/N=(NumberOfNewTraces) MinWave, MaxWave, Offsets, MulOffsets
			
	//	Calculates the minimum based on the background wave and the maximum based on the fit wave, if a fit exists
	MultiThread MinWave=WaveMin(YWaves[p])
	MultiThread MaxWave=WaveMax(YWaves[p])
	

//	COMMENTIZED THIS TO BE ABLE TO COMPILE
//	StrSearch
//	NaN problem???
//	NumType(xxx)=0 means normal
	
	//	Finds the status of the Offset and Scale buttons (1 = clicked, 0 = not clicked)
	Variable OffsetStatus=Str2Num(GetUserData(ActiveWindow, "OffsetButton", "Status"))
	Variable ScaleStatus=Str2Num(GetUserData(ActiveWindow, "ScaleButton", "Status"))
	
	//	Calculates the offsets and muloffsets for the waves based on the Offset and Scale buttons. If Waterfall is active the offsets will be overwritten, but the muloffsets are kept
	Variable StatusBit=OffsetStatus*2^0+ScaleStatus*2^1
	switch (StatusBit)
	
		//	Neither Offset nor Scale is active. Scaling is removed from all waves
		case 0:
			FastOP Offsets=0
			FastOP MulOffsets=1
			break

		//	Only Offset is active. All waves are offset by their minimum value
		case 1:
			FastOP Offsets=-1*MinWave
			FastOP MulOffsets=1
			break

		//	Only Scale is active. All waves are divided by their maxium value
		case 2:
			FastOP Offsets=0
			MultiThread MulOffsets=(MaxWave[p]==0) ? (1) : (1/Abs(MaxWave[p]))
			break

		//	Offset and Scale are both active. All waves are scaled between zero and one
		case 3:
			MultiThread MulOffsets=(MaxWave[p]==MinWave[p]) ? (1) : (1/(MaxWave[p]-MinWave[p]))
			MultiThread Offsets=-MinWave*MulOffsets
			break
	endswitch

	//	Adds the traces to the graph one at a time, in reverse order to make the last trace appear on top
	for (i=NumberOfNewTraces-1; i>-1; i+=-1)
		
		//	The waves to append to the graph
		Wave YWave=YWaves[i]
		Wave XWave=XWaves[i]
	
		//	Appends the XY trace to the graph
		AppendToGraph /W=$ActiveWindow YWave vs XWave
		
		//	Finds the group the trace belongs to
		GroupNum=WhichGroup[i]
		
		//	Hides the trace if either the trace or group hidden checkbox is checked
		Hidden=(TraceHidden[i] || GroupHidden[GroupNum])
		
		//	Gives each trace a different colour and updates the hidden status
		TraceName=NameOfWave(YWave)
		ModifyGraph /W=$ActiveWindow rgb($TraceName)=(ColourWave[TraceColour[i]][0], ColourWave[TraceColour[i]][1], ColourWave[TraceColour[i]][2]), hideTrace($TraceName)=Hidden, offset($TraceName)={0, Offsets[i]}, muloffset($TraceName)={1, MulOffsets[i]}
	endfor
	
	//	Changes the traces to lines and circular markers with black outlines around the markers.
	ModifyGraph /W=$ActiveWindow mode=4, marker=19, useMrkStrokeRGB=1
	
	//	The default axis labels
	String YLabel="y-Axis (\\u)"
	String XLabel="x-Axis (\\u)"
	
	//	Variables used to determine the autoscaling of the axes
	Variable YAutoScaleMode=0
	Variable XAutoScaleMode=0
	Variable ReverseYAxis=0			//	Not used, but allows the same function to be used for both axes
	Variable ReverseXAxis=0
	
	//	Finds the first not-hidden trace
	FindValue /V=0 TraceHidden
	Variable TraceNum=V_value
	
	//	Sets the axis labels based on the first not-hidden trace
	if (TraceNum>-1)
	
		//	Finds the displayed coefficients
		String YCoefStr=Traces[TraceNum][0]
		String XCoefStr=Traces[TraceNum][2]
		
		//	Finds the display mode
		ControlInfo /W=$ActiveWindow DisplayTypePopUp
		String DisplayMode=S_Value
	
		//	Finds the axis units corresponding to the selected display mode
		String YAxisUnit=""
		String XAxisUnit=""
		XPSFitOverviewAxisUnit(DisplayMode, YAxisUnit, XAxisUnit)
	
		//	Finds the axis labels and autoscale modes
		XPSFitOverviewAxisLabels(YCoefStr, YLabel, YAutoScaleMode, ReverseYAxis, DisplayMode, YAxisUnit, XAxisUnit)
		XPSFitOverviewAxisLabels(XCoefStr, XLabel, XAutoScaleMode, ReverseXAxis, DisplayMode, YAxisUnit, XAxisUnit)
	endif

	//	Sets the axis labels
	Label /W=$ActiveWindow left YLabel
	Label /W=$ActiveWindow bottom XLabel
	
	//	Autoscales the left axis from zero when displaying peak areas
	SetAxis/W=$ActiveWindow /A=1 /E=(YAutoScaleMode) left

	//	Reverses the bottom axis in XPS mode if peak positions are displayed, and autoscales from zero when displaying peak areas
	if (ReverseXAxis)
		SetAxis/W=$ActiveWindow /R/A=1 /E=(XAutoScaleMode) bottom
	else
		SetAxis/W=$ActiveWindow /A=1 /E=(XAutoScaleMode) bottom
	endif
end



ThreadSafe Static Function XPSFitOverviewAxisLabels(CoefStr, LabelStr, AutoScaleMode, ReverseAxis, DisplayMode, YAxisUnit, XAxisUnit)
//	Sets the axis labels based on the first not-hidden trace
String CoefStr, &LabelStr, DisplayMode, YAxisUnit, XAxisUnit
Variable &AutoScaleMode, &ReverseAxis	//	DisplayHelpTopic "Pass-By-Reference"

	//	Normal autoscale (zero is not special)
	AutoScaleMode=0
	ReverseAxis=0
	
	//	Determines the axis label based on the displayed coefficient
	strswitch(CoefStr)

		//	The linear background intercept is displayed
		case "Intercept":
			LabelStr="Intercept (\\u"+YAxisUnit+")"
			break

		//	The linear background slope is displayed
		case "Slope":
			LabelStr="Slope (\\u"+YAxisUnit+"/"+XAxisUnit+")"
			break
			
		//	The position of the spectrum in the group or data folder is displayed
		case "Wave No.":
			LabelStr="Spectrum Number (\\u)"
			break

		//	The sum of all peak areas in the group or data folder is displayed (autoscaled from zero)
		case "Area Sum":
			LabelStr="Area (\\u"+YAxisUnit+""+XAxisUnit+")"
			AutoScaleMode=3
			break

		//	The XSW exitation energy is displayed
		case "XSW Energy":
			LabelStr="Photon Energy (\\ueV)"
			break

		//	The peak position is displayed
		case "Position":
			LabelStr="Position (\\u"+XAxisUnit+")"
			
			//	Displays binding energies in reverse mode for XPS
			if (CmpStr(DisplayMode, "XPS")==0)
				ReverseAxis=1
			endif
			break

		//	The peak area is displayed (autoscaled from zero)
		case "Area":
			LabelStr="Area (\\u"+YAxisUnit+""+XAxisUnit+")"
			AutoScaleMode=3
			break

		//	The FWHM is displayed
		case "FWHM":
			LabelStr="FWHM (\\u"+XAxisUnit+")"
			break

		//	The GL ratio is displayed
		case "GL Ratio":
			LabelStr="GL Ratio (\\u)"
			break

		//	The first asymmetry coefficient is displayed
		case "Asym1":
			LabelStr="Asymmetry 1 (\\u)"
			break

		//	The first asymmetry coefficient is displayed
		case "Asym2":
			LabelStr="Asymmetry 2 (\\u)"
			break

		//	The Shirley background coefficient is displayed
		case "Shirley Coef":
			
			//	Tests if the selected display format is Infrared
			if (CmpStr(DisplayMode, "IR")==0)
				LabelStr="Shirley Coefficient (\\ucm)"
			else
				LabelStr="Shirley Coefficient (\\u"+XAxisUnit+"\\S-1\\M)"
			endif
			break
		
		//	None of the above is true
		default:
		
			//	Checks if the selected axis is another trace
			Variable SelectedTraceNum=-1
			String OverflowStr=""
			SScanF CoefStr, "Trace %u%s", SelectedTraceNum, OverflowStr
			
			if (V_flag==1)
			
				//	Finds the coefficient selection of the selected trace
				DFREF ProgramFolder=root:Programming
				Wave/T Traces=ProgramFolder:XPSFitOverviewTraces
				String SelectedTraceCoefStr=Traces[SelectedTraceNum][0]
				
				//	Determines the axis label based on the selected trace. This requires the function to be ThreadSafe!
				//	Potential infinite loop danger: Solved by only allowing links to traces with lower trace numbers
				XPSFitOverviewAxisLabels(SelectedTraceCoefStr, LabelStr, AutoScaleMode, ReverseAxis, DisplayMode, YAxisUnit, XAxisUnit)
			endif
	endswitch
end



Static Function XPSFitOverviewAxisUnit(DisplayMode, YAxisUnit, XAxisUnit)
//	Finds the axis units corresponding to the selected display mode
String DisplayMode, &YAxisUnit, &XAxisUnit

	strswitch(DisplayMode)

		//	Auger electon spectroscopy
		case "AES":
			YAxisUnit="Counts"
			XAxisUnit="eV"
			break

		//	Infrared spectroscopy
		case "IR":
			YAxisUnit=""
			XAxisUnit="cm\\S-1\\M"
			break
			
		//	Near-edge X-ray absorption fine structure
		case "NEXAFS":
			YAxisUnit="Counts"
			XAxisUnit="eV"
			break

		//	Scan analog mas spectrum
		case "QMS":
			YAxisUnit="A"
			XAxisUnit="amu"
			break

		//	X-ray photoelectron spectroscopy
		case "XPS":
			YAxisUnit="Counts"
			XAxisUnit="eV"
			break
			
		//	If none of the above
		default:
			YAxisUnit=""
			XAxisUnit=""
	endswitch
end



Static Function XPSFitOverviewGiveColour()
//	Gives the new trace the first unused colour

	//	First colour used as default
	Variable ColourValue=0

	//	The waves determining the colours of the selected traces
	DFREF ProgramFolder=root:Programming
	Wave/W/U ColourWave=ProgramFolder:XPSFitOverviewColours
	Wave/B/U TraceColour=ProgramFolder:XPSFitOverviewTraceColour
	
	if (NumPnts(TraceColour)>0)

		//	Finds the number of different colours in ColourWave
		Variable NumberOfColours=DimSize(ColourWave, 0)
	
		//	Creates a histogram wave to hold information about how often a colour is choosen
		Make/O/FREE/N=(NumberOfColours) HistogramWave
		Histogram /B=2 TraceColour, HistogramWave

		//	Finds the x location of the minimum y value (the least used colour)
		WaveStats/Q HistogramWave
	
		//	Picks the least used colour
		ColourValue=V_minloc
	endif
	
	//	Returns the least used colour
	Return ColourValue
end



Static Function XPSFitOverviewHideCheckBox(CB_Struct) : CheckBoxControl
//	Toggles the Fit Overview hide checkboxes
STRUCT WMCheckboxAction & CB_Struct

	//	If the state of the checkbox has changed. eventCode 2 = mouse up
	if (CB_Struct.eventCode==2)
	
		//	Ignores further calls to the checkbox procedure until this one has finished
		CB_Struct.blockReentry=1
		
		DFREF ProgramFolder=root:Programming
		
		//	Check if the check box is for a group or trace
		String CheckBoxName=CB_Struct.ctrlname
		if (CmpStr(CheckBoxName[0,16], "HideTraceCheckBox")==0)

			//	Finds the trace number
			Variable TraceNum=Str2Num(CB_Struct.userdata)
		
			//	Updates the hidden wave
			Wave/B/U TraceHidden=ProgramFolder:XPSFitOverviewTraceHidden
			TraceHidden[TraceNum]=(CB_Struct.checked==0)
			
			//	Hides the trace colour if the trace is not displayed
			XPSFitOverviewUpdateColourBox(TraceNum)
		else

			//	Finds the group Number
			Variable GroupNum=Str2Num(CB_Struct.userdata)
		
			//	Updates the hidden wave
			Wave/B/U GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden
			GroupHidden[GroupNum]=(CB_Struct.checked==0)
			
			//	Finds the number of traces in the group
			Wave/B/U WhichGroup=ProgramFolder:XPSFitOverviewWhichGroup
			Extract/O/FREE/INDX WhichGroup, IndexWave, (WhichGroup==GroupNum)
			Variable TracesInGroup=NumPnts(IndexWave)

			//	Updates all traces in the group			
			Variable i=0
			for (i=0; i<TracesInGroup; i+=1)
			
				//	Hides the trace colour if the trace is not displayed
				XPSFitOverviewUpdateColourBox(IndexWave[i])
			endfor
		endif
		
		//	Updates the Fit Overview graph
		XPSFitOverviewUpdateGraph()
	endif
	Return 0
end



Static Function XPSFitOverviewUpdateColourBox(TraceNum)
//	Hides the trace colours if the trace is not displayed
Variable TraceNum

	//	The waves used to construct the Fit Overview controls
	DFREF ProgramFolder=root:Programming
	Wave/B/U GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden
	Wave/B/U WhichGroup=ProgramFolder:XPSFitOverviewWhichGroup
	Wave/B/U TraceHidden=ProgramFolder:XPSFitOverviewTraceHidden
	Wave/B/U TraceColour=ProgramFolder:XPSFitOverviewTraceColour
	Wave/W/U ColourWave=ProgramFolder:XPSFitOverviewColours
	
	String ActiveWindow="XPSFitOverviewWindow#ControlPanel"
	
	Variable Red=0
	Variable Green=0
	Variable Blue=0
	
	if ((TraceHidden[TraceNum]==0) && (GroupHidden[WhichGroup[TraceNum]]==0))
		
		//	If the trace is not hidden the colour determined by TraceColour is used
		Red=ColourWave[TraceColour[TraceNum]][0]
		Green=ColourWave[TraceColour[TraceNum]][1]
		Blue=ColourWave[TraceColour[TraceNum]][2]
	else
		
		//	The default light grey colour
		Red=60000
		Green=60000
		Blue=60000
	endif

	//	The suffix used to make the control names unique
	String Suffix=Num2iStr(TraceNum)
			
	//	Updates the colour of the ColourBox
	ValDisplay $("ColourBox"+Suffix) barBackColor=(Red, Green, Blue), win=$ActiveWindow
end



Static Function XPSFitOverviewSetDF(PU_Struct) : PopupMenuControl
//	Selects the active data folder or group
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Ignores further calls to the popup menu procedure until this one has finished
		PU_Struct.blockReentry=1

		//	Finds the folder or group selection
		String FolderOrGroup=PU_Struct.popStr
		
		//	Finds the group number
		Variable GroupNum=Str2Num(PU_Struct.userdata)
		
		//	Creates the waves to hold the information about the groups and traces to display		
		XPSFitOverviewChangeGroup(FolderOrGroup, GroupNum)

		//	Creates or recreates the controls for the Fit Overview window
		XPSFitOverviewBuildControls()
		
		//	Updates the Fit Overview graph
		XPSFitOverviewUpdateGraph()
		
		//Update popup menu value
	endif
	Return 0
end


Static Function XPSFitOverviewSelectionPop(PU_Struct) : PopupMenuControl
//	Updates the Fit Overview selection waves and the graph
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Ignores further calls to the popup menu procedure until this one has finished
		PU_Struct.blockReentry=1
	
		//	The wave with the trace selections
		DFREF ProgramFolder=root:Programming
		Wave/T Traces=ProgramFolder:XPSFitOverviewTraces

		//	Finds the trace number
		Variable TraceNum=Str2Num(PU_Struct.userdata)
		
		//	Finds the selection
		String Selection=PU_Struct.popStr
		if ((CmpStr(Selection, " ")==0) && (CmpStr(Selection, "- select coef -")==0) && (CmpStr(Selection, "- select peak -")==0))
			Selection=""
		endif

		//	Updates the selection wave
		String PopupMenuName=PU_Struct.ctrlname
		strswitch(PopupMenuName[0,9])
			case "YCoefPopUp":
				Traces[TraceNum][0]=Selection
				break
				
			case "YPeakPopUp":
				Traces[TraceNum][1]=Selection
				break

			case "XCoefPopUp":
				Traces[TraceNum][2]=Selection
				break

			case "XPeakPopUp":
				Traces[TraceNum][3]=Selection
				break
		endswitch
		
		//	Updates the Fit Overview graph
		XPSFitOverviewUpdateGraph()
		
		//	If the user scrolls through the items in the list too fast, the selection will change before the update has finished, and the selection will not match the displayed data. This will prevent that
		//	The selection defaults to the first item ("- select coef -" or "- select peak -") if the popmatch value is invalid
		PopupMenu $PU_Struct.ctrlName popmatch=Selection, win=$PU_Struct.win
		
		//	Doesn't work properly.....????
	endif
	
	//	Needed, but not used
	Return 0
end



Static Function/WAVE XPSFitOverviewCreateSingleTrace(CoefStr, PeakStr, GroupV, TargetName, TargetFolder)
//	Creates the x and y-waves used to display the Fit Overview traces
String CoefStr, PeakStr
Wave GroupV
String TargetName
DFREF TargetFolder

	//	Creates an empty place holder wave
	Duplicate/O/R=[][0] GroupV, TargetFolder:$TargetName/WAVE=TargetWave
	FastOP TargetWave=(NaN)
	
	//	Finds the coefficient to display as trace
	strswitch(CoefStr)

		//	Displays the linear background intercept as trace
		case "Intercept":
			Duplicate/O/R=[][1] GroupV, TargetWave
			break

		//	Displays the linear background slope as trace
		case "Slope":
			Duplicate/O/R=[][3] GroupV, TargetWave
			break
			
		//	Displays the position of the spectrum in the group or data folder as trace
		case "Wave No.":
			TargetWave=p
			break

		//	Displays the sum of all peak areas in the group or data folder as trace
		case "Area Sum":
			Duplicate/O/R=[][DimSize(GroupV, 1)-2] GroupV, TargetWave
			break

		//	Displays the XSW exitation energy as trace (saved as the last coefficient in the saved group)
		case "XSW Energy":
			Duplicate/O/R=[][DimSize(GroupV, 1)-1] GroupV, TargetWave
			break

		//	If none of  the above is true
		default:
		
			//	Checks if the selected axis is another trace
			Variable SelectedTraceNum=-1
			String OverflowStr=""
			SScanF CoefStr, "Trace %u%s", SelectedTraceNum, OverflowStr
			if (V_flag==1)
				Wave/Z SelectedTrace=TargetFolder:$("XPSFitOverviewYWave"+Num2iStr(SelectedTraceNum))
				if (WaveExists(SelectedTrace))
					Duplicate/O SelectedTrace, TargetWave
				endif

			//	If another trace has not been selected
			else

				//	Checks that a valid peak has been selected
				Variable PeakNum=Str2Num(PeakStr)
				if (NumType(PeakNum)==0)

					//	Finds the coefficient to display as axis
					strswitch(CoefStr)

						//	Displays the position of peak PeakNum as trace
						case "Position":
							Duplicate/O/R=[][5+14*PeakNum] GroupV, TargetWave
							break

						//	Displays the area of peak PeakNum as trace
						case "Area":
							Duplicate/O/R=[][7+14*PeakNum] GroupV, TargetWave
							break

						//	Displays the FWHM of peak PeakNum as trace
						case "FWHM":
							Duplicate/O/R=[][9+14*PeakNum] GroupV, TargetWave
							break

						//	Displays the GL ratio of peak PeakNum as trace
						case "GL Ratio":
							Duplicate/O/R=[][11+14*PeakNum] GroupV, TargetWave
							break

						//	Displays the Asym1 coefficient of peak PeakNum as trace
						case "Asym1":
							Duplicate/O/R=[][13+14*PeakNum] GroupV, TargetWave
							break

						//	Displays the Asym2 coefficient of peak PeakNum as trace
						case "Asym2":
							Duplicate/O/R=[][15+14*PeakNum] GroupV, TargetWave
							break

						//	Displays the Shirley background coefficient of peak PeakNum as trace
						case "Shirley Coef":
							Duplicate/O/R=[][17+14*PeakNum] GroupV, TargetWave
							break
					endswitch
				endif
			endif
	endswitch
	Return TargetWave
end



Static Function XPSFitOverviewExtractGroupWaves(FolderOrGroup, GroupNum)
//	Collects the saved constraints and fit coefficients for the selected group or data folder and saves the numeric and text information in two separate waves
String FolderOrGroup
Variable GroupNum

	Variable MaxNumberOfPeaks=-1
	String ListOfWaves=""

	//	The waves used to construct the Fit Overview controls
	DFREF ProgramFolder=root:Programming
	DFREF ActiveGroupWaves=ProgramFolder:FitOverviewActiveGroupWaves
	
	//	Creates empty group waves. These waves will be overwritten if the group selection is valid.
	Make/O/N=0 ActiveGroupWaves:$("XPSFitOverviewGroupV"+Num2iStr(GroupNum))/WAVE=GroupV
	Make/O/T/N=0 ActiveGroupWaves:$("XPSFitOverviewGroupS"+Num2iStr(GroupNum))/WAVE=GroupS
	
	//	Tests if FolderOrGroup refers to a data folder
	DFREF ActiveFolder=$FolderOrGroup
	if (DataFolderRefStatus(ActiveFolder)!=0)
	
		//	The folder is valid: Returns a list of the waves in the folder
		ListOfWaves=XPSViewListWaves(ActiveFolder, 1)
	else

		//	If the data folder is invalid it is assumed to be the name of a saved group
		//	Returns the data folder for the selected group
		String GroupInfo=XPSViewGetGroupInfo(FolderOrGroup)
		ActiveFolder=XPSViewGroupDataFolder(GroupInfo)
		
		if (DataFolderRefStatus(ActiveFolder)!=0)
		
			//	The group data folder is valid: Creates a semicolon separated list of all waves in the selected group
			ListOfWaves=XPSViewGroupWaveList(GroupInfo)
		
		//	If the data folder is still invalid, the selection is assumed to be a saved fit series in root:Programming:FitOverviewSavedFits
		else

			//	Finds the saved Fit Overview group, if it exists
			DFREF SavedGroupWaves=ProgramFolder:FitOverviewSavedGroupWaves
			Wave/Z SavedGroupV=SavedGroupWaves:$(FolderOrGroup+"_V")		//	Contains variables, e.g. positions, areas, etc...
			Wave/T/Z SavedGroupS=SavedGroupWaves:$(FolderOrGroup+"_S")	//	Contains strings, e.g. constraints
		
			//	Copies the saved fit series into the active fit series
			if (WaveExists(SavedGroupV) && WaveExists(SavedGroupS))
				Duplicate/O SavedGroupV, GroupV
				Duplicate/O/T SavedGroupS, GroupS
			endif
		endif
	endif


	//	The data folder is valid and the group waves are extracted from the saved Lst and Sel waves
	if (DataFolderRefStatus(ActiveFolder)!=0)

		//	Finds the number of waves in the list and creates wave reference waves to hold the references to the saved Lst and Sel coefficient waves
		Variable NumberOfWaves=ItemsInList(ListOfWaves, ";")
		Make/O/FREE/WAVE/N=(NumberOfWaves) ListWaves, SelWaves
		
		//	Creates a temporary wave to holds the information about the number of peaks in each fit
		Make/O/FREE/N=(NumberOfWaves) NumberOfPeaksWave
		FastOP NumberOfPeaksWave=-1
		
		//	The folder containing the folders containing the fits
		DFREF SavedFitsFolder=ActiveFolder:SavedFits
							
		//	Adds the saved Lst and Sel coefficient waves of the waves in the list to the reference waves
		Variable i=0, a=0, b=0
		String WaveNameStr=""
		for (i=0; i<NumberOfWaves; i+=1)
		
			//	Finds the name of the next wave in the list
			b=StrSearch(ListOfWaves, ";", a)
			WaveNameStr=ListOfWaves[a, b-1]
			
			//	The folder containing the fit of the next wave in the list
			DFREF FitFolder=SavedFitsFolder:$WaveNameStr
			if (DataFolderRefStatus(FitFolder)!=0)
			
				//	Finds the Lst and Sel waves with the saved fit
				ListWaves[i]=FitFolder:$(WaveNameStr+"_lst")
				SelWaves[i]=FitFolder:$(WaveNameStr+"_sel")
				
				//	Finds the number of peaks in the fit
				NumberOfPeaksWave[i]=XPSFitAssNumberOfPeaks(ListWaves[i])
			else
				ListWaves[i]=$""
				SelWaves[i]=$""
				
				//	-1 indicates no fit exists
				NumberOfPeaksWave[i]=-1
			endif
			a=b+1
		endfor
		
		//	Finds the maximum number of peaks in the fits
		MaxNumberOfPeaks=WaveMax(NumberOfPeaksWave)
		
		if (MaxNumberOfPeaks>-1)
		
			//	Finds the required sizes of the GroupV and GroupS waves. The first data point in GroupV is the number of peaks for that fit and the last data point is the XSW excitation energy, if applicable. The second to last data point is the sum of all peak areas
			//	GroupV includes hold values, hence *14
			Variable SizeV=7+MaxNumberOfPeaks*14
			Variable SizeS=SizeV-3
		
			//	Creates the wave to hold the numeric information of the fits, e.g. positions, areas, etc...
			Make/O/N=(NumberOfWaves, SizeV) ActiveGroupWaves:$("XPSFitOverviewGroupV"+Num2iStr(GroupNum))/WAVE=GroupV
			FastOP GroupV=(NaN)
		
			//	Copies the number of peaks for each fit into the first column of the groupV wave
			ImageTransform/D=NumberOfPeaksWave /G=0 putCol GroupV
		
			//	Creates the text wave to hold the constraints for each fit. These will not be displayed anywhere, but are needed to be able to overwrite the saved fit coefficients with the coefficients from a group
			Make/O/T/N=(NumberOfWaves, SizeS) ActiveGroupWaves:$("XPSFitOverviewGroupS"+Num2iStr(GroupNum))/WAVE=GroupS
			GroupS=""
		
			//	Creates a dummy wave used to make multithreading easier
			Duplicate/FREE NumberOfPeaksWave, MultiThreadWave

			//	Uses the collected list and sel waves to extract and save the fit coefficients in the GroupV and GroupS waves
			//	Assigning values to a text wave is apparently NOT threadsafe. Hence, this task cannot be multithreaded. Any attempt to MultiThread this calculation crashes Igor
			MultiThreadWave[]=XPSFitOverviewExtractFits(p, GroupV, GroupS, ListWaves[p], SelWaves[p], MaxNumberOfPeaks)	//	1300 us for 12 spectra with 4 peaks
		endif
	endif

	//	Returns the number of peaks in the coefficient waves
	Return MaxNumberOfPeaks
end



Static Function XPSFitOverviewMaxNumPeaks(GroupNum)
//	Returns the maximum number of peaks in group GroupNum
Variable GroupNum

	//	Finds the GroupS wave belonging to group GroupNum
	DFREF ProgramFolder=root:Programming
	DFREF ActiveGroupWaves=ProgramFolder:FitOverviewActiveGroupWaves
	Wave/T/Z GroupS=ActiveGroupWaves:$("XPSFitOverviewGroupS"+Num2iStr(GroupNum))
	
	//	Finds the maximum number of peaks in the fit based on the size of the GroupS wave. A value of -1 indicates that no fit exists
	Variable MaxNumberOfPeaks=-1
	if (WaveExists(GroupS))
		Variable n=DimSize(GroupS, 1)
		if (n>=4)
			MaxNumberOfPeaks=(n-4)/14
		endif
	endif
	Return MaxNumberOfPeaks
end



Static Function XPSFitOverviewExtractFits(WaveNum, GroupV, GroupS, ListWave, SelWave, MaxNumberOfPeaks)
//	Uses the collected list and sel waves to extract and save the fit coefficients in the GroupV and GroupS waves
//	Assigning values to a text wave is apparently NOT threadsafe. Hence, this task cannot be multithreaded. Any attempt to MultiThread this calculation crashes Igor
Variable WaveNum, MaxNumberOfPeaks
Wave GroupV
Wave/T GroupS
Wave/T/Z ListWave
Wave/Z SelWave
		
	//	Finds the number of peaks in the fit
	Variable NumberOfPeaks=GroupV[WaveNum][0]
	
	//	Checks if a fit exists
	if (NumberOfPeaks>-1)
		
		//	Copies the linear background fit coefficients
		GroupV[WaveNum][1]=Str2Num(ListWave[1][1])
		GroupV[WaveNum][3]=Str2Num(ListWave[1][5])
			
		//	Copies the linear background hold values
		GroupV[WaveNum][2]=SelWave[1][2][0]
		GroupV[WaveNum][4]=SelWave[1][6][0]

		//	Copies the linear background constraints
		GroupS[WaveNum][0]=ListWave[1][3]
		GroupS[WaveNum][1]=ListWave[1][4]
		GroupS[WaveNum][2]=ListWave[1][7]
		GroupS[WaveNum][3]=ListWave[1][8]

		Variable i=0, a=0, c=0
		for (i=0; i<NumberOfPeaks; i+=1)
		
			//	For faster calculations
			a=14*i
			c=i+4

			//	Copies the peak fit coefficients	(index out of range, wave read)
			GroupV[WaveNum][a+5,a+17;2]=Str2Num(ListWave[c][(q-a)*2-9])
			
			//	Copies the peak hold values
			GroupV[WaveNum][a+6,a+18;2]=SelWave[c][(q-a)*2-10][0]
				
			//	Copies the peak constraints
			GroupS[WaveNum][a+4,a+16;2]=ListWave[c][(q-a)*2-5]
			GroupS[WaveNum][a+5,a+17;2]=ListWave[c][(q-a)*2-6]
		endfor
	endif
	
	//	Finds the size of the GroupV wave
	Variable SizeV=DimSize(GroupV, 1)
	
	//	Copies the sum of all peak areas, if any peaks exist
	if (NumberOfPeaks>0)
		GroupV[WaveNum][SizeV-2]=Str2Num(ListWave[NumberOfPeaks+6][5])
	else
		GroupV[WaveNum][SizeV-2]=NaN
	endif

	//	Copies the XSW excitation energy
	GroupV[WaveNum][SizeV-1]=NaN
	
	//	Returns 0 to the dummy wave used for multithreading
	Return 0
end



Static Function XPSFitOverviewCreateColourWave()
//	Creates the wave used to define the colours of the Fit Overview traces.
DFREF ProgramFolder=root:Programming

	Make/W/U/O/N=(9, 3) ProgramFolder:XPSFitOverviewColours /WAVE=Colours
	
	//	The colours of the traces in the Fit Overview graph
	Colours[0][0]=65535;	Colours[0][1]=0;		Colours[0][2]=0		//	Red
	Colours[1][0]=0;		Colours[1][1]=0;		Colours[1][2]=65535	//	Blue
	Colours[2][0]=0;		Colours[2][1]=0;		Colours[2][2]=0		//	Black
	Colours[3][0]=0;		Colours[3][1]=52428;	Colours[3][2]=0		//	Green
	Colours[4][0]=39321;	Colours[4][1]=26214;	Colours[4][2]=13107	//	Brown
	Colours[5][0]=0;		Colours[5][1]=39321;	Colours[5][2]=52428	//	Teal
	Colours[6][0]=65535;	Colours[6][1]=26214;	Colours[6][2]=0		//	Orange
	Colours[7][0]=54998;	Colours[7][1]=0;		Colours[7][2]=37779	//	Purple
	Colours[8][0]=60535;	Colours[8][1]=60535;	Colours[8][2]=0		//	Yellow
end



Function XPSFitOverviewHookFunc(s)
//	The hook function for the Fit Overview window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1
			
			//	The waves used to construct the Fit Overview controls
			DFREF ProgramFolder=root:Programming
			Wave/T Groups=ProgramFolder:XPSFitOverviewGroups
			Wave/T Traces=ProgramFolder:XPSFitOverviewTraces
			Wave/B/U GroupHidden=ProgramFolder:XPSFitOverviewGroupHidden
			Wave/B/U WhichGroup=ProgramFolder:XPSFitOverviewWhichGroup
			Wave/B/U TraceHidden=ProgramFolder:XPSFitOverviewTraceHidden
			Wave/B/U TraceColour=ProgramFolder:XPSFitOverviewTraceColour
			Wave/W/U Colours=ProgramFolder:XPSFitOverviewColours
		
			//	Kills the waves
			KillWaves/Z Groups, Traces, GroupHidden, WhichGroup, TraceHidden, TraceColour, Colours

			//	Kills the waves displayed in the Fit Overview window, but delays the execution until the window has been killed
			Execute/P/Q "KillWaves/Z root:Programming:XPSFitOverviewEmptyWave"
			Execute/P/Q "KillDataFolder/Z root:Programming:FitOverviewActiveTraceWaves"
			Execute/P/Q "KillDataFolder/Z root:Programming:FitOverviewActiveGroupWaves"
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function XPSFitOverviewEditWaves(B_Struct) : ButtonControl
//	WORK IN PROGRESS
//	Creates a table with the displayed traces. Useful for copying and pasting into other programs
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

	endif
	Return 0
end





//     -----<<<<<     Load Elettra NEXAFS     >>>>>-----
//	This section contains the functions used by the Load Elettra NEXAFS menu item

Static Function LoadNEXAFS(DataPathList)
//	Loads the raw Elettra KolXPD NEXAFS images
String DataPathList
String FileExtension="Elettra KolXPD NEXAFS Files (*.exp):.exp;All Files (*.*):.*;"

	//	Creates the necessay folders, if they do not already exist
	LoadNEXAFSCreateFolders()
	
	//	Specifies the default data directory any open file dialog will start in
	CreateDataPath(DataPathList)
	
	//	Display an open file dialog, showing only files with the extension .exp. Allows multiple files to be selected
	Open/D/F=FileExtension /R/MULT=1 /P=EccentricXPSDataPath RefNum
	
	if (CmpStr(S_FileName, "")!=0)

		//	Calculates the number of files selected to be loaded
		String AllFileNames=S_FileName
		Variable NumberOfFiles=ItemsInList(AllFileNames, "\r")
		
		//	Creates the temporary data folder to store the loaded NEXAFS images in until they are given permanent names.
		//	Storing them in a subfolder ensures that the images are not accidentally deleted by another function call using KillWavesInFolder(TempFolder)
		DFREF TempFolder=root:Programming:Temp
		NewDataFolder/O TempFolder:LoadNEXAFS
		DFREF TempLoadFolder=TempFolder:LoadNEXAFS
		
		//	Creates two temporary waves used when prompting the user for the permanent names of the loaded waves
		Make/O/T/N=(2, 4) TempLoadFolder:LoadNEXAFSListBoxWave/WAVE=ListBoxWave
		Make/O/B/U/N=(2, 4, 2) TempLoadFolder:LoadNEXAFSListBoxSelWave/WAVE=ListBoxSelWave

		ListBoxWave=""
		FastOP ListBoxSelWave=0

		//	Sets the titles of the three coloumns
		ListBoxWave[1][1]="File Name"
		ListBoxWave[1][2]="Region Name"
		ListBoxWave[1][3]="Select Wave Name (max 25 characters)"
		
		//	Makes the colours affect the background colour
		SetDimLabel 2, 1, backColors, ListBoxSelWave
		
		//	Sets the colour of the titles to dark blue
		ListBoxSelWave[1][1,3][1]=3
		
		//	Loads each file one at a time
		Variable i=0, RegionNumber=0
		for (i=0; i<NumberOfFiles; i+=1)
			LoadNEXAFSSingleFile(StringFromList(i, AllFileNames, "\r"), i, RegionNumber, TempFolder, TempLoadFolder, ListBoxWave, ListBoxSelWave)
		endfor
		
		//	Indicates the loading is finished
		Print("\rDone Loading")
		
		//	Deletes all temporary waves
		DFREF TempFolder=root:Programming:Temp
		KillWavesInFolder(TempFolder)
		
		//	Prompts the user for permanent names for the loaded waves, if any were loaded
		if (RegionNumber>0)
			LoadNEXAFSPromtNames(ListBoxWave, ListBoxSelWave, TempLoadFolder)
		endif
	endif
end



Static Function LoadNEXAFSCreateFolders()
//	Creates the necessay NEXAFS folders, if they do not already exist

	//	The main NEXAFS folder to hold all the subfolders. This folder should not contain any waves
	NewDataFolder/O root:NEXAFS
	
	//	The folder to hold all the raw two-dimensional spectral X, Y images created by KolXPD
	NewDataFolder/O root:NEXAFS:RawImages
	
	//	The folder to hold the X axis waves for the raw X, Y images
	NewDataFolder/O root:NEXAFS:RawImages:XWaves
	
	//	The folder to hold the Y axis waves for the raw X, Y images
	NewDataFolder/O root:NEXAFS:RawImages:YWaves
	
	//	The folder to hold the photon flux for the raw X, Y images
	NewDataFolder/O root:NEXAFS:RawImages:PhotonFlux
	
	//	The folder to hold all the processed two-dimensional waveform images
	NewDataFolder/O root:NEXAFS:Images
	
	//	The folder to hold the photon flux for the waveform images
	NewDataFolder/O root:NEXAFS:Images:PhotonFlux
	
	//	The folder to hold all the NEXAFS spectra
	NewDataFolder/O root:NEXAFS:Spectra
	
	//	The folder to hold all the permanent waves, used for different bureaucratic purposes, such as listboxes, displaying waves, etc...
	NewDataFolder/O root:Programming
	
	//	The folder to hold all the temporary waves, created during a function call and deleted again at the end of the function call.
	NewDataFolder/O root:Programming:Temp
end



Static Function/S LoadNEXAFSSingleFile(FullFileName, FileNumber, RegionNumber, TempFolder, TempLoadFolder, ListBoxWave, ListBoxSelWave)
//	Loads the Elettra KolXPD NEXAFS file named FullFileName
String FullFileName
Variable FileNumber, &RegionNumber
DFREF TempFolder, TempLoadFolder
Wave/T ListBoxWave
Wave/B/U ListBoxSelWave

	//	Removes the path information from the file name, e.g. 'c:data:xps:20141028xps0005.txt' becomes '20141028xps0005.txt'
	String FileName=ParseFilePath(0, FullFileName, ":", 1, 0)
	
	//	Creates the information string to be printed after a file has been loaded
	String PrintString="\r\rLoading file: "+FileName+"\r"
	
	//	Kills any waves already present in the temporary folder
	KillWavesInfolder(TempFolder)

	//	Saves the active data folder
	DFREF CurrentFolder=GetDataFolderDFR()

	//	Sets the temporary folder as the active folder
	SetDataFolder TempFolder

	//	Loads the file as a one long text wave
	LoadWave /O/N=LoadedTextWave /K=2 /L={0, 0, 0, 0, 1} /Q /J /V={"", "", 0, 0 } FullFileName
	Wave/T LoadedTextWave=TempFolder:LoadedTextWave0
	Variable NumberOfLines=NumPnts(LoadedTextWave)
	
	//	Loads the file as one long numeric wave
	LoadWave /O/N=LoadedNumericWave /K=1 /L={0, 0, NumberOfLines, 0, 1} /Q /J FullFileName
	Wave  LoadedNumericWave=TempFolder:LoadedNumericWave0
	
	//	Creates a temporary wave used to speed up the search
	Make/O/T/FREE/N=0 TempGrepWave
	
	//	Searches for lines starting with [ and ending with ]. This will significantly reduce the number of lines to search during the next Grep, thereby resulting in a much faster search
	//	8 ms
	Grep/INDX/E="(^\\[)(.+)(\\]$)" LoadedTextWave as TempGrepWave
	
	//	Saves the index wave for the first search
	Wave W_Index=TempFolder:W_Index
	Duplicate/O/FREE W_Index, TempIndexWave
	
	//	Searches for lines equal to [Folder], [EndFolder], [XPD] or [EndXPD]
	//	0.6 ms
	Grep/Q/INDX/E="(^\\[Folder\\]$)|(^\\[EndFolder\\])|(^\\[XPD\\])|(^\\[EndXPD\\])" TempGrepWave
	
	//	Saves the index wave, and converts the values to lines in the original text wave
	Wave W_Index=TempFolder:W_Index
	Duplicate/O/FREE W_Index, ItemIndexWave
	ItemIndexWave[]=TempIndexWave[ItemIndexWave[p]]
	
	//	Calculates the number of items in the index wave
	Variable ItemsInFile=NumPnts(ItemIndexWave)
	
	//	Returns the active data folder to the original folder
	SetDataFolder CurrentFolder
	
	//	Loads the root folder of the text file, including all subfolders and spectra
	Variable ActiveItem=0, Error=0, NewFile=1
	LoadNEXAFSLoadFolder(ActiveItem, ItemsInFile, Error, ItemIndexWave, RegionNumber, NewFile, "", LoadedTextWave, LoadedNumericWave, TempLoadFolder, ListBoxWave, ListBoxSelWave, FileName, PrintString)
	
	//	Adds an error message if any errors occurred while loading the file
	if (Error!=0)
		PrintString+="\r\t[An unspecified error occurred while loading the file]"
	endif

	//	Prints the names of the loaded regions
	Print(PrintString)
end



Static Function LoadNEXAFSFindValue(StartSearchAt, TextStartingWith, TextWave)
//	Searches TextWave, starting from position StartSearchAt, looking for elements starting with TextStartingWith
Variable StartSearchAt
String TextStartingWith
Wave/T TextWave

	//	Starts the search at StartSearchAt
	Variable SearchFrom=StartSearchAt
	
	//	Searches until an element starting with TextStartingWith has been found or the end of the wave has been reached
	do
		FindValue /S=(SearchFrom) /TEXT=TextStartingWith /TXOP=1 TextWave
		SearchFrom=V_value+1
	while ((V_value!=-1) && (V_startPos!=0))

	//	Returns the result of the search
	Return V_value
end



Static Function LoadNEXAFSLoadFolder(ActiveItem, ItemsInFile, Error, ItemIndexWave, RegionNumber, NewFile, ActiveFolder, LoadedTextWave, LoadedNumericWave, TempLoadFolder, ListBoxWave, ListBoxSelWave, FileName, PrintString)
//	Loads a folder from the KolXPD .exp NEXAFS text files from Elettra, including all spectra, subfolders and spectra within the subfolders
Variable &ActiveItem, ItemsInFile, &Error	//	& = Pass-by-reference. Changes to these variables are passed back to the parent function
Wave ItemIndexWave
Variable &RegionNumber, &NewFile
String ActiveFolder, FileName, &PrintString
Wave/T LoadedTextWave
Wave LoadedNumericWave
DFREF TempLoadFolder
Wave/T ListBoxWave
Wave/B/U ListBoxSelWave

	//	Checks that the active item is [Folder] and that at least one more item exists in the file
	if ((CmpStr(LoadedTextWave[ItemIndexWave[ActiveItem]], "[Folder]")!=0) || (ActiveItem>=ItemsInFile))
		Error=1
	else
	
		//	The position of [Folder] where the next search will start from
		Variable FilePosition=ItemIndexWave[ActiveItem]
	
		//	The position of the next item will serve as a limit to many of the searches below
		Variable NextItemPosition=ItemIndexWave[ActiveItem+1]

		//	Finds the position of the folder's name in the file
		Variable SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Title=", LoadedTextWave)
	
		if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
			Error=1
		else
	
			FilePosition=SearchResult

			//	Adds the name of the subfolder to that of the parent folder
			if (StrLen(ActiveFolder)==0)
				ActiveFolder=(LoadedTextWave[FilePosition])[6,StrLen(LoadedTextWave[FilePosition])-1]
			else
				ActiveFolder+=":"+(LoadedTextWave[FilePosition])[6,StrLen(LoadedTextWave[FilePosition])-1]
			endif

			//	Finds the position of the number of items (folders and spectra) in the folder
			SearchResult=LoadNEXAFSFindValue(FilePosition+1, "ItemCount=", LoadedTextWave)
				
			if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
				Error=1
			else
		
				FilePosition=SearchResult
		
				//	Stores the number of items (spectra and subfolders) in the folder
				Variable ItemsInFolder=0
				SScanF LoadedTextWave[FilePosition], "ItemCount=%f", ItemsInFolder
		
				if (V_flag!=1)
					Error=1
				else

					//	Loads all items (spectra and subfolders) in the folder
					Variable ItemsLoaded=0, EndFolderReached=0
					do
						
						//	Start loading the next item in the list
						ActiveItem+=1

						//	Sets the position to that of the next item in the folder
						FilePosition=ItemIndexWave[ActiveItem]
							
						//	Checks if the next item is a spectrum [XPD] or a subfolder [Folder]. All other items are ignored
						strswitch(LoadedTextWave[FilePosition])
					
							case "[XPD]":
							
								//	Loads a single 2D spectrum. After loading, FilePosition will be at [EndXPD]
								LoadNEXAFSLoadXPD(ActiveItem, ItemsInFile, Error, ItemIndexWave, RegionNumber, NewFile, ActiveFolder, LoadedTextWave, LoadedNumericWave, TempLoadFolder, ListBoxWave, ListBoxSelWave, FileName, PrintString)
								ItemsLoaded+=1
									
								//	Resets any errors while loading the spectrum, which could be caused, e.g., if the aquisition of a spectrum was cancelled by the user.
								if (Error!=0)
									PrintString+="\r\t[An error occurred while loading a region. This will happen if the acquisition of the region was cancelled by the user]"
									Error=0
								endif
								break

							case "[Folder]":
							
								//	Loads the subfolder, including all subfolders and spectra within. After loading, FilePosition will be at [EndFolder]
								LoadNEXAFSLoadFolder(ActiveItem, ItemsInFile, Error, ItemIndexWave, RegionNumber, NewFile, ActiveFolder, LoadedTextWave, LoadedNumericWave, TempLoadFolder, ListBoxWave, ListBoxSelWave, FileName, PrintString)
								ItemsLoaded+=1
								break

							case "[EndFolder]":
							
								//	Terminates the loop
								EndFolderReached=1
								break
						endswitch
						
					//	Stops loading if the last item in the file has been reached, if an error has occured or if [EndFolder] has been reached
					while ((ActiveItem<ItemsInFile) && (Error==0) && (EndFolderReached==0))
					
					//	Adds an error if the number of items loaded doesn't match the expected number of items in the folder, or if [EndFolder] has not been reached
					if ((ItemsLoaded!=ItemsInFolder) || (EndFolderReached==0))
						Error=1
					endif
				endif
			endif
		endif
	endif
end



Static Function LoadNEXAFSLoadXPD(ActiveItem, ItemsInFile, Error, ItemIndexWave, RegionNumber, NewFile, ActiveFolder, LoadedTextWave, LoadedNumericWave, TempLoadFolder, ListBoxWave, ListBoxSelWave, FileName, PrintString)
//	Loads a single spectrum from the KolXPD .exp NEXAFS text files from Elettra
Variable &ActiveItem, ItemsInFile, &Error	//	& = Pass-by-reference. Changes to these variables are passed back to the parent function
Wave ItemIndexWave
Variable &RegionNumber, &NewFile
String ActiveFolder, FileName, &PrintString
Wave/T LoadedTextWave
Wave LoadedNumericWave
DFREF TempLoadFolder
Wave/T ListBoxWave
Wave/B/U ListBoxSelWave

	//	Checks that the active item is [XPD] and that at least one more item exists in the file
	if ((CmpStr(LoadedTextWave[ItemIndexWave[ActiveItem]], "[XPD]")!=0) || (ActiveItem>=ItemsInFile))
		Error=1
	else
	
		//	The position of [XPD] where the next search will start from
		Variable FilePosition=ItemIndexWave[ActiveItem]
	
		//	Checks that the next item is [EndXPD]
		if (CmpStr(LoadedTextWave[ItemIndexWave[ActiveItem+1]], "[EndXPD]")!=0)
			Error=1
		else

			//	The position of [EndXPD] will serve as a limit to many of the searches below
			Variable NextItemPosition=ItemIndexWave[ActiveItem+1]
	
			//	Finds the position of the spectrum's region name in the file
			Variable SearchResult=LoadNEXAFSFindValue(FilePosition+1, "Title=", LoadedTextWave)
	
			if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
				Error=1
			else
	
				FilePosition=SearchResult

				//	Finds the region name. "Title=C K edge (NI, Q=150)"
				String RegionName=(LoadedTextWave[FilePosition])[6,StrLen(LoadedTextWave[FilePosition])-1]
		
				//	Adds the active folder to the region name
				String FullRegionName=ActiveFolder+":"+RegionName

				//	Finds the photon energy information
				SearchResult=LoadNEXAFSFindValue(FilePosition+1, "azStart=", LoadedTextWave)
		
				if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
					Error=1
				else
		
					FilePosition=SearchResult
			
					//	Finds the approximate photon energy of the first data point
					Variable PhotoEnergyStart=0
					SScanF LoadedTextWave[FilePosition], "azStart=%f", PhotoEnergyStart
	
					if (V_flag!=1)
						Error=1
					else

						//	Finds the approximate photon energy of the last data point
						Variable PhotoEnergyEnd=0
						SScanF LoadedTextWave[FilePosition+1], "azEnd=%f", PhotoEnergyEnd
		
						if (V_flag!=1)
							Error=1
						else

							//	Finds the approximate photon energy step size
							Variable PhotoEnergyStep=0
							SScanF LoadedTextWave[FilePosition+2], "azStep=%f", PhotoEnergyStep
			
							if (V_flag!=1)
								Error=1
							else

								//	Calculates the number of data points along the photon energy axis
								Variable PhotonEnergyNumPoints=Abs((PhotoEnergyEnd-PhotoEnergyStart)/PhotoEnergyStep)+1
						
								if (PhotonEnergyNumPoints<2)
									Error=1
								else
		
									//	Finds the start of the kinetic energy axis data points. The kinetic energy axis data points are not uniformly spaced!!!
									FindValue /S=(FilePosition+1) /TEXT="[XPD XData]" /TXOP=5 LoadedTextWave

									if ((V_Value==-1) || (V_Value>=NextItemPosition))
										Error=1
									else

										FilePosition=V_value
	
										//	Finds the end of the kinetic energy axis data points
										FindValue /S=(FilePosition+1) /TEXT="[Calib XData]" /TXOP=5 LoadedTextWave
						
										if ((V_Value==-1) || (V_Value>=NextItemPosition))
											Error=1
										else

											//	Calculates the number of data points along the kinetic energy axis
											Variable KineticEnergyNumPoints=V_Value-FilePosition-1
									
											if (KineticEnergyNumPoints<2)
												Error=1
											else
									
												//	Copies the kinetic energy axis
												Duplicate/O/FREE/R=[FilePosition+1,V_Value-1] LoadedNumericWave RawKineticEnergyWave
												FilePosition=V_value
								
												//	Creates the waves to hold the photon energy, mesh current, kinetic energy and the 2D NEXAFS spectrum								
												Make/O/FREE/N=(PhotonEnergyNumPoints) RawMeshCurrentWave
												Make/O/FREE/N=(PhotonEnergyNumPoints) RawPhotonEnergyWave
												Make/O/FREE/N=(PhotonEnergyNumPoints, KineticEnergyNumPoints) RawNEXAFSWave
												FastOP RawMeshCurrentWave=(NaN)
												FastOP RawPhotonEnergyWave=(NaN)
												FastOP RawNEXAFSWave=(NaN)
							
												//	Loads the 2D NEXAFS spectrum line-by-line
												Variable SpectrumLine=0, PhotonEnergy=0, MeshCurrent=0, i=0
												for (i=0; (i<PhotonEnergyNumPoints) && (Error==0); i+=1)
										
													//	Finds the position in the file of the next 1D Auger spectrum. DANGER this may search into the next region!!!!!
													SearchResult=LoadNEXAFSFindValue(FilePosition+1, "#Note #", LoadedTextWave)
		
													if ((SearchResult==-1) || (SearchResult>=NextItemPosition))
														Error=1
													else
								
														FilePosition=SearchResult
		
														//	Extracts the mesh current and photon energy from the note string
														SScanF LoadedTextWave[FilePosition], "#Note #%f: E=%f, mc=%f", SpectrumLine, PhotonEnergy, MeshCurrent
								
														if (V_flag!=3)
															Error=1
														else
										
															//	Checks that the spectrum line/note number is correct
															if (SpectrumLine!=i+1)
																Error=1
															else
													
																//	Checks that the function does not attempt to load beyond the [EndXPD] position
																if (FilePosition+KineticEnergyNumPoints>=NextItemPosition)
																	Error=1
																else
				
																	//	Extracts the 1D Auger spectrum for the given photon energy
																	Duplicate/O/FREE/R=[FilePosition+1, FilePosition+KineticEnergyNumPoints] LoadedNumericWave AugerWave
															
																	//	Tests the loaded wave for NaN values
																	WaveStats/Q AugerWave
															
																	if (V_numNans>0)
																		Error=1
																	else
												
																		//	Inserts the 1D Auger spectrum into the 2D NEXAFS spectrum at the given photon energy
																		ImageTransform/D=AugerWave /G=(i) putRow RawNEXAFSWave
															
																		//	Copies the photon energy and mesh current
																		RawPhotonEnergyWave[i]=PhotonEnergy
																		RawMeshCurrentWave[i]=MeshCurrent
	
																		//	Sets the file position at the last data point of the 1D Auger spectrum
																		FilePosition+=KineticEnergyNumPoints
																	endif
																endif
															endif
														endif
													endif
												endfor
											
												if (Error==0)

													//	Test that the line after the last 1D Auger spectrum in the region is [EndXPD] as expected
													if (FilePosition+1!=NextItemPosition)
														Error=1
													else
												
														//	Tests the loaded waves for NaN values
														WaveStats/Q RawMeshCurrentWave

														if (V_numNans>0)
															Error=1
														else

															//	Tests the loaded waves for NaN values
															WaveStats/Q RawPhotonEnergyWave
										
															if (V_numNans>0)
																Error=1
															else
										
																//	Tests the loaded waves for NaN values
																WaveStats/Q RawNEXAFSWave
									
																if (V_numNans>0)
																	Error=1
																else
												
																	//	Increases the number of data points in the photon and kinetic energy axis waves by one. The extra data point is needed, because the waves will later be used as axes for a 2D plot
																	InsertPoints 0, 1, RawPhotonEnergyWave, RawKineticEnergyWave
									
																	//	Creates the temporary loaded waves to be given permanent names later
																	String BaseWaveName="LoadedNEXAFSWave"+Num2iStr(RegionNumber)
																	Duplicate/O RawNEXAFSWave, TempLoadFolder:$BaseWaveName
																	Duplicate/O RawPhotonEnergyWave, TempLoadFolder:$(BaseWaveName+"_X")/WAVE=PhotonEnergyWave
																	Duplicate/O RawKineticEnergyWave, TempLoadFolder:$(BaseWaveName+"_Y")/WAVE=KineticEnergyWave
																	Duplicate/O RawMeshCurrentWave, TempLoadFolder:$(BaseWaveName+"_mc")
										
																	//	Interpolates the photon energy wave into the format needed for 2D axis waves
																	RawPhotonEnergyWave[0]=2*RawPhotonEnergyWave[1]-RawPhotonEnergyWave[2]
																	Rotate -1, PhotonEnergyWave
																	PhotonEnergyWave[PhotonEnergyNumPoints]=2*PhotonEnergyWave[PhotonEnergyNumPoints-1]-PhotonEnergyWave[PhotonEnergyNumPoints-2]
																	FastOP PhotonEnergyWave=PhotonEnergyWave+RawPhotonEnergyWave
																	FastOP PhotonEnergyWave=0.5*PhotonEnergyWave
															
																	//	Interpolates the photon and kinetic energy waves into the format needed for 2D axis waves
																	RawKineticEnergyWave[0]=2*RawKineticEnergyWave[1]-RawKineticEnergyWave[2]
																	Rotate -1, KineticEnergyWave
																	KineticEnergyWave[KineticEnergyNumPoints]=2*KineticEnergyWave[KineticEnergyNumPoints-1]-KineticEnergyWave[KineticEnergyNumPoints-2]
																	FastOP KineticEnergyWave=KineticEnergyWave+RawKineticEnergyWave
																	FastOP KineticEnergyWave=0.5*KineticEnergyWave
															
																	//	Increases the size of the listbox waves by one to make room for the new spectrum
																	Variable ListBoxSize=DimSize(ListBoxWave, 0)
																	InsertPoints /M=0 ListBoxSize, 1, ListBoxWave, ListBoxSelWave
															
																	if (RegionNumber==0)
															
																		//	Sets the colour of the file and region names to light blue and the wave names to yellow for the first region
																		ListBoxSelWave[ListBoxSize][1,2][1]=2
																		ListBoxSelWave[ListBoxSize][3][1]=4
																
																		//	Adds file name to the listbox wave
																		ListBoxWave[ListBoxSize][1]=FileName
																
																		//	The next spectrum will no longer be the first spectrum in the file
																		NewFile=0
																	else
															
																		if (NewFile)
				
																			//	Adds file name to the listbox wave
																			ListBoxWave[ListBoxSize][1]=FileName
																	
																			//	Alternates the colours from those of the previous region
																			ListBoxSelWave[ListBoxSize][1,2][1]=(ListBoxSelWave[ListBoxSize-1][1][1]==2 ? 5 : 2)
																			ListBoxSelWave[ListBoxSize][3][1]=(ListBoxSelWave[ListBoxSize-1][3][1]==1 ? 4 : 1)
																	
																			//	The next spectrum will no longer be the first spectrum in the file	
																			NewFile=0
																		else
			
																			//	The colours of the previous region are used
																			ListBoxSelWave[ListBoxSize][1,2][1]=ListBoxSelWave[ListBoxSize-1][1][1]
																			ListBoxSelWave[ListBoxSize][3][1]=ListBoxSelWave[ListBoxSize-1][3][1]
																		endif
																	endif
															
																	//	Adds folder and region names to the listbox wave
																	ListBoxWave[ListBoxSize][2]=FullRegionName
															
																	//	Adds the folder and region names to the string to be printed
																	PrintString+="\r\t["+FullRegionName+"]"
															
																	//	The number of the next region to be added
																	RegionNumber+=1
																endif
															endif
														endif
													endif
												endif
											endif
										endif
									endif
								endif
							endif
						endif
					endif
				endif
			endif
			
			//	Sets the file position at [EndXPD]
			FilePosition=NextItemPosition
		endif
	endif
end



Static Function LoadNEXAFSPromtNames(ListBoxWave, ListBoxSelWave, TempLoadFolder)
//	Prompts the user for permanent names for the loaded waves
Wave/T ListBoxWave
Wave/B/U ListBoxSelWave
DFREF TempLoadFolder

	//	Creates the wave used to define the colours of the listbox boxes
	Make/W/U/O/N=(3, 6) TempLoadFolder:LoadNEXAFSColours={{0, 0, 0}, {65000, 58000,45000}, {60000, 60000, 65535}, {51000, 51000, 65535}, {65000, 60000,43000}, {62000, 58000, 65535}}	//	Black (Not Used), Orange, Light Blue, Dark Blue, Light Yellow, Light Purple
	Wave/W/U Colours=TempLoadFolder:LoadNEXAFSColours
	MatrixTranspose Colours

	//	Sets the beginning of the region names as starting points for the wave names
	ListBoxWave[2,*][3]=CleanupName((ListBoxWave[p][2])[0, 19], 1)		//	An index being out-of-range is only a problem for a wave, not a string
	//	Makes the wave names editable
	ListBoxSelWave[2,*][3][0]=2

	//	Creates a new window
	Variable n=120/ScreenResolution
	NewPanel /K=1 /N=LoadNEXAFSNameWindow /W=(10*n, 40*n, 940*n, 540*n)
	//	Creates a listbox with file and region names, prompting for wave names for each region
	ListBox LoadNEXAFSListBox, colorWave=Colours, editStyle=0, frame=0, listWave=ListBoxWave, mode=0, noproc, pos={0, 0}, selWave=ListBoxSelWave,  size={930*n,500*n-50}, special={0, 0, 1 }, win=LoadNEXAFSNameWindow
	//	Cancel button. The temporary waves will be deleted
	Button LoadNEXAFSCancelB Disable=0, proc=EccentricXPS#CloseWindow, pos={400*n,500*n-40}, size={75,20}, title="Cancel", win=LoadNEXAFSNameWindow
	//	OK button. The temporary waves will be given permanent names
	Button LoadNEXAFSOKB Disable=0, proc=EccentricXPS#LoadNEXAFSOKB, pos+={20,0}, size={75,20}, title="OK", win=LoadNEXAFSNameWindow
	
	//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
	SetWindow LoadNEXAFSNameWindow hook(KillHook)=LoadNEXAFSNameHookFunc
end



Function LoadNEXAFSNameHookFunc(s)
//	The hook function for the LoadNEXAFSNameWindow. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1
			
			//	Deletes all temporary waves, but delays the execution until the window has been killed
			Execute/P/Q "KillDataFolder/Z root:Programming:Temp:LoadNEXAFS"
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function LoadNEXAFSOKB(B_Struct) : ButtonControl
//	OK button. The temporary waves in root:Programming:Temp will be given permanent names
STRUCT WMButtonAction &B_Struct
DFREF RawNEXAFSFolder=root:NEXAFS:RawImages
DFREF XWaves=RawNEXAFSFolder:XWaves, YWaves=RawNEXAFSFolder:YWaves, PhotonFlux=RawNEXAFSFolder:PhotonFlux
DFREF TempLoadFolder=root:Programming:Temp:LoadNEXAFS
	
	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
		
		//	Finds the number of spectra loaded
		Wave/T ListBoxWave=TempLoadFolder:LoadNEXAFSListBoxWave	
		Variable NumberOfWaves=DimSize(ListBoxWave, 0)-2

		//	Temporary wave used to check for duplicate wave names
		Duplicate/O/T/FREE/R=[][3] ListBoxWave, DuplicateTestWave

		//	Checks if any of the suggested permanent names are illegal or duplicates
		Variable i=0, IllegalName=0, DuplicateName=0
		for (i=0; (i<NumberOfWaves) && (IllegalName==0) && (DuplicateName==0); i+=1)
		
			//	Checks if any of the suggested permanent names are illegal. I use CleanupName instead of CheckName to allow for conflicts with existing waves (which will then be overwritten)
			IllegalName=CmpStr(ListBoxWave[i+2][3], CleanupName(ListBoxWave[i+2][3], 1))
			
			//	Checks all elements below element i+2 for duplicates of element i+2
			if (i<NumberOfWaves-1)
				FindValue /S=(i+3) /TEXT=ListBoxWave[i+2][3] /TXOP=5 DuplicateTestWave
				if (V_value!=-1)
					DuplicateName=1
				endif
			endif
		endfor

		if (IllegalName!=0)
			ErrorMessage("One or more of the suggested wave names are too long or contain illegal characters! ( : ; or \" )")
		else
		
			if (DuplicateName!=0)
				ErrorMessage("Two or more of the suggested wave names are identical!")
			else
	
				String BaseName=""
				for (i=0; i<NumberOfWaves; i+=1)
		
					//	Finds the base name of the temporary waves (file a, region b)
					BaseName="LoadedNEXAFSWave"+Num2Str(i)
					Wave ImageWave=TempLoadFolder:$BaseName
					Wave XWave=TempLoadFolder:$(BaseName+"_X")
					Wave YWave=TempLoadFolder:$(BaseName+"_Y")
					Wave mcWave=TempLoadFolder:$(BaseName+"_mc")
			
					//	Gives the waves their permanent names
					Duplicate/O ImageWave, RawNEXAFSFolder:$ListBoxWave[i+2][3]
					Duplicate/O XWave, XWaves:$ListBoxWave[i+2][3]
					Duplicate/O YWave, YWaves:$ListBoxWave[i+2][3]
					Duplicate/O mcWave, PhotonFlux:$ListBoxWave[i+2][3]
				endfor
		
				//	Closes the listbox window
				DoWindow /K $B_Struct.win

				//	Deletes all temporary waves
				KillDataFolder /Z TempLoadFolder
			endif
		endif
	endif
	Return 0
end





//     -----<<<<<     Convert to Waveform     >>>>>-----
//	This section contains the functions used by the Convert to Waveform menu item

Static Function RawNEXAFS(FolderName, DisplayedName, ForceSelection)
//	Displays the raw 2D NEXAFS images acquired at Elettra
String FolderName, DisplayedName
Variable ForceSelection

	//	Creates a list of existing View NEXAFS windows
	String ListOfWindows=WinList("RawNEXAFSWindow*", ";", "WIN:1")

	if (StrLen(ListOfWindows)==0)
	
		//	Creates a new window
		RawNEXAFSCreateWindow(FolderName, DisplayedName, "RawNEXAFSWindow0", ForceSelection)

	elseif (ForceSelection==0)
		
		//	Brings the first window instance in the list to the front
		String ActiveWindow=StringFromList(0, ListOfWindows, ";")
		DoWindow/F $ActiveWindow
		
		//	Finds the displayed wave
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
		
		//	Updates the View NEXAFS window	
		RawNEXAFSUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
	else

		//	Creates a new window
		RawNEXAFSCreateWindow(FolderName, DisplayedName, RawNEXAFSNewWinName("RawNEXAFSWindow"), ForceSelection)
	endif
end



Static Function/S RawNEXAFSNewWinName(BaseName)
//	Finds the first unused NEXAFS window instance
String BaseName
Variable WindowInstance=-1
String NewWinName=""

	//	Finds the first unused window instance
	do
		WindowInstance+=1
		NewWinName=BaseName+Num2iStr(WindowInstance)
		DoWindow $NewWinName
	while (V_Flag!=0)

	//	Returns the full name of the new window
	Return NewWinName
end



Static Function RawNEXAFSCreateWindow(FolderName, DisplayedName, ActiveWindow, ForceSelection)
//	Creates the window to display the raw NEXAFS images, suffix is used to create multiple copies, so images can be compared
String FolderName, DisplayedName, ActiveWindow
Variable ForceSelection
Variable n=120/ScreenResolution

	//	Finds  the active window instance
	String Suffix=""
	SScanF ActiveWindow, "RawNEXAFSWindow%s", Suffix
	Variable WindowInstance=Str2Num(Suffix)

	//	Creates the waves to hold the displayed NEXAFS Images
	DFREF ProgramFolder=root:Programming
	Make/O ProgramFolder:$("RawNEXAFSDataWave"+Suffix)={{0}}, ProgramFolder:$("RawNEXAFSXWave"+Suffix)={0, 1}, ProgramFolder:$("RawNEXAFSYWave"+Suffix)={0, 1}
	Wave RawDataWave=ProgramFolder:$("RawNEXAFSDataWave"+Suffix)
	Wave XWave=ProgramFolder:$("RawNEXAFSXWave"+Suffix)
	Wave YWave=ProgramFolder:$("RawNEXAFSYWave"+Suffix)
		
	//	Creates the graph to display the NEXAFS images.
	DoWindow /K $ActiveWindow
	Display /W=(5+WindowInstance*50, 40+WindowInstance*10, 5+365*n+WindowInstance*50, 40+300*n+WindowInstance*10) /K=1 /N=$ActiveWindow
	AppendImage/W=$ActiveWindow RawDataWave vs {XWave, YWave}
	ModifyImage/W=$ActiveWindow '' ctab= {*,*,ColdWarm,0}, ctabAutoscale=1, lookup= $""	//	' ' indicates all images in graph
	ModifyGraph /W=$ActiveWindow margin(top)=40*n
	Label /W=$ActiveWindow bottom "Photon Energy (\\ueV)"
	Label /W=$ActiveWindow left "Kinetic Energy (\\ueV)"
	
	//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
	SetWindow $ActiveWindow hook(KillHook)=RawNEXAFSHookFunc
	
	//	Selects the active data folder, listing root: and all it's subfolders with the exception of root:Programming and it's subfolders. To list the default folder (set with SetDataFolder) and all it's subfolders instead with no exceptions, use XPSViewDataFoldersList(:, $"")
	PopupMenu DataFolderPopUp bodyWidth=250, mode=1, pos={210,5}, proc=EccentricXPS#XPSViewDataFolderPopupMenu, value=EccentricXPS#XPSViewDataFoldersList(root:, root:Programming, "SavedFits"), userdata="EccentricXPS#RawNEXAFSUpdateVariables", win=$ActiveWindow, help={"Selects the data folder to look for waves in"}
	//	Popup menu to select the NEXAFS image to display
	PopupMenu DisplayedWavePopUp bodyWidth=200, mode=1, pos={415,5}, proc=EccentricXPS#XPSViewDisplayedWavePopupMenu, value=#("EccentricXPS#RawNEXAFSPopUpWaveList(\""+ActiveWindow+"\")"), userdata="EccentricXPS#RawNEXAFSUpdateVariables", win=$ActiveWindow, help={"Selects the NEXAFS image to display"}
	//	Popup menu to select the colour scheme to be used. Default is ColdWarm
	PopupMenu ColourSchemePopUp bodyWidth=130, mode=3, pos={550,5},  proc=EccentricXPS#XPSViewImageColours, value="Grays;YellowHot;ColdWarm;SpectrumBlack;", win=$ActiveWindow,help={"Selects the colour scheme to use for the image"}
	
	//	Button and variables used to convert the raw NEXAFS image to a waveform with uniform x and y steps
	//	\[0 will store the current font, font size, etc., at the point where you insert it, \F'Symbol'D inserts a delta character, and finally \]0 restores the former font and appearance
	//	Using the symbol font to write the slash / ensure that all the SetVariable boxes have the same height
	SetVariable SetXStart, limits={-inf, inf, 0}, pos={10,33}, size={99, 20}, pos+={0, 2}, title="Xo / \[0\F'Symbol'D\]0X", value=_NUM:0, win=$ActiveWindow, help={"Selects the location of the first point along the X-axis for the new waveform image"}
	SetVariable SetXStep, limits={-inf, inf, 0}, pos+={-8,0}, size={68, 20}, title="\F'Symbol'/", value=_NUM:0, win=$ActiveWindow, help={"Selects the uniform step size along the X-axis for the new waveform image"}
	SetVariable SetYStart, limits={-inf, inf, 0},pos+={0,0}, size={99, 20}, title="Yo / \[0\F'Symbol'D\]0Y", value=_NUM:0, win=$ActiveWindow, help={"Selects the location of the first point along the Y-axis for the new waveform image"}
	SetVariable SetYStep, limits={-inf, inf, 0}, pos+={-8,0}, size={68, 20}, title="\F'Symbol'/", value=_NUM:0, win=$ActiveWindow, help={"Selects the uniform step size along the Y-axis for the new waveform image"}

	//	Converts the image to waveform
	Button WaveformButton, proc=EccentricXPS#RawNEXAFSWaveform, size={70, 20}, pos+={0, -2}, title="Waveform", win=$ActiveWindow, help={"Converts the displayed NEXAFS image into a waveform image"}
	//	Creates another instances of the NEXAFS image window
	Button CloneButton, proc=EccentricXPS#RawNEXAFSClone, size={70, 20}, title="New Win", win=$ActiveWindow, help={"Creates a new independent instance of the Waveform window"}
	//	Opens the help file
	Button HelpButton proc=EccentricXPS#XPSViewHelp, size={70,20}, title="Help", userdata="Convert to Waveform", win=$ActiveWindow, help={"Opens the help file"}
	
	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$ActiveWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$ActiveWindow B
	
	//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or the default data folder is used instead, depending on the value of ForceSelection
	PopupMenu DataFolderPopUp popmatch=FolderName, win=$ActiveWindow
	ControlInfo /W=$ActiveWindow DataFolderPopUp
	if (CmpStr(FolderName, S_Value)!=0)
		if (ForceSelection==1)
			PopupMenu DataFolderPopUp mode=1, popvalue=FolderName, win=$ActiveWindow
		else
			PopupMenu DataFolderPopUp popmatch=GetDataFolder(1), win=$ActiveWindow
		endif
	endif
	
	//	Updates the list of wave selections and select DisplayedName, or if DisplayedName doesn't exist in the list, the first item in the list is selected
	PopupMenu DisplayedWavePopUp mode=1, popmatch=DisplayedName, win=$ActiveWindow

	//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
	if (StrLen(DisplayedName)>0)
		ControlInfo /W=$ActiveWindow DisplayedWavePopUp
		if (CmpStr(DisplayedName, S_Value)!=0)
			if (ForceSelection==1)
				PopupMenu DisplayedWavePopUp mode=1, popvalue=DisplayedName, win=$ActiveWindow
			endif
		endif
	endif

	DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
	Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
	
	//	Updates the raw NEXAFS window
	RawNEXAFSUpdateVariables(ActiveWindow, ActiveFolder, DisplayedWave)
end



Function RawNEXAFSHookFunc(s)
//	The hook function for the Raw NEXAFS window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1
			
			//	Finds the active window
			String ActiveWindow=s.winName
	
			//	Extract the window instance
			String Suffix=""
			SScanF ActiveWindow, "RawNEXAFSWindow%s", Suffix
	
			//	KIlls the waves associated with the window, but delays the execution until the window has been killed
			Execute/P/Q "KillWaves/Z root:Programming:RawNEXAFSDataWave"+Suffix+", root:Programming:RawNEXAFSXWave"+Suffix+", root:Programming:RawNEXAFSYWave"+Suffix
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function RawNEXAFSClone(B_Struct) : ButtonControl
//	Creates another instances of the NEXAFS image window
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
	
		//	Copies the selected folder to the new window
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		DFREF ActiveFolder=$S_Value

		//	Copies the selected wave to the new window
		ControlInfo /W=$B_Struct.win DisplayedWavePopUp
		String ActiveName=S_Value

		//	Creates another Raw NEXAFS window and saves the name of the window created
		String NewWindow=RawNEXAFSNewWinName("RawNEXAFSWindow")
		RawNEXAFSCreateWindow(GetDataFolder(1, ActiveFolder), ActiveName, NewWindow, 1)

		//	Copies the selected colour scheme to the new window
		ControlInfo /W=$B_Struct.win ColourSchemePopUp
		PopupMenu ColourSchemePopUp  popmatch=S_Value, win=$NewWindow
		ModifyImage/W=$NewWindow '' ctab= {*,*,$S_Value,0}	//	' ' indicates all images in graph
		
		//	Copies the selected x and y start and step values to the new window
		ControlInfo /W=$B_Struct.win SetXStart
		SetVariable SetXStart value=_NUM:V_Value, win=$NewWindow
		ControlInfo /W=$B_Struct.win SetXStep
		SetVariable SetXStep value=_NUM:V_Value, win=$NewWindow
		ControlInfo /W=$B_Struct.win SetYStart
		SetVariable SetYStart value=_NUM:V_Value, win=$NewWindow
		ControlInfo /W=$B_Struct.win SetYStep
		SetVariable SetYStep value=_NUM:V_Value, win=$NewWindow
	endif
	Return 0
end



Static Function RawNEXAFSWaveform(B_Struct) : ButtonControl
//	Converts the raw NEXAFS image into the waveform format with uniform x and y steps
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
		
		//	Finds the selected folder
		DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
		DFREF XFolder=ActiveFolder:XWaves, YFolder=ActiveFolder:YWaves
		
		if ((DataFolderRefStatus(ActiveFolder)!=0) && (DataFolderRefStatus(XFolder)!=0) && (DataFolderRefStatus(YFolder)!=0))

			//	Finds the selected wave
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")
			String DisplayedWaveName=NameOfWave(DisplayedWave)
			Wave/Z XWave=XFolder:$DisplayedWaveName
			Wave/Z YWave=YFolder:$DisplayedWaveName
			
			if (WaveExists(DisplayedWave) && WaveExists(XWave) && WaveExists(YWave))
			
				//	If the raw image is located in root:Folder1:RawImages the waveform image will be saved in root:Folder1:Images if such a folder exiists
				//	If not the waveform image will be saved in the same folder as the original
				DFREF ParentFolder=$(GetDataFolder(1, ActiveFolder)+":")
				DFREF DestinationFolder=ParentFolder:Images
				
				String SaveName=DisplayedWaveName

				if ((DataFolderRefStatus(ParentFolder)==0) || (DataFolderRefStatus(DestinationFolder)==0) || (CmpStr("RawImages", GetDataFolder(0, ActiveFolder))!=0))
					DestinationFolder=ActiveFolder
					SaveName+="_"
				endif

				//	Creates a save wave dialog where the OK button will execute XPSViewNEXAFSCreateWaveform
				XPSViewSaveWaveDialog("Save waveform images as", GetDataFolder(1, DestinationFolder), SaveName, "EccentricXPS#RawNEXAFSCreateWaveform", "", B_Struct.win)
			endif
		endif
	endif
	Return 0
end



Static Function RawNEXAFSCreateWaveform(SaveFolder, SaveName, ActiveWindow)
//	Converts the raw NEXAFS image into the waveform format with uniform x and y steps
DFREF SaveFolder
String SaveName, ActiveWindow

	if ((DataFolderRefStatus(SaveFolder)!=0) && (StrLen(SaveName)>0))

		//	Finds the selected folder
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		DFREF XFolder=ActiveFolder:XWaves, YFolder=ActiveFolder:YWaves
		
		if ((DataFolderRefStatus(ActiveFolder)!=0) && (DataFolderRefStatus(XFolder)!=0) && (DataFolderRefStatus(YFolder)!=0))

			//	Finds the selected wave
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
			String DisplayedWaveName=NameOfWave(DisplayedWave)
			Wave/Z XWave=XFolder:$DisplayedWaveName
			Wave/Z YWave=YFolder:$DisplayedWaveName
			
			if (WaveExists(DisplayedWave) && WaveExists(XWave) && WaveExists(YWave))
					
				//	Finds the X and Y start and step sizes
				ControlInfo /W=$ActiveWindow SetXStart
				Variable XStart=V_Value
				ControlInfo /W=$ActiveWindow SetXStep
				Variable XStep=V_Value
				ControlInfo /W=$ActiveWindow SetYStart
				Variable YStart=V_Value
				ControlInfo /W=$ActiveWindow SetYStep
				Variable YStep=V_Value
		
				Variable XSize=NumPnts(XWave)
				Variable YSize=NumPnts(YWave)
		
				//	Calculates the position of the last datapoint in the new wave, as the last datapoint still inside the range of the old wave
				Variable XEnd = XStart+XStep*Trunc(((XWave[XSize-1]+XWave[XSize-2])/2-XStart)/XStep)
				Variable YEnd = YStart+YStep*Trunc(((YWave[YSize-1]+YWave[YSize-2])/2-YStart)/YStep)
		
				//	Converts the image to waveform with the name designated by target name
				ImageInterpolate /DEST=SaveFolder:$SaveName /S={XStart,XStep,XEnd,YStart,YStep,YEnd } /W={XWave, YWave} XYWaves DisplayedWave
				Wave DestWave=SaveFolder:$SaveName
		
				//	Sets the x and y scaling of the new wave
				SetScale/P x, XStart, XStep, "", DestWave
				SetScale/P y, YStart, YStep, "", DestWave
	
				//	Finds the raw photon flux wave
				DFREF RawFluxFolder=ActiveFolder:PhotonFlux
				Wave/Z RawFluxWave=RawFluxFolder:$DisplayedWaveName
			
				if ((DataFolderRefStatus(RawFluxFolder)!=0) && WaveExists(RawFluxWave))
			
					//	Creates the data folder to hold the photon flux
					NewDataFolder/O SaveFolder:PhotonFlux
					DFREF DestFluxFolder=SaveFolder:PhotonFlux
				
					//	Calculates the unevenly stepped mesh current X wave from the 2D NEXAFS image X wave
					//	The image X wave refers to the corners of the rectangles used to draw the image. The actual X value for the first data point in the image is therefore the average of the first two X values.
					Duplicate/FREE/O XWave, XTemp1, XTemp2
					DeletePoints 0, 1, XTemp1
					DeletePoints NumPnts(XTemp2)-1, 1, XTemp2
					FastOP XTemp1=0.5*XTemp1+0.5*XTemp2
		
					//	Creates two new waves to hold the waveform mesh current
					Make/O/N=((XEnd-XStart)/XStep+1) DestFluxFolder:$SaveName /Wave=DestFluxWave
					SetScale/P x, XStart, XStep, "", DestFluxWave
		
					//	Calculates the mesh current at the new, evenly stepped X values
					Interpolate2 /I=3 /Y=DestFluxWave XTemp1, RawFluxWave
				endif
			endif
		endif
	endif
end



Static Function RawNEXAFSUpdateGraph(ActiveWindow, ActiveFolder, ActiveWave)
//	Updates the active raw NEXAFS window
String ActiveWindow
DFREF ActiveFolder
Wave/Z ActiveWave
DFREF ProgramFolder=root:Programming
	
	//	Extract the window instance
	String Suffix=""
	SScanF ActiveWindow, "RawNEXAFSWindow%s", Suffix

	//	Finds the x and y wave folders
	DFREF XFolder=ActiveFolder:XWaves, YFolder=ActiveFolder:YWaves

	//	Finds the x and y waves
	String ActiveName=NameOfWave(ActiveWave)
	Wave/Z XWave=XFolder:$ActiveName
	Wave/Z YWave=YFolder:$ActiveName
		
	if ((DataFolderRefStatus(ActiveFolder)==0) || (DataFolderRefStatus(XFolder)==0) || (DataFolderRefStatus(YFolder)==0) || (WaveExists(ActiveWave)==0) || (WaveExists(XWave)==0) || (WaveExists(YWave)==0))

		//	Overwites the displayed NEXAFS Image with a blank wave
		Make/O ProgramFolder:$("RawNEXAFSDataWave"+Suffix)={{0}}, ProgramFolder:$("RawNEXAFSXWave"+Suffix)={0, 1}, ProgramFolder:$("RawNEXAFSYWave"+Suffix)={0, 1}
	else
			
		//	Copies the selected waves into the displayed waves
		Duplicate/O ActiveWave ProgramFolder:$("RawNEXAFSDataWave"+Suffix)/WAVE=DataWave
		Duplicate/O XWave ProgramFolder:$("RawNEXAFSXWave"+Suffix)
		Duplicate/O YWave ProgramFolder:$("RawNEXAFSYWave"+Suffix)
		
		//	Removes any unit labels that would otherwise result in double axis labels
		SetScale/P x, DimOffset(DataWave, 0), DimDelta(DataWave, 0), "", DataWave
		SetScale/P y, DimOffset(DataWave, 0), DimDelta(DataWave, 1), "", DataWave
	endif
end



Static Function RawNEXAFSUpdateVariables(ActiveWindow, ActiveFolder, ActiveWave)
//	Updates the SetVariable controls in the Convert to Waveform (Raw NEXAFS) window, based on the selected wave
String ActiveWindow
DFREF ActiveFolder
Wave/Z ActiveWave
Variable XStart=0, YStart=0, XStep=0, YStep=0

	//	Finds the x and y wave folders
	DFREF XFolder=ActiveFolder:XWaves, YFolder=ActiveFolder:YWaves		
	
	if ((DataFolderRefStatus(ActiveFolder)!=0) && (DataFolderRefStatus(XFolder)!=0) && (DataFolderRefStatus(YFolder)!=0))
		
		//	Finds the x and y waves
		String ActiveName=NameOfWave(ActiveWave)
		Wave/Z XWave=XFolder:$ActiveName
		Wave/Z YWave=YFolder:$ActiveName

		if (WaveExists(ActiveWave) && WaveExists(XWave) && WaveExists(YWave))

			//	Differentiates the X and Y waves to find the smallest step size
			DFREF TempFolder=root:Programming:Temp
			Differentiate /EP=1 /METH=2 XWave /D=TempFolder:XTempWave
			Differentiate /EP=1 /METH=2 YWave /D=TempFolder:YTempWave
			
			Wave XTempWave=TempFolder:XTempWave
			Wave YTempWave=TempFolder:YTempWave
				
			//	Finds the smallest X step size, to use as the step size for the waveform image
			if (XTempWave[0]>0)
				XStep=WaveMin(XTempWave)
			else
				XStep=WaveMax(XTempWave)
			endif
			
			//	Finds the smallest Y step size, to use as the step size for the waveform image
			if (YTempWave[0]>0)
				YStep=WaveMin(YTempWave)
			else
				YStep=WaveMax(YTempWave)
			endif

			//	Finds the starting X and Y values. The X and Y waves used for display defines the corners of the rectangles used to draw the image.
			//	The actual location of the first data point is therefore the average of the two first values.
			XStart=0.5*(XWave[0]+XWave[1])
			YStart=0.5*(YWave[0]+YWave[1])
			
			//	Kills the temporary waves
			KillWavesInFolder(TempFolder)
		endif
	endif

	//	Updates the start and step values
	SetVariable SetXStart value=_NUM:(XStart), win=$ActiveWindow
	SetVariable SetXStep value=_NUM:(XStep), win=$ActiveWindow
	SetVariable SetYStart value=_NUM:(YStart), win=$ActiveWindow
	SetVariable SetYStep value=_NUM:(YStep), win=$ActiveWindow
	
	//	Updates the Convert to Waveform (Raw NEXAFS) window
	RawNEXAFSUpdateGraph(ActiveWindow, ActiveFolder, ActiveWave)
	
	//	Needed, but not used
	Return 0
end



Static Function/S RawNEXAFSPopUpWaveList(ActiveWindow)
//	Returns a list of all valid raw image waves
String ActiveWindow
String ReturnString="", ImageName="", ImageWaveList=""
Variable NumberOfWaves=0, a=0, b=0, i=0

	//	Finds the selected group or folder in the active window
	ControlInfo /W=$ActiveWindow DataFolderPopUp
	DFREF Folder=$S_Value
	
	if (DataFolderRefStatus(Folder)==0)
	
		//	Returns data folder does not exist, if the data folder does not exist
		ReturnString="Data folder does not exist;"
	else

		//	If both the X and Y folders exist
		DFREF XFolder=Folder:XWaves, YFolder=Folder:YWaves
		if ((DataFolderRefStatus(XFolder)!=0) && (DataFolderRefStatus(YFolder)!=0))
	
			//	Creates an alpha-numerically sorted list of all two-dimensinal, numeric waves in the folder
			ImageWaveList=XPSViewListWaves(Folder, 2)
	
			//	Calculates the number of waves in the image wave list
			NumberOfWaves=ItemsInList(ImageWaveList, ";")

			//	Removes waves without the necessary NEXAFS companion waves
			for (i=0; i<NumberOfWaves; i+=1)
				b=StrSearch(ImageWaveList, ";", a)
				ImageName=ImageWaveList[a, b-1]
				Wave/Z ImageWave=Folder:$ImageName
				Wave/Z XWave=XFolder:$ImageName
				Wave/Z YWave=YFolder:$ImageName
		
				//	Checks that the companion waves XWave and YWave exist and are of the right type and size
				if ((WaveExists(XWave)==1) && (WaveExists(YWave)==1))
					if ((WaveType(XWave, 1)==1) && (WaveType(YWave, 1)==1))
						if ((WaveDims(XWave)==1) && (WaveDims(YWave)==1))
							if ((DimSize(ImageWave, 0)+1==NumPnts(XWave)) && (DimSize(ImageWave, 1)+1==NumPnts(YWave)))
								ReturnString+=ImageName+";"
							endif
						endif
					endif
				endif
				a=b+1
			endfor
		endif
	
		//	If no valid waves exist "No waves in folder" is returned
		if (CmpStr(ReturnString, "")==0)
			ReturnString="No waves in folder;"
		endif
	endif
	Return ReturnString
end





//     -----<<<<<     New NEXAFS Viewer     >>>>>-----
//	This sections contains the functions used by the New NEXAFS Viewer menu item

Static Function NewNEXAFSViewer(FolderName, DisplayedName, ForceSelection)
//	Creates the new display for the kinetic and binding energy images and NEXAFS, Auger and XPS spectra, for the 2D NEXAFS images acquired at Elettra
String FolderName, DisplayedName
Variable ForceSelection

	//	Creates a list of existing New NEXAFS Viewer windows
	String ListOfWindows=WinList("NewNEXAFSViewWindow*", ";", "WIN:64")

	if (StrLen(ListOfWindows)==0)
	
		//	Creates a new window
		NewNEXAFSCreateWindow(FolderName, DisplayedName, "NewNEXAFSViewWindow0", ForceSelection)

	elseif (ForceSelection==0)
		
		//	Brings the first window instance in the list to the front
		String ActiveWindow=StringFromList(0, ListOfWindows, ";")
		DoWindow/F $ActiveWindow
		
		//	Finds the displayed wave
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow+"#ControlPanel", "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow+"#ControlPanel", ActiveFolder, "DisplayedWavePopUp")
		
		//	Updates the New NEXAFS Viewer window
		NewNEXAFSUpdateGraph(ActiveWindow+"#ControlPanel", ActiveFolder, DisplayedWave)
	else

		//	Creates a new window
		NewNEXAFSCreateWindow(FolderName, DisplayedName, RawNEXAFSNewWinName("NewNEXAFSViewWindow"), ForceSelection)
	endif
end



Static Function NewNEXAFSCreateWindow(FolderName, DisplayedName, ParentWindow, ForceSelection)
//	Creates the new display for the kinetic and binding energy images and NEXAFS, Auger and XPS spectra. Suffix is used to create multiple copies, so images can be compared
String FolderName, DisplayedName, ParentWindow
Variable ForceSelection
Variable n=120/ScreenResolution

	//	Creates the necessary folders, if they do not already exist
	//XPSCreateFolders()
	//	WORTH CREATING??
	
	
	
	//	----    Creates the necessary data folders    ----

	//	Creates the folder to hold all the permanent waves, used for different bureaucratic purposes, such as listboxes, displaying waves, etc...
	NewDataFolder/O root:Programming
	DFREF ProgramFolder=root:Programming
	
	//	Creates the folder to hold all the temporary waves, created during a function call and deleted again at the end of the function call.
	NewDataFolder/O root:Programming:Temp

	//	Creates the data folder to hold the test waves
	NewDataFolder/O ProgramFolder:NewNEXAFS
	DFREF NewNEXAFSFolder=ProgramFolder:NewNEXAFS
	//	CHANGE NAME OF FOLDER






	//	----    Creates the waves to display    ----

	//	Finds  the active window instance
	String BaseName="", Suffix=""
	SplitString /E="^([A-Za-z]+)([0-9]+)$" ParentWindow, BaseName, Suffix
	Variable WindowInstance=Str2Num(Suffix)

	//	Creates empty place holder waves for the kinetic energy and binding energy images and NEXAFS, Auger and XPS waves
	Make/O/N=(2,2) NewNEXAFSFolder:$("KineticEnergyImage"+Suffix)/WAVE=KineticEnergyImage, NewNEXAFSFolder:$("BindingEnergyImage"+Suffix)/WAVE=BindingEnergyImage
	Make/O/N=0 NewNEXAFSFolder:$("NEXAFSWave"+Suffix)/WAVE=NEXAFSWave, NewNEXAFSFolder:$("AugerWave"+Suffix)/WAVE=AugerWave, NewNEXAFSFolder:$("XPSWave"+Suffix)/WAVE=XPSWave
	FastOP KineticEnergyImage=(NaN)
	FastOP BindingEnergyImage=(NaN)




	//	----    Creates the parent window to hold the five subwindows    ----

	//	Creates the host window for all the images and spectra
	Variable XPos=5+WindowInstance*80, YPos=53+WindowInstance*20
	Variable Width=2*440, Height=3*220
	
	DoWindow/K $ParentWindow
	NewPanel/K=1 /W=(XPos, YPos, XPos+Width, YPos+Height) /N=$ParentWindow
		
	//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
	SetWindow $ParentWindow hook(KillHook)=NewNEXAFSHookFunc
			
	//	Creates three new horizontal and one new vertical guide to help arrange the subwindows with the images and spectra
	//	The standard frame guide names are FL, FR, FT, and FB for the left, right, top, and bottom frame guides, respectively
	//	The new guides are FLT (frame lower top), FCV (frame center vertical), FUH (frame upper horizontal) and FLH (frame lower horizontal)
	DefineGuide /W=$ParentWindow FLT={FT, 59}, FCV={FL, 0.5, FR}, FUH={FLT, 0.33, FB}, FLH={FLT, 0.67, FB}
			
		//	Different box around active window?? red, blue or black???





	//	----    Creates the five subwindows    ----
		
	//	Selects the framestyle to use with the graphs. Framestyle 7 for graphs and 4 for panels both refer to Text Well
	Variable frameInsetNum=0, graphFrameStyleNum=7, panelFrameStyleNum=4

	//	Creates the kinetic energy image subwindow. The \u#2 in the labels prevents Igor from displaying the units of the wave as specified by SetScale
	String SubWinName="KineticEnergyImage"
	String SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow /N=$SubWinName /FG=(FL, FLT, FCV, FUH)
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	AppendImage /W=$SubWindow KineticEnergyImage
	ModifyImage /W=$SubWindow $NameOfWave(KineticEnergyImage), ctab= {*,*,ColdWarm,0}, ctabAutoscale=1, lookup= $""
	Label /W=$SubWindow bottom "Photon Energy (\\ueV)"
	Label /W=$SubWindow left "Kinetic Energy (\\ueV)"
		
	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$SubWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$SubWindow B

	//	Creates the binding energy image subwindow
	SubWinName="BindingEnergyImage"
	SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow /N=$SubWinName /FG=(FL, FUH, FCV, FB)
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	AppendImage /W=$SubWindow BindingEnergyImage
	ModifyImage /W=$SubWindow $NameOfWave(BindingEnergyImage), ctab= {*,*,ColdWarm,0}, ctabAutoscale=1, lookup= $""
	SetAxis /W=$SubWindow /A/R left
	Label /W=$SubWindow bottom "Photon Energy (\\ueV)"
	Label /W=$SubWindow left "Binding Energy (\\ueV)"

	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$SubWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$SubWindow B

	//	Creates the NEXAFS wave subwindow
	SubWinName="NEXAFSWave"
	SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow /N=$SubWinName /FG=(FCV, FLT, FR, FUH) NEXAFSWave
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	Label /W=$SubWindow bottom "Photon Energy (\\ueV)"
	Label /W=$SubWindow left "Intensity (\\uCounts)"

	//	Creates the Auger wave subwindow
	SubWinName="AugerWave"
	SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow /N=$SubWinName /FG=(FCV, FUH, FR, FLH) AugerWave
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	Label /W=$SubWindow bottom "Kinetic Energy (\\ueV)"
	Label /W=$SubWindow left "Intensity (\\uCounts)"

	//	Creates the XPS wave subwindow
	SubWinName="XPSWave"
	SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow/N=$SubWinName /FG=(FCV, FLH, FR, FB) XPSWave
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	SetAxis /W=$SubWindow /A/R bottom
	Label /W=$SubWindow bottom "Binding Energy (\\ueV)"
	Label /W=$SubWindow left "Intensity (\\uCounts)"
		

	//	NOTE: control for 1st and second order light needed


	//	----    Creates the controls (buttons etc...) for the window    ----
	
	//	Creates a panel as a visual element identical to the other subwindows to place the buttons in
	SubWinName="ControlPanel"
	SubWindow=ParentWindow+"#"+SubWinName
	NewPanel/K=2 /HOST=$ParentWindow /N=$SubWinName /FG=(FL, FT, FR, FLT)
	ModifyPanel /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(panelFrameStyleNum), noEdit=1, cbRGB=(65534, 65534, 65534)

	//	Selects the active data folder, listing root: and all it's subfolders with the exception of root:Programming and it's subfolders. To list the default folder (set with SetDataFolder) and all it's subfolders instead with no exceptions, use XPSViewDataFoldersList(:, $"")
	PopupMenu DataFolderPopUp bodyWidth=250, mode=1, pos={207,7}, proc=EccentricXPS#XPSViewDataFolderPopupMenu, value=EccentricXPS#XPSViewGroupsAndFoldersList(root:, root:Programming, "SavedFits"), userdata="EccentricXPS#NewNEXAFSUpdateGraph", win=$SubWindow, help={"Selects the data folder to look for waves in"}
	//	Popup menu to select the NEXAFS image to display. The complicated value call #(String) indicates that the value ActiveWindow had when the control was created should be used
	PopupMenu DisplayedWavePopUp bodyWidth=200, mode=1, pos={412,7}, proc=EccentricXPS#XPSViewDisplayedWavePopupMenu, value=#("EccentricXPS#XPSViewPopUpWaveList(2, \""+SubWindow+"\", \"DataFolderPopUp\")"), userdata="EccentricXPS#NewNEXAFSUpdateGraph", win=$SubWindow, help={"Selects the NEXAFS image to display"}
	//	Popup menu to select the colour scheme to be used. Default is ColdWarm
	PopupMenu ColourSchemePopUp bodyWidth=130, mode=3, pos={547,7},  proc=EccentricXPS#NewNEXAFSColourPop, value="Grays;YellowHot;ColdWarm;SpectrumBlack;", userdata=ParentWindow+"#KineticEnergyImage;"+ParentWindow+"#BindingEnergyImage;", win=$SubWindow, help={"Selects the colour scheme to use for the image"}

	//	The Save Button userdata string used to create the Save Waves window
	String SaveUserData="Titles=NEXAFS Spectrum;Auger Spectrum;XPS Spectrum;Kinetic Energy Image;Binding Energy Image;"
	SaveUserData+=",SaveWaves=1;0;0;0;0;"
	String SourceFolderName=GetDataFolder(1, NewNEXAFSFolder)
	SaveUserData+=",SourceFolders="+SourceFolderName+";"+SourceFolderName+";"+SourceFolderName+";"+SourceFolderName+";"+SourceFolderName+";"
	SaveUserData+=",SourceWaves=NEXAFSWave"+Suffix+";AugerWave"+Suffix+";XPSWave"+Suffix+";KineticEnergyImage"+Suffix+";BindingEnergyImage"+Suffix+";"
	SaveUserData+=",TargetFolders=root:NEXAFS:Spectra:;root:NEXAFS:Spectra:;root:NEXAFS:Spectra:;root:NEXAFS:Images:;root:NEXAFS:Images:;"
	SaveUserData+=",TargetWaves="+DisplayedName+";"+DisplayedName+"_AES;"+DisplayedName+"_XPS;"+DisplayedName+";"+DisplayedName+"_BE;"

	//	Saves a spectrum or image
	Button SaveButton, proc=EccentricXPS#NewNEXAFSSaveButton, pos+={20, 0}, size={70, 20}, title="Save", userdata=SaveUserData, win=$SubWindow, help={"Saves a spectrum or image"}
	//	Creates a new independent instance of the New NEXAFS Viewer
	Button NewWinButton, proc=EccentricXPS#NewNEXAFSNewWin, size={70, 20}, title="New Win", win=$SubWindow, help={"Creates a new independent instance of the New NEXAFS Viewer"}
	//	Opens the help file
	Button HelpButton proc=EccentricXPS#XPSViewHelp, size={70,20}, title="Help", userdata="New NEXAFS Viewer", win=$SubWindow, help={"Opens the help file"}

	//	Opens the kinetic energy image in NEXAFSCleanup and cleans up photoemission (diagonal) features from the image
	Button CleanupButton proc=EccentricXPS#NewNEXAFSCleanupButton, pos={7, 32}, size={70, 20}, title="Cleanup", win=$SubWindow, help={"Opens the kinetic energy image in NEXAFSCleanup and cleans up photoemission (diagonal) features from the image"}
	
	//	Displays the images and spectra assuming all light is second order light
	CheckBox LightCheckBox pos+={20, 2}, proc=EccentricXPS#NewNEXAFSLightCheckBox, value=0,  title="2nd Order Light ", userdata="EccentricXPS#NewNEXAFSUpdateGraph", win=$SubWindow, help={"Displays the images and spectra assuming all light is second order light"}

	//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or the default data folder is used instead, depending on the value of ForceSelection
	PopupMenu DataFolderPopUp popmatch=FolderName, win=$SubWindow
	ControlInfo /W=$SubWindow DataFolderPopUp
	if (CmpStr(FolderName, S_Value)!=0)
		if (ForceSelection==1)
			PopupMenu DataFolderPopUp mode=1, popvalue=FolderName, win=$SubWindow
		else
			PopupMenu DataFolderPopUp popmatch=GetDataFolder(1), win=$SubWindow
		endif
	endif
	
	//	Updates the list of wave selections and select DisplayedName, or if DisplayedName doesn't exist in the list, the first item in the list is selected
	PopupMenu DisplayedWavePopUp mode=1, popmatch=DisplayedName, win=$SubWindow

	//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
	if (StrLen(DisplayedName)>0)
		ControlInfo /W=$SubWindow DisplayedWavePopUp
		if (CmpStr(DisplayedName, S_Value)!=0)
			if (ForceSelection==1)
				PopupMenu DisplayedWavePopUp mode=1, popvalue=DisplayedName, win=$SubWindow
			endif
		endif
	endif
	
	//	Updates the New NEXAFS Viewer window
	DFREF ActiveFolder=XPSViewGetDataFolder(SubWindow, "DataFolderPopUp")
	Wave/Z DisplayedWave=XPSViewGetDisplayedWave(SubWindow, ActiveFolder, "DisplayedWavePopUp")
	NewNEXAFSUpdateGraph(SubWindow, ActiveFolder, DisplayedWave)
	
	//	Sets the kinetic energy image as the active sub window
	SetActiveSubwindow $ParentWindow#KineticEnergyImage
end



Function NewNEXAFSHookFunc(s)
//	The hook function for the new NEXAFS viewer. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1
			
			//	Finds  the active window instance
			String BaseName="", Suffix=""
			SplitString /E="^([A-Za-z]+)([0-9]+)$" s.winName, BaseName, Suffix

			
			//	KIlls the the kinetic energy and binding energy images and NEXAFS, Auger and XPS waves, but delays the execution until the window has been killed
			Execute/P/Q "KillWaves/Z root:Programming:NewNEXAFS:KineticEnergyImage"+Suffix+", root:Programming:NewNEXAFS:BindingEnergyImage"+Suffix
			Execute/P/Q "KillWaves/Z root:Programming:NewNEXAFS:NEXAFSWave"+Suffix+", root:Programming:NewNEXAFS:AugerWave"+Suffix+", root:Programming:NewNEXAFS:XPSWave"+Suffix
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function NewNEXAFSCleanupButton(B_Struct) : ButtonControl
//	Opens the kinetic energy image in NEXAFSCleanup and cleans up photoemission (diagonal) features from the image
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
		
		//	Finds the selected data folder
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		String FolderName=S_Value

		//	Finds the selected wave
		ControlInfo /W=$B_Struct.win DisplayedWavePopUp
		String DisplayedName=S_Value

		//	Opens the kinetic energy image in the photoemission cleanup window, which allows photoemission (diagonal) features to be removed from the image
		String CleanupWindow=NEXAFSCleanup(FolderName, DisplayedName, 1)
		
		//	Copies the 2nd order light setting
		ControlInfo /W=$B_Struct.win LightCheckBox
		CheckBox LightCheckBox value=(V_Value), win=$CleanupWindow#ControlPanel
		
		//	Updates the cleanup window
		DFREF ActiveFolder=XPSViewGetDataFolder(CleanupWindow+"#ControlPanel", "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(CleanupWindow+"#ControlPanel", ActiveFolder, "DisplayedWavePopUp")
		NEXAFSCleanupUpdateGraph(CleanupWindow+"#ControlPanel", ActiveFolder, DisplayedWave)
	endif
	Return 0
end



Static Function NewNEXAFSColourPop(PU_Struct) : PopupMenuControl
//	Changes the colour schemes of the kinetic energy and binding energy images in the New NEXAFS Viewer window
//	NOTE: change all older versions to this
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed (mouseup = 2)
	if  (PU_Struct.eventCode==2)
	
		//	Finds the number of subwindows to change the colour scheme in
		Variable NumberOfSubWindows=ItemsInList(PU_Struct.userData, ";")
		
		//	Changes the colour schemes one subwindow at a time
		String SubWindow=""
		Variable i=0
		for (i=0; i<NumberOfSubWindows; i+=1)
			SubWindow=StringFromList(i, PU_Struct.userData, ";")
			ModifyImage /W=$SubWindow '', ctab= {*,*,$PU_Struct.PopStr,0}
		endfor
	endif
	Return 0
end



Static Function NewNEXAFSNewWin(B_Struct) : ButtonControl
//	Creates another instances of the New NEXAFS View window
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
		
		//	Finds the parent window of the subwindow hosting the controls
		String OldParentWindow=StringFromList(0, B_Struct.win, "#")

		//	Finds  the base name of the parent window
		String OldBaseName="", OldSuffix=""
		SplitString /E="^([A-Za-z]+)([0-9]+)$" OldParentWindow, OldBaseName, OldSuffix

		//	Finds the selected data folder
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		String FolderName=S_Value

		//	Finds the selected wave
		ControlInfo /W=$B_Struct.win DisplayedWavePopUp
		String DisplayedName=S_Value

		//	Creates another New NEXAFS View window and saves the name of the window created
		String NewParentWindow=RawNEXAFSNewWinName(OldBaseName)
		NewNEXAFSCreateWindow(FolderName, DisplayedName, NewParentWindow, 1)
		
		//	Copies the 2nd order light setting
		ControlInfo /W=$B_Struct.win LightCheckBox
		CheckBox LightCheckBox value=(V_Value), win=$NewParentWindow#ControlPanel

		//	Copies the selected colour schemes to the new windows
		ControlInfo /W=$B_Struct.win ColourSchemePopUp
		PopupMenu ColourSchemePopUp popmatch=S_Value, win=$NewParentWindow#ControlPanel
		ModifyImage/W=$NewParentWindow#KineticEnergyImage '' ctab= {*,*,$S_Value,0}			//	' ' indicates all images in graph
		ModifyImage/W=$NewParentWindow#BindingEnergyImage '' ctab= {*,*,$S_Value,0}		//	' ' indicates all images in graph

		//	Updates the display
		DFREF ActiveFolder=XPSViewGetDataFolder(NewParentWindow+"#ControlPanel", "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(NewParentWindow+"#ControlPanel", ActiveFolder, "DisplayedWavePopUp")
		NewNEXAFSUpdateGraph(NewParentWindow+"#ControlPanel", ActiveFolder, DisplayedWave)
	endif
	Return 0
end



Static Function NewNEXAFSUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
//	Updates the New NEXAFS Viewer window. ActiveFolder is not needed, but is included to allow the update functions for all graphs to share the same FUNCREF
String ActiveWindow
DFREF ActiveFolder
Wave/Z DisplayedWave

	//	Finds the data folder which holds the displayed waves
	DFREF ProgramFolder=root:Programming
	DFREF NewNEXAFSFolder=ProgramFolder:NewNEXAFS
	
	//	Updates the save button's userdata
	String SaveButtonUserData=GetUserData(ActiveWindow, "SaveButton", "")
	String DisplayedName=NameOfWave(DisplayedWave)
	String NewUserData=DisplayedName+";"+DisplayedName+"_AES;"+DisplayedName+"_XPS;"+DisplayedName+";"+DisplayedName+"_BE;"
	SaveButtonUserData=ReplaceStringByKey("TargetWaves", SaveButtonUserData, NewUserData, "=", ",", 1)
	Button SaveButton userData=(SaveButtonUserData), win=$ActiveWindow
	
	//	Finds the parent window of the subwindow hosting the controls
	String ParentWindow=StringFromList(0, ActiveWindow, "#")

	//	Finds  the active window instance
	String BaseName="", Suffix=""
	SplitString /E="^([A-Za-z]+)([0-9]+)$" ParentWindow, BaseName, Suffix
	
	//	Finds the x and y positions of any existing cursors
	Variable NumberOfSubWindows=5
	Make/O/FREE/T SubWindowNames={"KineticEnergyImage", "BindingEnergyImage", "NEXAFSWave", "AugerWave", "XPSWave"}
	Make/O/FREE/N=(NumberOfSubWindows, 2, 2) CursorPositions
	FastOP CursorPositions=(NaN)
	
	String SubWindow=""

	//	Counts through all subwindows and saves the cursor positions
	Variable i=0
	for (i=0; i<NumberOfSubWindows; i+=1)
	
		//	The name of subwindow i
		SubWindow=ParentWindow+"#"+SubWindowNames[i]

		//	Checks if Cursor A is placed on the graph
		if (StrLen(CsrInfo(A, SubWindow))>0)
		
			//	Saves the position of Cursor A
			CursorPositions[i][0][0]=xcsr(A, SubWindow)		//	Returns NaN if cursor does not exists??
			CursorPositions[i][1][0]=vcsr(A, SubWindow)
			
			//	Removes cursor A from the graph
			Cursor /K /W=$SubWindow A
		else
			CursorPositions[i][0][0]=NaN
			CursorPositions[i][1][0]=NaN
		endif
		
		//	Checks if Cursor B is placed on the graph
		if (StrLen(CsrInfo(B, SubWindow))>0)

			//	Saves the position of Cursor B
			CursorPositions[i][0][1]=xcsr(B, SubWindow)
			CursorPositions[i][1][1]=vcsr(B, SubWindow)
			
			//	Removes cursor B from the graph
			Cursor /K /W=$SubWindow B
		else
			CursorPositions[i][0][1]=NaN
			CursorPositions[i][1][1]=NaN
		endif
	endfor
	
	//	Checks that the selected wave exists
	Variable Error=0
	String ErrorStr=""
	if (WaveExists(DisplayedWave)==0)

		//	Returns an error
		Error=1
		ErrorStr="Selected image does not exist!"

		//	Displays an empty place holder wave for the kinetic energy image
		Make/O/N=(2,2) NewNEXAFSFolder:$("KineticEnergyImage"+Suffix)/WAVE=KineticEnergyImage
		FastOP KineticEnergyImage=(NaN)
	else

		//	Displays the selected image
		Duplicate/O DisplayedWave, NewNEXAFSFolder:$("KineticEnergyImage"+Suffix)/WAVE=KineticEnergyImage
		
		//	Checks if the 2nd order light checkbox has been clicked
		ControlInfo /W=$ActiveWindow LightCheckBox
		if (V_Value==1)
			SetScale/P x, 2*DimOffset(DisplayedWave, 0), 2*DimDelta(DisplayedWave, 0), "", KineticEnergyImage
		endif

		//	Finds the number of data points along the photon energy and kinetic energy axes
		Variable PhotonEnergyNumPnts=DimSize(KineticEnergyImage, 0)
		Variable KineticEnergyNumPnts=DimSize(KineticEnergyImage, 1)
		
		//	Checks if the image contains any NaN values
		WaveStats/Q KineticEnergyImage

		//	The image must be at least 2 x 2 points and contain no NaN values
		if ((V_numNaNs!=0) || (PhotonEnergyNumPnts<2) || (KineticEnergyNumPnts<2))
	
			//	Returns an error
			Error=1
			ErrorStr="\JCImage must be at least 2 x 2 points\r\JCand contain no NaN values!"
		else

			//	Finds the step sizes along the photon energy and kinetic energyaxes
			Variable PhotonEnergyDelta=DimDelta(KineticEnergyImage, 0)
			Variable KineticEnergyDelta=DimDelta(KineticEnergyImage, 1)
			
			//	Forces the photon energy to start at the lowest value
			if (PhotonEnergyDelta<0)
				PhotonEnergyDelta=Abs(PhotonEnergyDelta)
				Reverse /DIM=0 KineticEnergyImage
			endif
		
			//	Forces the kinetic energy to start at the lowest value
			if (KineticEnergyDelta<0)
				KineticEnergyDelta=Abs(KineticEnergyDelta)
				Reverse /DIM=1 KineticEnergyImage
			endif
		
			//	Calculates the photon energy step size as an integer times the kinetic energy step size
			//	This is required for a fast calculation of the binding energy image
			Variable PhotonStepInteger=PhotonEnergyDelta/KineticEnergyDelta
		
			//	Checks that the photon energy step size really is an integer times the kinetic energy step size
			if (Abs(Mod(PhotonStepInteger, 1))>1e-5)
		
				//	Returns an error
				Error=1
				ErrorStr="\JCPhoton energy step size must be an integer\r\JCtimes the kinetic energy step size!"
			else
		
				//	The other displayed waves
				Wave BindingEnergyImage=NewNEXAFSFolder:$("BindingEnergyImage"+Suffix)
				Wave NEXAFSWave=NewNEXAFSFolder:$("NEXAFSWave"+Suffix), AugerWave=NewNEXAFSFolder:$("AugerWave"+Suffix), XPSWave=NewNEXAFSFolder:$("XPSWave"+Suffix)

				//	Uses the kinetic energy image to calculate the binding energy image and NEXAFS, Auger and XPS spectra
				//	11.4 ms
				NewNEXAFSCalculateAllSpectra(KineticEnergyImage, BindingEnergyImage, NEXAFSWave, AugerWave, XPSWave, $"")
				
				//	Finds the photon energy and kinetic energy range of the image
				Variable PhotonEnergyLow=DimOffset(KineticEnergyImage, 0)
				Variable PhotonEnergyHigh=PhotonEnergyLow+PhotonEnergyDelta*(PhotonEnergyNumPnts-1)
				Variable KineticEnergyLow=DimOffset(KineticEnergyImage, 1)
				Variable KineticEnergyHigh=KineticEnergyLow+KineticEnergyDelta*(KineticEnergyNumPnts-1)
				Variable BindingEnergyHigh=DimOffset(XPSWave, 0)
				Variable BindingEnergyLow=BindingEnergyHigh+DimDelta(XPSWave, 0)*(DimSize(XPSWave, 0)-1)
				
				//	Checks if the old cursor x and y-positions are still valid on the new images
				CursorPositions[0,2][0][]=((limit(CursorPositions[p][q][r], PhotonEnergyLow, PhotonEnergyHigh)==CursorPositions[p][q][r]) ? (limit(CursorPositions[p][q][r], PhotonEnergyLow, PhotonEnergyHigh)) : (NaN))
				CursorPositions[3][0][]=((limit(CursorPositions[p][q][r], KineticEnergyLow, KineticEnergyHigh)==CursorPositions[p][q][r]) ? (limit(CursorPositions[p][q][r], KineticEnergyLow, KineticEnergyHigh)) : (NaN))
				CursorPositions[4][0][]=((limit(CursorPositions[p][q][r], BindingEnergyLow, BindingEnergyHigh)==CursorPositions[p][q][r]) ? (limit(CursorPositions[p][q][r], BindingEnergyLow, BindingEnergyHigh)) : (NaN))
				CursorPositions[0][1][]=limit(CursorPositions[p][q][r], KineticEnergyLow, KineticEnergyHigh)
				CursorPositions[1][1][]=limit(CursorPositions[p][q][r], BindingEnergyLow, BindingEnergyHigh)
				
				//	Places the cursors back on the graph
				for (i=0; i<NumberOfSubWindows; i+=1)
				
					//	The name of subwindow i
					SubWindow=ParentWindow+"#"+SubWindowNames[i]
					
					//	Checks if the Cursor A existed on the graph before and is still valid
					if (NumType(CursorPositions[i][0][0])==0)
					
						//	Places Cursor A back on the graph
						if (i>1)
							Cursor/W=$SubWindow A $(SubWindowNames[i]+Suffix), CursorPositions[i][0][0]
						else
							Cursor/I /W=$SubWindow A $(SubWindowNames[i]+Suffix), CursorPositions[i][0][0], CursorPositions[i][1][0]
						endif
					endif

					//	Checks if the Cursor B existed on the graph before
					if (NumType(CursorPositions[i][0][1])==0)
					
						//	Places Cursor B back on the graph
						if (i>1)
							Cursor/W=$SubWindow B $(SubWindowNames[i]+Suffix), CursorPositions[i][0][1]
						else
							Cursor/I /W=$SubWindow B $(SubWindowNames[i]+Suffix), CursorPositions[i][0][1], CursorPositions[i][1][1]
						endif
					endif
				endfor				
			endif
		endif
	endif

	//	If an error occurred empty waves are displayed
	if (Error!=0)

		//	Displays empty place holder waves for the binding energy image and NEXAFS, Auger and XPS waves
		Make/O/N=(2,2) NewNEXAFSFolder:$("BindingEnergyImage"+Suffix)/WAVE=BindingEnergyImage
		Make/O/N=0 NewNEXAFSFolder:$("NEXAFSWave"+Suffix)/WAVE=NEXAFSWave, NewNEXAFSFolder:$("AugerWave"+Suffix)/WAVE=AugerWave, NewNEXAFSFolder:$("XPSWave"+Suffix)/WAVE=XPSWave
		FastOP BindingEnergyImage=(NaN)
		
		//	Creates an error message box
		TextBox /W=$(ParentWindow+"#KineticEnergyImage") /K /N=ErrorBox
		TextBox /W=$(ParentWindow+"#KineticEnergyImage") /A=MC /N=ErrorBox ErrorStr
	else

		//	Removes the error message box if one exists
		TextBox /W=$(ParentWindow+"#KineticEnergyImage") /K /N=ErrorBox
	endif
end



Static Function NewNEXAFSCalculateAllSpectra(KineticEnergyImage, BindingEnergyImage, NEXAFSWave, AugerWave, XPSWave, IndexWave)
//	Uses the kinetic energy image to calculate the binding energy image and NEXAFS, Auger and XPS spectra
//	NOTE: The scaling of the kinetic energy source image MUST be from low-to-high photon energy and kinetic energy
//	NOTE: The photon energy step size MUST be an integer times the kinetic energy step size
Wave/Z KineticEnergyImage, BindingEnergyImage, NEXAFSWave, AugerWave, XPSWave, IndexWave

	Variable Error=0

	//	Checks that the source wave exists
	if (WaveExists(KineticEnergyImage)==0)
		Error=1
	else
	
		//	Finds the number of data points along the photon energy and kinetic energy axes
		Variable PhotonEnergyNumPnts=DimSize(KineticEnergyImage, 0)
		Variable KineticEnergyNumPnts=DimSize(KineticEnergyImage, 1)
		
		//	Finds the step sizes along the photon energy and kinetic energyaxes
		Variable PhotonEnergyDelta=DimDelta(KineticEnergyImage, 0)
		Variable KineticEnergyDelta=DimDelta(KineticEnergyImage, 1)
		
		//	Calculates the photon energy step size as an integer times the kinetic energy step size
		//	This is required for a fast calculation of the binding energy image
		Variable PhotonStepInteger=PhotonEnergyDelta/KineticEnergyDelta
		
		//	Checks that the photon energy step size really is an integer times the kinetic energy step size
		if (Abs(Mod(PhotonStepInteger, 1))>1e-5)
			Error=1
		else
			
			//	Ensures that the photon energy step size really is an integer times the kinetic energy step size
			PhotonStepInteger=Round(PhotonStepInteger)

			//	Finds the offsets of the photon energy and kinetic energy axes
			Variable PhotonEnergyStart=DimOffset(KineticEnergyImage, 0)
			Variable KineticEnergyStart=DimOffset(KineticEnergyImage, 1)
		
			//	Will not calculate the NEXAFS spectrum unless the target wave exists
			if (WaveExists(NEXAFSWave))

				//	Flattens the kinetic energy image, removing the kinetic energy axis, creating a one-dimensional NEXAFS spectrum
				//	0.16 ms
				MatrixOP /NTHR=0/O NEXAFSWave=SumRows(KineticEnergyImage)

				//	Normalizes with the number of kinetic energy data points
				//	0.002 ms
				FastOP NEXAFSWave=(1/KineticEnergyNumPnts)*NEXAFSWave
						
				//	Ensures the NEXAFS wave has the correct scaling
				//	0.002 ms
				SetScale/P x, PhotonEnergyStart, PhotonEnergyDelta, "", NEXAFSWave
			endif
		
		
		
			//	Will not calculate the Auger spectrum unless the target wave exists
			if (WaveExists(AugerWave))

				//	Flattens the kinetic energy image, removing the photon energy axis, creating a one-dimensional Auger spectrum
				//	0.12 ms
				MatrixOP /NTHR=0/O AugerWave=(SumCols(KineticEnergyImage))^t

				//	Normalizes with the number of photon energy data points
				//	0.0015 ms
				FastOP AugerWave=(1/PhotonEnergyNumPnts)*AugerWave

				//	Ensures the Auger wave has the correct scaling
				//	0.0015 ms
				SetScale/P x, KineticEnergyStart, KineticEnergyDelta, "", AugerWave
			endif
				
				

			//	Will not calculate the binding energy image unless the target wave exists
			if (WaveExists(BindingEnergyImage))

				//	Finds the number of data points needed along the binding energy axis
				Variable BindingEnergyNumPnts=PhotonStepInteger*(PhotonEnergyNumPnts-1)+KineticEnergyNumPnts

				//	Sets the step size along the binding energy axis
				Variable BindingEnergyDelta=-Abs(KineticEnergyDelta)										//	Is negative

				//	Finds the offset of the binding energy axis
				Variable PhotonEnergyEnd=PhotonEnergyStart+PhotonEnergyDelta*(PhotonEnergyNumPnts-1)
				Variable BindingEnergyStart=PhotonEnergyEnd-KineticEnergyStart								//	Starts at highest value

				//	Converts the kinetic energy image to a one-dimensional wave as required for the MatrixOP waveMap operation
				//	0.25 ms
				Variable KineticEnergyImageNumPnts=NumPnts(KineticEnergyImage)
				Duplicate/O/FREE KineticEnergyImage, KineticEnergy1DImage
				Redimension /N=(KineticEnergyImageNumPnts+2) KineticEnergy1DImage
				Rotate 1, KineticEnergy1DImage
		
				//	Sets the first and last data point to NaN. This will make all references to out-of-bounds indexes during the MatrixOP waveMap operation equal to NaN
				//	0.0025 ms
				KineticEnergy1DImage[0]=NaN
				KineticEnergy1DImage[KineticEnergyImageNumPnts+1]=NaN



				//	If ithe index wave does not exist, a free placeholder wave is created
				if (WaveExists(IndexWave)==0)
					Make/O/FREE/N=0 IndexWave
				endif
					
				//	If the index wave needed to create the wave map is invalid, it is recreated. This is time consuming, hence the possibility to reuse a previous one.
				if ((DimSize(IndexWave, 0)!=PhotonEnergyNumPnts) && (DimSize(IndexWave, 1)!=BindingEnergyNumPnts))

					//	Temporary waves used to create the index wave for the MatrixOP waveMap operation
					//	0.007 ms
					Make/O/FREE/N=(PhotonEnergyNumPnts) XWave
					Make/O/FREE/N=(1, BindingEnergyNumPnts) YWave

					//	Creates the first half of the index wave for the MatrixOP waveMap operation
					//	1.67 ms
					XWave=p
					FastOP XWave=(PhotonStepInteger*PhotonEnergyNumPnts+1)*XWave
					FastOP YWave=1
					MatrixOP/O/FREE/NTHR=0 XImage=(XWave x YWave)	

					//	Creates the second half of the index wave for the MatrixOP waveMap operation
					//	1.8 ms
					FastOP XWave=1
					YWave=q
					FastOP YWave=(PhotonEnergyNumPnts)*YWave
					MatrixOP/O/FREE/NTHR=0 YImage=(XWave x YWave)

					//	Combines the two halves to create the final index wave. Equal to IndexWave[][]=xx*p+xxx*q, but faster				!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
					//	1.0 ms
					Duplicate/O/FREE XImage, IndexWave
					FastOP IndexWave=XImage+YImage

					//	Shifts the index wave values to position the first data point in the kinetic energy image at the correct binding energy position
					//	0.17 ms
					Variable IndexOffset=(BindingEnergyNumPnts-KineticEnergyNumPnts)*PhotonEnergyNumPnts-1
					FastOP IndexWave=IndexWave-(IndexOffset)
				endif

				//	Uses the index wave to calculate the binding energy image from the kinetic energy image
				//	2.2 ms
				MatrixOP/O/NTHR=0 BindingEnergyImage=waveMap(KineticEnergy1DImage, IndexWave)

				//	Ensures the binding energy image has the correct scaling
				//	0.006 ms
				SetScale/P x, PhotonEnergyStart, PhotonEnergyDelta, "", BindingEnergyImage
				SetScale/P y, BindingEnergyStart, BindingEnergyDelta, "", BindingEnergyImage



				//	Will not calculate the XPS spectrum unless the target wave exists
				if (WaveExists(XPSWave))
			
					//	Creates four temporary waves used to calculate the number of photon energy data points for each binding energy in the XPS spectrum
					//	0.027 ms
					Make/O/FREE/N=(BindingEnergyNumPnts) XPSWeightWave, BeforeFirstPointWave, LastPointWave, StepWave
						
					//	Temporary calculation
					//	0.055 ms
					StepWave=trunc(p/PhotonStepInteger)

					//	Calculates the point BEFORE the first photon energy point in the sum
					//	0.018 ms
					FastOP BeforeFirstPointWave=(PhotonEnergyNumPnts-2)-StepWave
					BeforeFirstPointWave[BindingEnergyNumPnts-KineticEnergyNumPnts, *]=-1

					//	Calculates the last photon energy point in the sum
					//	0.016 ms
					FastOP LastPointWave=StepWave
					Reverse LastPointWave
					LastPointWave[0, KineticEnergyNumPnts-1]=PhotonEnergyNumPnts-1

					//	Calculates the number of photon energy points as a function of binding energy
					//	0.0022 ms
					FastOP XPSWeightWave=LastPointWave-BeforeFirstPointWave

					//	Flattens the binding energy image, removing the photon energy axis, creating a one-dimensional XPS spectrum. Also normalizes with the number of photon energy data points for each binding energy in the XPS spectrum
					//	1.45 ms
					MatrixOP /NTHR=0/O XPSWave=((SumCols(ReplaceNaNs(BindingEnergyImage, 0)))^t)/XPSWeightWave
						
					//	Ensures the XPS wave has the correct scaling
					//	0.003 ms
					SetScale/P x, BindingEnergyStart, BindingEnergyDelta, "", XPSWave
				endif
			endif
		endif
	endif
	
	//	If the kinetic energy image is invalid all values in the output waves are set to NaN
	if (Error)
		if (WaveExists(NEXAFSWave))
			FastOP NEXAFSWave=(NaN)
		endif
		if (WaveExists(AugerWave))
			FastOP AugerWave=(NaN)
		endif
		if (WaveExists(BindingEnergyImage))
			FastOP BindingEnergyImage=(NaN)
		endif
		if (WaveExists(IndexWave))
			FastOP IndexWave=(NaN)
		endif
		if (WaveExists(XPSWave))
			FastOP XPSWave=(NaN)
		endif
	endif
end



Function NewNEXAFSLightCheckBox(CB_Struct) : CheckBoxControl
//	Displays the images and spectra assuming all light is second order light
	STRUCT WMCheckboxAction &CB_Struct
	
	//	//	If the checkbox was pressed. eventCode 2 = mouse up
	if (CB_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		CB_Struct.blockReentry=1
		
		//	Finds the displayed folder and wave
		DFREF ActiveFolder=XPSViewGetDataFolder(CB_Struct.win, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(CB_Struct.win, ActiveFolder, "DisplayedWavePopUp")

		//	The default update function
		FUNCREF XPSViewProtoUpdate UpdateFunction=$CB_Struct.userdata

		//	Runs the update function
		UpdateFunction(CB_Struct.win, ActiveFolder, DisplayedWave)
	endif
	Return 0
end



Static Function NewNEXAFSSaveButton(B_Struct) : ButtonControl
//	Create a window where the user can select which of the spectra and images to save
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1

		//	Extracts the information from the button's userdata
		String Titles=StringByKey("Titles", B_Struct.userData, "=", ",", 1)
		String SaveWaves=StringByKey("SaveWaves", B_Struct.userData, "=", ",", 1)
		String SourceFolders=StringByKey("SourceFolders", B_Struct.userData, "=", ",", 1)
		String SourceWaves=StringByKey("SourceWaves", B_Struct.userData, "=", ",", 1)
		String TargetFolders=StringByKey("TargetFolders", B_Struct.userData, "=", ",", 1)
		String TargetWaves=StringByKey("TargetWaves", B_Struct.userData, "=", ",", 1)
		
		//	Finds the number of waves to be saved
		Variable NumberOfWaves=ItemsInList(Titles, ";")
		
		//	Finds the parent window of the subwindow hosting the button
		String ParentWindow=StringFromList(0, B_Struct.win, "#")
		
		//	Calculates the position of the Save Waves window based on the size and position of the parent window
		Variable n=ScreenResolution/72
		GetWindow $ParentWindow wsize
		Variable Width=915
		Variable Height=7+NumberOfWaves*25+40+7
		Variable XPos=Round((V_left+V_right)*n/2-Width/2), YPos=Round((V_top+V_bottom)*n/2-Height/2)

		//	Creates a prompt window
		String SaveWindow="NewNEXAFSSaveWindow"
		DoWindow/K $SaveWindow
		NewPanel/K=1 /W=(XPos, YPos, XPos+Width, YPos+Height) /N=$SaveWindow as "Save Waves"

		//	Variables used in the loop
		String Title=""
		Variable SaveWave=0
		String SourceFolder=""
		String SourceWave=""
		String TargetFolder=""
		String TargetWave=""
		
		//	More variables used in the loop
		String Suffix=""
		String CurrentDataFolder=GetDataFolder(1)
		Variable ControlYPos=0
		Variable DisableControl=0
		
		//	Adds one set of controls per wave
		Variable i=0
		for (i=0; i<NumberOfWaves; i+=1)
		
			//	Finds the information for wave number i
			Title=StringFromList(i, Titles, ";")
			SaveWave=Str2Num(StringFromList(i, SaveWaves, ";"))
			SourceFolder=StringFromList(i, SourceFolders, ";")
			SourceWave=StringFromList(i, SourceWaves, ";")
			TargetFolder=StringFromList(i, TargetFolders, ";")
			TargetWave=StringFromList(i, TargetWaves, ";")
			Suffix=Num2iStr(i)
			DisableControl=(SaveWave==0)*2
			
			//	The y-position of the i set of controls
			ControlYPos=9+25*i
		
			//	Check this box to save the wave
			CheckBox $("SaveCheckBox"+Suffix), pos={20, ControlYPos}, value=(SaveWave), disable=0, title="", proc=EccentricXPS#NewNEXAFSSaveCheckBox, userData=(Num2iStr(NumberOfWaves)), win=$SaveWindow, help={"Check this box to save the wave"}
			//	Displays the title of the source wave
			SetVariable $("TitleDisplay"+Suffix), pos={200, ControlYPos}, noproc, styledText=1, value=_STR:("\JC"+Title), bodyWidth=200, disable=(DisableControl), noedit=1, win=$SaveWindow, help={"Displays the title of the source wave"}
			//	Selects the data folder to save the new wave in
			PopupMenu $("TargetFolderPopup"+Suffix), pos={480, ControlYPos-2}, bodyWidth=250, title="as  ", mode=1, noproc, value=EccentricXPS#XPSViewGroupsAndFoldersList(root:, root:Programming, "SavedFits"), disable=(DisableControl), win=$SaveWindow, help={"Selects the data folder to save the new wave in"}
			//	Enter the name of the new wave to be created
			SetVariable $("SetTargetName"+Suffix), pos={840, ControlYPos}, noproc, value=_STR:(TargetWave), bodyWidth=350, disable=(DisableControl), win=$SaveWindow, help={"Enter the name of the new wave to be created"}
			
			//	Sets the target data folder selection to TargetFolder and reads the value of the control back to check if the selection was valid. If the selection was invalid the default data folder is used instead
			PopupMenu $("TargetFolderPopup"+Suffix) popmatch=TargetFolder, win=$SaveWindow
			ControlInfo /W=$SaveWindow $("TargetFolderPopup"+Suffix)
			if (CmpStr(TargetFolder, S_Value)!=0)
				PopupMenu $("TargetFolderPopup"+Suffix) popmatch=CurrentDataFolder, win=$SaveWindow
			endif
		endfor

		//	The y-position of the Cancel and Save buttons
		ControlYPos+=35
		
		//	The userdata to be passed to the save button
		String SaveButtonUserData="SourceFolders="+SourceFolders+",SourceWaves="+SourceWaves
		
		//	Creates the Save and Cancel buttons
		Button CancelButton proc=EccentricXPS#CloseWindow, pos={Width/2-135,ControlYPos}, size={125, 25}, title="Cancel", win=$SaveWindow
		Button SaveButton proc=EccentricXPS#NewNEXAFSSaveOK, pos={Width/2+10,ControlYPos}, size={125, 25}, title="Save", userdata=(SaveButtonUserData), win=$SaveWindow
	endif
	Return 0
end



Function NewNEXAFSSaveCheckBox(CB_Struct) : CheckBoxControl
//	Disables (greys out) the save window controls for waves that are not checked to be saved
	STRUCT WMCheckboxAction &CB_Struct
	
	//	//	If the checkbox was pressed. eventCode 2 = mouse up
	if (CB_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		CB_Struct.blockReentry=1

		//	Finds the number of waves
		Variable NumberOfWaves=Str2Num(CB_Struct.userData)
	
		Variable DisableControl=0
		String Suffix=""
	
		//	Counts through the controls
		Variable i=0
		for (i=0; i<NumberOfWaves; i+=1)
		
			Suffix=Num2iStr(i)
	
			//	Reads back the checkbox state
			ControlInfo /W=$CB_Struct.win $("SaveCheckBox"+Suffix)
			DisableControl=(V_Value==0)*2

			//	Disables/enables the controls
			SetVariable $("TitleDisplay"+Suffix), disable=(DisableControl), win=$CB_Struct.win
			PopupMenu $("TargetFolderPopup"+Suffix), disable=(DisableControl), win=$CB_Struct.win
			SetVariable $("SetTargetName"+Suffix), disable=(DisableControl), win=$CB_Struct.win
		endfor
	endif
	Return 0
end



Static Function NewNEXAFSSaveOK(B_Struct) : ButtonControl
//	Saves the waves selected in the Save Waves window
STRUCT WMButtonAction &B_Struct
	
	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
		
		//	Extracts the information from the button's userdata
		String SourceFolders=StringByKey("SourceFolders", B_Struct.userData, "=", ",", 1)
		String SourceWaves=StringByKey("SourceWaves", B_Struct.userData, "=", ",", 1)
		
		//	Finds the number of waves to be saved
		Variable NumberOfWaves=ItemsInList(SourceFolders, ";")
		
		//	Temporary waves
		Make/O/FREE/N=(NumberOfWaves) SaveWaves, IllegalNames, DuplicateNames
		Make/O/FREE/T/N=(NumberOfWaves) FullTargetNames="", TargetDataFolderNames="", TargetWaveNames=""
		FastOP SaveWaves=0
		FastOP IllegalNames=0
		FastOP DuplicateNames=0
		
		String Suffix=""
		
		//	Extracts the suggested target wave names
		Variable i=0, a=0
		for (i=0; i<NumberOfWaves; i+=1)
		
			Suffix=Num2iStr(i)
		
			//	Reads back the checkbox state
			ControlInfo /W=$B_Struct.win $("SaveCheckBox"+Suffix)
			SaveWaves[i]=V_Value

			//	Checks if the wave is to be saved
			if (SaveWaves[i]==1)

				//	Finds the target data folder and wave name. The wave name may contain additional data folders which may have to be created
				ControlInfo /W=$B_Struct.win $("TargetFolderPopup"+Suffix)
				FullTargetNames[i]=S_Value
				ControlInfo /W=$B_Struct.win $("SetTargetName"+Suffix)
				FullTargetNames[i]+=S_Value
				
				//	Splits the FullTargetName into a data folder part and a wave name part
				a=StrSearch(FullTargetNames[i], ":", StrLen(FullTargetNames[i])-2, 1)
				if (a>-1)
					TargetDataFolderNames[i]=(FullTargetNames[i])[0, a]
					TargetWaveNames[i]=(FullTargetNames[i])[a+1, StrLen(FullTargetNames[i])-1]
				endif
			endif
		endfor
	
		//	Checks if the suggested target wave names are illegal. I use CleanupName instead of CheckName to allow for conflicts with existing waves (which will then be overwritten)
		IllegalNames[]=((CmpStr(TargetWaveNames[p], CleanupName(TargetWaveNames[p], 1))!=0) && (SaveWaves[p]==1))

		//	Checks if the suggested target wave names are duplicates
		for (i=0; i<NumberOfWaves-1; i+=1)
			FindValue /S=(i+1) /TEXT=FullTargetNames[i] /TXOP=5 FullTargetNames
			if (V_value!=-1)
				if ((SaveWaves[i]==1) && (SaveWaves[V_Value]==1))
					DuplicateNames[i]=1
				endif
			endif
		endfor

		if (WaveMax(IllegalNames)!=0)
			ErrorMessage("One or more of the suggested wave names are too long or contain illegal characters! ( : ; or \" )")
		else
		
			if (WaveMax(DuplicateNames)!=0)
				ErrorMessage("Two or more of the suggested wave names are identical!")
			else

				String TargetDataFolderName=""

				//	xxxx
				Variable DataFolderError=0
				for (i=0; i<NumberOfWaves; i+=1)
	
					//	Checks if the wave is to be saved
					if (SaveWaves[i]==1)
					
						//	Searches for the first subfolder
						a=StrSearch(TargetDataFolderNames[i], ":", 0)
						do
						
							TargetDataFolderName=(TargetDataFolderNames[i])[0, a]
							
							//	Checks if the data folder exist, if not it is created
							DFREF TargetFolder=$TargetDataFolderName
							if (DataFolderRefStatus(TargetFolder)==0)
							
								//	The ending : has to be removed to be able to create the folder...
								NewDataFolder/O $(TargetDataFolderName[0, a-1])
								DFREF TargetFolder=$TargetDataFolderName
							endif

							//	Checks if the data folder was created successfully, if not an error is returned
							if (DataFolderRefStatus(TargetFolder)==0)
								DataFolderError=1								
							endif
							
							//	Searches for the next subfolder
							a=StrSearch(TargetDataFolderNames[i], ":", a+1)

						while ((a!=-1) && (DataFolderError==0))
					
						if (DataFolderError==0)
					
							//	Finds the source wave
							DFREF SourceFolder=$StringFromList(i, SourceFolders, ";")
							Wave SourceWave=SourceFolder:$StringFromList(i, SourceWaves, ";")
				
							//	Saves a copy of the wave
							Duplicate/O SourceWave, TargetFolder:$TargetWaveNames[i]
						endif
					endif
				endfor
				
				if (DataFolderError!=0)
					ErrorMessage("An error occurred while trying to create a data folder!")
				else

					//	Closes the Save window
					DoWindow /K $B_Struct.win
				endif
			endif
		endif
	endif
	Return 0
end	


		
	

//     -----<<<<<     Photoemission Cleanup     >>>>>-----
//	This sections contains the functions used by for cleaning up photoemission features in the 2D NEXAFS spectra

Static Function/S NEXAFSCleanup(FolderName, DisplayedName, ForceSelection)
//	Creates the display for the photoemission cleanup of the 2D NEXAFS images acquired at Elettra
String FolderName, DisplayedName
Variable ForceSelection

	String NewWindow=""

	//	Creates a list of existing photoemission cleanup windows
	String ListOfWindows=WinList("NEXAFSCleanupWindow*", ";", "WIN:64")
	
	if (StrLen(ListOfWindows)==0)
	
		//	Creates a new window
		NewWindow="NEXAFSCleanupWindow0"
		NEXAFSCleanupCreateWindow(FolderName, DisplayedName, NewWindow, ForceSelection)

	elseif (ForceSelection==0)
	
		//	Brings the first window instance in the list to the front
		NewWindow=StringFromList(0, ListOfWindows, ";")
		DoWindow/F $NewWindow
		
		//	Finds the displayed wave
		DFREF ActiveFolder=XPSViewGetDataFolder(NewWindow+"#ControlPanel", "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(NewWindow+"#ControlPanel", ActiveFolder, "DisplayedWavePopUp")
		
		//	Updates the cleanup window
		NEXAFSCleanupUpdateGraph(NewWindow+"#ControlPanel", ActiveFolder, DisplayedWave)
	else
	
		//	Creates a new window
		NewWindow=RawNEXAFSNewWinName("NEXAFSCleanupWindow")
		NEXAFSCleanupCreateWindow(FolderName, DisplayedName, NewWindow, ForceSelection)
	endif
	
	//	Returns the name of the new window created
	Return NewWindow
end



Static Function NEXAFSCleanupCreateWindow(FolderName, DisplayedName, ParentWindow, ForceSelection)
//	Creates the display for the photoemission cleanup of the 2D NEXAFS images acquired at Elettra. Suffix is used to create multiple copies, so images can be compared
String FolderName, DisplayedName, ParentWindow
Variable ForceSelection
Variable n=120/ScreenResolution

	//	Creates the necessary folders, if they do not already exist
	//XPSCreateFolders()
	//	WORTH CREATING??
	
	//	----    Creates the necessary data folders    ----

	//	Creates the folder to hold all the permanent waves, used for different bureaucratic purposes, such as listboxes, displaying waves, etc...
	NewDataFolder/O root:Programming
	DFREF ProgramFolder=root:Programming
	
	//	Creates the folder to hold all the temporary waves, created during a function call and deleted again at the end of the function call.
	NewDataFolder/O root:Programming:Temp

	//	Creates the data folder to hold the test waves
	NewDataFolder/O ProgramFolder:NEXAFSCleanup
	DFREF NEXAFSCleanupFolder=ProgramFolder:NEXAFSCleanup
	//	CHANGE NAME OF FOLDER



	//	----    Creates the waves to display    ----

	//	Finds  the active window instance
	String BaseName="", Suffix=""
	SplitString /E="^([A-Za-z]+)([0-9]+)$" ParentWindow, BaseName, Suffix
	Variable WindowInstance=Str2Num(Suffix)

	//	Creates empty place holder waves for the kinetic energy and binding energy images and NEXAFS, Auger and XPS waves
	Make/O/N=(2,2) NEXAFSCleanupFolder:$("NEXAFSImageResult"+Suffix)/WAVE=NEXAFSImageResult, NEXAFSCleanupFolder:$("NEXAFSImageSim"+Suffix)/WAVE=NEXAFSImageSim
	Make/O/N=(2,2) NEXAFSCleanupFolder:$("XPSImageResult"+Suffix)/WAVE=XPSImageResult, NEXAFSCleanupFolder:$("XPSImageSim"+Suffix)/WAVE=XPSImageSim
	Make/O/N=(2,2) NEXAFSCleanupFolder:$("BindingEnergyImageResult"+Suffix)/WAVE=BindingEnergyImageResult, NEXAFSCleanupFolder:$("OriginalNEXAFSImage"+Suffix)/WAVE=OriginalNEXAFSImage
	Make/O/N=0 NEXAFSCleanupFolder:$("NEXAFSWaveResult"+Suffix)/WAVE=NEXAFSWaveResult, NEXAFSCleanupFolder:$("AugerWaveResult"+Suffix)/WAVE=AugerWaveResult, NEXAFSCleanupFolder:$("XPSWaveResult"+Suffix)/WAVE=XPSWaveResult

	FastOP NEXAFSImageResult=(NaN)
	FastOP NEXAFSImageSim=(NaN)
	FastOP XPSImageResult=(NaN)
	FastOP XPSImageSim=(NaN)
	FastOP BindingEnergyImageResult=(NaN)
	FastOP OriginalNEXAFSImage=(NaN)




	//	----    Creates the parent window to hold the five subwindows    ----

	//	Creates the host window for all the images and spectra
	Variable XPos=500+WindowInstance*80, YPos=100+WindowInstance*20
	Variable Width=2*440, Height=3*220
	
	DoWindow/K $ParentWindow
	NewPanel/K=1 /W=(XPos, YPos, XPos+Width, YPos+Height) /N=$ParentWindow
		
	//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
	SetWindow $ParentWindow hook(KillHook)=NEXAFSCleanupHookFunc
			
	//	Creates three new horizontal and one new vertical guide to help arrange the subwindows with the images and spectra
	//	The standard frame guide names are FL, FR, FT, and FB for the left, right, top, and bottom frame guides, respectively
	//	The new guides are FLT (frame lower top), FCV (frame center vertical), FUH (frame upper horizontal) and FLH (frame lower horizontal)
	DefineGuide /W=$ParentWindow FLT={FT, 59}, FCV={FL, 0.5, FR}, FUH={FLT, 0.33, FB}, FCH={FLT, 0.5, FB}, FLH={FLT, 0.67, FB}




	//	----    Creates the six subwindows    ----
		
	//	Selects the framestyle to use with the graphs. Framestyle 7 for graphs and 4 for panels both refer to Text Well
	Variable frameInsetNum=0, graphFrameStyleNum=7, panelFrameStyleNum=4

	//	Creates the resulting NEXAFS wave subwindow
	String SubWinName="NEXAFSWaveResult"
	String SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow/N=$SubWinName /FG=(FL, FLT, FCV, FUH) NEXAFSWaveResult
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	Label /W=$SubWindow bottom "Photon Energy (\\ueV)"
	Label /W=$SubWindow left "Intensity (\\uCounts)"
		
	//	Creates the resulting NEXAFS image subwindow
	SubWinName="NEXAFSImageResult"
	SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow/N=$SubWinName /FG=(FL, FUH, FCV, FLH)
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	AppendImage /W=$SubWindow NEXAFSImageResult
	ModifyImage /W=$SubWindow $NameOfWave(NEXAFSImageResult), ctab= {*,*,ColdWarm,0}, ctabAutoscale=1, lookup= $""
	Label /W=$SubWindow bottom "Photon Energy (\\ueV)"
	Label /W=$SubWindow left "Kinetic Energy (\\ueV)"
		
	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$SubWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$SubWindow B
		
	//	Creates the simulated NEXAFS image subwindow
	SubWinName="NEXAFSImageSim"
	SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow/N=$SubWinName /FG=(FL, FLH, FCV, FB)
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	AppendImage /W=$SubWindow NEXAFSImageSim
	ModifyImage /W=$SubWindow $NameOfWave(NEXAFSImageSim), ctab= {*,*,ColdWarm,0}, ctabAutoscale=1, lookup= $""
	Label /W=$SubWindow bottom "Photon Energy (\\ueV)"
	Label /W=$SubWindow left "Kinetic Energy (\\ueV)"
		
	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$SubWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$SubWindow B

	//	Creates the resulting XPS wave subwindow
	SubWinName="XPSWaveResult"
	SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow/N=$SubWinName /FG=(FCV, FLT, FR, FUH) XPSWaveResult
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	SetAxis /W=$SubWindow /A/R bottom
	Label /W=$SubWindow bottom "Binding Energy (\\ueV)"
	Label /W=$SubWindow left "Intensity (\\uCounts)"

	//	Creates the resulting XPS image subwindow
	SubWinName="XPSImageResult"
	SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow/N=$SubWinName /FG=(FCV, FUH, FR, FLH)
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	AppendImage /W=$SubWindow XPSImageResult
	ModifyImage /W=$SubWindow $NameOfWave(XPSImageResult), ctab= {*,*,ColdWarm,0}, ctabAutoscale=1, lookup= $""
	Label /W=$SubWindow bottom "Photon Energy (\\ueV)"
	Label /W=$SubWindow left "Kinetic Energy (\\ueV)"
		
	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$SubWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$SubWindow B
		
	//	Creates the simulated XPS image subwindow
	SubWinName="XPSImageSim"
	SubWindow=ParentWindow+"#"+SubWinName
	Display/K=2 /HOST=$ParentWindow/N=$SubWinName /FG=(FCV, FLH, FR, FB)
	ModifyGraph /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(graphFrameStyleNum)
	AppendImage /W=$SubWindow XPSImageSim
	ModifyImage /W=$SubWindow $NameOfWave(XPSImageSim), ctab= {*,*,ColdWarm,0}, ctabAutoscale=1, lookup= $""
	Label /W=$SubWindow bottom "Photon Energy (\\ueV)"
	Label /W=$SubWindow left "Kinetic Energy (\\ueV)"
		
	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$SubWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$SubWindow B




	//	----    Creates the controls (buttons etc...) for the window    ----
	
	//	Creates a panel as a visual element identical to the other subwindows to place the buttons in
	SubWinName="ControlPanel"
	SubWindow=ParentWindow+"#"+SubWinName
	NewPanel/K=2 /HOST=$ParentWindow /N=$SubWinName /FG=(FL, FT, FR, FLT)
	ModifyPanel /W=$SubWindow frameInset=(frameInsetNum), frameStyle=(panelFrameStyleNum), noEdit=1, cbRGB=(65534, 65534, 65534)

	//	Selects the active data folder, listing root: and all it's subfolders with the exception of root:Programming and it's subfolders. To list the default folder (set with SetDataFolder) and all it's subfolders instead with no exceptions, use XPSViewDataFoldersList(:, $"")
	PopupMenu DataFolderPopUp bodyWidth=250, mode=1, pos={207,7}, proc=EccentricXPS#XPSViewDataFolderPopupMenu, value=EccentricXPS#XPSViewGroupsAndFoldersList(root:, root:Programming, "SavedFits"), userdata="EccentricXPS#NEXAFSCleanupUpdateGraph", win=$SubWindow, help={"Selects the data folder to look for waves in"}
	//	Popup menu to select the NEXAFS image to display. The complicated value call #(String) indicates that the value ActiveWindow had when the control was created should be used
	PopupMenu DisplayedWavePopUp bodyWidth=200, mode=1, pos={412,7}, proc=EccentricXPS#XPSViewDisplayedWavePopupMenu, value=#("EccentricXPS#XPSViewPopUpWaveList(2, \""+SubWindow+"\", \"DataFolderPopUp\")"), userdata="EccentricXPS#NEXAFSCleanupUpdateGraph", win=$SubWindow, help={"Selects the NEXAFS image to display"}
	//	Popup menu to select the colour scheme to be used. Default is ColdWarm
	PopupMenu ColourSchemePopUp bodyWidth=130, mode=3, pos={547,7},  proc=EccentricXPS#NewNEXAFSColourPop, value="Grays;YellowHot;ColdWarm;SpectrumBlack;", userdata=ParentWindow+"#NEXAFSImageResult;"+ParentWindow+"#NEXAFSImageSim;"+ParentWindow+"#XPSImageResult;"+ParentWindow+"#XPSImageSim;", win=$SubWindow, help={"Selects the colour scheme to use for the image"}

	//	The Save Button userdata string used to create the Save Waves window
	String SaveUserData="Titles=NEXAFS spectrum;Auger spectrum;XPS Spectrum;Original image;NEXAFS image;XPS image;Simulated NEXAFS image;Simulated XPS image;"
	SaveUserData+=",SaveWaves=1;0;0;0;0;0;0;0;"
	String SourceFolderName=GetDataFolder(1, NEXAFSCleanupFolder)
	SaveUserData+=",SourceFolders="+SourceFolderName+";"+SourceFolderName+";"+SourceFolderName+";"+SourceFolderName+";"+SourceFolderName+";"+SourceFolderName+";"+SourceFolderName+";"+SourceFolderName+";"
	SaveUserData+=",SourceWaves=NEXAFSWaveResult"+Suffix+";AugerWaveResult"+Suffix+";XPSWaveResult"+Suffix+";OriginalNEXAFSImage"+Suffix+";NEXAFSImageResult"+Suffix+";XPSImageResult"+Suffix+";NEXAFSImageSim"+Suffix+";XPSImageSim"+Suffix+";"
	SaveUserData+=",TargetFolders=root:NEXAFS:Spectra:;root:NEXAFS:Spectra:;root:NEXAFS:Spectra:;root:NEXAFS:Images:;root:NEXAFS:Images:;root:NEXAFS:Images:;root:NEXAFS:Images:;root:NEXAFS:Images:;"
	SaveUserData+=",TargetWaves="+DisplayedName+";"+DisplayedName+"_AES;"+DisplayedName+"_XPS;"+DisplayedName+";"+DisplayedName+"_NXF;"+DisplayedName+"_XPS;"+DisplayedName+"_SNX;"+DisplayedName+"_SXP;"

	//	Saves a spectrum or image
	Button SaveButton, proc=EccentricXPS#NewNEXAFSSaveButton, pos+={20, 0}, size={70, 20}, title="Save", userData=SaveUserData, win=$SubWindow, help={"Saves a spectrum or image"}
	//	Creates a new independent instance of the New NEXAFS Viewer
	Button NewWinButton, proc=EccentricXPS#NEXAFSCleanupNewWin, size={70, 20}, title="New Win", win=$SubWindow, help={"Creates a new independent instance of the New NEXAFS Viewer"}
	//	Opens the help file
	Button HelpButton proc=EccentricXPS#XPSViewHelp, size={70,20}, title="Help", userdata="Photoemission Cleanup", win=$SubWindow, help={"Opens the help file"}

	//	Cleans up photoemission (diagonal) features from the NEXAFS image
	Button CleanupButton proc=EccentricXPS#NEXAFSCleanupCleanupButton, pos={7, 32}, size={115, 20}, title="Cleanup", win=$SubWindow, help={"Cleans up photoemission (diagonal) features from the NEXAFS image"}
	//	Reloads the NEXAFS image, undoing any photoemission cleanup which may have been done
	Button ReloadButton Proc=EccentricXPS#NEXAFSCleanupReloadButton, pos+={8, 0}, size={115, 20}, title="Reload", win=$SubWindow, help={"Reloads the NEXAFS image, undoing any photoemission cleanup which may have been done"}
	//	Selects the type of photoemission cleanup to use
	PopupMenu MethodPopUp bodyWidth=335, mode=2, pos={547, 32}, noproc, value="Subtract NEXAFS first;Subtract NEXAFS first and correct for Auger;Subtract XPS first;Subtract XPS first and correct for Auger;", win=$SubWindow, help={"Selects the type of photoemission cleanup to use"}
	//	Sets the number of iterations for the photoemission cleanup
	SetVariable SetNumIterations, limits={0, inf, 1}, noproc, bodyWidth=50, pos={676, 34}, title="Iterations", value=_NUM:30, win=$SubWindow, help={"Sets the number of iterations for the photoemission cleanup"}
	//	Displays the images and spectra assuming all light is second order light
	CheckBox LightCheckBox pos={806, 35}, proc=EccentricXPS#NewNEXAFSLightCheckBox, value=0,  side=1, title="2nd Order Light ", userdata="EccentricXPS#NEXAFSCleanupUpdateGraph", win=$SubWindow, help={"Displays and cleans up the images and spectra assuming all light is second order light"}



	//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or the default data folder is used instead, depending on the value of ForceSelection
	PopupMenu DataFolderPopUp popmatch=FolderName, win=$SubWindow
	ControlInfo /W=$SubWindow DataFolderPopUp
	if (CmpStr(FolderName, S_Value)!=0)
		if (ForceSelection==1)
			PopupMenu DataFolderPopUp mode=1, popvalue=FolderName, win=$SubWindow
		else
			PopupMenu DataFolderPopUp popmatch=GetDataFolder(1), win=$SubWindow
		endif
	endif
	
	//	Updates the list of wave selections and select DisplayedName, or if DisplayedName doesn't exist in the list, the first item in the list is selected
	PopupMenu DisplayedWavePopUp mode=1, popmatch=DisplayedName, win=$SubWindow

	//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
	if (StrLen(DisplayedName)>0)
		ControlInfo /W=$SubWindow DisplayedWavePopUp
		if (CmpStr(DisplayedName, S_Value)!=0)
			if (ForceSelection==1)
				PopupMenu DisplayedWavePopUp mode=1, popvalue=DisplayedName, win=$SubWindow
			endif
		endif
	endif
	
	//	Sets the resulting NEXAFS image as the active sub window
	SetActiveSubwindow $ParentWindow#NEXAFSImageResult

	//	Updates the photoemission cleanup window
	DFREF ActiveFolder=XPSViewGetDataFolder(SubWindow, "DataFolderPopUp")
	Wave/Z DisplayedWave=XPSViewGetDisplayedWave(SubWindow, ActiveFolder, "DisplayedWavePopUp")
	NEXAFSCleanupUpdateGraph(SubWindow, ActiveFolder, DisplayedWave)
end



Function NEXAFSCleanupHookFunc(s)
//	The hook function for the photoemission cleanup window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1
			
			//	Finds  the active window instance
			String BaseName="", Suffix=""
			SplitString /E="^([A-Za-z]+)([0-9]+)$" s.winName, BaseName, Suffix

			//	KIlls the the resulting and simulated NEXAFS and XPS images and resulting NEXAFS, Auger and XPS waves, but delays the execution until the window has been killed
			Execute/P/Q "KillWaves/Z root:Programming:NEXAFSCleanup:NEXAFSImageResult"+Suffix+", root:Programming:NEXAFSCleanup:NEXAFSImageSim"+Suffix
			Execute/P/Q "KillWaves/Z root:Programming:NEXAFSCleanup:XPSImageResult"+Suffix+", root:Programming:NEXAFSCleanup:XPSImageSim"+Suffix
			Execute/P/Q "KillWaves/Z root:Programming:NEXAFSCleanup:BindingEnergyImageResult"+Suffix+", root:Programming:NEXAFSCleanup:OriginalNEXAFSImage"+Suffix
			Execute/P/Q "KillWaves/Z root:Programming:NEXAFSCleanup:NEXAFSWaveResult"+Suffix+", root:Programming:NEXAFSCleanup:AugerWaveResult"+Suffix+", root:Programming:NEXAFSCleanup:XPSWaveResult"+Suffix
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function NEXAFSCleanupReloadButton(B_Struct) : ButtonControl
//	Reloads the NEXAFS image, undoing any photoemission cleanup which may have been done
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
		
		//	Updates the photoemission cleanup window
		DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")
		NEXAFSCleanupUpdateGraph(B_Struct.win, ActiveFolder, DisplayedWave)
	endif
	Return 0
end



Static Function NEXAFSCleanupCleanupButton(B_Struct) : ButtonControl
//	Opens the kinetic energy image in NEXAFSCleanup and cleans up photoemission (diagonal) features from the image
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
		
		//	Finds the data folder which holds the displayed waves
		DFREF ProgramFolder=root:Programming
		DFREF NEXAFSCleanupFolder=ProgramFolder:NEXAFSCleanup
	
		//	Finds the parent window of the subwindow hosting the controls
		String ParentWindow=StringFromList(0, B_Struct.win, "#")

		//	Finds  the active window instance
		String BaseName="", Suffix=""
		SplitString /E="^([A-Za-z]+)([0-9]+)$" ParentWindow, BaseName, Suffix

		//	The simulated and resulting NEXAFS and XPS images and resulting NEXAFS, Auger and XPS waves to be used for the calculation
		Wave OriginalNEXAFSImage=NEXAFSCleanupFolder:$("OriginalNEXAFSImage"+Suffix)
		Wave NEXAFSImageResult=NEXAFSCleanupFolder:$("NEXAFSImageResult"+Suffix), NEXAFSImageSim=NEXAFSCleanupFolder:$("NEXAFSImageSim"+Suffix)
		Wave XPSImageResult=NEXAFSCleanupFolder:$("XPSImageResult"+Suffix), XPSImageSim=NEXAFSCleanupFolder:$("XPSImageSim"+Suffix)
		Wave BindingEnergyImageResult=NEXAFSCleanupFolder:$("BindingEnergyImageResult"+Suffix)
		Wave NEXAFSWaveResult=NEXAFSCleanupFolder:$("NEXAFSWaveResult"+Suffix), AugerWaveResult=NEXAFSCleanupFolder:$("AugerWaveResult"+Suffix), XPSWaveResult=NEXAFSCleanupFolder:$("XPSWaveResult"+Suffix)
		
		//	Checks if the input wave NEXAFSImageResult contains any NaN values
		WaveStats/Q NEXAFSImageResult
		
		//	Only attemps the cleanup if the wave contains no NaN values
		if (V_numNaNs==0)
		
			//	Uses the position of the cursor on the resulting NEXAFS image as the pre-resonance edge
			//	NOTE: Check that cursor is placed
			Variable PreResonanceEdge=pcsr(A, ParentWindow+"#NEXAFSImageResult")

			//	Finds the number of iterations
			//	NOTE: Check that value is valid
			ControlInfo /W=$B_Struct.win SetNumIterations
			Variable NumberOfIterations=V_Value

			//	Finds the selected cleanup method
			ControlInfo /W=$B_Struct.win MethodPopUp
			String Method=S_Value
			
			Variable SubtractNEXAFSFirst=1
			Variable NoAuger=1

			//	Sets the variables according to the cleanup selected method
			strswitch(Method)
				case "Subtract NEXAFS first":
					SubtractNEXAFSFirst=1
					NoAuger=1
					break
				case "Subtract NEXAFS first and correct for Auger":
					SubtractNEXAFSFirst=1
					NoAuger=0
					break
				case "Subtract XPS first":
					SubtractNEXAFSFirst=0
					NoAuger=1
					break
				case "Subtract XPS first and correct for Auger":
					SubtractNEXAFSFirst=0
					NoAuger=0
					break
			endswitch

			//	Calculates the position of the busy box based on the size and position of the parent window
			Variable n=ScreenResolution/72
			GetWindow $ParentWindow wsize
			Variable Width=200, Height=70
			Variable XPos=Round((V_left+V_right)*n/2-Width/2), YPos=Round((V_top+V_bottom)*n/2-Height/2)

			//	Creates a panel to indicate when a calculation is running
			NewPanel/K=2 /FLT=2 /N=NEXAFSCleanupBusyBox /W=(XPos, YPos, XPos+Width, YPos+Height)
			ModifyPanel /W=NEXAFSCleanupBusyBox fixedSize=1, noEdit=1
			SetDrawEnv /W=NEXAFSCleanupBusyBox  textxjust=1, textyjust=1, xcoord=rel, ycoord=rel
			DrawText /W=NEXAFSCleanupBusyBox 0.5, 0.5, "Calculating..."
			DoUpdate /W=NEXAFSCleanupBusyBox
			
			//	Clean up photoemission features from the NEXAFS image using an iterative loop
			NEXAFSCleanupDoCleanup(OriginalNEXAFSImage, NEXAFSImageResult, NEXAFSImageSim, XPSImageResult, XPSImageSim, BindingEnergyImageResult, NEXAFSWaveResult, AugerWaveResult, XPSWaveResult, PreResonanceEdge, NumberOfIterations, SubtractNEXAFSFirst, NoAuger, ParentWindow)
			
			//	Kills the busy box
			SetActiveSubwindow _endfloat_
			KillWindow NEXAFSCleanupBusyBox
		endif
	endif
	Return 0
end



Static Function NEXAFSCleanupUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave)
//	Updates the photoemission cleanup window. ActiveFolder is not needed, but is included to allow the update functions for all graphs to share the same FUNCREF
String ActiveWindow
DFREF ActiveFolder
Wave/Z DisplayedWave

	//	Finds the data folder which holds the displayed waves
	DFREF ProgramFolder=root:Programming
	DFREF NEXAFSCleanupFolder=ProgramFolder:NEXAFSCleanup
	
	//	Updates the save button's userdata
	String SaveButtonUserData=GetUserData(ActiveWindow, "SaveButton", "")
	String DisplayedName=NameOfWave(DisplayedWave)
	String NewUserData=DisplayedName+";"+DisplayedName+"_AES;"+DisplayedName+"_XPS;"+DisplayedName+";"+DisplayedName+"_NXF;"+DisplayedName+"_XPS;"+DisplayedName+"_SNX;"+DisplayedName+"_SXP;"
	SaveButtonUserData=ReplaceStringByKey("TargetWaves", SaveButtonUserData, NewUserData, "=", ",", 1)
	Button SaveButton userData=(SaveButtonUserData), win=$ActiveWindow
	
	//	Finds the parent window of the subwindow hosting the controls
	String ParentWindow=StringFromList(0, ActiveWindow, "#")

	//	Finds  the active window instance
	String BaseName="", Suffix=""
	SplitString /E="^([A-Za-z]+)([0-9]+)$" ParentWindow, BaseName, Suffix
	
	//	Finds the x and y positions of any existing cursors
	Variable NumberOfSubWindows=6
	Make/O/FREE/T SubWindowNames={"NEXAFSWaveResult", "NEXAFSImageResult", "NEXAFSImageSim", "XPSWaveResult", "XPSImageResult", "XPSImageSim"}
	Make/O/FREE/N=(NumberOfSubWindows, 2, 2) CursorPositions
	FastOP CursorPositions=(NaN)
	
	String SubWindow=""

	//	Counts through all subwindows and saves the cursor positions
	Variable i=0
	for (i=0; i<NumberOfSubWindows; i+=1)
	
		//	The name of subwindow i
		SubWindow=ParentWindow+"#"+SubWindowNames[i]

		//	Checks if Cursor A is placed on the graph
		if (StrLen(CsrInfo(A, SubWindow))>0)
		
			//	Saves the position of Cursor A
			CursorPositions[i][0][0]=xcsr(A, SubWindow)		//	Returns NaN if cursor does not exists??
			CursorPositions[i][1][0]=vcsr(A, SubWindow)
			
			//	Removes cursor A from the graph
			Cursor /K /W=$SubWindow A
		else
			CursorPositions[i][0][0]=NaN
			CursorPositions[i][1][0]=NaN
		endif
		
		//	Checks if Cursor B is placed on the graph
		if (StrLen(CsrInfo(B, SubWindow))>0)

			//	Saves the position of Cursor B
			CursorPositions[i][0][1]=xcsr(B, SubWindow)
			CursorPositions[i][1][1]=vcsr(B, SubWindow)
			
			//	Removes cursor B from the graph
			Cursor /K /W=$SubWindow B
		else
			CursorPositions[i][0][1]=NaN
			CursorPositions[i][1][1]=NaN
		endif
	endfor
	
	//	Checks that the selected wave exists
	Variable Error=0
	String ErrorStr=""
	if (WaveExists(DisplayedWave)==0)
		
		//	Returns an error
		Error=1
		ErrorStr="Selected image does not exist!"
		
		//	Displays an empty place holder waves for the original and resulting EXAFS images
		Make/O/N=(2,2) NEXAFSCleanupFolder:$("OriginalNEXAFSImage"+Suffix)/WAVE=OriginalNEXAFSImage, NEXAFSCleanupFolder:$("NEXAFSImageResult"+Suffix)/WAVE=NEXAFSImageResult
		FastOP OriginalNEXAFSImage=(NaN)
		FastOP NEXAFSImageResult=(NaN)
	else

		//	Displays the selected image, initially as both the resulting NEXAFS and XPS image
		Duplicate/O DisplayedWave, NEXAFSCleanupFolder:$("OriginalNEXAFSImage"+Suffix)/WAVE=OriginalNEXAFSImage, NEXAFSCleanupFolder:$("NEXAFSImageResult"+Suffix)/WAVE=NEXAFSImageResult, NEXAFSCleanupFolder:$("XPSImageResult"+Suffix)/WAVE=XPSImageResult
		
		//	Checks if the 2nd order light checkbox has been clicked
		ControlInfo /W=$ActiveWindow LightCheckBox
		if (V_Value==1)
			SetScale/P x, 2*DimOffset(DisplayedWave, 0), 2*DimDelta(DisplayedWave, 0), "", OriginalNEXAFSImage, NEXAFSImageResult, XPSImageResult
		endif
		
		//	Empty place holders for all other waves
		Make/O/N=(2,2) NEXAFSCleanupFolder:$("NEXAFSImageSim"+Suffix)/WAVE=NEXAFSImageSim, NEXAFSCleanupFolder:$("XPSImageSim"+Suffix)/WAVE=XPSImageSim, NEXAFSCleanupFolder:$("BindingEnergyImageResult"+Suffix)/WAVE=BindingEnergyImageResult
		Make/O/N=0 NEXAFSCleanupFolder:$("NEXAFSWaveResult"+Suffix)/WAVE=NEXAFSWaveResult, NEXAFSCleanupFolder:$("AugerWaveResult"+Suffix)/WAVE=AugerWaveResult, NEXAFSCleanupFolder:$("XPSWaveResult"+Suffix)/WAVE=XPSWaveResult

		FastOP NEXAFSImageSim=(NaN)
		FastOP XPSImageSim=(NaN)
		FastOP BindingEnergyImageResult=(NaN)

		//	Finds the number of data points along the photon energy and kinetic energy axes
		Variable PhotonEnergyNumPnts=DimSize(OriginalNEXAFSImage, 0)
		Variable KineticEnergyNumPnts=DimSize(OriginalNEXAFSImage, 1)
		
		//	Checks if the image contains any NaN values
		WaveStats/Q NEXAFSImageResult

		//	The image must be at least 2 x 2 points and contain no NaN values
		if ((V_numNaNs!=0) || (PhotonEnergyNumPnts<2) || (KineticEnergyNumPnts<2))

			//	Returns an error
			Error=1
			ErrorStr="\JCImage must be at least 2 x 2 points\r\JCand contain no NaN values!"
		else

			//	Finds the step sizes along the photon energy and kinetic energyaxes
			Variable PhotonEnergyDelta=DimDelta(OriginalNEXAFSImage, 0)
			Variable KineticEnergyDelta=DimDelta(OriginalNEXAFSImage, 1)
			
			//	Forces the photon energy to start at the lowest value
			if (PhotonEnergyDelta<0)
				PhotonEnergyDelta=Abs(PhotonEnergyDelta)
				Reverse /DIM=0 NEXAFSImageResult, XPSImageResult
			endif
		
			//	Forces the kinetic energy to start at the lowest value
			if (KineticEnergyDelta<0)
				KineticEnergyDelta=Abs(KineticEnergyDelta)
				Reverse /DIM=1 NEXAFSImageResult, XPSImageResult
			endif
		
			//	Calculates the photon energy step size as an integer times the kinetic energy step size
			//	This is required for a fast calculation of the binding energy image
			Variable PhotonStepInteger=PhotonEnergyDelta/KineticEnergyDelta
		
			//	Checks that the photon energy step size really is an integer times the kinetic energy step size
			if (Abs(Mod(PhotonStepInteger, 1))>1e-5)
	
				//	Returns an error
				Error=1
				ErrorStr="\JCPhoton energy step size must be an integer\r\JCtimes the kinetic energy step size!"
			else
		
				//	Uses the original kinetic energy image to calculate the binding energy image and NEXAFS, Auger and XPS spectra
				//	11.4 ms
				NewNEXAFSCalculateAllSpectra(OriginalNEXAFSImage, BindingEnergyImageResult, NEXAFSWaveResult, AugerWaveResult, XPSWaveResult, $"")
				
				//	Finds the photon energy and kinetic energy range of the image
				Variable PhotonEnergyLow=DimOffset(OriginalNEXAFSImage, 0)
				Variable PhotonEnergyHigh=PhotonEnergyLow+PhotonEnergyDelta*(PhotonEnergyNumPnts-1)
				Variable KineticEnergyLow=DimOffset(OriginalNEXAFSImage, 1)
				Variable KineticEnergyHigh=KineticEnergyLow+KineticEnergyDelta*(KineticEnergyNumPnts-1)
				Variable BindingEnergyHigh=DimOffset(XPSWaveResult, 0)
				Variable BindingEnergyLow=BindingEnergyHigh+DimDelta(XPSWaveResult, 0)*(DimSize(XPSWaveResult, 0)-1)
				
				//	Checks if the old cursor x and y-positions are still valid on the new images
				CursorPositions[0,2][0][]=((limit(CursorPositions[p][q][r], PhotonEnergyLow, PhotonEnergyHigh)==CursorPositions[p][q][r]) ? (limit(CursorPositions[p][q][r], PhotonEnergyLow, PhotonEnergyHigh)) : (NaN))
				CursorPositions[4,*][0][]=((limit(CursorPositions[p][q][r], PhotonEnergyLow, PhotonEnergyHigh)==CursorPositions[p][q][r]) ? (limit(CursorPositions[p][q][r], PhotonEnergyLow, PhotonEnergyHigh)) : (NaN))
				CursorPositions[*][1][]=limit(CursorPositions[p][q][r], KineticEnergyLow, KineticEnergyHigh)
				CursorPositions[3][0][]=((limit(CursorPositions[p][q][r], BindingEnergyLow, BindingEnergyHigh)==CursorPositions[p][q][r]) ? (limit(CursorPositions[p][q][r], BindingEnergyLow, BindingEnergyHigh)) : (NaN))
				
				//	The cursors should never be placed back on the simulated images since they start out being NaN
				CursorPositions[2][][]=NaN
				CursorPositions[5][][]=NaN
				
				//	Places the cursors back on the graph
				for (i=0; i<NumberOfSubWindows; i+=1)
				
					//	The name of subwindow i
					SubWindow=ParentWindow+"#"+SubWindowNames[i]
					
					//	Checks if the Cursor A existed on the graph before and is still valid
					if (NumType(CursorPositions[i][0][0])==0)
					
						//	Places Cursor A back on the graph
						if ((i==0) || (i==3))
							Cursor/W=$SubWindow A $(SubWindowNames[i]+Suffix), CursorPositions[i][0][0]
						else
							Cursor/I /W=$SubWindow A $(SubWindowNames[i]+Suffix), CursorPositions[i][0][0], CursorPositions[i][1][0]
						endif
					else
					
						//	Even if a previous cursor did not exist, a cursor is always placed on the resulting NEXAFS image
						if (CmpStr(SubWindowNames[i], "NEXAFSImageResult")==0)
							Cursor/I/P/A=1 /W=$SubWindow A $(SubWindowNames[i]+Suffix), round(0.10*PhotonEnergyNumPnts), round(0.3*KineticEnergyNumPnts)
						endif
					endif

					//	Checks if the Cursor B existed on the graph before
					if (NumType(CursorPositions[i][0][1])==0)
					
						//	Places Cursor B back on the graph
						if ((i==0) || (i==3))
							Cursor/W=$SubWindow B $(SubWindowNames[i]+Suffix), CursorPositions[i][0][1]
						else
							Cursor/I /W=$SubWindow B $(SubWindowNames[i]+Suffix), CursorPositions[i][0][1], CursorPositions[i][1][1]
						endif
					endif
				endfor
			endif
		endif
	endif

	//	If an error occurred empty waves are displayed
	if (Error!=0)

		//	Displays empty place holder waves for the resulting XPS image
		Make/O/N=(2,2) NEXAFSCleanupFolder:$("XPSImageResult"+Suffix)/WAVE=XPSImageResult
		FastOP XPSImageResult=(NaN)
		
		//	Creates an error message box
		TextBox /W=$(ParentWindow+"#NEXAFSImageResult") /K /N=ErrorBox
		TextBox /W=$(ParentWindow+"#NEXAFSImageResult") /A=MC /N=ErrorBox ErrorStr
	else

		//	Removes the error message box if one exists
		TextBox /W=$(ParentWindow+"#NEXAFSImageResult") /K /N=ErrorBox
	endif
end



Static Function NEXAFSCleanupDoCleanup(OriginalNEXAFSImage, NEXAFSImageResult, NEXAFSImageSim, XPSImageResult, XPSImageSim, BindingEnergyImageResult, NEXAFSWaveResult, AugerWaveResult, XPSWaveResult, PreResonanceEdge, NumberOfIterations, SubtractNEXAFSFirst, NoAuger, ParentWindow)
//	Clean up photoemission features from the NEXAFS image using an iterative loop
Wave OriginalNEXAFSImage, NEXAFSImageResult, NEXAFSImageSim, XPSImageResult, XPSImageSim, BindingEnergyImageResult, NEXAFSWaveResult, AugerWaveResult, XPSWaveResult
Variable PreResonanceEdge, NumberOfIterations, SubtractNEXAFSFirst, NoAuger
String ParentWindow

	//	Starts a timer
	Print("\r")
	Print("Photoemission cleanup started...")
	Variable t=StartMsTimer

	//	Creates an empty wave reference needed for the loop
	Wave/Z NoWave=$""
	
	//	Creates placeholders for two waves used by the MatrixOP waveMap operation that converts the kinetic energy image to a binding energy image and vice versa
	Make/O/FREE/N=0 IndexWave, ReverseIndexWave

	//	Checks if this is the first cleanup, or if cleanup has been pressed before for this data set. If this is the first cleanup all values in the simulated images will be NaN
	Variable FirstCleanup=(NumType(NEXAFSImageSim[0][0])==2)
	
	//	For the first cleanup the NEXAFS and Auger waves will have been generated without subtracting the pre-resonance offset, and hence has to be recalculated.
	if (FirstCleanup)

		//	Subtracts the pre-resonance offset from the resulting NEXAFS image. This is needed when normalizing the image with the Auger wave
		Duplicate/O/FREE/R=[0, PreResonanceEdge][] NEXAFSImageResult, PreResonanceImage
		FastOP NEXAFSImageResult=NEXAFSImageResult-(mean(PreResonanceImage))

		//	Uses the original kinetic energy image to calculate the binding energy image and NEXAFS, Auger and XPS spectra
		//	11.4 ms
		NewNEXAFSCalculateAllSpectra(NEXAFSImageResult, BindingEnergyImageResult, NEXAFSWaveResult, AugerWaveResult, XPSWaveResult, IndexWave)
	endif
	
	Variable i=0
	
	//	Checks if the user has choosen to subtract the NEXAFS or XPS features first in the iterative loop
	if (SubtractNEXAFSFirst)

		//	Starts the loop by simulating and subtracting the NEXAFS features
		for (i=0; i<NumberOfIterations; i+=1)
		
			//	Updates the photoemission cleanup window to allow the user to see the progress of the iteration
			DoUpdate /W=$ParentWindow
		
			//	Forces the Auger spectrum to be a flat line, if the No Auger mode has been selected
			if (NoAuger)
				FastOP AugerWaveResult=(mean(AugerWaveResult))
			endif
	
			//	Simulates a NEXAFS image from the resulting NEXAFS and Auger spectra
			NEXAFSCleanupSimNEXAFSImage(NEXAFSWaveResult, AugerWaveResult, NEXAFSImageSim)
		
			//	Calculate the resulting XPS image by subtractiung the simulated NEXAFS image from the original kinetic energy image
			FastOP XPSImageResult=OriginalNEXAFSImage-NEXAFSImageSim
		
			//	Calculates the XPS spectrum from the resulting XPS image
			NewNEXAFSCalculateAllSpectra(XPSImageResult, BindingEnergyImageResult, NoWave, NoWave, XPSWaveResult, IndexWave)
		
			//	Simulates an XPS image from the resulting XPS spectrum. The original image is only needed to get the scaling of the simulated image correct
			NEXAFSCleanupSimXPSImage(XPSWaveResult, XPSImageSim, ReverseIndexWave, OriginalNEXAFSImage)

			//	Subtracts the simulated XPS image from the original kinetic energy image to calculate the resulting NEXAFS image
			FastOP NEXAFSImageResult=OriginalNEXAFSImage-XPSImageSim

			//	Subtracts the pre-resonance offset from the resulting NEXAFS image. This is needed when normalizing the image with the Auger wave
			Duplicate/O/FREE/R=[0, PreResonanceEdge][] NEXAFSImageResult, PreResonanceImage
			FastOP NEXAFSImageResult=NEXAFSImageResult-(mean(PreResonanceImage))
		
			//	Calculates the NEXAFS and Auger spectra from the resulting NEXAFS image
			NewNEXAFSCalculateAllSpectra(NEXAFSImageResult, NoWave, NEXAFSWaveResult, AugerWaveResult, NoWave, NoWave)
		endfor
	else
	
		//	Starts the loop by simulating and subtracting the XPS features
		for (i=0; i<NumberOfIterations; i+=1)
		
			//	Updates the photoemission cleanup window to allow the user to see the progress of the iteration
			DoUpdate /W=$ParentWindow
	
			//	Simulates an XPS image from the resulting XPS spectrum. The original image is only needed to get the scaling of the simulated image correct
			NEXAFSCleanupSimXPSImage(XPSWaveResult, XPSImageSim, ReverseIndexWave, OriginalNEXAFSImage)
			
			//	Subtracts the simulated XPS image from the original kinetic energy image to calculate the resulting NEXAFS image
			FastOP NEXAFSImageResult=OriginalNEXAFSImage-XPSImageSim
			
			//	Subtracts the pre-resonance offset from the resulting NEXAFS image. This is needed when normalizing the image with the Auger wave
			Duplicate/O/FREE/R=[0, PreResonanceEdge][] NEXAFSImageResult, PreResonanceImage
			FastOP NEXAFSImageResult=NEXAFSImageResult-(mean(PreResonanceImage))
		
			//	Calculates the NEXAFS and Auger spectra from the resulting NEXAFS image
			NewNEXAFSCalculateAllSpectra(NEXAFSImageResult, NoWave, NEXAFSWaveResult, AugerWaveResult, NoWave, NoWave)

			//	Forces the Auger spectrum to be a flat line, if the No Auger mode has been selected
			if (NoAuger)
				FastOP AugerWaveResult=(mean(AugerWaveResult))
			endif

			//	Simulates a NEXAFS image from the resulting NEXAFS and Auger spectra
			NEXAFSCleanupSimNEXAFSImage(NEXAFSWaveResult, AugerWaveResult, NEXAFSImageSim)
		
			//	Subtracts the simulated NEXAFS image from the original kinetic energy image to calculate the resulting XPS image
			FastOP XPSImageResult=OriginalNEXAFSImage-NEXAFSImageSim
		
			//	Calculates the XPS spectrum from the resulting XPS image
			NewNEXAFSCalculateAllSpectra(XPSImageResult, BindingEnergyImageResult, NoWave, NoWave, XPSWaveResult, IndexWave)
		endfor
	endif
	
	//	Stops the timer
	Print(Num2iStr(i)+" iterations finished in "+Num2Str(StopMSTimer(t)/1e6)+" seconds")
end



Static Function NEXAFSCleanupSimNEXAFSImage(NEXAFSWaveResult, AugerWaveResult, NEXAFSImageSim)
//	Simulates a NEXAFS image from the resulting NEXAFS and Auger spectra
//	NOTE: The scaling of the NEXAFS and Auger source waves MUST be from low-to-high photon energy and kinetic energy
Wave/Z NEXAFSWaveResult, AugerWaveResult, NEXAFSImageSim

	//	NOTE: Make the error propagate through loop
	Variable Error=0

	//	Checks that the source and output waves exist
	if ((WaveExists(NEXAFSWaveResult)==0) || (WaveExists(AugerWaveResult)==0) || (WaveExists(NEXAFSImageSim)==0))

		//	At least one wave is invalid
		Error=1
	else

		//	Checks that the NEXAFS and Auger spectra will produce at least a 2 x 2 image
		if ((NumPnts(NEXAFSWaveResult)<2) || (NumPnts(AugerWaveResult)<2))
		
			//	The NEXAFS or Auger spectra are invalid
			Error=1
		else

			//	Creates a normalized version of the NEXAFS spectrum
			//	0.006 ms	
			Duplicate/O/FREE NEXAFSWaveResult, NormalizedNEXAFSWaveResult
			FastOP NormalizedNEXAFSWaveResult=(1/Mean(NEXAFSWaveResult))*NEXAFSWaveResult

			//	Simulates a NEXAFS image from the resulting NEXAFS and Auger spectra
			//	0.37 ms
			MatrixOP/O/NTHR=0 NEXAFSImageSim=(NormalizedNEXAFSWaveResult x (AugerWaveResult^t))

			//	Ensures the simulated image has the correct scaling
			//	0.003 ms
			SetScale/P x, DimOffset(NEXAFSWaveResult, 0), DimDelta(NEXAFSWaveResult, 0), "", NEXAFSImageSim
			SetScale/P y, DimOffset(AugerWaveResult, 0), DimDelta(AugerWaveResult, 0), "", NEXAFSImageSim
		endif
	endif
	
	//	If the NEXAFS or Auger spectra are invalid all values in the output wave are set to NaN
	if (Error!=0)
		if (WaveExists(NEXAFSImageSim))
			FastOP NEXAFSImageSim=(NaN)
		endif
	endif
end



Static Function NEXAFSCleanupSimXPSImage(XPSWaveResult, XPSImageSim, ReverseIndexWave, KineticEnergyImage)
//	Simulates an XPS image from the resulting XPS spectrum
//	NOTE: The scaling of the XPS source wave MUST be from high-to-low binding energy. The kinetic energy image is only used to get the correct scaling for the simulated image
Wave/Z XPSWaveResult, XPSImageSim, ReverseIndexWave, KineticEnergyImage

	//	NOTE: Make the error propagate through loop
	Variable Error=0

	//	Checks that the source and output waves exist
	if ((WaveExists(XPSWaveResult)==0) || (WaveExists(XPSImageSim)==0) || (WaveExists(KineticEnergyImage)==0))

		//	At least one wave is invalid
		Error=1
	else

		//	Checks that the XPS spectrum is valid....
		//	NOTE: ehhh???
		if (2==4)
		
			//	The XPS spectrum is invalid
			Error=1
		else

			Variable PhotonEnergyNumPnts=DimSize(KineticEnergyImage, 0)
			Variable KineticEnergyNumPnts=DimSize(KineticEnergyImage, 1)
	
			//	Calculates the photon energy step size as an integer times the kinetic energy step size
			//	This is required for a fast calculation of the binding energy image
			//	NOTE: Move to parent function
			Variable PhotonEnergyDelta=DimDelta(KineticEnergyImage, 0)
			Variable KineticEnergyDelta=DimDelta(KineticEnergyImage, 1)
			Variable PhotonStepInteger=Round(PhotonEnergyDelta/KineticEnergyDelta)

			//	If ithe index wave does not exist, a free placeholder wave is created
			if (WaveExists(ReverseIndexWave)==0)
				Make/O/FREE/N=0 ReverseIndexWave
			endif
	
			//	If the index wave needed to create the wave map is invalid, it is recreated. This is time consuming, hence the possibility to reuse a previous one.
			if ((DimSize(ReverseIndexWave, 0)!=PhotonEnergyNumPnts) && (DimSize(ReverseIndexWave, 1)!=KineticEnergyNumPnts))
			
				//	Temporary waves used to create the index wave for the MatrixOP waveMap operation
				//	0.005 ms
				Make/O/FREE/N=(PhotonEnergyNumPnts) XWave
				Make/O/FREE/N=(1, KineticEnergyNumPnts) YWave

				//	Creates the first half of the index wave for the MatrixOP waveMap operation
				//	0.54 ms
				XWave=p
				FastOP XWave=(PhotonStepInteger)*XWave
				Reverse XWave
				FastOP YWave=1
				MatrixOP/O/FREE/NTHR=0 XImage=(XWave x YWave)	

				//	Creates the second half of the index wave for the MatrixOP waveMap operation
				//	0.48 ms
				FastOP XWave=1
				YWave=q
				MatrixOP/O/FREE/NTHR=0 YImage=(XWave x YWave)

				//	Combines the two halves to create the final index wave
				//	0.16 ms
				Duplicate/O/FREE XImage, ReverseIndexWave
				FastOP ReverseIndexWave=XImage+YImage
			endif

			//	Simulates an XPS image from the resulting XPS spectrum. Identical to XPSImageSim=XPSWaveResult(x-y), but faster
			//	0.51 ms
			MatrixOP/O/NTHR=0 XPSImageSim=waveMap(XPSWaveResult, ReverseIndexWave)

			//	Ensures the simulated image has the correct scaling
			SetScale/P x, DimOffset(KineticEnergyImage, 0), DimDelta(KineticEnergyImage, 0), "", XPSImageSim
			SetScale/P y, DimOffset(KineticEnergyImage, 1), DimDelta(KineticEnergyImage, 1), "", XPSImageSim
		endif
	endif
	
	//	If the one of the source waves are invalid all values in the output wave are set to NaN
	if (Error!=0)
		if (WaveExists(XPSImageSim))
			FastOP XPSImageSim=(NaN)
		endif
	endif
end



Static Function NEXAFSCleanupNewWin(B_Struct) : ButtonControl
//	Creates another instances of the photoemission cleanup window
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
		
		//	Finds the parent window of the subwindow hosting the controls
		String OldParentWindow=StringFromList(0, B_Struct.win, "#")

		//	Finds  the base name and suffix of the old parent window
		String OldBaseName="", OldSuffix=""
		SplitString /E="^([A-Za-z]+)([0-9]+)$" OldParentWindow, OldBaseName, OldSuffix

		//	Finds the selected data folder
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		String FolderName=S_Value

		//	Finds the selected wave
		ControlInfo /W=$B_Struct.win DisplayedWavePopUp
		String DisplayedName=S_Value

		//	Creates another photoemission cleanup window and saves the name of the window created
		String NewParentWindow=RawNEXAFSNewWinName(OldBaseName)
		NEXAFSCleanupCreateWindow(FolderName, DisplayedName, NewParentWindow, 1)
		
		//	Finds  the base name and suffix of the new parent window
		String NewBaseName="", NewSuffix=""
		SplitString /E="^([A-Za-z]+)([0-9]+)$" NewParentWindow, NewBaseName, NewSuffix

		//	Copies the selected colour schemes to the new windows
		ControlInfo /W=$B_Struct.win ColourSchemePopUp
		PopupMenu ColourSchemePopUp popmatch=S_Value, win=$NewParentWindow#ControlPanel
		ModifyImage/W=$NewParentWindow#NEXAFSImageResult '' ctab= {*,*,$S_Value,0}			//	' ' indicates all images in graph
		ModifyImage/W=$NewParentWindow#NEXAFSImageSim '' ctab= {*,*,$S_Value,0}		//	' ' indicates all images in graph
		ModifyImage/W=$NewParentWindow#XPSImageResult '' ctab= {*,*,$S_Value,0}			//	' ' indicates all images in graph
		ModifyImage/W=$NewParentWindow#XPSImageSim '' ctab= {*,*,$S_Value,0}		//	' ' indicates all images in graph
		
		//	Copies the cleanup values to the new window
		ControlInfo /W=$B_Struct.win SetNumIterations
		SetVariable SetNumIterations, value=_NUM:(V_Value), win=$NewParentWindow#ControlPanel
		ControlInfo /W=$B_Struct.win MethodPopUp
		PopupMenu MethodPopUp popmatch=S_Value, win=$NewParentWindow#ControlPanel
		ControlInfo /W=$B_Struct.win LightCheckBox
		CheckBox LightCheckBox value=(V_Value), win=$NewParentWindow#ControlPanel

		//	Finds the data folder which holds the displayed waves
		DFREF ProgramFolder=root:Programming
		DFREF NEXAFSCleanupFolder=ProgramFolder:NEXAFSCleanup

		//	Duplicates all relevant waves
		Duplicate/O NEXAFSCleanupFolder:$("NEXAFSImageResult"+OldSuffix), NEXAFSCleanupFolder:$("NEXAFSImageResult"+NewSuffix)
		Duplicate/O NEXAFSCleanupFolder:$("NEXAFSImageSim"+OldSuffix), NEXAFSCleanupFolder:$("NEXAFSImageSim"+NewSuffix)
		Duplicate/O NEXAFSCleanupFolder:$("XPSImageResult"+OldSuffix), NEXAFSCleanupFolder:$("XPSImageResult"+NewSuffix)
		Duplicate/O NEXAFSCleanupFolder:$("XPSImageSim"+OldSuffix), NEXAFSCleanupFolder:$("XPSImageSim"+NewSuffix)
		Duplicate/O NEXAFSCleanupFolder:$("BindingEnergyImageResult"+OldSuffix), NEXAFSCleanupFolder:$("BindingEnergyImageResult"+NewSuffix)
		Duplicate/O NEXAFSCleanupFolder:$("OriginalNEXAFSImage"+OldSuffix), NEXAFSCleanupFolder:$("OriginalNEXAFSImage"+NewSuffix)
		Duplicate/O NEXAFSCleanupFolder:$("NEXAFSWaveResult"+OldSuffix), NEXAFSCleanupFolder:$("NEXAFSWaveResult"+NewSuffix)
		Duplicate/O NEXAFSCleanupFolder:$("AugerWaveResult"+OldSuffix), NEXAFSCleanupFolder:$("AugerWaveResult"+NewSuffix)
		Duplicate/O NEXAFSCleanupFolder:$("XPSWaveResult"+OldSuffix), NEXAFSCleanupFolder:$("XPSWaveResult"+NewSuffix)
	endif
	Return 0
end



Static Function RawNEXAFSRoundOffsets(Offset, Delta)
//	Rounds the offset and step sizes to 'nicer' values
//	BUILD INTO NORMAL RAWNEXAFS VIEWER
Variable &Offset, &Delta

//	//	Only allows the step sizes to have two decimals and the second decimal can only be 0 or 5, e.g. 1.0, 1.5, 2.0, 2.5, etc...
//	Variable Scale=10^Floor(Log(PhotonEnergyDelta))	
//	PhotonEnergyDelta=Round(2*PhotonEnergyDelta/Scale)*Scale/2
//	
//	//	Change the kinetic energy step size to be an integer fraction of the photon energy step size. This make it much easier to convert the kinetic energy image into a binding energy image to extract the XPS spectrum
//	KineticEnergyDelta=PhotonEnergyDelta/Round(PhotonEnergyDelta/KineticEnergyDelta)
//	
//	//	Changes the photon energy and kinetic energy starting values to be an integer times the step sizes
//	PhotonEnergyStart=Round(PhotonEnergyStart/PhotonEnergyDelta)*PhotonEnergyDelta
//	KineticEnergyStart=Round(KineticEnergyStart/KineticEnergyDelta)*KineticEnergyDelta
end





//     -----<<<<<     View NEXAFS     >>>>>-----
//	This sections contains the functions used by the View NEXAFS menu item

Static Function ViewNEXAFS(FolderName, DisplayedName, SubtractName, FluxName, ForceSelection)
//	Displays the waveform 2D NEXAFS images acquired at Elettra
String FolderName, DisplayedName, SubtractName, FluxName
Variable ForceSelection

	//	Creates a list of existing View NEXAFS windows
	String ListOfWindows=WinList("ViewNEXAFSWindow*", ";", "WIN:1")

	if (StrLen(ListOfWindows)==0)
	
		//	Creates a new window
		ViewNEXAFSCreateWindow(FolderName, DisplayedName, SubtractName, FluxName, "ViewNEXAFSWindow0", ForceSelection)

	elseif (ForceSelection==0)
		
		//	Brings the first window instance in the list to the front
		String ActiveWindow=StringFromList(0, ListOfWindows, ";")
		DoWindow/F $ActiveWindow
		
		//	Finds the displayed wave
		DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
		Wave/Z SubtractWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "SubtractWavePopUp")
		Wave/Z FluxWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder:PhotonFlux, "FluxWavePopUp")
		
		//	Updates the View NEXAFS window	
		ViewNEXAFSUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
	else

		//	Creates a new window
		ViewNEXAFSCreateWindow(FolderName, DisplayedName, SubtractName, FluxName, RawNEXAFSNewWinName("ViewNEXAFSWindow"), ForceSelection)
	endif
end



Static Function ViewNEXAFSCreateWindow(FolderName, DisplayedName, SubtractName, FluxName, ActiveWindow, ForceSelection)
//	Creates the window to display the waveform NEXAFS images, suffix is used to create multiple copies, so images can be compared
String FolderName, DisplayedName, SubtractName, FluxName, ActiveWindow
Variable ForceSelection
Variable n=120/ScreenResolution

	//	Finds  the active window instance
	String Suffix=""
	SScanF ActiveWindow, "ViewNEXAFSWindow%s", Suffix
	Variable WindowInstance=Str2Num(Suffix)

	//	Creates the folder to hold all the permanent waves, used for different bureaucratic purposes, such as listboxes, displaying waves, etc...
	NewDataFolder/O root:Programming
	DFREF ProgramFolder=root:Programming
	
	//	Creates the folder to hold all the temporary waves, created during a function call and deleted again at the end of the function call.
	NewDataFolder/O root:Programming:Temp

	//	Creates the 2D wave to hold the displayed NEXAFS Image and the 1D wave to hold the resulting spectrum
	Make/O ProgramFolder:$("ViewNEXAFSDataWave"+Suffix)={{0}},  ProgramFolder:$("ViewNEXAFSSpectrumWave"+Suffix)={0}
	Wave DataWave=ProgramFolder:$("ViewNEXAFSDataWave"+Suffix)
		
	//	Creates the graph to display the NEXAFS images. The \u#2 in the labels prevents Igor from displaying the units of the wave as specified by SetScale
	DoWindow /K $ActiveWindow
	Display /W=(5+WindowInstance*50, 40+WindowInstance*10, 5+365*n+WindowInstance*50, 40+320*n+WindowInstance*10) /K=1 /N=$ActiveWindow
	AppendImage/W=$ActiveWindow DataWave
	ModifyImage/W=$ActiveWindow '' ctab= {*,*,ColdWarm,0}, ctabAutoscale=1, lookup= $""
	ModifyGraph /W=$ActiveWindow margin(top)=60*n
	Label /W=$ActiveWindow bottom "Photon Energy (\\ueV)"
	Label /W=$ActiveWindow left "Kinetic Energy (\\ueV)"
	
	//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
	SetWindow $ActiveWindow hook(KillHook)=ViewNEXAFSHookFunc

	//	Selects the active data folder, listing root: and all it's subfolders with the exception of root:Programming and it's subfolders. To list the default folder (set with SetDataFolder) and all it's subfolders instead with no exceptions, use XPSViewDataFoldersList(:, $"")
	PopupMenu DataFolderPopUp bodyWidth=250, mode=1, pos={210,5}, proc=EccentricXPS#ViewNEXAFSFolderPopupMenu, value=EccentricXPS#XPSViewGroupsAndFoldersList(root:, root:Programming, "SavedFits"), userdata="EccentricXPS#ViewNEXAFSUpdateGraph", win=$ActiveWindow, help={"Selects the data folder to look for waves in"}
	//	Popup menu to select the NEXAFS image to display. The complicated value call #(String) indicates that the value ActiveWindow had when the control was created should be used
	PopupMenu DisplayedWavePopUp bodyWidth=200, mode=1, pos={415,5}, proc=EccentricXPS#ViewNEXAFSWavePopupMenu, value=#("EccentricXPS#XPSViewPopUpWaveList(2, \""+ActiveWindow+"\", \"DataFolderPopUp\")"), userdata="EccentricXPS#ViewNEXAFSUpdateGraph", win=$ActiveWindow, help={"Selects the NEXAFS image to display"}
	//	Popup menu to select the colour scheme to be used. Default is ColdWarm
	PopupMenu ColourSchemePopUp bodyWidth=130, mode=3, pos={550,5},  proc=EccentricXPS#XPSViewImageColours, value="Grays;YellowHot;ColdWarm;SpectrumBlack;", win=$ActiveWindow, help={"Selects the colour scheme to use for the image"}
	//	Popup menu to select the NEXAFS image to subtract from the displayed image. The complicated value call #(String) indicates that the value ActiveWindow had when the control was created should be used
	PopupMenu SubtractWavePopUp bodyWidth=200, mode=1, pos={160,33}, proc=EccentricXPS#ViewNEXAFSWavePopupMenu, value=#("\"- select wave to subtract -;\"+EccentricXPS#ViewNEXAFSPopUpSubtractWaveList(\""+ActiveWindow+"\")"), userdata="EccentricXPS#ViewNEXAFSUpdateGraph", win=$ActiveWindow, help={"Selects the background NEXAFS image to subtract from the main image"}
	
	//	Variables used to subtract a reference spectrum
	SetVariable SetMult, limits={-inf, inf, 0}, proc=EccentricXPS#ViewNEXAFSSetMult, size={85,20}, pos+={0, 2}, title="Mult. ", value=_NUM:1, userdata="EccentricXPS#ViewNEXAFSUpdateGraph", win=$ActiveWindow, help={"Selects the value to multiply the background image with before subtracting it"}
	SetVariable SetXShift, limits={-inf, inf, -0.02}, proc=EccentricXPS#ViewNEXAFSSetVariable, size={115,20}, title="Shift (X/Y) ", value=_NUM:0, userdata="EccentricXPS#ViewNEXAFSUpdateGraph", win=$ActiveWindow, help={"Selects how much the background image should be shifted in the X-direction before subtracting it"}
	SetVariable SetYShift, limits={-inf, inf, -0.02}, proc=EccentricXPS#ViewNEXAFSSetVariable, size={65,20}, pos+={-7, 0}, title="/", value=_NUM:0, userdata="EccentricXPS#ViewNEXAFSUpdateGraph", win=$ActiveWindow, help={"Selects how much the background image should be shifted in the Y-direction before subtracting it"}
	SetVariable SetYTilt, limits={-inf, inf, -0.005}, proc=EccentricXPS#ViewNEXAFSSetVariable, size={90,20}, title="Tilt (Y) ", value=_NUM:0, userdata="EccentricXPS#ViewNEXAFSUpdateGraph", win=$ActiveWindow, help={"Selects the how much the background image should be tilted in the Y-direction before subtracting it"}

	//	Popup menu to select the photon flux reference
	PopupMenu FluxWavePopUp bodyWidth=200, mode=1, pos={160,61}, proc=EccentricXPS#ViewNEXAFSWavePopupMenu, value=#("\"- select photon flux wave -;\"+EccentricXPS#ViewNEXAFSPopUpFluxWaveList(\""+ActiveWindow+"\")"), userdata="EccentricXPS#ViewNEXAFSUpdateGraph", win=$ActiveWindow, help={"Selects the photon flux wave to divide the image with"}
	//	Changes the y axis between kinetic and binding energy
	Button YAxisButton proc=EccentricXPS#ViewNEXAFSYAxis, disable=2, size={68, 20}, title="Y Axis", userdata="0", win=$ActiveWindow, help={"Toggles the Y-axis between kinetic energy and binding energy"}
	//	Creates a copy of the wave
	Button SaveButton, proc=EccentricXPS#ViewNEXAFSSaveImage, size={67, 20}, title="Save", win=$ActiveWindow, help={"Saves the resulting NEXAFS image"}
	//	Opens a window to view the resulting 1D spectrum
	Button SpectrumButton proc=EccentricXPS#ViewNEXAFSSpectrum, size={68, 20}, title="Spectrum", win=$ActiveWindow, help={"Displays the resulting 1D spectrum"}
	//	Creates another instances of the NEXAFS image window
	Button NewWinButton, proc=EccentricXPS#ViewNEXAFSNewWin, size={67, 20}, title="New Win", win=$ActiveWindow, help={"Creates a new independent instance of the View NEXAFS window"}
	//	Opens the help file
	Button HelpButton proc=EccentricXPS#XPSViewHelp, size={68,20}, title="Help", userdata="View NEXAFS", win=$ActiveWindow, help={"Opens the help file"}

	//	Changes the colours of the cursors to red and blue with dashed crosshairs
	Cursor/M/C=(65535, 0, 0) /H=1 /L=1 /W=$ActiveWindow A
	Cursor/M/C=(0, 0, 65535) /H=1 /L=1 /W=$ActiveWindow B
	
	//	Sets the data folder selection to FolderOrGroup and reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or the default data folder is used instead, depending on the value of ForceSelection
	PopupMenu DataFolderPopUp popmatch=FolderName, win=$ActiveWindow
	ControlInfo /W=$ActiveWindow DataFolderPopUp
	if (CmpStr(FolderName, S_Value)!=0)
		if (ForceSelection==1)
			PopupMenu DataFolderPopUp mode=1, popvalue=FolderName, win=$ActiveWindow
		else
			PopupMenu DataFolderPopUp popmatch=GetDataFolder(1), win=$ActiveWindow
		endif
	endif
	
	//	Updates the list of wave selections and select DisplayedName, or if DisplayedName doesn't exist in the list, the first item in the list is selected
	PopupMenu DisplayedWavePopUp mode=1, popmatch=DisplayedName, win=$ActiveWindow

	//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
	if (StrLen(DisplayedName)>0)
		ControlInfo /W=$ActiveWindow DisplayedWavePopUp
		if (CmpStr(DisplayedName, S_Value)!=0)
			if (ForceSelection==1)
				PopupMenu DisplayedWavePopUp mode=1, popvalue=DisplayedName, win=$ActiveWindow
			endif
		endif
	endif

	//	Updates the list of wave selections and select SubtractName, or if SubtractName doesn't exist in the list, the first item in the list is selected
	PopupMenu SubtractWavePopUp mode=1, popmatch=SubtractName, win=$ActiveWindow

	//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
	if (StrLen(SubtractName)>0)
		ControlInfo /W=$ActiveWindow SubtractWavePopUp
		if (CmpStr(SubtractName, S_Value)!=0)
			if (ForceSelection==1)
				PopupMenu SubtractWavePopUp mode=1, popvalue=SubtractName, win=$ActiveWindow
			endif
		endif
	endif

	//	Updates the list of wave selections and select FluxName, or if FluxName doesn't exist in the list, the first item in the list is selected
	PopupMenu FluxWavePopUp mode=1, popmatch=FluxName, win=$ActiveWindow

	//	Reads the value of the control back to check if the selection was valid. If the selection was invalid it is either forced or first wave in the list is used instead, depending on the value of ForceSelection
	if (StrLen(FluxName)>0)
		ControlInfo /W=$ActiveWindow FluxWavePopUp
		if (CmpStr(FluxName, S_Value)!=0)
			if (ForceSelection==1)
				PopupMenu FluxWavePopUp mode=1, popvalue=FluxName, win=$ActiveWindow
			endif
		endif
	endif

	DFREF ActiveFolder=XPSViewGetDataFolder(ActiveWindow, "DataFolderPopUp")
	Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
	Wave/Z SubtractWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "SubtractWavePopUp")
	Wave/Z FluxWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder:PhotonFlux, "FluxWavePopUp")

	//	Updates the View NEXAFS window
	ViewNEXAFSUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
end



Function ViewNEXAFSHookFunc(s)
//	The hook function for the View NEXAFS window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1
			
			//	Finds the active window
			String ActiveWindow=s.winName
	
			//	Extract the window instance
			String Suffix=""
			SScanF ActiveWindow, "ViewNEXAFSWindow%s", Suffix
			
			//	Kills the spectrum window associated with the image window
			DoWindow/K $("ViewNEXAFSSpectrumWindow"+Suffix)
	
			//	KIlls the waves associated with the window, but delays the execution until the window has been killed
			Execute/P/Q "KillWaves/Z root:Programming:ViewNEXAFSDataWave"+Suffix+", root:Programming:ViewNEXAFSSpectrumWave"+Suffix
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function ViewNEXAFSFolderPopupMenu(PU_Struct) : PopupMenuControl
//	Runs the update function if the selection is changed
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		PU_Struct.blockReentry=1
		
		//	Finds the active folder
		DFREF ActiveFolder=$PU_Struct.popStr
		
		//	Changes the wave selections to the first items in the lists
		PopupMenu DisplayedWavePopUp mode=1, win=$PU_Struct.win
		PopupMenu SubtractWavePopUp mode=1, win=$PU_Struct.win
		PopupMenu FluxWavePopUp mode=1, win=$PU_Struct.win
		
		//	Finds the name of the waves
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(PU_Struct.win, ActiveFolder, "DisplayedWavePopUp")
		Wave/Z SubtractWave=XPSViewGetDisplayedWave(PU_Struct.win, ActiveFolder, "SubtractWavePopUp")
		Wave/Z FluxWave=XPSViewGetDisplayedWave(PU_Struct.win, ActiveFolder:PhotonFlux, "FluxWavePopUp")

		//	The default update function
		FUNCREF ViewNEXAFSProtoUpdate UpdateFunction=$PU_Struct.userdata

		//	Runs the update function
		UpdateFunction(PU_Struct.win, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
		
		//	If the user scrolls through the items in the list too fast, the selection will change before the update has finished, and the selection will not match the displayed data. This will prevent that
		PopupMenu $PU_Struct.ctrlName popmatch=PU_Struct.popStr, win=$PU_Struct.win
	endif
	
	//	Needed, but not used
	Return 0
end



Static Function ViewNEXAFSWavePopupMenu(PU_Struct) : PopupMenuControl
//	Runs the update function if the selection is changed
STRUCT WMPopupAction &PU_Struct

	//	If the selection was changed
	if  (PU_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		PU_Struct.blockReentry=1
		
		//	Finds the active folder	
		DFREF ActiveFolder=XPSViewGetDataFolder(PU_Struct.win, "DataFolderPopUp")

		//	Checks if the data folder exists, if not the wave selections are updated to display the relevant errors
		if (DataFolderRefStatus(ActiveFolder)==0)
			ControlUpdate /W=$PU_Struct.win DisplayedWavePopUp
			ControlUpdate /W=$PU_Struct.win SubtractWavePopUp
			ControlUpdate /W=$PU_Struct.win FluxWavePopUp
		else

			//	Finds the name of the displayed wave
			if (CmpStr(PU_Struct.ctrlName, "DisplayedWavePopUp")==0)
				Wave/Z DisplayedWave=ActiveFolder:$PU_Struct.popStr

				//	Keeps the subtract and flux wave selections, if they are still valid
				ControlInfo/W=$PU_Struct.win SubtractWavePopUp
				PopupMenu SubtractWavePopUp mode=1, popmatch=S_Value, win=$PU_Struct.win
				ControlInfo/W=$PU_Struct.win FluxWavePopUp
				PopupMenu FluxWavePopUp mode=1, popmatch=S_Value, win=$PU_Struct.win
			else
				Wave/Z DisplayedWave=XPSViewGetDisplayedWave(PU_Struct.win, ActiveFolder, "DisplayedWavePopUp")
			endif
		
			//	Finds the name of the wave to subtract
			if (CmpStr(PU_Struct.ctrlName, "SubtractWavePopUp")==0)
				Wave/Z SubtractWave=ActiveFolder:$PU_Struct.popStr
			else
				Wave/Z SubtractWave=XPSViewGetDisplayedWave(PU_Struct.win, ActiveFolder, "SubtractWavePopUp")
			endif
		
			//	Finds the name of the photon flux wave
			if (CmpStr(PU_Struct.ctrlName, "FluxWavePopUp")==0)
				DFREF FluxFolder=ActiveFolder:PhotonFlux
				if (DataFolderRefStatus(FluxFolder)!=0)
					Wave/Z FluxWave=FluxFolder:$PU_Struct.popStr
				else
					Wave/Z FluxWave=$""
				endif
			else
				Wave/Z FluxWave=XPSViewGetDisplayedWave(PU_Struct.win, ActiveFolder:PhotonFlux, "FluxWavePopUp")
			endif
		endif

		//	The default update function
		FUNCREF ViewNEXAFSProtoUpdate UpdateFunction=$PU_Struct.userdata

		//	Runs the update function
		UpdateFunction(PU_Struct.win, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
		
		//	If the user scrolls through the items in the list too fast, the selection will change before the update has finished, and the selection will not match the displayed data. This will prevent that
		PopupMenu $PU_Struct.ctrlName popmatch=PU_Struct.popStr, win=$PU_Struct.win
	endif
	
	//	Needed, but not used
	Return 0
end



Static Function ViewNEXAFSSetVariable(SV_Struct) : SetVariableControl
//	Runs the update function if the value of the control is changed
STRUCT WMSetVariableAction &SV_Struct

	//	If a value was changed
	if ((SV_Struct.eventCode==1) || (SV_Struct.eventCode==2))
	
		//	Prevent multiple instances of the function call to run simultaneously
		SV_Struct.blockReentry=1
		
		//	Prevent the user from changing the value of the control faster than it can update
		SetVariable $SV_Struct.ctrlName disable=2

		//	Finds the active folder	
		DFREF ActiveFolder=XPSViewGetDataFolder(SV_Struct.win, "DataFolderPopUp")

		//	Finds the waves
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(SV_Struct.win, ActiveFolder, "DisplayedWavePopUp")
		Wave/Z SubtractWave=XPSViewGetDisplayedWave(SV_Struct.win, ActiveFolder, "SubtractWavePopUp")
		Wave/Z FluxWave=XPSViewGetDisplayedWave(SV_Struct.win, ActiveFolder:PhotonFlux, "FluxWavePopUp")

		//	The default update function
		FUNCREF ViewNEXAFSProtoUpdate UpdateFunction=$SV_Struct.userdata

		//	Runs the update function
		UpdateFunction(SV_Struct.win, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
		
		//	Prevent the user from changing the value of the control faster than it can update
		SetVariable $SV_Struct.ctrlName disable=0
	endif

	//	This is required for a variable control, but not used for anything
	return 0
end



Static Function ViewNEXAFSSetMult(SV_Struct) : SetVariableControl
//	Runs the update function if the value of the control is changed
STRUCT WMSetVariableAction &SV_Struct

	//	If a value was changed, or the mouse scroll wheel used
	if ((SV_Struct.eventCode==1) || (SV_Struct.eventCode==2) || (SV_Struct.eventCode==4) || (SV_Struct.eventCode==5))
	
		//	Prevent multiple instances of the function call to run simultaneously
		SV_Struct.blockReentry=1
		
		//	Prevent the user from changing the value of the control faster than it can update
		SetVariable $SV_Struct.ctrlName disable=2

		//	Mouse scroll wheel up
		if (SV_Struct.eventCode==4)

			//	Increases the value by 5%. Changing the value with SetVariable will NOT force another run of the function, hence the graph will still have to be updated.
			SetVariable $SV_Struct.ctrlName, value=_NUM:(SV_Struct.dval*1.05), win=$SV_Struct.win
	
		//	Mouse scroll wheel down
		elseif (SV_Struct.eventCode==5)

			//	Decreases the value by ~5%. Changing the value with SetVariable will NOT force another run of the function, hence the graph will still have to be updated.
			SetVariable $SV_Struct.ctrlName, value=_NUM:(SV_Struct.dval/1.05), win=$SV_Struct.win
		endif

		//	Finds the active folder	
		DFREF ActiveFolder=XPSViewGetDataFolder(SV_Struct.win, "DataFolderPopUp")

		//	Finds the waves
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(SV_Struct.win, ActiveFolder, "DisplayedWavePopUp")
		Wave/Z SubtractWave=XPSViewGetDisplayedWave(SV_Struct.win, ActiveFolder, "SubtractWavePopUp")
		Wave/Z FluxWave=XPSViewGetDisplayedWave(SV_Struct.win, ActiveFolder:PhotonFlux, "FluxWavePopUp")

		//	The default update function
		FUNCREF ViewNEXAFSProtoUpdate UpdateFunction=$SV_Struct.userdata

		//	Runs the update function
		UpdateFunction(SV_Struct.win, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
		
		//	Prevent the user from changing the value of the control faster than it can update
		SetVariable $SV_Struct.ctrlName disable=0
	endif

	//	This is required for a variable control, but not used for anything
	return 0
end



Function ViewNEXAFSProtoUpdate(ActiveWindow, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
//	Proto function used to define the update function type used in ViewNEXAFSWavePopupMenu. Proto functions are not alllowed to be static
String ActiveWindow
DFREF ActiveFolder
Wave/Z DisplayedWave, SubtractWave, FluxWave
end



Static Function ViewNEXAFSUpdateGraph(ActiveWindow, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
//	Updates the View NEXAFS window
String ActiveWindow
DFREF ActiveFolder
Wave/Z DisplayedWave, SubtractWave, FluxWave

	//	Finds  the active window instance
	String Suffix=""
	SScanF ActiveWindow, "ViewNEXAFSWindow%s", Suffix
	
	//	Finds the multiplication, shift and tilt values from the active window
	ControlInfo /W=$ActiveWindow SetMult
	Variable Mult=V_Value
	ControlInfo /W=$ActiveWindow SetXShift
	Variable XShift=V_Value
	ControlInfo /W=$ActiveWindow SetYShift
	Variable YShift=V_Value
	ControlInfo /W=$ActiveWindow SetYTilt
	Variable YTilt=V_Value
	
	DFREF ProgramFolder=root:Programming
	
	//	Displays an empty wave if the selected wave is invalid
	if ((WaveExists(DisplayedWave)==0) || (DataFolderRefStatus(ActiveFolder)==0))
	
		//	Displays empty waves for the 2D and 1D spectra
		Make/O ProgramFolder:$("ViewNEXAFSDataWave"+Suffix)={{0}}
		Make/O ProgramFolder:$("ViewNEXAFSSpectrumWave"+Suffix)={0}
		
		//	Deactivates the cursors
		Cursor/K/W=$ActiveWindow A
		Cursor/K/W=$ActiveWindow B
	else
	
		//	Determines if the cursors are placed on the graph. If they are, their positions are saved
		Variable CursorAX=0, CursorBX=0, CursorAY=0, CursorBY=0
		String CursorAInfo=CsrInfo(A, ActiveWindow)
		if (StrLen(CursorAInfo)>0)
			CursorAX=hcsr(A, ActiveWindow)
			CursorAY=vcsr(A, ActiveWindow)
		endif
		String CursorBInfo=CsrInfo(B, ActiveWindow)
		if (StrLen(CursorBInfo)>0)
			CursorBX=hcsr(B, ActiveWindow)
			CursorBY=vcsr(B, ActiveWindow)
		endif
	
		//	The step size of the resulting wave is determined by the displayed wave
		Variable XStep=DimDelta(DisplayedWave, 0)
		Variable YStep=DimDelta(DisplayedWave, 1)
		Variable XOffset=DimOffset(DisplayedWave, 0)
		Variable YOffset=DimOffset(DisplayedWave, 1)

		//	Finds the x-range of the displayed wave
		Variable Temp1=XOffset
		Variable Temp2=Temp1+(DimSize(DisplayedWave, 0)-1)*XStep
		Variable XMin=Min(Temp1, Temp2)
		Variable XMax=Max(Temp1, Temp2)
		
		//	Finds the y-range of the displayed wave
		Temp1=YOffset
		Temp2=Temp1+(DimSize(DisplayedWave, 1)-1)*YStep
		Variable YMin=Min(Temp1, Temp2)
		Variable YMax=Max(Temp1, Temp2)

		if (WaveExists(SubtractWave))

			//	Finds the x-range of the subtract wave
			Temp1=DimOffset(SubtractWave, 0)+XShift
			Temp2=Temp1+(DimSize(SubtractWave, 0)-1)*DimDelta(SubtractWave, 0)
			Variable SubtractXMin=Min(Temp1, Temp2)
			Variable SubtractXMax=Max(Temp1, Temp2)
			
			//	Updates the combined overlap
			XMin=Max(XMin, SubtractXMin)
			XMax=Min(XMax, SubtractXMax)

			//	Finds the y-range of the subtract wave
			Temp1=DimOffset(SubtractWave, 1)+YShift
			Temp2=Temp1+(DimSize(SubtractWave, 1)-1)*DimDelta(SubtractWave, 1)
			Variable SubtractYMin=Min(Temp1, Temp2)
			Variable SubtractYMax=Max(Temp1, Temp2)

			//	Updates the combined overlap
			YMin=Max(YMin, SubtractYMin)
			YMax=Min(YMax, SubtractYMax)
		endif
		
		if (WaveExists(FluxWave))

			//	Finds the x-range of the flux wave
			Temp1=LeftX(FluxWave )
			Temp2=Pnt2X(FluxWave, NumPnts(FluxWave)-1)
			Variable FluxXMin=Min(Temp1, Temp2)
			Variable FluxXMax=Max(Temp1, Temp2)

			//	Updates the combined overlap
			XMin=Max(XMin, FluxXMin)
			XMax=Min(XMax, FluxXMax)
		endif
		
		//	Reduces the overlap to the nearest x and y values of the displayed wave within the overlapping region
		if (XStep>0)
			XMin=XOffset+XStep*Ceil((XMin-XOffset)/XStep)
			XMax=XOffset+XStep*Floor((XMax-XOffset)/XStep)
		else
			XMin=XOffset+XStep*Floor((XMin-XOffset)/XStep)
			XMax=XOffset+XStep*Ceil((XMax-XOffset)/XStep)
		endif
		if (YStep>0)
			YMin=YOffset+YStep*Ceil((YMin-YOffset)/YStep)
			YMax=YOffset+YStep*Floor((YMax-YOffset)/YStep)
		else
			YMin=YOffset+YStep*Floor((YMin-YOffset)/YStep)
			YMax=YOffset+YStep*Ceil((YMax-YOffset)/YStep)
		endif


		if ((XMin>=XMax) || (YMin>=YMax))
		
			//	Displays empty waves for the 2D and 1D spectra
			Make/O ProgramFolder:$("ViewNEXAFSDataWave"+Suffix)={{0}}
			Make/O ProgramFolder:$("ViewNEXAFSSpectrumWave"+Suffix)={0}
			
			//	Deactivates the cursors
			Cursor/K/W=$ActiveWindow A
			Cursor/K/W=$ActiveWindow B
		
			ErrorMessage("The selected waves have no overlap!")
		else
		
			//	Reduces the range of the datawave to the range of the combined overlap.
			Duplicate/O/R=(XMin, XMax)(YMin, YMax) DisplayedWave, ProgramFolder:$("ViewNEXAFSDataWave"+Suffix)/WAVE=DataWave
		
			//	Calculates the new start and end values for the reduced data wave
			Variable InterpolateXStart=DimOffset(DataWave, 0)-XShift
			Variable InterpolateXEnd=InterpolateXStart+(DimSize(DataWave, 0)-0.5)*XStep			//	the 0.5 is used to prevent clipping the last datapoint in the interpolated waves
			Variable InterpolateYStart=DimOffset(DataWave, 1)-YShift
			Variable InterpolateYEnd=InterpolateYStart+(DimSize(DataWave, 1)-0.5)*YStep		//	the 0.5 is used to prevent clipping the last datapoint in the interpolated waves
		
			//	Creates the waves used to tilt the subtract wave
			Duplicate/FREE/O/R=[][0] DataWave Temp1DTiltXWave
			FastOP Temp1DTiltXWave=1
			Duplicate/FREE/O/R=[0][] DataWave Temp1DTiltYWave
		
			DFREF TempFolder=root:Programming:Temp

			if (WaveExists(SubtractWave))
		
				//	Calculates the value of the subtract wave at the x and y positions of the data wave. TempSubtractWave = SubtractWave(x)(y) doesn't interpolate for 2D waves as it does for 1D waves. The nearest data point is chosen instead
				ImageInterpolate /DEST=TempFolder:TempSubtractWave /S={InterpolateXStart,XStep,InterpolateXEnd,InterpolateYStart,YStep,InterpolateYEnd} Bilinear SubtractWave
				Wave TempSubtractWave=TempFolder:TempSubtractWave

				//	Calculates the tilt wave
				Variable y0=DimOffset(DataWave, 1)
				Variable yd=DimDelta(DataWave,1)
				Variable yc=y0+0.5*(DimSize(DataWave, 1)-1)*yd
				Variable y0c=y0-yc
				Variable a=Mult*YTilt*yd
				Variable b=Mult*YTilt*y0c+Mult
				Temp1DTiltYWave=a*q+b
			else
				//	Subtracts a wave equal to zero with no tilt, if no subtract wave exists
				Duplicate/O DataWave TempFolder:TempSubtractWave
				Wave TempSubtractWave=TempFolder:TempSubtractWave
				FastOP TempSubtractWave=0
				FastOP Temp1DTiltYWave=1
			endif
		
			//	Calculates the values of the photon flux wave at the x and y positions of the data wave. If the photon flux wave does not exists, a wave equal to 1 is used instead
			Duplicate/FREE/O/R=[][0] DataWave Temp1DFluxXWave
			Duplicate/FREE/O/R=[0][] DataWave Temp1DFluxYWave
			FastOP Temp1DFluxYWave=1
			if (WaveExists(FluxWave))
				Temp1DFluxXWave[]=FluxWave(x)
			else
				FastOP Temp1DFluxXWave=1
			endif

			//	Calculates the resulting 2D NEXAFS image			
			Duplicate/FREE/O DataWave, TempDataWave
			MatrixOP /NTHR=0/O/S DataWave=(TempDataWave-(Temp1DTiltXWave x Temp1DTiltYWave)*TempSubtractWave)/(Temp1DFluxXWave x Temp1DFluxYWave)

			//	Removes any unit labels that would otherwise result in double axis labels
			SetScale/P x, DimOffset(DataWave, 0), DimDelta(DataWave, 0), "", DataWave
			SetScale/P y, DimOffset(DataWave, 1), DimDelta(DataWave, 1), "", DataWave
		
			//	Flattens the 2D spectrum to a 1D spectrum, limited by the cursors
			if ((StrLen(CursorAInfo)>0) && (StrLen(CursorBInfo)>0))
				Duplicate/O/FREE/R=(CursorAX, CursorBX)(CursorAY, CursorBY) DataWave, TempDataWave
				MatrixOP /NTHR=0/O ProgramFolder:$("ViewNEXAFSSpectrumWave"+Suffix)=SumRows(TempDataWave)
				Wave SpectrumWave=ProgramFolder:$("ViewNEXAFSSpectrumWave"+Suffix)
				SetScale/P x, DimOffset(TempDataWave, 0), DimDelta(TempDataWave, 0), "", SpectrumWave
			else
				MatrixOP /NTHR=0/O ProgramFolder:$("ViewNEXAFSSpectrumWave"+Suffix)=SumRows(DataWave)
				Wave SpectrumWave=ProgramFolder:$("ViewNEXAFSSpectrumWave"+Suffix)
				SetScale/P x, DimOffset(DataWave, 0), DimDelta(DataWave, 0), "", SpectrumWave
			endif
	
			//	Moves the cursors back to where they were before the axis scaling changed.
			if (StrLen(CursorAInfo)>0)
				Cursor/I/W=$ActiveWindow A $NameOfWave(DataWave) CursorAX, CursorAY
			endif
			if (StrLen(CursorBInfo)>0)
				Cursor/I/W=$ActiveWindow B $NameOfWave(DataWave) CursorBX, CursorBY
			endif
		endif
		
		//	Cleans up the temporary waves
		KillWavesInFolder(TempFolder)
	endif
end



Static Function ViewNEXAFSNewWin(B_Struct) : ButtonControl
//	Creates another instances of the NEXAFS image window
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
	
		//	Copies the selected data folder to the new window
		ControlInfo /W=$B_Struct.win DataFolderPopUp
		String FolderName=S_Value

		//	Copies the selected wave to the new window
		ControlInfo /W=$B_Struct.win DisplayedWavePopUp
		String DisplayedName=S_Value

		//	Copies the wave to subtract to the new window
		ControlInfo /W=$B_Struct.win SubtractWavePopUp
		String SubtractName=S_Value

		//	Copies the wave to subtract to the new window
		ControlInfo /W=$B_Struct.win FluxWavePopUp
		String FluxName=S_Value

		//	Creates another Raw NEXAFS window and saves the name of the window created
		String NewWindow=RawNEXAFSNewWinName("ViewNEXAFSWindow")
		ViewNEXAFSCreateWindow(FolderName, DisplayedName, SubtractName, FluxName, NewWindow, 1)

		//	Copies the selected colour scheme to the new window
		ControlInfo /W=$B_Struct.win ColourSchemePopUp
		PopupMenu ColourSchemePopUp popmatch=S_Value, win=$NewWindow
		ModifyImage/W=$NewWindow '' ctab= {*,*,$S_Value,0}	//	' ' indicates all images in graph
		
		//	Copies the variables used to subtract a reference spectrum
		ControlInfo /W=$B_Struct.win SetMult
		SetVariable SetMult value=_NUM:(V_Value), win=$NewWindow
		ControlInfo /W=$B_Struct.win SetXShift
		SetVariable SetXShift value=_NUM:(V_Value), win=$NewWindow
		ControlInfo /W=$B_Struct.win SetYShift
		SetVariable SetYShift value=_NUM:(V_Value), win=$NewWindow
		ControlInfo /W=$B_Struct.win SetYTilt
		SetVariable SetYTilt value=_NUM:(V_Value), win=$NewWindow

		DFREF ActiveFolder=XPSViewGetDataFolder(NewWindow, "DataFolderPopUp")
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(NewWindow, ActiveFolder, "DisplayedWavePopUp")
		Wave/Z SubtractWave=XPSViewGetDisplayedWave(NewWindow, ActiveFolder, "SubtractWavePopUp")
		Wave/Z FluxWave=XPSViewGetDisplayedWave(NewWindow, ActiveFolder:PhotonFlux, "FluxWavePopUp")
		
		//	Updates the display
		ViewNEXAFSUpdateGraph(NewWindow, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
	endif
	Return 0
end



Static Function/S ViewNEXAFSPopUpFluxWaveList(ActiveWindow)
//	Returns a list of all flux waves that overlap with the displayed wave
String ActiveWindow
String ReturnString="", FluxName=""
Variable FluxA=0, FluxB=0

	//	Finds the selected group or folder in the active window
	ControlInfo /W=$ActiveWindow DataFolderPopUp
	DFREF ActiveFolder=$S_Value
	
	if (DataFolderRefStatus(ActiveFolder)==0)
	
		//	Returns data folder does not exist, if the data folder does not exist
		ReturnString="Data folder does not exist;"
	else

		//	Finds the displayed wave
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
		
		if (WaveExists(DisplayedWave))

			//	Finds the x-range of the displayed wave
			Variable XA=DimOffset(DisplayedWave, 0)
			Variable XB=XA+(DimSize(DisplayedWave, 0)-1)*DimDelta(DisplayedWave, 0)
			Variable XMin=Min(XA, XB)
			Variable XMax=Max(XA, XB)

			//	Finds the photon flux folder
			DFREF FluxFolder=ActiveFolder:PhotonFlux
			if (DataFolderRefStatus(FluxFolder)==0)

				//	Returns data folder does not exist, if the data folder does not exist
				ReturnString="Data folder does not exist;"
			else
	
				//	Creates an alpha-numerically sorted list of all one-dimensional, numeric waves in the folder
				String FluxWaveList=XPSViewListWaves(FluxFolder, 1)
	
				//	Calculates the number of waves in the image wave list
				Variable NumberOfWaves=ItemsInList(FluxWaveList, ";")

				//	Removes waves that do not overlap with the x-range of the displayed wave
				Variable a=0, b=0, i=0
				for (i=0; i<NumberOfWaves; i+=1)
					b=StrSearch(FluxWaveList, ";", a)
					FluxName=FluxWaveList[a, b-1]

					//	Finds the x-range of the flux wave
					Wave FluxWave=FluxFolder:$FluxName
					FluxA=leftx(FluxWave )
					FluxB=pnt2x(FluxWave, NumPnts(FluxWave)-1)
				
					//	Checks that the range of the flux wave overlaps with the displayed wave's
					if (Limit(FluxA, XMin, XMax)!=Limit(FluxB, XMin, XMax))
						ReturnString+=FluxName+";"
					endif
					a=b+1
				endfor
			endif
		endif
	
		//	If no valid waves exist "No waves in folder" is returned
		if (CmpStr(ReturnString, "")==0)
			ReturnString="No waves in folder;"
		endif
	endif
	Return ReturnString
end



Static Function/S ViewNEXAFSPopUpSubtractWaveList(ActiveWindow)
//	Returns a list of all image waves which overlap with the displayed image
String ActiveWindow
String ReturnString="", SubtractName=""
Variable SubtractX1=0, SubtractX2=0, SubtractY1=0, SubtractY2=0

	//	Finds the selected group or folder in the active window
	ControlInfo /W=$ActiveWindow DataFolderPopUp
	DFREF ActiveFolder=$S_Value
	
	if (DataFolderRefStatus(ActiveFolder)==0)
	
		//	Returns data folder does not exist, if the data folder does not exist
		ReturnString="Data folder does not exist;"
	else

		//	Finds the displayed wave
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ActiveWindow, ActiveFolder, "DisplayedWavePopUp")
		
		if (WaveExists(DisplayedWave))

			//	Finds the x-range of the displayed wave
			Variable Temp1=DimOffset(DisplayedWave, 0)
			Variable Temp2=Temp1+(DimSize(DisplayedWave, 0)-1)*DimDelta(DisplayedWave, 0)
			Variable XMin=Min(Temp1, Temp2)
			Variable XMax=Max(Temp1, Temp2)
			
			Temp1=DimOffset(DisplayedWave, 1)
			Temp2=Temp1+(DimSize(DisplayedWave, 1)-1)*DimDelta(DisplayedWave, 1)
			Variable YMin=Min(Temp1, Temp2)
			Variable YMax=Max(Temp1, Temp2)

			//	Creates an alpha-numerically sorted list of all two-dimensional, numeric waves in the active folder
			String SubtractWaveList=XPSViewListWaves(ActiveFolder, 2)
	
			//	Calculates the number of waves in the image wave list
			Variable NumberOfWaves=ItemsInList(SubtractWaveList, ";")

			//	Removes waves that do not overlap with the x and y-range of the displayed wave
			Variable a=0, b=0, i=0
			for (i=0; i<NumberOfWaves; i+=1)
			
				//	Finds the next wave in the list
				b=StrSearch(SubtractWaveList, ";", a)
				SubtractName=SubtractWaveList[a, b-1]
				Wave SubtractWave=ActiveFolder:$SubtractName

				//	Finds the x and y-range of the subtract wave
				SubtractX1=DimOffset(SubtractWave, 0)
				SubtractX2=SubtractX1+(DimSize(SubtractWave, 0)-1)*DimDelta(SubtractWave, 0)
				SubtractY1=DimOffset(SubtractWave, 1)
				SubtractY2=SubtractY1+(DimSize(SubtractWave, 1)-1)*DimDelta(SubtractWave, 1)
				
				//	Checks that the range of the flux wave overlaps with the displayed wave's
				if ((Limit(SubtractX1, XMin, XMax)!=Limit(SubtractX2, XMin, XMax)) && (Limit(SubtractY1, YMin, YMax)!=Limit(SubtractY2, YMin, YMax)))
					ReturnString+=SubtractName+";"
				endif
				a=b+1
			endfor
		endif
	
		//	If no valid waves exist "No waves in folder" is returned
		if (CmpStr(ReturnString, "")==0)
			ReturnString="No waves in folder;"
		endif
	endif
	Return ReturnString
end



Static Function ViewNEXAFSSaveImage(B_Struct) : ButtonControl
//	Creates a copy of the displayed image
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)

		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
	
		//	Suggests the selected folder as the folder to save the new image in
		DFREF ActiveFolder=XPSViewGetDataFolder(B_Struct.win, "DataFolderPopUp")
		
		//	Finds the selected wave
		Wave/Z DisplayedWave=XPSViewGetDisplayedWave(B_Struct.win, ActiveFolder, "DisplayedWavePopUp")
		
		if ((DataFolderRefStatus(ActiveFolder)!=0) && WaveExists(DisplayedWave))

			String DisplayedWaveName=NameOfWave(DisplayedWave)

			//	Finds the first unused wave name of the type NewName_##
			Variable i=0
			Wave/Z NewWave=ActiveFolder:$(DisplayedWaveName+"_0")
			for (i=1; WaveExists(NewWave); i+=1)
				Wave/Z NewWave=ActiveFolder:$(DisplayedWaveName+"_"+Num2iStr(i))
			endfor
		
			//	Updates the suggested wave name
			String NewName=DisplayedWaveName+"_"+Num2iStr(i)
			
			//	Creates a save wave dialog where the OK button will execute XPSViewNEXAFSSaveImageOK
			XPSViewSaveWaveDialog("Save copy of image as", GetDataFolder(1, ActiveFolder), NewName, "EccentricXPS#ViewNEXAFSSaveImageOK", "", B_Struct.win)
		else
			ErrorMessage("Selected wave does not exist!")		
		endif
	endif
	Return 0
end



Static Function ViewNEXAFSSaveImageOK(SaveFolder, SaveName, ActiveWindow)
//	Saves a copy of the resulting image displayed in the active View NEXAFS window
DFREF SaveFolder
String SaveName, ActiveWindow
DFREF ProgramFolder=root:Programming
String Suffix=""

	//	Finds  the active window instance
	SScanF ActiveWindow, "ViewNEXAFSWindow%s", Suffix
	
	//	Creates the copy of the displayed image
	Duplicate/O ProgramFolder:$("ViewNEXAFSDataWave"+Suffix) SaveFolder:$SaveName
end



Static Function ViewNEXAFSSpectrum(B_Struct) : ButtonControl
//	Displays the resulting 1D NEXAFS spectrum
STRUCT WMButtonAction &B_Struct
String ActiveWindow="", Suffix=""

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1

		//	Finds the name of the active image window
		ActiveWindow=B_Struct.win
		
		//	Finds  the active window instance
		SScanF ActiveWindow, "ViewNEXAFSWindow%s", Suffix
		Variable WindowInstance=Str2Num(Suffix)
		
		//	Finds the name of the spectrum window
		String ActiveSpectrumWindow="ViewNEXAFSSpectrumWindow"+Suffix
		
		//	Variable used to make the graph appear the same on different screens
		Variable n=120/ScreenResolution
		
		//	Bings the spectrum window to the front or creates it if it does not exist
		DoWindow /F $ActiveSpectrumWindow
		if (V_Flag==0)
		
			//	Creates the graph to display the 1D NEXAFS spectrum
			DFREF ProgramFolder=root:Programming
			Wave SpectrumWave=ProgramFolder:$("ViewNEXAFSSpectrumWave"+Suffix)
			DoWindow /K $ActiveSpectrumWindow
			Display /W=(15+365*n+WindowInstance*50, 40+WindowInstance*10, 15+765*n+WindowInstance*50, 40+240*n+WindowInstance*10) /K=1 /N=$ActiveSpectrumWindow SpectrumWave
			ModifyGraph /W=$ActiveSpectrumWindow margin(top)=20*n
			Label /W=$ActiveSpectrumWindow bottom "Photon Energy (eV)"
			Label /W=$ActiveSpectrumWindow left "Intensity (arb unit)"
			
			//	Creates a button to save the displayed spectrum
			Button SaveButton proc=EccentricXPS#ViewNEXAFSSaveSpectrum, size={70, 20}, title="Save", win=$ActiveSpectrumWindow, help={"Saves the displayed spectrum"}
			//	Creates a button to refresh the displayed spectrum
			Button RefreshButton proc=EccentricXPS#ViewNEXAFSRefreshSpectrum, size={70, 20}, title="Refresh", win=$ActiveSpectrumWindow, help={"Refreshes the displayed spectrum"}
		endif
	endif
	return 0
end



Static Function ViewNEXAFSRefreshSpectrum(B_Struct) : ButtonControl
//	Refreshes the View NEXAFS image and spectrum
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevent multiple instance of the function call to run simultaneously
		B_Struct.blockReentry=1
		
		//	Finds the active window
		String ActiveSpectrumWindow=B_Struct.win
		
		//	Finds  the active window instance
		String Suffix=""
		SScanF ActiveSpectrumWindow, "ViewNEXAFSSpectrumWindow%s", Suffix
		
		//	Finds the name of the parent NEXAFS image window
		String ImageWindowName="ViewNEXAFSWindow"+Suffix
		
		DoWindow $ImageWindowName
		if (V_Flag==0)
			ErrorMessage("The corresponding image window could not be found!")		
		else
	
			//	Finds the selections in the image window
			DFREF ActiveFolder=XPSViewGetDataFolder(ImageWindowName, "DataFolderPopUp")
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ImageWindowName, ActiveFolder, "DisplayedWavePopUp")
			Wave/Z SubtractWave=XPSViewGetDisplayedWave(ImageWindowName, ActiveFolder, "SubtractWavePopUp")
			Wave/Z FluxWave=XPSViewGetDisplayedWave(ImageWindowName, ActiveFolder:PhotonFlux, "FluxWavePopUp")
		
			//	Updates the display
			ViewNEXAFSUpdateGraph(ImageWindowName, ActiveFolder, DisplayedWave, SubtractWave, FluxWave)
		endif
	endif
	Return 0
end



Static Function ViewNEXAFSSaveSpectrum(B_Struct) : ButtonControl
//	Makes a copy of the selected wave
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Prevents multiple instances of the function to run simultaneously
		B_Struct.blockReentry=1
	
		//	Finds the active window
		String ActiveSpectrumWindow=B_Struct.win
		
		//	Finds  the active window instance
		String Suffix=""
		SScanF ActiveSpectrumWindow, "ViewNEXAFSSpectrumWindow%s", Suffix
		
		//	Finds the name of the parent NEXAFS image window
		String ImageWindowName="ViewNEXAFSWindow"+Suffix
		
		DoWindow $ImageWindowName
		if (V_Flag==0)
			ErrorMessage("The corresponding image window could not be found!")		
		else
		
			//	Finds the selected folder in the parent NEXAFS image window
			DFREF ActiveFolder=XPSViewGetDataFolder(ImageWindowName, "DataFolderPopUp")
		
			//	Finds the selected wave in the parent NEXAFS image window
			Wave/Z DisplayedWave=XPSViewGetDisplayedWave(ImageWindowName, ActiveFolder, "DisplayedWavePopUp")
		
			if ((DataFolderRefStatus(ActiveFolder)!=0) && WaveExists(DisplayedWave))

				String DisplayedWaveName=NameOfWave(DisplayedWave)

				//	If the image is located in root:XXX:Images the spectrum will be saved in root:XXX:Spectra if such a folder exiists
				DFREF ParentFolder=$(GetDataFolder(1, ActiveFolder)+":")
				DFREF SaveFolder=ParentFolder:Spectra
				String SaveName=DisplayedWaveName
			
				if ((DataFolderRefStatus(ParentFolder)==0) || (DataFolderRefStatus(SaveFolder)==0))
					SaveFolder=ActiveFolder
					SaveName+="_"
				endif

				//	Creates a save wave dialog where the OK button will execute XPSViewNEXAFSSaveImageOK
				XPSViewSaveWaveDialog("Save spectrum as", GetDataFolder(1, SaveFolder), SaveName, "EccentricXPS#ViewNEXAFSSaveSpectrumOK", "", ActiveSpectrumWindow)
			else
				ErrorMessage("Selected wave does not exist!")
			endif
		endif
	endif
	
	//	This is required for a button control, but not used for anything
	return 0
end



Static Function ViewNEXAFSSaveSpectrumOK(SaveFolder, SaveName, ActiveWindow)
//	Makes a copy of the selected wave
DFREF SaveFolder
String SaveName, ActiveWindow
DFREF ProgramFolder=root:Programming
String Suffix=""

	//	Finds  the active window instance
	SScanF ActiveWindow, "ViewNEXAFSSpectrumWindow%s", Suffix

	//	Creates the copy of the displayed image
	Duplicate/O ProgramFolder:$("ViewNEXAFSSpectrumWave"+Suffix) SaveFolder:$SaveName
end





//     -----<<<<<     Coverage     >>>>>-----
//	This section contains the functions used by the Coverage menu items

Static Function XPSCoverageAB()
//	Calculates the coverage or mean free path of a two-component system
String ActiveWindow="XPSCoverageABPanel"

	DoWindow /F $ActiveWindow
	if (V_flag==0)

		//	Creates the XPSCoveragePanel
		DoWindow /K $ActiveWindow
		NewPanel /K=1 /N=$ActiveWindow /W=(350, 300, 715, 475) as "Coverage AB"

		SetVariable InfiniteA bodyWidth=85, pos={190,10}, title="Infinite Adsorbate Signal: ", value=_NUM:0, win=$ActiveWindow
		SetVariable InfiniteB bodyWidth=85, pos={190,30}, title="Infinite Substrate Signal: ", value=_NUM:0, win=$ActiveWindow

		SetVariable SignalA bodyWidth=85, pos={190,60}, title="Adsorbate Signal: ", value=_NUM:0, win=$ActiveWindow
		SetVariable SignalB bodyWidth=85, pos={190,80}, title="Substrate Signal: ", value=_NUM:0, win=$ActiveWindow

		SetVariable MeanFreePath bodyWidth=85, pos={190,110}, title="Mean Free Path (ML): ", value=_NUM:0, win=$ActiveWindow
		Button CalculateMeanFreePath pos={260,108}, proc=EccentricXPS#XPSCoverageABCalculate, size={70,20}, title="Calculate", win=$ActiveWindow

		SetVariable CoverageA bodyWidth=85, pos={190,140}, title="Adsorbate Coverage (ML): ", value=_NUM:0, win=$ActiveWindow
		Button CalculateCoverage pos={260,138}, proc=EccentricXPS#XPSCoverageABCalculate, size={70,20}, title="Calculate", win=$ActiveWindow
	endif
end



Static Function XPSCoverageABCalculate(B_Struct) : ButtonControl
//	Calculates the coverage or mean free path of a two-component system
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Reads back the values of the controls
		ControlInfo/W=$B_Struct.win InfiniteA
		Variable InfiniteA=V_Value

		ControlInfo/W=$B_Struct.win InfiniteB
		Variable InfiniteB=V_Value

		ControlInfo/W=$B_Struct.win SignalA
		Variable SignalA=V_Value

		ControlInfo/W=$B_Struct.win SignalB
		Variable SignalB=V_Value

		ControlInfo/W=$B_Struct.win MeanFreePath
		Variable MeanFreePath=V_Value

		ControlInfo/W=$B_Struct.win CoverageA
		Variable CoverageA=V_Value

		Variable InfiniteAB=InfiniteA/InfiniteB
		Variable SignalAB=SignalA/SignalB
	
		if (CmpStr("CalculateMeanFreePath", B_Struct.ctrlName)==0)
	
			//	Calculates the mean free path and updates the control
			MeanFreePath=-CoverageA/ln(InfiniteAB/(InfiniteAB+SignalAB))
			SetVariable MeanFreePath value=_NUM:(MeanFreePath), win=$B_Struct.win
		else

			//	Calculates the coverage and updates the control
			CoverageA=-MeanFreePath*ln(InfiniteAB/(InfiniteAB+SignalAB))
			SetVariable CoverageA value=_NUM:(CoverageA), win=$B_Struct.win
		endif
	endif
	
	Return 0
end



Function XPSCoverageABC()
//	Calculates the coverage or mean free path of a three-component system
String ActiveWindow="XPSCoverageABCPanel"

	DoWindow /F $ActiveWindow
	if (V_Flag==0)
		//	Creates the XPSCoverageMgOPanel
		DoWindow /K $ActiveWindow
		NewPanel /K=1 /N=$ActiveWindow /W=(350, 300, 715, 525) as "Coverage ABC"

		SetVariable InfiniteA bodyWidth=85, pos={190,50}, title="Infinite C 1s Signal: ", value=_NUM:0, win=$ActiveWindow
		SetVariable InfiniteB bodyWidth=85, pos={190,30}, title="Infinite Mg KLL Signal: ", value=_NUM:0, win=$ActiveWindow
		SetVariable InfiniteC bodyWidth=85, pos={190,10}, title="Infinite Ag 3d Signal: ", value=_NUM:0, win=$ActiveWindow

		SetVariable SignalA bodyWidth=85, pos={190,120}, title="C 1s Signal: ", value=_NUM:0, win=$ActiveWindow
		SetVariable SignalB bodyWidth=85, pos={190,100}, title="Mg KLL Signal: ", value=_NUM:0, win=$ActiveWindow
		SetVariable SignalC bodyWidth=85, pos={190,80}, title="Ag 3d Signal: ", value=_NUM:0, win=$ActiveWindow

		SetVariable MeanFreePath bodyWidth=85, pos={190,150}, title="Mean Free Path (ML): ", value=_NUM:0, win=$ActiveWindow

		SetVariable CoverageA bodyWidth=85, disable=0, pos={190,180}, title="Carbon Coverage (ML): ", value=_NUM:0, win=$ActiveWindow
		SetVariable CoverageB bodyWidth=85, disable=0, pos={190, 200}, title="Thickness of MgO (ML): ", value=_NUM:0, win=$ActiveWindow
		Button CalculateCoverage pos={260,187}, proc=EccentricXPS#XPSCoverageABCCalculate, size={70,20}, title="Calculate", win=$ActiveWindow
	EndIf
end



Function XPSCoverageABCCalculate(B_Struct) : ButtonControl
//	Calculates the coverage or mean free path of a three-component system
STRUCT WMButtonAction &B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1

		//	Reads back the values of the controls
		ControlInfo/W=$B_Struct.win InfiniteA
		Variable InfiniteA=V_Value

		ControlInfo/W=$B_Struct.win InfiniteB
		Variable InfiniteB=V_Value

		ControlInfo/W=$B_Struct.win InfiniteC
		Variable InfiniteC=V_Value

		ControlInfo/W=$B_Struct.win SignalA
		Variable SignalA=V_Value

		ControlInfo/W=$B_Struct.win SignalB
		Variable SignalB=V_Value

		ControlInfo/W=$B_Struct.win SignalC
		Variable SignalC=V_Value

		ControlInfo/W=$B_Struct.win MeanFreePath
		Variable MeanFreePath=V_Value

		//	Calculates the coverages
		Variable Frac1 = InfiniteB * InfiniteC * SignalA
		Variable Frac2 = InfiniteA * InfiniteC * SignalB + InfiniteA * InfiniteB * SignalC
		Variable Arg = 1 + Frac1 / Frac2

		Variable CoverageA = MeanFreePath * ln(Arg)
		Variable CoverageB = -MeanFreePath * (ln(InfiniteA*SignalC/(InfiniteC*SignalA)) + ln(1 - exp(-CoverageA/MeanFreePath))) - CoverageA
	
		//	Updates the controls
		SetVariable CoverageA value=_NUM:(CoverageA), win=$B_Struct.win
		SetVariable CoverageB value=_NUM:(CoverageB), win=$B_Struct.win
	endif
	
	Return 0
end





//     -----<<<<<     Shared XPS and NEXAFS Functions     >>>>>-----
//	This section contains general functions that are shared between several of the XPS and NEXAFS menu items
//	+ miscellaneous functions

Static Function XPSViewCallXPSTools(Folder, ActiveWave)
//	Opens the specified wave in Martin Schmid's XPS Tools for peak fitting, see http://www.igorexchange.com/project/XPStools
//	XPS Tools was clearly not designed to be called by another procedure in this way, and I only expect this function to work for XPST version 1.1, which is the version I have used for testing..
DFREF Folder
Wave ActiveWave

	//	Checks if the XPST Fit Assistant window is open (I have no idea why it's called CursorPanel)
	DoWindow CursorPanel
	if (V_flag!=0)
		ErrorMessage("Close down XPS Tools first")
	else
	
		//	Checks if the XPST functions exist
		#if ((Exists("ProcGlobal#LaunchCursorPanel")==6) && (Exists("ProcGlobal#CallSTNewProjectPanel")==6))

			//	XPST only work in the default data folder
			SetDataFolder Folder

			//	Creates the main XPST panel
			ProcGlobal#LaunchCursorPanel()
		
			//	Open the new project panel, by simulating that the New Fit Project button has been clicked
			ProcGlobal#CallSTNewProjectPanel("LoadNewBtn")
		
			//	Sets the data wave shown in the panel to ActiveWave. This is purely visual.
			ControlUpdate /W=STNewProjectPanel WavePop
			PopupMenu WavePop popMatch=NameOfWave(ActiveWave), win=STNewProjectPanel
		
			//	Sets the actual data wave used to ActiveWave
			SVAR value = root:STFitAssVar:PR_nameWorkWave    
			value = NameOfWave(ActiveWave)
		#else
			ErrorMessage("The XPST procedure cannot be found")		
		#endif
	endif
end



Static Function XPSOpenNoteBook(NoteBookName)
//	Brings the notebook window to the front or, if it doesn't exist, creates it
String NoteBookName
Variable n=120/ScreenResolution
	DoWindow/F $NoteBookName
	if (V_Flag==0)
		NewNoteBook /F=0 /K=3 /N=$NoteBookName /W=(700, 45, 45+900*n, 45+500*n)
		Notebook $NoteBookName, setData="Notebook Example\r________________\r\r20141104_1  :  Clean Cu(111)\r\r20141104_2  :  0.8 ML 2HTPP / Cu(111) at 300 K\r\r20141104_3  :  Heated to 400 K for 5 min\r\r20141104_4  :  Heated to 500 K for 5 min\r\r20141110_5  :  Heated to 600 K for 5 min\r\r20141110_6  :  0.7 ML CuTPP / Cu(111) at 300 K"
	endif
end



Static Function XPSMinimizeGraphs(ListOfWindows)
//	Minimizes all windows in the list
String ListOfWindows
Variable n=0, i=0
String WindowName=""
	n=ItemsInList(ListOfWindows, ";")
	for (i=0; i<n; i+=1)
		WindowName=StringFromList(i, ListOfWindows, ";")
		DoWindow $WindowName
		if (V_flag==1)
			MoveWindow /W=$WindowName 0,0,0,0
		endif
	endfor
end



Static Function XPSRestoreGraphs(ListOfWindows)
//	Restores all windows in the list
String ListOfWindows
Variable n=0, i=0
String WindowName=""
	n=ItemsInList(ListOfWindows, ";")
	for (i=0; i<n; i+=1)
		WindowName=StringFromList(i, ListOfWindows, ";")
		DoWindow/F $WindowName
	endfor
end



Static Function KillWavesInFolder(Path)
//	Kills all waves in the specified folder, if it exists
DFREF Path
DFREF CurrentFolder=GetDataFolderDFR()
	if (DataFolderRefStatus(Path)!=0)
		SetDataFolder Path
		KillWaves/A/Z
		SetDataFolder CurrentFolder
	endif
end



Static Function/S XPSViewPath(Path)
//	Changes root:Folder1:Folder2:Wave1 to root:'Folder1':'Folder2':'Wave1', which allows liberal names that start with numbers or contain blank spaces
String Path
	Path=ReplaceString("''", "'"+ReplaceString(":", Path, "':'")+"'", "'")
	if (CmpStr(Path[0, 5], "'root'")==0)
		Path="root"+Path[6, StrLen(Path)-1]
	endif
	Return Path
end



Static Function XPSViewSaveWaveDialog(Title, SaveInFolder, NewName, FunctionName, CleanUpFunctionName, ActiveWindow)
//	Prompts the user for the location and name of the new wave
String Title, SaveInFolder, NewName, FunctionName, CleanUpFunctionName, ActiveWindow

	//	Creates the dialog panel
	DoWindow /K XPSViewSaveDialogWindow
	NewPanel /K=1 /W=(200, 200, 830, 290) /N=XPSViewSaveDialogWindow as Title
	
	//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
	SetWindow XPSViewSaveDialogWindow, hook(KillHook)=XPSViewSaveDialogHookFunc, userdata(Win)=ActiveWindow
	SetWindow XPSViewSaveDialogWindow, userdata(CleanUp)=CleanUpFunctionName

	//	Selects the data folder to save in, listing root: and all it's subfolders with the exception of root:Programming and it's subfolders
	PopupMenu XPSViewSaveDialogFolder bodyWidth=300, Disable=0, mode=1, pos={260, 10}, noproc, value=EccentricXPS#XPSViewDataFoldersList(root:, root:Programming, "SavedFits"), win=XPSViewSaveDialogWindow
	PopupMenu XPSViewSaveDialogFolder popmatch=SaveInFolder, win=XPSViewSaveDialogWindow

	//	Selects the wave name
	SetVariable XPSViewSaveDialogName bodyWidth=300, disable=0, pos={570,12},  title=" ", value=_STR:NewName, win=XPSViewSaveDialogWindow

	//	OK and Cancel buttons
	Button XPSViewSaveDialogCancel proc=EccentricXPS#CloseWindow, pos={205,50}, size={100, 20}, title="Cancel", win=XPSViewSaveDialogWindow
	Button XPSViewSaveDialogOK proc=EccentricXPS#XPSViewSaveWaveOK, pos={325,50}, size={100, 20}, title="OK", userdata(Func)=FunctionName, win=XPSViewSaveDialogWindow
end


Function XPSViewSaveDialogHookFunc(s)
//	The hook function for the View Images window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1

			//	Executes the clean up function specified in the userdata, but delays the execution until the window has been killed
			String CleanUpFunction=GetUserData(s.winName, "", "CleanUp")
			if (StrLen(CleanUpFunction)>0)
				Execute/P/Q CleanUpFunction+"(\""+GetUserData(s.winName, "", "Win")+"\")"
			endif
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Static Function XPSViewSaveWaveOK(B_Struct) : ButtonControl
//	Checks if the name of the new wave is in conflict with an existing object
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Ignores further calls to the button procedure until this one has finished
		B_Struct.blockReentry=1
		
		//	Finds the function and window specified in the userdata of the button
		String FuncUserData=GetUserData(B_Struct.win, B_Struct.ctrlName, "Func")
		String WinUserData=GetUserData(B_Struct.win, "", "Win")
		String CleanUpUserData=GetUserData(B_Struct.win, "", "CleanUp")
		
		//	Finds the selected folder
		ControlInfo /W=XPSViewSaveDialogWindow XPSViewSaveDialogFolder
		String ExistingFolder=S_Value
		
		//	Finds the selected new wavename
		ControlInfo /W=XPSViewSaveDialogWindow XPSViewSaveDialogName
		String NewFoldersAndName=S_Value
		
		//	If the new wavename starts with root: it is taken as an absolute name and the folder selection is ignored
		String FullName=""
		if (CmpStr(NewFoldersAndName[0,4], "root:")==0)
			FullName=NewFoldersAndName
		else
			FullName=ExistingFolder+NewFoldersAndName
		endif
		
		//	Separates the full path into a folder and a wave name
		Variable a=StrLen(FullName)
		Variable b=StrSearch(FullName, ":", a-1, 1)
		String SaveFolderStr=XPSViewPath(FullName[0, b-1])+":"
		String SaveName=FullName[b+1, a-1]
		
		//	Hides the dialog window
		DoWindow /HIDE=1 XPSViewSaveDialogWindow

		//	Checks if the proposed name conflicts with an existing wave (and the user has not been given the overwrite warning)
		if ((Exists(XPSViewPath(FullName))!=0) && (CmpStr(B_Struct.win, "XPSViewOverwriteDialogWindow")!=0))
		
			//	Creates the prompt panel
			DoWindow /K XPSViewOverwriteDialogWindow
			NewPanel /K=1 /W=(370, 220, 660, 260) /N=XPSViewOverwriteDialogWindow as "Name Already Exists!"
			
			//	Associates a hook function with the window. This will be used to clean up waves when the window is killed
			SetWindow XPSViewOverwriteDialogWindow, hook(KillHook)=XPSViewOverwriteDialogHookFunc, userdata(Win)=WinUserData
			SetWindow XPSViewOverwriteDialogWindow, userdata(CleanUp)=CleanUpUserData

			//	Creates Overwrite and Cancel buttons
			Button XPSViewSaveDialogCancel proc=EccentricXPS#XPSViewOverwriteCancel, pos={35,10}, size={100, 20}, title="Cancel", win=XPSViewOverwriteDialogWindow
			Button XPSViewSaveWaveOK proc=EccentricXPS#XPSViewSaveWaveOK, pos={155,10}, size={100, 20}, title="Overwrite", userdata(Func)=FuncUserData, userdata(Win)=WinUserData, win=XPSViewOverwriteDialogWindow
		else
		
			//	Creates any extra folders in the beginning of the name, e.g folder1:folder2:wave3
			XPSViewCreateDataFolders(SaveFolderStr)

			//	Finds the function specified in the userdata of the button
			FUNCREF XPSViewSaveWaveOKFunc SaveFunction=$FuncUserData
			
			//	Executes the function specified in the userdata of the button
			SaveFunction($SaveFolderStr, SaveName, WinUserData)
			
			//	Kills the overwrite and dialog windows, this has to be at the end or the cleanup function will remove the temporary waves
			DoWindow /K XPSViewOverwriteDialogWindow
			DoWindow /K XPSViewSaveDialogWindow
		endif
	endif
end



Function XPSViewOverwriteDialogHookFunc(s)
//	The hook function for the Overwrite window. This is used for cleaning up waves associated with the window when the window is killed
STRUCT WMWinHookStruct &s
Variable hookResult=0

	switch(s.eventCode)
	
		//	The window was killed
		case 2:

			//	Indicates that an action has taken place
			hookResult=1

			//	Kills the dialog window, if it is hidden
			DoWindow XPSViewSaveDialogWindow
			if (V_flag==2)
				DoWindow/K XPSViewSaveDialogWindow
			endif
			break
	endswitch

	//	If you handle a particular event and you want Igor to ignore it, return 1 from the hook function. Return 0 if nothing was done
	return hookResult
end



Function XPSViewSaveWaveOKFunc(SaveFolder, SaveName, ActiveWindow)
//	A proto function used to define the type of functions that can be called by the XPSViewSaveWaveOK button control. Proto functions are not alllowed to be static
//	If the called function is invalid or does not exist, this function will be called instead
DFREF SaveFolder
String SaveName, ActiveWindow
end



Function XPSViewSaveWaveOKCleanFunc(ActiveWindow)
//	A proto function used to define the type of clean up functions that can be called by the XPSViewSaveWaveOK button control. Proto functions are not alllowed to be static
//	If the called function is invalid or does not exist, this function will be called instead
String ActiveWindow
end



Static Function XPSViewCreateDataFolders(Path)
//	Creates the full path of data folders contained in Path. Each single path, seperated by ;, contained in Path must be a full path starting in root:
String Path
String SinglePath=""
Variable n=ItemsInList(Path, ";"), i=0, a=0
	for (i=0; i<n; i+=1)
		SinglePath=StringFromList(i, Path, ";")
		for (a=StrSearch(SinglePath, ":", 5); a!=-1; a=StrSearch(SinglePath, ":", a+1))
			NewDataFolder/O $XPSViewPath(SinglePath[0, a-1])
		endfor
	endfor
end



Static Function XPSViewOverwriteCancel(B_Struct) : ButtonControl
//	Brings back the initial dialog window
STRUCT WMButtonAction & B_Struct

	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
	
		//	Brings back the dialog window
		DoWindow /F /HIDE=0 XPSViewSaveDialogWindow
		
		//	Kills the overwrite window, this has to be at the end or the clean up function for the overwrite window will kill the dialog window as well
		DoWindow /K XPSViewOverwriteDialogWindow
	endif
end



Static Function ErrorMessage(Text)
//	Displays a short error message
String Text
	DoWindow /K ErrorWindow
	NewPanel /K=1 /N=ErrorWindow /W=(560-round(StrLen(Text)*2.7),300,640+round(StrLen(Text)*2.7),380) as "Error!"	
	DrawText /W=ErrorWindow 40,20, Text
	Button CloseErrorWindow proc=EccentricXPS#CloseWindow, pos={20+round(StrLen(Text)*2.7),40}, title="OK", win=ErrorWindow
end



Static Function CloseWindow(B_Struct) : ButtonControl
//	Closes the window the button is in
STRUCT WMButtonAction &B_Struct

	//	//	If the button was pressed. eventCode 2 = mouse up
	if (B_Struct.eventCode==2)
		DoWindow /K $B_Struct.win
	endif
	return 0
end



Static Function CreateDataPath(DataPathList)
//	Specifies the default data directory any open file dialog will start in. Tries the items in the semicolon separated list until a valid path has been created
String DataPathList

	Variable n=ItemsInList(DataPathList, ";")
	Variable i=0

	//	Tries the items in the list in order until a valid path has been created
	if (n>0)
		do
			NewPath/Q/O/Z EccentricXPSDataPath StringFromList(i, DataPathList, ";")
			i+=1
		while ((V_Flag!=0) && (i<n))
	endif
	
	//	Will use EccentricXPSDataPath with the next Open command. This is identical to Open/P=EccentricXPSDataPath, but it prevents a bug in Igor 6.36 or, most likely, Windows 10
	 PathInfo/S EccentricXPSDataPath
end



Static Function ReduceDatapoints(OldWaveName, Number)
//	Resamples a one-dimensional wave to a lower number of data points
//	This is an optional function not used by any of the menu items
string OldWaveName
variable Number
variable i=0, ii=0, n=0
	wave OldWave=$OldWaveName
	Make/O /N=(trunc(NumPnts(OldWave)/Number)) $OldWaveName+"_RD"
	wave NewWave=$OldWaveName+"_RD"
	NewWave=0
	n=NumPnts(NewWave)
	for (i=0; i<n; i=i+1)
		for (ii=0; ii<Number; ii=ii+1)
			NewWave[i]=NewWave[i]+OldWave[i*Number+ii]
		endfor
	endfor
	Setscale /P x, leftx(OldWave)+deltax(OldWave)*(Number-1)/2, deltax(OldWave)*Number, NewWave
end



Static Function XPSUpdateProcedureHistory()
//	Updates the history of EccentricXPS procedure versions used with the current experiment
Variable i=0, a=-1, b=-1, VersionNum=0

	//	Extracts the file name of the currently used procedure
	String ActiveProcName=ParseFilePath(0, FunctionPath("EccentricXPS#XPSUpdateProcedureHistory"), ":", 1, 0)
	
	//	Extracts the full text of the procedure code
	String ProcText = ProcedureText("", 0, ActiveProcName)		//	How much time does this take?? 4 ms

	//	Searches throgh the first 50 lines of code for the "#Pragma version =" statement
	for (i=0; i<50; i+=1)
	
		//	Finds the next end-of-line character
		a=b
		b=strsearch(ProcText, "\r", a+1)
		
		//	Terminates the loop if the current line is the last line in the procedure file
		if (b==-1)
			b=StrLen(ProcText)
			i=50
		endif
		
		//	Tests the curent line for the "#Pragma version =" statement and saves the version
		SScanF LowerStr(ProcText[a+1, b-1]), "#pragma version = %f", VersionNum
		
		//	Terminates the loop if a version number was found
		if (V_Flag==1)
			i=50
		endif
	endfor
	
	//	If no version number was found a value of NaN is assigned
	if (V_Flag!=1)
		VersionNum=NaN
	endif

	//	Finds the history of the used procedures
	DFREF ProgramFolder=root:Programming
	SVAR/Z VersionHistory=ProgramFolder:EccentricXPSVersionHistory
	if (SVAR_Exists(VersionHistory)==0)
		String /G ProgramFolder:EccentricXPSVersionHistory=""
		SVAR VersionHistory=ProgramFolder:EccentricXPSVersionHistory
		
		//	Updates the version history with the file name of the currently used procedure
		VersionHistory="\t"+ActiveProcName+"\tVersion "+Num2Str(VersionNum)+"\t("+Date()+")\r"
	else

		//	Extracts the file name of the last used procedure and version
		b=StrSearch(VersionHistory, "\t", StrLen(VersionHistory)-1, 1)
		a=StrSearch(VersionHistory, "\r", b-1, 1)
		String LastProcVerName=VersionHistory[a+1, b-1]

		//	Checks if the currently used procedure file and version is different from the last used procedure file
		if (CmpStr("\t"+ActiveProcName+"\tVersion "+Num2Str(VersionNum), LastProcVerName)!=0)
	
			//	Updates the version history with the file name of the currently used procedure
			VersionHistory+="\t"+ActiveProcName+"\tVersion "+Num2Str(VersionNum)+"\t("+Date()+")\r"
		endif
	endif
end



Static Function XPSPrintProcedureHistory()
//	Prints the history of procedures used with the current experiment

	//	Updates the history of procedures used with the current experiment
	XPSUpdateProcedureHistory()

	//	Finds the history of used procedures
	DFREF ProgramFolder=root:Programming
	SVAR/Z VersionHistory=ProgramFolder:EccentricXPSVersionHistory
	if (SVAR_Exists(VersionHistory))
	
		//	Prints the history of used procedures
		Print("\r\rEccentricXPS procedures used with the current experiment file:\r"+VersionHistory)
	endif
end