// This file is part of the Localizer package
// Copyright 2009-2011 Peter Dedecker

#pragma IgorVersion = 6.20
#pragma rtGlobals=3	// Use strict wave reference mode
#pragma IndependentModule = PALMAnalysis

#include <Resize Controls>
#include <WaveSelectorWidget>
#include <SaveRestoreWindowCoords>

#define USE_WAITING_LIST				// used in the bleaching (subtraction) analysis. If defined, do not consider a new emitter as being discovered
										// unless it shows up in two consecutive frames. Uncomment the line to use a waiting list.
constant SAME_EMITTER_THRESHOLD_FACTOR = 0.5	// used in the bleaching analysis. If two positions are localized within a distance of this factor time the fitted standard deviation
														// then consider them to be the same emitter

// camera types (for data loading)
constant CAMERA_TYPE_WINSPEC = 0
constant CAMERA_TYPE_ANDOR = 1
constant CAMERA_TYPE_HAMAMATSU = 2
constant CAMERA_TYPE_TIFF = 3
constant CAMERA_TYPE_PDE = 4	// a custom, very simple image format.
constant CAMERA_TYPE_ZEISS = 5		// Zeiss .lsm files. Currently unused since these are really TIFF files
constant CAMERA_TYPE_IGOR_WAVE = 6	// load data from an Igor wave

// threshold methods
constant THRESHOLD_METHOD_GLRT = 0
constant THRESHOLD_METHOD_ISODATA = 1
constant THRESHOLD_METHOD_TRIANGLE = 2
constant THRESHOLD_METHOD_DIRECT = 3

// preprocessing
constant  PREPROCESSOR_NONE = 0
constant  PREPROCESSOR_3X3MEDIAN = 1
constant  PREPROCESSOR_5X5MEDIAN = 2
constant  PREPROCESSOR_1X1GAUSSIAN = 3
constant  PREPROCESSOR_2X2GAUSSIAN = 4
constant  PREPROCESSOR_3X3MEAN = 5
constant  PREPROCESSOR_5X5MEAN = 6

// postprocessing
constant POSTPROCESSOR_NONE = 0
constant POSTPROCESSOR_REMOVE_ISOLATED = 1

// localization methods
constant LOCALIZATION_GAUSS_FITTING = 0
constant LOCALIZATION_GAUSS_FITTING_FIX = 1
constant LOCALIZATION_MULTIPLICATION = 2
constant LOCALIZATION_CENTROID = 3
constant LOCALIZATION_ZEISSPALM = 4
constant LOCALIZATION_ELLIPSOIDAL2DGAUSS = 5
constant LOCALIZATION_MLEWG = 6

// particle finding methods
constant PARTICLEFINDER_ADJACENT4 = 0
constant PARTICLEFINDER_ADJACENT8 = 1
constant PARTICLEFINDER_RADIUS = 2

// particle verification methods
constant PARTICLEVERIFIER_NONE = 0
constant PARTICLEVERIFIER_2DGAUSS = 1
constant PARTICLEVERIFIER_ELLIPSEGAUSS = 2
constant PARTICLEVERIFIER_OVERLAP = 3

// CCD processing methods
constant PROCESSCCDIMAGE_SUBAVG = 0
constant PROCESSCCDIMAGE_DIFFIMAGE = 1
constant PROCESSCCDIMAGE_CONVERTFORMAT = 2
constant PROCESSCCDIMAGE_CROP = 3
constant PROCESSCCDIMAGE_CONVERTPHOTONS = 4

// CCD processing output formats
constant IMAGE_OUTPUT_TYPE_TIFF = 0;
constant IMAGE_OUTPUT_TYPE_COMPR_TIFF = 1
constant IMAGE_OUTPUT_TYPE_IGOR = 2
constant IMAGE_OUTPUT_TYPE_PDE = 3

// CCD images analysis methods
constant ANALYZECCD_SUMMEDINTENSITYTRACE = 0
constant ANALYZECCD_AVERAGETRACE = 1
constant ANALYZECCD_AVERAGE_IMAGE = 2
constant ANALYZECCD_STANDARD_DEV_IMAGE = 3
constant ANALYZECCD_BLEACHING_ANALYSIS = 4

// PALM bitmap error estimators
constant PALMBITMAP_DEV_SAME = 0
constant PALMBITMAP_DEV_FITUNCERTAINTY = 1
constant PALMBITMAP_DEV_GAUSSIANMASK = 2

// lucky imaging methods
constant LUCKYIMAGING_AMPLITUDE = 0
constant LUCKYIMAGING_SMALLESTWIDTH = 1
constant LUCKYIMAGING_POSUNCERTAINTY = 2

// possible error codes (possibly Igor error codes from IgorXOP.h)
constant kUserAbort = 57

// miscellaneous constants
constant kScrollForward = 1
constant kScrollBackward = -1

// structures that make it more easy to pass data to e.g. the fitting routines
structure FitData
	Variable directThresholdLevel
	Variable cameraType
	variable background		// currently unused
	variable currentImage
	variable minDistanceBetweenParticles
	variable PFA
	variable PSFWidth
	variable xSize
	variable ySize
	variable numberOfImages
	variable firstFrameToAnalyze
	variable lastFrameToAnalyze
	variable thresholdMethod
	variable preprocessing
	variable postprocessing
	variable particlefinder
	variable localizationMethod
	string CCDFilePath
	
	// use or don't use a particular particle verification algorithm
	// to not use it set the corresponding value to zero
	// to use it set it to the corresponding constant defined above
	variable PVerOverlap
	variable PVerSymm
	variable PVerEllipse
endstructure

Menu "Macros"
	Submenu "CCD Analysis"
		SubMenu "Read CCD Data"
			"Read CCD data from disk...", /Q, SetupImageViewer_loadFromDisk()
			"Read CCD data from Igor wave...", /Q, SetupImageViewer_IgorWaves()
			"Read entire CCD file into memory...", /Q, ReadCCDFramesIntoMemory()
			"-"
			"Load localized positions from file...", /Q, LoadPositionsFromTextFile()
		End
		SubMenu "Manipulate localized positions"
			"Merge positions...", /Q, MergePositions()
			"Consolidate identical emitters...", /Q, ConsolidateSameEmitters_menu()
			"Extract spatial subset of positions...", /Q, ExtractSpatialSubsetOfPositions(0)
			//"Extract positions within limits....", /Q, ExtractPositionsWithinLimits()
			"Correct positions for drift...", /Q, CorrectDrift_menu()
		End
		SubMenu "Analyze Localized Positions"
			"Generate positions report...", /Q, GeneratePositionsReport()
			"Clustering analysis...", /Q, DoLClustering()
			"-"
			"Save positions to text file...", /Q, SavePositionsToTextFile()
		End
		SubMenu "Output Localization Images"
			"Generate scatter plot...", /Q, GenerateScatterPlot_Menu()
			"Generate accumulated image...", /Q, MakeAccumulatedImage_Menu()
			"Generate bitmap image...", /Q, MakePALMImage_Menu()
			"Generate 'normal widefield' image...", /Q, MakeWideFieldImage_menu()
		End
		"Save open CCD movie as...", /Q, SaveCCDImagesAs()
		"Process CCD movies...", /Q, SetupCCDProcessingPanel()
		"Run batch localization...", /Q, SetupBatchLocalizationPanel()
		"Bring controls to front/1", /Q, BringImageViewerToFront()
	End
End

Menu "GraphMarquee"
	SubMenu "Localization Analysis"
		"Summed Intensity Trajectory", /Q, MakeSummedTraceFromMarquee()
		"Average Intensity Trajectory", /Q, MakeAverageTraceFromMarquee()
		"Extract Positions", /Q, ExtractSpatialSubsetOfPositions(1)
		"Crop Images", /Q, MakeCroppedStackFromMarquee()
		SubMenu "Clustering"
			"Single cluster", /Q, SingleClusterStats_Marquee()
			"L function", /Q, LFunctionClustering_Marquee()
		End
	End
End

Menu "TracePopup", dynamic
	"-"
	Submenu "Localization Analysis"
		RemoveFitPositionMenuName(), /Q, RemoveFittedPositionContextMenu()
		RemovePositionsInFrameMenuName(), /Q, RemovePositionsFrameContextMenu()
	End
End

Function Assert(condition, [errMessage])
	variable condition
	string errMessage
	
	string abortMessage
	
	if (!condition)
		if (ParamIsDefault(errMessage))
			abortMessage = "An assert failed"
		else
			abortMessage = "Assert failed: " + errMessage
		endif
		Debugger
		Abort abortMessage
	endif
End

Function GetFileFormat(filePath)
	string filePath
	// determine the file format used by the file at the provided path
	
	variable cameraType
	
	if (stringmatch(filePath, "*.spe") == 1)
		cameraType = CAMERA_TYPE_WINSPEC	// Princeton file
	elseif (stringmatch(filePath, "*.sif") == 1)
		cameraType = CAMERA_TYPE_ANDOR	// Andor file
	elseif (stringmatch(filePath, "*.his") == 1)
		cameraType = CAMERA_TYPE_HAMAMATSU	// Hamamatsu file
	elseif (stringmatch(filePath, "*.tif*") == 1)
		cameraType = CAMERA_TYPE_TIFF	// TIFF file
	elseif (stringmatch(filePath, "*.pde") == 1)
		cameraType = CAMERA_TYPE_PDE	// PDE output file
	elseif (stringmatch(filePath, "*.lsm") == 1)
		cameraType = CAMERA_TYPE_TIFF	// Zeiss file,  just a tiff file
	else
		Abort "This doesn't seem to be a CCD file that I can understand"
	endif
	
	return cameraType
	
End

Function GetCCDFrame(filePath, camera_type, n)
	string filePath
	variable camera_type, n
	// this function will read the n-th image from the file at filePath, assuming that the camera type is given by camera_type
	
	// the requested image will be put into a wave called "Viewer_Temp"
	Assert(n >= 0)
	
	variable /G root:Packages:PALM:V_numberOfImages
	string /G root:Packages:PALM:S_CCD_file_path
	variable /G root:Packages:PALM:V_xSize
	variable /G root:Packages:PALM:V_ySize
	variable /G root:Packages:PALM:V_firstImageLoaded
	variable /G root:Packages:PALM:V_lastImageLoaded
	
	SVAR CCDFilePath = root:Packages:PALM:S_CCD_file_path
	NVAR nImages = root:Packages:PALM:V_numberOfImages
	NVAR xSize = root:Packages:PALM:V_xSize
	NVAR ySize = root:Packages:PALM:V_ySize
	NVAR firstImageLoaded = root:Packages:PALM:V_firstImageLoaded
	NVAR lastImageLoaded = root:Packages:PALM:V_lastImageLoaded
	
	DFREF packageDF = root:Packages:PALM
	
	string MacintoshFilePath = ParseFilePath(5, filePath, ":", 0, 0)
	wave M_CCDFrames
	wave Viewer_Temp
							
	// some sanity checks
	if (n < 0)
		Abort "Requested an image at a negative index"
	endif
	if (n > (nImages - 1))
		Abort "Requested an image beyond the number of images in the sequence"
	endif
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	ReadCCDImages /Y=(camera_type) /S=(n) /C=1 /O /DEST=packageDF:M_CCDFrames MacintoshFilePath
	
	nImages = V_numberOfImages
	xSize = V_xSize
	ySize = V_ySize
	firstImageLoaded = V_firstImageLoaded
	lastImageLoaded = V_lastImageLoaded
	
	wave M_CCDFrames = packageDF:M_CCDFrames
	Duplicate /O M_CCDFrames, packageDF:Viewer_Temp
	Redimension /N=(DimSize(M_CCDFrames, 0), DimSize(M_CCDFrames, 1)) packageDF:Viewer_Temp
	CCDFilePath = MacintoshFilePath
	
	return 0
End

Function SetupImageViewer_IgorWaves()
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	// make a listbox window to choose from
	DoWindow /F candidateWavesViewer
	if (V_flag == 0)
		NewPanel /K=1/W=(101,286,401,486) /N=candidateWavesViewer as "Choose a matrix wave"
		ListBox ListBoxCandidateWaves,pos={1,2},size={298,183}
		MakeListIntoWaveSelector("candidateWavesViewer", "ListBoxCandidateWaves", content = WMWS_Waves, selectionMode = WMWS_SelectionSingle,  nameFilterProc="WaveSelectorFilter2DAnd3DWaves")
		WS_SetNotificationProc("candidateWavesViewer", "ListBoxCandidateWaves", "wavesViewer_NotificationProc")
		ListBox ListBoxCandidateWaves,userdata(ResizeControlsInfo)= A"!!,<7!!#7a!!#BO!!#AFz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
		ListBox ListBoxCandidateWaves,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
		ListBox ListBoxCandidateWaves,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		SetWindow kwTopWin,hook(ResizeControls)=ResizeControls#ResizeControlsHook
		SetWindow kwTopWin,userdata(ResizeControlsInfo)= A"!!*'\"z!!#BP!!#AWzzzzzzzzzzzzzzzzzzzzz"
		SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzzzzzzzz"
		SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzz!!!"
		
		// if the window was opened previously then restore its size and position
		WC_WindowCoordinatesRestore("candidateWavesViewer")
		SetWindow candidateWavesViewer hook(WindowCoordsHook)=WC_WindowCoordinatesNamedHook
	endif
End

Function WaveSelectorFilter2DAnd3DWaves(aName, contents)
	String aName
	Variable contents
	
	if (contents != WMWS_Waves)	// only accept waves
		return 0
	endif
	
	wave proposedWave = $aName
	
	// only accept if the wave is 2D or 3D
	if ((DimSize(proposedWave, 1) != 0) && (DimSize(proposedWave, 3) == 0))
		return 1
	endif
	return 0
end

Function wavesViewer_NotificationProc(SelectedItem, EventCode)
	String selectedItem
	Variable eventCode
	
	switch (eventCode)
		case WMWS_DoubleClick:
			string fullPath = selectedItem
			wave dataWave = $fullPath
			if (WaveExists(dataWave))
				DoWindow /K candidateWavesViewer
				SetupImageViewer(fullPath, CAMERA_TYPE_IGOR_WAVE)
			else
				Abort "Unable to find the selected wave!"
			endif
			break
		default:
			return 0
	endswitch
End
	
Function SetupImageViewer_loadFromDisk()
	// prompt the user for a data file
	variable RefNum, cameraType
	string FilePath, MacintoshFilePath, fileName
	
	Open /D/R/T="????" RefNum

	if (strlen(S_Filename) == 0)	// The user cancelled the dialog
		return -1
	endif
	
	FilePath = S_FileName
	
	MacintoshFilePath = ParseFilePath(5, FilePath, ":", 0, 0)
	cameraType = GetFileFormat(MacintoshFilePath)
	
	SetupImageViewer(MacintoshFilePath, cameraType)
	
End

Function ReadCCDFramesIntoMemory()
	
	variable RefNum, cameraType, nFilesToLoad, i
	variable skipThisFile
	string filePath, filePaths, MacintoshFilePath, fileName
	string wName
	string errorMessage
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	Variable /G root:Packages:PALM:V_numberOfImages
	Variable /G root:Packages:PALM:V_xSize
	Variable /G root:Packages:PALM:V_ySize
	
	NVAR nImages = root:Packages:PALM:V_numberOfImages
	NVAR xSize = root:Packages:PALM:V_xSize
	NVAR ySize = root:Packages:PALM:V_ySize
	
	Open /D/R/MULT=1/T="????" RefNum

	if (strlen(S_Filename) == 0)	// The user cancelled the dialog
		return -1
	endif
	
	filePaths = S_FileName
	nFilesToLoad = ItemsInList(filePaths, "\r")
	
	for (i = 0; i < nFilesToLoad; i+=1)
		skipThisFile = 0
		filePath = StringFromList(i, filePaths, "\r")
		MacintoshFilePath = ParseFilePath(5, FilePath, ":", 0, 0)
		cameraType = GetFileFormat(MacintoshFilePath)
		
		// get the number of images in the file
		ReadCCDImages /Y=(cameraType) /Z /H MacintoshFilePath
		if (V_flag != 0)		// did we fail to read the data?
			errorMessage = "Unable to read from  " + MacintoshFilePath
			Abort errorMessage
		endif
		nImages = V_numberOfImages
		xSize = V_xSize
		ySize = V_ySize
		
		wName = ParseFilePath(3, MacintoshFilePath, ":", 0, 0)
		wName = CleanupName(wName, 1)
		for ( ; ; )
			if (WaveExists(root:'PALM Movies':$wName))
				wName = GetOutputWaveName(wName, root:'PALM Movies', "Name already exists, please provide a different one")
				if (strlen(wName) == 0)
					skipThisFile = 1
					nFilesToLoad -= 1
					break
				else
					break
				endif
			else
				break
			endif
		endfor
		
		if (skipThisFile == 1)
			continue
		endif
		
		NewDataFolder /O root:'PALM Movies'
		ReadCCDImages /Y=(cameraType) /O /DEST=root:'PALM Movies':$wName MacintoshFilePath
		
	endfor
	
	if (nFilesToLoad == 1)
		SetupImageViewer("root:PALM Movies:" + wName, CAMERA_TYPE_IGOR_WAVE)
	endif
	
End


Function SetupImageViewer(MacintoshFilePath, cameraType)
	string MacintoshFilePath
	variable cameraType
	// set up some controls so we can easily look at the spe images in a single file
	
	// if path != 0 then we should prompt the user to type a path directly, otherwise we should present a dialog
	Assert(strlen(macintoshFilePath) > 0)
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	variable /G root:Packages:PALM:V_dontSetDefaultValues
	variable /G root:Packages:PALM:V_currentImage = 0
	variable /G root:Packages:PALM:V_currentImageStartingFromOne=1
	SetFormula root:Packages:PALM:V_currentImageStartingFromOne, "root:Packages:PALM:V_currentImage + 1"
	variable /G root:Packages:PALM:V_minDistanceBetweenParticles
	variable /G root:Packages:PALM:V_numberOfImages
	variable /G root:Packages:PALM:V_xSize
	variable /G root:Packages:PALM:V_ySize
	string /G root:Packages:PALM:S_CCD_file_path
	string /G root:Packages:PALM:S_CCD_ProcessedImage_out_path
	variable /G root:Packages:PALM:V_cameraType = cameraType
	variable /G root:Packages:PALM:V_CCDPixelSize
	variable /G root:Packages:PALM:V_currentThreshold
	variable /G root:Packages:PALM:V_PFA
	variable /G root:Packages:PALM:V_gaussianWidth
	variable /G root:Packages:PALM:V_particleVerifierOverlap
	variable /G root:Packages:PALM:V_particleVerifierSymm
	variable /G root:Packages:PALM:V_particleVerifierEllipse
	variable /G root:Packages:PALM:V_nFramesToSkip
	
	NVAR dontSetDefaultValues = root:Packages:PALM:V_dontSetDefaultValues
	NVAR currentImageStartingFromOne = root:Packages:PALM:V_currentImageStartingFromOne
	NVAR minDistanceBetweenParticles = root:Packages:PALM:V_minDistanceBetweenParticles
	NVAR nImages = root:Packages:PALM:V_numberOfImages
	NVAR xSize = root:Packages:PALM:V_xSize
	NVAR ySize = root:Packages:PALM:V_ySize
	NVAR gCameraType = root:Packages:PALM:V_cameraType
	NVAR pixelSize = root:Packages:PALM:V_CCDPixelSize
	NVAR currentThreshold = root:Packages:PALM:V_currentThreshold
	NVAR pfa = root:Packages:PALM:V_PFA
	NVAR gaussianWidth = root:Packages:PALM:V_gaussianWidth
	NVAR nFramesToSkip = root:Packages:PALM:V_nFramesToSkip
	NVAR particleVerifierOverlap = root:Packages:PALM:V_particleVerifierOverlap
	
	// if this is the first movie being opened then set some defaults
	if (dontSetDefaultValues == 0)
		pfa = 40
		minDistanceBetweenParticles = 8
		gaussianWidth = 2
		dontSetDefaultValues = 1
		particleVerifierOverlap = 1
	endif
	
	variable status
	string fileName = ParseFilePath(3, MacintoshFilePath, ":", 0, 0)
	string listPositionsFunc = GetIndependentModuleName() + "#GetPossiblePositionsWaves()"
	string listParticleVerifiersFunc = GetIndependentModuleName() + "#PMParticleVerificationListItems()"
	
	ReadCCDImages /Y=(cameraType) /H /Z MacintoshFilePath
	if (V_flag != 0)		// did we fail to read the data?
		string errorMessage = "Unable to read from  " + MacintoshFilePath
		Abort errorMessage
	endif
	nImages = V_numberOfImages
	xSize = V_xSize
	ySize = V_ySize
	
	// load the initial image
	GetCCDFrame(MacintoshFilePath, gCameraType, 0)
	
	wave Viewer_Temp = root:Packages:PALM:Viewer_Temp
	
	DoWindow /F CCDViewer
	
	if (V_flag == 0)	// the window does not exist
		#if (stringmatch(StringByKey("OS", IgorInfo(3)), "*Macintosh*") == 1)	// running on Macintosh
			Display /K=1/N=CCDViewer /W=(0,44,649,636) as "CCD Frame"
		#else	// running on Windows
			Display /K=1/N=CCDViewer /W=(3.75,37.25,457.5,455) as "CCD Frame"
		#endif
		ModifyGraph width={Aspect, (DimSize(Viewer_Temp, 0) / DimSize(Viewer_Temp, 1))}, height=0
		AppendImage /W=CCDViewer Viewer_Temp
		SetWindow kWTopWin,hook(storeRightClickCoordinates)=StoreRightClickCoordinatesHook
	else
		// if the viewer window already exists, make sure the aspect ratio is correct
		ModifyGraph width={Aspect, (DimSize(Viewer_Temp, 0) / DimSize(Viewer_Temp, 1))}, height=0
	endif
	
	// update the title of the window to contain the file name
	DoWindow /T CCDViewer, fileName
	
	// construct the histogram viewer
	wave M_CCDHistogram = MakeImageHistogram(Viewer_Temp)
	DoWindow /F CCD_Histogram
	if (V_flag == 0)
		#if (stringmatch(StringByKey("OS", IgorInfo(3)), "*Macintosh*") == 1)	// running on Macintosh
			Display /K=1 /N=CCD_Histogram /W=(618,44,999,261) M_CCDHistogram as "CCD Histogram"
		#else
			Display /K=1 /N=CCD_Histogram /W=(474.75,49.25,802.5,221) M_CCDHistogram as "CCD Histogram"
		#endif
		ModifyGraph mode=7,hbFill=2
		AutoPositionWindow /E /M=0 /R=CCDViewer CCD_Histogram
	endif
	
	// construct the CCD analysis panel
	DoWindow /F AnalyzeImages
	
	if (V_flag == 0)
		NewPanel /K=1 /N=AnalyzeImages /W=(640,284,989,334) as "Analyze Images"
		PopupMenu PMChooseCCDAnalysisMethod,pos={9,12},size={175,20},title="Procedure:"
		PopupMenu PMChooseCCDAnalysisMethod,mode=1,value= #"\"Average Intensity Trace;Summed Intensity Trace;Average Image;Standard Deviation;Bleaching Analysis\""
		Button DoAnalyzeCCDImages,pos={267,12},size={75,20},proc=BTAnalyzeCCDImagesProc,title="Do it!"
		
		AutoPositionWindow /E /M=1 /R=CCD_Histogram AnalyzeImages
	endif
	
	// construct the main control panel
	DoWindow /F CCD_Control
	
	if (V_Flag == 0)
		NewPanel /K=1 /N=CCD_Control /W=(640,357,924,725) as "CCD Viewer"
		SetDrawLayer UserBack
		SetDrawEnv fsize= 9
		DrawText 54,201,"Particle verification:"
		Slider SLFrameSlider,pos={7,40},size={262,16},proc=SLViewerSliderProc
		Slider SLFrameSlider,limits={0,nImages - 1,1},value= 0,vert= 0,ticks= 0
		SetVariable SVCurrentFrame,pos={164,9},size={66,15},proc=SVCurrentFrameFromOneProc,title="Frame:"
		SetVariable SVCurrentFrame,limits={0,inf,0},value= currentImageStartingFromOne
		ValDisplay VDMaxFrames,pos={232,9},size={39,14},limits={0,0,0},barmisc={0,1000}
		ValDisplay VDMaxFrames,value= #"root:Packages:PALM:V_numberOfImages"
		Button BTNext,pos={85,6},size={30,20},proc=BTViewerProc,title=">"
		Button BTPrevious,pos={48,6},size={30,20},proc=BTViewerProc,title="<"
		Button BTFirst,pos={11,6},size={30,20},proc=BTViewerProc,title="<<"
		Button BTLast,pos={122,6},size={30,20},proc=BTViewerProc,title=">>"
		CheckBox CBThresholdCheck,pos={9,76},size={62,14},proc=CBShowThresholdProc,title="threshold:"
		CheckBox CBThresholdCheck,value= 0
		SetVariable SVSetThreshold,pos={77,76},size={55,15},title=" "
		SetVariable SVSetThreshold,value= currentThreshold,disable=2,proc=SVUpdateThreshold
		Button BTAnalyzePALM,pos={199,341},size={70,20},proc=BTAnalyzePALMProc,title="Analyze!"
		CheckBox CBDisplayFittedPositions,pos={9,239},size={120,14},proc=CBDisplayFittedPositionsProc,title="Display Fitted Positions"
		CheckBox CBDisplayFittedPositions,value= 0
		PopupMenu PMSelectPositionsWave,pos={142,237},size={125,20},bodyWidth=125,proc=PMUpdateSegmentationProc
		PopupMenu PMSelectPositionsWave,mode=1,value=#listPositionsFunc
		SetVariable SVSetPixelSize,pos={9,263},size={195,15},title="CCD pixel size (nm, optional):"
		SetVariable SVSetPixelSize,value= pixelSize
		PopupMenu PMThresholdMethod,pos={142,74},size={125,20},bodyWidth=125,proc=PMSelectThresholdMethodProc
		PopupMenu PMThresholdMethod,mode=1,value= #"\"GLRT;Direct\""
		SetVariable SVSetGaussianWidth,pos={9,280},size={230,15},title="Standard deviation of the PSF (pixels):"
		SetVariable SVSetGaussianWidth,limits={0.1,inf,0.1},value= gaussianWidth,proc=SVUpdateThreshold
		SetVariable SVSetPFA,pos={10,299},size={125,15},disable=0,title="GLRT Sensitivity:"
		SetVariable SVSetPFA,limits={1,inf,1},value= pfa,proc=SVUpdateThreshold
		PopupMenu PMThresholdPreprocessing,pos={27,115},size={240,20},bodyWidth=125,proc=PMSelectPreprocessingProc,title="Threshold preprocessing: "
		PopupMenu PMThresholdPreprocessing,mode=1,popvalue="None",value= #"\"None;3x3 Median Filter;5x5 Median Filter;1x1 Gaussian Smoothing;2x2 Gaussian Smoothing;3x3 Mean Filter;5x5 Mean Filter\""
		PopupMenu PMThresholdPostProcessing,pos={22,138},size={246,20},bodyWidth=125,proc=PMSelectPostProcessingProc,title="Threshold postprocessing: "
		PopupMenu PMThresholdPostProcessing,mode=1,popvalue="None",value= #"\"None;Remove Isolated Pixels\""
		CheckBox CBShowParticles,pos={27,97},size={128,14},proc=CBCheckShowParticlesProc,title="Show estimated particles"
		CheckBox CBShowParticles,value= 0,disable=0
		PopupMenu PMParticleFinder,pos={73,161},size={195,20},bodyWidth=125,proc=PMSelectParticleFinderProc,title="Particle finding:"
		PopupMenu PMParticleFinder,mode=1,value= #"\"8-way adjacency;4-way adjacency\""
		PopupMenu PMParticleVerification,pos={53,185},size={215,20},bodyWidth=125,proc=PMParticleVerificationProc,title="Choose..."
		PopupMenu PMParticleVerification,mode=0,value=#listParticleVerifiersFunc
		SetVariable SVSetMinimumRadius,pos={9,273},size={195,15},title="Minimum distance between pixels:"
		SetVariable SVSetMinimumRadius,limits={1,inf,1},value= minDistanceBetweenParticles,disable=1,proc=SVUpdateThreshold
		SetVariable SVNFramesToSkip,pos={10,320},size={130,15},bodyWidth=60,title="Frames to skip:",value=nFramesToSkip
		PopupMenu PMLocalizationMethod,pos={41,209},size={227,20},bodyWidth=125,title="Localization algorithm:",proc=PMUpdateSegmentationProc
		PopupMenu PMLocalizationMethod,mode=1,popvalue="Gaussian Fitting",value= #"\"Gaussian Fitting;Gaussian Fitting (Fixed Width);Ellipsoidal Gaussian Fitting;Iterative Multiplication;Center-of-Mass;MLEwG\""
		Button BTCopyPositions,pos={11,341},size={100,20},proc=BTCopyPositionsProc,title="Copy Positions"
		SetWindow kwTopWin,hook(keycontrol)=KeyboardViewControls_loadDisk
		
		AutoPositionWindow /E /M=1 /R=AnalyzeImages CCD_Control
	else		// the window already exists, we need to update it
			// we start out without any positions
		CheckBox CBDisplayFittedPositions, value=0
		CheckBox CBThresholdCheck, value=0
		CheckBox CBShowParticles, value = 0
		Slider SLFrameSlider,limits={0,nImages - 1,1},value= 0
	endif
	
	// make sure that the viewer control panel is in the foreground
	DoWindow /F CCD_Control
	
	// if we were previously showing a threshold or some other overlay, remove it
	GenerateGUIThreshold()
	
End

Function BringImageViewerToFront()
	// this function merely brings the controls and images that are created by SetupImageViewer_loadFromDisk
	// so that they can be conviently retrieved
	
	DoWindow /F AnalyzeImages
	DoWindow /F CCD_Histogram
	DoWindow /F CCDViewer
	DoWindow /F CCD_Control
End


Function ChangeCurrentFrame(n)
	variable n
	
	Assert(n >= 0)
	
	string savDF = GetDataFolder(1)
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	
	variable /G root:Packages:PALM:V_numberOfImages
	variable /G root:Packages:PALM:V_currentImage
	variable /G root:Packages:PALM:V_currentImageStartingFromOne
	variable /G root:Packages:PALM:V_currentThreshold
	variable /G root:Packages:PALM:V_cameraType
	string /G root:Packages:PALM:S_CCD_file_path
	
	NVAR nImages = root:Packages:PALM:V_numberOfImages
	NVAR currentImage = root:Packages:PALM:V_currentImage
	NVAR cameraType = root:Packages:PALM:V_cameraType
	SVAR CCDFilePath = root:Packages:PALM:S_CCD_file_path
	
	if (n >= nImages)
		return -1
	endif
	
	if (n == currentImage)	// we are already looking at the requested frame
		return 0
	endif
	
	GetCCDFrame(CCDFilePath, cameraType, n)
	
	currentImage = n
	
	GenerateGUIThreshold()
	Slider SLFrameSlider win=CCD_Control, value=currentImage

	// update the histogram
	MakeImageHistogram(root:Packages:PALM:Viewer_Temp)
	
	// explicitly update the display if we're running on Windows
	#if (StringMatch(IgorInfo(2), "Windows") == 1)
		DoUpdate
	#endif
End

Function BTViewerProc(ctrlName) : ButtonControl
	string ctrlName
	
	NVAR currentImage = root:Packages:PALM:V_currentImage
	NVAR nImages = root:Packages:PALM:V_numberOfImages
	
	strswitch(ctrlName)
	case "BTNext":
		if (currentImage >= nImages)
			break
		endif
		ChangeCurrentFrame(currentImage + 1)
		break
		
	case "BTPrevious":
		if (currentImage == 0)
			break
		endif
		ChangeCurrentFrame(currentImage - 1)
		break
		
	case "BTFirst":
		ChangeCurrentFrame(0)
		break
		
	case "BTLast":
		ChangeCurrentFrame(nImages - 1)
		break
	endswitch
End

Function SLViewerSliderProc(sa) : SliderControl
	STRUCT WMSliderAction &sa
	
	variable curval

	switch( sa.eventCode )
		case -1: // kill
			break
		default:
			if( sa.eventCode & 1 ) // value set
				curval = sa.curval
				ChangeCurrentFrame(curval)
			endif
			break
	endswitch

	return 0
End

Function KeyboardViewControls_loadDisk(s)
	STRUCT WMWinHookStruct &s
	
	NVAR currentImage = root:Packages:PALM:V_currentImage
	NVAR nImages = root:Packages:PALM:V_numberOfImages
	
	switch (s.eventcode)
		case 11:	// keyboard event
			variable key_struck = s.keycode
			
			if (key_struck == 29)	// go forward 1 image
				if (currentImage < (nImages - 1))
					ChangeCurrentFrame(currentImage + 1)
				endif
			endif
			
			if (key_struck == 28)	// go back 1 image
				if (currentImage > 0)
					ChangeCurrentFrame(currentImage - 1)
				endif
			endif
			
			if (key_struck == 13)	// return
				GenerateGUIthreshold()
			endif
			return 0
			break
		case 22:	// scroll event
			// accept both horizontal and vertical scrolling
			variable scrollDirection, nFramesToScroll
			if (s.wheelDx != 0)
				nFramesToScroll = round(abs(s.wheelDx))
				if (s.wheelDx > 0)
					scrollDirection = kScrollForward
				else
					scrollDirection = kScrollBackward
				endif
			endif
			
			if (s.wheelDy != 0)
				nFramesToScroll = round(abs(s.wheelDx))
				if (s.wheelDy > 0)
					scrollDirection = kScrollForward
				else
					scrollDirection = kScrollBackward
				endif
			endif
			
			if (scrollDirection == kScrollForward)
				if (currentImage - nFramesToScroll < 0)
					ChangeCurrentFrame(0)
				else
					ChangeCurrentFrame(currentImage - nFramesToScroll)
				endif
			else
				if (currentImage + nFramesToScroll >= nImages)
					ChangeCurrentFrame(nImages - 1)
				else
					ChangeCurrentFrame(currentImage + nFramesToScroll)
				endif
			endif
			return 0
			break
		default:
			return 0
			break
	endswitch
	
	return 0
End	

Function StoreRightClickCoordinatesHook(s)
	STRUCT WMWinHookStruct &s
	
	variable hookResult = 0
	
	switch(s.eventCode)
		case 3:	// mousedown
			if (s.EventMod & 2^4)	// right click
				NewDataFolder /O root:Packages
				NewDataFolder /O root:Packages:PALM
				
				variable /G root:Packages:PALM:V_RightClickX
				variable /G root:Packages:PALM:V_RightClickY
				
				NVAR xCoordinate = root:Packages:PALM:V_RightClickX
				NVAR yCoordinate = root:Packages:PALM:V_RightClickY
			
				variable xLoc = s.mouseLoc.h
				variable yLoc = s.mouseLoc.v
				
				xCoordinate = AxisValFromPixel("", "Bottom", xLoc)
				yCoordinate = AxisValFromPixel("", "Left", yLoc)
			endif
			hookResult = 0
			break
		default:
			// ignore all other options
	endswitch
	
	return hookresult
End
	
End

Function PMSelectThresholdMethodProc(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	switch( pa.eventCode )
		case 2: // mouse up
			Variable popNum = pa.popNum
			String popStr = pa.popStr
			
			ModifyControl SVSetThreshold, disable = 2
			ModifyControl SVSetPFA, disable = 2
			
			StrSwitch (pa.popStr)
				case "Direct":
					ModifyControl SVSetThreshold, disable = 0
					break
				case "GLRT":
					ModifyControl SVSetPFA, disable = 0
					break
			EndSwitch
			
			// now update the images
			GenerateGUIthreshold()
			
			break
	endswitch
End

Function PMSelectPreprocessingProc(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	switch( pa.eventCode )
		case 2: // mouse up
			Variable popNum = pa.popNum
			String popStr = pa.popStr
			// now calculate a new threshold image if we are currently showingthresholding	
			GenerateGUIthreshold()
			break
	endswitch
	
	return 0
End

Function PMSelectPostProcessingProc(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	switch( pa.eventCode )
		case 2: // mouse up
			Variable popNum = pa.popNum
			String popStr = pa.popStr
			
			// now calculate a new threshold image if we are currently showingthresholding
			GenerateGUIthreshold()
			
			break
	endswitch
	
	return 0
End

Function SVUpdateThreshold(sva) : SetVariableControl
	STRUCT WMSetVariableAction &sva

	switch( sva.eventCode )
		case 1: // mouse up
		case 2: // Enter key
		case 3: // Live update
			Variable dval = sva.dval
			String sval = sva.sval
			
			GenerateGUIthreshold()
			
			break
	endswitch

	return 0
End

Function CBShowThresholdProc(cba) : CheckBoxControl
	STRUCT WMCheckboxAction &cba

	switch( cba.eventCode )
		case 2: // mouse up
			Variable checked = cba.checked
			GenerateGUIThreshold()
			break
	endswitch

	return 0
End

Function Generatethreshold(image, fittedPositions, fitParams, showThreshold, showPositions, showFittedPositions, frameNumber)
	wave image, fittedPositions
	struct FitData &fitParams
	variable showThreshold, showPositions, showFittedPositions, frameNumber
	
	Assert(frameNumber >= 0)
	Assert(WaveExists(image))
	
	// take the filtering of the localization into account:
	// if the localization method is 2D Gaussian fitting, then enable the corresponding particle verifier
	// and similar for ellipsoidal Gaussian fitting
	if (fitParams.localizationMethod == LOCALIZATION_GAUSS_FITTING)
		fitParams.PVerSymm = PARTICLEVERIFIER_2DGAUSS
	elseif (fitParams.localizationMethod == LOCALIZATION_ELLIPSOIDAL2DGAUSS)
		fitParams.PVerEllipse = PARTICLEVERIFIER_ELLIPSEGAUSS
	endif
	
	DFREF savDF = GetDataFolderDFR()
	SetDataFolder root:Packages:PALM
	
	EmitterSegmentation /M=(fitParams.thresholdMethod) /G={fitParams.preprocessing, fitParams.postprocessing} /PFA=(fitParams.PFA) /ABS=(fitParams.directThresholdLevel) /WDTH=(fitParams.PSFWidth) /S=(showPositions) /F=(fitParams.particlefinder) /PVER={fitParams.PVerSymm, fitParams.PVerEllipse, fitParams.PVerOverlap} /R=(fitParams.minDistanceBetweenParticles) image
	
	SetDataFolder savDF
	
	if (showFittedPositions != 0)
		GenerateFittedPositionsWaves(fittedPositions, frameNumber)
	endif
	
	if (showPositions != 0)
		MakeParticleWavesAndLabel()
	endif
End

Function GenerateGUIThreshold()
	
	NVAR currentImage = root:Packages:PALM:V_currentImage
	variable showThreshold, showPositions, showFittedPositions
	wave image = root:Packages:PALM:Viewer_Temp
	wave fittedPositions
	string tracesInGraph, imagesInGraph
	
	DFREF packageFolder = root:Packages:PALM
	
	DoWindow CCD_Control
	Assert(V_flag == 1)
	
	DoWindow CCDViewer
	Variable CCDViewerExists = (V_flag != 0)
	
	Struct FitData fitParams 
	GetGUIFitSettingsAndOptions(fitParams)
	
	ControlInfo /W=CCD_Control CBThresholdCheck
	if (V_value == 1)	// show the threshold
		showThreshold = 1
	else
		showThreshold = 0
	endif
	
	ControlInfo /W=CCD_Control CBShowParticles
	if (V_value == 1)
		showPositions = 1
	else
		showPositions = 0
	endif
	
	ControlInfo /W=CCD_Control CBDisplayFittedPositions
	if (V_value == 1)
		// does the wave with the fitted positions exist?
		// we can only show the positions if they exist
		ControlInfo /W=CCD_Control PMSelectPositionsWave
		
		wave fittedPositions = GetPositionsWaveReference(S_Value)
		
		if (WaveExists(fittedPositions) == 0)
			showFittedPositions = 0
		else
			showFittedPositions = 1
		endif
	else
		showFittedPositions = 0
	endif
	
	// do we need to calculate the threshold or the positions?
	// if not then there is no reason to run the calculation
	if ((showThreshold == 1) || (showPositions == 1) || (showFittedPositions == 1))
		Generatethreshold(image, fittedPositions, fitParams, showThreshold, showPositions, showFittedPositions, currentImage)
	endif
	
	// if the CCD Viewer exists then adjust the calculations, otherwise do nothing
	if (CCDViewerExists)
		// what traces are currently displayed?
		tracesInGraph = TraceNameList("CCDViewer",";",1)
		imagesInGraph = ImageNameList("CCDViewer",";")
		
		// check if we need to show threshold traces
		if (showThreshold == 1)	// show the threshold
			wave M_SegmentedImage
			// is the threshold already on the graph? If they are not on the graph, then show them
			if  (stringmatch(imagesInGraph,"*M_SegmentedImage*") != 1)
				AppendImage /W=CCDViewer packageFolder:M_SegmentedImage
				ModifyImage  /W=CCDViewer M_SegmentedImage  ctab= {0.5,1,RedWhiteGreen,1}, minRGB=NaN,maxRGB=0
			endif
			
		else		// don't show the threshold
			if  (stringmatch(imagesInGraph,"*M_SegmentedImage*") == 1)
				RemoveImage /W=CCDViewer M_SegmentedImage
			endif
		endif
		
		// check if we need to show positions
		DoWindow CCDViewer
		if (showPositions == 1)	// show the particles
			
			// are the particles already on the graph? If not, show them
			if  (stringmatch(tracesInGraph,"*W_particles*") != 1)
				AppendToGraph /W=CCDViewer packageFolder:W_particles_Y vs packageFolder:W_particles_X
				ModifyGraph /W=CCDViewer mode(W_particles_Y)=3,msize(W_particles_Y)=4,rgb(W_particles_Y)=(0,0,65535)
				ModifyGraph /W=CCDViewer textMarker(W_particles_Y)=0
			endif
			
		else		// don't show the particles. If they are on the graph then remove them
			if  (stringmatch(tracesInGraph,"*W_particles*") == 1)
				RemoveFromGraph /W=CCDViewer W_particles_Y
			endif
		endif
		
		// check if we need to show the fitted positions
		if (showFittedPositions == 1)
			// are the fitted positions already on the graph? If not, show them
			if  (stringmatch(tracesInGraph,"*ps_temp*") != 1)
				AppendToGraph /W=CCDViewer packageFolder:ps_temp_Y vs packageFolder:ps_temp_X
				ModifyGraph /W=CCDViewer mode(ps_temp_Y)=3,marker(ps_temp_Y)=8, rgb(ps_temp_Y)=(65535,65535,0)
			endif
				
		else		// we need to remove the positions plot if it's on the graph
			tracesInGraph = TraceNameList("CCDViewer",";",1)
			if  (stringmatch(tracesInGraph,"*ps_temp*") == 1)
				RemoveFromGraph /W=CCDViewer $"ps_temp_Y"
			endif
		endif
	endif
	
	// explicitly update the display if we're running on Windows
	#if (StringMatch(IgorInfo(2), "Windows") == 1)
		DoUpdate
	#endif
	
End

Function /WAVE MakeImageHistogram(image)
	wave image
	
	Assert(WaveExists(image))
	// do not accept images that contain more than one plane
	Assert((DimSize(image, 2) == 0) || (DimSize(image, 2) == 1))
	
	DFREF savDF = GetDataFolderDFR()
	SetDataFolder root:Packages:PALM
	ImageHistogram image
	SetDataFolder savDF
	
	return root:Packages:PALM:W_ImageHist
End

Function /S PrettyPrintTimeElapsed(secondsElapsed)
	variable secondsElapsed
	
	Assert(secondsElapsed >= 0)
	
	variable days = 0, hours = 0, minutes = 0, seconds = 0
	variable remainder = secondsElapsed
	string prettyString = ""
	
	if (remainder >= 24 * 3600)
		days = floor(remainder / (24 * 3600))
		if (days > 1)
			prettyString += num2str(days) + " days, "
		else
			prettyString += num2str(days) + " day, "
		endif
		remainder -= days * 24 * 3600
	endif
	
	if (remainder >= 3600)
		hours = floor(remainder / 3600)
		if (hours > 1)
			prettyString += num2str(hours) + " hours, "
		else
			prettyString += num2str(hours) + " hour, "
		endif
		remainder -= hours * 3600
	endif
	
	if (remainder >= 60)
		minutes = floor(remainder / 60)
		if (minutes > 1)
			prettyString += num2str(minutes) + " minutes, "
		else
			prettyString += num2str(minutes) + " minute, "
		endif
		remainder -= minutes * 60
	endif
	
	// always show the seconds
	if (remainder != 1)
		prettyString += num2str(remainder) + " seconds"
	else
		prettyString += num2str(remainder) + " second"
	endif
	
	return prettyString
End

Function BTAnalyzePALMProc(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	variable msTimer, sec, err

	switch( ba.eventCode )
		case 2: // mouse up
				// click code here
				
				Struct FitData fitParams
				
				// get all the fit options and parameters the user defined in the GUI
				GetGUIFitSettingsAndOptions(fitParams)
				
				msTimer = startMSTimer
				
				err = DoPALMFitting(fitParams, 1)
				if (err != 0)	// an error occured
					Abort "An error occurred during the analysis"
				endif
				
				sec = stopMSTimer(msTimer)
				sec /= 1e6
				Printf "Elapsed time is %s\r", PrettyPrintTimeElapsed(sec)
				
				ControlUpdate /W=CCD_Control PMSelectPositionsWave
				PopupMenu PMSelectPositionsWave win=CCD_Control, mode=1, popMatch="POS_out"
				break

	endswitch
	return 0
End

Function GetGUIFitSettingsAndOptions(fitParams)
	STRUCT FitData &fitParams
	
	NVAR currentThreshold = root:Packages:PALM:V_currentThreshold
	NVAR cameraType = root:Packages:PALM:V_cameraType
	NVAR background = root:Packages:PALM:V_background
	NVAR currentImage = root:Packages:PALM:V_currentImage
	NVAR minDistanceBetweenParticles = root:Packages:PALM:V_minDistanceBetweenParticles
	NVAR PFA = root:Packages:PALM:V_PFA
	NVAR gaussianWidth = root:Packages:PALM:V_gaussianWidth
	NVAR xSize = root:Packages:PALM:V_xSize
	NVAR ySize = root:Packages:PALM:V_ySize
	NVAR nImages = root:Packages:PALM:V_numberOfImages
	NVAR nFramesToSkip = root:Packages:PALM:V_nFramesToSkip
	SVAR CCDFilePath = root:Packages:PALM:S_CCD_file_path
	
	NVAR particleVerifierOverlap = root:Packages:PALM:V_particleVerifierOverlap
	NVAR particleVerifierSymm = root:Packages:PALM:V_particleVerifierSymm
	NVAR particleVerifierEllipse = root:Packages:PALM:V_particleVerifierEllipse
	
	variable thresholdMethod, preprocessing, postprocessing, particlefinder, localizationMethod
	
	ControlInfo /W=CCD_Control PMThresholdMethod
	StrSwitch (S_Value)
		case "GLRT":
			thresholdMethod = THRESHOLD_METHOD_GLRT
			break
		case "Isodata":
			thresholdMethod = THRESHOLD_METHOD_ISODATA
			break
		case "Triangle":
			thresholdMethod = THRESHOLD_METHOD_TRIANGLE
			break
		case "Direct":
			thresholdMethod = THRESHOLD_METHOD_DIRECT
			break
		default:
			Abort "Unknown segmentation method"
			break
	EndSwitch
	
	ControlInfo /W=CCD_Control PMThresholdPreprocessing
	StrSwitch (S_Value)
		case "None":
			preprocessing = PREPROCESSOR_NONE
			break
		case "3x3 Median Filter":
			preprocessing = PREPROCESSOR_3X3MEDIAN
			break
		case "5x5 Median Filter":
			preprocessing = PREPROCESSOR_5X5MEDIAN
			break
		case "1x1 Gaussian Smoothing":
			preprocessing = PREPROCESSOR_1X1GAUSSIAN
			break
		case "2x2 Gaussian Smoothing":
			preprocessing = PREPROCESSOR_2X2GAUSSIAN
			break
		case "3x3 Mean Filter":
			preprocessing = PREPROCESSOR_3X3MEAN
			break
		case "5x5 Mean Filter":
			preprocessing = PREPROCESSOR_5X5MEAN
			break
		default:
			Abort "Unknown preprocessing method"
			break
	EndSwitch
	
	ControlInfo  /W=CCD_Control PMThresholdPostProcessing
	StrSwitch (S_Value)
		case "None":
			postprocessing =  POSTPROCESSOR_NONE
			break
		case "Remove Isolated Pixels":
			postprocessing =  POSTPROCESSOR_REMOVE_ISOLATED
			break
		default:
			Abort "Unknown postprocessing method"
			break
	EndSwitch
	
	ControlInfo /W=CCD_Control PMParticleFinder
	StrSwitch (S_Value)
		case "4-way adjacency":
			particlefinder = PARTICLEFINDER_ADJACENT4
			break
		case "8-way adjacency":
			particlefinder = PARTICLEFINDER_ADJACENT8
			break
		default:
			Abort "Unknown particle verifier method"
			break
	EndSwitch
	
	ControlInfo /W=CCD_Control PMLocalizationMethod
	StrSwitch (S_Value)
		case "Gaussian Fitting":
			localizationMethod = LOCALIZATION_GAUSS_FITTING
			break
		case "Gaussian Fitting (Fixed Width)":
			localizationMethod = LOCALIZATION_GAUSS_FITTING_FIX
			break
		case "Iterative Multiplication":
			localizationMethod = LOCALIZATION_MULTIPLICATION
			break
		case "Center-of-Mass":
			localizationMethod = LOCALIZATION_CENTROID
			break
		case "Ellipsoidal Gaussian Fitting":
			localizationMethod = LOCALIZATION_ELLIPSOIDAL2DGAUSS
			break
		case "MLEwG":
			localizationMethod = LOCALIZATION_MLEWG
			break
		default:
			Abort "Unknown localization method in GetGUIFitSettingsAndOptions()"
	EndSwitch
	
	// do a bunch of ugly crap with the particle verification since it's pretty hard to procedurally generate a multi-keyword flag
	if (particleVerifierSymm != 0)
		fitParams.PVerSymm = PARTICLEVERIFIER_2DGAUSS
	else
		fitParams.PVerSymm = PARTICLEVERIFIER_NONE
	endif
	
	if (particleVerifierEllipse != 0)
		fitParams.PVerEllipse = PARTICLEVERIFIER_ELLIPSEGAUSS
	else
		fitParams.PVerEllipse = PARTICLEVERIFIER_NONE
	endif
	
	if (particleVerifierOverlap != 0)
		fitParams.PVerOverlap = PARTICLEVERIFIER_OVERLAP
	else
		fitParams.PVerOverlap = PARTICLEVERIFIER_NONE
	endif
	
	fitParams.directThresholdLevel = currentThreshold
	fitParams.cameraType = cameraType
	fitParams.background = background
	fitParams.currentImage = currentImage
	fitParams.minDistanceBetweenParticles = minDistanceBetweenParticles
	fitParams.PFA = PFA
	fitParams.PSFWidth = gaussianWidth
	fitParams.xSize = xSize
	fitParams.ySize = ySize
	fitParams.firstFrameToAnalyze = nFramesToSkip
	fitParams.lastFrameToAnalyze = -1
	fitParams.numberOfImages = nImages
	fitParams.thresholdMethod = thresholdMethod
	fitParams.preprocessing = preprocessing
	fitParams.postprocessing = postprocessing
	fitParams.particlefinder = particlefinder
	fitParams.localizationMethod = localizationMethod
	fitParams.CCDFilePath = CCDFilePath
End

Function DoPALMFitting(fp, abortOnError)
	STRUCT FitData &fp
	variable abortOnError	// if nonzero then allow LocalizationAnalysis to abort with an error
							// if zero then do not cause aborts, but report a possible error as a nonzero return value
	
	NVAR pixelSize = root:Packages:PALM:V_CCDPixelSize
	
	DFREF packageDF = root:Packages:PALM
	
	// fit the data but also add the waveNote
	string waveNote
	variable err
	
	if (abortOnError != 0)
		LocalizationAnalysis /M=(fp.localizationMethod)/D=(fp.thresholdMethod)/Y=(fp.cameraType)/G={fp.preprocessing, fp.postprocessing}/F=(fp.particlefinder)/PVER={fp.PVerSymm,fp.PVerEllipse,fp.PVerOverlap}/RNG={fp.firstFrameToAnalyze,fp.lastFrameToAnalyze}/PFA=(fp.PFA)/T=(fp.directThresholdLevel)/R=(fp.minDistanceBetweenParticles)/W=(fp.PSFWidth) /DEST=packageDF:POS_out fp.CCDFilePath
	else	// don't abort for errors, but return them as an error code (use the /Z flag for LocalizationAnalysis)
		LocalizationAnalysis /M=(fp.localizationMethod)/D=(fp.thresholdMethod)/Y=(fp.cameraType)/G={fp.preprocessing, fp.postprocessing}/F=(fp.particlefinder)/PVER={fp.PVerSymm,fp.PVerEllipse,fp.PVerOverlap}/RNG={fp.firstFrameToAnalyze,fp.lastFrameToAnalyze}/PFA=(fp.PFA)/T=(fp.directThresholdLevel)/R=(fp.minDistanceBetweenParticles)/W=(fp.PSFWidth) /Z /DEST=packageDF:POS_out fp.CCDFilePath
		err = V_flag
	endif
	
	// check if the output wave exists
	// it might not exist if the user immediately aborted the procedure
	// since it is by request, do not treat a user abort as an error
	wave POS_out = root:Packages:PALM:POS_out
	
	if (WaveExists(POS_out) != 0)
		// append a note with information on the fit
		// most of the information is now appended by the XOP
		waveNote = ""
		
		if (pixelSize != 0)
			waveNote += "X PIXEL SIZE:" + num2str(pixelSize) + ";"
			waveNote += "Y PIXEL SIZE:" + num2str(pixelSize) + ";"
		endif
		waveNote += "CALCULATION DATE:" + date() + ";"
		
		Note /NOCR POS_out, waveNote
	endif
	
	if (abortOnError == 0)
		return err
	else
		return 0
	endif
End

Function GetColumnsForEmitterPositions(pos, xCol, yCol, zCol)
	wave pos
	variable &xCol, &yCol, &zCol
	
	Assert(WaveExists(pos))
	
	// The different types of localization routines in the software all return their own types of parameters for every positions
	// depending on what kind of information is offered by the routine
	// This means that the column containing the x, y, and possibly z positions could be located in different columns of the positions wave
	// this function will inspect the type of positions and return the column indices containing that information by reference
	// if a particular type does not support one of these then -1 will be returned
	
	// the kind of localization used in generating the positions must be set in the wave note
	
	// special case: if LOCALIZATION METHOD is not set, then assume that the positions originate from a symmetric 2D Gaussian
	// this might be the case if the user is using positions fitted with a previous version of the package
	if (NumType(NumberByKey("LOCALIZATION METHOD", note(pos))) == 2)	// NaN
		xCol = 3
		yCol = 4
		zCol = -1
		// append the information for future use
		Note /NOCR pos, "LOCALIZATION METHOD:" + num2str(LOCALIZATION_GAUSS_FITTING) + ";"
	endif
	
	switch (NumberByKey("LOCALIZATION METHOD", note(pos)))
		case LOCALIZATION_GAUSS_FITTING:
			xCol = 3
			yCol = 4
			zCol = -1
			break
		case LOCALIZATION_GAUSS_FITTING_FIX:
			xCol = 2
			yCol = 3
			zCol = -1
			break
		case LOCALIZATION_CENTROID:
			xCol = 1
			yCol = 2
			zCol = -1
			break
		case LOCALIZATION_MULTIPLICATION:
			xCol = 2
			yCol = 3
			zCol = -1
			break
		case LOCALIZATION_ZEISSPALM:
			xCol = 2
			yCol = 3
			zCol = -1
			break
		case LOCALIZATION_ELLIPSOIDAL2DGAUSS:
			xCol = 4
			yCol = 5
			zCol = -1
			break
		case LOCALIZATION_MLEWG:
			xCol = 3
			yCol = 4
			zCol = -1
			break
		default:
			Abort "Unable to deduce the kind of localized positions (check the wavenote)"
			break
		endswitch
End

Function GetColumnsForLocalizationError(pos, xDeviation, yDeviation, zDeviation)
	wave pos
	variable &xDeviation, &yDeviation, &zDeviation
	
	Assert(WaveExists(pos))
	
	// The different types of localization routines in the software all return their own types of parameters for every positions
	// depending on what kind of information is offered by the routine
	// This means that the column containing the deviations on the x, y, and possibly z positions could be located in different columns of the positions wave
	// this function will inspect the type of positions and return the column indices containing that information by reference
	// if a particular type does not support one of these then -1 will be returned
	
	// the kind of localization used in generating the positions must be set in the wave note
	
	// special case: if LOCALIZATION METHOD is not set, then assume that the positions originate from a symmetric 2D Gaussian
	// this might be the case if the user is using positions fitted with a previous version of the package
	if (NumType(NumberByKey("LOCALIZATION METHOD", note(pos))) == 2)	// NaN
		xDeviation = 8
		yDeviation = 9
		zDeviation = -1
		// append the information for future use
		Note /NOCR pos, "LOCALIZATION METHOD:" + num2str(LOCALIZATION_GAUSS_FITTING) + ";"
	endif
	
	switch (NumberByKey("LOCALIZATION METHOD", note(pos)))
		case LOCALIZATION_GAUSS_FITTING:
			xDeviation = 8
			yDeviation = 9
			zDeviation = -1
			break
		case LOCALIZATION_GAUSS_FITTING_FIX:
			xDeviation = 6
			yDeviation = 7
			zDeviation = -1
			break
		case LOCALIZATION_CENTROID:
			xDeviation = -1
			yDeviation = -1
			zDeviation = -1
			break
		case LOCALIZATION_MULTIPLICATION:
			xDeviation = -1
			yDeviation = -1
			zDeviation = -1
			break
		case LOCALIZATION_ZEISSPALM:
			xDeviation = 4
			yDeviation = 4
			zDeviation = -1
			break
		case LOCALIZATION_ELLIPSOIDAL2DGAUSS:
			xDeviation = 11
			yDeviation = 12
			zDeviation = -1
			break
		case LOCALIZATION_MLEWG:
			xDeviation = 6
			yDeviation = 6
			zDeviation = -1
			break
		default:
			Abort "Unable to deduce the kind of localized positions (check the wavenote)"
			break
		endswitch
End

Function GetColumnForNFramesPresent(pos, nFramesPresentCol)
	wave pos
	variable &nFramesPresentCol
	
	Assert(WaveExists(pos))
	
	// The different types of localization routines in the software all return their own types of parameters for every positions
	// depending on what kind of information is offered by the routine
	// This means that the column containing the number of frames it is present could be located in a different column of the positions wave
	// this function will inspect the type of positions and return the column indices containing that information by reference
	// if a particular type does not support one of these then -1 will be returned
	
	// the kind of localization used in generating the positions must be set in the wave note
	
	// special case: if LOCALIZATION METHOD is not set, then assume that the positions originate from a symmetric 2D Gaussian
	// this might be the case if the user is using positions fitted with a previous version of the package
	if (NumType(NumberByKey("LOCALIZATION METHOD", note(pos))) == 2)	// NaN
		nFramesPresentCol = 11
		// append the information for future use
		Note /NOCR pos, "LOCALIZATION METHOD:" + num2str(LOCALIZATION_GAUSS_FITTING) + ";"
	endif
	
	switch (NumberByKey("LOCALIZATION METHOD", note(pos)))
		case LOCALIZATION_GAUSS_FITTING:
			nFramesPresentCol = 11
			break
		case LOCALIZATION_GAUSS_FITTING_FIX:
			nFramesPresentCol = 9
			break
		case LOCALIZATION_CENTROID:
			nFramesPresentCol = 3
			break
		case LOCALIZATION_MULTIPLICATION:
			nFramesPresentCol = 4
			break
		case LOCALIZATION_ZEISSPALM:
			nFramesPresentCol = 5
			break
		case LOCALIZATION_ELLIPSOIDAL2DGAUSS:
			nFramesPresentCol = 15
			break
		case LOCALIZATION_MLEWG:
			nFramesPresentCol = 7
			break
		default:
			Abort "Unable to deduce the kind of localized positions (check the wavenote)"
			break
		endswitch
End

Function GetColumnForIntegratedIntensity(pos, integral)
	wave pos
	variable &integral
	
	Assert(WaveExists(pos))
	
	// The different types of localization routines in the software all return their own types of parameters for every positions
	// depending on what kind of information is offered by the routine
	// This means that the column containing integrated intensity could be located in different columns of the positions wave
	// this function will inspect the type of positions and return the column indices containing that information by reference
	// if a particular type does not support one of these then -1 will be returned
	
	// the kind of localization used in generating the positions must be set in the wave note
	
	// special case: if LOCALIZATION METHOD is not set, then assume that the positions originate from a symmetric 2D Gaussian
	// this might be the case if the user is using positions fitted with a previous version of the package
	if (NumType(NumberByKey("LOCALIZATION METHOD", note(pos))) == 2)	// NaN
		integral = 1
		// append the information for future use
		Note /NOCR pos, "LOCALIZATION METHOD:" + num2str(LOCALIZATION_GAUSS_FITTING) + ";"
	endif
	
	switch (NumberByKey("LOCALIZATION METHOD", note(pos)))
		case LOCALIZATION_GAUSS_FITTING:
		case LOCALIZATION_GAUSS_FITTING_FIX:
			integral = 1
			break
		case LOCALIZATION_CENTROID:
			integral = -1
			break
		case LOCALIZATION_MULTIPLICATION:
			integral = -1
			break
		case LOCALIZATION_ZEISSPALM:
			integral = 1
			break
		case LOCALIZATION_ELLIPSOIDAL2DGAUSS:
			integral = 1
			break
		case LOCALIZATION_MLEWG:
			integral = 1
			break
		default:
			Abort "Unable to deduce the kind of localized positions (check the wavenote)"
			break
		endswitch
End

Function GetColumnForFittedWidth(pos, xWidthColumn, yWidthColumn)
	wave pos
	variable &xWidthColumn, &yWidthColumn
	
	Assert(WaveExists(pos))
	
	// The different types of localization routines in the software all return their own types of parameters for every positions
	// depending on what kind of information is offered by the routine
	// This means that the columns containing the fitted widths could be located in different columns of the positions wave
	// this function will inspect the type of positions and return the column indices containing that information by reference
	// if a particular type does not support one of these then -1 will be returned
	
	// the kind of localization used in generating the positions must be set in the wave note
	
	// special case: if LOCALIZATION METHOD is not set, then assume that the positions originate from a symmetric 2D Gaussian
	// this might be the case if the user is using positions fitted with a previous version of the package
	if (NumType(NumberByKey("LOCALIZATION METHOD", note(pos))) == 2)	// NaN
		xWidthColumn = 2
		yWidthColumn = 2
		// append the information for future use
		Note /NOCR pos, "LOCALIZATION METHOD:" + num2str(LOCALIZATION_GAUSS_FITTING) + ";"
	endif
	
	switch (NumberByKey("LOCALIZATION METHOD", note(pos)))
		case LOCALIZATION_GAUSS_FITTING:
			xWidthColumn = 2
			yWidthColumn = 2
			break
		case LOCALIZATION_GAUSS_FITTING_FIX:
			xWidthColumn = -1
			yWidthColumn = -1
			break
		case LOCALIZATION_CENTROID:
			xWidthColumn = -1
			yWidthColumn = -1
			break
		case LOCALIZATION_MULTIPLICATION:
			xWidthColumn = -1
			yWidthColumn = -1
			break
		case LOCALIZATION_ZEISSPALM:
			xWidthColumn = -1
			yWidthColumn = -1
			break
		case LOCALIZATION_ELLIPSOIDAL2DGAUSS:
			xWidthColumn = 2
			yWidthColumn = 3
			break
		case LOCALIZATION_MLEWG:
			xWidthColumn = 2
			yWidthColumn = 3
			break
		default:
			Abort "Unable to deduce the kind of localized positions (check the wavenote)"
			break
		endswitch
End

Function CBDisplayFittedPositionsProc(cba) : CheckBoxControl
	STRUCT WMCheckboxAction &cba

	switch( cba.eventCode )
		case 2: // mouse up
			Variable checked = cba.checked
			
			GenerateGUIThreshold()
				
			break
	endswitch

	return 0
End

Function PMUpdateSegmentationProc(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	switch( pa.eventCode )
		case 2: // mouse up
			GenerateGUIThreshold()
			break
	endswitch
	return 0
End

Function SaveCCDImagesAs()
	
	NVAR cameraType = root:Packages:PALM:V_cameraType
	SVAR CCDFilePath = root:Packages:PALM:S_CCD_File_PATH
	
	string outputFormatStr, outputFilePath, extension
	variable outputFormat
	
	DoWindow /F CCDViewer
	if (V_flag != 1)
		Abort "No CCD file seems to have been opened"
	endif
	
	Prompt outputFormatStr, "Output format:", popup, "TIFF File;Compressed TIFF File;PDE File;QuickTime Movie"
	DoPrompt "Choose the output format", outputFormatStr
	if (V_flag == 1)
		return 0
	endif
	
	StrSwitch (outputFormatStr)
		case "TIFF File":
			outputFormat = IMAGE_OUTPUT_TYPE_TIFF
			extension = ".tif"
			break
		case "Compressed TIFF File":
			outputFormat = IMAGE_OUTPUT_TYPE_COMPR_TIFF
			extension = ".tif"
			break
		case "PDE File":
			outputFormat = IMAGE_OUTPUT_TYPE_PDE
			extension = ".pde"
			break
		case "QuickTime Movie":
			MakeMovieFromOpenFile()
			return 0
			break
		default:
			Abort "Internal error in SaveCCDImagesAs(): no such format"
	EndSwitch
	
	variable refNum
	String fileFilter = "Image files (" + extension + "):" + extension
	Open /F=(fileFilter) /D refNum
	
	outputFilePath = S_fileName
	if (strlen(S_fileName) == 0)
		return 0
	endif
	
	ProcessCCDImages /O /Y=(cameraType) /M=(PROCESSCCDIMAGE_CONVERTFORMAT) /OUT=(outputFormat) CCDFilePath, outputFilePath
	
End

Function SetupCCDProcessingPanel()
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	Variable /G root:Packages:PALM:V_cameraMultiplicationFactor
	Variable /G root:Packages:PALM:V_cameraOffset
	Variable /G root:Packages:PALM:V_nFramesRollingAverage
	
	NVAR cameraMultiplicationFactor = root:Packages:PALM:V_cameraMultiplicationFactor
	NVAR cameraOffset = root:Packages:PALM:V_cameraOffset
	NVAR nFramesRollingAverage = root:Packages:PALM:V_nFramesRollingAverage
	
	if (cameraMultiplicationFactor == 0)
		cameraMultiplicationFactor = 1
	endif
	
	Make /O/N=(0, 2) /T root:Packages:PALM:M_BatchProcessListWave
	Make /O/B/U/N=(0, 2, 2) root:Packages:PALM:M_BatchProcessSelectionWave
	Make /O/W/U/N=(2, 3) root:Packages:PALM:M_BatchProcessColorWave
	Make /O/N=2 /T root:Packages:PALM:M_BatchProcessColumnHeadings
	wave /T M_BatchProcessColumnHeadings = root:Packages:PALM:M_BatchProcessColumnHeadings
	wave M_BatchProcessColorWave = root:Packages:PALM:M_BatchProcessColorWave
	wave M_BatchProcessSelectionWave = root:Packages:PALM:M_BatchProcessSelectionWave
	
	M_BatchProcessColumnHeadings[0] = "CCD Files"
	M_BatchProcessColumnHeadings[1] = "Save file as"
	
	M_BatchProcessColorWave = 0
	M_BatchProcessColorWave[1][0] = 65535
	SetDimLabel 2,1, backColors, M_BatchProcessSelectionWave
	M_BatchProcessSelectionWave[][][1] = 0	// provide the default Igor colors
	
	
	DoWindow /F CCDProcessPanel
	if (V_flag != 1)
		NewPanel /K=1 /N=CCDProcessPanel /W=(81,56,638,470) as "Process CCD data"
		PopupMenu PMSelectProcessingMethod,pos={9,8},size={238,20},bodyWidth=250,title="Processing method:",proc=PMSelectProcessingMethodProc
		PopupMenu PMSelectProcessingMethod,mode=1,value= #"\"Subtract Pixelwise Average;Difference Image;Convert to Different Format;Convert to Photons\""
		PopupMenu PMSelectProcessingMethod,userdata(ResizeControlsInfo)= A"!!*'\"!!#:b!!#Bc!!#<Xz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
		PopupMenu PMSelectProcessingMethod,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
		PopupMenu PMSelectProcessingMethod,userdata(ResizeControlsInfo) += A"zzz!!#u:Du]k<zzzzzzzzzzzzzz!!!"
		PopupMenu PMSelectProcessingOutputType,pos={36,36},size={261,20},bodyWidth=200,title="Save data as:"
		PopupMenu PMSelectProcessingOutputType,mode=1,value= #"\"TIFF file;Zip-Compressed TIFF File;PDE file;Igor wave\""
		SetVariable SVRollingAverageFrames,pos={7,64},size={250,15},title="Frames to average over (0 = all frames)"
		SetVariable SVRollingAverageFrames,limits={0,inf,1},value=nFramesRollingAverage
		SetVariable SVCameraMultiplication,pos={7,64},size={200,15},title="Camera multiplication factor:"
		SetVariable SVCameraMultiplication,limits={1e-06,inf,1},value=cameraMultiplicationFactor,disable=1
		SetVariable SVCameraMultiplication,userdata(ResizeControlsInfo)= A"!!,@C!!#?9!!#AW!!#<(z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
		SetVariable SVCameraMultiplication,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
		SetVariable SVCameraMultiplication,userdata(ResizeControlsInfo) += A"zzz!!#u:Duafnzzzzzzzzzzzzzz!!!"
		SetVariable SVCameraOffset,pos={7,84},size={200,15},title="Camera offset:"
		SetVariable SVCameraOffset,value=cameraOffset,disable=1
		SetVariable SVCameraOffset,userdata(ResizeControlsInfo)= A"!!,@C!!#?a!!#AW!!#<(z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
		SetVariable SVCameraOffset,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
		SetVariable SVCameraOffset,userdata(ResizeControlsInfo) += A"zzz!!#u:Duafnzzzzzzzzzzzzzz!!!"
		Button BTDoCCDFileProcessing,pos={447,383},size={100,20},title="Process!",proc=BTDoCCDFileProcessing
		Button BTDoCCDFileProcessing,userdata(ResizeControlsInfo)= A"!!,IEJ,hsOJ,hpW!!#<Xz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
		Button BTDoCCDFileProcessing,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		Button BTDoCCDFileProcessing,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		Button BTBatchProcessAddFiles,pos={334,383},size={100,20},proc=BTBatchAddDataFiles,title="Add files..."
		Button BTBatchProcessAddFiles,userdata(ResizeControlsInfo)= A"!!,Hb!!#C$J,hpW!!#<Xz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
		Button BTBatchProcessAddFiles,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		Button BTBatchProcessAddFiles,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		ListBox LBBatchListCCDFilesToProcess,pos={7,114},size={542,263},proc=LBDeleteDoubleClickedRowProc
		ListBox LBBatchListCCDFilesToProcess,frame=2
		ListBox LBBatchListCCDFilesToProcess,listWave=root:Packages:PALM:M_BatchProcessListWave
		ListBox LBBatchListCCDFilesToProcess,selWave=root:Packages:PALM:M_BatchProcessSelectionWave
		ListBox LBBatchListCCDFilesToProcess,colorWave=root:Packages:PALM:M_BatchProcessColorWave
		ListBox LBBatchListCCDFilesToProcess,titleWave=root:Packages:PALM:M_BatchProcessColumnHeadings
		ListBox LBBatchListCCDFilesToProcess,mode= 1,selRow= 0,userColumnResize= 1
		ListBox LBBatchListCCDFilesToProcess,userdata(ResizeControlsInfo)= A"!!,@C!!#@H!!#ClJ,hrhJ,fQL!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
		ListBox LBBatchListCCDFilesToProcess,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
		ListBox LBBatchListCCDFilesToProcess,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		SetWindow kwTopWin,hook(ResizeControls)=ResizeControls#ResizeControlsHook
		SetWindow kwTopWin,userdata(ResizeControlsInfo)= A"!!*'\"z!!#Cp5QF0tzzzzzzzzzzzzzzzzzzzzz"
		SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzzzzzzzz"
		SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzz!!!"
	endif
End

Function PMSelectProcessingMethodProc(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	switch( pa.eventCode )
		case 2: // mouse up
			if (StringMatch(pa.popStr, "Subtract Pixelwise Average"))
				ModifyControl SVRollingAverageFrames win=CCDProcessPanel,disable=0
			else
				ModifyControl SVRollingAverageFrames win=CCDProcessPanel,disable=1
			endif
			
			if (StringMatch(pa.popStr, "Convert to Photons"))
				ModifyControl  SVCameraMultiplication win=CCDProcessPanel,disable=0
				ModifyControl  SVCameraOffset win=CCDProcessPanel,disable=0
			else
				ModifyControl  SVCameraMultiplication win=CCDProcessPanel,disable=1
				ModifyControl  SVCameraOffset win=CCDProcessPanel,disable=1
			endif
		break
	endswitch
End

Function BTDoCCDFileProcessing(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	switch( ba.eventCode )
		case 2: // mouse up
				// click code here
				NVAR cameraMultiplicationFactor = root:Packages:PALM:V_cameraMultiplicationFactor
				NVAR cameraOffset = root:Packages:PALM:V_cameraOffset
				NVAR nFramesRollingAverage = root:Packages:PALM:V_nFramesRollingAverage
				
				variable cameraType
				variable processMethod
				variable outputFormat
				variable newCameraType
				string outputFilePath, firstProcessedFilePath, inputFile
				
				wave /T M_BatchProcessListWave = root:Packages:PALM:M_BatchProcessListWave
				wave M_BatchProcessSelectionWave = root:Packages:PALM:M_BatchProcessSelectionWave
				wave M_BatchProcessColorWave = root:Packages:PALM:M_BatchProcessColorWave
				
				variable nFiles = DimSize(M_BatchProcessListWave, 0), i, j, identicalnames
				if (nFiles == 0)
					Abort "Please add some files first"
				endif
			
				// make sure that none of the cells are highlighted
				M_BatchProcessSelectionWave[][][1] = 0	// provide the default Igor colors
				
				// do any of the positions have identical names?
				for (i = 0; i < nFiles - 1; i+=1)
					for (j = i + 1; j < nFiles; j+=1)
						if (StringMatch(M_BatchProcessListWave[i][1], M_BatchProcessListWave[j][1]) == 1)
							M_BatchProcessSelectionWave[i][1][1] = 1// give the cell a red color
							M_BatchProcessSelectionWave[j][1][1] = 1
							identicalNames = 1
						endif
					endfor
				endfor
				
				if (identicalNames == 1)
					Abort "Some of the output file paths are identical, please correct these"
				endif
				
				ControlInfo /W=CCDProcessPanel PMSelectProcessingMethod
				StrSwitch (S_Value)
					case "Subtract Pixelwise Average":
						processMethod = PROCESSCCDIMAGE_SUBAVG
						break
					case "Difference Image":
						processMethod = PROCESSCCDIMAGE_DIFFIMAGE
						break
					case "Convert to Different Format":
						processMethod = PROCESSCCDIMAGE_CONVERTFORMAT
						break
					case "Convert to photons":
						processMethod = PROCESSCCDIMAGE_CONVERTPHOTONS
						break
					default:
						Abort "Unknown CCD processing method in StrSwitch"
				EndSwitch
				
				ControlInfo /W=CCDProcessPanel PMSelectProcessingOutputType
				StrSwitch (S_Value)
					case "TIFF file":
						outputFormat = IMAGE_OUTPUT_TYPE_TIFF
						newCameraType = CAMERA_TYPE_TIFF
						break
					case "Zip-Compressed TIFF File":
						outputFormat = IMAGE_OUTPUT_TYPE_COMPR_TIFF
						newCameraType = CAMERA_TYPE_TIFF
						break
					case "PDE File":
						outputFormat = IMAGE_OUTPUT_TYPE_PDE
						newCameraType = CAMERA_TYPE_PDE
						break
					case "Igor wave":
						outputFormat = IMAGE_OUTPUT_TYPE_IGOR
						newCameraType = CAMERA_TYPE_IGOR_WAVE
						break
					default:
						Abort "Unknown processing output type in StrSwitch"
				EndSwitch
				
				// check the provided settings
				if (processMethod == PROCESSCCDIMAGE_SUBAVG)
					if ((nFramesRollingAverage != 0) && (mod(nFramesRollingAverage, 2) == 0))
						Abort "Subtraction of a rolling average requires an odd number of frames for the average"
					endif
				endif
				
				if (outputFormat == IMAGE_OUTPUT_TYPE_IGOR)
					NewDataFolder /O root:'PALM Movies'
				endif
				
				DoWindow /K CCDProcessPanel
				
				// do the actual processing
				for (i = 0; i < nFiles; i+=1)
					inputFile = M_BatchProcessListWave[i][0]
					if (outputFormat == IMAGE_OUTPUT_TYPE_IGOR)
						outputFilePath = "root:'PALM Movies':" + M_BatchProcessListWave[i][1]
					else
						outputFilePath = M_BatchProcessListWave[i][1]
					endif
					Printf "Now processing %s\t\t(file %d out of %d)\r", inputFile, i + 1, nFiles
					
					// determine the input format of the data
					cameraType = GetFileFormat(inputFile)
					
					ProcessCCDImages /O /Y=(cameraType) /M=(processMethod) /CAL={cameraOffset, cameraMultiplicationFactor} /AVG=(nFramesRollingAverage) /OUT=(outputFormat) inputFile, outputFilePath
					if (i == 0)
						firstProcessedFilePath = outputFilePath
					endif
				endfor
				
				SetupImageViewer(firstProcessedFilePath, newCameraType)
				
				return 0
						
	endswitch
End

Function /WAVE ExtractPositionsInFrame(PositionsWave, frameNumber)
	wave PositionsWave
	variable frameNumber
	
	Assert(WaveExists(PositionsWave))
	Assert(frameNumber >= 0)
	
	// given a list of fitted positions, extract all positions that are present in frame frameNumber
	// do not return just the positions wave, but rather return a wave containing 2 wave references
	// the first reference is the positions wave, the second wave contains the indices of the found positions in the
	// original positions wave
	
	variable nPos = DimSize(PositionsWave, 0)
	variable start = 0, stop = nPos - 1, middle
	variable i, positionsHaveBeenConsolidated
	variable nExtractedPos
	
	Make /O/N=2 /WAVE /FREE M_ExtractedPositions
	
	variable nFramesPresentCol
	GetColumnForNFramesPresent(PositionsWave, nFramesPresentCol)
	
	// check if the positions have been consolidated before
	// if they haven't then use a binary search, if they have then perform a full loop
	ImageTransform /G=(nFramesPresentCol) sumCol, PositionsWave
	positionsHaveBeenConsolidated = V_value != DimSize(PositionsWave, 0)
	
	if (positionsHaveBeenConsolidated == 0)
		ImageTransform /G=0 getCol, PositionsWave
		wave W_ExtractedCol
		middle = BinarySearch(W_ExtractedCol, frameNumber)
		if ((middle < 0) || (PositionsWave[middle][0] != frameNumber))	// not found
			Make /D/FREE/O/N=(0, DimSize(PositionsWave, 1)) M_Positions
			Make /I/U/FREE/O/N=0 W_PositionIndices
		else
			for (i = middle; (i >= 0) && (PositionsWave[i][0] == frameNumber); i -= 1)
			endfor
			start = i + 1
			for (i = middle; (i < nPos) && (PositionsWave[i][0] == frameNumber); i += 1)
			endfor
			stop = i - 1
			
			nExtractedPos = stop - start + 1
			
			Make /D/FREE/O/N=(nExtractedPos, DimSize(PositionsWave, 1)) M_Positions
			Make /FREE/O/N=(nExtractedPos)/I/U W_PositionIndices
			
			M_Positions = PositionsWave[p + start][q]
			W_PositionIndices = p + start
		endif
	else		// positions have been consolidated
			// run over all frames up to the current frame to see if any of those emitters is still active
		Make /D/FREE/O/N=(0, DimSize(PositionsWave, 1)) M_Positions
		Make /I/U/FREE/O/N=0 W_PositionIndices
		variable offset = 0
		for (i = 0; (i < nPos) && (PositionsWave[i][0] <= frameNumber); i+=1)
			if ((PositionsWave[i][0] <= frameNumber) && ((PositionsWave[i][0] + PositionsWave[i][nFramesPresentCol]) > frameNumber))
				Redimension /N=(DimSize(M_Positions, 0) + 1, DimSize(M_Positions, 1)) M_Positions
				Redimension /N=(DimSize(W_PositionIndices, 0) + 1) W_PositionIndices
				M_Positions[offset][] = PositionsWave[i][q]
				W_PositionIndices[offset] = i
				offset += 1
			endif
		endfor
	endif
	
	M_ExtractedPositions[0] = M_Positions
	M_ExtractedPositions[1] = W_PositionIndices
	Note /NOCR /K M_ExtractedPositions[0], note(PositionsWave)
	
	KillWaves /Z W_ExtractedCol
	
	return M_ExtractedPositions
End


Function GenerateFittedPositionsWaves(PositionsWave, n)
	wave PositionsWave
	variable n
	
	Assert(WaveExists(PositionsWave))
	Assert(n >= 0)
	
	variable numberOfPositions, xCol, yCol, zCol
	
	wave /WAVE M_ExtractedPositions = ExtractPositionsInFrame(PositionsWave, n)
	wave positions = M_ExtractedPositions[0]
	
	
	numberOfPositions = DimSize(positions, 0)
	
	if (numberOfPositions == 0)	// no particles were found in this image
									// due to the (perhaps rather poor) design of the GUI
									// we need to create the waves anyway
									// we give them a size of 1 point, set to NaN
		Make /D/O/N=(numberOfPositions) root:Packages:PALM:ps_temp_X = NaN
		Make /D/O/N=(numberOfPositions) root:Packages:PALM:ps_temp_Y = NaN
	endif
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(PositionsWave, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to GenerateFittedPositionsWaves() do not appear to contain any (x,y) information"
	endif
	
	// are these positions subsets starting from a particular origin?
	variable xStart, yStart
	if (NumType(NumberByKey("X START", note(PositionsWave))) != 2)
		xStart = NumberByKey("X START", note(PositionsWave))
	else
		xStart = 0
	endif
	
	if (NumType(NumberByKey("Y START", note(PositionsWave))) != 2)
		yStart = NumberByKey("Y START", note(PositionsWave))
	else
		yStart = 0
	endif
	
	Make /D/O/N=(numberOfPositions) root:Packages:PALM:ps_temp_X = positions[p][xCol] + xStart
	Make /D/O/N=(numberOfPositions) root:Packages:PALM:ps_temp_Y = positions[p][yCol] + yStart
End

Function BTAnalyzeCCDImagesProc(ba) : ButtonControl
	STRUCT WMButtonAction &ba

	switch( ba.eventCode )
		case 2: // mouse up
			// click code here
			DFREF PALMPackage= root:Packages:PALM
			DFREF savDF = GetDataFolderDFR()
			
			variable analysisMethod
			NVAR cameraType = PALMPackage:V_cameraType
			SVAR CCDFilePath = PALMPackage:S_CCD_File_PATH
			
			string fileName = ParseFilePath(5, CCDFilePath, ":", 0, 0)
			fileName = ParseFilePath(3, fileName, ":", 0, 0)
			
			ControlInfo /W=AnalyzeImages PMChooseCCDAnalysisMethod
			StrSwitch (S_Value)
				case "Average Intensity Trace":
					analysisMethod = ANALYZECCD_AVERAGETRACE
					break
				case "Summed Intensity Trace":
					analysisMethod = ANALYZECCD_SUMMEDINTENSITYTRACE
					break
				case "Average Image":
					analysisMethod = ANALYZECCD_AVERAGE_IMAGE
					break
				case "Standard Deviation":
					analysisMethod = ANALYZECCD_STANDARD_DEV_IMAGE
					break
				case "Bleaching Analysis":
					analysisMethod = ANALYZECCD_BLEACHING_ANALYSIS
					break
				default:
					Abort "Unknown CCD analysis method"
			EndSwitch
			
			switch (analysisMethod)
				case ANALYZECCD_SUMMEDINTENSITYTRACE:
					AnalyzeCCDImages /Y=(cameraType) /M=(analysisMethod) /DEST=root:Packages:PALM:W_SummedIntensityTrace CCDFilePath
					// now display the wave
					DoWindow /F Intensity_Trace
					if (V_flag == 0)
						wave W_SummedIntensityTrace = PALMPackage:W_SummedIntensityTrace
						Display /K=1 /N=Intensity_Trace W_SummedIntensityTrace as "Summed intensity trace from " + fileName
					endif
					break
				case ANALYZECCD_AVERAGETRACE:
					AnalyzeCCDImages /Y=(cameraType) /M=(analysisMethod) /DEST=root:Packages:PALM:W_AverageIntensityTrace CCDFilePath
					// now display the wave
					DoWindow /F Average_Trace
					if (V_flag == 0)
						wave W_AverageIntensityTrace = PALMPackage:W_AverageIntensityTrace
						Display /K=1 /N=Average_Trace W_AverageIntensityTrace as "Average intensity trace from " + fileName
					endif
					break
				case ANALYZECCD_AVERAGE_IMAGE:	// calculate a image containing an average of the entire trace
					AnalyzeCCDImages /Y=(cameraType) /M=(analysisMethod) /DEST=root:Packages:PALM:M_AverageImage CCDFilePath
					// now display the wave
					wave M_AverageImage = PALMPackage:M_AverageImage
					DoWindow /F AverageImage
					if (V_flag == 0)
						Display /K=1 /N=AverageImage as "Average image from " + fileName
						AppendImage /W=AverageImage M_AverageImage
						ModifyGraph /W=$S_name width={Aspect, DimSize(M_AverageImage, 0) / DimSize(M_AverageImage, 1)}, height=0
					endif
					break
				case ANALYZECCD_STANDARD_DEV_IMAGE:	// calculate an image with the standard deviation of each pixel
					AnalyzeCCDImages /Y=(cameraType) /M=(analysisMethod) /DEST=root:Packages:PALM:M_StandardDeviation CCDFilePath
					DoWindow /F StandardDeviation
					wave M_StandardDeviation = PALMPackage:M_StandardDeviation
					if (V_flag == 0)
						display /K=1 /N=StandardDeviation as "Standard deviation from " + fileName
						AppendImage /W=StandardDeviation M_StandardDeviation
						ModifyGraph /W=$S_name width={Aspect, DimSize(M_StandardDeviation, 0) / DimSize(M_StandardDeviation, 1)}, height=0
					endif
					break
				case ANALYZECCD_BLEACHING_ANALYSIS:	// try to localize overlapping emitters by running a bleaching analysis
					RecoverEmittersByBleachingSteps()
					break
				default:
					Abort "Unknown CCD analysis method"
					break
			endswitch
			
			break
	endswitch
	
	return 0
End

Function GetCCDDimensionsFromPositions(positions, xSize, ySize, pixelSize)
	wave positions
	variable &xSize, &ySize, &pixelSize
	
	Assert(WaveExists(positions))
	
	// given a wave containing positions, try to deduce the x and y size (in pixels) of the original data based on metadata
	// if the pixel size is known then extract that as well, otherwise return 0
	// the primary focus of this function is to isolate upstream functions from having to manipulate the wavenote directly
	
	xSize = str2num(StringByKey("X SIZE", note(positions)))
	ySize = str2num(StringByKey("Y SIZE", note(positions)))
	
	if ((NumType(xSize) == 2) || (NumType(ySize) == 2))	
		// we didn't find the xSize and/or ySize from the waveNote. 
		// Try to get the x- and y-sizes from the variables defined when loading the images
		NVAR loadingXSize = root:Packages:PALM:V_xSize
		NVAR loadingYSize = root:Packages:PALM:V_ySize
		
		if ((NVAR_Exists(loadingXSize) + NVAR_Exists(loadingYSize)) != 2)
			Abort "Unable to determine the x- and y-size of the original data"
		endif
		
		Print "Warning: using the x- and y-dimensions determined from the CCD file that you loaded last, which may be incorrect if you loaded another CCD file since then."
		
		xSize = loadingXSize
		ySize = loadingYSize
	endif
	
	if (numtype(NumberByKey("X PIXEL SIZE", note(positions))) != 2)
		pixelSize = NumberByKey("X PIXEL SIZE", note(positions))
	else
		pixelSize = 0
	endif
End

Function GenerateScatterPlot(positions, xSize, ySize, pixelSize, colorScaleString)
	wave positions
	variable xSize, ySize, pixelSize
	string colorScaleString	// the name of an Igor color table
	
	Assert(WaveExists(positions))
	Assert((xSize > 0) && (ySize > 0))
	
	variable n_positions = DimSize(positions, 0)
	variable xCol, yCol, zCol
	
	variable number_of_images = positions[n_positions - 1][0] - positions[0][0] + 1
	variable pixelSizeUM = pixelSize / 1000
	
	Make /D/O/N=(n_positions) Scatter_positions_X
	Make /D/O/N=(n_positions) Scatter_positions_Y
	wave Scatter_positions_X
	wave Scatter_positions_Y
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(positions, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to GenerateScatterPlot() do not appear to contain any (x,y) information"
	endif
	
	Scatter_positions_X = positions[p][xCol]
	Scatter_positions_Y = positions[p][yCol]
	
	if (pixelSize != 0)
		Scatter_positions_X *= pixelSizeUM
		Scatter_positions_Y *= pixelSizeUM
		SetScale /P x, 0, 1, "um", Scatter_positions_X
		SetScale /P x, 0, 1, "um", Scatter_positions_Y
	else
		SetScale /P x, 0, 1, "pixel", Scatter_positions_X
		SetScale /P x, 0, 1, "pixel", Scatter_positions_Y
	endif
	
	if (strlen(colorScaleString) != 0)
		// choose the colors
		ColorTab2Wave $colorScaleString
		wave M_Colors
		Make /O/N=(n_positions, 3) Scatter_positions_color
		wave Scatter_positions_color = Scatter_positions_color
		
		variable number_of_colors = DimSize(M_Colors, 0)
		variable color_step_size = (number_of_colors - 1) / number_of_images
		
		Scatter_positions_color = M_Colors[floor(positions[p][0] * color_step_size)][q]
	endif
	
	KillWaves /Z M_Colors
End


Function GenerateScatterPlot_Menu()
	string posName
	string windowName
	variable colorScaleIndex
	string colorScaleString = "None;BlueRedGreen;Rainbow;YellowHot;"
	
	Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
	Prompt colorScaleIndex, "Color scale:", popup, colorScaleString
	
	DoPrompt "Select the positions wave", posName, colorScaleIndex
	if (V_flag == 1)	// cancel
		return 0
	endif
	
	colorScaleString = StringFromList(colorScaleIndex - 1, colorScaleString)
	if (StringMatch(colorScaleString, "None") == 1)
		colorScaleString = ""
	endif
	
	wave pos = GetPositionsWaveReference(posName)
	if (WaveExists(pos) == 0)
		Abort "The selected wave doesn't seem to exist!"
	endif
	
	variable xSize, ySize, pixelSize
	GetCCDDimensionsFromPositions(pos, xSize, ySize, pixelSize)
	
	DFREF savDF = GetDataFolderDFR()
	NewDataFolder /O root:'PALM Images'
	NewDataFolder /S/O root:'PALM Images':$posName
	GenerateScatterPlot(pos, xSize, ySize, pixelSize, colorScaleString)
	wave Scatter_positions_X =  Scatter_positions_X 
	wave Scatter_positions_Y =  Scatter_positions_Y
	wave Scatter_positions_color = Scatter_positions_color
	SetDataFolder savDF
	
	windowName = "Scatter plot from " + posName
	
	Display /K=1 /N=Scatter_Viewer Scatter_positions_Y vs Scatter_positions_X as windowName
	if (strlen(colorScaleString) != 0)
		ModifyGraph zColor(Scatter_positions_Y)={Scatter_positions_color,*,*,directRGB,0}
	endif
	ModifyGraph mode=3,marker=41,msize=2,tickUnit=1
	ModifyGraph /W=$S_name width={Aspect, xSize / ySize}, height=0
	
	if (pixelSize == 0)
		SetAxis /W=$S_name bottom, 0, xSize - 1
		SetAxis /W=$S_name left, 0, ySize - 1
		Label /W=$S_name left "Distance (pixel)"
		Label /W=$S_name bottom "Distance (pixel)"
	else
		SetAxis /W=$S_name bottom, 0, (xSize - 1) * pixelSize / 1000
		SetAxis /W=$S_name left, 0, (ySize - 1) * pixelSize / 1000
		Label /W=$S_name left "Distance (m)"
		Label /W=$S_name bottom "Distance (m)"
	endif
	
End

Function /S RemoveFitPositionMenuName()
	
	// get the name of the topmost graph
	string windowName = WinName(0, 1, 1)
	
	if (StringMatch(windowName, "CCDViewer") != 1)
		// this not the CCDViewer, do not show the menu entry
		return ""
	endif
	
	// check if we are showing positions
	string traceNames = TraceNameList("CCDViewer", ";", 1)
	if (StringMatch(traceNames, "*ps_temp_Y;*") != 1)
		// not showing positions, do not show the entry
		return ""
	endif
	
	// if we're still here then everything appears fine
	return "Remove closest localized position"
End

Function /S RemovePositionsInFrameMenuName()
	
	// get the name of the topmost graph
	string windowName = WinName(0, 1, 1)
	
	if (StringMatch(windowName, "CCDViewer") != 1)
		// this not the CCDViewer, do not show the menu entry
		return ""
	endif
	
	// check if we are showing positions
	string traceNames = TraceNameList("CCDViewer", ";", 1)
	if (StringMatch(traceNames, "*ps_temp_Y;*") != 1)
		// not showing positions, do not show the entry
		return ""
	endif
	
	// if we're still here then everything appears fine
	return "Remove positions fitted in this frame"
End

Function RemoveFittedPositionContextMenu()
	// allow the user to delete a specific fit in a specific frame
	// this function assumes that the coordinates where the user right-clicked
	// are provided by the hook function in CCDViewer
	
	NVAR clickX = root:Packages:PALM:V_RightClickX
	NVAR clickY = root:Packages:PALM:V_RightClickY
	NVAR currentImage = root:Packages:PALM:V_currentImage
	
	variable nearestX, nearestY, index
	
	// get a reference to the positions wave
	DoWindow CCD_Control
	if (V_flag != 1)
		// the CCD_Control window does not exist or is invisible
		// we're not sure what positions wave the user wants
		Abort "The localization analysis control panel is not open, I'm not sure of what positions to take"
	endif
	
	ControlInfo /W=CCD_Control PMSelectPositionsWave
	string positionsWaveName = S_Value
	wave positions = GetPositionsWaveReference(positionsWaveName)
	
	if (WaveExists(positions) != 1)
		Abort "Unable to find the positions wave"
	endif
	
	// get the positions in the current frame
	wave /WAVE M_ExtractedPositions = ExtractPositionsInFrame(positions, currentImage)
	wave extractedPositions = M_ExtractedPositions[0]
	wave extractedPositionsIndices = M_ExtractedPositions[1]
	
	if (DimSize(extractedPositions, 0) == 0)
		// no positions found in this frame, this should not occur
		return 0
	endif
	
	returnPointNearestFitPositions(extractedPositions, clickX, clickY, nearestX, nearestY, index)
	
	DeletePoints /M=0 (extractedPositionsIndices[index]), 1, positions
	
	// update the threshold
	GenerateGUIThreshold()
End

Function RemovePositionsFrameContextMenu()
	NVAR currentImage = root:Packages:PALM:V_currentImage
	
	// get a reference to the positions wave
	DoWindow CCD_Control
	if (V_flag != 1)
		// the CCD_Control window does not exist or is invisible
		// we're not sure what positions wave the user wants
		Abort "The localization analysis control panel is not open, I'm not sure of what positions to take"
	endif
	
	ControlInfo /W=CCD_Control PMSelectPositionsWave
	string positionsWaveName = S_Value
	wave positions = GetPositionsWaveReference(positionsWaveName)
	
	if (WaveExists(positions) != 1)
		Abort "Unable to find the positions wave"
	endif
	
	// get the positions in the current frame
	wave /WAVE M_ExtractedPositions = ExtractPositionsInFrame(positions, currentImage)
	wave extractedPositions = M_ExtractedPositions[0]
	wave extractedPositionsIndices = M_ExtractedPositions[1]
	
	variable nPositionsToDelete = DimSize(extractedPositions, 0)
	if (nPositionsToDelete == 0)
		// no positions found in this frame, this should not occur
		return 0
	endif
	
	DoAlert 1, "Delete " + num2str(nPositionsToDelete) + " positions?"
	if (V_flag != 1)	// user did not pick 'yes'
		return 0
	endif
	
	variable i
	for (i = nPositionsToDelete - 1; i >= 0; i-=1)
		DeletePoints /M=0 extractedPositionsIndices[i], 1, positions
	endfor
	
	 // update the threshold
	GenerateGUIThreshold()
End

Function selectPositionsWithinBox(positions, startX, endX, startY, endY)
	wave positions
	variable startX, startY, endX, endY
	
	Assert(WaveExists(positions))
	Assert((endX > startX) && (endY > startY))
	
	variable n_pos = DimSize(positions, 0), n_pos_within_box = 0
	variable i, offset = 0, xCol, yCol, zCol
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(positions, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to selectPositionsWithinBox() do not appear to contain any (x,y) information"
	endif
	
	for (i = 0; i < n_pos; i+=1)
		if ((positions[i][xCol] >= startX) && (positions[i][xCol] <= endX) && (positions[i][yCol] >= startY) && (positions[i][yCol] <= endY))
			n_pos_within_box += 1
		endif
	endfor
	
	Make /O/D/N=(n_pos_within_box) Box_positions_X
	Make /O/D/N=(n_pos_within_box) Box_positions_Y
	
	for (i = 0; i < n_pos; i+=1)
		if ((positions[i][xCol] >= startX) && (positions[i][xCol] <= endX) && (positions[i][yCol] >= startY) && (positions[i][yCol] <= endY))
			Box_positions_X[offset] = positions[i][xCol]
			Box_positions_Y[offset] = positions[i][yCol]
			offset += 1
		endif
	endfor
	
End

Function MakeSummedTraceFromMarquee()
	
	SVAR CCDFilePath = root:Packages:PALM:S_CCD_file_path
	NVAR cameraType = root:Packages:PALM:V_cameraType
	NVAR xSize = root:Packages:PALM:V_xSize
	NVAR ySize = root:Packages:PALM:V_ySize
	
	if ((SVAR_Exists(CCDFilePath) == 0) || (NVAR_Exists(xSize) == 0) || (NVAR_Exists(ySize) == 0) || (NVAR_Exists(cameraType) == 0))
		Abort "It doesn't look like a CCD file has been opened in this experiment"
	endif
	
	Variable bottom, left, right, top
	
	GetMarquee /Z left, bottom
	if (V_flag == 0)
		return -1
	endif
	
	if (cmpstr(S_marqueeWin, "CCDViewer") != 0)
		Abort "Summing intensities should only be called on the window showing the CCD frames"
	endif
	
	bottom = round(min(V_bottom, V_top))
	top = round(max(V_bottom, V_top))
	left = round(min(V_left, V_right))
	right = round(max(V_left, V_right))
	
	bottom = (bottom < 0) ? 0 : bottom
	left = (left < 0) ? 0 : left
	top = (top >= ySize) ? ySize - 1: top
	right = (right >= xSize) ? xSize - 1: right
	
	if ((top == bottom) || (left == right))	// it's not a real box
		return -1
	endif
	
	// run the operation that will generate the trace
	AnalyzeCCDImages /Y=(cameraType) /M=(ANALYZECCD_SUMMEDINTENSITYTRACE) /ROI={left, right, bottom, top} /DEST=root:Packages:PALM:W_SummedIntensityTrace CCDFilePath
	
	// now display the resulting image
	DoWindow /F Intensity_Trace
	if (V_flag == 0)
		Display /K=1 /N=Intensity_Trace root:Packages:PALM:W_SummedIntensityTrace as "Summed intensity trace"
		Label /W=Intensity_Trace left, "Summed intensity"
		Label /W=Intensity_Trace bottom, "Frame number"
	endif
	
End

Function MakeAverageTraceFromMarquee()
	
	SVAR CCDFilePath = root:Packages:PALM:S_CCD_file_path
	NVAR cameraType = root:Packages:PALM:V_cameraType
	NVAR xSize = root:Packages:PALM:V_xSize
	NVAR ySize = root:Packages:PALM:V_ySize
	
	if ((SVAR_Exists(CCDFilePath) == 0) || (NVAR_Exists(xSize) == 0) || (NVAR_Exists(ySize) == 0) || (NVAR_Exists(cameraType) == 0))
		Abort "It doesn't look like a CCD file has been opened in this experiment"
	endif
	
	Variable bottom, left, right, top
	
	GetMarquee /Z left, bottom
	if (V_flag == 0)
		return -1
	endif
	
	if (cmpstr(S_marqueeWin, "CCDViewer") != 0)
		Abort "Averaging intensities should only be called on the window showing the CCD frames"
	endif
	
	bottom = round(min(V_bottom, V_top))
	top = round(max(V_bottom, V_top))
	left = round(min(V_left, V_right))
	right = round(max(V_left, V_right))
	
	bottom = (bottom < 0) ? 0 : bottom
	left = (left < 0) ? 0 : left
	top = (top >= ySize) ? ySize - 1 : top
	right = (right >= xSize) ? xSize - 1 : right
	
	if ((top == bottom) || (left == right))	// it's not a real box
		return -1
	endif
	
	// run the operation that will generate the trace
	AnalyzeCCDImages /Y=(cameraType) /M=(ANALYZECCD_AVERAGETRACE) /ROI={left, right, bottom, top} /DEST=root:Packages:PALM:W_AverageIntensityTrace CCDFilePath
	
	// now display the resulting image
	DoWindow /F Average_Trace
	if (V_flag == 0)
		Display /K=1 /N=Average_Trace root:Packages:PALM:W_AverageIntensityTrace as "Average intensity trace"
		Label /W=Average_Trace left, "Average intensity"
		Label /W=Average_Trace bottom, "Frame number"
	endif
	
End

Function CBCheckShowParticlesProc(cba) : CheckBoxControl
	STRUCT WMCheckboxAction &cba

	switch( cba.eventCode )
		case 2: // mouse up
			Variable checked = cba.checked
			
			GenerateGUIThreshold()
			
			break
	endswitch

	return 0
End

Function PMSelectParticleFinderProc(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	switch( pa.eventCode )
		case 2: // mouse up
			Variable popNum = pa.popNum
			String popStr = pa.popStr
			
			variable particleFinderMethod = popNum - 1
			if (particleFinderMethod == PARTICLEFINDER_RADIUS)
				SetVariable SVSetMinimumRadius disable=0
			else
				SetVariable SVSetMinimumRadius disable=1
			endif
			GenerateGUIThreshold()
			
			break
	endswitch

	return 0
End

Function PMParticleVerificationProc(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	switch( pa.eventCode )
		case 2: // mouse up
			Variable popNum = pa.popNum
			String popStr = pa.popStr
			
			NVAR particleVerifierOverlap = root:Packages:PALM:V_particleVerifierOverlap
			NVAR particleVerifierSymm = root:Packages:PALM:V_particleVerifierSymm
			NVAR particleVerifierEllipse = root:Packages:PALM:V_particleVerifierEllipse
			
			StrSwitch (popStr)
				case "Remove Overlapping Particles":
					particleVerifierOverlap = !particleVerifierOverlap
					break
				case "Gaussian Fitting":
					particleVerifierSymm = !particleVerifierSymm
					break
				case "Ellipsoidal Gaussian Fitting":
					particleVerifierEllipse = !particleVerifierEllipse
					break
				default:
					Abort "Unknown particle verifier"
			EndSwitch
			
			GenerateGUIThreshold()
			
			break
	endswitch

	return 0
End

Function /S PMParticleVerificationListItems()
	NVAR particleVerifierOverlap = root:Packages:PALM:V_particleVerifierOverlap
	NVAR particleVerifierSymm = root:Packages:PALM:V_particleVerifierSymm
	NVAR particleVerifierEllipse = root:Packages:PALM:V_particleVerifierEllipse
	
	string listString = ""
	string checkMark = "\\M0:!" + num2char(18)+":"
	
	if (particleVerifierOverlap != 0)
		listString += checkMark + "Remove Overlapping Particles" + ";"
	else
		listString += "Remove Overlapping Particles" + ";"
	endif
	
	if (particleVerifierSymm != 0)
		listString += checkMark + "Gaussian Fitting" + ";"
	else
		listString += "Gaussian Fitting" + ";"
	endif
	
	if (particleVerifierEllipse != 0)
		listString += checkMark + "Ellipsoidal Gaussian Fitting" + ";"
	else
		listString += "Ellipsoidal Gaussian Fitting" + ";"
	endif
	
	return listString
		
End

Function SVCurrentFrameFromOneProc(sva) : SetVariableControl
	STRUCT WMSetVariableAction &sva

	switch( sva.eventCode )
		case 1: // mouse up
		case 2: // Enter key
		case 3: // Live update
			Variable dval = sva.dval
			String sval = sva.sval
			
			ChangeCurrentFrame(dval - 1)
			
			break
	endswitch

	return 0
End


Function MakeParticleWavesAndLabel()
	
	DFREF savDF = GetDataFolderDFR()
	SetDataFolder root:Packages:PALM
	
	wave M_locatedParticles
	
	variable nParticles = DimSize(M_locatedParticles, 0)
	
	Make /D/O/N=(nParticles) W_particles_X
	Make /D/O/N=(nParticles) W_particles_Y
	
	W_particles_X = M_locatedParticles[p][1]
	W_particles_Y = M_locatedParticles[p][2]
	SetDataFolder savDF
End


Function PALM_clear()
	
	string savDF = GetDataFolder(1)
	SetDataFolder root:Packages:PALM:
	String /G S_CCD_file_path
	
	S_CCD_file_path = ""
	
	// close all windows
	DoWindow /Z/K CCDViewer
	DoWindow /Z/K CCD_Histogram
	DoWindow /Z/K AnalyzeImages
	DoWindow /Z/K CCD_Control
	
	// kill waves that could possibly be in the packages folder
	KillWaves /Z Viewer_Temp, M_CCDFrames, M_SegmentedImage, M_locatedParticles, W_particles_X, W_particles_Y
	KillWaves /Z ps_temp_X, ps_temp_Y, Scatter_positions_X, Scatter_positions_Y, Scatter_positions_color
	
	// kill any generated images (we don't kill positions!)
	NewDataFolder /O/S root:'PALM Images'
	ZapDataInFolderTree("root:'PALM Images'")
	
	SetDataFolder savDF
End

Function ZapDataInFolderTree(path)
	String path
	
	Assert(strlen(path) > 0)

	String savDF= GetDataFolder(1)
	SetDataFolder path

	KillWaves/A/Z
	KillVariables/A/Z
	KillStrings/A/Z

	Variable i
	Variable numDataFolders = CountObjects(":",4)
	for(i=0; i<numDataFolders; i+=1)
		String nextPath = GetIndexedObjName(":",4,i)
		ZapDataInFolderTree(nextPath)
	endfor

	SetDataFolder savDF
End

Function MakePALMImage_Menu()
	string savDF = GetDataFolder(1)
	NewDataFolder /O root:Packages
	NewDataFolder /O/S root:Packages:PALM
	
	Variable /G root:Packages:PALM:V_PALMBitmapConstantWidth
	Variable /G root:Packages:PALM:V_PALMBitmapFitUncertainty
	Variable /G root:Packages:PALM:V_PALMBitmapCamMagnification
	Variable /G root:Packages:PALM:V_PALMCCDPixelSize
	Variable /G root:Packages:PALM:V_PALMBitmapImageWidth
	Variable /G root:Packages:PALM:V_gaussianWidth
	Variable /G root:Packages:PALM:V_PALMBitmapSelectedMethod
	
	NVAR constantWidth = root:Packages:PALM:V_PALMBitmapConstantWidth
	NVAR fitUncertainty = root:Packages:PALM:V_PALMBitmapFitUncertainty
	NVAR camMagnification = root:Packages:PALM:V_PALMBitmapCamMagnification
	NVAR imageWidth = root:Packages:PALM:V_PALMBitmapImageWidth
	NVAR PSFWidth = root:Packages:PALM:V_gaussianWidth
	NVAR SelectedMethod = root:Packages:PALM:V_PALMBitmapSelectedMethod
	
	// provide initial values if this is the first time that we've used this function
	if (constantWidth == 0)
		constantWidth = 0.5
	endif
	if (fitUncertainty == 0)
		fitUncertainty = 3
	endif
	if (imageWidth == 0)
		imageWidth = 2096
	endif
	if (PSFWidth == 0)
		PSFWidth = 2
	endif
	if (camMagnification == 0)
		camMagnification = 1
	endif
	
	SetDataFolder savDF
	
	DoWindow /F PALMBitmapWindow
	if (V_flag == 0)
		NewPanel /N=PALMBitmapWindow /K=1 /W=(137,135,396,474) as "Enter calculation parameters"
		GroupBox GBPALMBitmapColorBox,pos={4,65},size={250,25}
		PopupMenu PMPALMBitmapSelectPositionsWave,pos={14,8},size={220,20},bodyWidth=150,title="Positions wave:"
		PopupMenu PMPALMBitmapSelectPositionsWave,mode=1,value= #"PALMAnalysis#GetPossiblePositionsWaves()"
		PopupMenu PMPALMBitmapSelectColorTable,pos={35,33},size={198,20},bodyWidth=145,title="Color table:"
		PopupMenu PMPALMBitmapSelectColorTable,mode=2,value= #"\"*COLORTABLEPOPNONAMES*\""
		
		GroupBox GBPALMBitmapEmitterWeighing,pos={4,102},size={250,57},title="Relative position weight:"
		CheckBox CBPALMBitmapSameEmitterWeight,pos={11,119},size={209,14},proc=PALMAnalysis#CBPALMBitmapImage,title="Each localized position has the same weight"
		CheckBox CBPALMBitmapSameEmitterWeight,value= 1,mode=1
		CheckBox CBPALMBitmapPreserveIntegral,pos={11,139},size={188,14},proc=PALMAnalysis#CBPALMBitmapImage,title="Preserve the fitted integrated intensity"
		CheckBox CBPALMBitmapPreserveIntegral,value= 0,mode=1
		
		GroupBox GBPALMBitmapDeviationBox,pos={3,171},size={250,130},title="Spot deviation:"
		CheckBox CBPALMBitmapConstantWidth,pos={10,191},size={168,14},proc=PALMAnalysis#CBPALMBitmapImage,title="Constant width:\t\t\t\tCCD pixels"
		CheckBox CBPALMBitmapConstantWidth,value= 0,mode=1
		CheckBox CBPALMBitmapFitUncertainty,pos={10,214},size={166,14},proc=PALMAnalysis#CBPALMBitmapImage,title="Scale fit uncertainty\t\t\t\ttimes"
		CheckBox CBPALMBitmapFitUncertainty,value= 0,mode=1
		CheckBox CBPALMBitmapGME,pos={10,239},size={130,14},proc=PALMAnalysis#CBPALMBitmapImage,title="Gaussian mask estimation"
		CheckBox CBPALMBitmapGME,value= 0,mode=1
		SetVariable SVPALMBitmapImageWidth,pos={7,69},size={140,15},title="Image width (pixels)"
		SetVariable SVPALMBitmapImageWidth,limits={32,inf,0},value=imageWidth
		SetVariable SVPALMBitmapConstantWidth,pos={103,191},size={30,15},title=" "
		SetVariable SVPALMBitmapConstantWidth,limits={0,inf,0},value=constantWidth
		SetVariable SVPALMBitmapFitUncertainty,pos={121,214},size={30,15},title=" "
		SetVariable SVPALMBitmapFitUncertainty,limits={0,inf,0},value=fitUncertainty
		SetVariable SVPALMCameraMagnification,pos={29,258},size={190,15},title="Camera magnification factor:"
		SetVariable SVPALMCameraMagnification,limits={1e-4,inf,0},value=camMagnification
		SetVariable SVPALMBitmapPSFWidth,pos={28,278},size={190,15},title="PSF width (in CCD pixels):"
		SetVariable SVPALMBitmapPSFWidth,limits={0,inf,0},value=PSFWidth
		Button BTPALMBitmapImageCalculate,pos={179,307},size={75,20},proc=PALMAnalysis#BTPALMBitmapImageCalculateProc,title="Calculate!"
		
		// if the window was opened previously then restore its size and position
		WC_WindowCoordinatesRestore("PALMBitmapWindow")
		SetWindow PALMBitmapWindow hook(WindowCoordsHook)=WC_WindowCoordinatesNamedHook
	endif
	
	// restore the selected method if we used it previously
	Switch (SelectedMethod)
		case 0:
			CheckBox CBPALMBitmapConstantWidth value= 1
			break
		case 1:
			CheckBox CBPALMBitmapFitUncertainty value= 1
			break
		case 2:
			CheckBox CBPALMBitmapGME value = 1
			break
	EndSwitch
	
End

Function /WAVE GetPositionsWaveReference(posName)
	string posName
	
	Assert(strlen(posName) > 0)
	
	if (StringMatch(posName, "POS_out") == 1)
		wave pos = root:Packages:PALM:POS_out
	else
		wave pos = root:'PALM Positions':$posName
	endif
	
	return pos
End

Function BTPALMBitmapImageCalculateProc(ba) : ButtonControl
	STRUCT WMButtonAction &ba

	switch( ba.eventCode )
		case 2: // mouse up
			// click code here
			string savDF = GetDataFolder(1)
			NVAR constantWidth = root:Packages:PALM:V_PALMBitmapConstantWidth
			NVAR uncertaintyScale = root:Packages:PALM:V_PALMBitmapFitUncertainty
			NVAR camMagnification = root:Packages:PALM:V_PALMBitmapCamMagnification
			NVAR imageWidth = root:Packages:PALM:V_PALMBitmapImageWidth
			NVAR PSFWidth = root:Packages:PALM:V_gaussianWidth
			string posName, windowName
			string colorTable
			variable method = -1
			variable weighingMethod = 0
			ControlInfo /W=PALMBitmapWindow PMPALMBitmapSelectPositionsWave
			posName = S_Value
			
			ControlInfo /W=PALMBitmapWindow PMPALMBitmapSelectColorTable
			colorTable = S_value
			
			ControlInfo /W=PALMBitmapWindow CBPALMBitmapSameEmitterWeight
			if (V_value == 1)
				weighingMethod = 0
			endif
			
			ControlInfo /W=PALMBitmapWindow CBPALMBitmapPreserveIntegral
			if (V_value == 1)
				weighingMethod = 1
			endif
			
			ControlInfo /W=PALMBitmapWindow CBPALMBitmapConstantWidth
			if (V_value == 1)
				method = PALMBITMAP_DEV_SAME
			endif
			ControlInfo /W=PALMBitmapWindow CBPALMBitmapFitUncertainty
			if (V_value == 1)
				method = PALMBITMAP_DEV_FITUNCERTAINTY
			endif
			ControlInfo /W=PALMBitmapWindow CBPALMBitmapGME
			if (V_value == 1)
				method = PALMBITMAP_DEV_GAUSSIANMASK
			endif
				
			
			
			wave pos = GetPositionsWaveReference(posName)
			
			if (WaveExists(pos) == 0)
				SetDataFolder savDF
				Abort "The selected wave doesn't seem to exist!"
			endif
			
			NewDataFolder /O root:'PALM Images'
			NewDataFolder /O/S root:'PALM Images':$posName
			
			// if there is already a PALM image in this data folder then the calculation will overwrite it
			// try to kill it before the actual calculation to maybe free up some memory
			KillWaves /Z $"M_LocalizationBitmap"
			
			variable xSize, ySize, pixelSize
			GetCCDDimensionsFromPositions(pos, xSize, ySize, pixelSize)
			
			variable imageScaleFactor = round(imageWidth / xSize)
			variable upperLimit = xSize
			
			switch (method)
				case PALMBITMAP_DEV_SAME:
					LocalizationBitmap /M=(method) /WGHT=(weighingMethod) /S=(constantWidth) /W={xSize, ySize, imageScaleFactor} pos
					break
				case PALMBITMAP_DEV_FITUNCERTAINTY:
					LocalizationBitmap /M=(method) /WGHT=(weighingMethod) /S=(uncertaintyScale) /L=(upperLimit) /W={xSize, ySize, imageScaleFactor} pos
					break
				case PALMBITMAP_DEV_GAUSSIANMASK:
					LocalizationBitmap /M=(method) /WGHT=(weighingMethod) /WDTH=(PSFWidth) /MULT=(camMagnification) /W={xSize, ySize, imageScaleFactor} pos
					break
				default:
					SetDataFolder savDF
					Abort "Unknown method"
					break
			endswitch
			
			wave M_LocalizationBitmap
			windowName = "PALM bitmap image from " + posName
			
			// Typically the histogram for the resulting image is very inhomogeneous
			// a reasonable guess for an initial colorscale appears to be to set min to auto
			// and then take 10% of the maximum for the upper limit
			WaveStats /Q /M=1 M_LocalizationBitmap
			
			Display /K=1 /N=PALMImageViewer as windowName
			AppendImage /W=$S_name M_LocalizationBitmap
			ModifyImage /W=$S_name M_LocalizationBitmap ctab= {*, V_max / 10,$colorTable,0}
			ModifyGraph /W=$S_name width={Aspect, xSize / ySize}, height=0
			ModifyGraph /W=$S_name btLen=2,tickUnit=1
			
			if (pixelSize == 0)
				SetScale /I x, 0, xSize -1, "pixel", M_LocalizationBitmap
				SetScale /I y, 0, ySize -1, "pixel", M_LocalizationBitmap
				Label /W=$S_name left "Distance (pixels)"
				Label /W=$S_name bottom "Distance (pixels)"
			else
				SetScale /I x, 0, (xSize -1) * pixelSize / 1000, "um", M_LocalizationBitmap
				SetScale /I y, 0, (ySize -1) * pixelSize / 1000, "um", M_LocalizationBitmap
				Label /W=$S_name left "Distance (m)"
				Label /W=$S_name bottom "Distance (m)"
			endif
			
			DoWindow /K PALMBitmapWindow
			
			SetDataFolder savDF
					
			break
	endswitch

	return 0
End

Function CBPALMBitmapImage(cba) : CheckBoxControl
	STRUCT WMCheckboxAction &cba

	switch( cba.eventCode )
		case 2: // mouse up
			String controlName = cba.ctrlName
			Variable checked = cba.checked
			NVAR selectedMethod = root:Packages:PALM:V_PALMBitmapSelectedMethod
			
			StrSwitch(controlName)
				// checkboxes that affect how different positions are weighed
				case "CBPALMBitmapSameEmitterWeight":
					CheckBox CBPALMBitmapPreserveIntegral value = 0
					break
				case "CBPALMBitmapPreserveIntegral":
					CheckBox CBPALMBitmapSameEmitterWeight value = 0
					break
				
				
				// checkboxes that affect the calculation of the spot standard deviation
				case "CBPALMBitmapConstantWidth":
					CheckBox CBPALMBitmapFitUncertainty value=0
					CheckBox CBPALMBitmapGME value=0
					selectedMethod = 0
					break
				case "CBPALMBitmapFitUncertainty":
					CheckBox CBPALMBitmapConstantWidth value=0
					CheckBox CBPALMBitmapGME value=0
					selectedMethod = 1
					break
				case "CBPALMBitmapGME":
					CheckBox CBPALMBitmapConstantWidth value=0
					CheckBox CBPALMBitmapFitUncertainty value=0
					selectedMethod = 2
					break
			EndSwitch
			
			break
	endswitch

	return 0
End

Function MakeAccumulatedImage_Menu()
	Variable /G root:Packages:PALM:V_AccumulatedBinSize
	
	NVAR binSize = root:Packages:PALM:V_AccumulatedBinSize
	if (binSize == 0)
		binSize = 0.2
	endif
	
	string posName
	string windowName
	variable colorScaleIndex, localBinSize
	string colorScaleString = "BlueHot;Grays;Rainbow;YellowHot;BlueRedGreen;PlanetEarth;RedWhiteBlue;Yellow"
	
	DFREF savDF = GetDataFolderDFR()
	
	localBinSize = binSize
	Prompt localBinSize, "Bin size (pixels):"
	Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
	Prompt colorScaleIndex, "Color scale:", popup, colorScaleString
	DoPrompt "Select the pixel size and positions wave", posName, localBinSize, colorScaleIndex
	if (V_flag == 1)	// cancel
		return 0
	endif
	binSize = localBinSize
	
	colorScaleString = StringFromList(colorScaleIndex - 1, colorScaleString)
	
	wave pos = GetPositionsWaveReference(posName)
	
	if (WaveExists(pos) == 0)
		Abort "The selected wave doesn't seem to exist!"
	endif
	
	NewDataFolder /O/S root:'PALM Images'
	NewDataFolder /S/O root:'PALM Images':$posName
	
	variable xSize, ySize, pixelSize
	GetCCDDimensionsFromPositions(pos, xSize, ySize, pixelSize)
	
	MakeAccumulatedImage(pos, xSize, ySize, binSize, pixelSize)
	
	wave M_AccumulatedImage
	windowName = "Accumulated image from " + posName
	
	// Typically the histogram for the resulting image is very inhomogeneous
	// a reasonable guess for an initial colorscale appears to be to set min to auto
	// and then take 10% of the maximum for the upper limit
	// this is similar to the bitmap image
	WaveStats /Q /M=1 M_AccumulatedImage
	
	Display /K=1 /N=AccumulatedImageViewer as windowName
	AppendImage /W=$S_name M_AccumulatedImage
	ModifyImage /W=$S_name M_AccumulatedImage ctab= {*,V_max / 10,$colorScaleString,0}
	ModifyGraph /W=$S_name width={Aspect, xSize / ySize}, height=0,tickUnit=1
	
	if (pixelSize == 0)
		Label /W=$S_name left "Distance (pixels)"
		Label /W=$S_name bottom "Distance (pixels)"
	else
		Label /W=$S_name left "Distance (m)"
		Label /W=$S_name bottom "Distance (m)"
	endif
	
	SetDataFolder savDF
	
End

Function MakeAccumulatedImage(pos, xSize, ySize, binSize, pixelSize)
	wave pos
	variable xSize, ySize, binSize, pixelSize
	
	Assert(WaveExists(pos))
	Assert((xSize > 0) && (ySize > 0) && (binSize > 0))
	
	variable nPoints = DimSize(pos, 0)
	variable xWidth = ceil(xSize / binSize)
	variable yWidth = ceil(ySize / binSize)
	
	variable px, py, i, xCol, yCol, zCol
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(pos, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to MakeAccumulatedImage() do not appear to contain any (x,y) information"
	endif
	
	Make /D/O/N=(xWidth, yWidth) M_AccumulatedImage
	if (pixelSize == 0)	// no pixel size specified
		SetScale /I x, 0, xSize - 1, "pixel", M_AccumulatedImage
		SetScale /I y, 0, ySize - 1, "pixel", M_AccumulatedImage
	else		// pixel size specified, rescale the nm pixel size into um
		SetScale /I x, 0, (xSize - 1) * pixelSize / 1000, "um", M_AccumulatedImage
		SetScale /I y, 0, (ySize - 1) * pixelSize / 1000, "um", M_AccumulatedImage
	endif
	
	M_AccumulatedImage = 0
	
	for (i = 0; i < nPoints; i+=1)
		px = floor(pos[i][xCol] / binSize)
		py = floor(pos[i][yCol] / binSize)
		
		M_AccumulatedImage[px][py] += 1
	endfor
	
End

Function MakeWideFieldImage_menu()
	variable localPSFWidth = 2
	
	string savDF = GetDataFolder(1)
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	Variable /G root:Packages:PALM:V_gaussianWidth
	
	NVAR PSFWidth = root:Packages:PALM:V_gaussianWidth
	if (PSFWidth == 0)
		PSFWidth = 2
	endif
	localPSFWidth = PSFWidth
	
	
	string posName
	string windowName
	variable colorScaleIndex
	string colorScaleString = "BlueHot;Grays;Rainbow;YellowHot;BlueRedGreen;PlanetEarth;Terrain;RedWhiteBlue;"
	
	Prompt localPSFWidth, "Standard deviation of the PSF (in CCD pixels):"
	Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
	Prompt colorScaleIndex, "Color scale:", popup, colorScaleString
	DoPrompt "Enter calculation parameters", posName, localPSFWidth, colorScaleIndex
	if (V_flag == 1)	// cancel
		return 0
	endif
	
	colorScaleString = StringFromList(colorScaleIndex - 1, colorScaleString)
	
	wave pos = GetPositionsWaveReference(posName)
	
	if (WaveExists(pos) == 0)
		Abort "The selected wave doesn't seem to exist!"
	endif
	
	NewDataFolder /O/S root:'PALM Images'
	NewDataFolder /S/O root:'PALM Images':$posName
	
	variable xSize, ySize, pixelSize
	GetCCDDimensionsFromPositions(pos, xSize, ySize, pixelSize)
	
	MakeWidefieldImage(pos, xSize, ySize, localPSFWidth, pixelSize)
	
	wave M_Widefield
	windowName = "Widefield image from " + posName
	
	Display /K=1 /N=WidefieldImageViewer as windowName
	AppendImage /W=$S_name M_Widefield
	ModifyImage /W=$S_name M_Widefield ctab= {*,*,$colorScaleString,0}
	ModifyGraph /W=$S_name width={Aspect, xSize / ySize}, height=0,tickUnit=1
	
	if (pixelSize == 0)
		Label /W=$S_name left "Distance (pixels)"
		Label /W=$S_name bottom "Distance (pixels)"
	else
		Label /W=$S_name left "Distance (m)"
		Label /W=$S_name bottom "Distance (m)"
	endif
	SetDataFolder savDF
End

Function MakeWidefieldImage(positions, xSize, ySize, PSFSize, pixelSize)
	wave positions
	variable xSize, ySize, PSFSize, pixelSize
	
	Assert(WaveExists(positions))
	Assert((xSize > 0) && (ySize > 0) && (PSFSize > 0))
	
	variable nPos = DimSize(positions, 0)
	variable i, px, py, xCol, yCol, zCol
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(positions, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to MakeWidefieldImage() do not appear to contain any (x,y) information"
	endif
	
	Make /D/O/N=(4 * xSize, 4 * ySize) emitterLocation
	Make /D/O/N=(4 * xSize, 4 * ySize) PSF
	
	FastOP emitterLocation = 0
	SetScale /P x, -1 * floor(xSize / 2), 0.25, PSF
	SetScale /P y, -1 * floor(xSize / 2), 0.25, PSF
	
	PSF = exp(-(x^2 + y^2) / (2 * PSFSize^2))
	
	for (i = 0; i < nPos; i+=1)
		px = floor(positions[i][xCol] * 4)
		py = floor(positions[i][yCol] * 4)
		
		emitterLocation[px][py] += 1
	endfor
	
	ConvolveImages emitterLocation, PSF
	wave M_Convolved
	
	ImageInterpolate /PXSZ={4, 4} Pixelate M_Convolved
	
	wave M_PixelatedImage
	Duplicate /O M_PixelatedImage, M_Widefield
	
	if (pixelSize == 0)
		SetScale /I x, 0, xSize - 1, "pixel", M_Widefield
		SetScale /I y, 0, ySize - 1, "pixel", M_Widefield
	else
		SetScale /I x, 0, (xSize - 1) * pixelSize / 1000, "um", M_Widefield
		SetScale /I y, 0, (ySize - 1) * pixelSize / 1000, "um", M_Widefield
	endif
	
	KillWaves emitterLocation, PSF, M_Convolved, M_PixelatedImage
End


Function /S GetPossiblePositionsWaves()
	string savDF = GetDataFolder(1)
	string listOfWaves = ""
	
	wave POS_out = root:Packages:PALM:POS_out
	
	if (WaveExists(POS_out) != 0)
		listOfWaves = "POS_out;"
	endif
	
	NewDataFolder /O/S root:'PALM Positions'
	
	listOfWaves += WaveList("*", ";", "DIMS:2")
	
	SetDataFolder savDF
	
	if (strlen(ListOfWaves) == 0)
		listOfWaves = "* No Positions *;"
	endif
	
	// sort the list alphabetically
	listOfWaves = SortList(listOfWaves, ";", 16)
	
	return listOfWaves
End


Function /S GetOutputWaveName(suggestedName, outputDataFolder, promptString)
	string suggestedName
	DFREF outputDataFolder
	string promptString
	// prompt the user for a wave name in dataFolder
	// check that the name is a valid wave name and return it
	// if the user canceled then return an empty string
	// the promptstring is used in the dialogs to let the user know what happens
	
	DFREF savDF = GetDataFolderDFR()
	
	string outputWaveName = suggestedName
	variable outputWaveFound = 0
	
	SetDataFolder outputDataFolder
	
	do
		Prompt outputWaveName, "Wave name:"
		DoPrompt promptString, outputWaveName
		if (V_flag == 1)	// the user canceled
			SetDataFolder savDF
			return ""
		endif
		
		// check if the name is not a blank string
		if (strlen(outputWaveName) == 0)
			continue
		endif
		
		// check if the name is acceptable to Igor
		if (stringmatch(outputWaveName, CleanupName(outputWaveName, 1)) != 1)
			outputWaveName = CleanupName(outputWaveName, 1)
			continue
		endif
		
		// check if the wave already exists
		wave outputWave = $outputWaveName
		if (WaveExists(outputWave) == 1)
			// the requested wave already exists, ask the user if he wants to overwrite it or not
			DoAlert 2, "A wave called \"" + outputWaveName + "\" already exists! Do you want to replace it with the new wave?"
			switch (V_flag)
				case 1:	// OK to replace
						// we're all set
					outputWaveFound = 1
					break
				case 2:	// Don't replace, use a different name instead
					continue
					break
				case 3:	// user cancelled, escape
					SetDataFolder savDF
					return ""
					break
				default:	// shouldn't happen
					SetDataFolder savDF
					Abort "Unknown value for V_flag in DoAlert"
					break
			endswitch
		endif
		
		// if we arrive here then the waveName looks okay
		outputWaveFound = 1
	while (outputWaveFound == 0)
	
	SetDataFolder savDF
	
	return outputWaveName
End


Function BTCopyPositionsProc(ba) : ButtonControl
	STRUCT WMButtonAction &ba

	switch( ba.eventCode )
		case 2: // mouse up
			NewDataFolder /O root:'PALM Positions'
			DFREF outputFolder = root:'PALM Positions'
			string posName
			
			ControlInfo /W=CCD_Control PMSelectPositionsWave
			posName = S_Value
			
			if (StringMatch(posName, "*No Positions*") == 1)
				SetDataFolder savDF
				return 0
			endif
			
			wave posWave = GetPositionsWaveReference(posName)
			
			if (WaveExists(posWave) == 0)
				SetDataFolder savDF
				Abort "The specified positions wave doesn't exist. Did you delete it?"
			endif
			
			posName = GetOutputWaveName("", outputFolder, "Enter the new name of the positions wave")
			if (strlen(posName) == 0)	// the user canceled the dialog
				return -1
			endif
			
			Duplicate /O posWave, root:'PALM Positions':$posName
			break
	endswitch

	return 0
End


Function MergePositions()
	
	string savDF = GetDataFolder(1)
	NewDataFolder /O root:Packages
	NewDataFolder /O/S root:Packages:PALM
	
	string listPositionsFunc = GetIndependentModuleName() + "#GetPossiblePositionsWaves()"
	
	
	Make /T/O/N=(0) candidatesListBoxWave
	Make /O/N=(0) candidatesListBoxSelWave
	
	DoWindow /F positionsWavesViewer
	if (V_flag == 0)
		NewPanel /K=1/W=(101,286,335,497) /N=positionsWavesViewer as "Select positions waves (double click to remove)"
		ListBox ListBoxCandidateWaves,pos={1,2},size={231,183},proc=LBDeleteDoubleClickedRowProc
		ListBox ListBoxCandidateWaves,listWave=candidatesListBoxWave,mode=4
		ListBox ListBoxCandidateWaves,selRow= 0,selWave=candidatesListBoxSelWave
		ListBox ListBoxCandidateWaves,userdata(ResizeControlsInfo)= A"!!,<7!!#7a!!#B!!!#AFz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
		ListBox ListBoxCandidateWaves,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
		ListBox ListBoxCandidateWaves,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		PopupMenu PMAddPositionsWaves,pos={2,190},size={109,20},title="Add positions",proc=PMAddPositionsWavesProc
		PopupMenu PMAddPositionsWaves,mode=0,value= #listPositionsFunc
		PopupMenu PMAddPositionsWaves,userdata(ResizeControlsInfo)= A"!!,=b!!#AM!!#@<!!#<Xz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
		PopupMenu PMAddPositionsWaves,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		PopupMenu PMAddPositionsWaves,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		Button BTCreateImageFromMultiplePos,pos={164,189},size={50,20},proc=BTMergePositionsProc,title="OK"
		Button BTCreateImageFromMultiplePos,userdata(ResizeControlsInfo)= A"!!,G4!!#AL!!#>V!!#<Xz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
		Button BTCreateImageFromMultiplePos,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		Button BTCreateImageFromMultiplePos,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		SetWindow kwTopWin,hook(closeControl)=SetupImageViewer_IgorWaves_hook
		SetWindow kwTopWin,hook(ResizeControls)=ResizeControls#ResizeControlsHook
		SetWindow kwTopWin,userdata(ResizeControlsInfo)= A"!!*'\"z!!#B$!!#Abzzzzzzzzzzzzzzzzzzzzz"
		SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzzzzzzzz"
		SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzz!!!"
		
		// if the window was opened previously then restore its size and position
		WC_WindowCoordinatesRestore("positionsWavesViewer")
		SetWindow positionsWavesViewer hook(WindowCoordsHook)=WC_WindowCoordinatesNamedHook
	endif
	
	SetDataFolder savDF
End


Function LBDeleteDoubleClickedRowProc(lba) : ListBoxControl
	STRUCT WMListboxAction &lba

	Variable row = lba.row
	Variable col = lba.col
	WAVE/T/Z listWave = lba.listWave
	WAVE/Z selWave = lba.selWave

	switch( lba.eventCode )
		case -1: // control being killed
			break
		case 3: // double click
			
			DeletePoints /M=0 row, 1, listWave
			if (WaveExists(selWave))
				DeletePoints /M=0 row, 1, selWave
			endif
			break
		case 4: // cell selection
		case 5: // cell selection plus shift key
			break
		case 6: // begin edit
			break
		case 7: // finish edit
			break
	endswitch

	return 0
End

Function PMAddPositionsWavesProc(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	switch( pa.eventCode )
		case 2: // mouse up
			string savDF = GetDataFolder(1)
			SetDataFolder root:Packages:PALM
			Variable popNum = pa.popNum
			String popStr = pa.popStr
			
			wave/T candidatesListBoxWave
			wave candidatesListBoxSelWave
			
			if (StringMatch(popStr, "*No Positions*") == 1)
				SetDataFolder savDF
				return 0
			endif
			
			wave newPos = GetPositionsWaveReference(popStr)
			
			if (WaveExists(newPos) == 0)
				SetDataFolder savDF
				Abort "Internal error: the positions wave does not exist"
			endif
			
			Redimension /N=(DimSize(candidatesListBoxWave, 0) + 1) candidatesListBoxWave
			Redimension /N=(DimSize(candidatesListBoxSelWave, 0) + 1) candidatesListBoxSelWave
			
			candidatesListBoxWave[DimSize(candidatesListBoxWave, 0) - 1] = popStr
			candidatesListBoxSelWave[DimSize(candidatesListBoxSelWave, 0) - 1] = 0
			
			ControlUpdate /W=positionsWavesViewer ListBoxCandidateWaves
			
			SetDataFolder savDF
			break
	endswitch

	return 0
End


Function BTMergePositionsProc(ba) : ButtonControl
	STRUCT WMButtonAction &ba

	switch( ba.eventCode )
		case 2: // mouse up
			// click code here
			
			DFREF positionsFolder = root:'PALM Positions'
			wave/T waveNames = root:Packages:PALM:candidatesListBoxWave
			string outputName = ""
			variable nWaves = DimSize(waveNames, 0)
			variable i, j
			variable offset = 0
			variable nPoints, totalNumberOfPoints = 0
			variable lastFrameIndex = 0
			string waveNote
			string originalFilePaths
			string originalCalculationDates
			variable totalCalculationTime
			
			if (nWaves < 2)
				DoWindow /K positionsWavesViewer
				return 0
			endif
			
			outputName = GetOutputWaveName("", positionsFolder, "Enter a name for the composite positions wave")
			if (strlen(outputName) == 0)	// the user canceled the dialog
				return -1
			endif
			
			for (i = 0; i < nWaves; i+=1)
				if (StringMatch(waveNames[i], "POS_out") == 1)
					waveNames[i] = "root:Packages:PALM:POS_out"
				else
					waveNames[i] = "root:'PALM Positions':" + PossiblyQuoteName(waveNames[i])
				endif
				wave pos = $waveNames[i]
				totalNumberOfPoints += DimSize(pos, 0)
			endfor
			
			// did the positions originate from waves with the same x and y sizes?
			variable firstXSize, firstYSize, xSize, ySize, pixelSize
			GetCCDDimensionsFromPositions(pos, firstXSize, firstYSize, pixelSize)
			
			for (i = 1; i < nWaves; i+=1)
				GetCCDDimensionsFromPositions($waveNames[i], xSize, ySize, pixelSize)
				if ((xSize != firstXSize) || (ySize != firstYSize))
					// the positions come from waves with different dimensions, throw an error
					DoAlert 1, "The selected positions came from CCD frames with different dimensions. Do you want to continue anyway?"
					if (V_flag == 1)
						// user clicked 'yes'
						break
					else	// user clicked 'no'
						DoWindow /K positionsWavesViewer
						return 0
					endif
				endif
			endfor 
			
			try
				Make /D/O/N=(totalNumberOfPoints, 12) positionsFolder:$outputName
			catch
				DoWindow /K positionsWavesViewer
				Abort "Failed to make the output wave. Is the output name OK for Igor?"
			endtry
			
			wave concatenatedPositions = positionsFolder:$outputName
			
			for (i = 0; i < nWaves; i+=1)
				if (i > 0)
					lastFrameIndex += pos[DimSize(pos, 0) - 1][0]
				endif
				wave pos = $waveNames[i]
				nPoints = DimSize(pos, 0)
				for (j = 0; j <  nPoints; j+=1)
					concatenatedPositions[offset][] = pos[j][q]
					concatenatedPositions[offset][0] += lastFrameIndex
					offset += 1
				endfor
			endfor
			
			// set up an appropriate wave note that contains the calculation information
			waveNote = note($waveNames[0])
			originalFilePaths = StringByKey("ORIGINAL FILE PATH", waveNote)
			originalCalculationDates = StringByKey("CALCULATION DATE", waveNote)
			totalCalculationTime = NumberByKey("CALCULATION DURATION", waveNote)
			for (i = 1; i < nWaves; i+=1)
				originalFilePaths += " and "
				originalFilePaths += StringByKey("ORIGINAL FILE PATH", note($waveNames[i]))
				
				originalCalculationDates += " and "
				originalCalculationDates += StringByKey("CALCULATION DATE", note($waveNames[i]))
				
				totalCalculationTime += NumberByKey("CALCULATION DURATION", note($waveNames[i]))
			endfor
			
			waveNote = ReplaceStringByKey("ORIGINAL FILE PATH", waveNote, originalFilePaths)
			waveNote = ReplaceStringByKey("CALCULATION DATE", waveNote, originalCalculationDates)
			waveNote = ReplaceNumberByKey("CALCULATION DURATION", waveNote, totalCalculationTime)
			
			Note /K concatenatedPositions, waveNote
			
			DoWindow /K positionsWavesViewer
			break
	endswitch

	return 0
End


Function ExtractSpatialSubsetOfPositions(calledFromMarquee)
	variable calledFromMarquee
	
	// extract all the positions that lie within given x and y coordinates in a fitted positions wave
	// if calledFromMarquee == 1 then extract the coordinates of the box from the marquee on the graph
	// if calledFromMarquee == 0  then prompt for the requested coordinates
	
	variable xStart, xEnd, yStart, yEnd, err
	variable xPixelSize, yPixelSize
	string posName, outputName, units
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:'PALM Positions'
	string savDF = GetDataFolder(1)
	
	if (calledFromMarquee == 0)	// prompt for the required values
		Prompt xStart, "Lower limit for x:"
		Prompt xEnd, "Upper limit for x:"
		Prompt yStart, "Lower limit for y:"
		Prompt yEnd, "Upper limit for y:"
		Prompt units, "Provided numbers are in:", popup, "pixels;um;"
		Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
		
		DoPrompt "Enter calculation parameters", xStart, xEnd, yStart, yEnd, units, posName
		if (V_flag == 1)
			return 0
		endif
		
		wave pos = GetPositionsWaveReference(posName)	
		if (WaveExists(pos) == 0)
			Abort "The selected positions wave doesn't seem to exist!"
		endif
		
		// if the user wants to use units of micrometers but the positions doesn't specify
		// a pixel size then abort
		strswitch (units)
			case "pixels":
				// we don't need to do anything with the values
				// in xStart etc.
				break
			case "um":
				xPixelSize = NumberByKey("X PIXEL SIZE", note(pos))
				yPixelSize = NumberByKey("Y PIXEL SIZE", note(pos))
				// did the wavenote contain the pixel sizes?
				if ((NumType(xPixelSize) == 2) || (NumType(yPixelSize) == 2))	// NaN
					Abort "The positions do not contain any information on the pixel size (wavenote: \"X/Y PIXEL SIZE\")"
				endif
				
				// convert the provided limits to pixels
				xStart /= (xPixelSize / 1e3)	// convert from nanometer to m
				yStart /= (xPixelSize / 1e3)
				xEnd /= (xPixelSize / 1e3)
				yEnd /= (xPixelSize / 1e3)
				break
			default:
				Abort "Internal error: unknown units string in ExtractSpatialSubsetOfPositions()"
		endswitch
	else		// get the coordinate values from the marquee
		// attempt to get the positions from the graph automatically
		wave pos = GetPositionsFromGraph()
		if (WaveExists(pos) != 1)
			Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
			
			DoPrompt "Choose a positions wave", posName
			if (V_flag == 1)
				return 0
			endif
			
			wave pos = GetPositionsWaveReference(posName)	
			if (WaveExists(pos) == 0)
				Abort "The selected positions wave doesn't seem to exist!"
			endif
		endif
		
		// GetMarqueeCoordinates will return the coordinates in units of pixels
		err = GetMarqueeCoordinates(pos, xStart, yStart, xEnd, yEnd)
		if (err != 0)	// no Marquee
			return 0
		endif
	endif
	
	 if ((xEnd <= xStart) || (yEnd <= yStart) || (xStart < 0) || (yStart < 0))
	 	Abort "Please enter valid numbers for the limits"
	 endif
	 
	 // prompt for an output wavename
	 outputName = GetOutputWaveName("", root:'PALM Positions', "Enter a name for the output wave")
	 if (strlen(outputName) == 0)
	 	return 0
	 endif
	
	variable nPos = DimSize(pos, 0), xCol, yCol, zCol
	variable nPositionsWithinLimits = 0, i, offset
	string waveNote
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(pos, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to ExtractSubsetOfPositions_menu() do not appear to contain any (x,y) information"
	endif
	
	for (i = 0; i < nPos; i+= 1)
		if ((pos[i][xCol] >= xStart) && (pos[i][yCol] >= yStart) && (pos[i][xCol] <= xEnd) && (pos[i][yCol] <= yEnd))
			nPositionsWithinLimits += 1
		endif
	endfor
	
	Make /D/O/N=(nPositionsWithinLimits, DimSize(pos, 1)) root:'PALM Positions':$outputName
	wave outputPos = root:'PALM Positions':$outputName
	
	waveNote = note(pos)
	waveNote = ReplaceNumberByKey("X SIZE", waveNote, xEnd - xStart + 1)
	waveNote = ReplaceNumberByKey("Y SIZE", waveNote, yEnd - yStart + 1)
	waveNote += "X START:" + num2str(xStart) + ";"
	waveNote += "Y START:" + num2str(yStart) + ";"
	Note outputPos, waveNote
	
	offset = 0
	for (i = 0; i < nPos; i+= 1)
		if ((pos[i][xCol] >= xStart) && (pos[i][yCol] >= yStart) && (pos[i][xCol] <= xEnd) && (pos[i][yCol] <= yEnd))
			outputPos[offset][] = pos[i][q]
			// subtract the xStart and yStart values from every position so that the position numbering is once again from pixel 0
			outputPos[offset][xCol] -= xStart
			outputPos[offset][yCol] -= yStart
			offset += 1
		endif
	endfor
	
	SetDataFolder savDF
End

Function /WAVE GetPositionsFromGraph()
	// try to get the positions wave
	// that was used to make the plot in the top graph
	
	// check if the top graph is a suitable window created by the PALM package
	string windowName
	windowName = WinName(0,1)
	if ((StringMatch(windowName, "CCDViewer*")) || (StringMatch(windowName, "Scatter_Viewer*")) || (StringMatch(windowName, "AccumulatedImageViewer*")) || (StringMatch(windowName, "PALMImageViewer*")) || (StringMatch(windowName, "WidefieldImageViewer*")) || (StringMatch(windowName, "AverageImage*")) || (StringMatch(windowName, "M_StandardDeviation*")))
		// suitable window, continue
	else
		return root:QQQDoesNotExist
	endif
	
	// get a list of traces in the graph
	string traceNames, imageNames
	traceNames = TraceNameList("", ";", 1)
	imageNames = ImageNameList("", ";")
	
	// how many traces and images do we have?
	variable nTraces = ItemsInList(traceNames), i
	variable nImages = ItemsInList(imageNames)
	
	// if there are no traces or images then return a null wave reference
	if ((nTraces == 0) && (nImages == 0))
		return root:QQQDoesNotExist
	endif
	
	// get the the full wave paths (names including the data folder)
	string fullWavePaths = ""
	
	for (i = 0; i < nTraces; i+=1)
		fullWavePaths += GetWavesDataFolder(TraceNameToWaveRef("", StringFromList(i, traceNames)), 2) + ";"
	endfor
	
	for (i = 0; i < nImages; i+=1)
		fullWavePaths += GetWavesDataFolder(ImageNameToWaveRef("", StringFromList(i, imageNames)), 2) + ";"
	endfor
	
	// find the wave paths that can provide us with a position
	// we assume that the generated image or traces are in 
	// root:'PALM Images'
	string possiblePositionsWaves
	variable nPossiblePositionsWaves
	possiblePositionsWaves = ListMatch(fullWavePaths, "root:'PALM Images':*")
	nPossiblePositionsWaves = ItemsInList(possiblePositionsWaves)
	
	// if we found no possibilities or more than one possibility then we cannot decide
	if (nPossiblePositionsWaves != 1)
		return root:QQQDoesNotExist
	endif
	
	// we found exactly one possibility. The name of the positions wave
	// is now contained in possiblePositionsWaves as :root:'PALM Images':<name>
	string positionsName = StringFromList(2, possiblePositionsWaves, ":")
	
	// the wave name may be quoted, in which case we have to remove them
	positionsName = ReplaceString("'", positionsName, "")
	
	wave positionsWave = GetPositionsWaveReference(positionsName)
	return positionsWave
End


Function GetMarqueeCoordinates(positions, xStart, yStart, xEnd, yEnd)
	wave positions
	variable &xStart, &yStart, &xEnd, &yEnd
	
	// if the user has drawn a marquee on the top graph, get the coordinates of that marquee
	// take into account the the graph might be in units of pixels or in um
	// also order the coordinates
	
	// return the coordinates in units of pixels by reference
	
	GetMarquee /Z left, bottom
		
	if (V_flag == 0)
		return -1
	endif
	
	// check if the window is a suitable window created by the PALM package
	if ((StringMatch(S_MarqueeWin, "CCDViewer*")) || (StringMatch(S_MarqueeWin, "Scatter_Viewer*")) || (StringMatch(S_MarqueeWin, "AccumulatedImageViewer*")) || (StringMatch(S_MarqueeWin, "PALMImageViewer*")) || (StringMatch(S_MarqueeWin, "WidefieldImageViewer*")) || (StringMatch(S_MarqueeWin, "AverageImage*")) || (StringMatch(S_MarqueeWin, "M_StandardDeviation*")))
		// suitable window, continue
	else
		return 0
	endif
	
	// determine if the plot is in units of pixels or um
	string traceNames, imageNames, unitsString
	traceNames = TraceNameList("", ";", 1)
	imageNames = ImageNameList("", ";")
	if (ItemsInList(traceNames) > 0)	// the plot contains some traces, use these to get the scale
		wave traceWave = TraceNameToWaveRef("", StringFromList(0, traceNames))
		unitsString = WaveUnits(traceWave, 0)
	elseif (ItemsInList(imageNames) > 0) // the plot does not contain any traces, try using an image instead
		wave imageWave = ImageNameToWaveRef("", StringFromList(0, imageNames))
		unitsString = WaveUnits(imageWave, 0)
	else	// no traces or images on the plot
		Abort "The graph does not seem to contain any traces or images"
	endif
	
	// adjust the marquee coordinates depending on whether the graph
	// is in units of pixels or um
	variable xPixelSize, yPixelSize
	strswitch (unitsString)
		case "pixel":
		case "":	// treat no units as pixels
			xPixelSize = 1
			yPixelSize = 1
			break
		case "um":
			// get the wave units
			xPixelSize = NumberByKey("X PIXEL SIZE", note(positions)) / 1000	// since it's units of um
			yPixelSize = NumberByKey("Y PIXEL SIZE", note(positions)) / 1000
			if ((NumType(xPixelSize) == 2) || (NumType(yPixelSize) == 2))	// NaN
				Abort "Unable to recognize the units of the plot. Try using \"Extract positions within limits\" from the \"Macros\" menu instead"
			endif
			break
		default:
			Abort "Unable to recognize the units of the plot. Try using \"Extract positions within limits\" from the \"Macros\" menu instead"
	endswitch
	
	xStart = min(V_left / yPixelSize, V_right / yPixelSize)
	xEnd = max(V_left / yPixelSize, V_right / yPixelSize)
	yStart = min(V_bottom / yPixelSize, V_top / yPixelSize)
	yEnd = max(V_bottom / yPixelSize, V_top / yPixelSize)
	
	return 0
End

Function returnPointNearestFitPositions(positionsWave, xLoc, yLoc, nearestX, nearestY, index)
	wave positionsWave
	variable xLoc, yLoc
	variable& nearestX
	variable& nearestY
	variable& index
	
	Assert(WaveExists(positionsWave))
	Assert((xLoc >= 0) && (yLoc >= 0))
	
	// given the coordinates of a point x, y, the function returns (by reference) the coordinates of the closest matching point in positionsWave and its index in positionsWave
	variable sq_distance, i
	variable closest_sq_distance = 1e100, closest_index = -1
	variable nPos = DimSize(positionsWave, 0), xCol, yCol, zCol
	
	if (nPos == 0)
		nearestX = -1
		nearestY = -1
		return 1
	endif
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(positionsWave, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to returnPointNearestFitPositions() do not appear to contain any (x,y) information"
	endif
	
	for (i = 0; i < nPos; i+=1)
		sq_distance = (positionsWave[i][xCol] - xLoc)^2 + (positionsWave[i][yCol] - yLoc)^2
		if (sq_distance < closest_sq_distance)
			closest_sq_distance = sq_distance
			closest_index = i
		endif
	endfor
	
	nearestX = positionsWave[closest_index][xCol]
	nearestY = positionsWave[closest_index][yCol]
	index = closest_index
	
	return 0
End

Function CorrectDrift_menu()
	
	Variable /G root:Packages:PALM:V_DriftCorrectionMaxJump
	NVAR currentImage = root:Packages:PALM:V_currentImage
	NVAR maxJump = root:Packages:PALM:V_DriftCorrectionMaxJump
	string positionsWaves
	
	if (maxJump == 0)	// provide a default value
		maxJump = 1
	endif
	
	DoWindow /F CCDViewer
	if (V_flag == 0)	// there is no CCDViewer window, no meaning to continue
		Abort "Please open a CCD file first"
	elseif (V_flag == 2)	// the window is hidden, show it
		DoWindow /HIDE=0 CCDViewer
	endif
	
	string positionsListFunc = GetIndependentModuleName() + "#GetPossiblePositionsWaves()"
	positionsWaves = GetPossiblePositionsWaves()
	if (ItemsInList(positionsWaves) == 0)
		Abort "Please make sure that there are some fitted positions in the experiment"
	endif
	
	if (currentImage != 0)
		Printf "Be sure that the currently displayed frame (%d) is the first frame containing useful PALM data\r", currentImage + 1
	endif
	
	// set up the waves that will contain the clicked positions and the listbox entries (both updated by the window hook function)
	Make /D/O/N=(0, 2) root:Packages:PALM:M_ClickedCoordinates
	Make /T/O/N=(0, 2) root:Packages:PALM:M_ClickedTextCoordinates
	Make /T/O/N=(2) root:Packages:PALM:M_DriftListBoxTitles = {"X", "Y"}
	
	wave M_ClickedCoordinates = root:Packages:PALM:M_ClickedCoordinates
	wave /T M_ClickedTextCoordinates = root:Packages:PALM:M_ClickedTextCoordinates
	wave /T M_DriftListBoxTitles = root:Packages:PALM:M_DriftListBoxTitles
	
	// draw the main panel
	DoWindow /K PALMDriftCorrection	// in case the user still had a panel floating around
	
	NewPanel /K=1 /N=PALMDriftCorrection /W=(789,73,962,323)
	SetDrawLayer UserBack
	SetDrawEnv fsize= 11
	DrawText 10,34,"Click on the fiducial markers\rin the CCD frame"
	ListBox LBDriftCorrectionMarkerPosition,pos={27,39},size={117,116}, listWave = M_ClickedTextCoordinates, proc=LBDeleteDoubleClickedRowProc, titleWave=M_DriftListBoxTitles
	PopupMenu PMDriftCorrectionFittedPosition,pos={6,164},size={163,20},bodyWidth=90,title="Fitted Positions:"
	PopupMenu PMDriftCorrectionFittedPosition,mode=1,value=#positionsListFunc
	Button BTDriftCorrectionContinue,pos={31,221},size={100,20},proc=BTDriftMarkerContinue,title="Continue"
	SetVariable SVDriftCorrectionMaxDistance,pos={3,193},size={170,15},title="Maximum jump distance (px):"
	SetVariable SVDriftCorrectionMaxDistance,value=maxJump,limits={0,inf,1}
	SetWindow PALMDriftCorrection, hook(KillHook) = PALMDriftCorrection_WindowHook
	
	AutoPositionWindow /R=CCDViewer /M=0 PALMDriftCorrection
	DoWindow /F CCDViewer
	
	// attach the window hook function to the CCD Viewer
	SetWindow CCDViewer, hook(DriftMarkerHook) = CaptureClickedCoordinates_Hook
	
	
End

Function PALMDriftCorrection_WindowHook(s)
	STRUCT WMWinHookStruct &s
	
	switch (s.eventcode)
		case 2:	// window kill, remove the window hook function from CCDViewer
			DoWindow CCDViewer
			if (V_flag != 0)
				SetWindow CCDViewer hook(DriftMarkerHook) = $""
			endif
			break
		default:
			return 0
	endswitch
	
	return 0
End


Function BTDriftMarkerContinue(ba) : ButtonControl
	STRUCT WMButtonAction &ba

	switch( ba.eventCode )
		case 2: // mouse up
			// click code here
			NVAR currentFrame = root:Packages:PALM:V_currentImage
			NVAR maxJump = root:Packages:PALM:V_DriftCorrectionMaxJump
			wave /T M_ClickedTextCoordinates = root:Packages:PALM:M_ClickedTextCoordinates
			variable nMarkers = DimSize(M_ClickedTextCoordinates, 0), nPositions
			string positionsWaveName
			variable i, nearestX, nearestY, index
			string outputName, shortOutputName
			
			ControlInfo /W=PALMDriftCorrection PMDriftCorrectionFittedPosition
			positionsWaveName = S_Value
			
			if (StringMatch(positionsWaveName, "* No Positions *") == 1)
				DoWindow /K PALMDriftCorrection
				KillWaves /Z M_ClickedCoordinatesText
				KillWaves /Z root:Packages:PALM:M_ClickedCoordinates
				return 0
			endif
			
			wave positions = GetPositionsWaveReference(positionsWaveName)
			nPositions = DimSize(positions, 0)
			
			if (nMarkers == 0)	// no markers indicated
				DoWindow /K PALMDriftCorrection
				KillWaves /Z M_ClickedCoordinatesText
				KillWaves /Z root:Packages:PALM:M_ClickedCoordinates
				return 0
			endif
			
			// check if the positions have been consolidated, if so then refuse
			variable nFramesPresentCol
			GetColumnForNFramesPresent(positions, nFramesPresentCol)
			ImageTransform /G=(nFramesPresentCol) sumCol, positions
			if (V_value != DimSize(positions, 0))
				Abort "The requested positions have been consolidated, there is no point in performing the drift correction"
			endif
			
			// extract the positions that are fitted in the current frame
			wave /WAVE M_ExtractedPositions = ExtractPositionsInFrame(positions, currentFrame)
			wave extractedPositions = M_ExtractedPositions[0]
			wave extractedPositionsIndices = M_ExtractedPositions[1]
			
			if (DimSize(extractedPositions, 0) == 0)	// no positions were fitted in this frame
				DoWindow /K PALMDriftCorrection
				KillWaves /Z M_ClickedCoordinatesText, extractedPositions
				KillWaves /Z root:Packages:PALM:M_ClickedCoordinates
				Abort "No positions were fitted in the current frame, not even the markers!"
			endif
			
			Make /D/O/N=(nMarkers, 2) root:Packages:PALM:RecoveredMarkerPositions	// the coordinates that we have recovered for the markers
			wave RecoveredMarkerPositions = root:Packages:PALM:RecoveredMarkerPositions
			RecoveredMarkerPositions = str2num(M_ClickedTextCoordinates[p][q])
			
			for (i = 0; i < nMarkers; i+=1)
				returnPointNearestFitPositions(extractedPositions, RecoveredMarkerPositions[i][0], RecoveredMarkerPositions[i][1], nearestX, nearestY, index)
				RecoveredMarkerPositions[i][0] = nearestX
				RecoveredMarkerPositions[i][1] = nearestY
			endfor
			
			Make /D/O/N=(nMarkers) root:Packages:PALM:W_Markers_X
			Make /D/O/N=(nMarkers) root:Packages:PALM:W_Markers_Y
			wave W_Markers_X = root:Packages:PALM:W_Markers_X
			wave W_Markers_Y = root:Packages:PALM:W_Markers_Y
			
			W_Markers_X = RecoveredMarkerPositions[p][0]
			W_Markers_Y = RecoveredMarkerPositions[p][1]
			
			DoWindow /F CCDViewer
			AppendToGraph /W=CCDViewer W_Markers_Y vs W_Markers_X
			ModifyGraph /W=CCDViewer mode(W_Markers_Y)=3
			ModifyGraph /W=CCDViewer rgb(W_Markers_Y)=(0,65535,0)
			
			DoUpdate
			
			DoAlert 1, "The green markers on the image highlight the presumed marker positions. Accept?"
			if (V_flag != 1)	// user didn't choose 'yes'
				DoWindow /K PALMDriftCorrection
				RemoveFromGraph /W=CCDViewer W_Markers_Y
				KillWaves /Z M_ClickedCoordinatesText, extractedPositions, RecoveredMarkerPositions, W_Markers_X, W_Markers_Y
				KillWaves /Z root:Packages:PALM:M_ClickedCoordinates
				return 0
			endif
			
			RemoveFromGraph /W=CCDViewer W_Markers_Y
			KillWaves /Z W_Markers_X, W_Markers_Y
			
			CorrectDrift(positions, RecoveredMarkerPositions, maxJump)
			wave M_CorrectedPositions = root:Packages:PALM:M_CorrectedPositions
			
			DoWindow /K PALMDriftCorrection
			
			// ask for an output name
			DFREF positionsFolder = root:'PALM Positions'
			shortOutputName = GetOutputWaveName("", positionsFolder, "Enter a name for the corrected positions wave")
			if (strlen(outputName) == 0)	// the user canceled the dialog
				DoWindow /K positionsWavesViewer
				return 0
			endif
			
			outputName = "root:'PALM Positions':" + PossiblyQuoteName(shortOutputName)
			
			Duplicate /O M_CorrectedPositions, $outputName
			
			wave M_AbsoluteDrift = root:Packages:PALM:M_AbsoluteDrift
			Make /D/O/N=(DimSize(M_AbsoluteDrift, 0)) W_AbsoluteDrift_X
			Make /D/O/N=(DimSize(M_AbsoluteDrift, 0)) W_AbsoluteDrift_Y
			W_AbsoluteDrift_X = M_AbsoluteDrift[p][0]
			W_AbsoluteDrift_Y = M_AbsoluteDrift[p][1]
			
			DoWindow /F AbsoluteDriftViewer
			if (V_flag == 0)
				Display /K=1 /N=AbsoluteDriftViewer W_AbsoluteDrift_Y as "Estimated drift"
				AppendToGraph W_AbsoluteDrift_X
				ModifyGraph rgb(W_AbsoluteDrift_X)=(0,0,65535)
				Legend/C/N=text0/J "\\s(W_AbsoluteDrift_Y) X\r\\s(W_AbsoluteDrift_X) Y"
				Label left "Drift (pixel)"
				Label bottom "Frame Number"
			endif
	endswitch
	
	
	return 0
End


Function CorrectDrift(positions, markers, maxStepSize)
	wave positions, markers
	variable maxStepSize
	
	Assert(WaveExists(positions))
	Assert(WaveExists(markers))
	Assert(maxStepSize >= 0)
	
	// given a set of positions and a series of marker locations, follow these marker throughout the movie and try to follow their movements and correct for that
	
	string savDF = GetDataFolder(1)
	NewDataFolder /O root:Packages
	NewDataFolder /O/S root:Packages:PALM
	
	variable nPositions = DimSize(positions, 0)
	variable nFrames
	variable i, j, k, offset, xCol, yCol, zCol
	variable previousAbsoluteDriftX = 0, previousAbsoluteDriftY = 0
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(positions, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to CorrectDrift() do not appear to contain any (x,y) information"
	endif
	
	Duplicate /O positions, M_CorrectedPositions
	
	CalculateDrift(positions, markers, maxStepSize)
	wave M_AbsoluteDrift = root:Packages:PALM:M_AbsoluteDrift
	
	nFrames = DimSize(M_AbsoluteDrift, 0)
	offset = 0
	
	for (i = 0; i < nFrames; i+=1)
		for (j = offset; ; j += 1)
			if ((j >= nPositions) || (positions[j][0] != i))
				break
			endif
			
			if ((NumType(M_AbsoluteDrift[i][0]) == 2) || (NumType(M_AbsoluteDrift[i][1]) == 2))
						M_CorrectedPositions[j][xCol] -= previousAbsoluteDriftX
						M_CorrectedPositions[j][yCol] -= previousAbsoluteDriftY
						offset += 1
						continue
			endif
			
			M_CorrectedPositions[j][xCol] -= M_AbsoluteDrift[i][0]
			M_CorrectedPositions[j][yCol] -= M_AbsoluteDrift[i][1]
			previousAbsoluteDriftX = M_AbsoluteDrift[i][0]
			previousAbsoluteDriftY = M_AbsoluteDrift[i][1]
			offset += 1
		endfor
	endfor
	
	SetDataFolder savDF
	
End

Function CaptureClickedCoordinates_Hook(s) 
	STRUCT WMWinHookStruct &s
	
	// this function attaches to a window of choice (typically the CCD viewer) and appends the X- and Y-coordinates where the user clicked the mouse to a specific wave
	// both a numeric and a text version of the wave is made (for use with listbox controls)
	
	switch (s.eventcode)
		case 3:	// mousedown
			variable xVal, yVal
			NVAR xSize = root:Packages:PALM:V_xSize
			NVAR ySize = root:Packages:PALM:V_ySize
			
			wave M_ClickedCoordinates = root:Packages:PALM:M_ClickedCoordinates
			wave /T M_ClickedTextCoordinates = root:Packages:PALM:M_ClickedTextCoordinates
			
			if (WaveExists(M_ClickedCoordinates) == 0)
				Make /D/N=(0, 2) root:Packages:PALM:M_ClickedCoordinates
				wave M_ClickedCoordinates = root:Packages:PALM:M_ClickedCoordinates
			endif
			if (WaveExists(M_ClickedTextCoordinates) == 0)
				Make /T /N=(0, 2) root:Packages:PALM:M_ClickedTextCoordinates
				wave /T M_ClickedTextCoordinates = root:Packages:PALM:M_ClickedTextCoordinates
			endif
			
			xVal = AxisValFromPixel("", "Bottom", s.mouseLoc.h)
			yVal = AxisValFromPixel("", "Left", s.mouseLoc.v)
			
			if ((xVal < 0) || (yVal < 0) || (xVal >= xSize) || (yVal >= ySize))
				return 0
			endif
			
			Redimension /N=(DimSize(M_ClickedCoordinates, 0) + 1, 2) M_ClickedCoordinates
			M_ClickedCoordinates[DimSize(M_ClickedCoordinates, 0) - 1][0] = xVal
			M_ClickedCoordinates[DimSize(M_ClickedCoordinates, 0) - 1][1] = yVal
			
			Redimension /N=(DimSize(M_ClickedTextCoordinates, 0) + 1, 2) M_ClickedTextCoordinates
			M_ClickedTextCoordinates[DimSize(M_ClickedTextCoordinates, 0) - 1][0] = num2str(xVal)
			M_ClickedTextCoordinates[DimSize(M_ClickedTextCoordinates, 0) - 1][1] = num2str(yVal)
			
			return 1
			break
		default:
			return 0
			break
	endswitch
End


Function CalculateDrift(positions, markers, maxStepSize)
	wave positions, markers
	variable maxStepSize
	
	Assert(WaveExists(positions))
	Assert(WaveExists(markers))
	Assert(maxStepSize >= 0)
	
	variable nMarkers = DimSize(markers, 0)
	variable nPositions = DimSize(positions, 0)
	variable nFrames = positions[nPositions - 1][0] - positions[0][0] + 1
	variable i
	variable currentFrame
	variable driftX, driftY, nearestX, nearestY, index
	variable nMarkersRecovered
	
	Make /D/O/N=(nFrames, 2) M_RelativeDrift = 0
	Make /D/O/N=(nFrames, 2) M_AbsoluteDrift = 0
	Make /D/O/N=(nMarkers, 2) M_previousPositions	// the positions of the markers in the previous frame, not taking drift into account
	M_previousPositions = markers[p][q]	// initialize the marker positions to the given values
	
	for (currentFrame = 0; currentFrame < nFrames; currentFrame += 1)
		// extract the positions found in the current frame
		wave /WAVE M_Extracted = ExtractPositionsInFrame(positions, currentFrame)
		wave extractedPositions = M_Extracted[0]
		
		if (DimSize(extractedPositions, 0) == 0)	// no positions in the current frame
			M_RelativeDrift[currentFrame][] = NaN
			M_AbsoluteDrift[currentFrame][] = NaN
			continue
		endif
		
		nMarkersRecovered = 0
		driftX = 0
		driftY = 0
		for (i = 0; i < nMarkers; i += 1)
				returnPointNearestFitPositions(extractedPositions, M_previousPositions[i][0], M_previousPositions[i][1], nearestX, nearestY, index)
				if (sqrt((nearestX - M_previousPositions[i][0])^2 + (nearestY - M_previousPositions[i][1])^2) > maxStepSize)
					// it looks like this particular marker was not recovered in the fit in this frame
					// we shouldn't use it
					continue
				else	// the marker was recovered
					nMarkersRecovered += 1
					driftX += nearestX - M_previousPositions[i][0]
					driftY += nearestY - M_previousPositions[i][1]
					M_previousPositions[i][0] = nearestX
					M_previousPositions[i][1] = nearestY
				endif
		endfor
		
		// take the average of the individual drifts as the absolute drift
		if (nMarkersRecovered == 0)
			M_RelativeDrift[currentFrame][] = NaN
			M_AbsoluteDrift[currentFrame][] = NaN
		else
			// store the relative drift compared to the previous frame
			M_RelativeDrift[currentFrame][0] = driftX / nMarkersRecovered
			M_RelativeDrift[currentFrame][1] = driftY / nMarkersRecovered
			
			// store the absolute drift compared to the initial frame
			if (currentFrame == 0)	// the first frame is special because there is no previous frame
				M_AbsoluteDrift[currentFrame][0] = M_RelativeDrift[currentFrame][0]
				M_AbsoluteDrift[currentFrame][1] = M_RelativeDrift[currentFrame][1]
			else
				M_AbsoluteDrift[currentFrame][0] = M_AbsoluteDrift[currentFrame - 1][0] + M_RelativeDrift[currentFrame][0]
				M_AbsoluteDrift[currentFrame][1] = M_AbsoluteDrift[currentFrame - 1][1] + M_RelativeDrift[currentFrame][1]
			endif
		endif
			
		
	endfor
		
	KillWaves M_previousPositions
End


Function MakeMovieFromOpenFile()
	
	SVAR dataFilePath = root:Packages:PALM:S_CCD_file_path
	NVAR nCCDFrames = root:Packages:PALM:V_numberOfImages
	NVAR cameraType = root:Packages:PALM:V_cameraType
	NVAR xSize = root:Packages:PALM:V_xSize
	NVAR ySize = root:Packages:PALM:V_ySize
	NVAR currentImage = root:Packages:PALM:V_currentImage
	
	variable firstFrameInMovie = 1, lastFrameInMovie = nCCDFrames		// use numbering starting from 1
	variable originalFrame = currentImage
	variable fps = 25, refnum, i
	string outputFilePath
	string error
	
	if ((SVAR_exists(dataFilePath) == 0) || (strlen(dataFilePath) == 0))	// no CCD file has been opened yet
		Abort "No CCD file seems to have been opened"
	endif
	
	DoWindow CCDViewer
	if (V_flag != 1)
		Abort "No CCD file seems to be open"
	endif
	
	Prompt firstFrameInMovie, "First frame in movie:"
	Prompt lastFrameInMovie, "Last frame in movie:"
	
	DoPrompt "Enter movie parameters", firstFrameInMovie, lastFrameInMovie
	if (V_Flag == 1)
		return 1
	endif
	
	// remove the extension of the data file and replace it with .mov
	outputFilePath  = ParseFilePath(3, dataFilePath, ":", 0, 0)
	outputFilePath += ".mov"
	
	if (firstFrameInMovie < 0)
		Abort "Invalid first frame"
	elseif (lastFrameInMovie > nCCDFrames)
		Abort "The data contains less frames than requested"
	elseif (firstFrameInMovie >= lastFrameInMovie)
		Abort "Invalid frame range"
	endif
	
	Open /D /T=".mov" refnum as outputFilePath
	if (strlen(S_fileName) == 0)
		return 1
	endif
	
	outputFilePath = S_fileName
	
	GetCCDFrame(dataFilePath, cameraType, firstFrameInMovie - 1)
	wave Viewer_Temp = root:Packages:PALM:Viewer_Temp
	
	// remove a pre-exisiting movie viewer if it exists
	DoWindow /K MovieViewer
	
	// first create the window as a copy of the CCDViewer
	string viewerRecreation = WinRecreation("CCDViewer", 0)
	Execute viewerRecreation
	DoWindow /T kwTopWin, "MoviePlayer"
	DoWindow /C MovieViewer
	ModifyGraph /W=MovieViewer margin=-1
	ModifyGraph /W=MovieViewer noLabel=2,axThick=0
	
	NewMovie /Z /O /L /F=(fps) /I as outputFilePath
	if (V_flag != 0)
		DoWindow /K MovieViewer
		error = GetErrMessage(V_flag, 3)
		Abort error
	endif
	
	for (i = firstFrameInMovie - 1; i < lastFrameInMovie; i+=1)
		GetCCDFrame(dataFilePath, cameraType, i)
		currentImage = i
		GenerateGUIThreshold()
		DoUpdate
		AddMovieFrame
	endfor
	
	CloseMovie
	DoWindow /K MovieViewer
	
	// restore the image we were originally looking at
	currentImage = originalFrame
	GetCCDFrame(dataFilePath, cameraType, originalFrame)
	GenerateGUIThreshold()
End


Function ConvertFolderOfCCDFilesToMovies()
	string savDF = GetDataFolder(1)
	NewDataFolder /O root:Packages
	NewDataFolder /O/S root:Packages:PALM
	
	NVAR currentImage = root:Packages:PALM:V_currentImage
	NVAR cameraType = root:Packages:PALM:V_cameraType
	NVAR xSize = root:Packages:PALM:V_xSize
	NVAR ySize = root:Packages:PALM:V_ySize
	NVAR nImages = root:Packages:PALM:V_numberOfImages
	
	variable oldImage, oldCameraType
	SVAR dataFilePath = root:Packages:PALM:S_CCD_file_path
	string dataFiles, baseFilePath, currentFilePath, outputFilePath, oldDataFilePath, baseOutputFilePath
	variable nCCDFiles, i, j
	string error
	
	if (SVAR_Exists(dataFilePath) != 0)	// did we already open a CCD file?
		oldDataFilePath = dataFilePath
		oldImage = currentImage
		oldCameraType = cameraType
	else
		oldDataFilePath = ""
	endif
	
	NewPath /Q/O/M="Select the folder containing the CCD files" inputPath
	NewPath /Q/O/M="Select the folder where the movies should be saved" outputPath
	
	PathInfo inputPath
	baseFilePath = S_path
	
	PathInfo outputPath
	baseOutputFilePath = S_path
	
	dataFiles = IndexedFile(inputPath, -1, ".spe")
	dataFiles += IndexedFile(inputPath, -1, ".sif")
	dataFiles += IndexedFile(inputPath, -1, ".his")
	dataFiles += IndexedFile(inputPath, -1, ".tif")
	
	nCCDFiles = itemsInList(dataFiles)
	
	if (nCCDFiles == 0)
		SetDataFolder savDF
		Abort "No valid CCD files found in the input folder"
	endif
	
	for (i = 0; i < nCCDFiles; i += 1)
		currentFilePath = baseFilePath + StringFromList(i, dataFiles)
		cameraType = GetFileFormat(currentFilePath)
		
		outputFilePath  = baseOutputFilePath + StringFromList(i, dataFiles)
		outputFilePath  = ParseFilePath(3, outputFilePath, ":", 0, 0)
		outputFilePath += "mov"
		
		// get some info on the file dimensions and so on
		ReadCCDImages /Y=(cameraType) /H currentFilePath
		if (V_flag != 0)		// did we fail to read the data?
			SetDataFolder savDF
			string errorMessage = "Unable to read from  " + currentFilePath
			Abort errorMessage
		endif
		xSize = V_xSize
		ySize = V_ySize
		nImages = V_numberOfImages
		
		// the first file is special since we will initialize the viewer window here
		if (i == 0)
			Make /O/N=(2,2) Viewer_Temp	// make a dummy Viewer_Temp that will be replaced as soon as the first file is loaded
			Display /K=1 /N=MovieViewer; AppendImage Viewer_Temp
			ModifyGraph margin=-1
			ModifyGraph noLabel=2,axThick=0
			ModifyGraph width=0, height={aspect, ySize / xSize},tickUnit=1
			NewMovie /Z /I /L /O as outputFilePath	// allow the user to change the movie conversion settings for the first CCD file
			if (V_flag != 0)
				SetDataFolder savDF
				DoWindow /K MovieViewer
				error = GetErrMessage(V_flag, 3)
				Abort error
			endif
		else
			ModifyGraph width=0, height={aspect, ySize / xSize}
			NewMovie /Z /L /O as outputFilePath
			if (V_flag != 0)
				DoWindow /K MovieViewer
				SetDataFolder savDF
				Abort "An error occured; could not make the movie " + outputFilePath
			endif
		endif
			
		for (j = 0; j < nImages; j+=1)
			GetCCDFrame(currentFilePath, cameraType, j)
			DoUpdate
			AddMovieFrame
		endfor
		CloseMovie
	endfor
	
	// cleanup
	DoWindow /K MovieViewer
	if (strlen(oldDataFilePath) != 0)
		GetCCDFrame(oldDataFilePath, oldCameraType, oldImage)
	endif
	SetDataFolder savDF
End


Function RecoverEmittersByBleachingSteps()
	// assume that there is already a movie open in the GUI viewer that we want to analyse
	
	NewDataFolder /O root:'PALM Positions'
	DFREF savDF = GetDataFolderDFR()
	
	Struct FitData fp
	// get all the fit options and parameters the user defined in the GUI
	GetGUIFitSettingsAndOptions(fp)
	
	// Currently only symmetric 2D Gaussian fitting is supported
	if ((fp.localizationMethod != LOCALIZATION_GAUSS_FITTING) && (fp.localizationMethod != LOCALIZATION_GAUSS_FITTING_FIX))
		Abort "Only 2D symmetric Gaussian fitting is supported for the bleaching analysis"
	endif
	
	variable previousLoadedImage = fp.currentImage
	variable i, j, nPositionsFound, nPositionsOnWaitingList, nearestX, nearestY, closestIndex
	variable xCol, yCol, zCol, intensityCol, widthCol, dummyCol
	variable firstRun = 1
	variable previousProgress = 0
	variable progress, offset
	
	Printf "%s", "Running bleaching analysis..."
	
	for (i = fp.numberOfImages - 1; i >= 0; i-=1)
		GetCCDFrame(fp.CCDFilePath, fp.cameraType, i)
		wave Viewer_Temp = root:Packages:PALM:Viewer_Temp
		
		 if (firstRun == 1)
			// set up a wave that will contain the subtracted frame
			Duplicate /O Viewer_Temp, root:Packages:PALM:M_Subtracted
			wave M_Subtracted = root:Packages:PALM:M_Subtracted
			Redimension /D M_Subtracted
			
			// set up a wave that will contain the contributions of all emitters that have been identified
			Duplicate /O Viewer_Temp, root:Packages:PALM:M_SummedEmitters
			Wave M_SummedEmitters = root:Packages:PALM:M_SummedEmitters
			Redimension /D M_SummedEmitters
			FastOP M_SummedEmitters = 0
		endif
		
		progress = (fp.numberOfImages - 1 - i) / (fp.numberOfImages - 1) * 100
		if (progress - previousProgress > 10)	// print a progress notification every 10%
			previousProgress = floor(progress / 10) * 10
			Printf " %d%%", previousProgress
		endif
		
		SetDataFolder root:Packages:PALM
		
		// subtract all emitters that have been localized in previous frames
		MatrixOP /O M_Subtracted = Viewer_Temp - M_SummedEmitters
		
		LocalizationAnalysis /M=(fp.localizationMethod) /D=(fp.thresholdMethod) /Y=(CAMERA_TYPE_IGOR_WAVE) /G={fp.preprocessing, fp.postprocessing} /F=(fp.particlefinder) /PFA=(fp.PFA) /T=(fp.directThresholdLevel) /R=(fp.minDistanceBetweenParticles) /W=(fp.PSFWidth) /PVER={fp.PVerSymm, fp.PVerEllipse, fp.PVerOverlap} /Z /Q /DEST=root:Packages:PALM:POS_out "M_Subtracted"
		SetDataFolder savDF
		
		if (V_flag != 0)	// an error occured during the fitting
			string errorMessage = "An error occurred while fitting frame " + num2str(i + 1)
			Abort errorMessage
		endif
		
		wave POS_out = root:Packages:PALM:POS_out
		 nPositionsFound = DimSize(POS_out, 0)
		 
		 if (firstRun == 1)
			// make the output wave for the positions
			Duplicate /O POS_out, root:'PALM Positions':BleachingAnalysis
			wave M_BleachingAnalysis = root:'PALM Positions':BleachingAnalysis
			Redimension /N=(0, DimSize(POS_out, 1)) M_BleachingAnalysis
			
			#ifdef USE_WAITING_LIST
			Duplicate /O POS_out, root:Packages:PALM:M_WaitingList			// a wave that will contain a list points encountered in the previous frame
			wave M_WaitingList = root:Packages:PALM:M_WaitingList			// the reason for its existence is that the frame in which we first encounter an emitter may not show its correct intensity since the emitter bleaches somewhere in that frame
			Redimension /N=(0, DimSize(POS_out, 1)) M_WaitingList			// so when a new point is found, add it to the waiting list first, and then only include it in the analysis if the next frame shows it again
			#endif
			
			// get some indices into the columns of the positions wave
			GetColumnsForEmitterPositions(POS_out, xCol, yCol, zCol)
			GetColumnForIntegratedIntensity(POS_out, intensityCol)
			GetColumnForFittedWidth(POS_out, widthCol, dummyCol)
			
			// if no width or amplitude is provided then abort
			if ((intensityCol < 0) || (widthCol < 0))
				Abort "The bleaching analysis requires a fitting method that provides amplitude and width information (e.g. Gaussian fitting)"
			endif
			
			// set up some viewer windows
			DoWindow /F BleachingAnalysis_Original
			if (V_flag == 0)
				Display /K=1 /N=BleachingAnalysis_Original ; AppendImage Viewer_Temp
				AppendToGraph /W=BleachingAnalysis_Original M_BleachingAnalysis[][yCol] vs M_BleachingAnalysis[][xCol]
				ModifyGraph /W=BleachingAnalysis_Original mode=3
			endif
			DoWindow /F BleachingAnalysis_Subtracted
			if (V_flag == 0)
				Display /K=1 /N=BleachingAnalysis_Subtracted ; AppendImage M_Subtracted
				AutoPositionWindow /M=0 /R=BleachingAnalysis_Original BleachingAnalysis_Subtracted
			endif
			
			ModifyGraph /W=BleachingAnalysis_Original width={Aspect, (DimSize(Viewer_Temp, 0) / DimSize(Viewer_Temp, 1))}, height=0
			ModifyGraph /W=BleachingAnalysis_Subtracted width={Aspect, (DimSize(Viewer_Temp, 0) / DimSize(Viewer_Temp, 1))}, height=0
			
			firstRun = 0
		endif
		 
		 #ifdef USE_WAITING_LIST 
			 if (i != 0)	// for all frames except the first, work with a waiting list
				 // did we localize any emitters that are on the waiting list?
				 for (j = 0; j < DimSize(M_WaitingList, 0); j+=1)
				 	// did we localize this position again?
				 	returnPointNearestFitPositions(POS_Out, M_WaitingList[j][xCol], M_WaitingList[j][yCol], nearestX, nearestY, closestIndex)
				 	
				 	if (sqrt((nearestX - M_WaitingList[j][xCol])^2 + (nearestY - M_WaitingList[j][yCol])^2) > SAME_EMITTER_THRESHOLD_FACTOR * fp.PSFWidth)	// is it close enough to count as the same emitter?
				 		// delete this point from the waiting list, we can't reproduce it
				 		DeletePoints j, 1, M_WaitingList
				 		j -= 1
				 		continue
				 		
				 	else		// the point corresponds to the same emitter as the one on the waiting list
				 			// add it to the list of recovered emitters, and remove it from the fitted positions list (otherwise it will end up on the waiting list again)
				 		Redimension /N=(DimSize(M_BleachingAnalysis, 0) + 1, DimSize(M_BleachingAnalysis, 1)) M_BleachingAnalysis
				 		M_BleachingAnalysis[DimSize(M_BleachingAnalysis, 0) - 1][] = POS_out[closestIndex][q]
				 		M_BleachingAnalysis[DimSize(M_BleachingAnalysis, 0) - 1][0] = i
				 		
				 		// add its contribution
				 		AddEmitter(M_SummedEmitters, POS_out[closestIndex][intensityCol], POS_out[closestIndex][widthCol], POS_out[closestIndex][xCol], POS_out[closestIndex][yCol])
				 		
				 		DeletePoints closestIndex, 1, POS_out
				 		DeletePoints j, 1, M_WaitingList
				 		j -= 1
				 	endif
				 endfor
			else	// for the very first frame, don't use the waiting list but just add all fitted points to the output waves
				// requested by Rob 30102009
				
				offset = DimSize(M_BleachingAnalysis, 0)
				Redimension /N=(DimSize(M_BleachingAnalysis, 0) + DimSize(POS_Out, 0), DimSize(M_BleachingAnalysis, 1)) M_BleachingAnalysis
		 		
		 		for (j = 0; j < DimSize(POS_Out, 0); j+=1)
		 			M_BleachingAnalysis[j + offset][] = POS_Out[j][q]
		 			M_BleachingAnalysis[j + offset][0] = i
		 			AddEmitter(M_SummedEmitters, POS_out[j][intensityCol], POS_out[j][widthCol], POS_out[j][xCol], POS_out[j][yCol])
		 		endfor
				
			endif
			 
			 // all points that are in the fitting list now count as new potential candidates
			 // add them to the waiting list
			 if (DimSize(POS_out, 0) > 0)
				Redimension /N=(DimSize(POS_out, 0), DimSize(POS_out, 1)) M_WaitingList
				M_WaitingList = POS_out
			endif
			 
			 DoUpdate
		#else	// don't use a waiting list, just included the localized positions directly
			offset = DimSize(M_BleachingAnalysis, 0)
			Redimension /N=(DimSize(M_BleachingAnalysis, 0) + DimSize(POS_Out, 0), DimSize(M_BleachingAnalysis, 1)) M_BleachingAnalysis
			
			for (j = 0; j < DimSize(POS_Out, 0); j+=1)
	 			M_BleachingAnalysis[j + offset][] = POS_Out[j][q]
	 			M_BleachingAnalysis[j + offset][0] = i
	 			AddEmitter(M_SummedEmitters, POS_out[j][intensityCol], POS_out[j][widthCol], POS_out[j][xCol], POS_out[j][yCol])
	 		endfor
			
			DoUpdate
		#endif
	endfor
	
	// now reverse the order of the fitted positions so the positions are located according to incrementing frame number
	Duplicate /FREE/O M_BleachingAnalysis, ShiftedBleaching
	ShiftedBleaching = M_BleachingAnalysis[DimSize(M_BleachingAnalysis, 0) - 1 - p][q]
	M_BleachingAnalysis = ShiftedBleaching
	
	GetCCDFrame(fp.CCDFilePath, fp.cameraType, previousLoadedImage)
	Printf "%s", "Calculation finished!\r"
	
End


Function SubtractEmitter(image, integral, width, xPos, yPos)
	wave image
	variable integral, width, xPos, yPos
	
	Assert(WaveExists(image))
	Assert((width > 0) && (xPos >= 0) && (yPos >= 0))
	
	// given image, subtract the contribution of an emitter centered at the given position with the given characteristics
	// we don't make an effort to check if the emitter is really there or not, we just assume that it is
	
	variable startP, endP
	variable startQ, endQ
	variable i, j
	variable xSize = DimSize(image, 0)
	variable ySize = DimSize(image, 1)
	variable value, distance, width_exp, amplitude
	
	startP = floor(xPos - 4 * width)
	endP = ceil(xPos + 4 * width)
	startQ = floor(yPos - 4 * width)
	endQ = ceil(yPos + 4 * width)
	
	startP = (startP >= 0) ? startP : 0
	startQ = (startQ >= 0) ? startQ : 0
	endP = (endP < xSize) ? endP : xSize - 1
	endQ = (endQ < ySize) ? endQ : ySize - 1
	
	amplitude = integral / (2 * pi * width * width);
	width_exp = 2 * width^2
	
	for (i = startP; i <= endP; i+=1)
		for (j = startQ; j <= endQ; j+=1)
			distance = (i - xPos)^2 + (j - yPos)^2
			value = amplitude * exp(-distance / width_exp)
			image[i][j] -= value
		endfor
	endfor
	
End

Function AddEmitter(image, integral, width, xPos, yPos)
	wave image
	variable integral, width, xPos, yPos
	
	SubtractEmitter(image, -1 * integral, width, xPos, yPos)
	
End

Function LuckyImaging_Menu()
	string posName, outputName, waveNote
	variable fraction, mode
	fraction = 0.1
	
	Prompt fraction, "Fraction of positions to retain (0 - 1):"
	Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
	Prompt mode, "'Best' is estimated as:", popup, "Largest Amplitude;Smallest Width;Lowest Fit Position Uncertainty;"

	
	do
		DoPrompt "Choose parameters", posName, mode, fraction
		if (V_flag == 1)	// cancel
			return 0
		endif
		
		if ((fraction < 0) || (fraction > 1))
			Abort "Please specify a a value between 0 and 1 for the fraction to retain"
			fraction = 0.1
			continue
		endif
	while (1)
	
	wave positions = GetPositionsWaveReference(posName)
	mode -= 1	// compensate for numbering from 1 in popup
	
	if (WaveExists(positions) == 0)
		Abort "Couldn't locate the positions wave"
	endif
	
	outputName = GetOutputWaveName("", root:'PALM Positions', "Output name:")
	if (strlen(outputName) == 0)
		return 0
	endif
	
	LuckyImaging(positions, fraction, mode)
	wave M_LuckyPositions = root:Packages:PALM:M_LuckyPositions
	Duplicate /O M_LuckyPositions, root:'PALM Positions':$outputName
	waveNote = note(positions)
	Note /K  root:'PALM Positions':$outputName
	Note root:'PALM Positions':$outputName, waveNote
	
	KillWaves M_LuckyPositions
End

Function LuckyImaging(positions, fraction, mode)
	wave positions
	variable fraction, mode
	
	Assert(WaveExists(positions))
	
	if ((fraction < 0) || (fraction > 1))
		Abort "Invalid fraction supplied to LuckyImaging()"
	endif
	
	// apply lucky imaging on the position by retaining only the $fraction best positions, where 'best' is defined by the supplied mode
	
	variable nPositions = DimSize(positions, 0)
	variable nLuckyPositions = round(fraction * nPositions)
	
	Make /I/U/O/N=(nPositions) W_SortedIndices
	Make /D/O/N=(nLuckyPositions, DimSize(positions, 1)) M_LuckyPositions_Unsorted
	
	// sort W_SortingIndices so that the best (luckiest) points come first
	switch (mode)
		case LUCKYIMAGING_AMPLITUDE:
			// retain only the spots that have the highest amplitude
			Make /D/O/N=(nPositions) W_Amplitudes
			
			W_Amplitudes = positions[p][1]
			W_SortedIndices = p
			
			Sort /R W_Amplitudes, W_SortedIndices
			KillWaves W_Amplitudes
			break
		case LUCKYIMAGING_SMALLESTWIDTH:
			// retain only the spots that have the smallest widths
			Make /D/O/N=(nPositions) W_Widths
			
			W_Widths = positions[p][2]
			W_SortedIndices = p
			
			Sort W_Widths, W_SortedIndices
			KillWaves W_Widths
			break
		case LUCKYIMAGING_POSUNCERTAINTY:
			// retain only the spots that have lowest reported position uncertainty
			Make /D/O/N=(nPositions) W_PositionUncertainty
			
			W_PositionUncertainty = (positions[p][8] + positions[p][9]) / 2
			W_SortedIndices = p
			
			Sort W_PositionUncertainty, W_SortedIndices
			KillWaves W_PositionUncertainty
			break
		default:	
			KillWaves W_SortedIndices, M_LuckyPositions_Unsorted
			Abort "Unknown lucky imaging mode"
			break
	endswitch
	
	// only retain the best points
	M_LuckyPositions_Unsorted = positions[W_SortedIndices[p]][q]
	
	// sort M_LuckyPositions so that they are again arranged according to frame numbers
	Duplicate /O M_LuckyPositions_Unsorted, root:Packages:PALM:M_LuckyPositions
	wave M_LuckyPositions = root:Packages:PALM:M_LuckyPositions
	Redimension /N=(nLuckyPositions)W_SortedIndices
	Make /I/U/O/N=(nLuckyPositions) W_FrameNumbers
	W_SortedIndices = p
	W_FrameNumbers = M_LuckyPositions_Unsorted[p][0]
	
	Sort W_FrameNumbers, W_SortedIndices
	
	M_LuckyPositions = M_LuckyPositions_Unsorted[W_SortedIndices[p]][q]
	
	KillWaves W_SortedIndices, W_FrameNumbers, M_LuckyPositions_Unsorted
	
End

Function SetupBatchLocalizationPanel()
	
	DoWindow CCDViewer
	if (V_flag != 1)
		Abort "Please open a CCD file and set the desired processing options in the \"CCDViewer\" window"
	endif
	
	Make /O/N=(0, 2) /T root:Packages:PALM:M_BatchProcessListWave
	Make /O/B/U/N=(0, 2, 2) root:Packages:PALM:M_BatchProcessSelectionWave
	Make /O/W/U/N=(2, 3) root:Packages:PALM:M_BatchProcessColorWave
	Make /O/N=2 /T root:Packages:PALM:M_BatchProcessColumnHeadings
	wave /T M_BatchProcessColumnHeadings = root:Packages:PALM:M_BatchProcessColumnHeadings
	wave M_BatchProcessColorWave = root:Packages:PALM:M_BatchProcessColorWave
	wave M_BatchProcessSelectionWave = root:Packages:PALM:M_BatchProcessSelectionWave
	
	M_BatchProcessColumnHeadings[0] = "CCD Files"
	M_BatchProcessColumnHeadings[1] = "Output position names"
	
	M_BatchProcessColorWave = 0
	M_BatchProcessColorWave[1][0] = 65535
	SetDimLabel 2,1, backColors, M_BatchProcessSelectionWave
	M_BatchProcessSelectionWave[][][1] = 0	// provide the default Igor colors
	
	DoWindow /F BatchProcess
	if (V_flag == 0)
		NewPanel /K=1 /N=BatchProcess /W=(238,189,792,491) as "Select the files to localize (double click to remove)"
		SetDrawLayer UserBack
		SetDrawEnv fsize= 11,fstyle= 2
		DrawText 9,20,"All files will be processed with the options selected in the \\\"CCD Viewer\\\" panel"
		ListBox LBBatchListLocalization,pos={7,27},size={541,246}, proc=LBDeleteDoubleClickedRowProc, listWave=root:Packages:PALM:M_BatchProcessListWave
		ListBox LBBatchListLocalization,frame=2,userColumnResize=1,titleWave=root:Packages:PALM:M_BatchProcessColumnHeadings
		ListBox LBBatchListLocalization,mode=1,selWave=root:Packages:PALM:M_BatchProcessSelectionWave
		ListBox LBBatchListLocalization,colorWave = root:Packages:PALM:M_BatchProcessColorWave
		ListBox LBBatchListLocalization,userdata(ResizeControlsInfo)= A"!!,@C!!#=;!!#Cl5QF/pz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
		ListBox LBBatchListLocalization,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
		ListBox LBBatchListLocalization,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		Button BTBatchStartAnalysis,pos={434,277},size={100,20},title="Start!",proc=BTStartBatchLocalization
		Button BTBatchStartAnalysis,userdata(ResizeControlsInfo)= A"!!,I?!!#BDJ,hpW!!#<Xz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
		Button BTBatchStartAnalysis,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		Button BTBatchStartAnalysis,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		Button BTBatchAddLocalizationFiles,pos={320,276},size={100,20},title="Add files",proc=BTBatchAddDataFiles
		Button BTBatchAddLocalizationFiles,userdata(ResizeControlsInfo)= A"!!,H[!!#BD!!#@,!!#<Xz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
		Button BTBatchAddLocalizationFiles,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		Button BTBatchAddLocalizationFiles,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		CheckBox CBBatchSaveExperiment,pos={6,279},size={175,14},title="Autosave experiment after each run"
		CheckBox CBBatchSaveExperiment,userdata(ResizeControlsInfo)= A"!!,@#!!#BEJ,hqi!!#;mz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
		CheckBox CBBatchSaveExperiment,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		CheckBox CBBatchSaveExperiment,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		CheckBox CBBatchSaveExperiment,value= 1
		
		// make the controls resize dynamically when the window is resized thanks to the <Resize Controls> package
		SetWindow BatchProcess,hook(ResizeControls)=ResizeControls#ResizeControlsHook
		SetWindow kwTopWin,userdata(ResizeControlsInfo)= A"!!*'\"z!!#CoJ,hs'zzzzzzzzzzzzzzzzzzzzz"
		SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzzzzzzzz"
		SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzz!!!"
		
		// if the window was opened previously then restore its size and position
		WC_WindowCoordinatesRestore("BatchProcess")
		SetWindow BatchProcess hook(WindowCoordsHook)=WC_WindowCoordinatesNamedHook
	endif
	
End


Function BTBatchAddDataFiles(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	string baseFileName, extension, outputFilePath, macintoshFilePath
	variable nFiles, nNewFiles, i, offset
	
	switch( ba.eventCode )
		case 2: // mouse up
				// click code here
			wave /T M_BatchProcessListWave = root:Packages:PALM:M_BatchProcessListWave
			wave M_BatchProcessSelectionWave = root:Packages:PALM:M_BatchProcessSelectionWave
			wave M_BatchProcessColorWave = root:Packages:PALM:M_BatchProcessColorWave
			
			Open /D/R /MULT=1/T="????" refNum
			if (strlen(S_fileName) == 0)
				// user canceled
				return 0
			endif
			
			string callingControlName = ba.ctrlName
			offset = DimSize(M_BatchProcessListWave, 0)
			nNewFiles = ItemsInList(S_fileName, "\r") 
			nFiles = DimSize(M_BatchProcessListWave, 0) + nNewFiles
			Redimension /N=(nFiles, 2) M_BatchProcessListWave
			Redimension /N=(nFiles, 2, 2) M_BatchProcessSelectionWave
			for (i = 0; i < nNewFiles; i+=1)
				M_BatchProcessListWave[offset][0] = StringFromList(i, S_fileName, "\r")
				
				// the requested output depends on what control called this
				// this allows this proc to be used for both batch processing and
				// batch localization
				strswitch (callingControlName)
					case "BTBatchProcessAddFiles":
						// will the output be saved as a processed file or as a wave?
						ControlInfo /W=CCDProcessPanel PMSelectProcessingOutputType
						StrSwitch (S_Value)
							case "TIFF file":
							case "Zip-Compressed TIFF File":
								macintoshFilePath = ParseFilePath(5, StringFromList(i, S_fileName, "\r"), ":", 0, 0)
								extension = "." + ParseFilePath(4, macintoshFilePath, ":", 0, 0)
								outputFilePath = ReplaceString(extension, macintoshFilePath, "_processed.tif")
								M_BatchProcessListWave[offset][1] = outputFilePath
								break
							case "PDE File":
								macintoshFilePath = ParseFilePath(5, StringFromList(i, S_fileName, "\r"), ":", 0, 0)
								extension = "." + ParseFilePath(4, macintoshFilePath, ":", 0, 0)
								outputFilePath = ReplaceString(extension, macintoshFilePath, "_processed.pde")
								M_BatchProcessListWave[offset][1] = outputFilePath
								break
							case "Igor wave":
								macintoshFilePath = ParseFilePath(5, StringFromList(i, S_fileName, "\r"), ":", 0, 0)
								baseFileName = ParseFilePath(3, macintoshFilePath, ":", 0, 0)
								//provide a valid Igor output name
								baseFileName = CleanupName(baseFileName, 1)
								M_BatchProcessListWave[offset][1] = baseFileName
								break
							default:
								Abort "Unknown processing output type in StrSwitch"
						EndSwitch
						
						// if we get here then some files were added
						// do not allow the user to change the output format anymore
						PopupMenu PMSelectProcessingOutputType win=CCDProcessPanel, disable=2
						break
					case "BTBatchAddLocalizationFiles":
						// the output data will be saved as igor waves
						baseFileName = ParseFilePath(3, StringFromList(i, S_fileName, "\r"), ":", 0, 0)
						//provide a valid Igor output name
						baseFileName = CleanupName(baseFileName, 1)
						M_BatchProcessListWave[offset][1] = baseFileName
						break
					default:
						Abort "Internal error: unknown control calling BTBatchAddDataFiles"
						break
				endswitch
				
				offset += 1
			endfor
			
			M_BatchProcessSelectionWave[][1][0] = 2	// be sure that the output names are editable
			
			break
	endswitch	
	
	return 0
End

Function BTStartBatchLocalization(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	variable nFiles, nNewFiles, i, j, startTime
	variable autoSave
	variable invalidOutputNames = 0
	variable identicalNames = 0
	Struct FitData fitParams
	variable err
	
	switch( ba.eventCode )
		case 2: // mouse up
				// click code here
			wave /T M_BatchProcessListWave = root:Packages:PALM:M_BatchProcessListWave
			wave M_BatchProcessSelectionWave = root:Packages:PALM:M_BatchProcessSelectionWave
			wave M_BatchProcessColorWave = root:Packages:PALM:M_BatchProcessColorWave
			
			Make /T/O/N=(0, 2) root:Packages:PALM:M_BatchProcessErrors	// keep track of data files that had errors
			wave /T M_BatchProcessErrors = root:Packages:PALM:M_BatchProcessErrors
			
			
			nFiles = DimSize(M_BatchProcessListWave, 0)
			
			// make sure that none of the cells are highlighted
			M_BatchProcessSelectionWave[][][1] = 0	// provide the default Igor colors
				
			// are any of the output file names invalid?
			for (i = 0; i < nFiles; i+=1)
				if ((StringMatch(M_BatchProcessListWave[i][1], CleanupName(M_BatchProcessListWave[i][1], 1)) != 1) || (strlen(M_BatchProcessListWave[i][1]) == 0))
					M_BatchProcessSelectionWave[i][1][1] = 1	// give the cell a red color
					invalidOutputNames = 1
				endif
			endfor
			
			if (invalidOutputNames == 1)
				Abort "Some of the output names are invalid, please correct these"
			endif
			
			// do any of the positions have identical names?
			for (i = 0; i < nFiles - 1; i+=1)
				for (j = i + 1; j < nFiles; j+=1)
					if (StringMatch(M_BatchProcessListWave[i][1], M_BatchProcessListWave[j][1]) == 1)
						M_BatchProcessSelectionWave[i][1][1] = 1// give the cell a red color
						M_BatchProcessSelectionWave[j][1][1] = 1
						identicalNames = 1
					endif
				endfor
			endfor
			
			if (identicalNames == 1)
				Abort "Some of the output names are identical, please correct these"
			endif
			
			// if we're here then we're all set to go
			ControlInfo /W=BatchProcess CBBatchSaveExperiment
			autoSave = V_Value
			
			DoWindow /K BatchProcess
			
			// if the user selected to autosave the experiment, save it now
			// this avoids the situation where the experiment was not saved before and the 'SaveExperiment'
			// operation opens up a save dialog, but only after the first run has finished
			if (autoSave != 0)
				SaveExperiment
			endif
			
			Print "Starting batch analysis..."
			startTime = StopMSTimer(-2)
			
			for (i = 0; i < nFiles; i+=1)
				Printf "Now analyzing %s\t\t(file %d out of %d)\r", M_BatchProcessListWave[i][0], i + 1, nFiles
				// get all the fit options and parameters the user defined in the GUI
				GetGUIFitSettingsAndOptions(fitParams)
				
				// set the data filepath to the requested file
				fitParams.CCDFilePath = M_BatchProcessListWave[i][0]
				
				// run the actual fit
				err = DoPALMFitting(fitParams, 0)
				if (err != 0)
					if (err == kUserAbort)
						// the user wants to abort the analysis
						Print "Batch fitting stopped due to user abort"
						KillWaves /Z M_BatchProcessListWave, M_BatchProcessSelectionWave, M_BatchProcessColorWave, M_BatchProcessErrors
						return 0
					endif
					
					// if we got here then it means that there was an error fitting the current data file
					// don't abort now, but remember the filePath that caused the error and report it to the user
					Redimension /N=(DimSize(M_BatchProcessErrors, 0) + 1, DimSize(M_BatchProcessErrors, 1)) M_BatchProcessErrors
					M_BatchProcessErrors[DimSize(M_BatchProcessErrors, 0) - 1][0] = M_BatchProcessListWave[i][0]
					M_BatchProcessErrors[DimSize(M_BatchProcessErrors, 0) - 1][1] = M_BatchProcessListWave[i][1]
					
					// move on to the next file
					continue
				endif
					
				wave POS_Out = root:Packages:PALM:POS_out
				// copy the wave to the requested output positions name
				Duplicate /O POS_Out, root:'PALM Positions':$M_BatchProcessListWave[i][1]
				
				// if the users selected to autosave the experiment, do so now
				if (autoSave != 0)
					SaveExperiment
				endif
			endfor
			
			// if we're here then the fitting is finished
			Print "Batch processing finished"
			Printf "Total duration is %g\n", PrettyPrintTimeElapsed((StopMSTimer(-2) - startTime) / 1e6)
			
			// if there were any errors then report them now
			if (DimSize(M_BatchProcessErrors, 0) != 0)
				Print "WARNING: the following data file(s) reported errors and should not be trusted:"
				for (i = 0; i < DimSize(M_BatchProcessErrors, 0); i+=1)
					Printf "%d. %s,  %s %s\r", i + 1, M_BatchProcessErrors[i][0], "positions were to be written in", M_BatchProcessErrors[i][1]
				endfor
			endif
			
			
			// clean up
			KillWaves /Z M_BatchProcessListWave, M_BatchProcessSelectionWave, M_BatchProcessColorWave, M_BatchProcessErrors
			
			break
	endswitch
	
	return 0
End


Function MakeCroppedStackFromMarquee()

	SVAR CCDFilePath = root:Packages:PALM:S_CCD_file_path
	NVAR cameraType = root:Packages:PALM:V_cameraType
	NVAR xSize = root:Packages:PALM:V_xSize
	NVAR ySize = root:Packages:PALM:V_ySize
	
	variable xStart, yStart, xEnd, yEnd
	
	String outputFilePath
	
	if ((SVAR_Exists(CCDFilePath) == 0) || (NVAR_Exists(cameraType) == 0))
		Abort "It doesn't look like a CCD file has been opened in this experiment"
	endif
	
	// for now only allow cropping to be done if the top window is the window containing the ccd images
	GetMarquee /K /Z left, bottom
	if (StringMatch(S_MarqueeWin, "CCDViewer*") != 1)
		Abort "Cropping should only be done on the window displaying the CCD frames"
	endif
	
	// get the coordinates
	xStart = floor(min(V_right, V_left))
	xEnd = ceil(max(V_right, V_left))
	yStart = floor(min(V_top, V_bottom))
	yEnd = ceil(max(V_top, V_bottom))
	
	// if the coordinates are outside the frame dimensions then correct for that
	xStart = (xStart < 0) ? 0 : xStart
	yStart = (yStart < 0) ? 0 : yStart
	xEnd = (xEnd >= xSize) ? xSize - 1 : xEnd
	yEnd = (yEnd >= ySize) ? ySize - 1 : yEnd
	
	
	String outputFormat
	Prompt outputFormat, "Format:", popup, "TIFF file;Zip-Compressed TIFF File;PDE File;Igor wave"
	DoPrompt "Select the output format", outputFormat
	if (V_flag != 0)
		return 0
	endif
	
	variable outputType, newCameraType
	
	// the way to select the output differs depending on whether we want an Igor wave or a file
	strswitch (outputFormat)
		case "TIFF file":
			Open /D /T=".tif" RefNum
			if (strlen(S_FileName) == 0)	// the user canceled
				return -1
			endif
			outputFilePath = S_FileName
			outputType = IMAGE_OUTPUT_TYPE_TIFF
			newCameraType = CAMERA_TYPE_TIFF
			break
		case "Zip-Compressed TIFF File":
			Open /D /T=".tif" RefNum
			if (strlen(S_FileName) == 0)	// the user canceled
				return -1
			endif
			outputFilePath = S_FileName
			outputType = IMAGE_OUTPUT_TYPE_COMPR_TIFF
			newCameraType = CAMERA_TYPE_TIFF
			break
		case "PDE File":
			Open /D /T=".pde" RefNum
			if (strlen(S_FileName) == 0)	// the user canceled
				return -1
			endif
			outputFilePath = S_FileName
			outputType = IMAGE_OUTPUT_TYPE_PDE
			newCameraType = CAMERA_TYPE_PDE
			break
		case "Igor wave":
			NewDataFolder /O root:'PALM Movies'
			DFREF positionsFolder = root:'PALM Movies'
			
			outputFilePath = GetOutputWaveName("", positionsFolder, "Please enter the name of the output wave")
			if (strlen(outputFilePath) == 0)	// the user canceled the dialog
				return 0
			endif
			outputFilePath = "root:'PALM Movies':" + outputFilePath
			outputType = IMAGE_OUTPUT_TYPE_IGOR
			newCameraType = CAMERA_TYPE_IGOR_WAVE
			break
		default:
			Abort "Internal error: unknown output type in MakeCroppedStackFromMarquee"
			break
	endswitch
	
	// do the actual processing
	ProcessCCDImages /O /Y=(cameraType) /ROI={xStart, xEnd, yStart, yEnd} /M=(PROCESSCCDIMAGE_CROP) /OUT=(outputType) CCDFilePath, outputFilePath
	
	
	SetupImageViewer(outputFilePath, newCameraType)
End


Function RunParticleTracking(positions, maxStepSize, maxOffDuration)
	wave positions
	variable maxStepSize	// the maximum step a particle is allowed to take. This is also the threshold/criteria for automatically joining tracks
	variable maxOffDuration	// the maximum number of frame a particle can not be present without interrupting the track. Currently unused.
	
	Assert(WaveExists(positions))
	Assert(maxStepSize >= 0)
	Assert(maxOffDuration >= 0)
	
	// TODO: add data folder code
	
	variable nPositions = DimSize(positions, 0)
	variable maxStepSizeSquared = maxStepSize * maxStepSize
	variable startingFrame = positions[0][0]
	variable endingFrame = positions[nPositions - 1][0]
	variable xCol, yCol, zCol
	variable i, posInCurrentFrame, j, k, currentX, currentY
	variable nPointsWithinStepSize
	variable trackLastX, trackLastY
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(positions, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to RunParticleTracking() do not appear to contain any (x,y) information"
	endif
	
	Make /WAVE /N=0 root:Packages:PALM:W_ParticleTraces_X
	Make /WAVE /N=0 root:Packages:PALM:W_ParticleTraces_Y	// two waves containing references to waves with the actual tracks
	Make /N=0 /B/U root:Packages:PALM:W_TrackIsFinished
	
	wave /WAVE W_ParticleTraces_X = root:Packages:PALM:W_ParticleTraces_X
	wave /WAVE W_ParticleTraces_Y = root:Packages:PALM:W_ParticleTraces_Y
	wave W_TrackIsFinished = root:Packages:PALM:W_TrackIsFinished
	
	for (i = startingFrame; i <= endingFrame; i+=1)
		wave /WAVE M_Extracted = ExtractPositionsInFrame(positions, 0)
		wave positionsInFrame = M_Extracted[0]
		
		// run over the active tracks that are in progress
		// if for any of these tracks there are two or more points within maxStepSize, terminate those tracks and print some information about that
		for (k = 0; k < DimSize(W_ParticleTraces_X, 0); k+=1)
			wave currentTrackX = W_ParticleTraces_X[k]
			wave currentTrackY = W_ParticleTraces_Y[k]
			trackLastX = currentTrackX[DimSize(currentTrackX, 0) - 1]
			trackLastY = currentTrackY[DimSize(currentTrackY, 0) - 1]
			
			nPointsWithinStepSize = 0
			for (j = 0; j < DimSize(positionsInFrame, 0); j+=1)
				currentX = positionsInFrame[j][xCol]
				currentY = positionsInFrame[j][yCol]
				
				if (((currentX - trackLastX)^2 + (currentY - trackLastY)^2) <= maxStepSizeSquared)
					// the current fitted point is a possible candidate for the continuation of the current track
					nPointsWithinStepSize += 1
				endif
				
			endfor
			
			// if we found more than one point within maxStepSize then terminate the tracks
			W_TrackIsFinished[k] = 1
			
			// TODO: print some information to inform the user about why the track was terminated
		endfor
	endfor
	
End

Function SavePositionsToTextFile()
	
	// get the positions we want to save
	String posName
	Prompt posName, "Positions wave: ", popup, GetPossiblePositionsWaves()
	DoPrompt "Select the positions to be saved", posName
	if (V_flag == 1)
		return 0
	endif
	
	wave pos = GetPositionsWaveReference(posName)
	if (WaveExists(pos) != 1)
		Abort "Internal error: cannot find the requested positions"
	endif
	
	// get the output filepath
	variable refNum
	string outputFilePath
	Open /D /F="Text File (*.txt):.txt;All Files:.*;" refNum
	if (strlen(S_fileName) == 0)
		return 0
	endif
	outputFilePath = S_fileName
	
	// construct a header that will identify the filetype and columns
	string header, columnLabels
	variable localizationMethod = NumberByKey("LOCALIZATION METHOD", note(pos))
	if (NumType(localizationMethod) == 2)
		localizationMethod = LOCALIZATION_GAUSS_FITTING
	endif
	
	switch (localizationMethod)
		case LOCALIZATION_GAUSS_FITTING:
			header = "Localized positions using symmetric 2D Gauss fitting\n"
			columnLabels = "First frame\tIntegrated intensity\tFitted PSF standard deviation\tX position (pixel)\tY position (pixel)\tBackground\tIntensity deviation\tPSF width deviation\tX position deviation\tY position deviation\tBackground deviation\tNumber of frames where this emitter is present\n"
			break
		case LOCALIZATION_GAUSS_FITTING_FIX:
			header = "Localized positions using symmetric 2D Gauss fitting with fixed PSF width\n"
			columnLabels = "First frame\tIntegrated intensity\tX position (pixel)\tY position (pixel)\tBackground\tIntensity deviation\tX position deviation\tY position deviation\tBackground deviation\tNumber of frames where this emitter is present\n"
			break
		case LOCALIZATION_MULTIPLICATION:
			header = "Localized positions using iterative multiplication\n"
			columnLabels = "First frame\tUsed PSF standard deviation\tX position (pixel)\tY position (pixel)\tNumber of frames where this emitter is present\n"
			break
		case LOCALIZATION_CENTROID:
			header = "Localized positions using centroid calculation\n"
			columnLabels = "First frame\tX position (pixel)\tY position (pixel)\tNumber of frames where this emitter is present\n"
			break
		case LOCALIZATION_ZEISSPALM:
			header = "Positions localized with the Zeiss PALM system\n"
			columnLabels = "First frame\tIntegrated number of photons\tX position (pixel)\tY position (pixel)\tLocalization precision (pixel)\tNumber of frames where this emitter is present\n"
			break
		case LOCALIZATION_ELLIPSOIDAL2DGAUSS:
			header = "Localized positions using ellipsoidal 2D Gauss fitting\n"
			columnLabels = "First frame\tIntegrated intensity\tFitted PSF standard deviation along x (pixel)\tFitted PSF standard deviation along y (pixel)\tX position (pixel)\tY position (pixel)\tCorrelation between x and y\t"
			columnLabels += "Background\tIntensity deviation\tPSF width deviation along x (pixel)\tPSF width deviation along y (pixel)\tX position deviation (pixel)\tY position deviation (pixel)\tCorrelation deviation\tBackground deviation\tNumber of frames where this emitter is present\n"
			break
		case LOCALIZATION_MLEWG:
			header = "Localized positions using MLEwG localization\n"
			columnLabels = "First frame\tIntegrated intensity\tFitted PSF standard deviation\tX position (pixel)\tY position (pixel)\tBackground\tPosition deviation\t\tNumber of frames where this emitter is present\n"
			break
		default:
			Abort "Unknown localization method"
	EndSwitch
	
	// open the file for writing
	Open refNum as outputFilePath
	
	// write the metadata
	fprintf refnum, "%s", header
	fprintf refnum, "%s\n", "IGOR WAVENOTE FOLLOWS"
	fprintf refnum, "%s\n", note(pos)
	fprintf refnum, "%s\n", "DATA FOLLOWS"
	fprintf refnum, "%s", columnLabels
	
	Close refNum
	
	// write the actual data using the Igor save operation
	Save /A=2 /J /M="\n" /U={0,0,0,0} pos as outputFilePath
End

Function LoadPositionsFromTextFile()
	// load PALM positions from a text file
	// currently only the Zeiss output format is supported
	
	string inputFilePaths
	variable refNum
	
	// allow the user to select a series of text files
	Open /D/R/F="Text Files:.txt; All Files:.*" /M="Select one or more text files" /MULT=1 refNum
	inputFilePaths = S_fileName
	if (strlen(inputFilePaths) == 0)	// cancelled
		return 0
	endif
	
	// for each of the files, check if it is a Zeiss file or a file created by this software
	string currentFilePath, singleLine
	string ZeissPALMFilepaths = ""
	string thisSoftwareFiles = ""
	variable i
	
	for (i = 0; i < ItemsInList(inputFilePaths, "\r"); i+=1)
		currentFilePath = StringFromList(i, inputFilePaths, "\r")
		Open /Z=1 /R refNum as currentFilePath
		if (V_flag != 0)
			// an error occurred opening this file
			Printf "An error occurred opening the file at %s, skipping\r", currentFilePath
			continue
		endif
		
		// check if it is a valid Zeiss PALM file by looking for the unusual [mm] units specified
		FReadLine refNum, singleLine
		Close refNum
		if (StringMatch(singleLine, "*[mm]*") == 1)
			// probably a Zeiss PALM file
			ZeissPALMFilepaths += currentFilePath + "\r"
			continue
		endif
		
		// check if it is a file created by this software
		if (StringMatch(singleLine, "*Localized positions using*") == 1)
			// probably created by this software
			thisSoftwareFiles += currentFilePath + "\r"
			continue
		endif
		
		// unknown or unsupported file type, ignore
		Printf "The file at %s does not seem to be a file containing PALM positions or is not supported, skipping\r", currentFilePath
		i -= 1
		inputFilePaths = RemoveFromList(inputFilePaths, currentFilePath, "\r")
	endfor
	
	// load the Zeiss files
	if (ItemsInList(ZeissPALMFilepaths, "\r") > 0)
		LoadZeissPALMPositions(ZeissPALMFilepaths)
	endif
	
	// load files created with this software
	if (ItemsInList(thisSoftwareFiles, "\r") > 0)
		LoadXOPPositionsFiles(thisSoftwareFiles)
	endif
	
	ControlUpdate /W=CCD_Control PMSelectPositionsWave
End	
			
Function LoadZeissPALMPositions(ZeissPALMFilepaths)
	// load PALM positions from the text output format used by the Zeiss software
	// Zeiss works in units of nanometers, but since we work in pixels we need to convert
	
	// the zeiss positions have their own custom positions format with 6 columns
	// signaled by LOCALIZATION METHOD:LOCALIZATION_ZEISSPALM
	
	// the files to load should be specified as a list separated by carriage returns (\r)
	string ZeissPALMFilepaths
	
	Assert(strlen(ZeissPALMFilepaths) > 0)
	
	variable nDataFiles, pixelSize = 100, nPositions, i, j, k, nEntries, offset, nFrames
	variable xSize = 256, ySize = 256
	string currentMacintoshFilePath, waveOutputName, promptString
	string waveNote
	
	NewDataFolder /O root:'PALM Positions'
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	nDataFiles = ItemsInList(ZeissPALMFilepaths, "\r")
	
	for (i = 0; i < nDataFiles; i+=1)
		currentMacintoshFilePath = ParseFilePath(5, StringFromList(i, ZeissPALMFilepaths, "\r"), ":", 0, 0)
		waveOutputName = ParseFilePath(3, currentMacintoshFilePath, ":", 0, 0)
		
		sprintf promptString, "Loading positions from %s", currentMacintoshFilePath
		Prompt pixelSize, "Optical CCD pixel size (in nm)"
		Prompt xSize, "X-size (in pixels)"
		Prompt ySize, "Y-size (in pixels)"
		DoPrompt promptString, pixelSize, xSize, ySize
		if (V_flag == 1)	// cancel
			continue
		endif
		
		LoadWave /Q/A/G/O/B="N='_skip_';N=ZeissFirstFrame,T=96;N=ZeissNFrames,T=96;N=ZeissFramesMissing;N=ZeissXPosition,T=4; N=ZeissYPosition,T=4; N=ZeissPrecision,T=4;N=ZeissPhotons;T=96" currentMacintoshFilePath
		
		wave ZeissFirstFrame, ZeissNFrames, ZeissFramesMissing, ZeissXPosition, ZeissYPosition, ZeissPrecision, ZeissPhotons
		nEntries = DimSize(ZeissFirstFrame, 0)
		
		// The Zeiss output format does not sort the positions according to the ascending frame number in which a positions first appears
		// some routines here assume that the positions are ordered
		// do the sorting now
		Sort ZeissFirstFrame, ZeissFirstFrame, ZeissNFrames, ZeissFramesMissing, ZeissXPosition, ZeissYPosition, ZeissPrecision, ZeissPhotons
		
		nPositions = 0
		for (j = 0; j < nEntries; j += 1)
			if ((ZeissFramesMissing[j] != 0) || (ZeissPhotons[j] == 0))
							// the 'ZeissFramesMissing' column is unusual, so ignore it for now
			continue		// occasionally the Zeiss file contains strange data positions containing zero photons
			endif
			nPositions += 1
		endfor
		
		Make /D/O/N=(nPositions,6) root:'PALM Positions':$waveOutputName
		wave outputWave = root:'PALM Positions':$waveOutputName
		FastOP outputWave = 0
		
		offset = 0
		for (j = 0; j < nEntries; j+=1)
			if ((ZeissFramesMissing[j] != 0) || (ZeissPhotons[j] == 0))
				continue
			endif
			
			outputWave[offset][0] = ZeissFirstFrame[j]
			outputWave[offset][1] = ZeissPhotons[j]
			outputWave[offset][2] = ZeissXPosition[j] / pixelSize * 1e6	// positions are reported in mm by the Zeiss software
			outputWave[offset][3] = ZeissYPosition[j] / pixelSize * 1e6
			outputWave[offset][4] = ZeissPrecision[j] / pixelSize	// precision is reported in nm
			outputWave[offset][5] = ZeissNFrames[j]
			
			offset += 1
		endfor
		
		// append a note with as much additional information as possible
		waveNote = "LOCALIZATION METHOD:" + num2str(LOCALIZATION_ZEISSPALM) + ";"
		waveNote += "ORIGINAL FILE PATH:" + currentMacintoshFilePath + ";"
		waveNote += "X SIZE:" + num2str(xSize) + ";"
		waveNote += "Y SIZE:" + num2str(ySize) + ";"
		waveNote += "NUMBER OF IMAGES:" + num2str(nFrames) + ";"
		waveNote += "X PIXEL SIZE:" + num2str(pixelSize) + ";"
		waveNote += "Y PIXEL SIZE:" + num2str(pixelSize) + ";"
		
		Note /K outputWave	// remove a previous note (should not occur)
		Note /NOCR outputWave, waveNote
	endfor
		
		
	KillWaves /Z ZeissFirstFrame, ZeissNFrames, ZeissFramesMissing, ZeissXPosition, ZeissYPosition, ZeissPrecision, ZeissPhotons
End

Function LoadXOPPositionsFiles(PALMFilePaths)
	// load PALM positions from the text output format used by the Zeiss software
	// Zeiss works in units of nanometers, but since we work in pixels we need to convert
	
	// the zeiss positions have their own custom positions format with 6 columns
	// signaled by LOCALIZATION METHOD:LOCALIZATION_ZEISSPALM
	
	// the files to load should be specified as a list separated by carriage returns (\r)
	string PALMFilePaths
	
	Assert(strlen(PALMFilePaths) > 0)
	
	variable refNum, nDataFiles, pixelSize = 100, nPositions, i, j, k, offset, nFrames
	variable firstLineWithData
	variable localizationMethod
	string currentMacintoshFilePath, waveOutputName, promptString, singleLine, waveNote
	DFREF savDF = GetDataFolderDFR()
	
	NewDataFolder /O root:'PALM Positions'
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	nDataFiles = ItemsInList(PALMFilePaths, "\r")
	
	for (i = 0; i < nDataFiles; i+=1)
		currentMacintoshFilePath = ParseFilePath(5, StringFromList(i, PALMFilePaths, "\r"), ":", 0, 0)
		
		waveOutputName = CleanupName(ParseFilePath(3, currentMacintoshFilePath, ":", 0, 0), 0)
		if (strlen(waveOutputName) > 27)
			// loadwave easily complains of a name that's too long, so guard for that
			waveOutputName = waveOutputName[0, 26]
		endif
		
		waveNote = ""
		
		// get some information from the header
		Open /R refNum as currentMacintoshFilePath
		FReadLine refNum, singleLine
		
		// determine the type of positions from the first line
		if (StringMatch(singleLine, "*symmetric 2D Gauss fitting with fixed PSF width*") == 1)
			localizationMethod = LOCALIZATION_GAUSS_FITTING_FIX
			waveNote = "LOCALIZATION METHOD:" + num2str(localizationMethod) + ";"
		elseif (StringMatch(singleLine, "*symmetric 2D Gauss fitting*") == 1)
			localizationMethod = LOCALIZATION_GAUSS_FITTING
			waveNote = "LOCALIZATION METHOD:" + num2str(localizationMethod) + ";"
		elseif (StringMatch(singleLine, "*ellipsoidal 2D Gauss fitting*") == 1)
			localizationMethod = LOCALIZATION_ELLIPSOIDAL2DGAUSS
			waveNote = "LOCALIZATION METHOD:" + num2str(localizationMethod) + ";"
		elseif (StringMatch(singleLine, "*centroid calculation*") == 1)
			localizationMethod = LOCALIZATION_CENTROID
			waveNote = "LOCALIZATION METHOD:" + num2str(localizationMethod) + ";"
		elseif (StringMatch(singleLine, "*iterative multiplication*") == 1)
			localizationMethod = LOCALIZATION_MULTIPLICATION
			waveNote = "LOCALIZATION METHOD:" + num2str(localizationMethod) + ";"
		else
			Printf "The file at %s does not appear to be a valid positions file, skipping...\r", currentMacintoshFilePath
			Close refNum
			continue
		endif
		
		// parse the waveNote from the file, but stop when "DATA FOLLOWS" is encountered
		for (j = 1; ; j+=1)
			FReadLine refNum, singleLine
			if (strlen(singleLine) == 1)	// empty line
				continue
			endif
			if (stringmatch(singleLine, "DATA FOLLOWS*") == 1)
				firstLineWithData = j + 2
				break
			endif
			waveNote += singleLine + ";"
		endfor
		
		Close refNum
		
		// get rid of possible extra carriage returns produced by Igor
		waveNote = ReplaceString("\r", waveNote, "")
		waveNote += "POSITIONS FILE PATH:" + currentMacintoshFilePath + ";"
		
		SetDataFolder root:'PALM Positions'
		LoadWave /Q /A=$waveOutputName /G/ K=1 /M /L={firstLineWithData, firstLineWithData, 0, 0, 0} currentMacintoshFilePath
		SetDataFolder savDF
		
		if (ItemsInList(S_waveNames) != 1)
			Abort "An error occurred loading the data at " + currentMacintoshFilePath
		endif
		
		wave loadedWave = root:'PALM Positions':$StringFromList(0, S_waveNames)
		Note /K loadedWave, waveNote
	endfor
End

Function GeneratePositionsReport()
	wave pos
	
	// automatically generate a report that contains some statistics on the localized points of interest
	// first get the positions wave that we are interested in
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	
	string posName
	Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
	DoPrompt "Which positions?", posName
	if (V_flag == 1)
		return 0
	endif
	
	wave pos = GetPositionsWaveReference(posName)
	if (WaveExists(pos) == 0)
		Abort "The selected wave doesn't seem to exist!"
	endif
	
	variable nPos = DimSize(pos, 0)
	variable nFrames = pos[nPos - 1][0] - pos[0][0] + 1
	variable firstFrame = pos[0][0]
	variable i, offset, xDeviation, yDeviation, zDeviation, integralColumn, xWidthColumn, yWidthColumn
	
	if (nPos == 0)
		Abort "The specified wave does not contain any points!"
	endif
	
	// Extract the localization error
	GetColumnsForLocalizationError(pos, xDeviation, yDeviation, zDeviation)
	if ((xDeviation != -1) && (yDeviation != -1))	// does this type of positions provide information on the uncertainty?
		ImageTransform /G=(xDeviation) getCol, pos
		wave W_ExtractedCol
		Duplicate /D/FREE /O W_ExtractedCol, W_LocalizationError
		
		ImageTransform /G=(yDeviation) getCol, pos
		wave W_ExtractedCol
		W_LocalizationError = (W_LocalizationError + W_ExtractedCol) / 2
	else
		Make /D/FREE /N=(nPos) W_LocalizationError
		FastOP W_LocalizationError = 0
	endif
	
	
	// extract the integral
	 GetColumnForIntegratedIntensity(pos, integralColumn)
	 if (integralColumn != -1)
		ImageTransform /G=1 getCol, pos
		wave W_ExtractedCol
		Duplicate /D/FREE /O W_ExtractedCol, W_Intensities
	else
		Make /D/FREE /N=(nPos) W_Intensities
		FastOP W_Intensities = 0
	endif
	
	// Extract the widths
	GetColumnForFittedWidth(pos, xWidthColumn, yWidthColumn)
	if ((xWidthColumn != -1) && (yWidthColumn != -1))
		ImageTransform /G=(xWidthColumn) getCol, pos
		wave W_ExtractedCol
		Duplicate /FREE /O W_ExtractedCol, W_EmitterWidths
		
		ImageTransform /G=(yWidthColumn) getCol, pos
		wave W_ExtractedCol
		W_EmitterWidths = (W_EmitterWidths + W_ExtractedCol) / 2
	else
		Make /FREE /N=(nPos) W_EmitterWidths
		FastOP W_EmitterWidths = 0
	endif
	
	// show the number of frames an emitter is present for
	variable nFramesPresentCol
	GetColumnForNFramesPresent(pos, nFramesPresentCol)
	if (nFramesPresentCol != -1)
		ImageTransform /G=(nFramesPresentCol) getCol, pos
		wave W_ExtractedCol
		Duplicate /O/FREE W_ExtractedCol, W_nFramesPresent
		
		Make /D/O/N=1 root:Packages:PALM:W_NFramesPresentHist
		wave nFramesPresentHist = root:Packages:PALM:W_NFramesPresentHist
		
		WaveStats /Q/M=1 W_nFramesPresent
		if (V_max == 1)	// emitters have not been consolidated
			nFramesPresentHist[0] = DimSize(pos, 0)
		else
			Histogram /B={0.5, 1, V_max} W_nFramesPresent, nFramesPresentHist
		endif
		SetScale /P x, 1, 1, nFramesPresentHist
	else
		Make /FREE /N=(nPos) W_nFramesPresent
		FastOP W_nFramesPresent = 1
	endif
	
	// make some histograms
	Make /O/N=1 root:Packages:PALM:W_LocalizationErrorHist
	Make /O/N=1 root:Packages:PALM:W_IntensitiesHist
	Make /O/N=1 root:Packages:PALM:W_EmitterWidthsHist
	Histogram /B=4 W_LocalizationError, root:Packages:PALM:W_LocalizationErrorHist
	Histogram /B=4 W_Intensities, root:Packages:PALM:W_IntensitiesHist
	Histogram /B=4 W_EmitterWidths, root:Packages:PALM:W_EmitterWidthsHist
	wave localizationHist =  root:Packages:PALM:W_LocalizationErrorHist
	wave intensitiesHist = root:Packages:PALM:W_IntensitiesHist
	wave W_EmitterWidthsHist = root:Packages:PALM:W_EmitterWidthsHist
	
	// show the number of points localized per frame
	Make /O/N=(nFrames) root:Packages:PALM:W_nPosPerFrame
	wave W_nPosPerFrame = root:Packages:PALM:W_nPosPerFrame
	W_nPosPerFrame = 0
	for (i = 0; i < nPos; i+=1)
		W_nPosPerFrame[(pos[i][0] - firstFrame)] += 1
	endfor
	
	// output the wave info string in a listbox
	string waveNote = note(pos)
	variable nPropertyKeys = ItemsInlist(waveNote)
	Make /O/T/N=(nPropertyKeys + 1, 2) root:Packages:PALM:M_LBPositionsInfo
	wave /T M_LBPositionsInfo = root:Packages:PALM:M_LBPositionsInfo
	M_LBPositionsInfo[0][0] = "Number of positions"
	M_LBPositionsInfo[0][1] = num2str(DimSize(pos, 0))
	for (i = 0; i < nPropertyKeys; i+=1)
		M_LBPositionsInfo[i+1][0] = StringFromList(0, StringFromList(i, waveNote, ";"), ":")
		M_LBPositionsInfo[i+1][1] = StringFromList(1, StringFromList(i, waveNote, ";"), ":")
	endfor
	
	DoWindow /K PositionsReport
	if (V_flag == 0)
		NewPanel /K=1 /W=(150,50,797,722)/N=PositionsReport
		DefineGuide LeftColLeft={FL,0.02,FR},RightColRight={FL,0.98,FR},LeftColRight={FL,0.49,FR}
		DefineGuide RightColLeft={FL,0.51,FR},TopRowTop={FT,0.02,FB},TopRowBottom={FT,0.24,FB}
		DefineGuide MiddleRowTop={FT,0.26,FB},MiddleRowBottom={FT,0.49,FB}
		DefineGuide BottomRowTop={FT,0.51,FB}, BottomRowBottom={FT,0.74,FB}
		Display /W=(16,17,309,187)/HOST=PositionsReport /FG=(LeftColLeft, TopRowTop, LeftColRight, TopRowBottom) localizationHist
		Label bottom "Reported position deviation (pixels)"
		Label left "Occurrence"
		Display /W=(16,17,309,187)/HOST=PositionsReport /FG=(RightColLeft, TopRowTop, RightColRight, TopRowBottom) intensitiesHist
		Label bottom "Emitter intensity"
		Label left "Occurrence"
		Display /W=(16,17,309,187)/HOST=PositionsReport /FG=(LeftColLeft, MiddleRowTop, LeftColRight, MiddleRowBottom) W_EmitterWidthsHist
		Label bottom "Fitted emitter width"
		Label left "Occurrence"
		Display /W=(16,17,309,187)/HOST=PositionsReport /FG=(RightColLeft, MiddleRowTop, RightColRight, MiddleRowBottom) W_nPosPerFrame
		Label bottom "Frame"
		Label left "Positions fitted per frame"
		Display /W=(16,17,309,187)/HOST=PositionsReport /FG=(LeftColLeft, BottomRowTop, LeftColRight, BottomRowBottom) nFramesPresentHist
		Label bottom "Number of frames a single emitter persists"
		Label left "Occurrence"
		Edit /N=WaveMetaData /W=(16,17,309,187)/HOST=PositionsReport /FG=(RightColLeft, BottomRowTop, RightColRight, BottomRowBottom) root:Packages:PALM:M_LBPositionsInfo
		ModifyTable /W=PositionsReport#WaveMetaData format(Point)=1, alignment=0,width=150
		ModifyTable /W=PositionsReport#WaveMetaData showParts=0xF0
	endif
	
	DoWindow /T PositionsReport, "Positions report for \"" + NameOfWave(pos) + "\""
	
	KillWaves W_ExtractedCol
End

Function ComparePositions(pos1, pos2, [tol])
	wave pos1, pos2
	variable tol
	
	// Given two positions waves, pos1 and pos2, try to ascertain if they are equal (within calculation tolerances)
	
	if (ParamIsDefault(tol))
		tol = 1e-3
	endif
	
	string pos1Note = Note(pos1)
	string pos2Note = Note(pos2)
	
	if (DimSize(pos1, 0) != DimSize(pos2, 0))
		Abort "The positions contain a different number of emitters"
	endif
	
	variable posType1 = NumberByKey("LOCALIZATION METHOD", pos1Note)
	variable posType2 = NumberByKey("LOCALIZATION METHOD", pos2Note)
	if (posType1 != posType2)
		Abort "The positions were localized using different localization methods"
	endif
	
	variable nPos = DimSize(pos1, 0)
	
	// Due to the multi-threaded nature of the fitting,
	// the positions in a frame can be reported in different ordering
	// so get around that by sorting
	variable xCol, yCol, zCol
	variable frameCol = 0
	GetColumnsForEmitterPositions(pos1, xCol, yCol, zCol)
	
	Make /D/O/D/N=(nPos)/FREE pos1FrameNumbers, pos2FrameNumbers, pos1XValues, pos1YValues, pos2XValues, pos2YValues
	pos1FrameNumbers = pos1[p][frameCol]
	pos2FrameNumbers = pos2[p][frameCol]
	pos1XValues = pos1[p][xCol]
	pos2XValues = pos2[p][xCol]
	pos1YValues = pos1[p][yCol]
	pos2YValues = pos2[p][yCol]
	
	// sort both positions waves
	Make /D/O/N=(DimSize(pos1, 0), DimSize(pos1, 1)) /D/FREE pos1Sorted, pos2Sorted
	Make /O/I/U /FREE/N=(nPos) SortIndex
	SortIndex = p
	MakeIndex {pos1FrameNumbers, pos1XValues, pos1YValues} SortIndex
	pos1Sorted = pos1[SortIndex[p]][q]
	SortIndex = p
	MakeIndex {pos2FrameNumbers, pos2XValues, pos2YValues} SortIndex
	pos2Sorted = pos2[SortIndex[p]][q]
	
	Make /D/O/N=(nPos, DimSize(pos1, 1)) /D M_Difference
	M_Difference = pos1Sorted - pos2Sorted
	
	WaveStats /Q/M=1 M_Difference
	if ((Abs(V_max) > tol) || (Abs(V_min) > tol))
		DoAlert 0, "Positions are NOT equal"
	else
		DoAlert 0, "Positions are equal"
	endif
End

Function ExtractPositionsWithinLimits()
	// allow the user to specify minimum and maximum limits for the amplitude, error deviation, and width
	string posName, units
	variable minAmplitude = -1, maxAmplitude = -1
	variable minWidth = -1, maxWidth = -1
	variable minReportedDeviation = -1, maxReportedDeviation = -1
	variable xPixelSize, yPixelSize
	
	Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
	Prompt minAmplitude, "Min amplitude"
	Prompt maxAmplitude, "Max amplitude"
	Prompt minWidth, "Min width"
	Prompt maxWidth, "Max width"
	Prompt minReportedDeviation, "Min reported deviation"
	Prompt maxReportedDeviation, "Max reported deviation"
	Prompt units, "Widths and deviations are in:", popup, "pixels;micrometer;"
	DoPrompt "Enter criteria (-1 to ignore)", minAmplitude, maxAmplitude, minWidth, maxWidth, minReportedDeviation, maxReportedDeviation, units, posName
	if (V_flag == 1)	// cancel
		return 0
	endif
	
	// if maximum limits are not requested, provide some unrealistically large values
	if (maxAmplitude < 0)
		maxAmplitude = 1e200
	endif
	if (maxWidth < 0)
		maxWidth = 1e200
	endif
	if (maxReportedDeviation < 0)
		maxReportedDeviation = 1e200
	endif
	
	// safety checks
	if ((minAmplitude > maxAmplitude) || (minWidth > maxWidth) || (minReportedDeviation > maxReportedDeviation))
		Abort "One or more minimum values are larger than the maximum values"
	endif
	
	wave pos = GetPositionsWaveReference(posName)
	if (WaveExists(pos) == 0)
		Abort "The selected wave doesn't seem to exist!"
	endif
	
	// if the user wants coordinates in pixels, check if the positions wave specifies them
	if (StringMatch(units, "micrometer") == 1)
		xPixelSize = NumberByKey("X PIXEL SIZE", note(pos))
		yPixelSize = NumberByKey("Y PIXEL SIZE", note(pos))
		// did the wavenote contain the pixel sizes?
		if ((NumType(xPixelSize) == 2) || (NumType(yPixelSize) == 2))	// NaN
			Abort "The positions do not contain any information on the pixel size (wavenote: \"X/Y PIXEL SIZE\")"
		endif
	
		// convert the provided limits to pixels
		maxWidth /= (xPixelSize / 1e3)	// convert from nanometer to m
		minWidth /= (xPixelSize / 1e3)
		maxReportedDeviation /= (xPixelSize / 1e3)
		minReportedDeviation /= (xPixelSize / 1e3)
	endif
	
	// get an output wave name
	String outputName = GetOutputWaveName("", root:'PALM Positions', "Please enter a name for the output positions")
	if (strlen(outputName) == 0)
		return 0
	endif
	
	// get the relevant columns
	variable integralColumn, xCol, yCol, zCol, xDeviationCol, yDeviationCol, zDeviationCol, deviation, xWidthColumn, yWidthColumn, width
	
	GetColumnForIntegratedIntensity(pos, integralColumn)
	GetColumnsForEmitterPositions(pos, xCol, yCol, zCol)
	GetColumnsForLocalizationError(pos, xDeviationCol, yDeviationCol, zDeviationCol)
	GetColumnForFittedWidth(pos, xWidthColumn, yWidthColumn)
	
	variable nPointsWithinLimits = 0
	variable nPos = DimSize(pos, 0), i
	for (i = 0; i < nPos; i+=1)
		if (xDeviationCol == -1)	// not supported
			deviation = (maxReportedDeviation + minReportedDeviation) / 2
		else
			deviation = (pos[i][xDeviationCol] + pos[i][yDeviationCol]) / 2
		endif
			
		if (xWidthColumn == -1)	// not supported
			width = (minWidth + maxWidth) / 2
		else
			width = (pos[i][xWidthColumn] + pos[i][yWidthColumn]) / 2
		endif
		
		if ((pos[i][integralColumn] >= minAmplitude) && (pos[i][integralColumn] <= maxAmplitude) && (width >= minWidth) && (width <= maxWidth)  && (deviation >= minReportedDeviation) && (deviation <= maxReportedDeviation))
			nPointsWithinLimits += 1
		endif
	endfor
	
	if (nPointsWithinLimits == 0)
		Abort "The selected criteria exclude all positions!"
	endif
	
	Make /D/O/N=(nPointsWithinLimits, DimSize(pos, 1)) root:'PALM Positions':$outputName
	wave outputWave = root:'PALM Positions':$outputName
	
	variable offset = 0
	
	for (i = 0; i < nPos; i+=1)
		if (xDeviationCol == -1)	// not supported
			deviation = (maxReportedDeviation + minReportedDeviation) / 2
		else
			deviation = (pos[i][xDeviationCol] + pos[i][yDeviationCol]) / 2
		endif
			
		if (xWidthColumn == -1)	// not supported
			width = (minWidth + maxWidth) / 2
		else
			width = (pos[i][xWidthColumn] + pos[i][yWidthColumn]) / 2
		endif
		
		if ((pos[i][integralColumn] >= minAmplitude) && (pos[i][integralColumn] <= maxAmplitude) && (width >= minWidth) && (width <= maxWidth)  && (deviation >= minReportedDeviation) && (deviation <= maxReportedDeviation))
			outputWave[offset][] = pos[i][q]
			offset += 1
		endif
	endfor
	
	string waveNote
	waveNote = Note(pos)
	if (minAmplitude > 0)
		waveNote += "FILTERED MIN AMPLITUDE:" + num2str(minAmplitude) + ";"
	endif
	if ((maxAmplitude >= 0) && (maxAmplitude < 1e200))
		waveNote += "FILTERED MAX AMPLITUDE:" + num2str(maxAmplitude) + ";"
	endif
	if (minWidth > 0)
		waveNote += "FILTERED MIN WIDTH:" + num2str(minWidth) + ";"
	endif
	if ((maxWidth >= 0) && (maxWidth < 1e200))
		waveNote += "FILTERED MAX WIDTH:" + num2str(maxWidth) + ";"
	endif
	if (minReportedDeviation > 0)
		waveNote += "FILTERED MIN DEVIATION:" + num2str(minReportedDeviation) + ";"
	endif
	if ((maxReportedDeviation >= 0) && (maxReportedDeviation < 1e200))
		waveNote += "FILTERED MAX DEVIATION:" + num2str(maxReportedDeviation) + ";"
	endif
	
	Note /K outputWave
	Note outputWave, waveNote
End

Function ConsolidateSameEmitters_menu()
	// present a graphical interface to ConsolidateSameEmitters
	string posName
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	NewDataFolder /O root:'PALM Positions'
	
	Variable /G root:Packages:PALM:V_consolidateMaxDistance
	Variable /G root:Packages:PALM:V_consolidateMaxBlinking
	
	NVAR maxPositionDifference = root:Packages:PALM:V_consolidateMaxDistance
	NVAR maxBlinking = root:Packages:PALM:V_consolidateMaxBlinking
	
	if (maxPositionDifference == 0)
		maxPositionDifference = 0.5
	endif
	
	variable maxPositionDifferenceLocal = maxPositionDifference
	variable maxBlinkingLocal = maxBlinking
	
	Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
	Prompt maxPositionDifferenceLocal, "Maximum difference in position (in pixels)"
	Prompt maxBlinkingLocal, "Maximum allowed gap due to blinking (in frames)"
	
	DoPrompt "Enter calculation parameters", posName, maxPositionDifferenceLocal, maxBlinkingLocal
	if (V_flag == 1)	// cancel
		return 0
	endif
	
	maxBlinking = maxBlinkingLocal
	maxPositionDifference = maxPositionDifferenceLocal
	
	wave pos = GetPositionsWaveReference(posName)
	if (WaveExists(pos) == 0)
		Abort "The selected wave doesn't seem to exist!"
	endif
	
	// get a wave name for the output positions
	string suggestedOutputName = CleanupName(NameOfWave(pos) + "_cons", 1)
	string outputName = GetOutputWaveName(suggestedOutputName, root:'PALM Positions', "Enter a name for the output wave")
	if (strlen(outputName) == 0)
		return 0
	endif
	
	// do the actual calculation
	ConsolidateSameEmitters(pos, "root:'PALM Positions':" + outputName, maxPositionDifference, maxBlinking)
	
End

Function ConsolidateSameEmitters(positions, posOutputName, maxPositionDifference, maxBlinking, [generateTracks])
	wave positions
	string posOutputName
	variable maxPositionDifference, maxBlinking, generateTracks
	
	if (ParamIsDefault(generateTracks))
		generateTracks = 0
	endif
	
	Assert(WaveExists(positions))
	Assert(maxPositionDifference >= 0)
	Assert(maxBlinking >= 0)
	
	// Go over all the positions in the positions wave and look for emitters that show up in
	// different frames but might actually be the same spot. The two emitters have to be close
	// enough to be counted as the same molecule (maxPositionDifference). To allow for blinking
	// a molecule can disappear from the images for a number of frames up to maxBlinking
	// and still be accepted as the same emitter
	
	// the algorithm works as follows: every new position is temporarily added to a 'waiting list'
	// that contains the index into positions where it is listed, its x and y position, and a counter
	// that is initially set to maxBlinking
	
	// when moving on to a new frame every position on the waiting list is checked to see if a point
	// within a distance of maxPositionDifference is present in the new frame. If so then the number of
	// frames associated with the position on the waiting list is incremented by one, the counter is reset
	// to maxBlinking, and the matching point is deleted from the positions wave.
	// If not then the counter is decremented. If the counter becomes negative then the point is deleted .
	
	// the format of the waiting list is index of first appearance / x position / y position / counter
	
	// in its original form the algorithm actively deleted points that were the same as a previous emitter
	// during the processing. But this introduces the overhead of copying the wave elements each time.
	// therefore in its current form the algorithm does not delete points directly, but works on a duplicate
	// of the positions where the entries to be deleted are marked by a negative frame number
	
	variable nFramesPresentColumn, xCol, yCol, zCol, intensityColumn, xUncertaintyCol, yUncertaintyCol, zUncertaintyCol
	GetColumnForNFramesPresent(positions, nFramesPresentColumn)	// handle different types of positions
	GetColumnsForEmitterPositions(positions, xCol, yCol, zCol)
	GetColumnForIntegratedIntensity(positions, intensityColumn)
	GetColumnsForLocalizationError(positions, xUncertaintyCol, yUncertaintyCol, zUncertaintyCol)
	
	// check if these positions have not been consolidated before
	// if so then the results are unpredictable
	ImageTransform /G=(nFramesPresentColumn) sumCol, positions
	if (V_value != DimSize(positions, 0))
		DoAlert 1, "These positions look as if they have already been combined, or were created using an old version of this program. In the latter case, would you like me to correct this and continue?"
		if (V_flag == 1)	// yes
			positions[][nFramesPresentColumn] = 1
		else
			return 0
		endif
	endif
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	NewDataFolder /O root:Packages:PALM:ConsolidateTracks
	
	// if the user wants to make tracks then be sure to clean out any previous tracks
	if (generateTracks != 0)
		DFREF savDF = GetDataFolderDFR()
		SetDataFolder root:Packages:PALM:ConsolidateTracks
		KillWaves /Z/A
		SetDataFolder savDF
	endif
	
	// Make a copy of the positions
	Duplicate /O positions, root:Packages:PALM:M_TempConsolidatedPositions
	wave M_TempConsolidatedPositions = root:Packages:PALM:M_TempConsolidatedPositions
	
	Make /D/N=(0, 4) /O root:Packages:PALM:M_WaitingList
	wave M_WaitingList = root:Packages:PALM:M_WaitingList
	// M_WaitingList contains the emitters that might still show up in this frame
	// which is all emitters that have been on within maxBlinking frames from this one
	// the format of this wave is
	// [0]: index of the entry in positions where this emitter was first reported
	// [1]: the x position of the emitter as first recorded
	// [2]: the y position of the emitter as first recorded
	// [3]: the number of remaining frames this emitter can be off while still being counted as active
	
	variable n, err, i
	variable startingFrame = positions[0][0]	// warning: magic numbers
	variable endingFrame = positions[DimSize(positions,0) - 1][0]
	
	variable startingIndexOfCurrentFrame = 0	// the index from which the current frame starts in M_TempConsolidatedPositions
	variable nearestX, nearestY, index, distance, offset, totalNEmittersInThisFrame, nPointsToBeDeleted = 0
	variable currentXPos, currentYPos, currentNFrames, currentXUncertainty, currentYUncertainty
	string trackWaveName
	
	for (n = startingFrame; n <= endingFrame; n+=1)
		// get all emitters localized in this frame
		wave /WAVE M_Extracted = ExtractPositionsInFrame(positions, n)
		wave M_emittersInCurrentFrame = M_Extracted[0]
		wave W_emittersInCurrentFrameIndices = M_Extracted[1]
		
		totalNEmittersInThisFrame = DimSize(M_emittersInCurrentFrame, 0)
		
		// loop over all the emitters that are present in the waiting list
		for (i = 0; i < DimSize(M_WaitingList, 0); i+=1)
			// get the point in the current frame that is the closest to one in the waiting list
			err = returnPointNearestFitPositions(M_emittersInCurrentFrame, M_WaitingList[i][1], M_WaitingList[i][2], nearestX, nearestY, index)
			if (err != 0)
				// some error looking for the nearest point
				// there are probably no more emitters in M_emittersInCurrentFrame
				// just subtract 1 from the counter value for every point remaining
				// and move on
				M_WaitingList[i][3] = M_WaitingList[i][3] - 1
				continue
			endif
			distance = sqrt((nearestX - M_WaitingList[i][1])^2 + (nearestY - M_WaitingList[i][2])^2)
			
			if (distance > maxPositionDifference)
				// the point on the waiting list is not present in the current frame
				// decrement its counter
				M_WaitingList[i][3] = M_WaitingList[i][3] - 1
			
			else // the point on the waiting list is also present in this frame
				// recalculate the emitter position as a weighted average
				currentXPos = M_TempConsolidatedPositions[M_WaitingList[i][0]][xCol]
				currentYPos = M_TempConsolidatedPositions[M_WaitingList[i][0]][yCol]
				currentNFrames = M_TempConsolidatedPositions[M_WaitingList[i][0]][nFramesPresentColumn]
				M_TempConsolidatedPositions[M_WaitingList[i][0]][xCol] = (currentXPos * currentNFrames + M_emittersInCurrentFrame[index][xCol]) / (currentNFrames + 1)
				M_TempConsolidatedPositions[M_WaitingList[i][0]][yCol] = (currentYPos * currentNFrames + M_emittersInCurrentFrame[index][yCol]) / (currentNFrames + 1)
				
				// add the integrated intensity
				M_TempConsolidatedPositions[M_WaitingList[i][0]][intensityColumn] = M_TempConsolidatedPositions[M_WaitingList[i][0]][intensityColumn] + M_emittersInCurrentFrame[index][intensityColumn]
				
				// adjust the x and y position uncertainty
				// we assume that the uncertainty is approximated by the average of the uncertainties divided by the the square root
				// of the number of frames where the emitter is present
				// distinguish the cases where there is no uncertainty or only 1 uncertainty value
				if ((xUncertaintyCol != -1) && (xUncertaintyCol != yUncertaintyCol))
					// the fitted positions contain separate uncertainties for the x and y coordinate
					currentXUncertainty = M_TempConsolidatedPositions[M_WaitingList[i][0]][xUncertaintyCol]
					currentYUncertainty = M_TempConsolidatedPositions[M_WaitingList[i][0]][yUncertaintyCol]
					
					// correct for the division by the square root of the number of emitters
					currentXUncertainty *= sqrt(currentNFrames)
					currentYUncertainty *= sqrt(currentNFrames)
					// and make a new average
					currentXUncertainty = (currentXUncertainty * currentNFrames + M_emittersInCurrentFrame[index][xUncertaintyCol]) / (currentNFrames + 1)
					currentYUncertainty = (currentYUncertainty * currentNFrames + M_emittersInCurrentFrame[index][yUncertaintyCol]) / (currentNFrames + 1)
					// make sure that the uncertainty goes down
					currentXUncertainty /= sqrt(currentNFrames + 1)
					currentYUncertainty /= sqrt(currentNFrames + 1)
					
					M_TempConsolidatedPositions[M_WaitingList[i][0]][xUncertaintyCol] = currentXUncertainty
					M_TempConsolidatedPositions[M_WaitingList[i][0]][yUncertaintyCol] = currentYUncertainty
					
				elseif ((xUncertaintyCol != -1) && (xUncertaintyCol == yUncertaintyCol))
					// the positions contain only one uncertainty value
					currentXUncertainty = M_TempConsolidatedPositions[M_WaitingList[i][0]][xUncertaintyCol]
					currentXUncertainty *= sqrt(currentNFrames)
					currentXUncertainty = (currentXUncertainty * currentNFrames + M_emittersInCurrentFrame[index][xUncertaintyCol]) / (currentNFrames + 1)
					currentXUncertainty /= sqrt(currentNFrames + 1)
					
					M_TempConsolidatedPositions[M_WaitingList[i][0]][xUncertaintyCol] = currentXUncertainty
				endif
				
				// increment the number of frames that emitter is visible for
				M_TempConsolidatedPositions[M_WaitingList[i][0]][nFramesPresentColumn] = M_TempConsolidatedPositions[M_WaitingList[i][0]][nFramesPresentColumn] + 1
				
				// and reset its counter to maxBlinking
				M_WaitingList[i][3] = maxBlinking
				
				// mark the matching position for deletion
				M_TempConsolidatedPositions[W_emittersInCurrentFrameIndices[index]][0] = -1
				nPointsToBeDeleted += 1
				
				// if a track is requested then set it up if required, and add this emitter
				if (generateTracks != 0)
					sprintf trackWaveName, "Track_%d",   M_WaitingList[i][0]
					wave trackWave = root:Packages:PALM:ConsolidateTracks:$trackWaveName
					if (WaveExists(trackWave) != 1)
						// the track wave doesn't exist yet
						// set it up and add the initial emitter
						Make /O/N=(1, 2) /D root:Packages:PALM:ConsolidateTracks:$trackWaveName
						wave trackWave = root:Packages:PALM:ConsolidateTracks:$trackWaveName
						trackWave[0][0] = positions[M_WaitingList[i][0]][xCol]
						trackWave[0][1] = positions[M_WaitingList[i][0]][yCol]
					endif
					// add a new entry to the track wave
					Redimension /N=(DimSize(trackWave, 0) + 1, DimSize(trackWave, 1)) trackWave
					trackWave[DimSize(trackWave, 0) - 1][0] = M_emittersInCurrentFrame[index][xCol]
					trackWave[DimSize(trackWave, 0) - 1][1] = M_emittersInCurrentFrame[index][yCol]
				endif
				
				// delete the matching point in the frame from the extracted positions list
				DeletePoints index, 1, M_emittersInCurrentFrame
				DeletePoints index, 1, W_emittersInCurrentFrameIndices
			endif
		endfor
		
		// cleanup: remove any emitter from the waiting list that has a negative counter value
		for (i = 0; i < DimSize(M_WaitingList, 0); i+=1)
			if (M_WaitingList[i][3] < 0)
				DeletePoints i, 1, M_WaitingList
				i -= 1
			endif
		endfor
		
		// any emitters that remain in the current frame are new, add them to the waiting list
		offset = DimSize(M_WaitingList, 0)
		Redimension /N=(DimSize(M_WaitingList, 0) + DimSize(M_emittersInCurrentFrame, 0), 4) M_WaitingList
		for (i = 0; i < Dimsize(M_emittersInCurrentFrame, 0); i+=1)
			M_WaitingList[i + offset][0] = W_emittersInCurrentFrameIndices[i]
			M_WaitingList[i + offset][1] = M_emittersInCurrentFrame[i][xCol]
			M_WaitingList[i + offset][2] = M_emittersInCurrentFrame[i][yCol]
			M_WaitingList[i + offset][3] = maxBlinking
		endfor
		
		startingIndexOfCurrentFrame += totalNEmittersInThisFrame
	endfor
	
	// make a final copy of the positions wave
	Make /D/O/N=(DimSize(positions, 0) - nPointsToBeDeleted, DimSize(positions, 1)) $posOutputName
	wave consolidatedPositions = $posOutputName
	
	offset = 0
	for (i = 0; i < DimSize(positions, 0); i+=1)
		if (M_TempConsolidatedPositions[i][0] >= 0)
			consolidatedPositions[offset][] = M_TempConsolidatedPositions[i][q]
			offset += 1
		endif
	endfor
	
	// add info on the calculation params to the wave note
	string waveNote = Note(positions) + "CONSOLIDATE MAX SHIFT:" + num2str(maxPositionDifference) + ";" + "CONSOLIDATE MAX BLINKING:" + num2str(maxBlinking) + ";"
	Note /K consolidatedPositions, waveNote
	
	KillWaves /Z M_TempConsolidatedPositions, M_emittersInCurrentFrame, M_WaitingList, W_emittersInCurrentFrameIndices
	
	// perform a quality check on the algorithm
	// the sum of the column containing the number of frames
	// an emitter persists must be equal to the number of entries
	// in the original positions wave
	ImageTransform /G=(nFramesPresentColumn) sumCol, positions
	if (DimSize(positions, 0) != V_Value)
		Abort "The calculation appears to have gone wrong, the results should not be trusted"
	endif
	
End

Function RepeatedLocalizationPrecision()
	// try to get an idea for the localization precision by looking at localizations in subsequent
	// frames that can be thought to correspond to the same emitter
	// the functions assume that the positions have been consolidated before and with the generateTracks argument
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:PALM
	NewDataFolder /O root:Packages:PALM:ConsolidateTracks
	DFREF trackFolder = root:Packages:PALM:ConsolidateTracks
	DFREF savDF = GetDataFolderDFR()
	
	variable /G root:Packages:PALM:V_RepPrecMinTrackLength
	variable /G root:Packages:PALM:V_RepPrecMaxTrackLength
	
	NVAR minTrackLength = root:Packages:PALM:V_RepPrecMinTrackLength
	NVAR maxTrackLength = root:Packages:PALM:V_RepPrecMaxTrackLength
	
	variable localMinTrackLength = minTrackLength, localMaxTrackLength = maxTrackLength
	variable i, xStdDev, yStdDev, offset
	
	Prompt localMinTrackLength, "Minimum track length (in frames):"
	Prompt localMaxTrackLength, "Maximum track length (in frames)"
	
	DoPrompt "Enter calculation parameters...", localMinTrackLength, localMaxTrackLength
	if (V_flag == 1)
		return 0
	endif
	
	minTrackLength = localminTrackLength
	maxTrackLength = localmaxTrackLength
	
	// see if there are any tracks available
	string waveListOptions = "MINROWS:" + num2str(minTrackLength) + ",MAXROWS:" + num2str(maxTrackLength)
	SetDataFolder trackFolder
	string trackWaveList = WaveList("*", ";", waveListOptions)
	SetDataFolder savDF
	
	variable nValidTracks = ItemsInList(trackWaveList)
	if (nValidTracks == 0)
		Abort "No valid tracks were found"
	endif
	
	// set up some output waves that will hold the localization uncertainties
	Make /D/O/N=(nValidTracks) W_StdDev
	
	// do the calculation
	offset = 0
	for (i = 0; i < nValidTracks; i+=1)
		wave currentTrack = trackFolder:$(StringFromList(i, trackWaveList))
		
		// calculate the standard deviation of the position
		Make /D/O/N=(DimSize(currentTrack, 0))/FREE/D W_OneCoordinates
		W_OneCoordinates = currentTrack[p][0]
		WaveStats /Q /M=2 W_OneCoordinates
		xStdDev = V_sdev
		W_OneCoordinates = currentTrack[p][1]
		WaveStats /Q /M=2 W_OneCoordinates
		yStdDev = V_sdev
		W_StdDev[offset] = sqrt(xStdDev^2 + yStdDev^2)
		
		offset += 1
	endfor
End

Function /WAVE GetNSampledPositions(positions, nSamples)
	wave positions
	variable nSamples
	
	Assert(WaveExists(positions))
	Assert(nSamples > 0)
	
	variable nPositions = DimSize(positions, 0)
	
	if (nPositions <= nSamples)
		Abort "The number of samples requested is larger than or equal to the total number of positions"
	endif
	
	DFREF savDF = GetDataFolderDFR()
	NewDataFolder /O root:Packages
	NewDataFolder /O/S root:Packages:PALM
	
	StatsSample /MC /N=(nSamples) positions
	
	SetDataFolder savDF
	
	wave sampledPositions = root:Packages:PALM:M_Sampled
	Note /K/NOCR sampledPositions, Note(positions)
	return sampledPositions
End

Function DoLClustering()
	string posName
	
	DFREF savDF = GetDataFolderDFR()
	NewDataFolder /O root:Packages
	NewDataFolder /O/S root:Packages:PALM
	
	Variable /G root:Packages:PALM:V_lClusteringRange
	Variable /G root:Packages:PALM:V_lClusteringNBins
	Variable /G root:Packages:PALM:V_lCluseringNPointsToSample
	NVAR calculationRange = root:Packages:PALM:V_lClusteringRange
	NVAR nBins = root:Packages:PALM:V_lClusteringNBins
	NVAR nPointsToSample = root:Packages:PALM:V_lCluseringNPointsToSample
	
	if (calculationRange == 0)
		calculationRange = 30
	endif
	if (nBins == 0)
		nBins = 100
	endif
	if (nPointsToSample == 0)
		nPointsToSample = 5000
	endif
	
	variable localCalculationRange = calculationRange
	variable localNBins = nBins
	variable localNPointsToSample = nPointsToSample
	
	Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
	Prompt localCalculationRange, "Calculation range (pixel):"
	Prompt localNBins, "Number of bins:"
	Prompt localNPointsToSample, "Number of points to randomly sample (-1 for all points):"
	
	DoPrompt "Calculation parameters", posName, localCalculationRange, localNBins, localNPointsToSample
	if (V_flag == 1)	// cancel
		return 0
	endif
	
	calculationRange = localCalculationRange
	nBins = localNBins
	nPointsToSample = localNPointsToSample
	
	wave pos = GetPositionsWaveReference(posName)
	if (WaveExists(pos) == 0)
		Abort "The selected wave doesn't seem to exist!"
	endif
	
	variable nPositions = DimSize(pos, 0)
	
	if ((localNPointsToSample < 0) || (localNPointsToSample > nPositions))
		localNPointsToSample = nPositions
	endif
	
	SetDataFolder root:Packages:PALM
	if (localNPointsToSample < nPositions)
		// calculate only some points
		wave sampledPositions = GetNSampledPositions(pos, localNPointsToSample)
		RipleyLFunctionClustering /RNGE={calculationRange, nBins} sampledPositions
		KillWaves /Z sampledPositions
	else
		// calculate over all points
		RipleyLFunctionClustering /RNGE={calculationRange, nBins} pos
	endif
	SetDataFolder savDF
	
	wave W_lFunction = root:Packages:PALM:W_lFunction
	Duplicate /O W_lFunction, root:Packages:PALM:W_LFunctionSubX
	wave W_LFunctionSubX = root:Packages:PALM:W_LFunctionSubX
	
	W_LFunctionSubX = W_lFunction - x
	
	DoWindow /F LFunction
	if (V_flag != 1)
		Display /K=1/N=LFunction W_LFunctionSubX
		Label /W=LFunction Left, "L Function - x"
		Label /W=LFunction Bottom, "Distance (pixel)"
	endif
End

Function SingleClusterStats_Marquee()
	
	variable xStart, xEnd, yStart, yEnd, err
	variable pixelSize
	string posName, units
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:'PALM Positions'
	
	// try to get the positions wave from the graph automatically
	wave pos = GetPositionsFromGraph()
	
	// if this failed then prompt the user
	if (WaveExists(pos) != 1)
		Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
		DoPrompt "Choose a positions wave", posName
		if (V_flag == 1)
			return 0
		endif
	
		wave pos = GetPositionsWaveReference(posName)	
		if (WaveExists(pos) == 0)
			Abort "The selected positions wave doesn't seem to exist!"
		endif
	endif
	
	// GetMarqueeCoordinates will return the coordinates in units of pixels
	err = GetMarqueeCoordinates(pos, xStart, yStart, xEnd, yEnd)
	if (err != 0)	// no Marquee
		return 0
	endif
	
	 if ((xEnd <= xStart) || (yEnd <= yStart) || (xStart < 0) || (yStart < 0))
	 	Abort "Invalid limits from the marquee"
	 endif
	
	variable nPos = DimSize(pos, 0), xCol, yCol, zCol
	variable nPositionsWithinLimits = 0, i, offset
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(pos, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to ExtractSubsetOfPositions_menu() do not appear to contain any (x,y) information"
	endif
	
	// if present get the pixel size
	// assume that the pixel size is symmetric
	pixelSize = NumberByKey("X PIXEL SIZE", note(pos))
	
	
	for (i = 0; i < nPos; i+= 1)
		if ((pos[i][xCol] >= xStart) && (pos[i][yCol] >= yStart) && (pos[i][xCol] <= xEnd) && (pos[i][yCol] <= yEnd))
			nPositionsWithinLimits += 1
		endif
	endfor
	
	Make /D/O/N=(nPositionsWithinLimits, DimSize(pos, 1)) root:Packages:PALM:M_SelectedPositions
	wave selectedPositions = root:Packages:PALM:M_SelectedPositions
	
	offset = 0
	for (i = 0; i < nPos; i+= 1)
		if ((pos[i][xCol] >= xStart) && (pos[i][yCol] >= yStart) && (pos[i][xCol] <= xEnd) && (pos[i][yCol] <= yEnd))
			selectedPositions[offset][] = pos[i][q]
			offset += 1
		endif
	endfor
	
	// we now have the positions within the marquee in selectedPositions
	// calculate some statistics from them
	
	// get the center of the cluster
	variable clusterCenterX = 0, clusterCenterY = 0
	for (i = 0; i < nPositionsWithinLimits; i+=1)
		clusterCenterX += selectedPositions[i][xCol]
		clusterCenterY += selectedPositions[i][yCol]
	endfor
	clusterCenterX /= nPositionsWithinLimits
	clusterCenterY /= nPositionsWithinLimits
	
	// calculate the distances of all the positions from the center
	Make /D/O/N=(nPositionsWithinLimits) /FREE W_CenterDistances
	for (i = 0; i < nPositionsWithinLimits; i+=1)
		W_CenterDistances[i] = sqrt((selectedPositions[i][xCol] - clusterCenterX)^2 + (selectedPositions[i][yCol] - clusterCenterY)^2)
	endfor
	
	WaveStats /Q W_CenterDistances
	variable stdDev = V_sdev
	variable maxDistance = V_max
	variable avgDistance = V_avg
	Printf "In pixels: %u points, center is at (%g, %g), average distance from the center is %g, standard deviation is %g, radius of enclosing circle is %g\r%u,%g,%g,%g,%g,%g\r", nPositionsWithinLimits, clusterCenterX, clusterCenterY, avgDistance, stdDev, maxDistance, nPositionsWithinLimits, clusterCenterX, clusterCenterY, avgDistance, stdDev, maxDistance
	
	// also print the information in pixels if available
	if (NumType(pixelSize) == 0)
		string formatString = "In nanometer: %u points, center is at (%g nm, %g nm), average distance from the center is %g nm, standard deviation is %g nm, radius of enclosing circle is %g nm\r%u,%g,%g,%g,%g,%g\r" // work around the stupid 400-character limitation in Igor
		Printf formatString, nPositionsWithinLimits, clusterCenterX * pixelSize, clusterCenterY * pixelSize, avgDistance * pixelSize, stdDev * pixelSize, maxDistance * pixelSize, nPositionsWithinLimits, clusterCenterX * pixelSize, clusterCenterY * pixelSize, avgDistance * pixelSize, stdDev * pixelSize, maxDistance * pixelSize
	endif
End

Function LFunctionClustering_Marquee()
	variable xStart, xEnd, yStart, yEnd, err
	variable pixelSize
	string posName, units
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:'PALM Positions'
	
	// try to get the positions wave from the graph automatically
	wave pos = GetPositionsFromGraph()
	
	// if this failed then prompt the user
	if (WaveExists(pos) != 1)
		Prompt posName, "Positions wave:", popup, GetPossiblePositionsWaves()
		DoPrompt "Choose a positions wave", posName
		if (V_flag == 1)
			return 0
		endif
	
		wave pos = GetPositionsWaveReference(posName)	
		if (WaveExists(pos) == 0)
			Abort "The selected positions wave doesn't seem to exist!"
		endif
	endif
	
	// GetMarqueeCoordinates will return the coordinates in units of pixels
	err = GetMarqueeCoordinates(pos, xStart, yStart, xEnd, yEnd)
	if (err != 0)	// no Marquee
		return 0
	endif
	
	 if ((xEnd <= xStart) || (yEnd <= yStart) || (xStart < 0) || (yStart < 0))
	 	Abort "Invalid limits from the marquee"
	 endif
	
	variable nPos = DimSize(pos, 0), xCol, yCol, zCol
	variable nPositionsWithinLimits = 0, i, offset
	
	// get the indices of the columns containing the x and y coordinates
	getColumnsForEmitterPositions(pos, xCol, yCol, zCol)
	if ((xCol == -1) || (yCol == -1))
		Abort "The positions passed to ExtractSubsetOfPositions_menu() do not appear to contain any (x,y) information"
	endif
	
	// if present get the pixel size
	// assume that the pixel size is symmetric
	pixelSize = NumberByKey("X PIXEL SIZE", note(pos))
	
	
	for (i = 0; i < nPos; i+= 1)
		if ((pos[i][xCol] >= xStart) && (pos[i][yCol] >= yStart) && (pos[i][xCol] <= xEnd) && (pos[i][yCol] <= yEnd))
			nPositionsWithinLimits += 1
		endif
	endfor
	
	Make /D/O/FREE/N=(nPositionsWithinLimits, DimSize(pos, 1)) M_SelectedPositions
	wave selectedPositions = M_SelectedPositions
	
	Note /K/NOCR selectedPositions, Note(pos)
	
	offset = 0
	for (i = 0; i < nPos; i+= 1)
		if ((pos[i][xCol] >= xStart) && (pos[i][yCol] >= yStart) && (pos[i][xCol] <= xEnd) && (pos[i][yCol] <= yEnd))
			selectedPositions[offset][] = pos[i][q]
			offset += 1
		endif
	endfor
	
	DFREF savDF = GetDataFolderDFR()
	SetDataFolder root:Packages:PALM
	RipleyLFunctionClustering /RNGE={(xEnd - xStart) / 2, round(xEnd - xStart) / 0.01} /REGN={xStart, xEnd, yStart, yEnd} M_SelectedPositions
	SetDataFolder savDF
	
	wave W_lFunction = root:Packages:PALM:W_lFunction
	Duplicate /O W_lFunction, root:Packages:PALM:W_LFunctionSubX
	wave W_LFunctionSubX = root:Packages:PALM:W_LFunctionSubX
	
	W_LFunctionSubX = W_lFunction - x
	
	// add some information to the wavenote
	string waveNote = ""
	waveNote += "CALCULATED FROM:" + NameOfWave(pos) + ";"
	waveNote += "X START:" + num2str(xStart) + ";"
	waveNote += "X END:" + num2str(xEnd) + ";"
	waveNote += "Y START:" + num2str(yStart) + ";"
	waveNote += "Y END:" + num2str(yEnd) + ";"
	
	// if the pixel sizes have been provided then add them as well
	if (strlen(StringByKey("X PIXEL SIZE", Note(pos))) > 0)
		waveNote += "X PIXEL SIZE:" + StringByKey("X PIXEL SIZE", Note(pos)) + ";"
	endif
	if (strlen(StringByKey("Y PIXEL SIZE", Note(pos))) > 0)
		waveNote += "Y PIXEL SIZE:" + StringByKey("Y PIXEL SIZE", Note(pos)) + ";"
	endif
	
	Note /K /NOCR W_lFunction, waveNote
	Note /K /NOCR W_LFunctionSubX, waveNote
	
	DoWindow /F LFunction
	if (V_flag != 1)
		Display /K=1/N=LFunction W_LFunctionSubX
		Label /W=LFunction Left, "L Function - x"
		Label /W=LFunction Bottom, "Distance (pixel)"
	endif
	
End
