// XML Note Tools
// Jeffrey J Weimer
// SVN History: $Revision: 615 $ on $Date: 2008-03-18 19:17:09 -0400 (Tue, 18 Mar 2008) $

// Accessible Functions

// XMLPrint(property,text,[attributes, values, pretty])
// prints XML string to history
//	property: string of propert(ies)	- "property1:property2: ... propertyN"
//	text: string text to put at final property
//	attributes: string list of attributes		- "attribute1;attribute2; ... attributeN"
//	values: list of values for each attribute
//	pretty: variable 	- 0 (default) returns without tabs and returns, 1 returns with tabs and returns

// XMLPrintf(property,text,[attributes, values, pretty])
// same as above but returns string rather than prints to history

// XMLWaveNote(twave,property,text,[attributes, values, pretty, replace, at])
// same as above but attaches text to wave twave (full path or current folder only!)
//	replace: variable		- 0 (default), N => replace forward at N, -N => replace reverse at N
//	at:	string	"end" (default), "start"
//	for a note containing T total properties, replace=(-N) is the same as replace=(T-N+1)

// XMLNoteText(ntbk,property,text,[attributes, values, pretty, replace, at])
// same as above but puts text to notebook ntbk
//	at:	string	"end" (default), "start", "current"
//	selection always remains immediately after insertion for next addition!

// XMLNoteObject(ntbk,property,object,[objSpecs, attributes, values, pretty, at])
// same as above but puts an object (graph, table, ...) to notebook ntbk
//	objSpecs:	string list of four parameters (see Notebook ... frame={}, scale={}, picture={})
//	frame (default=0); scale (default=100); mode (default=0); flag (default=1); ;

// NT_LoadXMLTemplate(template,[filename])
// loads an XML file as a template (empty places contain #)

// NT_FillXMLTemplate(template,paramlist)
// returns text that is the template filled with paramlist in each #

// VARIATIONS
// text or values prefaced with % are interpreted as STRING commands to be executed
// to override this, use \% (escape code sequence)

#pragma rtGlobals=1
#pragma IgorVersion=6.03
#pragma version = 0.40

StrConstant xmlnt_ts="@"
StrConstant xmlnt_ps="#"

Menu "Misc", hideable
	Submenu "Panels"
		"XML Note Tools Panel",/Q,ShowXMLPanel()
	end
End

// Tool Panel Sizes

STRUCTURE XMLNoteToolsGlobals
	variable width
	variable height
ENDSTRUCTURE

Static Function InitXMLNTGlobals(xmlNT)
	STRUCT XMLNoteToolsGlobals &xmlNT
	xmlNT.width = 230
	xmlNT.height = 255
end

Function XMLPrint(property,text,[attributes, values, pretty])
	string property, text, attributes, values
	variable pretty

	string theText
	
	if (ParamIsDefault(pretty))
		pretty = 0
	endif
	
	if (ParamIsDefault(attributes))
		theText = xmlprintf(property,text, pretty=pretty)
	else
		if (ParamIsDefault(values))
			theText = xmlprintf(property, text, attributes=attributes, values="", pretty=pretty)
		else
			theText = xmlprintf(property, text, attributes=attributes, values=values, pretty=pretty)
		endif
	endif
	print theText
	return 0
end

Function/S XMLPrintF(property,text,[attributes, values, pretty])
	string property, text, attributes, values
	variable pretty
	
	string theText
	
	if (ParamIsDefault(pretty))
		pretty = 0
	endif
	
	if (ParamIsDefault(attributes))
		theText = parseXMLinput(property,text, pretty=pretty)
	else
		if (ParamIsDefault(values))
			theText = parseXMLinput(property, text, attributes=attributes, values="", pretty=pretty)
		else
			theText = parseXMLinput(property, text, attributes=attributes, values=values, pretty=pretty)
		endif
	endif
	return theText
end

Function XMLWaveNote(twave,property,text,[attributes, values, pretty, replace, at])
	wave twave
	string property, text, attributes, values, at
	variable pretty, replace

	string theText="", theNote=""
	
	if (!WaveExists(twave))
		return -1
	endif

	if (ParamIsDefault(replace))
		replace = 0
	endif
	
	if (ParamIsDefault(at))
		at = "end"
	endif
	
	switch(replace)
		case 0:
			if (ParamIsDefault(pretty))
				pretty = 0
			endif
			if (ParamIsDefault(attributes))
				theText = xmlprintf(property,text, pretty=pretty)
			else
				if (ParamIsDefault(values))
					theText = xmlprintf(property, text, attributes=attributes, values="", pretty=pretty)
				else
					theText = xmlprintf(property, text, attributes=attributes, values=values, pretty=pretty)
				endif
			endif
			break
		default:
			theNote = note(twave)
			theText = ReplaceXMLProperty(property,theNote,text,index=replace)
			note/k twave
			at = "end"	
			break
	endswitch
	strswitch(at)
		case "start":
			sprintf theText, "%s\r%s", theText, note(twave)
			note/k twave
			break
		default:
			break
	endswitch
	note twave, theText
	return 0
end

Function XMLNoteText(ntbk,property,text,[attributes, values, pretty, replace, at])
	string ntbk, property, text, attributes, values, at
	variable pretty, replace

	string theText
//	variable st, en,
	
	variable sPg,sPs,ePg,ePs, tN, nLines
	
	if (strlen(ntbk)==0)
		ntbk = WinName(0,16)
	else		
		if (strlen(WinList(ntbk,";","WIN:16"))==0)
			return -1
		endif
	endif
	
	Notebook $ntbk, findText={"",1}
	GetSelection notebook, $ntbk,1
	sPg = V_startParagraph
	sPs = V_startPos
	ePg = V_endParagraph
	ePs = V_endPos
	Notebook $ntbk, selection={endofFile,endofFile}
	GetSelection notebook, $ntbk, 1
	nLines = V_startParagraph	
	Notebook $ntbk, selection={(sPg,sPs),(ePg,ePs)}
	
	if (ParamIsDefault(replace))
		replace = 0
	endif
	
	if (ParamIsDefault(at))
		at = "current"
	endif

	if (ParamIsDefault(pretty))
		pretty = 0
	endif
	
	switch(replace)
		case 0:		
			if (ParamIsDefault(attributes))
				theText = xmlprintf(property,text, pretty=pretty)
			else
				if (ParamIsDefault(values))
					theText = xmlprintf(property, text, attributes=attributes, values="", pretty=pretty)
				else
					theText = xmlprintf(property, text, attributes=attributes, values=values, pretty=pretty)
				endif
			endif
			strswitch(at)
				case "current":
					break
				case "next":
					Notebook $ntbk, selection={(sPg+1,0),(sPg+1,0)}
					break
				case "previous":
					Notebook $ntbk, selection={(sPg-1,0),(sPg-1,0)}
					break
				case "start":
					Notebook $ntbk, selection={startOfFile,startOfFile}
					break
				case "end":
					Notebook $ntbk, selection={endOfFile,endOfFile}
					break
				default:
					tN = str2num(at)
					if ((sPg+tN)<0)
						tN = -sPg
					endif
					if ((sPg+tN)>nLines)
						tN = nLines - sPg
					endif
					Notebook $ntbk, selection={(sPg+tN,0),(sPg+tN,0)}
					break
			endswitch
			sprintf theText, "%s\r", theText
			notebook $ntbk text=theText
			break
		default:
			ReplaceXMLNtbkProperty(property,ntbk,text,index=replace)
			Notebook $ntbk selection={(sPg,sPs),(ePg,ePs)}
			break
	endswitch
	
	return 0
end

Function XMLNoteObject(ntbk,property,obj,[oSpecs,attributes, values, pretty, replace, at])
	string ntbk, property, obj, oSpecs, attributes, values, at
	variable pretty, replace

	string theInText, oT
	variable nt, ic

	if (strlen(ntbk)==0)
		ntbk = WinName(0,16)
	else		
		if (strlen(WinList(ntbk,";","WIN:16"))==0)
			return -1
		endif
	endif
	
	if (ParamIsDefault(oSpecs))
		oSpecs = ""
	endif
	
	if (ParamIsDefault(replace))
		replace = 0
	endif
	
	if (ParamIsDefault(at))
		at = "current"
	endif
	
	if (ParamIsDefault(values))
		values = ""
	endif
	
	if (ParamIsDefault(attributes))
		attributes = ""
	endif
	
	nt = ItemsInList(property,":")
	
	theInText = "xmlNTinsertobject(\"" + ntbk + "\",\"" + obj + "\",\"" + oSpecs +"\")"
	
	switch(replace)
		case 0:
			if (ParamIsDefault(pretty))
				pretty = 0
				sprintf oT, "%s%s%s", xmlnt_ts, "OBJECT", xmlnt_ts
			else
				if (pretty)
					sprintf oT, "\r%s%s%s\r", xmlnt_ts, "OBJECT", xmlnt_ts
					for (ic=0;ic<nt-1;ic+=1)
						oT += "\t"
					endfor
				else
					sprintf oT, "%s%s%s", xmlnt_ts, "OBJECT", xmlnt_ts
				endif
			endif
			if (ParamIsDefault(attributes))
				xmlnotetext(ntbk,property,oT,pretty=pretty, at=at)
			else
				if (ParamIsDefault(values))
					xmlnotetext(ntbk,property,oT, attributes=attributes, values="", pretty=pretty, at=at)
				else
					xmlnotetext(ntbk,property, oT, attributes=attributes, values=values, pretty=pretty, at=at)
				endif
			endif
			break
		default:
			ReplaceXMLNtbkProperty(property,ntbk,theInText,index=replace)
			break
	endswitch
	
	sprintf oT, "%s%s%s", xmlnt_ts, "OBJECT", xmlnt_ts
	Notebook $ntbk findText={"",1}
	GetSelection notebook, $ntbk, 1
	Notebook $ntbk findText={oT,2^4}
	Execute/Q theInText
	Notebook $ntbk selection={(V_startParagraph,V_startPos),(V_endParagraph,V_endPos)}
	
	return 0
end

Function NT_LoadXMLTemplate(template,[filename])
	string template, filename
	
	string cdf = GetDataFolder(1), theText
	variable refNum
	
	SetDataFolder root:
	
	if (!DataFolderExists("Packages:XMLNoteTools"))
		SetXMLNTGlobals()
	endif

	NewDataFolder/O/S root:Packages:XMLNoteTools:Templates
	
	if (ParamIsDefault(filename))
		theText="Select an XML file for template " + template
		open/D/R/T=".xml"/M=theText refNum
		if (strlen(S_filename)==0)
			close refNum
			SetDataFolder $cdf
			return -1
		endif
		filename = S_filename
	endif
	
	open/R refNum as filename
	
	string/G $template=""
	
	SVAR tstr = $template
	FSetPos refNum, 0
	do
		FReadLine refNum, theText
		if (strlen(theText)==0)
			break
		endif
		tstr += theText
	while(1)
	close refNum
	SetDataFolder $cdf
	return 0
end

Function/S NT_FillXMLTemplate(template,paramlist)
	string template, paramlist
	
	string rtStr, cdf = GetDataFolder(1), theText
	variable ic=0, nt
	
	theText = "root:Packages:XMLNoteTools:Templates:" + template
	
	if (exists(theText)==0)
		NT_LoadXMLTemplate(template)
	endif
	
	SVAR tstr = $theText

	SetDataFolder root:
	
	rtStr = tstr
	
	nt = ItemsInList(paramlist)
	do
		theText=StringFromList(ic,paramlist)
		if (strlen(theText)==0)
			break
		endif
		theText = parseXMLvalue(theText)
		rtStr = ReplaceString("#",rtStr,theText,0,1)
		ic += 1
	while(ic<nt)
	SetDataFolder $cdf
	return rtStr
end

Function/S XMLNTInsertObject(ntbk, obj, oSpecs)
	string ntbk, obj, oSpecs
	
	string theCmd = "Notebook " + ntbk
	string mode, flag, frame, scale
	variable st
	
	if (strlen(obj) == 0)
		obj = WinName(0,1)
	else
		st = strsearch(obj,"(",0)
		if (st>0)
			mode = obj[0,st-1]
			flag = obj[st,strlen(obj)-1]
		else
			mode = obj
			flag = ""
		endif
		switch(WinType(mode))
			case 1:
			case 3:
				obj = mode + flag
				break
			case 2:
			case 7:
				obj = mode
				break
			default:
				obj = WinName(0,1)
				break
		endswitch
	endif
	
	if (ItemsInList(oSpecs)<4)
		oSpecs = "0;100;0;1;"
	endif
	
	frame = StringFromList(0,oSpecs)
	if (strlen(frame)==0)
		frame = "0"
	endif
	scale = StringFromList(1,oSpecs)
	if (strlen(scale)==0)
		scale="100"
	endif
	mode = StringFromList(2,oSpecs)
	if (strlen(mode)==0)
		mode = "0"
	endif
	flag = StringFromList(3,oSpecs)
	if (strlen(flag)==0)
		flag = "1"
	endif
	
	sprintf theCmd, "%s, frame=%s, scaling={%s,%s}, picture={%s,%s,%s}", theCmd,frame,scale,scale, obj, mode,flag
	Execute/Q theCmd
	
	return (theCmd)
end

Static Function/S ReplaceXMLProperty(theproperty,theText,theReplace,[start, index])
	string theproperty, theText, theReplace
	variable start, index
	
	variable ic, st, en, jc
	string theStr, rtnStr, reStr, thePStr
	
	if (!ParamIsDefault(index))
		if (abs(index)>ItemsInXML(theproperty,theText))
			return theText
		endif
		if (index>0)
			if (ParamIsDefault(start))
				st = 0
			else
				st = start
			endif
			en = st
			for (ic=0;ic<index;ic+=1)
				en = GetXMLOffset(theproperty,theText,start=en)+1
				if (en < 0)
					return theText
				endif
			endfor
			st = en
		else
			if (ParamIsDefault(start))
				st = INF
			else
				st = start
			endif
			en = st
			thePStr = StringFromList(0,theproperty,":")
			for (ic=0;ic>index;ic-=1)
				en = GetXMLOffset(theproperty,theText,start=en,option=1)-1
				if (en < 0)
					return theText
				endif
				en = GetXMLOffset(thePStr,theText,start=en,option=1)-1
			endfor
			en = GetXMLOffset(theproperty,theText,start=en+1)+1
			st = en
		endif
	else
		st = GetXMLOffset(theproperty,theText,start=st)
	endif
	
	jc = ItemsInList(theproperty,":") - 1
	st = strsearch(theText,">",st)+1
	sprintf thePStr, "</%s>", StringFromList(jc,theproperty,":")
	en = strsearch(theText,thePStr,st)
	sprintf theStr, "%s%s",theText[st,en-1], thePStr
	sprintf reStr, "%s%s", parseXMLvalue(theReplace), thePStr
	rtnStr = ReplaceString(theStr,theText,reStr,0,1)
	return rtnStr
end

Static Function ReplaceXMLNtbkProperty(theproperty,theNtbk,theReplace, [index])
	string theproperty, theNtbk, theReplace
	variable index
	
	variable ic, en
	string thePStr
	
	if (!ParamIsDefault(index))
		if (index>0)
			Notebook $theNtbk, selection={startOfFile,startOfFile}
			for (ic=0;ic<index;ic+=1)
				en = SetXMLNtbkOffset(theproperty,theNtbk)
				if (en < 0)
					return -1
				endif
			endfor
		else
			Notebook $theNtbk, selection={endOfFile,endOfFile}
			thePStr = StringFromList(0,theproperty,":")
			for (ic=0;ic>index;ic-=1)
				en = SetXMLNtbkOffset(thePStr,theNtbk,flag=4)
				GetSelection notebook, $theNtbk, 1
				Notebook $theNtbk, selection={(V_startParagraph,0),(V_startParagraph,0)}
				if (en < 0)
					return 0
				endif
			endfor
			thePStr=StringFromList(ItemsInList(theproperty,":")-1,theproperty,":")
			en = SetXMLNtbkOffset(thePStr,theNtbk)
		endif
	else
		Notebook $theNtbk, selection={startOfFile,startOfFile}
		en = SetXMLNtbkOffset(theproperty,theNtbk)
	endif
	Notebook $theNtbk, text=theReplace
end

Static Function GetXMLOffset(theproperty,theText, [start, option])
	string theproperty, theText
	variable start, option
	
	variable ic, nt, st
	string theStr
	
	if (ParamIsDefault(option))
		option=0
	endif
	
	if ((option==1)||(option==3))
		if (ParamIsDefault(start))
			start = INF
		endif
	else
		if (ParamIsDefault(start))
			start = 0
		endif
	endif
	
	st = start
	nt = ItemsInList(theproperty, ":")
	if (nt > 1)
		for (ic=0;ic<nt-1;ic+=1)
			sprintf theStr, "<%s", StringFromList(ic,theproperty,":")
			if (!option)
				st = strsearch(theText,theStr,st)
			else
				st = strsearch(theText,theStr,st,option)
			endif
			if (st < 0)
				return st
			endif
		endfor
		sprintf theStr, "<%s", StringFromList(ic,theproperty,":")
		st = strsearch(theText,theStr,st)
	else
		sprintf theStr, "<%s", StringFromList(0,theproperty,":")
		if (!option)
			st = strsearch(theText,theStr,st)
		else
			st = strsearch(theText,theStr,st,option)
		endif
	endif
	return st
end

Static Function SetXMLNtbkOffset(theproperty,theNtbk, [flag])
	string theproperty, theNtbk
	variable flag
	
	variable ic, nt, st, en, pst, pen
	string theStr, theProp
	
	if (ParamIsDefault(flag))
		flag=-1
	endif
	
	nt = ItemsInList(theproperty, ":")
	if (nt > 1)
		for (ic=0;ic<nt;ic+=1)
			theProp = StringFromList(ic,theproperty,":")
			sprintf theStr, "<%s", theProp
			if (flag<0)
				Notebook $theNtbk, findText={theStr,4}
			else
				Notebook $theNtbk, findText={theStr,2^flag}
			endif
			if (!V_flag)
				return -1
			endif
		endfor
	else
		theProp = StringFromList(0,theproperty,":")
		sprintf theStr, "<%s", theProp
		if (flag<0)
			Notebook $theNtbk, findText={theStr,4}
		else
			Notebook $theNtbk, findText={theStr,2^flag}
		endif
		if (!V_flag)
			return -1
		endif
	endif
	sprintf theStr, "/%s>", theProp
	Notebook $theNtbk, findText={theStr,4}
	GetSelection notebook, $theNtbk, 1
	en = V_startPos-1; pen=V_startParagraph
	sprintf theStr, "<%s",theProp
	Notebook $theNtbk, findText={theStr,2^4}
	GetSelection notebook, $theNtbk, 1
	Notebook $theNtbk, findText={">",2}                            
	GetSelection notebook, $theNtbk, 1
	st = V_endPos; pst = V_endParagraph
	Notebook $theNtbk, selection={(pst,st),(pen,en)}
	return (st)
end

Static Function ItemsInXML(property,theText)
	string property, theText
	
	variable ic=0, en=0
	
	do
		en = GetXMLOffset(property,theText, start=en)+1
		if (en == 0)
			break
		endif
		ic +=1
	while(1)	
	return ic
end

Static Function/S parseXMLinput(property,text,[attributes, values, pretty])
	string property, text, attributes, values
	variable pretty

	string tS0 = "<", ts1 = "</", tE = ">", tR = "\r"
	string tT = "\t", tQ = "\"", tB="/"
	
	string theProp ="", rtnStr = "", theEnd = "", theAtt="", itsV="", tStr=""
	
	variable ic, nt, nv
	
	switch(pretty)
		case -1:
			return text
			break
		case 0:
			tR = ""
			tT = ""
			break
		case 1:
			break
	endswitch
	
	if (ParamIsDefault(attributes))
		theAtt = ""
	else
		nt = ItemsInList(attributes)
		if (nt==1)
			itsV = parseXMLvalue(StringFromList(0,values))
			sprintf theAtt, " %s=%s%s%s",StringFromList(0,attributes),tQ,itsV,tQ
		else
			nv = ItemsInList(values)
			if (nv < nt)
				values = ""
				for(ic=0;ic<nt;ic+=1)
					values += ";"
				endfor
			endif
			theAtt = ""
			for(ic=0;ic<nt;ic+=1)
				theProp = StringFromList(ic,attributes)
				itsV = parseXMLvalue(StringFromList(ic,values))
				sprintf theAtt, "%s %s=%s%s%s", theAtt, theProp,tQ,itsV,tQ
			endfor
		endif
	endif
	
	nt = ItemsInList(property,":")
	
	switch(nt)
		case 0:
			rtnStr = parseXMLvalue(text)
			break
		case 1:
			theProp = StringFromList(0,property,":")
			if (strlen(text)==0)
				sprintf rtnStr, "%s%s%s %s%s", ts0,theProp,theAtt,tB,tE
			else
				sprintf rtnStr, "%s%s%s%s%s%s%s%s", ts0,theProp,theAtt,tE,parseXMLvalue(text),ts1,theProp,tE
			endif
			break
		default:
			for (ic=0;ic<nt-1;ic+=1)
				theProp = StringFromList(ic,property,":")
				theEnd += tT
				sprintf rtnStr, "%s%s%s%s%s%s",rtnStr,ts0,theProp,tE,tR,theEnd
			endfor
			theProp = StringFromList(nt-1,property,":")
			if (strlen(text)==0)
				sprintf rtnStr, "%s%s%s%s %s%s", rtnStr,ts0,theProp,theAtt,tB,tE
			else
				sprintf rtnStr, "%s%s%s%s%s%s%s%s%s", rtnStr,ts0,theProp,theAtt,tE,parseXMLvalue(text),ts1,theProp,tE
			endif
			for (ic=nt-2;ic>=0;ic-=1)
				theEnd = RemoveEnding(theEnd,"\t")
				theProp = StringFromList(ic,property,":")
				sprintf rtnStr, "%s%s%s%s%s%s", rtnStr, tR,theEnd,ts1,theProp,tE
			endfor
			break
	endswitch
	
	return rtnStr
end

Static Function/S parseXMLvalue(vStr)
	string vStr
	
	string rtnStr, theCmd
	
	theCmd = vStr[0,1]
	if (cmpstr(theCmd,"\%")==0)
		rtnStr = vStr[1,strlen(vStr)-1]
		return rtnStr
	endif
	
	if (cmpstr(vStr[0],"%") == 0)
		string/G tmpXMLToolStr
		theCmd =  "tmpXMLToolStr = " + vStr[1,strlen(vStr)-1]
		Execute/Q/Z theCmd
		if (V_flag)
			tmpXMLToolStr = "Execution Error: " + vStr[1,strlen(vStr)-1]
		endif
		rtnStr = tmpXMLToolStr
		killstrings/Z tmpXMLToolStr
	else
		rtnStr = vStr
	endif
	
	return rtnStr
end

Function ShowXMLPanel()

	if (strlen(WinList("XMLNoteToolsPanel",";","WIN:64"))!=0)
		DoWindow/F XMLNoteToolsPanel
		return 0
	endif

//	if (!DataFolderExists("root:Packages:XMLNoteTools"))
	SetXMLNTGlobals()
//	endif
	
	NVAR gtemplate =  root:Packages:XMLNoteTools:PanelGlobals:template
	NVAR gdynamic =  root:Packages:XMLNoteTools:PanelGlobals:dynamic
	NVAR greplace = root:Packages:XMLNoteTools:PanelGlobals:replace
	NVAR gobject = root:Packages:XMLNoteTools:PanelGlobals:object
	NVAR gnotebk =  root:Packages:XMLNoteTools:PanelGlobals:notebk
	NVAR gframe = root:Packages:XMLNoteTools:PanelGlobals:frame
	NVAR gsize = root:Packages:XMLNoteTools:PanelGlobals:size
	NVAR gmode = root:Packages:XMLNoteTools:PanelGlobals:mode
	NVAR gpretty = root:Packages:XMLNoteTools:PanelGlobals:pretty
	NVAR gxsize = root:Packages:XMLNoteTools:PanelGlobals:xsize
	NVAR gysize =  root:Packages:XMLNoteTools:PanelGlobals:ysize
	SVAR gnotes0 = root:Packages:XMLNoteTools:PanelGlobals:notes0
	SVAR gnotes1 = root:Packages:XMLNoteTools:PanelGlobals:notes1
	SVAR gnotes2 = root:Packages:XMLNoteTools:PanelGlobals:notes2
	SVAR gproperty = root:Packages:XMLNoteTools:PanelGlobals:property
	SVAR gnewntbk = root:Packages:XMLNoteTools:PanelGlobals:newnotebk
	SVAR gtitle = root:Packages:XMLNoteTools:PanelGlobals:title

	STRUCT XMLNoteToolsGlobals xmlNT
	STRUCT WMPopupAction pa
	
	InitXMLNTGlobals(xmlNT)
	
#if Exists("ScreenSizer#Initialize")
	ScreenSizer#Initialize(0)
	ScreenSizer#MakePanel(70,10,xmlNT.width,xmlNT.height,"XML Note Tools",other="/N=XMLNoteToolsPanel/K=1")
#else
	NewPanel/W=(0,0,xmlNT.width,xmlNT.height)/N=XMLNoteToolsPanel/K=1 as "XML Note Tools"
#endif

//	ModifyPanel noEdit=1, fixedSize=1
	SetDrawLayer UserBack
	DrawLine 5,55,220,55
	DrawLine 5,83,220,83
	DrawLine 5,154,220,154
	DrawLine 5,227,220,227

	TabControl NotePad,pos={1,2},size={80,20},tabLabel(0)="Note Pad"
//	TabControl NotePad,tabLabel(1)="Templater",value= 0

	PopupMenu tb0_template,pos={8,30},size={197,21},title="Template", proc=xmlnt_template
	PopupMenu tb0_template,mode=gtemplate,bodyWidth=148,value= #"xmlnt_tlist()"
	CheckBox tb0_cbdynamic title="!",pos={206,33}, variable=gdynamic

	CheckBox tb0_cbtitle,pos={8,62},size={46,14},title="Title", value=1, proc=xmlnt_settitle
	SetVariable tb0_title,pos={47,62},size={85,16},value=gtitle,bodyWidth=85, title=" ", disable=1
	CheckBox tb0_cbdate,pos={135,62},size={46,14},title="Date", value=1
	CheckBox tb0_cbtime,pos={180,62},size={46,14},title="Time", value=1
	SetVariable tb0_property,pos={8,62},size={161,16},value=gproperty,bodyWidth=132, title="Prop:", disable=1
	CheckBox tb0_cbpretty,pos={175,63},size={45,14},title="Pretty", variable=gpretty, disable=1

	CheckBox tb0_cbobject,pos={8,89},size={15,15},title="Object",value=1, proc=xmlnt_getobject

	PopupMenu tb0_object,pos={59,86},size={165,21},title=""
	PopupMenu tb0_object,mode=gobject,bodyWidth=165,value= #"xmlnt_olist()"

	SetVariable tb0_frame,pos={6,120},size={60,16},title="Frame"
	SetVariable tb0_frame,limits={0,4,1},value=gframe

	Slider tb0_scale,pos={66,110},size={80,52},limits={5,100,5},variable=gsize,vert= 0

	ValDisplay tb0_showscale,pos={143,122},size={25,15}, value=#"root:Packages:XMLNoteTools:PanelGlobals:size"
	
	SetVariable tb0_mode,pos={6,136},size={60,16},title="Mode"
	SetVariable tb0_mode,limits={-5,8,1},value=gmode, proc=xmlnt_setmode

	CheckBox tb0_cbbw,pos={8,105},size={42,14},title="B&W",value=0

	SetVariable tb0_height,pos={178,114},size={47,16},title="|",value=gysize,bodyWidth= 42, limits={0,8200,50}, proc=xmlnt_psize
	SetVariable tb0_width,pos={178,133},size={47,16},title="-",value=gxsize,bodyWidth= 42, limits={0,8200,50}, proc=xmlnt_psize
	
	CheckBox tb0_cbnotes,pos={8,161},size={46,14},title="Notes", value=1, proc=xmlnt_getnotes

	PopupMenu tb0_separator,pos={175,156},size={46,21},title="", value="cr; ;tab;", mode=1

	Button tb0_clearnotes,pos={105,157},size={45,20},title="Clear",fSize=12, proc=XMLNT_ClearNotes
	SetVariable tb0_replace,pos={162,158},size={60,16},title=">#<"
	SetVariable tb0_replace,limits={-inf,inf,1},value=greplace, disable=1, proc=xmlnt_setreplace

	SetVariable tb0_notes0,pos={8,177},size={215,16},value=gnotes0,bodyWidth=215, title=" "
	SetVariable tb0_notes1,pos={8,193},size={215,16},value=gnotes1,bodyWidth=215, title=" "
	SetVariable tb0_notes2,pos={8,209},size={215,16},value=gnotes2,bodyWidth=215, title=" "

	Button tb0_tonotebook,pos={8,230},size={20,20},title="->",fSize=12,fStyle=1, proc=XMLNT_SendToNotebook

	PopupMenu tb0_notebook,pos={28,230},size={175,21},title="",proc=xmlnt_setnotebk
	PopupMenu tb0_notebook,mode=gnotebk,value= #"xmlnt_nlist()"

	PopupMenu tb0_location,pos={163,230},size={152,21},title="at"
	PopupMenu tb0_location,mode=3,value="|<<;<;!;>;>>|;"

	SetVariable tb0_newntbk,pos={100,232},size={120,21},title=" ",value=gnewntbk, disable=1
	
	pa.eventCode=99
	pa.popNum=gtemplate
	xmlnt_template(pa)
	SetWindow XMLNoteToolsPanel hook=xmlnt_panelhook
	return 0
End

Function XMLNT_ClearNotes(ba) : ButtonControl
	STRUCT WMButtonAction &ba

	SVAR gnotes0 = root:Packages:XMLNoteTools:PanelGlobals:notes0
	SVAR gnotes1 = root:Packages:XMLNoteTools:PanelGlobals:notes1
	SVAR gnotes2 = root:Packages:XMLNoteTools:PanelGlobals:notes2
	switch( ba.eventCode )
		case 2:
			ControlInfo tb0_template
			switch(V_value)
				case 1:
					gnotes0=""
					gnotes1=""
					gnotes2=""
					SetVariable tb0_notes0 activate
					break
				case 2:
					gnotes0=""
					SetVariable tb0_notes0 activate
					break
				case 3:
					gnotes1=""
					gnotes2=""
					break
				case 4:
					gnotes1=""
					gnotes2=""
					SetVariable tb0_notes0 activate
					break
			endswitch
	endswitch
	return 0
end

Static Function SetXMLNTGlobals()
 	
	string cdf = GetDataFolder(1)
	
	SetDataFolder root:
	
	NewDataFolder/O/S Packages
	NewDataFolder/O/S XMLNoteTools
	NewDataFolder/O/S PanelGlobals
	variable/G template=1, object=1, notebk=1, replace=0
	variable/G frame=0, size=100, mode=0, pretty=0
	variable/G xsize=0, ysize=0, dynamic=1
	string/G notes0="", notes1="", notes2="", tmpStr=""
	string/G property="", newnotebk="MyNotebook", title="Observation"
	
	SetDataFolder root:Packages:XMLNoteTools:
	
	SetDataFolder $cdf
	return 0
end

Function/S XMLNT_TList()

	string theStr ="", rtStr = "None;property | object | notes;property | object | options;property | text | options;"
	variable ic=0	
//	do
//		theStr = GetIndexedObjName("root:Packages:XMLNoteTools:Templates",3,ic)
//		if (strlen(theStr)==0)
//			break
//		endif
//		rtStr += theStr + ";"
//		ic+=1
//	while(1)	
	return rtStr	
end

Function/S XMLNT_OList()

	string theStr, rtStr = "None;"
	theStr = WinList("*",";","WIN:1")
	theStr += WinList("*",";","WIN:2")
	theStr += WinList("*",";","WIN:4")
	theStr += PICTList("*",";","")
	if (strlen(theStr)==0)
		return rtStr
	else
		return theStr
	endif
end

Function/S XMLNT_NList()

	return ("New;" + WinList("*",";","WIN:16"))
end

Function XMLNT_SendToNotebook(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	NVAR gtemplate =  root:Packages:XMLNoteTools:PanelGlobals:template
	NVAR gobject = root:Packages:XMLNoteTools:PanelGlobals:object
	NVAR gpretty = root:Packages:XMLNoteTools:PanelGlobals:pretty
	NVAR greplace = root:Packages:XMLNoteTools:PanelGlobals:replace
	NVAR gnotebk =  root:Packages:XMLNoteTools:PanelGlobals:notebk
	NVAR gframe = root:Packages:XMLNoteTools:PanelGlobals:frame
	NVAR gsize = root:Packages:XMLNoteTools:PanelGlobals:size
	NVAR gmode = root:Packages:XMLNoteTools:PanelGlobals:mode
	SVAR gnotes0 = root:Packages:XMLNoteTools:PanelGlobals:notes0
	SVAR gnotes1 = root:Packages:XMLNoteTools:PanelGlobals:notes1
	SVAR gnotes2 = root:Packages:XMLNoteTools:PanelGlobals:notes2
	SVAR gproperty = root:Packages:XMLNoteTools:PanelGlobals:property
	SVAR gnewntbk = root:Packages:XMLNoteTools:PanelGlobals:newnotebk
	SVAR gtitle = root:Packages:XMLNoteTools:PanelGlobals:title
	NVAR gysize = root:Packages:XMLNoteTools:PanelGlobals:ysize
	NVAR gxsize = root:Packages:XMLNoteTools:PanelGlobals:xsize

	STRUCT WMPopupAction pa
	
	string theNtbk="", theTmplate="", theObject="", theNotes="", theDTStr="", cr="\r"
	variable cobj=0, cnot=0, bw=1, itsFormat=0, stb=0, ic, nt
	string at="current", oSpecs=""
	
	switch( ba.eventCode )
		case 2:
			
			// get template

			ControlInfo tb0_template
			gtemplate = V_value
			theTmplate = StringFromList(gtemplate-1,XMLNT_TList())
			
			// get the object
			
			ControlInfo tb0_cbobject
			if (V_value)			
				ControlInfo tb0_object
				gobject = V_value
				theObject = StringFromList(gobject-1,XMLNT_OList())
				cobj=1
				ControlInfo tb0_cbbw
				bw = !V_value
			endif
			
			// get notebook and its format

			ControlInfo tb0_notebook
			gnotebk = V_value
			if (gnotebk == 1)
				if (strlen(gnewntbk)==0)
					gnewntbk = "Notebook" + ReplaceString(":",time(),"")
				else
					gnewntbk = CleanUpName(gnewntbk,0)
				endif
				if (strlen(WinList(gnewntbk,";","WIN:16"))==0)
					NewNotebook/F=1/N=$gnewntbk/W=(0,0,20,100)
					itsFormat = 1
					at = "start"
					theNtbk = gnewntbk
				else
					theNtbk = gnewntbk
					SplitString/E=("/F=./") WinRecreation(theNtbk,0)
					itsFormat = str2num(S_value[3])
				endif
				pa.popNum = WhichListItem(theNtbk,XMLNT_NList())+1
				pa.eventCode = 98
				xmlnt_setnotebk(pa)
			else
				theNtbk = StringFromList(gnotebk-1,XMLNT_NList())
				SplitString/E=("/F=./") WinRecreation(theNtbk,0)
				itsFormat = str2num(S_value[3])
			endif	
			
			// get where
			
			ControlInfo tb0_location
			at = StringFromList(V_Value-1,"start;previous;current;next;end;")
			stb = 0
			
			// set the property format
			
			if (cmpstr(gproperty[strlen(gproperty)-1],":")!=0)
				gproperty += ":"
			endif
				
			// put the stuff
			
			switch(gtemplate)
				case 1:	// none
					theDTStr = ""
					ControlInfo tb0_cbtitle
					if (V_value)
						theDTStr = gtitle + ": "
					endif
					ControlInfo tb0_cbdate
					if (V_value)
						theDTStr += date()
					endif
					ControlInfo tb0_cbtime
					if (V_value)
						if (strlen(theDTStr)==0)
							theDTStr += time()
						else
							theDTStr += " / " + time()
						endif
					endif
					XMLNoteText(theNtbk,"",theDTStr,pretty=-1, at=at)
					at = "current"
					if (cobj)
						if (itsFormat)
							Notebook $theNtbk, frame=gframe, scaling={gsize,gsize}, picture={$theObject(0,0,gxsize,gysize),gmode,bw}
							Notebook $theNtbk, text="\r"
						else
							sprintf theDTStr, "OBJECT %s would be here in a formatted notebook\r", theObject
							Notebook $theNtbk, text=theDTStr
						endif
					endif
					ControlInfo tb0_cbnotes
					if (V_value)
						ControlInfo tb0_separator
						switch(V_value)
							case 1:
								cr = "\r"
								break
							case 2:
								cr = " "
								break
							case 3:
								cr = "\t"
								break
						endswitch
						if (strlen(gnotes0)!=0)
							theNotes+= parsexmlvalue(gnotes0) + cr
						endif
						if (strlen(gnotes1)!=0)
							theNotes+= parsexmlvalue(gnotes1) + cr
						endif
						if (strlen(gnotes2)!=0)
							theNotes+= parsexmlvalue(gnotes2) + cr
						endif
						if (cmpstr(cr,"\r")!=0)
							theNotes+="\r"
						endif
						Notebook $theNtbk, text=theNotes
					endif
					break
				case 2:	// property | object | notes
					ControlInfo tb0_cbobject
					if (V_value)
						if (itsFormat)
							sprintf oSpecs, "%d;%d;%d;%d;", gframe, gsize, gmode, bw
							sprintf theObject, "%s(0,0,%d,%d)", theObject,gxsize,gysize
							XMLNoteObject(theNtbk,(gproperty+"object:"),theObject,oSpecs=oSpecs,pretty=gpretty, at=at)
						else
							sprintf theDTStr, "OBJECT %s would be here in a formatted notebook", theObject
							XMLNoteText(theNtbk,(gproperty+"object:"),theDTStr,pretty=gpretty, at=at)
						endif
						stb = 1				
					endif						
					ControlInfo tb0_cbnotes
					if (V_Value)
						if (gpretty)
							nt = ItemsInList(gproperty,":")
							theDTStr=""
							for (ic=0;ic<nt;ic+=1)
								theDTStr += "\t"
							endfor
							if (stb)
								XMLNoteText(theNtbk,"notes:",gnotes0,pretty=gpretty, at=num2str(-nt))
								Notebook $theNtbk, findText={"",1}
								GetSelection notebook, $theNtbk, 1
								Notebook $theNtbk selection={(V_startParagraph-1,0),(V_startParagraph-1,0)}
								Notebook $theNtbk, text=theDTStr		
								Notebook $theNtbk selection={(V_startParagraph+nt,0),(V_startParagraph+nt,0)}
							else
								XMLNoteText(theNtbk,(gproperty+"notes:"),gnotes0,pretty=gpretty, at=at)
							endif
						else
							Notebook $theNtbk, findText={"/object>",2^4}
							GetSelection notebook, $theNtbk, 1
							Notebook $theNtbk selection={(V_startParagraph,V_endPos),(V_startParagraph,V_endPos)}
							Notebook $theNtbk text=("<notes>"+gnotes0+"</notes>")
							Notebook $theNtbk selection={(V_startParagraph+1,0),(V_startParagraph+1,0)}
						endif
					endif
					break
				case 3:	// property | object | options
					if (itsFormat)
						sprintf oSpecs, "%d;%d;%d;%d;", gframe, gsize, gmode, bw
						sprintf theObject, "%s(0,0,%d,%d)", theObject,gxsize,gysize
						XMLNoteObject(theNtbk,(gproperty+"object:"),theObject,attributes=gnotes1,values=gnotes2,oSpecs=oSpecs,pretty=gpretty, at=at)
					else
						sprintf theDTStr, "OBJECT %s would be here in a formatted notebook", theObject
						XMLNoteText(theNtbk,(gproperty+"object:"),theDTStr,attributes=gnotes1,values=gnotes2,pretty=gpretty, at=at)
					endif
					break
				case 4:	// property | text | options
					XMLNoteText(theNtbk,gproperty,gnotes0,attributes=gnotes1,values=gnotes2,pretty=gpretty,replace=greplace,at=at)
					break
				default:
					break
			endswitch
			
	endswitch

	return 0
End

Function xmlnt_template(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	NVAR gtemplate =  root:Packages:XMLNoteTools:PanelGlobals:template
	NVAR gnotebk =  root:Packages:XMLNoteTools:PanelGlobals:notebk
	NVAR gpretty = root:Packages:XMLNoteTools:PanelGlobals:pretty
	NVAR greplace = root:Packages:XMLNoteTools:PanelGlobals:replace
	SVAR gnotes0 = root:Packages:XMLNoteTools:PanelGlobals:notes0
	SVAR gtmpStr = root:Packages:XMLNoteTools:PanelGlobals:tmpStr
	
	gtmpStr = "OBJECT"
	
//	STRUCT WMSetVariableAction sva
	
	switch( pa.eventCode )
		case 2:
			if (pa.popNum==gtemplate)
				break
			endif
		case 99:
			// set for the given template
			
			gtemplate=pa.PopNum
			ModifyControlList ControlNameList("",";","tb0_*") disable=0
//			gpretty = 0
//			greplace = 0
			PopupMenu tb0_location,pos={163,230},size={152,21},title="at",disable=0, bodyWidth=0
			switch(pa.popNum)
				case 1:	// none
					SetVariable tb0_title, disable=0
					Checkbox tb0_cbtitle, disable=0
					Checkbox tb0_cbdate, disable=0
					Checkbox tb0_cbtime, disable=0
					Checkbox tb0_cbpretty, disable=1
					SetVariable tb0_replace, disable=1
					SetVariable tb0_property, disable=1
					Checkbox tb0_cbnotes, title="Notes"
					SetVariable tb0_notes0, value=gnotes0
					SetVariable tb0_notes1,bodyWidth=215, title=" ",pos={8,193},size={215,16}, disable=0
					SetVariable tb0_notes2,bodyWidth=215, title=" ",pos={8,209},size={215,16}, disable=0
					SetVariable tb0_setwidth, disable=0
					SetVariable tb0_setheight, disable=0
					break
				case 2:	// property | object | notes
					SetVariable tb0_title, disable=1
					Checkbox tb0_cbtitle, disable=1
					Checkbox tb0_cbpretty, disable=0
					SetVariable tb0_replace, disable=1
					SetVariable tb0_property, disable=0
					Checkbox tb0_cbdate, disable=1
					Checkbox tb0_cbtime, disable=1
					Checkbox tb0_cbobject, disable=0
					PopupMenu tb0_object, disable=0
					SetVariable tb0_frame, disable=0
					SetVariable tb0_mode, disable=0
					Slider tb0_scale, disable=0
					Checkbox tb0_cbbw, disable=0
					Checkbox tb0_cbnotes, title="Notes"
					SetVariable tb0_notes0, value=gnotes0
					SetVariable tb0_notes1,disable=1
					SetVariable tb0_notes2,disable=1
					ValDisplay tb0_showscale, disable=0
					PopupMenu tb0_separator, disable=1
					SetVariable tb0_width, disable=0
					SetVariable tb0_height, disable=0
					break
				case 3:	// property | object | options
					SetVariable tb0_title, disable=1
					Checkbox tb0_cbtitle, disable=1
					Checkbox tb0_cbpretty, disable=0
					SetVariable tb0_replace, disable=1
					SetVariable tb0_property, disable=0
					Checkbox tb0_cbdate, disable=1
					Checkbox tb0_cbtime, disable=1
					Checkbox tb0_cbnotes, title="Template Input", value=1, disable=2
					Checkbox tb0_cbobject, value=1, disable=2
					SetVariable tb0_notes0, value=gtmpStr
					SetVariable tb0_notes1, title="attributes", bodyWidth=155, pos={8,193},size={215,16}, disable=0
					SetVariable tb0_notes2, title="values", bodyWidth=155, pos={8,209},size={215,16}, disable=0
					SetVariable tb0_notes0, disable=2
					PopupMenu tb0_separator, disable=1
					SetVariable tb0_width, disable=0
					SetVariable tb0_height, disable=0
					break
				case 4:	// property | text | options
					SetVariable tb0_title, disable=1
					Checkbox tb0_cbtitle, disable=1
					Checkbox tb0_cbpretty, disable=0
					SetVariable tb0_replace, disable=0
					SetVariable tb0_property, disable=0
					Checkbox tb0_cbdate, disable=1
					Checkbox tb0_cbtime, disable=1
					Checkbox tb0_cbobject, disable=1
					PopupMenu tb0_object, disable=1
					SetVariable tb0_frame, disable=1
					SetVariable tb0_mode, disable=1
					Slider tb0_scale, disable=1
					Checkbox tb0_cbbw, disable=1
					Checkbox tb0_cbnotes, title="Text", value=1, disable=2
					SetVariable tb0_notes0, value=gnotes0
					SetVariable tb0_notes1, title="attributes", bodyWidth=155, pos={8,193},size={215,16}, disable=0
					SetVariable tb0_notes2, title="values", bodyWidth=155, pos={8,209},size={215,16}, disable=0
					ValDisplay tb0_showscale, disable=1
					PopupMenu tb0_separator, disable=1
					SetVariable tb0_width, disable=1
					SetVariable tb0_height, disable=1
					break
			endswitch
			
			// modify for the given notebook
			
			pa.PopNum = gnotebk
			pa.eventCode = 99
			xmlnt_setnotebk(pa)
			break
	endswitch

	return 0
End

Function xmlnt_getobject(cba) : CheckBoxControl
	STRUCT WMCheckboxAction &cba

	switch( cba.eventCode )
		case 2: // mouse up
			PopupMenu tb0_object, disable=!cba.checked
			SetVariable tb0_frame, disable=!cba.checked
			SetVariable tb0_mode, disable=!cba.checked
			Checkbox tb0_cbbw, disable=!cba.checked
			ValDisplay tb0_showscale, disable=!cba.checked
			Slider tb0_scale, disable=!cba.checked
			SetVariable tb0_width, disable=!cba.checked
			SetVariable tb0_height, disable=!cba.checked
			break
	endswitch

	return 0
End

Function xmlnt_getnotes(cba) : CheckBoxControl
	STRUCT WMCheckboxAction &cba

	switch(cba.eventCode)
		case 2: // mouse up
			PopupMenu tb0_separator, disable=!cba.checked
			Button tb0_clearnotes, disable=!cba.checked
			SetVariable tb0_notes0, disable=!cba.checked
			SetVariable tb0_notes1, disable=!cba.checked
			SetVariable tb0_notes2, disable=!cba.checked
			break
	endswitch

	return 0
End

Function xmlnt_settitle(cba) : CheckBoxControl
	STRUCT WMCheckboxAction &cba

	switch(cba.eventCode)
		case 2: // mouse up
			SetVariable tb0_title, disable=!cba.checked
			break
	endswitch

	return 0
End

Function xmlnt_setreplace(sva) : SetVariableControl
	STRUCT WMSetVariableAction &sva

	NVAR greplace = root:Packages:XMLNoteTools:PanelGlobals:replace

	switch(sva.eventCode)
		case 1: // mouse up
		case 2: // Enter key
		case 3: // Live update
			if (sva.dval==0)
				PopupMenu tb0_location disable=0
				SetVariable tb0_notes1 disable=0
				SetVariable tb0_notes2 disable=0
				PopupMenu tb0_location,pos={163,230},size={152,21},title="at",disable=0, bodyWidth=0
				ControlInfo tb0_template
				if (V_value==1)
					Checkbox tb0_cbpretty disable=1
				else
					Checkbox tb0_cbpretty disable=0
				endif				
			else
				ControlInfo tb0_template
				if (V_value==3)
					greplace=0
					break
				endif
				PopupMenu tb0_location disable=1	
				SetVariable tb0_notes1 disable=1
				SetVariable tb0_notes2 disable=1
				PopupMenu tb0_location title="REPLACING",bodyWidth=1,disable=2
				Checkbox tb0_cbpretty disable=1
			endif
			break
	endswitch

	return 0
End

Function xmlnt_setnotebk(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	NVAR gnotebk =  root:Packages:XMLNoteTools:PanelGlobals:notebk
	SVAR gnewntbk = root:Packages:XMLNoteTools:PanelGlobals:newnotebk
	
	switch( pa.eventCode )
		case 2: // mouse up
			if (pa.popNum==gnotebk)
				break
			endif
		case 98: // new notebook just created
			gnotebk = pa.popNum
			PopupMenu tb0_notebook, mode=pa.popNum						
		case 99: // new template chosen
			if (gnotebk==1)
				PopupMenu tb0_location disable=1
				SetVariable tb0_newntbk disable=0
				SetVariable tb0_replace disable=1
				gnewntbk = ""
			else
				PopupMenu tb0_location disable=0
				SetVariable tb0_newntbk disable=1
				ControlInfo tb0_template
				SetVariable tb0_replace disable=!(V_value==4)
			endif
			break
	endswitch

	return 0
End

Function xmlnt_setmode(sva) : SetVariableControl
	STRUCT WMSetVariableAction &sva

	NVAR gmode = root:Packages:XMLNoteTools:PanelGlobals:mode

	switch( sva.eventCode )
		case 1: // mouse up
		case 2: // Enter key
		case 3: // Live update
			strswitch(IgorInfo(2))
				case "Windows":
					switch(sva.dval)
						case -5:
						case -4:
							break
						case -1:
						case -2:
						case -3:
							if (sva.dval<-2)
								gmode=0
							else
								gmode=-4
							endif
							break
						default:
							gmode = 0
							break
					endswitch				
					break
				default:
					switch(sva.dval)
						case -5:
						case -4:
						case -2:
							break
						case -3:
							gmode = -4
							break
						case -1:
							gmode = 0
							break
						case 3:
							gmode = 4
							break
						case 5:
						case 6:
						case 7:
							if (sva.dval<6)
								gmode = 8
							else
								gmode = 4
							endif
							break
						default:
							gmode = 0
							break
					endswitch				
					break
			endswitch
		endswitch

	return 0
End

Function xmlnt_psize(sva) : SetVariableControl
	STRUCT WMSetVariableAction &sva

	NVAR gysize = root:Packages:XMLNoteTools:PanelGlobals:ysize
	NVAR gxsize = root:Packages:XMLNoteTools:PanelGlobals:xsize

	switch( sva.eventCode )
		case 1: // mouse up
		case 2: // Enter key
		case 3: // Live update
			if (sva.dval<50)
				strswitch(sva.ctrlName)
					case "tb0_width":
						gxsize = 0
						break
					default:
						gysize = 0
						break
				endswitch
			endif
			break
	endswitch

	return 0
End

Function xmlnt_panelhook(infoStr)
	string infoStr

	NVAR gnotebk =  root:Packages:XMLNoteTools:PanelGlobals:notebk
	NVAR gobject = root:Packages:XMLNoteTools:PanelGlobals:object
	NVAR gdynamic = root:Packages:XMLNoteTools:PanelGlobals:dynamic
	
	string event=StringByKey("EVENT",infoStr)
	variable et
	
	strswitch(event)
		case "activate":
			if (gdynamic)
				PopupMenu tb0_notebook,mode=gnotebk,value= #"xmlnt_nlist()"
				PopupMenu tb0_object,mode=gobject,value= #"xmlnt_olist()"
			else
				ControlInfo tb0_notebook
				et = WhichListItem(S_value,xmlnt_nlist())
				PopupMenu tb0_notebook, popValue=S_value, mode=et+1
				ControlInfo tb0_object
				et = WhichListItem(S_value,xmlnt_olist())
				PopupMenu tb0_object, popValue=S_value, mode=et+1
			endif
			break
	endswitch
	
	return 0
	
end