﻿#pragma TextEncoding = "UTF-8"
#pragma rtGlobals=3
#pragma DefaultTab={3,20,4}
#pragma ModuleName=Hyperspec
#pragma IgorVersion=8
#pragma version=1.40

#include <Resize Controls>
#include <Varimax>

// Project Updater header
static constant kProjectID = 22408 // the project node on IgorExchange
static strconstant ksShortTitle = "Hyperspec" // the project short title on IgorExchange

//#define dynamic
//#define dev

static strconstant ksPackageName   = "Hyperspec"
static strconstant ksPrefsFileName = "acwHyperspec.bin"
static constant    kPrefsVersion   = 103

// execute hyperspec#demo() to create a demo dataset
// Create user funtions with the names HyperspecPixelTracesStyle() and
// HyperspecMapStyle() to be applied as style macros for the graph
// subwindows

menu "Windows"
	"Hyperspectral browser...", /Q, Hyperspec#NewHyperspecBrowser()
end

menu "New"
	"Hyperspectral browser...", /Q, Hyperspec#NewHyperspecBrowser()
end

#if IgorVersion() >= 9
#ifdef dynamic
menu "DataBrowserObjectsPopup", dynamic
	Hyperspec#ValidBrowserWave("Hyperspectral browser", 1, 1), /Q, Hyperspec#PlotDataBrowserSelection()
end
#else
menu "DataBrowserObjectsPopup"
	"Hyperspectral browser", /Q, Hyperspec#PlotDataBrowserSelection()
end
#endif
#endif

menu "TracePopup", dynamic
	"-"
	Hyperspec#InGraphMenu("Save All Pixel Traces", "HyperspecBrowser#PixelTraces"), /Q, Hyperspec#SaveAllPixelTraces()
	Hyperspec#InGraphMenu("Save Average...", "HyperspecBrowser#PixelTraces"), /Q, Hyperspec#SaveAveragePixelTrace()
	
	Hyperspec#ModeMenu("Draw ROI", 5, 0, "HyperspecBrowser#map"), /Q, Hyperspec#DrawPath(1, 0)
	Hyperspec#ModeMenu("Draw ROI linear", 5, 0, "HyperspecBrowser#map"), /Q, Hyperspec#DrawPath(1, 1)
	Hyperspec#ModeMenu("Stop ROI Drawing", 5, 1, "HyperspecBrowser#map"), /Q, Hyperspec#DrawPath(0, 0)
	
	Hyperspec#ModeMenu("Select Pixels", 6, 0, "HyperspecBrowser#map"), /Q, Hyperspec#PixelEdit(1)
	Hyperspec#ModeMenu("Stop Select Pixels", 6, 1, "HyperspecBrowser#map"), /Q, Hyperspec#PixelEdit(0)
	
	Hyperspec#ModeMenu("Save Path", 5, 1, "HyperspecBrowser#map"), /Q, Hyperspec#SavePath()
	
	Hyperspec#InGraphMenu("Draw Shapes", "HyperspecBrowser#map"), /Q, Hyperspec#SetIgorDrawingMode(1)
	Hyperspec#InGraphMenu("ROI from Shapes", "HyperspecBrowser#map"), /Q, Hyperspec#AveragePixelsROI($"", alert=1)
	Hyperspec#InGraphMenu("Clear Shapes", "HyperspecBrowser#map"), /Q, Hyperspec#DrawPath(0, 0); SetDrawLayer/W=HyperspecBrowser#map/K ProgFront
	Hyperspec#InGraphMenu("Save ROI...", "HyperspecBrowser#map"), /Q, Hyperspec#SaveROI()
	
	Hyperspec#InGraphMenu("Save map", "HyperspecBrowser#ProfileMap"), /Q, Hyperspec#SaveMap()
	Hyperspec#InGraphMenu("Save Profile", "HyperspecBrowser#Profiles"), /Q, Hyperspec#SaveProfile()
end

menu "GraphPopup", dynamic
	"-"
	Hyperspec#InGraphMenu("Save Average...", "HyperspecBrowser#PixelTraces"), /Q, Hyperspec#SaveAveragePixelTrace()
	Hyperspec#InGraphMenu("Save All Pixel Traces", "HyperspecBrowser#PixelTraces"), /Q, Hyperspec#SaveAllPixelTraces()
	Hyperspec#InGraphMenu("Save Profile", "HyperspecBrowser#Profiles"), /Q, Hyperspec#SaveProfile()
end

menu "GraphMarquee", dynamic
	MarqueeMenu("Save Average...", "HyperspecBrowser#map"), /Q, Hyperspec#SaveAverage()
	MarqueeMenu("Save ROI...", "HyperspecBrowser#map"), /Q, Hyperspec#SaveROI()
	MarqueeMenu("Truncate data...", "HyperspecBrowser#PixelTraces"), /Q, Hyperspec#TruncateData()
end

function/S MarqueeMenu(string s, string win)
	GetMarquee/Z
	if (cmpstr(S_marqueeWin, win))
		return ""
	endif
	return s
end

static function/S InGraphMenu(string str, string win)
	GetLastUserMenuInfo
	if (cmpstr(S_graphName, win))
		return ""
	endif
	return str
end

static function/S ModeMenu(string str, int mode, int active, string win)
	GetLastUserMenuInfo
	if (cmpstr(S_graphName, win))
		return ""
	endif
	wave/Z/SDFR=GetDFR() wStruct
	if (!WaveExists(wStruct))
		return ""
	endif
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	int good = active ? prefs.mode == mode : prefs.mode != mode
	return SelectString(good, "", str)
end

// *** package prefs ***

static structure PackagePrefs
	// general settings, 7 + 50 + 24 = 81
	uint16 version // 2
	char   mode // 0 = cursors, 4 = marquee, 5 = draw ROI path, 6 = select pixels, 7 = ROI shapes, 8 = edit ROI
	char   cursormode // 0 = crosshair, 1 = line
	char   mapmode // 0 = color table, 1 = RGB map
	uint16 options // 1: reverse spectrum, 2: reverse image y, 4: display all spectra,
						// 8: hide y axis, 16: autoscale y, 32: don't label axes, 64: line profile
						// 128: inverse mask, 256: swap line profile axes, 512: line profile map
	char   hlabel[25]
	char   vlabel[25]
	STRUCT RGBColor csrrgb  // 6 bytes, crosshair and line cursor color
	STRUCT RGBColor specrgb  // 6 bytes, spectrum color
	STRUCT point crosshair // 4 bytes, stored image pixel location for cursor
	STRUCT rect line // 8 bytes, stored image pixel locations for cursors
	
	// color table settings - 1 + 2 + 4 + 4 + 4 = 15
	char   ctoptions // 1 = auto min, 2 = auto max, 4 = log scale
	uint16 ctnum
	uint32 ctlayer // 4
	float  ctmin // minimum index value
	float  ctmax
	
	// rgb map settings // 3 * (1 + 4 + 4 + 4) = 39
	char   rgboptions[3] // 1 = use, 2 = auto, 4 = log
	uint32 rgblayer[3]
	float  rgbmax[3]
	float  rgbmin[3]
	
	float  fg // position of frame guide dividing spectra graph and line profile graph
	float  width		
	char   reserved[256 - 143] // 143 bytes used
endstructure

static function PrefsSetDefaults(STRUCT PackagePrefs &prefs)
	prefs.version    = kPrefsVersion
	prefs.cursormode = 0
	prefs.mapmode    = 0
	prefs.options    = 0 // 1: reverse spectrum, 2: reverse image y, 4: display all spectra,
						// 8: hide y axis, 16: autoscale y, 32: don't label axes
						// 64: line profile, 128: inverse mask
	prefs.hlabel     = "wavenumber"
	prefs.vlabel     = "absorbance"
	
	// white cursors
	prefs.csrrgb.red    = 65535
	prefs.csrrgb.green  = 65535
	prefs.csrrgb.blue   = 65535
	
	// black trace
	prefs.specrgb.red   = 0
	prefs.specrgb.green = 0
	prefs.specrgb.blue  = 0
	
	prefs.ctoptions     = 3 // auto min, auto max, no log scale
	prefs.ctnum         = 2 // yellowhot
	
	prefs.rgboptions[0] = 3 // red cursor active, auto limits, no log scale
	prefs.rgboptions[1] = 3 // green cursor active, auto limits, no log scale
	prefs.rgboptions[2] = 3 // blue cursor active, auto limits, no log scale
	
	prefs.mode          = 0 // 0 = cursors, 4 = marquee, 7 = ROI
	
	prefs.fg            = 0.5
	prefs.width         = 0
		
	int i
	for(i=0;i<(256-143);i+=1)
		prefs.reserved[i] = 0
	endfor
end

static function PrefsUpdate(STRUCT PackagePrefs &prefs)
	if (prefs.version != kPrefsVersion)
		PrefsSetDefaults(prefs)
		prefs.version = kPrefsVersion
	endif
end

static function PrefsLoad(STRUCT PackagePrefs &prefs)
	LoadPackagePreferences/MIS=1 ksPackageName, ksPrefsFileName, 0, prefs
	if (V_flag!=0 || V_bytesRead==0)
		PrefsSetDefaults(prefs)
	elseif (prefs.version != kPrefsVersion)
		// prefs definition may be changed
		PrefsUpdate(prefs)
	endif
	prefs.ctnum = limit(prefs.ctnum, 0, ItemsInList(CTabList())-1)
	prefs.cursormode = limit(prefs.cursormode, 0, 1)
end

static function PrefsSave()
	if (str2num(GetUserData("HyperspecBrowser", "", "version")) != GetThisVersion())
		return 0
	endif
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
end

static function SetBitValue(int var, int bit, int value)
	return (var&~2^bit) | value*2^bit
end

static function MakePrefsPanel()
	STRUCT PackagePrefs prefs
	wave/SDFR=GetDFR() wStruct
	StructGet prefs, wStruct
	
	int option = GetKeyState(0) & 2
	
	variable height = option ? 330 : 290, width = 260 // window coordinates
	variable expand = PanelResolution("HyperspecBrowser")/PanelResolution("")
	// exterior subwindows have controls drawn with host's expansion, but size is not recalculated.
	// NewPanel/EXP=(expand) doesn't change size, so:
	height *= expand
	width  *= expand
	NewPanel/K=1/N=HyperspecPrefsPanel/W=(0,height,width,height)/Host=HyperspecBrowser/EXT=0 as "Hyperspec Settings"
	ModifyPanel/W=HyperspecBrowser#HyperspecPrefsPanel noEdit=1
	variable y0 = 10
	CheckBox chkInvertMask pos={10,y0},size={100,16},title="Save inverted ROI mask waves",fSize=12,value=(prefs.options&256)>0, Proc=hyperspec#checkboxproc
	CheckBox chkInvertMask help={"Save ROI mask waves with positive exterior values for image processing compatibility"}
	y0 += 20
	CheckBox chkReverseImageY pos={10,y0},size={100,16},title="Reverse image y-axis",fSize=12,value=(prefs.options&2)>0, Proc=hyperspec#checkboxproc
	y0 += 20
	CheckBox chkShowAll pos={10,y0},size={100,16},title="Show traces for all pixels in ROI",fSize=12,value=(prefs.options&4)>0, Proc=hyperspec#checkboxproc
	y0 += 20
	CheckBox chkProfileMap pos={10,y0},size={100,16},title="Show profile map",fSize=12,value=(prefs.options&512)>0, Proc=hyperspec#checkboxproc
	CheckBox chkProfileMap help={"Replaces plot of spectra"}
	
	SetVariable svWidth pos={140,y0},size={100,16},title="Width",fSize=12,value=_NUM:prefs.width, Proc=hyperspec#SetvarProc
	SetVariable svWidth, help={"Sets width in pixels used by ImageLineProfile to create the profile map"},limits={0,inf,1}
	
	y0 += 20
	CheckBox chkProfile pos={10,y0},size={100,16},title="Show profile",fSize=12,value=(prefs.options&64)>0, Proc=hyperspec#checkboxproc
	CheckBox chkProfile help={"Display a plot of data values from the selected plane(s) vs distance for pixels that fall along pathway"}
	y0 += 20
	CheckBox chkSwapProfileAxes pos={10,y0},size={100,16},title="Swap profile axes",fSize=12,value=(prefs.options&128)>0, Proc=hyperspec#checkboxproc
	CheckBox chkSwapProfileAxes help={""}
	y0 += 20
	CheckBox chkAutoscale pos={10,y0},size={100,16},title="Autoscale trace plots",fSize=12,value=(prefs.options&16)>0, Proc=hyperspec#checkboxproc
	CheckBox chkAutoscale help={"Autoscale plot, otherwise limits are max and min values for selected plane"}
	y0 += 20
	CheckBox chkReverseSpectrum pos={10,y0},size={100,16},title="Reverse pixel-trace x-axis",fSize=12,value=prefs.options&1, Proc=hyperspec#checkboxproc
	CheckBox chkReverseSpectrum help={"Reverse the axis showing z-values for datacube"}
	
	if (option)
		y0 += 20
		CheckBox chkHideYaxis pos={10,y0},size={100,16},title="Suppress y-axis for trace plots",fSize=12,value=(prefs.options&8)>0, Proc=hyperspec#checkboxproc
		y0 += 20
		CheckBox chkLabels pos={10,y0},size={100,16},title="Don't label axes",fSize=12,value=(prefs.options&32)>0, Proc=hyperspec#checkboxproc
		CheckBox chkLabels help={"Suppress labelling of axes using labels defined below"}
	endif
	
	y0 += 20
	PopupMenu popTraceColor,pos={10,y0},size={50,20},mode=3,value=#"\"*COLORPOP*\"", Proc=Hyperspec#PopupProc
	PopupMenu popTraceColor,title="Default trace color",fsize=12,popColor=(prefs.specrgb.red,prefs.specrgb.green,prefs.specrgb.blue)
	y0 += 20
	PopupMenu popCursorColor,pos={10,y0},size={50,20},mode=3,value=#"\"*COLORPOP*\"", Proc=Hyperspec#PopupProc
	PopupMenu popCursorColor,title="Map cursor color",fsize=12,popColor=(prefs.csrrgb.red,prefs.csrrgb.green,prefs.csrrgb.blue)
	y0 += 20
	SetVariable svHlabel, pos={10,y0}, size={240,16}, title="Pixel-trace x-axis label", Proc=hyperspec#SetvarProc
	SetVariable svHlabel, help={"Label for spectrum independent variable"}, fSize=12, value=_STR:prefs.hlabel
	y0 += 20
	SetVariable svVlabel, pos={10,y0}, size={240,16}, title="Pixel-trace y-axis label", Proc=hyperspec#SetvarProc
	SetVariable svVlabel, help={"Label for spectrum dependent variable"}, fSize=12, value=_STR:prefs.vlabel
	y0 += 25
	Button btnDone, pos={198,y0}, size={50,22}, title="Done", Proc=hyperspec#ButtonProc
end

// returns truth that this procedure file has been updated since initialisation
static function CheckUpdated(int restart)
	if (str2num(GetUserData("HyperspecBrowser", "", "version")) != GetThisVersion())
		if (restart)
			DoAlert 0, "You have updated the package since this panel was created.\r\rThe package will restart to update the control panel."
			MakeHyperspecBrowser(GetImageWave())
		else
			DoAlert 0, "You have updated the package since this panel was created.\r\rPlease restart to continue."
		endif
		return 1
	endif
	return 0
end

static function demo()
	Make/O/N=(51,51,501) wDemo3d
	// make a broad Gaussian peak that increases in intensity towards the centre of the field of view
	wDemo3d  = 1000 * (60-sqrt((x-32)^2 + (y-32)^2)) * Gauss(z,350,400)
	// add a linear baseline whose slope varies depending on position
	wDemo3d += (0.0002 * x + 0.0005 * y) * z
	// add a peak at position 200
	wDemo3d += 500 * Gauss(z,200,20)
	// add a peak at position 150 to spectra that fall within a circular area
	wDemo3d += sqrt((x-20)^2 + (y-20)^2) < 10 ? 1000 * Gauss(z,150,20) : 0
	// add some noise
	wDemo3d += gnoise(1)
	SetScale/I x, 0, 1, wDemo3d
	SetScale/I y, 0, 1, wDemo3d
	SetScale/I z, 1000, 2000, wDemo3d
	MakeHyperspecBrowser(wDemo3d)
end

static function/S ValidBrowserWave(string str, int selectOne, int datacube)
	if (strlen(GetBrowserSelection(-1)) == 0 || (strlen(GetBrowserSelection(1)) && selectOne))
		return "" // Data Browser is not open/multiple objects selected
	endif
	wave/Z w1 = $GetBrowserSelection(0)
	if (WaveExists(w1) && WaveType(w1,1)==1)
		if (datacube == 0)
			return str
		elseif (WaveDims(w1)==3)
			return str
		endif
	endif
	return ""
end

static function Valid3DWave(wave/Z w)
	return WaveExists(w) && WaveType(w,1)==1 && WaveDims(w)==3
end

static function PlotDataBrowserSelection()
	wave/Z w = $GetBrowserSelection(0)
	if (WaveExists(w) && WaveDims(w)==3)
		MakeHyperspecBrowser(w)
	endif
end

static function ResetProfileMap(STRUCT PackagePrefs &prefs)
	
	if (!prefs.options & 512)
		return 0
	endif
	
	if (prefs.mode == 5)
		wave/Z/SDFR=GetDFR():DrawPath ROI_x, ROI_y
		return ResetPathProfileMap(prefs, ROI_x, ROI_y)
	endif
	
	Make/free xTrace={hcsr(A, "HyperspecBrowser#map"),hcsr(B, "HyperspecBrowser#map")}
	Make/free yTrace={vcsr(A, "HyperspecBrowser#map"),vcsr(B, "HyperspecBrowser#map")}
	
	WaveStats/Q/M=1 xTrace
	if (V_numNans)
		return 0
	endif
	
	DFREF dfSav = GetDataFolderDFR()
	SetDataFolder GetDFR()
	ImageLineProfile/SC/P=-2 srcWave=GetImageWave(), xWave=xTrace, yWave=yTrace, width=prefs.width
	wave M_ImageLineProfile
	wave W_LineProfileDisplacement
	MatrixTranspose M_ImageLineProfile
	variable delta = W_LineProfileDisplacement[numpnts(W_LineProfileDisplacement)-1]/(numpnts(W_LineProfileDisplacement)-1)
	SetScale/P y, 0, delta, M_ImageLineProfile
	
	if (IgorVersion()<9)
		wave w3d = GetImageWave()
		SetScale/P x, DimOffset(w3D, 2), DimDelta(w3D, 2), M_ImageLineProfile
	endif
		
	SetDataFolder dfSav
	string strTextBox = ""
	sprintf strTextBox, "%s Profile Map (%0.3g,%0.3g)-(%0.3g,%0.3g)", SelectString(prefs.mode&5, "Line", "Path"), xTrace[0], yTrace[0], xTrace[1], yTrace[1]
	if (WinType("HyperspecBrowser#ProfileMap"))
		TextBox/C/W=HyperspecBrowser#ProfileMap/N=textLineMap strTextBox
	endif
end

static function ResetPathProfileMap(STRUCT PackagePrefs &prefs, wave xPath, wave yPath)
	DFREF dfSav = GetDataFolderDFR()
	SetDataFolder GetDFR()
	
	wave w3D = GetImageWave()
	
	if (numpnts(xPath) < 2)
		Make/O/N=(2,2) M_ImageLineProfile = NaN
		variable z1 = IndexToScale(w3D, 0, 2)
		variable z2 = IndexToScale(w3D, DimSize(w3D, 2) - 1, 2)
		variable range = z2 - z1		
		SetScale/I x, z1 + range/4, z2-range/4, M_ImageLineProfile
	else
		ImageLineProfile/SC/P=-2 srcWave=w3D, xWave=xPath, yWave=yPath, width=prefs.width
		wave/Z M_ImageLineProfile
		wave/Z W_LineProfileDisplacement
		MatrixTranspose M_ImageLineProfile
		variable delta = W_LineProfileDisplacement[numpnts(W_LineProfileDisplacement)-1]/(numpnts(W_LineProfileDisplacement)-1)
		SetScale/P y, 0, delta, M_ImageLineProfile
		
		if (IgorVersion()<9)
			wave w3d = GetImageWave()
			SetScale/P x, DimOffset(w3D, 2), DimDelta(w3D, 2), M_ImageLineProfile
		endif
				
	endif
		
	SetDataFolder dfSav
	string strTextBox = ""
	sprintf strTextBox, "Path Profile Map"
	if (WinType("HyperspecBrowser#ProfileMap"))
		TextBox/C/W=HyperspecBrowser#ProfileMap/N=textLineMap strTextBox
	endif
end


static function ResetPathProfiles(STRUCT PackagePrefs &prefs, wave/Z xPath, wave/Z yPath)

	if (!(prefs.options&64 && WaveExists(xPath) && WaveExists(yPath)))
		return 0
	endif
	
	DFREF dfr = GetDFR()
	DFREF dfSav = GetDataFolderDFR()
	NewDataFolder/O/S dfr:pathprofiles
	int i
	
	if (prefs.mapmode == 0)
		if (numpnts(xPath) < 2)
			Make/free/N=1 W_ImageLineProfile = NaN, W_LineProfileDisplacement = 0
		else
			ImageLineProfile/SC/P=(prefs.ctlayer) srcWave=GetImageWave(), xWave=xPath, yWave=yPath
			wave/Z W_ImageLineProfile, W_LineProfileDisplacement
		endif
		Duplicate/O W_ImageLineProfile dfr:LineProfile
		Duplicate/O W_LineProfileDisplacement dfr:distanceCTline
	elseif (prefs.mapmode == 1)
		for(i=0;i<3;i++)
			if (prefs.rgboptions[i] & 1)
				if (numpnts(xPath) < 2)
					Make/free/N=1 W_ImageLineProfile = NaN, W_LineProfileDisplacement = NaN
				else
					ImageLineProfile/SC/P=(prefs.rgblayer[i]) srcWave=GetImageWave(), xWave=xPath, yWave=yPath
					wave W_ImageLineProfile, W_LineProfileDisplacement
				endif
				Duplicate/O W_ImageLineProfile dfr:$("LineProfile" + "RGB"[i])
				Duplicate/O W_LineProfileDisplacement dfr:distanceRGBline
			endif
		endfor
	endif
	
	DoUpdate/W=HyperspecBrowser#Profiles	
	SetDataFolder dfSav
	TextBox/C/W=HyperspecBrowser#Profiles/N=textLineProfile "Path Profile"
end

static function/DF getDFR()
	DFREF dfr = root:Packages:Hyperspec
	if (DataFolderRefStatus(dfr) != 1)
		return ResetDFR()
	endif
	return dfr
end

static function/DF ResetDFR()
	NewDataFolder/O root:Packages
	NewDataFolder/O root:Packages:Hyperspec
	DFREF dfr=root:Packages:Hyperspec
	return dfr
end

static function NewHyperspecBrowser()
	CreateBrowser/M Prompt="Select a 3D wave that represents a hyperspectral datacube"
	ModifyBrowser/M showWaves=1, showVars=0, showStrs=0, ShowInfo=0, showPlot=1
	ModifyBrowser/M showModalBrowser
	if (ItemsInList(S_BrowserList) == 1)
		wave/Z wSel = $StringFromList(0, S_BrowserList)
		MakeHyperspecBrowser(wSel)
	endif
	return 0
end

static function MakeHyperspecBrowser(wave/Z w3D)
	
	if (!Valid3DWave(w3D))
		NewHyperspecBrowser()
		return 0
	endif
			
	variable left, top, width, height, oldright, oldbottom
	int restart = 0
	width  = 385 // cpu
	height = 720 //cpu
	GetWindow/Z HyperspecBrowser wsizeRM // points, even for a panel
	if (V_flag == 0)
		left      = point2cpu(v_left)
		top       = point2cpu(v_top)
		oldright  = point2cpu(v_right)
		oldbottom = point2cpu(v_bottom)
		restart   = 1
	else
		GetMouse // pixels
		left = v_left - 200 // who knows
		top  = v_top - 300
	endif
	
	if (WinType("HyperspecBrowser") == 7)
		PrefsSave()
	endif
	KillWindow/Z HyperspecBrowser // saves prefs... or does it?
	STRUCT PackagePrefs prefs
	PrefsLoad(prefs)
	
	prefs.mode = 0
	prefs.fg   = 0.5 // default position for PixelTraces/Profiles divider
	
	// clear deprecated options
	prefs.ctoptions = SetBitValue(prefs.ctoptions, 3, 0)
	prefs.ctoptions = SetBitValue(prefs.ctoptions, 4, 0)
	
	prefs.options   = SetBitValue(prefs.options, 3, 0) // hide y
	prefs.options   = SetBitValue(prefs.options, 5, 0) // don't label
	
	// reset data folder
	DFREF dfr = GetDFR()
	Make/O/N=0/B/U dfr:wStruct/wave=wStruct
	
	Make/O/B/U/N=(DimSize(w3D, 0),DimSize(w3D, 1)) dfr:wMask/wave=wMask
	Make/O/N=(DimSize(w3D, 0),DimSize(w3D, 1),4) dfr:wDisplayMaskRGBA/wave=wDisplayMaskRGBA
	CopyScales w3D, wDisplayMaskRGBA, wMask
	wDisplayMaskRGBA[][][0,2] = 65535 // white
	wDisplayMaskRGBA[][][3]   = 0 // transparent
	
	Make/O/U/N=(DimSize(w3D, 0),DimSize(w3D, 1),3) dfr:wRGB/wave=wRGB
	CopyScales w3D, wRGB // sync x and y scales
	SetScale/P z, 0, 1, wRGB
	
	Make/O/N=(DimSize(w3D, 2))/Y=(WaveType(w3D)) dfr:spectrum/wave=spectrum
	SetScale/P x, DimOffset(w3D, 2), DimDelta(w3D, 2), spectrum
	
	Make/O/N=(2,2) dfr:wLine/wave=wLine
	Make/O/N=0 dfr:distanceCTline/wave=distanceCTline
	Make/O/N=0/Y=(WaveType(w3D)) dfr:LineProfile/wave=LineProfile
	Make/O/N=0 dfr:distanceRGBline/wave=distanceRGBline
	Make/O/N=0/Y=(WaveType(w3D)) dfr:LineProfileR/wave=LineProfileR
	Make/O/N=0/Y=(WaveType(w3D)) dfr:LineProfileG/wave=LineProfileG
	Make/O/N=0/Y=(WaveType(w3D)) dfr:LineProfileB/wave=LineProfileB
		
	string strColorTable = StringFromList(prefs.ctnum, CTabList())
	int logScale = prefs.ctoptions & 4 ? 1 : 0
		
	variable pmax = DimSize(w3D,0) - 1
	variable qmax = DimSize(w3D,1) - 1
	variable rmax = DimSize(w3D,2) - 1
	
	variable planeStart = IndexToScale(w3D, 0, 2)
	variable planeEnd   = IndexToScale(w3D, rMax, 2)
	variable planeMin   = min(planeStart, planeEnd)
	variable planeMax   = max(planeStart, planeEnd)
	
	if (prefs.ctlayer==0 || prefs.ctlayer==rMax || prefs.ctlayer!=limit(prefs.ctlayer, 0, rMax))
		prefs.ctlayer = rMax / 2
	endif
	variable ctplane = IndexToScale(w3D, prefs.ctlayer, 2)
			
	if (prefs.crosshair.h != limit(prefs.crosshair.h, 1, pMax))
		prefs.crosshair.h = pMax / 2
	endif
	if (prefs.crosshair.v != limit(prefs.crosshair.v, 1, qMax))
		prefs.crosshair.v = qMax / 2
	endif
	if (prefs.line.left != limit(prefs.line.left, 1, pMax))
		prefs.line.left = pMax / 4
	endif
	if (prefs.line.right != limit(prefs.line.right, 1, pMax))
		prefs.line.right = pMax * 3 / 4
	endif
	if (prefs.line.top != limit(prefs.line.top, 1, qMax))
		prefs.line.top = qMax / 4
	endif
	if (prefs.line.bottom != limit(prefs.line.bottom, 1, qMax))
		prefs.line.bottom = qMax * 3 / 4
	endif
	
	variable x0 = IndexToScale(w3D, prefs.line.left, 0)
	variable x1 = IndexToScale(w3D, prefs.line.right, 0)
	variable y0 = IndexToScale(w3D, prefs.line.top, 0)
	variable y1 = IndexToScale(w3D, prefs.line.bottom, 0)
	
	wLine = {{x0, x1}, {y0, y1}}
	
	WaveStats/Q/M=1 w3D
	variable vGlobalMin = v_min
	variable vGlobalMax = v_max
	
	WaveStats/Q/M=1/RMD=[][][prefs.ctlayer] w3D
	prefs.ctmin = prefs.ctoptions & 1 ? v_min : prefs.ctmin
	prefs.ctmax = prefs.ctoptions & 2 ? v_max : prefs.ctmax
		
	int i
	for (i=0;i<3;i++)
		WaveStats/Q/M=1/RMD=[][][prefs.rgblayer[i]] w3D
		prefs.rgbmin[i]   = prefs.rgboptions[i]&2 ? v_min : prefs.rgbmin[i]
		prefs.rgbmax[i]   = prefs.rgboptions[i]&2 ? v_max : prefs.rgbmax[i]
		prefs.rgblayer[i] = prefs.rgblayer[i]>0 && prefs.rgblayer[i]<rMax && prefs.rgblayer[i]==limit(prefs.rgblayer[i], 0, rMax) ? prefs.rgblayer[i] : (i+1)/4 * rMax
	endfor
	
	// for now we keep this here, so that fucntions that pull prefs from package folder will work
	StructPut prefs, wStruct
		
	// create the panel
	NewPanel/W=(left,top,left+width,top+height)/K=1/N=HyperspecBrowser as "Hyperspectral Browser"
	ModifyPanel/W=HyperspecBrowser cbRGB=(65535,65535,65534)
		
	// add controls
	
	// top of panel
	SetVariable svDatacube,win=HyperspecBrowser,pos={5,5},size={330,14},title="",value=_STR:GetWavesDataFolder(w3D, 2), Proc=Hyperspec#SetvarProc
	SetVariable svDatacube,win=HyperspecBrowser, help={"Select a 3D wave that represents a hyperspectral datacube"}
	SetVariable svDatacube,win=HyperspecBrowser, userdata=GetWavesDataFolder(w3D, 2)
	Button btnInfo, win=HyperspecBrowser, pos={345,5}, size={15,15}, Proc=hyperspec#ButtonProc, title=""
	Button btnInfo, win=HyperspecBrowser, help={"Show Info"}, Picture=hyperspec#pInfo
	Button btnSettings, win=HyperspecBrowser, pos={365,5}, size={15,15}, Proc=hyperspec#ButtonProc, title=""
	Button btnSettings, win=HyperspecBrowser, help={"Change Settings"}, Picture=hyperspec#pCog
	
	// tabs
	variable xpos = 10
	variable controlsY0 = height - 175
	variable ypos = controlsY0
	
	TabControl tabs, pos={xpos,ypos}, size={260,165}, tabLabel(0)="Color Table", tabLabel(1)="RGB Map", value=prefs.mapmode, Proc=Hyperspec#TabControlProc
	
// *** Controls for color table tab ***
	xpos  = 20
	ypos += 30
	SetVariable svPlane_ct,win=HyperspecBrowser,pos={28, ypos},size={200,180}, value=_NUM:ctplane,Proc=Hyperspec#SetvarProc
	SetVariable svPlane_ct,win=HyperspecBrowser,fSize=12,limits={planeMin,planeMax,abs(DimDelta(w3D,2))}
	
	if (strlen(prefs.hlabel))
		SetVariable svPlane_ct,win=HyperspecBrowser,title=prefs.hlabel
	else
		SetVariable svPlane_ct,win=HyperspecBrowser,title="Layer"
	endif
	ypos += 25
	PopupMenu popCT_ct,win=HyperspecBrowser,pos={xpos,ypos},size={160,20}, Proc=Hyperspec#PopupProc
	PopupMenu popCT_ct,win=HyperspecBrowser,mode=prefs.ctnum+1,value=#"\"*COLORTABLEPOPNONAMES*\"", bodyWidth=160
	CheckBox chkLog_ct,pos={200,ypos},size={42,16},title="log",fSize=12,value=logScale,Proc=Hyperspec#checkBoxProc
	CheckBox chkLog_ct, help={"Use logarithmic color scale"}
	ypos += 20
	CheckBox chkAutoMin_ct,pos={200,ypos},size={42,16},title="auto",fSize=12,value=(prefs.ctoptions&1),Proc=Hyperspec#checkBoxProc
	CheckBox chkAutoMin_ct,help={"Set minimum color table value to lowest value in selected layer"}
	SetVariable svMin_ct,pos={32,ypos},size={160,18},bodyWidth=70,value=_NUM:prefs.ctmin, disable=(prefs.ctoptions&1) ? 2 : 0
	SetVariable svMin_ct,title="First color index",fSize=12,limits={-Inf,Inf,0},Proc=Hyperspec#SetvarProc
	ypos += 20
	Slider sliderMin_ct pos={xpos,ypos},size={230,30}, vert=0,side=0,limits={prefs.ctmin,prefs.ctmax,0},ticks=0,value=prefs.ctmin
	Slider sliderMin_ct disable=prefs.ctoptions&1 ? 2 : 0, Proc=Hyperspec#SliderProc
	ypos += 20
	CheckBox chkAutoMax_ct,pos={200,ypos},size={42,16},title="auto",fSize=12,value=(prefs.ctoptions&2),Proc=Hyperspec#checkBoxProc
	CheckBox chkAutoMax_ct,help={"Set maximum color table value to highest value in selected layer"}
	SetVariable svMax_ct,pos={32,ypos},size={160,18},bodyWidth=70,value=_NUM:prefs.ctmax, disable=(prefs.ctoptions&2) ? 2 : 0
	SetVariable svMax_ct,title="Last color index ",fSize=12,limits={-Inf,Inf,0},Proc=Hyperspec#SetvarProc
	ypos += 20
	Slider sliderMax_ct pos={xpos,ypos},size={230,30}, vert=0,side=0,limits={prefs.ctmin,prefs.ctmax,0},ticks=0,value=prefs.ctmax
	Slider sliderMax_ct disable=prefs.ctoptions&2 ? 2 : 0, Proc=Hyperspec#SliderProc
	
// *** Controls for RGB map tab ***
	xpos = 20
	ypos = 565
	ypos = controlsY0 + 30

	TitleBox TitlePlane_rgb,pos={xpos+35,ypos},size={95,170},frame=0,title="plane",fsize=12,disable=1
	TitleBox TitleMin_rgb,pos={xpos+90,ypos},size={95,170},frame=0,title="iMin",fsize=12,disable=1
	TitleBox TitleMax_rgb,pos={xpos+145,ypos},size={95,170},frame=0,title="iMax",fsize=12,disable=1
	TitleBox TitleAuto_rgb,pos={xpos+190,ypos},size={95,170},frame=0,title="auto",fsize=12,disable=1
	TitleBox TitleLog_rgb,pos={xpos+219,ypos},size={95,170},frame=0,title="log",fsize=12,disable=1
	ypos += 20
	CheckBox chkR_rgb, align=0, pos={xpos,ypos},size={42,16},title="R",fColor=(65535,0,0),fSize=12,side=1,value=prefs.rgboptions[0]&1,Proc=Hyperspec#checkBoxProc,disable=1
	CheckBox chkR_rgb, help={"Fill red values in RGB map"}
	ypos += 2
	SetVariable svplane0_rgb,win=HyperspecBrowser,pos={xpos+30, ypos},size={50,16}, value=_NUM:IndexToScale(w3D, prefs.rgblayer[0], 2),Proc=Hyperspec#SetvarProc,disable=1
	SetVariable svplane0_rgb,win=HyperspecBrowser,limits={planeMin,planeMax,abs(DimDelta(w3D,2))}
	SetVariable svmin0_rgb,win=HyperspecBrowser,pos={xpos+85, ypos},size={50,16}, value=_NUM:prefs.rgbmin[0],Proc=Hyperspec#SetvarProc,disable=1
	SetVariable svmax0_rgb,win=HyperspecBrowser,pos={xpos+140, ypos},size={50,16}, value=_NUM:prefs.rgbmax[0],Proc=Hyperspec#SetvarProc,disable=1
	ypos -= 2
	CheckBox chkauto0_rgb,pos={xpos+197,ypos},size={42,16},title=" ",fSize=12,value=prefs.rgboptions[0]&2 ,Proc=Hyperspec#checkBoxProc,disable=1
	CheckBox chkauto0_rgb,help={"Set min and max red values to lowest and highest values in selected plane"}
	CheckBox chklog0_rgb,pos={xpos+220,ypos},size={42,16},title=" ",fSize=12,value=prefs.rgboptions[0]&4,Proc=Hyperspec#checkBoxProc,disable=1
	ypos += 30
	CheckBox chkG_rgb, align=0, pos={xpos,ypos},size={42,16},title="G",fColor=(0,65535,0),fSize=12,side=1,value=prefs.rgboptions[1]&1,Proc=Hyperspec#checkBoxProc,disable=1
	CheckBox chkG_rgb, help={"Fill green values in RGB map"}
	ypos += 2
	SetVariable svplane1_rgb,win=HyperspecBrowser,pos={xpos+30, ypos},size={50,16}, value=_NUM:IndexToScale(w3D, prefs.rgblayer[1], 2),Proc=Hyperspec#SetvarProc,disable=1
	SetVariable svplane1_rgb,win=HyperspecBrowser,limits={planeMin,planeMax,abs(DimDelta(w3D,2))}
	SetVariable svmin1_rgb,win=HyperspecBrowser,pos={xpos+85, ypos},size={50,16}, value=_NUM:prefs.rgbmin[1],Proc=Hyperspec#SetvarProc,disable=1
	SetVariable svmax1_rgb,win=HyperspecBrowser,pos={xpos+140, ypos},size={50,16}, value=_NUM:prefs.rgbmax[1],Proc=Hyperspec#SetvarProc,disable=1
	ypos -= 2
	CheckBox chkauto1_rgb,pos={xpos+197,ypos},size={42,16},title=" ",fSize=12,value=prefs.rgboptions[1]&2,Proc=Hyperspec#checkBoxProc,disable=1
	CheckBox chkauto1_rgb,help={"Set min and max green values to lowest and highest values in selected plane"}
	CheckBox chklog1_rgb,pos={xpos+220,ypos},size={42,16},title=" ",fSize=12,value=prefs.rgboptions[1]&4,Proc=Hyperspec#checkBoxProc,disable=1
	ypos += 30
	CheckBox chkB_rgb, align=0, pos={xpos,ypos},size={42,16},title="B",fColor=(0,0,65535),fSize=12,side=1,value=prefs.rgboptions[2]&1,Proc=Hyperspec#checkBoxProc,disable=1
	CheckBox chkB_rgb, help={"Fill blue values in RGB map"}
	ypos += 2
	SetVariable svplane2_rgb,win=HyperspecBrowser,pos={xpos+30, ypos},size={50,16}, value=_NUM:IndexToScale(w3D, prefs.rgblayer[2], 2),Proc=Hyperspec#SetvarProc,disable=1
	SetVariable svplane2_rgb,win=HyperspecBrowser,limits={planeMin,planeMax,abs(DimDelta(w3D,2))}
	SetVariable svmin2_rgb,win=HyperspecBrowser,pos={xpos+85, ypos},size={50,16}, value=_NUM:prefs.rgbmin[2],Proc=Hyperspec#SetvarProc,disable=1
	SetVariable svmax2_rgb,win=HyperspecBrowser,pos={xpos+140, ypos},size={50,16}, value=_NUM:prefs.rgbmax[2],Proc=Hyperspec#SetvarProc,disable=1
	ypos -= 2
	CheckBox chkauto2_rgb,pos={xpos+197,ypos},size={42,16},title=" ",fSize=12,value=prefs.rgboptions[2]&2,Proc=Hyperspec#checkBoxProc,disable=1
	CheckBox chkauto2_rgb,help={"Set min and max green values to lowest and highest values in selected plane"}
	CheckBox chklog2_rgb,pos={xpos+220,ypos},size={42,16},title=" ",fSize=12,value=prefs.rgboptions[2]&4,Proc=Hyperspec#checkBoxProc,disable=1
	
	// cursor controls
	ypos = controlsY0 +10
	xpos = 274
	GroupBox grpPixel,pos={xpos,ypos},size={95,155},title="",fsize=12
	xpos += 10
	ypos += 7
	TitleBox TitleCsr,pos={xpos,ypos},size={80,10},frame=0,title="Cursor mode",fsize=12
	ypos += 20
	CheckBox chkCrosshair,pos={xpos,ypos},size={42,16},title="Crosshair",fSize=12,mode=1,value=prefs.cursormode==0,Proc=Hyperspec#checkBoxProc
	ypos += 20
	CheckBox chkLine,pos={xpos,ypos},size={42,16},title="Line",fSize=12,mode=1,value=prefs.cursormode==1,Proc=Hyperspec#checkBoxProc
	ypos += 22
	Button btnZap,pos={xpos+10,ypos},size={50.00,22.00},Proc=Hyperspec#buttonproc
	Button btnZap,title="Zap", help={"Sets crosshair pixel to average of surrounding pixels"}
	ypos += 25
	Button btnCopy,pos={xpos+10,ypos},size={50.00,22.00},Proc=Hyperspec#buttonproc
	Button btnCopy,title="Copy", help={"Copy crosshair pixel"}
	ypos += 25
	Button btnPaste,pos={xpos+10,ypos},size={50.00,22.00},Proc=Hyperspec#buttonproc
	Button btnPaste,title="Paste", help={"Paste copied pixel at crosshair location"}
		
// *** map ***
	
	Preferences/Q 0
	variable oldPrefsState = v_flag
	
	spectrum = w3D[prefs.crosshair.h][prefs.crosshair.v][p]
	string strTextBox = ""
		
	DefineGuide/W=HyperspecBrowser FGT={FT,30}, FGM={FT,0.25,FB},FGB={FB,-185}, FGHDIV={FR, -0}//,FGR={FL,0.9,FR}
	Display/W=(21,178,447,433)/FG=(FL,FGM,FR,FGB)/HOST=HyperspecBrowser/N=map
	
	AppendImage/W=HyperspecBrowser#map w3D
	ModifyImage/W=HyperspecBrowser#map ''#0 ctab={*,*,$strColorTable,0}, plane=prefs.ctlayer, ctabAutoscale=3
	ModifyGraph/W=HyperspecBrowser#map margin(left)=8,margin(bottom)=1,margin(right)=38 // or zero for auto margin, allows more space for external color scale
	ModifyGraph/W=HyperspecBrowser#map tick=3,mirror=1,noLabel=2,standoff=0
	AppendImage/W=HyperspecBrowser#map wDisplayMaskRGBA
	ColorScale/W=HyperspecBrowser#map/C/N=scale/F=0/A=RC/X=0/Y=0.00/E/B=1 image=$NameOfWave(w3D), heightPct=100, width=10, tickLen=4, minor=1
	
	if (prefs.options & 2) // reverse image Y
		SetAxis/A/R/W=HyperspecBrowser#map left
	endif
	
	// crosshair cursor
	Cursor/W=HyperspecBrowser#map/P/S=0/I/H=1/C=(prefs.csrrgb.red,prefs.csrrgb.green,prefs.csrrgb.blue)/N=1 C $NameOfWave(w3D) prefs.crosshair.h, prefs.crosshair.v
	
	if (prefs.mapmode==0 && (DimOffset(w3D, 2)!=0 || DimDelta(w3D, 2)!=1))
		sprintf strTextBox, "%s[][][%g], z = %g", NameOfWave(w3D), prefs.ctlayer, ctplane//, prefs.hlabel
	else
		sprintf strTextBox, "%s[][][%g]", NameOfWave(w3D), prefs.ctlayer
	endif
	TextBox/C/N=text0/F=0/A=LT/X=8/Y=0/E=2 strTextBox
		
	#if (exists("HyperspecMapStyle") == 5 || exists("HyperspecMapStyle") == 6)
		Execute/Z "HyperspecMapStyle()"
	#endif

// *** Spectra Graph ***
	
	// rather than plotting w3D[xpnt][ypnt][*] we plot a 1D wave so that it can be easily exported
	Display/FG=(FL,FGT,FGHDIV,FGM)/HOST=HyperspecBrowser/N=PixelTraces spectrum/TN=spectrum
	ModifyGraph/W=HyperspecBrowser#PixelTraces rgb(spectrum)=(prefs.specrgb.red,prefs.specrgb.green,prefs.specrgb.blue), mode(spectrum)=0
	ModifyGraph/W=HyperspecBrowser#PixelTraces standoff=0,tick=2,btLen=3 //, margin(right)=8
	
	if (prefs.options & 16) // autoscale
		SetAxis/W=HyperspecBrowser#PixelTraces/A=2 left
	else
		SetAxis/W=HyperspecBrowser#PixelTraces left, vGlobalMin, vGlobalMax
	endif
	
	if (prefs.options & 8)
		ModifyGraph/W=HyperspecBrowser#PixelTraces tick(left)=3,noLabel(left)=2,axThick(left)=0//, margin(left)=8
	endif
	
	if (prefs.options & 32)
		Label/W=HyperspecBrowser#PixelTraces left, ""
		Label/W=HyperspecBrowser#PixelTraces bottom, ""
	else
		Label/W=HyperspecBrowser#PixelTraces left, prefs.vlabel
		Label/W=HyperspecBrowser#PixelTraces bottom, prefs.hlabel
	endif
	
	if (prefs.options & 1)
		SetAxis/A/R/W=HyperspecBrowser#PixelTraces bottom
	else
		SetAxis/A/W=HyperspecBrowser#PixelTraces bottom
	endif
	
	// set textbox position
	sprintf strTextBox, "%s[%g][%g]", NameOfWave(w3D), prefs.crosshair.h, prefs.crosshair.v
	TextBox/C/N=text1/F=0/A=LT/X=8/Y=0/E=2 strTextBox
	
	ModifyGraph/W=HyperspecBrowser#PixelTraces mirror=1, axisOnTop=1
	
	#if (exists("HyperspecPixelTracesStyle") == 5 || exists("HyperspecPixelTracesStyle") == 6)
		Execute/Z "HyperspecPixelTracesStyle()"
	#endif
	
// *** Line Profile Graph ***
	
	if (prefs.cursormode==1 && prefs.options&64)
		DefineGuide/W=HyperspecBrowser FGHDIV={FL, prefs.fg, FR}
	endif
	
	Display/FG=(FGHDIV,FGT,FR,FGM)/HOST=HyperspecBrowser/N=Profiles LineProfile vs distanceCTline
	ModifyGraph/W=HyperspecBrowser#Profiles standoff=0,tick=2,btLen=3 //, margin(right)=8
	ModifyGraph/W=HyperspecBrowser#Profiles rgb(LineProfile)=(prefs.specrgb.red,prefs.specrgb.green,prefs.specrgb.blue), mode(LineProfile)=0
	ModifyGraph/W=HyperspecBrowser#Profiles hideTrace(LineProfile)=(prefs.mapmode!=0)
	
	for (i=0;i<3;i++)
		string strProfileWave = "LineProfile" + "RGB"[i]
		AppendToGraph/W=hyperspecbrowser#Profiles dfr:$strProfileWave vs distanceRGBline
		ModifyGraph/Z/W=HyperspecBrowser#Profiles hideTrace($strProfileWave)=!(prefs.rgboptions[i]&1 && prefs.mapmode==1)
		ModifyGraph/W=HyperspecBrowser#Profiles rgb($strProfileWave)=(i==0?65535:0, i==1?65535:0, i==2?65535:0)
	endfor
	
	SetAxis/A/W=HyperspecBrowser#Profiles bottom
	if (prefs.options & 16) // autoscale
		SetAxis/W=HyperspecBrowser#Profiles/A=2 left
	else
		SetAxis/W=HyperspecBrowser#Profiles left, vGlobalMin, vGlobalMax
	endif
	
	ModifyGraph/W=HyperspecBrowser#Profiles mirror=1, axisOnTop=1
	
	if (prefs.options & 8)
		ModifyGraph/W=HyperspecBrowser#Profiles tick(left)=3,noLabel(left)=2,axThick(left)=0//, margin(left)=8
	endif
	
	if (prefs.options & 32)
		Label/W=HyperspecBrowser#Profiles left, ""
		Label/W=HyperspecBrowser#Profiles bottom, ""
	else
		Label/W=HyperspecBrowser#Profiles left, prefs.vlabel
		Label/W=HyperspecBrowser#Profiles bottom, "A <- distance -> B"
	endif
	
	ModifyGraph/W=HyperspecBrowser#Profiles swapXY=(prefs.options&128) > 0
	DoUpdate/W=HyperspecBrowser#Profiles
	ResetLineProfiles(prefs, w3D)
	
	TextBox/W=HyperspecBrowser#Profiles/C/N=textLineProfile/F=0/A=LT/X=8/Y=0/E=2 "Path Profile"
	
	Preferences/Q oldPrefsState
	StructPut prefs, wStruct
		
	MapMode(prefs)
	ShowCursors(prefs, 1)
	ReplotPixelTraces(prefs)
	ShowProfileMap(prefs) // optionally create line map graph
		
	SetWindow HyperspecBrowser userdata(version)=num2str(GetThisVersion())
	SetWindow HyperspecBrowser hook(hHyperspec)=Hyperspec#hookHyperspec
	#if exists("ScrollToZoom#hookScrollToZoom") == 6
	SetWindow HyperspecBrowser hook(hScrollToZoom)=ScrollToZoom#hookScrollToZoom
	#endif
	
	SetActiveSubwindow ##
	SetWindow HyperspecBrowser activeChildFrame=0
	
	SetVariable svDatacube,userdata(ResizeControlsInfo)=A"!!,?X!!#9W!!#B_!!#;mz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
	SetVariable svDatacube,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#u:Duafnzzzzzzzzzzz"
	SetVariable svDatacube,userdata(ResizeControlsInfo)+=A"zzz!!#u:Duafnzzzzzzzzzzzzzz!!!"
	Button btnInfo,userdata(ResizeControlsInfo)=A"!!,HgJ,hj-!!#<(!!#<(z!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	Button btnInfo,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#u:Duafnzzzzzzzzzzz"
	Button btnInfo,userdata(ResizeControlsInfo)+=A"zzz!!#u:Duafnzzzzzzzzzzzzzz!!!"
	Button btnSettings,userdata(ResizeControlsInfo)=A"!!,HqJ,hj-!!#<(!!#<(z!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	Button btnSettings,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#u:Duafnzzzzzzzzzzz"
	Button btnSettings,userdata(ResizeControlsInfo)+=A"zzz!!#u:Duafnzzzzzzzzzzzzzz!!!"

	TabControl tabs,userdata(ResizeControlsInfo)=A"!!,A.!!#Cm5QF0'!!#A4z!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
	TabControl tabs,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	TabControl tabs,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"

	TitleBox TitlePlane_rgb,userdata(ResizeControlsInfo)=A"!!,Dk!!#Ct^]6^8!!#A9z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	TitleBox TitlePlane_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	TitleBox TitlePlane_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	TitleBox TitleMin_rgb,userdata(ResizeControlsInfo)=A"!!,FA!!#Ct^]6^8!!#A9z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	TitleBox TitleMin_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	TitleBox TitleMin_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	TitleBox TitleMax_rgb,userdata(ResizeControlsInfo)=A"!!,G5!!#Ct^]6^8!!#A9z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	TitleBox TitleMax_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	TitleBox TitleMax_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	TitleBox TitleAuto_rgb,userdata(ResizeControlsInfo)=A"!!,Gb!!#Ct^]6^8!!#A9z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	TitleBox TitleAuto_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	TitleBox TitleAuto_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	TitleBox TitleLog_rgb,userdata(ResizeControlsInfo)=A"!!,H*!!#Ct^]6^8!!#A9z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	TitleBox TitleLog_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	TitleBox TitleLog_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkR_rgb,userdata(ResizeControlsInfo)=A"!!,BY!!#D$^]6\\L!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chkR_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkR_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svplane0_rgb,userdata(ResizeControlsInfo)=A"!!,DW!!#D%5QF,A!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svplane0_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svplane0_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svmin0_rgb,userdata(ResizeControlsInfo)=A"!!,F7!!#D%5QF,A!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svmin0_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svmin0_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svmax0_rgb,userdata(ResizeControlsInfo)=A"!!,G0!!#D%5QF,A!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svmax0_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svmax0_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkauto0_rgb,userdata(ResizeControlsInfo)=A"!!,Gi!!#D$^]6\\L!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chkauto0_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkauto0_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chklog0_rgb,userdata(ResizeControlsInfo)=A"!!,H+!!#D$^]6\\L!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chklog0_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chklog0_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkG_rgb,userdata(ResizeControlsInfo)=A"!!,BY!!#D,5QF,!!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chkG_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkG_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svplane1_rgb,userdata(ResizeControlsInfo)=A"!!,DW!!#D,^]6\\l!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svplane1_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svplane1_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svmin1_rgb,userdata(ResizeControlsInfo)=A"!!,F7!!#D,^]6\\l!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svmin1_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svmin1_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svmax1_rgb,userdata(ResizeControlsInfo)=A"!!,G0!!#D,^]6\\l!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svmax1_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svmax1_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkauto1_rgb,userdata(ResizeControlsInfo)=A"!!,Gi!!#D,5QF,!!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chkauto1_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkauto1_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chklog1_rgb,userdata(ResizeControlsInfo)=A"!!,H+!!#D,5QF,!!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chklog1_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chklog1_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkB_rgb,userdata(ResizeControlsInfo)=A"!!,BY!!#D3^]6\\L!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chkB_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkB_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svplane2_rgb,userdata(ResizeControlsInfo)=A"!!,DW!!#D45QF,A!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svplane2_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svplane2_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svmin2_rgb,userdata(ResizeControlsInfo)=A"!!,F7!!#D45QF,A!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svmin2_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svmin2_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svmax2_rgb,userdata(ResizeControlsInfo)=A"!!,G0!!#D45QF,A!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svmax2_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svmax2_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkauto2_rgb,userdata(ResizeControlsInfo)=A"!!,Gi!!#D3^]6\\L!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chkauto2_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkauto2_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chklog2_rgb,userdata(ResizeControlsInfo)=A"!!,H+!!#D3^]6\\L!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chklog2_rgb,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chklog2_rgb,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkCrosshair,userdata(ResizeControlsInfo)=A"!!,HI!!#D!J,hor!!#<(z!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	CheckBox chkCrosshair,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkCrosshair,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkLine,userdata(ResizeControlsInfo)=A"!!,HI!!#D&J,hn]!!#<(z!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	CheckBox chkLine,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkLine,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"

	SetVariable svPlane_ct,userdata(ResizeControlsInfo)=A"!!,CD!!#Ct^]6_m!!#<pz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svPlane_ct,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svPlane_ct,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	PopupMenu popCT_ct,userdata(ResizeControlsInfo)=A"!!,BY!!#D&!!#A/!!#<Xz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	PopupMenu popCT_ct,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	PopupMenu popCT_ct,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkLog_ct,userdata(ResizeControlsInfo)=A"!!,GX!!#D&!!#=k!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chkLog_ct,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkLog_ct,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkAutoMin_ct,userdata(ResizeControlsInfo)=A"!!,GX!!#D+!!#>6!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chkAutoMin_ct,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkAutoMin_ct,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svMin_ct,userdata(ResizeControlsInfo)=A"!!,C<!!#D+!!#A4!!#<Hz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svMin_ct,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svMin_ct,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	CheckBox chkAutoMax_ct,userdata(ResizeControlsInfo)=A"!!,GX!!#D5!!#>6!!#<8z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	CheckBox chkAutoMax_ct,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	CheckBox chkAutoMax_ct,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetVariable svMax_ct,userdata(ResizeControlsInfo)=A"!!,C4!!#D5!!#A5!!#<Hz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	SetVariable svMax_ct,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	SetVariable svMax_ct,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	Slider sliderMin_ct,userdata(ResizeControlsInfo)=A"!!,BY!!#D0!!#Au!!#<Hz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
	Slider sliderMin_ct,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Slider sliderMin_ct,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	Slider sliderMax_ct,userdata(ResizeControlsInfo)=A"!!,BY!!#D:!!#Au!!#<Hz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
	Slider sliderMax_ct,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Slider sliderMax_ct,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	GroupBox grpPixel,userdata(ResizeControlsInfo)=A"!!,HD!!#Co^]6^8!!#A*z!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	GroupBox grpPixel,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	GroupBox grpPixel,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	TitleBox TitleCsr,userdata(ResizeControlsInfo)=A"!!,HI!!#CqJ,hp#!!#<(z!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	TitleBox TitleCsr,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	TitleBox TitleCsr,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	Button btnZap,userdata(ResizeControlsInfo)=A"!!,HN!!#D,!!#>V!!#<hz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	Button btnZap,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Button btnZap,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	Button btnCopy,userdata(ResizeControlsInfo)=A"!!,HN!!#D25QF,A!!#<hz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	Button btnCopy,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Button btnCopy,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	Button btnPaste,userdata(ResizeControlsInfo)=A"!!,HN!!#D8J,ho,!!#<hz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	Button btnPaste,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Button btnPaste,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"

	SetWindow kwTopWin,userdata(ResizeControlsInfo)=A"!!*'\"z!!#C%J,htozzzzzzzzzzzzzzzzzzzzz"
	SetWindow kwTopWin,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzzzzzzzzzzzzzzz"
	SetWindow kwTopWin,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzzzzzzzzz!!!"
	SetWindow kwTopWin,userdata(ResizeControlsGuides)="FGT;FGM;FGB;FGHDIV;"
	
	// resizing panel hook
	SetWindow HyperspecBrowser hook(ResizeControls)=ResizeControls#ResizeControlsHook
		
	if (restart)
		DoUpdate
		MoveWindow/W=HyperspecBrowser left, top, oldright, oldbottom // points
	endif
	DoIgorMenu "control", "retrieve window"

	SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
end

static function TabControlProc(STRUCT WMTabControlAction &s)

	if (s.eventcode != 2)
		return 0
	endif
	
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	prefs.mapmode = s.tab
	StructPut prefs, wStruct
	
	ReplotPixelTraces(prefs)
	MapMode(prefs)
	ShowProfileMap(prefs)
	ShowProfile(prefs)
		
	return 0
end

static function cpu2point(variable cpu)
	return ScreenResolution > 96 ? cpu  : cpu * 72 / ScreenResolution
end

static function point2cpu(variable point)
	return ScreenResolution > 96 ? point : point * ScreenResolution / 72
end

function InfoMenu()
	string cmd = "(Version: " + num2str(GetThisVersion()/100) + ";"
	cmd += "Visit web page;Email the developer;"
	PopupContextualMenu cmd
	if (V_flag == 2)
		BrowseURL /Z "https://www.wavemetrics.com/node/" + num2str(kProjectID)
	elseif (V_flag == 3)
		Make/T/free/N=5 wt = "mail"
		string newline = "\r\n"
		wt[3] = "subject=" + URLEncode(ksShortTitle + " Package for Igor Pro")
		wt[4] = "Type your message here" + newline + newline
		wt[4] += "PROJECT VERSION:" + num2str(GetThisVersion()/100) + newline
		wt[4] += ReplaceString(";", IgorInfo(3), newline) + newline
		wt[4] = "body=" + URLEncode(wt[4])
		wt[1] = "tony withers"
		wt[2] = "uni-bayreuth de"
		sprintf cmd, "%sto:%s%%%d%s?%s&%s" wt[0], wt[1], 40, wt[2], wt[3], wt[4]
		BrowseURL/Z ReplaceString(" ", cmd, ".")
	endif
end



static function SaveAverage()
	DFREF dfr = getDFR()
	wave w = dfr:spectrum
	SaveWaveUI(w, basename=NameOfWave(GetImageWave()) + "_av")
end

static function TruncateData()
	GetMarquee/Z/W=HyperspecBrowser#PixelTraces bottom
	if (!V_flag)
		return 0
	endif
	
	wave w3D = GetImageWave()
	int rmax = DimSize(w3D, 2)-1
	int p0 = scaleToIndex(w3D, v_left, 2)
	int p1 = scaleToIndex(w3D, v_right, 2)
	int pmin = max(0, min(p0, p1))
	int pmax = min(rmax, max(p0, p1))
	
	if (pmin > 0 || pmax < rmax)
		string strAlert = "Do you really want to delete data from " + NameOfWave(w3D) + "?\r"
		strAlert += "Data outside x-range of marquee will be permanently deleted."
		DoAlert 1, strAlert
		if (v_flag == 2)
			return 0
		endif
		
		STRUCT PackagePrefs prefs
		wave/SDFR=GetDFR() wStruct
		StructGet prefs, wStruct
		
		DeletePoints/M=2 pmax+1, rmax-pmax, w3D
		if (prefs.ctlayer > pmax)
			prefs.ctlayer = pmax
		endif
		DeletePoints/M=2 0, pmin, w3D
		prefs.ctlayer -= min(pmin, prefs.ctlayer)
		StructPut prefs, wStruct
		
		MakeHyperspecBrowser(w3D) // start from scratch to clean up controls and plots
	endif
	return 0
end

// display a spectrum representing average of selected pixels
static function AveragePixelsInRect(STRUCT Rect &pix, STRUCT PackagePrefs &prefs)
	wave w3D = GetImageWave()
	
	wave/SDFR=GetDFR() wMask, spectrum
	FastOp wMask = 0
	wMask[pix.left,pix.right][pix.bottom,pix.top] = 1
	
	Duplicate/free/RMD=[pix.left,pix.right][pix.bottom,pix.top][] w3D wROI
	MatrixOP/free w = sumRows(sumCols(wROI)/fp64(numCols(wROI)))/fp64(numRows(wROI))
	Redimension/D spectrum
	spectrum = w[0][0][p]
	
	string strTextBox = ""
	sprintf strTextBox, "%s[%g,%g][%g,%g], %d pixels", NameOfWave(w3D), pix.left, pix.right, pix.bottom, pix.top, sum(wMask)
	TextBox/C/W=HyperspecBrowser#PixelTraces/N=text1 strTextBox
	note/K spectrum "source=" + strTextBox
	
	if (prefs.options & 4)
		wMask = DisplayPixelTrace(p, q, w3D, wMask) // doesn't change wMask
		ReorderTraces/W=HyperspecBrowser#PixelTraces _front_, {spectrum}
	endif
end

#ifdef dev
function PrintStruct()
	STRUCT PackagePrefs prefs
	wave/SDFR=GetDFR() wStruct, spectrum
	StructGet prefs, wStruct
	Print prefs
end
#endif

// point and rect structures must have same units
static function PointInRect(STRUCT point &pnt, STRUCT rect &r)
	return (pnt.h>r.left && pnt.h<r.right && pnt.v>r.top && pnt.v<r.bottom)
end

static function point2pixel(variable point [string win])
	variable expansion = ParamIsDefault(win) ? 1 : PanelResolution(win)/PanelResolution("")
	return expansion * point * ScreenResolution / 72
end

static function cpu2pixel(variable cpu, [string win])
	variable expansion = ParamIsDefault(win) ? 1 : PanelResolution(win)/PanelResolution("")
	return expansion * (ScreenResolution > 96 ? cpu * ScreenResolution / 72 : cpu)
end

static function hookHyperspec(STRUCT WMWinHookStruct &s)

	if (s.eventcode == 2) // kill
		PrefsSave()
		return 0
	endif
	
	STRUCT PackagePrefs prefs
	wave/SDFR=GetDFR() wStruct, spectrum
	StructGet prefs, wStruct
	
	variable xval
	variable GuideDelta = 10 // width of zone between plots where we will be able to grab a window guide, pixels
	variable mouseV, winHeight, pos, GuidePos, expand, winWidth, mouseH, lim1, lim2
		
// *** hook frameguides ***
	
	if (s.eventcode == 4) // mousemoved
		// most likely s.winrect and s.mouseloc refer to a subwindow
		GetMouse/W=HyperspecBrowser
		mouseV = v_top // pixel relative to window origin
		mouseH = v_left
					
		if (str2num(GetUserData("HyperspecBrowser", "", "DragGuide")) == 1)
			GetWindow HyperspecBrowser wsizeDC
			winHeight = (v_bottom - v_top)
			pos = mouseV / winHeight
			if (pos == limit(pos, 0.2, 0.5))
				DefineGuide/W=HyperspecBrowser FGM={FT,pos,FB}
			endif
			s.cursorcode = 6
			s.doSetCursor = 1
		
		elseif (str2num(GetUserData("HyperspecBrowser", "", "DragGuide")) == 2)
			GetWindow HyperspecBrowser wsizeDC
			winWidth = (v_right - v_left)
			pos = mouseH / winWidth
			if (pos == limit(pos, 0.2, 0.8))
				DefineGuide/W=HyperspecBrowser FGHDIV={FL,pos,FR}
				StructGet prefs, wStruct
				prefs.fg = pos
				StructPut prefs, wStruct
			endif
			s.cursorcode = 5
			s.doSetCursor = 1
		else
			GuidePos = NumberByKey("POSITION", GuideInfo("HyperspecBrowser", "FGM"))
			GuidePos = cpu2pixel(GuidePos, win="HyperspecBrowser")
			if (abs(mouseV - GuidePos) < GuideDelta)
				s.cursorcode = 6
				s.doSetCursor = 1
			elseif (mouseV < GuidePos)
				GuidePos = NumberByKey("POSITION", GuideInfo("HyperspecBrowser", "FGHDIV"))
				GuidePos = cpu2pixel(GuidePos, win="HyperspecBrowser")
				GetWindow HyperspecBrowser wsizeDC
				winWidth = (v_right - v_left)
				if (GuidePos<winWidth && abs(s.mouseloc.h - GuidePos)<GuideDelta)
					s.cursorcode = 5
					s.doSetCursor = 1
				endif
			endif
		endif
//		return 0
	endif
	
	// map subwindow updated
	if (s.eventcode == 8 && cmpstr(s.winname, "HyperspecBrowser#map")==0)
		// may need to rescale colours
		StructGet prefs, wStruct
		RecolorMap(prefs)
		ResetRGBmap(prefs, GetImageWave())
	endif
		
// *** hook cursors ***
	
	if (s.eventcode == 7) // cursor
		// get local prefs
		StructGet prefs, wStruct
		int i = 0
		string strControl = ""
		
		strswitch(s.winName)
			case "HyperspecBrowser#map" : // cursor in image plot
				strswitch(s.cursorname)
					case "C" : // crosshair
						// plot the spectrum for the pixel under the cursor
						prefs.crosshair.h = numtype(s.pointnumber) ? prefs.crosshair.h : s.pointnumber
						prefs.crosshair.v = numtype(s.ypointnumber) ? prefs.crosshair.v :s.ypointnumber
						PlotCrosshairPixel(prefs)
						break
					case "A" : // start of line
						prefs.line.left = numtype(s.pointnumber) ? prefs.line.left : s.pointnumber
						prefs.line.top = numtype(s.ypointnumber) ? prefs.line.top : s.ypointnumber
						PlotLineAndPixelTraces(prefs)
						ResetProfileMap(prefs)
						ResetLineProfiles(prefs, GetImageWave())
						break
					case "B" : // end of line
						prefs.line.right = numtype(s.pointnumber) ? prefs.line.right : s.pointnumber
						prefs.line.bottom = numtype(s.ypointnumber) ? prefs.line.bottom : s.ypointnumber
						PlotLineAndPixelTraces(prefs)
						ResetProfileMap(prefs)
						ResetLineProfiles(prefs, GetImageWave())
						break
				endswitch
				break
			case "HyperspecBrowser#PixelTraces" : // cursor in spectrum plot
			case "HyperspecBrowser#ProfileMap" : // cursor in profile map plot
				strswitch(s.cursorname)
					case "B" :
						if (strlen(CsrInfo(B, s.winname)))
							// sync setvar with cursor
							xval = hcsr($s.cursorname, s.winName)
													
							ControlInfo/W=HyperspecBrowser svPlane_ct
							SplitString/E="limits={.*}" S_recreation
							sscanf s_value, "limits={%g,%g,", lim1, lim2
							xval = limit(xval, min(lim1, lim2), max(lim1, lim2))
							
							SetVariable svPlane_ct win=HyperspecBrowser, value=_NUM:xval // this doesn't trigger update event in setvar proc
							prefs.ctlayer = scaleToIndex(spectrum, xval, 0) // pcsr($s.cursorname, s.winName)
							prefs.ctlayer = limit(prefs.ctlayer, 0, numpnts(spectrum)-1)
							RecolorMap(prefs)
							ResetLineProfiles(prefs, GetImageWave())
							
						endif
						break
					case "E" :
						i += 1
					case "D" :
						i += 1
					case "C" :
						if (strlen(CsrInfo($s.cursorname, "HyperspecBrowser#PixelTraces")))
							wave w3D = GetImageWave()
							xval = hcsr($s.cursorname, s.winName)
							strControl = "svplane" + num2str(i) + "_rgb"
							ControlInfo/W=HyperspecBrowser $strControl
							SplitString/E="limits={.*}" S_recreation
							sscanf s_value, "limits={%g,%g,", lim1, lim2
							xval = limit(xval, min(lim1, lim2), max(lim1, lim2))
							
							SetVariable $strControl win=HyperspecBrowser, value=_NUM:xval // this doesn't trigger update event in setvar proc
							prefs.rgblayer[i] = scaleToIndex(spectrum, xval, 0) // pcsr($s.cursorname, s.winName)
							prefs.rgblayer[i] = limit(prefs.rgblayer[i], 0, DimSize(w3D, 2)-1)
							
							ResetRGBmap(prefs, GetImageWave())
							ResetLineProfiles(prefs, GetImageWave())
						endif
						break
				endswitch
				break
			break
		endswitch
		 
		StructPut prefs, wStruct
	endif
		
	if (s.eventcode == 3) // mousedown
		GuidePos = NumberByKey("POSITION", GuideInfo("HyperspecBrowser", "FGM"))
		GuidePos = cpu2pixel(GuidePos, win="HyperspecBrowser")
		if (abs(s.mouseloc.v - GuidePos) < GuideDelta)
			SetWindow HyperspecBrowser userdata(DragGuide) = "1"
		elseif (s.mouseloc.v < GuidePos)
			GuidePos = NumberByKey("POSITION", GuideInfo("HyperspecBrowser", "FGHDIV"))
			GetWindow HyperspecBrowser wsizeDC
			winWidth = (v_right - v_left)
			GuidePos = cpu2pixel(GuidePos, win="HyperspecBrowser")
			if (GuidePos<winWidth && abs(s.mouseloc.h - GuidePos)<GuideDelta)
				SetWindow HyperspecBrowser userdata(DragGuide) = "2"
			endif
		endif
	endif
		
// *** hook keyboard ***
	
	if (s.eventcode == 11)	// Keyboard event

		if (s.keycode == 65)
			SetAxis/W=HyperspecBrowser#map/A
			
			wave/SDFR=GetDFR() wStruct
			StructGet prefs, wStruct
			
			if (prefs.options & 1)
				SetAxis/A/R/W=HyperspecBrowser#PixelTraces bottom
			else
				SetAxis/A/W=HyperspecBrowser#PixelTraces bottom
			endif
			SetAxis/A=2/W=HyperspecBrowser#PixelTraces left
			
			SetAxis/A=2/W=HyperspecBrowser#Profiles left
			SetAxis/A/W=HyperspecBrowser#Profiles bottom
			
			if (WinType("HyperspecBrowser#ProfileMap"))
				if (prefs.options&1)
					SetAxis/A/R/W=HyperspecBrowser#ProfileMap bottom
				else
					SetAxis/A/W=HyperspecBrowser#ProfileMap bottom
				endif
				SetAxis/A/W=HyperspecBrowser#ProfileMap left
			endif
			
			return 1
		endif
		
		if (s.keycode == 90)
			UndoZap()
			return 1
		endif
	endif
	
// *** hook mouseup ***
	
	if (s.eventcode == 5) // mouseup
		SetWindow HyperspecBrowser userdata(DragGuide) = "0"

		if (cmpstr(s.winname, "HyperspecBrowser#map"))
			if (GrepString(WinRecreation("HyperspecBrowser", 0), "ShowTools"))
				SetIgorDrawingMode(0)
			endif
		endif
	endif

	if (s.eventcode==12 && s.mouseloc.h<0) // moved, we get here by clicking tools button when showtools didn't use /A flag
		SetIgorDrawingMode(0)
	endif
	
	if (prefs.mode==6 && !cmpstr(s.winname, "HyperspecBrowser#map"))
		return PixelBrush(s)
	endif
	
	if (prefs.mode==5 || prefs.mode==8) // drawing a path or editing a ROI
		if (!cmpstr(s.winname, "HyperspecBrowser#map"))
			return DoPath(s) // skip further checks for mouseup/mousedown
		endif
	endif
			
	// mousedown in map
	if (s.eventcode==3 && !(s.eventmod&16) && !cmpstr(s.winname, "HyperspecBrowser#map")) // mousedown in map
		// get local prefs
		StructGet prefs, wStruct
		int oldmode = prefs.mode
			
		if (prefs.mode == 7) //
			prefs.mode = 0
			StructPut prefs, wStruct
			ShowProfileMap(prefs)
			ShowProfile(prefs)
		endif
			
		ShowCursors(prefs, 1)
		if (prefs.mode == 0 && oldmode)
			SetOverlayTransparency(0)
			ReplotPixelTraces(prefs)
		endif
	endif
	
	if (s.eventcode==5 && !cmpstr(s.winname, "HyperspecBrowser#map")) // mouseup in map
		// get local prefs
		StructGet prefs, wStruct
		oldmode = prefs.mode
		
		STRUCT Rect MQPixels
		if (MarqueeToImagePixels(MQPixels))
			// a marquee is present
			prefs.mode = 4
			StructPut prefs, wStruct
			AveragePixelsInRect(MQPixels, prefs) // display a spectrum representing average of selected pixels
			SetOverlayTransparency(1)
			ShowCursors(prefs, 1)
			ShowProfile(prefs)
			ShowProfileMap(prefs)
		else
			if (prefs.mode == 4 && !(s.eventmod&16)) // exiting marquee mode
				prefs.mode = 0
				StructPut prefs, wStruct
				SetOverlayTransparency(0)
				ShowCursors(prefs, 1)
				ReplotPixelTraces(prefs)
				ShowProfile(prefs)
				ShowProfileMap(prefs)
			endif
		endif
	endif

	if (s.eventcode==5 && prefs.mode==5) // mouseup outside of map in ROI drawing mode
		if (cmpstr(s.winname, "HyperspecBrowser")==0) // clicked in parent panel
			DrawPath(0, 0) // quit drawing
		endif
	endif
	
	return 0
end

// plot the spectrum for the pixel under the cursor
static function PlotCrosshairPixel(STRUCT PackagePrefs &prefs)
	
	CleanupPixelTraces(0)
		
	if (prefs.mode || prefs.cursormode)
		return 0
	endif
	
	if (strlen(CsrInfo(C, "HyperspecBrowser#map"))==0)
		ShowCursors(prefs, 1)
		return 0
	endif
	
	wave w3D = GetImageWave()
	wave/SDFR=GetDFR() spectrum, wStruct
	
	if(WaveType(spectrum) != WaveType(w3D))
		Redimension/Y=(WaveType(w3D)) spectrum
	endif
	
	spectrum = w3D[prefs.crosshair.h][prefs.crosshair.v][p]
	string strTextBox = ""
	if (DimOffset(w3D, 0)!=0 || DimDelta(w3D, 0)!=1 || DimOffset(w3D, 1)!=0 || DimDelta(w3D, 1)!=1)
		sprintf strTextBox, "%s[%g][%g], x=%g y=%g", NameOfWave(w3D), prefs.crosshair.h, prefs.crosshair.v, hcsr(C, "HyperspecBrowser#map"), vcsr(C, "HyperspecBrowser#map")
	else
		sprintf strTextBox, "%s[%g][%g]", NameOfWave(w3D), prefs.crosshair.h, prefs.crosshair.v
	endif
	TextBox/C/W=HyperspecBrowser#PixelTraces/N=text1 strTextBox
	
	string strNote = ""
	sprintf strNote "source=%s[%d][%d]", NameOfWave(w3D), prefs.crosshair.h, prefs.crosshair.v
	note/K spectrum strNote

	return 1
end

static function MarqueeToImagePixels(STRUCT Rect &MQPixels)
	GetMarquee/Z/W=HyperspecBrowser#map left, bottom
	if (V_flag)
		wave w3D = GetImageWave()
		int p1, p2, q1, q2, pmin, pmax, qmin, qmax
		q1 = scaleToIndex(w3D, v_top, 1)
		q2 = scaleToIndex(w3D, v_bottom, 1)
		p1 = scaleToIndex(w3D, v_left, 0)
		p2 = scaleToIndex(w3D, v_right, 0)
		pmin = 0
		qmin = 0
		pmax = DimSize(w3D, 0) - 1
		qmax = DimSize(w3D, 1) - 1
		MQPixels.left = max(pmin, min(p1, p2))
		MQPixels.right = min(pmax, max(p1, p2))
		MQPixels.bottom = max(min(q1, q2), qmin)
		MQPixels.top = min(max(q1, q2), qmax)
	endif
	return v_flag
end

// called from window hook function when prefs.mode = 6
static function PixelBrush(STRUCT WMWinHookStruct &s)
	
	if (cmpstr(s.winname, "HyperspecBrowser#map"))
		return 0
	endif
	
	if (s.eventMod & 16) // allow contextual menu
		return 0
	endif
	
	// need to fix this
	if (s.eventMod & 13) // cmd/ctrl
		s.doSetCursor = 1
		s.cursorCode = (s.eventMod & 4) ? 19 : 18
	endif
	
	if (s.eventCode==3 && !(s.eventMod&12) )
		PixelEdit(0)
	endif
		
	if (! (s.eventCode==3 && (s.eventMod&12)) ) // mousedown, alt/ctrl/shift-ctrl
		return 0
	endif
	
	wave w_img = GetImageWave()
	int pmax = DimSize(w_img, 0)
	int qmax = DimSize(w_img, 1)
	
	DFREF dfr = GetDFR()
	wave/SDFR=dfr wMask, wDisplayMaskRGBA
	
	variable v_X, v_Y, p_X, p_Y

	int ctrl = (s.eventMod & 8) > 0
	int opt =  (s.eventMod & 4) > 0
	int keys, shift
	keys = ctrl + 2*opt

	int mousebutton = 1
	int oldLeft = s.mouseLoc.h, oldTop = s.mouseLoc.v
	int horiz = 0, vert = 0
	int firstLoop = 1
	for (;mousebutton && (keys&3);)
		keys = GetKeyState(0)
		GetMouse/W=HyperspecBrowser //$s.WinName
		mousebutton = V_flag & 1
		
		if (keys & 4) // shift
			if ((horiz + vert) == 0)
				if (abs(v_left-oldLeft) > abs(v_top-oldTop))
					vert = 1
				elseif (abs(v_top-oldTop) > abs(v_left-oldLeft))
					horiz = 1
				endif
			endif
			if (vert)
				v_top = oldTop
			elseif (horiz)
				v_left = oldLeft
			endif
		else
			horiz = 0
			vert = 0
		endif
				
		if (v_left==oldLeft && v_top==oldTop && (!firstLoop))
			continue
		endif
		firstLoop = 0
		
		oldLeft = v_left
		oldTop = v_top
		
		ctrl = keys & 1
		opt = keys & 2
		if (!(mousebutton && (ctrl||opt)))
			continue
		endif
		
		variable hh = AxisValFromPixel("HyperspecBrowser#map", "bottom", v_left)
		variable vv = AxisValFromPixel("HyperspecBrowser#map", "left", v_top)
		
		int pp = scaleToIndex(w_img, hh, 0)
		int qq = scaleToIndex(w_img, vv, 1)
		
		if (pp<0 || qq<0 || pp>=pmax || qq>=qmax)
			continue
		endif
	
		wMask[pp][qq] = ctrl
		wDisplayMaskRGBA[pp][qq][3] = 40000 * !ctrl
		DoUpdate/W=HyperspecBrowser#map

	endfor
	AveragePixelsROI(wMask)
	
	return 0
end

static function PopupProc(STRUCT WMPopupAction &s) : PopupMenuControl
	
	if (s.eventCode != 2)
		return 0
	endif
	
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	strswitch (s.ctrlname)
		case "popCT_ct" :
			prefs.ctnum = s.popnum - 1
			RecolorMap(prefs)
			
			string strColorTable = StringFromList(prefs.ctnum, CTabList())
			if (WinType("HyperspecBrowser#ProfileMap"))
				ModifyImage/W=HyperspecBrowser#ProfileMap M_ImageLineProfile ctab={*,*,$strColorTable,0}, ctabAutoscale=1
			endif
			
			break
		case "popCursorColor" :
			ControlInfo/W=$s.win $s.ctrlname
			variable red, green, blue
			sscanf s_value, "(%d,%d,%d)", red, green, blue
			prefs.csrrgb.red = red
			prefs.csrrgb.green = green
			prefs.csrrgb.blue = blue
			StructPut prefs, wStruct
			ShowCursors(prefs, 7)
			break
		case "popTraceColor" :
			ControlInfo/W=$s.win $s.ctrlname
			sscanf s_value, "(%d,%d,%d)", red, green, blue
			prefs.specrgb.red = red
			prefs.specrgb.green = green
			prefs.specrgb.blue = blue
			ModifyGraph/Z/W=HyperspecBrowser#PixelTraces rgb(spectrum)=(red,green,blue)
			ModifyGraph/Z/W=HyperspecBrowser#Profiles rgb(LineProfile)=(red,green,blue)
	endswitch
	StructPut prefs, wStruct
	return 0
end

static function SetvarProc(STRUCT WMSetVariableAction &s) : SetVariableControl

	if (s.eventCode<1 || s.eventCode>8)
		return 0
	endif
	
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	if (s.eventcode == 7)
		if (cmpstr(s.ctrlname, "svDatacube") == 0)
			CreateBrowser/M Prompt="Select a 3D wave that represents a hyperspectral datacube"
			ModifyBrowser/M showWaves=1, showVars=0, showStrs=0
			ModifyBrowser/M ShowInfo=0, showPlot=0
			ModifyBrowser/M clearSelection
			if (strlen(s.sval))
				ModifyBrowser/M selectList=s.sval
			endif
			ModifyBrowser/M showModalBrowser
			if (V_Flag == 0)
				// use ControlUpdate to deselect the setvar
				ControlUpdate/W=$s.win $s.ctrlname
				return 0
			endif
			if (ItemsInList(S_BrowserList) == 1)
				wave/Z wSel = $StringFromList(0, S_BrowserList)
				MakeHyperspecBrowser(wSel)
				ControlUpdate/W=$s.win $s.ctrlname
			endif
		endif
		return 0 // prefs may have been updated and saved
	endif
		
	int i = 0	
	strswitch (s.ctrlName)
		case "svPlane_ct":
			if (prefs.mapmode==0) // (prefs.mapmode==0 && prefs.cursormode != 2)
				GetAxis/W=HyperspecBrowser#PixelTraces/Q bottom
				Cursor/W=HyperspecBrowser#PixelTraces/F/P B spectrum (s.dval-V_min)/(V_max-V_min), 0.5
			endif
			wave/SDFR=GetDFR() spectrum
			prefs.ctlayer = scaleToIndex(spectrum, s.dval, 0)
			StructPut prefs, wStruct
			RecolorMap(prefs)
			ShowCursors(prefs, 4)
			break
		case "svMin_ct":
			prefs.ctmin = s.dval
			StructPut prefs, wStruct
			RecolorMap(prefs)
			break
		case "svMax_ct": // plane, min or max
			prefs.ctmax = s.dval
			StructPut prefs, wStruct
			RecolorMap(prefs)
			break
			
		case "svplane2_rgb":
		case "svplane1_rgb":
		case "svplane0_rgb":
			i = str2num((s.ctrlname)[7])
			GetAxis/W=HyperspecBrowser#PixelTraces/Q bottom
			string csr = "CDE"[i]
			Cursor/W=HyperspecBrowser#PixelTraces/F/P $csr spectrum (s.dval-V_min)/(V_max-V_min), 0.5
			ShowCursors(prefs, 4)
			wave/SDFR=GetDFR() spectrum
			prefs.rgblayer[i] = scaleToIndex(spectrum, s.dval, 0)
			StructPut prefs, wStruct
			ResetRGBmap(prefs, GetImageWave())
			break
		case "svmin2_rgb":
		case "svmin1_rgb":
		case "svmin0_rgb":
			i = str2num((s.ctrlname)[5])
			prefs.rgbmin[i] = s.dval
			StructPut prefs, wStruct
			ResetRGBmap(prefs, GetImageWave())
			break
		case "svmax2_rgb":
		case "svmax1_rgb":
		case "svmax0_rgb":
			i = str2num((s.ctrlname)[5])
			prefs.rgbmax[i] = s.dval
			StructPut prefs, wStruct
			ResetRGBmap(prefs, GetImageWave())
			break

		case "svHlabel" :
			prefs.hlabel = s.sval
			StructPut prefs, wStruct
			if (strlen(prefs.hlabel))
				SetVariable svPlane_ct,win=HyperspecBrowser,title=prefs.hlabel
			else
				SetVariable svPlane_ct,win=HyperspecBrowser,title="Layer"
			endif
			if (!(prefs.options & 32))
				Label/W=HyperspecBrowser#PixelTraces bottom, prefs.hlabel
				if (WinType("HyperspecBrowser#ProfileMap"))
					Label/W=HyperspecBrowser#ProfileMap bottom, prefs.hlabel
				endif
			endif
			break
		case "svVlabel" :
			prefs.vlabel = s.sval
			StructPut prefs, wStruct
			if (!(prefs.options & 32))
				Label/W=HyperspecBrowser#PixelTraces left, prefs.vlabel
				if (WinType("HyperspecBrowser#Profiles"))
					if (prefs.options & 256)
						Label/W=HyperspecBrowser#Profiles bottom, prefs.vlabel
					else
						Label/W=HyperspecBrowser#Profiles left, prefs.vlabel
					endif
				endif
			endif		
			break		
		case "svWidth" :	
			prefs.width = s.dval
			if (prefs.options & 512)
				ResetProfileMap(prefs)
			endif	
			break
	endswitch
	
	StructPut prefs, wStruct
	return 0
end

static function SliderProc(STRUCT WMSliderAction &s) : SliderControl

	if (s.eventcode>0 && s.eventcode&1)
		wave/SDFR=GetDFR() wStruct
		STRUCT PackagePrefs prefs
		StructGet prefs, wStruct
		if (cmpstr(s.ctrlname, "sliderMin_ct") == 0)
			SetVariable svMin_ct win=HyperspecBrowser, value=_NUM:s.curval
			prefs.ctmin = s.curval
		elseif (cmpstr(s.ctrlname, "sliderMax_ct") == 0)
			SetVariable svMax_ct win=HyperspecBrowser, value=_NUM:s.curval
			prefs.ctmax = s.curval
		endif
		StructPut prefs, wStruct
		RecolorMap(prefs)
	endif
	return 0
end

static function ResetRGBmap(STRUCT PackagePrefs &prefs, wave w3D)
	wave/SDFR=GetDFR() wRGB
	FastOp wRGB = 0
	int i
	string ctrl
	
	int p0 = 0, p1 = DimSize(w3D, 0) - 1, q0 = 0, q1 = DimSize(w3D, 1) - 1
	Make/free/n=2/D prange, qrange
	GetAxis/W=HyperspecBrowser#map/Q bottom
	prange = {scaleToIndex(w3D, v_min, 0), scaleToIndex(w3D, v_max, 0)}
	GetAxis/W=HyperspecBrowser#map/Q left
	qrange = {scaleToIndex(w3D, v_min, 1), scaleToIndex(w3D, v_max, 1)}
	prange = (limit(prange, p0, p1))
	qrange = (limit(qrange, q0, q1))
	Sort prange, prange
	Sort qrange, qrange
		
	for(i=0;i<3;i++)
		if (prefs.rgboptions[i] & 1)
			if (prefs.rgboptions[i] & 2)
				WaveStats/Q/M=1/RMD=[prange[0], prange[1]][qrange[0], qrange[1]][prefs.rgblayer[i]] w3D
				prefs.rgbmin[i] = v_min
				prefs.rgbmax[i] = v_max
				ctrl = "svmax" + num2str(i) + "_rgb"
				SetVariable $ctrl win=HyperspecBrowser, value=_NUM:prefs.rgbmax[i]
				ctrl = "svmin" + num2str(i) + "_rgb"
				SetVariable $ctrl win=HyperspecBrowser, value=_NUM:prefs.rgbmin[i]
			endif
			if (prefs.rgboptions[i] & 4) // log colors
				// scale values linearly from 1 to 10 so that log values vary logarithmically from 0 to 1
				wRGB[][][i] = log(1 + (w3D[p][q][prefs.rgblayer[i]]-prefs.rgbmin[i])/(prefs.rgbmax[i]-prefs.rgbmin[i])*9) * 65535
			else
				wRGB[][][i] = limit((w3D[p][q][prefs.rgblayer[i]]-prefs.rgbmin[i])/(prefs.rgbmax[i]-prefs.rgbmin[i])*65535, 0, 65535)
			endif
		endif
	endfor
end

static function RecolorMap(STRUCT PackagePrefs &prefs)

	wave w3D = GetImageWave()
	if (prefs.mapmode != 0) // RGB map
		return 0
	endif

	variable plane = IndexToScale(w3D, prefs.ctlayer, 2)
	
	int p0 = 0, p1 = DimSize(w3D, 0) - 1, q0 = 0, q1 = DimSize(w3D, 1) - 1
	GetAxis/W=HyperspecBrowser#map/Q bottom
	Make/free/n=2/D prange, qrange
	prange = {scaleToIndex(w3D, v_min, 0), scaleToIndex(w3D, v_max, 0)}
	GetAxis/W=HyperspecBrowser#map/Q left
	qrange = {scaleToIndex(w3D, v_min, 1), scaleToIndex(w3D, v_max, 1)}
	prange = (limit(prange, p0, p1))
	qrange = (limit(qrange, q0, q1))
	Sort prange, prange
	Sort qrange, qrange
	
	WaveStats/Q/M=1/RMD=[prange[0], prange[1]][qrange[0], qrange[1]][prefs.ctlayer] w3D
	variable planemin = v_min
	variable planemax = v_max
	if (prefs.ctoptions & 3)
		if (prefs.ctoptions & 1)
			prefs.ctmin = v_min
			SetVariable svMin_ct win=HyperspecBrowser, value=_NUM:prefs.ctmin
		endif
		if (prefs.ctoptions & 2)
			prefs.ctmax = v_max
			SetVariable svMax_ct win=HyperspecBrowser, value=_NUM:prefs.ctmax
		endif
	endif
	Slider sliderMin_ct win=HyperspecBrowser, value=limit(prefs.ctmin, planemin, planemax), limits={planemin,planemax,0}
	Slider sliderMax_ct win=HyperspecBrowser, value=limit(prefs.ctmax, planemin, planemax), limits={planemin,planemax,0}
	string strColorTable = StringFromList(prefs.ctnum, CTabList())
	
	ModifyImage/W=HyperspecBrowser#map ''#0 plane=prefs.ctlayer, ctab={prefs.ctmin,prefs.ctmax,$strColorTable,0}, ctabAutoscale=3
	
	string strTextBox = ""
	if (prefs.mapmode==0 && (DimOffset(w3D, 2)!=0 || DimDelta(w3D, 2)!=1))
		sprintf strTextBox, "%s[][][%g], z = %g", NameOfWave(w3D), prefs.ctlayer, plane
	else
		sprintf strTextBox, "%s[][][%g]", NameOfWave(w3D), prefs.ctlayer
	endif
	TextBox/C/W=HyperspecBrowser#map/N=text0 strTextBox

end

static function ResetColorTableMapTextbox(STRUCT PackagePrefs &prefs, wave w3D)
	string strTextBox = ""
	if (prefs.mapmode==0 && (DimOffset(w3D, 2)!=0 || DimDelta(w3D, 2)!=1))
		sprintf strTextBox, "%s[][][%g], z = %g", NameOfWave(w3D), prefs.ctlayer, IndexToScale(w3D, prefs.ctlayer, 2)
	else
		sprintf strTextBox, "%s[][][%g]", NameOfWave(w3D), prefs.ctlayer
	endif
	TextBox/C/W=HyperspecBrowser#map/N=text0 strTextBox
end

// *** show line profile ***

static function ShowProfileMap(STRUCT PackagePrefs &prefs)
		
//	if (prefs.mapmode==0 && prefs.mode==0 && prefs.options&512 && prefs.cursormode==1)
	if ((prefs.mode==0  && prefs.options&512 && prefs.cursormode==1)   || (prefs.mode==5 && prefs.options&512))
				
		if (WinType("hyperspecbrowser#ProfileMap"))
			ShowCursors(prefs, 4)
			ResetProfileMap(prefs)
			Label/W=hyperspecbrowser#ProfileMap left SelectString(prefs.mode & 5, "A  <-   distance   ->  B", "distance")
			return 0
		endif

		Display/FG=(FL,FGT,FGHDIV,FGM)/HOST=HyperspecBrowser/N=ProfileMap
		// set textbox position
		TextBox/C/N=textLineMap/F=0/A=LT/X=8/Y=0/E=2 ""
		
		ResetProfileMap(prefs) // creates M_ImageLineProfile and updates textbox
		wave/Z/SDFR=GetDFR() M_ImageLineProfile
		
		AppendImage/W=hyperspecbrowser#ProfileMap M_ImageLineProfile
		ModifyGraph/W=HyperspecBrowser#ProfileMap standoff=0,tick=2,btLen=3 //, margin(right)=8
		string strColorTable = StringFromList(prefs.ctnum, CTabList())
		// autoscale colors - different scaling to map
		ModifyImage/W=HyperspecBrowser#ProfileMap M_ImageLineProfile ctab={*,*,$strColorTable,0}, ctabAutoscale=1
		
		if (prefs.options & 1)
			SetAxis/A/R/W=HyperspecBrowser#ProfileMap bottom
		else
			SetAxis/A/W=HyperspecBrowser#ProfileMap bottom
		endif
		SetAxis/A/W=HyperspecBrowser#ProfileMap left
			
		// add cursor
		ControlInfo/W=HyperspecBrowser svPlane_ct
		DoUpdate/W=hyperspecbrowser#ProfileMap
		GetAxis/W=HyperspecBrowser#ProfileMap/Q bottom
		Cursor/W=HyperspecBrowser#ProfileMap/P/F/S=2/H=2/N=1/I/C=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue) B M_ImageLineProfile (v_value-V_min)/(V_max-V_min), 0.5

		if (prefs.options & 8)
			ModifyGraph/W=HyperspecBrowser#ProfileMap tick(left)=3,noLabel(left)=2,axThick(left)=0//, margin(left)=8
		endif

		if (!(prefs.options & 32)) // label axes
			if (!(prefs.options & 8)) // hide Y axis
				Label/W=HyperspecBrowser#ProfileMap bottom, prefs.hlabel
			endif
			Label/W=hyperspecbrowser#ProfileMap left SelectString(prefs.mode & 5, "A  <-   distance   ->  B", "distance")
		endif
		ModifyGraph/W=HyperspecBrowser#ProfileMap mirror=1, axisOnTop=1
		ShowCursors(prefs, 4)
	else
		KillWindow/Z HyperspecBrowser#ProfileMap
	endif
end

static function ShowProfile(STRUCT PackagePrefs &prefs)
	
	if (prefs.options&64 && ((prefs.mode==0 && prefs.cursormode==1) || prefs.mode==5)) // line mode with profile
		if (prefs.mapmode==1 && !(prefs.rgboptions[0]&1 || prefs.rgboptions[1]&1 || prefs.rgboptions[2]&1))
			DefineGuide/W=HyperspecBrowser FGHDIV={FL,1,FR}
			return 0
		endif
		DefineGuide/W=HyperspecBrowser FGHDIV={FL,prefs.fg,FR}
		ResetLineProfiles(prefs, GetImageWave())
	else
		DefineGuide/W=HyperspecBrowser FGHDIV={FL,1,FR}
		return 0
	endif
	
	ModifyGraph/Z/W=HyperspecBrowser#Profiles hideTrace(LineProfile)=(prefs.mapmode!=0)
	int i
	for (i=0;i<3;i++)
		string strProfileWave = "LineProfile" + "RGB"[i]
		ModifyGraph/Z/W=HyperspecBrowser#Profiles hideTrace($strProfileWave)=!(prefs.rgboptions[i]&1 && prefs.mapmode==1)
	endfor
	
	ModifyGraph/W=HyperspecBrowser#Profiles swapXY=(prefs.options&128)>0
	
	if (prefs.options & 16) // autoscale
		DoUpdate/W=HyperspecBrowser#Profiles
		if (prefs.options & 128)			
			SetAxis/A=2/W=HyperspecBrowser#Profiles bottom
		else
			SetAxis/A=2/W=HyperspecBrowser#Profiles left
		endif
	else
		WaveStats/Q/M=1 GetImageWave()
		if (prefs.options & 128)
			SetAxis/W=HyperspecBrowser#Profiles bottom, v_min, v_max
		else
			SetAxis/W=HyperspecBrowser#Profiles left, v_min, v_max
		endif
	endif
	
	string StrLabel = SelectString(prefs.mode & 5, "A  <-   distance   ->  B", "distance")
	if (prefs.options & 128)			
		Label/W=HyperspecBrowser#Profiles left StrLabel
	else
		Label/W=HyperspecBrowser#Profiles bottom StrLabel
	endif
	TextBox/W=HyperspecBrowser#Profiles/C/N=textLineProfile/F=0/A=LT/X=8/Y=0/E=2 SelectString(prefs.mode & 5, "Line Profile", "Path Profile")
		
end

// returns position (proportion of axis) at axis value = val
static function AxisValToPosition(variable val)
	GetAxis/W=HyperspecBrowser#PixelTraces/Q bottom
	return (val-V_min)/(V_max-V_min)
end

static function CheckBoxProc(STRUCT WMCheckBoxAction &s) : CheckboxControl
	
	if (s.eventCode != 2)
		return 0
	endif
	
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	int i = 0
	
	strswitch (s.ctrlName)
		case "chkAutoMin_ct":
			SetVariable svMin_ct  win=HyperspecBrowser, disable=2*s.checked
			Slider sliderMin_ct   win=HyperspecBrowser, disable=2*s.checked
			prefs.ctoptions = SetBitValue(prefs.ctoptions, 0, s.checked)
			if (s.checked)
				RecolorMap(prefs)
			endif
			break
		case "chkAutoMax_ct":
			SetVariable svMax_ct  win=HyperspecBrowser, disable=2*s.checked
			Slider sliderMax_ct   win=HyperspecBrowser, disable=2*s.checked
			prefs.ctoptions = SetBitValue(prefs.ctoptions, 1, s.checked)
			if (s.checked)
				RecolorMap(prefs)
			endif
			break
		case "chkLog_ct" :
			prefs.ctoptions = SetBitValue(prefs.ctoptions, 2, s.checked)
			logcolors(s.checked)
			break
		case "chkCrosshair" :
			CheckBox chkLine win=HyperspecBrowser, value=0
			
			if (prefs.mode==5 || prefs.mode==8)
				DrawPath(0, 0)
			endif
			
			prefs.cursormode = 0
			prefs.mode = 0
			StructPut prefs, wStruct
			PixelEdit(0) // also clears overlay
			ShowCursors(prefs, 1)
			PlotCrosshairPixel(prefs)
			ShowProfile(prefs)
			ShowProfileMap(prefs)
			break
		case "chkLine" :
			CheckBox chkCrosshair win=HyperspecBrowser, value=0
			if (prefs.mode==5 || prefs.mode==8)
				DrawPath(0, 0)
			endif
			prefs.cursormode = 1
			prefs.mode = 0
			StructPut prefs, wStruct
			PixelEdit(0) // also clears overlay
			ShowCursors(prefs, 1)
			PlotLineAndPixelTraces(prefs)
			ShowProfile(prefs)
			ShowProfileMap(prefs)
			break
		case "chkauto2_rgb" :
		case "chkauto1_rgb" :
		case "chkauto0_rgb" :
			i = str2num((s.ctrlname)[strlen(s.ctrlname)-5])
			// i = 0 for red, 1 for green, 2 for blue
			string ctrl = ReplaceString("chkauto", s.ctrlname, "svmin")
			SetVariable $ctrl win=HyperspecBrowser, disable=2*s.checked
			ctrl = ReplaceString("chkauto", s.ctrlname, "svmax")
			SetVariable $ctrl win=HyperspecBrowser, disable=2*s.checked
			prefs.rgboptions[i] = SetBitValue(prefs.rgboptions[i], 1, s.checked)
			ResetRGBmap(prefs, GetImageWave())
			break
		case "chkB_rgb" :
			i += 1
		case "chkG_rgb" :
			i += 1
		case "chkR_rgb" :
			// i = 0 for red, 1 for green, 2 for blue
			string strExp = "*" + num2str(i) + "_rgb"
			ModifyControlList/Z ControlNameList("HyperspecBrowser",";",strExp), win=HyperspecBrowser, disable=2*!s.checked
			prefs.rgboptions[i] = SetBitValue(prefs.rgboptions[i], 0, s.checked)
			if (s.checked)
				strExp = "svmin" + num2str(i) + "_rgb"
				SetVariable $strExp win=HyperspecBrowser, disable=prefs.rgboptions[i]&2 ? 2 : 0
				strExp = "svmax" + num2str(i) + "_rgb"
				SetVariable $strExp win=HyperspecBrowser, disable=prefs.rgboptions[i]&2 ? 2 : 0
			endif
			ShowCursors(prefs, 6)
			ResetRGBmap(prefs, GetImageWave())
			ShowProfile(prefs)
			break
		case "chklog2_rgb" :
		case "chklog1_rgb" :
		case "chklog0_rgb" :
			i = str2num((s.ctrlname)[strlen(s.ctrlname)-5])
			// i = 0 for red, 1 for green, 2 for blue
			prefs.rgboptions[i] = SetBitValue(prefs.rgboptions[i], 2, s.checked)
			ResetRGBmap(prefs, GetImageWave())
			break
				
// Checkboxes in prefs panel
			
		case "chkReverseSpectrum" :
			prefs.options = SetBitValue(prefs.options, 0, s.checked)
			if (s.checked)
				SetAxis/A/R/W=HyperspecBrowser#PixelTraces bottom
				if (WinType("HyperspecBrowser#ProfileMap"))
					SetAxis/A/R/W=HyperspecBrowser#ProfileMap bottom
				endif
			else
				SetAxis/A/W=HyperspecBrowser#PixelTraces bottom
				if (WinType("HyperspecBrowser#ProfileMap"))
					SetAxis/A/W=HyperspecBrowser#ProfileMap bottom
				endif
			endif
			
			// update axis before we place cursor
			DoUpdate/W=HyperspecBrowser
			ShowCursors(prefs, 6)
			break
		case "chkReverseImageY" :
			prefs.options = SetBitValue(prefs.options, 1, s.checked)
			if (s.checked) // reverse image Y
				SetAxis/A/R/W=HyperspecBrowser#map left
			else
				SetAxis/A/W=HyperspecBrowser#map left
			endif
			break
		case "chkShowAll" :
			prefs.options = SetBitValue(prefs.options, 2, s.checked)
			StructPut prefs, wStruct
			CleanupPixelTraces(0)
			
			if (prefs.mode == 0 && prefs.cursormode==1)
				PlotLineAndPixelTraces(prefs)
			endif
			
			if (prefs.mode == 4)	
				STRUCT Rect MQPixels
				if (MarqueeToImagePixels(MQPixels)) // a marquee is present
					AveragePixelsInRect(MQPixels, prefs)
				endif
				break
			endif
			
			if (prefs.mode == 5) // path
				PlotLineAndPixelTraces(prefs)
			endif
			
			if (prefs.mode == 6)
				wave/SDFR=GetDFR() wMask
				AveragePixelsROI(wMask)
				break
			endif
			
			if (prefs.mode == 8 || prefs.mode==7)
				AveragePixelsROI($"")
				break
			endif
				
			break
		case "chkHideYaxis" :
			prefs.options = SetBitValue(prefs.options, 3, s.checked)
			if (s.checked)
				ModifyGraph/W=HyperspecBrowser#PixelTraces tick(left)=3,noLabel(left)=2,axThick(left)=0, mirror=0 //, margin(left)=8
				ModifyGraph/W=HyperspecBrowser#Profiles tick(left)=3,noLabel(left)=2,axThick(left)=0, mirror=0 //, margin(left)=8
				if (WinType("HyperspecBrowser#ProfileMap"))
					ModifyGraph/W=HyperspecBrowser#ProfileMap tick(left)=3,noLabel(left)=2,axThick(left)=0, mirror=0 //, margin(left)=8
				endif
			else
				ModifyGraph/W=HyperspecBrowser#PixelTraces tick(left)=0,noLabel(left)=0,axThick(left)=1, mirror=1 //, margin(left)=0
				ModifyGraph/W=HyperspecBrowser#Profiles tick(left)=0,noLabel(left)=0,axThick(left)=1, mirror=1 //, margin(left)=0
				if (WinType("HyperspecBrowser#ProfileMap"))
					ModifyGraph/W=HyperspecBrowser#ProfileMap tick(left)=0,noLabel(left)=0,axThick(left)=1, mirror=1 //, margin(left)=0
				endif
			endif
			break
		case "chkAutoscale" :
			prefs.options = SetBitValue(prefs.options, 4, s.checked)
			if (s.checked)
				SetAxis/A=2/W=HyperspecBrowser#PixelTraces left

				if (prefs.options & 128)
					SetAxis/A=2/W=HyperspecBrowser#Profiles bottom
				else
					SetAxis/A=2/W=HyperspecBrowser#Profiles left
				endif
			else
				wave w3D = GetImageWave()
				WaveStats/Q/M=1 w3D
				SetAxis/W=HyperspecBrowser#PixelTraces left, v_min, v_max

				if (prefs.options & 128)
					SetAxis/W=HyperspecBrowser#Profiles bottom, v_min, v_max
				else
					SetAxis/W=HyperspecBrowser#Profiles left, v_min, v_max
				endif
			endif
			break
		case "chkLabels" :
			prefs.options = SetBitValue(prefs.options, 5, s.checked)
			if (s.checked)
				Label/W=HyperspecBrowser#PixelTraces left, ""
				Label/W=HyperspecBrowser#PixelTraces bottom, ""
				Label/W=HyperspecBrowser#Profiles left, ""
				Label/W=HyperspecBrowser#Profiles bottom, ""
				if (WinType("HyperspecBrowser#ProfileMap"))
					Label/W=HyperspecBrowser#ProfileMap left, ""
					Label/W=HyperspecBrowser#ProfileMap bottom, ""
				endif
			else
				Label/W=HyperspecBrowser#PixelTraces left, prefs.vlabel
				Label/W=HyperspecBrowser#PixelTraces bottom, prefs.hlabel
				string StrLabel = SelectString(prefs.mode & 5, "A  <-   distance   ->  B", "distance")
				if (prefs.options & 128) // swap xy in line profile graph
					Label/W=HyperspecBrowser#Profiles bottom, prefs.vlabel
					Label/W=HyperspecBrowser#Profiles left, StrLabel
				else
					Label/W=HyperspecBrowser#Profiles left, prefs.vlabel
					Label/W=HyperspecBrowser#Profiles bottom, StrLabel
				endif
				if (WinType("HyperspecBrowser#ProfileMap"))
					Label/W=HyperspecBrowser#ProfileMap left, StrLabel
					Label/W=HyperspecBrowser#ProfileMap bottom, prefs.hlabel
				endif
			endif
			break
		case "chkProfile" :
			prefs.options = SetBitValue(prefs.options, 6, s.checked) // bit 6 = 64
			DefineGuide/W=HyperspecBrowser FGHDIV={FL,s.checked?prefs.fg:1,FR}
			ShowProfile(prefs)
			if (s.checked)
				DoWindow/F HyperspecBrowser
			endif
			break
		case "chkSwapProfileAxes" :	// bit 7 = 128
			prefs.options = SetBitValue(prefs.options, 7, s.checked)
			ModifyGraph/W=HyperspecBrowser#Profiles swapXY=s.checked
			break
		case "chkInvertMask" :	// bit 8 = 256
			prefs.options = SetBitValue(prefs.options, 8, s.checked)
			break
		case "chkProfileMap" :
			prefs.options = SetBitValue(prefs.options, 9, s.checked)
			if (prefs.cursormode==1 || prefs.mode==5)
				ShowProfileMap(prefs)
			endif
			if (!s.checked)
				PlotLineAndPixelTraces(prefs)
			endif
			break
	endswitch
	StructPut prefs, wStruct
	
	// 0 1 2 3  4  5  6   7   8   9
	// 1 2 4 8 16 32 64 128 256 512
	
	return 0
end

static function MapMode(STRUCT PackagePrefs &prefs)
	EnableMapControls(prefs)
	ShowCursors(prefs, 2)
	wave w3D = GetImageWave()
	if (prefs.mapmode == 1) // RGB mode
		wave/SDFR=getDFR() wRGB
		if (!WaveExists(TraceNameToWaveRef("HyperspecBrowser#map", "wRGB")))
			AppendImage/W=HyperspecBrowser#map wRGB
		endif
		ReorderImages/W=HyperspecBrowser#map wDisplayMaskRGBA, {wRGB}
		ColorScale/W=HyperspecBrowser#map/K/N=scale
		ModifyGraph/Z/W=HyperspecBrowser#PixelTraces zColor(spectrum)=0
		TextBox/C/W=HyperspecBrowser#map/N=text0 NameOfWave(w3D) + " RGB map"
	else // color table mode
		RemoveImage/W=HyperspecBrowser#map/Z wRGB
		ColorScale/W=HyperspecBrowser#map/C/N=scale/F=0/A=RC/X=0/Y=0.00/E/B=1 image=$NameOfWave(w3D), heightPct=100, width=10, tickLen=4, minor=1
//		ResetColorTableMapTextbox(prefs, w3D) // don't need this because resetting cursors triggers update
	endif
end

static function EnableMapControls(STRUCT PackagePrefs &prefs)
	TabControl tabs  win=HyperspecBrowser, value=prefs.mapmode
	ModifyControlList/Z ControlNameList("HyperspecBrowser",";","*_ct"), win=HyperspecBrowser, disable=prefs.mapmode
	ModifyControlList/Z ControlNameList("HyperspecBrowser",";","*_rgb"), win=HyperspecBrowser, disable=!prefs.mapmode
	int i
	string ctrl
	if (prefs.mapmode) // RGB mode
		for (i=0;i<3;i++)
			if (!(prefs.rgboptions[i] & 1))
				sprintf ctrl, "*%d_rgb", i
				ModifyControlList/Z ControlNameList("HyperspecBrowser",";",ctrl), win=HyperspecBrowser, disable=2
			elseif (prefs.rgboptions[i] & 2)
				sprintf ctrl, "svm*%d_rgb", i
				ModifyControlList/Z ControlNameList("HyperspecBrowser",";",ctrl), win=HyperspecBrowser, disable=2
			endif
		endfor
	else // color table
		SetVariable svMin_ct, win=HyperspecBrowser, disable=prefs.ctoptions & 1 ? 2 : 0
		Slider sliderMin_ct win=HyperspecBrowser, disable=prefs.ctoptions & 1 ? 2 : 0
		SetVariable svMax_ct, win=HyperspecBrowser, disable=prefs.ctoptions & 2 ? 2 : 0
		Slider sliderMax_ct win=HyperspecBrowser, disable=prefs.ctoptions & 2 ? 2 : 0
	endif
end

static function logcolors(int isLog)
	string strImage = StringFromList(0,ImageNameList("HyperspecBrowser#map", ";"))
	ModifyImage/W=HyperspecBrowser#map $strImage log=isLog
	ColorScale/W=HyperspecBrowser#map/C/N=scale log=isLog
	if (WinType("HyperspecBrowser#ProfileMap"))
		ModifyImage/W=HyperspecBrowser#ProfileMap M_ImageLineProfile log=isLog
	endif
end

static function ButtonProc(STRUCT WMButtonAction &s) : ButtonControl
	if (s.eventCode != 2)
		return 0
	endif
	
	strswitch (s.ctrlName)
		case "btnZap":
			ZapPixel()
			break
		case "btnCopy":
			CopyPastePixel(0)
			break
		case "btnPaste":
			CopyPastePixel(1)
			break
		case "btnInfo":
			InfoMenu()
			break
		case "btnSettings":
			if (!WinType("HyperspecPrefsPanel"))
				MakePrefsPanel()
			else
				KillWindow/Z HyperspecPrefsPanel
			endif
			break
		case "BtnDone": // prefs panel
			KillWindow/Z HyperspecPrefsPanel
			break
	endswitch
	
	return 0
end

static function CopyPastePixel(int paste)
	DFREF dfr = GetDFR()
	wave/Z pixelcopy = dfr:pixelcopy
	wave/Z w3D = GetImageWave()
	variable pp = pcsr(C, "HyperspecBrowser#map")
	variable qq = qcsr(C, "HyperspecBrowser#map")
	
	if (paste)
		if (!WaveExists(pixelcopy) || numpnts(pixelcopy)!=DimSize(w3D, 2))
			return 0
		endif
		
		Duplicate/O/RMD=[pp][qq][] w3D dfr:backup/wave=bup
		string strNote = ""
		sprintf strNote, "%s;%g;%g;", NameOfWave(w3D), pp, qq
		note/K bup strNote
		
		w3D[pp][qq][] = pixelcopy[r]
		
		wave/SDFR=GetDFR() wStruct
		STRUCT PackagePrefs prefs
		StructGet prefs, wStruct
		PlotCrosshairPixel(prefs)
		if (prefs.mapmode == 1)
			ResetRGBmap(prefs, w3D)
		endif
	else // copy pixel
		Duplicate/O/RMD=[pp][qq][] w3D dfr:pixelcopy
	endif
end

static function ZapPixel()
	
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	// better to check wavenote?
	string str = ""
	#ifdef WINDOWS
	str = "Note: pixel-zapping is for ratioed data!\r\rDo you want to continue?\r\rCtrl-Z to undo."
	#else
	str = "Note: pixel-zapping is for ratioed data!\r\rDo you want to continue?\r\rCmd-Z to undo."
	#endif
	DoAlert 1, str
	if (v_flag == 2)
		return 0
	endif
	
	variable pp = pcsr(C, "HyperspecBrowser#map")
	variable qq = qcsr(C, "HyperspecBrowser#map")
	wave/Z w3D = GetImageWave()
	
	if (numtype(pp) || numtype(qq) || !WaveExists(w3D))
		return 0
	endif
	
	variable i, j
	Make/free/D/N=(DimSize(w3D, 2)) total
	variable count
		
	variable newX, newY
	variable xMax = DimSize(w3D, 0)
	variable yMax = DimSize(w3D, 1)
	
	for (i=-1;i<2;i++)
		for (j=-1;j<2;j++)
			newX = pp + i
			newY = qq + j
			if (newX>=0 && newY>=0 && newX<xMax && newY<Ymax && (i || j))
				count += 1
				total += w3D[newX][newY][p]
			endif
		endfor
	endfor
	total /= count
	DFREF dfr = GetDFR()
	Duplicate/O total dfr:backup/wave=bup
	bup = w3D[pp][qq][p]
	string strNote = ""
	sprintf strNote, "%s;%g;%g;", NameOfWave(w3D), pp, qq
	note/K bup strNote
	w3D[pp][qq][] = total[r]
	PlotCrosshairPixel(prefs)
	if (prefs.mapmode == 1)
		ResetRGBmap(prefs, w3D)
	endif
end

static function/wave GetImageWave()
	ControlInfo/W=HyperspecBrowser svDatacube
	wave/Z w3D = $s_value
	return w3D
end

static function UndoZap()
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	wave/Z w3D = GetImageWave()
	wave/Z/SDFR=GetDFR() backup
	if (!WaveExists(backup) || !WaveExists(w3D))
		return 0
	endif
	
	string strNote = note(backup)
	if (cmpstr(NameOfWave(w3D), StringFromList(0, strNote)))
		return 0
	endif
	
	variable pp = str2num(StringFromList(1, strNote))
	variable qq = str2num(StringFromList(2, strNote))
	
	w3D[pp][qq][] = backup[r]
	PlotCrosshairPixel(prefs)
end

// *** save wave ***

static function/WAVE SaveWaveUI(wave w, [string basename, int overwrite])
	
	overwrite = ParamIsDefault(overwrite) ? 1 : overwrite // default is to allow user to overwrite a wave
	if (ParamIsDefault(basename))
		basename = NameOfWave(w)
		if (cmpstr(basename, "_free_") == 0)
			basename = "wave"
		endif
	endif
	KillWaves/Z wwSavedWaves
	DoWindow/K SaveWaveDialog
	GetMouse
	variable left, top, right, bottom
	left = pixel2cpu(V_left) - 50
	top = pixel2cpu(V_top) - 50
	right = left + 220
	bottom = top + 150
	NewPanel/K=1/W=(left,top,right,bottom)/N=SaveWaveDialog as "Save Wave"
	SetVariable svName,pos={25,20},size={157,14},title="Wave Name", userdata(base) = basename
	SetVariable svName,value=_STR:UniqueName(basename, 1, 0), fsize=12, Proc=Hyperspec#SaveWaveSetVars
	PopupMenu popFolder,pos={25,50},size={157,14},title="Data Folder", fsize=12
	PopupMenu popFolder,value=#"GetDataFolder(0)", userdata(folder)=GetDataFolder(1), Proc=Hyperspec#SaveWavePopups
	Button btnCancel,pos={40,100},size={55,20},title="Cancel", Proc=Hyperspec#SaveWaveButtons
	Button btnDoIt,pos={130,100},size={50,20},title="Do It", Proc=Hyperspec#SaveWaveButtons, userdata=num2str(overwrite)

	SetVariable svName,userdata(ResizeControlsInfo)=A"!!,C,!!#<X!!#A,!!#<Hz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
	SetVariable svName,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	SetVariable svName,userdata(ResizeControlsInfo)+=A"zzz!!#u:Du]k<zzzzzzzzzzzzzz!!!"
	PopupMenu popFolder,userdata(ResizeControlsInfo)=A"!!,C,!!#>V!!#@`!!#<Xz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
	PopupMenu popFolder,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	PopupMenu popFolder,userdata(ResizeControlsInfo)+=A"zzz!!#u:Du]k<zzzzzzzzzzzzzz!!!"
	Button btnCancel,userdata(ResizeControlsInfo)=A"!!,D/!!#@,!!#>j!!#<Xz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	Button btnCancel,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Button btnCancel,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	Button btnDoIt,userdata(ResizeControlsInfo)=A"!!,Fg!!#@,!!#>V!!#<Xz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	Button btnDoIt,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Button btnDoIt,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	SetWindow kwTopWin,userdata(ResizeControlsInfo)=A"!!*'\"z!!#Ak!!#A%zzzzzzzzzzzzzzzzzzzzz"
	SetWindow kwTopWin,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzzzzzzzzzzzzzzz"
	SetWindow kwTopWin,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzzzzzzzzz!!!"
	
	// resizing panel hook
	SetWindow SaveWaveDialog hook(ResizeControls)=ResizeControls#ResizeControlsHook

	PauseForUser SaveWaveDialog
	wave/Z/WAVE wwSavedWaves
	if (WaveExists(wwSavedWaves))
		wave/Z saved = wwSavedWaves[0]
		string strPathToNewWave = GetWavesDataFolder(saved, 2)
		KillWaves/Z saved
		Duplicate/O w $strPathToNewWave /wave=saved
		KillWaves wwSavedWaves
		return saved
	endif
	return $""
end

static function SaveWavePopups(STRUCT WMPopupAction &s) : PopupMenuControl
	
	if (s.eventCode != 3)
		return 0
	endif

	string strFolder = StringFromList(0, SelectFolder(s.popStr))
	if (strlen(strFolder))
		PopupMenu popFolder, win=$s.win, userdata(folder)=strFolder
		PopupMenu popFolder, win=$s.win, value=#"Hyperspec#GetSelectedFolder(0)"
	endif
	
	CheckNameSetvar()

	return 0
end

static function SaveWaveButtons(STRUCT WMButtonAction &s) : ButtonControl

	if (s.eventcode != 2)
		return 0
	endif
		
	strswitch (s.ctrlName)
		case "btnDoIt":
			int overwrite = str2num(s.userdata)
			ControlInfo/W=$s.win svName
			if (!overwrite && exists(GetSelectedFolder(1)+PossiblyQuoteName(s_value)))
				DoAlert 0, s_value + " exists."
				return 0
			endif
			KillWaves/Z $GetSelectedFolder(1)+PossiblyQuoteName(s_value)
			wave/Z w = $GetSelectedFolder(1)+PossiblyQuoteName(s_value)
			if (WaveExists(w))
				DoAlert 0, s_value + " exists and is in use, cannot overwrite."
				return 0
			endif
			Make/O/N=0 $GetSelectedFolder(1)+PossiblyQuoteName(s_value) /wave=w
			Make/O/WAVE wwSavedWaves = {w}
		case "btnCancel" : // Cancel or Do It
			DoWindow/K $s.win
			break
	endswitch
	return 0
end

static function SaveWaveSetVars(STRUCT WMSetVariableAction &s)
	
	if (s.eventCode < 0) // excludes Igor8 eventcodes -2 and -3
		return 0
	endif
	
	if (cmpstr(s.ctrlName, "svName") == 0)
		CheckNameSetvar()
	endif
	return 0
end

static function CheckNameSetvar()
	ControlInfo/W=SaveWaveDialog svName
	if (strlen(S_Value) == 0)
		S_Value = UniqueName(GetUserData("SaveWaveDialog", "svName", "base"), 1, 0)
		SetVariable svName win=SaveWaveDialog, value=_STR:S_Value
	endif
	if (exists(GetSelectedFolder(1) + S_Value))
		SetVariable svName win=SaveWaveDialog, valueBackColor=(0xFFFF,0,0)
	else
		SetVariable svName win=SaveWaveDialog, valueBackColor=0
	endif
end

static function/T GetSelectedFolder(int fullpath)
	string strFolder = GetUserData("SaveWaveDialog", "popFolder", "folder")
	return SelectString(fullpath, ParseFilePath(0, strFolder, ":", 1, 0), strFolder)
end

static function/T SelectFolder(string selection)
	CreateBrowser/M Prompt="Select Data Folder"
	ModifyBrowser/M showWaves=0, showVars=0, showStrs=0
	ModifyBrowser/M ShowInfo=0, showPlot=0
	ModifyBrowser/M select=selection
	ModifyBrowser/M showModalBrowser
	if (V_Flag == 0)
		return "" // User cancelled
	endif
	return S_BrowserList
end

static function pixel2cpu(variable pixel, [string win])
	variable expansion = ParamIsDefault(win) ? 1 : PanelResolution(win)/PanelResolution("")
	return (ScreenResolution > 96 ? pixel * 72 / ScreenResolution : pixel) / expansion
end

static function/S PolyMenu(string str)
	
	DFREF dfr = GetDFR()
	wave/Z polyX = dfr:ROI_x
	wave/Z polyY = dfr:ROI_y
	
	if (WaveExists(polyY)==0 || numpnts(polyY)<3)
		return ""
	endif
	
	GetLastUserMenuInfo
	if (cmpstr(S_graphName, "HyperspecBrowser#map"))
		return ""
	endif
	
	CheckDisplayed/W=HyperspecBrowser#map polyY
	if (!V_flag)
		return ""
	endif
	
	if (abs(polyy[numpnts(polyy)-1]-polyy[0]) > 1e-5)
		return ""
	elseif (abs(polyx[numpnts(polyx)-1]-polyx[0]) > 1e-5)
		return ""
	endif
	
	Make/free wx={AxisValFromPixel(S_graphName, "bottom", V_mouseX)}
	Make/free wy={AxisValFromPixel(S_graphName, "left", V_mouseY)}
	FindPointsInPoly wx, wy, polyx, polyy
	wave W_inPoly
	int inPoly = W_inPoly[0]
	KillWaves/Z W_inPoly
	return SelectString(inPoly, "", str)
end

static function inPoly(variable x, variable y)
	DFREF dfr = GetDFR()
	wave wx = dfr:ROI_x
	wave wy = dfr:ROI_y
	Make/free wxp = {x}
	Make/free wyp = {y}
	
	FindPointsInPoly wxp, wyp, wx, wy
	wave W_inPoly
	int inPoly = W_inPoly[0]
	KillWaves/Z W_inPoly
	return inPoly
end



// save a copy of profile map and display it
static function SaveMap()
	DFREF dfr = getDFR()
	wave/SDFR=dfr wStruct, M_ImageLineProfile
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	string strName = UniqueName("ProfileMap", 1, 0)
	duplicate M_ImageLineProfile $strName/wave = w
	Display/K=1; AppendImage w
	ModifyGraph standoff=0,tick=2,btLen=3 
	string strColorTable = StringFromList(prefs.ctnum, CTabList())
	string strTrace = stringfromlist(0, ImageNameList("", ";"))
	ModifyImage $strTrace ctab={*,*,$strColorTable,0}, ctabAutoscale=1	
	if (prefs.options & 1)
		SetAxis/A/R bottom
	else
		SetAxis/A bottom
	endif
	SetAxis/A left
	Label bottom, prefs.hlabel
	
	Label left SelectString(prefs.mode & 5, "A  <-   distance   ->  B", "distance")
	ModifyGraph mirror=1, axisOnTop=1	
end

// create copies of displayed profiles and display them
static function SaveProfile()
	
	DFREF dfr = getDFR()
	wave/SDFR=dfr/Z wStruct, distanceCTline, distanceRGBline, LineProfile, LineProfileR, LineProfileG, LineProfileB
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	string strNewName = UniqueName("Profile", 1, 0)
	string strProfileWave = ""
	if (prefs.mapmode == 1)
		if (!(prefs.rgboptions[0]&1 || prefs.rgboptions[1]&1 || prefs.rgboptions[2]&1))
			return 0
		endif
		make/N=(numpnts(distanceRGBline), 4) $strNewName/wave=w
		SetDimLabel 1, 0, distance, w
		SetDimLabel 1, 1, red, w
		SetDimLabel 1, 2, green, w
		SetDimLabel 1, 3, blue, w
		FastOp w = (NaN)
		w[][0] = distanceRGBline[p]
		Display/K=1
		int i
		for (i=0;i<3;i++)
			if (prefs.rgboptions[i] & 1)
				strProfileWave = "LineProfile" + "RGB"[i]
				wave profileWave = dfr:$strProfileWave
				w[][i+1] = profileWave[p]
				AppendToGraph w[][i+1]/TN=$("Profile"+"RGB"[i]) vs w[][0]
				ModifyGraph rgb($("Profile"+"RGB"[i]))=(i==0?65535:0, i==1?65535:0, i==2?65535:0)
			endif
		endfor
	else
		Make/N=(numpnts(distanceCTline), 2) $strNewName/wave=w
		w[][0] = distanceCTline[p]
		w[][1] = LineProfile[p]
		Display/K=1 w[][1] vs w[][0]
		ModifyGraph rgb=(prefs.specrgb.red,prefs.specrgb.green,prefs.specrgb.blue), mode=0
	endif
	
	ModifyGraph standoff=0,tick=2,btLen=3
	SetAxis/A bottom
	if (prefs.options & 16) // autoscale
		doupdate
		SetAxis/A=2 left
	else
		WaveStats/Q/M=1 GetImageWave()
		SetAxis left, v_min, v_max
	endif	
	ModifyGraph mirror=1, axisOnTop=1	
	Label left, prefs.vlabel
	
	Label bottom, SelectString(prefs.mode & 5, "A  <-   distance   ->  B", "distance")
//	ModifyGraph swapXY=(prefs.options&128) > 0	
end

static function SetIgorDrawingMode(int start)
	wave/SDFR=getDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	if (start == 1)
		prefs.mode = 7
		StructPut prefs, wStruct
		
		ShowCursors(prefs, 1)
		ShowTools/W=HyperspecBrowser arrow
		SetDrawLayer/W=HyperspecBrowser#map ProgFront
		SetDrawEnv/W=HyperspecBrowser#map linefgc=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue), fillpat=0, xcoord=bottom, ycoord=left	, Save

	else
		GraphNormal/W=HyperspecBrowser#map
		HideTools/W=HyperspecBrowser
		if (start == 0)
			if (!AveragePixelsROI($"")) // no shapes
				prefs.mode = 0
				ShowCursors(prefs, 1)
				ReplotPixelTraces(prefs) // restore spectra plot
				// restore profile plots
				ShowProfileMap(prefs)
				ShowProfile(prefs)
			endif	
		endif
	endif
end

static function SaveAveragePixelTrace()
	wave w = TraceNameToWaveRef("HyperspecBrowser#PixelTraces", "spectrum")
	SaveWaveUI(w)
end

// add a dialog to select basename
static function SaveAllPixelTraces()
	string strTraces = TraceNameList("HyperspecBrowser#PixelTraces", ";", 1)
	string strTrace = ""
	strTraces = RemoveFromList("spectrum", strTraces)
	int iNumTraces = ItemsInList(strTraces)
	
	if (!iNumTraces)
		return 0
	endif
	
	wave w3D = GetImageWave()
	wave/SDFR=getDFR() wMask, wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	Duplicate/free wMask wTest
	string strBaseName = NameOfWave(w3D)
	wTest = CheckOverwrite(p, q, wMask[p][q], strBaseName)
	
	if (WaveMax(wTest) > 0)
		DoAlert 1, "Some waves will be overwritten. Continue?"
		if (v_flag == 2)
			return 0
		endif
	endif
		
	int IsLine = prefs.mode==0 && prefs.cursormode==1
	
	wMask = Save1DPixelWave(p, q, w3D, wMask, IsLine)
end

static function SavePath()
	string strBaseName = UniqueName("Path", 1, 0)
	wave/Z/SDFR=GetDFR():DrawPath ROI_x, ROI_y
	if (!waveexists(ROI_x) || !waveexists(ROI_y))
		return 0
	endif
	make/N=(numpnts(ROI_x),2) $strBaseName/wave=wPath
	wPath[][0] = ROI_x[p]
	wPath[][1] = ROI_y[p]
end

static function SaveROI()
	wave w3D = GetImageWave()
	wave/SDFR=getDFR() wMask, wDisplayMaskRGBA, wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	int pathwayROI = prefs.mode==5 || (prefs.mode==0 && prefs.cursormode==1)
	
	if (pathwayROI)
		SetOverlayTransparency(1)
		DoUpdate/W=HyperspecBrowser#map
	else
		wMask = wDisplayMaskRGBA[p][q][3] == 0
	endif
		
	if (WaveMax(wMask) == 0 || WaveMin(wMask) == 1)
		DoAlert 0, "No ROI selected"
		return 0
	endif
	
	if (prefs.options & 256)
		Duplicate/free wMask wInverseMask
		FastOp wInverseMask = 255 * wInverseMask
		ImageTransform/O invert wInverseMask
		wave wMask = wInverseMask
	endif
	
	wave/Z wROI = $NameOfWave(w3D) + "Mask"
	if (WaveExists(wROI))
		DoAlert 1, "Append layer to " + NameOfWave(wROI) + "?"
		if (v_flag == 2)
			SaveWaveUI(wMask, basename=NameOfWave(wROI))
			
			if (pathwayROI)
				SetOverlayTransparency(-1)
			endif
			
			return 0
		endif
		Concatenate/NP=2/free {wROI, wMask}, w
		Duplicate/O w wROI
		
		if (pathwayROI)
			SetOverlayTransparency(-1)
		endif
		
		return 0
	else
		DoAlert 0, "Saving ROI to " + NameOfWave(w3D) + "Mask"	
	endif	
	Duplicate wMask $NameOfWave(w3D) + "Mask"
	
	if (pathwayROI)
		SetOverlayTransparency(-1)
	endif	
end

static function CheckOverwrite(int row, int col, variable DoIt, string strBaseName)
	if (!DoIt)
		return 0
	endif
	string strWave = ""
	sprintf strWave, "%s_row%dcol%d", strBaseName, row, col
	return WaveExists($strWave)
end

static function Save1DPixelWave(int row, int col, wave w3D, variable doIt, int line)
	string strWave = ""
	sprintf strWave, "%s_row%dcol%d", NameOfWave(w3D), row, col
	variable distance, x0, y0, x1, y1
	
	if (doIt)
		Make/O/N=(DimSize(w3D, 2))/Y=(WaveType(w3D)) $strWave/wave=spectrum
		SetScale/P x, DimOffset(w3D, 2), DimDelta(w3D, 2), spectrum
		spectrum = w3D[row][col][p]
		if (line)
			x0 = hcsr(A, "HyperspecBrowser#map") // inefficient
			y0 = vcsr(A, "HyperspecBrowser#map")
			x1 = IndexToScale(w3D, row, 0)
			y1 = IndexToScale(w3D, col, 1)
			distance = sqrt((y1-y0)^2 + (x1-x0)^2)
			note spectrum, "distance=" + num2str(distance)
		endif
	endif
	return doIt
end

static function ShowCursors(STRUCT PackagePrefs &prefs, int winnum)
	
	string strWin = ""
	
	if (winnum & 1) // bit zero, map graph
		if (prefs.mode==0 && prefs.cursormode==0) // crosshair cursor
		 	if (strlen(CsrInfo(C, "HyperspecBrowser#map"))==0)
				wave w3D = GetImageWave()
				Cursor/W=HyperspecBrowser#map/P/I/S=0/H=1/C=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue)/N=1 C $NameOfWave(w3D) prefs.crosshair.h, prefs.crosshair.v
			endif
			Button btnZap, win=HyperspecBrowser, disable=0
			Button btnCopy, win=HyperspecBrowser, disable=0
			Button btnPaste, win=HyperspecBrowser, disable=0
		else
			Cursor/W=HyperspecBrowser#map/K C
			Button btnZap, win=HyperspecBrowser, disable=2
			Button btnCopy, win=HyperspecBrowser, disable=2
			Button btnPaste, win=HyperspecBrowser, disable=2
		endif
				
		if (prefs.mode==0 && prefs.cursormode==1) // line cursors
			wave w3D = GetImageWave()
			Cursor/W=HyperspecBrowser#map/P/I/C=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue)/N=1/S=1 A $NameOfWave(w3D) prefs.line.left, prefs.line.top
			Cursor/W=HyperspecBrowser#map/P/I/C=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue)/N=1/S=1 B $NameOfWave(w3D) prefs.line.right, prefs.line.bottom
			wave/SDFR=GetDFR() wLine
			RemoveFromGraph/W=HyperspecBrowser#map/Z wLine
			AppendToGraph/W=HyperspecBrowser#map wLine[][1] vs wLine[][0]
			ModifyGraph/W=HyperspecBrowser#map rgb(wLine)=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue)
		else
			Cursor/W=HyperspecBrowser#map/K A
			Cursor/W=HyperspecBrowser#map/K B
			RemoveFromGraph/W=HyperspecBrowser#map/Z wLine
		endif
		
		if (WaveExists(TraceNameToWaveRef("HyperspecBrowser#map", "ROI_nodes")))
			ModifyGraph/Z rgb(ROI_nodes)=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue)
			wave/Z/SDFR=GetDFR() wStruct
			wave/Z/SDFR=GetDFR():DrawPath ROI_x, ROI_y
			StructPut prefs, wStruct // wave2poly() reads prefs from package folder
			Wave2Poly(ROI_x, ROI_y, "HyperspecBrowser#map", "bottom", "left", layer="ProgFront", name="ROI")
		endif
		
	endif
	
	if (winnum & 2) // Pixel traces plot

		if (prefs.mapmode==0) // (prefs.mapmode==0 && prefs.cursormode!=2) // color table map, not line profile
			ControlInfo/W=HyperspecBrowser svPlane_ct
			GetAxis/W=HyperspecBrowser#PixelTraces/Q bottom
			if (WaveExists(TraceNameToWaveRef("HyperspecBrowser#PixelTraces", "spectrum")))
				Cursor/W=HyperspecBrowser#PixelTraces/P/F/S=2/H=2/N=1/C=(0, 0, 0) B spectrum (v_value-V_min)/(V_max-V_min), 0.5
			endif
		else
			Cursor/W=HyperspecBrowser#PixelTraces/K B
		endif
	
		if (prefs.mapmode == 1)	// RGB map
			int i
			string csr
			variable plane
			GetAxis/W=HyperspecBrowser#PixelTraces/Q bottom
			for (i=0;i<3;i++)
				csr = "CDE"[i]
				if (prefs.rgboptions[i] & 1)
					ControlInfo/W=HyperspecBrowser $"svplane" + num2str(i) + "_rgb"
					plane = v_value
					Cursor/W=HyperspecBrowser#PixelTraces/P/F/S=2/H=2/N=1/C=(i==0?65535:0, i==1?65535:0, i==2?65535:0) $csr spectrum (plane-V_min)/(V_max-V_min), 0.5
				else
					Cursor/W=HyperspecBrowser#PixelTraces/K $csr
				endif
			endfor
		else
			Cursor/W=HyperspecBrowser#PixelTraces/K C
			Cursor/W=HyperspecBrowser#PixelTraces/K D
			Cursor/W=HyperspecBrowser#PixelTraces/K E
		endif
	endif
		
	if (winnum&4 && WinType("HyperspecBrowser#ProfileMap")) // line map graph
		GetAxis/W=HyperspecBrowser#ProfileMap/Q bottom
		
		if (prefs.mapmode == 0)
			ControlInfo/W=HyperspecBrowser svPlane_ct
			
//			if (0)//(IgorVersion() < 9)
//				Cursor/W=HyperspecBrowser#ProfileMap/F/S=2/H=2/N=1/I/C=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue) B M_ImageLineProfile v_value, 0.5
//			else

			Cursor/W=HyperspecBrowser#ProfileMap/P/F/S=2/H=2/N=1/I/C=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue) B M_ImageLineProfile (v_value-V_min)/(V_max-V_min), 0.5
//			endif
			Cursor/W=HyperspecBrowser#ProfileMap/K C
			Cursor/W=HyperspecBrowser#ProfileMap/K D
			Cursor/W=HyperspecBrowser#ProfileMap/K E
		elseif (prefs.mapmode ==1)
			GetAxis/W=HyperspecBrowser#ProfileMap/Q bottom
			for (i=0;i<3;i++)
				csr = "CDE"[i]
				if (prefs.rgboptions[i] & 1)
					ControlInfo/W=HyperspecBrowser $"svplane" + num2str(i) + "_rgb"
					plane = v_value
					
					
//					if (IgorVersion() < 9)
////						Cursor/W=HyperspecBrowser#ProfileMap/P/F/S=2/H=2/N=1/I/C=(i==0?65535:0, i==1?65535:0, i==2?65535:0) $csr M_ImageLineProfile plane, 0.5
//						Cursor/W=HyperspecBrowser#ProfileMap/P/F/S=2/H=2/N=1/I/C=(i==0?65535:0, i==1?65535:0, i==2?65535:0) $csr M_ImageLineProfile (plane-V_min)/(V_max-V_min), 0.5
//					else
//					
//					
						Cursor/W=HyperspecBrowser#ProfileMap/P/F/S=2/H=2/N=1/I/C=(i==0?65535:0, i==1?65535:0, i==2?65535:0) $csr M_ImageLineProfile (plane-V_min)/(V_max-V_min), 0.5
//					endif				
				else
					Cursor/W=HyperspecBrowser#ProfileMap/K $csr
				endif
			endfor
			Cursor/W=HyperspecBrowser#ProfileMap/K B
		endif
	endif
	
end

// show = 0: clear mask & overlay; show = 1: sync with wMask
static function SetOverlayTransparency(int show)
	wave/SDFR=GetDFR() wDisplayMaskRGBA, wMask
	if (show == -1)
		wDisplayMaskRGBA[][][3] = 0
	elseif (show)
		wDisplayMaskRGBA[][][3] = 40000 * !wMask[p][q]
	else
		FastOp wMask = 0
		wDisplayMaskRGBA[][][3] = 0
	endif
end


// if wMask is null, an image mask is created from drawing layer object(s)
static function AveragePixelsROI(wave/Z wMask, [int alert])
	alert = ParamIsDefault(alert) ? 0 : alert
	
	DFREF dfr = GetDFR()
	
	if (!WaveExists(wMask))
		KillWaves/Z M_ROIMask
		ImageGenerateROIMask/W=HyperspecBrowser#map $StringFromList(0,ImageNameList("HyperspecBrowser#map", ";"))
		if (!v_flag)
			if (alert)
				DoAlert 0, "No drawing objects to analyse.\r\rSelect 'Draw Shapes' to create a drawing object."
			endif
			return 0
		endif
		wave/SDFR=dfr wMask
		wave M_ROIMask
		wMask = M_ROIMask
		KillWaves/Z M_ROIMask
		SetOverlayTransparency(1)
	endif
	
	wave w3D = GetImageWave()
	
	MatrixOP/free w = sumRows(sumCols(w3D*wMask))
	variable numSpectra = sum(wMask)
	
	wave/SDFR=dfr spectrum
	Redimension/D spectrum
	spectrum = w[0][0][p] / numSpectra
	
	string strTextBox = ""
	sprintf strTextBox, "Masked Region, %d pixels", numSpectra
	TextBox/C/W=HyperspecBrowser#PixelTraces/N=text1 strTextBox
	note/K spectrum "source=" + NameOfWave(w3D) + " masked region"
	
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct // load local prefs
	
	if (prefs.mode!=5 && prefs.mode!=8 && prefs.mode!=6)
		prefs.mode = 7
	endif
	
	StructPut prefs, wStruct
	
	ShowProfileMap(prefs)
	ShowProfile(prefs)
	
	CleanupPixelTraces(0)
	if (prefs.options & 4) // plot all spectra
		wMask = DisplayPixelTrace(p, q, w3D, wMask)
		ReorderTraces/W=HyperspecBrowser#PixelTraces _front_, {spectrum}
	endif
	return 1
end

static function PlotPathPixelTraces(STRUCT PackagePrefs &prefs, wave xPath, wave yPath)
	
	DFREF dfr = GetDFR()
	DFREF dfSav = GetDataFolderDFR()
	NewDataFolder/O/S dfr:pathprofiles
	int i
	
	wave/SDFR=dfr wMask, spectrum
	wave w3D = GetImageWave()
		
		
	if (numpnts(xPath) == 0)
		Make/O/N=0 W_LineProfileX
		Make/O/N=0 W_LineProfileY
		return 0
	elseif (numpnts(xPath) == 1)
		Make/O W_LineProfileX = {xPath[0]}
		Make/O W_LineProfileY = {yPath[0]}
	else
		// plane is arbitrary, because here we're only interested in the path
		ImageLineProfile/SC/P=0 srcWave=GetImageWave(), xWave=xPath, yWave=yPath
	endif
	
	wave/Z W_LineProfileX, W_LineProfileY
	
	Concatenate/free {W_LineProfileX, W_LineProfileY}, wIndex
	wIndex = scaleToIndex(w3D, wIndex, q)
	for (i=DimSize(wIndex,0)-2;i>=0;i--)
		if (wIndex[i][0] == wIndex[i+1][0] && wIndex[i][1] == wIndex[i+1][1])
			DeletePoints/M=0 i, 1, wIndex
		endif
	endfor

	FastOp spectrum = 0
	FastOp wMask = 0
	wMask[wIndex] = 1

	CleanupPixelTraces(0)
	variable numSpectra = DimSize(wIndex, 0)
	for (i=0;i<numSpectra;i++)
		if (prefs.options & 4)
			DisplayPixelTrace(wIndex[i][0], wIndex[i][1], w3D, 1)
		endif
		spectrum += w3D[wIndex[i][0]][wIndex[i][1]][p]
	endfor
	spectrum /= numSpectra

	SetDataFolder dfSav
	ReorderTraces/W=HyperspecBrowser#PixelTraces _front_, {spectrum}
	
	string strTextBox = ""
	sprintf strTextBox, "Path, %d pixels", numSpectra
	TextBox/C/W=HyperspecBrowser#PixelTraces/N=text1 strTextBox
	note/K spectrum "source=" + NameOfWave(w3D) + ", " + strTextBox
end

static function ReplotPixelTraces(STRUCT PackagePrefs &prefs)
	CleanupPixelTraces(0)
	if (prefs.mode == 0 && (prefs.cursormode==1))
		PlotLineAndPixelTraces(prefs)
	endif
	
	if (prefs.mode == 0 && prefs.cursormode==0)
		PlotCrosshairPixel(prefs)
	endif
	
	if (prefs.mode)
		wave/SDFR=GetDFR() wMask
		AveragePixelsROI(wMask)
	endif
end

static function PlotLineAndPixelTraces(STRUCT PackagePrefs &prefs)	
	CleanupPixelTraces(0)	
	if (prefs.mode == 5)
		wave/Z/SDFR=GetDFR():DrawPath ROI_x, ROI_y
		return PlotPathPixelTraces(prefs, ROI_x, ROI_y)
	elseif (prefs.mode==0 && prefs.cursormode==1)
		return PlotLineAverageAndPixelTraces(prefs)
	endif
	return 0
end

// calculate average spectrum for Bresenham line defined by prefs.line rect
// sync mask wave with line pixels
// optionally plot spectrum for each pixel
static function PlotLineAverageAndPixelTraces(STRUCT PackagePrefs &prefs)
	
	wave/SDFR=GetDFR() wLine, spectrum
	wave w3D = GetImageWave()
	
	variable x0 = IndexToScale(w3D, prefs.line.left, 0)
	variable x1 = IndexToScale(w3D, prefs.line.right, 0)
	variable y0 = IndexToScale(w3D, prefs.line.top, 0)
	variable y1 = IndexToScale(w3D, prefs.line.bottom, 0)
	wLine = {{x0, x1}, {y0, y1}}
	
	int numPixels	
	if (abs(prefs.line.bottom - prefs.line.top) < abs(prefs.line.right - prefs.line.left))
		numPixels = BresenhamLineLow(prefs, w3D)
	else
		numPixels = BresenhamLineHigh(prefs, w3D)
	endif
			
	ReorderTraces/W=HyperspecBrowser#PixelTraces _front_, {spectrum}
	
	string strTextBox = ""
	sprintf strTextBox, "Line, [%d][%d]-[%d][%d], %d pixels", prefs.line.left, prefs.line.top, prefs.line.right, prefs.line.bottom, numPixels
	TextBox/C/W=HyperspecBrowser#PixelTraces/N=text1 strTextBox
	note/K spectrum "source=" + NameOfWave(w3D) + ", " + strTextBox
	
	return numPixels
end

static function BresenhamLineLow(STRUCT PackagePrefs &prefs, wave w3D)
	int p0, q0, p1, q1
	if (prefs.line.left > prefs.line.right)
		p0 = prefs.line.right
		q0 = prefs.line.bottom
		p1 = prefs.line.left
		q1 = prefs.line.top
	else
		p0 = prefs.line.left
		q0 = prefs.line.top
		p1 = prefs.line.right
		q1 = prefs.line.bottom
	endif
		
	DFREF dfr = GetDFR()
	wave/SDFR=dfr spectrum, wMask
	FastOp spectrum = 0
	FastOp wMask = 0
	Redimension/D spectrum
		
	int dp = p1 - p0
	int dq = q1 - q0
	int yi = 1
	if (dq < 0)
		yi = -1
		dq = -dq
	endif
	int D = (2 * dq) - dp
	int qq = q0

	int pp
	variable i = 0
	for (pp=p0;pp<=p1;pp++)
		
		i += 1
		wMask[pp][qq] = 1
		if (prefs.options & 4)
			DisplayPixelTrace(pp, qq, w3D, 1)
		endif
		spectrum += w3D[pp][qq][p]
		
		if (D > 0)
			qq = qq + yi
			D = D + (2 * (dq - dp))
		else
			D = D + 2*dq
		endif
	endfor
	spectrum /= i
	return i
end

static function BresenhamLineHigh(STRUCT PackagePrefs &prefs, wave w3D)
	int p0, q0, p1, q1
	if (prefs.line.top > prefs.line.bottom)
		p0 = prefs.line.right
		q0 = prefs.line.bottom
		p1 = prefs.line.left
		q1 = prefs.line.top
	else
		p0 = prefs.line.left
		q0 = prefs.line.top
		p1 = prefs.line.right
		q1 = prefs.line.bottom
	endif
	
	DFREF dfr = GetDFR()
	wave/SDFR=dfr spectrum, wMask
	FastOp spectrum = 0
	FastOp wMask = 0
	Redimension/D spectrum
	
	int dp = p1 - p0
	int dq = q1 - q0
	
	int p_i = 1
	if (dp < 0)
		p_i = -1
		dp = -dp
	endif
	int D = (2 * dp) - dq
	int pp = p0

	int qq
	variable i = 0
	for (qq=q0;qq<=q1;qq++)
		
		i += 1
		wMask[pp][qq] = 1
		if (prefs.options & 4)
			DisplayPixelTrace(pp, qq, w3D, 1)
		endif
		spectrum += w3D[pp][qq][p]
		
		if (D > 0)
			pp = pp + p_i
			D = D + (2 * (dp - dq))
		else
			D = D + 2*dp
		endif
	endfor
	spectrum /= i
	return i
end

// plot a trace representing one pixel of datacube
static function DisplayPixelTrace(int row, int col, wave w3D, variable doIt)
	string strTrace = ""
	sprintf strTrace, "row%dcol%d", row, col
	RemoveFromGraph/Z/W=HyperspecBrowser#PixelTraces $strTrace
	if (doIt)
		AppendToGraph/W=HyperspecBrowser#PixelTraces w3D[row][col][]/TN=$strTrace
		ModifyGraph/Z/W=HyperspecBrowser#PixelTraces rgb($strTrace)=(52428,52428,52428)
	endif
	return doIt
end

static function CleanupPixelTraces(int all)
	string strTraces = TraceNameList("HyperspecBrowser#PixelTraces", ";", 1)
	int numTraces = ItemsInList(strTraces)
	int i
	string strTrace = ""
	for (i=numTraces-1;i>=0;i--)
		strTrace = StringFromList(i, strTraces)
		if (all || cmpstr(strTrace, "spectrum"))
			RemoveFromGraph/W=HyperspecBrowser#PixelTraces $strTrace
		endif
	endfor
end

// uses integer maths. This is a teeny bit slower than adding points to the output waves during the Bresenham calculation.
static function ResetLineProfiles(STRUCT PackagePrefs &prefs, wave w3D)
	
	if (!(prefs.options & 64))
		return 0
	endif
	
	if (prefs.mode == 5)
		wave/Z/SDFR=GetDFR():DrawPath ROI_x, ROI_y
		return ResetPathProfiles(prefs, ROI_x, ROI_y)
	endif
	
	Make/free/N=(2,2) wPath
	wPath = {{prefs.line.left, prefs.line.right},{prefs.line.top, prefs.line.bottom}}
	wave w2D = GetBresenhamPath(wPath)
		
	DFREF dfr = GetDFR()
	int numPixels = DimSize(w2D, 0)
	int type = WaveType(w3D)
	
	variable originX = IndexToScale(w3D, prefs.line.left, 0)
	variable originY = IndexToScale(w3D, prefs.line.top, 1)
	
	if (prefs.mapmode == 0)
		Make/O/N=(numPixels) dfr:distanceCTline/wave=distanceCTline
		Make/O/N=(numPixels)/Y=(type) dfr:LineProfile/wave=LineProfile
		LineProfile = w3D[ w2D[p][0] ][ w2D[p][1] ][prefs.ctlayer]
		distanceCTline = sqrt(( IndexToScale(w3D, w2D[p][0], 0) - originX)^2 + (IndexToScale(w3D, w2D[p][1], 1) - originY)^2)
	elseif (prefs.mapmode == 1)
		Make/O/N=(numPixels) dfr:distanceRGBline/wave=distanceRGBline
		Make/O/N=(numPixels)/Y=(type) dfr:LineProfileR/wave=LineProfileR
		Make/O/N=(numPixels)/Y=(type) dfr:LineProfileG/wave=LineProfileG
		Make/O/N=(numPixels)/Y=(type) dfr:LineProfileB/wave=LineProfileB
		distanceRGBline = sqrt(( IndexToScale(w3D, w2D[p][0], 0) - originX)^2 + (IndexToScale(w3D, w2D[p][1], 1) - originY)^2)
		if (prefs.rgboptions[0] & 1)
			LineProfileR = w3D[ w2D[p][0] ][ w2D[p][1] ][prefs.rgblayer[0]]
		else
			FastOp LineProfileR = (NaN)
		endif
		if (prefs.rgboptions[1] & 1)
			LineProfileG = w3D[ w2D[p][0] ][ w2D[p][1] ][prefs.rgblayer[1]]
		else
			FastOp LineProfileG = (NaN)
		endif
		if (prefs.rgboptions[2] & 1)
			LineProfileB = w3D[ w2D[p][0] ][ w2D[p][1] ][prefs.rgblayer[2]]
		else
			FastOp LineProfileB = (NaN)
		endif
	endif
	
	TextBox/C/W=HyperspecBrowser#Profiles/N=textLineProfile "Line Profile"
end

static function PixelEdit(int start)
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	if (start)
		wave/SDFR=GetDFR() wMask, wDisplayMaskRGBA
				
		if (prefs.mode == 5 || prefs.mode==8)
			SetOverlayTransparency(1) // for mode 5
			Duplicate/free wDisplayMaskRGBA wDisplayMaskBUP
			DrawPath(0, 0)
			FastOp wDisplayMaskRGBA = wDisplayMaskBUP
		endif
		prefs.mode = 6
		ShowCursors(prefs, 1)
		StructPut prefs, wStruct
		
		// if we are displaying a masked area, use pixel editing to adjust the displayed mask
		WaveStats/Q/M=1/RMD=[][][3,3] wDisplayMaskRGBA
		if (v_min==0 && v_Max)
			wMask = !wDisplayMaskRGBA[p][q][3]
		else // otherwise, make sure mask is cleared
			FastOp wMask = 0
		endif
		SetOverlayTransparency(1)
		ResetMapTextbox(prefs)
		AveragePixelsROI(wMask)
		// clear profile plots
		ShowProfileMap(prefs)
		ShowProfile(prefs)
	else
		prefs.mode = 0
		ResetMapTextbox(prefs)
		SetOverlayTransparency(0)
		ShowCursors(prefs, 1)
		ReplotPixelTraces(prefs) // restore spectra plot
		// restore profile plots
		ShowProfileMap(prefs)
		ShowProfile(prefs)
	endif
	StructPut prefs, wStruct
end

static function ResetMapTextbox(STRUCT PackagePrefs &prefs)
	
	string strTitle = ""
	
	if (prefs.mode == 6)		
		#ifdef WINDOWS
		sprintf strTitle, "\\K(%d,%d,%d)Ctrl to select, Alt to deselect", prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue
		#else
		sprintf strTitle, "\\K(%d,%d,%d)Cmd to select, Option to deselect", prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue
		#endif
		TextBox/W=HyperspecBrowser#map/C/N=textPixels/F=0/A=LT/B=1 strTitle
	else
		TextBox/W=HyperspecBrowser#map/K/N=textPixels
	endif
	
	if (prefs.mode == 5 || prefs.mode == 8)
		sprintf strTitle, "\\K(%d,%d,%d)Click %sto add a point", prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue, Selectstring(prefs.mode==5, "path ", "")
		strTitle += "\rDrag points to move"
		#ifdef WINDOWS
		strTitle += "\rCtrl-click to delete points"
		#else
		strTitle += "\rCmd-click to delete points"
		#endif
		if (prefs.mode == 5)
			strTitle += "\rHold shift to move origin"
		endif
		TextBox/W=HyperspecBrowser#map/C/N=textROI/F=0/A=LT/B=1 strTitle
	else
		TextBox/W=HyperspecBrowser#map/K/N=textROI
	endif
	
end


// *** draw shape ***

// create a polygon in drawing layer from a pair of x-y coordinate waves
static function Wave2Poly(wave/Z wx, wave/Z wy, string win, string hAxis, string vAxis, [string layer, string name])
	
	layer = SelectString(ParamIsDefault(layer), layer, "ProgFront")
	name  = SelectString(ParamIsDefault(name), name, "")
	
	if (DimSize(wy, 0)<2)
		return 0
	endif
	
	name = SelectString(strlen(name)==0, name, NameOfWave(wy) + "Poly")
	
	wave/SDFR=GetDFR() wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	DrawAction/L=$layer/W=$win getgroup=$name, delete
	SetDrawLayer/W=$win $layer
	SetDrawEnv/W=$win gstart, gname=$name
	SetDrawEnv/W=$win xcoord=$hAxis, ycoord=$vAxis
	SetDrawEnv/W=$win fillpat= 0, linefgc= (prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue)	 
	SetDrawEnv/W=$win arrow=(wx[0]!=wx[numpnts(wx)-1] || wy[0]!=wy[numpnts(wy)-1])
	
	DrawPoly/W=$win wx[0], wy[0], 1, 1, wx, wy
	SetDrawEnv/W=$win gstop, gname=$name

	WaveStats/Q/M=1 wy
	return ((V_numNaNs + V_numInfs) == 0)
end


static function DrawPath(int start, int linear)
	
	DFREF dfrPackage = GetDFR()
	wave/SDFR=dfrPackage wStruct
	STRUCT PackagePrefs prefs
	StructGet prefs, wStruct
	
	NewDataFolder/O dfrPackage:DrawPath
	DFREF dfr = dfrPackage:DrawPath
		
	if (start)
		PixelEdit(0) // this sets mode to 0
		
		prefs.mode = 5
		StructPut prefs, wStruct
		
		Make/O/D/N=(0,2) dfr:ROI_nodes/wave=ROI_nodes
		Make/O/D/N=(0) dfr:ROI_x/wave=ROI_x, dfr:ROI_y/wave=ROI_y
			
		RemoveFromGraph/Z/W=HyperspecBrowser#map ROI_y
		AppendToGraph/W=HyperspecBrowser#map ROI_y vs ROI_x
		ModifyGraph/Z mode(ROI_y)=0
		
		RemoveFromGraph/Z/W=HyperspecBrowser#map ROI_nodes // keep trace on top
		AppendToGraph/W=HyperspecBrowser#map ROI_nodes[][1] vs ROI_nodes[][0]
		
		ModifyGraph/W=HyperspecBrowser#map mode(ROI_nodes)=3, marker(ROI_nodes)=8, rgb(ROI_nodes)=(prefs.csrrgb.red, prefs.csrrgb.green, prefs.csrrgb.blue)
		ModifyGraph/W=HyperspecBrowser#map rgb(ROI_y)=(65535,0,0,0)
		
		// create global variables in DrawPath date folder
		variable/G dfr:status = 1
		variable/G dfr:pointNum = NaN
		variable/G dfr:mode = 0
		variable/G dfr:polysuccess = 1
		variable/G dfr:linear = linear
		
		SetOverlayTransparency(0)
		ShowCursors(prefs, 1)
		ShowProfileMap(prefs)
		ShowProfile(prefs)
		ResetMapTextbox(prefs)
		
		DoWindow/F HyperspecBrowser
		SetActiveSubwindow HyperspecBrowser#map
		
		wave/SDFR=dfrPackage wMask
		FastOp wMask = 0
		AveragePixelsROI(wMask) // a bit inefficient, but works to reset display
	else
		prefs.mode = 0
		StructPut prefs, wStruct

		RemoveFromGraph/Z/W=HyperspecBrowser#map ROI_nodes, ROI_y
		DrawAction/L=ProgFront/W=HyperspecBrowser#map getgroup=ROI, delete

		SetOverlayTransparency(0)
		ShowCursors(prefs, 1)
		ReplotPixelTraces(prefs)
		ShowProfileMap(prefs)
		ShowProfile(prefs)
		ResetMapTextbox(prefs)
	endif
	
end

// called from hyperspec hook function when mode is 5 or 8.
// active subwindow must be map graph!
static function DoPath(STRUCT WMWinHookStruct &s)
	
	DFREF dfrPackage = getDFR()
	DFREF dfr = getDFR():DrawPath

	// we will use cursor codes to define the mode
	variable Insert    = 18 // insert node
	variable Move      = 21  // move node
	variable Zap       = 19 // delete node
	variable CloseLoop = 33
	if (IgorVersion() < 9)
		CloseLoop = 12
	endif
	variable Idle      = 0
	variable MoveOrigin = 8
	
	// status definitions
	variable wavedraw  = 0
	variable polydraw  = 1
	variable editnodes = 2
	
	NVAR/SDFR=dfr status, mode, pointNum, polysuccess, linear
	
	wave/Z nodes = dfr:ROI_nodes
	wave/Z wx    = dfr:ROI_x
	wave/Z wy    = dfr:ROI_y
	
	if (!(WaveExists(nodes) && WaveExists(wy) && WaveExists(wx)))
		DrawPath(0, linear)
		return 0
	endif
			
	switch(s.eventCode)
		case 4: // mousemoved
			if (s.eventmod & 8) // command key
				mode = Zap // delete node
				s.cursorCode = Zap
			elseif (mode == Move) // move node
				s.cursorCode = Move
			
			elseif (mode == MoveOrigin)
				s.cursorCode = Move
			
			
			else // zap mode will be decided based on cursor position and status
				mode = Idle
				pointNum = NaN
			endif
			
			if (mode==Move || mode==MoveOrigin) // move node and update spine
				nodes[pointNum][] = {{AxisValFromPixel(s.winName, "bottom", s.mouseloc.h)},{AxisValFromPixel(s.winName, "left", s.mouseloc.v)}}
				
				if (linear)
					Redimension/N=(DimSize(nodes, 0)) wx, wy
					wx = nodes[p][0]
					wy = nodes[p][1]
					if (status==editnodes)
						wx[numpnts(wx)] = {wx[0]}
						wy[numpnts(wy)] = {wy[0]}
					endif
				else
					CatmullRomSpline(nodes, wx, wy, closed=(status==editnodes), aspect=AspectRatio(s.winName, "bottom", "left"))
				endif
				
				// redraw poly if we move the anchor point
				if (pointnum==0)
					Wave2Poly(wx, wy, s.winname, "bottom", "left", name="ROI")
				endif
			else
				pointNum = NodeUnderCursor(s, nodes)
				
				if (mode != Zap)
					if (numtype(pointNum) == 0)
						
						if (status == editnodes || pointnum>0 || DimSize(nodes, 0)<=2)
							s.cursorCode = Move // cursor indicates that node can be moved
							mode = Idle // don't enter move mode until mousedown
						elseif (pointnum==0 && s.eventmod&2) // shift key 
							s.cursorCode = Move
						elseif (pointnum==0 && DimSize(nodes, 0)>2) // ensure that we enclose a finite volume
							s.cursorCode = CloseLoop // cursor indicates that loop can be closed
							mode = CloseLoop
						endif
					else
						pointNum = SegmentUnderCursor(s, wx, wy, linear)
						if (numtype(pointNum) == 0) // cursor over segment
							mode = Insert
						elseif (status != editnodes && MouseInPlotFrame(s, "bottom", "left"))
							pointNum = DimSize(nodes, 0)
							mode = Insert
						endif

						if (mode == Insert) // in position to insert a point
							mode = Insert
							s.cursorCode = Insert
						endif
					endif
				endif
			endif
			break
		case 3: // mousedown
			
			if (s.eventmod & 16) // right click
				return 0
			endif
			
			if (numtype(pointnum)==0 && mode==Idle) // grab node beneath cursor for moving
				mode = Move
			elseif (numtype(pointnum)==0 && mode==CloseLoop && (s.eventmod&2)) // shift key to move origin
				mode = MoveOrigin
			endif
			
			if (mode == Insert) // insert a node
				InsertPoints/M=0 pointnum, 1, nodes
				nodes[pointnum][0] = AxisValFromPixel(s.winName, "bottom", s.mouseloc.h)
				nodes[pointnum][1] = AxisValFromPixel(s.winName, "left", s.mouseloc.v)
				// allow the inserted node to be dragged while the mouse button remains down
				mode = Move
				s.cursorCode = Move
			endif
			
			// stop adjusting points after clicking elsewhere in window
			if ( (mode!=Move && mode!=Zap) && status == editnodes)
				DrawPath(0, 0)
				return 0
			endif
						
			break
		case 5: // mouseup
			if (mode == Zap && numtype(pointNum) == 0) // delete a node
				DeletePoints/M=0 pointNum, 1, nodes
				
				if (pointnum==0) // && strlen(layer))
					polysuccess = 0 // trigger an update of the drawing object
				endif
				
				pointNum = NaN
			elseif (mode == CloseLoop) // status changes from 'draw' to 'edit' after closing the loop
				status = editnodes
				s.cursorCode = Move
				polysuccess = 0 // force a redraw of poly without arrow
				
				wave/SDFR=GetDFR() wStruct
				STRUCT PackagePrefs prefs
				StructGet prefs, wStruct
				prefs.mode = 8 // change hyperspec mode from draw path to edit ROI polygon
				StructPut prefs, wStruct
				ShowProfileMap(prefs)
				ShowProfile(prefs)
				ResetMapTextbox(prefs)

			else
				// stop moving nodes after mouseup
				mode = Idle
			endif
						
			if (linear)
				Redimension/N=(DimSize(nodes, 0)) wx, wy
				wx = nodes[p][0]
				wy = nodes[p][1]
				if (status==editnodes)
					wx[numpnts(wx)] = {wx[0]}
					wy[numpnts(wy)] = {wy[0]}
				endif
			else
				CatmullRomSpline(nodes, wx, wy, closed=(status==editnodes), aspect=AspectRatio(s.winName, "bottom", "left"))
			endif
			
			if (polysuccess == 0 || (status==polydraw && DimSize(nodes, 0)>1))
				// draw the poly (the drawing object becomes dependent upon wx & wy waves) and revert to wavedraw status on success
				polysuccess = Wave2Poly(wx, wy, s.winname, "bottom", "left", name="ROI")
				if (polysuccess && (status==polydraw))
					status = wavedraw
				endif
			endif
			
			if (status == editnodes)
				AveragePixelsROI($"")
			else
				wave/SDFR=GetDFR() wStruct
				StructGet prefs, wStruct
				PlotPathPixelTraces(prefs, wx, wy)
				ResetPathProfiles(prefs, wx, wy)
				ResetPathProfileMap(prefs, wx, wy)
			endif
			
			break
		case 6: //resize
			if (!linear) // even though we're using axis coordinates, if the window aspect ratio changes, the path shape will change
				CatmullRomSpline(nodes, wx, wy, closed=(status==editnodes), aspect=AspectRatio(s.winName, "bottom", "left"))
				if (status == editnodes)
					AveragePixelsROI($"")
				endif
			endif
			break
	endswitch
	
	if (s.cursorCode)
		s.doSetCursor = 1
	endif
	
	return 0
end

// corrects for aspect ratio of graph area and relative scale of axes.
static function AspectRatio(string strGraph, string hAxis, string vAxis)
	GetAxis/W=$strGraph/Q $hAxis
	variable horizontalAx = V_max - V_min
	
	string strInfo = AxisInfo(strGraph, hAxis)
	if( NumberByKey("log(x)",strInfo,"="))
		horizontalAx = log(V_max) - log(V_min)
	endif
	
	GetAxis/W=$strGraph/Q $vAxis
	variable verticalAx = V_max - V_min
		
	strInfo = AxisInfo(strGraph, vAxis)
	if( NumberByKey("log(x)",strInfo,"="))
		verticalAx = log(V_max) - log(V_min)
	endif
		
	GetWindow $strGraph psize
	return abs(horizontalAx/verticalAx) / abs((V_left - V_right)/(V_top - V_bottom))
end

static function/S parent(string str)
	return ParseFilePath(0, str, "#", 0, 0)
end

static function MouseInPlotFrame(STRUCT WMWinHookStruct &s, string hAxis, string vAxis)
	variable xx = AxisValFromPixel(s.winName, hAxis, s.mouseloc.h)
	variable yy = AxisValFromPixel(s.winName, vAxis, s.mouseloc.v)
	GetAxis/W=$s.winName/Q $hAxis
	variable xmax = max(v_min,v_max)
	variable xmin = min(v_min,v_max)
	GetAxis/W=$s.winName/Q $vAxis
	variable ymax = max(v_min,v_max)
	variable ymin = min(v_min,v_max)
	return (xx == limit(xx, xmin, xmax) && yy == limit(yy, ymin, ymax))
end

static function NodeUnderCursor(STRUCT WMWinHookStruct &s, wave nodes)
	string info = TraceFromPixel(s.mouseloc.h, s.mouseloc.v, "WINDOW:"+s.winName+";DELTAX:4;DELTAY:4;")
	string tname = StringByKey("TRACE", info)
	wave target = TraceNameToWaveRef(s.winName, tname)
	if (WaveRefsEqual(nodes, target))
		return NumberByKey("HITPOINT", info)
	endif
	return NaN
end

// figure out where we want to insert a point in wx, wy
static function SegmentUnderCursor(STRUCT WMWinHookStruct &s, wave wx, wave wy, variable linear)
	string info = TraceFromPixel(s.mouseloc.h, s.mouseloc.v, "WINDOW:"+s.winName+";DELTAX:4;DELTAY:4;")
	string tname = StringByKey("TRACE", info)
	wave target = TraceNameToWaveRef(s.winName, tname)
		
	if (WaveRefsEqual(wy, target))
		if (linear)
			variable hitpoint = NumberByKey("HITPOINT", info)
			
			if (numtype(hitpoint))
				return NaN
			elseif (hitpoint == 0) // must be segment 1
				return 1
			elseif (hitpoint < (numpnts(wx) - 1))
				if (DistanceToSegment(hitpoint+1, wx, wy, s.mouseloc) < DistanceToSegment(hitpoint, wx, wy, s.mouseloc))
					hitpoint += 1
				endif
			endif
			return hitpoint
		else
			// this works provided we select a point at least 1/40th of the distance from start of segment....
			return ceil(NumberByKey("HITPOINT", info) / 20)
		endif
	endif
	return NaN
end

// calculate distance from mouse cursor to a finite line
static function DistanceToSegment(variable segment, wave wx, wave wy, STRUCT point &mouse)	
	variable segmentlength, tostart, toend
	variable mouseX = AxisValFromPixel("HyperspecBrowser#map", "bottom", mouse.h)
	variable mouseY = AxisValFromPixel("HyperspecBrowser#map", "left", mouse.v)
	segmentlength = sqrt((wx[segment]-wx[segment-1])^2 + (wy[segment]-wy[segment-1])^2)
	tostart = sqrt((mouseX-wx[segment-1])^2 + (mouseY-wy[segment-1])^2)
	toend = sqrt((mouseX-wx[segment])^2 + (mouseY-wy[segment])^2)
	
	variable cosalpha, cosbeta // cosalpha is cosine of angle line from startpoint to mouse and segment
	cosbeta = (segmentlength^2 + toend^2 - tostart^2) / (2 * segmentlength * toend) // cosine rule
	if (cosbeta <= 0)
		return toend
	endif
	cosalpha = (segmentlength^2 + tostart^2 - toend^2) / (2 * segmentlength * tostart) // cosine rule
	if (cosalpha <= 0)
		return tostart
	endif
	return 1/(2*segmentlength)*sqrt((segmentlength+tostart+toend)*(tostart+toend-segmentlength)*(toend+segmentlength-tostart)*(segmentlength+tostart-toend))
end

// Create a spline that passes through a series of x-y points,
// optionally looping back to the start point.
// Can be used with ConvexHull to make a loop around a cloud of points.
static function CatmullRomSpline(nodes, wx, wy, [segPoints, alpha, overwrite, closed, name, aspect, logx, logy])
	wave nodes
	wave/Z/D wx, wy
	variable segPoints, alpha
	int overwrite, closed
	string name
	variable aspect, logx, logy
	
	segPoints = ParamIsDefault(segPoints) ? 20 : segPoints
	alpha     = ParamIsDefault(alpha) ? 0.5 : alpha
	overwrite = ParamIsDefault(overwrite) ? 0 : overwrite
	closed    = ParamIsDefault(closed) ? 0 : closed
	name      = SelectString(ParamIsDefault(name), name, NameOfWave(nodes)+"_CR")
	aspect    = ParamIsDefault(aspect) ? 1 : aspect
	logx      = ParamIsDefault(logx) ? 0 : logx
	logy      = ParamIsDefault(logy) ? 0 : logy
		
	variable np = DimSize(nodes, 0)
	
	if (!(WaveExists(wy) && WaveExists(wx)))
		wave/Z/D wx = $name+"_x"
		wave/Z/D wy = $name+"_y"
		if (WaveExists(wx) || WaveExists(wy))
			if (!overwrite)
				DoAlert 1, "overwrite existing wave?"
				if (V_Flag==2)
					return 0
				endif
			endif
			Redimension/N=(0)/D wx, wy
		else
			Make/D/O/N=(0) $name+"_x" /WAVE=wx
			Make/D/O/N=(0) $name+"_y" /WAVE=wy
		endif
	else
		Redimension/N=(0)/D wx, wy
	endif
		
	if (np < 2)
		return 0
	endif
		
	Make/free/N=(np+2,2)/D temp
	temp[1,np][] = nodes[p-1][q]
	
	if (logx == 1)
		temp[1,np][0] = log(temp[p][0])
	elseif (logx == 2) // this is rather pointless. doesn't matter which log base you use.
		temp[1,np][0] = log(temp[p][0])/log(2)
	endif
	if (logy == 1)
		temp[1,np][1] = log(temp[p][1])
	elseif (logy == 2)
		temp[1,np][1] = log(temp[p][1])/log(2)
	endif
	if (aspect != 1)
		temp[1,np][1] *= aspect
	endif
		
	if (closed)
		// check whether nodes form a closed or open loop
		if (nodes[0][0]!=nodes[np-1][0] || nodes[0][1]!=nodes[np-1][1])
			np += 1
			Redimension/N=(np+2,2) temp
			temp[np][] = temp[1][q]
		endif
		
		// we have made sure the nodes wrap around in points 1 ... np,
		// figure out values for points 0 and np+1.
		temp[0][] = temp[np-1][q]
		temp[np+1][] = temp[2][q]
	else // extrapolate beyond first and last node
		temp[0][] = temp[1][q] - (temp[2][q] - temp[1][q])
		temp[np+1][] = temp[np][q] + (temp[np][q] - temp[np-1][q])
	endif
		
	Make/free/N=(segPoints,2) xy_out
	Make/free/N=(4,2) xy_in
	
	variable i
	for (i=1;i<np;i+=1)
		xy_in[][] = temp[i-1+p][q]
		CatmullRomSegment(xy_in, xy_out, alpha)
		SplitWave/free/OREF=refs xy_out
		Concatenate/NP {refs[0]}, wx
		Concatenate/NP {refs[1]}, wy
	endfor
	
	// make sure first and last points are really equal when path is closed
	wx[0] = temp[1][0]
	wy[0] = temp[1][1]
	
	wx[numpnts(wx)]={temp[np][0]}
	wy[numpnts(wy)]={temp[np][1]}
	
	if (aspect != 1)
		wy /= aspect
	endif
	if (logx == 1)
		wx = 10^wx
	elseif (logx == 2)
		wx = 2^wx
	endif
	if (logy == 1)
		wy = 10^wy
	elseif (logy == 2)
		wy = 2^wy
	endif
end

static function CatmullRomSegment(wave xy_in, wave xy_out, variable alpha)
	// xy_in has x and y values of four control points
	// alpha=0 for standard (uniform) Catmull-Rom spline
	// alpha=0.5 for centripetal Catmull-Rom spline
	// alpha=1 for chordal Catmull-Rom spline
	
	int nPoints = DimSize(xy_out, 0)
	Make/D/free/N=4 T4 = 0
	T4[1,3] = sqrt((xy_in[p][0]-xy_in[p-1][0])^2 + (xy_in[p][1]-xy_in[p-1][1])^2)^alpha + T4[p-1]
	Make/D/free/N=(nPoints, 3, 2) wA
	Make/D/free/N=(nPoints, 2, 2) wB
	Make/D/free/N=(nPoints) w_t
		
	w_t = T4[1] + p/(nPoints)*(T4[2]-T4[1])
	wA = (T4[q+1]-w_t[p]) / (T4[q+1]-T4[q]) * xy_in[q][r] + (w_t[p]-T4[q]) / (T4[q+1]-T4[q]) * xy_in[q+1][r]
	wB = (T4[q+2]-w_t[p]) / (T4[q+2]-T4[q]) * wA[p][q][r] + (w_t[p]-T4[q]) / (T4[q+2]-T4[q]) * wA[p][q+1][r]
	xy_out = (T4[2]-w_t[p])/(T4[2]-T4[1])*wB[p][0][q] + (w_t[p]-T4[1])/(T4[2]-T4[1])*wB[p][1][q]
end


// uses integer maths
static function/wave GetBresenhamPath(wave wPath)
	Make/I/free/N=(1,2) wOut
	wOut = wPath[0][q]
	Make/I/free/N=(2,2) wSegment
	int i
	int startpoint
	for(i=1;i<DimSize(wPath, 0); i++)
		wSegment = wPath[i-1+p][q]
		
		if (abs(wSegment[1][1] - wSegment[0][1]) < abs(wSegment[1][0] - wSegment[0][0]))
			wave w = GetLineLow(wSegment)
		else
			wave w = GetLineHigh(wSegment)
		endif
		DeletePoints/M=0 0, 1, w
		startpoint = DimSize(wOut, 0)
		InsertPoints/M=0 startpoint, DimSize(w, 0), wOut
		wOut[startpoint,][] = w[p-startpoint][q]
	endfor
	return wOut // returns 2D pixel coordinate wave
end

static function/wave GetLineLow(wave wSegment)
	int p0, q0, p1, q1
	if (wSegment[0][0] > wSegment[1][0])
		p0 = wSegment[1][0]
		q0 = wSegment[1][1]
		p1 = wSegment[0][0]
		q1 = wSegment[0][1]
	else
		p0 = wSegment[0][0]
		q0 = wSegment[0][1]
		p1 = wSegment[1][0]
		q1 = wSegment[1][1]
	endif
	
	Make/I/N=(0,2)/free wOut
		
	int dp = p1 - p0
	int dq = q1 - q0
	int qi = 1
	if (dq < 0)
		qi = -1
		dq = -dq
	endif
	int D = (2 * dq) - dp
	int qq = q0

	int pp
	variable i = 0
	for (pp=p0;pp<=p1;pp++)
		
		wOut[i][] = {{pp},{qq}}
		
		if (D > 0)
			qq = qq + qi
			D = D + (2 * (dq - dp))
		else
			D = D + 2*dq
		endif
		
		i += 1
	endfor
		
	if (wSegment[0][0] > wSegment[1][0])
		Reverse/DIM=0 wOut
	endif
	return wOut
end

static function/wave GetLineHigh(wave wSegment)
	int p0, q0, p1, q1
	if (wSegment[0][1] > wSegment[1][1])
		p0 = wSegment[1][0]
		q0 = wSegment[1][1]
		p1 = wSegment[0][0]
		q1 = wSegment[0][1]
	else
		p0 = wSegment[0][0]
		q0 = wSegment[0][1]
		p1 = wSegment[1][0]
		q1 = wSegment[1][1]
	endif
	
	Make/I/N=(0,2)/free wOut
	
	int dp = p1 - p0
	int dq = q1 - q0
	
	int p_i = 1
	if (dp < 0)
		p_i = -1
		dp = -dp
	endif
	int D = (2 * dp) - dq
	int pp = p0

	int qq
	variable i = 0
	for (qq=q0;qq<=q1;qq++)
		
		wOut[i][] = {{pp},{qq}}
		
		if (D > 0)
			pp = pp + p_i
			D = D + (2 * (dp - dq))
		else
			D = D + 2*dp
		endif
		i += 1
	endfor
	
	if (wSegment[0][1] > wSegment[1][1])
		Reverse/DIM=0 wOut
	endif
	return wOut
end

static function GetThisVersion()

	variable procVersion
	// try the quick way
	#if exists("ProcedureVersion") == 3
	procVersion = ProcedureVersion("")
	if (procVersion)
		return round(100 * procVersion)
	endif
	#endif
	
	int maxLines = 30 // number of lines to search for version pragma
	int refNum, i
	string strHeader = ""
	string strLine = ""
	
	Open/R/Z refnum as FunctionPath("")
	if (refnum == 0)
		return 0
	endif
	for (i=0;i<maxLines;i+=1)
		FReadLine refNum, strLine
		strHeader += strLine
	endfor
	Close refnum
	wave/T ProcText = ListToTextWave(strHeader, "\r")
	string S_Value = "" // workaround for Grep bug for Igor 8
	Grep/Q/E="(?i)^#pragma[\s]*version[\s]*="/LIST/Z ProcText
	if (v_flag != 0)
		return 0
	endif
	s_value = LowerStr(TrimString(s_value, 1))
	sscanf s_value, "#pragma version = %f", procVersion
	ProcVersion = (V_flag!=1 || ProcVersion<=0) ? 0 : ProcVersion

	return round(100 * ProcVersion) // 100 * 1.12 is not equal to 112!
end


// PNG: width= 90, height= 30
static Picture pInfo
	ASCII85Begin
	M,6r;%14!\!!!!.8Ou6I!!!"&!!!!?#R18/!3BT8GQ7^D&TgHDFAm*iFE_/6AH5;7DfQssEc39jTBQ
	=U#t@KS5u_NKc'hjQ<7_k>^N6#'iu's#@iL6?EN#,SR%ORqWY?g_DQ6ZakX-9m6UF`Y7B5's^*$E%B
	u3<(a:i;5<Ct\*^a)hT'K%0QC]d[0jPV;N@+9cG@fjB`p38^oR<*NngmMP4QXaLkr6j/Tp8k\$ZiA^
	F[J/@;&;!fa/0Y"8EKlO!*92tV8I!509L_n[prdb[0DD]b^77\4Qh+<X22WSL=^Tpl!>%R?bOV1l`F
	.H/kj.Q0YHO!p0Z_o!(t!"2!0>1<o&\&oX/k93#9I:/\ol*"]HFk2"&^kI,Op^Po$LJ:qE[=Zdn*%1
	/%c)$r:,4%r)#UgVCL9-4fb:)2uc00,SCF2g"FKhP:)ciSp&'<om1=hD5F)*"ddDT8%a$Z"blrKB1V
	t"bqBnEcNrlWWRU];Q?b;\6c%+`!0V&SGg^%NRV$e":,J(6+Q,'.qs8WXZ?]HQd?\@;-q_mAJ,]9,i
	ondb3ItHg(Zjc?GW[$h>t<@#!7L*gWDnag1eK/1,*>Cr1P3*D[BX`3`d.+2D9NUG(..rXX^7$1"@82
	!1KE^Ub4(b^NWMsF;Ht6&&;b$b#dun1L:o\#UFM.3gNK`n[l7Cp+c[X/cDaa?O<b.!b=-Pr2.^?1Nd
	)e2#U:Q=,SCE:&8^%*L$dX(UQj\B/Bea\(..rm00Qspp'-Bu;ci@[g2Ab(7h`\ZA=S45.(4!_jMmH)
	qUNK^MN[#3[D>08n\uMdd,!N;U*ru@<Dd(/5&X;O&;3@n7RotRD4?;%HAGk=G+fW.>tiHTqLp0W7=4
	1t\u)fWAJng`QIdQ"8TJr%$paau+6@''?G#H(RO^<AUs^^G[]\cSB@c5aI;=JigY9^3e#1T9SXPucg
	hq>mG`bOMiSe2FC2.Jjp@_T>`piIOWDnc;k%;&=gQn1M0p,[l:OIIe[!*"QkrNrEAXG*0N(r07]$fZ
	h/$5=fV3Tb,=$\psV/^EJ)4IFB]PEW'\qbg(SsMlA:JXaHWi@OR/f1s5k:MsaPoMc&J--#d&-rCGg1
	@0&:")(CjpXC94`bia)`#WJ(G?agO/a)!%!Za'1b>,JjlN'!=]&V5U;CI@Et9%o2-h+N"!1EAB5`,M
	Th&'lHW;8P&FJlPBUVCZ77C(9D;1qiXUW&&O5KD=a,V0/D%4B*&."qC)Xk?1%Zs:M/7mre"!%<ET2<
	bOD<cB%HGLYk/]p$:.Z,bn[7sj`?B9ZXXV6Rq]_Pti2gd\J3f(i:R&5_pm4KmDXm$aS/B^J!/^)#uG
	%6sfYult1cH[_1nRTfUip]H>d<$0>4r9NH8sWQ9)djkQAg3E#:]C\]o'uTN1rJBf7mE:cKPfJ<US@o
	sfC=1##U)g+pYUIm80gMI'&gcc'97IQ?8%%Ad+-[#D;.]AGGtPU1e&m.iUpnuW^=]Xm<&%d+jRgmR6
	q<!"K4)c%%V>s6.boU<k:W9@PN6o+0)@'G&kQ@q=8),*BN8CpR@_H@2O[8@=!9Ifs"n;A,h79HFe;'
	<AQ<<*SW3\4hYn'7c+!LCK<HgOtdE6\7T^B_C4`os.\g?L1-<OUon"8QFD=fRPcBo,tYs-%!5e<il-
	gK-hM%Q-;cW@=dI0o(g+V`I[7Hll^gmF1C8U`9;n:SDf?'4kXX)YA6CN6o]r]qT!s1+^3pK)B6pt_B
	a+O@`IRg8=:@XkJL"[Wg"k-TK%RqsK*5:)MTX360]U/4Wi@77;Jt3uP7<CJ.#Pn:q"VM]0-o&(8.o9
	`MEo2(5'MS@J+Rq@q<5kKI!A\?`c`\j5Xs.C`^\u^];Gr<!]0NL(T`^;0>7#+XNrGfbj/DFR`<:aKO
	an5L'f'H$C6<N=ieH_Ns:8r/-F>E>*(o2.ET1%BpgNq9:F_:'kIT8!)),C3GlcmYHU;j&0aK<:WMel
	3Ze-Om%!i`\)#_:DlKDdZeWg@P_C[O5M'iUB8>_-2HE;O1>jM;$8=V+SNHPXHb;/UA2:]pC+19D\Bd
	qtPsUXEa5qKga#B2SZ=.SRp%A!RdqL(!Ea7bt3ql@F;Uj=M08??L+[=OW75>HDQ5.GO:69br-505tC
	KaJC@jCMg**JRJQascSIH0iWNK&pUTkRi35A[F<CY2`i#$#_f\:6RcLI9_:V"&,4[]HN!OfQlP*Qtb
	VGkuA?F_!6F<)ocgU#D#V(':e98CFiHpC[p2PjB,D#;1";i6n,<g%nVGOf<>lBjHXs8BA(fD;DdgE&
	_+N0,P*1$jPnaB'W+c:JFR2daE`AbT2\$"\Atc30k=3]2,.R>E*Nf"c^j@=F?-EhKuZgDlI9;bg[c+
	#@$`/2$kB@MhsTDp>G]B0oZ#`%fgeAhHcB?i5($LInM,P9MS?gJ,auQDYK=UG;=]pX[t0`Ddn+h)@5
	*f*'#BK+[7+e+25ad[WOTm!ShS%o^pU.1GgqQgF=j%Y7Q;ZTcp"RlDZ'E)=kG_qYSr#Wb]AY@*-(nJ
	s#A#q"YFMfsgRh]Kt<,gI6=12lg#`?;TLp_*KtthTY]\gR%W:!I8"sroVaPFBo;RFjF5*7K$9'/M"a
	&=8b8<:r],W]a;6"MF%U=g2!m3l-^kmCMU]X>\[CWf<8QErql0Iq<+A?>3\a$'Y&&M_Vt+amGf@imo
	bQW3d]\44-GFK,M+U?MapjQn$rZ2MO]A366&(1N))Y5K\G-`9XF4l;%/:LK#,8h*dQfYG3mlOSIcZr
	>bLn*pC6ZP>8Kg\bG_@CKr2I%"\bg,9T7gnab8=Dm!=-WrDiH(kdbct`7o4FNN0!X<E0c>,]?+,R:L
	D]PZ0,$>?`&92Jr:c#b4?2.q.C==rdOGT-GEe&/PNlY926R6do>OS3)p0O\R3Y8&I`m-S_GCD(TPUp
	#Oh0$f?=CMoA(S7gE=*>mMVCq0YVL.YYcK,Y=`@\X=h:G[ec2^3h=YF;C/g`2VT*c?qXpKj]]U?cVq
	fC?DNujYDHgQ#<&Y/Z3l"'p#PjIED$6^p\*"iPGcpqYD$G(gq`=S8``85)IVK,7k,W='%mZ6N^%W9s
	ZH4Ib4'RZ77*WgpE7az8OZBBY!QNJ
	ASCII85End
end

// PNG: width= 90, height= 30
static Picture pCog
	ASCII85Begin
	M,6r;%14!\!!!!.8Ou6I!!!"&!!!!?#R18/!3BT8GQ7^D&TgHDFAm*iFE_/6AH5;7DfQssEc39jTBQ
	=U%adj95u_NKeXCo&'5*oW5;QerS2ddE(FP;'6Ql#oBiuS1dW/'TBu9uWoJd,<?@,Ku[OT_*C6>N&V
	5fgKO-Z!0m]BHZFE$kSY/b+).^UIV80KE-KEW"FQ9_,eF^bT7+UPEE3;1R;IlL+VJZGMtc/)57aYSB
	X^5kLMq7Sp9q0YBG@U214ph]9b)&eA4@(!#Ip%g0/F0_UH>\E'7M.=ui]mI6`m'G"T)B'OA[IVmQDm
	=?Eat`-Qd[5bR1c#F]`cT(Se*XgJen[DI=<Y8YF7N!d-X+iY2pq;SVDt;siWS_bs$!E]WT^pRhs]aH
	M%csq\akmJ4tMo<HGU1WqT,5cib%XrHM`>*]/&U:](RQ7QDj\u&#cMNi8>`?8;-?rCcXX>+1^gW10F
	n!h-2G-M;R\a+TD2op'?`R]"%W!DpJmO],i(@-(/F?,;L7Vkt%YOa,f\8+CV@lHVOEMgMZPnh$6>!V
	MTYBm^8f[O,?Z$2MnH6.T'JWSM4HtSissRo-):4d:ecoe5Tn^(gUEQm+o94@;L(/[91%aXk:!pP;mm
	\kh$s.7qbe%=-p1eBtDs*CHp:+$CUY\0A,jM0:4J2&pY-HWBG?nb`"BE/M-#*)+E?I*C/(r;J]APNh
	3!Ea<(u)`o?0R`ma=QV<n?GV/s3:I0Wf2_M0p@:__T%OEl+sL@10K8&ViQgR(0Q3qMLYA':/iba:,;
	]Y$@ACMV&9b[fD4A`Vq5+A!37VD0na`;0#fWNWKq#f5N>Mt)$S['[2:?=(p2$Q$$NX_cXoJ`iVOcHm
	Rb+"_b#*b4@tp)Xq9r*1_<^IVlpMJ=kE>MhiHa2]]q9<d*4(lA_8$4ej2NM5Z!#`oc=+Ttk-]%D5"O
	Yiu,o$V/I<=@2fN3Ds,PNfIEnqn6C?^[OYDs4q2k*s6TFu+@1>SKUmdko@B5>Pp)-]8`l_Ig,/1c.T
	K'Z+asa)qDc*mqZAKmijlOd;;&H$MEMWY1:\q<G#aaNVlho?TWCGL35!G658MH$RpQ,/[:S#==eP-@
	T+s%'h-7&.0)\eW6j@1gNW'FYlgRIid1g1dP0.MtL)"o@*4A+B&XU\9bRSJWg?B!keI%b6T7FS'?W(
	@7j-a-n[,Adkh%U((6"oN9G"iRV$rmb0"2lqXk08!N&V_b13Oo*tc)h=[L]E@W<ihr:]%Dbs-*cCbc
	T^`<b81D1(d_gue7JX)rMl-Q!ag!0.a4mbCL5'MQH2X<p3`1nA#69QNiWDnN[J^:kIm`JPCXo#W8lo
	?KFN_dOMp#7s\i!;q:1Vb`q^Za5j'0Sg8ALVn\tm:OM*.G/IF;=K4S+O/0U]^`u\O,i'69Y*0'f,SH
	:Kp)mH3.EQ3hDf%?f;W[LbJbR7R5Lb$8U4I7,[8ZM7fU7H5(>6BlSlr1cG6"263q!"YH.\`>aLYN(!
	e3hZYm666$Mq_c(_GHO?%CE(rnI-UV=I6M\e$%CXt$`9q"IB8d`/4e)0&Dcf`43oobf6MqdV?d>\73
	X/dI(2jZ+#[Nt'sQg5KRFl_^rEVX>cZ5h3g1gR#*IhTZf+KqmsVrW[`UcE6>MM*N0fMcT[S!:h'=ju
	EaL@5>OfU]WXd+d/JISLYCQZ_BPkB$IiAYUBq1l^ecC4a8EYJ'WJ,pak5V59k6$F23m\(d</D&W$.c
	&:n64N(\`j0%h;m7iB=j)(R^kh8BUCfn<.JP[26K1F7\>0JOc56jWCOX(6k=i$m^+A%<G5QZ.i$e8]
	01cU/k$R?&@c[1P\L>tK[OkSMmQ7lT_puI&4&#-'R99q+p;3\Sn`Ic2G%kDj/"N6oB<E0?Z6Kl"@,Y
	?4P5G,Nu\q^LTs(Vj(g3YCoZEl;W9!T)>ePG,S!10n_Y\rP(FBr90.J3;jVP^MYp<-ML6>(=)*bFsB
	&uWX-VWg'<P8G@$+\A?J1@G'F,/Z.<q/;.,=8I?;gFp>>;IjEQPE_:7`.R+3bElA@DB6<k@ksJ9lg(
	=CVM=g<G(^E#SiiFHZ8.qF-^ppkE&\[U*_);<'Lfk*Fq]^#W339=Q'IWpj"KgUWU]1,ETW<&G^OtOH
	MkGSX1r0@A!=Gd&>m2;$s(nF8b!K?8R`eZjO8V38_I$<1AcnP!)B*`T#3.X=hE[s8PMoFf*/H2@32u
	N'Len`GUk?n:K*@=9gMMi1ES9gF7flQcCD`2n^,h:`S5=GNQ^G#@^/a:?]W`PV50n4Xue>QVk8E1=]
	lWKB?uV(SiXjL_hVC,G-+W-bHd)[C`^u(BPM:VV58ltJcZ8d$CEhp-Ek)Qb.'^'f)4eT`-]7*:O[1>
	WO?>HRRZ3%&Aum4gNS.jU=.^S*B#H\'0H3Z)tG>O;)WW1_IO09[>@PI4Y3R7J0]^!UgQhj+u1-,O_#
	4q7aj2?Y5.V]ph=D*J`g8?n%JH:8P)K%MLrfVTs(X1]A:d+mFtdNBG"";'8siHNZC4&bKJq`%mNb7r
	SW;=`2-+n=L)HDOsFHoS$CX_6m<3W758iRSt7"9?7u`s%5^"&NsfV]&fKiR),nod"F>!-ZZj257RH'
	BCpkU2>pgE:BYW?=m,Fa:(R[)N0g+8U_d1?h6?87.J?1.S8QM+N.?c/5H^[K9Qq/KSe*09PFi*)kOs
	Cpa8151hB![K\`b9:/2t#`0h)TQGGW^_mOCaj@jCA@uU*q95,uIW@7!X&<O\"P/_=YGg"!\PP(fcMs
	XX8]B01I/5GVm/oHal.qLkA^e?r`*cq5<6cMYpEN]i/%/Vl(p'^fIMd8C\oHLnT1!)7oQm]mKeID*1
	Nn<=:2#ZkEe#YpHHH,[1j)T%7I4;@/)4cu_Q1)PbZM:[@i"UEI%;r>p03XoVZ_RkU=+$'7#=USG/bL
	8-+m<=>h,iqN=A8kNQ3E0-ce+TlUO7L$\:&5CW07\^Y5(=Lpj3bn5fXf]+hD?H]7Wq?&[-c"7hNK0#
	/)B'Mj<HT8ma3CurC,*bh\'aHR7Z[QbX-ZmAk800m):lpO:?P+(!;TY'R\j"AmjWGZ^'4n^bf?W3J@
	>&NBK1HqWkVhSJ@2:%U$9,hq:d,G1cCmI-TcrP\J*Ut[2@6?BcK3XN6]^DH?sm>]m;PWk0+t]M3*pb
	_i5ToaNr1&dko4ib1O7G-^#a3R58IWd+6c;6ULrU<E06&]A7B"H::^+p=jM"Cht@E-\k9W-F%RN7[f
	g9`s&hll-^kfa*7KZfMd!Yc("^(?tb@(G_h8BF>IEA!<RhTlND,!db&r!Y3m>2$M&6dHhJmnR;"(TN
	7k9deNFM:rt_%M:_\bI5MO2h<A0N#R:ZSrC"&ps\iY*%&:=-;@IrX+"G9!l_&sOI?=_'7)$hsk)[Og
	CfZ9V$@mNB]AS#G_>V6^Z_/)$iF?19\*_+U8'Lh!@O$@74\ogtP<K2K,l#,B;0e6NJ*#oS35C1Gqa?
	[/#EMYbdp\'g083t^Hm.OC']=?TCh[+\@'/CD^$lb9k>s0TnKIaqh4fI!&lDq*\4*U*,*??/2AnId;
	.P@%q^Y_gV7L#<Y@CP!Nm,EKn=&i8<r@!PTa5]H_PQ](fBqn;iMUE,Pl5E-<#!*W9G0Gifi6\]*<o4
	FnfrX,WF^_\F$1nF]^fU-bLO!=bm%5lG.k3$IWMqUu'c@l,R*B4I#7$5RGsBAo;S"q!W&r%8C2/"PK
	bsa<4<J73edT0;DW#W4)GM@uFDONL"9R+_N^([T!(k&'aJc*VLHV&^R;!(_#4Z$g7@#[rnEd4b][m6
	"s2Cf6FX#Yth+![-Bc9;DCc35!#ZOe]>R5l%A3s9r*"E3cZ^HAq!@!X3ZLR*.A7oQ8om.\p[kV)RYJ
	YS_VJ,ke(!91A):`eg`H3<A0s%C//=2RBq,pC2:S]*\P@U_Oa*3.T[g!Hf.uI$^Z574:Ig+a&Rh&b)
	g:i!IBPVCY]Y$?Mc-nM/==coe'#A=jP*M;$C2,4.LP+[KAA[:ZiG]VW6ipmf;5gRtUogbYmG#,M=Y+
	23\;kLm>:;.qMlKqn)uClJ![.i',6Vo?VRu"<5C39QHia.rIUXNcq/492/E<t4:q=5j]#0#BT^2C8R
	r=7NOaA&EGCimE'I"(o(b72sE7cRPmWBnj]tHh/;(=(Hs(iTH$*GN/REC6P4"-OQ)1Z*S9OlNXqYG,
	/qFo"%Xt5WcH)DMDo_@'gn2mp\l)\(b!Y0Pa!2n,Nj%-R@Yjn?WT$E#t(FUbj=.R08ON,:0qYL%:/M
	0]<Q1)2)G':0@s*h8ZZ<4MLPu4'A3d&SI6l9j+cCI%0ltDh?G(&13<0N\Q=?t??;p9Q8$59_n3Rli=
	5b`N"1`$#>;[JNr+$!*^fli%qGlHBaGd#r_gndbH0<a<pR0sE5L0:j)I_t)\EH/7Wqt8QJMd<r<&WK
	8J3cuoH9hij#22_bS-?/1q+bUC@(DjDc_1Dg*LCYK([C$_m"OB=44C54XF6CiRHM)#JSik-Qi#lgdX
	C:AAV;n0(-%L_0m!T,"d5MVG@HhTKZ87K.p%WH^Xh3kCpXcTY<@p]%p!H!PcGm8Ma`dWI-i:%O`.@A
	\EMhGlp>Rk7O<G2m`*r,h[u\8;4r,bUaQp%EDN*J]D4B1hFXuppq^tpModARV5%<QlNN?L%hAEkIlW
	/#`^]Bs#-d.f-97RH2#l<o@ZXZ&T?iRH-<%N9SU+'so7AAgW2mil0fWaM)A,6@)4Rp@WFC0@Y,uIN:
	5uCJkMPAJFd6VVd/UR6[rI=?0fVgq/kI,s^(YARDN54WD\PD#"bV<C5XKS<`]Fql#m@)ul]O#Nn]-2
	[lB);<HM,sPARkJs:;d4_S!/nh?qGe7@I:AsnMi7DjM_D$2XW>fsY^ZQI8$;`nm!f"mg^^mq'BNs/!
	!!!j78?7R6=>B
	ASCII85End
end

// PNG: width= 90, height= 30
static Picture pHelp
	ASCII85Begin
	M,6r;%14!\!!!!.8Ou6I!!!"&!!!!?#R18/!3BT8GQ7^D&TgHDFAm*iFE_/6AH5;7DfQssEc39jTBQ
	=U%&ule5u_NK]pa@g']g@U^Upt%GFQ2iG6aJpJmdFX78;,cVIC:7E<8Td:b6cN,?9UjVf\f%WZ;[/V
	%KP#JO*M&WN?THKK9cU)`=`H0[O9,)L8;-_'G*Bak\80]?@nUmHm(8j1N'qpQ@[GqTSkLm[iV;4l?5
	9p34hHca--pCc9?`a1`N@?0>MW\`C_cK87>p!!)X0(AW7k(<k1!Z6U5Zb%=PW#*`9'?8(R.l9\9B%n
	9#Xi,CY"\p?O1oCK4p_#dS:<['Ueqmu'!DdI1i5jrP3+p\-^L%B)Rl&&6R@eFq>=L[qbCBEtmmNr/L
	*e`#,Pc(S2@J/p9W2QZ)-J]W[Rki??laDuDX[@&P,(^^j"-[%#SONo\V1=G'!!%70KV'(>nd;T\r9s
	nW!!']u`J45Y;-RJO#^h#IS8@)s?-6E37K\jr"J&9mNoQd1W?So9&&7!4QcfPj`<*6p)q!4%Z!kNSb
	ab*S0C)+!-4P&qo6Jd<2$QkkZlF^m=^($s8m)tsn,tIZn\F/q!JI,[b(u%<c'JV4aX<^Bf%*k%J6&^
	>;,IqtHG)FHLBE0]KCF+J!.aqBH9Cu\K<fV!+1qo2IdrI^jN)fkTP5;J-4NXp%T0D`+g<s/YW;Xf"@
	\,T!Yqbd?E6KnN%F4r'*)'g*uD%cKYq4thknG`:Fo$Eh;$dZNZ:%ld4Hbsa^bHA79m^g0PiWdCc346
	JH/)8KM&)(k36QF'1-*7>&j)l`JADS2'.,2Z*_)fH$KMmMP'\-](',%?j1@)>F]S9TO0p.T!W[#6cg
	fTP-7;t(+Zf_JqtSn,aMH!n\';3U2ZD&S;P.=h1rnGXk<CB$fKN,kd+9PO4u%TJ&Hu4&4Crs9Y@_nH
	"MR&l)1>b[1u<1&g\hMaNi(kl1-rkL5)pKfeO6U'VM\7&0/`!A7PfqQOE]am/tCKELH9=]@#NnO'jY
	tqYGLh>IH)^mct,(P9&uXOhUj'bDC#sD;2^*qE?dh;P68RpF'`l^&V357%l%naPeh(A($.glRQtugt
	ZGl%$*W+%Ia6#,81..8g^]id,Kck$t@][$aBOjq/OasMhNuG2M,LC'G_J("Z[P4_r5QBmFntXjd',6
	lZt\G^OJI;7Rj%OX5iIM#6uCX!]FoLK`(amh5S'MpXVq/O#EQ=C>KI[/s=orf,T\.V5"J]r\=n:<P^
	dbK.Jkf0au0o*Zjc5<u!+J$%sur)3hjN3<hR:W<>kX3+#G"^cXsdL/hY`gSKQ2hQ^Pq(l!UpCdqH+(
	.;#abis6/fk6q:$aRh5_>*eM9NS:6RPelrZ>?$aa)Nf&c!V@m]6<R6e=FIgEcWdq5MWd4Wd(EGH-(t
	7'6n3kpg)VSEUsmM3$iUNPR)sTB+NfY$cV71VX=4n!'f7b&o2]5j7tq;=*jFX8d1ff/P>NYa\kIC.H
	0D<]`$GJ&fmmVZ]hoTDId<i+X\WBThm/_*d[#n'GhQY]#TjRa0oj>Y^?+^*&qn&>%qpqLF8^A1;Q1&
	-O1QL#&^mh-Vg1nFHc5Yg_^Fm5Mj%+,Mn_\Yol9dGTnRi%0.uNj'\-e(in+;$'enZ/=ITkLsojIs'f
	5J4*Sf)rJ-0)p,hFVP)c6^;1hlKdF&nZ-FC.eJ7Vk#%#29.N7pubh'DZ93h9r&DjT2'SNDUuSfXk:T
	hm!T6a,*oAQcCI[)@&bnCG;H\^W#'+J+VfO[ZNjXF'@+Pe?YI;kq8O!7WJ6/>GbW`_[MnT91E/eol'
	bTm^=u,IJJ5lD.poMum`/$SP;Lj,6doEk."\m=nfH#0X:Yfbb'MOIr-edqrknBHJ'.M!5h;8]P4A(2
	C5q?EY.TLP&:@>Ph$?Yg`,3D[$dC9?0\Lot/&E.W?j`_I5L)S`l("cMF\gETWuSY0H>CU9[07$'*X?
	.ECCH8hd=RSR5*cX/cU:_gpc,KYucE`q+RG\$$<TcCB`,F1^8,XK7BFNf9bRLZ:?J%"ljN:<;Dmlr<
	<hnCr)p-KUd:j9K#ZR[0<)\a0F9.RDN?6IID#?^a.u`5-q,0f8$Rlf7RA:d.C+<VOe$lIo%e;iX5o1
	`08MhTJ)r/NmJY-DDFd-.d!K_Rt0c0JkU#f>m(GTtJ#XKnY4Yh:U@^b&M*`#/`.YmHm=QB$'%\qX#r
	@k9m_?!'^:&'rj398\D*jgar,Hmr+Pss6/[R]X5WiKS!V5MLl\*=I@$r(oGZ,`<UmS89I+G&ERttG\
	?uGTYJ4QI94'p5&O$_NuCGa:bXG:/cc`G"U[\/_&o4D^R9qQ.&+2[(Oj,+Wf%(,h7@a%l,]TuYHJJ]
	c-7[nR\D2Y'FNK/2Is\:pn`I55s*LoL%75rm!\:SWNQ!$7lE$\oEiU0_1I2l(qmg9mW>fI686_Rj.5
	jeJn6.1(2"3)s(MJI+9@R=1DURoE!UFZS]K(#"X+#C.&IfNW]P,?eu`-]QAr!;/J\^(*&mqjc^33aT
	ho-+@KOjAb]rUUj[oC9NQ;ek29;O&*`'Q"/";k+Tg7'`6o+_GJCHppi2q`P`s):,hk.sAA13FNUdVO
	CqH6RTVp:'Y5Q5n4OSA+pK<_\dBuc<WYCcMO?Q@,LgL<7c;$W^J0;HuM+9W5^S^q4aWOpsd@r>fnk2
	So.FRL?T3's"<s%-?!6l<9^F=LQcfaIk'F`fAVN=fHg&+@:55-A/$,0Q;a!e\;h*Lf(XaXKL[!4-!c
	aqqcW-G6]?LB/PBBUY6(jb9r[dt)"0%,6E?I7ZC,@h&U9d0N'0B"I*cP&=W6+,5Q+jm*N\UX=lkGu&
	=k,OT/8_;BQ/7_Ej-C_!.H&/7RV2C0;(69I-DBB1!=ngL'RTsV0@Thl"/7EIHaEf0:a;]5Cc66VSZQ
	6I"!Xu@+C"F9t&6)h$QNJ4Z`l)J2%pja/;XcRC:V`\qo7%>:kj303LOY7gGU9[=BWi@Q<4J,J?\$`W
	[j`9>Hfs<YZk"J7Y4O9e.Z4_2'41CGG!H$J1+c^co`cG$^kE[l+]TgZQ_ZQAl-Ye&1(foL%GJNZubA
	I=\jNa+r25c`$-\_d!-SF>`c$*"bnA5Pd&@G[NL1-s#NT^QJE4K3ch;$eqe%2otU'SNkO0)Wq;?3TU
	rR^1M8Y>J.e=2WqS,EtAVsr+7Ye3:5n)T/gm&69l`j'CTj4Lg;;));SI-5M&;0PNH(V+qhEd59(q77
	:UfXZ&NG4+.BMM_c:\fMCIl=*8Wl$0*1n"*V,VrbVTeamc>-QOiL!'a[j8ho<N"<e&.<_R:uIoHWYJ
	V2XkU$7DS=[%EK"=$\$kMQ0&ZlWD`GVR/k$j8kX2:&F1\IrM\).p2mX&c?cjWb$4(kt>!Fg\$7MYDA
	UHms%/2@dgf]sf_>hK$8KIS*p\IF@UM*pf19J:qKEM*`+C1)HeU%mj[5%,n0(A&?QmTqML1]D_'#VU
	*PTY.VJl],a[1oi.[R2gCCs)EWsbZM?gV]Lt.iOP5c/>+CgY](/@X'N)9IE>TW8![nBi9rGA-g7>Hp
	7+aXH[X[R9gD:QXZ-eK.]oBqplJdKII8NiR<EiZXD*$*VkL$%p@CP!"\(botnDUa&J]RXWD/Cjs8kK
	_<Vt(nb'K\+MDAp)9hWgu3=Kl-_8`3<f3W%U'e"ljrNP(a<ThrgW8>ReE0a-h(!L0!@m!e[%Zs0JWq
	t?0?mV_+Ij[j)-=03DO_eQTRXK2hGf<1D3EY\ZChfAPHFN"fO]NbF:l!M_*b:=[),oo-29QT>^V:Jn
	:-m.rqHKcs0/5Y;,CW^2$p-3Z_&F&EBJB;>bhJ?i+bO_rAXf/4u_28sXAEi7<jcOgQrqM6=VYgP@_8
	H64h2ZJOlBs?ebm(55\!.MW!DM.KJ-<m>:brccmn<T3VCVWDrT)dYhZZGG0Gk5H\]2"fJq/]7#nTGM
	d*Rkr\k#n?2".f$$2_u>^Ei3E4V/;GBQFSYB4Z]qWhhRd8-hNf"q^:7iL-29.seIBTbbt(s41u^CK<
	qq#AJ*pLEHlLC:2KZR-9u#'FI,;"9\c,GW[%O/;p>tQ_L@Sk*gC1'^rD[Ng<&QqT8!\!;\90X=aNKD
	r`TCVbt6obG&O5$4H$sZ/XWa124<T4?Q;4T0A86rdItO<p@AD3h6a-P6iN]-!7Zr+m.(@HX1ct2t-b
	HR`A77"JcKsz8OZBBY!QNJ
	ASCII85End
end
