﻿#pragma rtGlobals=3		// Use modern global access method and strict wave access.
#pragma igorVersion=8
#pragma TextEncoding = "UTF-8"

STATIC CONSTANT kProjectID=22194
STATIC STRCONSTANT ksShortTitle="KML tool"

#pragma version=2.10		// to help with version control

STATIC CONSTANT kVersion=2.1
STATIC STRCONSTANT ksVersion_Number="2.1.0"

STATIC STRCONSTANT kAuthor="Kyle J Zarzana"
STATIC STRCONSTANT kAuthorEmail="kzarzana@ucar.edu"
STATIC STRCONSTANT kWebPage="https://www.wavemetrics.com/node/22194"

//Code to create KMLs that are either lines or a set of markers.  The markers can be colored and/or sized by some f(z) wave.
//When using the panel, all waves must be in the same data folder.  As of version 1.3.0, you can call the maker functions without using the panel
//		in which case the waves can be in whatever folder you want.  
//All function names start with KML_.  
//Somewhat tested, use at your own risk, etc, etc, etc...

//v1.0.0 First public release
//v1.1.0: Better navigation in the help notebook.  
//v1.1.1: 20250716: Added help button and text for the color scale legend.
//v1.2.0: 20250717: Added a scale factor for the altitude.  Changed altitude mode for lines.  Absolute is used if there is an altitude wave, relative to ground
//					is used if there isn't an altitude wave.
//v2.0.0: 20250816: Made separate functions to make the KML files so you can run them in a loop without messing with the panel.  Added wind barbs.  Added time for lines.  
//v2.1.0: 20251015: Hopefully made the indentations all consistent and correct.  Fixed a bug where the lookat tag wasn't written for lines with times.  

//Built in Google Earth icons can be found here: http://kml4earth.appspot.com/icons.html

STATIC CONSTANT kvDefault_Notebook_Size=12			//Controls the default (minimum) font size in the help notebook

STATIC CONSTANT kvScroll_Direction=-1				//Change this to -1 if the scroll direction offends you

STATIC STRCONSTANT kstrGoogle_Earth_Symbols="GE circle;GE shaded_dot;"
STATIC STRCONSTANT kstrIgor_Symbols="Circle;Square;Diamond;Hexagon;"			//If you want to add more markers, you also need to edit the code in the maker function.

STATIC STRCONSTANT kstrHelp_Notebook_Name="KML_Creator_Panel_Help"


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


MENU "Useful Tools"
	
	SUBMENU "KML Panel"
		"Make panel", KML_Panel_Creation()
		"-"
		"Get help", KML_Help_Open_Notebook("")
	END
	
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Makes the panel
FUNCTION KML_Panel_Creation()

	DefaultGUIFont/Mac all={"Arial", 12, 0}
	DefaultGUIFont/Win all={"Arial", 12, 0}
	
	String strWindow_Name=UniqueName("KML_Panel_", 9, 0)
	
	NewPanel/N=$strWindow_Name/W=(73,158,487,669) as strWindow_Name

	TitleBox P0_TitleBox,pos={13.00,1.00},size={87,17},title="KML Creator "+ksVersion_Number,fSize=14,frame=0,fStyle=1

	Button P0_Button_1dot_Wave_Selection,pos={374.00,4},size={35.00,15.00},proc=KML_Panel_Button_Get_Help,title="Help",help={"Opens the help notebook"},fSize=11,fStyle=1,fColor=(49151,65535,49151)
	
	GroupBox P0_GroupBox_Waves,pos={6.00,15.00},size={403,214.00},title="Waves",frame=0
		
	PopupMenu P0_Pop_Wave_Latitude,pos={62,31.00},size={98,17.00},title="Latitude:",fSize=12,mode=1,popvalue="none",value= #"\"none;\"+sortlist(WaveList(\"*\", \";\", \"TEXT:0,DIMS:1\"), \";\", 16)",appearance={default,all}
	PopupMenu P0_Pop_Wave_Longitude,pos={52,53.00},size={108,17.00},title="Longitude:",fSize=12,mode=1,popvalue="none",value= #"\"none;\"+sortlist(WaveList(\"*\", \";\", \"TEXT:0,DIMS:1\"), \";\", 16)",appearance={default,all}
	PopupMenu P0_Pop_Wave_Altitude,pos={33,75.00},size={127,17.00},title="Altitude (opt.):",fSize=12,mode=1,popvalue="none",value= #"\"none;\"+sortlist(WaveList(\"*\", \";\", \"TEXT:0,DIMS:1\"), \";\", 16)",appearance={default,all}
	PopupMenu P0_Pop_Wave_Altitude,help={"KML altitude units are meters.  If you select \"none\" then theopt.points will be on the ground instead of at the actual altitude."}
	
	PopupMenu P0_Pop_Wave_Time,pos={47,97},size={113.00,17.00},title="Time (opt.):",fSize=12,mode=1,popvalue="none",value= #"\"none;\"+sortlist(WaveList(\"*\", \";\", \"TEXT:0,DIMS:1\"), \";\", 16)",appearance={default,all}	
	
	PopupMenu P0_Pop_Wave_WindSpeed,pos={10.00,119},size={130.00,23.00},title="Wind speed (opt.):",fSize=12,mode=1,popvalue="none",value= #"\"none;\"+sortlist(WaveList(\"*\", \";\", \"TEXT:0,DIMS:1\"), \";\", 16)",appearance={default,all}
	PopupMenu P0_Pop_Wave_WindSpeed, help={"Speed in meters/second"}
	PopupMenu P0_Pop_Wave_WindDir,pos={29,141},size={131.00,17.00},title="Wind dir (opt.):",fSize=12,mode=1,popvalue="none",value= #"\"none;\"+sortlist(WaveList(\"*\", \";\", \"TEXT:0,DIMS:1\"), \";\", 16)",appearance={default,all}
	PopupMenu P0_Pop_Wave_WindDir,help={"Direction in degrees, with 0=north"}
	
	PopupMenu P0_Pop_Wave_Fz,pos={52,163},size={108.00,23.00},proc=KML_Marker_Pop_Fz,title="F(z) (opt.):",fSize=12,mode=1,popvalue="none",value= #"\"none;\"+sortlist(WaveList(\"*\", \";\", \"TEXT:0,DIMS:1\"), \";\", 16)",appearance={default,all}
	PopupMenu P0_Pop_Wave_Fz, help={"If you select \"none\" then the markers will be the same size and color."}
	
	SetVariable P0_SetVar_DataRes,pos={11.00,185},size={144.00,17.00},bodyWidth=45,title="Use every N pnts:",fSize=12,limits={1,inf,1},value=_NUM:10	
	SetVariable P0_SetVar_UTC_Offset,pos={177,185},size={114.00,17.00},bodyWidth=50,title="UTC offset",fSize=12,limits={-12,14,1},value=_NUM:0
	SetVariable P0_SetVar_UTC_Offset,help={"Only needed if you have a time wave."}
	SetVariable P0_SetVar_Alt_Scale,pos={308,185},size={93,18.00},bodyWidth=40,title="Alt. scale",help={"Altitude exaggeration factor."},fSize=12,limits={0.0001,inf,1},value=_NUM:1
	SetVariable P0_SetVar_Alt_Scale,help={"Scale factor to make the altitude stand out more."}
	SetVariable P0_SetVar_WindRes,pos={11,207},size={201,17.00},bodyWidth=45,title="Wind barb every M kml pnts:",fSize=12,limits={1,inf,1},value=_NUM:30
	SetVariable P0_SetVar_WindRes,help={"Wind barbs added to every Mth point in the kml, not in the original wave!"}
	
	
	TabControl P0_Tab_Control,pos={1.00,231.00},size={413,305},tabLabel(0)="Line",tabLabel(1)="Markers",tabLabel(2)="Wind barbs",value= 0,proc=KML_Panel_Tab_Control,fstyle=1

	//Tab 0
	GroupBox T0_GroupBox_Line_Settings,pos={6.00,259},size={160,71.00},title="Line settings",frame=0
	SetVariable T0_SetVar_Line_Width,pos={14.00,281.00},size={108.00,19.00},bodyWidth=50,title="Line width",fSize=12,limits={0,inf,0.1},value= _NUM:2
	PopupMenu T0_Pop_Line_Color,pos={17.00,306.00},size={105.00,17.00},title="Line color",fSize=12,mode=2,popColor= (65535,65535,65535),value= #"\"*COLORPOP*\""

	Button T0_Button_2dot_Line,pos={178.00,272.00},size={35.00,15.00},proc=KML_Panel_Button_Get_Help,title="Help",help={"Opens the help notebook"},fSize=11,fStyle=1,fColor=(49151,65535,49151)
	
	SetVariable T0_SetStr_File_Name,pos={9,336.00},size={400,18.00},bodyWidth=341,title="File name",fSize=12,value= _STR:"", help={"Don't add the .kml extension to the file name."}
	TitleBox T0_Title_File_Name_Note,pos={331,356.00},size={82.00,15.00},title="Don't add .kml",fSize=12,frame=0

	Button T0_Button_Make_KML,pos={9.00,360},size={120,25.00},title="Make KML file",proc=KML_Line_Button_Make_File,appearance={os9,all},fSize=16,fstyle=1,fColor=(0,0,65535),valueColor=(65535,65535,65535)

	//Tab 1
	Button T1_Button_3dot_Markers,pos={371,273},size={35.00,15.00},proc=KML_Panel_Button_Get_Help,title="Help",help={"Opens the help notebook"},fSize=11,fStyle=1,fColor=(49151,65535,49151),disable=1

	//Fz values
	GroupBox T1_GroupBox_Fz_Values,pos={6.00,259},size={140,80},title="Fz min/max",frame=0,disable=1
	SetVariable T1_SetVar_Fz_Min,pos={17,276.00},size={123,19.00},bodyWidth=100,title="Min",fSize=12,value= _NUM:0,disable=1,limits={-inf,inf,0}
	SetVariable T1_SetVar_Fz_Min,help={"Must be a real number (no NaNs or Infs).\rIf doing log scaling, must be a postive number."}
	SetVariable T1_SetVar_Fz_Max,pos={15,296.00},size={125,19.00},bodyWidth=100,title="Max",fSize=12,value= _NUM:0,disable=1,limits={-inf,inf,0}
	SetVariable T1_SetVar_Fz_Max,help={"Must be a real number (no NaNs or Infs).\rIf doing log scaling, must be a postive number."}
	Button T1_Button_Get_Min_Max_Values,pos={60,316.00},size={80,20.00},proc=KML_Marker_Button_Fz_Min_Max,title="Get min/max",disable=1
	Button T1_Button_Get_Min_Max_Values,help={"Gets the min/max values from the f(z) wave, assuming it exists."}
	
	//Markers
	GroupBox T1_GroupBox_Markers,pos={153.00,259.00},size={256.00,80.00},title="Markers",frame=0,disable=1
	
	String strMarker_List="\""+kstrGoogle_Earth_Symbols+kstrIgor_Symbols+"\""
	PopupMenu T1_Pop_Symbol,pos={158.00,274},size={113.00,17.00},title="Symbol:",fSize=12,mode=1,popvalue="GE circle",value= #strMarker_List,disable=1
	PopupMenu T1_Pop_Symbol help={"GE circle has a black border, and GE shaded_dot has shading, so you don't quite\rget the color you want.  The other shapes are only one color."}
	CheckBox T1_Check_SizeFz,pos={158,296},size={122.00,23.00},title="Marker size as f(z)",fSize=14,value= 1,proc=KML_Marker_Check_Fz_Size_Color,disable=1
	CheckBox T1_Check_Log_Size,pos={300,274},size={64.00,16.00},title="Log size",fSize=12,value= 0,disable=1
	CheckBox T1_Check_Log_Size,help={"If checked (and both Fz min and max values are positive), then use log\rscaling for the marker sizes.  If unchecked (or either min/max values\rare negative), use linear scaling."}
	SetVariable T1_SetVar_Marker_Size_Min,pos={161,315.00},size={108,18.00},bodyWidth=60,title="Min size",fSize=12,value= _NUM:0.5,limits={0,inf,0.1},disable=1	
	SetVariable T1_SetVar_Marker_Size_Max,pos={285,315},size={110,18.00},bodyWidth=60,title="Max size",fSize=12,value= _NUM:2.5,limits={0,inf,0.1},disable=1
	SetVariable T1_SetVar_Marker_Size_Default,pos={161,315},size={116,18.00},bodyWidth=50,title="Marker size",fSize=12,value= _NUM:1,limits={0,inf,0.1},disable=1
	
	//Colors
	GroupBox T1_GroupBox_Colors,pos={6,338},size={403,62},title="Colors",frame=0,disable=1
	CheckBox T1_Check_ColorFz,pos={13,355},size={97.00,17.00},title="Color as f(z)",fSize=14,value= 1,proc=KML_Marker_Check_Fz_Size_Color,disable=1
	
	PopupMenu T1_Pop_Marker_Color,pos={12,375},size={82.00,17.00},title="Marker color",fSize=12,mode=2,popColor= (65535,65535,65535),value= #"\"*COLORPOP*\"",disable=1
	
	CheckBox T1_Check_Reverse_Colors,pos={117,355},size={101.00,16.00},title="Reverse colors",fSize=12,value= 0,disable=1
	CheckBox T1_Check_Log_Colors,pos={225,355},size={76.00,16.00},title="Log colors",fSize=12,value= 0,disable=1
	CheckBox T1_Check_Log_Colors,help={"If checked (and both Fz min and max values are positive), then use log\rscaling for the marker sizes.  If unchecked (or either min/max values\rare negative), use linear scaling."}
	
	CheckBox T1_Check_Custom_Color_Table,pos={13,377},size={61.00,16.00},title="Custom",fSize=12,value= 0,proc=KML_Marker_Check_Fz_Size_Color,disable=1
	CheckBox T1_Check_Custom_Color_Table, help={"You need to load/create the custom color table before it can be used.\rIgor has a lot of color tables stored as .ibw files in the Igor Pro folder,\rwhich can be accessed from the Help menu."}
	
	PopupMenu T1_Pop_Color_Table,pos={83,375.00},size={250.00,20.00},bodyWidth=250,fSize=12,mode=2,value=#"\"*COLORTABLEPOP*\"",disable=1
	PopupMenu T1_Pop_Custom_Color_Table_Wave,pos={83,375},size={58.00,23.00},fSize=12,mode=1,popvalue="none",value= #"\"none;\"+sortlist(WaveList(\"*\", \";\", \"TEXT:0,DIMS:2\"), \";\", 16)",disable=1
	PopupMenu T1_Pop_Custom_Color_Table_Wave, help={"This only selects 2-D non-text waves."}

	//Color scale
	Button T1_Button_4dot_Color_Scale,pos={371.00,415},size={35.00,15.00},proc=KML_Panel_Button_Get_Help,title="Help",help={"Opens the help notebook"},fSize=11,fStyle=1,fColor=(49151,65535,49151),disable=1

	GroupBox T1_GroupBox_Color_Scale,pos={6.00,400},size={403,60.00},title="Color scale",frame=0,disable=1
	CheckBox T1_Check_Color_Scale,pos={14,417},size={80.00,16.00},title="Color scale",fSize=12,value= 0,proc=KML_Marker_Check_Fz_Size_Color,disable=1
		
	PopupMenu T1_Pop_Color_Scale_Label_Side,pos={99,415},size={80,23.00},title="Side",fSize=12,mode=1,popvalue="Left",value= #"\"Left;Right;Top;Bottom\"",disable=1
	CheckBox T1_Check_Transparent,pos={203,417},size={86.00,16.00},title="Transparent",fSize=12,value= 0,disable=1
	CheckBox T1_Check_Transparent,help={"Makes any white colors on the plot transparent\rwhen loaded into Google Earth."}
	Button T1_Button_Make_Color_Scale_Plot,pos={297,415},size={60,20.00},proc=KML_Marker_Button_Make_CS_Plot,title="Make plot",fSize=10,fStyle=1,disable=1
	Button T1_Button_Make_Color_Scale_Plot, help={"Makes a plot that will get used as the color scale in Google Earth.\rYou will want to tweak the settings to make it look nice."}

	PopupMenu T1_Pop_CS_Graph,pos={14,436},size={160.00,22.00},title="Graph",fSize=12,mode=1,popvalue="none",value= #"\"none;\"+sortlist(WinList(\"*\", \";\", \"WIN:1\"))",disable=1
	
	SetVariable T1_SetStr_File_Name,pos={9,463.00},size={400,18.00},bodyWidth=341,title="File name",fSize=12,value= _STR:"",disable=1
	SetVariable T1_SetStr_File_Name, help={"Don't add the .kml extension to the file name."}
	TitleBox T1_Title_File_Name_Note,pos={331,485},size={82.00,15.00},title="Don't add .kml",fSize=12,frame=0,disable=1

	Button T1_Button_Make_KML,pos={9.00,484.00},size={120.00,25.00},proc=KML_Marker_Button_Make_File,title="Make KML file",disable=1,appearance={os9,all},fSize=16,fstyle=1,fColor=(0,0,65535),valueColor=(65535,65535,65535)
	
	SetWindow kwTopWin,hook(KML_Scroll_Hook)=KML_Panel_Scroll_Hook
	
	
	//Tab 3 wind barbs
	CheckBox T2_Check_Barb_Use_Line_Marker_Color,pos={11.00,261.00},size={134.00,15.00},proc=KML_Wind_Check_Color_Selector,title="Use line/marker color",value=1,mode=1,disable=1
	CheckBox T2_Check_Barb_Use_Custom_Color,pos={11.00,282.00},size={113.00,15.00},proc=KML_Wind_Check_Color_Selector,title="Use custom color",value=0,mode=1,disable=1
	PopupMenu T2_Pop_Barb_Color,pos={154.00,280.00},size={109.00,19.00},title="Barb color",fSize=12,mode=2,popColor=(65535,65535,65535),value=#"\"*COLORPOP*\"",disable=1

	SetVariable T2_SetVar_Barb_Length,pos={12.00,311.00},size={105.00,17.00},limits={1,inf,0.1},bodyWidth=40,title="Barb length",value=_NUM:5,disable=1
	SetVariable T2_SetVar_Barb_Thick,pos={20.00,331.00},size={97.00,17.00},limits={1,inf,0.1},bodyWidth=40,title="Barb thick",value=_NUM:3,disable=1
	SetVariable T2_SetVar_Barb_Scale,pos={16.00,351.00},size={101.00,17.00},bodyWidth=40,title="Barb scale",limits={1,inf,0.1},value=_NUM:3,disable=1

	
	Button T2_Button_Make_Test_Barb,pos={26.00,373.00},size={90.00,20.00},proc=KML_Wind_Button_Make_Test_Barb,title="Make test barb",disable=1

END


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


FUNCTION KML_Panel_Scroll_Hook(s)
	STRUCT WMWinHookStruct &s

	STRSWITCH(s.eventName)
		CASE "mouseWheel":
			String strWindow_Name=s.winName
		
			Variable vMouse_X_Pos=s.mouseLoc.h
			Variable vMouse_Y_Pos=s.mouseLoc.v
			Variable vScroll_Mode=s.wheelDy<0
			Variable vIncrement=SelectNumber(vScroll_Mode, -1, 1)*kvScroll_Direction
			
			//Checks to see if we are on a setvariable
			WAVE/T twSetVar_Controls=ListToTextWave(ControlNameList(strWindow_Name, ";", "*_SetVar_*"), ";")
			Variable vNum_SetVars=numpnts(twSetVar_Controls)
			Variable iSetVar_Dex=0, vSetVar_Flag=0
			DO
				String strSetVar_Ctrl_Name=twSetVar_Controls[iSetVar_Dex]		//Loop over the setvars until we find the active one the mouse is over
				ControlInfo/W=$strWindow_Name $strSetVar_Ctrl_Name
				vSetVar_Flag=(vMouse_X_Pos>=V_left && vMouse_X_Pos<=(V_left+V_Width) && vMouse_Y_Pos>=(V_top) && vMouse_Y_Pos<=(V_top+V_Height) && V_disable==0)
				iSetVar_Dex+=1
			WHILE(!vSetVar_Flag && iSetVar_Dex<vNum_SetVars)
			
			//Checks to see if we are on a popup
			WAVE/T twPopup_Controls=ListToTextWave(ControlNameList(strWindow_Name, ";", "*_Pop_*"), ";")		//This will have problems if you have embedded subwindows, which we don't have here
			Variable vNum_Popups=numpnts(twPopup_Controls)
			Variable iPop_Dex=0, vPop_Flag=0, vWidth_Mod
			DO
				String strPop_Ctrl_Name=twPopup_Controls[iPop_Dex]		//Loop over the popups until we find the active one the mouse is over
				IF(StringMatch(strPop_Ctrl_Name, "*P0_Pop_Wave_*"))
					vWidth_Mod=250
				ELSEIF(cmpstr(strPop_Ctrl_Name, "T1_Pop_Symbol")==0)
					vWidth_Mod=150
				ELSEIF(cmpstr(strPop_Ctrl_Name, "T1_Pop_Color_Table")==0)
					vWidth_Mod=250
				ENDIF
				
				ControlInfo/W=$strWindow_Name $strPop_Ctrl_Name
				vPop_Flag=(vMouse_X_Pos>=V_left && vMouse_X_Pos<=(V_left+vWidth_Mod) && vMouse_Y_Pos>=(V_top) && vMouse_Y_Pos<=(V_top+V_Height) && V_disable==0)
				iPop_Dex+=1
			WHILE(!vPop_Flag && iPop_Dex<vNum_Popups)
			
			IF(vSetVar_Flag)
				ControlInfo/W=$strWindow_Name $strSetVar_Ctrl_Name
				Variable vSetVar_Value=V_Value
				
				Variable vSetVar_Limits_Start=strsearch(S_Recreation, "limits={", 0, 2)
				Variable vSetVar_Limits_Stop=strsearch(S_Recreation, "}", vSetVar_Limits_Start, 2)
				
				IF(vSetVar_Limits_Start>=0 && vSetVar_Limits_Stop>=0)
					String strSetVar_Limits=S_Recreation[vSetVar_Limits_Start+8, vSetVar_Limits_Stop-1]
					Variable vSetVar_Min=str2num(StringFromList(0, strSetVar_Limits, ","))
					Variable vSetVar_Max=str2num(StringFromList(1, strSetVar_Limits, ","))
					Variable vSetVar_Increment=str2num(StringFromList(2, strSetVar_Limits, ","))*SelectNumber(s.eventmod & 8, 1, 10)
					
					IF(cmpstr(StringFromList(2, strSetVar_Limits, ","), "0")!=0)		//A little redundant, but better safe than sorry
						Variable vSetVar_New_Value=limit(vSetVar_Value+vSetVar_Increment*vIncrement, vSetVar_Min, vSetVar_Max)
						SetVariable/Z $strSetVar_Ctrl_Name, win=$strWindow_Name, value=_NUM:vSetVar_New_Value
					ENDIF
				ENDIF
			ENDIF
			
			
			IF(vPop_Flag)
				IF(StringMatch(strPop_Ctrl_Name, "*P0_Pop_Wave_*"))
					String strAll_Waves="none;"+sortlist(WaveList("*", ";", "TEXT:0,DIMS:1"))
					Variable vNum_Pop_Items=ItemsInList(strAll_Waves)
					
					ControlInfo/W=$strWindow_Name $strPop_Ctrl_Name
					Variable vPop_New_Value=limit(V_Value-vIncrement, 1, vNum_Pop_Items)
					String strPop_New_Value=StringFromList(vPop_New_Value-1, strAll_Waves)
					
					PopUpMenu/Z $strPop_Ctrl_Name, win=$strWindow_Name, mode=vPop_New_Value
					ControlUpdate/W=$strWindow_Name $strPop_Ctrl_Name
					
					IF(StringMatch(strPop_Ctrl_Name, "P0_Pop_Wave_Fz") && vPop_New_Value==(V_Value-vIncrement))
						
						STRUCT WMPopupAction Fz_pa
						Fz_pa.win=strWindow_Name
						Fz_pa.eventCode=2
						Fz_pa.popStr=strPop_New_Value
						
						KML_Marker_Pop_Fz(Fz_pa)
					ENDIF
					
				ELSEIF(cmpstr(strPop_Ctrl_Name, "T1_Pop_Symbol")==0)
					vNum_Pop_Items=ItemsInList(kstrGoogle_Earth_Symbols)+ItemsInList(kstrIgor_Symbols)
					
					ControlInfo/W=$strWindow_Name T1_Pop_Symbol	
					vPop_New_Value=limit(V_Value-vIncrement, 1, vNum_Pop_Items)		//Minus increment so scrolling down goes down the popup list (which is actually increasing the index)
					
					PopUpMenu/Z T1_Pop_Symbol, win=$strWindow_Name, mode=vPop_New_Value
					ControlUpdate/W=$strWindow_Name T1_Pop_Symbol
					
				ELSEIF(cmpstr(strPop_Ctrl_Name, "T1_Pop_Color_Table")==0)
					Variable vNum_Color_Tables=ItemsInList(CTabList())
					
					ControlInfo/W=$strWindow_Name T1_Pop_Color_Table
					vPop_New_Value=limit(V_Value-vIncrement, 1, vNum_Color_Tables)		//Minus increment so scrolling down goes down the popup list (which is actually increasing the index)
					
					PopUpMenu/Z T1_Pop_Color_Table, win=$strWindow_Name, mode=vPop_New_Value
					ControlUpdate/W=$strWindow_Name T1_Pop_Color_Table
					
				ELSEIF(cmpstr(strPop_Ctrl_Name, "T1_Pop_Color_Scale_Label_Side")==0)
					ControlInfo/W=$strWindow_Name T1_Pop_Color_Scale_Label_Side
					vPop_New_Value=limit(V_Value-vIncrement, 1, 4)
					
					PopUpMenu/Z T1_Pop_Color_Scale_Label_Side, win=$strWindow_Name, mode=vPop_New_Value
					ControlUpdate/W=$strWindow_Name T1_Pop_Color_Table
				
				ELSEIF(cmpstr(strPop_Ctrl_Name, "T1_Pop_CS_Graph")==0)
					String strAll_Graph_Windows="none;"+sortlist(WinList("*", ";", "WIN:1"))
					Variable vNum_Graph_Windows=ItemsInList(strAll_Graph_Windows)
					
					ControlInfo/W=$strWindow_Name T1_Pop_CS_Graph
					vPop_New_Value=limit(V_Value-vIncrement, 1, vNum_Graph_Windows)
					strPop_New_Value=StringFromList(vPop_New_Value-1, strAll_Graph_Windows)
					
					PopUpMenu/Z T1_Pop_CS_Graph, win=$strWindow_Name, mode=vPop_New_Value
					ControlUpdate/W=$strWindow_Name T1_Pop_CS_Graph
				ENDIF
			
			ENDIF
			
			
		BREAK
	ENDSWITCH
		
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Launches the help notebook
FUNCTION KML_Panel_Button_Get_Help(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	String nb=kstrHelp_Notebook_Name
	
	SWITCH(ba.eventcode)
		CASE 2:
		
			String strControl_Name=ba.ctrlName
			String strSection=RemoveListItem(0, RemoveListItem(0, strControl_Name, "_"), "_")
			strSection=ReplaceString("dot", strSection, ".")
			strSection=ReplaceString("_", strSection, " ")
			strSection="* "+strSection
			
			IF(strlen(WinList(nb,";","WIN:16")) > 0 )
				DoWindow/F $nb
			
			ELSE
				KML_Help_Open_Notebook("")
			ENDIF
			
			//I can't encode the whole section name in the control name (unless I use User_Data, which is a pain), so I have to make the note book and then go to the right spot			
			Notebook $nb, findText={"***END***",9}		//Go to the end of the notebook
			
			Notebook $nb, findText={strSection,25}			//Search backwards here so the section ends up at the top of the displayed text
			Notebook $nb, findText={"*", 9}
			GetSelection notebook, $nb, 1
			
			Notebook $nb, selection={(V_startParagraph,0),(V_endParagraph,V_EndPos)}
		BREAK
	ENDSWITCH
	
	Return 0
	
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


FUNCTION KML_Help_Open_Notebook(String strSection)

	String nb=kstrHelp_Notebook_Name			//nb is the Igor name when you make the notebook commands, so using this to avoid a bunch of replacements.
	
	IF(strlen(WinList(nb,";","WIN:16"))>0)
		DoWindow/F $nb
		
		IF(strlen(strSection)>0)
			Notebook $nb, findText={strSection,9}
		ENDIF
		
		return 0
	
	ELSE
		String strUpdate_Date="Glorious 25th of May, Year of the Dancing Dog"
		String strIpf_path=FunctionPath(GetRTStackInfo(1))
		GetFileFolderInfo/Q/Z strIpf_path
		IF(V_Flag==0 && V_IsFile)
			strUpdate_Date=Secs2Date(V_modificationDate, -2)
		ENDIF
		
		String strGoToSection_Cmd_Start="Notebook "+nb+" findText={\"***END***\",9}\r"
		String strGoToSection_Cmd_End="GetSelection notebook, "+nb+", 1"
		String strGoToSection_Cmd_Middle=""
		
		NewNotebook/N=$nb/F=1/V=1/K=1/ENCG={3,1}/OPTS=4
		Notebook $nb showRuler=0
		Notebook $nb newRuler=Normal, justification=0, margins={0,0,670}, spacing={0,0,0}, tabs={}, rulerDefaults={"Arial",kvDefault_Notebook_Size,0,(0,0,0)}
		Notebook $nb ruler=Normal, fSize=kvDefault_Notebook_Size+3, fStyle=1, text="KML Creator documentation, v"+ksVersion_Number+"\r"
		
		Notebook $nb text=""
		Notebook $nb fSize=-1, fStyle=-1, text="Last updated "+strUpdate_Date+"\r"
		Notebook $nb text="\r"
		
		Notebook $nb fSize=kvDefault_Notebook_Size+1, fStyle=1, text="Table of Contents\r"
//		Notebook $nb fSize=-1, fStyle=-1, text="0. "
		NotebookAction commands=("KML_Help_Notebook_Nav(1,\"* 0. Panel features *\")"), linkStyle=1, quiet=1, title="0. Panel features"
		
		Notebook $nb text="\r"
		NotebookAction commands=("KML_Help_Notebook_Nav(1,\"* 1. Wave selection *\")"), linkStyle=1, quiet=1, title="1. Wave selection"
		
		Notebook $nb text="\r"
		strGoToSection_Cmd_Middle="Notebook "+nb+", findText={\"* 2. Line *\",25}\r"
		NotebookAction commands=("KML_Help_Notebook_Nav(1,\"* 2. Line *\")"), linkStyle=1, quiet=1, title="2. Line"
		
		Notebook $nb text="\r"
		strGoToSection_Cmd_Middle="Notebook "+nb+", findText={\"* 3. Markers *\",25}\r"
		NotebookAction commands=("KML_Help_Notebook_Nav(1,\"* 3. Markers *\")"), linkStyle=1, quiet=1, title="3. Markers"
		
		Notebook $nb text="\r"
		strGoToSection_Cmd_Middle="Notebook "+nb+", findText={\"* 4. Color Scale *\",25}\r"
		NotebookAction commands=("KML_Help_Notebook_Nav(1,\"* 4. Color Scale *\")"), linkStyle=1, quiet=1, title="4. Color Scale"
		
		Notebook $nb text="\r"
		strGoToSection_Cmd_Middle="Notebook "+nb+", findText={\"* 5. Using in loops *\",25}\r"
		NotebookAction commands=("KML_Help_Notebook_Nav(1,\"* 5. Using in loops *\")"), linkStyle=1, quiet=1, title="5. Using in loops"
		Notebook $nb text="\r\r"
		Notebook $nb text="\r"
		
		Notebook $nb fSize=kvDefault_Notebook_Size+1, fStyle=1, text="* 0. Panel features *", fSize=-1, fStyle=0, text=" "
		NotebookAction commands=("KML_Help_Notebook_Nav(0,\"\")"), linkStyle=1, quiet=1, title="(return to top)"
		Notebook $nb text="\r"
		Notebook $nb text="Mouse wheel scrolling works for all popups except the color palette selector.  Mouse wheel scrolling also works for set variables with increment arrows.  "
		Notebook $nb text="Hold down the control/command key to increment by 10x the default increment.  \r"
		Notebook $nb text="\r\r"
		
		Notebook $nb fSize=kvDefault_Notebook_Size+1, fStyle=1, text="* 1. Wave selection *", fSize=-1, fStyle=0, text=" "
		NotebookAction commands=("KML_Help_Notebook_Nav(0,\"\")"), linkStyle=1, quiet=1, title="(return to top)"
		Notebook $nb text="\r"
		Notebook $nb text="Select the waves you want to use.  ALL waves used (position waves, time waves, color table waves) need to be in the current data folder.  "
		Notebook $nb text="KML uses meters for the altitude, and there is a scale factor setvar if you want to exaggerate the heights.  " 
		Notebook $nb text="The f(z) and time wave options will not do anything if you make a line kml.  \r"
		Notebook $nb text="\r"
		Notebook $nb text="Adding every point to the KML will probably be overkill, and there is an option to use every Nth point (note that this does NOT average the data). Select 1 if you want all the points to be in the KML.  \r"
		Notebook $nb text="\r"
		Notebook $nb text="If you are using a timewave, enter the UTC offset of the time wave.  This will ensure that the times in Google Earth are correct.  \r" 
		Notebook $nb text="\r"
		Notebook $nb text="Adding a wind barb on every point in the KML will be overkill, and there is an option to set the frequency of the barbs.  If you set \"Use every N pnts\" to 10 and "
		Notebook $nb text="\"Wind barbs every M kml pnts\" to 10, then the KML will only have every tenth point from the original dataset, and only 1 in every 10 of those points will have a wind barb.\r"
		 Notebook $nb text="\r\r"
		
		Notebook $nb fSize=kvDefault_Notebook_Size+1, fStyle=1, text="* 2. Line *", fSize=-1, fStyle=0, text=" "
		NotebookAction commands=("KML_Help_Notebook_Nav(0,\"\")"), linkStyle=1, quiet=1, title="(return to top)"
		Notebook $nb text="\r"
		Notebook $nb text="This option makes a KML file with the track as a continuous line.  You can add a altitude wave if desired.  "
		Notebook $nb text="The only things you can change are the line color and width.  "
		Notebook $nb text="I haven't figured out how to make the line anything other than one color, and I haven't figured out how to add the time slider bar for lines, "
		Notebook $nb text="so selecting an f(z) trace or a time trace won't do anything.  \r"
		Notebook $nb text="\r\r"
		
		Notebook $nb fSize=kvDefault_Notebook_Size+1, fStyle=1, text="* 3. Markers *", fSize=-1, fStyle=0, text=" "
		NotebookAction commands=("KML_Help_Notebook_Nav(0,\"\")"), linkStyle=1, quiet=1, title="(return to top)"
		Notebook $nb text="\r"
		Notebook $nb text="This option will make a KML file with the track as markers.  You can color and size the markers by some f(z) trace, similar to the f(z) options in the Modify Trace Appearance window. \r"
		Notebook $nb text="\r"
		Notebook $nb text="If you select a time wave a slider bar will appear in Google Earth to animate the track.  \r"
		Notebook $nb text="\r"
		Notebook $nb text="For the markers, you can use one of the "
		NotebookAction commands=("BrowseUrl/Z \"http://kml4earth.appspot.com/icons.html\""), linkStyle=1, quiet=1, title="Google Earth built-in options", helpText="http://kml4earth.appspot.com/icons.html"
		Notebook $nb text=", or one of the Igor shapes.  Only a few Igor shapes are in the code right now, but in principle any shape can be added.  "
		Notebook $nb text="If you use an Igor shape, the code will create a png file with the shape template.  This png file must accompany the KML.  \r"
		Notebook $nb text="\r"
		Notebook $nb text="You can use one of the built-in Igor color tables, or you can use a custom one.  Igor has a ton of other color tables available as .ibw files in the "
		NotebookAction commands=("DoIgorMenu \"Help\", \"Show Igor Pro Folder\""), linkStyle=1, quiet=1, title="Igor Pro Folder", helpText="Opens the Igor Pro folder on the disk"
		Notebook $nb text=".  "	
		Notebook $nb text="You will need to load the .ibw file into the experiment, and it needs to be in the same data folder as the position waves.  "
		Notebook $nb text="More information about color tables is here: "
		NotebookAction commands=("DisplayHelpTopic/Z \"Color Table Waves in the Igor Pro Folder\""), linkStyle=1, quiet=1, title="Color Table Waves in the Igor Pro Folder", helpText="Opens the help for Color Table waves"
		Notebook $nb text="."
		Notebook $nb text="\r\r\r"
		
		Notebook $nb fSize=kvDefault_Notebook_Size+1, fStyle=1, text="* 4. Color Scale *", fSize=-1, fStyle=0, text=" "
		NotebookAction commands=("KML_Help_Notebook_Nav(0,\"\")"), linkStyle=1, quiet=1, title="(return to top)"
		Notebook $nb text="\r"
		Notebook $nb text="If you want a color scale for the f(z) values, you will need to create an image of the color scale that gets displayed by the kml.  "
		Notebook $nb text="\r\rFirst decide on the layout of the scale.  Choosing left or right from the Side menu will make a vertically oriented color scale, while top and bottom will make a horizontal color scale.  "
		Notebook $nb text="\r\rSelect whether you want the background to be transparent or not, and then click the Make plot button.  A graph of the color scale will appear that you can customize.  "
		Notebook $nb text="\r\rWhen you make the kml file, the code will save a png of the graph, which will then get linked to the kml.  This png file also needs to be in the same folder as the kml.  "
		Notebook $nb text="\r\r"
		
		Notebook $nb fSize=kvDefault_Notebook_Size+1, fStyle=1, text="* 5. Using in loops *", fSize=-1, fStyle=0, text=" "
		NotebookAction commands=("KML_Help_Notebook_Nav(0,\"\")"), linkStyle=1, quiet=1, title="(return to top)"
		Notebook $nb text="\r"
		Notebook $nb text="If you want to make kmls in a loop, you can call the functions for making the files without using the panel.  You need to make waves with the text and numeric inputs beforehand.  "
		Notebook $nb text="To make a kml file where the path is a line, you need two waves with the following inputs:\r"
		Notebook $nb text="wUser_Numeric_Inputs[0]=resolution\r"
		Notebook $nb text="wUser_Numeric_Inputs[1]=altitude scale factor\r"
		Notebook $nb text="wUser_Numeric_Inputs[2]=UTC offset\r"
		Notebook $nb text="wUser_Numeric_Inputs[3]=line width\r"
		Notebook $nb text="wUser_Numeric_Inputs[4]=overwrite flag, 1=overwrite file with same name\r\r"

		Notebook $nb text="twUser_Text_Inputs[0]=full path to latitude wave\r"
		Notebook $nb text="twUser_Text_Inputs[1]=full path to longitude wave\r"
		Notebook $nb text="twUser_Text_Inputs[2]=full path to altitude wave (if using, if not using leave blank or set to \"none\")\r"
		Notebook $nb text="twUser_Text_Inputs[3]=full path to time wave (if using, if not using leave blank or set to \"none\")\r"
		Notebook $nb text="twUser_Text_Inputs[4]=line color, can be hex format or RGB (Red,Green,Blue)\r"
		Notebook $nb text="twUser_Text_Inputs[5]=file name\r"
		Notebook $nb text="twUser_Text_Inputs[6]=path to folder for file\r\r"
		
		Notebook $nb text="To make a kml file where the path is a marker, you need two waves with the following inputs:\r"
		Notebook $nb text="wUser_Numeric_Inputs[0]=resolution\r"
		Notebook $nb text="wUser_Numeric_Inputs[1]=altitude scale factor\r"
		Notebook $nb text="wUser_Numeric_Inputs[2]=UTC offset\r"
		Notebook $nb text="wUser_Numeric_Inputs[3]=F(z) min\r"
		Notebook $nb text="wUser_Numeric_Inputs[4]=F(z) max\r"
		Notebook $nb text="wUser_Numeric_Inputs[5]=Color by F(z) (=1)\r"
		Notebook $nb text="wUser_Numeric_Inputs[6]=Reverse color scale (=1)\r"
		Notebook $nb text="wUser_Numeric_Inputs[7]=Log colors (=1)\r"
		Notebook $nb text="wUser_Numeric_Inputs[8]=Transparent color scale (=1)\r"
		Notebook $nb text="wUser_Numeric_Inputs[9]=Size by F(z) (=1)\r"
		Notebook $nb text="wUser_Numeric_Inputs[10]=Marker size 1\r"
		Notebook $nb text="wUser_Numeric_Inputs[11]=Marker size 2\r"
		Notebook $nb text="wUser_Numeric_Inputs[12]=Log marker size (=1)\r"
		Notebook $nb text="wUser_Numeric_Inputs[13]=Marker default size\r"
		Notebook $nb text="wUser_Numeric_Inputs[14]=overwrite flag, 1=overwrite file with same name\r\r"

		Notebook $nb text="twUser_Text_Inputs[0]=full path to latitude wave\r"
		Notebook $nb text="twUser_Text_Inputs[1]=full path to longitude wave\r"
		Notebook $nb text="twUser_Text_Inputs[2]=full path to altitude wave (if using, if not using leave blank or set to \"none\")\r"
		Notebook $nb text="twUser_Text_Inputs[3]=full path to time wave (if using, if not using leave blank or set to \"none\")\r"
		Notebook $nb text="twUser_Text_Inputs[4]=full path to fz wave (if using)\r"
		Notebook $nb text="twUser_Text_Inputs[5]=color table info.  If using a built in color table, use Color table=[name of color table].  If using a custom color table, use Custom=[Full path to color table wave]\r"
		Notebook $nb text="twUser_Text_Inputs[6]=name of color scale graph window\r"
		Notebook $nb text="twUser_Text_Inputs[7]=marker color (if not doing fz), can be hex format or RGB (Red,Green,Blue)\r"
		Notebook $nb text="twUser_Text_Inputs[8]=file name\r"
		Notebook $nb text="twUser_Text_Inputs[9]=file path\r"
		Notebook $nb text="twUser_Text_Inputs[10]=marker name\r\r"

		NotebookAction commands=("KML_Help_Notebook_Nav(0,\"\")"), linkStyle=1, quiet=1, title="(return to top)"
		
		Notebook $nb findText={"KML Creator documentation",9}		//sets the notebook position to the top
		Notebook $nb selection={(0,0),(0,0)}  
	ENDIF
	
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


FUNCTION KML_Help_Notebook_Nav(vMode, strSearch_Key)
	Variable vMode
	String strSearch_Key
	
	String nb=kstrHelp_Notebook_Name
	
	SWITCH(vMode)
		CASE 0:
			Notebook $nb selection={(0,0),(0,0)},findText={"",1}		
		BREAK
		
		CASE 1:
			Notebook $nb, selection={endOfFile, endOfFile},findText={"",1}
			Notebook $nb, findText={strSearch_Key,25}
			GetSelection notebook, $nb, 1
		BREAK
		
		DEFAULT:
	
		BREAK
	ENDSWITCH
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	

//Shows and hides various controls based on the tab selection
FUNCTION KML_Panel_Tab_Control(tca) : TabControl
	STRUCT WMTabControlAction &tca

	SWITCH( tca.eventCode )
		CASE 2: // mouse up
			
			String strWindow_Name=tca.win
			
			String strControls_2_Disable=ControlNameList(strWindow_Name,";", "T*")											//Makes a list of all the controls on the tabs				Took out the Panel_Red_Baron in the ControlNameList command.  Not sure it's needed
			String strControls_2_Enable=ControlNameList(strWindow_Name,";", "T"+num2str(tca.tab)+"*")						//Makes a list of the controls on the active tab
				
			ModifyControlList strControls_2_Disable, win=$strWindow_Name, disable=1									//Disables all the controls on all the  tabs
			ModifyControlList strControls_2_Enable, win=$strWindow_Name, disable=0	
			
			IF(tca.tab==1)
				STRUCT WMCheckBoxAction cba
				cba.eventcode=2
				cba.win=strWindow_Name
				
				cba.ctrlName="T1_Check_SizeFz"
				ControlInfo/W=$strWindow_Name T1_Check_SizeFz
				cba.checked=V_Value
				KML_Marker_Check_Fz_Size_Color(cba)
				
				cba.ctrlName="T1_Check_ColorFz"
				ControlInfo/W=$strWindow_Name T1_Check_ColorFz
				cba.checked=V_Value
				KML_Marker_Check_Fz_Size_Color(cba)
			
			ELSEIF(tca.tab==2)
				ControlInfo/W=$tca.win T2_Check_Barb_Use_Custom_Color
				PopupMenu T2_Pop_Barb_Color, win=$tca.win, disable=SelectNumber(V_Value, 2, 0)
			ENDIF
		BREAK
		
		CASE -1: // control being killed
		
		BREAK
	ENDSWITCH

	return 0
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


FUNCTION KML_Panel_SetVar_Control(sva) : SetVariableControl
	STRUCT WMSetVariableAction &sva

	SWITCH( sva.eventCode )
		CASE 1: // mouse up
		CASE 2: // Enter key
		CASE 3: // Live update
			Variable dval = sva.dval
			String sval = sva.sval
		BREAK
		
		CASE 4:		//Mouse scroll wheel up
		CASE 5:		//Mouse scroll wheel down
			String strWindow_Name=sva.win
			String strControl_Name=sva.ctrlName
			
			ControlInfo/W=$strWindow_Name strControl_Name
			
			
		CASE -1: // control being killed
		BREAK
	ENDSWITCH

	return 0
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


FUNCTION KML_Wind_Check_Color_Selector(cba) : CheckBoxControl
	STRUCT WMCheckboxAction &cba

	switch( cba.eventCode )
		case 2: // mouse up
			CheckBox T2_Check_Barb_Use_Line_Marker_Color, win=$cba.Win, value=StringMatch(cba.ctrlName, "T2_Check_Barb_Use_Line_Marker_Color")
			CheckBox T2_Check_Barb_Use_Custom_Color, win=$cba.Win, value=StringMatch(cba.ctrlName, "T2_Check_Barb_Use_Custom_Color")
		
			PopupMenu T2_Pop_Barb_Color, win=$cba.win, disable=SelectNumber(StringMatch(cba.ctrlName, "T2_Check_Barb_Use_Custom_Color"), 2, 0)
		
			Variable checked = cba.checked
			break
		case -1: // control being killed
			break
	endswitch

	return 0
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Makes the KML file with markers for the track
FUNCTION KML_Wind_Button_Make_Test_Barb(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	SWITCH(ba.eventcode)
		CASE 2:
			ControlInfo/W=$ba.win T2_SetVar_Barb_Length
			Variable vBarb_Length=V_Value
			
			ControlInfo/W=$ba.win T2_SetVar_Barb_Thick
			Variable vBarb_Thick=V_Value
			
			ControlInfo/W=$ba.win T2_SetVar_Barb_Scale
			Variable vBarb_Scale=V_Value
			
			Variable vSpeed_Bin=15
			
			String strBarb_Name
			sprintf strBarb_Name, "%03d", vSpeed_Bin
			
			String strBarb_Graph_Name=UniqueName("Dummy_Graph_For_Barb_", 6, 0)
		
			DFREF dfrCurrent_Data_Folder=GetDataFolderDFR()
		
			NewDataFolder/O/S root:Dummy_Temp_Barb_Folder
		
			Make/O/D/N=1 Dummy_Temp_Wave_Lat=0, Dummy_Temp_Wave_Lon=0
			Make/O/D/N=(1,3) Dummy_Temp_Wave_Barb
			SetDimLabel 1,2,windBarb,Dummy_Temp_Wave_Barb
			Dummy_Temp_Wave_Barb[0][0]=10*vBarb_Length
			Dummy_Temp_Wave_Barb[0][1]=pi/2			//Make the barb for northerly wind, and then let Google Earth handle the rotation
			Dummy_Temp_Wave_Barb[0][2]=vSpeed_Bin/5
			
			Display /W=(353.25,203,487.5,461.75)/N=$strBarb_Graph_Name/K=1 Dummy_Temp_Wave_Lat vs Dummy_Temp_Wave_Lon as strBarb_Graph_Name
			ModifyGraph margin(left)=1,margin(bottom)=1,margin(top)=1,margin(right)=1
			ModifyGraph wbRGB=(0,0,0,0)
			ModifyGraph gbRGB=(0,0,0,0)
			ModifyGraph mode=3
			ModifyGraph rgb=(0,0,0)
			ModifyGraph arrowMarker(Dummy_Temp_Wave_Lat)={Dummy_Temp_Wave_Barb,vBarb_Thick,10,1,0}
			ModifyGraph standoff=0
			ModifyGraph notation=1
			ModifyGraph axRGB=(65535,65535,65535)
			ModifyGraph tlblRGB=(65535,65535,65535)
			ModifyGraph alblRGB=(65535,65535,65535)
			ModifyGraph axisOnTop=1
			ModifyGraph btLen=4
			Label bottom " "
		
			DoUpdate/W=$strBarb_Graph_Name
			
			SetDataFolder dfrCurrent_Data_Folder
		BREAK
		
		DEFAULT:
		BREAK
	ENDSWITCH
	
END

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Makes the KML where the track is just a line
FUNCTION KML_Line_Button_Make_File(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	SWITCH(ba.eventcode)
		CASE 2:
			Close/A
		
			String strWindow_Name=ba.win
			
			Make/O/T/FREE/N=0 twKML_Inputs
			
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Latitude
			String strLatitude_Wave = S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Lat="+strLatitude_Wave
						
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Longitude
			String strLongitude_Wave = S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Lon="+strLongitude_Wave
		
			ControlInfo/W=$strWindow_Name P0_SetVar_DataRes
			Variable vResolution = V_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="DataRes="+num2istr(vResolution)
		
			WAVE/Z Latitude_Wave=$strLatitude_Wave, Longitude_Wave=$strLongitude_Wave
		
			IF(!WaveExists(Latitude_Wave) || !WaveExists(Longitude_Wave))
				DoAlert/T="ERROR ID-10T", 0, "One or more waves don't exist!"
				Return -1
			ENDIF
			
			Variable vNum_Data_Pnts = numpnts(Latitude_Wave)
			Variable vNum_KML_Pnts=ceil(vNum_Data_Pnts/vResolution)		//This should be okay
			
			IF((numpnts(Longitude_Wave)!= vNum_Data_Pnts))
				DoAlert/T="ERROR ID-10T", 0, "Waves must all have same number of points!"
				Return -1
			ENDIF
									
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Altitude
			String strAltitude_Wave = S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Alt="+strAltitude_Wave
			
			Variable vAltitude_Flag=cmpstr(strAltitude_Wave, "none")!=0 && strlen(strAltitude_Wave)>0
			IF(vAltitude_Flag)
				WAVE/Z Altitude_Wave=$strAltitude_Wave
				
				IF(numpnts(Altitude_Wave)!= vNum_Data_Pnts)
					DoAlert/T="ERROR ID-10T", 0, "Waves must all have same number of points!"
					Return -1
				ENDIF
				
				ControlInfo/W=$strWindow_Name P0_SetVar_Alt_Scale
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="AltScale="+num2str(V_Value)		
			ENDIF
			
			//Time wave
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Time
			String strTime_Wave=S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Time="+strTime_Wave
						
			Variable vTime_Flag=cmpstr(strTime_Wave, "none")!=0
			IF(vTime_Flag)
				WAVE/Z Time_Wave=$strTime_Wave
				
				IF(numpnts(Time_Wave)!= vNum_Data_Pnts)
					DoAlert/T="ERROR ID-10T", 0, "Waves must all have same number of points!"
					Return -1
				ENDIF
												
				ControlInfo/W=$strWindow_Name P0_SetVar_UTC_Offset
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="UTC offset="+num2str(V_Value)
			ENDIF
			
			//Wind parameters
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_WindSpeed
			String strWindSpeed_Wave=S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="WindSpeed="+strWindSpeed_Wave
						
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_WindDir
			String strWindDir_Wave=S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="WindDir="+strWindDir_Wave
			
			Variable vWind_Flag=cmpstr(strWindSpeed_Wave, "none")!=0 && cmpstr(strWindDir_Wave, "none")!=0
			IF(vWind_Flag)
				WAVE/Z WindSpeed_Wave=$strWindSpeed_Wave
				WAVE/Z WindDir_Wave=$strWindDir_Wave
				
				IF(numpnts(WindSpeed_Wave)!= vNum_Data_Pnts || numpnts(WindDir_Wave)!= vNum_Data_Pnts)
					DoAlert/T="ERROR ID-10T", 0, "Waves must all have same number of points!"
					Return -1
				ENDIF
				
				ControlInfo/W=$strWindow_Name P0_SetVar_WindRes
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="WindRes="+num2istr(V_Value)
				
				ControlInfo/W=$strWindow_Name T2_SetVar_Barb_Length
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Barb length="+num2str(V_Value)
				
				ControlInfo/W=$strWindow_Name T2_SetVar_Barb_Thick
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Barb thick="+num2str(V_Value)
				
				ControlInfo/W=$strWindow_Name T2_SetVar_Barb_Scale
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Barb scale="+num2str(V_Value)
				
				ControlInfo/W=$strWindow_Name T2_Check_Barb_Use_Custom_Color
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Wind color mode="+num2istr(V_Value)
								
				IF(V_Value)
					ControlInfo/W=$strWindow_Name T2_Pop_Barb_Color
					String strBarb_Color=KML_Convert_RGB_to_Hex(V_Red, V_Green, V_Blue)
					
					Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
					twKML_Inputs[numpnts(twKML_Inputs)-1]="Wind color="+strBarb_Color
				ENDIF
			ENDIF
			
			ControlInfo/W=$strWindow_Name T0_SetVar_Line_Width
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Line width="+num2str(V_Value)
			
			ControlInfo/W=$strWindow_Name T0_Pop_Line_Color
			String strLine_Color=KML_Convert_RGB_to_Hex(V_Red, V_Green, V_Blue)
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Line color="+strLine_Color
						
			ControlInfo/W=$strWindow_Name T0_SetStr_File_Name
			String strFile_Name=S_Value
			IF(strlen(strFile_Name)==0)
				Prompt strFile_Name, "File name (no extension)"
				DoPrompt "Enter a file name", strFile_Name
				IF(V_Flag)
					Return -1
				ENDIF
				
				strFile_Name=KML_Cleanup_File_Name(strFile_Name)
				
				SetVariable T0_SetStr_File_Name, value=_STR:strFile_Name
			ENDIF
			
			strFile_Name=KML_Cleanup_File_Name(strFile_Name)
			
			strFile_Name+=SelectString(StringMatch(strFile_Name, "*.kml"), ".kml", "")
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="File name="+strFile_Name
			
			NewPath/O/M="Choose folder to save kml file in:"/Q pKML_Path 
			IF(V_flag!=0)
				Return -1
			ENDIF
			PathInfo pKML_Path
			String strKML_Path=S_Path
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="File path="+strKML_Path
			
			GetFileFolderInfo/P=pKML_Path/Q/Z strFile_Name
			IF(V_Flag==0)
				DoAlert/T="READ CAREFULLY!!!!!" 1, "READ CAREFULLY!!!!\r\rThe file already exists!  Do you want to OVERWRITE the file (YES) or CANCEL (NO)?\r\rREAD CAREFULLY!"
				
				IF(V_Flag==2)
					Return -1
				ENDIF
			ENDIF
			
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Overwrite=1"
			
			Variable vFlag=KML_Line_Make_File(twKML_Inputs)
			
			IF(vFlag!=-1)
				Print "Done creating "+strKML_Path+strFile_Name
			ELSE
				Print "Problems creating the file!"
			ENDIF
			
		BREAK
	ENDSWITCH
	
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Function that actually makes the KML line file
FUNCTION KML_Line_Make_File(WAVE/T/Z twUser_Inputs)

	String strUser_Inputs
	wfprintf strUser_Inputs, "%s;", twUser_Inputs

	String strLatitude_Wave = StringByKey("Lat", strUser_Inputs, "=", ";")		
	String strLongitude_Wave = StringByKey("Lon", strUser_Inputs, "=", ";")		

	WAVE/Z Latitude_Wave=$strLatitude_Wave, Longitude_Wave=$strLongitude_Wave

	IF(!WaveExists(Latitude_Wave) || !WaveExists(Longitude_Wave))
		Print "One or more waves don't exist!"
		Return -1
	ENDIF
	
	Variable vResolution = str2num(StringByKey("DataRes", strUser_Inputs, "=", ";"))
	
	Variable vNum_Data_Pnts = numpnts(Latitude_Wave)
	Variable vNum_KML_Pnts=ceil(vNum_Data_Pnts/vResolution)		//This should be okay
	
	IF((numpnts(Longitude_Wave)!= vNum_Data_Pnts))
		Print "Waves must all have same number of points!"
		Return -1
	ENDIF
	
	Make/O/B/U/FREE/N=(vNum_Data_Pnts) Flag_Wave=numtype(Latitude_Wave[p])==0 && numtype(Longitude_Wave[p])==0

	Make/O/D/FREE/N=(vNum_KML_Pnts) Latitude_Wave_KML=Latitude_Wave[p*vResolution]
	Make/O/D/FREE/N=(vNum_KML_Pnts) Longitude_Wave_KML=Longitude_Wave[p*vResolution]
	
	String strInfo="Experiment name:"+IgorInfo(1)+";Resolution:"+num2str(vResolution)+";Latitude wave:"+strLatitude_Wave+";Longitude wave:"+strLongitude_Wave+";"
	
	String strAltitude_Wave = StringByKey("Alt", strUser_Inputs, "=", ";")
	WAVE/Z Altitude_Wave=$strAltitude_Wave
	
	Variable vAltitude_Flag=WaveExists(Altitude_Wave)
	IF(vAltitude_Flag)
		IF(numpnts(Altitude_Wave)!= vNum_Data_Pnts)
			Print "Waves must all have same number of points!"
			Return -1
		ENDIF
		
		Variable vAltitude_Scale=str2num(StringByKey("AltScale", strUser_Inputs, "=", ";"))
		
		Flag_Wave=Flag_Wave[p] && numtype(Altitude_Wave[p])==0
		Make/O/D/FREE/N=(vNum_KML_Pnts) Altitude_Wave_KML=Altitude_Wave[p*vResolution]*vAltitude_Scale
		
		strInfo+="Altitude wave:"+strAltitude_Wave+";"
	ENDIF
	
	String strUTC_Offset
	
	String strTime_Wave=StringByKey("Time", strUser_Inputs, "=", ";")
	WAVE/Z Time_Wave=$strTime_Wave
		
	Variable vTime_Flag=WaveExists(Time_Wave)
	IF(vTime_Flag)
		IF(numpnts(Time_Wave)!= vNum_Data_Pnts)
			Print "Waves must all have same number of points!"
			Return -1
		ENDIF
		
		Flag_Wave=Flag_Wave[p] && numtype(Time_Wave[p])==0
		Make/O/T/FREE/N=(vNum_KML_Pnts) TimeStamp_GE_Format_KML=Secs2Date(Time_Wave[p*vResolution], -2)+"T"+Secs2Time(Time_Wave[p*vResolution], 3)
		
		strInfo+="Time wave:"+strTime_Wave+";"
		
		Variable vUTC_Offset=str2num(StringByKey("UTC offset", strUser_Inputs, "=", ";"))
		Variable vUTC_Offset_hr=floor(vUTC_Offset)
		Variable vUTC_Offset_Frac=round(60*(vUTC_Offset-vUTC_Offset_hr))
		
		sprintf strUTC_Offset, "%+0.2d", vUTC_Offset_hr
		IF(vUTC_Offset_Frac==0)
			strUTC_Offset+=":00"
		ELSE
			String strUTC_Offset_Frac
			sprintf strUTC_Offset_Frac, "%0.2d", vUTC_Offset_Frac
			strUTC_Offset+=":"+strUTC_Offset_Frac
		ENDIF
		
		IF(cmpstr(strUTC_Offset, "+00:00")==0)
			strUTC_Offset="Z"
		ENDIF
	ELSE
		strUTC_Offset="Z"
	ENDIF
	
	String strWindSpeed_Wave=StringByKey("WindSpeed", strUser_Inputs, "=", ";")
	String strWindDirection_Wave=StringByKey("WindDir", strUser_Inputs, "=", ";")
	Variable vWind_Resolution=str2num(StringByKey("WindRes", strUser_Inputs, "=", ";"))
	
	WAVE/Z Wind_Speed=$strWindSpeed_Wave
	WAVE/Z Wind_Direction=$strWindDirection_Wave
	
	Variable vWind_Flag=WaveExists(Wind_Speed) && WaveExists(Wind_Direction) && vWind_Resolution>0
	IF(vWind_Flag)
		IF(numpnts(Wind_Speed)!= vNum_Data_Pnts || numpnts(Wind_Direction)!=vNum_Data_Pnts)
			Print "Waves must all have same number of points!"
			Return -1
		ENDIF
		
		Flag_Wave=Flag_Wave[p] && numtype(Wind_Speed[p])==0 && numtype(Wind_Direction[p])==0
		Make/O/D/FREE/N=(vNum_KML_Pnts) Wind_Speed_KML=Wind_Speed[p*vResolution]
		Make/O/D/FREE/N=(vNum_KML_Pnts) Wind_Direction_KML=Wind_Direction[p*vResolution]
	ENDIF
	
	
	Variable vLine_Width=str2num(StringByKey("Line width", strUser_Inputs, "=", ";"))
	
	String strLine_Color=StringByKey("Line color", strUser_Inputs, "=", ";")
	IF(ItemsInList(strLine_Color, ",")>1)
		strLine_Color=KML_Convert_RGB_to_Hex(str2num(StringFromList(0,strLine_Color, ",")), str2num(StringFromList(1,strLine_Color, ",")), str2num(StringFromList(2,strLine_Color, ",")))
	ENDIF
	
	String strFile_Name=StringByKey("File name", strUser_Inputs, "=", ";")
	strFile_Name+=SelectString(StringMatch(strFile_Name, "*.kml"), ".kml", "")
	
	strFile_Name=KML_Cleanup_File_Name(strFile_Name)
		
	Make/O/D/FREE/N=(vNum_KML_Pnts) Flag_Wave_KML=Flag_Wave[p*vResolution]	
	
	Extract/O/FREE Latitude_Wave_KML, Latitude_Wave_KML, Flag_Wave_KML
	Extract/O/FREE Longitude_Wave_KML, Longitude_Wave_KML, Flag_Wave_KML
	
	IF(vAltitude_Flag)
		Extract/O/FREE Altitude_Wave_KML, Altitude_Wave_KML, Flag_Wave_KML
	ELSE
		Make/O/D/FREE/N=(numpnts(Latitude_Wave_KML)) Altitude_Wave_KML=0
	ENDIF
	
	IF(vTime_Flag)
		Extract/O/T/FREE TimeStamp_GE_Format_KML, TimeStamp_GE_Format_KML, Flag_Wave_KML
		TimeStamp_GE_Format_KML=TimeStamp_GE_Format_KML+strUTC_Offset
		
		String strTime_Start=TimeStamp_GE_Format_KML[0]
		String strTime_Stop=TimeStamp_GE_Format_KML[numpnts(TimeStamp_GE_Format_KML)-1]
	ENDIF
	
	IF(vWind_Flag)
		Extract/O/FREE Wind_Speed_KML, Wind_Speed_KML, Flag_Wave_KML
		Extract/O/FREE Wind_Direction_KML, Wind_Direction_KML, Flag_Wave_KML
	ENDIF
	
	vNum_KML_Pnts=numpnts(Latitude_Wave_KML)
				
	WaveStats/Q Latitude_Wave_KML
	Variable vLat_Avg = V_avg
	Variable vLat_Min=V_Min
	Variable vLat_Max=V_Max
	
	WaveStats/Q Longitude_Wave_KML
	Variable vLon_Avg = V_avg
	Variable vLon_Min=V_Min
	Variable vLon_Max=V_Max
	
	Variable vHeight_Scale_Factor=2.5		//This seems to work
	Variable vHeight=vHeight_Scale_Factor*max(KML_Get_Bounds(vLat_Max, vLon_Avg, vLat_Min, vLon_Avg), KML_Get_Bounds(vLat_Avg, vLon_Min, vLat_Avg, vLon_Max))
	
	String strKML_Path=StringByKey("File path", strUser_Inputs, "=", ";")
	
	Variable vOverwrite=str2num(StringByKey("Overwrite", strUser_Inputs, "=", ";"))
	
	NewPath/O/Q/Z pKML_Path, strKML_Path
	GetFileFolderInfo/P=pKML_Path/Q/Z strFile_Name
	IF(V_Flag==0 && !vOverwrite)
		Print "File already exists and overwrite flag not set!"
		Return -1
	ENDIF
	
	Variable refnum
	Open/P=pKML_Path/C="IGR0" refNum as strFile_Name
	
	//Ken made a long string, with one line of Igor code for each line of KML code, and then printed that string to the file.  I'm not sure why you would do that versus just using fprintf for each line.
	fprintf refNum, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\r"
	fprintf refNum, "<kml xmlns=\"http://www.opengis.net/kml/2.2\" xmlns:gx=\"http://www.google.com/kml/ext/2.2\" xmlns:kml=\"http://www.opengis.net/kml/2.2\" xmlns:atom=\"http://www.w3.org/2005/Atom\">\r"
	fprintf refNum, "<Document>\r"
	fprintf refNum, "	<name>"+strFile_Name+"</name>\r"
	fprintf refnum, "	<ExtendedData>"+strInfo+"</ExtendedData>\r"
	
	IF(!vTime_Flag)
		fprintf refNum, "	<StyleMap id=\"m_ylw-pushpin\">\r"
		fprintf refNum, "		<Pair>\r"
		fprintf refNum, "			<key>normal</key>\r"
		fprintf refNum, "			<styleUrl>#s_ylw-pushpin</styleUrl>\r"
		fprintf refNum, "		</Pair>\r"
		fprintf refNum, "		<Pair>\r"
		fprintf refNum, "			<key>highlight</key>\r"
		fprintf refNum, "			<styleUrl>#s_ylw-pushpin_hl</styleUrl>\r"
		fprintf refNum, "		</Pair>\r"
		fprintf refNum, "	</StyleMap>\r"
		fprintf refNum, "	<Style id=\"s_ylw-pushpin\">\r"
		fprintf refNum, "		<IconStyle>\r"
		fprintf refNum, "			<scale>1.1</scale>\r"
		fprintf refNum, "			<Icon>\r"
		fprintf refNum, "				<href>http://maps.google.com/mapfiles/kml/pushpin/ylw-pushpin.png</href>\r"
		fprintf refNum, "			</Icon>\r"
		fprintf refNum, "			<hotSpot x=\"20\" y=\"2\" xunits=\"pixels\" yunits=\"pixels\"/>\r"
		fprintf refNum, "		</IconStyle>\r"
		fprintf refNum, "		<LineStyle>\r"
		fprintf refNum, "			<color>"+strLine_Color+"</color>\r"
		fprintf refNum, "			<width>"+num2str(vLine_Width)+"</width>\r"
		fprintf refNum, "		</LineStyle>\r"
		fprintf refNum, "	</Style>\r"
		fprintf refNum, "	<Style id=\"s_ylw-pushpin_hl\">\r"
		fprintf refNum, "		<IconStyle>\r"
		fprintf refNum, "			<scale>1.3</scale>\r"
		fprintf refNum, "			<Icon>\r"
		fprintf refNum, "				<href>http://maps.google.com/mapfiles/kml/pushpin/ylw-pushpin.png</href>\r"
		fprintf refNum, "			</Icon>\r"
		fprintf refNum, "			<hotSpot x=\"20\" y=\"2\" xunits=\"pixels\" yunits=\"pixels\"/>\r"
		fprintf refNum, "		</IconStyle>\r"
		fprintf refNum, "		<LineStyle>\r"
		fprintf refNum, "			<color>"+strLine_Color+"</color>\r"
		fprintf refNum, "			<width>"+num2str(vLine_Width)+"</width>\r"
		fprintf refNum, "		</LineStyle>\r"
		fprintf refNum, "	</Style>\r"
		fprintf refNum, "	<Folder>\r"
		fprintf refNum, "		<Placemark>\r"
		fprintf refNum, "			<name>1</name>\r"
		fprintf refNum, "			<styleUrl>#m_ylw-pushpin</styleUrl>\r"
		fprintf refNum, "			<LineString>\r"
		fprintf refNum, "				<extrude>1</extrude>\r"
		fprintf refNum, "				<tessellate>1</tessellate>\r"
		IF(vAltitude_Flag)
			fprintf refNum, "				<altitudeMode>absolute</altitudeMode>\r"
		ELSE
			fprintf refNum, "				<altitudeMode>relativeToGround</altitudeMode>\r"
		ENDIF
		fprintf refNum, "				<coordinates>\r"
					
		wfprintf refNum, "					%0.8f,%0.8f,%0.8f\r", Longitude_Wave_KML, Latitude_Wave_KML, Altitude_Wave_KML
		
		fprintf refNum, "				</coordinates>\r"
		fprintf refNum, "			</LineString>\r"
		fprintf refNum, "		</Placemark>\r"
		
	ELSE
		String strStyle_Name="Line_Time"
	
		fprintf refnum, "	<Style id=\""+strStyle_Name+"_n\">\r"
		fprintf refnum, "		<IconStyle>\r"
		fprintf refnum, "			<Icon>\r"
		fprintf refnum, "				<href>http://earth.google.com/images/kml-icons/track-directional/track-0.png</href>\r"
		fprintf refnum, "			</Icon>\r"
		fprintf refnum, "		</IconStyle>\r"
		fprintf refnum, "		<LineStyle>\r"
		fprintf refnum, "			<color>"+strLine_Color+"</color>\r"
		fprintf refnum, "			<width>"+num2str(vLine_Width)+"</width>\r"
		fprintf refnum, "		</LineStyle>\r"
		fprintf refnum, "	</Style>\r"
		
		fprintf refnum, "	<Style id=\""+strStyle_Name+"_h\">\r"
		fprintf refnum, "		<IconStyle>\r"
		fprintf refnum, "			<scale>1.2</scale>\r"
		fprintf refnum, "			<Icon>\r"
		fprintf refnum, "				<href>http://earth.google.com/images/kml-icons/track-directional/track-0.png</href>\r"
		fprintf refnum, "			</Icon>\r"
		fprintf refnum, "		</IconStyle>\r"
		fprintf refnum, "		<LineStyle>\r"
		fprintf refnum, "			<color>"+strLine_Color+"</color>\r"
		fprintf refnum, "			<width>"+num2str(vLine_Width)+"</width>\r"
		fprintf refnum, "		</LineStyle>\r"
		fprintf refnum, "	</Style>\r"
		fprintf refnum, "	<StyleMap id=\""+strStyle_Name+"\">\r"
		fprintf refnum, "		<Pair>\r"
		fprintf refnum, "			<key>normal</key>\r"
		fprintf refnum, "			<styleUrl>#"+strStyle_Name+"_n</styleUrl>\r"
		fprintf refnum, "		</Pair>\r"
		fprintf refnum, "		<Pair>\r"
		fprintf refnum, "			<key>highlight</key>\r"
		fprintf refnum, "			<styleUrl>#"+strStyle_Name+"_h</styleUrl>\r"
		fprintf refnum, "		</Pair>\r"
		fprintf refnum, "	</StyleMap>\r"
		fprintf refnum, "	<Style id=\""+strStyle_Name+"_line\">\r"
		fprintf refnum, "		<LineStyle>\r"
		fprintf refnum, "			<color>"+strLine_Color+"</color>\r"
		fprintf refnum, "			<width>"+num2str(vLine_Width)+"</width>\r"
		fprintf refnum, "		</LineStyle>\r"
		fprintf refnum, "	</Style>\r"
		
		fprintf refNum, "	<Folder>\r"	
		fprintf refnum, "		<Placemark>\r"
		fprintf refnum, "			<styleUrl>#"+strStyle_Name+"</styleUrl>\r"
		fprintf refnum, "			<gx:Track>\r"
		
		IF(vAltitude_Flag)
			fprintf refNum, "				<gx:altitudeMode>absolute</gx:altitudeMode>\r"
		ELSE
			fprintf refNum, "				<gx:altitudeMode>relativeToGround</gx:altitudeMode>\r"
		ENDIF
		
		wfprintf refNum, "				<when>%s</when>\r", TimeStamp_GE_Format_KML
		wfprintf refNum, "				<gx:coord>%0.8f %0.8f %0.8f</gx:coord>\r", Longitude_Wave_KML, Latitude_Wave_KML, Altitude_Wave_KML
		fprintf refnum, "			</gx:Track>\r"
		fprintf refnum, "		</Placemark>\r"
	ENDIF
	
	IF(vWind_Flag)
		Make/O/T/FREE/N=0 twBarb_List
		String strBarb_Icon_Folder_Path=strKML_Path+SelectString(StringMatch(strKML_Path, "*:"), ":", "")+RemoveEnding(strFile_Name, ".kml")+"_WindBarbs"
		String strBarb_Icon_Folder_Name=RemoveEnding(strFile_Name, ".kml")+"_WindBarbs"
		String strBarb_Icon_Base_Name=RemoveEnding(strFile_Name, ".kml")
		
		NewPath/O/C/Q/Z pBarb_Icon_Path, strBarb_Icon_Folder_Path
		
		Variable vBarb_Length=str2num(StringByKey("Barb length", strUser_Inputs, "=", ";"))
		Variable vBarb_Thick=str2num(StringByKey("Barb thick", strUser_Inputs, "=", ";"))
		Variable vBarb_Scale=str2num(StringByKey("Barb scale", strUser_Inputs, "=", ";"))
		
		Variable vWind_Color_Mode=str2num(StringByKey("Wind color mode", strUser_Inputs, "=", ";"))
		
		IF(vWind_Color_Mode)
			String strBarb_Color=StringByKey("Wind color", strUser_Inputs, "=", ";")
			IF(ItemsInList(strBarb_Color, ",")>1)
				strBarb_Color=KML_Convert_RGB_to_Hex(str2num(StringFromList(0,strBarb_Color, ",")), str2num(StringFromList(1,strBarb_Color, ",")), str2num(StringFromList(2,strBarb_Color, ",")))
			ENDIF
		ELSE
			strBarb_Color=strLine_Color
		ENDIF
			
		Variable vNum_Barbs=floor(vNum_KML_Pnts/vWind_Resolution)
			
		Variable iBarbDex
		FOR(iBarbDex=0;iBarbDex<vNum_Barbs;iBarbDex+=1)
			String strBarb_Icon_Name=KML_Wind_Make_Barb_Icon(twBarb_List, Wind_Speed_KML[iBarbDex*vWind_Resolution], strBarb_Icon_Folder_Path, strBarb_Icon_Base_Name, vBarb_Length, vBarb_Thick)
			fprintf refnum, "		<Placemark>\r"
			
			IF(vTime_Flag)
				fprintf refNum, "			<TimeStamp><when>"+TimeStamp_GE_Format_KML[iBarbDex*vWind_Resolution]+"</when></TimeStamp>\r"
			ENDIF
			
			fprintf refnum, "			<Style>\r"
			fprintf refnum, "				<IconStyle>\r"
			fprintf refnum, "					<Icon><href>"+strBarb_Icon_Folder_Name+"/"+strBarb_Icon_Name+".png</href></Icon>\r"
			fprintf refnum, "					<scale>"+num2str(vBarb_Scale)+"</scale>\r"
			fprintf refnum, "					<heading>%0.8f</heading>\r",Wind_Direction_KML[iBarbDex*vWind_Resolution]
			fprintf refnum, "					<color>"+strBarb_Color+"</color>\r"
			fprintf refnum, "				</IconStyle>\r"
			fprintf refnum, "			</Style>\r"
			fprintf refnum, "			<Point>\r"
			fprintf refnum, "				<coordinates>%0.8f,%0.8f,%0.8f</coordinates>\r", Longitude_Wave_KML[iBarbDex*vWind_Resolution], Latitude_Wave_KML[iBarbDex*vWind_Resolution], Altitude_Wave_KML[iBarbDex*vWind_Resolution]
			IF(vAltitude_Flag)
				fprintf refNum, "				<altitudeMode>absolute</altitudeMode>\r"
			ELSE
				fprintf refNum, "				<altitudeMode>relativeToGround</altitudeMode>\r"
			ENDIF
			fprintf refnum, "			</Point>\r"
			fprintf refnum, "		</Placemark>\r"
		ENDFOR
		
		KillPath/Z pBarb_Icon_Path
	ENDIF
	
	fprintf refNum, "	</Folder>\r"
	
	fprintf refNum, "	<LookAt>\r"
	fprintf refNum, "		<longitude>%.10f</longitude>\r", vLon_Avg
	fprintf refNum, "		<latitude>%.10f</latitude>\r", vLat_Avg
	fprintf refNum, "		<range>%d</range>\r", vHeight
	fprintf refNum, "		<tilt>0</tilt>\r"
	fprintf refNum, "		<heading>0</heading>\r"
	
	IF(vTime_Flag)
		fprintf refNum, "		<gx:TimeSpan>\r"
		fprintf refNum, "		<begin>"+strTime_Start+"</begin>\r"
		fprintf refNum, "		<end>"+strTime_Stop+"</end>\r"
		fprintf refNum, "		</gx:TimeSpan>\r"
	ENDIF
	
	fprintf refNum, "	</LookAt>\r"
	
	fprintf refnum, "</Document>\r</kml>\r"
	Close refNum
	
	KillPath/Z pKML_Path
	
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Enables or disables various controls based on checkbox selections
FUNCTION KML_Marker_Check_Fz_Size_Color(cba) : CheckBoxControl
	STRUCT WMCheckBoxAction &cba
	
	SWITCH(cba.eventcode)
		CASE 2:
	
			String strControlName=cba.ctrlName
			String strWindow_Name=cba.win
	
			STRSWITCH(cba.ctrlName)
				CASE "T1_Check_SizeFz":
					SetVariable T1_SetVar_Marker_Size_Min, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
					SetVariable T1_SetVar_Marker_Size_Max, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
					CheckBox T1_Check_Log_Size, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
					SetVariable T1_SetVar_Marker_Size_Default, win=$strWindow_Name, disable=SelectNumber(cba.checked, 0, 1)
				BREAK
				
				CASE "T1_Check_ColorFz":					
					PopupMenu T1_Pop_Color_Table, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
					CheckBox T1_Check_Custom_Color_Table, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
					CheckBox T1_Check_Reverse_Colors, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
					Checkbox T1_Check_Log_Colors, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
										
					GroupBox T1_GroupBox_Color_Scale,win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
					CheckBox T1_Check_Color_Scale, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
					
					ControlInfo/W=$strWindow_Name T1_Check_Color_Scale
					Button T1_Button_Make_Color_Scale_Plot, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, SelectNumber(V_Value, 2,0))
					PopupMenu T1_Pop_Color_Scale_Label_Side, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, SelectNumber(V_Value, 2,0))
					CheckBox T1_Check_Transparent, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, SelectNumber(V_Value, 2,0))
					PopupMenu T1_Pop_CS_Graph, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, SelectNumber(V_Value, 2,0))
					
					ControlInfo/W=$strWindow_Name T1_Check_Custom_Color_Table
					PopUpMenu T1_Pop_Custom_Color_Table_Wave, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, SelectNumber(V_Value, 1,0))
					PopUpMenu T1_Pop_Color_Table, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, SelectNumber(V_Value, 0,1))
					
					PopupMenu T1_Pop_Marker_Color, win=$strWindow_Name, disable=SelectNumber(cba.checked, 0, 1)
				BREAK
				
				CASE "T1_Check_Color_Scale":
					Button T1_Button_Make_Color_Scale_Plot, win=$strWindow_Name, disable=SelectNumber(cba.checked, 2, 0)
					CheckBox T1_Check_Transparent, win=$strWindow_Name, disable=SelectNumber(cba.checked, 2, 0)
					PopupMenu T1_Pop_CS_Graph, win=$strWindow_Name, disable=SelectNumber(cba.checked, 2, 0)
					PopupMenu T1_Pop_Color_Scale_Label_Side, win=$strWindow_Name, disable=SelectNumber(cba.checked, 2, 0)
				BREAK
				
				CASE "T1_Check_Custom_Color_Table":
					PopUpMenu T1_Pop_Custom_Color_Table_Wave, win=$strWindow_Name, disable=SelectNumber(cba.checked, 1, 0)
					PopUpMenu T1_Pop_Color_Table, win=$strWindow_Name, disable=SelectNumber(cba.checked, 0, 1)
				BREAK
			ENDSWITCH
		BREAK
	ENDSWITCH

END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////



FUNCTION KML_Marker_Pop_Fz(pa) : PopupMenuControl
	STRUCT WMPopupAction &pa

	SWITCH( pa.eventCode )
		CASE 2: // mouse up
			String strWindow_Name=pa.win
			String strFz_Wave = pa.popStr
			
			WAVE/Z Fz_Wave=$strFz_Wave
			IF(WaveExists(Fz_Wave))
				WaveStats/Q/M=1 Fz_Wave
				
				IF(V_npnts==0)
					 Print Secs2Date(DateTime, -2)+" "+Secs2Time(DateTime, 3)+" F(z) wave has no valid points!"
					
					SetVariable T1_SetVar_Fz_Min, win=$strWindow_Name, value=_NUM:NaN		//Wavemin and Wavemax choke on infs
					SetVariable T1_SetVar_Fz_Max, win=$strWindow_Name, value=_NUM:NaN
					
					Return -1
				ENDIF
				
				SetVariable T1_SetVar_Fz_Min, win=$strWindow_Name, value=_NUM:V_Min		//Wavemin and Wavemax choke on infs
				SetVariable T1_SetVar_Fz_Max, win=$strWindow_Name, value=_NUM:V_Max
				
			ELSE
				SetVariable T1_SetVar_Fz_Min, win=$strWindow_Name, value=_NUM:NaN		//Wavemin and Wavemax choke on infs
				SetVariable T1_SetVar_Fz_Max, win=$strWindow_Name, value=_NUM:NaN
				
				IF(!StringMatch(strFz_Wave, "None"))
					Print Secs2Date(DateTime, -2)+" "+Secs2Time(DateTime, 3)+" F(z) wave doesn't exist!"
				ENDIF
			ENDIF
			
		BREAK
	ENDSWITCH
	
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Gets the min and max of the F(z) wave
FUNCTION KML_Marker_Button_Fz_Min_Max(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	SWITCH(ba.eventcode)
		CASE 2:
			String strWindow_Name=ba.win
			
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Fz	
			WAVE/Z Fz_Wave=$S_Value
			IF(WaveExists(Fz_Wave))
				WaveStats/Q/M=1 Fz_Wave
				
				IF(V_npnts==0)
					DoAlert/T="ERROR ID-10T" 0, "F(z) wave has no valid points!"
					Return -1
				ENDIF
				
				SetVariable T1_SetVar_Fz_Min, win=$strWindow_Name, value=_NUM:V_Min		//Wavemin and Wavemax choke on infs
				SetVariable T1_SetVar_Fz_Max, win=$strWindow_Name, value=_NUM:V_Max
				
			ELSE
				DoAlert/T="ERROR ID-10T" 0, "F(z) wave doesn't exist!"
			ENDIF
		BREAK
	ENDSWITCH
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Makes a plot for the color scale in the KML
FUNCTION KML_Marker_Button_Make_CS_Plot(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	SWITCH(ba.eventcode)
		CASE 2:
			String strWindow_Name=ba.win
			
			ControlInfo/W=$strWindow_Name T1_Check_Custom_Color_Table
			Variable vCustom_Color_Table=V_Value
			IF(vCustom_Color_Table)
				ControlInfo/W=$strWindow_Name T1_Pop_Custom_Color_Table_Wave
				WAVE/Z Color_Table_Wave=$S_Value
				
				IF(!WaveExists(Color_Table_Wave))
					DoAlert/T="ERROR ID-10T" 0, "Invalid color table wave!"
					Return -1
				ENDIF
				
				Variable vNum_Colors=DimSize(Color_Table_Wave, 0)
				
			ELSE		//Get the color table wave and then convert it into a free wave
				ControlInfo/W=$strWindow_Name T1_Pop_Color_Table
				String strColor_Table=S_Value
				
				String strCurrent_Data_Folder=GetDataFolder(1)
				NewDataFolder/O/S root:Stupidly_Named_Temp_Folder
				ColorTab2Wave $strColor_Table
				WAVE/Z M_Colors
				vNum_Colors=DimSize(M_Colors, 0)
				
				SetDataFolder $strCurrent_Data_Folder
				KillDataFolder/Z root:Stupidly_Named_Temp_Folder
			ENDIF
			
			ControlInfo/W=$strWindow_Name T1_Check_Reverse_Colors
			Variable vReverse_Colors = V_Value
			
			ControlInfo/W=$strWindow_Name T1_Check_Log_Colors
			Variable vLog_Colors=V_Value
			
			ControlInfo/W=$strWindow_Name T1_SetVar_Fz_Min
			Variable vFz_Min=V_Value
			
			ControlInfo/W=$strWindow_Name T1_SetVar_Fz_Max
			Variable vFz_Max=V_Value
			
			String strFz_Min, strFz_Max				//Don't want to compare floats.
			sprintf strFz_Min, "%0.6f", vFz_Min
			sprintf strFz_Max, "%0.6f", vFz_Max
			
			IF(numtype(vFz_Min)!=0 || numtype(vFz_Max)!=0)
				DoAlert/T="ERROR ID-10T" 0, "Fz value limits must be real numbers!"
				Return -1
			ENDIF
			
			IF(cmpstr(strFz_Min, strFz_Max)==0)
				DoAlert/T="ERROR ID-10T" 0, "Min and max fz values must be different"
				Return -1
			ENDIF
			
				IF(vLog_Colors)
					IF(vFz_Max<=0 || vFz_Min<=0)
						Print "One or both of the Fz limits is less than or equal to zero!  Using linear scaling of the colors!"
						vLog_Colors=0
					ENDIF
				ENDIF
			
			ControlInfo/W=$strWindow_Name T1_Pop_Color_Scale_Label_Side
			String strCS_Label_Side=S_Value
			
			Make/O/D/N=(vNum_Colors) KML_Color_Table_Wave_Y=1, KML_Color_Table_Wave_X
			
			IF(vLog_Colors)
				SetScale/I x log(vFz_Min),log(vFz_Max),"", KML_Color_Table_Wave_X
				KML_Color_Table_Wave_X=10^x
			ELSE
				SetScale/I x vFz_Min,vFz_Max,"", KML_Color_Table_Wave_X
				KML_Color_Table_Wave_X=x
			ENDIF
								
			String strGraph_Name
			
			STRSWITCH(strCS_Label_Side)
				CASE "Left":
					strGraph_Name=UniqueName("CS_Left_Label_Vertical_", 6, 0)
				
					Display/N=$strGraph_Name /W=(398,200,448,310)/K=1 /VERT KML_Color_Table_Wave_Y vs KML_Color_Table_Wave_X as strGraph_Name
					ModifyGraph margin(left)=23,margin(bottom)=3,margin(top)=1,margin(right)=1,width=8.64,height=108
					ModifyGraph mode=7
					ModifyGraph hbFill=2
					IF(vCustom_Color_Table)
						ModifyGraph zColor(KML_Color_Table_Wave_Y)={KML_Color_Table_Wave_X,*,*,ctableRGB, vReverse_Colors, Color_Table_Wave}
					ELSE
						ModifyGraph zColor(KML_Color_Table_Wave_Y)={KML_Color_Table_Wave_X,*,*,$strColor_Table, vReverse_Colors}
					ENDIF
					ModifyGraph logZColor=vLog_Colors
					ModifyGraph tick(bottom)=3
					ModifyGraph mirror=2
					ModifyGraph noLabel(bottom)=2
					ModifyGraph fSize(left)=6
					ModifyGraph log(left)=vLog_Colors
					ModifyGraph standoff=0
					ModifyGraph axisOnTop=1
					ModifyGraph btLen=4
					SetAxis left vFz_Min,vFz_Max
					SetAxis bottom 0,1
				BREAK
				
				CASE "Right":
					strGraph_Name=UniqueName("CS_Right_Label_Vertical_", 6, 0)
				
					Display/N=$strGraph_Name /W=(325,201,375,311)/K=1 /VERT/R KML_Color_Table_Wave_Y vs KML_Color_Table_Wave_X as strGraph_Name
					ModifyGraph margin(left)=1,margin(bottom)=3,margin(top)=1,margin(right)=23,width=8.64
					ModifyGraph height=108
					ModifyGraph mode=7
					ModifyGraph hbFill=2
					IF(vCustom_Color_Table)
						ModifyGraph zColor(KML_Color_Table_Wave_Y)={KML_Color_Table_Wave_X,*,*,ctableRGB, vReverse_Colors, Color_Table_Wave}
					ELSE
						ModifyGraph zColor(KML_Color_Table_Wave_Y)={KML_Color_Table_Wave_X,*,*,$strColor_Table, vReverse_Colors}
					ENDIF
					ModifyGraph logZColor=vLog_Colors
					ModifyGraph tick(bottom)=3
					ModifyGraph mirror=2
					ModifyGraph noLabel(bottom)=2
					ModifyGraph fSize(right)=6
					ModifyGraph log(right)=vLog_Colors
					ModifyGraph standoff=0
					ModifyGraph axisOnTop=1
					ModifyGraph btLen=4
					SetAxis right vFz_Min,vFz_Max
					SetAxis bottom 0,1
				BREAK
				
				CASE "Top":
					strGraph_Name=UniqueName("CS_Top_Label_Horizonal_", 6, 0)
					Display/N=$strGraph_Name/W=(254,292,370,316)/K=1 /T KML_Color_Table_Wave_Y vs KML_Color_Table_Wave_X as strGraph_Name
					ModifyGraph margin(left)=1,margin(bottom)=1,margin(top)=14,margin(right)=7,width=108
					ModifyGraph height=8.64
					ModifyGraph mode=7
					ModifyGraph hbFill=2
					IF(vCustom_Color_Table)
						ModifyGraph zColor(KML_Color_Table_Wave_Y)={KML_Color_Table_Wave_X,*,*,ctableRGB, vReverse_Colors, Color_Table_Wave}
					ELSE
						ModifyGraph zColor(KML_Color_Table_Wave_Y)={KML_Color_Table_Wave_X,*,*,$strColor_Table, vReverse_Colors}
					ENDIF
					ModifyGraph logZColor=vLog_Colors
					ModifyGraph tick(left)=3
					ModifyGraph mirror(left)=1,mirror(top)=2
					ModifyGraph noLabel(left)=2
					ModifyGraph fSize(top)=6
					ModifyGraph log(top)=vLog_Colors
					ModifyGraph standoff=0
					ModifyGraph axisOnTop=1
					ModifyGraph btLen(top)=4
					SetAxis left 0,1
				BREAK
				
				CASE "Bottom":
					strGraph_Name=UniqueName("CS_Bottom_Label_Horizonal_", 6, 0)
					
					Display/N=$strGraph_Name/W=(253,251,369,375)/K=1  KML_Color_Table_Wave_Y vs KML_Color_Table_Wave_X as strGraph_Name
					ModifyGraph margin(left)=1,margin(bottom)=14,margin(top)=1,margin(right)=7,width=108
					ModifyGraph height=8.64
					ModifyGraph mode=7
					ModifyGraph hbFill=2
					IF(vCustom_Color_Table)
						ModifyGraph zColor(KML_Color_Table_Wave_Y)={KML_Color_Table_Wave_X,*,*,ctableRGB, vReverse_Colors, Color_Table_Wave}
					ELSE
						ModifyGraph zColor(KML_Color_Table_Wave_Y)={KML_Color_Table_Wave_X,*,*,$strColor_Table, vReverse_Colors}
					ENDIF
					ModifyGraph logZColor=vLog_Colors
					ModifyGraph tick(left)=3
					ModifyGraph mirror(left)=1,mirror(bottom)=2
					ModifyGraph noLabel(left)=2
					ModifyGraph fSize(bottom)=6
					ModifyGraph log(bottom)=vLog_Colors
					ModifyGraph standoff=0
					ModifyGraph axisOnTop=1
					ModifyGraph btLen(bottom)=4
					SetAxis left 0,1
				BREAK
			ENDSWITCH
			
			String strGraph_Name_List=WinList("*", ";", "WIN:1")
			Variable vNum_Graphs=ItemsInList(strGraph_Name_List)
			String strGraph_Name_List_Sorted=SortList(strGraph_Name_List)
			Variable vGraph_Index=WhichListItem(strGraph_Name, strGraph_Name_List_Sorted)
			
			PopupMenu T1_Pop_CS_Graph,win=$strWindow_Name, value= #"\"none;\"+sortlist(WinList(\"*\", \";\", \"WIN:1\"))", mode=vGraph_Index+2,popvalue=strGraph_Name
			ControlUpdate/W=$strWindow_Name T1_Pop_CS_Graph
		BREAK
	ENDSWITCH
		
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Makes the KML file with markers for the track
FUNCTION KML_Marker_Button_Make_File(ba) : ButtonControl
	STRUCT WMButtonAction &ba
	
	SWITCH(ba.eventcode)
		CASE 2:
			Close/A
			
			String strWindow_Name=ba.win
			
			Make/O/T/FREE/N=0 twKML_Inputs
			
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Latitude
			String strLatitude_Wave = S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Lat="+strLatitude_Wave
			
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Longitude
			String strLongitude_Wave = S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Lon="+strLongitude_Wave
		
			WAVE/Z Latitude_Wave=$strLatitude_Wave, Longitude_Wave=$strLongitude_Wave
			
			IF(!WaveExists(Latitude_Wave) || !WaveExists(Longitude_Wave))
				DoAlert/T="ERROR ID-10T", 0, "One or more waves don't exist!"
				Return -1
			ENDIF
			
			Variable vNum_Data_Pnts = numpnts(Latitude_Wave)
			
			ControlInfo/W=$strWindow_Name P0_SetVar_DataRes
			Variable vResolution = V_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="DataRes="+num2istr(vResolution)
			
			IF((numpnts(Longitude_Wave)!= vNum_Data_Pnts))
				DoAlert/T="ERROR ID-10T", 0, "Waves must all have same number of points!"
				Return -1
			ENDIF
			
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Altitude
			String strAltitude_Wave = S_Value
			Variable vAltitude_Flag=cmpstr(strAltitude_Wave, "none")!=0
			
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Alt="+strAltitude_Wave
						
			IF(vAltitude_Flag)
				WAVE/Z Altitude_Wave=$strAltitude_Wave
				
				IF(numpnts(Altitude_Wave)!= vNum_Data_Pnts)
					DoAlert/T="ERROR ID-10T", 0, "Waves must all have same number of points!"
					Return -1
				ENDIF
				
				ControlInfo/W=$strWindow_Name P0_SetVar_Alt_Scale
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="AltScale="+num2str(V_Value)
			ENDIF
			
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Time
			String strTime_Wave=S_Value
			
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Time="+strTime_Wave
			
			Variable vTime_Flag=cmpstr(strTime_Wave, "none")!=0
			IF(vTime_Flag)
				WAVE/Z Time_Wave=$strTime_Wave
				
				IF(numpnts(Time_Wave)!= vNum_Data_Pnts)
					DoAlert/T="ERROR ID-10T", 0, "Waves must all have same number of points!"
					Return -1
				ENDIF
				
				ControlInfo/W=$strWindow_Name P0_SetVar_UTC_Offset
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="UTC offset="+num2str(V_Value)
			ENDIF
			
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_Fz
			String strFz_Wave = S_Value
			
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Fz="+strFz_Wave
			
			Variable vFz_Flag=cmpstr(strFz_Wave, "none")!=0
			
			IF(vFz_Flag)
				WAVE/Z Fz_Wave=$strFz_Wave
				
				IF(numpnts(Fz_Wave)!= vNum_Data_Pnts)
					DoAlert/T="ERROR ID-10T", 0, "Waves must all have same number of points!"
					Return -1
				ENDIF
				
				ControlInfo/W=$strWindow_Name T1_SetVar_Fz_Min
				Variable vFz_Min = V_Value
				String strFz_Min
				sprintf strFz_Min, "%.15g", vFz_Min
				
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Fz min="+strFz_Min
			
				ControlInfo/W=$strWindow_Name T1_SetVar_Fz_Max
				Variable vFz_Max = V_Value
				String strFz_Max
				sprintf strFz_Max, "%.15g", vFz_Max
				
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Fz max="+strFz_Max
				
				IF(numtype(vFz_Min)!=0 || numtype(vFz_Max)!=0)
					DoAlert/T="ERROR ID-10T" 0, "Fz limits must be real numbers!"
					Return -1
				ENDIF
				
				IF(cmpstr(strFz_Min, strFz_Max)==0)
					DoAlert/T="ERROR ID-10T" 0, "Min and max Fz values must be different"
					Return -1
				ENDIF
			ENDIF
			
			ControlInfo/W=$strWindow_Name T1_Pop_Symbol
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Symbol="+S_Value
			
			//	if color-coding is checked, get the color table into a wave for use with this data
			Variable vColor_Scale=0				
			ControlInfo/W=$strWindow_Name T1_Check_ColorFz
			Variable vColor_Flag = V_Value
			
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Color by Fz="+num2istr(vColor_Flag)
			
			IF(vColor_Flag)
				ControlInfo/W=$strWindow_Name T1_Check_Custom_Color_Table
				
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Color table type="+num2istr(V_Value)
				
				IF(V_Value)
					ControlInfo/W=$strWindow_Name T1_Pop_Custom_Color_Table_Wave
					WAVE/Z Color_Table_Wave=$S_Value
					
					IF(!WaveExists(Color_Table_Wave))
						DoAlert/T="ERROR ID-10T" 0, "Invalid color table wave!"
						Return -1
					ENDIF
					
					Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
					twKML_Inputs[numpnts(twKML_Inputs)-1]="Color table="+GetWavesDataFolder(Color_Table_Wave, 2)
										
				ELSE
					ControlInfo/W=$strWindow_Name T1_Pop_Color_Table
					String strColor_Table=S_Value
					
					Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
					twKML_Inputs[numpnts(twKML_Inputs)-1]="Color table="+strColor_Table
				ENDIF
				
				ControlInfo/W=$strWindow_Name T1_Check_Reverse_Colors
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Reverse colors="+num2istr(V_Value)
								
				ControlInfo/W=$strWindow_Name T1_Check_Log_Colors
				Variable vLog_Colors=V_Value
				
				IF(vLog_Colors)
					IF(vFz_Max<=0 || vFz_Min<=0)
						Print "One or both of the Fz limits is less than or equal to zero!  Using linear scaling of the colors!"
						vLog_Colors=0
					ENDIF
				ENDIF
				
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Log colors="+num2istr(vLog_Colors)
				
				ControlInfo/W=$strWindow_Name T1_Check_Color_Scale
				vColor_Scale=V_Value
				IF(vColor_Scale)
					ControlInfo/W=$strWindow_Name T1_Pop_CS_Graph
					String strColor_Scale_Graph_Name=S_Value
					
					Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
					twKML_Inputs[numpnts(twKML_Inputs)-1]="Color scale graph="+strColor_Scale_Graph_Name
					
					IF(WinType(strColor_Scale_Graph_Name)==0)
						DoAlert/T="ERROR ID-10T" 0, "Invalid color scale graph!"
						Return -1
					ENDIF
					
					ControlInfo/W=$strWindow_Name T1_Check_Transparent
					Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
					twKML_Inputs[numpnts(twKML_Inputs)-1]="Transparent color scale="+num2istr(V_Value)
				ENDIF
				
			ELSE
				ControlInfo/W=$strWindow_Name T1_Pop_Marker_Color
				String strMarker_Hex_Color_Default=KML_Convert_RGB_to_Hex(V_Red, V_Green, V_Blue)
				
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Marker color="+strMarker_Hex_Color_Default				
			ENDIF
		
			ControlInfo/W=$strWindow_Name T1_Check_SizeFz
			Variable vSize_Flag = V_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Size by Fz="+num2istr(vSize_Flag)
			
			IF(vSize_Flag)		
				String strMarker_Size_Min, strMarker_Size_Max
				ControlInfo/W=$strWindow_Name T1_SetVar_Marker_Size_Min
				Variable vMarker_Size_1=V_Value
				
				ControlInfo/W=$strWindow_Name T1_SetVar_Marker_Size_Max
				Variable vMarker_Size_2=V_Value
				
				Variable vMarker_Size_Min=min(vMarker_Size_1, vMarker_Size_2)
				Variable vMarker_Size_Max=max(vMarker_Size_1, vMarker_Size_2)
				
				sprintf strMarker_Size_Min, "%0.6f", vMarker_Size_Min
				sprintf strMarker_Size_Max, "%0.6f", vMarker_Size_Max
				
				IF(cmpstr(strMarker_Size_Min, strMarker_Size_Max)==0)
					DoAlert/T="ERROR ID-10T" 0, "Min and max marker sizes must be different"
					Return -1
				ENDIF
				
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Marker size min="+strMarker_Size_Min
				
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Marker size max="+strMarker_Size_Max
				
				ControlInfo/W=$strWindow_Name T1_Check_Log_Size
				Variable vLog_Size=V_Value
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Log marker size="+num2istr(vLog_Size)
			
			ELSE
				ControlInfo/W=$strWindow_Name T1_SetVar_Marker_Size_Default
				Variable vMarker_Size_Default=V_Value
			
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Marker size="+num2istr(vMarker_Size_Default)
			ENDIF
			
			//This is redundant, but I should add a notice to the user if this happens
			IF(!vFz_Flag && (vColor_Flag || vSize_Flag))			//The logic for the fz_flag, color_flag, and size_flag is a little fuzzy, but I think it's fine and I don't plan to fix it
				DoAlert/T="ERROR ID-10T" 0, "You cannot color/size by f(z) if your f(z) wave does not exist!"
				Return -1
			ENDIF
			
			//Wind parameters
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_WindSpeed
			String strWindSpeed_Wave=S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="WindSpeed="+strWindSpeed_Wave
						
			ControlInfo/W=$strWindow_Name P0_Pop_Wave_WindDir
			String strWindDir_Wave=S_Value
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="WindDir="+strWindDir_Wave
			
			Variable vWind_Flag=cmpstr(strWindSpeed_Wave, "none")!=0 && cmpstr(strWindDir_Wave, "none")!=0
			IF(vWind_Flag)
				WAVE/Z WindSpeed_Wave=$strWindSpeed_Wave
				WAVE/Z WindDir_Wave=$strWindDir_Wave
				
				IF(numpnts(WindSpeed_Wave)!= vNum_Data_Pnts || numpnts(WindDir_Wave)!= vNum_Data_Pnts)
					DoAlert/T="ERROR ID-10T", 0, "Waves must all have same number of points!"
					Return -1
				ENDIF
				
				ControlInfo/W=$strWindow_Name P0_SetVar_WindRes
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="WindRes="+num2istr(V_Value)
				
				ControlInfo/W=$strWindow_Name T2_SetVar_Barb_Length
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Barb length="+num2str(V_Value)
				
				ControlInfo/W=$strWindow_Name T2_SetVar_Barb_Thick
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Barb thick="+num2str(V_Value)
				
				ControlInfo/W=$strWindow_Name T2_SetVar_Barb_Scale
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Barb scale="+num2str(V_Value)
				
				ControlInfo/W=$strWindow_Name T2_Check_Barb_Use_Custom_Color
				Variable vUse_Custom_Color=V_Value
				
				ControlInfo/W=$strWindow_Name T2_Pop_Barb_Color
				String strBarb_Color=KML_Convert_RGB_to_Hex(V_Red, V_Green, V_Blue)
				
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Wind color mode="+SelectString(vColor_Flag, num2istr(vUse_Custom_Color), "0")		//If coloring by f(z), use only one color
				
				Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
				twKML_Inputs[numpnts(twKML_Inputs)-1]="Wind color="+strBarb_Color
			ENDIF
			
			ControlInfo/W=$strWindow_Name T1_SetStr_File_Name
			String strFile_Name=S_Value
			IF(strlen(strFile_Name)==0)
				Prompt strFile_Name, "File name (no extension)"
				DoPrompt "Enter a file name", strFile_Name
				IF(V_Flag)
					Return -1
				ENDIF
				
				strFile_Name=KML_Cleanup_File_Name(strFile_Name)
				
				SetVariable T0_SetStr_File_Name, value=_STR:strFile_Name
			ENDIF
			
			strFile_Name=KML_Cleanup_File_Name(strFile_Name)
			strFile_Name+=SelectString(StringMatch(strFile_Name, "*.kml"), ".kml", "")
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="File name="+strFile_Name
			
			//Get the folder path and do anything that creates files after all the checks that could lead to aborting the function
			NewPath/O/M="Choose folder to save kml file in:"/Q/Z pKML_Path 
			IF(V_flag!=0)
				Return -1
			ENDIF
			
			PathInfo pKML_Path
			String strKML_Path=S_Path
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="File path="+strKML_Path
			
			GetFileFolderInfo/P=pKML_Path/Q/Z strFile_Name
			IF(V_Flag==0)
				DoAlert/T="READ CAREFULLY!!!!!" 1, "READ CAREFULLY!!!!\r\rThe file already exists!  Do you want to OVERWRITE the file (YES) or CANCEL (NO)?\r\rREAD CAREFULLY!"
				
				IF(V_Flag==2)
					Return -1
				ENDIF
			ENDIF
			
			Redimension/N=(numpnts(twKML_Inputs)+1) twKML_Inputs
			twKML_Inputs[numpnts(twKML_Inputs)-1]="Overwrite=1"
		
			KillPath/Z pKML_Path
						
			Variable vFlag=KML_Marker_Make_File(twKML_Inputs)
			
			IF(vFlag!=-1)
				Print "Done creating "+strKML_Path+strFile_Name
			ELSE
				Print "Problems creating the file!"
			ENDIF
			
		BREAK
	ENDSWITCH
	
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Function that actually makes the KML marker file
FUNCTION KML_Marker_Make_File(WAVE/T/Z twUser_Inputs)
	
	Close/A
	
	String strUser_Inputs
	wfprintf strUser_Inputs, "%s;", twUser_Inputs
					
	String strLatitude_Wave = StringByKey("Lat", strUser_Inputs, "=", ";")		
	
	String strLongitude_Wave = StringByKey("Lon", strUser_Inputs, "=", ";")		

	WAVE/Z Latitude_Wave=$strLatitude_Wave, Longitude_Wave=$strLongitude_Wave
	
	IF(!WaveExists(Latitude_Wave) || !WaveExists(Longitude_Wave))
		Print "One or more waves don't exist!"
		Return -1
	ENDIF
	
	Variable vNum_Data_Pnts = numpnts(Latitude_Wave)
	
	Variable vResolution = str2num(StringByKey("DataRes", strUser_Inputs, "=", ";"))	
	Variable vNum_KML_Pnts=ceil(vNum_Data_Pnts/vResolution)		//This should be okay
	
	IF((numpnts(Longitude_Wave)!= vNum_Data_Pnts))
		Print "Waves must all have same number of points!"
		Return -1
	ENDIF
	
	Make/O/D/FREE/N=(vNum_KML_Pnts) Latitude_Wave_KML=Latitude_Wave[p*vResolution]
	Make/O/D/FREE/N=(vNum_KML_Pnts) Longitude_Wave_KML=Longitude_Wave[p*vResolution]
	
	Make/O/B/U/FREE/N=(numpnts(Latitude_Wave)) Flag_Wave=(numtype(Latitude_Wave[p])==0) && numtype(Longitude_Wave[p])==0
	
	String strInfo="Experiment name:"+IgorInfo(1)+";Resolution:"+num2str(vResolution)+";Latitude wave:"+strLatitude_Wave+";Longitude wave:"+strLongitude_Wave+";"
	
	String strAltitude_Wave = StringByKey("Alt", strUser_Inputs, "=", ";")
	WAVE/Z Altitude_Wave=$strAltitude_Wave
	Variable vAltitude_Flag=WaveExists(Altitude_Wave)
					
	IF(vAltitude_Flag)
		IF(numpnts(Altitude_Wave)!= vNum_Data_Pnts)
			Print "Waves must all have same number of points!"
			Return -1
		ENDIF
		
		Variable vAltitude_Scale=str2num(StringByKey("AltScale", strUser_Inputs, "=", ";"))
		
		Flag_Wave=Flag_Wave[p] && numtype(Altitude_Wave[p])==0
		Make/O/D/FREE/N=(vNum_KML_Pnts) Altitude_Wave_KML=Altitude_Wave[p*vResolution]*vAltitude_Scale
		
		strInfo+="Altitude wave:"+strAltitude_Wave+";"
	ENDIF
		
	String strTime_Wave=StringByKey("Time", strUser_Inputs, "=", ";")
	WAVE/Z Time_Wave=$strTime_Wave
	Variable vTime_Flag=WaveExists(Time_Wave)
	
	IF(vTime_Flag)
		IF(numpnts(Time_Wave)!= vNum_Data_Pnts)
			Print "Waves must all have same number of points!"
			Return -1
		ENDIF
		
		Flag_Wave=Flag_Wave[p] && numtype(Time_Wave[p])==0
		Make/O/T/FREE/N=(vNum_KML_Pnts) TimeStamp_GE_Format_KML=Secs2Date(Time_Wave[p*vResolution], -2)+"T"+Secs2Time(Time_Wave[p*vResolution], 3)
		
		strInfo+="Time wave:"+strTime_Wave+";"
		
		Variable vUTC_Offset=str2num(StringByKey("UTC offset", strUser_Inputs, "=", ";"))
		Variable vUTC_Offset_hr=floor(vUTC_Offset)
		Variable vUTC_Offset_Frac=round(60*(vUTC_Offset-vUTC_Offset_hr))
		
		String strUTC_Offset
		sprintf strUTC_Offset, "%+0.2d", vUTC_Offset_hr
		IF(vUTC_Offset_Frac==0)
			strUTC_Offset+=":00"
		ELSE
			String strUTC_Offset_Frac
			sprintf strUTC_Offset_Frac, "%0.2d", vUTC_Offset_Frac
			strUTC_Offset+=":"+strUTC_Offset_Frac
		ENDIF
		
		IF(cmpstr(strUTC_Offset, "+00:00")==0)
			strUTC_Offset="Z"
		ENDIF
		
	ELSE
		strUTC_Offset="Z"
	ENDIF
	
	String strFz_Wave = StringByKey("Fz", strUser_Inputs, "=", ";")
	WAVE/Z Fz_Wave=$strFz_Wave
	Variable vFz_Flag=WaveExists(Fz_Wave)
	
	IF(vFz_Flag)
		IF(numpnts(Fz_Wave)!= vNum_Data_Pnts)
			Print "Waves must all have same number of points!"
			Return -1
		ENDIF
		
		Variable vFz_Min = str2num(StringByKey("Fz min", strUser_Inputs, "=", ";"))
		Variable vFz_Max = str2num(StringByKey("Fz max", strUser_Inputs, "=", ";"))
		
		IF(numtype(vFz_Min)!=0 || numtype(vFz_Max)!=0)
			Print "Fz limits must be real numbers!"
			Return -1
		ENDIF
		
		String strFz_Min, strFz_Max				//Don't want to compare floats.
		sprintf strFz_Min, "%0.6f", vFz_Min
		sprintf strFz_Max, "%0.6f", vFz_Max
		
		IF(cmpstr(strFz_Min, strFz_Max)==0)
			Print "Min and max Fz values must be different"
			Return -1
		ENDIF
		
		Flag_Wave=Flag_Wave[p] && numtype(Fz_Wave[p])==0
		Make/O/D/FREE/N=(vNum_KML_Pnts) Fz_Wave_KML=Fz_Wave[p*vResolution]
		
		strInfo+="F(z) wave:"+strFz_Wave+";"
	ENDIF
	
	String strWindSpeed_Wave=StringByKey("WindSpeed", strUser_Inputs, "=", ";")
	String strWindDirection_Wave=StringByKey("WindDir", strUser_Inputs, "=", ";")
	Variable vWind_Resolution=str2num(StringByKey("WindRes", strUser_Inputs, "=", ";"))
	
	WAVE/Z Wind_Speed=$strWindSpeed_Wave
	WAVE/Z Wind_Direction=$strWindDirection_Wave
	
	Variable vWind_Flag=WaveExists(Wind_Speed) && WaveExists(Wind_Direction) && vWind_Resolution>0
	IF(vWind_Flag)
		IF(numpnts(Wind_Speed)!= vNum_Data_Pnts || numpnts(Wind_Direction)!=vNum_Data_Pnts)
			Print "Waves must all have same number of points!"
			Return -1
		ENDIF
		
		Flag_Wave=Flag_Wave[p] && numtype(Wind_Speed[p])==0 && numtype(Wind_Direction[p])==0
		Make/O/D/FREE/N=(vNum_KML_Pnts) Wind_Speed_KML=Wind_Speed[p*vResolution]
		Make/O/D/FREE/N=(vNum_KML_Pnts) Wind_Direction_KML=Wind_Direction[p*vResolution]
	ENDIF

				
	Make/O/D/FREE/N=(vNum_KML_Pnts) Flag_Wave_KML=Flag_Wave[p*vResolution]
	
	Extract/O/FREE Latitude_Wave_KML, Latitude_Wave_KML, Flag_Wave_KML
	Extract/O/FREE Longitude_Wave_KML, Longitude_Wave_KML, Flag_Wave_KML
	
	IF(vFz_Flag)
		Extract/O/FREE Fz_Wave_KML, Fz_Wave_KML, Flag_Wave_KML
	ENDIF
	
	IF(vAltitude_Flag)
		Extract/O/FREE Altitude_Wave_KML, Altitude_Wave_KML, Flag_Wave_KML
	ELSE
		Make/O/D/FREE/N=(numpnts(Latitude_Wave_KML)) Altitude_Wave_KML=0
	ENDIF
	
	IF(vTime_Flag)
		Extract/O/T/FREE TimeStamp_GE_Format_KML, TimeStamp_GE_Format_KML, Flag_Wave_KML
		TimeStamp_GE_Format_KML=TimeStamp_GE_Format_KML+strUTC_Offset
		
		String strTime_Start=TimeStamp_GE_Format_KML[0]
		String strTime_Stop=TimeStamp_GE_Format_KML[numpnts(TimeStamp_GE_Format_KML)-1]
	ENDIF
	
	IF(vWind_Flag)
		Extract/O/FREE Wind_Speed_KML, Wind_Speed_KML, Flag_Wave_KML
		Extract/O/FREE Wind_Direction_KML, Wind_Direction_KML, Flag_Wave_KML
	ENDIF
	
	vNum_KML_Pnts=numpnts(Latitude_Wave_KML)
				
	WaveStats/Q Latitude_Wave_KML
	Variable vLat_Avg = V_avg
	Variable vLat_Min=V_Min
	Variable vLat_Max=V_Max
	
	WaveStats/Q Longitude_Wave_KML
	Variable vLon_Avg = V_avg
	Variable vLon_Min=V_Min
	Variable vLon_Max=V_Max
	
	Variable vHeight_Scale_Factor=2.5		//This seems to work
	Variable vHeight=vHeight_Scale_Factor*max(KML_Get_Bounds(vLat_Max, vLon_Avg, vLat_Min, vLon_Avg), KML_Get_Bounds(vLat_Avg, vLon_Min, vLat_Avg, vLon_Max))
	
	//	if color-coding is checked, get the color table into a wave for use with this data
	Variable vColor_by_Fz=str2num(StringByKey("Color by fz", strUser_Inputs, "=", ";"))
	
	IF(vColor_by_Fz)
		String strColor_Table=StringByKey("Color table", strUser_Inputs, "=", ";")
		Variable vColor_Table_Type=str2num(StringByKey("Color table type", strUser_Inputs, "=", ";"))
		
		IF(vColor_Table_Type)
			WAVE/Z Custom_Color_Table=$strColor_Table
			
			IF(!WaveExists(Custom_Color_Table))
				Print "Invalid color table wave!"
				Return -1
			ENDIF
			
			Duplicate/O/FREE Custom_Color_Table, wColor_Table_Wave
		
		ELSE
			String strAll_Color_Tables=CTabList()		//ColorTab2Wave doesn't have a Z flag, so we need to check the full list of color tables
			IF(WhichListItem(strColor_Table, strAll_Color_Tables, ";", 0, 0)<0)
				Print "Invalid color table name!"
				Return -1
			ENDIF
			
			String strCurrent_Data_Folder=GetDataFolder(1)
			NewDataFolder/O/S root:Stupidly_Named_Temp_Folder
			ColorTab2Wave $strColor_Table			
			WAVE/Z M_Colors
			Duplicate/O/FREE M_Colors, wColor_Table_Wave
			
			SetDataFolder $strCurrent_Data_Folder
			KillDataFolder/Z root:Stupidly_Named_Temp_Folder
		ENDIF
		
		Variable vNum_Colors=DimSize(wColor_Table_Wave, 0)
		
		IF(str2num(StringByKey("Reverse colors", strUser_Inputs, "=", ";")))
			MatrixOp/O/FREE wColor_Table_Wave=ReverseCols(wColor_Table_Wave)
		ENDIF
		
		Variable vLog_Colors=str2num(StringByKey("Log colors", strUser_Inputs, "=", ";"))
		
		IF(!vLog_Colors)
			Variable vColor_Slope=(vNum_Colors-1)/(vFz_Max-vFz_Min)
			Variable vColor_Intercept=0-vColor_Slope*vFz_Min
		ELSE
			IF(vFz_Max<=0 || vFz_Min<=0)
				Print "One or both of the Fz limits is less than or equal to zero!  Using linear scaling of the colors!"
				vLog_Colors=0
				vColor_Slope=(vNum_Colors-1)/(vFz_Max-vFz_Min)
				vColor_Intercept=0-vColor_Slope*vFz_Min
			ELSE
				vColor_Slope=(vNum_Colors-1)/(log(vFz_Max)-log(vFz_Min))
				vColor_Intercept=0-vColor_Slope*log(vFz_Min)
			ENDIF
		ENDIF
		
		String strColor_Scale_Graph=StringByKey("Color scale graph", strUser_Inputs, "=", ";")
		Variable vColor_Scale=strlen(strColor_Scale_Graph)>0
		
		IF(vColor_Scale)
			IF(WinType(strColor_Scale_Graph)!=1)
				Print "Invalid color scale graph!"
				Return -1
			ENDIF
			
			Variable vTransparent_Color_Scale=str2num(StringByKey("Transparent color scale", strUser_Inputs, "=", ";"))
			
			String strColor_Scale_Position=StringByKey("Color scale position", strUser_Inputs, "=", ";")
		ENDIF
		
	ELSE
		String strMarker_Hex_Color_Default=StringByKey("Marker color", strUser_Inputs, "=", ";")
		IF(ItemsInList(strMarker_Hex_Color_Default, ",")>1)
			strMarker_Hex_Color_Default=KML_Convert_RGB_to_Hex(str2num(StringFromList(0,strMarker_Hex_Color_Default, ",")), str2num(StringFromList(1,strMarker_Hex_Color_Default, ",")), str2num(StringFromList(2,strMarker_Hex_Color_Default, ",")))
		ENDIF
	ENDIF
	
	Variable vSize_Flag = str2num(StringByKey("Size by Fz", strUser_Inputs, "=", ";"))
	IF(vSize_Flag)
		String strMarker_Size_Min, strMarker_Size_Max
		Variable vMarker_Size_1=str2num(StringByKey("Marker size min", strUser_Inputs, "=", ";"))
		
		Variable vMarker_Size_2=str2num(StringByKey("Marker size max", strUser_Inputs, "=", ";"))
		
		Variable vMarker_Size_Min=min(vMarker_Size_1, vMarker_Size_2)
		Variable vMarker_Size_Max=max(vMarker_Size_1, vMarker_Size_2)
		
		sprintf strMarker_Size_Min, "%0.6f", vMarker_Size_Min
		sprintf strMarker_Size_Max, "%0.6f", vMarker_Size_Max
		
		IF(cmpstr(strMarker_Size_Min, strMarker_Size_Max)==0)
			Print "Min and max marker sizes must be different"
			Return -1
		ENDIF
		
		Variable vLog_Size=str2num(StringByKey("Log marker size", strUser_Inputs, "=", ";"))
		IF(!vLog_Size)
			Variable vMarker_Slope=(vMarker_Size_Max-vMarker_Size_Min)/(vFz_Max-vFz_Min)
			Variable vMarker_Intercept=vMarker_Size_Min-vMarker_Slope*vFz_Min
		ELSE
			IF(vMarker_Size_Min==0 || vMarker_Size_Max==0)
				Print "One or more marker sizes is zero.  Using linear scaling of the marker sizes!"
				vLog_Size=0
				vMarker_Slope=(vMarker_Size_Max-vMarker_Size_Min)/(vFz_Max-vFz_Min)
				vMarker_Intercept=vMarker_Size_Min-vMarker_Slope*vFz_Min
			ELSE
				vMarker_Slope=(vMarker_Size_Max-vMarker_Size_Min)/(log(vFz_Max)-log(vFz_Min))
				vMarker_Intercept=vMarker_Size_Min-vMarker_Slope*log(vFz_Min)
			ENDIF
		ENDIF
		
	ELSE
		Variable vMarker_Size_Default = str2num(StringByKey("Marker size", strUser_Inputs, "=", ";"))	
	ENDIF
	
	//This is redundant, but I should add a notice to the user if this happens
	IF(!vFz_Flag && (vColor_by_Fz || vSize_Flag))			//The logic for the fz_flag, color_flag, and size_flag is a little fuzzy, but I think it's fine and I don't plan to fix it
		Print "You cannot color/size by f(z) if your f(z) wave does not exist!"
		Return -1
	ENDIF
	
	String strFile_Name=StringByKey("File name", strUser_Inputs, "=", ";")
	strFile_Name+=SelectString(StringMatch(strFile_Name, "*.kml"), ".kml", "")
	
	strFile_Name=KML_Cleanup_File_Name(strFile_Name)
	
	String strKML_Path=StringByKey("File path", strUser_Inputs, "=", ";")
	NewPath/O/Q/Z pKML_Path, strKML_Path
	
	Variable vOverwrite=str2num(StringByKey("Overwrite", strUser_Inputs, "=", ";"))
	
	GetFileFolderInfo/P=pKML_Path/Q/Z strFile_Name
	IF(V_Flag==0 && !vOverwrite)
		Print "File already exists and overwrite flag not set!"
		Return -1
	ENDIF
	
	String strSymbol = StringByKey("Symbol", strUser_Inputs, "=", ";")
	
	String strKML_Symbol_Line
	IF(strsearch(strSymbol, "GE ", 0)!=0)		//User icons.  Doing this in the IF case so if the user makes a typo there is a valid icon
		strKML_Symbol_Line="<href>"+RemoveEnding(strFile_Name, ".kml")+"_Icon.png</href>"
				
		String strMarker_Graph_Name=UniqueName("Dummy_Graph_For_Marker_", 6, 0)
		
		//The different sizes and margins of the graphs is intentional
		STRSWITCH(strSymbol)		//Makes icons without an outline using the predefined Igor markers.  It's a little kludgy, but way more flexible that trying to make the shapes by hand.
			CASE "Diamond":
				Display /W=(536,602,628,694)/N=$strMarker_Graph_Name/K=1 as strMarker_Graph_Name
				ModifyGraph margin(left)=1,margin(bottom)=1,margin(top)=1,margin(right)=1,width=90
				ModifyGraph height=90
				TextBox/C/N=text0/F=0/M/H={15,2,10}/B=1/A=MC/X=0.00/Y=0.00 "\\K(0,0,0)\\Z99\\W5018"
			BREAK
			
			CASE "Circle":
				Display /W=(511,381,585,455)/N=$strMarker_Graph_Name/K=1 as strMarker_Graph_Name
				ModifyGraph margin(left)=1,margin(bottom)=1,margin(top)=1,margin(right)=1,width=72
				ModifyGraph height=72
				TextBox/C/N=text0/F=0/M/H={15,2,10}/B=1/A=MC/X=0.00/Y=0.00 "\\K(0,0,0)\\Z99\\W5019"
			BREAK
			
			CASE "Square":			//No idea why I have to do this, but if it's all 65535 then it doesn't work
				Display /W=(511,381,585,455)/N=$strMarker_Graph_Name/K=1 as strMarker_Graph_Name
				ModifyGraph margin(left)=1,margin(bottom)=1,margin(top)=1,margin(right)=1,width=72
				ModifyGraph height=72
				TextBox/C/N=text0/F=0/M/H={15,2,10}/B=1/A=MC/X=0.00/Y=0.00 "\\K(0,0,0)\\Z99\\W5016"
			BREAK
			
			CASE "Hexagon":
				Display /W=(616,848,679,911)/N=$strMarker_Graph_Name/K=1 as strMarker_Graph_Name
				ModifyGraph margin(left)=1,margin(bottom)=1,margin(top)=1,margin(right)=1,width=61.2
				ModifyGraph height=61.2
				TextBox/C/N=text0/F=0/M/H={15,2,10}/B=1/A=MC/X=0.00/Y=0.00 "\\K(0,0,0)\\Z99\\W5055"
			BREAK
			
			DEFAULT:
				Print "Invalid marker name"
				Return -1
			BREAK
				
		ENDSWITCH
		
		DoUpdate/W=$strMarker_Graph_Name

		SavePICT/O/E=-5/B=288/WIN=$strMarker_Graph_Name/P=pKML_Path as RemoveEnding(strFile_Name, ".kml")+"_Icon.png"		//Save the graph with the marker.  Use the eventual name for the icon so we don't have to kill the file later.
		KillWindow/Z $strMarker_Graph_Name		//Kill the graph
		
		ImageLoad/T=png/O/Q/N=Dummy_Marker_Matrix_Wave/P=pKML_Path RemoveEnding(strFile_Name, ".kml")+"_Icon.png"		//Load the image as a matrix
		
		WAVE/Z Dummy_Marker_Matrix_Wave
		Make/O/D/FREE/N=(DimSize(Dummy_Marker_Matrix_Wave,0), DimSize(Dummy_Marker_Matrix_Wave,1), 4) mMarker			//Convert the image matrix to the proper form (white inside the marker, transparent outside)
		mMarker[][][0]=Dummy_Marker_Matrix_Wave[p][q][0]>100 ? 0 : 65535			//I probably could use layers 1 and 2 as well, but this works.  
		mMarker[][][1]=Dummy_Marker_Matrix_Wave[p][q][0]>100 ? 0 : 65535
		mMarker[][][2]=Dummy_Marker_Matrix_Wave[p][q][0]>100 ? 0 : 65535
		mMarker[][][3]=Dummy_Marker_Matrix_Wave[p][q][0]>100 ? 0 : 65535

		ImageSave/T="png"/O/P=pKML_Path mMarker as RemoveEnding(strFile_Name, ".kml")+"_Icon.png"		//Save the wave as an image.  Overwrite instead of killing the file.
		
		KillWaves/Z Dummy_Marker_Matrix_Wave
	
	ELSE		//Icons defined in GE.  Have black outlines.
		STRSWITCH(strSymbol)
			CASE "GE shaded_dot":
				strKML_Symbol_Line= "<href>http://maps.google.com/mapfiles/kml/shapes/shaded_dot.png</href>"	
			BREAK
			
			CASE "GE circle":
				strKML_Symbol_Line= "<href>http://maps.google.com/mapfiles/kml/pal2/icon26.png</href>"	
			BREAK
			
//			CASE "GE square":																										//This doesn't seem to work
//				strKML_Symbol_Line= "<href>http://maps.google.com/mapfiles/kml/pal2/icon15.png</href>"	
//			BREAK

			DEFAULT:
				strKML_Symbol_Line= "<href>http://maps.google.com/mapfiles/kml/pal2/icon26.png</href>"	
			BREAK
		ENDSWITCH
	ENDIF
	
	IF(vColor_Scale)
		SavePICT/O/E=-5/TRAN=(vTransparent_Color_Scale)/B=288/P=pKML_Path/WIN=$strColor_Scale_Graph as RemoveEnding(strFile_Name, ".kml")+"_Color_Scale.png"
	ENDIF
	
	Variable refnum
	Open/P=pKML_Path refNum as strFile_Name		// Prompt for file name to save into.
	
	fprintf refNum, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\r"
	fprintf refNum, "<kml xmlns=\"http://www.opengis.net/kml/2.2\" xmlns:gx=\"http://www.google.com/kml/ext/2.2\" xmlns:kml=\"http://www.opengis.net/kml/2.2\" xmlns:atom=\"http://www.w3.org/2005/Atom\">\r"
	fprintf refNum, "<Document>\r"
	fprintf refnum, "	<ExtendedData>"+strInfo+"</ExtendedData>\r"
	fprintf refNum, "	<name>"+strFile_Name+"</name>\r"		

	Variable iKML_Index			
	IF(vSize_Flag || vColor_by_Fz)
		fprintf refNum, "	<Folder>\r"		//write this string to file
		
		FOR(iKML_Index=0;iKML_Index<vNum_KML_Pnts;iKML_Index+=1)
			
			IF(vSize_Flag)
				IF(!vLog_Size)
					Variable vMarker_Size=limit(vMarker_Slope*FZ_Wave_KML[iKML_Index]+vMarker_Intercept, vMarker_Size_Min, vMarker_Size_Max)
				ELSE
					vMarker_Size=limit(vMarker_Slope*log(FZ_Wave_KML[iKML_Index])+vMarker_Intercept, vMarker_Size_Min, vMarker_Size_Max)
				ENDIF
			ELSE
				vMarker_Size=vMarker_Size_Default
			ENDIF
			
			IF(vColor_by_Fz)
				IF(!vLog_Colors)
					Variable vColor_Index=round(limit(vColor_Slope*FZ_Wave_KML[iKML_Index]+vColor_Intercept, 0, vNum_Colors-1))		//I suppose I could use selectnumber, but both values get computed, which will be slower
				ELSE
					vColor_Index=round(limit(vColor_Slope*log(FZ_Wave_KML[iKML_Index])+vColor_Intercept, 0, vNum_Colors-1))
				ENDIF
				
				String strHex_Color=KML_Convert_RGB_to_Hex(wColor_Table_Wave[vColor_Index][0], wColor_Table_Wave[vColor_Index][1], wColor_Table_Wave[vColor_Index][2])
			ELSE
				strHex_Color=strMarker_Hex_Color_Default
			ENDIF
			
			fprintf refNum, "		<Placemark>\r"
			
			IF(vTime_Flag)
				fprintf refNum, "			<TimeStamp><when>"+TimeStamp_GE_Format_KML[iKML_Index]+"</when></TimeStamp>\r"
			ENDIF
			
			fprintf refNum, "			<Style>\r"
			fprintf refNum, "				<IconStyle>\r"
			fprintf refNum, "					<Icon>"+strKML_Symbol_Line+"</Icon>\r"
			fprintf refNum, "					<scale>"+num2str(vMarker_Size)+"</scale>\r"
			fprintf refNum, "					<heading>0</heading>\r"
			fprintf refNum, "					<color>"+strHex_Color+"</color>\r"
			fprintf refNum, "				</IconStyle>\r"
			fprintf refNum, "			</Style>\r"
			
			fprintf refNum, "			<Point>\r"
			IF(vAltitude_Flag)		//Alt
				fprintf refNum, "				<coordinates>%0.8f,%0.8f,%0.8f</coordinates>\r", Longitude_Wave_KML[iKML_Index], Latitude_Wave_KML[iKML_Index], Altitude_Wave_KML[iKML_Index]
				fprintf refNum, "				<altitudeMode>absolute</altitudeMode>\r"
			ELSE		//No alt
				fprintf refNum, "				<coordinates>%0.8f,%0.8f,0</coordinates>\r", Longitude_Wave_KML[iKML_Index], Latitude_Wave_KML[iKML_Index]
			ENDIF
			
			fprintf refNum, "			</Point>\r"
			fprintf refNum, "		</Placemark>\r"
		ENDFOR
				
		fprintf refNum, "</Folder>\r"					//There was a square bracket after the >, that might be a typo
	
	ELSE		//If you want single color/size markers.  This is copied from a KML modified in Google Earth so each marker has the same style.
		fprintf refnum, "	<Style id=\"mySym1\">\r"
		fprintf refnum, "		<IconStyle>\r"
		fprintf refnum, "			<color>"+strMarker_Hex_Color_Default+"</color>\r"
		fprintf refnum, "			<scale>"+num2str(vMarker_Size_Default)+"</scale>\r"
		fprintf refnum, "			<Icon>"+strKML_Symbol_Line+"</Icon>\r"
		fprintf refnum, "  	</IconStyle>\r"
		fprintf refnum, "	</Style>\r"
		fprintf refnum, "	<StyleMap id=\"mySym10\">\r"
		fprintf refnum, "		<Pair>\r"
		fprintf refnum, "			<key>normal</key>\r"
		fprintf refnum, "			<styleUrl>#mySym1</styleUrl>\r"
		fprintf refnum, "		</Pair>\r"
		fprintf refnum, "		<Pair>\r"
		fprintf refnum, "			<key>highlight</key>\r"
		fprintf refnum, "			<styleUrl>#mySym11</styleUrl>\r"
		fprintf refnum, "		</Pair>\r"
		fprintf refnum, "	</StyleMap>\r"
		fprintf refnum, "	<Style id=\"mySym11\">\r"
		fprintf refnum, "		<IconStyle>\r"
		fprintf refnum, "			<color>"+strMarker_Hex_Color_Default+"</color>\r"
		fprintf refnum, "			<scale>"+num2str(vMarker_Size_Default)+"</scale>\r"
		fprintf refnum, "			<Icon>"+strKML_Symbol_Line+"</Icon>\r"
		fprintf refnum, "		</IconStyle>\r"
		fprintf refnum, "	</Style>\r"

		fprintf refNum, "	<Folder>\r"		//write this string to file
		FOR(iKML_Index=0;iKML_Index<vNum_KML_Pnts;iKML_Index+=1)
			fprintf refNum, "		<Placemark>\r"
			
			IF(vTime_Flag)
				fprintf refNum, "			<TimeStamp><when>"+TimeStamp_GE_Format_KML[iKML_Index]+"</when></TimeStamp>\r"
			ENDIF
			
			fprintf refNum, "			<styleUrl>#mySym10</styleUrl>\r"
			fprintf refNum, "			<Point>\r"
			IF(vAltitude_Flag)		//Alt
				fprintf refnum, "				<coordinates>%0.8f,%0.8f,%0.8f</coordinates>\r", Longitude_Wave_KML[iKML_Index], Latitude_Wave_KML[iKML_Index], Altitude_Wave_KML[iKML_Index]
				fprintf refnum, "				<altitudeMode>absolute</altitudeMode>\r" 
			ELSE
				fprintf refNum, "				<coordinates>%0.8f,%0.8f,0</coordinates>\r", Longitude_Wave_KML[iKML_Index], Latitude_Wave_KML[iKML_Index]
			ENDIF
			fprintf refNum, "			</Point>\r"
			fprintf refNum, "		</Placemark>\r"
		ENDFOR
		
		fprintf refNum, "	</Folder>\r"					//There was a square bracket after the >, that might be a typo
	ENDIF
	
	fprintf refNum, "	<LookAt>\r"
	fprintf refNum, "		<longitude>%.10f</longitude>\r", vLon_Avg
	fprintf refNum, "		<latitude>%.10f</latitude>\r", vLat_Avg
	fprintf refNum, "		<range>%d</range>\r", vHeight
	fprintf refNum, "		<tilt>0</tilt>\r"
	fprintf refNum, "		<heading>0</heading>\r"
	
	
	IF(vTime_Flag)
		fprintf refNum, "		<gx:TimeSpan>\r"
		fprintf refNum, "		<begin>"+strTime_Start+"</begin>\r"
		fprintf refNum, "		<end>"+strTime_Stop+"</end>\r"
		fprintf refNum, "		</gx:TimeSpan>\r"
	ENDIF
	
	fprintf refNum, "	</LookAt>\r"
	
	IF(vWind_Flag)
		Make/O/T/FREE/N=0 twBarb_List
		String strBarb_Icon_Folder_Path=strKML_Path+SelectString(StringMatch(strKML_Path, "*:"), ":", "")+RemoveEnding(strFile_Name, ".kml")+"_WindBarbs"
		String strBarb_Icon_Folder_Name=RemoveEnding(strFile_Name, ".kml")+"_WindBarbs"
		String strBarb_Icon_Base_Name=RemoveEnding(strFile_Name, ".kml")
		
		NewPath/O/C/Q/Z pBarb_Icon_Path, strBarb_Icon_Folder_Path
		
		Variable vBarb_Length=str2num(StringByKey("Barb length", strUser_Inputs, "=", ";"))
		Variable vBarb_Thick=str2num(StringByKey("Barb thick", strUser_Inputs, "=", ";"))
		Variable vBarb_Scale=str2num(StringByKey("Barb scale", strUser_Inputs, "=", ";"))
		
		Variable vWind_Color_Mode=str2num(StringByKey("Wind color mode", strUser_Inputs, "=", ";"))
		
		IF(vWind_Color_Mode)
			String strBarb_Color=StringByKey("Wind color", strUser_Inputs, "=", ";")
			IF(ItemsInList(strBarb_Color, ",")>1)
				strBarb_Color=KML_Convert_RGB_to_Hex(str2num(StringFromList(0,strBarb_Color, ",")), str2num(StringFromList(1,strBarb_Color, ",")), str2num(StringFromList(2,strBarb_Color, ",")))
			ENDIF
		ELSE
			IF(vColor_by_Fz)
				strBarb_Color=StringByKey("Wind color", strUser_Inputs, "=", ";")
				IF(ItemsInList(strBarb_Color, ",")>1)
					strBarb_Color=KML_Convert_RGB_to_Hex(str2num(StringFromList(0,strBarb_Color, ",")), str2num(StringFromList(1,strBarb_Color, ",")), str2num(StringFromList(2,strBarb_Color, ",")))
				ENDIF
			ELSE
				strBarb_Color=strMarker_Hex_Color_Default
			ENDIF
		ENDIF
			
		Variable vNum_Barbs=floor(vNum_KML_Pnts/vWind_Resolution)
			
		Variable iBarbDex
		FOR(iBarbDex=0;iBarbDex<vNum_Barbs;iBarbDex+=1)
			String strBarb_Icon_Name=KML_Wind_Make_Barb_Icon(twBarb_List, Wind_Speed_KML[iBarbDex*vWind_Resolution], strBarb_Icon_Folder_Path, strBarb_Icon_Base_Name, vBarb_Length, vBarb_Thick)
			fprintf refnum, "	<Placemark>\r"
			
			IF(vTime_Flag)
				fprintf refNum, "		<TimeStamp><when>"+TimeStamp_GE_Format_KML[iBarbDex*vWind_Resolution]+"</when></TimeStamp>\r"
			ENDIF
			
			fprintf refnum, "		<Style>\r"
			fprintf refnum, "			<IconStyle>\r"
			fprintf refnum, "				<Icon><href>"+strBarb_Icon_Folder_Name+"/"+strBarb_Icon_Name+".png</href></Icon>\r"
			fprintf refnum, "				<scale>"+num2str(vBarb_Scale)+"</scale>\r"
			fprintf refnum, "				<heading>%0.8f</heading>\r",Wind_Direction_KML[iBarbDex*vWind_Resolution]
			fprintf refnum, "				<color>"+strBarb_Color+"</color>\r"
			fprintf refnum, "			</IconStyle>\r"
			fprintf refnum, "		</Style>\r"
			fprintf refnum, "		<Point>\r"
			fprintf refnum, "			<coordinates>%0.8f,%0.8f,%0.8f</coordinates>\r", Longitude_Wave_KML[iBarbDex*vWind_Resolution], Latitude_Wave_KML[iBarbDex*vWind_Resolution], Altitude_Wave_KML[iBarbDex*vWind_Resolution]
			IF(vAltitude_Flag)
				fprintf refNum, "			<altitudeMode>absolute</altitudeMode>\r"
			ELSE
				fprintf refNum, "			<altitudeMode>relativeToGround</altitudeMode>\r"
			ENDIF
			fprintf refnum, "		</Point>\r"
			fprintf refnum, "	</Placemark>\r"
		ENDFOR
		
		KillPath/Z pBarb_Icon_Path
	ENDIF
	
	IF(vColor_Scale)
		fprintf refnum, "	<ScreenOverlay>\r"
		fprintf refnum, "		<name>Scale</name>\r"
		fprintf refnum, "		<Icon> <href>"+RemoveEnding(strFile_Name, ".kml")+"_Color_Scale.png"+"</href></Icon>\r"
		fprintf refnum, "		<overlayXY x=\"0\" y=\"0\" xunits=\"fraction\" yunits=\"fraction\"/>\r"
		fprintf refnum, "		<screenXY x=\"0\" y=\"0\" xunits=\"pixels\" yunits=\"pixels\"/>\r"
		fprintf refnum, "		<rotationXY x=\"0.5\" y=\"0.5\" xunits=\"fraction\" yunits=\"fraction\"/>\r"
		fprintf refnum, "		<size x=\"0\" y=\"0\" xunits=\"pixels\" yunits=\"pixels\"/>\r"
		fprintf refnum, "	</ScreenOverlay>\r"
	ENDIF

	fprintf refNum, "</Document>\r</kml>\r"
	Close refNum
	
	KillPath/Z pKML_Path
	
	Return 0

END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


FUNCTION/S KML_Wind_Make_Barb_Icon(WAVE/T twBarb_List, VARIABLE vSpeed_m_s, STRING strBarb_Folder_Path, STRING strBase_Name, VARIABLE vBarb_Length, VARIABLE vBarb_Thick)
	
	Variable vSpeed_knots=vSpeed_m_s*1.95
	Variable vSpeed_Bin=5*round(vSpeed_knots/5)
	
	String strBarb_Name
	sprintf strBarb_Name, "%03d", vSpeed_Bin
	
	String strBarb_Pic_Name=strBase_Name+"_Barb"+strBarb_Name
	FindValue/TEXT=(strBarb_Pic_Name)/TXOP=4 twBarb_List
	
	IF(V_Value<0)
		String strBarb_Graph_Name=UniqueName("Dummy_Graph_For_Barb_", 6, 0)
	
		NewPath/O/Q/Z pBarb_Path, strBarb_Folder_Path
	
		Redimension/N=(numpnts(twBarb_List)+1) twBarb_List
		twBarb_List[numpnts(twBarb_List)-1]=strBarb_Pic_Name
		Sort/A=2 twBarb_List, twBarb_List
		
		Make/O/D/N=1 Dummy_Temp_Wave_Lat=0, Dummy_Temp_Wave_Lon=0
		Make/O/D/N=(1,3) Dummy_Temp_Wave_Barb
		SetDimLabel 1,2,windBarb,Dummy_Temp_Wave_Barb
		Dummy_Temp_Wave_Barb[0][0]=10*vBarb_Length
		Dummy_Temp_Wave_Barb[0][1]=pi/2			//Make the barb for northerly wind, and then let Google Earth handle the rotation
		Dummy_Temp_Wave_Barb[0][2]=vSpeed_Bin/5
		
		Display /W=(353.25,203,487.5,461.75)/N=$strBarb_Graph_Name/K=1 Dummy_Temp_Wave_Lat vs Dummy_Temp_Wave_Lon as strBarb_Graph_Name
		ModifyGraph margin(left)=1,margin(bottom)=1,margin(top)=1,margin(right)=1
		ModifyGraph wbRGB=(0,0,0,0)
		ModifyGraph gbRGB=(0,0,0,0)
		ModifyGraph mode=3
		ModifyGraph rgb=(0,0,0)
		ModifyGraph arrowMarker(Dummy_Temp_Wave_Lat)={Dummy_Temp_Wave_Barb,vBarb_Thick,10,1,0}
		ModifyGraph standoff=0
		ModifyGraph notation=1
		ModifyGraph axRGB=(65535,65535,65535)
		ModifyGraph tlblRGB=(65535,65535,65535)
		ModifyGraph alblRGB=(65535,65535,65535)
		ModifyGraph axisOnTop=1
		ModifyGraph btLen=4
		Label bottom " "
	
		DoUpdate/W=$strBarb_Graph_Name

		SavePICT/O/E=-5/B=288/WIN=$strBarb_Graph_Name/P=pBarb_Path as strBarb_Pic_Name+".png"		//Save the graph with the marker.  Use the eventual name for the icon so we don't have to kill the file later.
		KillWindow/Z $strBarb_Graph_Name		//Kill the graph
		
		ImageLoad/T=png/O/Q/N=Dummy_Barb_Matrix_Wave/P=pBarb_Path strBarb_Pic_Name+".png"		//Load the image as a matrix
		
		WAVE/Z Dummy_Barb_Matrix_Wave
		Make/O/D/FREE/N=(DimSize(Dummy_Barb_Matrix_Wave,0), DimSize(Dummy_Barb_Matrix_Wave,1), 4) mBarb			//Convert the image matrix to the proper form (white inside the marker, transparent outside)
		mBarb[][][0]=Dummy_Barb_Matrix_Wave[p][q][0]>100 ? 0 : 65535			//I probably could use layers 1 and 2 as well, but this works.  
		mBarb[][][1]=Dummy_Barb_Matrix_Wave[p][q][0]>100 ? 0 : 65535
		mBarb[][][2]=Dummy_Barb_Matrix_Wave[p][q][0]>100 ? 0 : 65535
		mBarb[][][3]=Dummy_Barb_Matrix_Wave[p][q][0]>100 ? 0 : 65535

		ImageSave/T="png"/O/P=pBarb_Path mBarb as strBarb_Pic_Name+".png"		//Save the wave as an image.  Overwrite instead of killing the file.
		
		KillWaves/Z Dummy_Barb_Matrix_Wave, Dummy_Temp_Wave_Lat, Dummy_Temp_Wave_Lon, Dummy_Temp_Wave_Barb
		KillPath/Z pBarb_Path
	
	ENDIF
	
	Return strBarb_Pic_Name
	
END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Converts the three 16-bit color values to 8 bit and then to one hex code in Google Earth format (BGR)
FUNCTION/S KML_Convert_RGB_to_Hex(VARIABLE vRed, VARIABLE vGreen, VARIABLE vBlue)

	Variable vRed_8bit=round(((vRed-1)/256)-1)		//Negative numbers don't convert nicely, but non-integers are fine.  Going to round anyways.
	Variable vGreen_8bit=round(((vGreen-1)/256)-1)
	Variable vBlue_8bit=round(((vBlue-1)/256)-1)
	
	String strHex_Color_Red, strHex_Color_Green, strHex_Color_Blue
	sprintf strHex_Color_Red, "%02X", SelectNumber(vRed_8bit<0, vRed_8bit, 0)	//GE uses 256 as full color value. Igor uses 65535
	sprintf strHex_Color_Green, "%02X",SelectNumber(vGreen_8bit<0, vGreen_8bit, 0)	//uses 2 characters if 0
	sprintf strHex_Color_Blue, "%02X", SelectNumber(vBlue_8bit<0, vBlue_8bit, 0)
	String strHex_Color = "ff"+strHex_Color_Blue+strHex_Color_Green+strHex_Color_Red	//GE uses BGR hex color	
	
	Return strHex_Color

END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


//Computes the straight line distance through the Earth between two points.  Would probably be better to use the Haversine equation to get the distance
FUNCTION KML_Get_Bounds(VARIABLE vLat1, VARIABLE vLon1, VARIABLE vLat2, VARIABLE vLon2)

	variable vEarth_rad = 6373e3 // meters
	Variable vRad_Conversion=pi/180

	Variable vX1=vEarth_rad*cos(vLat1*vRad_Conversion)*cos(vLon1*vRad_Conversion)
	Variable vY1=vEarth_rad*cos(vLat1*vRad_Conversion)*sin(vLon1*vRad_Conversion)
	Variable vZ1=vEarth_rad*sin(vLat1*vRad_Conversion)
	
	Variable vX2=vEarth_rad*cos(vLat2*vRad_Conversion)*cos(vLon2*vRad_Conversion)
	Variable vY2=vEarth_rad*cos(vLat2*vRad_Conversion)*sin(vLon2*vRad_Conversion)
	Variable vZ2=vEarth_rad*sin(vLat2*vRad_Conversion)
	
	Variable vDistance=sqrt((vX2-vX1)^2+(vY2-vY1)^2+(vZ2-vZ1)^2)
	
	Return vDistance

END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


FUNCTION/S KML_Cleanup_File_Name(STRING strOld_File_Name)

	String strNew_File_Name=strOld_File_Name

	IF(StringMatch(strOld_File_Name, " *"))
		DO
			strNew_File_Name=strNew_File_Name[1,strlen(strNew_File_Name)-1]

		WHILE(StringMatch(strNew_File_Name, " *") && strlen(strNew_File_Name)>0)
	ENDIF

	Return strNew_File_Name

END


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
