// LogBook, J J Weimer
#pragma rtGlobals=3
#pragma rtFunctionErrors=1
#pragma IgorVersion=8.00
#pragma TextEncoding="UTF-8"
#pragma hide=1

// Project Updater
Static Constant kProjectID=2352
Static Strconstant ksShortTitle="LogBook"

// Procedure
#pragma IndependentModule=LogBook
#pragma version=4.50

Static StrConstant thePackageFolder="root:Packages:LogBook"
Static Constant thePackageVersion=4.50

Static StrConstant kPanelName = "LogBookPanel"
Static StrConstant kPanelTitle = "LogBook Panel"

// panel width/height constants

Static Constant lbpwidth=300
Static Constant lbpheight=425
Static Constant lbpswidth=45
Static Constant lbpsheight=45

// *** MENU

Menu "Windows", hideable
	"LogBook Panel", /Q, show_LBPanel()
End

// *** ENTRY FUNCTIONS

// @brief clear and reset panel cache settings when an experiment starts
Function AfterFileOpenHook(variable refNum,string file, string pathName, string type, string creator, variable kind)
	
	// only do the rest when the package folder exists
	if (!DataFolderExists(thePackageFolder))
		return 0
	endif
	
	DFREF pdf = $thePackageFolder
	DFREF gdf = pdf:PanelGlobals
	
	NVAR/SDFR=gdf hrefnum
	SVAR/SDFR=gdf history

	// panel displayed -- update package if needed
	if (WinType(kPanelName)==7)
		if (update_tocurrentversion() == 1)
			KillWindow/Z $kPanelName
			create_LBPanel()
		endif
	endif

	// reset the history	
	if (NVAR_exists(hrefnum))
		hrefnum = nan
		history = ""
	endif
	return 0
end

// @brief menu entry point to show panel
Function show_LBPanel()
	
	variable updated = 0

	// create or update the package folder?
	if (!DataFolderExists(thePackageFolder))		
		setup_LBPackage()
		updated = 1
	else
		updated = update_tocurrentversion()
	endif
	
	// show or create the panel?	
	DoWindow/F $kPanelName
	if (v_flag == 1)
		if (updated == 1)
			KillWindow/Z $kPanelName
			create_LBpanel()
		endif
	else
		create_LBpanel()	
	endif
	
	return 0
end

// *** CREATE PANEL

// @brief set up logbook package
Static Function setup_LBPackage()
		
	// first time set up here
	DFREF cdf = GetDataFolderDFR()
	SetDataFolder root:
	NewDataFolder/O/S Packages
	NewDataFolder/O/S LogBook
	variable/G version=thePackageVersion
	make/T/N=1 graphiclist = ""
	make/N=1 graphicsel = 32
	NewDataFolder/O/S PanelGlobals
	variable/G notebk = 1
	variable/G size = 100
	variable/G hrefnum = nan
	string/G recentdate = ""
	string/G title = "Notes"
	string/G notes = ""
	string/G history = ""
	SetDataFolder cdf
	
	return 0
end

// @brief update to current version
Static Function update_toCurrentVersion()

	DFREF pdf = $thePackageFolder
	DFREF gdf = pdf:PanelGlobals
	
	string gList
	variable nitems
	variable vrtn=0

	// update to 4+
	NVAR/Z/SDFR=pdf version
	if (!NVAR_Exists(version))
		DFREF cdf = GetDataFolderDFR()
		SetDataFolder pdf
		gList = f_LoGraphics()
		nitems = ItemsInList(gList)
		wave/T w = ListToTextWave(gList,";")
		variable/G version = thePackageVersion
		duplicate w graphiclist
		make/N=(nitems) graphicsel=32
		SetDataFolder gdf
		KillVariables/Z odynamic, template, graphic
		SetDataFolder cdf
		print date(), " updated LogBook package to version ", thePackageVersion
		vrtn = 1
	else
		// update to all others
		if (version < 4.20)
			DFREF cdf = GetDataFolderDFR()
			SetDataFolder gdf
			killvariables/Z frame, ndynamic, omode, xsize, ysize
			killstrings/Z graphicname
			SetDataFolder cdf
		endif
		if (version < 4.50)
			DFREF cdf = GetDataFolderDFR()
			SetDataFolder root:
			killwaves/Z graphicsel, graphiclist
			SetDataFolder cdf
		endif
		if (version != thePackageVersion)
			version = thePackageVersion
			print date(), " updated LogBook package to version ", thePackageVersion
			vrtn = 1
		endif
	endif
	return vrtn
end

// @brief create the logbook panel
Static Function create_LBPanel()
	
	string ptitle = kPanelTitle

	NewPanel/W=(20,80,20+lbpwidth,80+lbpheight)/N=$kPanelName/K=1 as ptitle
	ModifyPanel/W=$kPanelName noEdit=1, fixedSize=1

	// panel controls
	
	put_PanelControls()
	
	// panel hook function
	SetWindow $kPanelName hook(LogBook)=panelhook
	
	// reset history record	
	startstop_History(0)
	
	return 0
end

// @brief put controls on panel
Static Function put_PanelControls()
	
	DFREF pdf = $thePackageFolder
	DFREF gdf = pdf:PanelGlobals
	
	NVAR/SDFR=gdf size, notebk //, xsize, ysize, omode, ndynamic, frame
	SVAR/SDFR=gdf title, notes // graphicname
	
	wave/T/SDFR=pdf tw = graphiclist
	wave/SDFR=pdf sw = graphicsel
	
	string tstr

	variable toffset = 4

	Button tb0_iconize, pos={10,toffset}, size={30,20}, fSize=14, title="[ ]", userData(state)="full"
	Button tb0_iconize, proc=setbuttons, help={"Shrink the panel to an icon.\rRegrow the icon to a full panel."}

	CheckBox tb0_cbnbmd pos={round(0.45*lbpwidth),toffset+2}, fSize=10, title="MD", value=1
	CheckBox tb0_cbnbmd help={"Use MarkDown format for all headers in the notebook."}
	CheckBox tb0_cbdosubs pos={round(0.45*lbpwidth+45),toffset+2}, fSize=10, title="3+", value=1
	CheckBox tb0_cbdosubs help={"Use 3+ level (sub)headers"}

	// help and settings
		
	Button tb0_help, pos={lbpwidth-45,toffset+2}, size={15,15}, title="", Picture=LogBook#pLB_Help, labelBack=0
	Button tb0_help, proc=setbuttons, help={"Click for help"}, focusRing=0
	Button tb0_settings, pos={lbpwidth-25,toffset+2}, size={15,15}, title="", Picture=LogBook#pLB_Cog, labelBack=0
	Button tb0_settings, Proc=setbuttons, help={"Do additional actions"}

	// new page setting

	toffset += 26
	
	SetDrawLayer UserBack
	SetDrawEnv fsize=9, dash=2
	DrawLine 10,toffset-3,lbpwidth-10,toffset-3

	PopupMenu popup_startpage, pos={5,toffset}, size={180,20}, title="Break", value="None;Heading;Line;New Page;", fSize=12, proc=setpopups
	PopupMenu popup_startpage, help={"Define the break input before the contents are added to the notebook."}
	SetVariable tb0_head, pos={130,toffset}, size={150,20}, title=" ", fSize=12, help={"Insert a heading here."}, value=_STR:"Experiment Information", disable=1
	
	// title block
	
	toffset += 23
	
	CheckBox tb0_cbtitle, pos={10,toffset}, size={40,20}, title="Title", value=1, fSize=12, proc=setcheckboxes
	CheckBox tb0_cbtitle, help={"Check to include a title for the notes."}
	SetVariable tb0_title, pos={60,toffset}, size={lbpwidth-100,20}, value=title, title=" ", fSize=12, proc=setvariables
	SetVariable tb0_title,help={"Insert a title here."}

	// notes block
	
	toffset += 22

	NewNotebook/F=1/N=notefield/W=(6,toffset,lbpwidth-6,toffset+93)/HOST=#/OPTS=(2)
	Notebook $(kPanelName)#notefield, statusWidth=0, autoSave=0, fSize=14, fStyle=0, textRGB=(0,0,0)
	
	toffset += 93

	CheckBox tb0_notefuncs,pos={10,toffset+2},size={42,14},title="functions in comments",value=0, fSize=10
	CheckBox tb0_notefuncs,help={"Check this box to allow function inputs in comment field.\rEnter functions with % or # prefix."}
	Button tb0_clearnotes,pos={lbpwidth-102,toffset},size={20,20}, title="", picture=LogBook#pLB_Eraser, proc=setbuttons, labelBack=0
	Button tb0_clearnotes,help={"Clear the comment field."}, disable=2
	CheckBox tb0_autoclear,pos={lbpwidth-72,toffset},size={42,14},title="auto clear",value=1, fSize=10, proc=setcheckboxes
	CheckBox tb0_autoclear,help={"Auto-clear the comment field each time \r button is clicked or Shift+Return is entered."}

	// graphic block
	
	toffset += 30
	
	SetDrawLayer UserBack
	SetDrawEnv fsize=9, dash=2
	DrawLine 10,toffset-3,lbpwidth-10,toffset-3

	PopupMenu tb0_graphic,pos={10,toffset},size={170,21},fsize=12,title="Graphics",proc=setpopups
	PopupMenu tb0_graphic,mode=1,value= #"LogBook#f_GraphicsMenu()"
	PopupMenu tb0_graphic,help={"This is a selection list for what graphics to include in the notebook."}

	Checkbox cb_titles, pos={162,toffset}, title="titles", fSize=10, font="Arial", value=0, proc=setcheckboxes
	Checkbox cb_titles, help={"List by titles rather than by names?"}
	
	Checkbox cb_wpicts, pos={220,toffset}, title="pictures?", fSize=10, font="Arial", value=0, proc=setcheckboxes, disable=0
	Checkbox cb_wpicts, help={"Include stored picures in All graphics and Select list?"}

	Checkbox cb_bffs, pos={190,toffset+20}, title="Fullscale @ 6.5\"", fSize=10, font="Arial", value=0, disable=1
	Checkbox cb_bffs, help={"Image widths are scaled using actual size.\rCheck this to scale image widths by 6.5 inches fullscale."}
	
	toffset += 20

	ListBox tb0_listbox pos={15,toffset}, listWave=tw, selWave=sw, size={165,92}, fSize=11, proc=SetListBox, userData(AllOn)="0", disable=1
	Setvariable tb0_frontmost pos={15,toffset}, value=_STR:"", size={140,20}, fSize=12, noedit=1, disable=1
	Checkbox tb0_WithSubWindow pos={15,toffset+25}, value=1, size={60,20}, fSize=12, title="with external subwindows", disable=1
	
	Slider tb0_scale,pos={190,toffset+20},size={100,45},limits={5,100,5},variable=size,vert=0, proc=setsliders, disable=1
	Slider tb0_scale,help={"Define the image scale size."}

	ValDisplay tb0_showscale,pos={190,toffset+45},size={25,15}, value=#"root:Packages:LogBook:PanelGlobals:size", disable=1

	// history block
	
	toffset += 102
	
	SetDrawLayer UserBack
	SetDrawEnv fsize=9, dash=2
	DrawLine 10,toffset-4,lbpwidth-10,toffset-4

	CheckBox tb0_cbhistory,pos={10,toffset},size={40,20},title="History ", value=0, fSize=12, font="Arial", proc=setcheckboxes
	CheckBox tb0_cbhistory,help={"Record the history window to the notebook."}

	CheckBox tb0_cbhistory0,pos={80,toffset},size={46,20},title="... <+> ...", value=1, mode=1,fSize=12, font="Arial", disable=2, proc=setcheckboxes
	CheckBox tb0_cbhistory0,help={"Record the history in one shot. Starts at button on click and ends with a log event."}
	CheckBox tb0_cbhistory1,pos={160,toffset},size={46,20},title="<... + ...>", value=0, mode=1,fSize=12, font="Arial", disable=2, proc=setcheckboxes
	CheckBox tb0_cbhistory1,help={"Record the history continuously. Starts at button on click and ends at button off/checkbox disable click."}
	
	// data folder files
	
	toffset += 20
	
	CheckBox tb0_cbdbrecord,pos={10,toffset},size={46,22},title="Folder Dump", value=0, fSize=12, font="Arial", proc=setcheckboxes
	CheckBox tb0_cbdbrecord, help={"Check this to include a directory listing from the current directory."}
	PopupMenu tb0_lodf,pos={100,toffset-2},size={165,21},mode=1,value= #"LogBook#f_LoDataFldrs()",title="", disable=2
	PopupMenu tb0_lodf,help={"Choose which data folder to print a directory listing."}
	
	// which notebook
	
	toffset += 29

	PopupMenu tb0_notebook,pos={10,toffset},size={165,21},bodywidth=0
	PopupMenu tb0_notebook,mode=notebk,value=#"LogBook#f_LoNtbks()",title="To:", fSize=12, proc=setpopups
	PopupMenu tb0_notebook,help={"Choose which notebook is used to log the information."}
	
	// send to notebook block

	PopupMenu tb0_location,pos={lbpwidth-135,lbpheight-30},size={152,25},title="@"
	PopupMenu tb0_location,mode=5,value="|<<;<;!;>;>>|;", fSize=12
	PopupMenu tb0_location,help={"Insert information at start, before current position, at current position,\rafter current position, or at end of notebook."}
	
	Button tb_noteit,pos={lbpwidth-60,lbpheight-30},size={40,20},title="->",fColor=(16385,28398,65535)
	Button tb_noteit fSize=12,fStyle=1, proc=setbuttons
	Button tb_noteit, help={"Send the information to the notebook."}

	// version and dashed line
	
	string theStr
	
	sprintf theStr, "v %3.2f", thePackageVersion
	
	SetDrawEnv fsize=9
	DrawText 5,lbpheight-5, theStr
	
	SetDrawLayer UserBack
	SetDrawEnv fsize=9
	DrawText lbpwidth,12,theStr
	SetDrawEnv dash=2
	DrawLine 10,lbpheight-60,lbpwidth-10,lbpheight-60

	return 0
end

// @brief Panel Hook Function
// update automatic settings when the panel is selected
Function PanelHook(STRUCT WMWinHookStruct &whs)
	
	DFREF pdf = $thePackageFolder
	DFREF gdf = pdf:PanelGlobals

	NVAR/SDFR=gdf notebk, hrefnum
	
	string mtxt = ""
	variable et
	variable rtnV = 0
	
	switch(whs.eventCode)
		case 0:	// activate
			//	print "panel active ", mtxt, odynamic, graphicname
			// dynamic notebook selection
			et = ItemsInList(f_LoNtbks())
//			if (ndynamic)
			if (et == 1)
				mtxt = "New"
				notebk = 1
			else
				mtxt = WinName(0,16)
				et = WhichListItem(mtxt,f_LoNtbks())+1
				notebk = et
			endif
			PopupMenu tb0_notebook,  win=$kPanelName, popValue=mtxt, value=#"LogBook#f_LoNtbks()", mode=et				
//			endif

			// dynamic graphic selection
			ControlInfo/W=$kPanelName tb0_graphic
//			print "active ", whs.eventCode, v_value
			switch(v_value)
				case 1:		// none
					break
				case 2:		// frontmost
					Setvariable tb0_frontmost, value=_STR:f_FrontMostWin()
					break
				case 3:		// all
					PopupMenu tb0_graphic, win=$kPanelName, value=#"LogBook#f_GraphicsMenu()"
					break
				case 4:		// select
					update_ListWaves()
					break
				case 5:		// clipboard
					
					break
				default:		// pictures
					PopupMenu tb0_graphic, win=$kPanelName, value=#"LogBook#f_GraphicsMenu()"
					break
			endswitch
			rtnV = 1
			break
		
		case 17:	// kill vote
			hrefnum = nan
			Checkbox tb0_cbhistory,  win=$kPanelName, value = 0
			Checkbox tb0_cbhistory0,  win=$kPanelName, disable = 2
			Checkbox tb0_cbhistory1,  win=$kPanelName, disable = 2
			rtnV = 1
			break
	
// @brief mac keyboard input block
#ifdef Macintosh
		case 11:	// keyboard input
			et = GetKeyState(0)
//			print "Mac keyboard input"
//			print et, whs.keycode
			// shift key?
			if (et&4)
				switch(whs.keycode)
					case 13:	// return key too
						Log_ToNotebook()
						rtnV = 1
						break
					default:
						break
				endswitch
			elseif(whs.keycode == 27) // escape key
				SetVariable tb0_title, activate
				rtnV = 1
			endif
			break
#endif // MACINTOSH

// @brief windows keyboard input block
#ifdef Windows
		case 11:
			et = GetKeyState(0)
//			print "Windows keyboard input"
//			print et, whs.keycode
			if ((et&4) && (et&1))
				switch(whs.keycode)
					case 187:	// = key Windows
						Log_ToNotebook()
						rtnV = 1
						break
					default:
						break
				endswitch
			endif
			break
#endif // WINDOWS
		default:
			break
	endswitch
	
//	print "panel hook function ", whs.eventCode, whs.keycode, rtnV
	
	return rtnV
end

// *** CONTROL FUNCTIONS

// @brief checkbox control
Function SetCheckBoxes(STRUCT WMCheckboxAction &cba) : CheckBoxControl
	
	switch( cba.eventCode )
		case 2: // mouse up
			strswitch(cba.ctrlName)
				case "tb0_cbtitle":
					SetVariable tb0_title, win=$kPanelName, disable=!cba.checked
					break
				case "tb0_autoclear":
					Button tb0_clearnotes, win=$kPanelName, disable=(2*cba.checked)
					break
				case "tb0_cbhistory":
					CheckBox tb0_cbhistory0, win=$kPanelName, disable=!cba.checked*2
					CheckBox tb0_cbhistory1, win=$kPanelName, disable=!cba.checked*2
					startstop_History(cba.checked)
					break
				case "tb0_cbhistory0":		// incremental
					CheckBox tb0_cbhistory1, value=!cba.checked
					startstop_History(2)
					break
				case "tb0_cbhistory1":		// accumulated
					CheckBox tb0_cbhistory0, win=$kPanelName, value=!cba.checked
					break
				case "tb0_cbdbrecord":		// folder dump
					PopupMenu tb0_lodf, win=$kPanelName, disable=((!cba.checked)*2)
					break
				case "cb_titles":	// fallthrough
				case "cb_wpicts":	// fallthrough
					ControlInfo/W=$kPanelName tb0_graphic
					switch(v_value)
						case 2:		// frontmost
							Setvariable tb0_frontmost, win=$kPanelName, value=_STR:f_FrontMostWin(), disable = 0
							break
						case 4:
							update_ListWaves()
							break
						default:
							break
					endswitch
					break
				default:
					break
			endswitch
			break
		default:
			break
	endswitch

	return 0
end

// @brief popup menu control
Function SetPopUps(STRUCT WMPopupAction &pa) : PopupMenuControl
		
	switch( pa.eventCode )
		case 2:
			strswitch( pa.ctrlName )
				// graphic selection
				case "popup_startpage":
					SetVariable tb0_head, win=$kPanelName, disable=(pa.popNum!=2)
					break
				case "tb0_graphic":
					update_PanelforGraphics(pa.popNum)
					break
				default:
					break
			endswitch
			break
		default:
			break
	endswitch
	return 0
end

// @brief listbox control
Function SetListBox(STRUCT WMListboxAction &lba) : ListBoxControl

	Variable row = lba.row
	Variable col = lba.col
	WAVE/T/Z listWave = lba.listWave
	WAVE/Z selWave = lba.selWave

	switch( lba.eventCode )
		case 13: // checkbox clicked (Igor 6.2 or later)
			update_SelectedGraphics()
			break
		default:
			break
	endswitch

	return 0
end

// @brief slider control
Function SetSliders(STRUCT WMSliderAction &sa) : SliderControl
	
	DFREF pdf = $thePackageFolder
	DFREF gdf = pdf:PanelGlobals
	
	NVAR/SDFR=gdf size

	switch( sa.eventCode )
		case -1: // control being killed
			break
		default:
			strswitch(sa.ctrlName)
				case "tb0_scale":
					size = sa.curval
					break
				default:
					break
			endswitch
			break
	endswitch

	return 0
end

// @brief set variable control
Function SetVariables(STRUCT WMSetVariableAction &sva) : SetVariableControl
	
	variable getout = 0

	switch( sva.eventCode )
		case 1: // fallthrough mouse up
		case 3: // fallthrough Live update
			strswitch(sva.ctrlName)

				// title - tab or return moves to and activates note field
				
				case "tb0_title":
					break
				default:
					break
			endswitch
			break
		case 8:
			SetActiveSubwindow #notefield
			break
		default:
			break
	endswitch
	
	return 0
end

// @brief button control
Function SetButtons(STRUCT WMButtonAction &ba) : ButtonControl
	
	switch( ba.eventCode )
		case 2:
			strswitch(ba.CtrlName)
				case "tb0_settings":
					show_NtbkSettingsMenu()
					break
				case "tb0_clearnotes":
					Notebook $(kPanelName)#notefield selection={startOfFile,endofFile}
					Notebook $(kPanelName)#notefield text=""
					break
				case "tb_noteit":
					Log_ToNotebook()
					break
				case "tb0_help":
					show_NtbkHelpMenu()
					break
				case "tb0_iconize":
					iconize_Panel()
					break
				case "tb_cleanup":
					ControlInfo/W=$kPanelName tb0_notebook
					AutoCorrect_ImageNumbering(s_value)
					break
				default:
					break
			endswitch
			break
		default:
			break
	endswitch
	
	return 0
end

// *** ADDITIONAL FUNCTIONS

// @brief parse for window title in an external sub window
Static Function/S parse_forTitle(string fullwinpath)

	string rtstr = ""
	string rexp = "New.*HOST=#.*  as \"([^\"]*)"
	SplitString/E=(rexp) WinRecreation(fullwinpath,0), rtstr
	if (strlen(rtstr) == 0)
		rtstr = fullwinpath
	endif
	return rtstr
end

// @brief called when switching popup for graphics
// @input mode 1 - none, 2 - frontmost, 3 - all, 4 - select
Static Function update_PanelforGraphics(variable mode)

	Checkbox tb0_WithSubWindow, win=$kPanelName, disable=1	
	switch(mode)
		case 1:		// none
			ListBox tb0_listbox, win=$kPanelName, disable=1
			Setvariable tb0_frontmost, win=$kPanelName, disable=1
			Slider tb0_scale, win=$kPanelName, disable=1
			Checkbox cb_titles, win=$kPanelName, disable=2
			Checkbox cb_bffs, win=$kPanelName, disable=1
			Checkbox cb_wpicts, win=$kPanelName, disable=0
			ValDisplay tb0_showscale, win=$kPanelName, disable=1
			break
		case 2:		// frontmost
			ListBox tb0_listbox, win=$kPanelName, disable=1
			Setvariable tb0_frontmost, win=$kPanelName, value=_STR:f_FrontMostWin(), disable=0
			Slider tb0_scale, win=$kPanelName, disable=0
			Checkbox cb_titles, win=$kPanelName, disable=0
			Checkbox cb_bffs, win=$kPanelName, disable=0
			Checkbox cb_wpicts, win=$kPanelName, disable=2
			ValDisplay tb0_showscale, win=$kPanelName, disable=0
			break
		case 3:		// all
			ListBox tb0_listbox, win=$kPanelName, disable=1
			Setvariable tb0_frontmost, win=$kPanelName, disable=1
			Slider tb0_scale, win=$kPanelName, disable=0
			Checkbox cb_titles, win=$kPanelName, disable=2
			Checkbox cb_bffs, win=$kPanelName, disable=0
			Checkbox cb_wpicts, win=$kPanelName, disable=2
			ValDisplay tb0_showscale, win=$kPanelName, disable=0
			break
		case 4:		// select
			ListBox tb0_listbox, win=$kPanelName, disable = 0
			Setvariable tb0_frontmost, win=$kPanelName, disable = 1
			Slider tb0_scale, win=$kPanelName, disable = 0
			Checkbox cb_titles, win=$kPanelName, disable=0
			Checkbox cb_bffs, win=$kPanelName, disable = 0
			Checkbox cb_wpicts, win=$kPanelName, disable=0
			ValDisplay tb0_showscale, win=$kPanelName, disable = 0
			update_ListWaves()
			break
		default:		// picture or clipboard
			ListBox tb0_listbox, win=$kPanelName, disable=1
			Setvariable tb0_frontmost, win=$kPanelName, disable=1
			Slider tb0_scale, win=$kPanelName, disable=0
			Checkbox cb_titles, win=$kPanelName, disable=0
			Checkbox cb_bffs, win=$kPanelName, disable=0
			Checkbox cb_wpicts, win=$kPanelName, disable=2
			ValDisplay tb0_showscale, win=$kPanelName, disable=0
			break
	endswitch
	return 0
end

// @brief called when action occurs in listbox
Static Function update_SelectedGraphics()
	
	DFREF pdf = $thePackageFolder
	
	wave/T/SDFR=pdf tw = graphiclist
	wave/SDFR=pdf sw = graphicsel
	
	variable onoff
	
	onoff = str2num(GetUserData(kPanelName,"tb0_listbox","AllOn"))
	switch(onoff)
		case 0:
			if (sw[0] != 32)
				sw = 48
				ListBox tb0_listbox, win=$kPanelName, userData(AllOn) = "1"
			endif
			break
		case 1:
			if (sw[0] != 48)
				sw = 32
				ListBox tb0_listbox, win=$kPanelName, userData(AllOn) = "0"
			endif
			break
		default:
			break
	endswitch
	
	return 0
end

// @brief called when action occurs to change listbox
Static Function update_ListWaves()
	
	DFREF pdf = $thePackageFolder
	wave/T/SDFR=pdf tw = graphiclist
	wave/SDFR=pdf sw = graphicsel
	
	variable ic, nitems
	string theItem, dispList
	
	dispList = f_LoGraphics()
	nitems = ItemsInList(dispList)

	// 	only change list if items are added or removed
	wave/T w = ListtoTextWave("All (toggle);" + dispList,";")
	if (DimSize(tw,0) != (nitems+1))
		duplicate/O/T w tw
		redimension/N=(nitems+1) sw
		sw = 32
	else
		tw = w[p]
	endif
	
	return 0
end

// @brief make panel into icon or return to full
Static Function iconize_Panel()

	string state, ntbk
	variable vl, vt, nl, nt
	
	state = GetUserData(kPanelName,"tb0_iconize","state")
	GetWindow $kPanelName, wsize
	vl = v_left; vt = v_top
	ControlInfo/W=$kPanelName tb0_notebook
	if (v_value!=1)	
		ntbk = s_value
		GetWindow $ntbk, wsize
		nl = (v_left<30) ? 0 : (v_left-30)
		nt = v_top
		DoWindow/F $ntbk
	else
		nl = vl
		nt = vt
	endif
	strswitch(state)
		case "full":
			MoveWindow/W=$kPanelName nl, nt, nl + lbpswidth, nt + lbpsheight
			Button tb0_iconize, win=$kPanelName, userData(state)="icon", title="< >"
			break
		case "icon":
			MoveWindow/W=$kPanelName vl, vt, vl + lbpwidth, vt + lbpheight
			Button tb0_iconize, win=$kPanelName, userData(state)="full", title="[ ]"
			break
		default:
			break
	endswitch
	DoWindow/F $kPanelName
	return 0
end

// @brief toggle capture of history
// @input onoff for history
// 0 - stop history
// 1 - start history
// 2 - restart history
Static Function startstop_History(variable onoff)
	
	DFREF pdf = $thePackageFolder
	DFREF gdf = pdf:PanelGlobals
	
	NVAR/SDFR=gdf hrefnum
	SVAR/SDFR=gdf history

	switch( onoff )
		case 0:		// turn it off
			if (numtype(hrefnum)!=2)
				try
					history = CaptureHistory(hrefnum,1)
				catch
				endtry
				history = ""
				hrefnum = nan
			endif
			break
		default:		// turn it on
			if (numtype(hrefnum)==2)	//  --> not yet on
				history = ""
				hrefnum = CaptureHistoryStart()
			else
				if (onoff == 2)				// --> restart it
					try
						history = CaptureHistory(hrefnum,1)
					catch
					endtry
					history = ""
					hrefnum = CaptureHistoryStart()
				endif
			endif
			break
	endswitch

	return 0	
end

// @brief send information to notebook
Static Function log_ToNotebook()		

	DFREF pdf = $thePackageFolder
	DFREF gdf = pdf:PanelGlobals

	NVAR/SDFR=gdf notebk, hrefnum
	SVAR/SDFR=gdf title, notes, history, recentdate

	string theHeader = ""
	string theDTStr=""
	string thegraphic=""
	string theNotebook=""
	string md1, md2, md3
	string tstr, estr, pURL
	variable itsFormat=1
	variable at, newntbk, cObj, cHist, grtype, newpage, bffs
	variable sParagraph, sPosition, dosubs, useMD

	// get notebook and its format
	
	ControlInfo/W=$kPanelName tb0_notebook
	notebk = V_value	
	ControlInfo/W=$kPanelName tb0_cbdosubs
	dosubs = v_value	
	ControlInfo/W=$kPanelName tb0_cbnbmd
	useMD = v_value
	ControlInfo/W=$kPanelName cb_bffs
	bffs = v_value
	
	md1 = SelectString(useMD,"","# ")
	md2 = SelectString(useMD,"","## ")
	md3 = SelectString(useMD,"","### ")
	
	notes = ""

	if (notebk == 1)
		// create a new notebook
		tstr = "Log Notebook"
		prompt tstr, "Give a title for the new notebook:"
		doprompt "Create New Notebook", tstr
		if (v_flag)
			return -1
		endif
		theNotebook = cleanupname(tstr,0)
		DoWindow $theNotebook
		if (v_flag==1)
			theNotebook = UniqueName(theNotebook,10,0)
		endif
		NewNotebook/F=1/N=$theNotebook
		newntbk = 1
		Notebook $theNotebook, showRuler=0
	else
		// use an existing notebook
		ControlInfo/W=$kPanelName tb0_notebook
		theNotebook = S_value
		// store current selection point
		GetSelection notebook, $theNotebook, 1
		sParagraph = V_startParagraph; sPosition = V_startPos
		Notebook $theNotebook selection={startOfFile,endOfFile}
		Notebook $theNotebook getData=2
		if (strlen(S_value)==0)
			newntbk = 1
		else
			newntbk = 0
		endif		
		Notebook $theNotebook selection={(sParagraph,sPosition),(sParagraph,sPosition)}
	endif
	
	// create rulers
	Create_NtbkRulers(theNotebook)
	
	// if a new (or blank) notebook, put starting information
	if (newntbk)
	
		tstr = md1 + "Start of Notebook\r"
		Notebook $theNotebook ruler=Heading
		Notebook $theNotebook text = tstr
		
		tstr = "#date " + date() + "\r"
		Notebook $theNotebook ruler=Normal
		Notebook $theNotebook text = tstr
		
		tstr = md2 + "Experiment Information\r"
		Notebook $theNotebook ruler=SubHeading
		Notebook $theNotebook text = tstr
		
		estr = IgorInfo(1)
		PathInfo home
		tstr = ReplaceString(":",s_path,"/")
		tstr = RemoveListItem(0,tstr,"/")
		sprintf pURL "[%s](<file:///%s%s.pxp>)", estr, tstr, estr
		tstr = "Project Name: " + IgorInfo(1)
#if IgorVersion()>=9
		GetFileFolderInfo/Q/Z/P=home igorinfo(12)
		if(!V_flag)
			tstr += "\rCreation: " +  Secs2Date(V_creationDate,-2) +"T" + Secs2Time(V_creationDate,3)
			tstr += "\rModification: " + Secs2Date(V_modificationDate,-2) +"T" + Secs2Time(V_modificationDate,3)
		endif
#endif // IGOR VERSION
		tstr += "\rExperiment File: " + pURL + "\r"
		Notebook $theNotebook ruler=Normal
		Notebook $theNotebook text = tstr

		tstr = md2 + "User Information\r"
		Notebook $theNotebook ruler=SubHeading
		Notebook $theNotebook text = tstr
		
		tstr = "#author User Name: " + IgorInfo(7)
		tstr += "\rPlatform: " + IgorInfo(2)
		tstr += "\rIgor Version Information: " + IgorInfo(3)
		tstr += "\rIgorPro SN: " + IgorInfo(5) + "\r"
		Notebook $theNotebook ruler=Normal
		Notebook $theNotebook text = tstr

		tstr = md1 + "Experiment Notes\r"
		Notebook $theNotebook, specialChar={1,1,""}
		Notebook $theNotebook ruler=Heading
		Notebook $theNotebook text = tstr

	endif
	
	SplitString/E=("/F=./") WinRecreation(theNotebook,0)
	itsFormat = str2num(S_value[3])
		
	// move to where the stuff is to go
	ControlInfo/W=$kPanelName tb0_location
	at = V_value-1
	DoWindow/F $theNotebook
	switch(at)
		case 0: // start
			Notebook $theNotebook selection={startOfFile,startOfFile}
			break
		case 1: // previous paragraph
			Notebook $theNotebook selection={startOfPrevParagraph,startOfPrevParagraph}
			break		
		case 2: // here
			break
		case 3: // next paragraph
			Notebook $theNotebook selection={startOfNextParagraph,startOfNextParagraph}
			break
		case 4: // end
			Notebook $theNotebook selection={endOfFile,endOfFile}
			break
		default:
			break
	endswitch
	
	// new page
	ControlInfo/W=$kPanelName popup_startpage
	switch(v_value)
		case 1:		// nothing
			break
		case 2:		// header
			ControlInfo/W=$kPanelName tb0_head
			Notebook $theNotebook ruler=Heading
			tstr = md1 + s_value + "\r"
			Notebook $theNotebook text = tstr
			Notebook $theNotebook ruler=Normal
			break
		case 3:		// new line
			Notebook $theNotebook ruler=Normal
			Notebook $theNotebook text = "---\r"
			break
		case 4:		// new page
			Notebook $theNotebook ruler=Normal
			Notebook $theNotebook, specialChar={1,1,""}
			break
		default:
			break
	endswitch
	
	// title
	ControlInfo/W=$kPanelName tb0_cbtitle
	if (V_value)
		theHeader = md2 + title + "\r"
		Notebook $theNotebook ruler=SubHeading
		Notebook $theNotebook text=theHeader
	endif
	
	// date and time
	theDTStr = date()
	if (cmpstr(theDTStr,recentdate)!=0)
		recentdate = theDTStr
		theDTStr = "#date " + theDTStr + "\r"
	else
		theDTStr = ""
	endif
	theDTStr += "#time " + time() + "\r"
	Notebook $theNotebook ruler=Normal
	Notebook $theNotebook text = theDTStr
	
	// comments	
	if (dosubs)
		tstr = md3 + "Comments\r"
		Notebook $theNotebook ruler=SubSubheading
		Notebook $theNotebook text=tstr
	endif
	Notebook $(kPanelName)#notefield getData=2
	if (strlen(S_value)!=0)
		tstr = S_value
		// do comments have functions?	
		ControlInfo/W=$kPanelName tb0_notefuncs
		if (v_value)
			tstr = Parse_ForExecution(tstr)
		endif
		notes += tstr + "\r"
		Notebook $theNotebook ruler=Normal
		Notebook $theNotebook text=notes	
	endif
	// autoclear notes???
	ControlInfo/W=$kPanelName tb0_autoclear
	if (V_value)
		Notebook $(kPanelName)#notefield selection={startOfFile,endofFile}
		Notebook $(kPanelName)#notefield text=""
	endif			
	
	// graphics
	cObj = 1
	ControlInfo/W=$kPanelName tb0_graphic
	switch(v_value)
		case 1:	// none
			cObj = 0
			break
		case 2:	// frontmost
			thegraphic = f_FrontMostWin(full=1)
			break
		case 3:	// all
			thegraphic = f_LoGraphics(full=1)
			break
		case 4: // selected
			thegraphic = f_LoGraphicsSelected()
			if (strlen(thegraphic) == 0)
				cObj = 0
			endif
			break
		case 5:	// clipboard
			thegraphic = "CLIPBOARD"
			break
		default:	// picture
			thegraphic = s_value
			break
	endswitch
	
	switch(cObj)
		case 0:	// no graphic
			break				
		case 1: // graphics
			if (dosubs)
				tstr = md3 + "Figures\r"
				Notebook $theNotebook ruler=SubSubheading
				Notebook $theNotebook text=tstr
			endif
			Notebook $theNotebook ruler=Normal
			put_GraphicinNtbk(thegraphic,theNotebook,bffs,dosubs,useMD)
			break
		default:
			break
	endswitch
	
	// history
	ControlInfo/W=$kPanelName tb0_cbhistory
	if (V_value)
		// history is being captured
		ControlInfo/W=$kPanelName tb0_cbhistory0
		if (V_value==1)		// incremental history
			if (dosubs)
				history = md3 + "Incremental History\r"
				Notebook $theNotebook ruler=SubSubheading
				Notebook $theNotebook text = history
			endif
			history = CaptureHistory(hrefnum,0) + "\r"
			Notebook $theNotebook ruler=Normal
			Notebook $theNotebook text = history
			startstop_History(0)
			Checkbox/Z tb0_cbhistory, win=$kPanelName, value=0
			Checkbox/Z tb0_cbhistory0, win=$kPanelName, disable=2
			Checkbox/Z tb0_cbhistory1, win=$kPanelName, disable=2
		else					// ongoing history
			if (dosubs)
				history = md3 + "Accumulated History\r"
				Notebook $theNotebook ruler=SubSubheading
				Notebook $theNotebook text = history
			endif
			history = CaptureHistory(hrefnum,0)	 + "\r"
			Notebook $theNotebook ruler=Normal
			Notebook $theNotebook text = history
		endif
	else
		startstop_History(0)
	endif	
	
	// data directory dump	
	ControlInfo/W=$kPanelName tb0_cbdbrecord
	if (v_value)
		ControlInfo/W=$kPanelName tb0_lodf
		if (cmpstr(s_value,"root")!=0)
			DFREF dfr = $("root:" + s_value)
		else
			DFREF dfr = $("root:")
		endif
		if (dosubs)
			history = md3 + "Data Directory Dump\r"
			Notebook $theNotebook ruler=SubSubheading
			Notebook $theNotebook text = history
		endif
		history = "Data Folder => " + GetDataFolder(1,dfr) + "\r"
		history += DataFolderDir(-1,dfr) + "\r"
		Notebook $theNotebook ruler=Normal
		Notebook $theNotebook text = history
	endif
	
	return 0
end

// @brief create rulers in the notebook
Static Function create_NtbkRulers(string theNtbk)
	
	Notebook $theNtbk newRuler=Heading, spacing={12,6,0}, rulerDefaults={"",14,1,(0,0,0)}
	Notebook $theNtbk newRuler=Subheading, spacing={6,6,0}, rulerDefaults={"",12,1,(0,0,0)}
	Notebook $theNtbk newRuler=SubSubheading, spacing={6,6,0}, rulerDefaults={"",12,2,(0,0,0)}
	Notebook $theNtbk Ruler=Normal, spacing={0,0,0}, rulerDefaults={"",12,0,(0,0,0)}
	
	return 0
end

// @brief put graphics in the notebook
// thegraphic --> window
// theNtbk --> the notebook
// bffs --> scale by fixed full size
Static Function put_GraphicinNtbk(string theList, string theNtbk, variable bffs, variable dosubs, variable useMD)

	DFREF pdf = $thePackageFolder
	DFREF gdf = pdf:PanelGlobals
		
	NVAR/SDFR=gdf size
	
	variable lmode, hwidth
	variable ic, ng, gwidth, fs, fig, ngraphics, gtype
	variable frame = 0
	string tStr, gname, gtitle, gPICTname, theGraphic
	string md4 = SelectString(useMD,"","#### ")
	
	// translate modes and size
	// put as png
	lmode = -5
	hwidth = 6.5*72
	
	// put the graphic

	ng = ItemsInList(theList)
	
	for (ic=0;ic<ng;ic+=1)
		fig = 1
		thegraphic = StringFromList(ic,theList)
		strswitch(thegraphic)
			case "CLIPBOARD":
				gtype = 8
				break
			case "PICTURE":
				gtype = 9
				break
			default:
				gtype = WinType(thegraphic)
				break
		endswitch
		switch(gtype)
			case 0:	// possible gallery picture
				GetWindow/Z $thegraphic, wtitle
				if (strlen(s_value) == 0)
					gtitle = parse_forTitle(thegraphic)
				else
					gtitle = thegraphic + " - " +  s_value
				endif
				gname = md4 + "Gallery Picture: " + gtitle + "\r"
				gPICTname = thegraphic
//				SavePICT/SNAP=1/O/WIN=$thegraphic/P=_PictGallery_/E=-5 as gPICTname
				gwidth = str2num(StringByKey("WIDTH",PICTInfo(gPICTname)))
				break
			case 1:	// graph
				GetWindow/Z $thegraphic, wtitle
				gtitle = thegraphic + " - " +  s_value
				GetWindow/Z $thegraphic wsize
				gwidth = V_right - V_left
				gname = md4 + "Graph: " + gtitle + "\r"
				gPICTname = thegraphic
				break
			case 2:	// table
				GetWindow/Z $thegraphic, wtitle
				gtitle = thegraphic + " - " + s_value
				GetWindow/Z $thegraphic wsize
				gwidth = V_right - V_left
				gname = md4 + "Table: " + gtitle + "\r"
				gPICTname = thegraphic
				break
			case 3:	// layout
				GetWindow/Z $thegraphic, wtitle
				gtitle = thegraphic + " - " + s_value
				GetWindow/Z $thegraphic wsize
				gwidth = V_right - V_left
				gname = md4 + "Layout: " + gtitle + "\r"
				gPICTname = thegraphic
				break					
			case 7:  // panel
				GetWindow/Z $thegraphic, wtitle
				// possible child panel
				if (strlen(s_value) == 0)
					gtitle = parse_forTitle(thegraphic)
				else
					gtitle = thegraphic + " - " +  s_value
				endif
				gname = md4 + "Panel: " + gtitle + "\r"
				gPICTname = "panelPict"
				SavePICT/SNAP=1/O/WIN=$thegraphic/P=_PictGallery_/E=-5 as gPICTname
				gwidth = str2num(StringByKey("WIDTH",PICTInfo(gPICTname)))
				break
			case 8:  // clipboard
				gPICTname = "clipPict"
				gtitle = "Clipboard Picture"
				gname = md4 + "Clipboard Picture: " + gtitle + "\r"
				// save clipboard to picture gallery
				LoadPict/O/Q "Clipboard", $gPICTname
				gwidth = str2num(StringByKey("WIDTH",PICTInfo(gPICTname)))
				break
			default:  // picture
				gPICTname = thegraphic
				gtitle = "Gallery Picture"
				gname = md4 + "Gallery Picture: " + gtitle + "\r"
				gwidth = str2num(StringByKey("WIDTH",PICTInfo(gPICTname)))
				gtype = 9
				break
		endswitch
		
		// put figure in notebook
		if (fig==1)
			if (dosubs)
				Notebook $theNtbk, text=gname
			endif
			if (bffs==0)		// at full scale
				Notebook $theNtbk, frame=frame, scaling={size,size}, picture={$gPICTname(0,0,0,0),lmode,1}
			else			// scaled at a specific size to letter paper
				fs = round((hwidth/gwidth)*size)
//				print thegraphic, hwidth, gwidth, fs
				Notebook $theNtbk, frame=frame, scaling={fs,fs}, picture={$gPICTname,lmode,1}
			endif
			ngraphics = str2num(GetUserData(theNtbk,"","GraphicsCount"))
			if (numtype(ngraphics)!=0)
				ngraphics = 0
			endif
			put_URLinNtbk(ngraphics + ic + 1, thegraphic, theNtbk)
			tStr = f_GraphicsInfo(thegraphic)
			Notebook $theNtbk, text = tStr
			if ((gtype > 0) && (gtype < 7))
				put_windowNotesinNtbk(thegraphic, theNtbk)
			endif
			Notebook $theNtbk, text="\r"
		endif
	endfor
	KillPICTs/Z panelPict, clipPict
	
	SetWindow $theNtbk userData(GraphicsCount)=num2str(ngraphics + ng)

	return 0
end

// @brief put URL for graphic in notebook
// @inputs: theOne - sequencial number, thegraphic - the local file name, theNtbk - the notebook name
Static Function put_URLinNtbk(variable theOne, string thegraphic, string theNtbk)

	string mdURL	
	sprintf mdURL "\r![local file: %s.png](<./%sMedia/Picture%d.png>)\r", theGraphic, theNtbk, theOne
	Notebook $theNtbk, text=mdURL
	return 0
end

// @brief add window notes if they exist
Static Function put_WindowNotesinNtbk(string thegraphic, string theNtbk)

	if (cmpstr(thegraphic,"CLIPBOARD") != 0)
		GetWindow $thegraphic, note
		if (strlen(S_value) != 0)
			sprintf S_value "\r#caption %s\r", S_value
			Notebook $theNtbk, text=S_value
		endif
	endif
	return 0
end

// @brief Parse for Execution
// returns string from execution
// [all]: 1 returns command and string, 0 (default) returns only executation string
Static Function/S parse_ForExecution(string vStr, [variable all])
	
	all = ParamIsDefault(all) ? 0 : all
	
	string theReturnStr=""
	string theCmd=""
	string enStr, theSCmd
	variable st=0
	variable en=0
	variable doff
	
	string/G rtnStr = ""

	// check for executable functions
	en = strsearch(vStr,"%",st)
	if (en >= 0)	
		do
			if (cmpstr(vStr[en-1],"\\")!=0)
				theReturnStr += vStr[st,en-1]
				st = en+1
				en = strsearch(vStr,")",st)
				if (en > st)
					theCmd = vStr[st,en]
					sprintf theCmd, "rtnStr = %s", theCmd
					Execute/Q/Z theCmd
					if (V_flag)
						sprintf theCmd, "rtnStr = ProcGlobal#%s", vStr[st,en]
						Execute/Q/Z theCmd
						if (V_flag)
							sprintf theCmd, "ERROR %d: %s", V_flag, vStr[st,en]
							rtnStr = theCmd
						endif
					endif
					theReturnStr += rtnStr
					st = en + 1
				else
					en = st + 7
					theCmd = vStr[st,en]
					strswitch(theCmd)
						case "clipTEXT":
							rtnStr = GetScrapText()
							if (strlen(rtnStr) != 0)
								theReturnStr += vstr[st,en] + ":\r" + rtnStr + "\r"
							else
								theReturnStr += vstr[st,en] + ": EMPTY "
							endif
							st += 8
							break
//						case "clippict":
//							LoadPict/O/Q/Z "Clipboard"
//							if (v_flag != 0)
//								LoadPict/O/Q "Clipboard", PICTclip
//								//theReturnStr += vstr[st,en]+"\r"
//								put_PICTinNtbk("PICTclip")
//							else
//								theReturnStr += vstr[st,en] + ": EMPTY "
//							endif
//							st += 8
//							break
						default:
							theReturnStr += "MISSING ): "
							break
					endswitch
				endif
			else
				theReturnStr += vStr[st,en-2] + "%"
				st = en + 1
			endif	
			en = strsearch(vStr,"%",st)
		while (en >= 0)
		theReturnStr += vStr[st,strlen(vStr)-1]
	else
		theReturnStr = vStr
	endif
	
	killstrings/Z rtnStr

	// check for executable function calls
	vStr = theReturnStr
	theReturnStr = ""
	st = 0
	en = strsearch(vStr,"#",st)
	if (en >= 0)	
		do
			if (cmpstr(vStr[en-1],"\\")!=0)
				if (cmpstr(vStr[en+1],"{")==0)
					enStr = "}"
					st = en + 2
					doff = 1
				else
					enStr = ")"
					st = en + 1
					doff = 0
				endif
				en = strsearch(vStr,enStr,st)
				if (en > st)
					theCmd = vStr[st,en-doff]
					Execute/Q/Z theCmd
					if (V_flag)
						switch(doff)
							case 0:
								Execute/Q/Z theCmd
								if (V_flag)
									sprintf theCmd, "ProcGlobal#%s",  vStr[st,en-doff]
									Execute/Q/Z theCmd
									if (V_flag)
										sprintf theCmd, "\rEXECUTION ERROR %d: #%s\r", V_flag, vStr[st,en-doff]
										theReturnStr += theCmd
									endif
								endif
								break
							case 1:
								sprintf theCmd, "\rEXECUTION ERROR %d: #%s\r", V_flag, vStr[st,en-doff]
								theReturnStr += theCmd
								break
							default:
								break
						endswitch
					endif
					st = en + 1
				else
					sprintf theReturnStr, "%s\rERROR MISSING %s: %s", theReturnStr, enStr, vStr[st-1-doff,st+5]
				endif
			else
				theReturnStr += vStr[st,en-2] + "#"
				st = en + 1
			endif	
			en = strsearch(vStr,"#",st)
		while (en >= 0)
		theReturnStr += vStr[st,strlen(vStr)-1]
	else
		theReturnStr = vStr
	endif
	
	return theReturnStr
end

// @brief return existence of x-wave for a given y-wave on a graph
Static Function/S parse_forXWaveName(string gname, string tname)

	string rStr = ""
	string tStr, wStr
	tStr = TraceInfo(gname,tname,0)
	wStr = StringByKey("XWAVE",tStr)
	if (strlen(wStr) != 0)
		tStr = StringByKey("XWAVEDF",tStr) + wStr
		rStr = " vs " + tStr
	endif
	return rStr
end

// @brief export the notebook as markdown
Static Function export_NtbktoMD(string theNtbk)

	string htmfilename, mdfilename //, tfoldername, nfoldername
	htmfilename = theNtbk + ".htm"
	mdfilename = theNtbk + ".md"
	//foldername = theNtbk + "Media"
	
	NewPath/O/Q/M="Choose location for markdown notebook" ntbkpath
	if (v_flag != 0)
		return 0
	endif
	
	AutoCorrect_ImageNumbering(theNtbk)
	
	SaveNotebook/O/P=ntbkpath/S=5/H={"UTF-8",0,0,0,0,32} $theNtbk as htmfilename
	SaveNotebook/O/P=ntbkpath/S=6 $theNtbk as mdfilename
	DeleteFile/P=ntbkpath htmfilename
	//MoveFolder/O/P=ntbkpath/Z tfoldername as nfoldername
	
	return 0
end

// @brief autoCorrect Image Numbering
// @input the notebook name
Static Function autoCorrect_ImageNumbering(string theNtbk)

	string theLine, thePre, theFront, theT, theN, theBack, theNewN
	string theREGEX="(.+<./)(.+)(Media/Picture)([[:digit:]]+)(.+)"
	variable ic = 0
	Notebook $theNtbk, selection={startOfFile, startofFile}
	do
		Notebook $theNtbk, findText={"/Picture",0}
		if (v_flag == 0)
			break
		endif
		Notebook $theNtbk, selection={startofParagraph,endOfParagraph}
		GetSelection notebook, $theNtbk, 2
		SplitString/E=(theREGEX) S_selection, thePre, theT, theFront, theN, theBack
		theNewN = num2str(ic)
		theLine = thePre + theNtbk + theFront + theNewN + theBack
		Notebook $theNtbk, text=theLine
		ic += 1
	while(1)
	SetWindow $theNtbk userData(GraphicsCount)=num2str(ic-1)
	return 0
end

// @brief clear MarkDown formats from notebook
// @input the notebook name
Static Function clear_MarkDown(string theNtbk)

	string pstr
	
	pstr = "Clear all markdown header tags from notebook " + theNtbk + "?\r\rThis action is not undoable!"
	DoAlert/T="Clear MarkDown Header Tags from Notebook" 1, pstr	
	if (v_flag != 1)
		return -1
	endif
	
	Notebook $theNtbk, selection={startOfFile, startofFile}
	do
		Notebook $theNtbk, findText={"#### ",0}
		if (v_flag == 1)
			Notebook $theNtbk, text=""
		endif
	while(v_flag!=0)
	Notebook $theNtbk, selection={startOfFile, startofFile}
	do
		Notebook $theNtbk, findText={"### ",0}
		if (v_flag == 1)
			Notebook $theNtbk, text=""
		endif
	while(v_flag!=0)
	Notebook $theNtbk, selection={startOfFile, startofFile}
	do
		Notebook $theNtbk, findText={"## ",0}
		if (v_flag == 1)
			Notebook $theNtbk, text=""
		endif
	while(v_flag!=0)
	Notebook $theNtbk, selection={startOfFile, startofFile}
	do
		Notebook $theNtbk, findText={"# ",0}
		if (v_flag == 1)
			Notebook $theNtbk, text=""
		endif
	while(v_flag!=0)
	
	return 0
end

// @brief clear 3+ subheadings from notebook
Static Function clear_SubHeaders(string theNtbk)

	string pstr
	
	pstr = "Clear 3+ markdown (sub) headers from notebook " + theNtbk + "?\r\rThis action is not undoable!"	
	DoAlert/T="Clear Markdown SubHeaders from Notebook" 1, pstr
	if (v_flag != 1)
		return 0
	endif
	
//	DoWindow/F $theNtbk
	Notebook $theNtbk, selection={startOfFile, startofFile}
	do
		Notebook $theNtbk, findText={"#### ",0}
		if (v_flag == 1)
			Notebook $theNtbk, selection={startofParagraph, startofNexTParagraph}
			Notebook $theNtbk, text=""
			v_flag=1
		endif
	while(v_flag!=0)
	Notebook $theNtbk, selection={startOfFile, startofFile}
	do
		Notebook $theNtbk, findText={"### ",0}
		if (v_flag == 1)
			Notebook $theNtbk, selection={startofParagraph, startofNexTParagraph}
			Notebook $theNtbk, text=""
			v_flag=1
		endif
	while(v_flag!=0)
	Notebook $theNtbk, selection={endOfFile, endofFile}
	
	return 0
end
//
//// @brief show Help
//Static Function show_LBHelp()
//
//	OpenHelp/INT=0/Z ParseFilePath(1, FunctionPath(""), ":", 1, 0) + "LogBookHelp.ihf"
//	DisplayHelpTopic/K=1/Z "Help for LogBook Package for Igor Pro"
//	return 0
//end

// @brief show and handle help for notebook panel
Static Function show_NtbkHelpMenu()

	string cmdstr, infostr

	cmdstr = "Open the help file;Visit the Webpage;(-;Email a BUG REPORT;Email a feature request;"
	PopupContextualMenu cmdstr

	switch(v_flag)
		case 1:	// open the help file
			OpenHelp/INT=0/Z ParseFilePath(1, FunctionPath(""), ":", 1, 0) + "LogBookHelp.ihf"
			break
		case 2:	// visit the Webpage
			BrowseURL /Z "https://www.wavemetrics.com/node/" + num2str(kProjectID)
			break
		case 4:	// email BUG REPORT
			sprintf infostr "LogBook PACKAGE VERSION: %2.2f\r\n", thePackageVersion
			infostr += IgorInfo(3)
			cmdstr = "mailto:"
			cmdstr += "Jeffrey.Weimer@UAH.edu"
			cmdstr += "?subject="
			cmdstr += "BUG REPORT: LogBook Package for Igor Pro"
			cmdstr += "&body="
			cmdstr += "\r\n--\r\nType Your Bug Report Above the Line"
			cmdstr += "\r\nPlease do your best with details to help me troubleshoot"
			cmdstr += "\r\nScreen snapshots or debug outputs are welcomed"
			cmdstr += "\r\n" + infostr + "\r\n--"
			BrowseURL/Z cmdstr
			break
		case 5:	// email feature request
			sprintf infostr "LogBook PACKAGE VERSION: %2.2f\r\n", thePackageVersion
			infostr += IgorInfo(3)
			cmdstr = "mailto:"
			cmdstr += "Jeffrey.Weimer@UAH.edu"
			cmdstr += "?subject="
			cmdstr += "Feature Request: LogBook Package for Igor Pro"
			cmdstr += "&body="
			cmdstr += "\r\n--\r\nType Your Feature Request Above the Line"
			cmdstr += "\r\nPlease do your best in describing what you would like to have"
			cmdstr += "\r\nSpecific suggestions or mock-ups are welcomed"
			cmdstr += "\r\n" + infostr + "\r\n--"
			BrowseURL/Z cmdstr
			break
		default:
			break
	endswitch
	return v_flag
end

// @brief show and handle settings for notebook panel
Static Function show_NtbkSettingsMenu()

	string ntbk, ntbkt, cmdstr, mflgstr, flgstr
	
	// is notebook present?
	ControlInfo/W=$kPanelName tb0_notebook
	if (v_value!=1)
		ntbk = s_value
		DoWindow $ntbk
		if (v_flag==1)
			mflgstr = ""
			GetWindow $ntbk wtitle
			ntbkt = s_value
		else
			mflgstr = "\M1("
			ntbkt = "No Active Notebook"
		endif
	else
		mflgstr = "\M1("
		ntbkt = "No Active Notebook"
	endif
	cmdstr = mflgstr + ntbkt + ";(-;"
	cmdstr += mflgstr + "Autocorrect image number sequences;"
	cmdstr += "(-;"
	
	// is markdown being used?
	ControlInfo/W=$kPanelName tb0_cbnbmd
	mflgstr = SelectString(v_value,"\M1(",mflgstr)
	cmdstr += mflgstr + "Clear all markdown;"
	cmdstr += mflgstr + "Remove all 3+ level markdown headings;"
	cmdstr += mflgstr + "Export notebook as markdown;"
	
	// show context menu
	PopupContextualMenu cmdstr
	
	// do command
	switch(v_flag)
		case 1:	// notebook
			DoWindow/F $ntbk
			break
		case 3:	// autocorrect image number sequences
			autocorrect_ImageNumbering(ntbk)
			break
		case 5:	//
			clear_MarkDown(ntbk)
			break
		case 6:	//
			clear_SubHeaders(ntbk)
			break
		case 7:	// export notebook as markdown
			export_NtbktoMD(ntbk)
			break
		default:
			break
	endswitch
	return v_flag
end

// *** f_functions

// @brief check whether window is an external sub window
Function f_isExtSubWindow(string fullwinpath)

	GetWindow $fullwinpath, exterior
	return v_value
end

// @brief return list of external sub windows if they exist
Function/S f_LoExtSubWindows(string winpath)

	variable nsw, ic
	string rtstr = ""
	string cwinList, theWin
	cwinList = ChildWindowList(winpath)
	if (strlen(cwinList) != 0)
		nsw = ItemsInList(cwinList)
		for (ic=0;ic<nsw;ic+=1)
			theWin = winpath + "#" + StringFromList(ic,cwinList)
			GetWindow $theWin, exterior
			if (v_value)
				rtstr += theWin + ";"
			endif
		endfor
	endif
	return rtstr
end

// @brief return string frontmost possible graphic
Function/S f_FrontMostWin([variable full])

	string rstr
	variable wt = 1+2+4+64+16384+65536
	string theWin
	sprintf theWin "WIN:%d,VISIBLE:1"	, wt
	rstr = StringFromList(0,RemoveFromList(kPanelName,WinList("*",";",theWin)))
	if (ParamIsDefault(full))
		ControlInfo/W=$kPanelName cb_titles
		rstr = shorten_String(rstr,v_value)
	endif
	return rstr
end

// @brief return string list of graphics
// @input full = 1 to avoid truncations
Function/S f_LoGraphics([variable full])
	
	string rStr, theWin, theStr, theGraphList, theGraphic
	string theSList = ""
	string theList = ""
	
	variable mode, ic, ng, slen
	
	full = SelectNumber(ParamIsDefault(full),0,1)
	
	ControlInfo/W=$kPanelName cb_titles
	mode = v_value
	
	// layouts
	theWin = "WIN:4,VISIBLE:1"
	theStr = WinList("*",";",theWin)
	if (strlen(theStr)!=0)
		theList = SortList(theStr)
	endif
	
	// graphs
	theWin = "WIN:1,VISIBLE:1"
	theStr = WinList("*",";",theWin)
	if (strlen(theStr)!=0)
		theStr = SortList(theStr)
		theList += theStr
		theGraphList = theStr
	endif
	
	// tables
	theWin = "WIN:2,VISIBLE:1"
	theStr = WinList("*",";",theWin)
	if (strlen(theStr)!=0)
		theList += SortList(theStr)
	endif
	
	// gizmo windows	
	theWin = "WIN:65536,VISIBLE:1"
	theStr = WinList("*",";",theWin)
	if (strlen(theStr)!=0)
		theList += SortList(theStr)
	endif
		
	// camera windows	
	theWin = "WIN:16384,VISIBLE:1"
	theStr = WinList("*",";",theWin)
	if (strlen(theStr)!=0)
		theList += SortList(theStr)
	endif

	// panels
	theWin = "WIN:64,VISIBLE:1"
	theStr = RemoveFromList(kPanelName,WinList("*",";",theWin))
	theStr = RemoveFromList("WNInput",theStr)
	if (strlen(theStr)!=0)
		theList += SortList(theStr)
	endif
	
	// child windows (panels) attached to graphs
	ng = ItemsInList(theGraphList)
	for (ic=0;ic<ng;ic+=1)
		theGraphic = StringFromList(ic,theGraphList)
		theStr = ChildWindowList(theGraphic)
		if (strlen(theStr) != 0)
			theList += theGraphic + "#" + theStr
		endif
	endfor
	
	// picture objects (including clipboard)
	ControlInfo/W=$kPanelName cb_wpicts
	if (v_value)
		LoadPict/Z/Q "Clipboard"
		if (v_flag == 1)
			theList += "clipboard;"
		endif
		theWin = ""
		theStr = PICTList("*",";",theWin)
		if (strlen(theStr)!=0)
			theList += SortList(theStr)
		endif
	endif
	
	// format based on menu options
	
	if (full == 0)	// full path list
		theSList = theList
	else
		wave/T theTList = ListToTextWave(theList,";")
		for (theGraphic : theTList)
			theSList += shorten_String(theGraphic, mode) + ";"
		endfor
	endif
	
	rStr = theSList
	
	return rStr
end

// @brief shorten a string
Function/S shorten_String(string theGraphic, variable mode)

	string rStr, theStr
	variable slen
	
	switch(mode)
		case 0:	// shorten name
			slen = strlen(theGraphic)
			if (slen>20)
				rStr = theGraphic[0,5] + "..." + theGraphic[slen-11,slen-1]
			else
				rStr = theGraphic
			endif
			break
		case 1:	// shorten title
			GetWindow/Z $theGraphic wtitle
			if (strlen(S_value) != 0)
				theStr = s_value
			else
				theStr = parse_forTitle(theGraphic)
			endif
			slen = strlen(theStr)
			if (slen>20)
				rStr = theStr[0,5] + "..." + theStr[slen-11,slen-1]
			else
				rStr = theStr
			endif
			break
		default:
			rStr = theStr
			break
	endswitch
	
	return rStr
end

// @brief return string list of noteboooks
Function/S f_LoNtbks()

	return ("New;" + WinList("*",";","WIN:16"))
end

// @brief return string list of data folders
Function/S f_LoDataFldrs()

	string lof = ""
	variable ic = 0
	variable nf = CountObjects("root:",4)
	for (ic=0;ic<nf;ic++)
		lof += GetIndexedObjName("root:",4,ic) + ";"
	endfor
	lof = "root;" + RemoveFromList("Packages",lof,";")
	return lof
end

// @brief return truth of having pictures 
Function f_HasPicts()

	variable vrtn = (ItemsInList(PICTList("*",";","")) == 0) ? 0 : 1
	return vrtn
end

// @brief returns string list of selected graphics
Function/S f_LoGraphicsSelected()
	
	DFREF pdf = $thePackageFolder
	wave/SDFR=pdf sw = graphicsel
	
	variable ic, npts
	string rtStr = ""
	
	wave/T fw = ListtoTextWave(f_LoGraphics(full=1),";")
	npts = DimSize(sw,0)
	for(ic=1;ic<npts;ic+=1)
		if (sw[ic] == 48)
			rtStr += fw[ic-1] + ";"
		endif
	endfor
	
	return rtStr
end

// @brief return string list of graphics menu options
Function/S f_GraphicsMenu()

	string rStr
	string tStr = ""
	variable nitems = ItemsInList(f_LoGraphics())
	
	// picture objects (including clipboard)
	ControlInfo/W=$kPanelName tb0_cbnbmd
	print v_value, " MD"
	ControlInfo/W=$kPanelName cb_titles
	print v_value, " titles"
//	Checkbox cb_wpicts, win=$kPanelName, value=1
	ControlInfo/W=$kPanelName cb_wpicts
	print v_value, " picts"
	if (v_value)
		LoadPict/Z/Q "Clipboard"
		if (v_flag == 1)
			tStr = "clipboard;"
		else
			tStr = "\M1(clipboard;"
		endif
		rStr = PICTList("*",";","")
		if (strlen(rStr)!=0)
			tStr += SortList(rStr)
		endif
	endif
	
	sprintf rStr, "None;Frontmost;All (%d);Select;%s", nitems, tStr
	return rStr
end

// @brief return string of graphics information
Function/S f_GraphicsInfo(string gname)
	
	string rStr = "\rGraphics Information"
	string tStr, icStr
	variable ic, nt
	
	if (cmpstr(gname,"CLIPBOARD") == 0)
		rStr += "\r- title: (from clipboard)"
	else
		GetWindow/Z $gname, wtitle
		if (strlen(s_value) == 0)
			rStr += "\r- title: (child or unknown window)"
		else
			rStr += "\r- title: " + S_value
		endif
	endif
	
	switch(WinType(gname))
		case 1:	// graph
			tStr = TraceNameList(gname,";",1+2+4+16+32)
			nt = ItemsInList(tStr)
			if (nt != 0)
				rStr += "\r- traces: "
				for (ic=0;ic<nt;ic+=1)
					icStr = StringFromList(ic,tStr)
					wave w = TraceNameToWaveRef(gname,icStr)
					rStr += GetWavesDataFolder(w,2) + parse_forXWaveName(gname,icStr) + ", "
				endfor
				rStr = RemoveEnding(rStr,", ")
			endif
			tStr = ImageNameList(gname,";")
			nt = ItemsInList(tStr)
			if (nt != 0)
				for (ic=0;ic<nt;ic+=1)
					rStr += "\r- images: "
					icStr = StringFromList(ic,tStr)
					wave w = ImageNameToWaveRef(gname,icStr)
					rStr += GetWavesDataFolder(w,2) + ", "
				endfor
				rStr = RemoveEnding(rStr,", ")
			endif
			break
		case 2:		// table
			rStr += "\r- waves: "
			tStr = TableInfo(gname,-2)
			nt = str2num(StringByKey("COLUMNS",tStr)) - 1
			for (ic=0;ic<nt;ic+=1)
				tStr = TableInfo(gname,ic)
				rStr += StringByKey("WAVE",tStr) + ", "
			endfor
			rStr = RemoveEnding(rStr,", ")
			break
		default:		// all else
			break	
	endswitch
	
	return (rStr + "\r")
end

// *** PICTURES

// PNG: width= 65, height= 22
Picture pLB_Eraser
	ASCII85Begin
	M,6r;%14!\!!!!.8Ou6I!!!!]!!!!5#R18/!7:mT.0'>J"EQn$<!Wa8#^cngL]@DT$#iF<ErZ1Lf)P
	d[f)\2dRs4o5!#]lI=EIJ[Aor6*Eb,5pGBYZRDJO<'@;od1DfTK[mQ;Bf!!sc>6pXdSKrp`sD]Mmo`
	ddk@F(W_BnZdLO43iO*D*nUuco/RmEc:.f"<dmT]MSeE49,F3PTg-&$P-85%fdd&#'fB*V,ltf;l(X
	/De\4,Nfteblm,>BO6<#hB4fKH4<3PD99O>KdQEJ#Xf_%rhcaga28VB:q#@Lh!#pG<TU%@;(//U[V)
	_,9j.+$=MLmMNPXI$a9.;)n(ha"!JL4e)=)e"#0m:ht1"B`YGC["BV*kToctifg$0^=T@F=Uh3$>Pf
	3&++PUeJ`V2RenP79@U/>1,%qFmaPCK#Ah/499[K"eW2gM<QAcb6%#M4t5TC:\]q.LgU2XI8o'2gZG
	!XH?YXjHI"pal5\L6X7kun#)mOY'#VZ,PiHDGS@Quj9lhdQc13KZJY'[MG6b!K;YZi`cB2BHeYUigi
	J8l]EY*)VD#<RF*kT&,@(gcUNVFnNm@7\Da:-D-E#`@.c8f]Y0j2Z1rd^VY>J^0drKk`P!VZ\EQO#\
	9":e+sdqE;d>YH$'82tW:iUk)ImF2\nP'WI(h*C0^%$f(p^@6M!oNq9V49W;%9QQ2X[aoJ,,iY6,O-
	3Zh[odE#C!,.\,!Yu$+R<*OnD-q"bS]1t2T<sIB*cB^"X)-49V*E8c[.%:W:tJ]$fYD6"dbI]<`kR7
	DRPqCcsV76`Mp1(B.l<?VPB\J6Q0cYkUoG1m6Ka`$"og5_?qaRjh2G4e/7$`]u\J8-7BY1pjb('@h$
	io3-b?Fp"HO7mB918>0#^_6i?Oo-`O6##bFNHMVMB/Q4[Jk@J`3FEm^NB2_Sm8_\tl\2e\ePAG4LuN
	JSuo`^?4N"W(Q8Vp7Eh#X&H@M5LJC?9aXU[7h&PiZEY58T!QYcl0!U;qE^K@@HTXaa*Za96b:AIZF@
	*.DA_b`gAd1r\[n?>O'6G_td?Qm#g+Ol1%9H?!trd1IF2Be!ueuET0?k&'ke\[HTInUt7ulFVC_c;3
	23.GmnflFHo!OXgeC=`qCJ3kVL!6]<rPeAs`ZBH]7K8#N'&J+rYo_00]BV4J*o+4)6jbk=nF<H^;BL
	2p1ktT:Y*W+8h8[hE,teZ5L[Vn>AK5bdo[4(.DL7!R?g?L,>Ku0g8>pHAq"M.%Hd&X.<1N,qAmFj[-
	mbZ%cWRAlZO_:iAm+dDOH(RXqis1:h*OUX)(rDf6*LqlERXjOOj5A%:u&?7]I#pB^($i*/T/ZZ'[ZI
	L\U8>P`\mn?jV44?UqWWIU%*^;K`^h<;nfF3E9;$f/":b&c!#9OiI?R?8"nNf9'lSNsRqHm&eqL@<i
	0O\kEXP^9oCYrnhc4!ZA%-n/8uT;ECDp?c@7kAR<NC'7GpCQ7duZjaJ;r&[Tb[i<#'4#a>`>3%p:ON
	R0b'E<BB"2EVWGH$7Y&h*rU.?m^iPOmB&LR@@YDh%A"9:`ma-UURjcoaP7bigdU,=cF<3f0Dq6"DFe
	jsZ^m?(4`RZb3.QWgB[Y'Ie`"8mR!jof6cGiT._d*-gHLQ5,>A\T:eENKtJu$u[4M`,la.oo01mS\^
	3uNmF?Xh'[AB.sAu880JQTUk<k$&(CII_gML)5K&D+U&,RIN?N-sM9epNF8Q0cYq]cGe%(0W5"8^2"
	^+YepN0=Om%q=tMdirAern3r>Q?aT4NG&qG#0\*d8DVqm,7@enQ-doHU%?-8@q^E7J?(tl"-#7,L6e
	D3eVY:`aGq1prEa(mb#H0S+!3(Le-!Z4O9?@Cb8eQd$gN%3B^(La(4S%guN"VTp=fZ9%'_=TW\U(er
	;9d89Hd0YmBcKQX1h,;bYo]K^G^u[@A'OiS5IJY[R.:,g?f_A'FTPqZHfX6Yi*G@Zj3>0c@i2)Q+s-
	IC9:]Fmq7E1D.lDP:XnCX>:D1,Yi-V?Qf)k18RN][&bJbJuB6B<6Ej6)-?<IU/SCbp04P#r;M/hcs"
	^`0hF1@c4Pr[9^m@m^d'RJROF&-2Og'oZ*$[nPBYkYBkW$JXcG6gB[DPZBXQNaAM!l8=6UbG,)RT(3
	2Y5@/ZC@Q<"^4+'UC>gYLfJOQb6KJ.0Rlk*j]9GX&:+E<,b,(PZjO7K!fUIB.+;`e%bQQ`o8=b'Hrq
	PDkniTD-Ns`iJXFj+A61UqP@*U@;sCWN1V!NVI0:Re$BKY/LqLjs(PBi&N+/T`_`d<WRpGM$ABcFQW
	+;F#RG3'_SJON_%jKAFjaGPNG=_J[7XK%,0D4E+MdSr[+7dEY1X']WjQ10[qoL4oJecoq/aR8R$5_9
	6kA(5OZlVuSqkYdrC)]]hOASHE`aU"s'C`"nf"-KqIpEY8X`1\i3R=W3psTF/lo/^I8.!I63sse&QC
	4ol[9HP#Ohp%E:r>R4h]KZ?-%$m0*ot>O&?<.,RABq_U_nj`?N,#pMoief6D8t:55sh[s%.B2Jq0dR
	PejdK&FXS=t.G]Pq/edReN"(YO6_.p<#ZNV%R<!a,V/b$j/!@.]ZXS6qF^FJl4YF.BQmL=EKg3%?n5
	_8tjC=6p&@d$EmFGpH4+OR*:Y'#!]RV%k!'MFG?iAO4Kcrck,;(F"g11Fk?41X&i+M(*PPoNfF_T!S
	)4U<GR>N;D5nrBNd0*S+/k`#d*SZ:"3j<AS1@ZRUPcOBhG`QD#(s(LrK+^kCNN#F1inpFjaq7:L+Zo
	l"QTm0hFGMZOX?/qZXX*8>0._gf:2OX%K?jFIlM;QiqR7LOk*%9hQg-3-4:3;G2D>M(U_76^ejN9pq
	kP$9[tV9h9.U%nl;ri`!>6DT#50Nfm/^d*=_+XTL(o9\>!$'QKm:V,U^070Hiq:tej%0EfuZ4PqM;P
	t@.7!t]E<JF\do!<VT&iLFsA(IDAogFJalpbKRWgGY3OkO;2APK\/ZCV`W$/I(uJ'#t=/npGi:SVJk
	HZ8ckEg&Z8Y9h/?q<6\_J`FH#M0$hY!+\Pa-dNWP+jB55QhB=9]_Xm:)+iR>M=O8JBb@QIqFarpd3c
	ZcZZ-9t?8AHiMl$ieO(3j.GM><#/!!!!j78?7R6=>B
	ASCII85End
End

// PNG: width= 90, height= 30
Picture pLB_Help
	ASCII85Begin
	M,6r;%14!\!!!!.8Ou6I!!!"&!!!!?#R18/!3BT8GQ7^D&TgHDFAm*iFE_/6AH5;7DfQssEc39jTBQ
	=U%&ule5u_NK]pa@g']g@U^Upt%GFQ2iG6aJpJmdFX78;,cVIC:7E<8Td:b6cN,?9UjVf\f%WZ;[/V
	%KP#JO*M&WN?THKK9cU)`=`H0[O9,)L8;-_'G*Bak\80]?@nUmHm(8j1N'qpQ@[GqTSkLm[iV;4l?5
	9p34hHca--pCc9?`a1`N@?0>MW\`C_cK87>p!!)X0(AW7k(<k1!Z6U5Zb%=PW#*`9'?8(R.l9\9B%n
	9#Xi,CY"\p?O1oCK4p_#dS:<['Ueqmu'!DdI1i5jrP3+p\-^L%B)Rl&&6R@eFq>=L[qbCBEtmmNr/L
	*e`#,Pc(S2@J/p9W2QZ)-J]W[Rki??laDuDX[@&P,(^^j"-[%#SONo\V1=G'!!%70KV'(>nd;T\r9s
	nW!!']u`J45Y;-RJO#^h#IS8@)s?-6E37K\jr"J&9mNoQd1W?So9&&7!4QcfPj`<*6p)q!4%Z!kNSb
	ab*S0C)+!-4P&qo6Jd<2$QkkZlF^m=^($s8m)tsn,tIZn\F/q!JI,[b(u%<c'JV4aX<^Bf%*k%J6&^
	>;,IqtHG)FHLBE0]KCF+J!.aqBH9Cu\K<fV!+1qo2IdrI^jN)fkTP5;J-4NXp%T0D`+g<s/YW;Xf"@
	\,T!Yqbd?E6KnN%F4r'*)'g*uD%cKYq4thknG`:Fo$Eh;$dZNZ:%ld4Hbsa^bHA79m^g0PiWdCc346
	JH/)8KM&)(k36QF'1-*7>&j)l`JADS2'.,2Z*_)fH$KMmMP'\-](',%?j1@)>F]S9TO0p.T!W[#6cg
	fTP-7;t(+Zf_JqtSn,aMH!n\';3U2ZD&S;P.=h1rnGXk<CB$fKN,kd+9PO4u%TJ&Hu4&4Crs9Y@_nH
	"MR&l)1>b[1u<1&g\hMaNi(kl1-rkL5)pKfeO6U'VM\7&0/`!A7PfqQOE]am/tCKELH9=]@#NnO'jY
	tqYGLh>IH)^mct,(P9&uXOhUj'bDC#sD;2^*qE?dh;P68RpF'`l^&V357%l%naPeh(A($.glRQtugt
	ZGl%$*W+%Ia6#,81..8g^]id,Kck$t@][$aBOjq/OasMhNuG2M,LC'G_J("Z[P4_r5QBmFntXjd',6
	lZt\G^OJI;7Rj%OX5iIM#6uCX!]FoLK`(amh5S'MpXVq/O#EQ=C>KI[/s=orf,T\.V5"J]r\=n:<P^
	dbK.Jkf0au0o*Zjc5<u!+J$%sur)3hjN3<hR:W<>kX3+#G"^cXsdL/hY`gSKQ2hQ^Pq(l!UpCdqH+(
	.;#abis6/fk6q:$aRh5_>*eM9NS:6RPelrZ>?$aa)Nf&c!V@m]6<R6e=FIgEcWdq5MWd4Wd(EGH-(t
	7'6n3kpg)VSEUsmM3$iUNPR)sTB+NfY$cV71VX=4n!'f7b&o2]5j7tq;=*jFX8d1ff/P>NYa\kIC.H
	0D<]`$GJ&fmmVZ]hoTDId<i+X\WBThm/_*d[#n'GhQY]#TjRa0oj>Y^?+^*&qn&>%qpqLF8^A1;Q1&
	-O1QL#&^mh-Vg1nFHc5Yg_^Fm5Mj%+,Mn_\Yol9dGTnRi%0.uNj'\-e(in+;$'enZ/=ITkLsojIs'f
	5J4*Sf)rJ-0)p,hFVP)c6^;1hlKdF&nZ-FC.eJ7Vk#%#29.N7pubh'DZ93h9r&DjT2'SNDUuSfXk:T
	hm!T6a,*oAQcCI[)@&bnCG;H\^W#'+J+VfO[ZNjXF'@+Pe?YI;kq8O!7WJ6/>GbW`_[MnT91E/eol'
	bTm^=u,IJJ5lD.poMum`/$SP;Lj,6doEk."\m=nfH#0X:Yfbb'MOIr-edqrknBHJ'.M!5h;8]P4A(2
	C5q?EY.TLP&:@>Ph$?Yg`,3D[$dC9?0\Lot/&E.W?j`_I5L)S`l("cMF\gETWuSY0H>CU9[07$'*X?
	.ECCH8hd=RSR5*cX/cU:_gpc,KYucE`q+RG\$$<TcCB`,F1^8,XK7BFNf9bRLZ:?J%"ljN:<;Dmlr<
	<hnCr)p-KUd:j9K#ZR[0<)\a0F9.RDN?6IID#?^a.u`5-q,0f8$Rlf7RA:d.C+<VOe$lIo%e;iX5o1
	`08MhTJ)r/NmJY-DDFd-.d!K_Rt0c0JkU#f>m(GTtJ#XKnY4Yh:U@^b&M*`#/`.YmHm=QB$'%\qX#r
	@k9m_?!'^:&'rj398\D*jgar,Hmr+Pss6/[R]X5WiKS!V5MLl\*=I@$r(oGZ,`<UmS89I+G&ERttG\
	?uGTYJ4QI94'p5&O$_NuCGa:bXG:/cc`G"U[\/_&o4D^R9qQ.&+2[(Oj,+Wf%(,h7@a%l,]TuYHJJ]
	c-7[nR\D2Y'FNK/2Is\:pn`I55s*LoL%75rm!\:SWNQ!$7lE$\oEiU0_1I2l(qmg9mW>fI686_Rj.5
	jeJn6.1(2"3)s(MJI+9@R=1DURoE!UFZS]K(#"X+#C.&IfNW]P,?eu`-]QAr!;/J\^(*&mqjc^33aT
	ho-+@KOjAb]rUUj[oC9NQ;ek29;O&*`'Q"/";k+Tg7'`6o+_GJCHppi2q`P`s):,hk.sAA13FNUdVO
	CqH6RTVp:'Y5Q5n4OSA+pK<_\dBuc<WYCcMO?Q@,LgL<7c;$W^J0;HuM+9W5^S^q4aWOpsd@r>fnk2
	So.FRL?T3's"<s%-?!6l<9^F=LQcfaIk'F`fAVN=fHg&+@:55-A/$,0Q;a!e\;h*Lf(XaXKL[!4-!c
	aqqcW-G6]?LB/PBBUY6(jb9r[dt)"0%,6E?I7ZC,@h&U9d0N'0B"I*cP&=W6+,5Q+jm*N\UX=lkGu&
	=k,OT/8_;BQ/7_Ej-C_!.H&/7RV2C0;(69I-DBB1!=ngL'RTsV0@Thl"/7EIHaEf0:a;]5Cc66VSZQ
	6I"!Xu@+C"F9t&6)h$QNJ4Z`l)J2%pja/;XcRC:V`\qo7%>:kj303LOY7gGU9[=BWi@Q<4J,J?\$`W
	[j`9>Hfs<YZk"J7Y4O9e.Z4_2'41CGG!H$J1+c^co`cG$^kE[l+]TgZQ_ZQAl-Ye&1(foL%GJNZubA
	I=\jNa+r25c`$-\_d!-SF>`c$*"bnA5Pd&@G[NL1-s#NT^QJE4K3ch;$eqe%2otU'SNkO0)Wq;?3TU
	rR^1M8Y>J.e=2WqS,EtAVsr+7Ye3:5n)T/gm&69l`j'CTj4Lg;;));SI-5M&;0PNH(V+qhEd59(q77
	:UfXZ&NG4+.BMM_c:\fMCIl=*8Wl$0*1n"*V,VrbVTeamc>-QOiL!'a[j8ho<N"<e&.<_R:uIoHWYJ
	V2XkU$7DS=[%EK"=$\$kMQ0&ZlWD`GVR/k$j8kX2:&F1\IrM\).p2mX&c?cjWb$4(kt>!Fg\$7MYDA
	UHms%/2@dgf]sf_>hK$8KIS*p\IF@UM*pf19J:qKEM*`+C1)HeU%mj[5%,n0(A&?QmTqML1]D_'#VU
	*PTY.VJl],a[1oi.[R2gCCs)EWsbZM?gV]Lt.iOP5c/>+CgY](/@X'N)9IE>TW8![nBi9rGA-g7>Hp
	7+aXH[X[R9gD:QXZ-eK.]oBqplJdKII8NiR<EiZXD*$*VkL$%p@CP!"\(botnDUa&J]RXWD/Cjs8kK
	_<Vt(nb'K\+MDAp)9hWgu3=Kl-_8`3<f3W%U'e"ljrNP(a<ThrgW8>ReE0a-h(!L0!@m!e[%Zs0JWq
	t?0?mV_+Ij[j)-=03DO_eQTRXK2hGf<1D3EY\ZChfAPHFN"fO]NbF:l!M_*b:=[),oo-29QT>^V:Jn
	:-m.rqHKcs0/5Y;,CW^2$p-3Z_&F&EBJB;>bhJ?i+bO_rAXf/4u_28sXAEi7<jcOgQrqM6=VYgP@_8
	H64h2ZJOlBs?ebm(55\!.MW!DM.KJ-<m>:brccmn<T3VCVWDrT)dYhZZGG0Gk5H\]2"fJq/]7#nTGM
	d*Rkr\k#n?2".f$$2_u>^Ei3E4V/;GBQFSYB4Z]qWhhRd8-hNf"q^:7iL-29.seIBTbbt(s41u^CK<
	qq#AJ*pLEHlLC:2KZR-9u#'FI,;"9\c,GW[%O/;p>tQ_L@Sk*gC1'^rD[Ng<&QqT8!\!;\90X=aNKD
	r`TCVbt6obG&O5$4H$sZ/XWa124<T4?Q;4T0A86rdItO<p@AD3h6a-P6iN]-!7Zr+m.(@HX1ct2t-b
	HR`A77"JcKsz8OZBBY!QNJ
	ASCII85End
end

// PNG: width= 90, height= 30
Picture pLB_Cog
	ASCII85Begin
	M,6r;%14!\!!!!.8Ou6I!!!"&!!!!?#R18/!3BT8GQ7^D&TgHDFAm*iFE_/6AH5;7DfQssEc39jTBQ
	=U%adj95u_NKeXCo&'5*oW5;QerS2ddE(FP;'6Ql#oBiuS1dW/'TBu9uWoJd,<?@,Ku[OT_*C6>N&V
	5fgKO-Z!0m]BHZFE$kSY/b+).^UIV80KE-KEW"FQ9_,eF^bT7+UPEE3;1R;IlL+VJZGMtc/)57aYSB
	X^5kLMq7Sp9q0YBG@U214ph]9b)&eA4@(!#Ip%g0/F0_UH>\E'7M.=ui]mI6`m'G"T)B'OA[IVmQDm
	=?Eat`-Qd[5bR1c#F]`cT(Se*XgJen[DI=<Y8YF7N!d-X+iY2pq;SVDt;siWS_bs$!E]WT^pRhs]aH
	M%csq\akmJ4tMo<HGU1WqT,5cib%XrHM`>*]/&U:](RQ7QDj\u&#cMNi8>`?8;-?rCcXX>+1^gW10F
	n!h-2G-M;R\a+TD2op'?`R]"%W!DpJmO],i(@-(/F?,;L7Vkt%YOa,f\8+CV@lHVOEMgMZPnh$6>!V
	MTYBm^8f[O,?Z$2MnH6.T'JWSM4HtSissRo-):4d:ecoe5Tn^(gUEQm+o94@;L(/[91%aXk:!pP;mm
	\kh$s.7qbe%=-p1eBtDs*CHp:+$CUY\0A,jM0:4J2&pY-HWBG?nb`"BE/M-#*)+E?I*C/(r;J]APNh
	3!Ea<(u)`o?0R`ma=QV<n?GV/s3:I0Wf2_M0p@:__T%OEl+sL@10K8&ViQgR(0Q3qMLYA':/iba:,;
	]Y$@ACMV&9b[fD4A`Vq5+A!37VD0na`;0#fWNWKq#f5N>Mt)$S['[2:?=(p2$Q$$NX_cXoJ`iVOcHm
	Rb+"_b#*b4@tp)Xq9r*1_<^IVlpMJ=kE>MhiHa2]]q9<d*4(lA_8$4ej2NM5Z!#`oc=+Ttk-]%D5"O
	Yiu,o$V/I<=@2fN3Ds,PNfIEnqn6C?^[OYDs4q2k*s6TFu+@1>SKUmdko@B5>Pp)-]8`l_Ig,/1c.T
	K'Z+asa)qDc*mqZAKmijlOd;;&H$MEMWY1:\q<G#aaNVlho?TWCGL35!G658MH$RpQ,/[:S#==eP-@
	T+s%'h-7&.0)\eW6j@1gNW'FYlgRIid1g1dP0.MtL)"o@*4A+B&XU\9bRSJWg?B!keI%b6T7FS'?W(
	@7j-a-n[,Adkh%U((6"oN9G"iRV$rmb0"2lqXk08!N&V_b13Oo*tc)h=[L]E@W<ihr:]%Dbs-*cCbc
	T^`<b81D1(d_gue7JX)rMl-Q!ag!0.a4mbCL5'MQH2X<p3`1nA#69QNiWDnN[J^:kIm`JPCXo#W8lo
	?KFN_dOMp#7s\i!;q:1Vb`q^Za5j'0Sg8ALVn\tm:OM*.G/IF;=K4S+O/0U]^`u\O,i'69Y*0'f,SH
	:Kp)mH3.EQ3hDf%?f;W[LbJbR7R5Lb$8U4I7,[8ZM7fU7H5(>6BlSlr1cG6"263q!"YH.\`>aLYN(!
	e3hZYm666$Mq_c(_GHO?%CE(rnI-UV=I6M\e$%CXt$`9q"IB8d`/4e)0&Dcf`43oobf6MqdV?d>\73
	X/dI(2jZ+#[Nt'sQg5KRFl_^rEVX>cZ5h3g1gR#*IhTZf+KqmsVrW[`UcE6>MM*N0fMcT[S!:h'=ju
	EaL@5>OfU]WXd+d/JISLYCQZ_BPkB$IiAYUBq1l^ecC4a8EYJ'WJ,pak5V59k6$F23m\(d</D&W$.c
	&:n64N(\`j0%h;m7iB=j)(R^kh8BUCfn<.JP[26K1F7\>0JOc56jWCOX(6k=i$m^+A%<G5QZ.i$e8]
	01cU/k$R?&@c[1P\L>tK[OkSMmQ7lT_puI&4&#-'R99q+p;3\Sn`Ic2G%kDj/"N6oB<E0?Z6Kl"@,Y
	?4P5G,Nu\q^LTs(Vj(g3YCoZEl;W9!T)>ePG,S!10n_Y\rP(FBr90.J3;jVP^MYp<-ML6>(=)*bFsB
	&uWX-VWg'<P8G@$+\A?J1@G'F,/Z.<q/;.,=8I?;gFp>>;IjEQPE_:7`.R+3bElA@DB6<k@ksJ9lg(
	=CVM=g<G(^E#SiiFHZ8.qF-^ppkE&\[U*_);<'Lfk*Fq]^#W339=Q'IWpj"KgUWU]1,ETW<&G^OtOH
	MkGSX1r0@A!=Gd&>m2;$s(nF8b!K?8R`eZjO8V38_I$<1AcnP!)B*`T#3.X=hE[s8PMoFf*/H2@32u
	N'Len`GUk?n:K*@=9gMMi1ES9gF7flQcCD`2n^,h:`S5=GNQ^G#@^/a:?]W`PV50n4Xue>QVk8E1=]
	lWKB?uV(SiXjL_hVC,G-+W-bHd)[C`^u(BPM:VV58ltJcZ8d$CEhp-Ek)Qb.'^'f)4eT`-]7*:O[1>
	WO?>HRRZ3%&Aum4gNS.jU=.^S*B#H\'0H3Z)tG>O;)WW1_IO09[>@PI4Y3R7J0]^!UgQhj+u1-,O_#
	4q7aj2?Y5.V]ph=D*J`g8?n%JH:8P)K%MLrfVTs(X1]A:d+mFtdNBG"";'8siHNZC4&bKJq`%mNb7r
	SW;=`2-+n=L)HDOsFHoS$CX_6m<3W758iRSt7"9?7u`s%5^"&NsfV]&fKiR),nod"F>!-ZZj257RH'
	BCpkU2>pgE:BYW?=m,Fa:(R[)N0g+8U_d1?h6?87.J?1.S8QM+N.?c/5H^[K9Qq/KSe*09PFi*)kOs
	Cpa8151hB![K\`b9:/2t#`0h)TQGGW^_mOCaj@jCA@uU*q95,uIW@7!X&<O\"P/_=YGg"!\PP(fcMs
	XX8]B01I/5GVm/oHal.qLkA^e?r`*cq5<6cMYpEN]i/%/Vl(p'^fIMd8C\oHLnT1!)7oQm]mKeID*1
	Nn<=:2#ZkEe#YpHHH,[1j)T%7I4;@/)4cu_Q1)PbZM:[@i"UEI%;r>p03XoVZ_RkU=+$'7#=USG/bL
	8-+m<=>h,iqN=A8kNQ3E0-ce+TlUO7L$\:&5CW07\^Y5(=Lpj3bn5fXf]+hD?H]7Wq?&[-c"7hNK0#
	/)B'Mj<HT8ma3CurC,*bh\'aHR7Z[QbX-ZmAk800m):lpO:?P+(!;TY'R\j"AmjWGZ^'4n^bf?W3J@
	>&NBK1HqWkVhSJ@2:%U$9,hq:d,G1cCmI-TcrP\J*Ut[2@6?BcK3XN6]^DH?sm>]m;PWk0+t]M3*pb
	_i5ToaNr1&dko4ib1O7G-^#a3R58IWd+6c;6ULrU<E06&]A7B"H::^+p=jM"Cht@E-\k9W-F%RN7[f
	g9`s&hll-^kfa*7KZfMd!Yc("^(?tb@(G_h8BF>IEA!<RhTlND,!db&r!Y3m>2$M&6dHhJmnR;"(TN
	7k9deNFM:rt_%M:_\bI5MO2h<A0N#R:ZSrC"&ps\iY*%&:=-;@IrX+"G9!l_&sOI?=_'7)$hsk)[Og
	CfZ9V$@mNB]AS#G_>V6^Z_/)$iF?19\*_+U8'Lh!@O$@74\ogtP<K2K,l#,B;0e6NJ*#oS35C1Gqa?
	[/#EMYbdp\'g083t^Hm.OC']=?TCh[+\@'/CD^$lb9k>s0TnKIaqh4fI!&lDq*\4*U*,*??/2AnId;
	.P@%q^Y_gV7L#<Y@CP!Nm,EKn=&i8<r@!PTa5]H_PQ](fBqn;iMUE,Pl5E-<#!*W9G0Gifi6\]*<o4
	FnfrX,WF^_\F$1nF]^fU-bLO!=bm%5lG.k3$IWMqUu'c@l,R*B4I#7$5RGsBAo;S"q!W&r%8C2/"PK
	bsa<4<J73edT0;DW#W4)GM@uFDONL"9R+_N^([T!(k&'aJc*VLHV&^R;!(_#4Z$g7@#[rnEd4b][m6
	"s2Cf6FX#Yth+![-Bc9;DCc35!#ZOe]>R5l%A3s9r*"E3cZ^HAq!@!X3ZLR*.A7oQ8om.\p[kV)RYJ
	YS_VJ,ke(!91A):`eg`H3<A0s%C//=2RBq,pC2:S]*\P@U_Oa*3.T[g!Hf.uI$^Z574:Ig+a&Rh&b)
	g:i!IBPVCY]Y$?Mc-nM/==coe'#A=jP*M;$C2,4.LP+[KAA[:ZiG]VW6ipmf;5gRtUogbYmG#,M=Y+
	23\;kLm>:;.qMlKqn)uClJ![.i',6Vo?VRu"<5C39QHia.rIUXNcq/492/E<t4:q=5j]#0#BT^2C8R
	r=7NOaA&EGCimE'I"(o(b72sE7cRPmWBnj]tHh/;(=(Hs(iTH$*GN/REC6P4"-OQ)1Z*S9OlNXqYG,
	/qFo"%Xt5WcH)DMDo_@'gn2mp\l)\(b!Y0Pa!2n,Nj%-R@Yjn?WT$E#t(FUbj=.R08ON,:0qYL%:/M
	0]<Q1)2)G':0@s*h8ZZ<4MLPu4'A3d&SI6l9j+cCI%0ltDh?G(&13<0N\Q=?t??;p9Q8$59_n3Rli=
	5b`N"1`$#>;[JNr+$!*^fli%qGlHBaGd#r_gndbH0<a<pR0sE5L0:j)I_t)\EH/7Wqt8QJMd<r<&WK
	8J3cuoH9hij#22_bS-?/1q+bUC@(DjDc_1Dg*LCYK([C$_m"OB=44C54XF6CiRHM)#JSik-Qi#lgdX
	C:AAV;n0(-%L_0m!T,"d5MVG@HhTKZ87K.p%WH^Xh3kCpXcTY<@p]%p!H!PcGm8Ma`dWI-i:%O`.@A
	\EMhGlp>Rk7O<G2m`*r,h[u\8;4r,bUaQp%EDN*J]D4B1hFXuppq^tpModARV5%<QlNN?L%hAEkIlW
	/#`^]Bs#-d.f-97RH2#l<o@ZXZ&T?iRH-<%N9SU+'so7AAgW2mil0fWaM)A,6@)4Rp@WFC0@Y,uIN:
	5uCJkMPAJFd6VVd/UR6[rI=?0fVgq/kI,s^(YARDN54WD\PD#"bV<C5XKS<`]Fql#m@)ul]O#Nn]-2
	[lB);<HM,sPARkJs:;d4_S!/nh?qGe7@I:AsnMi7DjM_D$2XW>fsY^ZQI8$;`nm!f"mg^^mq'BNs/!
	!!!j78?7R6=>B
	ASCII85End
end
