﻿#pragma TextEncoding="UTF-8"
#pragma rtGlobals=3
#pragma IgorVersion=8
#pragma IndependentModule=Updater
#pragma version=3.30
#include <Resize Controls>

// updater headers
static constant kProjectID=8197 // the project node on IgorExchange
static strconstant ksShortTitle="Updater" // the project short title on IgorExchange

// Use this software at your own risk. Installing and updating projects 
// requires that this software downloads from the web, and overwrites 
// files on your computer. No warranty is offered against error or 
// vulnerability to exploitation of this software or of third-party (web)
// services.

// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL 
// WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED 
// WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE 
// AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL 
// DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR 
// PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER 
// TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR 
// PERFORMANCE OF THIS SOFTWARE.

// ---------------------------------------------------------------------------

// This file should be saved in the Igor Procedures folder (look in the 
// Help menu for "Show Igor Pro User Files").

// Look in the Misc menu for "IgorExchange Projects..." to open a control panel 
// that allows you to browse and install user-contributed packages. 

// If you use the control panel to update projects, it's best to set 
// the setting for Text Editing like this:
// Look in the Misc menu for Miscellaneus Settings, then select Text 
// Editing -> External Editor and check Reload Automatically -> As Soon 
// as Modifiucation is Detected.

// If you're developing a package and want to make sure that it's 
// compatible with IgorExchange installer:

// Compress (zip) your project file(s) before uploading a project release.
// This is recommended even for single-file projects.

// Make sure that the version number fields are filled out correctly. If 
// your file version is 1.03, set the patch version to 03, NOT 3!

// Be aware that this information is recorded in the install log on the 
// user's computer at the time of installation. If files are moved or 
// replaced by the user this will interfere with the updating function of
// the installer.

// Design your project so that files do not have to be moved into other 
// locations after installation. You can provide an itx script as part of
// your package if you need Igor to create shortcuts for XOPs, help 
// files, etc.

// ---------------------------------------------------------------------------
// Version details
// 3.30 14/6/20
// Bug fix, forgot that duplicating a 2 D wave with zero rows does not 
// preserve the column dimension labels 
// Added a button to settings pane to clear cache. Will help if cache is corrupted!
// ---------------------------------------------------------------------------

// Command line options

// Updater#Install("archull")
// Installs archull project in location chosen by user

// Updater#Install("https://www.wavemetrics.com/project-releases/7399")
// same as above

// Updater#Install("archull;baselines;", path=SpecialDirPath("Igor Pro User Files",0,0,0)+"User Procedures:")
// Installs archull and baselines projects in specified location

// Updater#CheckAndUpdate(ProcFilePath)
// Checks for updates, downloads and installs if found. Minimal dialog.

// updater#InstallFile(filename) function for use in itx scripts

// ----------------------- Details ----------------------------------

// An install log and cache file are saved in the User Procedures folder.
// When projects are installed using the IgorExchange installer, details 
// of the installation are recorded in the install log. Moving or deleting
// the install log or any of the installed files will interfere with the 
// update-checking functions.

// I'm grateful to Jim Prouty and Jeff Weimer, both of whom have made 
// suggestions and tested development versions of this project. Please 
// let me know about any bugs you find.

// Required igor version is not acccesible for recent project releases. 
// In older releases this information was encoded in the version string, 
// but that's not currently enforced

// Procedure files are not allowed to self-check for updates more 
// frequently than the value set in updater preferences. The default 
// value is weekly. Procedure files do not need to ask updater to check 
// for updates if they are part of a package that has been installed 
// using the IgorExchange installer.

// feedback? ideas? send me a note: https: www.wavemetrics.com/user/tony

// -----------------------------------------------------------------------------

constant kResizablePanel=1	// if non-zero, the Install User Project panel will be resizable
constant kNumProjects=230 // provides a rough idea of the minimum number of user-contributed projects that can be found at wavemetrics.com
constant kRemoveSuffix=1 // if a single file is downloaded that has a name that looks like it has
 								// a suffix added to make the file unique, remove that suffix.
strconstant ksIgnoreFilesList=".DS_Store;" // list of files that shouldn't be copied, wildcards okay
strconstant ksIgnoreFoldersList="__MACOSX;" // list of folders that shouldn't be copied, wildcards okay
strconstant ksBackupLocation="Desktop" // set the value of dirIDStr for SpecialDirPath here
strconstant ksDownloadLocation="Temporary" // displayHelpTopic "SpecialDirPath" for details
strconstant ksLogPath="User Procedures:IgorExchange Installer:" // path to location for log files, starting from User Files

// Other settings are accessible from the control panel: Misc -> IgorExchange Projects...

menu "Misc"
	"-"
	"IgorExchange Projects...", /Q, updater#makeInstallerPanel()
	"-"
end


// --------- Hook and related functions for initating periodic checks ------------------

// strangely, in an indepependent module this must be static
static function IgorStartOrNewHook(igorApplicationNameStr)
	string igorApplicationNameStr

	Variable startTicks = ticks + 60*2	// start in 2 seconds - plenty of time to allow an experiment to be opened 
	CtrlNamedBackground BGUpdaterDelayedStartDL, proc=updater#StartBackgroundCheck, start=startTicks	
	ExperimentModified 0 // this will allow an experiment to be loaded by double-clicking	
	return 0
end

// if we always open Igor by double-clicking an experiment file, this will enable 
// checking update status in a preemptive thread
static function AfterFileOpenHook(refNum, fileNameStr, pathNameStr, fileTypeStr, fileCreatorStr, fileKind)
	variable refNum, fileKind
	string fileNameStr, pathNameStr, fileTypeStr, fileCreatorStr
	
	CtrlNamedBackground BGUpdaterStartDL, proc=updater#StartBackgroundCheck, start	
	return 0
end

// start premptive download and cooperative bg task
function StartBackgroundCheck(s)
	STRUCT WMBackgroundStruct &s
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)	
	if( (datetime-prefs.lastCheck) < prefs.frequency)
		return 1 // Getting here should be blazingly fast!
	endif
	
	string ProjectsList=LogProjectsList()
	if(strlen(ProjectsList)==0)
		prefs.lastCheck=datetime
		SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
		return 1 // task doesn't repeat
	endif
	
	// create an input folder for preemptive task
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:Installer
	NewDataFolder /O root:Packages:Installer:dfrIn
	DFREF dfrIn = root:Packages:Installer:dfrIn
	
	// put a list of projects from the log file and the timeout prefs setting 
	// into preemptive task input queue
	wave/T w_Projects=ListToTextWave(ProjectsList,";")
	MoveWave w_Projects dfrIn:w_Projects
	variable /G dfrIn:v_timeout=prefs.pagetimeout
	
	WAVEClear w_Projects
	DFREF dfrIn = $""
			
	Variable/G root:Packages:Installer:threadGroupID
	NVAR threadGroupID=root:Packages:Installer:threadGroupID
	threadGroupID = ThreadGroupCreate(1)	
	ThreadStart threadGroupID,0,PreemptiveDownloader()
	ThreadGroupPutDF threadGroupID, root:Packages:Installer:dfrIn
	// preemptive thread is now downloading
		
	// start cooperative BG task to deal with downloads when they're done
	CtrlNamedBackground BGCheck,period=2*60,proc=BackgroundCheck,start
	
	return 1 // task doesn't repeat
end

threadsafe function PreemptiveDownloader()

	DFREF dfrIn = ThreadGroupGetDFR(0,inf) // waits here for dfr
	if( DataFolderRefStatus(dfrIn) == 0 )
		return -1		// Thread is being killed
	endif

	NVAR timeout=dfrIn:v_timeout
	wave /T w_Projects=dfrIn:w_Projects
	
	// create output folder
	NewDataFolder /S resultsDF
	Make /T/N=(DimSize(w_Projects, 0)) w_Downloads
	
	// save some text from project web pages in w_Downloads
	w_Downloads=getLatestReleaseInfoFromWeb(w_Projects[p], timeout)
	
	// reload the projects list
	wave /T w_ProjectsList=downloadProjectsList(timeout)
	Duplicate w_ProjectsList :w_ProjectsList

	// ThreadGroupPutDF requires that no waves in the data folder be referenced
	WAVEClear w_Downloads, w_projects, w_ProjectsList
	ThreadGroupPutDF 0,: // Send output data folder to output queue
	KillDataFolder dfrIn
	return 0
end

threadsafe function /WAVE downloadProjectsList(timeout)
	variable timeout
	
	Make /free/T/N=(0,7) w
	setDimLabels("projectID;name;author;published;views;type;userNum", 1, w)
		
	variable pageNum, projectNum, pStart, pEnd, selStart, selEnd
	string baseURL, projectsURL, url
	string projectID, strName, strUserNum, strAuthor, strProjectURL
	
	baseURL="https://www.wavemetrics.com"
	projectsURL="/projects?os_compatibility=All&project_type=All&field_supported_version_target_id=All&page="
	
	// loop through listPages
	for(pageNum=0;pageNum<100;pageNum+=1)
		sprintf url "%s%s%d", baseURL, projectsURL, pageNum

		URLRequest /time=(timeout)/Z url=url
		if(V_flag)
			return w
		endif

		pStart=strsearch(S_serverResponse, "<section class=\"Project-teaser-wrapper\">", 0, 2)
		if (pStart==-1)
			break // no more projects
		endif
		pEnd=0
		
		// loop through projects on listPage
		for (projectNum=0;projectNum<50; projectNum+=1)
			pStart=strsearch(S_serverResponse, "<section class=\"Project-teaser-wrapper\">", pEnd, 2)
			pEnd=strsearch(S_serverResponse, "<div class=\"Project-teaser-footer\">", pStart, 2)
			if (pEnd==-1 || pStart==-1)
				break // no more projects on this listPage
			endif
			
			selStart=strsearch(S_serverResponse, "<a class=\"user-profile-compact-wrapper\" href=\"/user/", pEnd, 3)
			if(selStart<pStart)
				continue
			endif
			
			selStart+=52
			selEnd=strsearch(S_serverResponse, "\">", selStart, 0)
			strUserNum=S_serverResponse[selStart,selEnd-1]
			
			selStart=strsearch(S_serverResponse, "<span class=\"username-wrapper\">", selEnd, 2)
			selStart+=31
			selEnd=strsearch(S_serverResponse, "</span>", selStart, 2)
			strAuthor=S_serverResponse[selStart,selEnd-1]
					
			selStart=strsearch(S_serverResponse, "<a href=\"", selEnd, 2)
			selStart+=9
			selEnd=strsearch(S_serverResponse, "\"><h2>", selStart, 2)
			strProjectURL=baseURL+S_serverResponse[selStart,selEnd-1]
			
			selStart=selEnd+6
			selEnd=strsearch(S_serverResponse, "</h2></a>", selStart, 2)
			strName=S_serverResponse[selStart,selEnd-1]
			
			if (strlen(strName)==0)
				continue
			endif
			
			// clean up project names that contain certain encoded characters
			strName=ReplaceString("&#039;",strName, "'")
			strName=ReplaceString("&amp;",strName, "&")
			
			projectID=ParseFilePath(0, strProjectURL, "/", 1, 0)
			
			InsertPoints /M=0 DimSize(w, 0), 1, w
			w[Inf][%projectID]=ParseFilePath(0, strProjectURL, "/", 1, 0)
			w[Inf][%name]=strName
			w[Inf][%author]=strAuthor
			w[Inf][%userNum]=strUserNum
			// search for other non-essential parameters
			
			// project types
			selEnd=pStart
			do
				selStart=strsearch(S_serverResponse, "/taxonomy/", selEnd, 2)
				selStart=strsearch(S_serverResponse, ">", selStart, 0)
				selEnd=strsearch(S_serverResponse, "<", selStart, 0)
				if(selStart<pStart || selEnd>pEnd || selEnd<1 )
					break
				endif
				w[Inf][%type]+=S_serverResponse[selStart+1,selEnd-1]+";"
			while (1)
			
			// date
			selStart=strsearch(S_serverResponse, "<span>", pEnd, 2)
			selEnd=strsearch(S_serverResponse, "</span>", pEnd, 2)
			if (selStart>0 && selEnd>0 && selEnd < (pEnd+80) )
				w[Inf][%published]=ParsePublishDate(S_serverResponse[selStart+6,selEnd-1])
			endif
			
			// views
			selEnd=strsearch(S_serverResponse, " views</span>", pEnd, 2)
			selStart=strsearch(S_serverResponse, "<span>", selEnd, 3)
			if (selStart>pEnd && selEnd < (pEnd+150) )
				w[Inf][%views]=S_serverResponse[selStart+6,selEnd-1]
			endif
			
		endfor	 // next project
	endfor	 // next page
	return w
end

function BackgroundCheck(s)
	STRUCT WMBackgroundStruct &s
	
	NVAR threadGroupID=root:Packages:Installer:threadGroupID
	DFREF dfr = ThreadGroupGetDFR(threadGroupID,0)	// Get free data folder from output queue
	if( DataFolderRefStatus(dfr) == 0 )
		return 0 // task repeats in main thread until folder is ready
	endif
	
	wave /T w_downloads=dfr:w_downloads
	wave /T w_ProjectsList=dfr:w_ProjectsList
	
	CachePutProjectsWave(w_ProjectsList)

	// check status of each project
	variable i, local, remote
	variable numProjects=DimSize(w_downloads, 0)
	string fileStatus, projectID, keyList
	variable UpdateAvailable=0
	
	for (i=0;i<numProjects;i+=1)
		keyList=w_downloads[i]
		if(strlen(keyList)==0)
			continue
		endif
		keyList=ReplaceStringByKey("ReleaseCacheDate", keyList, num2istr(datetime))
		CachePutKeylist(keyList) // insert download into cache file
		projectID=StringByKey("projectID", keyList)
		fileStatus=getInstallStatus(projectID)
		if(stringmatch(fileStatus, "complete")==0)
			continue
		endif
		local=str2num(LogGetVersion(projectID))
		remote=str2num(StringByKey("remote", keyList))
		if (local<remote)
			//if(releaseIgorVersion<=currentIgorVersion)
			UpdateAvailable=1
			// if the new release is not compatible with curent Igor version will not find out until an update is attempted.
		endif
	endfor
	
	variable tstatus=ThreadGroupRelease(threadGroupID)
	if( tstatus == -2 )
		Print "Updater thread would not Quit normally, had to force kill it. Restart Igor."
	endif
	KillVariables threadGroupID
	
	STRUCT PackagePrefs prefs
		
	if(UpdateAvailable)
		DoAlert 1, "An IgorExchange Project update is available.\rDo you want to view updates?"
		if(v_flag==1)
			LoadPrefs(prefs)
			prefs.paneloptions += (prefs.paneloptions&1) ? 0 : 1 // set tab to 1
			SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
			makeInstallerPanel()	// use the data we stashed in the cache to create panel
		endif
	else // kill package folder?
		DoWindow InstallerPanel
		if(v_flag==0)
			KillDataFolder /Z root:packages:installer:
		endif
	endif
	
	LoadPrefs(prefs)
	prefs.lastCheck=datetime
	SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
		
	return 1 // bg task doesn't repeat
end

// ------------------------ package preferences ----------------------

// define some constants for saving preferences for this package
strconstant ksPackageName=Updater
strconstant ksPrefsFileName=UpdaterPrefs.bin
constant kPrefsVersion=100

structure PackagePrefs
	uint32	 version		// 4 bytes, structure version 
	uint32 frequency	// 4 bytes
	uint16 pagetimeout	// 2 bytes
	uint16 filetimeout	// 2 bytes	
	uchar options		// 1 byte, 8 bits to set
	uchar paneloptions	// 1 byte	
	uchar dateFormat	// 1 byte
	uchar tab		// 1 byte to record active tab
	STRUCT Rect win // window position and size, 8 bytes
				// 24 bytes
	uint32 lastCheck	// 4 bytes
	uint32 reserved[121]	// Reserved for future use
endStructure // 512 bytes

// set prefs structure to default values
function PrefsSetDefault(prefs)
	STRUCT PackagePrefs &prefs
	
	prefs.version=kPrefsVersion
	prefs.frequency=604800 // weekly
	prefs.pagetimeout=3
	prefs.filetimeout=10
	prefs.options=1 // bit 0 save backups, 1 use more DoAlerts
	prefs.paneloptions=0 // bit 0: tab, bit 1: bigger panel
	prefs.win.left=20
	prefs.win.top=20
	prefs.win.right=20+520
	prefs.win.bottom=20+355
	prefs.lastCheck=0 // last check time, rounded to seconds
	variable i
	for(i=0;i<(121);i+=1)
		prefs.reserved[i]=0
	endfor
end

function LoadPrefs(prefs)
	STRUCT PackagePrefs &prefs
	
	LoadPackagePreferences ksPackageName, ksPrefsFileName, 0, prefs	
	if (V_flag!=0 || V_bytesRead==0 || prefs.version!=kPrefsVersion)
		PrefsSetDefault(prefs)
	endif
end

function getScreenHeight()
	
	 variable numItems
    string strInfo
   
    strInfo = StringByKey("SCREEN1",IgorInfo(0))
    numItems = ItemsInList(strInfo,",")
    return str2num(StringFromList(numItems-1,strInfo,","))
end
	
function makePrefsPanel()
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	DoWindow /K UpdaterPrefsPanel	
	
	variable WL=150, WT=100 // window coordinates
	GetWindow /Z InstallerPanel  wsize	
	if(v_flag==0)
		WL=V_right-100; WT=V_bottom-310
	endif
	
	variable sHeight=getScreenHeight(), sMinHeight=700	
	if(prefs.paneloptions&2)
		if(sHeight>=sMinHeight)		
			execute/Q/Z "SetIgorOption PanelResolution=?"
			NVAR vf=V_Flag
			variable oldResolution = vf	
			execute/Q/Z "SetIgorOption PanelResolution=96"
			// make a correction to window coordinates
			WL*=ScreenResolution/96; WT*=ScreenResolution/96
		else
			prefs.paneloptions-=2
		endif
	endif
	
	variable frequencyPopMode
	switch (prefs.frequency)
		case 0:
			frequencyPopMode=1 // always
			break
		case 86400:
			frequencyPopMode=2 // daily
			break
		case 604800:
			frequencyPopMode=3 // weekly
			break
		case 2592000:
			frequencyPopMode=4 // monthly
			break
		default:
			frequencyPopMode=5 // never
	endswitch
	
	NewPanel /K=1/N=UpdaterPrefsPanel/W=(WL,WT,WL+230,WT+310) as "Settings"
	ModifyPanel /W=UpdaterPrefsPanel, fixedSize=1, noEdit=1
	
	variable left=15
	variable top=10
	
	Button btnClearCache, win=UpdaterPrefsPanel,pos={left,top},title="Clear Cache",size={90,22},proc=updater#PrefsButtonProc	
	top+=30
	PopupMenu popDate, win=UpdaterPrefsPanel,pos={left,top},value="mm/dd/year;dd/mm/year;Sunday May 25, 1980;year mm dd;year/mm/dd"
	PopupMenu popDate, win=UpdaterPrefsPanel,title="Date Format", mode=prefs.dateformat+1, fsize=12
	PopupMenu popDate, win=UpdaterPrefsPanel,help={"Format for project release dates in control panel"}
	top+=30
	PopupMenu popFrequency, win=UpdaterPrefsPanel,pos={left,top},value="always*;daily;weekly;monthly;never;"
	PopupMenu popFrequency, win=UpdaterPrefsPanel,title="Check for Updates", mode=frequencyPopMode, fsize=12
	PopupMenu popFrequency, win=UpdaterPrefsPanel,help={"How frequently to check for updates for installed packages"}
	top+=25
	TitleBox titleFreq, win=UpdaterPrefsPanel,pos={left,top}, frame=0, fsize=12
	TitleBox titleFreq, win=UpdaterPrefsPanel,title="Checking for updates more\rfrequently than weekly is not\rrecommended."
	top+=50
	TitleBox titleFreq2, pos={left,top},frame=0,win=UpdaterPrefsPanel,title="*always is for testing only", fsize=12
	top+=20
	SetVariable setvarPage, win=UpdaterPrefsPanel,pos={left,top}, title="Timeout for Page Download (s)", value=_NUM:prefs.pagetimeout
	SetVariable setvarPage, win=UpdaterPrefsPanel,size={200,20}, limits={1,25,0}, fsize=12
	top+=20
	SetVariable setvarFile, win=UpdaterPrefsPanel,pos={left,top}, title="Timeout for File Download (s)  ", value=_NUM:prefs.filetimeout
	SetVariable setvarFile, win=UpdaterPrefsPanel,size={200,20}, limits={1,25,0}, fsize=12
	top+=25
	CheckBox checkBackups,win=UpdaterPrefsPanel,pos={left,top},size={141.00,16.00},title="Save Backups (update only)"
	CheckBox checkBackups,win=UpdaterPrefsPanel,fSize=12,value=prefs.options&1
	top+=20
	CheckBox checkBackups,win=UpdaterPrefsPanel,help={"Backup files to a folder on the desktop"}
	CheckBox checkDoAlerts,win=UpdaterPrefsPanel,pos={left,top},size={141.00,16.00},title="More Interactive"
	CheckBox checkDoAlerts,win=UpdaterPrefsPanel,help={"Present user with many alerts when checking for updates"}
	CheckBox checkDoAlerts,win=UpdaterPrefsPanel,fSize=12,value=prefs.options&2
	top+=20
	CheckBox checkBigger,win=UpdaterPrefsPanel,pos={left,top},size={141.00,16.00},title="Bigger Panel"
	CheckBox checkBigger,win=UpdaterPrefsPanel,fSize=12,value=prefs.paneloptions&2
	CheckBox checkBigger,win=UpdaterPrefsPanel,help={"Control panel size will be slightly increased"}
	CheckBox checkBigger,win=UpdaterPrefsPanel,disable=2*(sHeight<sMinHeight)
	top+=25	
	Button ButtonSave, win=UpdaterPrefsPanel,pos={15,top},size={100,22},title="Save Settings", valueColor=(65535,65535,65535), fColor=(0,0,65535), proc=updater#PrefsButtonProc
	Button ButtonCancel, win=UpdaterPrefsPanel,pos={135,top},size={70,22},title="Cancel", proc=updater#PrefsButtonProc
	
	SetWindow UpdaterPrefsPanel, hook(hEnter)=updater#hookPrefsPanel
	
	if(prefs.paneloptions&2) 
		// reset panel resolution
		execute/Q/Z "SetIgorOption PanelResolution="+num2istr(oldResolution)
	endif
	
	PauseForUser UpdaterPrefsPanel
end

function PrefsButtonProc(s)
	STRUCT WMButtonAction &s
	
	if(s.eventCode!=2)
		return 0
	endif
		
	if(cmpstr(s.ctrlName, "BtnClearCache")==0)
		return CacheClearAll()
	endif
	
	if(stringmatch(s.ctrlName, "ButtonSave"))
		// run PrefsSync, return value tells us whether updates need to be made 
		variable redraw=PrefsSync()
	endif
	KillWindow /Z UpdaterPrefsPanel
	
	if(redraw)
		makeInstallerPanel()
	endif
	
	UpdateListboxWave(fGetStub())
	
	return 0
end

// hook makes panel act as if save Button has focus
function hookPrefsPanel(s)
	STRUCT WMWinHookStruct &s
	
	if(s.eventCode!=11)
		return 0
	endif
	
	if(s.keycode==13 || s.keycode==3) // enter or return
		variable redraw=PrefsSync()
		KillWindow /Z UpdaterPrefsPanel
		if(redraw)
			makeInstallerPanel()
		endif
		UpdateListboxWave(fGetStub())		
		return 1
	endif
	return 0
end

function PrefsSync()
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)

	prefs.options=0	
	ControlInfo /W=UpdaterPrefsPanel checkBackups
	prefs.options+=1*(v_value)
	ControlInfo /W=UpdaterPrefsPanel checkDoAlerts
	prefs.options+=2*(v_value)
	
	ControlInfo /W=UpdaterPrefsPanel checkBigger
	variable redrawPanel=((prefs.paneloptions&2)!=(2*v_value))
	prefs.paneloptions-=prefs.paneloptions&2
	prefs.paneloptions+=2*(v_value)
	
	ControlInfo /W=UpdaterPrefsPanel setvarPage
	prefs.pagetimeout=v_value
	ControlInfo /W=UpdaterPrefsPanel setvarFile
	prefs.filetimeout=v_value
	
	ControlInfo /W=UpdaterPrefsPanel popDate
	prefs.dateformat=v_value-1
	ControlInfo /W=UpdaterPrefsPanel popFrequency	
	switch (v_value)
		case 1:
			prefs.frequency=0 // always
			break
		case 2:
			prefs.frequency=86400 // daily
			break
		case 3:
			prefs.frequency=604800 // weekly
			break
		case 4:
			prefs.frequency=2592000 // monthly
			break
		default:
			prefs.frequency=inf  // actually, 2^32, ie never
	endswitch
		
	SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
	
	return redrawPanel
end

// save window position in package prefs
function PrefsSaveWindowPosition(strWin)
	string strWin
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	string recreation=winRecreation(strWin, 0)
	variable index, top, left, bottom, right
	// save window position
	index=strsearch(recreation, "/W=(", 0)
	if (index>0)
		sscanf recreation[index,strlen(recreation)-1], "/W=(%g,%g,%g,%g)", left, top, right, bottom
		prefs.win.top=top
		prefs.win.left=left 
		prefs.win.bottom=bottom
		prefs.win.right=right
	endif
	
	ControlInfo /W=InstallerPanel tabs
	prefs.paneloptions-=(prefs.paneloptions&1)
	prefs.paneloptions+=v_value
	
	DFREF dfr = root:Packages:Installer
	
	SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
end

// ---------------- prefs files to hold last update times --------------

// We'll keep a very small binary file in the updater preferences folder to
// store the last update check time for each project that works with updater
// These will not be used in future versions
structure LastUpdateStructure
	uint32	 version		// Preferences structure version number. 100 means 1.00.
	uint32 lastUpdate	// 32 bits should be good for another 20 years or so...
	uint32 reserved[8]	// Reserved for future use
endStructure // 40 bytes

function setDefaultUpdatePrefs(s)
	STRUCT LastUpdateStructure &s
	
	s.version=200
	s.lastUpdate=0
	variable i
	for(i=0;i<8;i+=1)
		s.reserved[i]=0
	endfor	
end

function setLastUpdate(projectID, lastUpdate)
	string projectID
	variable lastUpdate
	
	string prefsFile=projectID+"LastUpdate.bin"
	STRUCT LastUpdateStructure s
	LoadPackagePreferences "updater", prefsFile, 1, s
	s.lastUpdate=lastUpdate
	SavePackagePreferences "updater", prefsFile, 1, s	
end

function getLastUpdate(projectID)
	string projectID
	
	string prefsFile=projectID+"LastUpdate.bin"
	
	// check for time of last update check
	STRUCT LastUpdateStructure s
	LoadPackagePreferences "updater", prefsFile, 1, s
	if (V_flag!=0 || V_bytesRead==0 || s.version!=200)
		setDefaultUpdatePrefs(s)
		SavePackagePreferences "updater", prefsFile, 1, s		
	endif
	return s.lastUpdate	
end

// ------------------ end of package preferences & related functions --------------------

function /DF setupPackageFolder()
	
	NewDataFolder /O root:Packages
	NewDataFolder /O root:Packages:Installer
	DFREF dfr = root:Packages:Installer
	variable /G dfr:stubLen=0

	// create wave to hold names of all available projects
	wave /T/SDFR=dfr/Z ProjectsFullList
	if(waveexists(ProjectsFullList)==0)
		make /O/T/N=(0,10) dfr:ProjectsFullList /WAVE=ProjectsFullList, dfr:ProjectsMatchList /WAVE=ProjectsMatchList
		setDimLabels("projectID;name;author;published;views;type;userNum;;;;", 1, ProjectsFullList)
		setDimLabels("projectID;name;author;published;views;type;userNum;;;;", 1, ProjectsMatchList)
	endif
	
	// create wave to hold names of projects that have updater compatibility
	wave /T/SDFR=dfr/Z UpdatesFullList
	if(waveexists(UpdatesFullList)==0)
		make /O/T/N=(0,15) dfr:UpdatesFullList /WAVE=UpdatesFullList, dfr:UpdatesMatchList /WAVE=UpdatesMatchList
		SetDimLabel 1, 0, projectID, UpdatesFullList, UpdatesMatchList
		SetDimLabel 1, 1, name, UpdatesFullList, UpdatesMatchList // short title if possible
		SetDimLabel 1, 2, status, UpdatesFullList, UpdatesMatchList // update available, up to date, etc
		SetDimLabel 1, 3, local, UpdatesFullList, UpdatesMatchList // version of installed project
		SetDimLabel 1, 4, remote, UpdatesFullList, UpdatesMatchList // remote version
		SetDimLabel 1, 5, system, UpdatesFullList, UpdatesMatchList // operating system compatibility
		SetDimLabel 1, 6, releaseDate, UpdatesFullList, UpdatesMatchList // last release date
		SetDimLabel 1, 7, installPath, UpdatesFullList, UpdatesMatchList // file path or install path
		SetDimLabel 1, 8, releaseURL, UpdatesFullList, UpdatesMatchList // url for new release
		SetDimLabel 1, 9, releaseIgorVersion, UpdatesFullList, UpdatesMatchList // can't retrieve this from web :(
		SetDimLabel 1, 10, installDate, UpdatesFullList, UpdatesMatchList // 
		SetDimLabel 1, 11, info, UpdatesFullList, UpdatesMatchList // 
	endif
	
	make /O/N=(1,4)/T dfr:ProjectsDisplayList /WAVE=ProjectsDisplayList, dfr:ProjectsHelpList /WAVE=ProjectsHelpList, dfr:ProjectsColTitles /WAVE=ProjectsColTitles
	ProjectsDisplayList={{"retrieving list..."},{""},{""},{""}}
	ProjectsHelpList={{"waiting for download"},{""},{""},{""}}
	ProjectsColTitles={{"Project (double-click for web page)"},{"Author"},{"\JRRelease Date"},{"\JRViews"}}
	
	make /O/N=(1,4) /T dfr:UpdatesDisplayList /WAVE=UpdatesDisplayList, dfr:UpdatesHelpList /WAVE=UpdatesHelpList, dfr:UpdatesColTitles /WAVE=UpdatesColTitles
	UpdatesDisplayList={{"retrieving list..."},{""},{""},{""}}
	UpdatesHelpList={{"waiting for download"},{""},{""},{""}}
	UpdatesColTitles={{"Project"},{"Status"},{"\JCLocal"},{"\JCRemote"}}
		
	if(strlen(LogGetVersion("8197"))==0) // updater not in log
		string filePath=FunctionPath("") // path to this file
		string fileName=ParseFilePath(0, filePath, ":", 1, 0) // name of this file
		string strVersion=num2str(getProcVersion(filePath)) // version of this procedure
		string installPath=ParseFilePath(1, filePath, ":", 1, 0) // location of this file
		LogUpdateProject(num2str(kProjectID), ksShortTitle, installPath, strVersion, fileName)
	endif
	
	return dfr	
end

threadsafe function setDimLabels(strList, dim, w)
	string strList
	variable dim
	wave w
	
	variable numLabels=ItemsInList(strList)
	variable i
		
	if(numLabels!=DimSize(w, dim))
		return 0
	endif
	for(i=0;i<numLabels;i+=1)
		SetDimLabel dim, i, $StringFromList(i, strList), w
	endfor
	return 1
end

// Wrapper function for update()
// This function is called from the file to be checked
// This is the old way of checking for updates.
function UpdateCheck(keyListOrFilePath)
	string keyListOrFilePath
		
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	variable interactive=prefs.options&2
	
	variable updated=0
	string url="", projectID=""
	// for reasons that have been lost to the mists of time, the filepath 
	// used to be passed to this function as part of a keylist. For 
	// backward compatibility, this is still acceptable.
	string filePath=StringByKey("fileloc", keyListOrFilePath) // path to procedure file to be updated	
	if(strlen(filePath)==0)
		filePath=keyListOrFilePath
	endif
	if(strlen(filePath)==0)
		return 0
	endif
		
	// first find out whether *this* file needs updating
	[projectID,url]=getReplacementFileURL(FunctionPath(""))
	if(strlen(url)&&interactive)
		DoAlert 1, "Download new version of updater?"
		if (v_flag==2) // print file location for manual download
			print "New version of updater available at " + url
		else
			updated+=update(FunctionPath(""), url, projectID)
		endif
	else
		updated+=update(FunctionPath(""), url, projectID)
	endif
	
	// now check the calling procedure file
	[projectID,url]=getReplacementFileURL(filePath)
	if(strlen(url)&&interactive)
		string shortTitle=getShortTitle(filePath)
		DoAlert 1, "Download new version of "+shortTitle+"?"
		if (v_flag==2) // print file location for manual download
			printf "New version of %s available at %s\r", shortTitle, url
		else
			updated+=update(filePath, url, projectID)
		endif
	else
		updated+=update(filePath, url, projectID)
	endif
 
	return updated
end

// checks for new release and provides a URL when a compatible release is
// available. This function is designed for periodic checking and is 
// called via UpdateCheck from file to be checked
function [string projectID, string url] getReplacementFileURL(string filePath)
	
	variable checkFrequency, isZip, isWin, lastUpdate
	variable releaseMajor, releaseMinor, releaseVersion, releaseIgorVersion
	variable localVersion=0
	variable currentIgorVersion=getIgorVersion()
	variable flagI, i
	
	string shortTitle="", cmd=""
	string prefsFile="", projectName="", system="", fileType="", releaseDate="", releaseExtra=""
	string packagePathStr="", archivePathStr=""
	string destPathStr=""
	string archiveName="", ipfName="", fileName=""
	string overwriteList="", fileList="", folderList=""
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	variable interactive=prefs.options&2

	if (strlen(filePath)==0)
		return ["", ""]
	endif
	getFileFolderInfo /Q/Z filePath
	if (V_isFile==0)
		return ["", ""] // shouldn't arrive here unless we've included a copy of updater in another namespace
	endif
	
	ipfName=ParseFilePath(0, filePath, ":", 1, 0) // name of procedure file to be updated
	packagePathStr=ParseFilePath(1, filePath, ":", 1, 0) // path to folder containing procedure file	
	system=selectString(stringmatch(StringByKey("OS", igorinfo(3))[0,2],"Mac"), "Windows", "Macintosh")
	isWin=stringmatch(system[0,2],"Win")
		
	url=getUpdateURLfromFile(filePath) // get all releases url defined by ksLocation or kProjectID
	
	// find project short title
	shortTitle=getShortTitle(filePath)
	if(strlen(shortTitle)==0)
		if (strlen(url)) // we have a download location defined, but not ksShortTitle
			// use filename in place of short title
			// may be a poor choice if procedure names are not unique.
			shortTitle=ParseFilePath(3, filePath, ":", 0, 0)
		else
			printf "Updater: could not find Project ID in %s\r" filePath
			return ["", ""]
		endif
	endif
	
	projectID=getProjectIDString(filePath) // defined by ksLocation or kProjectID
	if(strlen(projectID)==0)
		// in future versions will quit here
		// for backward compatibility, switch to short title for now
		projectID=shortTitle
	endif
		
	// find out how frequently to run check	
	checkFrequency = prefs.frequency
	lastUpdate=getLastUpdate(projectID)
	if ((datetime-lastUpdate)<checkFrequency)
		return ["", ""] // silently give up
	endif
	
	// find location of all releases page
	if(strlen(url)==0) // try to guess url based on short title
		url=guessURLfromShortTitle(shortTitle) // downloads project page and extracts all releases url
	endif
	if(strlen(url)==0)
		return ["", ""]
	endif
	
	// reset Project ID to what it should be
	projectID=ParseFilePath(0, url, "/", 1, 0)
	
	// version number of local file	
	localVersion=getProcVersion(filePath)
	if (localVersion==0)
		printf "Could not find version pragma in %s\r", ipfName
		return ["", ""]
	endif
	
	// now we have enough info to check for an update	
	if(interactive)
		sprintf cmd, "Check for updates to %s?", shortTitle
		DoAlert 1, cmd
		if (v_flag==2)
			return ["", ""]
		endif
	endif
	printf "%s: ", shortTitle
	
	// save new update time in prefs file
	setLastUpdate(projectID, datetime) // will be rounded to 32 bit integer
	
	// download the "all releases" page	
	URLRequest /TIME=(prefs.pageTimeout)/Z url=url
	if(V_flag)
		// sometimes first use of URLRequest fails.
		// Updater is checked first, so if we failed on Updater have another go
		if(stringmatch(shortTitle, "Updater"))
			URLRequest /TIME=(prefs.pageTimeout)/Z url=url
		endif
		if(V_flag)
			updateFailMsg("Could not load "+url); return ["", ""]
		endif
	endif
	
	// Look for new releases. If the web page location is not IgorExchange, 
	// ParseReleases will have to be changed to take into account the format 
	// of the web page.	
	wave /T w_releases=ParseReleases(S_serverResponse)
	// w_releases contains version info and file locations for all releases
	if (dimsize(w_releases,0)==0)
		sprintf cmd, "Could not find packages at %s.\r", url
		cmd+="\tPlease check for new versions at https://www.wavemetrics.com/projects"	
		updateFailMsg(cmd); return ["", ""]
	endif
		
	url=""
	for(i=0;i<dimsize(w_releases, 0);i+=1) // step through releases, starting with most recent
		
		projectName = w_releases[i][0]
		
		sscanf (w_releases[i][1])[5,inf], "%f.x-%f", releaseIgorVersion, releaseVersion
		if(V_flag!=2) // version string doesn't have strict formatting from old IgorExchange site
			releaseIgorVersion=0 // no way to figure out required Igor version prior to download
		endif
		
		releaseMajor=str2num(w_releases[i][2])
		releaseMinor=str2num(w_releases[i][3])
		releaseVersion=releaseMajor+releaseMinor/100
		releaseExtra=w_releases[i][7]
		releaseDate=w_releases[i][6]
		
		if(i==0 && localVersion>=releaseVersion)
			sprintf cmd, "%s %0.2f is the most recent version\r", shortTitle, localVersion
			updateFailMsg(cmd);return ["", ""]
		endif
		
		if(releaseVersion>localVersion)
			if(currentIgorVersion<releaseIgorVersion)
				sprintf cmd, "%s %0.2f%s requires Igor Pro version >= %g\r", shortTitle, releaseVersion, releaseExtra, releaseIgorVersion
				updateFailMsg(cmd); return ["", ""]
			elseif(FindListItem(system, w_releases[i][5])==-1)
				sprintf cmd, "%s %0.2f%s not available for %s\r", shortTitle, releaseVersion, releaseExtra, system
				updateFailMsg(cmd); return ["", ""]
			else
				url=w_releases[i][4]
				break
			endif
		endif	
	endfor
	
	if (strlen(url)==0)
		sprintf cmd, "No new version available\r"
		updateFailMsg(cmd); return ["", ""]
	endif
	
	// procedure-file requested check for updates needs to find an ipf
	if(grepString(url,"((?i)(.ipf|.zip)$)")==0)
		sprintf cmd, "New version of %s released\r", projectName
		sprintf cmd, "%sUpdater couldn't find a useable file at %s\r", cmd, url
		updateFailMsg(cmd); return ["", ""]
	endif
	
	printf "New release: version %0.2f of %s released %s\r", releaseVersion, shortTitle, releaseDate	
	return [projectID, url]	
end

// Download file from supplied url and replace original
// InstallPath is either path to file, or path to folder (installPath from log)
// If we're updating something from the install log, must supply shortTitle and localversion
// Returns full path to replaced file
function /S updateFile(InstallPath, url, projectID, [shortTitle, localVersion, newVersion])
	string InstallPath, url, projectID
	string shortTitle
	variable localversion, newVersion
				
	string cmd="", filePath="", fileExt=""
	string fileName
	variable flagI
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	variable interactive=prefs.options&2
	
	string backupPathStr=selectstring(prefs.options&1, "", SpecialDirPath(ksBackupLocation,0,0,0))
	// this location must exist, a subfolder may be created
	
	string downloadFileName=ParseFilePath(0, url, "/", 1, 0)
	string downloadFileExt=ParseFilePath(4, url, "/", 0, 0)
	
	if(stringmatch(InstallPath[strlen(InstallPath)-1], ":"))
		// get the filepath from install log
		filePath=LogGetFilePath(projectID)
	else
		filePath=InstallPath
	endif
	
	InstallPath=ParseFilePath(1, filePath, ":", 1, 0)
	fileName=ParseFilePath(0, filePath, ":", 1, 0) // name of file to be updated
	fileExt=ParseFilePath(4, filePath, ":", 0, 0)
	
	if(stringmatch(fileExt, downloadFileExt)==0)
		sprintf cmd, "Uploader could not match file type at\r%s\rand %s\r", url, filePath
		if (interactive) // in interactive mode this was already noted
			DoAlert 0, cmd
		endif
		print cmd
		return ""
	endif
	
	if(ParamIsDefault(shortTitle))
		shortTitle=getShortTitle(FilePath)
		if(strlen(shortTitle)==0) // use filename
			shortTitle=ParseFilePath(3, FilePath, ":", 0, 0)
		endif
	endif
	if(ParamIsDefault(localVersion))
		localVersion=getProcVersion(FilePath)
	endif
	
	// check with user before overwriting file			
	if (isFile(filePath))
		if (!interactive) // in interactive mode this was already noted
			sprintf cmd, "New update found for %s\r", shortTitle
		endif
		cmd+="Do you want to replace "+fileName+" with\r"
		cmd+="the file from\r"+url+"?"
		DoAlert 1, cmd
		if(v_flag==2)
			printf "Update for %s cancelled\r", shortTitle
			return ""
		endif
	endif	
	
	if(strlen(backupPathStr) && isFile(FilePath)) // save a backup of the old procedure file	
		// figure out full path to backup file
		sprintf backupPathStr, "%s%s%g.%s", backupPathStr, ParseFilePath(3, fileName, ":", 0, 0), localVersion, ParseFilePath(4, fileName, ":", 0, 0) 
		flagI=0
		getFileFolderInfo /Q/Z backupPathStr
		if(v_flag==0) // file already exists in archive location
			flagI=2 // check before overwriting
		endif
		movefile /O/S="Archive current file"/I=(flagI)/Z  FilePath as backupPathStr
		if(v_flag==0)
			printf "Saved copy of %s version %g to %s\r", shortTitle, localVersion, s_path
		endif
	endif
			
	// download new file and overwrite exisiting one
	URLRequest /TIME=(prefs.fileTimeout)/Z/O/FILE=FilePath url=url
	if(V_flag)
		updateFailMsg("Could not download "+url); return ""
	elseif(strlen(S_fileName)==0)
		updateFailMsg("Could not write file to "+FilePath); return ""
	endif		
	printf "Downloaded %s\r", url
	printf "Saved file to %s\r", S_fileName
	
	if(ParamIsDefault(newVersion))
		newVersion=getProcVersion(FilePath)
	endif
	
	// write to install log
	LogUpdateProject(projectID, shortTitle, InstallPath, num2str(newVersion), fileName)
	
	return S_fileName
end

// Update procedure and associated files with contents of zip file to be 
// downloaded from url. If we're updating something from the install log,
// must supply all the optional parameters. In that case filepath is a 
// path to install location, not to a file.
function /s updateZip(FilePath, url, projectID, [shortTitle, localVersion, newVersion])
	string FilePath, url, projectID, shortTitle
	variable localversion, newVersion
	
	string archivePathStr, archiveName, unzipPathStr, fileList, folderList, fileName
	string ipfName, packagePathStr, InstallPathStr
	string oldFiles="", staleFiles="", cmd =""
	string downloadPathStr=SpecialDirPath(ksDownloadLocation,0,0,0)+"TonyInstallerTemp:"
	string backupPathStr=SpecialDirPath(ksBackupLocation,0,0,0)	
	variable i
	
	if(ParamIsDefault(localVersion))
		localVersion=getProcVersion(FilePath)
	endif
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	variable interactive=prefs.options&2
	
	if(cmpstr(FilePath[strlen(filePath)-1], ":")) // we have path to a file
		ipfName=ParseFilePath(0, FilePath, ":", 1, 0) // name of procedure file to be updated
		packagePathStr=ParseFilePath(1, FilePath, ":", 1, 0) // path to folder containing procedure file
	else // path to folder
		ipfName=""
		packagePathStr=FilePath
		InstallPathStr=FilePath
	endif
	
	if(ParamIsDefault(shortTitle)) // not a log file based update
		shortTitle=getShortTitle(FilePath)
		if(strlen(shortTitle)==0) // use filename
			shortTitle=ParseFilePath(3, FilePath, ":", 0, 0)
		endif
	endif
	
	// create a temporary folder in the download location
	newpath /C/O/Q tempPathIXI, downloadPathStr; killpath /z tempPathIXI
			
	// download zip file to temporary location
	URLRequest /TIME=(prefs.fileTimeout)/Z/O/FILE=downloadPathStr+ParseFilePath(0, url,"/", 1, 0) url=url
	if(V_flag)
		updateFailMsg("Could not download "+url); return ""
	elseif(strlen(S_fileName)==0)
		updateFailMsg("Could not write file to "+FilePath); return ""
	endif
	
	archivePathStr=S_fileName // path to zip file
	archiveName=ParseFilePath(0, S_fileName, ":", 1, 0)
	printf "Downloaded new version of %s from %s\r", shortTitle, url
	printf "Saved temporary file %s\r", archiveName
	
	// create a temporary directory for the uncompressed files
	unzipPathStr=CreateUniqueDir(downloadPathStr, "Install")

	// inflate archive
	variable success = unzipArchive(archivePathStr, unzipPathStr)
	if (success==0)
		updateFailMsg("unzipArchive failed to inflate "+S_fileName); return ""
	endif		
	printf "Inflated %s to %s\r", archiveName, unzipPathStr
	
	// get list of files and folders
	newpath /O/Q unzipPath, unzipPathStr
	fileList=IndexedFile(unzipPath, -1, "????")
	folderList=IndexedDir(unzipPath, -1, 0)
	killpath /Z unzipPath
		
	// ignore pesky __MACOSX folder, and other folders listed in ignoreFoldersList
	folderList=RemoveFromListWC(folderList, ksIgnoreFoldersList)	
	fileList=RemoveFromListWC(fileList, ksIgnoreFilesList)
	
	// check number of files and folders at root of inflated archive
	variable numFiles=ItemsInList(fileList), numFolders=ItemsInList(folderList)

	// skip the following checks if we're using log file		
	if(strlen(ipfname))
		// use FindFile to look for the file that replaces the target procedure file
		string unzippedIPFpathStr=FindFile(unzipPathStr, ipfName)
		if(strlen(unzippedIPFpathStr)==0) // abort installation if we don't have the right file!
			sprintf cmd, "Could not find %s in %s\r", ipfName, archiveName
			updateFailMsg(cmd)
			InstallerCleanup(downloadPathStr)
			return ""
		endif
		
		// getting rid of this code block ensures that InstallPath is the 
		// correct install path for the installer log
							
//		if (numFiles==0 && numFolders==1)
//			// inflated files are inside a subfolder of unzipPathStr
//			unzipPathStr+=StringFromList(0, folderList)+":"
//			newpath /O/Q unzipPath, unzipPathStr
//			fileList=IndexedFile(unzipPath, -1, "????")
//			folderList=IndexedDir(unzipPath, -1, 0)
//			killpath /Z unzipPath
//			folderList=RemoveFromListWC(folderList, ksIgnoreFoldersList)	
//			fileList=RemoveFromListWC(fileList, ksIgnoreFilesList)
//			numFiles=ItemsInList(fileList)
//			numFolders=ItemsInList(folderList)
//		endif
			
		// figure out local path from unzipPathStr to replacement procedure file
		unzippedIPFpathStr=unzippedIPFpathStr[strlen(unzipPathStr),inf]
		variable subfolders=ItemsInList(unzippedIPFpathStr, ":")-1
			
		if(subfolders>0) // check that directory structure at destination matches that from archive
			if(stringmatch(FilePath, "*"+unzippedIPFpathStr)==0)
				// target directory doesn't match folder in archive
				sprintf cmd, "Could not match directory structure in %s to %s\r", archiveName, packagePathStr
				updateFailMsg(cmd)
				InstallerCleanup(downloadPathStr)
				return ""
			endif
		endif		
	
		// move up directory structure at destination
		InstallPathStr=ParseFilePath(1, FilePath, ":", 1, 0) // path to folder containing file
		for(i=0;i<subfolders;i+=1)
			InstallPathStr=ParseFilePath(1, InstallPathStr, ":", 1, 0)
		endfor
		
	endif
	
	// test to find out which files will be overwritten
	fileList=mergeFolder(unzipPathStr, InstallPathStr, test=1)
	numFiles=ItemsInList(fileList)
	
	oldFiles=LogGetFileList(projectID) // file list from log (if log exists)
	staleFiles=RemoveFromList(fileList, oldFiles, ";", 0) // files to be removed
	
	oldFiles=fileList // files to be overwritten
	if(strlen(staleFiles))
		oldFiles=AddListItem(staleFiles,oldFiles)
	endif
	variable numOldFiles=ItemsInList(oldFiles)
	
	// check with user before overwriting files
	if (numOldFiles)
		if (!interactive) // in interactive mode this was already noted
			sprintf cmd, "New update found for %s\r", shortTitle
		endif
		cmd+="The following files will be overwritten"
		if(strlen(staleFiles))
			cmd+=" or deleted"
		endif
		cmd+=":\r"
		string tooMany= "\r*** TOO MANY FILES TO LIST ***\r"
		string cont="\rContinue?"
		variable cmdLen = strlen(cmd)+strlen(tooMany)+strlen(cont)
		for (i=0;i<numFiles;i+=1)
			fileName=StringFromList(i, oldFiles)
			fileName=fileName[strlen(InstallPathStr),inf]
			if( cmdLen+strlen(fileName) >= 1023 )
				cmd+=tooMany
				break
			endif
			cmd+=fileName+"\r"
			cmdLen+=strlen(fileName+"\r")
		endfor
		cmd+=cont
		DoAlert 1, cmd
		if(v_flag==2)
			printf "Update for %s cancelled\r", shortTitle
			InstallerCleanup(downloadPathStr)
			return ""
		endif
	endif
	
	// create backup folder
	if(numOldFiles && strlen(backupPathStr))				
		string folderName
		sprintf folderName, "%s_%0.2f",shortTitle,localVersion
		folderName=cleanupname(folderName,0) // making it Igor-legal should hopefully ensure the name is good for the OS
		sprintf backupPathStr, "%s%s:", backupPathStr, folderName
		NewPath /Q/C/O/Z tempPathIXI, backupPathStr; killpath /Z tempPathIXI
		printf "Created backup folder %s\r", backupPathStr
	endif
		
	if(strlen(staleFiles)) 
		// remove any files from previous install that won't be overwritten
		string deletePath
		if(strlen(backupPathStr))
			deletePath=backupPathStr
		else
			deletePath=CreateUniqueDir(SpecialDirPath("Temporary",0,0,0), shortTitle)
		endif		
		moveFiles(staleFiles, InstallPathStr, deletePath)
	endif
		
	// move files into user procedures folder
	fileList=mergeFolder(unzipPathStr, InstallPathStr, backupPathStr=backupPathStr, killSourceFolder=0)
	
	for (i=0;i<ItemsInList(fileList);i+=1)
		fileName=StringFromList(i, fileList) // this is full path to file
		printf "Moved %s to %s\r", ParseFilePath(0, fileName, ":", 1, 0), ParseFilePath(1, fileName, ":", 1, 0)
	endfor	
	InstallerCleanup(downloadPathStr)	
		
	if(ParamIsDefault(newVersion)) // not a log-based update
		newVersion=getProcVersion(FilePath)
	endif
	
	printf "Updated %s to version %g\r", shortTitle, newVersion
		
	// write to install log
	LogUpdateProject(projectID, shortTitle, InstallPathStr, num2str(newVersion), fileList)
	
	return fileList
end

function moveFiles(fileList, fromPath, toPath)
	string fileList, fromPath, toPath
	
	string fileName, filePath, subPath, destinationPath
	variable i, j
		
	for(i=0;i<ItemsInList(fileList);i+=1)	
		FilePath=StringFromList(i, fileList)
		if(stringmatch(FilePath, fromPath+"*")==0)
			continue
		endif
		FileName=ParseFilePath(0, filePath, ":", 1, 0)
		subPath=ParseFilePath(1, filePath, ":", 1, 0)
		subPath=subPath[strlen(fromPath), inf]
		destinationPath=toPath
		for(j=0;j<ItemsInList(subPath, ":");j+=1)
			destinationPath+=StringFromList(j, subPath, ":") + ":"
			newPath /C/O/Q/Z tempPathIXI, destinationPath
		endfor
		MoveFile /Z/O FilePath as destinationPath+FileName
		if (v_flag==0)
			printf "Moved %s to %s\r", fileName, destinationPath
		endif
	endfor
	killPath /Z tempPathIXI
	return 1
end

// for commandline use
function CheckAndUpdate(FilePath)
	string FilePath
	
	string url, projectID
	[projectID,url]=getReplacementFileURL(FilePath)	
	return update(FilePath, url, projectID)
end

function update(FilePath, url, projectID)
	string FilePath, url, projectID
	
	string fileType=ParseFilePath(4, url,":", 0, 0)
	string fileList=""
	if(stringmatch(fileType, "zip"))
		fileList=updateZip(FilePath, url, projectID)
	elseif(stringmatch(fileType, "ipf"))
		fileList=updateFile(FilePath, url, projectID)
	elseif(stringmatch(fileType, "xop"))
		// can only update xop if we have info from install log
	endif
	return (strlen(fileList)>0)
end

// this is for updates requested by IgorExchange Installer Panel
// for backward compatibility, filepath can be path to a file or to install location
// this will be the preferred method 
function UpdateSelection()

	string projectID, InstallPath, url, shortTitle
	variable localVersion, newVersion
		
	DFREF dfr = root:Packages:Installer
	wave /SDFR=dfr/T UpdatesMatchList
	ControlInfo /W=InstallerPanel listboxUpdate
	
	if(v_value>-1 && v_value<dimsize(UpdatesMatchList,0))
		setPanelStatus("Updating "+ UpdatesMatchList[v_value][%name])
		
		projectID=UpdatesMatchList[v_value][%projectID]
		localVersion=str2num(UpdatesMatchList[v_value][%local])
		newVersion=str2num(UpdatesMatchList[v_value][%remote])
		url=UpdatesMatchList[v_value][%releaseURL]
		shortTitle=UpdatesMatchList[v_value][%name]	
		InstallPath=UpdatesMatchList[v_value][%installPath] // full path
		
		string fileType=ParseFilePath(4, url,":", 0, 0)
		string fileList=""
		if(stringMatch(fileType, "zip"))
			fileList=updateZip(InstallPath, url, projectID, shortTitle=shortTitle, localVersion=localVersion, newVersion=newVersion)
		else
			fileList=updateFile(InstallPath, url, projectID, shortTitle=shortTitle, localVersion=localVersion, newVersion=newVersion)		
		endif
		if(strlen(fileList))		
			LogUpdateProject(projectID, shortTitle, InstallPath, num2str(newVersion), fileList)
			reloadUpdatesList(0, 1)
			UpdateListboxWave(fGetStub())
			setPanelStatus(shortTitle+" Update Complete")
		else
			setPanelStatus("Selected: "+UpdatesMatchList[v_value][%name])
		endif
	endif
	
	return (strlen(fileList)>0)
end

// delete temporary installation folder if required
function InstallerCleanup(downloadPathStr)
	string downloadPathStr
	
	if(stringmatch(downloadPathStr, SpecialDirPath("Temporary",0,0,0)+"*"))
		return 0
	endif
	deleteFolder /M="OK to delete temporary installation files?"/Z removeEnding(downloadPathStr,":")
	if(v_flag==0)
		printf "Deleted temporary folder %s\r", downloadPathStr
	endif
end

function /S getFilePathFromProcWin(strWin)
	string strWin
	
	GetWindow /Z $strWin file
	return StringFromList(1, S_value)+StringFromList(0, S_value)
end

function getIgorVersion() // windows-safe
	
	string strIgorVersion=StringByKey("IGORFILEVERSION", IgorInfo(3))
	strIgorVersion=ReplaceString(".", strIgorVersion, "*", 0, 1)
	strIgorVersion=ReplaceString(".", strIgorVersion, "")
	strIgorVersion=ReplaceString("*", strIgorVersion, ".")
	return str2num(strIgorVersion)
end

// wrapper for GetFileFolderInfo
function isFile(FilePath)
	string FilePath
	
	GetFileFolderInfo /Q/Z FilePath
	return V_isFile
end

function getProcVersion(FilePath)
	string FilePath
	
	variable procVersion
	Grep /Q/E="(?i)^#pragma[\s]*version[\s]*=" /LIST/Z FilePath
	s_value=LowerStr(TrimString(s_value, 1))
	sscanf s_value, "#pragma version = %f", procVersion
	if (V_flag!=1 || procVersion<=0)
		return 0
	endif
	return procVersion
end

// extract project ID from file and return as string
function /S getProjectIDString(FilePath)
	string FilePath

	variable projectID=getUpdaterConstantFromFile("kProjectID", FilePath)
	if(numtype(projectID)==0)
		return num2istr(projectID)
	endif
	// for backward compatibility, try getting projectID from all releases URL
	string url=getStringConst("ksLocation", FilePath)
	return ParseFilePath(0, url, "/", 1, 0) // returns "" on failure
end
	
function /S getPragmaString(strPragma, FilePath)
	string strPragma, FilePath
	
	if(isFile(FilePath)==0)
		return ""
	endif
	
	string exp="(?i)^#pragma[\s]*"+strPragma+"[\s]*="
	Grep /Q/E=exp /LIST/Z FilePath
	string str=RemoveEnding(s_value, ";")
	variable vpos=strsearch(str, "=", 0)
	if(vpos==-1)
		return ""
	endif
	str=str[vpos+1,Inf]
	vpos=strsearch(str, "//", 0)
	if(vpos>2)
		str=str[0,vpos-1]
	endif
	return TrimString(str, 1)
end

function /S getShortTitle(FilePath)
	string FilePath
	
	variable selStart, selEnd

	string shortTitle=getStringConst("ksShortTitle", FilePath)
	if (strlen(shortTitle)==0)
		shortTitle=getStringConst("ksShortName", FilePath) // for backward compatibility
	endif
	
	if (strlen(shortTitle)==0)
		Grep /Q/E="(?i)#pragma[\s]*moduleName[\s]*=" /LIST/Z FilePath
		if (strlen(s_value)==0)
			Grep /Q/E="(?i)#pragma[\s]*IndependentModule[\s]*=" /LIST/Z FilePath
			if (strlen(s_value)==0)
				return ""
			endif
		endif
		s_value=RemoveEnding(s_value,";")
		selStart=strsearch(s_value, "=", 0)+1
		selEnd=strsearch(s_value, "//", 0)
		selEnd = selEnd==-1 ? strlen(s_value)-1 : selEnd-1
		shortTitle=TrimString(s_value[selStart,selEnd])
	endif
	return shortTitle
end

// an alternative to getPragmaString("ksLocation"....
// used only when neither ksLocation nor kProjectID are defined
function /s guessURLfromShortTitle(shortTitle, [noHist])
	string shortTitle
	variable noHist
	
	noHist = ParamIsDefault(NoHist) ? 0 : 1
	
	if(strlen(shortTitle)==0)
		return ""
	endif
	
	variable selStart, selEnd
	string url
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
		
	// check the project page on IgorExchange
	sprintf url, "https://www.wavemetrics.com/project/%s", shortTitle
	URLRequest /time=(prefs.pageTimeout)/Z url=url
	if(V_flag && NoHist==0)
		printf "Installer could not load %s\r", url
		return ""
	endif
	// web page text should be in S_serverResponse

	// find the link to the 'all releases' page for this project
	selEnd=strsearch(S_serverResponse, ">View all releases</a>", 0, 2)
	selStart=strsearch(S_serverResponse, "href=", selEnd, 3)
	if ((selEnd==-1 || selStart==-1) && NoHist==0)
		printf "Installer could not find all releases page for %s\r", shortTitle
		return ""
	endif
	url="https://www.wavemetrics.com"+S_serverResponse[selStart+6,selEnd-2]
	return url
end

// extract a static string constant from a procedure file by reading from disk
function /T getStringConst(constantNameStr, FilePath)
	string constantNameStr, FilePath
	
	string s_exp="", s_out=""
	sprintf s_exp, "(?i)^[\s]*static[\s]*strconstant[\s]*%s[\s]*=", constantNameStr
	Grep /Q/E=s_exp/LIST/Z FilePath
	variable start, stop
	start=strsearch(s_value, "\"", 0)
	stop=strsearch(s_value, "\"", start+1)
	if (start<0 || stop<0)
		return ""
	endif
	return s_value[start+1,stop-1]
end

// extract a static constant from a procedure file by reading from disk
function getUpdaterConstantFromFile(strName, FilePath)
	string strName, FilePath
	
	if(isFile(FilePath)==0)
		return NaN
	endif
	
	string s_exp="", s_out=""
	sprintf s_exp, "(?i)^[\s]*static[\s]*constant[\s]*%s[\s]*=", strName
	Grep /Q/E=s_exp/LIST/Z FilePath
	variable start, stop
	start=strsearch(s_value, "=", 0)
	if (start<0)
		return NaN
	endif
	return str2num(s_value[start+1,Inf])
end

// strip trailing stuff in the output from winlist
function /S FileNameFromProcName(str)
	string str
	
	SplitString/E=".*\.ipf" str // remove module names
	return removeending(s_value, ".ipf")
end

function updateFailMsg(strMsg)
	string strMsg
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	variable interactive=prefs.options&2
	if(strlen(strMsg)>0)
		print strMsg
		if (interactive)
			DoAlert 0, strMsg
		endif
	endif
end	

function /S CreateUniqueDir(pathStr, baseName)
	string pathStr, baseName
	
	pathStr=ParseFilePath(2, pathStr, ":", 0, 0)
	GetFileFolderInfo /Q/Z pathStr
	if(v_isFolder==0)
		return ""
	endif
	
	variable i=0
	string strOut=""
	do
		sprintf strOut,"%s%s%d:", pathStr, baseName, i
		GetFileFolderInfo /Q/Z strOut
		i+=1
	while(V_Flag==0)			
	newpath /C/O/Q tempPathIXI, strOut; killPath /Z tempPathIXI
	if (v_flag==0)
		return strOut
	endif
	return ""
end

// returns listStr, purged of any items that match an item in ZapListStr.
// Wildcards okay! Case insensitive.
function /S RemoveFromListWC(listStr, zapListStr)
	string listStr, zapListStr
	
	string removeStr=""
	variable i
	for(i=0;i<ItemsInList(zapListStr);i+=1)
		removeStr+=ListMatch(listStr, StringFromList(i, zapListStr))
	endfor
	return RemoveFromList(removeStr, listStr, ";", 0)
end

// Parse html in WebPageText to find releases. 
threadsafe function /WAVE ParseReleases(WebPageText)
	string WebPageText
	
	make /T/Free/N=(0,8) w_releases
		
	variable selStart=-1, selEnd=-1, blockStart, blockEnd, selFound
	string projectName="", requiredVersion="", releaseMajor="", releaseMinor="", releaseExtra=""
	string url="", platform="", versionDate=""
	variable i=0
	
	// locate project name
	selStart=strsearch(WebPageText, "<h1 class=\"page-title\" title=\"Releases for ", blockStart, 2)
	selStart+=43
	selEnd=strsearch(WebPageText, "\"", selStart, 2)
	if(selStart<43 || selEnd>selStart+150)
		return w_releases
	endif
	projectName=WebPageText[selStart, selEnd-1]

	do
		// find start and end of project release fields
		selStart= strsearch(WebPageText, "<div class=\"project-release-info\">", selStart, 2)
		if(selStart==-1)
			break
		endif
		blockStart=selStart
		selEnd=strsearch(WebPageText, "<div class=\"project-release-footer\">", selStart, 2)
		if(selEnd==-1)
			break
		endif
		blockEnd=selEnd
		
		// locate version info
		selStart=strsearch(WebPageText, "\"field-paragraph-version\">", blockStart, 2)
		selStart+=26
		selEnd=strsearch(WebPageText, "<", selStart, 2)
		if(selStart<blockStart || selEnd>blockEnd)
			break
		endif
		requiredVersion=WebPageText[selStart, selEnd-1]	
		
		// locate version date
		selStart=strsearch(WebPageText, "\"field-paragraph-version-date\">", blockStart, 2)
		selStart+=31
		selEnd=strsearch(WebPageText, "<", selStart, 2)
		if(selStart<blockStart || selEnd>blockEnd)
			break
		endif
		versionDate=WebPageText[selStart, selEnd-1]	
		
		selStart=strsearch(WebPageText, "\"field-paragraph-version-major\">", blockStart, 2)
		selStart+=32
		selEnd=strsearch(WebPageText, "<", selStart, 2)
		if(selStart<blockStart || selEnd>blockEnd)
			break
		endif
		releaseMajor=WebPageText[selStart, selEnd-1]
		
		selStart=strsearch(WebPageText, "\"field-paragraph-version-patch\">", blockStart, 2)
		selStart+=32
		selEnd=strsearch(WebPageText, "<", selStart, 2)
		if(selStart<blockStart || selEnd>blockEnd)
			releaseMinor="0" // patch not found - field may be missing
		else
			releaseMinor=WebPageText[selStart, selEnd-1]
		endif
		
		selStart=strsearch(WebPageText, "\"field-paragraph-version-extra\">", blockStart, 2)
		selStart+=32
		selEnd=strsearch(WebPageText, "<", selStart, 2)
		if(selStart<blockStart || selEnd>blockEnd)
			releaseExtra="" // field not present for this release
		else
			releaseExtra=WebPageText[selStart, selEnd-1]
		endif
		
		// find download link
		selStart=strsearch(WebPageText, "\"field-paragraph-file\"", blockStart, 2)
		selStart=strsearch(WebPageText, "<a href=\"", selStart, 2)
		selStart+=9
		selEnd=strsearch(WebPageText, "\"", selStart, 2)
		if(selEnd<0 || selEnd>blockEnd)
			break
		endif
		url=WebPageText[selStart, selEnd-1]
				
		platform=""
		selFound=strsearch(WebPageText, "<span class=\"entity-reference\">Windows</span>", blockStart, 2)
		if(selFound>blockStart && selFound<blockEnd)
			platform+="Windows;"
		endif
		selFound=strsearch(WebPageText, "<span class=\"entity-reference\">Mac-", blockStart, 2)
		if(selFound>blockStart && selFound<blockEnd)
			platform+="Macintosh;"
		endif
		if(strlen(projectName)==0 || strlen(releaseMajor)==0 || strlen(releaseMinor)==0 ||strlen(url)==0)
			break
		endif
		w_releases[DimSize(w_releases,0)][]={{projectName},{requiredVersion},{releaseMajor},{releaseMinor},{url},{platform},{versionDate},{releaseExtra}}
		selStart=blockEnd
		i+=1
	while (i<1000)
	return w_releases
end
 
// utility function, inflates a zip archive
function unzipArchive(archivePathStr, unzipPathStr)
	string archivePathStr, unzipPathStr
		
	string validExtensions="zip;" // set to "" to skip check
	variable verbose=1 // choose whether to print output from executescripttext
	string msg, unixCmd, cmd
		
	GetFileFolderInfo /Q/Z archivePathStr

	if(V_Flag || V_isFile==0)
		printf "Could not find file %s\r", archivePathStr
		return 0
	endif

	if(ItemsInList(validExtensions) && FindListItem(ParseFilePath(4, archivePathStr, ":", 0, 0), validExtensions, ";", 0, 0)==-1)
		printf "%s doesn't appear to be a zip archive\r", ParseFilePath(0, archivePathStr, ":", 1, 0)
		return 0
	endif
	
	if(strlen(unzipPathStr)==0)
		unzipPathStr=SpecialDirPath("Desktop",0,0,0)+ParseFilePath(3, archivePathStr, ":", 0, 0)
		sprintf msg, "Unzip to %s:%s?", ParseFilePath(0, unzipPathStr, ":", 1, 1), ParseFilePath(0, unzipPathStr, ":", 1, 0)
		DoAlert 1, msg
		if (v_flag==2)
			return 0
		endif
	else
		GetFileFolderInfo /Q/Z unzipPathStr
		if(V_Flag || V_isFolder==0)
			sprintf msg, "Could not find unzipPathStr folder\rCreate %s?", unzipPathStr
			DoAlert 1, msg
			if (v_flag==2)
				return 0
			endif
		endif
	endif
	
	// make sure unzipPathStr folder exists - necessary for mac
	NewPath /C/O/Q acw_tmpPath, unzipPathStr
	KillPath /Z acw_tmpPath

	if(stringmatch(StringByKey("OS", IgorInfo(3))[0,2],"Win")) // Windows
		// The following works with .Net 4.5, which is available in Windows 8 and up.
		// current versions of Windows with Powershell 5 can use the more succinct PS command
		// 'Expand-Archive -LiteralPath C:\archive.zip -DestinationPath C:\Dest'
		string strVersion=StringByKey("OSVERSION", IgorInfo(3))
		variable WinVersion=str2num(strVersion) // turns "10.1.2.3" into 10.1 and 6.23.111 into 6.2 (windows 8.0)
		if (WinVersion<6.2)
			Print "unzipArchive requires Windows 8 or later"
			return 0
		endif
		
		archivePathStr=ParseFilePath(5, archivePathStr, "\\", 0, 0)
		unzipPathStr=ParseFilePath(5, unzipPathStr, "\\", 0, 0)
		cmd="powershell.exe -nologo -noprofile -command \"& { Add-Type -A 'System.IO.Compression.FileSystem';"
		sprintf cmd "%s [IO.Compression.ZipFile]::ExtractToDirectory('%s', '%s'); }\"", cmd, archivePathStr, unzipPathStr
	else // Mac
		sprintf unixCmd, "unzip %s -d %s", ParseFilePath(5, archivePathStr, "/", 0,0), ParseFilePath(5, unzipPathStr, "/", 0,0)
		sprintf cmd, "do shell script \"%s\"", unixCmd
	endif
	
	ExecuteScriptText /B/UNQ/Z cmd
	if(verbose)
		Print S_value // output from executescripttext
	endif
	
	return (v_flag==0)
end

// returns full path to alias if found, otherwise ""
function /T FindAlias(folderPathStr, targetPathStr)
	String folderPathStr, targetPathStr
	
	variable maxLevels=5 // quit after recursion through this many sublevels
	variable folderLimit=20 // quit after looking in this many folders.
	Variable folderIndex, fileIndex, subfolderIndex, folderCount=1, sublevels=0
	string folderList

	if (strlen(folderPathStr)==0 || strlen(targetPathStr)==0)
		return ""
	endif
		
	make /free/T/n=0 w_folders, w_subfolders
	w_folders={folderPathStr}

	do
		for(folderIndex=0;folderIndex<numpnts(w_folders);folderIndex+=1)
			// check files at current level
			newPath /O/Q/Z tempPathIXI, w_folders[folderIndex]
			fileindex=0
			do
				getFileFolderInfo /P=tempPathIXI/Q/Z IndexedFile(tempPathIXI, fileindex, "????")
				if(V_isFile==0)
					break
				endif
				if(V_isAliasShortcut && stringmatch(S_aliasPath, targetPathStr))
					killpath /Z tempPathIXI
					return s_path // set by getFileFolderInfo
				endif
				fileindex+=1
			while(1)
			// make a list of subfolders in current folder
			folderList=IndexedDir(tempPathIXI, -1, 0)
			for (subfolderIndex=0;subfolderIndex<ItemsInList(folderList);subfolderIndex+=1)
				w_subfolders[numpnts(w_subfolders)]={w_folders[folderIndex]+StringFromList(subfolderIndex,folderList)+":"}				
				folderCount+=1
			endfor			
		endfor
		duplicate /T/O/free w_subfolders, w_folders
		redimension /N=0 w_subfolders
		sublevels+=1
		
		if(numpnts(w_folders)==0)
			break
		endif
		
		if(sublevels>maxLevels || (folderCount-folderIndex)>folderLimit)
			break
		endif

	while(1)
	killpath /Z tempPathIXI
	return ""
end

// Search recursively for fileName within folder described by folderPathStr.
// Don't expect this to be quick.
// Returns path to file. Wildcards okay.
function /T FindFile(folderPathStr, fileName)
	String folderPathStr, fileName
	
	variable maxLevels=10 // quit after recursion through this many sublevels
	variable folderLimit=5000 // quit after looking in this many folders.
	Variable folderIndex, fileIndex, subfolderIndex, folderCount=1, sublevels=0
	string folderList, indexFileName

	if (strlen(folderPathStr)==0)
		// it would be nice to default to the current save location, but I couldn't find a way to do that
		folderPathStr=SpecialDirPath("Documents", 0, 0, 0)
	endif
		
	make /free/T/n=0 w_folders, w_subfolders
	w_folders={folderPathStr}

	do
		for(folderIndex=0;folderIndex<numpnts(w_folders);folderIndex+=1)
			// check files at current level
			newPath /O/Q/Z tempPathIXI, w_folders[folderIndex]

			fileindex=0
			do	
				indexFileName=IndexedFile(tempPathIXI, fileindex, "????")
				if(stringmatch(indexFileName, fileName))
					killpath /Z tempPathIXI
					return w_folders[folderIndex]+indexFileName
				endif
				fileindex+=1
			while(strlen(indexFileName))
			// make a list of subfolders in current folder
			folderList=IndexedDir(tempPathIXI, -1, 0)
			for (subfolderIndex=0;subfolderIndex<ItemsInList(folderList);subfolderIndex+=1)
				w_subfolders[numpnts(w_subfolders)]={w_folders[folderIndex]+StringFromList(subfolderIndex,folderList)+":"}				
				folderCount+=1
			endfor			
		endfor
		duplicate /T/O/free w_subfolders, w_folders
		redimension /N=0 w_subfolders
		sublevels+=1
		
		if(numpnts(w_folders)==0)
			break
		endif
		
		if(sublevels>maxLevels || (folderCount-folderIndex)>folderLimit)
			break
		endif

	while(1)
	killpath /Z tempPathIXI
	return ""
end

// search recursively in folderPathStr for files matching fileName.
// returns list of complete paths to files
function /T RecursiveFileList(folderPathStr, fileName)
	String folderPathStr, fileName
	
	variable maxLevels=5 // quit after recursion through this many sublevels
	variable folderLimit=500 // quit after looking in this many folders
	Variable folderIndex, fileIndex, subfolderIndex, folderCount=1, sublevels=0
	string folderList, indexFileName
	string fileList=""

	if (strlen(folderPathStr)==0)
		return ""
	endif
		
	make /free/T/n=0 w_folders, w_subfolders
	w_folders={folderPathStr}

	do
		for(folderIndex=0;folderIndex<numpnts(w_folders);folderIndex+=1)
			// check files at current level
			newPath /O/Q/Z tempPathIXI, w_folders[folderIndex]

			fileindex=0
			do	
				indexFileName=IndexedFile(tempPathIXI, fileindex, "????")
				if(stringmatch(indexFileName, fileName))
					fileList=AddListItem(w_folders[folderIndex]+indexFileName, fileList)
				endif
				fileindex+=1
			while(strlen(indexFileName))
			// make a list of subfolders in current folder
			folderList=IndexedDir(tempPathIXI, -1, 0)
			for (subfolderIndex=0;subfolderIndex<ItemsInList(folderList);subfolderIndex+=1)
				w_subfolders[numpnts(w_subfolders)]={w_folders[folderIndex]+StringFromList(subfolderIndex,folderList)+":"}				
				folderCount+=1
			endfor			
		endfor
		duplicate /T/O/free w_subfolders, w_folders
		redimension /N=0 w_subfolders
		sublevels+=1
		
		if(numpnts(w_folders)==0)
			break
		endif
		
		if(((maxlevels && sublevels>maxLevels) || (folderLimit && (folderCount-folderIndex)>folderLimit)))
			break
		endif

	while(1)
	killpath /Z tempPathIXI
	return fileList
end

// Merges folders like copyFolder on Windows, deletes source files
// Files in destination folder are overwritten by files from source
// Setting test=1 doesn't move anything but generates a list of files
// that would be overwritten
function /S mergeFolder(source, destination, [killSourceFolder, backupPathStr, test])
	string source, destination
	variable killSourceFolder, test
	string backupPathStr // must be no more than one sublevel below an existing folder
	
	killSourceFolder = ParamIsDefault(killSourceFolder) ? 1 : killSourceFolder
	test = ParamIsDefault(test) ? 0 : test
	if(ParamIsDefault(backupPathStr))
		backupPathStr=""
	endif
	variable backup=(strlen(backupPathStr)>0)
	
	// clean up paths
	source=ParseFilePath(2, source, ":", 0, 0)
	destination=ParseFilePath(2, destination, ":", 0, 0)
	if (backup)
		backupPathStr=ParseFilePath(2, backupPathStr, ":", 0, 0)
	endif
	
	// check that souce and destination folders exist
	GetFileFolderInfo /Q/Z source
	variable sourceOK=V_isFolder
	GetFileFolderInfo /Q/Z destination
	if(sourceOK==0 || V_isFolder==0)
		return ""
	endif
	
	Variable folderIndex, fileIndex, subfolderIndex, folderCount=1, sublevels=0
	string folderList, fileList, fileName
	string movedfileList="", destFolderStr=""
	string subPathStr=""
	
	Make /free/T/n=0 w_folders, w_subfolders
	w_folders={source}
	do
		// step through folders at current sublevel
		for(folderIndex=0;folderIndex<numpnts(w_folders);folderIndex+=1)
			// figure out destination folder to match current source folder
			subPathStr=(w_folders[folderIndex])[strlen(source),strlen(w_folders[folderIndex])-1]
			destFolderStr=destination+subPathStr
			
			// make sure that folder exists at destination
			if(test==0)
				NewPath /C/O/Q/Z tempPathIXI, destination+subPathStr
				if (backup)
					NewPath /C/O/Q/Z tempPathIXI, backupPathStr+subPathStr
				endif
			endif
					
			// get list of source files in indexth folder at current sublevel
			NewPath /O/Q/Z tempPathIXI, w_folders[folderIndex]
			fileList=IndexedFile(tempPathIXI, -1, "????")
			// remove files from list if they match an entry in ignorefileList
			fileList=RemoveFromListWC(fileList, ksIgnoreFilesList)
			// move files
			for(fileindex=0;fileIndex<ItemsInList(fileList);fileIndex+=1)
				FileName=StringFromList(fileindex, fileList)
				if(test)
					GetFileFolderInfo /Q/Z destFolderStr+FileName
					if(v_flag==0 && v_isFile) // file is to be overwritten
						movedfileList=AddListItem(destFolderStr+FileName, movedfileList)
					endif
				else
					if (backup) // back up any files that are to be overwritten
						GetFileFolderInfo /Q/Z destFolderStr+FileName
						if(v_flag==0 && v_isFile) // file is to be overwritten
							//newPath /C/O/Q/Z tempPathIXI, backupPathStr+subPathStr; killpath /Z tempPathIXI
							MoveFile /Z/O destFolderStr+FileName as backupPathStr+subPathStr+FileName
						endif
					endif
					MoveFile /Z/O w_folders[folderIndex]+FileName as destFolderStr+FileName
					movedfileList=AddListItem(destFolderStr+FileName, movedfileList)
				endif
			endfor
			
			// make a list of subfolders in current folder
			folderList=IndexedDir(tempPathIXI, -1, 0)
			
			// remove folders that we don't want to copy
			folderList=RemoveFromListWC(folderList, ksIgnoreFoldersList)
			
			// add the list of folders to the subfolders wave
			for (subfolderIndex=0;subfolderIndex<ItemsInList(folderList);subfolderIndex+=1)
				w_subfolders[numpnts(w_subfolders)]={w_folders[folderIndex]+StringFromList(subfolderIndex,folderList)+":"}
				folderCount+=1
			endfor
		endfor
		// prepare for next sublevel iteration
		Duplicate /T/O/free w_subfolders, w_folders
		Redimension /N=0 w_subfolders
		sublevels+=1
	
		if(numpnts(w_folders)==0)
			break
		endif

	while(1)
	KillPath /Z tempPathIXI
	
	if((test==0) && killSourceFolder)
		DeleteFolder /Z RemoveEnding(source, ":")
	endif
	
	return SortList(movedfileList)
end

// ------------------------- Project Installer ---------------------

// install a user-contributed package hosted on Wavemetrics.com
// packageList is a list of project short names.
// The project URL can be used in place of the short name.
// Looks for install.itx within package folder and executes if found.
// install.itx should create shortcuts for help files and XOPs as needed.
// Use installWindows.itx or installMacintosh.itx for OS specific installations.

// install("foo") installs IgorExchange project with short title foo in a 
// location within the Igor User Files folder chosen by the user. 

// install("foo;bar;", path=SpecialDirPath("Igor Pro User 
// Files",0,0,0)+"User Procedures:") installs packages foo and bar in the
// User Procedures folder.

// install("foo;bar;", path="User Procedures") is an acceptable substitution.
// "User Procedures" is the only shortened alternative to a full path.

// if path is supplied for zip archive(s), subfolders may or may not 
// not be created, depending on the structure of the archive. 

function install(packageList, [path, gui])
	string packageList, path
	variable gui
	
	// gui is not intended to be set when running from commandline
	gui = paramIsDefault(gui) ? 0 : gui // default to gui=0: don't use DoAlert unless necessary
	path=selectstring(paramisdefault(path), path, "") // default to ""
	if(stringmatch(path, "User Procedures"))
		path = SpecialDirPath("Igor Pro User Files",0,0,0)+"User Procedures:"
	endif
	
	string url="", cmd="", projectName="", projectID=""
	variable selStart, selEnd, ListIndex
	variable nameIsURL=0, successfulInstalls=0
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	for(ListIndex=0;ListIndex<ItemsInList(packageList);ListIndex+=1)
		
		projectName=StringFromList(ListIndex, packageList)
		
		// projectName can be package short title, package URL
		// for new releases, or filepath for a local install.
		// URL IS THE PACKAGE URL, NOT THE ALL RELEASES PAGE
		if (isFile(projectName))			
			successfulInstalls+=InstallFile(projectName, path=path)
			continue			
		elseif (stringmatch(projectName, "https://www.wavemetrics.com/*"))
			nameIsURL=1
			url=projectName
		else
			sprintf url, "https://www.wavemetrics.com/project/%s", projectName
		endif
		

		// check the project page on IgorExchange	
		URLRequest /TIME=(prefs.pageTimeout)/Z url=url
		if(V_flag)
			sprintf cmd, "Installer could not load %s\r", url
			InstallFailMsg(cmd, gui)			
			continue
		endif
					
		if(nameIsURL) // try to extract a project name from web page
			// this should work for projects older than updater, ie. projectID <= 8197 
			selStart=strsearch(S_serverResponse, "<link rel=\"canonical\" href=\"", 0, 2)
			selStart+=28
			selEnd=strsearch(S_serverResponse, "\"", selStart, 0)
			if (selEnd==-1 || selStart<28)
				sprintf cmd, "Installer could not find releases at %s\r", url
				InstallFailMsg(cmd, gui)
				continue
			endif
			url=S_serverResponse[selStart,selEnd-1]
			projectName=ParseFilePath(0, url, "/", 1, 0) // projectName is short title of project
		endif
		
		if( GrepString(projectName,"(?i)[a-z]+") == 0 ) // projectID > 8197 
			// projectName is numeric
			selStart=strsearch(S_serverResponse, "<h1 class=\"page-title\" title=\"", 0, 2)
			if(selStart>-1)
				selStart+=30
				selEnd=strsearch(S_serverResponse, "\"", selStart, 0)
				projectName=S_serverResponse[selStart,selEnd-1] // set projectName to full title
			else
				projectName="Project "+projectName
			endif
		endif
				
		printf "Found %s at %s\r", projectName, url
	
		// find the link to the 'all releases' page for this project
		selEnd=strsearch(S_serverResponse, ">View all releases</a>", 0, 2)
		selStart=strsearch(S_serverResponse, "href=", selEnd, 3)
		if (selEnd==-1 || selStart==-1)
			sprintf cmd, "Installer could not find releases at %s\r", url
			InstallFailMsg(cmd, gui)	
			continue
		endif
		url="https://www.wavemetrics.com"+S_serverResponse[selStart+6,selEnd-2]		
		projectID=ParseFilePath(0, url, "/", 1, 0)

		successfulInstalls+=installProject(projectID, gui=gui, path=path, shortTitle=projectName)
	endfor	
	return successfulInstalls	
end

// url supplies the file to be installed
// this is to be used for local install
function installProject(projectID, [gui, path, shortTitle, url])
	string projectID
	variable gui
	string path, shortTitle, url
	
	// gui is not intended to be set when running from commandline
	gui = paramIsDefault(gui) ? 0 : gui
	// if we don't know the short title that's unfortunate
	string packageName=selectstring(paramisdefault(shortTitle), shortTitle, "project "+projectID)
	// packageName will be used until we get a better idea for shortTitle. projectName is the full title.
	shortTitle=selectstring(paramisdefault(shortTitle), shortTitle, "")
	path=selectstring(paramisdefault(path), path, "")
	url=selectstring(paramisdefault(url), url, "")
	
	// where to save downloaded files - a temporary location
	string downloadPathStr=SpecialDirPath(ksDownloadlocation,0,0,0)+"TonyInstaller:"	
		
	string projectName, cmd
	string IncludeProcStr="", packagePathStr="", releaseExtra=""
	string fileList="", fileName="", fileExtension="", destFileName="", fileNameNoExt=""
	variable releaseIgorVersion, releaseVersion, releaseMajor, releaseMinor
	variable selStart, selEnd, i
	variable ListIndex=0, success=0, restricted=0, isZip=0
	
	variable currentIgorVersion=getIgorVersion()
	string system=selectString(stringmatch(StringByKey("OS", igorinfo(3))[0,2],"Mac"), "Windows", "Macintosh")
//	variable isWin=stringmatch(system[0,2],"Win")
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	if(strlen(url)==0)
		// try to download 'all releases' page
		sprintf url, "https://www.wavemetrics.com/project-releases/%s" projectID
		if(gui)
			setPanelStatus("Downloading "+url)	
		endif
		URLRequest /TIME=(prefs.pageTimeout)/Z url=url
		if(V_flag)
			sprintf cmd, "Could not load %s\r", url
			InstallFailMsg(cmd, gui)
			return 0
		endif
		printf "Looking for releases at %s\r", url
		
		// parse page to populate w_releases
		wave /T w_releases=ParseReleases(S_serverResponse) // free text wave
		if (dimsize(w_releases,0)==0)
			sprintf cmd, "Could not find packages at %s.\r", url
			InstallFailMsg(cmd, gui)
			return 0
		endif	
		
		// find most recent OS- and version-compatible release
		url=""
		for(i=0;i<dimsize(w_releases, 0);i+=1)
			projectName = w_releases[i][0]
			sscanf (w_releases[i][1])[5,inf], "%f.x-%f", releaseIgorVersion, releaseVersion
			if(V_flag!=2)
				releaseIgorVersion=0
			endif
			releaseMajor=str2num(w_releases[i][2])
			releaseMinor=str2num(w_releases[i][3])
			releaseExtra=w_releases[i][7]
			releaseVersion=str2num(w_releases[i][2]+"."+w_releases[i][3])
					
			if(currentIgorVersion<releaseIgorVersion)
				sprintf cmd, "%s %0.2f%s requires Igor Pro version >= %g\r", projectName, releaseVersion, releaseExtra, releaseIgorVersion
				InstallFailMsg(cmd, gui)
			elseif(FindListItem(system, w_releases[i][5])==-1)
				sprintf cmd, "%s %0.2f%s not available for %s\r", projectName, releaseVersion, releaseExtra, system
				InstallFailMsg(cmd, gui)
			else
				url=w_releases[i][4]
				break
			endif
		endfor
		if (strlen(url)==0)
			return 0
		endif
	endif
	
	// create the temporary folder
	newpath /C/O/Q DLPathTemp, downloadPathStr; killpath /z DLPathTemp
	
	// download package
	if(gui)
		setPanelStatus("Downloading "+url)	
	endif
	URLRequest /TIME=(prefs.fileTimeout)/Z/O/FILE=downloadPathStr+ParseFilePath(0, url,"/", 1, 0) url=url
	if(V_flag)
		sprintf cmd, "Could not load %s\r", url
		InstallFailMsg(cmd, gui)
		return 0
	endif
	fileName=S_fileName
	printf "Downloaded %s\r", url
	printf "Saved temporary file %s\r", fileName
	
	fileExtension=ParseFilePath(4, fileName,":", 0, 0)
	isZip=stringmatch(ParseFilePath(4, fileName,":", 0, 0), "zip")
	
	if(isZip)
		//string archivePathStr=fileName // path to zip file
		string archiveName=ParseFilePath(0, fileName, ":", 1, 0)

		// create a temporary directory for the uncompressed files
		string unzipPathStr=CreateUniqueDir(downloadPathStr, "Install")

		// inflate archive
		success = unzipArchive(fileName, unzipPathStr)
		if (success==0)
			printf "unzipArchive failed to inflate %s\r", fileName
			return 0
		endif
		printf "Inflated %s to %s\r", archiveName, unzipPathStr
		fileList=RecursiveFileList(unzipPathStr, "*")
	else
		fileList=fileName
	endif
	
	if(strlen(grepList(fileList,"((?i)(.ipf|.xop)$)")))
		restricted=1
	endif
	
	// determine destination for file(s)
	if(strlen(path)==0)	
		packagePathStr=ChooseInstallLocation(projectName, restricted)
		if (strlen(packagePathStr)==0)
			return 0
		endif	
	else
		getfileFolderInfo /Q/Z path
		if(v_flag!=0 || V_isFolder==0)
			sprintf cmd, "Install path not found: %s\r", path
			InstallFailMsg(cmd, gui)
			return 0
		endif
		packagePathStr=path
	endif
		
	fileList="" // keep a list of installed files	
	
	// move files/folders to destination	
	if(isZip)
		// test to find out which files will overwritten
		fileList=mergeFolder(unzipPathStr, packagePathStr, test=1)
		variable numFiles=ItemsInList(fileList)
		if (numFiles)
			cmd="The following files will be overwritten:\r"
			string tooMany= "\r*** TOO MANY FILES TO LIST ***\r"
			string cont="\rContinue?"
			variable cmdLen = strlen(cmd)+strlen(tooMany)+strlen(cont)
			for (i=0;i<numFiles;i+=1)
				fileName=StringFromList(i, fileList)
				fileName=fileName[strlen(SpecialDirPath("Igor Pro User Files",0,0,0)),inf]
				// avoid too-long alert
				if( cmdLen+strlen(fileName) >= 1023 )
					cmd+=tooMany
					break
				endif
				cmd+=fileName+"\r"
				cmdLen+=strlen(fileName+"\r")
			endfor
			cmd+=cont
			DoAlert 1, cmd
			if(v_flag==2)
				printf "Install cancelled for %s\r", packageName
				InstallerCleanup(downloadPathStr)
				return 0
			endif
		endif
		// move files into user procedures folder
		fileList=mergeFolder(unzipPathStr, packagePathStr, killSourceFolder=0)
		for (i=0;i<ItemsInList(fileList);i+=1)
			fileName=StringFromList(i, fileList) // this is full path to file
			printf "Moved %s to %s\r", ParseFilePath(0, fileName, ":", 1, 0), ParseFilePath(1, fileName, ":", 1, 0)
		endfor
		
	else // a single file		
		
		// remove suffix if needed
		fileNameNoExt=ParseFilePath(3, fileName, ":", 0, 0) // filename without extension
		if(kRemoveSuffix && grepstring(fileNameNoExt, "_[0-9]$"))	
			fileNameNoExt=removeending(removeending(fileNameNoExt)) // remove two characters
		endif
		destFileName=fileNameNoExt+"."+fileExtension
		destFileName=removeEncoding(destFileName)
		
		getFileFolderInfo /Q/Z packagePathStr+destFileName
		if (v_flag==0) // already exists
			DoAlert 1, "overwrite "+destFileName+"?"
			if (v_flag==2)
				return 0
			endif
		endif
		movefile /O/Z 	fileName as packagePathStr+destFileName
		if(V_flag!=0)
			return 0
		endif
		fileList=packagePathStr+destFileName
		printf "Saved %s in %s\r", destFileName, packagePathStr
	endif
			
	// figure out if we have a 'master' procedure file
	string IncludeProcPath=""
	string ipfList=listMatch(fileList, "*.ipf")
	variable numIPFs=ItemsInList(ipfList)	
	if(numIPFs==1) // only one procedure file
		IncludeProcPath=StringFromList(0, ipfList)
		IncludeProcStr=ParseFilePath(3, IncludeProcPath, ":", 0, 0)
	elseif(numIPFs>1)
		if(WhichListItem(packagePathStr+packageName+".ipf", ipfList)!=-1)
			// package folder includes a procedure with project name
			IncludeProcPath=packagePathStr+packageName+".ipf"
			IncludeProcStr=packageName
		elseif(WhichListItem(packagePathStr+replacestring(" ", packageName, "")+".ipf", ipfList)!=-1)
			IncludeProcPath=packagePathStr+replacestring(" ", packageName, "")+".ipf"
			IncludeProcStr=replacestring(" ", packageName, "")
		endif
	endif
		
	string msg
	
	// if there's an XOP in the package, don't try to open any procedure file
	string xopList=listMatch(fileList, "*.xop")
	if(strlen(xopList))
		IncludeProcStr=""
	endif
	
	// problem: if package has mac and windows xops, but mac xop has 
	// resource fork instead of extension, will offer to install an alias 
	// for the windows xop!
	
	// such packages are unlikely to be 64-bit compatible anyway
	
//	// if only one xop in package offer to create alias
//	if(ItemsInList(xopList)==1)
//		filename=StringFromList(0, xopList)
//		string ExtensionsPath=getIgorExtensionsPath()
//		// find out whether an alias already exists
//		variable aliasExists=(strlen(FindAlias(ExtensionsPath, filename))>0)
//		variable isInExtensions=stringmatch(filename, ExtensionsPath+"*")
//		if(!(aliasExists || isInExtensions))
//			sprintf msg, "Do you want to create an alias for\r%s\r)", ParseFilePath(0, filename, ":", 1, 0)
//			msg+="in the Igor Extensions folder?"
//			DoAlert 1, msg
//			if(v_flag==1)
//				string aliasName=ParseFilePath(3, filename, ":", 0, 0)+selectstring(isWin, " Alias", " Shortcut")
//				createAliasShortcut filename as ExtensionsPath+aliasName
//				printf "Created %s in %s\r", aliasName, ExtensionsPath
//			endif
//		endif
//	endif

					
	// Look for and run itx-format installation script. Script will run
	// from within the package folder saved in location of user's 
	// choice within Igor Pro User Files folder. 	
	string pathToScriptStr=listMatch(fileList, "*:install.itx")
	if(strlen(pathToScriptStr)==0)
		pathToScriptStr=listMatch(fileList, "*:install"+system+".itx")
	endif
	// listMatch returns a list, and will always have separator at end
	pathToScriptStr= StringFromList(0,pathToScriptStr)		// keep only the first item from the list, no trailing separator.
	if(strLen(pathToScriptStr)) // script could do naughty things, so ask before allowing it to run
		sprintf msg, "Run installation script?\r%s was downloaded from %s", ParseFilePath(0, pathToScriptStr, ":", 1, 0), url 
		
		// security check here?
		
		DoAlert 1, msg
		if (v_flag==1)
			sprintf cmd, "Loadwave /T \"%s\"", pathToScriptStr
			execute cmd // will be printed to history
		endif
	endif
	
	if (strlen(IncludeProcStr)>0 && isLoaded(IncludeProcPath)==0) 
		// 'master' procedure file is not loaded		
		sprintf cmd "Do you want to include %s in the current experiment?", IncludeProcStr
		DoAlert 1, cmd
		if(v_flag==1)
			// load the procedure
			sprintf cmd, "INSERTINCLUDE \"%s\"", IncludeProcStr
			Execute/P/Q/Z cmd
			Execute/P/Q/Z "COMPILEPROCEDURES "
			printf "Included %s in current experiment\r", IncludeProcStr
		endif			
	endif
			
	LogUpdateProject(projectID, shortTitle, packagePathStr, num2str(releaseVersion), fileList)
	
	InstallerCleanup(downloadPathStr)	
	return 1	
end

// install from a local file
// filePath is full path to file
// (local path from itx script works too)
// path is install location, default is User Procedures
// path can be full path, "Igor Procedures", "Igor Extensions"
// if projectID, shortTitle, and version can be determined
// project will be added to install log

// Multi-file projects must be packed into a zip archive.
// Usually filePathList will be path to one file, typically a zip archive.
// Provide a list (without setting projectID, shortTitle, and version) to
// install multiple packages.

function InstallFile(filePathList, [path, projectID, shortTitle, version])
	string filePathList, path, projectID, shortTitle, version
	
	variable numPackagesToInstall=ItemsInList(filePathList)
	variable numInstalled=0, fileNum=0
	string filePath
	
	for(fileNum=0;fileNum<numPackagesToInstall;fileNum+=1)
		filePath=StringFromList(fileNum, filePathList)
	
		if(isFile(filePath)==0)
			// if InstallFile has been executed from an ITX script, 
			// look in location of itx file for install files
			SVAR ITXpath=s_path
			if(SVAR_Exists(ITXpath))
				filePath=ITXpath+filePath
			endif
			if(isFile(filePath)==0)
				print "File not found "+filePath
				continue
			endif
		endif	
		
		string fileList, ipfList, ipfFile, fileName, cmd
		variable success, isZip, i
		projectID=selectstring(ParamIsDefault(projectID), projectID, "")
		shortTitle=selectstring(ParamIsDefault(shortTitle), shortTitle, "")
		version=selectstring(ParamIsDefault(version), version, "")
		
		if(ParamIsDefault(path) || stringmatch(path, "User Procedures") || strlen(path)==0)
			path = SpecialDirPath("Igor Pro User Files",0,0,0)+"User Procedures:"
		elseif(stringmatch(path, "Igor Procedures"))
			path = SpecialDirPath("Igor Pro User Files",0,0,0)+"Igor Procedures:"
		elseif(stringmatch(path, "Igor Extensions"))
			path = getIgorExtensionsPath()
		endif
		
		isZip=grepstring(filePath,"(?i).zip$")
		
		if(isZip)
			string archiveName=ParseFilePath(0, filePath, ":", 1, 0)
			string archiveFolder=ParseFilePath(1, filePath, ":", 1, 0)
			// create a temporary directory for the uncompressed files
			string unzipPathStr=CreateUniqueDir(archiveFolder, "Install")
			if(strlen(unzipPathStr)==0)
				printf "could not create directory in %s\r", archiveFolder
				continue
			endif
			// inflate archive
			success = unzipArchive(filePath, unzipPathStr)
			if (success==0)
				printf "unzipArchive failed to inflate %s\r", filePath
				continue
			endif
			printf "Inflated %s to %s\r", archiveName, unzipPathStr
			fileList=RecursiveFileList(unzipPathStr, "*")
		else
			fileList=filePath
		endif
		
		ipfList=listMatch(fileList, "*.ipf")
		
		if(ItemsInList(ipfList)==1)
			ipfFile=StringFromList(0,ipfList)
			projectID=selectstring(strlen(projectID)==0, projectID, getProjectIDString(ipfFile))
			shortTitle=selectstring(strlen(shortTitle)==0, shortTitle, getShortTitle(ipfFile))
			version=selectstring(strlen(version)==0, version, num2str(getProcVersion(ipfFile)))
		endif
			
		// check destination for file(s)
		getfileFolderInfo /Q/Z path
		if(v_flag!=0 || V_isFolder==0)
			printf "Install path not found: %s\r", path
			continue
		endif
			
		fileList="" // keep a list of installed files	
		
		// move files/folders to destination	
		if(isZip)
			// test to find out which files will overwritten
			fileList=mergeFolder(unzipPathStr, path, test=1)
			variable numFiles=ItemsInList(fileList)
			if (numFiles)
				cmd="The following files will be overwritten:\r"
				string tooMany= "\r*** TOO MANY FILES TO LIST ***\r"
				string cont="\rContinue?"
				variable cmdLen = strlen(cmd)+strlen(tooMany)+strlen(cont)
				for (i=0;i<numFiles;i+=1)
					fileName=StringFromList(i, fileList)
					fileName=fileName[strlen(path),inf]
					// avoid too-long alert
					if( cmdLen+strlen(fileName) >= 1023 )
						cmd+=tooMany
						break
					endif
					cmd+=fileName+"\r"
					cmdLen+=strlen(fileName+"\r")
				endfor
				cmd+=cont
				DoAlert 1, cmd
				if(v_flag==2)
					printf "Install cancelled\r"
					continue
				endif
			endif
			// move files into user procedures folder
			fileList=mergeFolder(unzipPathStr, path, killSourceFolder=0)
			for (i=0;i<ItemsInList(fileList);i+=1)
				fileName=StringFromList(i, fileList) // this is full path to file
				printf "Moved %s to %s\r", ParseFilePath(0, fileName, ":", 1, 0), ParseFilePath(1, fileName, ":", 1, 0)
			endfor
			
		else // a single file		
			fileName=ParseFilePath(0, filePath, ":", 1, 0)
			getFileFolderInfo /Q/Z path+fileName
			if (v_flag==0) // already exists
				DoAlert 1, "overwrite "+fileName+"?"
				if (v_flag==2)
					continue
				endif
			endif
			movefile /O/Z 	filePath as path+fileName
			if(V_flag!=0)
				continue
			endif
			fileList=path+fileName
			printf "Saved %s in %s\r", fileName, path
		endif			
		
		// if we have the required parameters, register this project in the install log 
		// to check for updated releases at wavemetrics.com 
		if(strlen(projectID)&&strlen(shortTitle)&&strlen(version))
			LogUpdateProject(projectID, shortTitle, path, version, fileList)
			printf "Added %s to install log\r", shortTitle
		endif
		numInstalled+=1
	endfor // next file
	return numInstalled
end

// probably not undoable
// folders and subfolders will be left in place
function uninstallProject(projectID)
	string projectID
	
	string installedfileList=LogGetFileList(projectID)
	string installPath=LogGetInstallPath(projectID)
	string deletefileList="", filePath="", fileName="", deletePath="", cmd=""
	variable numFiles, i
	numFiles=ItemsInList(installedfileList)
	for(i=0;i<numFiles;i+=1)
		filePath=StringFromList(i, installedfileList)
		deletefileList+=selectstring(isFile(filePath), "", filePath+";")
	endfor
	numFiles=ItemsInList(deletefileList)
	if(numFiles)
		cmd="The following files will be deleted:\r"
		string tooMany= "\r*** TOO MANY FILES TO LIST ***\r"
		string cont="\rContinue?"
		variable cmdLen = strlen(cmd)+strlen(tooMany)+strlen(cont)
		for (i=0;i<numFiles;i+=1)
			fileName=StringFromList(i, deletefileList)
			if(stringmatch(filename, installPath+"*"))
				filename=filename[strlen(installPath),inf]
			endif
			if( cmdLen+strlen(fileName) >= 1023 )
				cmd+=tooMany
				break
			endif
			cmd+=fileName+"\r"
			cmdLen+=strlen(fileName+"\r")
		endfor
		cmd+=cont
		DoAlert 1, cmd
		if(v_flag==2)
			print "Uninstall cancelled"
			return 0
		endif
		deletePath=CreateUniqueDir(SpecialDirPath("Temporary",0,0,0), projectID)
	endif
	
	moveFiles(deletefileList, installPath, deletePath)
	logRemoveProject(projectID)
	return 1
end

// removes some potential url encodings
function /s removeEncoding(s)
	string s
	
	make /free/t w_encoded={"%24","%26","%2B","%2C","%2D","%2E","%3D","%40","%20","%23","%25"}
	make /free/t w_unencoded={"$","&","+",",","-",".","=","@"," ","#","%"}
	variable i, imax=numpnts(w_encoded)
	for(i=0;i<imax;i+=1)
		s=ReplaceString(w_encoded[i], s, w_unencoded[i])
	endfor
	return s
end

function /s getIgorExtensionsPath()
	
	newPath /O/Q/Z tempPathIXI, SpecialDirPath("Igor Pro User Files", 0, 0, 0)
	string folderList=IndexedDir(tempPathIXI, -1, 0)
	killpath /Z tempPathIXI
	string ExtensionsPath=listMatch(folderList, "Igor Extensions*")
	return SpecialDirPath("Igor Pro User Files", 0, 0, 0)+StringFromList(0, ExtensionsPath)+":"
end

// -------------- functions for writing to and querying install-log file ------------

// File is named install.log and is located in a subfolder of User Procedures folder
// First line records the version of this procedure that created the file.
// Subsequent lines are semicolon-separated lists, terminated by carriage return.
// Each list starts with projectID. 
// First six items are projectID;shortTitle;version;installDate;dirIDStr;installPath;
// Next four items are reserved for future use
// List items 10 onward are paths to installed files.
// installPath is either path from directory specified by dirIDStr, or 
// full path for installations outside of User Files. Paths to files 
// start from folder given by installPath. dirIDStr specifies a file 
// system directory parameter, as used by SpecialDirPath. When dirIDStr 
// is empty, InstallPath is full path. Setting dirIDStr allows files to 
// be copied to new installations.

// creates a log file if one doesn't exist
function /s getPathToLog()
	
	string UserFilesPath=SpecialDirPath("Igor Pro User Files", 0, 0, 0)
	string InstallerPath=UserFilesPath+ksLogPath
	if(isFile(InstallerPath+"install.log"))
		return InstallerPath+"install.log"
	else
		string tmpPath=UserFilesPath // 
		variable i, subfolders
		subfolders=ItemsInList(ksLogPath, ":")
		for(i=0;i<subfolders;i+=1)
			tmpPath+=StringFromList(i, ksLogPath, ":")+":"
			newPath /C/O/Q/Z tempPathIXI, tmpPath
		endfor
		killpath /Z tempPathIXI
	endif
	// InstallerPath folder should now exist
	// before we create install.log check that we have a human-readable 
	// file in the same location
	if(isFile(InstallerPath+"installer-readme.txt")==0)
		variable refnum
		open /A/Z refnum as InstallerPath+"installer-readme.txt"
		if(V_flag==0)
			fprintf refnum, "The files in this folder were created by IgorExchange Installer\r"
			fprintf refnum, "https://www.wavemetrics.com/project/Updater\r"
			fprintf refnum, "Do not delete or move install.log!\r"
			close refnum
		endif
	endif
	// now create the log file
	open /A/Z refnum as InstallerPath+"install.log"
	if(V_flag==0)
		variable version=getProcVersion(functionpath(""))
		fprintf refnum, "Install log created by IgorExchange Installer %0.2f\r", version
		close refnum
	endif
	
	return InstallerPath+"install.log"
end

// replaces any existing log entry for projectID
function LogUpdateProject(projectID, shortTitle, installPath, newVersion, fileList)
	string projectID, shortTitle, installPath, newVersion, fileList
	variable restricted
	
	LogRemoveProject(projectID)
	
	variable i
	string s_out, file, root="", installSubPath=""
	string UserFilesPath=SpecialDirPath("Igor Pro User Files", 0, 0, 0)
	string DocumentsPath=SpecialDirPath("Documents", 0, 0, 0)
	if(stringmatch(installPath,UserFilesPath+"*"))
		root="Igor Pro User Files"
		installSubPath=installPath[strlen(UserFilesPath), inf]
	elseif(stringmatch(installPath,DocumentsPath+"*"))
		root="Documents"
		installSubPath=installPath[strlen(DocumentsPath), inf]
	endif
	
	sprintf s_out "%s;%s;%s;%s;%s;%s;;;;;", projectID, shortTitle, newVersion, num2istr(datetime), root, installSubPath
	for(i=0;i<ItemsInList(fileList);i+=1)
		file=StringFromList(i, fileList)
		if(stringmatch(file, installPath+"*")) // this should always be true
			file=file[strlen(installPath), inf]
		endif
		s_out+=file+";"
	endfor
	s_out+="\r"
	return LogAppend(s_out)
end

function LogUpdateInstallPath(projectID, installPath)
	string projectID, installPath
	
	if(strlen(installPath)==0)
		return 0
	endif
	
	string root=""
	string UserFilesPath=SpecialDirPath("Igor Pro User Files", 0, 0, 0)
	string DocumentsPath=SpecialDirPath("Documents", 0, 0, 0)
	if(stringmatch(installPath,UserFilesPath+"*"))
		root="Igor Pro User Files"
		installPath=installPath[strlen(UserFilesPath), inf]
	elseif(stringmatch(installPath,DocumentsPath+"*"))
		root="Documents"
		installPath=installPath[strlen(DocumentsPath), inf]
	endif
		
	string logEntry=LogGetProject(projectID)
	LogRemoveProject(projectID)
	
	variable i
	string s_out=""
	for(i=0;i<ItemsInList(logEntry);i+=1)
		switch(i)
			case 4:
				s_out+=root+";"
				break
			case 5:
				s_out+=installPath+";"
	 			break
	 		default:
	 			s_out+=StringFromList(i, logEntry)+";"
		endswitch
	endfor
	return LogAppend(s_out)
end

// s is a complete line
function LogAppend(s)
	string s
	
	string filePath=getPathToLog()
	variable refnum
	open /A/Z refnum as filePath
	if(V_flag)
		return 0
	endif
	
	s=removeEnding(s, "\r")+"\r"
	fprintf refnum, "%s", s
	close refnum
	return 1
end

function /S LogGetProject(projectID)
	string projectID
	
	string filePath=getPathToLog()	
	if(isFile(filePath)==0)
		return ""
	endif
	
	make /T/N=0/free w
	string s_exp=""
	sprintf s_exp, "^%s;", projectID
	// extract lines starting with projectID into free text wave
	Grep /O/Z/E=s_exp FilePath as w
	if(V_value)
		return w[0]
	endif
	return ""
end

// returns currently installed version as string
function /S LogGetVersion(projectID)
	string projectID
	
	string filePath=getPathToLog()	
	if(isFile(filePath)==0)
		return ""
	endif
	
	make /T/N=0/free w
	string s_exp=""
	sprintf s_exp, "^%s;", projectID
	// extract lines starting with projectID into free text wave
	Grep /O/Z/E=s_exp FilePath as w
	if(V_value)
		return StringFromList(2, w[0])
	endif
	return ""
end

// for single file projects, get full path to file
function /S LogGetFilePath(projectID)
	string projectID
	
	string logEntry=LogGetProject(projectID)
	if(strlen(logEntry)==0)
		return ""
	endif
		
	string root=StringFromList(4, logEntry)	
	if(strlen(root))
		root=SpecialDirPath(root, 0, 0, 0)
	endif
	return root+StringFromList(5, logEntry)+StringFromList(10, logEntry)
end	

function /S LogGetInstallPath(projectID)
	string projectID
	
	string logEntry=LogGetProject(projectID)
	if(strlen(logEntry)==0)
		return ""
	endif
		
	string root=StringFromList(4, logEntry)	
	if(strlen(root))
		root=SpecialDirPath(root, 0, 0, 0)
	endif
	return root+StringFromList(5, logEntry)
end	

function /S LogGetFileList(projectID)
	string projectID
	
	string logEntry=LogGetProject(projectID)
	if(strlen(logEntry)==0)
		return ""
	endif
		
	string installPath=StringFromList(4, logEntry)
	if(strlen(installPath))
		installPath=SpecialDirPath(installPath, 0, 0, 0)
	endif
	installPath+=StringFromList(5, logEntry)
		
	string fileList=""
	variable i
	variable imax=ItemsInList(logEntry)
	for(i=10;i<imax;i+=1)
		fileList+=installPath+StringFromList(i, logEntry)+";"
	endfor
	return fileList
end	

// remove missing projects from log file
function LogCleanup()
	
	string filePath=getPathToLog()	
	if(isFile(filePath)==0)
		return 0
	endif

	make /T/N=0/free w
	// extract lines into free text wave
	Grep /O/Z/E="" FilePath as w
	
	variable i=0,j=0, numFiles=0
	string root="", installPath=""	
	do
		root=StringFromList(4, w[0])
		if(strlen(root))
			InstallPath=SpecialDirPath(root, 0, 0, 0)
		endif
		InstallPath+=StringFromList(5, w[0]) // full path to install location	
		// fileList starts at item 10 in list
		for(j=10;j<ItemsInList(w[i]);j+=1)
			numFiles+=isFile(installPath+StringFromList(j, w[i]))
		endfor
		if(numFiles==0)
			deletepoints i, 1, w
		else
			i+=1
		endif
	while(i<numpnts(w))
		
	// overwrite log file with lines from textwave
	Grep /O/Z/E="" w as filePath
	return V_value // number of projects remaining in log 
end

function /s LogProjectsList()
	
	string filePath=getPathToLog()
	if(isFile(filePath)==0)
		return ""
	endif
	
	string projectList="", strLine=""
	variable i, numLines
	// use grep to read all project lines into a string
	Grep /Q/O/Z/E=";"/LIST="\r" FilePath
	numLines=ItemsInList(s_value, "\r")
	for (i=0;i<numLines;i+=1)
		strLine=StringFromList(i, s_value, "\r")
		projectList=AddListItem(StringFromList(0, strLine), projectList)
	endfor
	return projectList
end

// examines log file to figure out whether project can be updated
// updates not allowed for incomplete or missing projects
function /S getUpdateStatus(projectID, localVersion, remoteVersion)
	string projectID
	variable localVersion, remoteVersion
	
	string status=getInstallStatus(projectID)
	if( remoteVersion>localVersion && stringmatch(status, "complete") )
		status += ", update available"
	elseif(stringmatch(status, "complete"))
		status += ", up to date"
	endif
	return status	
end

// returns "complete", "incomplete", or "missing" depending on how many of the installed files can be located
function /s getInstallStatus(projectID)
	string projectID
	
	if(strlen(projectID)==0)
		return ""
	endif
	
	string filePath=getPathToLog()
	if(isFile(filePath)==0)
		return ""
	endif
	
	// extract line starting with projectID into free text wave
	string s_exp=""
	variable numfiles=0, i=0
	sprintf s_exp, "^%s;", projectID
	make /T/N=0/free w
	Grep /O/Z/E=s_exp FilePath as w
	
	if(numpnts(w)==0)
		return ""
	endif
	
	string root="", InstallPath=""
	root=StringFromList(4, w[0])
	if(strlen(root))
		InstallPath=SpecialDirPath(root, 0, 0, 0)
	endif
	InstallPath+=StringFromList(5, w[0]) // full path to install location
		
	for(i=10;i<ItemsInList(w[0]);i+=1) // list of files starts at item 10
		numFiles+=isFile(InstallPath+StringFromList(i, w[0]))
	endfor
	if(numFiles==0)
		return "missing"
	elseif(numFiles<(ItemsInList(w[0])-10))
		return "incomplete"
	endif
	return "complete"
end

function /s getInfo(projectID)
	string projectID
	
	string filePath=getPathToLog()
	if(isFile(filePath)==0)
		return ""
	endif
	
	// extract line starting with projectID into free text wave
	string s_exp="", fileList=""
	sprintf s_exp, "^%s;", projectID
	make /T/N=0/free w
	Grep /O/Z/E=s_exp FilePath as w
	
	if(numpnts(w)==0)
		return ""
	endif
	
	string root="", subpath="", InstallPath="", info=""
	root=StringFromList(4, w[0])
	subpath=StringFromList(5, w[0])
	if(strlen(root))
		InstallPath=SpecialDirPath(root, 0, 0, 0)
	endif
	InstallPath+=subpath // full path to install location
		
	string strFile=""
	variable i=0, numFiles=0, numIPF=0, numXOP=0, numPXP=0, numOther=0, numMissing=0
		
	for(i=10;i<ItemsInList(w[0]);i+=1) // list of files starts at item 10
		strFile=InstallPath+StringFromList(i, w[0])
		numFiles+=1
		if(isFile(strFile)==0)
			numMissing+=1
		elseif(grepString(strFile,"(?i).ipf$"))
			numIPF+=1
		elseif(grepString(strFile,"(?i).xop$"))
			numXOP+=1
		elseif(grepString(strFile,"(?i).pxp$"))
			numPXP+=1	
		else
			numOther+=1
		endif
	endfor
	
	sprintf info, "%s\r", subpath
	if(numIPF)
		info+=" "+num2str(numIPF)+" ipf,"
	endif
	if(numXOP)
		info+=" "+num2str(numXOP)+" xop,"
	endif
	if(numPXP)
		info+=" "+num2str(numPXP)+" pxp,"
	endif
	if(numOther)
		info+=" "+num2str(numOther)+" other,"
	endif
	if(numMissing)
		info+=" "+num2str(numMissing)+" missing,"
	endif

	return removeending(info, ",")
end

// clear a project from the log file
function LogRemoveProject(projectID)
	string projectID
		
	string filePath=getPathToLog()
	if(isFile(filePath)==0)
		return 0
	endif
	
	make /T/N=0/free w
	string s_exp=""
	sprintf s_exp, "^%s;", projectID
	// extract lines not starting with projectID into free text wave
	Grep /O/Z/E={s_exp, 1} FilePath as w
	// overwrite log file with all lines from textwave
	Grep /O/Z/E="" w as filePath
	return V_value // number of projects remaining in log
end

// lineNum is zero based line number
// projects start on line 1
function LogRemoveLine(lineNum)
	variable lineNum
	
	string filePath=getPathToLog()
	if(isFile(filePath)==0)
		return 0
	endif
	
	make /T/N=0/free w
	// extract lines into free text wave
	Grep /O/Z/E="" FilePath as w
	deletepoints linenum, 1, w
	// overwrite log file with lines from textwave
	Grep /O/Z/E="" w as filePath
	return V_value // number of projects remaining in log 
end

// --------- functions for writing to and retrieving projects from cache file --------

// Cache file structure:
// Semicolon-separated list for each project, list starts with projectID,
// terminated by carriage return.
// 15 items in list:
// projectID;ProjectCacheDate;title;author;published;views;type;userNum;
// ReleaseCacheDate;shortTitle;remote;system;releaseDate;releaseURL;releaseIgorVersion

// creates a cache file if one doesn't exist
function /s getPathToCache()
	
	string UserFilesPath=SpecialDirPath("Igor Pro User Files", 0, 0, 0)
	string InstallerPath=UserFilesPath+ksLogPath
	if(isFile(InstallerPath+"cache.txt"))
		return InstallerPath+"cache.txt"
	else
		string tmpPath=UserFilesPath // 
		variable i, subfolders
		subfolders=ItemsInList(ksLogPath, ":")
		for(i=0;i<subfolders;i+=1)
			tmpPath+=StringFromList(i, ksLogPath, ":")+":"
			newPath /C/O/Q/Z tempPathIXI, tmpPath
		endfor
		killpath /Z tempPathIXI
	endif
	// InstallerPath folder should now exist
	// before we create cache, check that we have a human-readable 
	// file in the same location
	if(isFile(InstallerPath+"installer-readme.txt")==0)
		variable refnum
		open /A/Z refnum as InstallerPath+"installer-readme.txt"
		if(V_flag==0)
			fprintf refnum, "The files in this folder were created by IgorExchange Installer\r"
			fprintf refnum, "https://www.wavemetrics.com/project/Updater\r"
			fprintf refnum, "Do not delete or move install.log!\r"
			close refnum
		endif
	endif
	// now create the cache file
	open /A/Z refnum as InstallerPath+"cache.txt"
	if(V_flag==0)
		variable version=getProcVersion(functionpath(""))
		fprintf refnum, "Cache file created by IgorExchange Installer %0.2f\r", version
		close refnum
	endif
	
	return InstallerPath+"cache.txt"
end

function /s CacheGetProjectsList()
	
	string filePath=getPathToCache()
	if(isFile(filePath)==0)
		return ""
	endif
	
	string projectList="", strLine=""
	variable i, numLines
	// use grep to read all project lines into a string
	Grep /Q/O/Z/E=";"/LIST="\r" FilePath
	numLines=ItemsInList(s_value, "\r")
	for (i=0;i<numLines;i+=1)
		strLine=StringFromList(i, s_value, "\r")
		projectList=AddListItem(StringFromList(0, strLine), projectList)
	endfor
	return projectList
end

// retrieve project from cache and return keylist of cached parameters
function /S CacheGetKeyList(projectID)
	string projectID
	
	string s=CacheGetProject(projectID)
	if(strlen(s)==0)
		return ""
	endif
	return CacheString2KeyList(CacheGetProject(projectID))
end

// retrieve project from cache and return list of cached parameters
function /S CacheGetProject(projectID)
	string projectID
	
	string filePath=getPathToCache()
	
	string s_exp
	sprintf s_exp, "^%s;", projectID
	grep /Z/Q/LIST /E=s_exp filePath
	
	if(strlen(s_value))
		return s_value
	endif
	return ""
end

//function /S CacheGetLastPutTime(projectID)
//	string projectID
//	
//	string keylist=CacheGetKeyList(projectID)
//	variable secs=str2num(stringbykey("ReleaseCacheDate", keylist))
//	return secs2date(secs, 0)+" "+secs2time(secs, 0)
//end

// turns a line from cache into a keylist
// avoids empty keypairs
function /S CacheString2KeyList(cacheEntry)
	string cacheEntry
	
	string keyList="", s="", key=""
	string keys="projectID;ProjectCacheDate;title;author;published;views;type;userNum;"
	keys+="ReleaseCacheDate;shortTitle;remote;system;releaseDate;releaseURL;releaseIgorVersion"
	variable i
	
	for(i=0;i<15;i+=1)	
		s=StringFromList(i, cacheEntry)
		if(strlen(s))
			key=StringFromList(i,keys)
			keyList=ReplaceStringByKey(key, keyList, s)
		endif
	endfor
	return keyList
end

// inserts keyList values into cache
// keyList should not have empty values
function CachePutKeylist(keyList)
	string keyList
	
	string projectID=StringByKey("projectID", keyList)
	if(strlen(projectID)==0)
		return 0
	endif
	
	string filePath=getPathToCache()
	
	string oldKeyList=CacheGetKeyList(projectID)
	keyList+=oldKeyList // items not in new keyList will be set to their value (if any) in oldKeyList

	string s_out="", key=""
	string keys="projectID;ProjectCacheDate;title;author;published;views;type;userNum;"
	keys+="ReleaseCacheDate;shortTitle;remote;system;releaseDate;releaseURL;releaseIgorVersion"
	variable i
	
	for(i=0;i<15;i+=1)	
		key=StringFromList(i,keys)
		s_out+=StringByKey(key, keyList)+";"
	endfor
	
	string s_exp
	sprintf s_exp, "^%s;", projectID		
	make /T/N=0/free w
	// extract lines not starting with projectID into free text wave
	Grep /O/Z/E={s_exp, 1} FilePath as w
	w[numpnts(w)]={s_out}
	sort /A w, w // header remains at point 0
	// overwrite cache file with all lines from textwave
	Grep /O/Z/E="" w as filePath
	
	return V_value // number of projects remaining in log
end

function CachePutProjectsWave(ProjectsWave)
	wave /T ProjectsWave

	if(dimsize(ProjectsWave, 0)==0)
		return 0
	endif
	
	// load cache, if it exists, as text wave
	string filePath=getPathToCache()
	make /T/N=0/free CacheWave
	Grep /O/Z/E="" FilePath as CacheWave 
	
	string s_exp="", strList="", projectID=""
	string installerListDate=num2istr(datetime)
	variable i
	
	for(i=0;i<DimSize(ProjectsWave, 0);i+=1)
		projectID=ProjectsWave[i][0]
		strList=projectID+";;;;;;;;;;;;;;;"
		sprintf s_exp, "^%s;", projectID
		Grep /Z/Q/LIST/INDX /E=s_exp CacheWave
		if(strlen(s_value))
			wave w_index
			// remove project from CacheWave
			DeletePoints /M=0 (w_index[0]), 1, CacheWave
			// get any values that have been set by project installer
			strList=s_value
		endif
		
		// insert new update info into list string
		string s
		sprintf s, "%s;%s;%s;%s;%s;" installerListDate,ProjectsWave[i][%name],ProjectsWave[i][%author],ProjectsWave[i][%published],ProjectsWave[i][%views]
		sprintf s, "%s%s;%s" s,ReplaceString(";",ProjectsWave[i][%type],","),ProjectsWave[i][%userNum]
		strList=ReplaceListItem(1, strList, s, numItems=7) // avoid trailing ;

		// write project to CacheWave
		CacheWave[numpnts(CacheWave)]={strList}
	endfor
	
	if(numpnts(CacheWave)==0)
		return 0
	endif
	
	sort /A CacheWave,CacheWave // header stays at point 0
	grep /O/Z/E="" CacheWave as filePath
	return v_value	
endÍ

// set item 8, the time that updates info was cached, to 0 for each project
function CacheClearUpdates()
	
	string filePath=getPathToCache()
	make /T/N=0/free CacheWave
	Grep /O/Z/E="" FilePath as CacheWave // load cache if it exists
	if(numpnts(CacheWave)==0)
		return 0
	endif
	CacheWave[1,inf]=ReplaceListItem(8, CacheWave[p], "0")
	grep /O/Z/E="" CacheWave as filePath
	return v_value	
end

// Reset cache file
function CacheClearAll()
	
	string filePath=getPathToCache()	
	variable refnum
	open /Z refnum as filePath
	if(V_flag==0)
		variable version=getProcVersion(functionpath(""))
		fprintf refnum, "Cache file created by IgorExchange Installer %0.2f\r", version
		close refnum
	endif
	return V_flag	
end

// replaces list item(s) in listStr, starting from itemNum, with contents of itemStr
// note that a trailing ; in itemStr will add an extra empty item
function /S ReplaceListItem(itemNum, listStr, itemStr, [numItems])
	variable itemNum
	string listStr, itemStr
	variable numItems
	numItems = ParamIsDefault(numItems) ? 1 : numItems
	variable i
	for(i=0;i<numItems;i+=1)
		listStr=RemoveListItem(itemNum, listStr)
	endfor
	
	return AddListItem(itemStr, listStr, ";", itemNum)
end

// -------------- end of Cache functions -----------

function isLoaded(procPath)
	string procPath
	
	string procName=ParseFilePath(3, procPath, ":", 0, 0)
	wave /T w_loadedProcs=ListToTextWave(winlist("*",";","WIN:128,INDEPENDENTMODULE:1"),";")
	if(numpnts(w_loadedProcs))
		w_loadedProcs=FileNameFromProcName(w_loadedProcs)
	endif
	FindValue /TEXT=procName/TXOP=4 /Z w_loadedProcs
	return (V_Value>-1)
end

function InstallFailMsg(cmd, gui)
	string cmd
	variable gui
	
	print cmd
	if(gui)
		DoAlert 0, cmd
	endif
	return 1
end

// returns a path selected by user
// path leads to a subfolder of Igor Pro User Files when restricted is non-zero
function /S ChooseInstallLocation(packageName, restricted)
	string packageName
	variable restricted
		
	variable success
	string cmd
	if(strlen(packageName))
		packageName=" "+packageName
	endif
	sprintf cmd, "Where would you like to install%s?", packageName
	do
		success=1
		if(restricted)
			NewPath /O/Q/Z TempInstallPath SpecialDirPath("Igor Pro User Files",0,0,0)+"User Procedures:"
			PathInfo /S TempInstallPath // set start directory to user procedures
		endif
		NewPath /M=cmd /O/Q/Z TempInstallPath
		if(v_flag)
			printf "Install cancelled\r"
			return ""
		endif
		PathInfo /S TempInstallPath
		if(restricted)
			if(stringmatch(S_path, SpecialDirPath("Igor Pro User Files",0,0,0)+"*")==0)
				DoAlert 0, "Please select a location within the Igor Pro User Files folder\r(usually in the User Procedures folder)"
				success=0
			endif
			if(stringmatch(S_path, SpecialDirPath("Igor Pro User Files",0,0,0)))
				DoAlert 0, "Please select a subfolder within the Igor Pro User Files folder\r(usually in the User Procedures folder)"
				success=0
			endif
		endif
	while(!success)
	killpath /Z TempInstallPath
	return S_path
end	

// -------------- A GUI for installing and updating user projects ------------

function makeInstallerPanel()
	
	DoWindow /k InstallerPanel
	
	// create package folder and waves
	DFREF dfr = setupPackageFolder()
	wave /T/SDFR=dfr/Z ProjectsFullList, ProjectsDisplayList, ProjectsHelpList, ProjectsColTitles
	wave /T/SDFR=dfr/Z UpdatesFullList, UpdatesDisplayList, UpdatesHelpList, UpdatesColTitles
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	// panel height is 338 points
	// if panelresolution is set to 96, height will be
	// 338*96/screenresolution
	variable BiggerPanel=prefs.paneloptions&2
	variable sHeight=getScreenHeight(), sMinHeight=700	
	if(BiggerPanel && sHeight<sMinHeight)
		BiggerPanel=0
		prefs.paneloptions-=2
		SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
	endif
		
	variable folder=1
//	NVAR/Z V_Pop=dfr:V_Pop // folder popup selection is retained in experiment
//	folder = NVAR_Exists(V_Pop) ? V_Pop : 1
		
	// make a control panel GUI
	variable pLeft=prefs.win.left, pTop=prefs.win.top
	variable pWidth=520, pHeight=338
		
	if( BiggerPanel ) 
		execute/Q/Z "SetIgorOption PanelResolution=?"
		NVAR vf=V_Flag
		variable oldResolution = vf
		execute/Q/Z "SetIgorOption PanelResolution=96"
	endif
	
	NewPanel /K=1 /W=(pLeft,pTop,pLeft+pWidth,pTop+pHeight)/N=InstallerPanel as "IgorExchange Projects"
	ModifyPanel /W=InstallerPanel, noEdit=1
	
	variable vTop=15,tabSelection=prefs.paneloptions&1
	
	// insert a notebook subwindow to be used for filtering lists
	DefineGuide/W=InstallerPanel UGV0={FR,-28}
	NewNotebook /F=1 /N=nb0 /HOST=InstallerPanel /W=(300,vTop,5200,vTop+25)/FG=($"",$"",UGV0,$"") /OPTS=3 
	Notebook InstallerPanel#nb0 fSize=12+6*BiggerPanel, showRuler=0
	Notebook InstallerPanel#nb0 spacing={4+2*BiggerPanel, 0, 5}
	Notebook InstallerPanel#nb0 margins={0,0,1000}
	SetWindow InstallerPanel#nb0, activeChildFrame=0
	fClearText(1) // sets notebook to its default appearance
	
	// make a Button for clearing text in notebook subwindow
	Button ButtonClear, win=InstallerPanel,pos={497,vTop+4},size={15,15},title=""
	Button ButtonClear, picture=Updater#fClearTextPic,proc=Updater#InstallerButtonProc, disable=1
	
	vTop+=35	
	TabControl tabs,win=InstallerPanel,pos={0,vTop},size={520,260},tabLabel(0)="Projects"
	TabControl tabs,win=InstallerPanel,tabLabel(1)="Updates", value=tabSelection, proc=updater#InstallerTabProc
	
	vTop+=25
	// Listbox for install tab
	ListBox listboxInstall, win=InstallerPanel, pos={0,vTop},size={520, 200}, listwave=ProjectsDisplayList, fsize=10+2*BiggerPanel
	ListBox listboxInstall, win=InstallerPanel, mode=1, proc=Updater#InstallerListBoxProc, selRow=-1, userColumnResize=1
	ListBox listboxInstall, win=InstallerPanel, helpWave=ProjectsHelpList, titlewave=ProjectsColTitles, disable=(tabSelection==1)
	ListBox listboxInstall, win=InstallerPanel, widths={8,3,2,1}, userdata(sortcolumn)="-3" // columns numbered from 1 
	
	// Listbox for update tab
	ListBox listboxUpdate, win=InstallerPanel, pos={0,vTop},size={520, 200}, listwave=UpdatesDisplayList, fsize=10+2*BiggerPanel
	ListBox listboxUpdate, win=InstallerPanel, mode=1, proc=Updater#InstallerListBoxProc, selRow=-1, userColumnResize=1
	ListBox listboxUpdate, win=InstallerPanel, helpWave=UpdatesHelpList, titlewave=UpdatesColTitles, disable=(tabSelection==0)
	ListBox listboxUpdate, win=InstallerPanel, widths={22,16,6,6}, userdata(sortcolumn)="-2" // columns numbered from 1
	
	vTop+=208
	// popup for install tab	
	PopupMenu popupType, win=InstallerPanel, pos={10,vTop}, value="all;", mode=1, title="Type", proc=Updater#InstallerPopupProc
	PopupMenu popupType, win=InstallerPanel, disable=(tabSelection==1)
		
	// popup for update tab
	PopupMenu popupFolder, win=InstallerPanel, pos={10,vTop}, value="Installed Projects;User Procedures Folder;Current Experiment;", mode=folder, title="Check files in ", proc=Updater#InstallerPopupProc
	PopupMenu popupFolder, win=InstallerPanel, disable=(tabSelection==0)
		
	Button btnInstallOrUpdate, win=InstallerPanel,pos={450,vTop-2},size={60,25},proc=Updater#InstallerButtonProc,title="Install"
	Button btnInstallOrUpdate, win=InstallerPanel,help={"Install selected user project"}, disable=2
		
	vTop+=35	
	TitleBox statusBox, win=InstallerPanel, pos={10,vTop}, frame=0, title=""
	Button btnSettings, win=InstallerPanel,pos={493,vTop-2},size={15,15}, picture=Updater#cog, labelBack=0, title=""
	Button btnSettings, win=InstallerPanel,proc=Updater#InstallerButtonProc, help={"Change Settings"}
		
	DoUpdate /W=InstallerPanel	
	SetWindow InstallerPanel hook(hInstallerHook)=updater#fHook	

	if( kResizablePanel )
		// resizing userdata for controls
		Button ButtonClear, win=InstallerPanel,userdata(ResizeControlsInfo)= A"!!,I^J,hm&!!#<(!!#<(z!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
		Button ButtonClear, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#r+D.Oh\\ASGdjF8u:@zzzzzzzzz"
		Button ButtonClear, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzz!!#r+D.Oh\\ASGdjF8u:@zzzzzzzzzzzz!!!"
		
		TabControl tabs, win=InstallerPanel,userdata(ResizeControlsInfo)= A"!!*'\"!!#>V!!#Cg!!#B<z!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAeBE/#4z"
		TabControl tabs, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du_\"OASGdjF8u:@zzzzzzzzz"
		TabControl tabs, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<!-A3SF8u:@zzzzzzzzzzzz!!!"
		
		ListBox listboxInstall, win=InstallerPanel,userdata(ResizeControlsInfo)= A"!!*'\"!!#?O!!#Cg!!#AWz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
		ListBox listboxInstall, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du_\"OASGdjF8u:@zzzzzzzzz"
		ListBox listboxInstall, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<!-A2@zzzzzzzzzzzzz!!!"
		
		ListBox listboxUpdate, win=InstallerPanel,userdata(ResizeControlsInfo)= A"!!*'\"!!#?O!!#Cg!!#AWz!!#](Aon\"Qzzzzzzzzzzzzzz!!#o2B4uAezz"
		ListBox listboxUpdate, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du_\"OASGdjF8u:@zzzzzzzzz"
		ListBox listboxUpdate, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<!-A2@zzzzzzzzzzzzz!!!"

		PopupMenu popupType, win=InstallerPanel,userdata(ResizeControlsInfo)= A"!!,A.!!#BGJ,hol!!#<pz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
		PopupMenu popupType, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		PopupMenu popupType, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
			
		PopupMenu popupFolder, win=InstallerPanel,userdata(ResizeControlsInfo)= A"!!,A.!!#BGJ,ho,!!#<Xz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
		PopupMenu popupFolder, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		PopupMenu popupFolder, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		
		Button btnInstallOrUpdate, win=InstallerPanel,userdata(ResizeControlsInfo)= A"!!,IG!!#BFJ,hoT!!#=+z!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
		Button btnInstallOrUpdate, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<!,6(ZF8u:@zzzzzzzzz"
		Button btnInstallOrUpdate, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
		
		TitleBox statusBox, win=InstallerPanel,userdata(ResizeControlsInfo)= A"!!,A.!!#BY!!#@&!!#;=z!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
		TitleBox statusBox, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		TitleBox statusBox, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
			
		Button btnSettings, win=InstallerPanel,userdata(ResizeControlsInfo)= A"!!,I\\J,hs.!!#<(!!#<(z!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
		Button btnSettings, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
		Button btnSettings, win=InstallerPanel,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
			
		// resizing userdata for panel
		SetWindow InstallerPanel,userdata(ResizeControlsInfo)= A"!!*'\"z!!#Cg!!#Bczzzzzzzzzzzzzzzzzzzzz"
		SetWindow InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzzzzzzzz"
		SetWindow InstallerPanel,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzz!!!"
		
		SetWindow InstallerPanel,userdata(ResizeControlsGuides)=  "UGV0;"
		SetWindow InstallerPanel,userdata(ResizeControlsInfoUGV0)=  "NAME:UGV0;WIN:InstallerPanel;TYPE:User;HORIZONTAL:0;POSITION:492.00;GUIDE1:FR;GUIDE2:;RELPOSITION:-28;"
			
		// resizing panel hook
		SetWindow InstallerPanel hook(ResizeControls)=ResizeControls#ResizeControlsHook
	endif
	
	if(tabSelection==0)
		reloadProjectsList() // populate ProjectsFullList
		if(dimsize(ProjectsFullList,0)==0)
			ProjectsDisplayList={{"Download failed."},{""},{""},{""}}
		else			
			Duplicate /free/T/RMD=[][5,5] ProjectsFullList, types
			types=replacestring(",",types,";")
			string typeList = "\"all;"+listOf(types)+"\""
			PopupMenu popupType, win=InstallerPanel, value=#typeList, mode=1
		endif
	endif
	
	if(tabSelection==1)
		reloadUpdatesList(1, 1) // populate UpdatesFullList
		if(dimsize(UpdatesFullList,0)==0)
			UpdatesDisplayList={{"Download failed."},{""},{""},{""}}
		endif
	endif
	
	updateListboxWave("")
	
	if( BiggerPanel ) 
		// reset panel resolution
		execute/Q/Z "SetIgorOption PanelResolution="+num2istr(oldResolution)
	endif	
end

// get a list of user contributed projects from Wavemetrics.com
// if we already have an up to date list loaded in ProjectsFullList, do nothing
// load projects from cache into ProjectsFullList
// check for more recent releases and add to ProjectsFullList
// update cache
function reloadProjectsList()

	DFREF dfr = root:Packages:Installer
	wave /T/SDFR=dfr ProjectsFullList, ProjectsDisplayList
	
	variable lastmod=NumberByKey("MODTIME", waveInfo(ProjectsFullList,0))
	variable oneDay=86400, oneWeek=86400*7, startPage=0, pnt=0, pnt2=0
	
	if(((datetime-lastmod) < oneDay) && (dimsize(ProjectsFullList, 0)>=kNumProjects)) // less than 1 day old
		// looks like we already have an up-to-date list of projects
		return 0
	endif
	
	// clear any incomplete or old list
	redimension /N=(0,-1) ProjectsFullList
	
	// load projects from cache
	string filePath=getPathToCache()
	make /T/N=0/free w
	if(isFile(filePath))
		Grep /O/Z/E=";" FilePath as w // load cache
	endif
	
	variable i
	variable numProjects=dimsize(w, 0)
	
	// check that cached list is reasonably recent	
	make/free/N=(numProjects) cacheTime
	if(numProjects)
		cacheTime=str2num(StringFromList(1, w[p]))
	endif
	
	if(numProjects>=kNumProjects && wavemin(cacheTime)>(datetime-oneWeek) ) // we have a fairly complete list
		Redimension /N=(numProjects, -1) ProjectsFullList
		ProjectsFullList[][0,6]=StringFromList(q+(q>0), w[p]) // skip second item in list (cache time)
	else
		numProjects=0 // start over by downloading complete list
	endif
	
	// now grab projects more recent than those in cache	
	// keep a list of new downloads to add to cache

//	duplicate /T/free ProjectsFullList toCache
//	redimension /N=(0,-1) toCache
	// be careful - duplicate may lose column dim labels if wave has zero rows!
	make /T/free/N=(0, 10) toCache
	setDimLabels("projectID;name;author;published;views;type;userNum;;;;", 1, toCache)
		
	string baseURL, projectsURL, URL //, listPageText, projectPageText
	baseURL="https://www.wavemetrics.com"
	projectsURL="/projects?os_compatibility=All&project_type=All&field_supported_version_target_id=All&page="
	variable pageNum, projectNum, success
	
	variable pStart, pEnd, selStart, selEnd=0, err
	string strAuthor, strName, ShortTitle, strProjectURL, strUserNum, strProjectNum
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	variable done=0
	
	// loop through listPages
	for(pageNum=0;pageNum<100;pageNum+=1)
		
		setPanelStatus("Downloading page "+num2str(pageNum+1)+"...")
		sprintf URL "%s%s%d", baseURL, projectsURL, pageNum
		
		URLRequest /TIME=(prefs.pageTimeout)/Z url=url
		if(V_flag)
			printf "Installer could not load %s\r", url		
			UpdateListboxWave("") // update based on anything we managed to download
			setPanelStatus("List may be incomplete")
			return 0		
		endif
			
		pStart=strsearch(S_serverResponse, "<section class=\"project-teaser-wrapper\">", 0, 2)
		if (pStart==-1)
			break // no more projects
		endif
		pEnd=0
		
		// loop through projects on listPage
		for (projectNum=0;projectNum<50; projectNum+=1)
			pStart=strsearch(S_serverResponse, "<section class=\"project-teaser-wrapper\">", pEnd, 2)	
			pEnd=strsearch(S_serverResponse, "<div class=\"project-teaser-footer\">", pStart, 2)
			if (pEnd==-1 || pStart==-1)
				break // no more projects on this listPage
			endif
			
			selStart=strsearch(S_serverResponse, "<a class=\"user-profile-compact-wrapper\" href=\"/user/", pEnd, 3)
			if(selStart<pStart)
				continue
			endif
			
			selStart+=52
			selEnd=strsearch(S_serverResponse, "\">", selStart, 0)
			strUserNum=S_serverResponse[selStart,selEnd-1]
			
			selStart=strsearch(S_serverResponse, "<span class=\"username-wrapper\">", selEnd, 2)
			selStart+=31
			selEnd=strsearch(S_serverResponse, "</span>", selStart, 2)		
			strAuthor=S_serverResponse[selStart,selEnd-1]
					
			selStart=strsearch(S_serverResponse, "<a href=\"", selEnd, 2)
			selStart+=9
			selEnd=strsearch(S_serverResponse, "\"><h2>", selStart, 2)
			strProjectURL=baseURL+S_serverResponse[selStart,selEnd-1]
			
			selStart=selEnd+6
			selEnd=strsearch(S_serverResponse, "</h2></a>", selStart, 2)
			strName=S_serverResponse[selStart,selEnd-1]
			
			if (strLen(strName)==0)
				continue
			endif
			
			// clean up project names that contain certain encoded characters
			strName=replaceString("&#039;",strName, "'")
			strName=replaceString("&amp;",strName, "&")


// Don't try to get project details at this point: too many pages to download			
			
//			projectPageText=FetchURL(strProjectURL); err = GetRTError(1)
//			if (err != 0)
//				continue
//			endif
//			selStart=strsearch(projectPageText, "<link rel=\"canonical\" href=\"", 0, 0)
//			selStart+=28
//			selEnd=strsearch(projectPageText, "\"", selStart, 0)
//			strProjectURL=projectPageText[selStart,selEnd-1]
//			ShortTitle=ParseFilePath(0, strProjectURL, "/", 1, 0)
//			
//			success=(strlen(ShortTitle)&&strlen(strProjectURL))
//			if(!success)
//				continue
//			endif
			
			string projectID=ParseFilePath(0, strProjectURL, "/", 1, 0)
			findvalue /TEXT=projectID/TXOP=2/RMD=[][0,0] ProjectsFullList
			if(V_Value>0) // we already have this one in cache
				done=1
				break
			endif
			
			InsertPoints /M=0 DimSize(ProjectsFullList, 0), 1, ProjectsFullList
			ProjectsFullList[inf][%projectID]=ParseFilePath(0, strProjectURL, "/", 1, 0)
			ProjectsFullList[inf][%name]=strName
			ProjectsFullList[inf][%author]=strAuthor
			ProjectsFullList[inf][%userNum]=strUserNum			
			// search for other non-essential parameters
			
			// project types
			selEnd=pStart
			do
				selStart=strsearch(S_serverResponse, "/taxonomy/", selEnd, 2)
				selStart=strsearch(S_serverResponse, ">", selStart, 0)
				selEnd=strsearch(S_serverResponse, "<", selStart, 0)
				if(selStart<pStart || selEnd>pEnd || selEnd<1 )
					break
				endif
				ProjectsFullList[inf][%type]+=S_serverResponse[selStart+1,selEnd-1]+";"
			while (1)
			
			// date	
			selStart=strsearch(S_serverResponse, "<span>", pEnd, 2)
			selEnd=strsearch(S_serverResponse, "</span>", pEnd, 2)
			if (selStart>0 && selEnd>0 && selEnd < (pEnd+80) )
				ProjectsFullList[inf][%published]=ParsePublishDate(S_serverResponse[selStart+6,selEnd-1])
			endif
			
			// views
			selEnd=strsearch(S_serverResponse, " views</span>", pEnd, 2)
			selStart=strsearch(S_serverResponse, "<span>", selEnd, 3)		
			if (selStart>pEnd && selEnd < (pEnd+150) )
				ProjectsFullList[inf][%views]=S_serverResponse[selStart+6,selEnd-1]
			endif
			
			pnt=DimSize(toCache, 0)
			pnt2=DimSize(ProjectsFullList, 0)-1	
			InsertPoints /M=0 pnt, 1, toCache
			toCache[pnt][]=ProjectsFullList[pnt2][q]
			
		endfor	 // next project
		
		resortWaves(0, 0)
		UpdateListboxWave("") // rebuild display list from time to time
		DoUpdate
		if(done)
			break
		endif
	endfor	 // next page
	
	// cache any newly added projects
	if(projectNum>0) // we have added new projects	
		CachePutProjectsWave(toCache)
	endif
	
	setPanelStatus("")
	return 1
end

function reloadUpdatesList(forced, gui)
	variable forced
	variable gui // 1 for installer panel, 2 for UpdateCheckProgressPanel
	
	// get project name, update status, local and remote versions, release
	// URL, OS compatibility and release date for each file with 
	// compatible update header
	DFREF dfr = root:Packages:Installer
	wave /T UpdatesFullList=dfr:UpdatesFullList
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	variable lastmod=NumberByKey("MODTIME", waveInfo(UpdatesFullList,0))
	variable oneDay=86400, oneWeek=604800
	variable checkFilesInUserProcsFolder=0, checkFilesInExperiment=0, checkInstalled=1
	
	if(gui==1)
		ControlInfo /W=InstallerPanel popupFolder
		checkInstalled=(V_Value==1)
		checkFilesInUserProcsFolder=(V_Value==2)
		checkFilesInExperiment=(V_Value==3)
	endif
	
	if(forced==0 && ((datetime-lastmod) < oneDay) && (dimsize(UpdatesFullList, 0)>0) && (dimsize(UpdatesFullList, 1)==15)) // less than 1 day old
		// looks like we already have a recent list of projects
		// update local info only, we only arrive here if installer panel is open
		
		if(checkInstalled)
			UpdatesFullList[][%local]=LogGetVersion(UpdatesFullList[p][%projectID])
			UpdatesFullList[][%status]=getUpdateStatus(UpdatesFullList[p][%projectID], str2num(UpdatesFullList[p][%local]), str2num(UpdatesFullList[p][%remote]))			
		else 	
			UpdatesFullList[][%local]=getPragmaString("version", UpdatesFullList[p][%installPath])
			// maybe a file has been updated
			UpdatesFullList[][%status]=selectstring(str2num(UpdatesFullList[p][%local])>=str2num(UpdatesFullList[p][%remote]), UpdatesFullList[p][%status], "up to date")
			resortWaves(1, 0)
		endif
		
		return 0
	endif
	
	// clear any incomplete or old list
	redimension /n=(0,-1) UpdatesFullList
		
	string procList="", cmd="", strExitStatus=""
	variable i,j
	string filePath, shortTitle, url, releaseURL, projectName, strDate, fileStatus
	string strRemVer, strLocVer, strSystem="", info=""
	string installDate=""
	string projectID, keyList
	variable CacheDate, localVersion, remoteVersion, releaseVersion, releaseMajor, releaseMinor, releaseIgorVersion
	variable currentIgorVersion=getIgorVersion() 
	string logEntry=""//, strCache=""
	
	if(checkInstalled)
		wave/T w=ListToTextWave(LogProjectsList(),";")
	else
		if(checkFilesInUserProcsFolder)
			wave/T w=getProcsRecursive(1)
		elseif(checkFilesInExperiment) // check only open files
			wave/T w=ListToTextWave(winlist("*",";","INDEPENDENTMODULE:1,INCLUDE:3"),";")
			w=getFilePathFromProcWin(w)
			TextWaveZapString(w, "")
		endif
		// remove this procedure from the list
		TextWaveZapString(w, FunctionPath(""))
		
		// add this file to start of list
		insertpoints 0, 1, w
		w[0]=FunctionPath("")	
	endif
	
	variable numProjects=dimsize(w,0)
	for(i=0;i<numProjects;i+=1)
		
		// first check status of local file
		
		if(checkInstalled)
			logEntry=LogGetProject(w[i])
			
			// fill UpdatesFullList
			projectID=StringFromList(0, logEntry)
			shortTitle=StringFromList(1, logEntry)
			strLocVer=StringFromList(2, logEntry)
			installDate=StringFromList(3, logEntry)		
			filePath=StringFromList(4, logEntry)
			if(strlen(filePath))
				filePath=SpecialDirPath(filePath,0,0,0)
			endif
			filePath+=StringFromList(5, logEntry)// path to install location	
			fileStatus=getInstallStatus(projectID)
			url="https://www.wavemetrics.com/project-releases/"+projectID
			localVersion=str2num(strLocVer)
			info=getInfo(projectID)
		else
			filePath=w[i]
		
			GetFileFolderInfo /Q/Z filePath
			if (V_isFile==0)
				continue
			endif
			
			// skip any file that doesn't have kProjectID set
			// we don't try to guess URL because download failures will slow the indexing too much
			url=getUpdateURLfromFile(filePath)
			if(strlen(url)==0)
				continue
			endif	
			projectID=ParseFilePath(0, url, "/", 1, 0)
			
			// find project short title
			shortTitle=getShortTitle(filePath)
			if(strlen(shortTitle)==0) // we found the URL, but not the short title
				shortTitle=ParseFilePath(3, filePath, ":", 0, 0) // use filename
			endif
			
			// version number of local file	
			localVersion=getProcVersion(filePath)
			if (localVersion==0)
				continue
			endif
			strLocVer=getPragmaString("version", filePath)
		endif
		
		// now check release version	
		// if release info was recently cached, use that
		keyList=cacheGetKeylist(projectID)
		cacheDate=str2num(StringByKey("ReleaseCacheDate", keyList))
		cacheDate = numtype(cacheDate)==0 ? cacheDate : 0
		if( cacheDate < (datetime-oneDay) )
			keyList=""
			// download the "all releases" page
			if(gui==1)
				setPanelStatus("Downloading "+url); DoUpdate /W=InstallerPanel
			elseif(gui==2)
				TitleBox statusBox, win=UpdateCheckProgressPanel, title="Downloading "+url
			endif
				
			// download all releases page, parse contents and format as keyList
			keyList=getLatestReleaseInfoFromWeb(projectID, prefs.pagetimeout)
			
			if(strlen(keyList)==0) // download failed
				strExitStatus+=shortTitle+","
				continue
			endif
				
			// store retrieved values in cache
			keyList=ReplaceStringByKey("ReleaseCacheDate", keyList, num2istr(datetime))
			CachePutKeylist(keyList)
		endif
			
		releaseVersion=str2num(StringByKey("remote", keyList))

		if(checkInstalled)	// checking projects in install log
			if(releaseVersion>localVersion)
				if(releaseIgorVersion<=currentIgorVersion && stringmatch(fileStatus, "complete"))
					fileStatus+=", update available" // don't allow updates for incomplete or missing projects
				elseif(stringmatch(fileStatus, "complete"))
					fileStatus+=", update incompatible"
				endif
			elseif(stringmatch(fileStatus, "complete"))
				fileStatus+=", up to date"
			endif
		else // checking based on a filepath
			if(localVersion>=releaseVersion)
				fileStatus="up to date"
			elseif(releaseIgorVersion<=currentIgorVersion)
				fileStatus="update available"
			else
				fileStatus="update incompatible"
			endif
		endif

		j=dimsize(UpdatesFullList,0)
		insertpoints /M=0 j, 1, UpdatesFullList
		UpdatesFullList[j][%projectID]=projectID
		UpdatesFullList[j][%name]=shortTitle
		UpdatesFullList[j][%status]=fileStatus
		UpdatesFullList[j][%local]=strLocVer
		UpdatesFullList[j][%remote]=StringByKey("remote", keyList)
		UpdatesFullList[j][%system]=StringByKey("system", keyList)
		UpdatesFullList[j][%releaseDate]=StringByKey("releaseDate", keyList)
		UpdatesFullList[j][%installPath]=filePath // full path to install folder or to ipf
		UpdatesFullList[j][%releaseURL]=StringByKey("releaseURL", keyList)
		UpdatesFullList[j][%releaseIgorVersion]=""
		UpdatesFullList[j][%installDate]=installDate
		UpdatesFullList[j][%info]=info
		
		if(gui==1)
			resortWaves(1, 0) // match current sort order for gui
			UpdateListboxWave(fGetStub())
		elseif(gui==2) // deprecated
			DoUpdate /W=UpdateCheckProgressPanel /E=1
			if( V_Flag == 2 )	// user cancel
				return 0 // stop trying to reload updates list in background
			endif
		endif		
	endfor
		
	if(gui==1)
		strExitStatus=selectstring(strlen(strExitStatus)>0,"","D/L failed for "+removeending(strExitStatus, ","))
		setPanelStatus(strExitStatus)
	endif
	return 1	
end

// returns keylist of release info
threadsafe function /S getLatestReleaseInfoFromWeb(projectID, timeout)
	string projectID
	variable timeout
	
	string keyList="" // be careful to avoid key and list separators in list items
	string url=""
	sprintf url, "https://www.wavemetrics.com/project-releases/%s", projectID
	
	URLRequest /TIME=(timeout)/Z url=url
	if(v_flag)
		return ""
	endif
	
	// extract version info from webpage
	wave /T w_releases=ParseReleases(S_serverResponse)
	if(dimsize(w_releases, 0)==0)
		return ""
	endif

	string strSystem="", releaseURL="",releaseExtra="",strRemVer="", strReleaseDate=""
	variable releaseIgorVersion, releaseVersion
	variable j=0 // inspect most recent release
	
	keyList=ReplaceStringByKey("projectID", keyList, projectID)
	keyList=ReplaceStringByKey("name", keyList, w_releases[j][0]) 
	sscanf (lowerStr(w_releases[j][1])), "igor.%f.x-%f", releaseIgorVersion, releaseVersion
	if(V_flag!=2) // version string doesn't have strict formatting from old IgorExchange site
		releaseIgorVersion=0 // no way to figure out required Igor version prior to download
	endif
	keyList=ReplaceStringByKey("releaseIgorVersion", keyList, num2str(releaseIgorVersion)) 
	releaseVersion=str2num(w_releases[j][2]+"."+w_releases[j][3])
	releaseExtra= selectstring( strlen(w_releases[j][7])>0 , "", "-"+w_releases[j][7])
	sprintf strRemVer, "%g%s", releaseVersion, releaseExtra
	strRemVer=replacestring(":", strRemVer, "-")
	keyList=ReplaceStringByKey("remote", keyList, strRemVer) 	
	keyList=ReplaceStringByKey("releaseURL", keyList, w_releases[j][4]) 
	strSystem=replacestring(";", w_releases[j][5], ",")
	keyList=ReplaceStringByKey("system", keyList, strSystem)
	strReleaseDate=ParseReleaseDate(w_releases[j][6])
	keyList=ReplaceStringByKey("releaseDate", keyList, strReleaseDate) 
	
	return keyList	
end

function resortWaves(tabNum, sortCol)
	variable tabNum , sortCol
	// tabNum 0 is install list, 1 is updates list
	// column: O resort according to user data
	// non-zero: sort by column, reversing sort if column matches previous
	
	string lb=SelectString(tabNum, "listboxInstall", "listboxUpdate")
	DFREF dfr = root:Packages:Installer
	if(tabNum==0)
		wave /T FullList=dfr:ProjectsFullList
	else
		wave /T FullList=dfr:UpdatesFullList
	endif
	
	variable oldSortCol=str2num(GetUserData("InstallerPanel", lb, "sortColumn"))
	
	if(sortCol==0)
		sortCol=oldSortCol
	elseif(sortCol==abs(oldSortCol))
		sortCol=-oldSortCol
	endif
	
	ListBox $lb win=InstallerPanel, userdata(sortcolumn)=num2str(sortCol)
	// use second and third sort columns to prevent weird resorting behaviour
	// when user re-clicks active tab.
	variable first, second, third
	first=abs(sortCol)
	second=1+(first<2)
	third=2+(first<3)
	if(sortCol<0) // reverse sort
		SortColumns /A/KNDX={first,second,third}/R sortwaves={FullList}
	else
		SortColumns /A/KNDX={first,second,third} sortwaves={FullList}
	endif
	updateListboxWave(fGetStub())
	return 1
end

// returns alphabetical list of textWave contents
function /S listOf(textWave, [separator])
	wave /T textWave
	string separator
	
	separator=SelectString(ParamIsDefault(separator), separator, ";")
	
	string outputList="", strItem
	variable i, j, items
	for (i=0;i<DimSize(textWave,0);i+=1)
		items=ItemsInList(textWave[i][0], separator)
		for(j=0;j<items;j+=1)
			strItem=StringFromList(j, textWave[i][0], separator)
			if (WhichListItem(strItem,outputList, separator)==-1)
				outputList=AddListItem(strItem, outputList, separator)
			endif
		endfor
	endfor
	
	return SortList(outputList, separator)
end

threadsafe function /S ParseReleaseDate(str)
	string str
	
	string strDay, strAMPM
	variable year, month, day, HH, MM
	sscanf str, "%s %g/%g/%d - %d:%d %s", strDay, month, Day, Year, HH, MM, strAMPM
	
	return num2istr(DateToJulian(year, month, day))
end

// returns Julian date as string
threadsafe function /S ParsePublishDate(str)
	string str
	
	string months="January;February;March;April;May;June;July;August;September;October;November;December"
	string strMon, strDay, strYear
	SplitString/E="([[:alpha:]]+) ([[:digit:]]+), ([[:digit:]]+)" str, strMon, strDay, strYear
	variable year, month, day
	month=WhichListItem(strMon, months)+1
	day=str2num(strDay)
	year=str2num(strYear)
	
	return num2istr(DateToJulian(year, month, day))
end

function InstallerPopupProc(s)
	STRUCT WMpopupAction &s
	
	if(s.eventCode==-1) // save some prefs before killing control panel
		PrefsSaveWindowPosition(s.win)
		KillDataFolder /Z root:packages:installer
		return 0
	endif
	
	if(s.eventCode!=2)
		return 0
	endif
	
	if(stringmatch(s.ctrlName, "popupFolder"))
		reloadUpdatesList(1, 1)
	else
		UpdateListboxWave(fGetStub())
	endif
	
	// send a keyboard event to filter hook
	// this updates text completion based on new popup selection
	STRUCT WMWinHookStruct hookstruct
	hookstruct.eventcode=11
	fHook(hookstruct)
	
	return 0
end

function InstallerTabProc(s)
	STRUCT WMTabControlAction& s
	
	if(s.eventCode!=2)
		return 0
	endif
	variable UpdatesTab=(s.tab==1), ProjectsTab=(s.tab==0)
	
	// controls in projects tab
	ListBox listboxInstall, win=InstallerPanel, disable=UpdatesTab
	PopupMenu popupType, win=InstallerPanel, disable=UpdatesTab
	
	// controls in updates tab
	ListBox listboxUpdate, win=InstallerPanel, disable=ProjectsTab
	PopupMenu popupFolder, win=InstallerPanel, disable=ProjectsTab
	
	// change Button title
	Button btnInstallOrUpdate, win=InstallerPanel, title= SelectString (projectsTab, "Update","Install")
		
	if(projectsTab)
		reloadProjectsList() // populate ProjectsFullList
		wave /T/SDFR=root:Packages:Installer ProjectsFullList, ProjectsDisplayList
		if(DimSize(ProjectsFullList,0)==0)
			ProjectsDisplayList={{"Download failed."},{""},{""}}
		else
			Duplicate /free/T /RMD=[][5,5] ProjectsFullList, types
			types=ReplaceString(",",types,";")
			string typeList = "\"all;"+listOf(types)+"\""
			PopupMenu popupType, win=InstallerPanel, value=#typeList, mode=1
		endif
	endif
	
	if(updatesTab)
		wave /T/SDFR=root:Packages:Installer UpdatesFullList, UpdatesDisplayList
		reloadUpdatesList(0, 1)
		if(DimSize(UpdatesFullList,0)==0)
			UpdatesDisplayList={{"Download failed."},{""},{""},{""}}
		endif
	endif
	
//	fClearText(1)
//	UpdateListboxWave("")
	UpdateListboxWave(fGetStub())
	
	// send a keyboard event to filter hook
	// this updates text completion based on new popup selection
	STRUCT WMWinHookStruct hookstruct
	hookstruct.eventcode=11
	fHook(hookstruct)
		
	if(projectsTab)
		ControlInfo/W=InstallerPanel listboxInstall
	else
		ControlInfo/W=InstallerPanel listboxUpdate
	endif
	wave /T listWave=$(S_DataFolder+s_value)
	if(V_value<DimSize(listwave,0) && V_value>-1)
		setPanelStatus("Selected: "+listWave[V_value][0])
		if(projectsTab || stringmatch(listWave[V_value][1], "*update available"))
			Button btnInstallOrUpdate, win=InstallerPanel, disable=0
		else
			Button btnInstallOrUpdate, win=InstallerPanel, disable=2
		endif
	else
//		setPanelStatus("")
		Button btnInstallOrUpdate, win=InstallerPanel, disable=2
	endif
	
	return 0
end

function InstallerButtonProc(s)
	STRUCT WMButtonAction &s
	
	if(s.eventCode!=2)
		return 0
	endif
	
	strswitch(s.ctrlName)
		case "btnInstallOrUpdate" :
			ControlInfo /W=InstallerPanel tabs
			if(v_value==0) // install something
				InstallSelection()
			else // update a project
				UpdateSelection()
			endif
			break
		case "ButtonClear" :
			fClearText(1)
			NVAR stubLen=root:Packages:Installer:stubLen
			stubLen=0
			Button ButtonClear, win=InstallerPanel, disable=3
			UpdateListboxWave("")
			break
		case "btnSettings" :
			makePrefsPanel()
	endswitch

	return 0
end

function InstallerListBoxProc(s)
	STRUCT WMListboxAction &s
	
	if(s.eventCode==-1)
		return 0
	endif
	
	DFREF dfr = root:Packages:Installer
	if (stringmatch(s.ctrlName, "ListBoxInstall"))
		wave /T matchlist=dfr:ProjectsMatchList
		wave /T fullList=dfr:ProjectsFullList
	else
		wave /T matchlist=dfr:UpdatesMatchList
		wave /T fullList=dfr:UpdatesFullList
	endif
	string status=""
	
	switch (s.eventCode)
		case 2: // mouseup
			
			if(s.row==-1 && s.eventmod!=1) // click in column heading - resort listbox waves
				resortWaves(stringmatch(s.ctrlName,"ListBoxUpdate"), s.col+1)
			endif
			
			// set status and enable Button based on selection
			ControlInfo/W=$(s.win) $(s.ctrlName)
			if(V_value<DimSize(s.listwave,0) && V_value>-1)
				setPanelStatus("Selected: "+s.listWave[V_value][0])
				if(stringmatch(s.ctrlName, "ListBoxInstall"))
					Button btnInstallOrUpdate, win=InstallerPanel, disable=0
				else
					Button btnInstallOrUpdate, win=InstallerPanel, disable=2-2*(stringmatch(s.listWave[v_value][1],"*update available"))
				endif
				return 0
			endif
			sprintf status "Showing %d of %d projects", DimSize(s.listWave, 0), DimSize(FullList, 0)
			setPanelStatus(status)
			Button btnInstallOrUpdate, win=InstallerPanel, disable=2
			break
			
		case 4: // Cell selection (mouse or arrow keys)
		case 5: // Cell selection + shift
			
			if(s.eventMod==17) // right click?
				if(s.row>=DimSize(s.listwave,0) || s.row<0)
					return 0
				endif
				if(stringmatch(s.ctrlName, "listboxUpdate"))
					string projectID=matchlist[s.row][%projectID]
					if(stringmatch(s.listWave[s.row][1], "missing"))
						PopupContextualMenu "Remove Missing Project from List;Locate Missing Project;"
						if(v_flag==1)
							LogRemoveProject(projectID)
							ReloadUpdateslist(1, 1)
							UpdateListboxWave(fGetStub())
						elseif(v_flag==2)
							NewPath /M="Reset install path" /O/Q/Z TempInstallPath
							if(v_flag)
								return 0
							endif
							PathInfo /S TempInstallPath
							LogUpdateInstallPath(projectID, S_path)
							if(stringmatch(getInstallStatus(projectID), "missing"))
								LogUpdateInstallPath(projectID, ParseFilePath(1, S_path, ":", 1, 0))
							endif
							ReloadUpdateslist(1, 1)
							UpdateListboxWave(fGetStub())
						endif
					else
						PopupContextualMenu "Show Files;Uninstall "+matchlist[s.row][%name]+";"
						if(v_flag==1)
							NewPath /O/Q/Z TempInstallPath matchlist[s.row][%installPath]
							PathInfo /SHOW TempInstallPath
						elseif(v_flag==2)
							Print "Uninstalling "+matchlist[s.row][%name]
							uninstallProject(matchlist[s.row][%projectID])
							ReloadUpdateslist(1, 1)
							UpdateListboxWave(fGetStub())
						endif
					endif
				endif
				KillPath /Z TempInstallPath
				return 0
			endif
			
			if(s.row<DimSize(s.listwave,0) && s.row>-1)
				setPanelStatus("Selected: "+s.listWave[s.row][0])
				if(stringmatch(s.ctrlName, "ListBoxInstall"))
					Button btnInstallOrUpdate, win=InstallerPanel, disable=0
				else
					Button btnInstallOrUpdate, win=InstallerPanel, disable=2-2*(cmpstr(s.listWave[s.row][1],"update available")==0)
				endif
				return 0
			endif
			sprintf status "Showing %d of %d projects", DimSize(s.listWave, 0), DimSize(FullList, 0)
			setPanelStatus(status)
			break
			
		case 3: // double-click

			if(s.row>=DimSize(s.listwave,0) || s.row<0)
				return 0
			endif
			string url
			if (stringmatch(s.ctrlName, "ListBoxInstall"))
				if(s.col==1)
					url="https://www.wavemetrics.com/user/"+matchlist[s.row][%userNum]
				else
					url="https://www.wavemetrics.com/node/"+matchlist[s.row][%projectID]
				endif
			else
				url="https://www.wavemetrics.com/project-releases/"+matchlist[s.row][%projectID]
			endif
			BrowseURL url
			break
			
		case 11: // column resize
			
			ControlInfo /W=$(s.win) $(s.ctrlName)
			variable c1, c2, c3, c4
			sscanf S_columnWidths, "%g,%g,%g,%g", c1, c2, c3, c4
			c1/=2;c2/=2;c3/=2;c4/=2
			ListBox $(s.ctrlName) win=$(s.win), widths={c1, c2, c3, c4}
			break
			
	endswitch
		
	return 0
end

// update listbox wave based on string str
function UpdateListboxWave(str)
	string str
	
	variable nameCol=1
	variable typeCol=5
	
	ControlInfo /W=InstallerPanel tabs
	variable SelectedTab=v_value
	variable col
	string listBoxName="", regEx=""
	
	DFREF dfr = root:Packages:Installer
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	switch(selectedTab)
		case 0:		// projects list
			wave /T FullList=dfr:ProjectsFullList, DisplayList=dfr:ProjectsDisplayList
			wave /T matchlist=dfr:ProjectsMatchList, HelpList=dfr:ProjectsHelpList
			wave /T titles=dfr:ProjectsColTitles
			listBoxName="listboxInstall"
			break
		case 1:		// updates list
			wave /T FullList=dfr:UpdatesFullList, DisplayList=dfr:UpdatesDisplayList
			wave /T matchlist=dfr:UpdatesMatchList, HelpList=dfr:UpdatesHelpList
			wave /T titles=dfr:UpdatesColTitles
			listBoxName="listboxUpdate"
			break
	endswitch
	Duplicate /free/T FullList, subList
	
	// save current selection
	string strSelection=""
	ControlInfo /W=InstallerPanel $listboxName
	if(V_Value>-1 && v_value<DimSize(DisplayList,0))
		strSelection=DisplayList[V_Value][0]
	endif
	
	if(selectedTab==0)
		ControlInfo /W=InstallerPanel popupType
		if (cmpstr(S_Value, "all"))
			regEx="\\b"+s_value+"\\b"
			Grep /GCOL=(typeCol)/Z/E=regEx subList as subList
		endif
	endif
	
	str=ReplaceString("+",str, "\+")
	regEx="(?i)"+str
	Grep /GCOL=(nameCol)/Z/E=regEx subList as matchList
	
	switch(selectedTab)
		case 0:
			if (DimSize(matchList,0)) // check for non-zero rows
				Duplicate /T/O/R=[][1,4] matchList, dfr:ProjectsDisplayList, dfr:ProjectsHelpList
				DisplayList[][2]=JulianToDate(str2num(DisplayList[p][2]),prefs.dateFormat)
				HelpList=""
				HelpList[][0]="Project "+matchList[p][%projectID]
			else
				Make /o/n=(0,4)/T dfr:ProjectsDisplayList, dfr:ProjectsHelpList
			endif
			break
		case 1:
			if (DimSize(matchList,0)) // check for non-zero rows
				Duplicate /T/O/R=[][1,4] matchList, dfr:UpdatesDisplayList, dfr:UpdatesHelpList
				HelpList=""
				HelpList[][0]=matchList[p][%info]
				HelpList[][1]=matchList[p][%info]
				HelpList[][2]=SelectString(strlen(matchList[p][%installDate]), "", "Installed on "+Secs2Date(str2num(matchList[p][%installDate]), 0))
				HelpList[][3]="Released on "+JulianToDate(str2num(matchList[p][%releaseDate]),prefs.dateFormat)
				HelpList[][3]+="\rFile type: "+ParseFilePath(4, matchList[p][%releaseURL], "/", 0, 0)
			else
				Make /o/n=(0,4)/T dfr:UpdatesDisplayList, dfr:UpdatesHelpList
			endif
			break
	endswitch
	
	if(DimSize(DisplayList,0))
		for(col=0;col<DimSize(DisplayList, 1);col+=1)
			strswitch( (titles[0][col])[0,2] )
				case "\JC":
					DisplayList[][col,col]="\JC"+DisplayList[p][q]
					break
				case "\JR":
					DisplayList[][col,col]="\JR"+DisplayList[p][q]
					break
			endswitch
		endfor
	endif
	
	string s
	if(strlen(strSelection)>0)
		FindValue /TEXT=strSelection/TXOP=4/RMD=[][0,0] DisplayList
		ListBox $listboxName, win=InstallerPanel, selRow=v_value, row=-1
		if(v_value<0)
			sprintf s "Showing %d of %d projects", DimSize(DisplayList, 0), DimSize(FullList, 0)
			setPanelStatus(s)
			Button btnInstallOrUpdate, win=InstallerPanel, disable=2
		endif
	else
		sprintf s "Showing %d of %d projects", DimSize(DisplayList, 0), DimSize(FullList, 0)
		setPanelStatus(s)
	endif
end

function setPanelStatus(strText)
	string strText
	
	TitleBox statusBox, win=InstallerPanel, title=strText
	return 0
end

function /S getUpdateURLfromFile(filePath)
	string filePath
	
	string url=""
	variable projectID=getUpdaterConstantFromFile("kProjectID", filePath)
	if(numtype(projectID)==0)
		sprintf url, "https://www.wavemetrics.com/project-releases/%d", projectID
		return url
	endif
	url=getStringConst("ksLOCATION", filePath)
	return url // "" on failure
end

function selectProject(projectID)
	string projectID
	
	ControlInfo /W=InstallerPanel tabs
	if(v_flag!=8)
		return 0
	endif
	
	DFREF dfr = root:Packages:Installer
	
	string strListbox=SelectString(v_value, "listboxInstall","listboxUpdate")
	string strListWave=SelectString(v_value, "ProjectsMatchList","UpdatesMatchList")
	wave /T ListWave=dfr:$strListWave
	FindValue /Z/TEXT=projectID/TXOP=2/RMD=[][0,0] ListWave
	if(v_value>-1)
		ListBox $strListbox win=InstallerPanel, selRow=v_value
		setPanelStatus("Selected: "+ListWave[v_value][1])
	endif
end

function InstallSelection()
		
	DFREF dfr = root:Packages:Installer
	wave /T matchlist=dfr:ProjectsMatchList
	
	ControlInfo /W=InstallerPanel listboxInstall
	if(v_value>-1 && v_value<dimsize(matchlist,0))
		variable success
		string url="https://www.wavemetrics.com/node/"+matchlist[v_value][%projectID]
		success=install(url, gui=2) 
		// we use install rather than installProject because install may be able to figure out short title.
		if (success)
			setPanelStatus("Reloading Updates List")
			reloadUpdatesList(1, 1)
			setPanelStatus("Install Complete")
		else
			setPanelStatus("Selected: "+matchlist[v_value][%name])
		endif
	endif
	return 0
end

// ----------------------- filter code ----------------------------

// intercept and deal with keyboard events in notebook subwindow
function fHook(s)
	STRUCT WMWinHookStruct &s

//	GetWindow $s.winName activeSW
//	if (CmpStr(S_value,"InstallerPanel#nb0") != 0)
//		return 0
//	endif
	if(s.eventcode==2) // window is being killed
		return 1
	endif
	
	DFREF dfr = root:Packages:Installer	
	NVAR stubLen=dfr:stubLen
	
	if(s.eventcode==3 && stubLen==0) // mousedown
		return 1
	endif

	if(s.eventcode==5) // mouseup 	
		GetSelection Notebook, InstallerPanel#nb0, 1 // get current position in notebook
		V_endPos=min(stubLen,V_endPos)
		V_startPos=min(stubLen,V_startPos)
		Notebook InstallerPanel#nb0 selection={(0,V_startPos),(0,V_endPos)}
		return 1
	endif
	
	if(s.eventcode==10) // menu 
		strswitch(s.menuItem)
			case "Paste":
				GetSelection Notebook, InstallerPanel#nb0, 1 // get current position in notebook		
				string strScrap=getscrapText()
				strScrap=replacestring("\r", strScrap, "")
				strScrap=replacestring("\n", strScrap, "")
				strScrap=replacestring("\t", strScrap, "")
				Notebook InstallerPanel#nb0 selection={(0,V_startPos),(0,V_endPos)}, text=strScrap
				stubLen+=strlen(strScrap)-abs(V_endPos-V_startPos)
				s.eventcode=11 
				// pretend this was a keyboard event to allow execution to continue
				break
			case "Cut":
				GetSelection Notebook, InstallerPanel#nb0, 3 // get current position in notebook
				PutScrapText s_selection
				Notebook InstallerPanel#nb0 selection={(0,V_startPos),(0,V_endPos)}, text=""
				stubLen-=strlen(s_selection)
				s.eventcode=11 
				break
			case "Clear":
				GetSelection Notebook, InstallerPanel#nb0, 3 // get current position in notebook
				Notebook InstallerPanel#nb0 selection={(0,V_startPos),(0,V_endPos)}, text="" // clear text
				stubLen-=strlen(s_selection)
				s.eventcode=11 
				break
		endswitch
		Button ButtonClear, win=InstallerPanel, disable=3*(stublen==0)
		fClearText((stubLen==0)); 
	endif
				
	if(s.eventcode!=11)		
		return 0
	endif	
	
	if(stubLen==0) // Remove "Filter" text before starting to deal with keyboard activity
		Notebook InstallerPanel#nb0 selection={startOfFile,endofFile}, text=""
	endif
	
	// deal with some non-printing characters
	switch(s.keycode)
		case 9:	// tab: jump to end
		case 3:
		case 13: // enter or return: jump to end
			Notebook InstallerPanel#nb0 selection={startOfFile,endofFile}, textRGB=(0,0,0)
			Notebook InstallerPanel#nb0 selection={endOfFile,endofFile}
			GetSelection Notebook, InstallerPanel#nb0, 1 // get current position in notebook
			stubLen=V_endPos
			break
		case 28: // left arrow
			fClearText((stubLen==0)); return 0
		case 29: // right arrow
			GetSelection Notebook, InstallerPanel#nb0, 1
			if(V_endPos>=stubLen)
				if(s.eventMod&2) // shift key
					Notebook InstallerPanel#nb0 selection={(0,V_startPos),(0,stubLen)}
				else
					Notebook InstallerPanel#nb0 selection={(0,stubLen),(0,stubLen)}
				endif
				fClearText((stubLen==0)); return 1
			endif
			fClearText((stubLen==0)); return 0
		case 8:
		case 127: // delete or forward delete
			GetSelection Notebook, InstallerPanel#nb0, 1
			if(V_startPos==V_endPos)
				V_startPos += (s.keycode==8) ? -1 : 1
			endif
			V_startPos=min(stubLen,V_startPos); V_endPos=min(stubLen,V_endPos)
			V_startPos=max(0, V_startPos); V_endPos=max(0, V_endPos)
			Notebook InstallerPanel#nb0 selection={(0,V_startPos),(0,V_endPos)}, text=""
			stubLen-=abs(V_endPos-V_startPos)
			break			
	endswitch
		
	// find and save current position	
	GetSelection Notebook, InstallerPanel#nb0, 1
	variable selEnd=V_endPos
		
	if(strlen(s.keyText)==1) // a one-byte printing character
		// insert character into current selection
		Notebook InstallerPanel#nb0 text=s.keyText, textRGB=(0,0,0)
		stubLen+=1-abs(V_endPos-V_startPos)
		// find out where we want to leave cursor
		GetSelection Notebook, InstallerPanel#nb0, 1 
		selEnd=V_endPos
	endif	
	
	string strStub="", strInsert="", strEnding=""
		
	// select and format stub
	Notebook InstallerPanel#nb0 selection={startOfFile,(0,stubLen)}, textRGB=(0,0,0)	
	// get stub text
	GetSelection Notebook, InstallerPanel#nb0, 3
	strStub=s_selection
	// get matches based on stub text
	UpdateListboxWave(strStub)
	
	// do auto-completion based on stubLen characters
	ControlInfo /W=InstallerPanel tabs
	if(v_value==0)
		wave /T  matchList=dfr:ProjectsMatchList
	else
		wave /T  matchList=dfr:UpdatesMatchList
	endif
	
	if(s.keycode==30 || s.keycode==31) // up or down arrow
		Notebook InstallerPanel#nb0 selection={(0,stubLen),endOfFile}
		GetSelection Notebook, InstallerPanel#nb0, 3
		strEnding=s_selection
		strInsert=fArrowKey(strStub, strEnding, 1-2*(s.keycode==30), matchList)
	else
		strInsert=fCompleteStr(strStub, matchList)
	endif
	// insert completion text in grey	
	Notebook InstallerPanel#nb0 selection={(0,stubLen),endOfFile}, textRGB=(50000,50000,50000), text=strInsert	
	Notebook InstallerPanel#nb0 selection={(0,selEnd),(0,selEnd)}, findText={"",1}
	
	Button ButtonClear, win=InstallerPanel, disable=3*(stublen==0)
	fClearText((stubLen==0))
	return 1 // tell Igor we've handled all keyboard events
end

function fClearText(doIt)
	variable doIt
	
	if(doIt)
		Notebook InstallerPanel#nb0 selection={startOfFile,endofFile}, textRGB=(50000,50000,50000), text="Filter"
		Notebook InstallerPanel#nb0 selection={startOfFile,startOfFile}
	endif
end

function /T fGetStub()
	
	DFREF dfr = root:Packages:Installer
	NVAR stubLen=dfr:stubLen
	
	// find and save current position	
	GetSelection Notebook, InstallerPanel#nb0, 1
	variable selEnd=V_endPos
	
	// select stub
	Notebook InstallerPanel#nb0 selection={startOfFile,(0,stubLen)}
	
	// get stub text
	GetSelection Notebook, InstallerPanel#nb0, 3
	string strStub=s_selection
	
	// reset position
	Notebook InstallerPanel#nb0 selection={(0,selEnd),(0,selEnd)}, findText={"",1}
	
	return strStub
end

// returns completion text for first match of string s in text wave w
function /T fCompleteStr(s, w)
	string s
	wave /T w
	
	variable col=1
	
	if (strlen(s)==0)
		return ""
	endif
	
	string stub=s[0,strlen(s)-1]
	variable stubLen=strlen(stub)
	if (stubLen==0)
		return ""
	endif
	
	make /free/T/N=1 w_out
	grep /GCOL=(col)/Z/E="(?i)^"+stub w as w_out
	if(dimsize(w_out,0)==0)
		return ""
	endif
	return (w_out[0][col])[stubLen,inf]	
end

// find next or previous matching entry in wList and return completion text
function /T fArrowKey(stub, ending, increment, wList)
	string stub, ending
	variable increment
	wave /T wList
	
	variable col=1
	
	if (strlen(stub)==0)
		return ""
	endif
	
	variable stubLen=strlen(stub)
	if (stubLen==0)
		return ""
	endif
	
	Make /free/T/N=1 w_out
	Grep /Z/GCOL=(col)/E="(?i)^"+stub/DCOL={0} wList as w_out
	if(DimSize(w_out,0)==0)
		return ""
	endif
	
	FindValue /RMD=[][0,0]/TEXT=stub+ending /TXOP=4/Z w_out
	if(v_value>-1)
		v_value+=increment
		v_value = V_value<0 ? DimSize(w_out,0)-1 : v_value
		v_value = V_value>=DimSize(w_out,0) ? 0 : v_value
	else
		return (w_out[0][0])[stubLen,Inf]
	endif
	return (w_out[v_value][0])[stubLen,Inf]
end

// PNG: width= 90, height= 30
picture fClearTextPic
	ASCII85Begin
	M,6r;%14!\!!!!.8Ou6I!!!"&!!!!?#R18/!3BT8GQ7^D&TgHDFAm*iFE_/6AH5;7DfQssEc39jTBQ
	=U"$&q@5u_NKm@(_/W^%DU?TFO*%Pm('G1+?)0-OWfgsSqYDhC]>ST`Z)0"D)K8@Ncp@>C,GnA#([A
	Jb0q`hu`4_P;#bpi`?T]j@medQ0%eKjbh8pO.^'LcCD,L*6P)3#odh%+r"J\$n:)LVlTrTIOm/oL'r
	#E&ce=k6Fiu8aXm1/:;hm?p#L^qI6J;D8?ZBMB_D14&ANkg9GMLR*Xs"/?@4VWUdJ,1MBB0[bECn33
	KZ1__A<"/u9(o<Sf@<$^stNom5GmA@5SIJ$^\D=(p8G;&!HNh)6lgYLfW6>#jE3aT_'W?L>Xr73'A#
	m<7:2<I)2%%Jk$'i-7@>Ml+rPk4?-&B7ph6*MjH9&DV+Uo=D(4f6)f(Z9"SdCXSlj^V?0?8][X1#pG
	[0-Dbk^rVg[Rc^PBH/8U_8QFCWdi&3#DT?k^_gU>S_]F^9g'7.>5F'hcYV%X?$[g4KPRF0=NW^$Z(L
	G'1aoAKLpqS!ei0kB29iHZJgJ(_`LbUX%/C@J6!+"mVIs6V)A,gbdt$K*g_X+Um(2\?XI=m'*tR%i"
	,kQIh51]UETI+HBA)DV:t/sl4<N*#^^=N.<B%00/$P>lNVlic"'Jc$p^ou^SLA\BS?`$Jla/$38;!#
	Q+K;T6T_?\3*d?$+27Ri'PYY-u]-gEMR^<d.ElNUY$#A@tX-ULd\IU&bfX]^T)a;<u7HgR!i2]GBpt
	SiZ1;JHl$jf3!k*jJlX$(ZroR:&!&8D[<-`g,)N96+6gSFVi$$Gr%h:1ioG%bZgmgbcp;2_&rF/[l"
	Qr^V@O-"j&UsEk)HgI'9`W31Wh"3^O,KrI/W'chm_@T!!^1"Y*Hknod`FiW&N@PIluYQdKILa)RK=W
	Ub4(Y)ao_5hG\K-+^73&AnBNQ+'D,6!KY/`F@6)`,V<qS#*-t?,F98]@h"8Y7Kj.%``Q=h4.L(m=Nd
	,%6Vs`ptRkJNBdbpk]$\>hR4"[5SF8$^:q=W([+TB`,%?4h7'ET[Y6F!KJ3fH"9BpILuUI#GoI.rl(
	_DAn[OiS_GkcL7QT`\p%Sos;F.W#_'g]3!!!!j78?7R6=>B
	ASCII85End
End

// --------------------------- end of filter functions --------------------------

// PNG: width= 90, height= 30
picture cog
	ASCII85Begin
	M,6r;%14!\!!!!.8Ou6I!!!"&!!!!?#R18/!3BT8GQ7^D&TgHDFAm*iFE_/6AH5;7DfQssEc39jTBQ
	=U%adj95u_NKeXCo&'5*oW5;QerS2ddE(FP;'6Ql#oBiuS1dW/'TBu9uWoJd,<?@,Ku[OT_*C6>N&V
	5fgKO-Z!0m]BHZFE$kSY/b+).^UIV80KE-KEW"FQ9_,eF^bT7+UPEE3;1R;IlL+VJZGMtc/)57aYSB
	X^5kLMq7Sp9q0YBG@U214ph]9b)&eA4@(!#Ip%g0/F0_UH>\E'7M.=ui]mI6`m'G"T)B'OA[IVmQDm
	=?Eat`-Qd[5bR1c#F]`cT(Se*XgJen[DI=<Y8YF7N!d-X+iY2pq;SVDt;siWS_bs$!E]WT^pRhs]aH
	M%csq\akmJ4tMo<HGU1WqT,5cib%XrHM`>*]/&U:](RQ7QDj\u&#cMNi8>`?8;-?rCcXX>+1^gW10F
	n!h-2G-M;R\a+TD2op'?`R]"%W!DpJmO],i(@-(/F?,;L7Vkt%YOa,f\8+CV@lHVOEMgMZPnh$6>!V
	MTYBm^8f[O,?Z$2MnH6.T'JWSM4HtSissRo-):4d:ecoe5Tn^(gUEQm+o94@;L(/[91%aXk:!pP;mm
	\kh$s.7qbe%=-p1eBtDs*CHp:+$CUY\0A,jM0:4J2&pY-HWBG?nb`"BE/M-#*)+E?I*C/(r;J]APNh
	3!Ea<(u)`o?0R`ma=QV<n?GV/s3:I0Wf2_M0p@:__T%OEl+sL@10K8&ViQgR(0Q3qMLYA':/iba:,;
	]Y$@ACMV&9b[fD4A`Vq5+A!37VD0na`;0#fWNWKq#f5N>Mt)$S['[2:?=(p2$Q$$NX_cXoJ`iVOcHm
	Rb+"_b#*b4@tp)Xq9r*1_<^IVlpMJ=kE>MhiHa2]]q9<d*4(lA_8$4ej2NM5Z!#`oc=+Ttk-]%D5"O
	Yiu,o$V/I<=@2fN3Ds,PNfIEnqn6C?^[OYDs4q2k*s6TFu+@1>SKUmdko@B5>Pp)-]8`l_Ig,/1c.T
	K'Z+asa)qDc*mqZAKmijlOd;;&H$MEMWY1:\q<G#aaNVlho?TWCGL35!G658MH$RpQ,/[:S#==eP-@
	T+s%'h-7&.0)\eW6j@1gNW'FYlgRIid1g1dP0.MtL)"o@*4A+B&XU\9bRSJWg?B!keI%b6T7FS'?W(
	@7j-a-n[,Adkh%U((6"oN9G"iRV$rmb0"2lqXk08!N&V_b13Oo*tc)h=[L]E@W<ihr:]%Dbs-*cCbc
	T^`<b81D1(d_gue7JX)rMl-Q!ag!0.a4mbCL5'MQH2X<p3`1nA#69QNiWDnN[J^:kIm`JPCXo#W8lo
	?KFN_dOMp#7s\i!;q:1Vb`q^Za5j'0Sg8ALVn\tm:OM*.G/IF;=K4S+O/0U]^`u\O,i'69Y*0'f,SH
	:Kp)mH3.EQ3hDf%?f;W[LbJbR7R5Lb$8U4I7,[8ZM7fU7H5(>6BlSlr1cG6"263q!"YH.\`>aLYN(!
	e3hZYm666$Mq_c(_GHO?%CE(rnI-UV=I6M\e$%CXt$`9q"IB8d`/4e)0&Dcf`43oobf6MqdV?d>\73
	X/dI(2jZ+#[Nt'sQg5KRFl_^rEVX>cZ5h3g1gR#*IhTZf+KqmsVrW[`UcE6>MM*N0fMcT[S!:h'=ju
	EaL@5>OfU]WXd+d/JISLYCQZ_BPkB$IiAYUBq1l^ecC4a8EYJ'WJ,pak5V59k6$F23m\(d</D&W$.c
	&:n64N(\`j0%h;m7iB=j)(R^kh8BUCfn<.JP[26K1F7\>0JOc56jWCOX(6k=i$m^+A%<G5QZ.i$e8]
	01cU/k$R?&@c[1P\L>tK[OkSMmQ7lT_puI&4&#-'R99q+p;3\Sn`Ic2G%kDj/"N6oB<E0?Z6Kl"@,Y
	?4P5G,Nu\q^LTs(Vj(g3YCoZEl;W9!T)>ePG,S!10n_Y\rP(FBr90.J3;jVP^MYp<-ML6>(=)*bFsB
	&uWX-VWg'<P8G@$+\A?J1@G'F,/Z.<q/;.,=8I?;gFp>>;IjEQPE_:7`.R+3bElA@DB6<k@ksJ9lg(
	=CVM=g<G(^E#SiiFHZ8.qF-^ppkE&\[U*_);<'Lfk*Fq]^#W339=Q'IWpj"KgUWU]1,ETW<&G^OtOH
	MkGSX1r0@A!=Gd&>m2;$s(nF8b!K?8R`eZjO8V38_I$<1AcnP!)B*`T#3.X=hE[s8PMoFf*/H2@32u
	N'Len`GUk?n:K*@=9gMMi1ES9gF7flQcCD`2n^,h:`S5=GNQ^G#@^/a:?]W`PV50n4Xue>QVk8E1=]
	lWKB?uV(SiXjL_hVC,G-+W-bHd)[C`^u(BPM:VV58ltJcZ8d$CEhp-Ek)Qb.'^'f)4eT`-]7*:O[1>
	WO?>HRRZ3%&Aum4gNS.jU=.^S*B#H\'0H3Z)tG>O;)WW1_IO09[>@PI4Y3R7J0]^!UgQhj+u1-,O_#
	4q7aj2?Y5.V]ph=D*J`g8?n%JH:8P)K%MLrfVTs(X1]A:d+mFtdNBG"";'8siHNZC4&bKJq`%mNb7r
	SW;=`2-+n=L)HDOsFHoS$CX_6m<3W758iRSt7"9?7u`s%5^"&NsfV]&fKiR),nod"F>!-ZZj257RH'
	BCpkU2>pgE:BYW?=m,Fa:(R[)N0g+8U_d1?h6?87.J?1.S8QM+N.?c/5H^[K9Qq/KSe*09PFi*)kOs
	Cpa8151hB![K\`b9:/2t#`0h)TQGGW^_mOCaj@jCA@uU*q95,uIW@7!X&<O\"P/_=YGg"!\PP(fcMs
	XX8]B01I/5GVm/oHal.qLkA^e?r`*cq5<6cMYpEN]i/%/Vl(p'^fIMd8C\oHLnT1!)7oQm]mKeID*1
	Nn<=:2#ZkEe#YpHHH,[1j)T%7I4;@/)4cu_Q1)PbZM:[@i"UEI%;r>p03XoVZ_RkU=+$'7#=USG/bL
	8-+m<=>h,iqN=A8kNQ3E0-ce+TlUO7L$\:&5CW07\^Y5(=Lpj3bn5fXf]+hD?H]7Wq?&[-c"7hNK0#
	/)B'Mj<HT8ma3CurC,*bh\'aHR7Z[QbX-ZmAk800m):lpO:?P+(!;TY'R\j"AmjWGZ^'4n^bf?W3J@
	>&NBK1HqWkVhSJ@2:%U$9,hq:d,G1cCmI-TcrP\J*Ut[2@6?BcK3XN6]^DH?sm>]m;PWk0+t]M3*pb
	_i5ToaNr1&dko4ib1O7G-^#a3R58IWd+6c;6ULrU<E06&]A7B"H::^+p=jM"Cht@E-\k9W-F%RN7[f
	g9`s&hll-^kfa*7KZfMd!Yc("^(?tb@(G_h8BF>IEA!<RhTlND,!db&r!Y3m>2$M&6dHhJmnR;"(TN
	7k9deNFM:rt_%M:_\bI5MO2h<A0N#R:ZSrC"&ps\iY*%&:=-;@IrX+"G9!l_&sOI?=_'7)$hsk)[Og
	CfZ9V$@mNB]AS#G_>V6^Z_/)$iF?19\*_+U8'Lh!@O$@74\ogtP<K2K,l#,B;0e6NJ*#oS35C1Gqa?
	[/#EMYbdp\'g083t^Hm.OC']=?TCh[+\@'/CD^$lb9k>s0TnKIaqh4fI!&lDq*\4*U*,*??/2AnId;
	.P@%q^Y_gV7L#<Y@CP!Nm,EKn=&i8<r@!PTa5]H_PQ](fBqn;iMUE,Pl5E-<#!*W9G0Gifi6\]*<o4
	FnfrX,WF^_\F$1nF]^fU-bLO!=bm%5lG.k3$IWMqUu'c@l,R*B4I#7$5RGsBAo;S"q!W&r%8C2/"PK
	bsa<4<J73edT0;DW#W4)GM@uFDONL"9R+_N^([T!(k&'aJc*VLHV&^R;!(_#4Z$g7@#[rnEd4b][m6
	"s2Cf6FX#Yth+![-Bc9;DCc35!#ZOe]>R5l%A3s9r*"E3cZ^HAq!@!X3ZLR*.A7oQ8om.\p[kV)RYJ
	YS_VJ,ke(!91A):`eg`H3<A0s%C//=2RBq,pC2:S]*\P@U_Oa*3.T[g!Hf.uI$^Z574:Ig+a&Rh&b)
	g:i!IBPVCY]Y$?Mc-nM/==coe'#A=jP*M;$C2,4.LP+[KAA[:ZiG]VW6ipmf;5gRtUogbYmG#,M=Y+
	23\;kLm>:;.qMlKqn)uClJ![.i',6Vo?VRu"<5C39QHia.rIUXNcq/492/E<t4:q=5j]#0#BT^2C8R
	r=7NOaA&EGCimE'I"(o(b72sE7cRPmWBnj]tHh/;(=(Hs(iTH$*GN/REC6P4"-OQ)1Z*S9OlNXqYG,
	/qFo"%Xt5WcH)DMDo_@'gn2mp\l)\(b!Y0Pa!2n,Nj%-R@Yjn?WT$E#t(FUbj=.R08ON,:0qYL%:/M
	0]<Q1)2)G':0@s*h8ZZ<4MLPu4'A3d&SI6l9j+cCI%0ltDh?G(&13<0N\Q=?t??;p9Q8$59_n3Rli=
	5b`N"1`$#>;[JNr+$!*^fli%qGlHBaGd#r_gndbH0<a<pR0sE5L0:j)I_t)\EH/7Wqt8QJMd<r<&WK
	8J3cuoH9hij#22_bS-?/1q+bUC@(DjDc_1Dg*LCYK([C$_m"OB=44C54XF6CiRHM)#JSik-Qi#lgdX
	C:AAV;n0(-%L_0m!T,"d5MVG@HhTKZ87K.p%WH^Xh3kCpXcTY<@p]%p!H!PcGm8Ma`dWI-i:%O`.@A
	\EMhGlp>Rk7O<G2m`*r,h[u\8;4r,bUaQp%EDN*J]D4B1hFXuppq^tpModARV5%<QlNN?L%hAEkIlW
	/#`^]Bs#-d.f-97RH2#l<o@ZXZ&T?iRH-<%N9SU+'so7AAgW2mil0fWaM)A,6@)4Rp@WFC0@Y,uIN:
	5uCJkMPAJFd6VVd/UR6[rI=?0fVgq/kI,s^(YARDN54WD\PD#"bV<C5XKS<`]Fql#m@)ul]O#Nn]-2
	[lB);<HM,sPARkJs:;d4_S!/nh?qGe7@I:AsnMi7DjM_D$2XW>fsY^ZQI8$;`nm!f"mg^^mq'BNs/!
	!!!j78?7R6=>B
	ASCII85End
End

// ------------------- from UserProcLoader, but modified ---------------

// search recursively for ipf files and return list of files as textwave
function /wave getProcsRecursive(fast)
	variable fast
	
	variable maxFolders=1000, fullpath=1
	variable folderIndex, i
	string strFile="", strFolder=""
	Make /free/T/n=0 w_folders, w_allFiles
	
	w_folders={SpecialDirPath("Igor Pro User Files",0,0,0)+"User Procedures:"} // search recursively starting in this folder

	// w_folders will grow as subfolders are added
	for(folderIndex=0;folderIndex<numpnts(w_folders);folderIndex+=1)
		// check files in folderIndexth folder from w_folders
		NewPath /O/Q/Z tempPathIXI, w_folders[folderIndex]
		
		if(fast) // don't worry about semicolons in filenames, or shortcuts ending in .ipf!
			// add list of ipf files in current folder to w_allFiles
			wave /T w=ListToTextWave(IndexedFile(tempPathIXI, -1, ".ipf"),";")
			if (fullpath)
				w=w_folders[folderIndex]+w
			endif
			TextWaveConcatenate(w_allFiles, w)
			// add list of folders in current folder to w_folders
			wave /T w=ListToTextWave(IndexedDir(tempPathIXI, -1, 1),";")
			w+=":"
			TextWaveConcatenate(w_folders, w)
			
#ifdef WINDOWS
			// get a list of shortcuts
			wave /T w=ListToTextWave(IndexedFile(tempPathIXI, -1, ".lnk"),";")
			if(numpnts(w))
				w=ResolveAlias(w_folders[folderIndex]+w)
				Duplicate /free w w_f
				// keep shortcuts to files in w, shortcuts to folders in w_f
				if(fullpath==0)
					w = ParseFilePath(0, w, ":", 1, 0)
				endif
				TextWaveStringMatch(w, "*.ipf")
				TextWaveConcatenate(w_allFiles, w)
				TextWaveStringMatch(w_f, "*:")
				TextWaveConcatenate(w_folders, w_f)
			endif
#else // mac
			// get a list of file aliases
			wave /T w=ListToTextWave(IndexedFile(tempPathIXI, -1, "alis"),";")
			if(numpnts(w))
				w=ResolveAlias(w_folders[folderIndex]+w)
				if(fullpath==0)
					w = ParseFilePath(0, w, ":", 1, 0)
				endif
				TextWaveStringMatch(w, "*.ipf")
				TextWaveConcatenate(w_allFiles, w)
			endif
			// get a list of folder aliases
			wave /T w=ListToTextWave(IndexedFile(tempPathIXI, -1, "fdrp"),";")
			if(numpnts(w))
				w=ResolveAlias(w_folders[folderIndex]+w)
				TextWaveZapString(w, "")
				TextWaveConcatenate(w_folders, w)
			endif
#endif
			continue // next folder
		endif
				
		// the slow way: loop through files in folder
		for(i=0;1;i+=1)
			strFile=IndexedFile(tempPathIXI, i, "????")
			if(strlen(strFile)==0)
				break
			endif
			GetFileFolderInfo /Q/Z w_folders[folderIndex]+strFile
			if(V_isAliasShortcut)
				strFile=ResolveAlias(w_folders[folderIndex]+strFile)
				if(stringmatch(strFile,"*:"))
					w_folders[numpnts(w_folders)]={strFile}
				endif
				if(stringmatch(strFile,"*.ipf"))
					if(fullpath==0)
						strFile=ParseFilePath(0, strFile, ":", 1, 0)
					endif
					w_allFiles[numpnts(w_allFiles)]={strFile}
				endif
				continue // next file
			endif
			
			if(stringmatch(strFile,"*.ipf"))
				if(fullpath)
					strFile=w_folders[folderIndex]+strFile
				endif
				w_allFiles[numpnts(w_allFiles)]={strFile}
				continue
			endif
		endfor // next file
		
		// add subfolders in current folder
		for(i=0;1;i+=1)
			strFolder=IndexedDir(tempPathIXI, i, 1)
			if(strlen(strFolder)==0)
				break
			endif
			w_folders[numpnts(w_folders)]={strFolder+":"}
		endfor // next subfolder
		if(folderIndex==maxFolders)
			Print "Warning: getProcsRecursive exceeding max iterations"
			break
		endif
	endfor // next folder
	
	KillPath /Z tempPathIXI
	return w_allFiles
end

// appends w2 to w
function TextWaveConcatenate(w, w2)
	wave /T w, w2
	
	variable oldPnts=numpnts(w), newPnts=numpnts(w2)
	if(newPnts)
		Redimension /n=(oldPnts+newPnts), w
		w[oldPnts,]=w2[p-oldPnts]
	endif
end

// Recursively resolve shortcuts to files/directories
// returns full path or an empty string if the file does not exist or the
// shortcut points to a non existing file/folder
function/S ResolveAlias(strPath)
	String strPath // full path to file, folder or alias

	GetFileFolderInfo/Q/Z RemoveEnding(strPath, ":")
	if(V_isAliasShortcut)
		return ResolveAlias(S_aliasPath)
	endif
	if(v_flag==0)
		return strPath // path to a file or folder
	endif
	return ""
end

// keep only points of textwave matching strMatch
function TextWaveStringMatch(w, strMatch)
	wave /T w
	string strMatch
	
	if(numpnts(w))
		w=SelectString(stringmatch(w,strMatch),"",w)
		TextWaveZapString(w, "")
	endif
end

// Remove points from textwave w (and optionally the corresponding points
// of waves in wave reference wave w_zapRefs) that match matchStr. Does
// not check numbers of points in w_zapRefs waves. Wildcards okay.
function TextWaveZapString(w, matchStr, [w_zapRefs])
	wave /T w
	string matchStr
	wave /WAVE w_zapRefs
	
	if(numpnts(w)==0)
		return 0
	endif
	variable i=0, j=0
	do
		if(stringmatch(w[i],matchStr))
			DeletePoints i,1,w
			if(WaveExists(w_zapRefs))
				for(j=0;j<numpnts(w_zapRefs);j+=1)
					DeletePoints i,1,w_zapRefs[j]
				endfor
			endif
		else
			i+=1
		endif
	while(i<numpnts(w))
	return numpnts(w)
end

function TextWaveZapList(w, matchList, [ w_zapRefs])
	wave /T w
	string matchList
	wave /Z/WAVE w_zapRefs
	
	Make /free/n=(ItemsInList(matchList)) index
	index=TextWaveZapString(w, StringFromList(p,matchList), w_zapRefs=w_zapRefs)
	return index[numpnts(index)-1]
end

// ------------------ end of stuff from UserProcLoader ------------------
