﻿#pragma TextEncoding="UTF-8"
#pragma rtGlobals=3
#pragma ModuleName=XYbrowser
#pragma IgorVersion=8
#pragma version=1.50

#include <Resize Controls>
#include <WaveSelectorWidget>

// A browser gui for previewing y vs x plots of paired waves.
// Waves are paired by specifying the required X wave in the
// wavenote of the Y wave, in the form "Xwave:wavename".
// The GUI can also be used to preview X-Y plots of specified columns of
// 2D waves.

// The browser is useful for building Y vs X plots of many pairs of
// waves. X-Y wave pairs are used for compatibility with various tools
// for spectroscopy that will not work with 2D waves. This is designed to
// be combined with a custom file loader that sets the wavenote. An
// example is the WiRE loader for Renishaw Raman files, https:
// www.wavemetrics.com/node/21301

// set this constant to format graphs for Raman spectroscopy
static constant kRaman = 0

// headers used by the updater project
static constant kProjectID=21303 // the project node on IgorExchange
static strconstant ksShortTitle="XYBrowser" // the project short title on IgorExchange

static strconstant ksPackageName = XYBrowser
static strconstant ksPrefsFileName = acwXYBrowser.bin
static constant kPrefsVersion = 101

menu "Data"
	"XY Browser", /Q, XYbrowser#MakeXYBrowserPanel()
end

menu "Windows"
	"XY Browser", /Q, XYbrowser#MakeXYBrowserPanel()
end

// ------------------ package preferences --------------------

static structure PackagePrefs
	uint32 version
	uint32 options
	uint16 mincols, maxcols, xcol, ycol
	char key[30]
	char keysep[5]
	char listsep[5]
	STRUCT Rect win // window position and size, 8 bytes
	char mode // line mode
	char plotPrefs
	char reserved[128 - 8 - 8 - 30 - 5 - 5 - 8 - 1 - 1]
endstructure

// set prefs structure to default values
static function PrefsSetDefaults(STRUCT PackagePrefs &prefs)
	prefs.version = kPrefsVersion
	prefs.options = 0
	prefs.mincols = 2
	prefs.maxcols = 2
	prefs.xcol = 0
	prefs.ycol = 1
	prefs.key = "Xwave"
	prefs.keysep = ":"
	prefs.listsep = "\r"
	prefs.win.left = 100
	prefs.win.top = 100
	prefs.win.right = 100 + 740
	prefs.win.bottom = 100 + 370
	prefs.mode = 1
	prefs.plotPrefs = 0
	int i
	for(i=0;i<(128-66);i+=1)
		prefs.reserved[i] = 0
	endfor
end

static function LoadPrefs(STRUCT PackagePrefs &prefs)
	LoadPackagePreferences /MIS=1 ksPackageName, ksPrefsFileName, 0, prefs
	if (V_flag!=0 || V_bytesRead==0 || prefs.version!=kPrefsVersion)
		PrefsSetDefaults(prefs)
	endif
end

// save window position in package prefs
static function SaveWindowPosition(string strWin)
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	string recreation = WinRecreation(strWin, 0)
	int index, top, left, bottom, right
	index = strsearch(recreation, "/W=(", 0)
	if (index > 0)
		sscanf recreation[index,strlen(recreation)-1], "/W=(%g,%g,%g,%g)", left, top, right, bottom
		prefs.win.top = top
		prefs.win.left = left
		prefs.win.bottom = bottom
		prefs.win.right = right
	endif
	SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
end

// returns waveref for the X wave paired with wave w
static function /WAVE getXWaveRef(wave w, STRUCT PackagePrefs &prefs)
	if(WaveExists(w)==0)
		return $""
	endif
	DFREF dfr = GetWavesDataFolderDFR(w)
	string s_x = StringByKey(prefs.key, note(w), prefs.keysep, prefs.listsep)
	return dfr:$s_x
end

static function MakeXYBrowserPanel()
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	DoWindow /F XYBrowserPanel
	if(v_flag)
		return 0
	endif
	
	NewPanel /K=1/N=XYBrowserPanel/W=(prefs.win.left,prefs.win.top,prefs.win.right,prefs.win.bottom) as "XY Browser"
	ModifyPanel /W=XYBrowserPanel, noEdit=1
	
	DefineGuide/W=XYBrowserPanel guideR={FR,-10}
	DefineGuide/W=XYBrowserPanel guideDiv={FL,0.3,Fr}
	DefineGuide/W=XYBrowserPanel guideL={guideDiv,5}
	DefineGuide/W=XYBrowserPanel guideT={FT,30}
	DefineGuide/W=XYBrowserPanel guideB={FB,-30}
	
	Display/FG=(guideL,guideT,guideR,guideB)/HOST=XYBrowserPanel /N=PreviewPlot
	ModifyGraph /W=XYBrowserPanel#PreviewPlot frameStyle=5
	SetActiveSubwindow XYBrowserPanel
	
	
	// insert a notebook subwindow to be used for filtering lists
	DefineGuide/W=XYBrowserPanel fgb={FB,-7}
	DefineGuide/W=XYBrowserPanel fgt={FB,-24}
	DefineGuide/W=XYBrowserPanel fgr={guideDiv,-20}
	NewNotebook /F=1/N=nbFilter/HOST=XYBrowserPanel/W=(10,500,5200,1000)/FG=($"",fgt,fgr,fgb)/OPTS=3
	variable isWin = 0
	#ifdef WINDOWS
		isWin = 1
	#endif
	Notebook XYBrowserPanel#nbFilter fSize=12-3*isWin, showRuler=0
	Notebook XYBrowserPanel#nbFilter spacing={1, 0, 0}
	Notebook XYBrowserPanel#nbFilter margins={0,0,1000}
	SetWindow XYBrowserPanel#nbFilter, activeChildFrame=0
	ClearText(1) // sets notebook to its default appearance
	SetActiveSubwindow XYBrowserPanel
	
	// make a Button for clearing text in notebook subwindow
	Button ButtonClear, win=XYBrowserPanel,pos={147,276},size={15,15},title=""
	Button ButtonClear, win=XYBrowserPanel, Picture=XYBrowser#ClearTextPicture,Proc=XYBrowser#ButtonProc, disable=1
	
	PopupMenu popX,pos={8.00,5},size={120,20}, Proc=XYBrowser#popupProc
	PopupMenu popX,mode=1+(prefs.options&1),value= #"\"Paired X-wave;2-D columns\""
	SetVariable setvarKey,pos={140,7},size={80,14},title="Key", disable=prefs.options&1
	SetVariable setvarKey,value= _STR:prefs.key, Proc=XYBrowser#setvarProc
	SetVariable setvarKeySep,pos={225,7},size={90,1}, disable=prefs.options&1
	SetVariable setvarKeySep,title="Key Separator",value= _STR:prefs.keysep, Proc=XYBrowser#setvarProc
	SetVariable setvarListSep,pos={320,7},size={90,14}, disable=prefs.options&1
	SetVariable setvarListSep,title="List Separator",value= _STR:prefs.listsep, Proc=XYBrowser#setvarProc
	SetVariable setvarMinCols,pos={140,7},size={100,14}, disable=(prefs.options&1)==0, Proc=XYBrowser#setvarProc
	SetVariable setvarMinCols,title="Columns: Min",limits={2,Inf,1},value= _NUM:prefs.mincols
	SetVariable setvarMinCols,help={"set minimum number of columns for 2D waves to be shown in list"}
	SetVariable setvarMaxCols,pos={245,7},size={60,14}, disable=(prefs.options&1)==0, Proc=XYBrowser#setvarProc
	SetVariable setvarMaxCols,title="Max",limits={2,Inf,1},value= _NUM:prefs.maxcols
	SetVariable setvarMaxCols,help={"set maximum number of columns for 2D waves to be shown in list"}
	SetVariable setvarXCol,pos={310,7},size={40,14},title="X", Proc=XYBrowser#setvarProc
	SetVariable setvarXCol,limits={0,Inf,1},value= _NUM:prefs.xcol, disable=(prefs.options&1)==0
	SetVariable setvarXCol,help={"column containing x values"}
	SetVariable setvarYCol,pos={355,7},size={40,14},title="Y", Proc=XYBrowser#setvarProc
	SetVariable setvarYCol,limits={0,Inf,1},value= _NUM:prefs.ycol, disable=(prefs.options&1)==0
	SetVariable setvarYCol,help={"column containing y values"}
	
	ListBox ListBoxDataWaves,pos={10, 30},size={150, 240}, mode=9
	ListBox ListBoxDataWaves, focusRing=0
	
	string strPrefs
	StructPut /S prefs strPrefs
	SetWindow XYBrowserPanel userdata(prefs)=strPrefs
	
	MakeListIntoWaveSelector("XYBrowserPanel", "ListBoxDataWaves", content=WMWS_Waves, nameFilterProc="XYBrowser#filterFunc")
	WS_SetNotificationProc("XYBrowserPanel", "ListBoxDataWaves", "XYBrowser#WS_NotificationProc")	
	WS_OpenAFolderFully("XYBrowserPanel", "ListBoxDataWaves", GetDataFolder(1))
			
	CheckBox checkPrefs,pos={420,5},size={69,16},title="Graph Prefs"
	CheckBox checkPrefs,value=prefs.plotprefs, Proc=XYBrowser#checkProc
	CheckBox checkPrefs,help={"use preferences for plotting"}
	PopupMenu popupMode,pos={170,275},size={160,20}, Proc=XYBrowser#popupProc
	PopupMenu popupMode,mode=prefs.mode,value= #"\"Lines between points;Sticks to zero;Dots;Markers;Lines and markers;Bars;Cityscape;Fill to zero;Sticks and markers\""
	Button ButtonNewGraph,win=XYBrowserPanel, pos={460,275},size={80,20},title="New Graph"
	Button ButtonNewGraph,win=XYBrowserPanel, disable=2,Proc=XYbrowser#buttonProc
	Button ButtonNewGraph,win=XYBrowserPanel, help={"Display the selected X-Y pairs in a graph window"}
	Button ButtonAppendToTop,win=XYBrowserPanel, pos={350,275},size={100,20},title="Append to top"
	Button ButtonAppendToTop,win=XYBrowserPanel, disable=2, Proc=XYbrowser#buttonProc
	Button ButtonAppendToTop,win=XYBrowserPanel, help={"Add the selected X-Y pairs to the top graph window"}

	SetWindow XYBrowserPanel, activeChildFrame=0	
	SetWindow XYBrowserPanel hook(hFilterHook)=XYBrowser#FilterHook

	// resizing user data for controls
	PopupMenu popX,userdata(ResizeControlsInfo)= A"!!,@c!!#9W!!#@L!!#<Xz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"Qzz"
	PopupMenu popX,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	PopupMenu popX,userdata(ResizeControlsInfo) += A"zzz!!#u:DuaGlB6@p!AO6@szzzzzzzzzzzz!!!"
	SetVariable setvarKey,userdata(ResizeControlsInfo)= A"!!,Fq!!#:B!!#?Y!!#;mz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"qB6@p!AOL6U"
	SetVariable setvarKey,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	SetVariable setvarKey,userdata(ResizeControlsInfo) += A"zzz!!#u:DuaGlB6@p!AO6@szzzzzzzzzzzz!!!"
	SetVariable setvarKeySep,userdata(ResizeControlsInfo)= A"!!,Gq!!#:B!!#?m!!#;mz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"qB6@p!AOL6U"
	SetVariable setvarKeySep,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	SetVariable setvarKeySep,userdata(ResizeControlsInfo) += A"zzz!!#u:DuaGlB6@p!AO6@szzzzzzzzzzzz!!!"
	SetVariable setvarListSep,userdata(ResizeControlsInfo)= A"!!,H[!!#:B!!#?m!!#;mz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"qB6@p!AOL6U"
	SetVariable setvarListSep,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	SetVariable setvarListSep,userdata(ResizeControlsInfo) += A"zzz!!#u:DuaGlB6@p!AO6@szzzzzzzzzzzz!!!"
	SetVariable setvarMinCols,userdata(ResizeControlsInfo)= A"!!,Fq!!#:B!!#@,!!#;mz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"qB6@p!AOL6U"
	SetVariable setvarMinCols,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	SetVariable setvarMinCols,userdata(ResizeControlsInfo) += A"zzz!!#u:DuaGlB6@p!AO6@szzzzzzzzzzzz!!!"
	SetVariable setvarMaxCols,userdata(ResizeControlsInfo)= A"!!,H0!!#:B!!#?)!!#;mz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"qB6@p!AOL6U"
	SetVariable setvarMaxCols,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	SetVariable setvarMaxCols,userdata(ResizeControlsInfo) += A"zzz!!#u:DuaGlB6@p!AO6@szzzzzzzzzzzz!!!"
	SetVariable setvarXCol,userdata(ResizeControlsInfo)= A"!!,HV!!#:B!!#>.!!#;mz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"qB6@p!AOL6U"
	SetVariable setvarXCol,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	SetVariable setvarXCol,userdata(ResizeControlsInfo) += A"zzz!!#u:DuaGlB6@p!AO6@szzzzzzzzzzzz!!!"
	SetVariable setvarYCol,userdata(ResizeControlsInfo)= A"!!,HlJ,hjm!!#>.!!#;mz!!#](Aon\"Qzzzzzzzzzzzzzz!!#](Aon\"qB6@p!AOL6U"
	SetVariable setvarYCol,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	SetVariable setvarYCol,userdata(ResizeControlsInfo) += A"zzz!!#u:DuaGlB6@p!AO6@szzzzzzzzzzzz!!!"
	ListBox ListBoxDataWaves,userdata(ResizeControlsInfo)= A"!!,A.!!#=S!!#A%!!#B*z!!#](Aon\"Qzzzzzzzzzzzzzz!!#N3Bk1ctB6@p!AOL6U"
	ListBox ListBoxDataWaves,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Du]k<zzzzzzzzzzz"
	ListBox ListBoxDataWaves,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<!-N!dAO6@szzzzzzzzzzzz!!!"
	CheckBox checkPrefs,userdata(ResizeControlsInfo)= A"!!,I8!!#9W!!#?C!!#<8z!!#](Aon\"qB6@p!AOL6Uzzzzzzzzzzzz!!#](Aon\"qB6@p!AOL6U"
	CheckBox checkPrefs,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#u:Duafnzzzzzzzzzzz"
	CheckBox checkPrefs,userdata(ResizeControlsInfo) += A"zzz!!#u:Duafnzzzzzzzzzzzzzz!!!"
	PopupMenu popupMode,userdata(ResizeControlsInfo)= A"!!,G:!!#BCJ,hqV!!#<Xz!!#N3Bk1ctB6@p!AOL6Uzzzzzzzzzzzz!!#N3Bk1ctB6@p!AOL6U"
	PopupMenu popupMode,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	PopupMenu popupMode,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	Button ButtonAppendToTop,userdata(ResizeControlsInfo)= A"!!,Hj!!#BCJ,hpW!!#<Xz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	Button ButtonAppendToTop,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Button ButtonAppendToTop,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	Button ButtonNewGraph,userdata(ResizeControlsInfo)= A"!!,IL!!#BCJ,hp/!!#<Xz!!#o2B4uAezzzzzzzzzzzzzz!!#o2B4uAezz"
	Button ButtonNewGraph,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Button ButtonNewGraph,userdata(ResizeControlsInfo) += A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	Button buttonClear,userdata(ResizeControlsInfo)=A"!!,G#!!#BD!!#<(!!#<(z!!#N3Bk1ctAnQ,azzzzzzzzzzzzz!!#N3Bk1ctAnQ,aAOL6U"
	Button buttonClear,userdata(ResizeControlsInfo)+=A"zzzzzzzzzzzz!!#?(FEDG<zzzzzzzzzzz"
	Button buttonClear,userdata(ResizeControlsInfo)+=A"zzz!!#?(FEDG<zzzzzzzzzzzzzz!!!"
	
	// resizing user data for panel window
	SetWindow kwTopWin,userdata(ResizeControlsInfo)= A"!!*'\"z!!#CnJ,hs&zzzzzzzzzzzzzzzzzzzzz"
	SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzzzzzzzz"
	SetWindow kwTopWin,userdata(ResizeControlsInfo) += A"zzzzzzzzzzzzzzzzzzz!!!"
	SetWindow kwTopWin,userdata(ResizeControlsGuides)=  "guideR;guideDiv;guideL;guideT;guideB;"
	SetWindow kwTopWin,userdata(ResizeControlsInfoguideR)=  "NAME:guideR;WIN:XYBrowserPanel;TYPE:User;HORIZONTAL:0;POSITION:540.00;GUIDE1:FR;GUIDE2:;RELPOSITION:-10;"
	SetWindow kwTopWin,userdata(ResizeControlsInfoguideDiv)=  "NAME:guideDiv;WIN:XYBrowserPanel;TYPE:User;HORIZONTAL:0;POSITION:165.00;GUIDE1:FL;GUIDE2:Fr;RELPOSITION:0.3;"
	SetWindow kwTopWin,userdata(ResizeControlsInfoguideL)=  "NAME:guideL;WIN:XYBrowserPanel;TYPE:User;HORIZONTAL:0;POSITION:170.00;GUIDE1:guideDiv;GUIDE2:;RELPOSITION:5;"
	SetWindow kwTopWin,userdata(ResizeControlsInfoguideT)=  "NAME:guideT;WIN:XYBrowserPanel;TYPE:User;HORIZONTAL:1;POSITION:30.00;GUIDE1:FT;GUIDE2:;RELPOSITION:30;"
	SetWindow kwTopWin,userdata(ResizeControlsInfoguideB)=  "NAME:guideB;WIN:XYBrowserPanel;TYPE:User;HORIZONTAL:1;POSITION:270.00;GUIDE1:FB;GUIDE2:;RELPOSITION:-30;"
	SetWindow kwTopWin,userdata(ResizeControlsInfofgb)= "NAME:fgb;WIN:XYBrowserPanel;TYPE:User;HORIZONTAL:1;POSITION:293.00;GUIDE1:FB;GUIDE2:;RELPOSITION:-7;"
	SetWindow kwTopWin,userdata(ResizeControlsInfofgt)= "NAME:fgt;WIN:XYBrowserPanel;TYPE:User;HORIZONTAL:1;POSITION:276.00;GUIDE1:FB;GUIDE2:;RELPOSITION:-24;"
	SetWindow kwTopWin,userdata(ResizeControlsInfofgr)= "NAME:fgr;WIN:XYBrowserPanel;TYPE:User;HORIZONTAL:0;POSITION:145.00;GUIDE1:guideDiv;GUIDE2:;RELPOSITION:-20;"
	
	// resizing panel hook
	SetWindow XYBrowserPanel hook(ResizeControls)=ResizeControls#ResizeControlsHook
	
	return 1
end

static function checkProc(STRUCT WMCheckboxAction &s)
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	prefs.plotPrefs = s.checked
	savePrefs(prefs)
	Replot()
	return 0
end

static function savePrefs(STRUCT PackagePrefs &prefs)
	SavePackagePreferences ksPackageName, ksPrefsFileName, 0, prefs
	string strPrefs // save structure to a userdata string for quick retrieval
	StructPut /S prefs strPrefs
	SetWindow XYBrowserPanel userdata(prefs)=strPrefs
end

static function popupProc(STRUCT WMPopupAction &s)

	if (s.eventCode == -1)
		SaveWindowPosition(s.win)
	endif
	
	if (s.eventCode!=2)
		return 0
	endif
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	if (cmpstr(s.ctrlName, "popX")==0)
	
		prefs.options -= prefs.options&1
		prefs.options += (s.popNum==2)
		
		SetVariable setvarKey,win=$s.win,disable=prefs.options&1
		SetVariable setvarKeySep,win=$s.win,disable=prefs.options&1
		SetVariable setvarListSep,win=$s.win,disable=prefs.options&1
		SetVariable setvarMinCols,win=$s.win,disable=(prefs.options&1)==0
		SetVariable setvarMaxCols,win=$s.win,disable=(prefs.options&1)==0
		SetVariable setvarXCol,win=$s.win,disable=(prefs.options&1)==0
		SetVariable setvarYCol,win=$s.win,disable=(prefs.options&1)==0
	else
		prefs.mode = s.popNum
	endif
	savePrefs(prefs)
	
	WS_UpdateWaveSelectorWidget("XYBrowserPanel", "ListBoxDataWaves")
	Replot()
end

static function setvarProc(STRUCT WMSetVariableAction &s)
	
	if(s.eventCode<0)
		return 0
	endif
	
	if (s.eventCode == 7 && GrepString(s.ctrlName, "Sep$"))
		SetVariable $s.ctrlName,win=$s.win,value=_STR:char2backslash(s.sval)
		return 0
	elseif(s.eventCode == 8 && GrepString(s.ctrlName, "Sep$"))
		s.sval=backslash2char(s.sval)
		SetVariable $s.ctrlName,win=$s.win,value=_STR:s.sval
	endif
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	if (s.eventcode < 1 || s.eventcode > 6)
		return 0
	endif
	
	strswitch (s.ctrlName)
		case "setvarKey":
			prefs.key = s.sval
			break
		case "setvarKeySep":
			prefs.keysep = s.sval
			break
		case "setvarListSep":
			prefs.listsep = s.sval
			break
		case "setvarMinCols":
			prefs.mincols = s.dval
			break
		case "setvarMaxCols":
			prefs.maxcols = s.dval
			break
		case "setvarXCol":
			prefs.xcol = s.dval
			break
		case "setvarYCol":
			prefs.ycol = s.dval
			break
	endswitch
	savePrefs(prefs)
	WS_UpdateWaveSelectorWidget("XYBrowserPanel", "ListBoxDataWaves")
	Replot()
	return 0
end
	
// function used by WaveSelectorWidget
static function filterFunc(string aName, variable contents)
	
	// loading prefs from disk is too much of a bottleneck in the filter function
	STRUCT PackagePrefs prefs
	StructGet /S prefs GetUserData("XYBrowserPanel", "", "prefs")

	wave /Z w = $aname
	if (WaveExists(w) == 0 || WaveType(w,1) != 1)
		return 0
	endif
	
	if (prefs.options & 1)
		variable cols = DimSize(w, 1)
		if (cols< prefs.mincols || cols > prefs.maxcols)
			return 0
		endif
	elseif (WaveExists(getXWaveRef(w, prefs))==0)
		return 0
	endif
		
	string strFilter = GetUserData("XYBrowserPanel", "", "filter")
	string leafName = ParseFilePath(0, aName, ":", 1, 0)
	int vFilter = 0
	DebuggerOptions
	int sav_debug = V_debugOnError
	DebuggerOptions debugOnError=0
	try
		vFilter = GrepString(leafName, "(?i)" + strFilter); AbortOnRTE
	catch
		int err = GetRTError(1)
	endtry
	DebuggerOptions debugOnError=sav_debug
	return vFilter	
end

static function ClearText(int doIt)
	if(doIt) // clear filter widget
		Notebook XYBrowserPanel#nbFilter selection={startOfFile,endofFile}, textRGB=(50000,50000,50000), text="Filter"
		Notebook XYBrowserPanel#nbFilter selection={startOfFile,startOfFile}
		Button buttonClear, win=XYBrowserPanel, disable=3		
		SetWindow XYBrowserPanel userdata(filter)=""
	endif
end

// intercept and deal with keyboard events in notebook subwindow
// also deal with killing or resizing panel
static function FilterHook(STRUCT WMWinHookStruct &s)
	
	if(s.eventcode == 4) // mousemoved
		return 0
	endif
		
	GetWindow /Z XYBrowserPanel activeSW
	if (cmpstr(s_value, "XYBrowserPanel#nbFilter"))
		return 0
	endif
	
	if(s.eventCode == 22 && cmpstr(s.WinName, "XYBrowserPanel#nbFilter")==0) // don't allow scrolling
		return 1
	endif

	string strFilter = GetUserData("XYBrowserPanel", "", "filter")	
	int vLength = strlen(strFilter)
	
	if(s.eventcode==3 && vLength==0) // mousedown
		return 1 // don't allow mousedown when we have 'filter' displayed in nb
	endif
		
	if(s.eventcode == 10) // menu
		strswitch(s.menuItem)
			case "Paste":
				string strScrap = GetScrapText()
				strScrap = ReplaceString("\r", strScrap, "")
				strScrap = ReplaceString("\n", strScrap, "")
				strScrap = ReplaceString("\t", strScrap, "")
				
				GetSelection Notebook, XYBrowserPanel#nbFilter, 1 // get current position in notebook
				if(vLength == 0)
					Notebook XYBrowserPanel#nbFilter selection={startOfFile,endofFile}, text=strScrap
				else
					Notebook XYBrowserPanel#nbFilter selection={(0,V_startPos),(0,V_endPos)}, text=strScrap
				endif
				vLength += strlen(strScrap) - abs(V_endPos - V_startPos)
				s.eventcode = 11
				// pretend this was a keyboard event to allow execution to continue
				break
			case "Cut":
				GetSelection Notebook, XYBrowserPanel#nbFilter, 3 // get current position in notebook
				PutScrapText s_selection
				Notebook XYBrowserPanel#nbFilter selection={(0,V_startPos),(0,V_endPos)}, text=""
				vLength -= strlen(s_selection)
				s.eventcode = 11
				break
			case "Clear":
				GetSelection Notebook, XYBrowserPanel#nbFilter, 3 // get current position in notebook
				Notebook XYBrowserPanel#nbFilter selection={(0,V_startPos),(0,V_endPos)}, text="" // clear text
				vLength -= strlen(s_selection)
				s.eventcode = 11
				break
		endswitch
		Button buttonClear, win=XYBrowserPanel, disable=3*(vLength == 0)
		ClearText((vLength == 0))
	endif
				
	if(s.eventcode!=11)
		return 0
	endif
	
	if(vLength == 0) // Remove "Filter" text before starting to deal with keyboard activity
		Notebook XYBrowserPanel#nbFilter selection={startOfFile,endofFile}, text=""
	endif
	
	// deal with some non-printing characters
	switch(s.keycode)
		case 9:	// tab: jump to end
		case 3:
		case 13: // enter or return: jump to end
			Notebook XYBrowserPanel#nbFilter selection={endOfFile,endofFile}
			break
		case 28: // left arrow
		case 29: // right arrow
			ClearText((vLength == 0)); return (vLength == 0)
		case 8:
		case 127: // delete or forward delete
			GetSelection Notebook, XYBrowserPanel#nbFilter, 1
			if(V_startPos == V_endPos)
				V_startPos -= (s.keycode == 8)
				V_endPos += (s.keycode == 127)
			endif
			V_startPos = min(vLength,V_startPos); V_endPos = min(vLength,V_endPos)
			V_startPos = max(0, V_startPos); V_endPos = max(0, V_endPos)
			Notebook XYBrowserPanel#nbFilter selection={(0,V_startPos),(0,V_endPos)}, text=""
			vLength -= abs(V_endPos - V_startPos)
			break
	endswitch
		
	// find and save current position
	GetSelection Notebook, XYBrowserPanel#nbFilter, 1
	int selEnd = V_endPos
		
	if(strlen(s.keyText) == 1) // a one-byte printing character
		// insert character into current selection
		Notebook XYBrowserPanel#nbFilter text=s.keyText, textRGB=(0,0,0)
		vLength += 1 - abs(V_endPos - V_startPos)
		// find out where we want to leave cursor
		GetSelection Notebook, XYBrowserPanel#nbFilter, 1
		selEnd = V_endPos
	endif
	
	// select and format text
	Notebook XYBrowserPanel#nbFilter selection={startOfFile,endOfFile}, textRGB=(0,0,0)
	// store text in panel userdata
	GetSelection Notebook, XYBrowserPanel#nbFilter, 3
	SetWindow XYBrowserPanel userdata(filter) = s_selection
	
	Notebook XYBrowserPanel#nbFilter selection={(0,selEnd),(0,selEnd)}, findText={"",1}
	
	Button buttonClear, win=XYBrowserPanel, disable=3*(vLength==0)
	ClearText((vLength == 0))
	
	WS_UpdateWaveSelectorWidget("XYBrowserPanel", "ListBoxDataWaves")
	Replot()
	
	return 1 // tell Igor we've handled all keyboard events
end

// PNG: width= 90, height= 30
static Picture ClearTextPicture
	ASCII85Begin
	M,6r;%14!\!!!!.8Ou6I!!!"&!!!!?#R18/!3BT8GQ7^D&TgHDFAm*iFE_/6AH5;7DfQssEc39jTBQ
	=U"$&q@5u_NKm@(_/W^%DU?TFO*%Pm('G1+?)0-OWfgsSqYDhC]>ST`Z)0"D)K8@Ncp@>C,GnA#([A
	Jb0q`hu`4_P;#bpi`?T]j@medQ0%eKjbh8pO.^'LcCD,L*6P)3#odh%+r"J\$n:)LVlTrTIOm/oL'r
	#E&ce=k6Fiu8aXm1/:;hm?p#L^qI6J;D8?ZBMB_D14&ANkg9GMLR*Xs"/?@4VWUdJ,1MBB0[bECn33
	KZ1__A<"/u9(o<Sf@<$^stNom5GmA@5SIJ$^\D=(p8G;&!HNh)6lgYLfW6>#jE3aT_'W?L>Xr73'A#
	m<7:2<I)2%%Jk$'i-7@>Ml+rPk4?-&B7ph6*MjH9&DV+Uo=D(4f6)f(Z9"SdCXSlj^V?0?8][X1#pG
	[0-Dbk^rVg[Rc^PBH/8U_8QFCWdi&3#DT?k^_gU>S_]F^9g'7.>5F'hcYV%X?$[g4KPRF0=NW^$Z(L
	G'1aoAKLpqS!ei0kB29iHZJgJ(_`LbUX%/C@J6!+"mVIs6V)A,gbdt$K*g_X+Um(2\?XI=m'*tR%i"
	,kQIh51]UETI+HBA)DV:t/sl4<N*#^^=N.<B%00/$P>lNVlic"'Jc$p^ou^SLA\BS?`$Jla/$38;!#
	Q+K;T6T_?\3*d?$+27Ri'PYY-u]-gEMR^<d.ElNUY$#A@tX-ULd\IU&bfX]^T)a;<u7HgR!i2]GBpt
	SiZ1;JHl$jf3!k*jJlX$(ZroR:&!&8D[<-`g,)N96+6gSFVi$$Gr%h:1ioG%bZgmgbcp;2_&rF/[l"
	Qr^V@O-"j&UsEk)HgI'9`W31Wh"3^O,KrI/W'chm_@T!!^1"Y*Hknod`FiW&N@PIluYQdKILa)RK=W
	Ub4(Y)ao_5hG\K-+^73&AnBNQ+'D,6!KY/`F@6)`,V<qS#*-t?,F98]@h"8Y7Kj.%``Q=h4.L(m=Nd
	,%6Vs`ptRkJNBdbpk]$\>hR4"[5SF8$^:q=W([+TB`,%?4h7'ET[Y6F!KJ3fH"9BpILuUI#GoI.rl(
	_DAn[OiS_GkcL7QT`\p%Sos;F.W#_'g]3!!!!j78?7R6=>B
	ASCII85End
end

static function WS_NotificationProc(string SelectedItem, variable EventCode)
	if (eventCode == 4 || eventCode==5)
		Replot()
	endif
end

// updates plot and buttons
static function Replot()
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
	
	string waves = WS_SelectedObjectsList("XYBrowserPanel", "ListBoxDataWaves")
	int numWaves = ItemsInList(waves)
	string s_topgraph = WinList("*", ";", "WIN:1")
	
	Button buttonNewGraph, win=XYBrowserPanel, disable=2*(numWaves==0)
	Button buttonAppendToTop, win=XYBrowserPanel, disable=2*( (numWaves==0) || (strlen(s_topgraph)==0) )
	
	clearPlot()
	Preferences /Q prefs.plotPrefs
	int savPref = V_flag
	
	int i
	for(i=0;i<numWaves;i+=1)
		wave /Z w = $StringFromList(i, waves)
		if (prefs.options & 1)
			AppendToGraph /W=XYBrowserPanel#PreviewPlot w[][prefs.ycol] vs w[][prefs.xcol]
		else
			wave /Z w_x = getXWaveRef(w, prefs)
			if(WaveExists(w) && WaveExists(w_x))
				AppendToGraph /W=XYBrowserPanel#PreviewPlot w vs w_x
			endif
		endif
	endfor
	SetAxis /Z/A/W=XYBrowserPanel#PreviewPlot
	ModifyGraph /W=XYBrowserPanel#PreviewPlot mode=prefs.mode-1
	Preferences /Q savPref
	return 0
end

static function clearPlot()
	#if IgorVersion() >= 9
		RemoveFromGraph /Z/W=XYBrowserPanel#PreviewPlot/ALL
	#else
		string plotList=TraceNameList("XYBrowserPanel#PreviewPlot",";",1)
		int i
		for (i=ItemsInList(plotList)-1;i>=0;i--))
			RemoveFromGraph /Z/W=XYBrowserPanel#PreviewPlot StringFromList(i, plotlist)
		endfor
	#endif
end

static function buttonProc(STRUCT WMbuttonAction &s)
		
	if (s.eventCode != 2)
		return 0
	endif
	
	STRUCT PackagePrefs prefs
	LoadPrefs(prefs)
		
	int hist = 1 // choose whether to record graph building commands to history
			
	if (cmpstr(s.ctrlName, "ButtonClear") == 0)
		ClearText(1)
		WS_UpdateWaveSelectorWidget("XYBrowserPanel", "ListBoxDataWaves")
		return 0
	endif
	
	if (cmpstr(s.ctrlName, "buttonNewGraph") == 0)
		Display
		if (hist)
			Print "•Display"
		endif
	endif
	
	string waves = WS_SelectedObjectsList("XYBrowserPanel", "ListBoxDataWaves")
	int numWaves = ItemsInList(waves)
	int i
	for(i=0;i<numWaves;i+=1)
		wave /Z w = $StringFromList(i, waves)
		if (prefs.options & 1)
			AppendToGraph w[][prefs.ycol] vs w[][prefs.xcol]
			if (hist) // record the graph building commands to history
				printf "•AppendToGraph %s[%d] vs %s[%d]\r", GetWavesDataFolder(w,4), prefs.ycol, GetWavesDataFolder(w,4), prefs.xcol
			endif
		else
			wave /Z w_x = getXWaveRef(w, prefs)
			if(WaveExists(w) && WaveExists(w_x))
				AppendToGraph w vs w_x
				if (hist) // record the graph building commands to history
					printf "•AppendToGraph %s vs %s\r", GetWavesDataFolder(w,4), GetWavesDataFolder(w_x,4)
				endif
			endif
		endif
	endfor
	
	if(kRaman && stringmatch(s.ctrlName, "buttonNewGraph"))
		Label/Z left "Intensity"
		Label/Z bottom "Wavenumber (cm\\S-1\\M)"
	endif
	return 0
end

static function /S backslash2char(string str)
	str = ReplaceString("\\r", str, "\r")
	str = ReplaceString("\\t", str, "\t")
	str = ReplaceString("\\n", str, "\n")
	return str
end

static function /S char2backslash(string str)
	str = ReplaceString("\r", str, "\\r")
	str = ReplaceString("\t", str, "\\t")
	str = ReplaceString("\n", str, "\\n")
	return str
end

// Provides a quick way to pair x and y waves to test the X-Y Browser
// Right-click on a trace in an X-Y plot and select "Pair X-wave as
// plotted." If both waves are in the same data folder the pair should now
// be selectable in the X-Y browser panel.
menu "TracePopup", dynamic
	XYbrowser#XYTraceMenu("Pair X-wave as plotted"), /Q,  XYbrowser#PairTrace()
end

menu "AllTracesPopup"
	"Pair all traces with X-waves as plotted", /Q,  XYbrowser#PairTrace()
end

static function PairTrace()
	// figure out graph and trace names
	GetLastUserMenuInfo  // sets s_tracename, s_graphname
	
	if (strlen(s_tracename)==0)
		s_tracename=traceNameList(S_graphName,";",1)
	endif
	string trace = ""
	int i
	for (i=itemsinlist(s_tracename)-1;i>=0;i--)
		trace = stringfromlist(i, S_traceName)
		wave /Z w_x = XWaveRefFromTrace(S_graphName,trace)
		wave w = TraceNameToWaveRef(S_graphName,trace)
		if (!waveexists(w_x))
			continue
		endif
		note /K w, ReplaceStringByKey("Xwave", note(w), NameOfWave(w_x), ":", "\r")
	endfor
	return 1
end

static function /s XYTraceMenu(string s)
	if (WinType("") != 1)
		return "" // don't do anything if Igor is just rebuilding the menu
	endif
	// figure out graph and trace names
	GetLastUserMenuInfo
	return SelectString(WaveExists(XWaveRefFromTrace(s_graphname, s_tracename)), "", s)
end