// Operation template: ccp4unpack /M /N[=name:basename] /O /P=name:pathName [string:fileNameStr]

#include "XOPStandardHeaders.h"			// Include ANSI headers, Mac headers, IgorXOP.h, XOP.h and XOPSupport.h

#include "ccp4.h"
#include "pack_c.h"

// Runtime param structure for ccp4unpack operation.
#include "XOPStructureAlignmentTwoByte.h"

struct ccp4unpackRuntimeParams {
    // Flag parameters.

    // Parameters for /M flag group.
    int MFlagEncountered;
    // There are no fields for this group because it has no parameters.
    
    // Parameters for /N flag group.
    int NFlagEncountered;
    char NBaseName[MAX_OBJ_NAME+1];			// Optional parameter.
    int NFlagParamsSet[1];
    
    // Parameters for /O flag group.
    int OFlagEncountered;
    // There are no fields for this group because it has no parameters.
    
    // Parameters for /P flag group.
    int PFlagEncountered;
    char pathName[MAX_OBJ_NAME+1];
    int PFlagParamsSet[1];
    
    // Main parameters.
    
    // Parameters for simple main group #0.
    int fileNameStrEncountered;
    Handle fileNameStr;					// Optional parameter.
    int fileNameStrParamsSet[1];
    
    // These are postamble fields that Igor sets.
    int calledFromFunction;					// 1 if called from a user function, 0 otherwise.
    int calledFromMacro;					// 1 if called from a macro, 0 otherwise.
};

typedef struct ccp4unpackRuntimeParams ccp4unpackRuntimeParams;
typedef struct ccp4unpackRuntimeParams* ccp4unpackRuntimeParamsPtr;
#include "XOPStructureAlignmentReset.h"

#ifdef __cplusplus
extern "C" int Executeccp4unpack(ccp4unpackRuntimeParamsPtr p);		// This is needed for C++ only.
#endif

#ifdef _WINDOWS_
	#define snprintf _snprintf
#endif

/* Some general defines: */

/*	GetLoadFile(initialDir, fullFilePath)
 
 GetLoadFile puts up a standard open dialog to get the name of the file
 that the user wants to open.
 
 initialDir is a full path to the directory to initially display in the
 dialog or "" if you don't care.
 
 It returns -1 if the user cancels or 0 if the user clicks Open.
 If the user clicks Open, returns the full path via fullFilePath.
 */
static int
GetLoadFile(const char *initialDir, char fullFilePath[MAX_PATH_LEN+1])
{
#ifdef _MACINTOSH_
    char* filterStr = "All Files:****:;";
#endif
#ifdef _WINDOWS_
    char* filterStr = "\0\0";	
#endif
    char * prompt = "Looking for a CCP4 compressed binary image file (e.g., a MAR345 image plate file)";
    int result;
    
    static int fileIndex = 1;		// This preserves the setting across invocations of the dialog. A more sophisticated XOP would save this as a preference.
    
    *fullFilePath = 0;				// Must be preset to zero because on Windows XOPOpenFileDialog passes this to GetOpenFileName which requires that it be preset.
    
    result = XOPOpenFileDialog(prompt, filterStr, &fileIndex, initialDir, fullFilePath);

    return result;
}

static int
ccp4unpack(int calledFromFunction, long fileLoaderFlags, char* baseName, const char* symbolicPathName, const char* fileParam, int isMARfile)
{
    char symbolicPathPath[MAX_PATH_LEN+1];		// Full path to the folder that the symbolic path refers to. This is a native path with trailing colon (Mac) or backslash (Win).
    char nativeFilePath[MAX_PATH_LEN+1];		// Full path to the file to load. Native path.
    char fileSystemFilePath[MAX_PATH_LEN+1];		// Full path to the file to load. file system path (POSIX on Mac OS X).
    int err = 0;
    
    long	dimensionSizes[MAX_DIMENSIONS+1];
    char	outName[MAX_OBJ_NAME+1];
    long	suffixNum = 0;
    int		nameChanged;
    int		doOverwrite;
    waveHndl	waveH;
    int		hState, hState2;
    long	dataOffset;
    WORD *	imageP;
    
    *symbolicPathPath = 0;
    if (*symbolicPathName != 0) {
	if (err = GetPathInfo2(symbolicPathName, symbolicPathPath))
	    return err;
    }
    
    if (GetFullPathFromSymbolicPathAndFilePath(symbolicPathName, fileParam, nativeFilePath) != 0)
	fileLoaderFlags |= FILE_LOADER_INTERACTIVE;		// Need dialog to get file name.
    
    if (!FullPathPointsToFile(nativeFilePath))			// File does not exist or path is bad?
	fileLoaderFlags |= FILE_LOADER_INTERACTIVE;		// Need dialog to get file name.
    
    if (fileLoaderFlags & FILE_LOADER_INTERACTIVE) {
	if (GetLoadFile(symbolicPathPath, nativeFilePath))
	    return -1;							// User cancelled.
    }
    
    if (err = SetFileLoaderOperationOutputVariables(calledFromFunction, nativeFilePath, 0, ""))	// Initialize Igor output variables.
	return err;
    
    if (*baseName == 0) {
	if (err = GetLeafName(nativeFilePath, baseName))
	    return err;
    }
    
    WatchCursor();

#ifdef _MACINTOSH_
    HFSToPosixPath(nativeFilePath, fileSystemFilePath, 0);
#else
    strncpy(fileSystemFilePath, nativeFilePath, sizeof(fileSystemFilePath) - 1);
#endif

    MemClear(dimensionSizes, sizeof(dimensionSizes));
    
    imsiz_c(fileSystemFilePath, &dimensionSizes[0], &dimensionSizes[1]);

    err = CreateValidDataObjectName(NULL, baseName, outName, &suffixNum, WAVE_OBJECT, 1, 
				    fileLoaderFlags & FILE_LOADER_OVERWRITE, 0, 1, &nameChanged, &doOverwrite);
    if (err) {
	return err;
    }
    
    err = MDMakeWave(&waveH, outName, NULL, dimensionSizes, NT_I16 | NT_UNSIGNED, fileLoaderFlags & FILE_LOADER_OVERWRITE);
    if (err) {
	return err;
    }
    
    if (err = MDAccessNumericWaveData(waveH, kMDWaveAccessMode0, &dataOffset)) {
	return err;
    }
    
    hState = MoveLockHandle(waveH); // Lock handle so data won’t move. 
    
    imageP = (WORD*)((char*)(*waveH) + dataOffset);
    readpack_word_c(imageP, fileSystemFilePath);

    if (isMARfile) {
	XOP_FILE_REF fileRef;
	Handle	noteH;
	int		i;
	unsigned long	BOM;
	int		swap;
	int		overflows;
		
	// Assign wave note from file header
	
	if (err = XOPOpenFile(nativeFilePath, 0, &fileRef))
	    return err;
	
	if (err = XOPSetFilePosition(fileRef, 128, -1))
	    return err;

	noteH = NewHandle(4096 - 128);
	hState2 = MoveLockHandle(noteH); // Lock handle so data won’t move. 
	if (err = XOPReadFile2(fileRef, 4096 - 128, *noteH, NULL))
	    return err;
	
	for (i = 0; i < 4096 - 128; i++) {
	    /* can't use '\n' and '\r' because the C "standard" is insane */
	    if (*noteH[i] == 10) {
		*noteH[i] == 13;
	    }
	}

	HSetState(noteH, hState2);

	SetWaveNote(waveH, noteH);
	
	// Apply overflows
	
	if (err = XOPSetFilePosition(fileRef, 0, -1))
	    return err;

	BOM = getw(fileRef);
	
	if (BOM == 1234) {
	    swap = FALSE;
	} else {
	    FixByteOrder(&BOM, sizeof(int), 1);
	    if (BOM == 1234) {
		swap = TRUE;
	    } else {
		return BAD_BINARY_FILE;
	    }
	}
	
	if (err = XOPSetFilePosition(fileRef, 2 * sizeof(int), -1))
	    return err;

	overflows = getw(fileRef);

	if (swap) {
	    FixByteOrder(&overflows, sizeof(int), 1);
	}
	
	if (err = XOPSetFilePosition(fileRef, 4096, -1))
	    return err;
	
	for (i = 0; i < overflows; i++) {
	    int	a, x, y;
	    
	    a = getw(fileRef);
	    if (swap) {
		FixByteOrder(&a, sizeof(int), 1);
	    }

	    x = a / dimensionSizes[1];
	    y = a % dimensionSizes[1];

	    a = getw(fileRef);
	    if (swap) {
		FixByteOrder(&a, sizeof(int), 1);
	    }

	    if (x >= 0 && x < dimensionSizes[0] &&
		y >= 0 && y < dimensionSizes[1])
		
		imageP[x * dimensionSizes[1] + y] = a;
	    
	    else
		
		return BAD_INDEX;
	}	     
	
	XOPCloseFile(fileRef);
    }
    
    HSetState(waveH, hState);
    
    ArrowCursor();
    
    // Store standard file loader output globals.
    if (err == 0) {
	char	waveNames[MAX_OBJ_NAME+2];
	
	snprintf(waveNames, sizeof(waveNames)-1, "%s;", outName);
	err = SetFileLoaderOperationOutputVariables(calledFromFunction, nativeFilePath, 1, waveNames);
    }
    
    return err;
}

static int
Executeccp4unpack(ccp4unpackRuntimeParamsPtr p)
{
    long fileLoaderFlags;
    char baseName[MAX_OBJ_NAME+1];
    char symbolicPathName[MAX_OBJ_NAME+1];
    char fileName[MAX_PATH_LEN+1];
    int err = 0;
    
    fileLoaderFlags = 0;
    *baseName = 0;
    *symbolicPathName = 0;
    *fileName = 0;
    
    // Flag parameters.
    
    if (p->NFlagEncountered) {
	fileLoaderFlags |= FILE_LOADER_AUTONAME;
	strcpy(baseName, "wave");
	if (p->NFlagParamsSet[0]) {
	    if (*p->NBaseName != 0)
		strcpy(baseName, p->NBaseName);
	}
    }
    
    if (p->OFlagEncountered)
	fileLoaderFlags |= FILE_LOADER_OVERWRITE;
    
    if (p->PFlagEncountered) {
	strcpy(symbolicPathName, p->pathName);
	
	if (*symbolicPathName != 0) {						// /P=$"" is a like no /P at all.
	    char symbolicPathPath[MAX_PATH_LEN+1];
	    
	    fileLoaderFlags |= FILE_LOADER_PATH;
	    
	    // This is just to check if the symbolic path name is valid.
	    if (err = GetPathInfo2(symbolicPathName, symbolicPathPath))
		return err;
	}
    }
    
    // Main parameters.
    
    if (p->fileNameStrEncountered) {
	if (err = GetCStringFromHandle(p->fileNameStr, fileName, sizeof(fileName)-1))
	    return err;
	if (err = GetNativePath(fileName, fileName))
	    return err;
    }

    return ccp4unpack(p->calledFromFunction, fileLoaderFlags, baseName, symbolicPathName, fileName, p->MFlagEncountered);
}

static int
Registerccp4unpack(void)
{
    char* cmdTemplate;
    char* runtimeNumVarList;
    char* runtimeStrVarList;
    
    // NOTE: If you change this template, you must change the ccp4unpackRuntimeParams structure as well.
    cmdTemplate = "ccp4unpack /M /N[=name:basename] /O /P=name:pathName [string:fileNameStr]";
    runtimeNumVarList = "V_flag;"; 
    runtimeStrVarList = "S_path;S_fileName;S_waveNames;"; 
    return RegisterOperation(cmdTemplate, runtimeNumVarList, runtimeStrVarList, sizeof(ccp4unpackRuntimeParams), Executeccp4unpack, 0);
}


static int
RegisterOperations(void)		// Register any operations with Igor.
{
	int result;
	
	// Register ccp4unpack operation.
	if (result = Registerccp4unpack())
		return result;
	
	// There are no more operations added by this XOP.
		
	return 0;
}

/*	XOPMenuItem()
 
 XOPMenuItem is called when the XOP's menu item is selected.
 */
static int
XOPMenuItem(void)
{
    int	    menuID, itemID;
    char    baseName[MAX_OBJ_NAME+1];
    char    symbolicPathName[MAX_OBJ_NAME+1];
    long    fileLoaderFlags;
    int	    isMARfile;
    char    fileName[MAX_PATH_LEN+1];
    int	    err = 0;
    
    fileLoaderFlags = 0;
    *baseName = 0;
    *symbolicPathName = 0;
    *fileName = 0;
    
    menuID = GetXOPItem(0);
    itemID = GetXOPItem(1);
    
    itemID = ActualToResourceItem(menuID, itemID);
    switch (itemID) {
	case 1:
	    isMARfile = FALSE;
	    break;
	case 2:
	    isMARfile = TRUE;
	    break;
    }
    
    err = ccp4unpack(0, fileLoaderFlags, baseName, symbolicPathName, fileName, isMARfile);
    return err;
}

/*	XOPEntry()

	This is the entry point from the host application to the XOP for all
	messages after the INIT message.
*/
static void
XOPEntry(void)
{	
	long result = 0;
	
	switch (GetXOPMessage()) {
	    case CLEANUP:								// XOP about to be disposed of.
		break;
		
	    case MENUITEM:								// XOPs menu item selected.
		result = XOPMenuItem();
		SetXOPType(TRANSIENT);					// XOP is done now.
		break;
	}
	SetXOPResult(result);
}

/*	main(ioRecHandle)

	This is the initial entry point at which the host application calls XOP.
	The message sent by the host must be INIT.
	
	main does any necessary initialization and then sets the XOPEntry field of the
	ioRecHandle to the address to be called for future messages.
*/
HOST_IMPORT void
main(IORecHandle ioRecHandle)
{
	int result;
	
	XOPInit(ioRecHandle);							// Do standard XOP initialization.

	SetXOPEntry(XOPEntry);							// Set entry point for future calls.

	if (result = RegisterOperations()) {
		SetXOPResult(result);
		return;
	}
	
	SetXOPResult(0);
}
